/*
 * Copyright (C) 2020 The zfoo Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except
 * in compliance with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License is distributed
 * on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and limitations under the License.
 */

package com.zfoo.storage.convert;

import com.zfoo.protocol.util.JsonUtils;
import com.zfoo.protocol.util.ReflectionUtils;
import com.zfoo.protocol.util.StringUtils;
import org.springframework.core.convert.TypeDescriptor;
import org.springframework.core.convert.converter.ConditionalGenericConverter;

import java.util.Collections;
import java.util.Set;

/**
 * 转换一个String到一个POJO对象，且这个对象不能继承如何接口
 *
 * @author godotg
 */
public class JsonToObjectConverter implements ConditionalGenericConverter {

    public static final JsonToObjectConverter INSTANCE = new JsonToObjectConverter();

    private JsonToObjectConverter() {
    }

    @Override
    public boolean matches(TypeDescriptor sourceType, TypeDescriptor targetType) {
        if (sourceType.getType() != String.class) {
            return false;
        }

        if (targetType.getType().isPrimitive()) {
            return false;
        }

        if (Number.class.isAssignableFrom(targetType.getType())) {
            return false;
        }

        if (CharSequence.class.isAssignableFrom(targetType.getType())) {
            return false;
        }

        return ReflectionUtils.isPojoClass(targetType.getType());
    }

    @Override
    public Set<ConvertiblePair> getConvertibleTypes() {
        return Collections.singleton(new ConvertiblePair(String.class, Object.class));
    }

    @Override
    public Object convert(Object source, TypeDescriptor sourceType, TypeDescriptor targetType) {
        String content = (String) source;
        if (StringUtils.isBlank(content)) {
            return null;
        }
        return JsonUtils.string2Object(content, targetType.getType());
    }
}
