<?php
/**
 * PrestaShop security vulnerability checker.
 *
 * @author    Maksim T. <zapalm@yandex.com>
 * @copyright 2019 Maksim T.
 * @license   https://opensource.org/licenses/MIT MIT
 * @link      https://github.com/zapalm/prestashop-security-vulnerability-checker GitHub
 * @link      https://prestashop.modulez.ru/en/tools-scripts/70-security-vulnerability-checker.html Homepage
 */

namespace zapalm;

use Db;

/**
 * PrestaShop security vulnerability checker.
 *
 * @author Maksim T. <zapalm@yandex.com>
 */
class PrestaShopSecurityVulnerabilityChecker
{
    /** The possibility of a vulnerability: exists. */
    const VULNERABILITY_EXISTS     = 'Exists';
    /** The possibility of a vulnerability: not exists. */
    const VULNERABILITY_NOT_EXISTS = 'Not exists';
    /** The possibility of a vulnerability: possible. */
    const VULNERABILITY_POSSIBLE   = 'Possible';

    /**
     * Returns an installed version of a module.
     *
     * @param int $moduleName The module name.
     *
     * @return string|null The installed version of the module or null, if the module isn't installed.
     *
     * @author Maksim T. <zapalm@yandex.com>
     *
     * @noinspection SqlResolve
     * @noinspection SqlNoDataSourceInspection
     */
    protected function getModuleInstalledVersion($moduleName) 
    {
        $version = Db::getInstance()->getValue('SELECT m.version FROM ' . _DB_PREFIX_ . 'module m WHERE m.name = "' . pSQL($moduleName) . '"');

        if (false === is_string($version)) {
            return null;
        }

        return $version;
    }

    /**
     * Prints a given report to a browser or to a console.
     *
     * @param array $report The report given by check().
     *
     * @author Maksim T. <zapalm@yandex.com>
     */
    public function printReport(array $report)
    {
        $message = 'SECURITY CHECK:' . PHP_EOL;

        $issuesCount = 0;
        foreach ($report as $vulnerabilityId => $testResult) {
            if (self::VULNERABILITY_NOT_EXISTS !== $testResult[0]) {
                $issuesCount++;
            }

            $message .= '[' . $testResult[0] . '] : ' . $vulnerabilityId . ' : ' . $testResult[1] . PHP_EOL;
        }

        $message .= '------' . PHP_EOL;
        if ($issuesCount > 0) {
            $message .= 'ATTENTION: Your website has ' . $issuesCount . ' security issues!' . PHP_EOL;
        } else {
            $message .= 'Your website has no security issues.' . PHP_EOL;
        }
        $message .= 'The vulnerability with the score of 10 is the most dangerous and vice versa for the score of 1.';

        if ('cli' !== php_sapi_name()) {
            $message = '<pre>' . $message . '</pre>';
        }

        echo $message;
    }

    /**
     * Returns files found under the directory specified by a pattern.
     *
     * @param string $pattern The directory specified by the pattern.
     *
     * @return string[] Files found directory specified by the pattern.
     *
     * @see glob() for pattern examples.
     *
     * @author Maksim T. <zapalm@yandex.com>
     */
    protected function findFiles($pattern)
    {
        $directory = realpath(dirname($pattern));
        if (false === $directory) {
            return array();
        }

        $pattern = $directory . DIRECTORY_SEPARATOR . basename($pattern);
        $files   = glob($pattern);

        $dirs = glob(dirname($pattern) . DIRECTORY_SEPARATOR . '*', GLOB_ONLYDIR | GLOB_NOSORT);
        if (is_array($dirs) && count($dirs)) {
            foreach ($dirs as $dir) {
                $files = array_merge($files, $this->findFiles($dir . DIRECTORY_SEPARATOR . basename($pattern)));
            }
        }

        return $files;
    }

    /**
     * Checks vulnerability issues.
     *
     * @return array The check result.
     *
     * @author Maksim T. <zapalm@yandex.com>
     */
    public function check()
    {
        return array_merge( // Add a new method's call to the top of the array (sort by newest vulnerability issues).
            $this->check_CVE_2022_31181(),
            $this->check_CVE_2018_19355(),
            $this->check_CVE_2018_19126(),
            $this->check_CVE_2018_13784(),
            $this->check_CVE_2018_8824(),
            $this->check_CVE_2018_7491(),
            $this->check_CVE_2018_5682(),
            $this->check_CVE_2018_5681(),
            $this->check_CVE_2017_9841(),
            $this->check_CVE_2015_1175(),
            $this->check_CVE_2012_6641(),
            $this->check_CVE_2012_5801(),
            $this->check_CVE_2011_4545(),
            $this->check_CVE_2011_4544(),
            $this->check_CVE_2011_3796(),
            $this->check_CVE_2008_5791(),
            $this->check_outdated()
        );
    }

    /**
     * Checks security vulnerability: CVE-2022-31181.
     *
     * @return array
     *
     * @see https://prestashop.modulez.ru/en/news/60
     * @see https://nvd.nist.gov/vuln/detail/CVE-2022-31181
     * @see https://github.com/PrestaShop/blockwishlist/pull/178/files
     *
     * @author Maksim T. <zapalm@yandex.com>
     */
    protected function check_CVE_2022_31181()
    {
        $testResult = self::VULNERABILITY_NOT_EXISTS;
        $moduleName = 'blockwishlist';
        if (file_exists(_PS_MODULE_DIR_ . $moduleName . '/' . $moduleName . '.php')) {
            $moduleVersion = $this->getModuleInstalledVersion($moduleName);
            if (null !== $moduleVersion
                && version_compare($moduleVersion, '2.0.0', '>=')
                && version_compare($moduleVersion, '2.1.0', '<=')
            ) {
                $testResult = self::VULNERABILITY_EXISTS;
            }
        }

        $report['CVE-2022-31181'] = array(
            $testResult,
            implode(' ', array(
                'Security vulnerability (9.8/10) with the "Wishlist" module v2.0.0-2.1.0.',
                'Solution: update the module or fix a vulnerable file.',
            )),
        );

        return $report;
    }

    /**
     * Checks security vulnerability: CVE-2018-19355.
     *
     * @return array
     *
     * @see https://www.cvedetails.com/cve/CVE-2018-19355/
     *
     * @author Maksim T. <zapalm@yandex.com>
     */
    protected function check_CVE_2018_19355()
    {
        $testResult = self::VULNERABILITY_NOT_EXISTS;
        if (file_exists(_PS_MODULE_DIR_ . 'orderfiles/upload.php')) {
            $testResult = self::VULNERABILITY_POSSIBLE;
        }

        $report['CVE-2018-19355'] = array(
            $testResult,
            implode(' ', array(
                'Security vulnerability (7.5/10) with the module OrderFiles.',
                'Solution: update the module, remove or fix vulnerable files.',
                'There is no information about vulnerable OrderFiles module version and about the author of the module.',
                'If you have the information, contact with me: https://prestashop.modulez.ru/en/contact-us?id_product=70',
            )),
        );

        return $report;
    }

    /**
     * Checks security vulnerability: CVE-2018-19126, CVE-2018-19125, CVE-2018-19124.
     *
     * @return array
     *
     * @see https://www.cvedetails.com/cve/CVE-2018-19126/
     * @see https://www.cvedetails.com/cve/CVE-2018-19125/
     * @see https://www.cvedetails.com/cve/CVE-2018-19124/
     *
     * @author Maksim T. <zapalm@yandex.com>
     */
    protected function check_CVE_2018_19126()
    {
        $report = array();

        $testResult = self::VULNERABILITY_NOT_EXISTS;
        if ((1.6 === (float)_PS_VERSION_ && version_compare(_PS_VERSION_, '1.6.1.23', '<')) || (1.7 === (float)_PS_VERSION_ && version_compare(_PS_VERSION_, '1.7.4.4', '<'))) {
            if (extension_loaded('phar') && false === in_array(strtolower(ini_get('phar.readonly')), array('1', 'on', 'yes', 'true'), true)) {
                $testResult = self::VULNERABILITY_EXISTS;
            } else {
                $testResult = self::VULNERABILITY_POSSIBLE;
            }
        }

        $report['CVE-2018-19126 | CVE-2018-19125 | CVE-2018-19124'] = array(
            $testResult,
            'Security vulnerability (7.5/10). The explanation: https://prestashop.modulez.ru/en/news/51',
        );

        return $report;
    }

    /**
     * Checks security vulnerability: CVE-2018-13784.
     *
     * @return array
     *
     * @see https://www.cvedetails.com/cve/CVE-2018-13784/
     *
     * @author Maksim T. <zapalm@yandex.com>
     */
    protected function check_CVE_2018_13784()
    {
        $report = array();

        $testResult = self::VULNERABILITY_NOT_EXISTS;
        if ((1.6 === (float)_PS_VERSION_ && version_compare(_PS_VERSION_, '1.6.1.20', '<')) || (1.7 === (float)_PS_VERSION_ && version_compare(_PS_VERSION_, '1.7.3.4', '<'))) {
            $testResult = self::VULNERABILITY_POSSIBLE;
        }

        $report['CVE-2018-13784'] = array(
            $testResult,
            'Security vulnerability (6.4/10). More info: http://build.prestashop.com/news/prestashop-1-7-3-4-1-6-1-20-maintenance-releases/',
        );

        return $report;
    }

    /**
     * Checks security vulnerability: CVE-2018-8824, CVE-2018-8823.
     *
     * @return array
     *
     * @see https://www.cvedetails.com/cve/CVE-2018-8824/
     * @see https://www.cvedetails.com/cve/CVE-2018-8823/
     *
     * @author Maksim T. <zapalm@yandex.com>
     */
    protected function check_CVE_2018_8824()
    {
        $testResult = self::VULNERABILITY_NOT_EXISTS;
        if (file_exists(_PS_MODULE_DIR_ . 'bamegamenu/ajax_phpcode.php')) {
            $moduleVersion = $this->getModuleInstalledVersion('bamegamenu');
            if (null !== $moduleVersion) {
                if (version_compare($moduleVersion, '1.0.32', '<')) {
                    $testResult = self::VULNERABILITY_POSSIBLE;
                } elseif (version_compare($moduleVersion, '1.0.32', '=')) {
                    $testResult = self::VULNERABILITY_EXISTS;
                }
            }
        }

        $report['CVE-2018-8824 | CVE-2018-8823'] = array(
            $testResult,
            implode(' ', array(
                'Security vulnerability (7.5/10) with the "Responsive Mega Menu (Horizontal+Vertical+Dropdown) Pro" module v1.0.32.',
                'Solution: update the module, remove or fix vulnerable files.',
                'The module on Addons: https://addons.prestashop.com/en/menu/20464-responsive-mega-menu-horizontaldropdownswipesticky.html',
            )),
        );

        return $report;
    }

    /**
     * Checks security vulnerability: CVE-2018-7491.
     *
     * @return array
     *
     * @see https://www.cvedetails.com/cve/CVE-2018-7491/
     *
     * @author Maksim T. <zapalm@yandex.com>
     */
    protected function check_CVE_2018_7491()
    {
        $report = array();

        $testResult = self::VULNERABILITY_NOT_EXISTS;
        if (1.7 === (float)_PS_VERSION_) { // @todo: May be all versions of PS are vulnerable.
            // @todo: Currently this issue is not resolved.
            $testResult = self::VULNERABILITY_EXISTS;
        }

        $report['CVE-2018-7491'] = array(
            $testResult,
            'Security vulnerability (5/10). This is UI-Redressing/Clickjacking vulnerability. More info: https://github.com/PrestaShop/PrestaShop/pull/8807',
        );

        return $report;
    }

    /**
     * Checks security vulnerability: CVE-2018-5682.
     *
     * @return array
     *
     * @see https://www.cvedetails.com/cve/CVE-2018-5682/
     *
     * @author Maksim T. <zapalm@yandex.com>
     */
    protected function check_CVE_2018_5682()
    {
        $report = array();

        // @todo: Currently this issue is not resolved. All versions of PrestaShop are vulnerable, except 1.7 front-office (may be not stable releases of 1.7 are also vulnerable).
        $testResult = self::VULNERABILITY_EXISTS;

        $report['CVE-2018-5682'] = array(
            $testResult,
            'Security vulnerability (5/10). The attacker can review existent emails of employees and customers. More info: http://forge.prestashop.com/browse/BOOM-4613',
        );

        return $report;
    }

    /**
     * Checks security vulnerability: CVE-2018-5681.
     *
     * @return array
     *
     * @see https://www.cvedetails.com/cve/CVE-2018-5681/
     *
     * @author Maksim T. <zapalm@yandex.com>
     */
    protected function check_CVE_2018_5681()
    {
        $report = array();

        // All versions of PrestaShop are vulnerable.
        // Currently this XSS issue is not resolved fully, but it has the small score.
        $testResult = self::VULNERABILITY_EXISTS;

        $report['CVE-2018-5681'] = array(
            $testResult,
            'Security vulnerability (3.5/10). More info: http://forge.prestashop.com/browse/BOOM-4612',
        );

        return $report;
    }

    /**
     * Checks security vulnerability: CVE-2017-9841.
     *
     * @return array
     *
     * @see https://www.cvedetails.com/cve/CVE-2017-9841/
     *
     * @author Maksim T. <zapalm@yandex.com>
     */
    protected function check_CVE_2017_9841()
    {
        $matchPattern = implode('|', array(
            '^eval-stdin.php$',
            '^sssp.php$',
            '^kill.php$',
            '^0x666.php$',
            '^f.php$',
            '^traber*',
            '^h0d3_g4nt3ng*',
            '^Raiz0WorM*',
            '^RCE_VULN*',
            '^probing*',
            '^Xsam_Xadoo*',
            '^XsamXadoo*',
            '^phpXXX000sdf*',
        ));

        $foundFiles = array();
        foreach ($this->findFiles(realpath(_PS_ROOT_DIR_) . DIRECTORY_SEPARATOR . '*') as $file) {
            if (1 === preg_match('/(' . $matchPattern . ')/i', basename($file))) {
                $shortPath = str_replace(realpath(_PS_ROOT_DIR_), '', $file);
                $shortPath = str_replace('\\', '/', $shortPath);

                $foundFiles[] = $shortPath;
            }
        }

        if (count($foundFiles)) {
            $testResult = self::VULNERABILITY_EXISTS;
            $message    = 'Potentially malicious files found: "' . implode(', ', $foundFiles) . '".';
        } else {
            $testResult = self::VULNERABILITY_NOT_EXISTS;
            $message    = '';
        }

        return array(
            'CVE-2017-9841' => array(
                $testResult,
                implode(' ', array(
                    'Security vulnerability (7.5/10).',
                    $message,
                    'Solution: remove or fix vulnerable files.',
                    'More info: https://prestashop.modulez.ru/en/news/58',
                )),
            ),
        );
    }

    /**
     * Checks security vulnerability: CVE-2015-1175.
     *
     * @return array
     *
     * @see https://www.cvedetails.com/cve/CVE-2015-1175/
     *
     * @author Maksim T. <zapalm@yandex.com>
     */
    protected function check_CVE_2015_1175()
    {
        $testResult = self::VULNERABILITY_NOT_EXISTS;
        if (file_exists(_PS_MODULE_DIR_ . 'blocklayered/blocklayered-ajax.php')) {
            $moduleVersion = $this->getModuleInstalledVersion('blocklayered');
            if (null !== $moduleVersion && version_compare($moduleVersion, '2.0.7', '<')) {
                $testResult = self::VULNERABILITY_EXISTS;
            }
        }

        $report['CVE-2015-1175'] = array(
            $testResult,
            implode(' ', array(
                'Security vulnerability (4.3/10) with the BlockLayered module v2.0.5 or less.',
                'Solution: update the module, remove or fix vulnerable files.',
                'The module on GitHub: https://github.com/PrestaShop/blocklayered',
            )),
        );

        return $report;
    }

    /**
     * Checks security vulnerability: CVE-2012-6641.
     *
     * @return array
     *
     * @see https://www.cvedetails.com/cve/CVE-2012-6641/
     *
     * @author Maksim T. <zapalm@yandex.com>
     */
    protected function check_CVE_2012_6641()
    {
        $testResult = self::VULNERABILITY_NOT_EXISTS;
        if (version_compare(_PS_VERSION_, '1.4.7.2', '<') && file_exists(_PS_MODULE_DIR_ . 'socolissimo/redirect.php')) {
            $testResult = self::VULNERABILITY_EXISTS;
        }

        $report['CVE-2012-6641'] = array(
            $testResult,
            implode(' ', array(
                'Security vulnerability (5/10) with the SoColissimo module in PrestaShop before 1.4.7.2.',
                'Solution: update the module, remove or fix vulnerable files.',
                'The module on GitHub: https://github.com/quadra-informatique/SoColissimo-3.x-Prestashop',
            )),
        );

        return $report;
    }

    /**
     * Checks security vulnerability: CVE-2012-5801, CVE-2012-5800, CVE-2012-5799.
     *
     * @return array
     *
     * @see https://www.cvedetails.com/cve/CVE-2012-5801/
     * @see https://www.cvedetails.com/cve/CVE-2012-5800/
     * @see https://www.cvedetails.com/cve/CVE-2012-5799/
     *
     * @author Maksim T. <zapalm@yandex.com>
     */
    protected function check_CVE_2012_5801()
    {
        $testResult = self::VULNERABILITY_NOT_EXISTS;
        if (file_exists(_PS_MODULE_DIR_ . 'paypal/paypal.php') || file_exists(_PS_MODULE_DIR_ . 'ebay/ebay.php') || file_exists(_PS_MODULE_DIR_ . 'canadapost/canadapost.php')) {
            // @todo: There is no information about issue for a concrete version of PrestaShop and a version of PayPal, ebay, CanadaPost (Presto-Changeo) modules.
            $testResult = self::VULNERABILITY_POSSIBLE;
        }

        $report['CVE-2012-5801 | CVE-2012-5800 | CVE-2012-5799'] = array(
            $testResult,
            implode(' ', array(
                'Security vulnerability (5.8/10) with the PayPal, ebay, CanadaPost (Presto-Changeo) modules.',
                'There is no information about issue for a concrete version of PrestaShop and a version of PayPal, ebay, CanadaPost (Presto-Changeo) modules.',
                'More info: https://www.cs.utexas.edu/~shmat/shmat_ccs12.pdf',
            )),
        );

        return $report;
    }

    /**
     * Checks security vulnerability: CVE-2011-4545.
     *
     * @return array
     *
     * @see https://www.cvedetails.com/cve/CVE-2011-4545/
     *
     * @author Maksim T. <zapalm@yandex.com>
     */
    protected function check_CVE_2011_4545()
    {
        $testResult = self::VULNERABILITY_NOT_EXISTS;
        if (version_compare(PHP_VERSION, '5.1.2', '<')) {
            $testResult = self::VULNERABILITY_EXISTS;
        }

        $report['CVE-2011-4545'] = array(
            $testResult,
            implode(' ', array(
                'Security vulnerability (5/10).',
                'Solution: update your PHP to at least the last of 5.6.',
                'More info: https://www.dognaedis.com/vulns/DGS-SEC-7.html',
            )),
        );

        return $report;
    }

    /**
     * Checks security vulnerability: CVE-2011-4544.
     *
     * @return array
     *
     * @see https://www.cvedetails.com/cve/CVE-2011-4544/
     *
     * @author Maksim T. <zapalm@yandex.com>
     */
    protected function check_CVE_2011_4544()
    {
        $testResult = self::VULNERABILITY_NOT_EXISTS;
        if (1.4 === (float)_PS_VERSION_) {
            if (file_exists(_PS_MODULE_DIR_ . 'mondialrelay/kit_mondialrelay/RechercheDetailPointRelais_ajax.php') || file_exists(_PS_MODULE_DIR_ . 'mondialrelay/kit_mondialrelay/SuiviExpedition_ajax.php')) {
                $testResult = self::VULNERABILITY_EXISTS;
            } else {
                $testResult = self::VULNERABILITY_POSSIBLE;
            }
        }

        $report['CVE-2011-4544'] = array(
            $testResult,
            implode(' ', array(
                'Security vulnerability (4.3/10) with the "mondialrelay" module and "ajaxfilemanager" script.',
                'More info: https://www.dognaedis.com/vulns/DGS-SEC-5.html',
            )),
        );

        return $report;
    }

    /**
     * Checks security vulnerability: CVE-2011-3796.
     *
     * @return array
     *
     * @see https://www.cvedetails.com/cve/CVE-2011-3796/
     *
     * @author Maksim T. <zapalm@yandex.com>
     */
    protected function check_CVE_2011_3796()
    {
        $testResult = self::VULNERABILITY_NOT_EXISTS;
        if (1.4 === (float)_PS_VERSION_ && version_compare(_PS_VERSION_, '1.4.0.6', '<=')) {
            $testResult = self::VULNERABILITY_EXISTS;
        }

        $report['CVE-2011-3796'] = array(
            $testResult,
            implode(' ', array(
                'Security vulnerability (5/10).',
                'Allows remote attackers to obtain sensitive information via a direct request to a .php file, which reveals the installation path in an error message.',
            )),
        );

        return $report;
    }

    /**
     * Checks security vulnerability: CVE-2008-6503, CVE-2008-5791.
     *
     * @return array
     *
     * @see https://www.cvedetails.com/cve/CVE-2008-6503/
     * @see https://www.cvedetails.com/cve/CVE-2008-5791/
     *
     * @author Maksim T. <zapalm@yandex.com>
     */
    protected function check_CVE_2008_5791()
    {
        $testResult = self::VULNERABILITY_NOT_EXISTS;
        if (1.1 === (float)_PS_VERSION_) {
            $testResult = self::VULNERABILITY_EXISTS;
        }

        $report['CVE-2008-5791 | CVE-2008-6503'] = array(
            $testResult,
            implode(' ', array(
                'Security vulnerability (10/10).',
                'Solution: upgrade or remove your PrestaShop 1.1 ;)',
            )),
        );

        return $report;
    }

    /**
     * Checks PrestaShop is outdated.
     *
     * @return array
     *
     * @author Maksim T. <zapalm@yandex.com>
     */
    protected function check_outdated()
    {
        $testResult = self::VULNERABILITY_NOT_EXISTS;
        if ((float)_PS_VERSION_ < 1.6) {
            $testResult = self::VULNERABILITY_POSSIBLE;
        }

        $report['Outdated version of PrestaShop'] = array(
            $testResult,
            implode(' ', array(
                'Actual versions of PrestaShop are 1.6 and 1.7.',
                'Solution: upgrade. How to do this: https://prestashop.modulez.ru/en/services/28-prestashop-upgrade-to-newest-16-17-by-zapalm-4-hours.html',
            )),
        );

        return $report;
    }
}
