unit HlpSnefru;

{$I ..\Include\HashLib.inc}

interface

uses
  SysUtils,
  HlpHashLibTypes,
{$IFDEF DELPHI}
  HlpHash,
  HlpHashBuffer,
{$ENDIF DELPHI}
  HlpBits,
  HlpHashSize,
  HlpConverters,
  HlpIHash,
  HlpIHashInfo,
  HlpHashCryptoNotBuildIn,
  HlpArrayUtils;

resourcestring
  SInvalidSnefruLevel =
    'Snefru Security Level Cannot be Less than 1. Standard Level is 8';
  SInvalidSnefruHashSize =
    'Snefru HashSize Must be Either 128 bit(16 byte) or 256 bit(32 byte)';
  SInvalidHashSize = 'Specified HashSize Is Invalid or UnSupported "%d"';

type

  TSnefru = class sealed(TBlockHash, ICryptoNotBuildIn, ITransformBlock)

  strict private
  var
    FState: THashLibUInt32Array;
    FSecurityLevel: Int32;

  const
    SShifts: array [0 .. 3] of Int32 = (16, 8, 16, 24);

    class var

      FSBoxes: THashLibMatrixUInt32Array;

    class constructor Snefru();

    function GetSnefruHashSize(AHashSize: Int32): THashSize; inline;

  strict protected
    function GetName: String; override;
    function GetResult(): THashLibByteArray; override;
    procedure Finish(); override;
    procedure TransformBlock(AData: PByte; ADataLength: Int32;
      AIndex: Int32); override;

  public
    /// <summary>
    ///
    /// </summary>
    /// <param name="ASecurityLevel">any Integer value greater than 0. Standard is 8. </param>
    /// <param name="AHashSize">128bit, 256bit</param>
    /// <returns></returns>
    constructor Create(ASecurityLevel: Int32; AHashSize: THashSize);
    procedure Initialize(); override;
    function Clone(): IHash; override;

  end;

implementation

{ TSnefru }

function TSnefru.GetSnefruHashSize(AHashSize: Int32): THashSize;
begin
  case AHashSize of
    16:
      Result := THashSize.hsHashSize128;
    32:
      Result := THashSize.hsHashSize256
  else
    begin
      raise EArgumentInvalidHashLibException.CreateResFmt(@SInvalidHashSize,
        [AHashSize]);
    end;
  end;
end;

function TSnefru.Clone(): IHash;
var
  LHashInstance: TSnefru;
begin
  LHashInstance := TSnefru.Create(FSecurityLevel, GetSnefruHashSize(HashSize));
  LHashInstance.FState := System.Copy(FState);
  LHashInstance.FBuffer := FBuffer.Clone();
  LHashInstance.FProcessedBytesCount := FProcessedBytesCount;
  Result := LHashInstance as IHash;
  Result.BufferSize := BufferSize;
end;

constructor TSnefru.Create(ASecurityLevel: Int32; AHashSize: THashSize);
begin
  Inherited Create(Int32(AHashSize), 64 - (Int32(AHashSize)));
  FSecurityLevel := Int32(ASecurityLevel);
  System.SetLength(FState, HashSize shr 2);
end;

procedure TSnefru.Finish;
var
  LBits: UInt64;
  LPadIndex: Int32;
  LPad: THashLibByteArray;
begin
  LBits := FProcessedBytesCount * 8;
  if FBuffer.Position > 0 then
  begin
    LPadIndex := 2 * BlockSize - FBuffer.Position - 8
  end
  else
  begin
    LPadIndex := BlockSize - FBuffer.Position - 8;
  end;

  System.SetLength(LPad, LPadIndex + 8);

  LBits := TConverters.be2me_64(LBits);

  TConverters.ReadUInt64AsBytesLE(LBits, LPad, LPadIndex);

  LPadIndex := LPadIndex + 8;

  TransformBytes(LPad, 0, LPadIndex);
end;

function TSnefru.GetName: String;
begin
  Result := Format('%s_%u_%u', [Self.ClassName, FSecurityLevel,
    Self.HashSize * 8]);
end;

function TSnefru.GetResult: THashLibByteArray;
begin
  System.SetLength(Result, System.Length(FState) * System.SizeOf(UInt32));
  TConverters.be32_copy(PCardinal(FState), 0, PByte(Result), 0,
    System.Length(Result));
end;

procedure TSnefru.Initialize;
begin
  TArrayUtils.ZeroFill(FState);
  Inherited Initialize();
end;

procedure TSnefru.TransformBlock(AData: PByte; ADataLength: Int32;
  AIndex: Int32);
var
  LSBox0, LSBox1: THashLibUInt32Array;
  LIdx, LJdx, LKdx, LShift: Int32;
  LWork: array [0 .. 15] of UInt32;
  LPtrWork: PCardinal;
begin
  LPtrWork := @(LWork[0]);
  System.Move(FState[0], LWork[0], System.Length(FState) *
    System.SizeOf(UInt32));

  TConverters.be32_copy(AData, AIndex, LPtrWork + System.Length(FState), 0,
    ADataLength);

  LIdx := 0;

  while LIdx < FSecurityLevel do
  begin
    LSBox0 := FSBoxes[LIdx * 2 + 0];
    LSBox1 := FSBoxes[LIdx * 2 + 1];

    LJdx := 0;
    while LJdx < 4 do
    begin
      LWork[15] := LWork[15] xor LSBox0[Byte(LWork[0])];
      LWork[1] := LWork[1] xor LSBox0[Byte(LWork[0])];
      LWork[0] := LWork[0] xor LSBox0[Byte(LWork[1])];
      LWork[2] := LWork[2] xor LSBox0[Byte(LWork[1])];
      LWork[1] := LWork[1] xor LSBox1[Byte(LWork[2])];
      LWork[3] := LWork[3] xor LSBox1[Byte(LWork[2])];
      LWork[2] := LWork[2] xor LSBox1[Byte(LWork[3])];
      LWork[4] := LWork[4] xor LSBox1[Byte(LWork[3])];
      LWork[3] := LWork[3] xor LSBox0[Byte(LWork[4])];
      LWork[5] := LWork[5] xor LSBox0[Byte(LWork[4])];
      LWork[4] := LWork[4] xor LSBox0[Byte(LWork[5])];
      LWork[6] := LWork[6] xor LSBox0[Byte(LWork[5])];
      LWork[5] := LWork[5] xor LSBox1[Byte(LWork[6])];
      LWork[7] := LWork[7] xor LSBox1[Byte(LWork[6])];
      LWork[6] := LWork[6] xor LSBox1[Byte(LWork[7])];
      LWork[8] := LWork[8] xor LSBox1[Byte(LWork[7])];
      LWork[7] := LWork[7] xor LSBox0[Byte(LWork[8])];
      LWork[9] := LWork[9] xor LSBox0[Byte(LWork[8])];
      LWork[8] := LWork[8] xor LSBox0[Byte(LWork[9])];
      LWork[10] := LWork[10] xor LSBox0[Byte(LWork[9])];
      LWork[9] := LWork[9] xor LSBox1[Byte(LWork[10])];
      LWork[11] := LWork[11] xor LSBox1[Byte(LWork[10])];
      LWork[10] := LWork[10] xor LSBox1[Byte(LWork[11])];
      LWork[12] := LWork[12] xor LSBox1[Byte(LWork[11])];
      LWork[11] := LWork[11] xor LSBox0[Byte(LWork[12])];
      LWork[13] := LWork[13] xor LSBox0[Byte(LWork[12])];
      LWork[12] := LWork[12] xor LSBox0[Byte(LWork[13])];
      LWork[14] := LWork[14] xor LSBox0[Byte(LWork[13])];
      LWork[13] := LWork[13] xor LSBox1[Byte(LWork[14])];
      LWork[15] := LWork[15] xor LSBox1[Byte(LWork[14])];
      LWork[14] := LWork[14] xor LSBox1[Byte(LWork[15])];
      LWork[0] := LWork[0] xor LSBox1[Byte(LWork[15])];

      LShift := SShifts[LJdx];

      LKdx := 0;

      while LKdx < 16 do
      begin
        LWork[LKdx] := TBits.RotateRight32(LWork[LKdx], LShift);
        System.Inc(LKdx);
      end;

      System.Inc(LJdx);
    end;

    System.Inc(LIdx);
  end;

  FState[0] := FState[0] xor LWork[15];
  FState[1] := FState[1] xor LWork[14];
  FState[2] := FState[2] xor LWork[13];
  FState[3] := FState[3] xor LWork[12];

  if (HashSize = 32) then
  begin
    FState[4] := FState[4] xor LWork[11];
    FState[5] := FState[5] xor LWork[10];
    FState[6] := FState[6] xor LWork[9];
    FState[7] := FState[7] xor LWork[8];
  end;

  System.FillChar(LWork, System.SizeOf(LWork), UInt32(0));
end;

class constructor TSnefru.Snefru;
begin
{$REGION 'Consts'}
  FSBoxes := THashLibMatrixUInt32Array.Create
    (THashLibUInt32Array.Create($64F9001B, $FEDDCDF6, $7C8FF1E2, $11D71514,
    $8B8C18D3, $DDDF881E, $6EAB5056, $88CED8E1, $49148959, $69C56FD5, $B7994F03,
    $0FBCEE3E, $3C264940, $21557E58, $E14B3FC2, $2E5CF591, $DCEFF8CE, $092A1648,
    $BE812936, $FF7B0C6A, $D5251037, $AFA448F1, $7DAFC95A, $1EA69C3F, $A417ABE7,
    $5890E423, $B0CB70C0, $C85025F7, $244D97E3, $1FF3595F, $C4EC6396, $59181E17,
    $E635B477, $354E7DBF, $796F7753, $66EB52CC, $77C3F995, $32E3A927, $80CCAED6,
    $4E2BE89D, $375BBD28, $AD1A3D05, $2B1B42B3, $16C44C71, $4D54BFA8, $E57DDC7A,
    $EC6D8144, $5A71046B, $D8229650, $87FC8F24, $CBC60E09, $B6390366, $D9F76092,
    $D393A70B, $1D31A08A, $9CD971C9, $5C1EF445, $86FAB694, $FDB44165, $8EAAFCBE,
    $4BCAC6EB, $FB7A94E5, $5789D04E, $FA13CF35, $236B8DA9, $4133F000, $6224261C,
    $F412F23B, $E75E56A4, $30022116, $BAF17F1F, $D09872F9, $C1A3699C, $F1E802AA,
    $0DD145DC, $4FDCE093, $8D8412F0, $6CD0F376, $3DE6B73D, $84BA737F, $B43A30F2,
    $44569F69, $00E4EACA, $B58DE3B0, $959113C8, $D62EFEE9, $90861F83, $CED69874,
    $2F793CEE, $E8571C30, $483665D1, $AB07B031, $914C844F, $15BF3BE8, $2C3F2A9A,
    $9EB95FD4, $92E7472D, $2297CC5B, $EE5F2782, $5377B562, $DB8EBBCF, $F961DEDD,
    $C59B5C60, $1BD3910D, $26D206AD, $B28514D8, $5ECF6B52, $7FEA78BB, $504879AC,
    $ED34A884, $36E51D3C, $1753741D, $8C47CAED, $9D0A40EF, $3145E221, $DA27EB70,
    $DF730BA3, $183C8789, $739AC0A6, $9A58DFC6, $54B134C1, $AC3E242E, $CC493902,
    $7B2DDA99, $8F15BC01, $29FD38C7, $27D5318F, $604AAFF5, $F29C6818, $C38AA2EC,
    $1019D4C3, $A8FB936E, $20ED7B39, $0B686119, $89A0906F, $1CC7829E, $9952EF4B,
    $850E9E8C, $CD063A90, $67002F8E, $CFAC8CB7, $EAA24B11, $988B4E6C, $46F066DF,
    $CA7EEC08, $C7BBA664, $831D17BD, $63F575E6, $9764350E, $47870D42, $026CA4A2,
    $8167D587, $61B6ADAB, $AA6564D2, $70DA237B, $25E1C74A, $A1C901A0, $0EB0A5DA,
    $7670F741, $51C05AEA, $933DFA32, $0759FF1A, $56010AB8, $5FDECB78, $3F32EDF8,
    $AEBEDBB9, $39F8326D, $D20858C5, $9B638BE4, $A572C80A, $28E0A19F, $432099FC,
    $3A37C3CD, $BF95C585, $B392C12A, $6AA707D7, $52F66A61, $12D483B1, $96435B5E,
    $3E75802B, $3BA52B33, $A99F51A5, $BDA1E157, $78C2E70C, $FCAE7CE0, $D1602267,
    $2AFFAC4D, $4A510947, $0AB2B83A, $7A04E579, $340DFD80, $B916E922, $E29D5E9B,
    $F5624AF4, $4CA9D9AF, $6BBD2CFE, $E3B7F620, $C2746E07, $5B42B9B6, $A06919BC,
    $F0F2C40F, $72217AB5, $14C19DF3, $F3802DAE, $E094BEB4, $A2101AFF, $0529575D,
    $55CDB27C, $A33BDDB2, $6528B37D, $740C05DB, $E96A62C4, $40782846, $6D30D706,
    $BBF48E2C, $BCE2D3DE, $049E37FA, $01B5E634, $2D886D8D, $7E5A2E7E, $D7412013,
    $06E90F97, $E45D3EBA, $B8AD3386, $13051B25, $0C035354, $71C89B75, $C638FBD0,
    $197F11A1, $EF0F08FB, $F8448651, $38409563, $452F4443, $5D464D55, $03D8764C,
    $B1B8D638, $A70BBA2F, $94B3D210, $EB6692A7, $D409C2D9, $68838526, $A6DB8A15,
    $751F6C98, $DE769A88, $C9EE4668, $1A82A373, $0896AA49, $42233681, $F62C55CB,
    $9F1C5404, $F74FB15C, $C06E4312, $6FFE5D72, $8AA8678B, $337CD129,
    $8211CEFD), THashLibUInt32Array.Create($074A1D09, $52A10E5A, $9275A3F8,
    $4B82506C, $37DF7E1B, $4C78B3C5, $CEFAB1DA, $F472267E, $B63045F6, $D66A1FC0,
    $400298E3, $27E60C94, $87D2F1B8, $DF9E56CC, $45CD1803, $1D35E098, $CCE7C736,
    $03483BF1, $1F7307D7, $C6E8F948, $E613C111, $3955C6FF, $1170ED7C, $8E95DA41,
    $99C31BF4, $A4DA8021, $7B5F94FB, $DD0DA51F, $6562AA77, $556BCB23, $DB1BACC6,
    $798040B9, $BFE5378F, $731D55E6, $DAA5BFEE, $389BBC60, $1B33FBA4, $9C567204,
    $36C26C68, $77EE9D69, $8AEB3E88, $2D50B5CE, $9579E790, $42B13CFC, $33FBD32B,
    $EE0503A7, $B5862824, $15E41EAD, $C8412EF7, $9D441275, $2FCEC582, $5FF483B7,
    $8F3931DF, $2E5D2A7B, $49467BF9, $0653DEA9, $2684CE35, $7E655E5C, $F12771D8,
    $BB15CC67, $AB097CA1, $983DCF52, $10DDF026, $21267F57, $2C58F6B4, $31043265,
    $0BAB8C01, $D5492099, $ACAAE619, $944CE54A, $F2D13D39, $ADD3FC32, $CDA08A40,
    $E2B0D451, $9EFE08AE, $B9D50FD2, $EA5CD7FD, $C9A749DD, $13EA2253, $832DEBAA,
    $24BE640F, $E03E926A, $29E01CDE, $8BF59F18, $0F9D00B6, $E1238B46, $1E7D8E34,
    $93619ADB, $76B32F9F, $BD972CEC, $E31FA976, $A68FBB10, $FB3BA49D, $8587C41D,
    $A5ADD1D0, $F3CF84BF, $D4E11150, $D9FFA6BC, $C3F6018C, $AEF10572, $74A64B2F,
    $E7DC9559, $2AAE35D5, $5B6F587F, $A9E353FE, $CA4FB674, $04BA24A8, $E5C6875F,
    $DCBC6266, $6BC5C03F, $661EEF02, $ED740BAB, $058E34E4, $B7E946CF, $88698125,
    $72EC48ED, $B11073A3, $A13485EB, $A2A2429C, $FA407547, $50B76713, $5418C37D,
    $96192DA5, $170BB04B, $518A021E, $B0AC13D1, $0963FA2A, $4A6E10E1, $58472BDC,
    $F7F8D962, $979139EA, $8D856538, $C0997042, $48324D7A, $447623CB, $8CBBE364,
    $6E0C6B0E, $D36D63B0, $3F244C84, $3542C971, $2B228DC1, $CB0325BB, $F8C0D6E9,
    $DE11066B, $A8649327, $FC31F83E, $7DD80406, $F916DD61, $D89F79D3, $615144C2,
    $EBB45D31, $28002958, $56890A37, $F05B3808, $123AE844, $86839E16, $914B0D83,
    $C506B43C, $CF3CBA5E, $7C60F5C9, $22DEB2A0, $5D9C2715, $C77BA0EF, $4F45360B,
    $C1017D8B, $E45ADC29, $A759909B, $412CD293, $D7D796B1, $00C8FF30, $23A34A80,
    $4EC15C91, $714E78B5, $47B9E42E, $78F3EA4D, $7F078F5B, $346C593A, $A3A87A1A,
    $9BCBFE12, $3D439963, $B2EF6D8E, $B8D46028, $6C2FD5CA, $62675256, $01F2A2F3,
    $BC96AE0A, $709A8920, $B4146E87, $6308B9E2, $64BDA7BA, $AFED6892, $6037F2A2,
    $F52969E0, $0ADB43A6, $82811400, $90D0BDF0, $19C9549E, $203F6A73, $1ACCAF4F,
    $89714E6D, $164D4705, $67665F07, $EC206170, $0C2182B2, $A02B9C81, $53289722,
    $F6A97686, $140E4179, $9F778849, $9A88E15D, $25CADB54, $D157F36F, $32A421C3,
    $B368E98A, $5A92CD0D, $757AA8D4, $C20AC278, $08B551C7, $849491E8, $4DC75AD6,
    $697C33BE, $BAF0CA33, $46125B4E, $59D677B3, $30D9C8F2, $D0AF860C, $1C7FD0FA,
    $FE0FF72C, $5C8D6F43, $57FDEC3B, $6AB6AD97, $D22ADF89, $18171785, $02BFE22D,
    $6DB80917, $80B216AF, $E85E4F9A, $7A1C306E, $6FC49BF5, $3AF7A11C, $81E215E7,
    $68363FCD, $3E9357C8, $EF52FD55, $3B8BAB4C, $3C8CF495, $BEFCEEBD, $FD25B714,
    $C498D83D, $0D2E1A8D, $E9F966AC, $0E387445, $435419E5, $5E7EBEC4, $AA90B8D9,
    $FF1A3A96), THashLibUInt32Array.Create($4A8FE4E3, $F27D99CD, $D04A40CA,
    $CB5FF194, $3668275A, $FF4816BE, $A78B394C, $4C6BE9DB, $4EEC38D2, $4296EC80,
    $CDCE96F8, $888C2F38, $E75508F5, $7B916414, $060AA14A, $A214F327, $BE608DAF,
    $1EBBDEC2, $61F98CE9, $E92156FE, $4F22D7A3, $3F76A8D9, $559A4B33, $38AD2959,
    $F3F17E9E, $85E1BA91, $E5EBA6FB, $73DCD48C, $F5C3FF78, $481B6058, $8A3297F7,
    $8F1F3BF4, $93785AB2, $477A4A5B, $6334EB5D, $6D251B2E, $74A9102D, $07E38FFA,
    $915C9C62, $CCC275EA, $6BE273EC, $3EBDDD70, $D895796C, $DC54A91B, $C9AFDF81,
    $23633F73, $275119B4, $B19F6B67, $50756E22, $2BB152E2, $76EA46A2, $A353E232,
    $2F596AD6, $0B1EDB0B, $02D3D9A4, $78B47843, $64893E90, $40F0CAAD, $F68D3AD7,
    $46FD1707, $1C9C67EF, $B5E086DE, $96EE6CA6, $9AA34774, $1BA4F48A, $8D01ABFD,
    $183EE1F6, $5FF8AA7A, $17E4FAAE, $303983B0, $6C08668B, $D4AC4382, $E6C5849F,
    $92FEFB53, $C1CAC4CE, $43501388, $441118CF, $EC4FB308, $53A08E86, $9E0FE0C5,
    $F91C1525, $AC45BE05, $D7987CB5, $49BA1487, $57938940, $D5877648, $A958727F,
    $58DFE3C3, $F436CF77, $399E4D11, $F0A5BFA9, $EF61A33B, $A64CAC60, $04A8D0BA,
    $030DD572, $B83D320F, $CAB23045, $E366F2F0, $815D008D, $C897A43A, $1D352DF3,
    $B9CC571D, $8BF38744, $72209092, $EBA124EB, $FB99CE5E, $3BB94293, $28DA549C,
    $AAB8A228, $A4197785, $33C70296, $25F6259B, $5C85DA21, $DF15BDEE, $15B7C7E8,
    $E2ABEF75, $FCC19BC1, $417FF868, $14884434, $62825179, $C6D5C11C, $0E4705DC,
    $22700DE0, $D3D2AF18, $9BE822A0, $35B669F1, $C42BB55C, $0A801252, $115BF0FC,
    $3CD7D856, $B43F5F9D, $C2306516, $A1231C47, $F149207E, $5209A795, $34B3CCD8,
    $67AEFE54, $2C83924E, $6662CBAC, $5EEDD161, $84E681AA, $5D57D26B, $FA465CC4,
    $7E3AC3A8, $BF7C0CC6, $E18A9AA1, $C32F0A6F, $B22CC00D, $3D280369, $994E554F,
    $68F480D3, $ADCFF5E6, $3A8EB265, $83269831, $BD568A09, $4BC8AE6A, $69F56D2B,
    $0F17EAC8, $772EB6C7, $9F41343C, $AB1D0742, $826A6F50, $FEA2097C, $1912C283,
    $CE185899, $E4444839, $2D8635D5, $65D0B1FF, $865A7F17, $326D9FB1, $59E52820,
    $0090ADE1, $753C7149, $9DDD8B98, $A5A691DA, $0D0382BB, $8904C930, $086CB000,
    $6E69D3BD, $24D4E7A7, $05244FD0, $101A5E0C, $6A947DCB, $E840F77B, $7D0C5003,
    $7C370F1F, $805245ED, $E05E3D3F, $7906880E, $BABFCD35, $1A7EC697, $8C052324,
    $0C6EC8DF, $D129A589, $C7A75B02, $12D81DE7, $D9BE2A66, $1F4263AB, $DE73FDB6,
    $2A00680A, $56649E36, $3133ED55, $90FA0BF2, $2910A02A, $949D9D46, $A0D1DCDD,
    $CFC9B7D4, $D2677BE5, $95CB36B3, $13CD9410, $DBF73313, $B7C6E8C0, $F781414B,
    $510B016D, $B0DE1157, $D6B0F62C, $BB074ECC, $7F1395B7, $EE792CF9, $EA6FD63E,
    $5BD6938E, $AF02FC64, $DAB57AB8, $8EDB3784, $8716318F, $164D1A01, $26F26141,
    $B372E6B9, $F8FC2B06, $7AC00E04, $3727B89A, $97E9BCA5, $9C2A742F, $BC3B1F7D,
    $7165B471, $609B4C29, $20925351, $5AE72112, $454BE5D1, $C0FFB95F, $DD0EF919,
    $6F2D70C9, $0974C5BF, $98AA6263, $01D91E4D, $2184BB6E, $70C43C1E, $4D435915,
    $AE7B8523, $B6FB06BC, $5431EE76, $FDBC5D26, $ED77493D, $C5712EE4, $A8380437,
    $2EEF261A), THashLibUInt32Array.Create($5A79392B, $B8AF32C2, $41F7720A,
    $833A61EC, $13DFEDAC, $C4990BC4, $DC0F54BC, $FEDD5E88, $80DA1881, $4DEA1AFD,
    $FD402CC6, $AE67CC7A, $C5238525, $8EA01254, $B56B9BD5, $862FBD6D, $AC8575D3,
    $6FBA3714, $DA7EBF46, $59CD5238, $8AC9DBFE, $353729FC, $E497D7F2, $C3AB84E0,
    $F05A114B, $7B887A75, $EDC603DD, $5E6FE680, $2C84B399, $884EB1DA, $1CB8C8BF,
    $AA51098A, $C862231C, $8BAC2221, $21B387E5, $208A430D, $2A3F0F8B, $A5FF9CD2,
    $6012A2EA, $147A9EE7, $F62A501D, $B4B2E51A, $3EF3484C, $C0253C59, $2B82B536,
    $0AA9696B, $BE0C109B, $C70B7929, $CE3E8A19, $2F66950E, $459F1C2C, $E68FB93D,
    $A3C3FF3E, $62B45C62, $300991CB, $01914C57, $7F7BC06A, $182831F5, $E7B74BCA,
    $FA50F6D0, $523CAA61, $E3A7CF05, $E9E41311, $280A21D1, $6A4297E1, $F24DC67E,
    $FC3189E6, $B72BF34F, $4B1E67AF, $543402CE, $79A59867, $0648E02A, $00A3AC17,
    $C6208D35, $6E7F5F76, $A45BB4BE, $F168FA63, $3F4125F3, $F311406F, $02706565,
    $BFE58022, $0CFCFDD9, $0735A7F7, $8F049092, $D98EDC27, $F5C5D55C, $E0F201DB,
    $0DCAFC9A, $7727FB79, $AF43ABF4, $26E938C1, $401B26A6, $900720FA, $2752D97B,
    $CFF1D1B3, $A9D9E424, $42DB99AB, $6CF8BE5F, $E82CEBE3, $3AFB733B, $6B734EB6,
    $1036414A, $975F667C, $049D6377, $BA587C60, $B1D10483, $DE1AEFCC, $1129D055,
    $72051E91, $6946D623, $F9E86EA7, $48768C00, $B0166C93, $9956BBF0, $1F1F6D84,
    $FB15E18E, $033B495D, $56E3362E, $4F44C53C, $747CBA51, $89D37872, $5D9C331B,
    $D2EF9FA8, $254917F8, $1B106F47, $37D75553, $B3F053B0, $7DCCD8EF, $D30EB802,
    $5889F42D, $610206D7, $1A7D34A1, $92D87DD8, $E5F4A315, $D1CF0E71, $B22DFE45,
    $B901E8EB, $0FC0CE5E, $2EFA60C9, $2DE74290, $36D0C906, $381C70E4, $4C6DA5B5,
    $3D81A682, $7E381F34, $396C4F52, $95AD5901, $1DB50C5A, $29982E9E, $1557689F,
    $3471EE42, $D7E2F7C0, $8795A1E2, $BC324D8D, $E224C3C8, $12837E39, $CDEE3D74,
    $7AD2143F, $0E13D40C, $78BD4A68, $A2EB194D, $DB9451F9, $859B71DC, $5C4F5B89,
    $CA14A8A4, $EF92F003, $16741D98, $33AA4444, $9E967FBB, $092E3020, $D86A35B8,
    $8CC17B10, $E1BF08AE, $55693FC5, $7680AD13, $1E6546E8, $23B6E7B9, $EE77A4B2,
    $08ED0533, $44FD2895, $B6393B69, $05D6CACF, $9819B209, $ECBBB72F, $9A75779C,
    $EAEC0749, $94A65AEE, $BDF52DC3, $D6A25D04, $82008E4E, $A6DE160F, $9B036AFB,
    $228B3A66, $5FB10A70, $CC338B58, $5378A9DF, $C908BCA9, $4959E25B, $46909A97,
    $66AE8F6E, $DD0683E9, $65F994B4, $6426CDA5, $C24B8840, $32539DA0, $63175650,
    $D0C815FF, $50CBC41E, $F7C774A3, $31B0C231, $8D0D8116, $24BEF16C, $D555D256,
    $DF47EA8C, $6D21ECCD, $A887A012, $84542AED, $A7B9C1BD, $914C1BB1, $A0D5B67D,
    $438CE937, $7030F873, $71F6B0C7, $574576BA, $F8BC4541, $9C61D348, $1960579D,
    $17C4DAAD, $96A4CB0B, $C193F2F6, $756EAFA2, $7C1D2F94, $F4FE2B43, $CB86E33A,
    $EBD4C728, $9D18AE64, $9FE13E30, $3CE0F5DE, $ABA1F985, $ADDC2718, $68CE6278,
    $D45E241F, $A15C82B7, $3B2293D4, $739EDD32, $674A6BF1, $5B5D587F, $4772DEAA,
    $4A63968F, $0BE68686, $513D6426, $939A4787, $BBA89296, $4EC20007, $818D0D08,
    $FF64DFD6), THashLibUInt32Array.Create($CB2297CB, $DB48A144, $A16CBE4B,
    $BBEA1D6C, $5AF6B6B7, $8A8110B6, $F9236EF9, $C98F83E6, $0F9C65B8, $252D4A89,
    $A497F068, $A5D7ED2D, $94C22845, $9DA1C8C4, $E27C2E2E, $6E8BA2B4, $C3DD17FB,
    $498CD482, $0DFE6A9F, $B0705829, $9A1E6DC1, $F829717C, $07BB8E3A, $DA3C0B02,
    $1AF82FC7, $73B70955, $7A04379C, $5EE20A28, $83712AE5, $F4C47C6D, $DF72BA56,
    $D794858D, $8C0CF709, $18F0F390, $B6C69B35, $BF2F01DB, $2FA74DCA, $D0CD9127,
    $BDE66CEC, $3DEEBD46, $57C88FC3, $CEE1406F, $0066385A, $F3C3444F, $3A79D5D5,
    $75751EB9, $3E7F8185, $521C2605, $E1AAAB6E, $38EBB80F, $BEE7E904, $61CB9647,
    $EA54904E, $05AE00E4, $2D7AC65F, $087751A1, $DCD82915, $0921EE16, $DD86D33B,
    $D6BD491A, $40FBADF0, $4232CBD2, $33808D10, $39098C42, $193F3199, $0BC1E47A,
    $4A82B149, $02B65A8A, $104CDC8E, $24A8F52C, $685C6077, $C79F95C9, $1D11FE50,
    $C08DAFCD, $7B1A9A03, $1C1F11D8, $84250E7F, $979DB248, $EBDC0501, $B9553395,
    $E3C05EA8, $B1E51C4C, $13B0E681, $3B407766, $36DB3087, $EE17C9FC, $6C53ECF2,
    $ADCCC58F, $C427660B, $EFD5867D, $9B6D54A5, $6FF1AEFF, $8E787952, $9E2BFFE0,
    $8761D034, $E00BDBAD, $AE99A8D3, $CC03F6E2, $FD0ED807, $0E508AE3, $B74182AB,
    $4349245D, $D120A465, $B246A641, $AF3B7AB0, $2A6488BB, $4B3A0D1F, $E7C7E58C,
    $3FAFF2EB, $90445FFD, $CF38C393, $995D07E7, $F24F1B36, $356F6891, $6D6EBCBE,
    $8DA9E262, $50FD520E, $5BCA9E1E, $37472CF3, $69075057, $7EC5FDED, $0CAB892A,
    $FB2412BA, $1728DEBF, $A000A988, $D843CE79, $042E20DD, $4FE8F853, $56659C3C,
    $2739D119, $A78A6120, $80960375, $70420611, $85E09F78, $ABD17E96, $1B513EAF,
    $1E01EB63, $26AD2133, $A890C094, $7613CF60, $817E781B, $A39113D7, $E957FA58,
    $4131B99E, $28B1EFDA, $66ACFBA7, $FF68944A, $77A44FD1, $7F331522, $59FFB3FA,
    $A6DF935B, $FA12D9DF, $C6BF6F3F, $89520CF6, $659EDD6A, $544DA739, $8B052538,
    $7C30EA21, $C2345525, $15927FB2, $144A436B, $BA107B8B, $1219AC97, $06730432,
    $31831AB3, $C55A5C24, $AA0FCD3E, $E5606BE8, $5C88F19B, $4C0841EE, $1FE37267,
    $11F9C4F4, $9F1B9DAE, $864E76D0, $E637C731, $D97D23A6, $32F53D5C, $B8161980,
    $93FA0F84, $CAEF0870, $8874487E, $98F2CC73, $645FB5C6, $CD853659, $2062470D,
    $16EDE8E9, $6B06DAB5, $78B43900, $FC95B786, $5D8E7DE1, $465B5954, $FE7BA014,
    $F7D23F7B, $92BC8B18, $03593592, $55CEF4F7, $74B27317, $79DE1FC2, $C8A0BFBD,
    $229398CC, $62A602CE, $BCB94661, $5336D206, $D2A375FE, $6A6AB483, $4702A5A4,
    $A2E9D73D, $23A2E0F1, $9189140A, $581D18DC, $B39A922B, $82356212, $D5F432A9,
    $D356C2A3, $5F765B4D, $450AFCC8, $4415E137, $E8ECDFBC, $ED0DE3EA, $60D42B13,
    $F13DF971, $71FC5DA2, $C1455340, $F087742F, $F55E5751, $67B3C1F8, $AC6B8774,
    $7DCFAAAC, $95983BC0, $489BB0B1, $2C184223, $964B6726, $2BD3271C, $72266472,
    $DED64530, $0A2AA343, $D4F716A0, $B4DAD6D9, $2184345E, $512C990C, $29D92D08,
    $2EBE709A, $01144C69, $34584B9D, $E4634ED6, $ECC963CF, $3C6984AA, $4ED056EF,
    $9CA56976, $8F3E80D4, $B5BAE7C5, $30B5CAF5, $63F33A64, $A9E4BBDE, $F6B82298,
    $4D673C1D), THashLibUInt32Array.Create($4B4F1121, $BA183081, $C784F41F,
    $D17D0BAC, $083D2267, $37B1361E, $3581AD05, $FDA2F6BC, $1E892CDD, $B56D3C3A,
    $32140E46, $138D8AAB, $E14773D4, $5B0E71DF, $5D1FE055, $3FB991D3, $F1F46C71,
    $A325988C, $10F66E80, $B1006348, $726A9F60, $3B67F8BA, $4E114EF4, $05C52115,
    $4C5CA11C, $99E1EFD8, $471B83B3, $CBF7E524, $43AD82F5, $690CA93B, $FAA61BB2,
    $12A832B5, $B734F943, $BD22AEA7, $88FEC626, $5E80C3E7, $BE3EAF5E, $44617652,
    $A5724475, $BB3B9695, $7F3FEE8F, $964E7DEB, $518C052D, $2A0BBC2B, $C2175F5C,
    $9A7B3889, $A70D8D0C, $EACCDD29, $CCCD6658, $34BB25E6, $B8391090, $F651356F,
    $52987C9E, $0C16C1CD, $8E372D3C, $2FC6EBBD, $6E5DA3E3, $B0E27239, $5F685738,
    $45411786, $067F65F8, $61778B40, $81AB2E65, $14C8F0F9, $A6B7B4CE, $4036EAEC,
    $BF62B00A, $ECFD5E02, $045449A6, $B20AFD28, $2166D273, $0D13A863, $89508756,
    $D51A7530, $2D653F7A, $3CDBDBC3, $80C9DF4F, $3D5812D9, $53FBB1F3, $C0F185C0,
    $7A3C3D7E, $68646410, $857607A0, $1D12622E, $97F33466, $DB4C9917, $6469607C,
    $566E043D, $79EF1EDB, $2C05898D, $C9578E25, $CD380101, $46E04377, $7D1CC7A9,
    $6552B837, $20192608, $B97500C5, $ED296B44, $368648B4, $62995CD5, $82731400,
    $F9AEBD8B, $3844C0C7, $7C2DE794, $33A1A770, $8AE528C2, $5A2BE812, $1F8F4A07,
    $2B5ED7CA, $937EB564, $6FDA7E11, $E49B5D6C, $B4B3244E, $18AA53A4, $3A061334,
    $4D6067A3, $83BA5868, $9BDF4DFE, $7449F261, $709F8450, $CAD133CB, $DE941C3F,
    $F52AE484, $781D77ED, $7E4395F0, $AE103B59, $922331BB, $42CE50C8, $E6F08153,
    $E7D941D0, $5028ED6B, $B3D2C49B, $AD4D9C3E, $D201FB6E, $A45BD5BE, $FFCB7F4B,
    $579D7806, $F821BB5B, $59D592AD, $D0BE0C31, $D4E3B676, $0107165A, $0FE939D2,
    $49BCAAFD, $55FFCFE5, $2EC1F783, $F39A09A5, $3EB42772, $19B55A5D, $024A0679,
    $8C83B3F7, $8642BA1D, $ACACD9EA, $87D352C4, $60931F45, $A05F97D7, $1CECD42C,
    $E2FCC87B, $B60F94E2, $67A34B0B, $FCDD40C9, $0B150A27, $D3EE9E04, $582E29E9,
    $4AC22B41, $6AC4E1B8, $BCCAA51A, $237AF30E, $EBC3B709, $C4A59D19, $284BC98A,
    $E9D41A93, $6BFA2018, $73B2D651, $11F9A2FA, $CE09BFF1, $41A470AA, $25888F22,
    $77E754E8, $F7330D8E, $158EAB16, $C5D68842, $C685A6F6, $E5B82FDE, $09EA3A96,
    $6DDE1536, $4FA919DA, $26C0BE9F, $9EED6F69, $F05555F2, $E06FC285, $9CD76D23,
    $AF452A92, $EFC74CB7, $9D6B4732, $8BE408EE, $22401D0D, $EE6C459D, $7587CB82,
    $E8746862, $5CBDDE87, $98794278, $31AFB94D, $C11E0F2F, $30E8FC2A, $CF3261EF,
    $1A3023E1, $AA2F86CF, $F202E24A, $8D08DCFF, $764837C6, $A26374CC, $9F7C3E88,
    $949CC57D, $DD26A07F, $C39EFAB0, $C8F879A1, $DCE67BB9, $F4B0A435, $912C9AE0,
    $D85603E4, $953A9BBF, $FB8290D6, $0AEBCD5F, $16206A9A, $6C787A14, $D9A0F16A,
    $29BF4F74, $8F8BCE91, $0E5A9354, $AB038CB1, $1B8AD11B, $E327FF49, $0053DA20,
    $90CF51DC, $DA92FE6D, $0390CA47, $A8958097, $A9DC5BAF, $3931E3C1, $840446B6,
    $63D069FB, $D7460299, $7124ECD1, $0791E613, $485918FC, $D635D04C, $DF96AC33,
    $66F2D303, $247056AE, $A1A7B2A8, $27D8CC9C, $17B6E998, $7BF5590F, $FE97F557,
    $5471D8A2), THashLibUInt32Array.Create($83A327A1, $9F379F51, $40A7D007,
    $11307423, $224587C1, $AC27D63B, $3B7E64EA, $2E1CBFA6, $09996000, $03BC0E2C,
    $D4C4478A, $4542E0AB, $FEDA26D4, $C1D10FCB, $8252F596, $4494EB5C, $A362F314,
    $F5BA81FD, $75C3A376, $4CA214CA, $E164DEDD, $5088FA97, $4B0930E0, $2FCFB7E8,
    $33A6F4B2, $C7E94211, $2D66C774, $43BE8BAE, $C663D445, $908EB130, $F4E3BE15,
    $63B9D566, $529396B5, $1E1BE743, $4D5FF63F, $985E4A83, $71AB9DF7, $C516C6F5,
    $85C19AB4, $1F4DAEE4, $F2973431, $B713DC5E, $3F2E159A, $C824DA16, $06BF376A,
    $B2FE23EC, $E39B1C22, $F1EECB5F, $08E82D52, $565686C2, $AB0AEA93, $FD47219F,
    $EBDBABD7, $2404A185, $8C7312B9, $A8F2D828, $0C8902DA, $65B42B63, $C0BBEF62,
    $4E3E4CEF, $788F8018, $EE1EBAB7, $93928F9D, $683D2903, $D3B60689, $AFCB0DDC,
    $88A4C47A, $F6DD9C3D, $7EA5FCA0, $8A6D7244, $BE11F120, $04FF91B8, $8D2DC8C0,
    $27F97FDB, $7F9E1F47, $1734F0C7, $26F3ED8E, $0DF8F2BF, $B0833D9E, $E420A4E5,
    $A423CAE6, $95616772, $9AE6C049, $075941F2, $D8E12812, $000F6F4F, $3C0D6B05,
    $6CEF921C, $B82BC264, $396CB008, $5D608A6F, $6D7782C8, $186550AA, $6B6FEC09,
    $28E70B13, $57CE5688, $ECD3AF84, $23335A95, $91F40CD2, $7B6A3B26, $BD32B3B6,
    $3754A6FB, $8ED088F0, $F867E87C, $20851746, $6410F9C6, $35380442, $C2CA10A7,
    $1ADEA27F, $76BDDD79, $92742CF4, $0E98F7EE, $164E931D, $B9C835B3, $69060A99,
    $B44C531E, $FA7B66FE, $C98A5B53, $7D95AAE9, $302F467B, $74B811DE, $F3866ABD,
    $B5B3D32D, $FC3157A4, $D251FE19, $0B5D8EAC, $DA71FFD5, $47EA05A3, $05C6A9E1,
    $CA0EE958, $9939034D, $25DC5EDF, $79083CB1, $86768450, $CF757D6D, $5972B6BC,
    $A78D59C9, $C4AD8D41, $2A362AD3, $D1179991, $601407FF, $DCF50917, $587069D0,
    $E0821ED6, $DBB59427, $73911A4B, $7C904FC3, $844AFB92, $6F8C955D, $E8C0C5BB,
    $B67AB987, $A529D96C, $F91F7181, $618B1B06, $E718BB0C, $8BD7615B, $D5A93A59,
    $54AEF81B, $772136E3, $CE44FD9C, $10CDA57E, $87D66E0B, $3D798967, $1B2C1804,
    $3EDFBD68, $15F6E62B, $EF68B854, $3896DB35, $12B7B5E2, $CB489029, $9E4F98A5,
    $62EB77A8, $217C24A2, $964152F6, $49B2080A, $53D23EE7, $48FB6D69, $1903D190,
    $9449E494, $BF6E7886, $FB356CFA, $3A261365, $424BC1EB, $A1192570, $019CA782,
    $9D3F7E0E, $9C127575, $EDF02039, $AD57BCCE, $5C153277, $81A84540, $BCAA7356,
    $CCD59B60, $A62A629B, $A25CCD10, $2B5B65CF, $1C535832, $55FD4E3A, $31D9790D,
    $F06BC37D, $4AFC1D71, $AEED5533, $BA461634, $BB694B78, $5F3A5C73, $6A3C764A,
    $8FB0CCA9, $F725684C, $4FE5382F, $1D0163AF, $5AA07A8F, $E205A8ED, $C30BAD38,
    $FF22CF1F, $72432E2E, $32C2518B, $3487CE4E, $7AE0AC02, $709FA098, $0A3B395A,
    $5B4043F8, $A9E48C36, $149A8521, $D07DEE6B, $46ACD2F3, $8958DFFC, $B3A1223C,
    $B11D31C4, $CD7F4D3E, $0F28E3AD, $E5B100BE, $AAC54824, $E9C9D7BA, $9BD47001,
    $80F149B0, $66022F0F, $020C4048, $6EFA192A, $67073F8D, $13EC7BF9, $3655011A,
    $E6AFE157, $D9845F6E, $DECC4425, $511AE2CC, $DF81B4D8, $D7809E55, $D6D883D9,
    $2CC7978C, $5E787CC5, $DD0033D1, $A050C937, $97F75DCD, $299DE580, $41E2B261,
    $EA5A54F1), THashLibUInt32Array.Create($7E672590, $BEA513BB, $2C906FE6,
    $86029C2B, $55DC4F74, $0553398E, $63E09647, $CAFD0BAB, $264C37DF, $8272210F,
    $67AFA669, $12D98A5F, $8CAB23C4, $75C68BD1, $C3370470, $33F37F4E, $283992FF,
    $E73A3A67, $1032F283, $F5AD9FC2, $963F0C5D, $664FBC45, $202BA41C, $C7C02D80,
    $54731E84, $8A1085F5, $601D80FB, $2F968E55, $35E96812, $E45A8F78, $BD7DE662,
    $3B6E6EAD, $8097C5EF, $070B6781, $B1E508F3, $24E4FAE3, $B81A7805, $EC0FC918,
    $43C8774B, $9B2512A9, $2B05AD04, $32C2536F, $EDF236E0, $8BC4B0CF, $BACEB837,
    $4535B289, $0D0E94C3, $A5A371D0, $AD695A58, $39E3437D, $9186BFFC, $21038C3B,
    $0AA9DFF9, $5D1F06CE, $62DEF8A4, $F740A2B4, $A2575868, $682683C1, $DBB30FAC,
    $61FE1928, $468A6511, $C61CD5F4, $E54D9800, $6B98D7F7, $8418B6A5, $5F09A5D2,
    $90B4E80B, $49B2C852, $69F11C77, $17412B7E, $7F6FC0ED, $56838DCC, $6E9546A2,
    $D0758619, $087B9B9A, $D231A01D, $AF46D415, $097060FD, $D920F657, $882D3F9F,
    $3AE7C3C9, $E8A00D9B, $4FE67EBE, $2EF80EB2, $C1916B0C, $F4DFFEA0, $B97EB3EB,
    $FDFF84DD, $FF8B14F1, $E96B0572, $F64B508C, $AE220A6E, $4423AE5A, $C2BECE5E,
    $DE27567C, $FC935C63, $47075573, $E65B27F0, $E121FD22, $F2668753, $2DEBF5D7,
    $8347E08D, $AC5EDA03, $2A7CEBE9, $3FE8D92E, $23542FE4, $1FA7BD50, $CF9B4102,
    $9D0DBA39, $9CB8902A, $A7249D8B, $0F6D667A, $5EBFA9EC, $6A594DF2, $79600938,
    $023B7591, $EA2C79C8, $C99D07EA, $64CB5EE1, $1A9CAB3D, $76DB9527, $C08E012F,
    $3DFB481A, $872F22E7, $2948D15C, $A4782C79, $6F50D232, $78F0728A, $5A87AAB1,
    $C4E2C19C, $EE767387, $1B2A1864, $7B8D10D3, $D1713161, $0EEAC456, $D8799E06,
    $B645B548, $4043CB65, $A874FB29, $4B12D030, $7D687413, $18EF9A1F, $D7631D4C,
    $5829C7DA, $CDFA30FA, $C5084BB0, $92CD20E2, $D4C16940, $03283EC0, $A917813F,
    $9A587D01, $70041F8F, $DC6AB1DC, $DDAEE3D5, $31829742, $198C022D, $1C9EAFCB,
    $5BBC6C49, $D3D3293A, $16D50007, $04BB8820, $3C5C2A41, $37EE7AF8, $8EB04025,
    $9313ECBA, $BFFC4799, $8955A744, $EF85D633, $504499A7, $A6CA6A86, $BB3D3297,
    $B34A8236, $6DCCBE4F, $06143394, $CE19FC7B, $CCC3C6C6, $E36254AE, $77B7EDA1,
    $A133DD9E, $EBF9356A, $513CCF88, $E2A1B417, $972EE5BD, $853824CD, $5752F4EE,
    $6C1142E8, $3EA4F309, $B2B5934A, $DFD628AA, $59ACEA3E, $A01EB92C, $389964BC,
    $DA305DD4, $019A59B7, $11D2CA93, $FAA6D3B9, $4E772ECA, $72651776, $FB4E5B0E,
    $A38F91A8, $1D0663B5, $30F4F192, $B50051B6, $B716CCB3, $4ABD1B59, $146C5F26,
    $F134E2DE, $00F67C6C, $B0E1B795, $98AA4EC7, $0CC73B34, $654276A3, $8D1BA871,
    $740A5216, $E0D01A23, $9ED161D6, $9F36A324, $993EBB7F, $FEB9491B, $365DDCDB,
    $810CFFC5, $71EC0382, $2249E7BF, $48817046, $F3A24A5B, $4288E4D9, $0BF5C243,
    $257FE151, $95B64C0D, $4164F066, $AAF7DB08, $73B1119D, $8F9F7BB8, $D6844596,
    $F07A34A6, $53943D0A, $F9DD166D, $7A8957AF, $F8BA3CE5, $27C9621E, $5CDAE910,
    $C8518998, $941538FE, $136115D8, $ABA8443C, $4D01F931, $34EDF760, $B45F266B,
    $D5D4DE14, $52D8AC35, $15CFD885, $CBC5CD21, $4CD76D4D, $7C80EF54, $BC92EE75,
    $1E56A1F6), THashLibUInt32Array.Create($BAA20B6C, $9FFBAD26, $E1F7D738,
    $794AEC8D, $C9E9CF3C, $8A9A7846, $C57C4685, $B9A92FED, $29CB141F, $52F9DDB7,
    $F68BA6BC, $19CCC020, $4F584AAA, $3BF6A596, $003B7CF7, $54F0CE9A, $A7EC4303,
    $46CF0077, $78D33AA1, $215247D9, $74BCDF91, $08381D30, $DAC43E40, $64872531,
    $0BEFFE5F, $B317F457, $AEBB12DA, $D5D0D67B, $7D75C6B4, $42A6D241, $1502D0A9,
    $3FD97FFF, $C6C3ED28, $81868D0A, $92628BC5, $86679544, $FD1867AF, $5CA3EA61,
    $568D5578, $4A2D71F4, $43C9D549, $8D95DE2B, $6E5C74A0, $9120FFC7, $0D05D14A,
    $A93049D3, $BFA80E17, $F4096810, $043F5EF5, $A673B4F1, $6D780298, $A4847783,
    $5EE726FB, $9934C281, $220A588C, $384E240F, $933D5C69, $39E5EF47, $26E8B8F3,
    $4C1C6212, $8040F75D, $074B7093, $6625A8D7, $36298945, $76285088, $651D37C3,
    $24F5274D, $DBCA3DAB, $186B7EE1, $D80F8182, $14210C89, $943A3075, $4E6E11C4,
    $4D7E6BAD, $F05064C8, $025DCD97, $4BC10302, $7CEDE572, $8F90A970, $AB88EEBA,
    $B5998029, $5124D839, $B0EEB6A3, $89DDABDC, $E8074D76, $A1465223, $32518CF2,
    $9D39D4EB, $C0D84524, $E35E6EA8, $7ABF3804, $113E2348, $9AE6069D, $B4DFDABB,
    $A8C5313F, $23EA3F79, $530E36A2, $A5FD228B, $95D1D350, $2B14CC09, $40042956,
    $879D05CC, $2064B9CA, $ACACA40E, $B29C846E, $9676C9E3, $752B7B8A, $7BE2BCC2,
    $6BD58F5E, $D48F4C32, $606835E4, $9CD7C364, $2C269B7A, $3A0D079C, $73B683FE,
    $45374F1E, $10AFA242, $577F8666, $DDAA10F6, $F34F561C, $3D355D6B, $E47048AE,
    $AA13C492, $050344FD, $2AAB5151, $F5B26AE5, $ED919A59, $5AC67900, $F1CDE380,
    $0C79A11B, $351533FC, $CD4D8E36, $1F856005, $690B9FDD, $E736DCCF, $1D47BF6A,
    $7F66C72A, $85F21B7F, $983CBDB6, $01EBBEBF, $035F3B99, $EB111F34, $28CEFDC6,
    $5BFC9ECD, $F22EACB0, $9E41CBB2, $E0F8327C, $82E3E26F, $FC43FC86, $D0BA66DF,
    $489EF2A7, $D9E0C81D, $68690D52, $CC451367, $C2232E16, $E95A7335, $0FDAE19B,
    $FF5B962C, $97596527, $C46DB333, $3ED4C562, $C14C9D9E, $5D6FAA21, $638E940D,
    $F9316D58, $47B3B0EA, $30FFCAD2, $CE1BBA7D, $1E6108E6, $2E1EA33D, $507BF05B,
    $FAFEF94B, $D17DE8E2, $5598B214, $1663F813, $17D25A2D, $EEFA5FF9, $582F4E37,
    $12128773, $FEF17AB8, $06005322, $BB32BBC9, $8C898508, $592C15F0, $D38A4054,
    $4957B7D6, $D2B891DB, $37BD2D3E, $34AD20CB, $622288E9, $2DC7345A, $AFB416C0,
    $1CF459B1, $DC7739FA, $0A711A25, $13E18A0C, $5F72AF4C, $6AC8DB11, $BE53C18E,
    $1AA569B9, $EF551EA4, $A02A429F, $BD16E790, $7EB9171A, $77D693D8, $8E06993A,
    $9BDE7560, $E5801987, $C37A09BE, $B8DB76AC, $E2087294, $6C81616D, $B7F30FE7,
    $BC9B82BD, $FBA4E4D4, $C7B1012F, $A20C043B, $DE9FEBD0, $2F9297CE, $E610AEF8,
    $70B06F19, $C86AE00B, $0E01988F, $41192AE0, $448C1CB5, $ADBE92EE, $7293A007,
    $1B54B5B3, $D61F63D1, $EAE40A74, $61A72B55, $EC83A7D5, $88942806, $90A07DA5,
    $D7424B95, $67745B4E, $A31A1853, $CA6021EF, $DFB56C4F, $CBC2D915, $3C48E918,
    $8BAE3C63, $6F659C71, $F8B754C1, $2782F3DE, $F796F168, $71492C84, $33C0F5A6,
    $3144F6EC, $25DC412E, $B16C5743, $83A1FA7E, $0997B101, $B627E6E8, $CF33905C,
    $8456FB65), THashLibUInt32Array.Create($B29BEA74, $C35DA605, $305C1CA3,
    $D2E9F5BC, $6FD5BFF4, $FF347703, $FC45B163, $F498E068, $B71229FC, $81ACC3FB,
    $78538A8B, $984ECF81, $A5DA47A4, $8F259EEF, $6475DC65, $081865B9, $49E14A3C,
    $19E66079, $D382E91B, $5B109794, $3F9F81E1, $4470A388, $41601ABE, $AAF9F407,
    $8E175EF6, $ED842297, $893A4271, $1790839A, $D566A99E, $6B417DEE, $75C90D23,
    $715EDB31, $723553F7, $9AFB50C9, $FBC5F600, $CD3B6A4E, $97ED0FBA, $29689AEC,
    $63135C8E, $F0E26C7E, $0692AE7F, $DBB208FF, $2EDE3E9B, $6A65BEBD, $D40867E9,
    $C954AFC5, $73B08201, $7FFDF809, $1195C24F, $1CA5ADCA, $74BD6D1F, $B393C455,
    $CADFD3FA, $99F13011, $0EBCA813, $60E791B8, $6597AC7A, $18A7E46B, $09CB49D3,
    $0B27DF6D, $CFE52F87, $CEF66837, $E6328035, $FA87C592, $37BAFF93, $D71FCC99,
    $DCAB205C, $4D7A5638, $48012510, $62797558, $B6CF1FE5, $BC311834, $9C2373AC,
    $14EC6175, $A439CBDF, $54AFB0EA, $D686960B, $FDD0D47B, $7B063902, $8B78BAC3,
    $26C6A4D5, $5C0055B6, $2376102E, $0411783E, $2AA3F1CD, $51FC6EA8, $701CE243,
    $9B2A0ABB, $0AD93733, $6E80D03D, $AF6295D1, $F629896F, $A30B0648, $463D8DD4,
    $963F84CB, $01FF94F8, $8D7FEFDC, $553611C0, $A97C1719, $B96AF759, $E0E3C95E,
    $0528335B, $21FE5925, $821A5245, $807238B1, $67F23DB5, $EA6B4EAB, $0DA6F985,
    $AB1BC85A, $EF8C90E4, $4526230E, $38EB8B1C, $1B91CD91, $9FCE5F0C, $F72CC72B,
    $C64F2617, $DAF7857D, $7D373CF1, $28EAEDD7, $203887D0, $C49A155F, $A251B3B0,
    $F2D47AE3, $3D9EF267, $4A94AB2F, $7755A222, $0205E329, $C28FA7A7, $AEC1FE51,
    $270F164C, $8C6D01BF, $53B5BC98, $C09D3FEB, $834986CC, $4309A12C, $578B2A96,
    $3BB74B86, $69561B4A, $037E32F3, $DE335B08, $C5156BE0, $E7EF09AD, $93B834C7,
    $A7719352, $59302821, $E3529D26, $F961DA76, $CB142C44, $A0F3B98D, $76502457,
    $945A414B, $078EEB12, $DFF8DE69, $EB6C8C2D, $BDA90C4D, $E9C44D16, $168DFD66,
    $AD64763B, $A65FD764, $95A29C06, $32D7713F, $40F0B277, $224AF08F, $004CB5E8,
    $92574814, $8877D827, $3E5B2D04, $68C2D5F2, $86966273, $1D433ADA, $8774988A,
    $3C0E0BFE, $DDAD581D, $2FD654ED, $0F4769FD, $C181EE9D, $5FD88F61, $341DBB3A,
    $528543F9, $D92235CF, $1EA82EB4, $B5CD790F, $91D24F1E, $A869E6C2, $61F474D2,
    $CC205ADD, $0C7BFBA9, $BF2B0489, $B02D72D8, $2B46ECE6, $E4DCD90A, $B8A11440,
    $EE8A63B7, $854DD1A1, $D1E00583, $42B40E24, $9E8964DE, $B4B35D78, $BEC76F6E,
    $24B9C620, $D8D399A6, $5ADB2190, $2DB12730, $3A5866AF, $58C8FADB, $5D8844E7,
    $8A4BF380, $15A01D70, $79F5C028, $66BE3B8C, $F3E42B53, $56990039, $2C0C3182,
    $5E16407C, $ECC04515, $6C440284, $4CB6701A, $13BFC142, $9D039F6A, $4F6E92C8,
    $A1407C62, $8483A095, $C70AE1C4, $E20213A2, $BACAFC41, $4ECC12B3, $4BEE3646,
    $1FE807AE, $25217F9C, $35DDE5F5, $7A7DD6CE, $F89CCE50, $AC07B718, $7E73D2C6,
    $E563E76C, $123CA536, $3948CA56, $9019DD49, $10AA88D9, $C82451E2, $473EB6D6,
    $506FE854, $E8BB03A5, $332F4C32, $FE1E1E72, $B1AE572A, $7C0D7BC1, $E1C37EB2,
    $F542AA60, $F1A48EA0, $D067B89F, $BBFA195D, $1A049B0D, $315946AA, $36D1B447,
    $6D2EBDF0), THashLibUInt32Array.Create($0D188A6D, $12CEA0DB, $7E63740E,
    $6A444821, $253D234F, $6FFC6597, $94A6BDEF, $33EE1B2F, $0A6C00C0, $3AA336B1,
    $5AF55D17, $265FB3DC, $0E89CF4D, $0786B008, $C80055B8, $6B17C3CE, $72B05A74,
    $D21A8D78, $A6B70840, $FE8EAE77, $ED69565C, $55E1BCF4, $585C2F60, $E06F1A62,
    $AD67C0CD, $7712AF88, $9CC26ACA, $1888053D, $37EB853E, $9215ABD7, $DE30ADFC,
    $1F1038E6, $70C51C8A, $8D586C26, $F72BDD90, $4DC3CE15, $68EAEEFA, $D0E9C8B9,
    $200F9C44, $DDD141BA, $024BF1D3, $0F64C9D4, $C421E9E9, $9D11C14C, $9A0DD9E4,
    $5F92EC19, $1B980DF0, $1DCC4542, $B8FE8C56, $0C9C9167, $4E81EB49, $CA368F27,
    $E3603B37, $EA08ACCC, $AC516992, $C34F513B, $804D100D, $6EDCA4C4, $FC912939,
    $29D219B0, $278AAA3C, $4868DA7D, $54E890B7, $B46D735A, $514589AA, $D6C630AF,
    $4980DFE8, $BE3CCC55, $59D41202, $650C078B, $AF3A9E7B, $3ED9827A, $9E79FC6E,
    $AADBFBAE, $C5F7D803, $3DAF7F50, $67B4F465, $73406E11, $39313F8C, $8A6E6686,
    $D8075F1F, $D3CBFED1, $69C7E49C, $930581E0, $E4B1A5A8, $BBC45472, $09DDBF58,
    $C91D687E, $BDBFFDA5, $88C08735, $E9E36BF9, $DB5EA9B6, $95559404, $08F432FB,
    $E24EA281, $64663579, $000B8010, $7914E7D5, $32FD0473, $D1A7F0A4, $445AB98E,
    $EC72993F, $A29A4D32, $B77306D8, $C7C97CF6, $7B6AB645, $F5EF7ADF, $FB2E15F7,
    $E747F757, $5E944354, $234A2669, $47E46359, $9B9D11A9, $40762CED, $56F1DE98,
    $11334668, $890A9A70, $1A296113, $B3BD4AF5, $163B7548, $D51B4F84, $B99B2ABC,
    $3CC1DC30, $A9F0B56C, $812272B2, $0B233A5F, $B650DBF2, $F1A0771B, $36562B76,
    $DC037B0F, $104C97FF, $C2EC98D2, $90596F22, $28B6620B, $DF42B212, $FDBC4243,
    $F3FB175E, $4A2D8B00, $E8F3869B, $30D69BC3, $853714C8, $A7751D2E, $31E56DEA,
    $D4840B0C, $9685D783, $068C9333, $8FBA032C, $76D7BB47, $6D0EE22B, $B546794B,
    $D971B894, $8B09D253, $A0AD5761, $EE77BA06, $46359F31, $577CC7EC, $52825EFD,
    $A4BEED95, $9825C52A, $EB48029A, $BAAE59F8, $CF490EE1, $BC990164, $8CA49DFE,
    $4F38A6E7, $2BA98389, $8228F538, $199F64AC, $01A1CAC5, $A8B51641, $5CE72D01,
    $8E5DF26B, $60F28E1E, $CD5BE125, $E5B376BF, $1C8D3116, $7132CBB3, $CB7AE320,
    $C0FA5366, $D7653E34, $971C88C2, $C62C7DD0, $34D0A3DA, $868F6709, $7AE6FA8F,
    $22BBD523, $66CD3D5B, $1EF9288D, $F9CF58C1, $5B784E80, $7439A191, $AE134C36,
    $9116C463, $2E9E1396, $F8611F3A, $2D2F3307, $247F37DD, $C1E2FF9D, $43C821E5,
    $05ED5CAB, $EF74E80A, $4CCA6028, $F0AC3CBD, $5D874B29, $6C62F6A6, $4B2A2EF3,
    $B1AA2087, $62A5D0A3, $0327221C, $B096B4C6, $417EC693, $ABA840D6, $789725EB,
    $F4B9E02D, $E6E00975, $CC04961A, $63F624BB, $7FA21ECB, $2C01EA7F, $B2415005,
    $2A8BBEB5, $83B2B14E, $A383D1A7, $5352F96A, $043ECDAD, $CE1918A1, $FA6BE6C9,
    $50DEF36F, $F6B80CE2, $4543EF7C, $9953D651, $F257955D, $87244914, $DA1E0A24,
    $FFDA4785, $14D327A2, $3B93C29F, $840684B4, $61AB71A0, $9F7B784A, $2FD570CF,
    $15955BDE, $38F8D471, $3534A718, $133FB71D, $3FD80F52, $4290A8BE, $75FF44C7,
    $A554E546, $E1023499, $BF2652E3, $7D20399E, $A1DF7E82, $177092EE, $217DD3F1,
    $7C1FF8D9), THashLibUInt32Array.Create($12113F2E, $BFBD0785, $F11793FB,
    $A5BFF566, $83C7B0E5, $72FB316B, $75526A9A, $41E0E612, $7156BA09, $53CE7DEE,
    $0AA26881, $A43E0D7D, $3DA73CA3, $182761ED, $BD5077FF, $56DB4AA0, $E792711C,
    $F0A4EB1D, $7F878237, $EC65C4E8, $08DC8D43, $0F8CE142, $8258ABDA, $F4154E16,
    $49DEC2FD, $CD8D5705, $6C2C3A0F, $5C12BB88, $EFF3CDB6, $2C89ED8C, $7BEBA967,
    $2A142157, $C6D0836F, $B4F97E96, $6931E969, $514E6C7C, $A7792600, $0BBBF780,
    $59671BBD, $0707B676, $37482D93, $80AF1479, $3805A60D, $E1F4CAC1, $580B3074,
    $30B8D6CE, $05A304BE, $D176626D, $EBCA97F3, $BB201F11, $6A1AFE23, $FFAA86E4,
    $62B4DA49, $1B6629F5, $F5D9E092, $F37F3DD1, $619BD45B, $A6EC8E4F, $29C80939,
    $0C7C0C34, $9CFE6E48, $E65FD3AC, $73613B65, $B3C669F9, $BE2E8A9E, $286F9678,
    $5797FD13, $99805D75, $CFB641C5, $A91074BA, $6343AF47, $6403CB46, $8894C8DB,
    $2663034C, $3C40DC5E, $00995231, $96789AA2, $2EFDE4B9, $7DC195E1, $547DADD5,
    $06A8EA04, $F2347A63, $5E0DC6F7, $8462DFC2, $1E6B2C3C, $9BD275B3, $91D419E2,
    $BCEFD17E, $B9003924, $D07E7320, $DEF0495C, $C36AD00E, $1785B1AB, $92E20BCF,
    $B139F0E9, $675BB9A1, $AECFA4AF, $132376CB, $E84589D3, $79A05456, $A2F860BC,
    $1AE4F8B5, $20DF4DB4, $A1E1428B, $3BF60A1A, $27FF7BF1, $CB44C0E7, $F7F587C4,
    $1F3B9B21, $94368F01, $856E23A4, $6F93DE3F, $773F5BBF, $8B22056E, $DF41F654,
    $B8246FF4, $8D57BFF2, $D57167EA, $C5699F22, $40734BA7, $5D5C2772, $033020A8,
    $E30A7C4D, $ADC40FD6, $76353441, $5AA5229B, $81516590, $DA49F14E, $4FA672A5,
    $4D9FAC5F, $154BE230, $8A7A5CC0, $CE3D2F84, $CCA15514, $5221360C, $AF0FB81E,
    $5BDD5873, $F6825F8F, $1113D228, $70AD996C, $93320051, $60471C53, $E9BA567B,
    $3A462AE3, $5F55E72D, $1D3C5AD7, $DCFC45EC, $34D812EF, $FA96EE1B, $369D1EF8,
    $C9B1A189, $7C1D3555, $50845EDC, $4BB31877, $8764A060, $8C9A9415, $230E1A3A,
    $B05E9133, $242B9E03, $A3B99DB7, $C2D7FB0A, $3333849D, $D27278D4, $B5D3EFA6,
    $78AC28AD, $C7B2C135, $0926ECF0, $C1374C91, $74F16D98, $2274084A, $3F6D9CFA,
    $7AC0A383, $B73AFF1F, $3909A23D, $9F1653AE, $4E2F3E71, $CA5AB22A, $E01E3858,
    $90C5A7EB, $3E4A17DF, $AA987FB0, $488BBD62, $B625062B, $2D776BB8, $43B5FC08,
    $1490D532, $D6D12495, $44E89845, $2FE60118, $9D9EF950, $AC38133E, $D3864329,
    $017B255A, $FDC2DD26, $256851E6, $318E7086, $2BFA4861, $89EAC706, $EE5940C6,
    $68C3BC2F, $E260334B, $98DA90BB, $F818F270, $4706D897, $212D3799, $4CF7E5D0,
    $D9C9649F, $A85DB5CD, $35E90E82, $6B881152, $AB1C02C7, $46752B02, $664F598E,
    $45AB2E64, $C4CDB4B2, $BA42107F, $EA2A808A, $971BF3DE, $4A54A836, $4253AECC,
    $1029BE68, $6DCC9225, $E4BCA56A, $C0AE50B1, $7E011D94, $E59C162C, $D8E5C340,
    $D470FA0B, $B2BE79DD, $D783889C, $1CEDE8F6, $8F4C817A, $DDB785C9, $860232D8,
    $198AAAD9, $A0814738, $3219CFFC, $169546D2, $FC0CB759, $55911510, $04D5CEC3,
    $ED08CC3B, $0D6CF427, $C8E38CCA, $0EEEE3FE, $9EE7D7C8, $F9F24FA9, $DB04B35D,
    $9AB0C9E0, $651F4417, $028F8B07, $6E28D9AA, $FBA96319, $8ED66687, $FECBC58D,
    $954DDB44), THashLibUInt32Array.Create($7B0BDFFE, $865D16B1, $49A058C0,
    $97ABAA3F, $CAACC75D, $ABA6C17D, $F8746F92, $6F48AEED, $8841D4B5, $F36A146A,
    $73C390AB, $E6FB558F, $87B1019E, $26970252, $246377B2, $CBF676AE, $F923DB06,
    $F7389116, $14C81A90, $83114EB4, $8B137559, $95A86A7A, $D5B8DA8C, $C4DF780E,
    $5A9CB3E2, $E44D4062, $E8DC8EF6, $9D180845, $817AD18B, $C286C85B, $251F20DE,
    $EE6D5933, $F6EDEF81, $D4D16C1E, $C94A0C32, $8437FD22, $3271EE43, $42572AEE,
    $5F91962A, $1C522D98, $59B23F0C, $D86B8804, $08C63531, $2C0D7A40, $B97C4729,
    $04964DF9, $13C74A17, $5878362F, $4C808CD6, $092CB1E0, $6DF02885, $A0C2105E,
    $8ABA9E68, $64E03057, $E5D61325, $0E43A628, $16DBD62B, $2733D90B, $3AE57283,
    $C0C1052C, $4B6FB620, $37513953, $FC898BB3, $471B179F, $DF6E66B8, $D32142F5,
    $9B30FAFC, $4ED92549, $105C6D99, $4ACD69FF, $2B1A27D3, $6BFCC067, $6301A278,
    $AD36E6F2, $EF3FF64E, $56B3CADB, $0184BB61, $17BEB9FD, $FAEC6109, $A2E1FFA1,
    $2FD224F8, $238F5BE6, $8F8570CF, $AEB5F25A, $4F1D3E64, $4377EB24, $1FA45346,
    $B2056386, $52095E76, $BB7B5ADC, $3514E472, $DDE81E6E, $7ACEA9C4, $AC15CC48,
    $71C97D93, $767F941C, $911052A2, $FFEA09BF, $FE3DDCF0, $15EBF3AA, $9235B8BC,
    $75408615, $9A723437, $E1A1BD38, $33541B7E, $1BDD6856, $B307E13E, $90814BB0,
    $51D7217B, $0BB92219, $689F4500, $C568B01F, $5DF3D2D7, $3C0ECD0D, $2A0244C8,
    $852574E8, $E72F23A9, $8E26ED02, $2D92CBDD, $DABC0458, $CDF5FEB6, $9E4E8DCC,
    $F4F1E344, $0D8C436D, $4427603B, $BDD37FDA, $80505F26, $8C7D2B8E, $B73273C5,
    $397362EA, $618A3811, $608BFB88, $06F7D714, $212E4677, $28EFCEAD, $076C0371,
    $36A3A4D9, $5487B455, $3429A365, $65D467AC, $78EE7EEB, $99BF12B7, $4D129896,
    $772A5601, $CCE284C7, $2ED85C21, $D099E8A4, $A179158A, $6AC0AB1A, $299A4807,
    $BE67A58D, $DC19544A, $B8949B54, $8D315779, $B6F849C1, $53C5AC34, $66DE92A5,
    $F195DD13, $318D3A73, $301EC542, $0CC40DA6, $F253ADE4, $467EE566, $EA5585EC,
    $3BAF19BB, $7DE9F480, $79006E7C, $A9B7A197, $A44BD8F1, $FB2BA739, $EC342FD4,
    $ED4FD32D, $3D1789BA, $400F5D7F, $C798F594, $4506A847, $034C0A95, $E2162C9D,
    $55A9CFD0, $692D832E, $CF9DB2CA, $5E2287E9, $D2610EF3, $1AE7ECC2, $48399CA0,
    $A7E4269B, $6EE3A0AF, $7065BFE1, $A6FFE708, $2256804C, $7476E21B, $41B0796C,
    $7C243B05, $000A950F, $1858416B, $F5A53C89, $E9FEF823, $3F443275, $E0CBF091,
    $0AF27B84, $3EBB0F27, $1DE6F7F4, $C31C29F7, $B166DE3D, $12932EC3, $9C0C0674,
    $5CDA81B9, $D1BD9D12, $AFFD7C82, $8962BCA7, $A342C4A8, $62457151, $82089F03,
    $EB49C670, $5B5F6530, $7E28BAD2, $20880BA3, $F0FAAFCD, $CE82B56F, $0275335C,
    $C18E8AFB, $DE601D69, $BA9B820A, $C8A2BE4F, $D7CAC335, $D9A73741, $115E974D,
    $7F5AC21D, $383BF9C6, $BCAEB75F, $FD0350CE, $B5D06B87, $9820E03C, $72D5F163,
    $E3644FC9, $A5464C4B, $57048FCB, $9690C9DF, $DBF9EAFA, $BFF4649A, $053C00E3,
    $B4B61136, $67593DD1, $503EE960, $9FB4993A, $19831810, $C670D518, $B05B51D8,
    $0F3A1CE5, $6CAA1F9C, $AACC31BE, $949ED050, $1EAD07E7, $A8479ABD, $D6CFFCD5,
    $936993EF), THashLibUInt32Array.Create($472E91CB, $5444B5B6, $62BE5861,
    $1BE102C7, $63E4B31E, $E81F71B7, $9E2317C9, $39A408AE, $518024F4, $1731C66F,
    $68CBC918, $71FB0C9E, $D03B7FDD, $7D6222EB, $9057EDA3, $1A34A407, $8CC2253D,
    $B6F6979D, $835675DC, $F319BE9F, $BE1CD743, $4D32FEE4, $77E7D887, $37E9EBFD,
    $15F851E8, $23DC3706, $19D78385, $BD506933, $A13AD4A6, $913F1A0E, $DDE560B9,
    $9A5F0996, $A65A0435, $48D34C4D, $E90839A7, $8ABBA54E, $6FD13CE1, $C7EEBD3C,
    $0E297602, $58B9BBB4, $EF7901E6, $64A28A62, $A509875A, $F8834442, $2702C709,
    $07353F31, $3B39F665, $F5B18B49, $4010AE37, $784DE00B, $7A1121E9, $DE918ED3,
    $C8529DCD, $816A5D05, $02ED8298, $04E3DD84, $FD2BC3E2, $AF167089, $96AF367E,
    $A4DA6232, $18FF7325, $05F9A9F1, $4FEFB9F9, $CD94EAA5, $BFAA5069, $A0B8C077,
    $60D86F57, $FE71C813, $29EBD2C8, $4CA86538, $6BF1A030, $A237B88A, $AA8AF41D,
    $E1F7B6EC, $E214D953, $33057879, $49CAA736, $FA45CFF3, $C063B411, $BA7E27D0,
    $31533819, $2A004AC1, $210EFC3F, $2646885E, $66727DCF, $9D7FBF54, $A8DD0EA8,
    $3447CACE, $3F0C14DB, $B8382AAC, $4ACE3539, $0A518D51, $95178981, $35AEE2CA,
    $73F0F7E3, $94281140, $59D0E523, $D292CB88, $565D1B27, $7EC8FBAF, $069AF08D,
    $C127FD24, $0BC77B10, $5F03E7EF, $453E99BA, $EED9FF7F, $87B55215, $7915AB4C,
    $D389A358, $5E75CE6D, $28D655C0, $DAD26C73, $2E2510FF, $9FA7EECC, $1D0629C3,
    $DC9C9C46, $2D67ECD7, $E75E94BD, $3D649E2A, $6C413A2B, $706F0D7C, $DFB0127B,
    $4E366B55, $2C825650, $24205720, $B5C998F7, $3E95462C, $756E5C72, $3259488F,
    $11E8771A, $A7C0A617, $577663E5, $089B6401, $8EAB1941, $AE55EF8C, $3AAC5460,
    $D4E6262F, $5D979A47, $B19823B0, $7F8D6A0C, $FFA08683, $0170CD0F, $858CD5D8,
    $53961C90, $C4C61556, $41F2F226, $CFCD062D, $F24C03B8, $EA81DF5B, $7BE2FA52,
    $B361F98B, $C2901316, $55BA4BBC, $93B234A9, $0FBC6603, $80A96822, $6D60491F,
    $22BD00F8, $BCAD5AAD, $52F3F13B, $42FD2B28, $B41DD01C, $C52C93BF, $FC663094,
    $8F58D100, $43FECC08, $C6331E5D, $E6480F66, $CA847204, $4BDF1DA0, $30CC2EFB,
    $13E02DEA, $FB49AC45, $F9D4434F, $F47C5B9C, $148879C2, $039FC234, $A3DB9BFC,
    $D1A1DC5C, $763D7CD4, $ED6D2F93, $AB13AF6E, $1E8E054A, $D68F4F9A, $C30484B3,
    $D7D50AFA, $6930855F, $CC07DB95, $CE746DB1, $744E967D, $F16CF575, $8643E8B5,
    $F0EAE38E, $E52DE1D1, $6587DAE0, $0C4B8121, $1C7AC567, $AC0DB20A, $36C3A812,
    $5B1A4514, $A9A3F868, $B9263BAA, $CB3CE9D2, $E44FB1A4, $9221BC82, $B29390FE,
    $6AB41863, $974A3E2E, $89F531C5, $255CA13E, $8B65D348, $EC248F78, $D8FC16F0,
    $50ECDEEE, $09010792, $3C7D1FB2, $EBA5426B, $847B417A, $468B40D9, $8DC4E680,
    $7CC1F391, $2F1EB086, $6E5BAA6A, $E0B395DA, $E31B2CF6, $D9690B0D, $729EC464,
    $38403DDE, $610B80A2, $5CF433AB, $B0785FC4, $D512E4C6, $BBB7D699, $5A86591B,
    $10CF5376, $12BF9F4B, $980FBAA1, $992A4E70, $20FA7AE7, $F7996EBB, $C918A2BE,
    $82DE74F2, $AD54209B, $F66B4D74, $1FC5B771, $169D9229, $887761DF, $00B667D5,
    $DB425E59, $B72F2844, $9B0AC1F5, $9C737E3A, $2B85476C, $6722ADD6, $44A63297,
    $0D688CED), THashLibUInt32Array.Create($ABC59484, $4107778A, $8AD94C6F,
    $FE83DF90, $0F64053F, $D1292E9D, $C5744356, $8DD1ABB4, $4C4E7667, $FB4A7FC1,
    $74F402CB, $70F06AFD, $A82286F2, $918DD076, $7A97C5CE, $48F7BDE3, $6A04D11D,
    $AC243EF7, $33AC10CA, $2F7A341E, $5F75157A, $F4773381, $591C870E, $78DF8CC8,
    $22F3ADB0, $251A5993, $09FBEF66, $796942A8, $97541D2E, $2373DAA9, $1BD2F142,
    $B57E8EB2, $E1A5BFDB, $7D0EFA92, $B3442C94, $D2CB6447, $386AC97E, $66D61805,
    $BDADA15E, $11BC1AA7, $14E9F6EA, $E533A0C0, $F935EE0A, $8FEE8A04, $810D6D85,
    $7C68B6D6, $4EDC9AA2, $956E897D, $ED87581A, $264BE9D7, $FF4DDB29, $823857C2,
    $E005A9A0, $F1CC2450, $6F9951E1, $AADE2310, $E70C75F5, $83E1A31F, $4F7DDE8E,
    $F723B563, $368E0928, $86362B71, $21E8982D, $DFB3F92B, $44676352, $99EFBA31,
    $2EAB4E1C, $FC6CA5E7, $0EBE5D4E, $A0717D0C, $B64F8199, $946B31A1, $5656CBC6,
    $CFFEC3EF, $622766C9, $FA211E35, $52F98B89, $6D01674B, $4978A802, $F651F701,
    $15B0D43D, $D6FF4683, $3463855F, $672BA29C, $BC128312, $4626A70D, $C8927A5A,
    $B8481CF9, $1C962262, $A21196BA, $BABA5EE9, $5BB162D0, $69943BD1, $0C47E35C,
    $8CC9619A, $E284D948, $271BF264, $C27FB398, $4BC70897, $60CF202C, $7F42D6AA,
    $A5A13506, $5D3E8860, $CEA63D3C, $63BF0A8F, $F02E9EFA, $B17B0674, $B072B1D3,
    $06E5723B, $3737E436, $24AA49C7, $0DED0D18, $DB256B14, $58B27877, $ECB49F54,
    $6C40256A, $6EA92FFB, $3906AA4C, $C9866FD5, $4549323E, $A7B85FAB, $1918CC27,
    $7308D7B5, $1E16C7AD, $71850B37, $3095FD78, $A63B70E6, $D880E2AE, $3E282769,
    $A39BA6BC, $98700FA3, $F34C53E8, $288AF426, $B99D930F, $F5B99DF1, $E9D0C8CF,
    $5AC8405D, $50E7217B, $511FBBBE, $2CA2E639, $C020301B, $356DBC00, $8E43DDB9,
    $4D327B4A, $F20FF3ED, $1DBB29BD, $43D44779, $A1B68F70, $6114455B, $E63D280B,
    $6BF6FF65, $10FC39E5, $3DAE126E, $C1D7CF11, $CB60B795, $1789D5B3, $9BCA36B7,
    $08306075, $84615608, $8B3A0186, $E88FBECD, $7BA47C4D, $2DE44DAC, $653FE58D,
    $CCA0B968, $D7FA0E72, $93901780, $1F2C26CC, $AE595B6B, $A9ECEA9B, $E3DBF8C4,
    $319CC130, $12981196, $01A3A4DE, $32C454B6, $755BD817, $3CD871E4, $A48BB8DA,
    $02FDEC09, $FD2DC2E2, $9E578088, $9A9F916D, $4065FE6C, $1853999E, $C7793F23,
    $DC1016BB, $969355FF, $7EF292F6, $CDCE4ADC, $05E24416, $85C16C46, $D441D37F,
    $57BD6855, $8746F54F, $9CA773DF, $770BAE22, $54828413, $B75E4B19, $04C35C03,
    $BF7CCA07, $2955C4DD, $721DB041, $B2394F33, $03F51387, $89B73C9F, $0B1737F3,
    $07E69024, $9231D245, $76193861, $88159C15, $DEB552D9, $D9767E40, $20C6C0C3,
    $4281977C, $F8AFE1E0, $D32A0751, $3FC27432, $DDF1DCC5, $68581F34, $3BCD5025,
    $0091B2EE, $4AEB6944, $1602E743, $EA09EB58, $EF0A2A8B, $641E03A5, $EB50E021,
    $5C8CCEF8, $802FF0B8, $D5E3EDFE, $C4DD1B49, $5334CD2A, $13F82D2F, $47450C20,
    $55DAFBD2, $BEC0C6F4, $B45D7959, $3AD36E8C, $0AA8AC57, $1A3C8D73, $E45AAFB1,
    $9F664838, $C6880053, $D0039BBF, $EE5F19EB, $CA0041D8, $BBEA3AAF, $DA628291,
    $9D5C95D4, $ADD504A6, $C39AB482, $5E9E14A4, $2BE065F0, $2A13FC3A, $9052E8EC,
    $AF6F5AFC), THashLibUInt32Array.Create($519AA8B5, $BB303DA9, $E00E2B10,
    $DFA6C1DB, $2E6B952E, $EE10DC23, $37936D09, $1FC42E92, $39B25A9F, $13FF89F4,
    $C8F53FEA, $18500BC7, $95A0379D, $98F751C2, $2289C42F, $A21E4098, $6F391F41,
    $F27E7E58, $0D0DF887, $4B79D540, $8E8409AA, $71FE46F8, $688A9B29, $3F08B548,
    $84ABE03A, $5E91B6C1, $FDE4C2AE, $251D0E72, $92D4FEE5, $F9371967, $9175108F,
    $E6E81835, $8C8CB8EE, $B55A67B3, $CEF138CC, $8B256268, $00D815F5, $E8810812,
    $77826189, $EA73267D, $19B90F8D, $45C33BB4, $82477056, $E1770075, $09467AA6,
    $A7C6F54A, $79768742, $61B86BCA, $D6644A44, $E33F0171, $C229FBCD, $41B08FEB,
    $D1903E30, $65EC9080, $563D6FBD, $F56DA488, $EBF64CD8, $4934426B, $7C8592FC,
    $6ACA8CF2, $1CEA111B, $3A57EE7A, $ACE11C0D, $9942D85E, $C4613407, $FA8E643B,
    $327FC701, $4CA9BE82, $3352526D, $2C047F63, $F3A8F7DD, $1A4A98A8, $762ED4D1,
    $27C75008, $BDF497C0, $7A7B84DF, $315C28AB, $801F93E3, $F19B0CA1, $8F14E46A,
    $E48BA333, $9605E625, $F03ECB60, $60385F2D, $902845BA, $7F96D66F, $24BFF05C,
    $2820730B, $947133CB, $D444828A, $B343F6F1, $0BEF4705, $8DA574F9, $01E25D6C,
    $1732793E, $4F0F7B27, $364B7117, $B2D1DA77, $A6C5F1E9, $574CA5B1, $386A3076,
    $AD6894D6, $1156D7FA, $A48D1D9A, $4794C0AF, $150C0AA0, $26D348AC, $29FDEABE,
    $A5DEDE53, $81671E8E, $594EE3BF, $A96C56E6, $3426A726, $C5976579, $BC22E5E4,
    $C1006319, $DAAFDD2A, $A1A1AA83, $3BADD0E7, $C3B14981, $D770B155, $CCD7C693,
    $42E944C5, $03E0064F, $CA95B4EF, $3DEE81C3, $FBBCD98C, $1E07E15B, $667CE949,
    $E7D6773F, $21B6124B, $6B2A6EF7, $D3278A9C, $9A988304, $75D2AE9B, $FE49E2FF,
    $9BC24F46, $74CC2CF6, $A3139F36, $6C9EF35A, $9FC1DFFE, $9E5FACDC, $AADC8BBB,
    $5ABDBC5F, $44B3B390, $F754EFA7, $5FE3BDB7, $4E59C886, $06A4C984, $A0338878,
    $CD513CD7, $63EBD27E, $8ABA80AD, $50DA144E, $5D9F4E97, $025B751C, $2D580200,
    $B6C05837, $580AA15D, $54022A6E, $B41A5415, $4863FAB6, $B0B79957, $46D0D159,
    $DC2B8650, $20A7BB0C, $4A032974, $EC8636A2, $8548F24C, $F6A2BF16, $1088F4B0,
    $0C2F3A94, $525DC396, $14065785, $2B4DCA52, $08AEED39, $ABEDFC99, $B1DBCF18,
    $87F85BBC, $AE3AFF61, $433CCD70, $5B23CC64, $7B453213, $5355C545, $9318EC0A,
    $78692D31, $0A21693D, $D5666814, $05FB59D9, $C71985B2, $2ABB8E0E, $CF6E6C91,
    $D9CFE7C6, $EFE7132C, $9711AB28, $3CE52732, $12D516D2, $7209A0D0, $D278D306,
    $70FA4B7B, $1D407DD3, $DB0BEBA4, $BFD97621, $A8BE21E1, $1B6F1B66, $30650DDA,
    $BA7DDBB9, $7DF953FB, $9D1C3902, $EDF0E8D5, $B8741AE0, $0F240565, $62CD438B,
    $C616A924, $AF7A96A3, $35365538, $E583AF4D, $73415EB8, $23176A47, $FC9CCEE8,
    $7EFC9DE2, $695E03CF, $F8CE66D4, $88B4781D, $67DD9C03, $3E8F9E73, $C0C95C51,
    $BE314D22, $55AA0795, $CB1BB011, $E980FDC8, $9C62B7CE, $DE2D239E, $042CADF3,
    $FFDF04DE, $5CE6A60F, $D8C831ED, $B7B5B9EC, $B9CBF962, $E253B254, $0735BA1F,
    $16AC917F, $DD607C2B, $64A335C4, $40159A7C, $869222F0, $6EF21769, $839D20A5,
    $D03B24C9, $F412601E, $6D72A243, $0E018DFD, $89F3721A, $C94F4134, $2F992F20,
    $4D87253C));

{$ENDREGION}
end;

end.
