/*
 * Copyright (c) 2015, 2016, Yutaka Tsutano
 *
 * Permission to use, copy, modify, and/or distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES WITH
 * REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, DIRECT,
 * INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
 * LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
 * OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */

#include "jitana/vm_core/insn_info.hpp"

using namespace jitana;
using namespace jitana::detail;

typedef insn_fmt_id f;
typedef insn_props p;

// clang-format off

static const insn_info insn_info_tbl[]{
    /* 0x00 */ { opcode::op_nop,                        "nop",                         f::fmt_10x , 1, p::can_continue },
    /* 0x01 */ { opcode::op_move,                       "move",                        f::fmt_12x , 1, p::can_continue | p::sets_register },
    /* 0x02 */ { opcode::op_move_from16,                "move/from16",                 f::fmt_22x , 2, p::can_continue | p::sets_register },
    /* 0x03 */ { opcode::op_move_16,                    "move/16",                     f::fmt_32x , 3, p::can_continue | p::sets_register },
    /* 0x04 */ { opcode::op_move_wide,                  "move-wide",                   f::fmt_12x , 1, p::can_continue | p::sets_register | p::sets_wide_register | p::reads_wide_register },
    /* 0x05 */ { opcode::op_move_wide_from16,           "move-wide/from16",            f::fmt_22x , 2, p::can_continue | p::sets_register | p::sets_wide_register | p::reads_wide_register },
    /* 0x06 */ { opcode::op_move_wide_16,               "move-wide/16",                f::fmt_32x , 3, p::can_continue | p::sets_register | p::sets_wide_register | p::reads_wide_register },
    /* 0x07 */ { opcode::op_move_object,                "move-object",                 f::fmt_12x , 1, p::can_continue | p::sets_register },
    /* 0x08 */ { opcode::op_move_object_from16,         "move-object/from16",          f::fmt_22x , 2, p::can_continue | p::sets_register },
    /* 0x09 */ { opcode::op_move_object_16,             "move-object/16",              f::fmt_32x , 3, p::can_continue | p::sets_register },
    /* 0x0a */ { opcode::op_move_result,                "move-result",                 f::fmt_11x , 1, p::can_continue | p::sets_register },
    /* 0x0b */ { opcode::op_move_result_wide,           "move-result-wide",            f::fmt_11x , 1, p::can_continue | p::sets_register | p::sets_wide_register },
    /* 0x0c */ { opcode::op_move_result_object,         "move-result-object",          f::fmt_11x , 1, p::can_continue | p::sets_register },
    /* 0x0d */ { opcode::op_move_exception,             "move-exception",              f::fmt_11x , 1, p::can_continue | p::sets_register },
    /* 0x0e */ { opcode::op_return_void,                "return-void",                 f::fmt_10x , 1, p::can_return },
    /* 0x0f */ { opcode::op_return,                     "return",                      f::fmt_11x , 1, p::can_return | p::sets_result },
    /* 0x10 */ { opcode::op_return_wide,                "return-wide",                 f::fmt_11x , 1, p::can_return | p::reads_wide_register | p::sets_result },
    /* 0x11 */ { opcode::op_return_object,              "return-object",               f::fmt_11x , 1, p::can_return | p::sets_result },
    /* 0x12 */ { opcode::op_const_4,                    "const/4",                     f::fmt_11n , 1, p::can_continue | p::sets_register },
    /* 0x13 */ { opcode::op_const_16,                   "const/16",                    f::fmt_21s , 2, p::can_continue | p::sets_register },
    /* 0x14 */ { opcode::op_const,                      "const",                       f::fmt_31i , 3, p::can_continue | p::sets_register },
    /* 0x15 */ { opcode::op_const_high16,               "const/high16",                f::fmt_21h , 2, p::can_continue | p::sets_register },
    /* 0x16 */ { opcode::op_const_wide_16,              "const-wide/16",               f::fmt_21s , 2, p::can_continue | p::sets_register | p::sets_wide_register },
    /* 0x17 */ { opcode::op_const_wide_32,              "const-wide/32",               f::fmt_31i , 3, p::can_continue | p::sets_register | p::sets_wide_register },
    /* 0x18 */ { opcode::op_const_wide,                 "const-wide",                  f::fmt_51l , 5, p::can_continue | p::sets_register | p::sets_wide_register },
    /* 0x19 */ { opcode::op_const_wide_high16,          "const-wide/high16",           f::fmt_21h , 2, p::can_continue | p::sets_register | p::sets_wide_register },
    /* 0x1a */ { opcode::op_const_string,               "const-string",                f::fmt_21c , 2, p::can_throw | p::can_continue | p::sets_register },
    /* 0x1b */ { opcode::op_const_string_jumbo,         "const-string/jumbo",          f::fmt_31c , 3, p::can_throw | p::can_continue | p::sets_register },
    /* 0x1c */ { opcode::op_const_class,                "const-class",                 f::fmt_21c , 2, p::can_throw | p::can_continue | p::sets_register },
    /* 0x1d */ { opcode::op_monitor_enter,              "monitor-enter",               f::fmt_11x , 1, p::can_throw | p::can_continue },
    /* 0x1e */ { opcode::op_monitor_exit,               "monitor-exit",                f::fmt_11x , 1, p::can_throw | p::can_continue },
    /* 0x1f */ { opcode::op_check_cast,                 "check-cast",                  f::fmt_21c , 2, p::can_throw | p::can_continue | p::sets_register },
    /* 0x20 */ { opcode::op_instance_of,                "instance-of",                 f::fmt_22c , 2, p::can_throw | p::can_continue | p::sets_register },
    /* 0x21 */ { opcode::op_array_length,               "array-length",                f::fmt_12x , 1, p::can_throw | p::can_continue | p::sets_register },
    /* 0x22 */ { opcode::op_new_instance,               "new-instance",                f::fmt_21c , 2, p::can_throw | p::can_continue | p::sets_register },
    /* 0x23 */ { opcode::op_new_array,                  "new-array",                   f::fmt_22c , 2, p::can_throw | p::can_continue | p::sets_register },
    /* 0x24 */ { opcode::op_filled_new_array,           "filled-new-array",            f::fmt_35c , 3, p::can_throw | p::can_continue | p::sets_result },
    /* 0x25 */ { opcode::op_filled_new_array_range,     "filled-new-array/range",      f::fmt_3rc , 3, p::can_throw | p::can_continue | p::sets_result },
    /* 0x26 */ { opcode::op_fill_array_data,            "fill-array-data",             f::fmt_31t , 3, p::can_continue },
    /* 0x27 */ { opcode::op_throw,                      "throw",                       f::fmt_11x , 1, p::can_throw },
    /* 0x28 */ { opcode::op_goto,                       "goto",                        f::fmt_10t , 1, p::none },
    /* 0x29 */ { opcode::op_goto_16,                    "goto/16",                     f::fmt_20t , 2, p::none },
    /* 0x2a */ { opcode::op_goto_32,                    "goto/32",                     f::fmt_30t , 3, p::none },
    /* 0x2b */ { opcode::op_packed_switch,              "packed-switch",               f::fmt_31t , 3, p::can_continue },
    /* 0x2c */ { opcode::op_sparse_switch,              "sparse-switch",               f::fmt_31t , 3, p::can_continue },
    /* 0x2d */ { opcode::op_cmpl_float,                 "cmpl-float",                  f::fmt_23x , 2, p::can_continue | p::sets_register },
    /* 0x2e */ { opcode::op_cmpg_float,                 "cmpg-float",                  f::fmt_23x , 2, p::can_continue | p::sets_register },
    /* 0x2f */ { opcode::op_cmpl_double,                "cmpl-double",                 f::fmt_23x , 2, p::can_continue | p::sets_register | p::reads_wide_register },
    /* 0x30 */ { opcode::op_cmpg_double,                "cmpg-double",                 f::fmt_23x , 2, p::can_continue | p::sets_register | p::reads_wide_register },
    /* 0x31 */ { opcode::op_cmp_long,                   "cmp-long",                    f::fmt_23x , 2, p::can_continue | p::sets_register | p::reads_wide_register },
    /* 0x32 */ { opcode::op_if_eq,                      "if-eq",                       f::fmt_22t , 2, p::can_continue },
    /* 0x33 */ { opcode::op_if_ne,                      "if-ne",                       f::fmt_22t , 2, p::can_continue },
    /* 0x34 */ { opcode::op_if_lt,                      "if-lt",                       f::fmt_22t , 2, p::can_continue },
    /* 0x35 */ { opcode::op_if_ge,                      "if-ge",                       f::fmt_22t , 2, p::can_continue },
    /* 0x36 */ { opcode::op_if_gt,                      "if-gt",                       f::fmt_22t , 2, p::can_continue },
    /* 0x37 */ { opcode::op_if_le,                      "if-le",                       f::fmt_22t , 2, p::can_continue },
    /* 0x38 */ { opcode::op_if_eqz,                     "if-eqz",                      f::fmt_21t , 2, p::can_continue },
    /* 0x39 */ { opcode::op_if_nez,                     "if-nez",                      f::fmt_21t , 2, p::can_continue },
    /* 0x3a */ { opcode::op_if_ltz,                     "if-ltz",                      f::fmt_21t , 2, p::can_continue },
    /* 0x3b */ { opcode::op_if_gez,                     "if-gez",                      f::fmt_21t , 2, p::can_continue },
    /* 0x3c */ { opcode::op_if_gtz,                     "if-gtz",                      f::fmt_21t , 2, p::can_continue },
    /* 0x3d */ { opcode::op_if_lez,                     "if-lez",                      f::fmt_21t , 2, p::can_continue },
    /* 0x3e */ { opcode::op_3e,                         "(unused)",                    f::fmt_10x , 1, p::none },
    /* 0x3f */ { opcode::op_3f,                         "(unused)",                    f::fmt_10x , 1, p::none },
    /* 0x40 */ { opcode::op_40,                         "(unused)",                    f::fmt_10x , 1, p::none },
    /* 0x41 */ { opcode::op_41,                         "(unused)",                    f::fmt_10x , 1, p::none },
    /* 0x42 */ { opcode::op_42,                         "(unused)",                    f::fmt_10x , 1, p::none },
    /* 0x43 */ { opcode::op_43,                         "(unused)",                    f::fmt_10x , 1, p::none },
    /* 0x44 */ { opcode::op_aget,                       "aget",                        f::fmt_23x , 2, p::can_throw | p::can_continue | p::sets_register },
    /* 0x45 */ { opcode::op_aget_wide,                  "aget-wide",                   f::fmt_23x , 2, p::can_throw | p::can_continue | p::sets_register | p::sets_wide_register },
    /* 0x46 */ { opcode::op_aget_object,                "aget-object",                 f::fmt_23x , 2, p::can_throw | p::can_continue | p::sets_register },
    /* 0x47 */ { opcode::op_aget_boolean,               "aget-boolean",                f::fmt_23x , 2, p::can_throw | p::can_continue | p::sets_register },
    /* 0x48 */ { opcode::op_aget_byte,                  "aget-byte",                   f::fmt_23x , 2, p::can_throw | p::can_continue | p::sets_register },
    /* 0x49 */ { opcode::op_aget_char,                  "aget-char",                   f::fmt_23x , 2, p::can_throw | p::can_continue | p::sets_register },
    /* 0x4a */ { opcode::op_aget_short,                 "aget-short",                  f::fmt_23x , 2, p::can_throw | p::can_continue | p::sets_register },
    /* 0x4b */ { opcode::op_aput,                       "aput",                        f::fmt_23x , 2, p::can_throw | p::can_continue },
    /* 0x4c */ { opcode::op_aput_wide,                  "aput-wide",                   f::fmt_23x , 2, p::can_throw | p::can_continue | p::reads_wide_register },
    /* 0x4d */ { opcode::op_aput_object,                "aput-object",                 f::fmt_23x , 2, p::can_throw | p::can_continue },
    /* 0x4e */ { opcode::op_aput_boolean,               "aput-boolean",                f::fmt_23x , 2, p::can_throw | p::can_continue },
    /* 0x4f */ { opcode::op_aput_byte,                  "aput-byte",                   f::fmt_23x , 2, p::can_throw | p::can_continue },
    /* 0x50 */ { opcode::op_aput_char,                  "aput-char",                   f::fmt_23x , 2, p::can_throw | p::can_continue },
    /* 0x51 */ { opcode::op_aput_short,                 "aput-short",                  f::fmt_23x , 2, p::can_throw | p::can_continue },
    /* 0x52 */ { opcode::op_iget,                       "iget",                        f::fmt_22c , 2, p::can_throw | p::can_continue | p::sets_register },
    /* 0x53 */ { opcode::op_iget_wide,                  "iget-wide",                   f::fmt_22c , 2, p::can_throw | p::can_continue | p::sets_register | p::sets_wide_register },
    /* 0x54 */ { opcode::op_iget_object,                "iget-object",                 f::fmt_22c , 2, p::can_throw | p::can_continue | p::sets_register },
    /* 0x55 */ { opcode::op_iget_boolean,               "iget-boolean",                f::fmt_22c , 2, p::can_throw | p::can_continue | p::sets_register },
    /* 0x56 */ { opcode::op_iget_byte,                  "iget-byte",                   f::fmt_22c , 2, p::can_throw | p::can_continue | p::sets_register },
    /* 0x57 */ { opcode::op_iget_char,                  "iget-char",                   f::fmt_22c , 2, p::can_throw | p::can_continue | p::sets_register },
    /* 0x58 */ { opcode::op_iget_short,                 "iget-short",                  f::fmt_22c , 2, p::can_throw | p::can_continue | p::sets_register },
    /* 0x59 */ { opcode::op_iput,                       "iput",                        f::fmt_22c , 2, p::can_throw | p::can_continue },
    /* 0x5a */ { opcode::op_iput_wide,                  "iput-wide",                   f::fmt_22c , 2, p::can_throw | p::can_continue | p::reads_wide_register },
    /* 0x5b */ { opcode::op_iput_object,                "iput-object",                 f::fmt_22c , 2, p::can_throw | p::can_continue },
    /* 0x5c */ { opcode::op_iput_boolean,               "iput-boolean",                f::fmt_22c , 2, p::can_throw | p::can_continue },
    /* 0x5d */ { opcode::op_iput_byte,                  "iput-byte",                   f::fmt_22c , 2, p::can_throw | p::can_continue },
    /* 0x5e */ { opcode::op_iput_char,                  "iput-char",                   f::fmt_22c , 2, p::can_throw | p::can_continue },
    /* 0x5f */ { opcode::op_iput_short,                 "iput-short",                  f::fmt_22c , 2, p::can_throw | p::can_continue },
    /* 0x60 */ { opcode::op_sget,                       "sget",                        f::fmt_21c , 2, p::can_throw | p::can_continue | p::sets_register },
    /* 0x61 */ { opcode::op_sget_wide,                  "sget-wide",                   f::fmt_21c , 2, p::can_throw | p::can_continue | p::sets_register | p::sets_wide_register },
    /* 0x62 */ { opcode::op_sget_object,                "sget-object",                 f::fmt_21c , 2, p::can_throw | p::can_continue | p::sets_register },
    /* 0x63 */ { opcode::op_sget_boolean,               "sget-boolean",                f::fmt_21c , 2, p::can_throw | p::can_continue | p::sets_register },
    /* 0x64 */ { opcode::op_sget_byte,                  "sget-byte",                   f::fmt_21c , 2, p::can_throw | p::can_continue | p::sets_register },
    /* 0x65 */ { opcode::op_sget_char,                  "sget-char",                   f::fmt_21c , 2, p::can_throw | p::can_continue | p::sets_register },
    /* 0x66 */ { opcode::op_sget_short,                 "sget-short",                  f::fmt_21c , 2, p::can_throw | p::can_continue | p::sets_register },
    /* 0x67 */ { opcode::op_sput,                       "sput",                        f::fmt_21c , 2, p::can_throw | p::can_continue },
    /* 0x68 */ { opcode::op_sput_wide,                  "sput-wide",                   f::fmt_21c , 2, p::can_throw | p::can_continue | p::reads_wide_register },
    /* 0x69 */ { opcode::op_sput_object,                "sput-object",                 f::fmt_21c , 2, p::can_throw | p::can_continue },
    /* 0x6a */ { opcode::op_sput_boolean,               "sput-boolean",                f::fmt_21c , 2, p::can_throw | p::can_continue },
    /* 0x6b */ { opcode::op_sput_byte,                  "sput-byte",                   f::fmt_21c , 2, p::can_throw | p::can_continue },
    /* 0x6c */ { opcode::op_sput_char,                  "sput-char",                   f::fmt_21c , 2, p::can_throw | p::can_continue },
    /* 0x6d */ { opcode::op_sput_short,                 "sput-short",                  f::fmt_21c , 2, p::can_throw | p::can_continue },
    /* 0x6e */ { opcode::op_invoke_virtual,             "invoke-virtual",              f::fmt_35c , 3, p::can_throw | p::can_continue | p::sets_result | p::can_invoke },
    /* 0x6f */ { opcode::op_invoke_super,               "invoke-super",                f::fmt_35c , 3, p::can_throw | p::can_continue | p::sets_result | p::can_invoke },
    /* 0x70 */ { opcode::op_invoke_direct,              "invoke-direct",               f::fmt_35c , 3, p::can_throw | p::can_continue | p::sets_result | p::can_invoke | p::can_initialize_reference },
    /* 0x71 */ { opcode::op_invoke_static,              "invoke-static",               f::fmt_35c , 3, p::can_throw | p::can_continue | p::sets_result | p::can_invoke },
    /* 0x72 */ { opcode::op_invoke_interface,           "invoke-interface",            f::fmt_35c , 3, p::can_throw | p::can_continue | p::sets_result | p::can_invoke },
    /* 0x73 */ { opcode::op_73,                         "(unused)",                    f::fmt_10x , 1, p::none },
    /* 0x74 */ { opcode::op_invoke_virtual_range,       "invoke-virtual/range",        f::fmt_3rc , 3, p::can_throw | p::can_continue | p::sets_result | p::can_invoke },
    /* 0x75 */ { opcode::op_invoke_super_range,         "invoke-super/range",          f::fmt_3rc , 3, p::can_throw | p::can_continue | p::sets_result | p::can_invoke },
    /* 0x76 */ { opcode::op_invoke_direct_range,        "invoke-direct/range",         f::fmt_3rc , 3, p::can_throw | p::can_continue | p::sets_result | p::can_invoke | p::can_initialize_reference },
    /* 0x77 */ { opcode::op_invoke_static_range,        "invoke-static/range",         f::fmt_3rc , 3, p::can_throw | p::can_continue | p::sets_result | p::can_invoke },
    /* 0x78 */ { opcode::op_invoke_interface_range,     "invoke-interface/range",      f::fmt_3rc , 3, p::can_throw | p::can_continue | p::sets_result | p::can_invoke },
    /* 0x79 */ { opcode::op_79,                         "(unused)",                    f::fmt_10x , 1, p::none },
    /* 0x7a */ { opcode::op_7a,                         "(unused)",                    f::fmt_10x , 1, p::none },
    /* 0x7b */ { opcode::op_neg_int,                    "neg-int",                     f::fmt_12x , 1, p::can_continue | p::sets_register },
    /* 0x7c */ { opcode::op_not_int,                    "not-int",                     f::fmt_12x , 1, p::can_continue | p::sets_register },
    /* 0x7d */ { opcode::op_neg_long,                   "neg-long",                    f::fmt_12x , 1, p::can_continue | p::sets_register | p::sets_wide_register | p::reads_wide_register },
    /* 0x7e */ { opcode::op_not_long,                   "not-long",                    f::fmt_12x , 1, p::can_continue | p::sets_register | p::sets_wide_register | p::reads_wide_register },
    /* 0x7f */ { opcode::op_neg_float,                  "neg-float",                   f::fmt_12x , 1, p::can_continue | p::sets_register },
    /* 0x80 */ { opcode::op_neg_double,                 "neg-double",                  f::fmt_12x , 1, p::can_continue | p::sets_register | p::sets_wide_register | p::reads_wide_register },
    /* 0x81 */ { opcode::op_int_to_long,                "int-to-long",                 f::fmt_12x , 1, p::can_continue | p::sets_register | p::sets_wide_register | p::reads_wide_register },
    /* 0x82 */ { opcode::op_int_to_float,               "int-to-float",                f::fmt_12x , 1, p::can_continue | p::sets_register },
    /* 0x83 */ { opcode::op_int_to_double,              "int-to-double",               f::fmt_12x , 1, p::can_continue | p::sets_register | p::sets_wide_register | p::reads_wide_register },
    /* 0x84 */ { opcode::op_long_to_int,                "long-to-int",                 f::fmt_12x , 1, p::can_continue | p::sets_register },
    /* 0x85 */ { opcode::op_long_to_float,              "long-to-float",               f::fmt_12x , 1, p::can_continue | p::sets_register },
    /* 0x86 */ { opcode::op_long_to_double,             "long-to-double",              f::fmt_12x , 1, p::can_continue | p::sets_register | p::sets_wide_register | p::reads_wide_register },
    /* 0x87 */ { opcode::op_float_to_int,               "float-to-int",                f::fmt_12x , 1, p::can_continue | p::sets_register },
    /* 0x88 */ { opcode::op_float_to_long,              "float-to-long",               f::fmt_12x , 1, p::can_continue | p::sets_register | p::sets_wide_register | p::reads_wide_register },
    /* 0x89 */ { opcode::op_float_to_double,            "float-to-double",             f::fmt_12x , 1, p::can_continue | p::sets_register | p::sets_wide_register | p::reads_wide_register },
    /* 0x8a */ { opcode::op_double_to_int,              "double-to-int",               f::fmt_12x , 1, p::can_continue | p::sets_register },
    /* 0x8b */ { opcode::op_double_to_long,             "double-to-long",              f::fmt_12x , 1, p::can_continue | p::sets_register | p::sets_wide_register | p::reads_wide_register },
    /* 0x8c */ { opcode::op_double_to_float,            "double-to-float",             f::fmt_12x , 1, p::can_continue | p::sets_register },
    /* 0x8d */ { opcode::op_int_to_byte,                "int-to-byte",                 f::fmt_12x , 1, p::can_continue | p::sets_register },
    /* 0x8e */ { opcode::op_int_to_char,                "int-to-char",                 f::fmt_12x , 1, p::can_continue | p::sets_register },
    /* 0x8f */ { opcode::op_int_to_short,               "int-to-short",                f::fmt_12x , 1, p::can_continue | p::sets_register },
    /* 0x90 */ { opcode::op_add_int,                    "add-int",                     f::fmt_23x , 2, p::can_continue | p::sets_register },
    /* 0x91 */ { opcode::op_sub_int,                    "sub-int",                     f::fmt_23x , 2, p::can_continue | p::sets_register },
    /* 0x92 */ { opcode::op_mul_int,                    "mul-int",                     f::fmt_23x , 2, p::can_continue | p::sets_register },
    /* 0x93 */ { opcode::op_div_int,                    "div-int",                     f::fmt_23x , 2, p::can_throw | p::can_continue | p::sets_register },
    /* 0x94 */ { opcode::op_rem_int,                    "rem-int",                     f::fmt_23x , 2, p::can_throw | p::can_continue | p::sets_register },
    /* 0x95 */ { opcode::op_and_int,                    "and-int",                     f::fmt_23x , 2, p::can_continue | p::sets_register },
    /* 0x96 */ { opcode::op_or_int,                     "or-int",                      f::fmt_23x , 2, p::can_continue | p::sets_register },
    /* 0x97 */ { opcode::op_xor_int,                    "xor-int",                     f::fmt_23x , 2, p::can_continue | p::sets_register },
    /* 0x98 */ { opcode::op_shl_int,                    "shl-int",                     f::fmt_23x , 2, p::can_continue | p::sets_register },
    /* 0x99 */ { opcode::op_shr_int,                    "shr-int",                     f::fmt_23x , 2, p::can_continue | p::sets_register },
    /* 0x9a */ { opcode::op_ushr_int,                   "ushr-int",                    f::fmt_23x , 2, p::can_continue | p::sets_register },
    /* 0x9b */ { opcode::op_add_long,                   "add-long",                    f::fmt_23x , 2, p::can_continue | p::sets_register | p::sets_wide_register | p::reads_wide_register },
    /* 0x9c */ { opcode::op_sub_long,                   "sub-long",                    f::fmt_23x , 2, p::can_continue | p::sets_register | p::sets_wide_register | p::reads_wide_register },
    /* 0x9d */ { opcode::op_mul_long,                   "mul-long",                    f::fmt_23x , 2, p::can_continue | p::sets_register | p::sets_wide_register | p::reads_wide_register },
    /* 0x9e */ { opcode::op_div_long,                   "div-long",                    f::fmt_23x , 2, p::can_throw | p::can_continue | p::sets_register | p::sets_wide_register | p::reads_wide_register },
    /* 0x9f */ { opcode::op_rem_long,                   "rem-long",                    f::fmt_23x , 2, p::can_throw | p::can_continue | p::sets_register | p::sets_wide_register | p::reads_wide_register },
    /* 0xa0 */ { opcode::op_and_long,                   "and-long",                    f::fmt_23x , 2, p::can_continue | p::sets_register | p::sets_wide_register | p::reads_wide_register },
    /* 0xa1 */ { opcode::op_or_long,                    "or-long",                     f::fmt_23x , 2, p::can_continue | p::sets_register | p::sets_wide_register | p::reads_wide_register },
    /* 0xa2 */ { opcode::op_xor_long,                   "xor-long",                    f::fmt_23x , 2, p::can_continue | p::sets_register | p::sets_wide_register | p::reads_wide_register },
    /* 0xa3 */ { opcode::op_shl_long,                   "shl-long",                    f::fmt_23x , 2, p::can_continue | p::sets_register | p::sets_wide_register | p::reads_wide_register },
    /* 0xa4 */ { opcode::op_shr_long,                   "shr-long",                    f::fmt_23x , 2, p::can_continue | p::sets_register | p::sets_wide_register | p::reads_wide_register },
    /* 0xa5 */ { opcode::op_ushr_long,                  "ushr-long",                   f::fmt_23x , 2, p::can_continue | p::sets_register | p::sets_wide_register | p::reads_wide_register },
    /* 0xa6 */ { opcode::op_add_float,                  "add-float",                   f::fmt_23x , 2, p::can_continue | p::sets_register },
    /* 0xa7 */ { opcode::op_sub_float,                  "sub-float",                   f::fmt_23x , 2, p::can_continue | p::sets_register },
    /* 0xa8 */ { opcode::op_mul_float,                  "mul-float",                   f::fmt_23x , 2, p::can_continue | p::sets_register },
    /* 0xa9 */ { opcode::op_div_float,                  "div-float",                   f::fmt_23x , 2, p::can_continue | p::sets_register },
    /* 0xaa */ { opcode::op_rem_float,                  "rem-float",                   f::fmt_23x , 2, p::can_continue | p::sets_register },
    /* 0xab */ { opcode::op_add_double,                 "add-double",                  f::fmt_23x , 2, p::can_continue | p::sets_register | p::sets_wide_register | p::reads_wide_register },
    /* 0xac */ { opcode::op_sub_double,                 "sub-double",                  f::fmt_23x , 2, p::can_continue | p::sets_register | p::sets_wide_register | p::reads_wide_register },
    /* 0xad */ { opcode::op_mul_double,                 "mul-double",                  f::fmt_23x , 2, p::can_continue | p::sets_register | p::sets_wide_register | p::reads_wide_register },
    /* 0xae */ { opcode::op_div_double,                 "div-double",                  f::fmt_23x , 2, p::can_continue | p::sets_register | p::sets_wide_register | p::reads_wide_register },
    /* 0xaf */ { opcode::op_rem_double,                 "rem-double",                  f::fmt_23x , 2, p::can_continue | p::sets_register | p::sets_wide_register | p::reads_wide_register },
    /* 0xb0 */ { opcode::op_add_int_2addr,              "add-int/2addr",               f::fmt_12x , 1, p::can_continue | p::sets_register },
    /* 0xb1 */ { opcode::op_sub_int_2addr,              "sub-int/2addr",               f::fmt_12x , 1, p::can_continue | p::sets_register },
    /* 0xb2 */ { opcode::op_mul_int_2addr,              "mul-int/2addr",               f::fmt_12x , 1, p::can_continue | p::sets_register },
    /* 0xb3 */ { opcode::op_div_int_2addr,              "div-int/2addr",               f::fmt_12x , 1, p::can_throw | p::can_continue | p::sets_register },
    /* 0xb4 */ { opcode::op_rem_int_2addr,              "rem-int/2addr",               f::fmt_12x , 1, p::can_throw | p::can_continue | p::sets_register },
    /* 0xb5 */ { opcode::op_and_int_2addr,              "and-int/2addr",               f::fmt_12x , 1, p::can_continue | p::sets_register },
    /* 0xb6 */ { opcode::op_or_int_2addr,               "or-int/2addr",                f::fmt_12x , 1, p::can_continue | p::sets_register },
    /* 0xb7 */ { opcode::op_xor_int_2addr,              "xor-int/2addr",               f::fmt_12x , 1, p::can_continue | p::sets_register },
    /* 0xb8 */ { opcode::op_shl_int_2addr,              "shl-int/2addr",               f::fmt_12x , 1, p::can_continue | p::sets_register },
    /* 0xb9 */ { opcode::op_shr_int_2addr,              "shr-int/2addr",               f::fmt_12x , 1, p::can_continue | p::sets_register },
    /* 0xba */ { opcode::op_ushr_int_2addr,             "ushr-int/2addr",              f::fmt_12x , 1, p::can_continue | p::sets_register },
    /* 0xbb */ { opcode::op_add_long_2addr,             "add-long/2addr",              f::fmt_12x , 1, p::can_continue | p::sets_register | p::sets_wide_register | p::reads_wide_register },
    /* 0xbc */ { opcode::op_sub_long_2addr,             "sub-long/2addr",              f::fmt_12x , 1, p::can_continue | p::sets_register | p::sets_wide_register | p::reads_wide_register },
    /* 0xbd */ { opcode::op_mul_long_2addr,             "mul-long/2addr",              f::fmt_12x , 1, p::can_continue | p::sets_register | p::sets_wide_register | p::reads_wide_register },
    /* 0xbe */ { opcode::op_div_long_2addr,             "div-long/2addr",              f::fmt_12x , 1, p::can_throw | p::can_continue | p::sets_register | p::sets_wide_register | p::reads_wide_register },
    /* 0xbf */ { opcode::op_rem_long_2addr,             "rem-long/2addr",              f::fmt_12x , 1, p::can_throw | p::can_continue | p::sets_register | p::sets_wide_register | p::reads_wide_register },
    /* 0xc0 */ { opcode::op_and_long_2addr,             "and-long/2addr",              f::fmt_12x , 1, p::can_continue | p::sets_register | p::sets_wide_register | p::reads_wide_register },
    /* 0xc1 */ { opcode::op_or_long_2addr,              "or-long/2addr",               f::fmt_12x , 1, p::can_continue | p::sets_register | p::sets_wide_register | p::reads_wide_register },
    /* 0xc2 */ { opcode::op_xor_long_2addr,             "xor-long/2addr",              f::fmt_12x , 1, p::can_continue | p::sets_register | p::sets_wide_register | p::reads_wide_register },
    /* 0xc3 */ { opcode::op_shl_long_2addr,             "shl-long/2addr",              f::fmt_12x , 1, p::can_continue | p::sets_register | p::sets_wide_register | p::reads_wide_register },
    /* 0xc4 */ { opcode::op_shr_long_2addr,             "shr-long/2addr",              f::fmt_12x , 1, p::can_continue | p::sets_register | p::sets_wide_register | p::reads_wide_register },
    /* 0xc5 */ { opcode::op_ushr_long_2addr,            "ushr-long/2addr",             f::fmt_12x , 1, p::can_continue | p::sets_register | p::sets_wide_register | p::reads_wide_register },
    /* 0xc6 */ { opcode::op_add_float_2addr,            "add-float/2addr",             f::fmt_12x , 1, p::can_continue | p::sets_register },
    /* 0xc7 */ { opcode::op_sub_float_2addr,            "sub-float/2addr",             f::fmt_12x , 1, p::can_continue | p::sets_register },
    /* 0xc8 */ { opcode::op_mul_float_2addr,            "mul-float/2addr",             f::fmt_12x , 1, p::can_continue | p::sets_register },
    /* 0xc9 */ { opcode::op_div_float_2addr,            "div-float/2addr",             f::fmt_12x , 1, p::can_continue | p::sets_register },
    /* 0xca */ { opcode::op_rem_float_2addr,            "rem-float/2addr",             f::fmt_12x , 1, p::can_continue | p::sets_register },
    /* 0xcb */ { opcode::op_add_double_2addr,           "add-double/2addr",            f::fmt_12x , 1, p::can_continue | p::sets_register | p::sets_wide_register | p::reads_wide_register },
    /* 0xcc */ { opcode::op_sub_double_2addr,           "sub-double/2addr",            f::fmt_12x , 1, p::can_continue | p::sets_register | p::sets_wide_register | p::reads_wide_register },
    /* 0xcd */ { opcode::op_mul_double_2addr,           "mul-double/2addr",            f::fmt_12x , 1, p::can_continue | p::sets_register | p::sets_wide_register | p::reads_wide_register },
    /* 0xce */ { opcode::op_div_double_2addr,           "div-double/2addr",            f::fmt_12x , 1, p::can_continue | p::sets_register | p::sets_wide_register | p::reads_wide_register },
    /* 0xcf */ { opcode::op_rem_double_2addr,           "rem-double/2addr",            f::fmt_12x , 1, p::can_continue | p::sets_register | p::sets_wide_register | p::reads_wide_register },
    /* 0xd0 */ { opcode::op_add_int_lit16,              "add-int/lit16",               f::fmt_22s , 2, p::can_continue | p::sets_register },
    /* 0xd1 */ { opcode::op_rsub_int,                   "rsub-int/lit16",              f::fmt_22s , 2, p::can_continue | p::sets_register },
    /* 0xd2 */ { opcode::op_mul_int_lit16,              "mul-int/lit16",               f::fmt_22s , 2, p::can_continue | p::sets_register },
    /* 0xd3 */ { opcode::op_div_int_lit16,              "div-int/lit16",               f::fmt_22s , 2, p::can_throw | p::can_continue | p::sets_register },
    /* 0xd4 */ { opcode::op_rem_int_lit16,              "rem-int/lit16",               f::fmt_22s , 2, p::can_throw | p::can_continue | p::sets_register },
    /* 0xd5 */ { opcode::op_and_int_lit16,              "and-int/lit16",               f::fmt_22s , 2, p::can_continue | p::sets_register },
    /* 0xd6 */ { opcode::op_or_int_lit16,               "or-int/lit16",                f::fmt_22s , 2, p::can_continue | p::sets_register },
    /* 0xd7 */ { opcode::op_xor_int_lit16,              "xor-int/lit16",               f::fmt_22s , 2, p::can_continue | p::sets_register },
    /* 0xd8 */ { opcode::op_add_int_lit8,               "add-int/lit8",                f::fmt_22b , 2, p::can_continue | p::sets_register },
    /* 0xd9 */ { opcode::op_rsub_int_lit8,              "rsub-int/lit8",               f::fmt_22b , 2, p::can_continue | p::sets_register },
    /* 0xda */ { opcode::op_mul_int_lit8,               "mul-int/lit8",                f::fmt_22b , 2, p::can_continue | p::sets_register },
    /* 0xdb */ { opcode::op_div_int_lit8,               "div-int/lit8",                f::fmt_22b , 2, p::can_throw | p::can_continue | p::sets_register },
    /* 0xdc */ { opcode::op_rem_int_lit8,               "rem-int/lit8",                f::fmt_22b , 2, p::can_throw | p::can_continue | p::sets_register },
    /* 0xdd */ { opcode::op_and_int_lit8,               "and-int/lit8",                f::fmt_22b , 2, p::can_continue | p::sets_register },
    /* 0xde */ { opcode::op_or_int_lit8,                "or-int/lit8",                 f::fmt_22b , 2, p::can_continue | p::sets_register },
    /* 0xdf */ { opcode::op_xor_int_lit8,               "xor-int/lit8",                f::fmt_22b , 2, p::can_continue | p::sets_register },
    /* 0xe0 */ { opcode::op_shl_int_lit8,               "shl-int/lit8",                f::fmt_22b , 2, p::can_continue | p::sets_register },
    /* 0xe1 */ { opcode::op_shr_int_lit8,               "shr-int/lit8",                f::fmt_22b , 2, p::can_continue | p::sets_register },
    /* 0xe2 */ { opcode::op_ushr_int_lit8,              "ushr-int/lit8",               f::fmt_22b , 2, p::can_continue | p::sets_register },
    /* 0xe3 */ { opcode::op_iget_volatile,              "+iget-volatile",              f::fmt_22c , 2, p::can_throw | p::odex_only | p::can_continue | p::sets_register | p::odexed_instance_volatile },
    /* 0xe4 */ { opcode::op_iput_volatile,              "+iput-volatile",              f::fmt_22c , 2, p::can_throw | p::odex_only | p::can_continue | p::odexed_instance_volatile },
    /* 0xe5 */ { opcode::op_sget_volatile,              "+sget-volatile",              f::fmt_21c , 2, p::can_throw | p::odex_only | p::can_continue | p::sets_register | p::odexed_static_volatile },
    /* 0xe6 */ { opcode::op_sput_volatile,              "+sput-volatile",              f::fmt_21c , 2, p::can_throw | p::odex_only | p::can_continue | p::odexed_static_volatile },
    /* 0xe7 */ { opcode::op_iget_object_volatile,       "+iget-object-volatile",       f::fmt_22c , 2, p::can_throw | p::odex_only | p::can_continue | p::sets_register | p::odexed_instance_volatile },
    /* 0xe8 */ { opcode::op_iget_wide_volatile,         "+iget-wide-volatile",         f::fmt_22c , 2, p::can_throw | p::odex_only | p::can_continue | p::sets_register | p::sets_wide_register | p::odexed_instance_volatile },
    /* 0xe9 */ { opcode::op_iput_wide_volatile,         "+iput-wide-volatile",         f::fmt_22c , 2, p::can_throw | p::odex_only | p::can_continue | p::reads_wide_register | p::odexed_instance_volatile },
    /* 0xea */ { opcode::op_sget_wide_volatile,         "+sget-wide-volatile",         f::fmt_21c , 2, p::can_throw | p::odex_only | p::can_continue | p::sets_register | p::sets_wide_register | p::odexed_static_volatile },
    /* 0xeb */ { opcode::op_sput_wide_volatile,         "+sput-wide-volatile",         f::fmt_21c , 2, p::can_throw | p::odex_only | p::can_continue | p::reads_wide_register | p::odexed_static_volatile },
    /* 0xec */ { opcode::op_ec,                         "+breakpoint",                 f::fmt_00x , 1, p::none },
    /* 0xed */ { opcode::op_throw_verification_error,   "+throw-verification-error",   f::fmt_20bc, 2, p::can_throw | p::odex_only },
    /* 0xee */ { opcode::op_execute_inline,             "+execute-inline",             f::fmt_35ms, 3, p::can_throw | p::odex_only | p::can_continue | p::sets_result },
    /* 0xef */ { opcode::op_execute_inline_range,       "+execute-inline/range",       f::fmt_3rms, 3, p::can_throw | p::odex_only | p::can_continue | p::sets_result },
    /* 0xf0 */ { opcode::op_invoke_object_init_range,   "+invoke-object-init/range",   f::fmt_35c , 3, p::can_throw | p::odex_only | p::can_continue | p::sets_result | p::can_initialize_reference | p::can_invoke },
    /* 0xf1 */ { opcode::op_return_void_barrier,        "+return-void-barrier",        f::fmt_10x , 1, p::odex_only | p::can_return },
    /* 0xf2 */ { opcode::op_iget_quick,                 "+iget-quick",                 f::fmt_22cs, 2, p::can_throw | p::odex_only | p::can_continue | p::sets_register | p::odexed_instance_quick },
    /* 0xf3 */ { opcode::op_iget_wide_quick,            "+iget-wide-quick",            f::fmt_22cs, 2, p::can_throw | p::odex_only | p::can_continue | p::sets_register | p::sets_wide_register | p::odexed_instance_quick },
    /* 0xf4 */ { opcode::op_iget_object_quick,          "+iget-object-quick",          f::fmt_22cs, 2, p::can_throw | p::odex_only | p::can_continue | p::sets_register | p::odexed_instance_quick },
    /* 0xf5 */ { opcode::op_iput_quick,                 "+iput-quick",                 f::fmt_22cs, 2, p::can_throw | p::odex_only | p::can_continue | p::odexed_instance_quick },
    /* 0xf6 */ { opcode::op_iput_wide_quick,            "+iput-wide-quick",            f::fmt_22cs, 2, p::can_throw | p::odex_only | p::can_continue | p::reads_wide_register | p::odexed_instance_quick },
    /* 0xf7 */ { opcode::op_iput_object_quick,          "+iput-object-quick",          f::fmt_22cs, 2, p::can_throw | p::odex_only | p::can_continue | p::odexed_instance_quick },
    /* 0xf8 */ { opcode::op_invoke_virtual_quick,       "+invoke-virtual-quick",       f::fmt_35ms, 3, p::can_throw | p::odex_only | p::can_continue | p::sets_result | p::can_invoke },
    /* 0xf9 */ { opcode::op_invoke_virtual_quick_range, "+invoke-virtual-quick/range", f::fmt_3rms, 3, p::can_throw | p::odex_only | p::can_continue | p::sets_result | p::can_invoke },
    /* 0xfa */ { opcode::op_invoke_super_quick,         "+invoke-super-quick",         f::fmt_35ms, 3, p::can_throw | p::odex_only | p::can_continue | p::sets_result | p::can_invoke },
    /* 0xfb */ { opcode::op_invoke_super_quick_range,   "+invoke-super-quick/range",   f::fmt_3rms, 3, p::can_throw | p::odex_only | p::can_continue | p::sets_result | p::can_invoke },
    /* 0xfc */ { opcode::op_iput_object_volatile,       "+iput-object-volatile",       f::fmt_22c , 2, p::can_throw | p::odex_only | p::can_continue | p::odexed_instance_volatile },
    /* 0xfd */ { opcode::op_sget_object_volatile,       "+sget-object-volatile",       f::fmt_21c , 2, p::can_throw | p::odex_only | p::can_continue | p::sets_register | p::odexed_static_volatile },
    /* 0xfe */ { opcode::op_sput_object_volatile,       "+sput-object-volatile",       f::fmt_21c , 2, p::can_throw | p::odex_only | p::can_continue | p::odexed_static_volatile },
    /* 0xff */ { opcode::op_ff,                         "(unused)",                    f::fmt_10x , 1, p::can_continue },
};

// clang-format on

const insn_info& jitana::info(opcode op)
{
    return insn_info_tbl[size_t(op)];
}
