﻿package playerio{
	/**
	 * Provides access to the BigDB service provided by Player.IO.
	 * 
	 */
	public interface BigDB{
		
		/**
		 * Load the DatabaseObject with the given key from the given table. 
		 * @param table The table to load the DatabaseObject from
		 * @param key The key of the DatabaseObject to load
		 * @param callback Function executed when the DatabaseObject is successfully loaded: function(o:DatabaseObject):void{...}
		 * @param errorHandler Function executed if an error occurs while loading the DatabaseObject.
		 * 
		 */
		function load(table:String, key:String, callback:Function=null, errorHandler:Function=null):void;
		/**
		 * Loads the DatabaseObjects with the given keys from the given table.
		 * @param table The table to load the DatabaseObject from
		 * @param keys Array of keys to load from the given table
		 * @param callback Function executed when the DatabaseObjects are successfully loaded. An array of DatabaseObjects are passed to the method: function(dbarr:Array):void{...}
		 * @param errorHandler Function executed if an error occurs while loading the DatabaseObjects
		 */
		function loadKeys(table:String, keys:Array, callback:Function=null, errorHandler:Function=null):void;
		/**
		 * Load a DatabaseObject from a table using the specified index.
		 * @param table The table to load the DatabaseObject from
		 * @param index The name of the index to query for the DatabaseObject
		 * @param indexValue An array of objects of the same types as the index properties, specifying which object to load
		 * @param callback Function executed when the DatabaseObject is successfully loaded: function(o:DatabaseObject):void{...}
		 * @param errorHandler Function executed if an error occurs while loading the DatabaseObject
		 * 
		 */
		function loadSingle(table:String, index:String, indexValue:Array, callback:Function=null, errorHandler:Function=null):void;
		/**
		 * Creates a new DatabaseObject in the given table with the specified key. If no key is specified, the object will receive an autogenerated id.
		 * @param table The table to create the DatabaseObject in.
		 * @param key The key of the newly created DatabaseObject
		 * @param data Initial data of the DatabaseObject passed as an object.
		 * @param callback Function executed when the DatabaseObject has successfully been created. The newly created DatabaseObject is passed to the callback handler: function(o:DatabaseObject):void{...}
		 * @param errorHandler Function executed if an error occurs while creating the DatabaseObject
		 * 
		 */
		function createObject(table:String, key:String, data:Object, callback:Function=null, errorHandler:Function=null):void;
		/**
		 * Loads the DatabaseObject corresponding to the player from the PlayerObjects table
		 * @param callback Function executed when the object has loaded successfully: function(o:DatabaseObject):void{...}
		 * @param errorHandler Function executed if an error occurs while loading the DatabaseObject
		 * @example The following example loads the current players DatabaseObject
		 * <listing version="3.0">
		 * client.bigDB.loadMyPlayerObject(function(myDBObject:DatabaseObject):void{
		 * 	trace("Loaded", myDBObject)
		 * })
		 * </listing>
		 */
		function loadMyPlayerObject(callback:Function=null, errorHandler:Function=null):void;
		/**
		 * Load a range of DatabaseObjects from a table using the specified index.
		 * @param table The table to load the DatabaseObject from
		 * @param index The name of the index to query for the DatabaseObject
		 * @param path Where in the index to start the range search: An array of objects of the same types as the index properties, specifying where in the index to start loading DatabaseObjects from. For instance, in the index [Mode,Map,Score] you might use ["expert","skyland"] as the indexPath and use the start and stop arguments to determine the range of scores you wish to return. IndexPath can be set to null if there is only one property in the index.
		 * @param start Where to start the range search. For instance, if the index is [Mode,Map,Score] and indexPath is ["expert","skyland"], then start defines the minimum score to include in the results
		 * @param stop Where to stop the range search. For instance, if the index is [Mode,Map,Score] and indexPath is ["expert","skyland"], then stop defines the maximum score to include in the results
		 * @param limit The max amount of objects to return
		 * @param callback Function executed when the DatabaseObjects are successfully loaded: An array of DatabaseObjects are passed to the method: function(dbarr:Array):void{...}
		 * @param errorHandler Function executed if an error occurs while loading the DatabaseObjects
		 * 
		 */
		function loadRange(table:String, index:String, path:Array, start:Object, stop:Object,  limit:int, callback:Function=null, errorHandler:Function=null):void;
		/**
		 * Delete a set of DatabaseObjects from a table
		 * @param table The table to delete the DatabaseObjects from
		 * @param keys The keys of the DatabaseObjects to delete
		 * @param callback Function executed when the DatabaseObjects are successfully deleted. No arguments are passed to the the callback methoh.
		 * @param errorHandler Function executed if an error occurs while deleting the DatabaseObjects
		 * 
		 */
		function deleteKeys(table:String, keys:Array, callback:Function=null, errorHandler:Function=null):void;
		/**
		 * Delete a range of DatabaseObjects from a table using an index
		 * @param table The table to delete the DatabaseObject from
		 * @param index The name of the index to query for the DatabaseObjects to delete
		 * @param path Where in the index to start the range delete: An array of objects of the same types as the index properties, specifying where in the index to start loading DatabaseObjects from. For instance, in the index [Mode,Map,Score] you might use new ["expert","skyland"] as the indexPath and use the start and stop arguments to determine the range of scores you wish to delete. IndexPath can be set to null instead of an empty array.
		 * @param start Where to start the range delete. For instance, if the index is [Mode,Map,Score] and indexPath is ["expert","skyland"], then start defines the minimum score to delete
		 * @param stop Where to stop the range delete. For instance, if the index is [Mode,Map,Score] and indexPath is ["expert","skyland"], then stop defines the maximum score to delete
		 * @param callback Function executed when the DatabaseObjects are successfully deleted. No arguments are passed to the the callback methoh.
		 * @param errorHandler Function executed if an error occurs while deleting the DatabaseObjects
		 * 
		 */
		function deleteRange(table:String,index:String, path:Array, start:Object, stop:Object, callback:Function=null, errorHandler:Function=null):void;
		/**
		 * Loads or creates the DatabaseObject with the given key from the given table. 
		 * @param table The table from which to load or create the DatabaseObject
		 * @param key The key of the DatabaseObject to load or create
		 * @param callback Function executed when the DatabaseObject is successfully loaded or created: function(o:DatabaseObject):void{...}
		 * @param errorHandler Function executed if an error occurs while loading or creating the DatabaseObject
		 * 
		 */
		function loadOrCreate(table:String, key:String, callback:Function=null, errorHandler:Function=null):void;
		/**
		 * Loads or creates DatabaseObjects with the given keys from the given table. New objects are created if there are no existing objects for the given keys.
		 * @param table The table to load the DatabaseObjects from
		 * @param keys They keys of the DatabaseObjects to load
		 * @param callback Function executed when the DatabaseObjects are successfully loaded or created: function(dbarr:Array):void{...}
		 * @param errorHandler Function executed if an error occurs while loading or creating the DatabaseObjects
		 * 
		 */
		function loadKeysOrCreate(table:String, keys:Array, callback:Function=null, errorHandler:Function=null):void;
		
	}
}

