/*****************************************************************************
MIT License

Copyright (c) 2020 Yahia Farghaly Ashour

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
SOFTWARE.
******************************************************************************/

/*
 * Author   : Yahia Farghaly Ashour
 *
 * Purpose  : Contains the implementation of various of OS_Task*() APIs.
 *
 * Language:  C
 * 
 * Set 1 tab = 4 spaces for better comments readability.
 */

/*
*******************************************************************************
*                               Includes Files                                *
*******************************************************************************
*/
#include "pretty_os.h"
#include "pretty_shared.h"

/*
*******************************************************************************
*                               static variables                              *
*******************************************************************************
*/

/* Array of TCBs, Each TCB Containing the task internal data.*/
static OS_TASK_TCB OS_TblTask[OS_CONFIG_TASK_COUNT];
static OS_TASK_TCB* volatile pTCBFreeList;


/*
*******************************************************************************
*                               static functions                              *
*******************************************************************************
*/

static OS_TASK_TCB*
OS_TCB_allocate (void)
{
	OS_TASK_TCB* ptcb;
	ptcb = pTCBFreeList;
	if(pTCBFreeList != OS_NULL(OS_TASK_TCB))
	{
		pTCBFreeList = pTCBFreeList->OSTCB_NextPtr;
	}
	return (ptcb);
}

static void
OS_TCB_free (OS_TASK_TCB* ptcb)
{
	ptcb->OSTCB_NextPtr 		= pTCBFreeList;
	pTCBFreeList 					= ptcb;
}


/*
*******************************************************************************
*                             external functions                              *
*******************************************************************************
*/

/*
 * Function:  OS_Event_FreeListInit
 * --------------------
 * Initialize the memory pool of TCB entries to their default values
 *
 * Arguments    : None.
 *
 * Returns      : None.
 *
 * Notes        :   1) This function for internal use.
 */
extern void
OS_TCB_ListInit (void)
{
    CPU_t32U idx;

    for(idx = 0; idx < OS_CONFIG_TASK_COUNT - 1; ++idx)
    {
        OS_TblTask[idx].TASK_Stat   = OS_TASK_STAT_DELETED;

#if(OS_CONFIG_EDF_EN == OS_CONFIG_DISABLE)
        OS_TblTask[idx].TASK_Ticks  = 0U;
#endif

#if (OS_AUTO_CONFIG_INCLUDE_EVENTS == OS_CONFIG_ENABLE)

        OS_TblTask[idx].TASK_Event    = OS_NULL(OS_EVENT);

#endif

        OS_TblTask[idx].OSTCB_NextPtr = &OS_TblTask[idx + 1];
        OS_tblTCBPrio[idx]          = OS_NULL(OS_TASK_TCB);
    }

    OS_TblTask[OS_CONFIG_TASK_COUNT - 1].TASK_Stat   	= OS_TASK_STAT_DELETED;

#if(OS_CONFIG_EDF_EN == OS_CONFIG_DISABLE)
    OS_TblTask[OS_CONFIG_TASK_COUNT - 1].TASK_Ticks  	= 0U;
#endif

#if (OS_AUTO_CONFIG_INCLUDE_EVENTS == OS_CONFIG_ENABLE)

    OS_TblTask[OS_CONFIG_TASK_COUNT - 1].TASK_Event    	= OS_NULL(OS_EVENT);

#endif

    OS_TblTask[OS_CONFIG_TASK_COUNT - 1].OSTCB_NextPtr 	= &OS_TblTask[idx + 1];

    OS_tblTCBPrio[OS_CONFIG_TASK_COUNT - 1]          	= OS_NULL(OS_TASK_TCB);

    pTCBFreeList = &OS_TblTask[0];						/* Point to the first free TCB's task object.	*/
}

/* ****************************************************************************
 *																			  *
 * 			Tasks APIs For Earliest Deadline First Scheduling based.		  *
 *																			  *
 * ****************************************************************************
 * */

#if(OS_CONFIG_EDF_EN == OS_CONFIG_ENABLE)

void
OS_TaskCreate (void (*TASK_Handler)(void* params),
                             void *params,
                             CPU_tSTK* pStackBase,
                             CPU_tSTK_SIZE  stackSize,
                             OS_OPT task_type, OS_TICK task_relative_deadline, OS_TICK task_period )
{

	CPU_tWORD* stack_top;
	OS_TASK_TCB* ptcb;

	CPU_SR_ALLOC();

	if(TASK_Handler == OS_NULL(void) || pStackBase == OS_NULL(CPU_tWORD) ||
			stackSize == 0U )
	{
		OS_ERR_SET(OS_ERR_PARAM);
		return;
	}

	if(task_type != OS_TASK_PERIODIC)	/* Current Implementation for the periodic tasks.			*/
	{
		OS_ERR_SET(OS_ERR_PARAM);
		return;
	}

	OS_CRTICAL_BEGIN();

	if(OS_IntNestingLvl > 0U)
	{
		OS_CRTICAL_END();
		OS_ERR_SET(OS_ERR_TASK_CREATE_ISR);
		return;
	}

	stack_top = OS_CPU_TaskStackInit(TASK_Handler, params, pStackBase, stackSize);

	ptcb = OS_TCB_allocate();

	if(ptcb == OS_NULL(OS_TASK_TCB) || OS_SystemTasksCount > OS_CONFIG_TASK_COUNT)
	{
		OS_CRTICAL_END();
		OS_ERR_SET(OS_ERR_TASK_POOL_EMPTY);
		return;
	}

	ptcb->TASK_SP       = stack_top;

#if (OS_CONFIG_CPU_SOFT_STK_OVERFLOW_DETECTION == OS_CONFIG_ENABLE)
	ptcb->TASK_SP_Limit = (void*)pStackBase;
#endif

	ptcb->TASK_Stat     = OS_TASK_STAT_READY;

#if (OS_AUTO_CONFIG_INCLUDE_EVENTS == OS_CONFIG_ENABLE)

	ptcb->TASK_PendStat = OS_STAT_PEND_OK;
	ptcb->OSTCB_NextPtr = OS_NULL(OS_TASK_TCB);
	ptcb->TASK_Event    = OS_NULL(OS_EVENT);

#endif

#if (OS_CONFIG_TCB_TASK_ENTRY_STORE_EN == OS_CONFIG_ENABLE)
	ptcb->TASK_EntryAddr = TASK_Handler;
	ptcb->TASK_EntryArg  = params;
#endif

	/* Fill The EDF Parameters in the TCB task.																					 */
	ptcb->EDF_params.tick_relative_deadline = task_relative_deadline;
	ptcb->EDF_params.task_period			= task_period;
	ptcb->EDF_params.task_type				= task_type;
	ptcb->EDF_params.task_yield				= OS_FAlSE;

	if(task_type == OS_TASK_PERIODIC)
	{
		ptcb->EDF_params.tick_absolute_deadline = OS_TickTime + task_relative_deadline;
	}

	/* Insert into the Ready List with relative deadlines.																		 */
	OS_TCBList[OS_SystemTasksCount].itemVal = task_relative_deadline;
	/* Link the List Item object to the allocated TCB.																			 */
	OS_TCBList[OS_SystemTasksCount].pOwner  = (void*)ptcb;
	/* Link TCB to its List Item.																								 */
	ptcb->pListItemOwner = &OS_TCBList[OS_SystemTasksCount];

    OS_tblTCBPrio[OS_SystemTasksCount] = ptcb;			/* Link to the allocated TCB. 											 */

	if(OS_SystemTasksCount != OS_IDLE_TASK_PRIO_LEVEL)
	{
	/* No Need to insert the Idle Task in the Ready List, It will be called when necessary.  								 	 */
		listItemInsert(&OS_ReadyList,&OS_TCBList[OS_SystemTasksCount]);
	}
	/* Increment The Number of created tasks.																					 */
	OS_SystemTasksCount++;

#if(OS_CONFIG_CPU_TASK_CREATED == OS_CONFIG_ENABLE)
		OS_CPU_Hook_TaskCreated (ptcb);
#endif

#if (OS_CONFIG_APP_TASK_CREATED == OS_CONFIG_ENABLE)
		App_Hook_TaskCreated (ptcb);
#endif

	if(OS_TRUE == OS_Running)
	{
	/* Schedule whose deadline is nearest.   																				     */
		OS_Sched();
	}

	OS_CRTICAL_END();

	OS_ERR_SET(OS_ERR_NONE);
}

/*
 * Function:  OS_TaskYield
 * --------------------
 * Give up the current task execution from the CPU & schedule another task.
 *
 * Arguments    :   None.
 *
 * Returns      :   None.
 *
 * Note(s)		:	1) This Function should be used @ the end of task execution.
 */
void
OS_TaskYield (void)
{
	CPU_SR_ALLOC();

	OS_CRTICAL_BEGIN();

	if(OS_currentTask != (OS_TASK_TCB*)OS_TCBList[0].pOwner)
	{
		if(OS_currentTask->EDF_params.task_type == OS_TASK_PERIODIC)
		{
			/* For a periodic task ...															*/
			/* ... The next call should be in its next period.									*/
			OS_currentTask->EDF_params.tick_arrive += OS_currentTask->EDF_params.task_period;
			/* ... The new absolute deadline. 													*/
			OS_currentTask->EDF_params.tick_absolute_deadline = OS_currentTask->EDF_params.tick_arrive + OS_currentTask->EDF_params.tick_relative_deadline;
			/* ... Indicate the the current task wants to yield (give up) the CPU resources.	*/
			OS_currentTask->EDF_params.task_yield	= OS_TRUE;
			/* Insert the current task in an inactive state till the next arrive time.			*/
			OS_currentTask->pListItemOwner->itemVal = OS_currentTask->EDF_params.tick_arrive;
			listItemInsert(&OS_InactiveList,OS_currentTask->pListItemOwner);
		}
	}

    OS_CRTICAL_END();

	OS_Sched();		/* Schedule the next least deadline time of a task.							*/
}

#else	/* OS_CONFIG_EDF_EN == OS_CONFIG_DISABLE 							  */

/* ****************************************************************************
 *																			  *
 * 				Tasks APIs For Priority Assignment Scheduling based.		  *
 *																			  *
 * ****************************************************************************
 * */

/*
 * Function:  OS_TaskCreate
 * --------------------
 * Normal Task Creation.
 *
 *
 * Arguments    :   TASK_Handler            is a function pointer to the task code.
 *                  params                  is a pointer to the user supplied data which is passed to the task.
 *                  pStackBase              is a pointer to the bottom of the task stack.
 *                  stackSize               is the task stack size.
 *                  priority                is the task priority. ( A unique priority must be assigned to each task )
 *                                              - A greater number means a higher priority
 *                                              - 0 => is reserved for the OS'Idle Task.
 *                                              - 1 => is reserved for OS use.
 *                                              - OS_LOWEST_PRIO_LEVEL(0) < Allowed value <= OS_HIGHEST_PRIO_LEVEL
 *
 * Returns      :   OS_RET_OK, OS_ERR_PARAM, OS_RET_ERROR_TASK_CREATE_ISR
 */
OS_tRet
OS_TaskCreate (void (*TASK_Handler)(void* params),
                             void *params,
                             CPU_tSTK* pStackBase,
                             CPU_tSTK_SIZE  stackSize,
                             OS_PRIO priority)

{
    CPU_tWORD* stack_top;
    CPU_SR_ALLOC();

    if(TASK_Handler == OS_NULL(void) || pStackBase == OS_NULL(CPU_tWORD) ||
            stackSize == 0U )
    {
    	OS_ERR_SET(OS_ERR_PARAM);
        return (OS_ERR_PARAM);
    }

    OS_CRTICAL_BEGIN();

    if(OS_IntNestingLvl > 0U)                                                     /* Don't Create a task from an ISR.                                                        */
    {
        OS_CRTICAL_END();
    	OS_ERR_SET(OS_ERR_TASK_CREATE_ISR);
        return (OS_ERR_TASK_CREATE_ISR);
    }

    stack_top = OS_CPU_TaskStackInit(TASK_Handler, params, pStackBase, stackSize);     /* Call low level function to Initialize the stack frame of the task.                      */

    if(OS_IS_VALID_PRIO(priority))
    {

        if(OS_tblTCBPrio[priority] != OS_NULL(OS_TASK_TCB))            				 	/* Check that the task is not in use.                                                      */
        {
            OS_CRTICAL_END();
            OS_ERR_SET(OS_ERR_TASK_CREATE_EXIST);
            return (OS_ERR_TASK_CREATE_EXIST);
        }

        OS_tblTCBPrio[priority] = OS_TCB_allocate();

        OS_tblTCBPrio[priority]->TASK_SP       = stack_top;

#if (OS_CONFIG_CPU_SOFT_STK_OVERFLOW_DETECTION == OS_CONFIG_ENABLE)
        OS_tblTCBPrio[priority]->TASK_SP_Limit = (void*)pStackBase;
#endif
        OS_tblTCBPrio[priority]->TASK_priority = priority;
        OS_tblTCBPrio[priority]->TASK_Stat     = OS_TASK_STAT_READY;

#if (OS_AUTO_CONFIG_INCLUDE_EVENTS == OS_CONFIG_ENABLE)

        OS_tblTCBPrio[priority]->TASK_PendStat = OS_STAT_PEND_OK;
        OS_tblTCBPrio[priority]->OSTCB_NextPtr = OS_NULL(OS_TASK_TCB);
        OS_tblTCBPrio[priority]->TASK_Event    = OS_NULL(OS_EVENT);

#endif

#if (OS_CONFIG_TCB_TASK_ENTRY_STORE_EN == OS_CONFIG_ENABLE)
        OS_tblTCBPrio[priority]->TASK_EntryAddr = TASK_Handler;
        OS_tblTCBPrio[priority]->TASK_EntryArg  = params;
#endif

#if(OS_CONFIG_CPU_TASK_CREATED == OS_CONFIG_ENABLE)
        OS_CPU_Hook_TaskCreated (OS_tblTCBPrio[priority]);						 /* Call port specific task creation code.													  */
#endif

#if (OS_CONFIG_APP_TASK_CREATED == OS_CONFIG_ENABLE)
        App_Hook_TaskCreated (OS_tblTCBPrio[priority]);							 /* Calls Application specific code for a successfully created task.						  */
#endif

        OS_SetReady(priority);                                                   /* Put in ready state.                                                                       */
    }
    else
    {
        OS_CRTICAL_END();
        OS_ERR_SET(OS_ERR_PRIO_INVALID);
        return (OS_ERR_PRIO_INVALID);
    }

    if(OS_TRUE == OS_Running)
    {
        OS_Sched();                                                             /* A higher priority task can be created inside another task. So, Schedule it immediately.    */
    }

    OS_CRTICAL_END();

    OS_ERR_SET(OS_ERR_NONE);
    return (OS_ERR_NONE);
}

/*
 * Function:  OS_TaskDelete
 * -------------------------
 * Delete a task given its priority. It can delete the calling task itself.
 * The deleted task is moved to a dormant state and can be re-activated again by creating the deleted task.
 *
 * Arguments    :   prio    is the task priority.
 *
 * Returns      :   OS_RET_OK, OS_ERR_TASK_DELETE_ISR, OS_ERR_TASK_DELETE_IDLE, OS_ERR_PRIO_INVALID, OS_ERR_TASK_NOT_EXIST.
 */
OS_tRet
OS_TaskDelete (OS_PRIO prio)
{
    OS_TASK_TCB* ptcb;
    CPU_SR_ALLOC();

    if(OS_IntNestingLvl > 0U)                                                      /* Don't delete from an ISR.                 */
    {
    	OS_ERR_SET(OS_ERR_TASK_DELETE_ISR);
        return (OS_ERR_TASK_DELETE_ISR);
    }
    if(prio == OS_IDLE_TASK_PRIO_LEVEL)                                            /* Don't delete the Idle task.               */
    {
    	OS_ERR_SET(OS_ERR_TASK_DELETE_IDLE);
        return (OS_ERR_TASK_DELETE_IDLE);
    }
    if(!OS_IS_VALID_PRIO(prio))                                                    /* Valid priority ?                          */
    {
    	OS_ERR_SET(OS_ERR_PRIO_INVALID);
        return (OS_ERR_PRIO_INVALID);
    }

    OS_CRTICAL_BEGIN();

    ptcb = OS_tblTCBPrio[prio];

    if(ptcb == OS_NULL(OS_TASK_TCB) || ptcb->TASK_Stat == OS_TASK_STAT_DELETED)   /* Task must exist.                           */
    {
        OS_CRTICAL_END();
        OS_ERR_SET(OS_ERR_TASK_NOT_EXIST);
        return (OS_ERR_TASK_NOT_EXIST);
    }

    OS_RemoveReady(prio);                                                         /* Remove the task from ready state.          */

#if (OS_AUTO_CONFIG_INCLUDE_EVENTS == OS_CONFIG_ENABLE)

    if(ptcb->TASK_Event != ((OS_EVENT*)0U))                                       /* If it is waiting for any event...          */
    {
        OS_Event_TaskRemove(ptcb, ptcb->TASK_Event);                              /* ... unlink it.                             */
    }

#endif

    if(ptcb->TASK_Stat & OS_TASK_STAT_DELAY)                                      /* If it's waiting due to a delay             */
    {
        OS_UnBlockTime(prio);                                                     /* Remove from the time blocked state.        */
    }

    ptcb->TASK_Ticks    = 0U;                                                     /* Remove any remaining ticks.                */

#if (OS_AUTO_CONFIG_INCLUDE_EVENTS == OS_CONFIG_ENABLE)

    ptcb->TASK_PendStat = OS_STAT_PEND_OK;                                        /* Remove any pend state.                     */

#endif

    ptcb->TASK_Stat     = OS_TASK_STAT_DELETED;                                   /* Make the task be Dormant.                  */

    OS_tblTCBPrio[prio] = OS_NULL(OS_TASK_TCB);                                   /* The task is no longer exist. 				*/

    /* At this point, the task is prevented from resuming or made ready from another higher task or an ISR.                     */
    if(OS_TRUE == OS_Running)
    {
        OS_Sched();                                                               /* Schedule a new higher priority task.       */
    }

#if(OS_CONFIG_CPU_TASK_DELETED == OS_CONFIG_ENABLE)
    OS_CPU_Hook_TaskDeleted (ptcb);												  /* Call port specific task deletion code.		*/
#endif

#if (OS_CONFIG_APP_TASK_DELETED == OS_CONFIG_ENABLE)
	App_Hook_TaskDeleted 	(ptcb);												  /* Calls Application specific code.			*/
#endif

    OS_TCB_free(ptcb);															  /* Return the TCB object to the Tasks pool.	*/

    OS_CRTICAL_END();

    OS_ERR_SET(OS_ERR_NONE);
    return (OS_ERR_NONE);
}

/*
 * Function:  OS_TaskChangePriority
 * --------------------
 * Change the priority of a task dynamically.
 *
 * Arguments    :   oldPrio     is the old priority
 *                  newPrio     is the new priority
 *
 * Returns      :   OS_RET_OK, OS_ERR_PRIO_INVALID, OS_ERR_PRIO_EXIST, OS_ERR_TASK_NOT_EXIST
 */
OS_tRet
OS_TaskChangePriority (OS_PRIO oldPrio, OS_PRIO newPrio)
{
    OS_TASK_TCB* ptcb;
    OS_EVENT*    pevent;
    CPU_SR_ALLOC();

    if(oldPrio == newPrio)                                                    /* Don't waste more cycles.                     */
    {
    	OS_ERR_SET(OS_ERR_PRIO_EXIST);
        return (OS_ERR_PRIO_EXIST);
    }

    if(OS_IS_RESERVED_PRIO(oldPrio))                                          /* Don't Change an OS reserved priority.        */
    {
    	OS_ERR_SET(OS_ERR_PRIO_EXIST);
        return (OS_ERR_PRIO_EXIST);
    }

    if(OS_IS_RESERVED_PRIO(newPrio))                                         /* Don't Change to an OS reserved priority.      */
    {
    	OS_ERR_SET(OS_ERR_PRIO_EXIST);
        return (OS_ERR_PRIO_EXIST);
    }

    if(!OS_IS_VALID_PRIO(oldPrio) && !OS_IS_VALID_PRIO(newPrio))             /* Priority within our acceptable range.         */
    {
    	OS_ERR_SET(OS_ERR_PRIO_INVALID);
        return (OS_ERR_PRIO_INVALID);
    }

    OS_CRTICAL_BEGIN();

    if(OS_tblTCBPrio[oldPrio] == OS_NULL(OS_TASK_TCB))                       /* Check that the old task is Created.          */
    {
        OS_CRTICAL_END();
        OS_ERR_SET(OS_ERR_TASK_NOT_EXIST);
        return OS_ERR_TASK_NOT_EXIST;
    }

    if(OS_tblTCBPrio[oldPrio] == OS_TCB_MUTEX_RESERVED)
    {
        OS_CRTICAL_END();                                                  /* old prio should not be reserved for a mutex.   */
        OS_ERR_SET(OS_ERR_TASK_NOT_EXIST);
        return OS_ERR_TASK_NOT_EXIST;
    }

    if(OS_tblTCBPrio[oldPrio]->TASK_Stat == OS_TASK_STAT_DELETED)          /* Be dummy in the checks !                       */
    {
        OS_CRTICAL_END();
        OS_ERR_SET(OS_ERR_TASK_NOT_EXIST);
        return OS_ERR_TASK_NOT_EXIST;
    }

    if(OS_tblTCBPrio[newPrio] != OS_NULL(OS_TASK_TCB))                      /* The new priority must be available.           */
    {
        OS_CRTICAL_END();
        OS_ERR_SET(OS_ERR_TASK_CREATE_EXIST);
        return OS_ERR_TASK_CREATE_EXIST;
    }

    if(OS_tblTCBPrio[newPrio] == OS_TCB_MUTEX_RESERVED)
    {
        OS_CRTICAL_END();                                                  /* new prio should not be reserved for a mutex.   */
        OS_ERR_SET(OS_ERR_TASK_NOT_EXIST);
        return OS_ERR_TASK_NOT_EXIST;
    }

    ptcb   = OS_tblTCBPrio[oldPrio];                                       /* Store a pointer to TCB entry at old priority.  */

#if (OS_AUTO_CONFIG_INCLUDE_EVENTS == OS_CONFIG_ENABLE)

    pevent = ptcb->TASK_Event;

#endif

    if(ptcb->TASK_Stat == OS_TASK_STAT_READY)
    {
        OS_RemoveReady(oldPrio);
        OS_SetReady(newPrio);
    }
    else
    {
        if(ptcb->TASK_Stat & OS_TASK_STAT_DELAY)
        {
            OS_UnBlockTime(oldPrio);
            OS_BlockTime(newPrio);
        }

#if (OS_AUTO_CONFIG_INCLUDE_EVENTS == OS_CONFIG_ENABLE)

        if(ptcb->TASK_Event != OS_NULL(OS_EVENT))                          /* If old priority is waiting for an event.        */
        {
            OS_Event_TaskRemove(ptcb, pevent);                             /* ... Remove at the old priority.                 */

            ptcb->TASK_priority    = newPrio;
            OS_Event_TaskInsert(OS_tblTCBPrio[newPrio], pevent);           /* ... Place event at the new priority.            */
        }

#endif

    }

    ptcb->TASK_priority    = newPrio;                                      /* Store new priority in TCB entry.                */
    OS_tblTCBPrio[oldPrio] = OS_NULL(OS_TASK_TCB);                         /* Unlink old priority pointer to TCB entry...     */
    OS_tblTCBPrio[newPrio] = ptcb;                                         /* ... Link to the new priority.                   */

    OS_CRTICAL_END();

    if(OS_TRUE == OS_Running)
    {
        OS_Sched();                                                      /* Call the scheduler, it may be a higher priority task.   */
    }

    OS_ERR_SET(OS_ERR_NONE);
    return (OS_ERR_NONE);
}

/*
 * Function:  OS_TaskSuspend
 * -------------------------
 * Suspend a task given its priority.
 * This function can suspend the calling task itself.
 *
 * Arguments    :   prio    is the task priority.
 *
 * Returns      :   OS_RET_OK, OS_RET_TASK_SUSPENDED, OS_ERR_TASK_SUSPEND_IDEL, OS_ERR_PRIO_INVALID, OS_ERR_TASK_SUSPEND_PRIO
 */
OS_tRet
OS_TaskSuspend (OS_PRIO prio)
{
    CPU_tWORD       selfTask;
    OS_TASK_TCB*    thisTask;
    CPU_SR_ALLOC();

    if(OS_IDLE_TASK_PRIO_LEVEL == prio)                     /* Don't suspend idle task                                                 */
    {
    	OS_ERR_SET(OS_ERR_TASK_SUSPEND_IDLE);
        return (OS_ERR_TASK_SUSPEND_IDLE);
    }

    if(OS_IS_VALID_PRIO(prio))
    {
        OS_CRTICAL_BEGIN();

        if(prio == OS_currentTask->TASK_priority)           /* Is the caller task will be the suspended ?                              */
        {
            selfTask = OS_TRUE;
        }
        else
        {
            selfTask = OS_FAlSE;
        }

        thisTask = OS_tblTCBPrio[prio];

        if(thisTask == OS_NULL(OS_TASK_TCB) ||
          thisTask->TASK_Stat == OS_TASK_STAT_DELETED)     /* Check that the suspended task is actually exist.                         */
        {
            OS_CRTICAL_END();
            OS_ERR_SET(OS_ERR_TASK_SUSPEND_PRIO);
            return (OS_ERR_TASK_SUSPEND_PRIO);
        }

        if(thisTask->TASK_Stat & OS_TASK_STAT_SUSPENDED)   /* If it's in a suspend state, why do extra work !                           */
        {
            OS_CRTICAL_END();
            OS_ERR_SET(OS_ERR_TASK_SUSPENDED);
            return (OS_ERR_TASK_SUSPENDED);
        }

        thisTask->TASK_Stat |= OS_TASK_STAT_SUSPENDED;

        OS_RemoveReady(prio);

        OS_CRTICAL_END();

        if(selfTask == OS_TRUE)                            /* Calls the scheduler only if the task being suspended is the calling task. */
        {
            OS_Sched();
        }

        OS_ERR_SET(OS_ERR_NONE);
        return OS_ERR_NONE;
    }

    OS_ERR_SET(OS_ERR_PRIO_INVALID);
    return (OS_ERR_PRIO_INVALID);
}

/*
 * Function:  OS_TaskResume
 * ------------------------
 * Resume a suspended task given its priority.
 *
 * Arguments    :   prio  is the task priority.
 *
 * Returns      :   OS_RET_OK, OS_ERR_TASK_RESUME_PRIO, OS_ERR_PRIO_INVALID.
 */
OS_tRet
OS_TaskResume (OS_PRIO prio)
{
    OS_TASK_TCB*    thisTask;
    CPU_SR_ALLOC();

    if(OS_IDLE_TASK_PRIO_LEVEL == prio)                                             /* Resume an suspended task !                                                 */
    {
    	OS_ERR_SET(OS_ERR_PRIO_INVALID);
        return (OS_ERR_PRIO_INVALID);
    }

    if(OS_IS_VALID_PRIO(prio))
    {
        OS_CRTICAL_BEGIN();

        if(prio == OS_currentTask->TASK_priority)                                   /* Resume self !                                                              */
        {
            OS_CRTICAL_END();
            OS_ERR_SET(OS_ERR_TASK_RESUME_PRIO);
            return (OS_ERR_TASK_RESUME_PRIO);
        }

        thisTask = OS_tblTCBPrio[prio];

        if(thisTask == OS_NULL(OS_TASK_TCB) ||
          thisTask->TASK_Stat == OS_TASK_STAT_DELETED)                              /* Check that the resumed task is actually exist.                             */
        {
            OS_CRTICAL_END();
            OS_ERR_SET(OS_ERR_TASK_RESUME_PRIO);
            return (OS_ERR_TASK_RESUME_PRIO);
        }

        if((thisTask->TASK_Stat & OS_TASK_STAT_SUSPENDED) != OS_TASK_STAT_READY)    /* Check it's already in suspend state and not in ready state.                */
        {
            thisTask->TASK_Stat &= ~(OS_TASK_STAT_SUSPENDED);                       /* Clear the suspend state.                                                   */
           if((thisTask->TASK_Stat & OS_TASK_STATE_PEND_ANY) == OS_TASK_STAT_READY) /* If it's not pending on any events ... */
           {
               if(thisTask->TASK_Ticks == 0U)                                       /* If it's not waiting a delay ...                                            */
               {
                   OS_SetReady(prio);
                   OS_CRTICAL_END();
                   if(OS_TRUE == OS_Running)
                   {
                       OS_Sched();                                                  /* Call the scheduler, it may be a higher priority task.                      */
                   }
               }
           }
        }

        OS_CRTICAL_END();
        OS_ERR_SET(OS_ERR_NONE);
        return (OS_ERR_NONE);
    }
    OS_ERR_SET(OS_ERR_PRIO_INVALID);
    return (OS_ERR_PRIO_INVALID);
}

/*
 * Function:  OS_TaskStatus
 * --------------------
 * Return Task Status.
 *
 * Arguments    :   prio  is the task priority.
 *
 * Returns      :   OS_STATUS
 */
OS_STATUS inline
OS_TaskStatus (OS_PRIO prio)
{
	if(OS_tblTCBPrio[prio] == OS_NULL(OS_TASK_TCB))
	{
		return OS_TASK_STAT_DELETED;
	}

	return OS_tblTCBPrio[prio]->TASK_Stat;
}

/*
 * Function:  OS_TaskRunningPriorityGet
 * -----------------------------------
 * Return The current running task priority.
 *
 * Arguments    :   None.
 *
 * Returns      :   The current running task priority.
 */
OS_PRIO
OS_TaskRunningPriorityGet (void)
{
	OS_PRIO running_prio;
	CPU_SR_ALLOC();

	OS_CRTICAL_BEGIN();

	running_prio = OS_currentTask->TASK_priority;

	OS_CRTICAL_END();

	return (running_prio);
}

#endif

/*
 * Function:  OS_TaskReturn
 * --------------------
 * This function should be called if a task is accidentally returns without deleting itself.
 * The address of this function should be set at the task stack register of the return address.
 *
 * Arguments    :   None.
 *
 * Returns      :   None.
 *
 * Note(s)      :   1) This function is for internal use and the application should never called it.
 * 					2) This function must be called from port layer if you want to support catching tasks return.
 */
void
OS_TaskReturn (void)
{

#if (OS_CONFIG_APP_TASK_RETURNED == OS_CONFIG_ENABLE)
	App_Hook_TaskReturned (OS_currentTask); 			/* Calls Application specific code when a task returns intentionally. 								*/
#endif

#if(OS_CONFIG_EDF_EN == OS_CONFIG_DISABLE)
    OS_TaskDelete(OS_currentTask->TASK_priority);       /* Delete a task.            																		*/
#endif
    for(;;)                                             /* If deletion fails, Loop Every OS_CONFIG_TICKS_PER_SEC											*/
    {
#if(OS_CONFIG_EDF_EN == OS_CONFIG_DISABLE)
        OS_DelayTicks(OS_CONFIG_TICKS_PER_SEC);
#else
        OS_TaskYield();									/* If we reached to this point, this means that the aperiodic task doesn't call OS_TaskYield() at
        													its epilogue. So we call it here.																*/
#endif
    }
}


