package operations

import (
	"errors"
	"fmt"
	"io"
	"os"
	"path/filepath"
	"strings"
	"text/template"
)

const DirectoryName = "operations"

func GetPaths(wunderGraphDir string) ([]string, error) {
	operationsDirectoryAbs := filepath.Join(wunderGraphDir, DirectoryName)
	var operationFilePaths []string
	err := filepath.Walk(operationsDirectoryAbs, func(path string, info os.FileInfo, err error) error {
		if info.IsDir() || strings.HasSuffix(info.Name(), ".d.ts") || !strings.HasSuffix(info.Name(), ".ts") {
			return nil
		}
		path, err = filepath.Rel(wunderGraphDir, path)
		if err != nil {
			return err
		}
		operationFilePaths = append(operationFilePaths, path)
		return nil
	})
	if err != nil {
		return nil, err
	}
	return operationFilePaths, nil
}

func Cleanup(wunderGraphDir string, paths []string) error {
	expected := make([]string, 0, len(paths)*2)
	for _, path := range paths {
		expected = append(expected,
			filepath.Join(wunderGraphDir, "generated", "bundle", strings.Replace(path, ".ts", ".js", 1)),
			filepath.Join(wunderGraphDir, "generated", "bundle", strings.Replace(path, ".ts", ".js.map", 1)),
		)
	}
	operationsBundlePath := filepath.Join(wunderGraphDir, "generated", "bundle", "operations")
	if _, err := os.Stat(operationsBundlePath); errors.Is(err, os.ErrNotExist) {
		return nil
	}
	err := filepath.Walk(operationsBundlePath, func(path string, info os.FileInfo, err error) error {
		if info == nil || info.IsDir() {
			return nil
		}
		if !contains(expected, path) {
			return os.Remove(path)
		}
		return nil
	})
	if err != nil {
		return err
	}
	return filepath.Walk(filepath.Join(wunderGraphDir, "generated", "bundle", "operations"), func(path string, info os.FileInfo, err error) error {
		if !info.IsDir() {
			return nil
		}
		// check if directory is empty
		empty, err := isDirEmpty(path)
		if err != nil {
			return err
		}
		if empty {
			return os.Remove(path)
		}
		return nil
	})
}

func isDirEmpty(name string) (bool, error) {
	f, err := os.Open(name)
	if err != nil {
		return false, err
	}
	defer f.Close()

	_, err = f.Readdirnames(1)
	if err == io.EOF {
		return true, nil
	}
	return false, err
}

func contains(paths []string, path string) bool {
	for _, p := range paths {
		if p == path {
			return true
		}
	}
	return false
}

const (
	wunderGraphFactoryTs = "wundergraph.factory.ts"
)

var (
	wunderGraphFactoryTemplate = template.Must(template.New(wunderGraphFactoryTs).Parse(
		`// Code generated by wunderctl. DO NOT EDIT.

import type { InternalOperationsClient } from "./wundergraph.internal.operations.client";
import type { ORM } from './orm'
import type { Role } from "./wundergraph.server";
import type { CustomClaims } from "./claims";
import { createOperationFactory } from "@wundergraph/sdk/operations";

{{ if .HasWunderGraphServerTs }}
import type server from '../wundergraph.server';

type ContextField = Required<Required<Required<typeof server>['context']>['request']>['create'];
export type ContextType = ContextField extends (...args: any) => any ? Awaited<ReturnType<ContextField>> : never;

{{ else }}
export type ContextType = never;
{{ end }}

import type { Queries, Mutations } from "./jsonschema";
import type { IOpenaiAgentFactory } from "@wundergraph/sdk/openai"
export type QueriesAndMutations = Queries & Mutations;
export type OpenApiAgentFactory = IOpenaiAgentFactory<QueriesAndMutations>;

export { z, AuthorizationError } from "@wundergraph/sdk/operations";
export const createOperation = createOperationFactory<Role, CustomClaims, InternalOperationsClient, ORM, OpenApiAgentFactory, ContextType>();`,
	))
)

type wunderGraphFactoryTemplateData struct {
	HasWunderGraphServerTs bool
}

func EnsureWunderGraphFactoryTS(wunderGraphDir string) error {
	st, err := os.Stat(filepath.Join(wunderGraphDir, "wundergraph.server.ts"))
	HasWunderGraphServerTs := err == nil && !st.IsDir()
	generated := filepath.Join(wunderGraphDir, "generated")
	if err := os.MkdirAll(generated, os.ModePerm); err != nil {
		return fmt.Errorf("error creating %s: %s", generated, err)
	}
	wunderGraphFactoryTsPath := filepath.Join(generated, wunderGraphFactoryTs)
	f, err := os.Create(wunderGraphFactoryTsPath)
	if err != nil {
		return fmt.Errorf("error creating %s: %s", wunderGraphFactoryTsPath, err)
	}
	defer f.Close()
	if err := wunderGraphFactoryTemplate.Execute(f, wunderGraphFactoryTemplateData{
		HasWunderGraphServerTs: HasWunderGraphServerTs,
	}); err != nil {
		return fmt.Errorf("error writing %s: %s", wunderGraphFactoryTsPath, err)
	}
	if err := f.Close(); err != nil {
		return fmt.Errorf("error writing %s: %s", wunderGraphFactoryTsPath, err)
	}
	return nil
}
