/*
 * Copyright (c) 2023 WunderGraph Inc.
 * All rights reserved.
 *
 * This file is licensed under the WunderGraph Enterprise License.
 * @see https://github.com/wundergraph/wundergraph/blob/main/LICENSE.ENTERPRISE.md
 */
package hooks

import (
	"github.com/wundergraph/wundergraph/pkg/operation"
	"github.com/wundergraph/wundergraph/pkg/wgpb"
)

type HookMatcher struct {
	OperationType operation.Type
	DataSources   []string
}

func (m *HookMatcher) containsDataSource(dataSourceID string) bool {
	for _, ds := range m.DataSources {
		if ds == dataSourceID {
			return true
		}
	}
	return false
}

// Hook contains a new-style hook that can match on arbitrary events.
// To use the hook create an Executor by calling Hook.Executor()
type Hook struct {
	ID      string
	Type    wgpb.HookType
	Matcher HookMatcher
}

type HookCheck struct {
	OperationType operation.Type
	DataSourceID  string
}

func (h *Hook) Executor() Executor {
	return &hookExecutor{
		matcher: h.Matcher,
		id:      h.ID,
	}
}

type hookExecutor struct {
	matcher HookMatcher
	id      string
}

func (e *hookExecutor) Matches(check *HookCheck) bool {
	if check.OperationType != operation.TypeInvalid && e.matcher.OperationType != operation.TypeInvalid && e.matcher.OperationType != check.OperationType {
		return false
	}
	if len(check.DataSourceID) > 0 && len(e.matcher.DataSources) > 0 && !e.matcher.containsDataSource(check.DataSourceID) {
		return false
	}
	return true
}

func (e *hookExecutor) HookID() string {
	return e.id
}

type Executor interface {
	Matches(check *HookCheck) bool
	HookID() string
}
