import crypto from 'crypto';
import path from 'path';

import execa from 'execa';
import Handlebars from 'handlebars';
import { JSONSchema7 as JSONSchema, JSONSchema7Definition } from 'json-schema';
import { capitalize } from 'lodash';

import { OperationType } from '@wundergraph/protobuf';
import { CodeGenerationConfig, GraphQLOperation, Template, TemplateOutputFile } from '@wundergraph/sdk';
import { hasInput } from '@wundergraph/sdk/internal/codegen';
import {
	configurationHash,
	operationInputTypename,
	operationResponseTypename,
} from '@wundergraph/sdk/dist/codegen/templates/typescript/helpers';
import Logger from '@wundergraph/sdk/internal/logger';

const logger = Logger.child({ plugin: 'rust-client' });

const clientPackageName = 'wundergraph-rust-client';
const clientPackageNamespace = clientPackageName.replace(/-/g, '_');

export interface RustClientTemplateConfig {
	/**
	 * Package name of the generated client
	 */
	packageName: string;
	/**
	 * Package version of the generated client
	 */
	packageVersion?: string;
}

const defaultTemplateConfig: RustClientTemplateConfig = {
	packageName: 'app',
};

const getAnyOf = (schema: JSONSchema) => {
	// Since anyOf is very unergonomic to express with the rust type system,
	// we handle both oneOf and anyOf as oneOf
	return schema.anyOf ?? schema.oneOf ?? [];
};

const rustfmt = (code: string) => {
	try {
		// check if gofmt is installed
		const formatter = execa.sync('rustfmt', ['--edition', '2021', '--emit', 'stdout'], {
			input: code,
			encoding: 'utf8',
		});
		if (formatter.exitCode === 0) {
			return formatter.stdout;
		}
	} catch (e: any) {
		// If the error is due to gofmt not being installed, we ignore it
		// on purpose. Otherwise we throw an error with both the code and
		// the error message returned by gofmt.
		if (e instanceof Error && e.message.indexOf('ENOENT') >= 0) {
			logger.warn('rustfmt is not installed. If you want to format the generated code, please install rustfmt');
		} else {
			throw new Error(`failed to format rust code: ${e}`);
		}
	}
	return code;
};

const fileHeader = (comment: string = '//') => `${comment} Code generated by wunderctl. DO NOT EDIT.\n\n`;

const capitalizeFirstChar = (name: string) => capitalize(name.substring(0, 1)) + name.substring(1);

const toCamelCase = (s: string) => {
	let out = '';
	for (const component of s.split('_')) {
		out += capitalizeFirstChar(component);
	}
	return out;
};

const toSnakeCase = (s: string) => {
	return s.replace(/[a-z][A-Z]/g, (s) => `${s[0]}_${s[1]}`).toLowerCase();
};

const clientRsTemplate = `use std::sync::Arc;

use crate::queries::Queries;
use crate::mutations::Mutations;
use crate::subscriptions::Subscriptions;
use crate::live_queries::LiveQueries;

#[derive(Clone)]
pub struct Client {
	client: Arc<${clientPackageNamespace}::Client>,
}

pub type Options = ${clientPackageNamespace}::ClientOptions;

impl Client {
	pub fn new(url: Option<${clientPackageNamespace}::Url>) -> Self {
		Client::new_with_options(Options{
			url,
			application_hash: None,
		})
	}

	pub fn new_with_options(options: Options) -> Self {
		let options = Options{
			url: options.url,
			application_hash: Some(options.application_hash.unwrap_or(String::from("{{apiHash}}"))),
		};
		Self {
			client: Arc::new(${clientPackageNamespace}::Client::new(options)),
		}
	}

	pub fn queries(&self) -> Queries {
		Queries::new(self.client.clone())
	}

	pub fn mutations(&self) -> Mutations {
		Mutations::new(self.client.clone())
	}

	pub fn subscriptions(&self) -> Subscriptions {
		Subscriptions::new(self.client.clone())
	}

	pub fn live_queries(&self) -> LiveQueries {
		LiveQueries::new(self.client.clone())
	}
}
`;

class ClientRs implements Template {
	constructor(private config: RustClientTemplateConfig) {}

	async generate(config: CodeGenerationConfig): Promise<TemplateOutputFile[]> {
		const tmpl = Handlebars.compile(clientRsTemplate);
		const content = tmpl({
			apiHash: configurationHash(config.config),
		});
		return Promise.resolve([
			{
				path: path.join('src', 'client.rs'),
				content: rustfmt(content),
				header: fileHeader(),
			},
		]);
	}
}

const libRsTemplate = `

type Result<T> = std::result::Result<T, ${clientPackageNamespace}::Error>;

pub use ${clientPackageNamespace}::{Error, Url, Stream, StreamExt};

mod client;
pub use client::{Client, Options};
mod inputs;
pub use inputs::*;
mod responses;
pub use responses::*;
mod queries;
mod mutations;
mod subscriptions;
mod live_queries;
`;

class LibRs implements Template {
	constructor(private config: RustClientTemplateConfig) {}

	async generate(_: CodeGenerationConfig): Promise<TemplateOutputFile[]> {
		return Promise.resolve([
			{
				path: path.join('src', 'lib.rs'),
				content: rustfmt(libRsTemplate),
				header: fileHeader(),
			},
		]);
	}
}

const cargoTomlTemplate = `
[package]
name = "{{packageName}}"
version = "{{packageVersion}}"
edition = "2021"

# See more keys and their definitions at https://doc.rust-lang.org/cargo/reference/manifest.html

[features]
tls = ["${clientPackageName}/tls"]

[dependencies]
serde = { version = "1.0.188", features = [ "serde_derive" ] }
serde_json = "1.0.105"
${clientPackageName} = { {{clientPackageProtocol}} = "{{clientPackageUrl}}" }
`;

class CargoToml implements Template {
	constructor(private config: RustClientTemplateConfig) {}

	async generate(_: CodeGenerationConfig): Promise<TemplateOutputFile[]> {
		const tmpl = Handlebars.compile(cargoTomlTemplate);
		let clientPackageProtocol: string | undefined;
		let clientPackageUrl: string | undefined;
		// Use this to override the Cargo.toml entry to point to a local
		// copy of the package while working on the rust-client crate
		if (process.env.RUST_PACKAGE_PATH) {
			clientPackageProtocol = 'path';
			clientPackageUrl = process.env.RUST_PACKAGE_PATH;
		} else {
			clientPackageProtocol = 'git';
			clientPackageUrl = 'https://github.com/wundergraph/rust-client.git';
		}
		const content = tmpl({
			packageName: this.config.packageName,
			packageVersion: this.config.packageVersion ?? '0.0.1',
			clientPackageProtocol,
			clientPackageUrl,
		});
		return Promise.resolve([
			{
				path: path.join('Cargo.toml'),
				content,
				header: fileHeader('#'),
			},
		]);
	}
}

const typeAnnotations = `#[derive(Clone, Debug, serde::Serialize, serde::Deserialize)]\n`;

class rustEncoder {
	private readonly types: Map<string, string>;
	private handledRefPaths: Set<string>;
	private structNameByTypeName: Map<string, string>;
	private encodedObjectTypeNames: Set<string>;
	private output: string = '';

	constructor(private rootSchema: JSONSchema, private rootTypeName: string, definedTypes?: Map<string, string>) {
		this.types = definedTypes ?? new Map();
		this.handledRefPaths = new Set<string>();
		this.structNameByTypeName = new Map<string, string>();
		this.encodedObjectTypeNames = new Set<string>();
	}

	encode() {
		let encoded = this.encodeType(this.rootSchema, this.rootTypeName);
		if (this.output == '') {
			// Output is not a struct, handle it here
			return `pub type ${this.encodeTypeName(this.rootTypeName)} = ${encoded};\n`;
		}
		return this.output;
	}

	definedTypes() {
		return new Map(this.types);
	}

	private encodeFieldName(name: string) {
		const fieldName = toSnakeCase(name.replace(/@/g, '_'));
		// From https://doc.rust-lang.org/reference/keywords.html
		const reserved = [
			// strict keywords
			'as',
			'break',
			'const',
			'continue',
			'crate',
			'else',
			'enum',
			'extern',
			'false',
			'fn',
			'for',
			'if',
			'impl',
			'in',
			'let',
			'loop',
			'match',
			'mod',
			'move',
			'mut',
			'pub',
			'ref',
			'return',
			'self',
			'Self',
			'static',
			'struct',
			'super',
			'trait',
			'true',
			'type',
			'unsafe',
			'use',
			'where',
			'while',
			// added in 2018 edition
			'async',
			'await',
			'dyn',
			// reserved
			'abstract',
			'become',
			'box',
			'do',
			'final',
			'macro',
			'override',
			'priv',
			'typeof',
			'unsized',
			'virtual',
			'yield',
			// added to reserved in 2018 edition
			'try',
			// weak keywords
			'macro_rules',
			'union',
			"'static",
		];
		if (reserved.includes(fieldName)) {
			return `_${fieldName}`;
		}
		return fieldName;
	}

	private isEncodableTypename(name: string) {
		return (
			name === 'ResponseError<"ResponseError">' || !(name.includes('<') || name.includes('{') || name.includes('|'))
		);
	}

	private encodeTypeName(name: string) {
		if (name === 'ResponseError<"ResponseError">') {
			return 'ResponseError';
		}
		if (!this.isEncodableTypename(name)) {
			const sum = crypto.createHash('sha1');
			sum.update(name);
			return `Anonymous${sum.digest('hex')}`;
		}
		return toCamelCase(name);
	}

	private outputTypeDefinition(typeName: string, typeDefinition: string) {
		const prevDefinition = this.types.get(typeName);
		if (prevDefinition) {
			if (prevDefinition !== typeDefinition) {
				throw new Error(`can't redefine type ${typeName} from ${prevDefinition} to ${typeDefinition}`);
			}
			// Nothing to add to the output
			return;
		}
		this.types.set(typeName, typeDefinition);
		this.output += typeDefinition;
	}

	private encodeObject(typeSchema: JSONSchema, optional: boolean, typeName?: string): string {
		if (!typeName) {
			throw new Error(`no type name for ${JSON.stringify(typeSchema)}`);
		}
		this.encodedObjectTypeNames.add(typeName);
		const structTypeName = this.encodeTypeName(typeName);
		let def = typeAnnotations;
		def += `pub struct ${structTypeName} {\n`;
		const typeProperties = typeSchema.properties ?? {};
		const typeRequiredProperties = typeSchema.required ?? [];
		for (const propName of Object.keys(typeProperties)) {
			const propSchema = typeProperties[propName];
			if (typeof propSchema === 'boolean') {
				throw new Error(`can't encode ${propName} in ${typeName}`);
			}
			// Provide a fallback typeName for anonymous embedded objects
			const propTypeName = `${typeName}_${propName}`;
			if ((propSchema.type as any) === 'undefined') {
				continue;
			}
			let propType = this.structNameByTypeName.get(propTypeName) || this.encodeType(propSchema, propTypeName);
			if (!propType) {
				throw new Error(`property ${propName} with schema ${JSON.stringify(propSchema)} returned an empty type`);
			}
			const isRequiredInType = typeRequiredProperties.includes(propName);
			if (!isRequiredInType && !propType.startsWith('Option<')) {
				propType = `Option<${propType}>`;
			}
			if (propType.startsWith('Option<')) {
				def += `\t#[serde(skip_serializing_if = "Option::is_none")]\n`;
			}
			def += `\t#[serde(rename(serialize = "${propName}", deserialize = "${propName}"))]\n`;
			def += `\tpub ${this.encodeFieldName(propName)}: ${propType},\n`;
		}
		def += `}\n\n`;
		this.outputTypeDefinition(structTypeName, def);
		return structTypeName;
	}

	private encodeAnyOf(typeSchema: JSONSchema, optional: boolean, typeName?: string): string {
		const anyOf = getAnyOf(typeSchema);
		if (!typeName) {
			throw new Error(`no type name for ${JSON.stringify(typeSchema)}`);
		}
		const enumTypeName = this.encodeTypeName(typeName);
		let def = typeAnnotations;
		def += `#[serde(untagged)]\n`;
		def += `pub enum ${enumTypeName} {\n`;
		for (let ii = 0; ii < anyOf.length; ii++) {
			const variantTypeName = `${typeName}_${ii}`;
			const variantSchema = anyOf[ii];
			if (typeof variantSchema === 'boolean') {
				throw new Error(`can't encode variant ${ii} in ${typeName}`);
			}
			const variantType = this.encodeType(variantSchema, variantTypeName);
			if (!variantType) {
				throw new Error(`variant ${ii} with schema ${JSON.stringify(variantSchema)} returned an empty type`);
			}
			def += `\t${toCamelCase(variantType.replace(/<|>/g, '_'))}(${variantType}),\n`;
		}
		def += `}\n\n`;
		this.outputTypeDefinition(enumTypeName, def);
		return enumTypeName;
	}

	private encodeType(typeSchema: JSONSchema, typeName?: string): string {
		if (typeSchema.$ref && !this.handledRefPaths.has(typeSchema.$ref)) {
			if (!typeSchema.$ref.startsWith('#/')) {
				throw new Error(`can't resolve ref ${typeSchema.$ref}`);
			}
			const refComponents = typeSchema.$ref.substring(2).split('/');
			let current = this.rootSchema;
			for (const component of refComponents) {
				const next = (current as any)[component];
				if (typeof next === 'undefined') {
					throw new Error(`can't resolve ${component} while looking up $ref ${typeSchema.$ref}`);
				}
				current = next;
			}
			let refTypeName = refComponents[refComponents.length - 1];
			if (refTypeName.includes('{')) {
				refTypeName = typeName ?? refTypeName;
			}
			this.handledRefPaths.add(typeSchema.$ref);
			return this.encodeType(current as JSONSchema, refTypeName);
		}
		if (typeSchema.$ref && typeName) {
			this.structNameByTypeName.set(typeName, this.encodeTypeName(typeName));
		}
		if (typeSchema.$ref && this.rootSchema.definitions) {
			const refComponents = typeSchema.$ref.substring(2).split('/');
			const refTypeName = refComponents[refComponents.length - 1];
			if (this.rootSchema.definitions[refTypeName]) {
				typeSchema = this.rootSchema.definitions[refTypeName] as JSONSchema;
				typeName = refTypeName;
			}
		}
		let type = typeSchema.type;
		let anyOf = getAnyOf(typeSchema);
		let optional = false;
		if (Array.isArray(type)) {
			if (type.includes('null')) {
				optional = true;
				type = type.filter((x) => x !== 'null');
			}
			if (type.length > 1) {
				anyOf = type as unknown as JSONSchema7Definition[];
				//				throw new Error(`can't encode type ${type}`);
			} else {
				type = type[0];
			}
		} else if (!type && (anyOf?.length ?? 0) == 0) {
			if (typeof typeSchema !== 'object' || Object.keys(typeSchema).length !== 0) {
				throw new Error(`invalid type schema ${JSON.stringify(typeSchema)}`);
			}
		}
		const ifOptional = (typeName: string) => {
			if (optional) {
				return `Option<${typeName}>`;
			}
			return typeName;
		};
		if ((anyOf?.length ?? 0) > 0) {
			const enumTypeName = this.encodeAnyOf(typeSchema, optional, typeName);
			return ifOptional(enumTypeName);
		}
		switch (type) {
			case 'boolean':
				return ifOptional('bool');
			case 'integer':
				return ifOptional('i64');
			case 'number':
				return ifOptional('f64');
			case 'string':
				return ifOptional('String');
			case 'array':
				const itemSchema = typeSchema.items;
				if (!itemSchema || typeof itemSchema === 'boolean') {
					throw new Error(`can't encode array with item schema ${itemSchema}`);
				}
				if (Array.isArray(itemSchema)) {
					throw new Error(`can't encode array with multiple item schemas ${itemSchema}`);
				}
				const itemType = this.encodeType(itemSchema, `${typeName}_item`);
				return ifOptional(`Vec<${itemType}>`);
			case 'object':
				if (typeName && this.encodedObjectTypeNames.has(typeName)) {
					return ifOptional(this.encodeTypeName(typeName));
				}
				const structTypeName = this.encodeObject(typeSchema, optional, typeName);
				return ifOptional(structTypeName);
			case undefined:
				return ifOptional('std::collections::HashMap<String, serde_json::Value>');
			default:
				throw new Error(`can't encode type ${type}`);
		}
	}
}

class Inputs implements Template {
	constructor(private config: RustClientTemplateConfig) {}

	async generate(generationConfig: CodeGenerationConfig): Promise<TemplateOutputFile[]> {
		let definedTypes: Map<string, string> | undefined;
		const contents: string[] = [];
		for (const op of generationConfig.config.application.Operations.filter(hasInput)) {
			const encoder = new rustEncoder(op.VariablesSchema, operationInputTypename(op), definedTypes);
			const opContent = encoder.encode();
			definedTypes = encoder.definedTypes();
			contents.push(opContent);
		}
		const content = contents.join('\n\n');
		return Promise.resolve([
			{
				path: path.join('src', 'inputs.rs'),
				content: rustfmt(content),
				header: fileHeader(),
			},
		]);
	}
}

class Responses implements Template {
	constructor(private config: RustClientTemplateConfig) {}

	async generate(generationConfig: CodeGenerationConfig): Promise<TemplateOutputFile[]> {
		let definedTypes: Map<string, string> | undefined;
		const contents: string[] = [];
		for (const op of generationConfig.config.application.Operations) {
			const responseSchema = op.ResponseSchema.properties?.['data'] as JSONSchema;
			if (typeof responseSchema === 'undefined') {
				throw new Error(`could to retrieve response schema for ${op.Name}`);
			}
			if (typeof responseSchema === 'boolean') {
				throw new Error(`can't encode response type for ${op.Name}`);
			}
			const encoder = new rustEncoder(responseSchema, operationResponseTypename(op), definedTypes);
			const opContent = encoder.encode();
			definedTypes = encoder.definedTypes();
			contents.push(opContent);
		}
		const content = contents.join('\n\n');
		return Promise.resolve([
			{
				path: path.join('src', 'responses.rs'),
				content: rustfmt(content),
				header: fileHeader(),
			},
		]);
	}
}

const createMethodsTemplate = (typeName: string, methodName: string, streaming?: boolean) => {
	const extraUse = streaming ?? false ? `use wundergraph_rust_client::Stream;\n` : '';
	const returnTypeTemplate =
		streaming ?? false ? `impl Stream<Item = Result<{{responseTypeName}}>>` : `{{responseTypeName}}`;
	const template = `use std::sync::Arc;

use crate::inputs::*;
use crate::responses::*;
use crate::Result;
${extraUse}

#[derive(serde::Serialize)]
struct Empty {}

pub struct ${typeName} {
	client: Arc<${clientPackageNamespace}::Client>,
}

impl ${typeName} {
	pub fn new(client: Arc<${clientPackageNamespace}::Client>) -> Self {
		Self{
			client,
		}
	}

	{{#each operations}}
	{{#if allowNoSnakeCase}}
	#[allow(non_snake_case)]
	{{/if}}
	pub async fn {{name}}(&self{{#if hasInput}}, input: {{inputTypeName}}{{/if}}) -> Result<${returnTypeTemplate}> {
		{{#if hasInput}}
		self.client.${methodName}("{{path}}", input).await
		{{else}}
		self.client.${methodName}("{{path}}", Empty{}).await
		{{/if}}
	}
	{{/each}}
}`;
	return template;
};

const operationContext = (op: GraphQLOperation) => {
	let responseTypeName = operationResponseTypename(op);
	const responseData = op.ResponseSchema.properties?.['data'] as JSONSchema;
	// If the returned type is just a $ref, we won't generate a type for the container
	// just the $ref itself
	if (responseData?.$ref) {
		responseTypeName = responseData.$ref.replace('#/definitions/', '');
	}
	return {
		name: toSnakeCase(op.Name),
		inputTypeName: toCamelCase(operationInputTypename(op)),
		responseTypeName: toCamelCase(responseTypeName),
		allowNoSnakeCase: toSnakeCase(op.Name).includes('__'),
		path: op.PathName,
		hasInput: hasInput(op),
	};
};

class Queries implements Template {
	constructor(private config: RustClientTemplateConfig) {}

	async generate(generationConfig: CodeGenerationConfig): Promise<TemplateOutputFile[]> {
		const config = generationConfig.config;
		const tmpl = Handlebars.compile(createMethodsTemplate('Queries', 'query'));
		const content = tmpl({
			operations: config.application.Operations.filter((op) => op.OperationType === OperationType.QUERY).map(
				operationContext
			),
		});
		return Promise.resolve([
			{
				path: path.join('src', 'queries.rs'),
				content: rustfmt(content),
				header: fileHeader(),
			},
		]);
	}
}

class Mutations implements Template {
	constructor(private config: RustClientTemplateConfig) {}

	async generate(generationConfig: CodeGenerationConfig): Promise<TemplateOutputFile[]> {
		const config = generationConfig.config;
		const tmpl = Handlebars.compile(createMethodsTemplate('Mutations', 'mutate'));
		const content = tmpl({
			operations: config.application.Operations.filter((op) => op.OperationType === OperationType.MUTATION).map(
				operationContext
			),
		});
		return Promise.resolve([
			{
				path: path.join('src', 'mutations.rs'),
				content: rustfmt(content),
				header: fileHeader(),
			},
		]);
	}
}

class Subscriptions implements Template {
	constructor(private config: RustClientTemplateConfig) {}

	async generate(generationConfig: CodeGenerationConfig): Promise<TemplateOutputFile[]> {
		const operations = generationConfig.config.application.Operations;
		const tmpl = Handlebars.compile(createMethodsTemplate('Subscriptions', 'subscribe', true));
		const content = tmpl({
			operations: operations.filter((op) => op.OperationType === OperationType.SUBSCRIPTION).map(operationContext),
		});
		return Promise.resolve([
			{
				path: path.join('src', 'subscriptions.rs'),
				content: rustfmt(content),
				header: fileHeader(),
			},
		]);
	}
}

class LiveQueries implements Template {
	constructor(private config: RustClientTemplateConfig) {}

	async generate(generationConfig: CodeGenerationConfig): Promise<TemplateOutputFile[]> {
		const operations = generationConfig.config.application.Operations;
		const tmpl = Handlebars.compile(createMethodsTemplate('LiveQueries', 'live_query', true));
		const content = tmpl({
			operations: operations
				.filter((op) => op.OperationType === OperationType.QUERY && op.LiveQuery?.enable === true)
				.map(operationContext),
		});
		return Promise.resolve([
			{
				path: path.join('src', 'live_queries.rs'),
				content: rustfmt(content),
				header: fileHeader(),
			},
		]);
	}
}

export const rustClient = (config: RustClientTemplateConfig = defaultTemplateConfig): Template[] => {
	return [
		new CargoToml(config),
		new LibRs(config),
		new ClientRs(config),
		new Inputs(config),
		new Responses(config),
		new Queries(config),
		new Mutations(config),
		new Subscriptions(config),
		new LiveQueries(config),
	];
};
