from flask import render_template, flash, redirect, url_for, request
from flask_login import current_user, login_user, logout_user, login_required
from werkzeug.urls import url_parse
from app.models import User
from app.auth import bp
from app import db
from app.auth.email import send_password_reset_email
from app.auth.forms import (
    LoginForm,
    RegistrationForm,
    ResetPasswordRequestForm,
    ResetPasswordForm,
)


@bp.route("/login", methods=["GET", "POST"])
def login():
    # prevent the logged user to navigate to "/login"
    if current_user.is_authenticated:
        return redirect(url_for("dashboard.overview"))

    form = LoginForm()
    if form.validate_on_submit():
        # load the user from the db
        user = User.query.filter_by(username=form.username.data).first()

        if user is None or not user.check_password(form.password.data):
            flash("Invalid username or password")

            return redirect(url_for("auth.login"))

        # register the user as logged in
        login_user(user, remember=form.remember_me.data)
        # redirect to the previous page or dashboard if not
        next_page = request.args.get("next")

        if not next_page or url_parse(next_page).netloc != "":
            next_page = url_for("dashboard.overview")

        return redirect(next_page)

    return render_template("auth/login.html", title="Login", form=form)


@bp.route("/register", methods=["GET", "POST"])
def register():
    # prevent the logged user to navigates to the /register URL
    if current_user.is_authenticated:
        return redirect(url_for("main.index"))

    users = User.query.all()
    # if there are useres, redirect to the index
    # (only the first user can use this form)
    if users:
        flash("To register, send an email to carlos.w.montecinos@gmail.com")
        return redirect(url_for("main.index"))

    form = RegistrationForm()
    if form.validate_on_submit():
        # register as administrator
        user = User(
            username=form.username.data, email=form.email.data, is_administrator=True
        )
        user.set_password(form.password.data)
        db.session.add(user)
        db.session.commit()
        flash("User {} has been registered.".format(form.username.data))

        return redirect(url_for("auth.login"))

    return render_template("auth/register.html", title="Register", form=form)


@bp.route("/reset_password_request", methods=["GET", "POST"])
def reset_password_request():
    # prevent the logged user to see this page
    if current_user.is_authenticated:
        return redirect(url_for("dashboard.overview"))

    form = ResetPasswordRequestForm()
    if form.validate_on_submit():
        user = User.query.filter_by(email=form.email.data).first()

        if user:
            send_password_reset_email(user)

        flash("Check your email for the instruction to reset your password.")
        return redirect(url_for("auth.login"))

    return render_template(
        "auth/reset_password_request.html", title="Reset Password", form=form
    )


@bp.route("/reset_password/<token>", methods=["GET", "POST"])
def reset_password(token):
    if current_user.is_authenticated:
        return redirect(url_for("dashboard.overview"))

    user = User.verify_reset_password_token(token)
    if not user:
        return redirect(url_for("main.index"))

    form = ResetPasswordForm()
    if form.validate_on_submit():
        user.set_password(form.password.data)
        db.session.commit()
        flash("Your password has been reset.")
        return redirect(url_for("auth.login"))

    return render_template(
        "auth/reset_password.html", form=form, title="Reset Password"
    )


@bp.route("/logout")
@login_required
def logout():
    logout_user()
    return redirect(url_for("auth.login"))
