use std::{
    net::{IpAddr, SocketAddr},
    str::FromStr,
    thread::{self, JoinHandle},
};

use crate::{app::PhosdbState, serial::Serial};

use super::{UiElement, UiIndicatorText, UiNode};

const DEFAULT_ADDRESS: &str = "127.0.0.1";
const DEFAULT_PORT: u16 = 8888;

pub struct UiSocket {
    is_connected: bool,
    request: bool,
    serial_thread: Option<JoinHandle<Serial>>,

    address: String,
    port: i32,

    indicator: UiIndicatorText,
}

impl UiSocket {
    pub fn new() -> Self {
        let mut indicator = UiIndicatorText::new();
        indicator.set_expiring(true);

        Self {
            is_connected: false,
            request: false,
            serial_thread: None,
            address: DEFAULT_ADDRESS.to_owned(),
            port: DEFAULT_PORT as i32,
            indicator,
        }
    }
}

impl UiNode for UiSocket {
    fn render(&mut self, ui: &imgui::Ui) {
        ui.window("Socket").always_auto_resize(true).build(|| {
            ui.disabled(self.request, || {
                ui.input_text("Address", &mut self.address).build();
                ui.input_int("Port", &mut self.port).build();

                if IpAddr::from_str(self.address.as_str()).is_err() {
                    self.address = DEFAULT_ADDRESS.to_owned();
                }

                if self.port < u16::MIN as i32 {
                    self.port = u16::MIN as i32;
                }
                if self.port > u16::MAX as i32 {
                    self.port = u16::MAX as i32;
                }

                if ui.button(if self.is_connected {
                    "Disconnect"
                } else {
                    "Connect"
                }) {
                    self.request = true;
                }
            });

            ui.same_line();

            self.indicator.render(ui);
        });
    }
}

impl UiElement<PhosdbState> for UiSocket {
    fn update(&mut self, state: &mut PhosdbState) {
        if self.serial_thread.is_some() {
            if self.serial_thread.as_ref().unwrap().is_finished() {
                state.serial = self
                    .serial_thread
                    .take()
                    .unwrap()
                    .join()
                    .expect("couldn't join serial connection thread");

                if state.serial.is_connected() {
                    self.indicator.set_success("successfully connected");
                } else {
                    self.indicator.set_error("failed to connect");
                }

                self.request = false;
            }
        } else if self.request {
            if self.is_connected {
                state.serial.disconnect();

                self.indicator.set_neutral("disconnected");

                self.request = false;
            } else {
                let address = self.address.clone();
                let port = self.port as u16;

                self.serial_thread = Some(thread::spawn(move || {
                    Serial::new(&SocketAddr::new(
                        IpAddr::from_str(address.as_str()).unwrap(),
                        port,
                    ))
                }));

                self.indicator.set_neutral("connecting...");
            }
        }

        self.is_connected = state.serial.is_connected();
    }
}
