use std::time::Instant;

use imgui::sys::*;

use crate::util::{norm_rgb, rgb_alpha};

pub mod about;
pub mod menu;
pub mod socket;

const INDICATOR_EXPIRE_TIME: f32 = 2.0;

pub trait UiNode {
    fn render(&mut self, ui: &imgui::Ui);
}

pub trait UiElement<T>: UiNode {
    fn update(&mut self, state: &mut T);
}

pub struct UiIndicatorText {
    text: String,
    color: [f32; 3],
    time: Instant,
    expiring: bool,
}

impl UiIndicatorText {
    pub fn new() -> Self {
        Self {
            text: String::default(),
            color: norm_rgb(&catppuccin::PALETTE.mocha.colors.text.rgb),
            time: Instant::now(),
            expiring: false,
        }
    }

    pub fn set_expiring(&mut self, expiring: bool) {
        self.expiring = expiring;
    }

    fn set_text(&mut self, text: &str) {
        self.text = text.to_owned();
        self.time = Instant::now();
    }

    pub fn set_neutral(&mut self, text: &str) {
        self.color = norm_rgb(&catppuccin::PALETTE.mocha.colors.text.rgb);
        self.set_text(text);
    }

    pub fn set_success(&mut self, text: &str) {
        self.color = norm_rgb(&catppuccin::PALETTE.latte.colors.green.rgb);
        self.set_text(text);
    }

    pub fn set_error(&mut self, text: &str) {
        self.color = norm_rgb(&catppuccin::PALETTE.latte.colors.red.rgb);
        self.set_text(text);
    }
}

impl UiNode for UiIndicatorText {
    fn render(&mut self, ui: &imgui::Ui) {
        let elapsed = self.time.elapsed().as_secs_f32();

        if !self.text.is_empty() && (!self.expiring || elapsed < INDICATOR_EXPIRE_TIME) {
            let color = if self.expiring {
                rgb_alpha(
                    self.color,
                    1.0 - (elapsed / INDICATOR_EXPIRE_TIME).min(1.0).powf(2.0),
                )
            } else {
                rgb_alpha(self.color, 1.0)
            };

            ui.text_colored(color, &self.text);
        }
    }
}

pub fn center_next_window() {
    unsafe {
        let io = &*igGetIO();

        igSetNextWindowPos(
            ImVec2::new(io.DisplaySize.x * 0.5, io.DisplaySize.y * 0.5),
            ImGuiCond_Always as i32,
            ImVec2::new(0.5, 0.5),
        )
    }
}

pub fn hcenter<R, F: FnOnce() -> R>(ui: &imgui::Ui, width: f32, f: F) -> R {
    let window_width = ui.window_size()[0];

    unsafe {
        igSetCursorPosX((window_width - width) * 0.5);
    }

    f()
}

pub fn text_hcenter<R, F: FnOnce(&str) -> R>(ui: &imgui::Ui, text: &str, f: F) -> R {
    let width = ui.calc_text_size(text)[0];

    hcenter(ui, width, || f(text))
}

pub fn button_hcenter<R, F: FnOnce(&str) -> R>(ui: &imgui::Ui, text: &str, f: F) -> R {
    let width = ui.calc_text_size(text)[0] + unsafe { ui.style() }.frame_padding[0] * 2.0;

    hcenter(ui, width, || f(text))
}
