use glow::{Context, HasContext};
use glutin::{
    config::{Config, ConfigTemplateBuilder},
    context::{ContextAttributesBuilder, NotCurrentGlContext, PossiblyCurrentContext},
    display::{GetGlDisplay, GlDisplay},
    surface::{GlSurface, Surface, SurfaceAttributesBuilder, WindowSurface},
};
use raw_window_handle::HasRawWindowHandle;
use std::{num::NonZeroU32, time::Instant};
use winit::{
    dpi::LogicalSize,
    error::EventLoopError,
    event_loop::EventLoop,
    window::{Window, WindowBuilder},
};

use crate::{
    serial::Serial,
    ui::{self, UiElement},
    util::{norm_rgb, premul_array, rgb_alpha},
};

pub struct PhosdbState {
    pub serial: Serial,
}

pub struct PhosdbApp {
    event_loop: EventLoop<()>,
    window: Window,
    gl_context: PossiblyCurrentContext,
    gl_surface: Surface<WindowSurface>,
    imgui_ctx: imgui::Context,
    imgui_winit_platform: imgui_winit_support::WinitPlatform,
    imgui_renderer: imgui_glow_renderer::AutoRenderer,
    last_frame: Instant,
    ui: Vec<Box<dyn UiElement<PhosdbState>>>,
    state: PhosdbState,
}

pub trait App {
    fn new(event_loop: EventLoop<()>) -> Self
    where
        Self: Sized;

    fn run(self) -> Result<(), EventLoopError>;
}

impl PhosdbApp {
    fn create_window_for_gl(event_loop: &EventLoop<()>) -> (Option<Window>, Config) {
        let window_builder = WindowBuilder::new()
            .with_title("Phosdb_rs")
            .with_inner_size(LogicalSize::new(1024.0, 576.0))
            .with_transparent(true);

        glutin_winit::DisplayBuilder::new()
            .with_window_builder(Some(window_builder))
            .build(&event_loop, ConfigTemplateBuilder::new(), |mut configs| {
                configs.next().unwrap()
            })
            .expect("failed to create window")
    }

    fn init_gl(
        window: &Window,
        gl_config: &Config,
    ) -> (PossiblyCurrentContext, Surface<WindowSurface>, Context) {
        let context_attribs =
            ContextAttributesBuilder::new().build(Some(window.raw_window_handle()));

        let context = unsafe {
            gl_config
                .display()
                .create_context(&gl_config, &context_attribs)
                .expect("failed to create OpenGL context")
        };

        let surface_attribs = SurfaceAttributesBuilder::<WindowSurface>::new()
            .with_srgb(Some(true))
            .build(
                window.raw_window_handle(),
                NonZeroU32::new(1024).unwrap(),
                NonZeroU32::new(576).unwrap(),
            );
        let surface = unsafe {
            gl_config
                .display()
                .create_window_surface(&gl_config, &surface_attribs)
                .expect("failed to create OpenGL surface")
        };

        let context = context
            .make_current(&surface)
            .expect("failed to make OpenGL context current");

        let gl = unsafe {
            glow::Context::from_loader_function_cstr(|s| {
                context.display().get_proc_address(s).cast()
            })
        };

        (context, surface, gl)
    }

    fn init_imgui(
        window: &Window,
        gl: Context,
    ) -> (
        imgui::Context,
        imgui_winit_support::WinitPlatform,
        imgui_glow_renderer::AutoRenderer,
    ) {
        let mut imgui_ctx = imgui::Context::create();
        imgui_ctx.set_ini_filename(None);

        let mut imgui_winit_platform = imgui_winit_support::WinitPlatform::init(&mut imgui_ctx);
        imgui_winit_platform.attach_window(
            imgui_ctx.io_mut(),
            &window,
            imgui_winit_support::HiDpiMode::Rounded,
        );

        imgui_ctx
            .fonts()
            .add_font(&[imgui::FontSource::DefaultFontData { config: None }]);

        imgui_ctx.io_mut().font_global_scale = (1.0 / imgui_winit_platform.hidpi_factor()) as f32;

        let imgui_renderer = imgui_glow_renderer::AutoRenderer::initialize(gl, &mut imgui_ctx)
            .expect("failed to create ImGui OpenGL renderer");

        (imgui_ctx, imgui_winit_platform, imgui_renderer)
    }

    #[rustfmt::skip]
    fn set_style(style: &mut imgui::Style) {
        style.window_rounding    = 0.0;
        style.child_rounding     = 0.0;
        style.frame_rounding     = 0.0;
        style.popup_rounding     = 0.0;
        style.scrollbar_rounding = 0.0;
        style.grab_rounding      = 0.0;
        style.tab_rounding       = 0.0;

        style.frame_border_size = 1.0;

        style[imgui::StyleColor::Text]                 = rgb_alpha(norm_rgb(&catppuccin::PALETTE.mocha.colors.text.rgb), 1.0);
        style[imgui::StyleColor::TextDisabled]         = rgb_alpha(norm_rgb(&catppuccin::PALETTE.mocha.colors.text.rgb), 0.5);
        style[imgui::StyleColor::WindowBg]             = rgb_alpha(norm_rgb(&catppuccin::PALETTE.mocha.colors.base.rgb), 0.75);
        style[imgui::StyleColor::PopupBg]              = rgb_alpha(norm_rgb(&catppuccin::PALETTE.mocha.colors.base.rgb), 0.75);
        style[imgui::StyleColor::Border]               = rgb_alpha(norm_rgb(&catppuccin::PALETTE.mocha.colors.surface0.rgb), 0.25);
        style[imgui::StyleColor::FrameBg]              = rgb_alpha(norm_rgb(&catppuccin::PALETTE.mocha.colors.overlay0.rgb), 0.5);
        style[imgui::StyleColor::FrameBgHovered]       = rgb_alpha(norm_rgb(&catppuccin::PALETTE.latte.colors.lavender.rgb), 0.5);
        style[imgui::StyleColor::FrameBgActive]        = rgb_alpha(norm_rgb(&catppuccin::PALETTE.latte.colors.lavender.rgb), 0.75);
        style[imgui::StyleColor::TitleBg]              = rgb_alpha(norm_rgb(&catppuccin::PALETTE.mocha.colors.crust.rgb), 0.75);
        style[imgui::StyleColor::TitleBgActive]        = rgb_alpha(norm_rgb(&catppuccin::PALETTE.mocha.colors.surface0.rgb), 0.75);
        style[imgui::StyleColor::TitleBgCollapsed]     = rgb_alpha(norm_rgb(&catppuccin::PALETTE.mocha.colors.crust.rgb), 0.5);
        style[imgui::StyleColor::MenuBarBg]            = rgb_alpha(norm_rgb(&catppuccin::PALETTE.mocha.colors.surface0.rgb), 0.75);
        style[imgui::StyleColor::ScrollbarBg]          = rgb_alpha(norm_rgb(&catppuccin::PALETTE.mocha.colors.crust.rgb), 0.5);
        style[imgui::StyleColor::ScrollbarGrab]        = rgb_alpha(norm_rgb(&catppuccin::PALETTE.mocha.colors.surface0.rgb), 0.5);
        style[imgui::StyleColor::ScrollbarGrabHovered] = rgb_alpha(norm_rgb(&catppuccin::PALETTE.mocha.colors.surface1.rgb), 0.75);
        style[imgui::StyleColor::ScrollbarGrabActive]  = rgb_alpha(norm_rgb(&catppuccin::PALETTE.mocha.colors.surface2.rgb), 1.0);
        style[imgui::StyleColor::CheckMark]            = rgb_alpha(norm_rgb(&catppuccin::PALETTE.latte.colors.lavender.rgb), 1.0);
        style[imgui::StyleColor::SliderGrab]           = rgb_alpha(norm_rgb(&catppuccin::PALETTE.latte.colors.lavender.rgb), 0.5);
        style[imgui::StyleColor::SliderGrabActive]     = rgb_alpha(norm_rgb(&catppuccin::PALETTE.latte.colors.lavender.rgb), 1.0);
        style[imgui::StyleColor::Button]               = rgb_alpha(norm_rgb(&catppuccin::PALETTE.latte.colors.lavender.rgb), 0.5);
        style[imgui::StyleColor::ButtonHovered]        = rgb_alpha(norm_rgb(&catppuccin::PALETTE.latte.colors.lavender.rgb), 0.75);
        style[imgui::StyleColor::ButtonActive]         = rgb_alpha(norm_rgb(&catppuccin::PALETTE.latte.colors.lavender.rgb), 1.0);
        style[imgui::StyleColor::Header]               = rgb_alpha(norm_rgb(&catppuccin::PALETTE.latte.colors.lavender.rgb), 0.5);
        style[imgui::StyleColor::HeaderHovered]        = rgb_alpha(norm_rgb(&catppuccin::PALETTE.latte.colors.lavender.rgb), 0.75);
        style[imgui::StyleColor::HeaderActive]         = rgb_alpha(norm_rgb(&catppuccin::PALETTE.latte.colors.lavender.rgb), 1.0);
        style[imgui::StyleColor::Separator]            = rgb_alpha(norm_rgb(&catppuccin::PALETTE.mocha.colors.overlay1.rgb), 0.5);
        style[imgui::StyleColor::SeparatorHovered]     = rgb_alpha(norm_rgb(&catppuccin::PALETTE.latte.colors.lavender.rgb), 0.75);
        style[imgui::StyleColor::SeparatorActive]      = rgb_alpha(norm_rgb(&catppuccin::PALETTE.latte.colors.lavender.rgb), 1.0);
        style[imgui::StyleColor::ResizeGrip]           = rgb_alpha(norm_rgb(&catppuccin::PALETTE.latte.colors.lavender.rgb), 0.5);
        style[imgui::StyleColor::ResizeGripHovered]    = rgb_alpha(norm_rgb(&catppuccin::PALETTE.latte.colors.lavender.rgb), 0.75);
        style[imgui::StyleColor::ResizeGripActive]     = rgb_alpha(norm_rgb(&catppuccin::PALETTE.latte.colors.lavender.rgb), 1.0);
        style[imgui::StyleColor::Tab]                  = rgb_alpha(norm_rgb(&catppuccin::PALETTE.latte.colors.lavender.rgb), 0.5);
        style[imgui::StyleColor::TabHovered]           = rgb_alpha(norm_rgb(&catppuccin::PALETTE.latte.colors.lavender.rgb), 0.75);
        style[imgui::StyleColor::TabActive]            = rgb_alpha(norm_rgb(&catppuccin::PALETTE.latte.colors.lavender.rgb), 1.0);
        style[imgui::StyleColor::TabUnfocused]         = rgb_alpha(norm_rgb(&catppuccin::PALETTE.mocha.colors.surface0.rgb), 0.5);
        style[imgui::StyleColor::TabUnfocusedActive]   = rgb_alpha(norm_rgb(&catppuccin::PALETTE.mocha.colors.overlay0.rgb), 1.0);
        style[imgui::StyleColor::PlotLines]            = rgb_alpha(norm_rgb(&catppuccin::PALETTE.mocha.colors.overlay2.rgb), 1.0);
        style[imgui::StyleColor::TableHeaderBg]        = rgb_alpha(norm_rgb(&catppuccin::PALETTE.mocha.colors.surface0.rgb), 0.75);
        style[imgui::StyleColor::TextSelectedBg]       = rgb_alpha(norm_rgb(&catppuccin::PALETTE.latte.colors.lavender.rgb), 0.25);
        style[imgui::StyleColor::NavHighlight]         = rgb_alpha(norm_rgb(&catppuccin::PALETTE.latte.colors.lavender.rgb), 1.0);
    }
}

impl App for PhosdbApp {
    fn new(event_loop: EventLoop<()>) -> Self {
        debug!("creating winit window");

        let (window, gl_config) = Self::create_window_for_gl(&event_loop);
        let window = window.unwrap();

        debug!("initializing OpenGL");

        let (gl_context, gl_surface, gl) = Self::init_gl(&window, &gl_config);

        debug!("initializing Dear ImGui");

        let (mut imgui_ctx, imgui_winit_platform, imgui_renderer) = Self::init_imgui(&window, gl);

        Self::set_style(imgui_ctx.style_mut());

        let last_frame = Instant::now();

        debug!("creating tokio runtime");

        Self {
            event_loop,
            window,
            gl_context,
            gl_surface,
            imgui_ctx,
            imgui_winit_platform,
            imgui_renderer,
            last_frame,
            ui: vec![
                Box::new(ui::menu::UiMenu::default()),
                Box::new(ui::socket::UiSocket::new()),
            ],
            state: PhosdbState {
                serial: Serial::default(),
            },
        }
    }

    fn run(mut self) -> Result<(), EventLoopError> {
        self.event_loop.run(|event, window_target| match event {
            winit::event::Event::NewEvents(_) => {
                let now = Instant::now();
                self.imgui_ctx
                    .io_mut()
                    .update_delta_time(now.duration_since(self.last_frame));
                self.last_frame = now;
            }
            winit::event::Event::AboutToWait => {
                self.imgui_winit_platform
                    .prepare_frame(self.imgui_ctx.io_mut(), &self.window)
                    .unwrap();
                self.window.request_redraw();
            }
            winit::event::Event::WindowEvent {
                event: winit::event::WindowEvent::RedrawRequested,
                ..
            } => {
                let gl = self.imgui_renderer.gl_context();

                unsafe {
                    gl.clear(glow::COLOR_BUFFER_BIT);

                    let rgba = premul_array(rgb_alpha(
                        norm_rgb(&catppuccin::PALETTE.mocha.colors.base.rgb),
                        0.25,
                    ));

                    gl.clear_color(rgba[0], rgba[1], rgba[2], rgba[3]);
                };

                let ui = self.imgui_ctx.new_frame();

                for ui_element in self.ui.iter_mut() {
                    ui_element.update(&mut self.state);
                    ui_element.render(ui);
                }

                self.imgui_winit_platform.prepare_render(ui, &self.window);
                let draw_data = self.imgui_ctx.render();

                if draw_data.draw_lists_count() > 0 {
                    self.imgui_renderer
                        .render(draw_data)
                        .expect("error rendering ImGui");

                    self.gl_surface
                        .swap_buffers(&self.gl_context)
                        .expect("failed to swap buffers");
                }
            }
            winit::event::Event::WindowEvent {
                event: winit::event::WindowEvent::CloseRequested,
                ..
            } => {
                info!("exiting");
                window_target.exit();
            }
            winit::event::Event::WindowEvent {
                event: winit::event::WindowEvent::Resized(new_size),
                ..
            } => {
                if new_size.width > 0 && new_size.height > 0 {
                    self.gl_surface.resize(
                        &self.gl_context,
                        NonZeroU32::new(new_size.width).unwrap(),
                        NonZeroU32::new(new_size.height).unwrap(),
                    );
                }
                self.imgui_winit_platform.handle_event(
                    self.imgui_ctx.io_mut(),
                    &self.window,
                    &event,
                );
            }
            mut event => {
                // https://github.com/imgui-rs/imgui-rs/issues/776
                if let winit::event::Event::WindowEvent {
                    event:
                        winit::event::WindowEvent::KeyboardInput {
                            device_id: _,
                            event: ref mut input_event,
                            is_synthetic: _,
                        },
                    ..
                } = event
                {
                    if !input_event.state.is_pressed() {
                        input_event.text = None;
                    }
                }

                self.imgui_winit_platform.handle_event(
                    self.imgui_ctx.io_mut(),
                    &self.window,
                    &event,
                );
            }
        })
    }
}
