//                           _       _
// __      _____  __ ___   ___  __ _| |_ ___
// \ \ /\ / / _ \/ _` \ \ / / |/ _` | __/ _ \
//  \ V  V /  __/ (_| |\ V /| | (_| | ||  __/
//   \_/\_/ \___|\__,_| \_/ |_|\__,_|\__\___|
//
//  Copyright © 2016 - 2024 Weaviate B.V. All rights reserved.
//
//  CONTACT: hello@weaviate.io
//

//go:build integrationTest
// +build integrationTest

package lsmkv

import (
	"context"
	"fmt"
	"math/rand"
	"testing"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
	"github.com/weaviate/sroar"
	"github.com/weaviate/weaviate/entities/cyclemanager"
	"github.com/weaviate/weaviate/entities/filters"
)

func compactionRoaringSetRangeStrategy_Random(ctx context.Context, t *testing.T, opts []BucketOption) {
	maxKey := uint64(100)
	maxValue := uint64(1_000_000)
	iterations := uint64(100_000)

	deleteRatio := 0.2   // 20% of all operations will be deletes, 80% additions
	flushChance := 0.001 // on average one flush per 1000 iterations

	r := getRandomSeed()

	instr := generateRandomRangeInstructions(r, maxKey, maxValue, iterations, deleteRatio)
	control := controlFromRangeInstructions(instr, maxKey)

	b, err := NewBucketCreator().NewBucket(ctx, t.TempDir(), "", nullLogger(), nil,
		cyclemanager.NewCallbackGroupNoop(), cyclemanager.NewCallbackGroupNoop(), opts...)
	require.Nil(t, err)

	defer b.Shutdown(testCtx())

	// so big it effectively never triggers as part of this test
	b.SetMemtableThreshold(1e9)

	compactions := 0
	for _, inst := range instr {
		if inst.addition {
			b.RoaringSetRangeAdd(inst.key, inst.value)
		} else {
			b.RoaringSetRangeRemove(inst.key, inst.value)
		}

		if r.Float64() < flushChance {
			require.Nil(t, b.FlushAndSwitch())
		}
	}

	var compacted bool
	for compacted, err = b.disk.compactOnce(); err == nil && compacted; compacted, err = b.disk.compactOnce() {
		compactions++
	}
	require.Nil(t, err)

	// this is a sanity check to make sure the test setup actually does what we
	// want. With the current setup, we expect on average to have ~100
	// compactions. It would be extremely unexpected to have fewer than 25.
	assert.Greater(t, compactions, 25)

	verifyBucketRangeAgainstControl(t, b, control)
}

func verifyBucketRangeAgainstControl(t *testing.T, b *Bucket, control []*sroar.Bitmap) {
	reader := b.ReaderRoaringSetRange()
	defer reader.Close()

	for i, controlBM := range control {
		actual, err := reader.Read(context.Background(), uint64(i), filters.OperatorEqual)
		require.Nil(t, err)

		require.Equal(t, controlBM.ToArray(), actual.ToArray(), "i = %d", i)
	}
}

type roaringSetRangeInstruction struct {
	key   uint64
	value uint64

	// true=addition, false=deletion
	addition bool
}

func generateRandomRangeInstructions(r *rand.Rand, maxKey, maxValue, iterations uint64,
	deleteRatio float64,
) []roaringSetRangeInstruction {
	instr := make([]roaringSetRangeInstruction, iterations)

	for i := range instr {
		instr[i].key = uint64(r.Intn(int(maxKey)))     // value
		instr[i].value = uint64(r.Intn(int(maxValue))) // docId
		instr[i].addition = r.Float64() > deleteRatio
	}

	return instr
}

func controlFromRangeInstructions(instr []roaringSetRangeInstruction, maxKey uint64) []*sroar.Bitmap {
	unique := make(map[uint64]uint64)
	for _, inst := range instr {
		if inst.addition {
			unique[inst.value] = inst.key
		} else {
			delete(unique, inst.value)
		}
	}

	out := make([]*sroar.Bitmap, maxKey)
	for i := range out {
		out[i] = sroar.NewBitmap()
	}
	for value, key := range unique {
		out[key].Set(value)
	}

	return out
}

func compactionRoaringSetRangeStrategy(ctx context.Context, t *testing.T, opts []BucketOption,
	expectedMinSize, expectedMaxSize int64,
) {
	maxKey := uint64(100)

	type kv struct {
		key       uint64
		additions []uint64
		deletions []uint64
	}
	// this segment is not part of the merge, but might still play a role in
	// overall results. For example if one of the later segments has a tombstone
	// for it
	var previous1 []kv
	var previous2 []kv

	var segment1 []kv
	var segment2 []kv
	var expected []kv
	var bucket *Bucket

	dirName := t.TempDir()

	t.Run("create test data", func(t *testing.T) {
		// The test data is split into 8 scenarios evenly:
		//
		// 0.) created in the first segment, never touched again
		// 1.) created in the first segment, appended to it in the second
		// 2.) created in the first segment, first element deleted in the second
		// 3.) created in the first segment, second element deleted in the second
		// 4.) not present in the first segment, created in the second
		// 5.) present in an unrelated previous segment, deleted in the first
		// 6.) present in an unrelated previous segment, deleted in the second
		// 7.) present in an unrelated previous segment, never touched again
		for k := uint64(0); k < maxKey; k++ {
			key := k
			values := []uint64{k + 10_000, k + 20_000}

			switch k % 8 {
			case 0:
				// add to segment 1
				segment1 = append(segment1, kv{
					key:       key,
					additions: values[:1],
				})

				// leave this element untouched in the second segment
				expected = append(expected, kv{
					key:       key,
					additions: values[:1],
				})

			case 1:
				// add to segment 1
				segment1 = append(segment1, kv{
					key:       key,
					additions: values[:1],
				})

				// update in the second segment
				segment2 = append(segment2, kv{
					key:       key,
					additions: values[1:],
				})

				expected = append(expected, kv{
					key:       key,
					additions: values,
				})

			case 2:
				// add both to segment 1, delete the first
				segment1 = append(segment1, kv{
					key:       key,
					additions: values,
				})

				// delete first element in the second segment
				segment2 = append(segment2, kv{
					key:       key,
					deletions: values[:1],
				})

				// only the 2nd element should be left in the expected
				expected = append(expected, kv{
					key:       key,
					additions: values[1:],
				})

			case 3:
				// add both to segment 1, delete the second
				segment1 = append(segment1, kv{
					key:       key,
					additions: values,
				})

				// delete second element in the second segment
				segment2 = append(segment2, kv{
					key:       key,
					deletions: values[1:],
				})

				// only the 1st element should be left in the expected
				expected = append(expected, kv{
					key:       key,
					additions: values[:1],
				})

			case 4:
				// do not add to segment 1

				// only add to segment 2 (first entry)
				segment2 = append(segment2, kv{
					key:       key,
					additions: values,
				})

				expected = append(expected, kv{
					key:       key,
					additions: values,
				})

			case 5:
				// only part of a previous segment, which is not part of the merge
				previous1 = append(previous1, kv{
					key:       key,
					additions: values[:1],
				})
				previous2 = append(previous2, kv{
					key:       key,
					additions: values[1:],
				})

				// delete in segment 1
				segment1 = append(segment1, kv{
					key:       key,
					deletions: values,
				})

				// should not have any values in expected at all, not even a key

			case 6:
				// only part of a previous segment, which is not part of the merge
				previous1 = append(previous1, kv{
					key:       key,
					additions: values[:1],
				})
				previous2 = append(previous2, kv{
					key:       key,
					additions: values[1:],
				})

				// delete in segment 2
				segment2 = append(segment2, kv{
					key:       key,
					deletions: values,
				})

				// should not have any values in expected at all, not even a key

			case 7:
				// part of a previous segment
				previous1 = append(previous1, kv{
					key:       key,
					additions: values[:1],
				})
				previous2 = append(previous2, kv{
					key:       key,
					additions: values[1:],
				})

				expected = append(expected, kv{
					key:       key,
					additions: values,
				})
			}
		}
	})

	t.Run("shuffle the import order for each segment", func(t *testing.T) {
		// this is to make sure we don't accidentally rely on the import order
		rand.Shuffle(len(segment1), func(i, j int) {
			segment1[i], segment1[j] = segment1[j], segment1[i]
		})
		rand.Shuffle(len(segment2), func(i, j int) {
			segment2[i], segment2[j] = segment2[j], segment2[i]
		})
	})

	t.Run("init bucket", func(t *testing.T) {
		b, err := NewBucketCreator().NewBucket(ctx, dirName, dirName, nullLogger(), nil,
			cyclemanager.NewCallbackGroupNoop(), cyclemanager.NewCallbackGroupNoop(), opts...)
		require.Nil(t, err)

		// so big it effectively never triggers as part of this test
		b.SetMemtableThreshold(1e9)

		bucket = b
	})

	t.Run("import and flush previous segments", func(t *testing.T) {
		for _, kv := range previous1 {
			err := bucket.RoaringSetRangeAdd(kv.key, kv.additions...)
			require.NoError(t, err)
		}

		require.NoError(t, bucket.FlushAndSwitch())

		for _, kv := range previous2 {
			err := bucket.RoaringSetRangeAdd(kv.key, kv.additions...)
			require.NoError(t, err)
		}

		require.NoError(t, bucket.FlushAndSwitch())
	})

	t.Run("import segment 1", func(t *testing.T) {
		for _, kv := range segment1 {
			if len(kv.additions) > 0 {
				err := bucket.RoaringSetRangeAdd(kv.key, kv.additions...)
				require.NoError(t, err)
			}
			if len(kv.deletions) > 0 {
				err := bucket.RoaringSetRangeRemove(kv.key, kv.deletions...)
				require.NoError(t, err)
			}
		}
	})

	t.Run("flush to disk", func(t *testing.T) {
		require.NoError(t, bucket.FlushAndSwitch())
	})

	t.Run("import segment 2", func(t *testing.T) {
		for _, kv := range segment2 {
			if len(kv.additions) > 0 {
				err := bucket.RoaringSetRangeAdd(kv.key, kv.additions...)
				require.NoError(t, err)
			}
			if len(kv.deletions) > 0 {
				err := bucket.RoaringSetRangeRemove(kv.key, kv.deletions...)
				require.NoError(t, err)
			}
		}
	})

	t.Run("flush to disk", func(t *testing.T) {
		require.NoError(t, bucket.FlushAndSwitch())
	})

	t.Run("verify control before compaction", func(t *testing.T) {
		reader := bucket.ReaderRoaringSetRange()
		defer reader.Close()

		var retrieved []kv
		for k := uint64(0); k < maxKey; k++ {
			bm, err := reader.Read(context.Background(), k, filters.OperatorEqual)
			require.NoError(t, err)

			if !bm.IsEmpty() {
				retrieved = append(retrieved, kv{
					key:       k,
					additions: bm.ToArray(),
				})
			}
		}

		assert.Equal(t, expected, retrieved)
	})

	t.Run("compact until no longer eligible", func(t *testing.T) {
		i := 0
		var compacted bool
		var err error
		for compacted, err = bucket.disk.compactOnce(); err == nil && compacted; compacted, err = bucket.disk.compactOnce() {
			if i == 1 {
				// segment1 and segment2 merged
				// none of them is root segment, so tombstones
				// will not be removed regardless of keepTombstones setting
				assertSecondSegmentOfSize(t, bucket, 2256, 2256)
			}
			i++
		}
		require.Nil(t, err)
	})

	t.Run("verify control after compaction", func(t *testing.T) {
		reader := bucket.ReaderRoaringSetRange()
		defer reader.Close()

		var retrieved []kv
		for k := uint64(0); k < maxKey; k++ {
			bm, err := reader.Read(context.Background(), k, filters.OperatorEqual)
			require.NoError(t, err)

			if !bm.IsEmpty() {
				retrieved = append(retrieved, kv{
					key:       k,
					additions: bm.ToArray(),
				})
			}
		}

		assert.Equal(t, expected, retrieved)
		assertSingleSegmentOfSize(t, bucket, expectedMinSize, expectedMaxSize)
	})
}

func compactionRoaringSetRangeStrategy_RemoveUnnecessary(ctx context.Context, t *testing.T, opts []BucketOption) {
	// in this test each segment reverses the action of the previous segment so
	// that in the end a lot of information is present in the individual segments
	// which is no longer needed. We then verify that after all compaction this
	// information is gone, thus freeing up disk space
	iterations := uint64(100)
	key := uint64(1)

	var bucket *Bucket
	dirName := t.TempDir()

	t.Run("init bucket", func(t *testing.T) {
		b, err := NewBucketCreator().NewBucket(ctx, dirName, "", nullLogger(), nil,
			cyclemanager.NewCallbackGroupNoop(), cyclemanager.NewCallbackGroupNoop(), opts...)
		require.Nil(t, err)

		// so big it effectively never triggers as part of this test
		b.SetMemtableThreshold(1e9)

		bucket = b
	})

	t.Run("write segments", func(t *testing.T) {
		for v := uint64(0); v < iterations; v++ {
			if v != 0 {
				// we can only delete an existing value if this isn't the first write
				err := bucket.RoaringSetRangeRemove(key, v-1)
				require.NoError(t, err)
			}

			err := bucket.RoaringSetRangeAdd(key, v)
			require.NoError(t, err)

			require.NoError(t, bucket.FlushAndSwitch())
		}
	})

	t.Run("verify control before compaction", func(t *testing.T) {
		reader := bucket.ReaderRoaringSetRange()
		defer reader.Close()

		bm, err := reader.Read(context.Background(), key, filters.OperatorEqual)
		require.NoError(t, err)

		assert.Equal(t, []uint64{iterations - 1}, bm.ToArray())
	})

	t.Run("compact until no longer eligible", func(t *testing.T) {
		var compacted bool
		var err error
		for compacted, err = bucket.disk.compactOnce(); err == nil && compacted; compacted, err = bucket.disk.compactOnce() {
		}
		require.Nil(t, err)
	})

	t.Run("verify control before compaction", func(t *testing.T) {
		reader := bucket.ReaderRoaringSetRange()
		defer reader.Close()

		bm, err := reader.Read(context.Background(), key, filters.OperatorEqual)
		require.NoError(t, err)

		assert.Equal(t, []uint64{iterations - 1}, bm.ToArray())
	})
}

func compactionRoaringSetRangeStrategy_FrequentPutDeleteOperations(ctx context.Context, t *testing.T, opts []BucketOption) {
	// In this test we are testing that the compaction works well for set collection
	maxSegments := 10

	for segments := 4; segments < maxSegments; segments++ {
		t.Run(fmt.Sprintf("compact %v segments", segments), func(t *testing.T) {
			var bucket *Bucket

			key := uint64(1)
			values := []uint64{1, 2}

			dirName := t.TempDir()

			t.Run("init bucket", func(t *testing.T) {
				b, err := NewBucketCreator().NewBucket(ctx, dirName, "", nullLogger(), nil,
					cyclemanager.NewCallbackGroupNoop(), cyclemanager.NewCallbackGroupNoop(), opts...)
				require.Nil(t, err)

				// so big it effectively never triggers as part of this test
				b.SetMemtableThreshold(1e9)

				bucket = b
			})

			t.Run("import and flush segments", func(t *testing.T) {
				for segment := 0; segment < segments; segment++ {
					err := bucket.RoaringSetRangeAdd(key, values...)
					require.Nil(t, err)

					if segments == 5 {
						// delete all
						err := bucket.RoaringSetRangeRemove(key, values[0])
						require.Nil(t, err)
						err = bucket.RoaringSetRangeRemove(key, values[1])
						require.Nil(t, err)
					} else if segments == 6 {
						// delete only one value
						err := bucket.RoaringSetRangeRemove(key, values[0])
						require.Nil(t, err)
					} else if segment != segments-1 {
						// don't delete from the last segment
						err := bucket.RoaringSetRangeRemove(key, values[0])
						require.Nil(t, err)
						err = bucket.RoaringSetRangeRemove(key, values[1])
						require.Nil(t, err)
					}

					require.Nil(t, bucket.FlushAndSwitch())
				}
			})

			t.Run("verify that objects exist before compaction", func(t *testing.T) {
				reader := bucket.ReaderRoaringSetRange()
				defer reader.Close()

				bm, err := reader.Read(context.Background(), key, filters.OperatorEqual)
				require.NoError(t, err)

				if segments == 5 {
					assert.Equal(t, 0, bm.GetCardinality())
				} else if segments == 6 {
					assert.Equal(t, 1, bm.GetCardinality())
				} else {
					assert.Equal(t, 2, bm.GetCardinality())
				}
			})

			t.Run("compact until no longer eligible", func(t *testing.T) {
				var compacted bool
				var err error
				for compacted, err = bucket.disk.compactOnce(); err == nil && compacted; compacted, err = bucket.disk.compactOnce() {
				}
				require.Nil(t, err)
			})

			t.Run("verify that objects exist after compaction", func(t *testing.T) {
				reader := bucket.ReaderRoaringSetRange()
				defer reader.Close()

				bm, err := reader.Read(context.Background(), key, filters.OperatorEqual)
				require.NoError(t, err)

				if segments == 5 {
					assert.Equal(t, 0, bm.GetCardinality())
				} else if segments == 6 {
					assert.Equal(t, 1, bm.GetCardinality())
				} else {
					assert.Equal(t, 2, bm.GetCardinality())
				}
			})
		})
	}
}

// Regression test to verify overwritten buffer in pread segment cursor.
// Since pread segment cursor reuses buffer to read data for single key, advancing to next
// key overwrites data loaded for previous key, therefore it is important to make sure data is utilized
// before moving to next key. As deletions are stored only in key 0 but merged with other keys' additions
// during compaction, compactor was improved to clone deletions before advancing from key 0.
//
// Data in this test come from compactionRoaringSetRangeStrategy_Random test for which test failed.
func compactionRoaringSetRangeStrategy_BugfixOverwrittenBuffer(ctx context.Context, t *testing.T, opts []BucketOption) {
	b, err := NewBucketCreator().NewBucket(ctx, t.TempDir(), "", nullLogger(), nil,
		cyclemanager.NewCallbackGroupNoop(), cyclemanager.NewCallbackGroupNoop(), opts...)
	require.Nil(t, err)
	defer b.Shutdown(testCtx())
	// so big it effectively never triggers as part of this test
	b.SetMemtableThreshold(1e9)

	maxKey := uint64(100)
	additions := []map[uint64]uint64{
		{582: 31, 2457: 1, 7014: 84, 10074: 19, 15857: 21, 17389: 59, 20641: 39, 22654: 31, 26615: 68, 29628: 77, 34629: 88, 36259: 41, 38356: 40, 41907: 12, 47389: 9, 54234: 67, 57477: 48, 63860: 52, 65936: 73, 66936: 83, 68719: 93, 69983: 73, 73312: 87, 79075: 87, 79750: 80, 84822: 62, 89404: 34, 92677: 8, 92920: 99, 114337: 93, 120795: 3, 137241: 65, 137566: 57, 141451: 12, 141590: 73, 141737: 1, 146854: 11, 157621: 7, 158380: 48, 168107: 22, 174002: 34, 174061: 12, 175938: 32, 176461: 28, 180894: 27, 185953: 99, 188827: 7, 189017: 33, 190976: 65, 193880: 9, 195549: 44, 201959: 39, 215286: 1, 227231: 70, 227853: 19, 233336: 60, 245655: 97, 247745: 2, 250395: 53, 255416: 85, 259850: 7, 261589: 67, 267553: 18, 276859: 5, 281001: 36, 281342: 89, 286924: 69, 289336: 97, 304796: 89, 308377: 83, 315617: 44, 319441: 65, 334869: 89, 349908: 40, 353155: 69, 354711: 27, 358443: 54, 364522: 57, 372484: 83, 372883: 5, 377380: 75, 378862: 63, 381456: 4, 382128: 5, 398907: 36, 403740: 68, 412506: 69, 418014: 67, 418588: 36, 419184: 30, 419861: 47, 420078: 57, 421345: 44, 422573: 2, 428339: 23, 429818: 37, 438506: 67, 443207: 71, 445249: 61, 448140: 47, 450461: 23, 451021: 40, 465803: 60, 468608: 34, 471398: 21, 471871: 57, 472561: 35, 477304: 57, 482443: 9, 482743: 1, 490800: 62, 494687: 64, 495412: 60, 504717: 40, 506351: 15, 506701: 96, 507051: 14, 508263: 35, 514357: 1, 514572: 9, 519761: 13, 521475: 1, 523911: 67, 523958: 24, 525325: 44, 532566: 71, 534416: 42, 537652: 48, 538026: 21, 538831: 69, 545807: 3, 546217: 22, 553210: 66, 556123: 94, 557398: 38, 565813: 87, 568596: 70, 570487: 26, 573210: 9, 580573: 21, 587063: 11, 589574: 79, 594780: 25, 595905: 52, 596992: 20, 597223: 72, 599964: 80, 621697: 25, 622681: 78, 626797: 61, 636896: 75, 638507: 67, 644633: 11, 645102: 76, 645249: 57, 647232: 50, 648061: 32, 655584: 16, 662134: 92, 662870: 51, 664021: 70, 664947: 53, 673987: 56, 674080: 14, 676228: 4, 677240: 84, 684016: 6, 684662: 63, 692925: 18, 693473: 90, 695394: 1, 701424: 29, 711145: 80, 719341: 54, 725949: 99, 739548: 15, 742095: 12, 744237: 4, 745522: 38, 748390: 2, 752683: 48, 764642: 8, 764744: 42, 778885: 78, 783438: 72, 783725: 32, 786208: 90, 793675: 74, 804884: 18, 808337: 69, 808850: 59, 811400: 18, 812434: 64, 813132: 74, 813638: 25, 818764: 63, 819782: 14, 819912: 79, 820983: 7, 821591: 89, 826149: 68, 831816: 1, 832127: 62, 833326: 71, 833399: 15, 834476: 5, 836818: 20, 842899: 76, 843875: 99, 847120: 90, 847236: 38, 855452: 73, 864728: 9, 865576: 16, 867235: 90, 874232: 86, 875925: 66, 882191: 62, 905422: 92, 906540: 17, 912249: 88, 912491: 31, 943672: 17, 949369: 64, 950691: 15, 953102: 33, 963659: 11, 970330: 62, 970821: 19, 975728: 60, 979035: 44, 981544: 67, 987386: 98, 989328: 36, 995279: 89},
		{556: 42, 1929: 87, 2090: 78, 2522: 5, 3908: 29, 10292: 63, 11934: 29, 12165: 48, 12614: 6, 12741: 44, 16143: 39, 16189: 16, 16859: 8, 18436: 40, 18735: 94, 18884: 25, 19843: 64, 21477: 53, 21927: 60, 24427: 16, 25665: 37, 26464: 21, 28448: 7, 29261: 9, 30539: 63, 32052: 42, 34871: 3, 35661: 8, 39015: 98, 39828: 75, 40404: 44, 41893: 73, 44348: 55, 46066: 98, 49015: 95, 49939: 24, 50607: 82, 51143: 4, 51556: 59, 51725: 6, 52673: 25, 54255: 57, 55139: 89, 55731: 5, 57361: 5, 57587: 84, 57666: 40, 59288: 49, 60448: 2, 61140: 92, 61621: 19, 62666: 29, 64023: 7, 67178: 80, 67560: 7, 69056: 17, 69847: 66, 70354: 53, 73765: 84, 74839: 93, 76108: 88, 76641: 4, 79954: 58, 80684: 61, 81779: 12, 82604: 0, 83696: 63, 84844: 60, 85213: 3, 89178: 53, 94228: 81, 94325: 57, 96947: 90, 98172: 69, 100058: 2, 101090: 69, 103769: 46, 105506: 92, 106996: 27, 107041: 70, 107440: 97, 108664: 59, 111724: 66, 112341: 88, 113543: 43, 115509: 61, 115562: 82, 120568: 17, 120913: 59, 122549: 32, 123819: 65, 124945: 39, 125770: 78, 129760: 91, 131788: 80, 132257: 70, 132843: 55, 133124: 10, 133645: 37, 134313: 34, 135084: 66, 136805: 96, 137340: 86, 138452: 93, 139825: 42, 139901: 52, 141056: 47, 141946: 95, 143782: 82, 144813: 12, 144988: 33, 146215: 26, 147526: 44, 148114: 14, 150797: 36, 151344: 29, 151752: 14, 152123: 12, 152366: 39, 153551: 69, 154340: 83, 160247: 10, 160538: 34, 161321: 48, 162746: 81, 163083: 8, 166246: 35, 166427: 73, 167370: 77, 170519: 51, 170977: 74, 170989: 77, 171601: 17, 172584: 35, 175825: 31, 176517: 36, 176543: 44, 178539: 73, 179579: 18, 179885: 19, 180301: 65, 180827: 37, 186639: 25, 188285: 46, 189236: 83, 190679: 83, 190872: 21, 191361: 53, 192654: 58, 195186: 56, 196043: 80, 196083: 18, 196157: 37, 196286: 2, 196539: 13, 197904: 45, 198578: 93, 198799: 15, 199253: 5, 204045: 65, 205000: 11, 205091: 84, 205859: 53, 207201: 81, 207407: 37, 209812: 46, 210970: 61, 211680: 33, 212009: 92, 213579: 86, 214882: 49, 217456: 2, 217693: 24, 221590: 77, 223766: 49, 224892: 61, 225984: 73, 230083: 97, 233015: 88, 235641: 6, 235984: 27, 238950: 29, 239328: 4, 240803: 43, 240904: 90, 241069: 79, 243321: 72, 245740: 96, 248387: 50, 248906: 11, 248917: 25, 249462: 47, 250268: 64, 254342: 74, 254975: 29, 257002: 69, 258317: 60, 260037: 83, 260775: 77, 261360: 22, 261458: 82, 266419: 63, 268315: 54, 274845: 47, 275153: 70, 275268: 48, 277489: 45, 277729: 61, 277798: 76, 277917: 40, 279463: 26, 279501: 56, 281123: 32, 282810: 99, 282832: 83, 283319: 2, 288491: 14, 289713: 51, 292441: 50, 292994: 6, 297789: 31, 299480: 47, 300270: 26, 303220: 60, 303998: 2, 304959: 6, 306655: 85, 307353: 63, 307496: 56, 307674: 83, 309754: 77, 309795: 0, 309938: 42, 311884: 86, 320379: 2, 320570: 40, 320608: 40, 326719: 40, 327601: 0, 328231: 53, 329456: 70, 330403: 71, 334787: 10, 336316: 58, 336766: 57, 336837: 80, 339109: 89, 339135: 77, 339840: 28, 340616: 18, 342396: 26, 343451: 17, 346573: 24, 346798: 77, 347182: 53, 349340: 73, 351484: 71, 351501: 84, 352038: 38, 352675: 80, 352964: 70, 353980: 67, 355200: 79, 355434: 90, 355629: 62, 355789: 83, 356394: 17, 356486: 59, 356783: 27, 356856: 34, 358588: 83, 358888: 76, 359701: 81, 360053: 85, 360206: 86, 360522: 80, 361274: 56, 363302: 99, 365867: 22, 366103: 12, 366779: 22, 368496: 84, 369262: 92, 371188: 61, 372863: 55, 373853: 96, 373975: 38, 377399: 0, 379796: 60, 381519: 87, 382434: 41, 384635: 79, 385074: 4, 385361: 12, 386388: 22, 386558: 70, 388000: 10, 390373: 90, 391106: 18, 391555: 78, 394091: 26, 394175: 13, 394240: 65, 396231: 75, 396596: 57, 398463: 72, 398517: 87, 398910: 59, 399363: 10, 402421: 99, 406169: 89, 411309: 74, 412308: 91, 412937: 16, 413814: 68, 414324: 81, 414521: 70, 414819: 4, 416955: 95, 420400: 64, 421770: 18, 422533: 87, 424069: 11, 424078: 26, 426124: 10, 428478: 35, 429913: 35, 429944: 27, 429955: 28, 430255: 63, 432974: 94, 434675: 33, 435184: 93, 437892: 61, 437932: 97, 439446: 94, 441344: 13, 441617: 74, 442721: 93, 442820: 81, 443200: 23, 443571: 1, 445216: 39, 448677: 30, 451938: 20, 452299: 80, 453183: 76, 453591: 43, 453988: 81, 454836: 90, 454908: 35, 456242: 48, 456702: 55, 457738: 32, 460827: 68, 463895: 36, 464650: 27, 464853: 39, 464994: 48, 468394: 20, 469508: 4, 470363: 37, 473019: 90, 474784: 21, 475502: 72, 476345: 91, 477441: 26, 478416: 63, 478591: 91, 479592: 28, 480764: 47, 482385: 16, 483784: 53, 484443: 0, 486715: 13, 487869: 76, 487918: 71, 490913: 22, 490935: 24, 493289: 93, 494013: 55, 496235: 32, 496337: 96, 496619: 7, 496800: 47, 498644: 70, 501955: 1, 502641: 97, 503108: 91, 504655: 32, 505345: 9, 505644: 24, 507678: 73, 507931: 23, 508455: 25, 509439: 83, 509810: 42, 511646: 3, 512856: 3, 514656: 37, 514689: 96, 514820: 15, 514853: 80, 515196: 94, 515891: 49, 517463: 10, 518513: 13, 520455: 22, 521635: 56, 524886: 70, 524980: 96, 525267: 24, 525907: 96, 526435: 84, 530003: 36, 530590: 28, 533627: 77, 533996: 86, 534005: 73, 534024: 83, 535260: 52, 536384: 96, 536762: 26, 539783: 19, 544941: 20, 546042: 57, 548084: 4, 548495: 31, 549317: 68, 549824: 20, 550005: 54, 550551: 73, 551805: 69, 552193: 23, 553261: 0, 557853: 97, 558651: 88, 561647: 34, 563626: 46, 566205: 85, 566623: 37, 566812: 36, 567389: 18, 567623: 4, 573626: 73, 573821: 13, 573967: 96, 575271: 94, 576665: 40, 579965: 45, 580222: 76, 581892: 4, 584918: 73, 586066: 20, 586119: 72, 590645: 18, 597250: 81, 597667: 3, 598059: 55, 599858: 68, 600572: 46, 603053: 4, 604060: 25, 606866: 42, 606929: 53, 609214: 26, 609675: 98, 609981: 11, 610055: 96, 610063: 73, 610426: 84, 610625: 13, 611569: 14, 613895: 19, 614065: 49, 614872: 42, 618084: 71, 618347: 34, 619259: 45, 619374: 24, 619914: 79, 620108: 11, 622534: 38, 624525: 24, 624802: 72, 625326: 16, 626628: 68, 627339: 12, 627393: 6, 628378: 74, 628923: 92, 630023: 68, 632141: 50, 633421: 39, 634273: 45, 634661: 46, 634790: 22, 635817: 27, 636409: 66, 636431: 30, 637743: 59, 640650: 77, 640887: 79, 641218: 86, 642080: 90, 643865: 15, 644272: 43, 645666: 84, 646106: 78, 646978: 70, 647541: 37, 650879: 38, 651263: 51, 651581: 60, 653623: 89, 654276: 2, 654769: 93, 657039: 35, 657181: 43, 657750: 56, 658645: 48, 659503: 41, 659750: 73, 660420: 68, 661926: 7, 664747: 17, 665133: 0, 668942: 94, 670854: 73, 673813: 46, 674536: 20, 677534: 56, 678539: 62, 680304: 69, 680423: 34, 681499: 55, 682809: 5, 682984: 22, 683188: 97, 683755: 8, 686379: 86, 686414: 30, 688609: 32, 689193: 88, 689834: 16, 689910: 79, 692825: 76, 693283: 66, 694118: 59, 695067: 76, 695685: 82, 697129: 62, 697890: 59, 699737: 99, 699830: 65, 702619: 54, 702712: 20, 706647: 3, 706800: 32, 709276: 97, 712284: 9, 715200: 32, 716270: 36, 717118: 31, 717786: 72, 718712: 58, 720005: 2, 720965: 57, 721770: 64, 721835: 89, 724545: 92, 725604: 43, 727008: 25, 732183: 77, 734947: 41, 737066: 76, 737321: 51, 737342: 5, 738840: 32, 738904: 32, 740760: 13, 743124: 17, 746716: 39, 747367: 68, 751529: 47, 752170: 60, 752757: 56, 753792: 7, 759656: 10, 760489: 70, 760944: 34, 761419: 75, 761570: 94, 761572: 34, 762682: 91, 762701: 38, 763172: 95, 763884: 21, 766708: 52, 768802: 6, 770483: 66, 771175: 9, 771285: 70, 772790: 70, 774912: 17, 775198: 58, 775273: 92, 775724: 23, 775907: 68, 776262: 85, 777096: 88, 777200: 90, 778932: 59, 779043: 44, 779753: 65, 782147: 99, 784200: 58, 784745: 73, 785135: 46, 785654: 23, 786223: 70, 787467: 33, 788198: 61, 789681: 81, 789814: 43, 790083: 43, 790257: 15, 790506: 98, 792019: 40, 795960: 71, 796760: 98, 797281: 96, 797982: 30, 798702: 85, 801428: 58, 802913: 93, 803042: 50, 803358: 24, 804312: 96, 805815: 94, 806164: 83, 807203: 64, 808838: 79, 809153: 7, 810941: 46, 811237: 7, 813198: 31, 815993: 83, 816779: 23, 816853: 51, 817044: 27, 817454: 86, 820601: 12, 822009: 3, 822103: 40, 824528: 84, 825128: 35, 825726: 77, 825967: 50, 828432: 36, 828662: 83, 828964: 1, 830985: 46, 834175: 15, 834773: 84, 834922: 62, 836598: 99, 843548: 46, 845534: 76, 845743: 1, 845963: 17, 847407: 52, 847431: 13, 848451: 34, 855235: 61, 857547: 9, 858901: 23, 867204: 6, 870558: 20, 871497: 68, 872621: 15, 875148: 67, 875763: 96, 878731: 61, 880861: 94, 881185: 8, 881535: 61, 883257: 66, 883297: 55, 883853: 72, 885307: 53, 886034: 70, 886703: 3, 887159: 71, 887346: 78, 887840: 44, 890576: 50, 892247: 48, 892335: 94, 892706: 41, 893667: 40, 895266: 64, 895452: 78, 895537: 69, 896929: 61, 897351: 27, 898093: 65, 898281: 67, 899680: 6, 900056: 53, 900630: 45, 903669: 26, 903907: 74, 909758: 47, 909775: 26, 914586: 1, 914671: 58, 916915: 90, 919245: 22, 922157: 63, 924937: 93, 925563: 28, 925762: 71, 925915: 22, 926626: 70, 928713: 73, 931411: 32, 932040: 19, 932433: 16, 932712: 51, 934707: 62, 936601: 44, 937207: 35, 937413: 44, 939328: 41, 940271: 31, 941832: 51, 943928: 2, 949561: 33, 949643: 84, 953454: 12, 953577: 48, 953790: 64, 956412: 42, 956736: 7, 957733: 96, 959630: 34, 959884: 93, 961126: 14, 961426: 62, 961479: 60, 962153: 77, 962250: 60, 964030: 98, 966876: 45, 967357: 65, 968048: 49, 968175: 42, 968878: 60, 969289: 38, 969878: 4, 970570: 13, 971187: 71, 971607: 1, 974056: 59, 974098: 63, 974834: 34, 975286: 59, 976309: 23, 977038: 77, 977510: 48, 978883: 55, 978885: 67, 981089: 95, 981695: 22, 982690: 95, 983348: 14, 983849: 37, 986240: 53, 986292: 4, 988121: 15, 988620: 92, 992366: 10, 992907: 95, 993168: 82, 993443: 62, 995180: 46, 995716: 70, 995960: 83, 997045: 88},
		{1335: 43, 1788: 48, 2902: 22, 3121: 5, 5555: 68, 7573: 18, 8590: 46, 12273: 7, 15262: 89, 16503: 77, 18155: 86, 23792: 91, 24338: 26, 24474: 29, 24821: 45, 27005: 57, 31420: 19, 31432: 23, 31748: 98, 36122: 24, 37255: 33, 37815: 61, 40727: 64, 42468: 18, 43098: 85, 43313: 82, 44320: 17, 45427: 19, 45504: 52, 45728: 61, 47022: 18, 48752: 33, 48847: 79, 48899: 63, 50000: 89, 50124: 12, 51314: 4, 54592: 33, 54997: 9, 55230: 91, 56247: 67, 57871: 66, 58854: 92, 67347: 38, 69640: 84, 70338: 11, 74616: 27, 75077: 6, 76111: 86, 79943: 38, 82466: 0, 84717: 42, 86107: 62, 86128: 86, 86532: 82, 86691: 8, 87906: 52, 91304: 9, 91583: 40, 93301: 61, 94802: 51, 96196: 2, 97022: 8, 97529: 18, 97835: 18, 98664: 48, 99251: 83, 99622: 63, 99667: 30, 101886: 5, 102154: 70, 102447: 57, 102457: 41, 105717: 27, 108022: 65, 108590: 15, 108843: 3, 111299: 17, 111972: 60, 117197: 27, 117775: 38, 118122: 28, 118269: 75, 119924: 59, 120127: 19, 120946: 56, 126087: 60, 127003: 12, 127860: 80, 130694: 1, 132668: 30, 136445: 49, 136496: 52, 138794: 86, 138885: 30, 139949: 15, 142343: 26, 142632: 51, 143202: 20, 145783: 22, 145844: 88, 146361: 79, 147895: 1, 148036: 46, 151005: 12, 155150: 9, 155431: 59, 156272: 13, 156601: 81, 156829: 46, 157825: 0, 157984: 71, 159374: 71, 160628: 55, 162502: 85, 162887: 38, 164551: 8, 166207: 38, 168933: 91, 170410: 24, 171358: 27, 172496: 65, 172695: 78, 174463: 52, 176039: 96, 176404: 83, 177358: 9, 177617: 64, 179004: 34, 179026: 4, 179238: 40, 179591: 32, 181144: 37, 181998: 53, 182376: 91, 185037: 92, 189408: 76, 189628: 33, 190375: 62, 191723: 23, 192271: 73, 193891: 16, 195157: 3, 195751: 26, 196388: 16, 199254: 66, 199440: 84, 201299: 76, 202814: 6, 204600: 69, 204931: 85, 204959: 80, 211379: 43, 211887: 28, 214305: 56, 216531: 48, 218311: 75, 218514: 85, 218660: 89, 219998: 20, 220557: 95, 220975: 24, 222505: 18, 224150: 67, 226389: 3, 226505: 0, 227573: 48, 228064: 40, 228310: 94, 231401: 28, 232316: 95, 234559: 54, 235156: 86, 235663: 77, 236333: 53, 239202: 69, 240667: 96, 240952: 0, 241029: 77, 244757: 59, 245082: 7, 247385: 86, 248016: 19, 248711: 36, 249511: 93, 252101: 8, 252484: 89, 253359: 12, 254488: 15, 257169: 47, 258316: 64, 263957: 55, 266490: 35, 266632: 1, 266758: 82, 267162: 11, 268088: 51, 269762: 18, 271116: 28, 271370: 60, 272648: 98, 273473: 12, 274646: 30, 276271: 15, 276540: 27, 276566: 68, 278953: 89, 280221: 51, 284383: 87, 284798: 76, 284866: 53, 285757: 94, 289898: 69, 292499: 16, 292565: 18, 297086: 52, 298832: 40, 300792: 43, 303096: 43, 303507: 4, 303626: 26, 304590: 57, 304949: 90, 306243: 10, 309270: 42, 316948: 6, 317599: 68, 317848: 70, 319518: 52, 321335: 67, 321521: 6, 321628: 55, 323959: 6, 325914: 25, 328502: 76, 330179: 49, 330433: 25, 331781: 25, 333674: 34, 334604: 83, 334607: 3, 336816: 50, 336901: 32, 337185: 36, 338826: 40, 339846: 3, 340445: 19, 340854: 86, 343974: 74, 350352: 80, 352253: 54, 355420: 10, 359841: 15, 360783: 83, 361411: 44, 362986: 91, 363821: 86, 363979: 37, 364253: 42, 365116: 55, 368895: 62, 369664: 59, 372578: 84, 374149: 17, 374391: 56, 374548: 96, 375111: 39, 376373: 55, 376649: 90, 377158: 71, 380732: 99, 382042: 66, 383503: 49, 383584: 15, 385740: 81, 386617: 25, 392482: 0, 395323: 83, 395784: 10, 398704: 11, 399405: 49, 400361: 11, 400729: 61, 401457: 42, 402526: 49, 403907: 97, 404296: 40, 404468: 8, 405951: 99, 408673: 55, 409633: 86, 411599: 39, 412059: 22, 420008: 66, 420447: 29, 425070: 47, 426710: 80, 430261: 11, 434850: 53, 437932: 28, 438663: 53, 439924: 42, 440331: 44, 444243: 72, 445251: 69, 446112: 40, 450807: 3, 451087: 45, 452421: 7, 452794: 66, 453257: 39, 464397: 34, 468127: 11, 468377: 90, 470576: 78, 472098: 69, 472322: 23, 477969: 14, 481103: 41, 481162: 97, 483536: 60, 483948: 5, 486591: 11, 487667: 75, 487812: 70, 489352: 18, 492833: 77, 493079: 7, 495390: 19, 496804: 0, 497330: 6, 500895: 52, 502910: 16, 503234: 70, 508892: 25, 509573: 99, 513672: 59, 516400: 55, 516736: 29, 518060: 56, 519956: 34, 521125: 11, 521702: 7, 523686: 11, 523752: 94, 524276: 28, 525010: 13, 526641: 25, 526673: 53, 527213: 67, 528283: 38, 528452: 35, 528742: 29, 530479: 17, 530558: 58, 532609: 92, 532857: 37, 533066: 73, 533676: 96, 533957: 3, 534800: 90, 535733: 37, 536705: 99, 537917: 38, 539178: 4, 540411: 41, 540749: 37, 542948: 82, 543979: 2, 548263: 83, 549059: 47, 550243: 80, 553885: 38, 556855: 50, 559903: 51, 560219: 49, 563869: 4, 564474: 46, 566533: 71, 566944: 60, 569802: 5, 570322: 1, 571633: 40, 572228: 61, 575163: 58, 576038: 92, 576278: 98, 577167: 60, 578011: 81, 578502: 56, 582462: 11, 584828: 79, 587601: 57, 587654: 65, 590225: 86, 591917: 97, 592104: 16, 593365: 83, 593852: 25, 599264: 24, 601191: 96, 603568: 39, 603619: 50, 604353: 60, 605673: 90, 606623: 68, 608910: 89, 611803: 5, 611841: 10, 612709: 66, 613636: 86, 614548: 41, 617031: 63, 617377: 6, 618936: 23, 619222: 33, 619376: 93, 619671: 38, 622241: 40, 624293: 42, 624356: 16, 627015: 42, 629556: 37, 629657: 73, 633350: 30, 634510: 97, 636133: 77, 637604: 15, 638669: 20, 639172: 35, 640322: 89, 640835: 17, 642314: 18, 642696: 57, 645423: 38, 646223: 78, 652957: 77, 655881: 71, 659020: 45, 659302: 83, 659761: 25, 662031: 40, 662863: 46, 663429: 86, 664241: 70, 664780: 64, 666337: 64, 668289: 86, 668895: 29, 670001: 24, 671217: 90, 671355: 29, 673385: 98, 674603: 72, 676435: 2, 677841: 72, 677913: 72, 678569: 15, 680514: 43, 680769: 91, 681942: 81, 682494: 16, 683743: 7, 685387: 65, 686556: 31, 687054: 25, 688169: 27, 690262: 92, 690270: 15, 693683: 74, 697817: 37, 701470: 73, 703049: 21, 708738: 29, 708965: 9, 709248: 66, 709337: 36, 709752: 29, 709766: 8, 709937: 89, 711943: 44, 712424: 29, 713056: 62, 714227: 79, 714847: 74, 717015: 61, 719310: 71, 723390: 7, 724823: 28, 725034: 41, 725801: 77, 727132: 40, 729486: 54, 732716: 45, 734666: 82, 735382: 7, 738272: 81, 742499: 85, 742519: 63, 742532: 98, 742918: 51, 745078: 90, 748399: 78, 752190: 64, 752399: 65, 757528: 94, 761499: 66, 763597: 56, 765666: 25, 766026: 48, 766449: 48, 766511: 7, 767723: 30, 769249: 6, 769599: 69, 770515: 18, 773412: 45, 773492: 9, 776185: 48, 779400: 27, 779971: 85, 781041: 41, 781176: 87, 784893: 26, 786955: 15, 790706: 12, 790821: 90, 791190: 16, 792032: 53, 794093: 0, 794609: 19, 796803: 90, 796819: 27, 797566: 35, 798148: 82, 799325: 13, 800404: 82, 806263: 73, 806677: 45, 808900: 75, 813668: 32, 814028: 83, 814466: 39, 816348: 63, 817702: 59, 817767: 16, 819169: 70, 820302: 57, 822724: 70, 824106: 27, 826680: 5, 830378: 51, 832365: 82, 837031: 16, 837479: 32, 838343: 83, 838894: 46, 841495: 6, 843805: 65, 844590: 11, 845704: 9, 845728: 14, 846595: 71, 847364: 86, 852700: 2, 854374: 56, 857609: 67, 859942: 88, 861390: 6, 865190: 2, 865642: 7, 866702: 42, 867147: 28, 868031: 93, 868465: 39, 868731: 82, 869485: 84, 872238: 18, 872663: 82, 873005: 32, 874555: 73, 875160: 96, 875293: 80, 876187: 54, 876961: 43, 879457: 53, 880223: 49, 880860: 67, 881327: 16, 883751: 3, 885097: 16, 886785: 26, 889284: 47, 889525: 68, 889860: 36, 891545: 46, 893905: 7, 894557: 19, 895924: 43, 896582: 56, 897299: 96, 901024: 43, 905196: 53, 907443: 54, 910630: 38, 912748: 75, 915067: 32, 917506: 28, 919668: 53, 920575: 35, 922142: 27, 924297: 88, 926503: 57, 928497: 92, 929160: 75, 932110: 49, 932114: 10, 942928: 15, 943552: 69, 945335: 32, 945558: 91, 945594: 74, 948319: 37, 949689: 4, 952132: 29, 952541: 10, 952809: 34, 954961: 76, 955500: 45, 956852: 53, 957192: 96, 958658: 41, 961082: 58, 961302: 65, 963013: 30, 963180: 9, 965592: 64, 965886: 56, 966073: 63, 968916: 27, 970289: 29, 971684: 59, 974027: 10, 976220: 50, 978734: 14, 980687: 66, 981069: 63, 981159: 75, 986089: 78, 986252: 29, 986384: 8, 987231: 99, 988360: 81, 988400: 99, 990370: 67, 990613: 81, 990965: 59, 997768: 95},
		{5692: 36, 5791: 32, 6158: 42, 6834: 53, 7277: 72, 13631: 65, 16436: 43, 17918: 34, 18441: 32, 19055: 36, 23336: 58, 24416: 25, 27233: 56, 29382: 37, 30556: 84, 32950: 26, 36909: 67, 36932: 24, 39564: 42, 42490: 39, 42525: 8, 43510: 35, 44632: 45, 44754: 98, 44828: 16, 45772: 4, 46188: 79, 46234: 59, 47280: 8, 47430: 44, 47527: 95, 48226: 20, 49005: 32, 49483: 26, 51845: 65, 52623: 26, 57331: 11, 61393: 38, 61789: 70, 64944: 45, 67152: 86, 67542: 53, 67599: 85, 69172: 27, 72267: 56, 76035: 64, 79632: 11, 81610: 28, 81758: 72, 83003: 28, 85266: 48, 86211: 83, 89029: 31, 89344: 86, 90228: 74, 94951: 36, 97763: 75, 103999: 19, 106558: 38, 106691: 58, 109813: 11, 111249: 12, 112107: 56, 113014: 65, 114506: 79, 115164: 36, 115373: 78, 117398: 41, 117448: 40, 118800: 79, 121603: 7, 125773: 1, 127533: 35, 129980: 58, 130058: 44, 130237: 0, 131241: 72, 131588: 46, 134651: 40, 134753: 42, 135290: 24, 136027: 56, 138574: 19, 143121: 71, 144101: 41, 147081: 0, 147765: 96, 153275: 44, 153576: 36, 155007: 7, 155105: 63, 156572: 26, 159431: 97, 159571: 50, 160124: 64, 160410: 35, 160941: 1, 162087: 32, 163482: 85, 164899: 47, 166442: 76, 172472: 51, 176974: 11, 177727: 17, 178481: 43, 180859: 99, 182562: 7, 184875: 89, 185935: 96, 187389: 76, 187842: 72, 190795: 28, 191950: 67, 192333: 80, 194245: 81, 197614: 42, 204852: 47, 207556: 56, 210492: 76, 211006: 91, 213217: 78, 213824: 83, 217166: 92, 221277: 63, 223144: 13, 224054: 13, 226583: 33, 227297: 12, 238363: 99, 238846: 28, 242153: 3, 242828: 78, 243936: 84, 244243: 63, 245115: 54, 246895: 77, 247151: 32, 250964: 30, 252353: 15, 252537: 19, 254894: 75, 257230: 49, 259170: 30, 259359: 73, 262588: 74, 264626: 60, 269627: 94, 272175: 26, 272640: 55, 274690: 36, 277993: 62, 286173: 98, 290549: 78, 291417: 82, 291594: 69, 294141: 53, 296733: 19, 297120: 7, 301292: 26, 301338: 98, 301637: 99, 304756: 30, 306590: 15, 307188: 8, 308459: 46, 316103: 50, 317017: 62, 317180: 30, 317559: 45, 317627: 55, 317903: 79, 320478: 51, 325723: 38, 327678: 67, 327693: 55, 328915: 86, 330690: 5, 335674: 41, 337980: 54, 338780: 15, 340558: 49, 346730: 36, 347382: 86, 350146: 88, 350337: 33, 351406: 56, 351552: 12, 357584: 12, 358238: 18, 358414: 41, 361887: 75, 362885: 72, 367002: 96, 367910: 80, 370394: 27, 370677: 98, 379234: 70, 383227: 89, 384698: 84, 390968: 41, 393968: 29, 398056: 30, 398607: 83, 400916: 61, 401314: 33, 405639: 40, 409000: 92, 409230: 64, 410873: 88, 417293: 22, 421253: 92, 421818: 73, 422884: 14, 424169: 17, 429980: 46, 433669: 98, 438692: 51, 439583: 48, 439697: 45, 441368: 99, 443093: 0, 443617: 53, 445237: 51, 445543: 93, 445838: 51, 448226: 54, 450659: 61, 450661: 24, 451909: 74, 456541: 71, 456763: 96, 457675: 9, 459230: 60, 460730: 74, 467425: 40, 471515: 94, 472543: 42, 473059: 68, 476002: 67, 476897: 87, 476944: 16, 477845: 76, 478153: 12, 481174: 66, 483918: 19, 486170: 29, 490044: 20, 491002: 38, 492244: 18, 492386: 87, 492505: 17, 493107: 49, 493125: 91, 496834: 91, 498345: 1, 499312: 67, 500302: 4, 500750: 53, 501343: 93, 502827: 0, 503614: 52, 504261: 8, 509365: 25, 512530: 81, 513242: 81, 514131: 26, 520223: 27, 520693: 34, 526780: 82, 528171: 9, 528706: 43, 530726: 0, 532076: 46, 532599: 61, 533867: 81, 535704: 91, 536458: 3, 536610: 76, 536789: 86, 537732: 63, 539100: 38, 539609: 31, 540648: 93, 542566: 18, 543765: 44, 546096: 89, 546409: 59, 547891: 73, 549590: 16, 550403: 24, 551191: 41, 553952: 85, 556178: 56, 556808: 95, 558712: 70, 558847: 17, 560726: 43, 565495: 73, 566573: 29, 569040: 10, 569665: 37, 569720: 42, 574102: 77, 577239: 71, 577422: 97, 578615: 99, 579292: 32, 579487: 43, 581671: 48, 582005: 25, 588052: 15, 591513: 22, 593964: 20, 600793: 23, 601711: 84, 603132: 35, 604442: 87, 604866: 58, 606611: 17, 607466: 30, 610596: 27, 612174: 37, 612486: 8, 614891: 6, 617504: 2, 624549: 78, 624957: 48, 627526: 73, 630133: 42, 631582: 76, 631950: 18, 634821: 40, 635020: 54, 635556: 32, 637862: 68, 639467: 50, 639755: 48, 639832: 46, 642616: 55, 644929: 29, 646006: 1, 647235: 90, 649760: 23, 651204: 93, 651991: 75, 652144: 88, 656345: 18, 659135: 94, 659293: 78, 659808: 50, 660870: 59, 661709: 24, 666124: 45, 668285: 62, 669425: 85, 671678: 74, 672842: 99, 673638: 12, 673803: 58, 674626: 56, 674763: 76, 676487: 13, 679847: 80, 682187: 32, 682690: 23, 682724: 36, 685148: 85, 685253: 38, 688048: 19, 689841: 62, 690439: 56, 694808: 68, 694936: 17, 695041: 29, 699706: 26, 700462: 60, 702025: 44, 702920: 15, 703331: 50, 706135: 78, 706156: 92, 706408: 71, 707425: 23, 707682: 35, 709528: 21, 711539: 79, 721337: 90, 723375: 34, 726355: 51, 728727: 96, 729653: 5, 730596: 77, 732511: 34, 733045: 30, 733566: 77, 734559: 80, 734636: 39, 735716: 12, 740950: 60, 747058: 57, 747279: 37, 748052: 89, 749952: 24, 754624: 18, 755567: 96, 758607: 83, 759355: 4, 760156: 24, 772596: 89, 774248: 89, 774327: 64, 774425: 62, 776457: 34, 776820: 69, 777257: 2, 786216: 36, 787272: 65, 787604: 34, 787723: 15, 787875: 25, 788116: 69, 791311: 9, 791858: 67, 794785: 58, 794894: 52, 795433: 71, 795589: 60, 796973: 62, 797684: 45, 800026: 41, 806507: 48, 807703: 23, 808095: 53, 808829: 67, 810389: 8, 816672: 74, 818621: 18, 819703: 0, 819868: 8, 824262: 51, 824702: 16, 826754: 94, 827834: 13, 829540: 30, 831103: 35, 833992: 70, 834286: 71, 837843: 97, 838991: 10, 844327: 1, 844604: 64, 848227: 83, 848747: 95, 852952: 87, 859333: 86, 861237: 75, 864199: 62, 864809: 81, 868631: 22, 870725: 13, 871645: 15, 875347: 11, 877680: 8, 878559: 24, 882713: 42, 884222: 81, 885633: 64, 886877: 49, 889165: 42, 892664: 95, 897796: 9, 899063: 67, 900057: 98, 900237: 51, 904604: 6, 906474: 55, 906498: 22, 912934: 85, 914630: 55, 915477: 65, 917797: 29, 918175: 7, 919674: 48, 923127: 68, 925155: 75, 928021: 32, 930689: 10, 932380: 76, 932405: 8, 935055: 61, 935187: 42, 937757: 91, 939189: 56, 939594: 84, 940607: 80, 945688: 21, 947403: 71, 947440: 72, 949002: 70, 949584: 40, 951834: 35, 955944: 9, 958498: 9, 958723: 19, 960331: 24, 961372: 77, 962008: 57, 963839: 2, 965831: 67, 966409: 49, 968484: 26, 970620: 69, 971168: 26, 971906: 55, 972506: 50, 972668: 81, 974723: 2, 975615: 95, 978133: 0, 978301: 65, 978559: 5, 979530: 38, 984208: 57, 985454: 5, 986156: 34, 986834: 24, 989027: 50, 989468: 27, 989569: 39, 990023: 82, 991282: 2, 992612: 48, 994317: 50, 994925: 5, 995029: 76, 998131: 21, 998625: 52, 998675: 21, 999918: 99},
		{2033: 3, 2553: 4, 2596: 84, 2617: 81, 4022: 99, 4632: 82, 5997: 44, 7201: 4, 7971: 7, 8770: 66, 8987: 80, 9179: 44, 11276: 99, 12357: 66, 12779: 40, 12860: 33, 13714: 80, 14105: 31, 14225: 45, 14251: 53, 14744: 67, 17232: 55, 18538: 17, 19650: 57, 19781: 24, 19816: 1, 20147: 33, 20540: 94, 25093: 26, 26394: 40, 27645: 53, 27990: 66, 28000: 12, 28215: 72, 29048: 40, 29209: 93, 30022: 35, 31844: 11, 31893: 39, 33471: 13, 33517: 30, 33612: 13, 33896: 92, 34246: 36, 35922: 8, 37841: 58, 39153: 78, 39258: 75, 41510: 94, 41750: 67, 42272: 72, 43389: 15, 44549: 50, 44632: 64, 44643: 78, 46140: 34, 46151: 82, 49758: 65, 50618: 87, 50680: 41, 50943: 96, 51031: 6, 51344: 48, 51373: 25, 52078: 49, 52697: 10, 53967: 78, 55206: 22, 55367: 52, 56602: 68, 56965: 80, 57402: 78, 57501: 26, 58162: 89, 58971: 52, 59624: 60, 59643: 71, 60431: 83, 60532: 92, 62365: 18, 65199: 18, 66070: 1, 68359: 26, 68753: 28, 69376: 80, 70401: 69, 70654: 96, 70850: 62, 71915: 24, 72285: 87, 72877: 2, 73108: 40, 73617: 59, 74039: 58, 76935: 34, 78328: 58, 78405: 94, 78656: 54, 79017: 7, 80196: 55, 80571: 63, 81270: 6, 82734: 86, 82854: 27, 84775: 71, 85501: 10, 86277: 10, 86361: 47, 86916: 11, 88152: 22, 89618: 23, 90195: 45, 90419: 58, 92833: 34, 95256: 20, 95669: 48, 96128: 13, 96964: 74, 97025: 17, 97094: 84, 97278: 16, 99508: 87, 100996: 89, 101372: 27, 102862: 55, 102995: 91, 106658: 22, 108449: 99, 109185: 91, 109858: 51, 110322: 30, 111124: 10, 112271: 27, 115064: 94, 116592: 4, 117489: 31, 118204: 32, 119469: 51, 120387: 96, 120467: 61, 121542: 81, 121670: 65, 123148: 80, 123322: 22, 123849: 2, 124075: 92, 124609: 9, 124929: 96, 125637: 48, 125933: 15, 126151: 19, 126531: 8, 126626: 75, 126998: 46, 128984: 48, 129277: 66, 129897: 18, 132865: 90, 133085: 23, 135356: 55, 135671: 41, 136647: 52, 137856: 51, 138360: 11, 138480: 77, 141856: 6, 142152: 45, 142669: 22, 143582: 15, 143802: 81, 144018: 58, 144203: 76, 145235: 79, 145639: 59, 145982: 0, 147151: 62, 147726: 19, 149051: 24, 152150: 16, 152700: 58, 153467: 98, 153959: 12, 154563: 77, 154996: 55, 155399: 85, 156605: 67, 156626: 48, 157097: 21, 158741: 28, 159123: 6, 159220: 96, 160326: 73, 163035: 90, 164205: 64, 165114: 97, 166166: 50, 166483: 21, 166877: 72, 168903: 5, 171316: 24, 172227: 63, 172716: 79, 174229: 10, 174231: 19, 175072: 33, 176899: 73, 179190: 17, 180457: 27, 180896: 6, 184415: 16, 184825: 34, 184855: 41, 185051: 15, 185381: 13, 185630: 49, 186352: 58, 186485: 35, 187557: 49, 188539: 43, 188634: 43, 190140: 31, 191909: 54, 193643: 98, 194401: 94, 194760: 47, 195006: 3, 195337: 81, 196854: 70, 197332: 11, 197438: 67, 197701: 88, 198846: 86, 200296: 72, 203107: 79, 203738: 49, 205063: 58, 206265: 55, 208619: 10, 208640: 76, 208673: 1, 209079: 8, 209944: 17, 211321: 65, 211722: 74, 212137: 46, 212247: 30, 212781: 46, 213575: 99, 213758: 85, 214648: 7, 216157: 64, 218238: 77, 218277: 73, 218731: 98, 219108: 43, 219290: 14, 219577: 39, 220589: 91, 222306: 35, 224405: 89, 225989: 19, 226438: 44, 227049: 2, 227241: 86, 227287: 53, 227298: 45, 227751: 21, 227850: 46, 228022: 39, 228067: 98, 228665: 43, 229984: 59, 230450: 73, 230681: 15, 230991: 2, 231535: 6, 231880: 77, 232827: 95, 232933: 12, 234603: 7, 234732: 5, 234904: 82, 235318: 78, 235972: 44, 236561: 88, 237759: 73, 238094: 64, 238162: 92, 239016: 45, 239066: 96, 241361: 86, 241999: 23, 242083: 23, 243559: 4, 243922: 8, 245748: 19, 246485: 85, 247399: 64, 247527: 10, 250702: 55, 252926: 98, 253091: 45, 254212: 84, 258566: 98, 258599: 23, 259079: 12, 260358: 67, 261004: 26, 262222: 85, 264914: 92, 265194: 47, 265581: 79, 266630: 21, 266799: 49, 267753: 49, 269429: 17, 269475: 19, 271572: 43, 271940: 38, 277723: 92, 277736: 33, 279190: 47, 280206: 82, 281080: 46, 281375: 64, 281647: 85, 282380: 10, 282438: 5, 282745: 88, 284149: 27, 284233: 13, 284327: 72, 285883: 96, 286057: 39, 286224: 17, 286563: 24, 288024: 94, 290494: 41, 291174: 70, 292385: 91, 292681: 59, 294003: 97, 294744: 45, 295196: 5, 296264: 85, 296698: 15, 296700: 55, 297262: 95, 297824: 13, 298604: 57, 298735: 51, 301325: 91, 301744: 70, 301811: 59, 307659: 62, 308899: 20, 309955: 82, 310603: 38, 310988: 58, 312405: 36, 314278: 79, 315722: 93, 315784: 88, 316279: 28, 317147: 26, 317286: 32, 319155: 57, 323039: 14, 323209: 96, 325695: 25, 326795: 65, 328295: 85, 328557: 7, 329043: 97, 329180: 0, 329504: 29, 329600: 51, 330069: 86, 331479: 15, 334363: 35, 338150: 83, 338278: 96, 339221: 73, 339267: 59, 339677: 2, 340637: 33, 341186: 63, 342045: 41, 342142: 18, 344219: 38, 344723: 14, 346446: 84, 347894: 70, 348364: 49, 349073: 66, 349703: 41, 351389: 33, 352803: 72, 352860: 56, 353026: 93, 354102: 52, 354547: 64, 355772: 60, 357243: 10, 358019: 78, 358550: 79, 359508: 97, 360316: 57, 360583: 72, 361330: 60, 361800: 79, 362339: 33, 362935: 32, 363307: 28, 364290: 27, 364711: 24, 367110: 18, 368431: 5, 370214: 7, 370341: 86, 371463: 48, 372050: 71, 372393: 71, 374062: 76, 374759: 99, 375955: 41, 376419: 73, 376517: 94, 378504: 29, 378622: 27, 378931: 98, 379003: 42, 379021: 13, 381257: 34, 381958: 65, 382021: 9, 382473: 51, 382605: 35, 384040: 1, 384363: 45, 385690: 27, 387244: 90, 387309: 27, 387823: 82, 388590: 2, 388822: 13, 388947: 12, 390026: 46, 390948: 67, 391214: 37, 391647: 32, 393110: 15, 393235: 18, 393249: 63, 394537: 4, 395611: 71, 397165: 37, 399424: 6, 399580: 97, 399620: 1, 400551: 12, 401626: 1, 402517: 69, 403968: 12, 404271: 27, 406345: 12, 408294: 0, 410175: 45, 412471: 13, 412802: 69, 412846: 45, 413755: 70, 415162: 72, 415396: 31, 416916: 55, 417836: 90, 417949: 74, 418892: 42, 419678: 50, 419731: 22, 420070: 50, 420214: 54, 420815: 44, 420955: 17, 421009: 32, 421308: 57, 421389: 65, 422050: 40, 422090: 25, 422354: 13, 422439: 87, 427587: 78, 427675: 1, 427698: 87, 428397: 51, 428638: 70, 429081: 71, 429382: 89, 429534: 57, 431829: 73, 432348: 89, 432360: 84, 432976: 66, 433854: 11, 434872: 40, 435962: 41, 436351: 68, 436499: 85, 436613: 95, 437449: 45, 438862: 97, 438906: 78, 439522: 59, 439624: 74, 439937: 57, 440371: 4, 441895: 72, 442381: 86, 442916: 87, 444679: 82, 444907: 91, 445265: 1, 445595: 74, 446736: 74, 448512: 34, 449082: 37, 449473: 7, 451164: 66, 452559: 99, 453676: 35, 455258: 90, 455388: 44, 457018: 74, 458746: 91, 459221: 12, 459536: 10, 463317: 69, 464085: 51, 466128: 46, 466762: 10, 467079: 12, 468680: 23, 469734: 59, 469741: 82, 470784: 41, 472784: 3, 473214: 80, 474603: 7, 474768: 66, 475790: 3, 476176: 76, 477111: 19, 479519: 64, 479745: 70, 480644: 83, 481450: 85, 481520: 40, 482221: 84, 483865: 68, 483971: 22, 484913: 91, 485270: 45, 485352: 45, 485650: 55, 486291: 33, 486924: 34, 487933: 88, 488106: 10, 488389: 40, 488401: 57, 489244: 78, 489741: 73, 490196: 51, 490212: 40, 490302: 88, 490564: 20, 492358: 44, 494366: 73, 497452: 1, 498337: 27, 498346: 58, 499565: 5, 500821: 28, 502460: 51, 504901: 55, 505042: 63, 506304: 65, 507760: 12, 508056: 84, 508240: 58, 509210: 49, 509732: 97, 509858: 42, 511633: 91, 512532: 15, 512726: 20, 515052: 63, 516656: 49, 519929: 75, 521186: 57, 521667: 88, 523820: 92, 523934: 75, 524119: 23, 525796: 55, 525902: 82, 527671: 18, 528050: 42, 530186: 53, 530957: 6, 531115: 96, 531652: 97, 531661: 39, 533281: 11, 535519: 41, 535765: 32, 535964: 83, 536687: 32, 537599: 30, 537753: 58, 537770: 40, 537771: 11, 538757: 90, 539376: 94, 541909: 79, 542715: 33, 543322: 44, 543972: 69, 544025: 11, 544029: 35, 544952: 16, 544998: 42, 545920: 62, 548335: 55, 549384: 25, 550379: 59, 550446: 64, 550764: 38, 551121: 59, 551411: 54, 551807: 23, 552378: 94, 553015: 52, 553461: 80, 553480: 39, 555277: 72, 556327: 57, 556408: 40, 556601: 46, 556854: 98, 559061: 68, 560827: 73, 560845: 11, 561289: 44, 562144: 57, 562872: 77, 563527: 52, 564509: 3, 565061: 71, 565588: 50, 565798: 68, 567185: 35, 567792: 38, 569200: 95, 571160: 84, 574908: 56, 574921: 68, 574962: 87, 576115: 38, 576815: 4, 578196: 99, 580343: 11, 582533: 6, 583069: 2, 584246: 31, 585759: 96, 586173: 37, 586692: 69, 587706: 0, 588035: 18, 589762: 92, 591854: 60, 593246: 31, 594093: 10, 595350: 60, 595663: 36, 597283: 35, 598453: 73, 599322: 44, 599546: 91, 599935: 7, 600511: 6, 601635: 37, 601799: 79, 603257: 33, 604497: 23, 604532: 72, 605212: 78, 605524: 6, 610876: 12, 611144: 73, 612197: 95, 612704: 53, 612893: 3, 613296: 3, 614144: 12, 618938: 35, 619194: 95, 620704: 83, 621725: 24, 621866: 72, 622746: 8, 623630: 21, 623938: 67, 624383: 10, 624442: 48, 624814: 9, 625449: 75, 625906: 57, 626452: 74, 628867: 44, 630091: 8, 630628: 86, 631730: 8, 632458: 46, 632731: 12, 633546: 76, 633590: 38, 636159: 87, 636193: 47, 636219: 72, 636572: 23, 636758: 74, 636892: 0, 637044: 27, 640214: 37, 640395: 26, 640551: 64, 642150: 56, 642834: 66, 643932: 5, 644686: 36, 645644: 42, 645908: 64, 646185: 75, 646472: 33, 647620: 19, 647814: 59, 648214: 20, 651049: 29, 651947: 6, 653294: 75, 653383: 29, 654050: 19, 657298: 12, 659957: 89, 660808: 29, 660928: 87, 661793: 99, 662216: 68, 662600: 82, 663113: 8, 665859: 46, 668283: 16, 668518: 19, 668582: 17, 670184: 35, 670669: 45, 671394: 38, 671749: 21, 671997: 58, 673590: 48, 673877: 44, 674308: 42, 675925: 33, 675934: 91, 677921: 31, 678751: 20, 678769: 88, 679661: 87, 680358: 8, 682552: 4, 683698: 0, 683991: 53, 684166: 69, 689358: 30, 689474: 16, 690403: 76, 690545: 65, 690984: 79, 692733: 86, 692923: 93, 693202: 19, 693291: 18, 693574: 95, 694080: 96, 694154: 64, 694757: 90, 695987: 14, 697875: 80, 698212: 98, 698919: 70, 699088: 51, 699400: 10, 699450: 59, 701321: 53, 701385: 5, 702341: 93, 702759: 35, 703058: 90, 704690: 1, 706799: 83, 707461: 80, 707782: 96, 709394: 65, 709674: 68, 710134: 39, 711321: 2, 711617: 48, 711697: 54, 711865: 27, 712280: 95, 712674: 93, 714395: 49, 715752: 65, 716594: 77, 718628: 6, 719717: 92, 719836: 98, 722326: 90, 727429: 77, 727486: 39, 727995: 17, 728725: 81, 730320: 97, 732662: 31, 732987: 54, 734077: 45, 734579: 17, 735287: 23, 735582: 54, 736172: 35, 736333: 21, 738337: 38, 740944: 78, 740955: 20, 742041: 67, 744781: 56, 747104: 45, 747629: 22, 747992: 29, 748677: 75, 748973: 0, 749012: 38, 749334: 70, 749404: 57, 749878: 87, 751771: 80, 753201: 76, 754298: 35, 756605: 81, 756641: 51, 757131: 11, 757454: 47, 757620: 69, 758402: 39, 758782: 35, 759085: 50, 759161: 34, 759562: 43, 761286: 62, 761908: 16, 762165: 77, 766751: 74, 767125: 9, 767654: 32, 768455: 99, 769620: 54, 769622: 87, 769895: 29, 769998: 22, 770716: 63, 771903: 93, 773959: 45, 775912: 60, 777448: 77, 778798: 70, 780896: 63, 781794: 43, 784133: 10, 785835: 17, 785951: 99, 786015: 91, 786774: 93, 787556: 39, 790828: 29, 791374: 94, 791638: 17, 793837: 77, 797673: 18, 797807: 77, 798273: 71, 798470: 92, 799642: 94, 800687: 30, 800972: 16, 801580: 62, 802470: 22, 803485: 99, 803527: 8, 807959: 77, 808066: 80, 808158: 8, 808318: 94, 811566: 19, 812314: 7, 813275: 25, 814647: 68, 815622: 79, 816386: 13, 817345: 31, 818114: 74, 818152: 21, 819731: 39, 819890: 97, 820113: 48, 820378: 98, 821034: 27, 821469: 34, 821727: 31, 821990: 46, 826765: 50, 827085: 16, 827387: 98, 829335: 62, 829776: 94, 830440: 38, 830510: 0, 835764: 94, 835904: 79, 837256: 6, 838071: 41, 838079: 46, 839644: 35, 839955: 11, 839970: 34, 840469: 66, 840678: 71, 840984: 73, 841770: 10, 842703: 13, 843215: 48, 845241: 99, 845720: 11, 846107: 9, 846255: 46, 846976: 28, 847685: 39, 849303: 96, 850295: 45, 851493: 28, 853398: 36, 853515: 35, 853887: 52, 855241: 51, 855487: 10, 856775: 89, 857068: 19, 858067: 71, 858851: 93, 859671: 8, 860218: 53, 861415: 62, 861450: 17, 862042: 50, 862302: 59, 863567: 22, 865658: 28, 867472: 6, 867601: 7, 868704: 83, 870271: 20, 871057: 0, 872726: 3, 873153: 91, 873787: 45, 875682: 40, 876394: 44, 878497: 85, 878579: 53, 878917: 36, 879259: 15, 879940: 48, 880396: 56, 886129: 0, 888904: 73, 889248: 28, 889997: 78, 892443: 87, 892894: 55, 893982: 74, 894195: 96, 895431: 8, 895727: 96, 895735: 62, 895920: 52, 898136: 23, 899851: 95, 900925: 12, 902396: 34, 904724: 48, 905590: 37, 907013: 29, 908843: 48, 909146: 12, 910456: 60, 911306: 6, 911398: 21, 915909: 0, 916743: 75, 917104: 74, 920164: 19, 920898: 25, 922146: 28, 922661: 71, 923060: 74, 923239: 81, 923639: 83, 924040: 59, 924702: 44, 926614: 16, 929311: 11, 930004: 53, 931459: 33, 931963: 18, 932798: 95, 933776: 72, 934793: 46, 936611: 46, 936672: 39, 936697: 92, 936883: 2, 938703: 16, 938983: 34, 939308: 18, 940777: 49, 942629: 44, 942998: 46, 943374: 60, 945307: 67, 945786: 63, 945984: 51, 946908: 30, 946973: 77, 947360: 8, 947698: 95, 947963: 32, 948269: 50, 948277: 82, 948336: 71, 948383: 97, 949580: 81, 950817: 67, 953986: 38, 954249: 9, 954415: 10, 954717: 19, 955029: 46, 955032: 95, 956264: 26, 956430: 11, 957404: 84, 958966: 7, 959004: 76, 959564: 79, 960542: 8, 961129: 67, 961620: 63, 962991: 25, 963001: 64, 963666: 1, 964929: 36, 965353: 74, 966547: 17, 968411: 29, 969960: 39, 970823: 12, 971050: 46, 973315: 62, 973528: 69, 974091: 64, 974279: 7, 974955: 49, 976428: 15, 977616: 66, 983977: 86, 984204: 72, 984439: 97, 984756: 54, 985182: 88, 986313: 99, 987209: 22, 989771: 32, 990139: 1, 991082: 43, 991264: 50, 993516: 82, 993858: 86, 994768: 21, 996406: 61, 996635: 49, 997225: 86, 999298: 11, 999821: 65},
		{3945: 65, 6587: 51, 7370: 2, 7700: 23, 7821: 39, 9337: 27, 13175: 57, 17375: 64, 22124: 39, 23294: 51, 25044: 37, 29220: 82, 33292: 75, 34810: 34, 36381: 39, 43120: 25, 43146: 76, 45066: 15, 48005: 66, 52733: 59, 53645: 96, 57492: 11, 57841: 82, 58406: 33, 58643: 76, 65165: 0, 65695: 48, 68192: 18, 70547: 57, 70977: 64, 72071: 51, 80134: 53, 80203: 1, 86323: 56, 86983: 42, 88293: 13, 90566: 5, 90746: 15, 91939: 13, 94620: 58, 95169: 42, 97302: 88, 97924: 66, 98281: 15, 100112: 71, 100506: 27, 102572: 12, 102688: 57, 102762: 41, 105041: 52, 110000: 86, 110626: 32, 114594: 34, 118174: 22, 119027: 51, 121789: 90, 122935: 23, 138784: 57, 139868: 94, 143428: 20, 147504: 30, 149329: 32, 153043: 38, 159597: 72, 166218: 49, 182985: 80, 184103: 66, 187374: 79, 190993: 24, 192570: 46, 197977: 82, 202705: 9, 204343: 87, 205864: 52, 206320: 91, 208402: 0, 209497: 60, 210209: 36, 211342: 99, 217546: 98, 223191: 40, 225549: 19, 228316: 67, 229132: 58, 231797: 10, 232095: 4, 233635: 56, 239339: 55, 246572: 5, 258277: 10, 263345: 53, 265899: 27, 272556: 85, 272687: 44, 273068: 2, 273727: 29, 276583: 39, 277248: 89, 277355: 86, 281640: 34, 285073: 93, 288066: 3, 301611: 63, 305099: 74, 307891: 77, 308616: 33, 314104: 15, 314464: 98, 317752: 57, 323406: 17, 324067: 50, 328547: 9, 330531: 92, 333527: 51, 335557: 16, 335637: 83, 340179: 23, 342232: 1, 346632: 84, 348963: 8, 349842: 98, 350970: 28, 355388: 93, 356031: 93, 357412: 2, 361335: 57, 361365: 24, 362473: 88, 362536: 86, 368831: 8, 369157: 2, 369854: 15, 374862: 84, 378383: 27, 380553: 14, 381838: 55, 382207: 84, 382309: 39, 383567: 13, 386141: 68, 391384: 34, 392404: 30, 393763: 7, 396467: 95, 401084: 39, 401312: 83, 406280: 92, 406353: 76, 407378: 64, 410876: 86, 411083: 61, 418229: 30, 422701: 67, 424307: 47, 428125: 85, 439168: 40, 442653: 20, 447677: 40, 453225: 23, 453564: 49, 456040: 51, 461723: 53, 465904: 53, 468100: 0, 469470: 47, 472075: 87, 474177: 10, 476675: 3, 481653: 29, 482070: 67, 482923: 9, 486889: 23, 489147: 92, 491258: 52, 495780: 38, 496119: 41, 496681: 97, 499295: 23, 501420: 97, 503220: 45, 505962: 73, 507419: 16, 510476: 89, 511330: 81, 513390: 63, 515508: 37, 517870: 4, 526612: 61, 529104: 71, 531081: 75, 536267: 36, 536291: 19, 538835: 39, 542698: 70, 544528: 81, 545799: 56, 551892: 80, 552121: 79, 554313: 12, 555754: 92, 565859: 15, 570099: 63, 571249: 78, 574156: 16, 575369: 76, 579318: 93, 582263: 1, 585518: 36, 585575: 83, 591339: 8, 591637: 20, 594239: 45, 594414: 26, 596943: 21, 597431: 54, 599002: 94, 600261: 90, 601109: 36, 603651: 38, 604498: 72, 604579: 0, 605048: 71, 608864: 77, 611510: 1, 614541: 22, 619335: 1, 620060: 8, 622683: 15, 625527: 60, 626004: 87, 626845: 54, 635726: 22, 639661: 13, 641287: 87, 644052: 91, 647145: 59, 648623: 87, 649390: 60, 649956: 97, 650156: 40, 651989: 44, 655724: 47, 666143: 79, 669110: 5, 671274: 18, 671530: 53, 672848: 78, 673295: 82, 673873: 89, 675010: 99, 681355: 25, 694231: 89, 696800: 51, 697627: 38, 697935: 42, 699204: 5, 699669: 78, 701847: 88, 701888: 35, 712135: 67, 713051: 27, 714032: 32, 716614: 58, 717199: 11, 722130: 87, 722172: 3, 725329: 56, 727355: 99, 730024: 69, 730409: 22, 735799: 81, 739335: 67, 741856: 86, 742378: 39, 743830: 91, 744609: 95, 746859: 74, 749754: 30, 750608: 17, 757619: 55, 759588: 26, 761640: 60, 762324: 58, 764017: 31, 767890: 50, 768895: 8, 772917: 93, 775528: 20, 778908: 96, 781148: 73, 790529: 14, 793090: 44, 793308: 38, 803564: 97, 811782: 14, 812016: 40, 812733: 81, 815668: 2, 818658: 35, 824115: 20, 832470: 83, 835259: 14, 835961: 28, 839229: 17, 839436: 44, 839530: 63, 839708: 48, 840122: 36, 843028: 92, 844050: 90, 844975: 82, 845995: 25, 848886: 18, 848951: 33, 850932: 97, 851105: 80, 851360: 37, 854049: 29, 856196: 71, 858700: 53, 861714: 24, 871229: 27, 874253: 8, 875515: 56, 879199: 0, 883684: 58, 885940: 54, 888507: 33, 890730: 11, 891307: 95, 891502: 55, 901943: 57, 902885: 51, 907538: 30, 909424: 13, 910885: 76, 913035: 4, 916384: 5, 916410: 78, 922463: 30, 923685: 12, 924012: 23, 924165: 94, 925071: 83, 925452: 91, 930741: 65, 934368: 62, 936301: 3, 936364: 86, 940995: 97, 941267: 52, 942942: 90, 944015: 30, 953073: 86, 964229: 15, 964758: 37, 967530: 68, 975105: 15, 978769: 49, 981272: 44, 982415: 0, 985883: 53, 988354: 81, 990979: 11, 991475: 40},
		{519: 67, 1875: 0, 2257: 63, 3246: 49, 3770: 61, 3773: 63, 3892: 48, 4026: 0, 4261: 13, 5489: 37, 7337: 54, 7838: 47, 7873: 8, 8505: 18, 9008: 45, 10211: 83, 10613: 24, 11187: 11, 11196: 33, 11825: 58, 11927: 60, 13209: 25, 13301: 2, 13510: 97, 14764: 41, 16246: 49, 16549: 3, 17146: 62, 17716: 77, 21137: 26, 21389: 58, 21423: 72, 21803: 35, 22433: 67, 22649: 60, 22665: 25, 23761: 85, 24171: 65, 25731: 92, 26216: 43, 27369: 59, 27941: 12, 28062: 99, 28640: 17, 30444: 27, 30755: 82, 31123: 44, 32408: 21, 32461: 89, 33294: 97, 34048: 64, 34952: 14, 35835: 77, 38739: 58, 38849: 21, 38877: 8, 39569: 31, 39657: 22, 41005: 68, 41094: 45, 43488: 90, 43760: 28, 44073: 68, 44099: 80, 44379: 64, 44508: 68, 44511: 64, 44917: 6, 46303: 0, 47827: 82, 48325: 37, 49201: 24, 49361: 79, 50880: 24, 51437: 91, 52712: 51, 52723: 32, 52828: 14, 52845: 57, 53634: 7, 53903: 59, 55105: 54, 55169: 12, 55934: 11, 56400: 39, 57351: 73, 57452: 24, 58327: 17, 58388: 13, 58495: 81, 58669: 56, 59647: 44, 60045: 3, 60743: 91, 60870: 10, 60876: 18, 61230: 4, 62352: 45, 63196: 69, 63264: 99, 63382: 10, 63543: 87, 63668: 19, 67396: 69, 67559: 9, 69366: 18, 69486: 5, 70535: 48, 71166: 42, 71781: 29, 73375: 13, 73870: 98, 73996: 16, 76279: 77, 76320: 8, 76572: 79, 76661: 25, 77975: 38, 79655: 33, 80372: 29, 80867: 72, 80951: 83, 83556: 81, 84602: 7, 85008: 47, 85520: 90, 85624: 73, 87229: 0, 88457: 91, 89401: 29, 92340: 91, 92753: 29, 93167: 65, 93356: 59, 95017: 99, 95442: 85, 97141: 66, 97641: 37, 98684: 15, 98778: 81, 98805: 93, 99722: 32, 99942: 41, 100031: 93, 102323: 38, 102548: 63, 102908: 85, 106273: 7, 107431: 49, 108312: 60, 108338: 65, 110071: 7, 110374: 16, 111615: 69, 112046: 32, 112393: 45, 115167: 62, 115283: 15, 115707: 93, 116062: 41, 116808: 13, 117386: 66, 117441: 66, 118465: 85, 119873: 85, 120977: 4, 121176: 57, 123319: 35, 123578: 13, 124723: 64, 125272: 72, 130734: 6, 131239: 73, 131514: 22, 132057: 4, 132679: 94, 132680: 4, 134265: 37, 135522: 87, 138286: 35, 141196: 6, 141771: 72, 141896: 0, 142100: 28, 142319: 74, 144170: 32, 148285: 74, 148605: 8, 148967: 94, 150173: 33, 151324: 2, 152096: 28, 152613: 41, 152834: 36, 153209: 23, 153768: 87, 155335: 20, 156522: 35, 158066: 37, 158576: 46, 160136: 46, 160158: 35, 160598: 82, 160894: 63, 162785: 97, 163061: 32, 166001: 27, 166254: 57, 166913: 74, 167823: 42, 169047: 52, 169194: 97, 169648: 52, 170412: 26, 172186: 89, 173991: 47, 174291: 68, 175151: 56, 175775: 61, 176274: 55, 177262: 95, 177359: 75, 179595: 96, 180357: 22, 181316: 41, 182054: 87, 183370: 55, 183386: 85, 184206: 28, 184346: 64, 184543: 45, 185244: 59, 185551: 84, 187619: 30, 189035: 21, 190318: 72, 191741: 10, 193654: 51, 194384: 39, 194842: 57, 196114: 67, 196451: 1, 196832: 89, 197191: 98, 197767: 56, 197889: 91, 199187: 15, 200954: 9, 201017: 33, 201510: 16, 202081: 62, 202459: 46, 203395: 42, 203442: 91, 203799: 95, 204808: 30, 206606: 28, 206664: 91, 207051: 3, 207265: 47, 208430: 41, 208457: 48, 211842: 43, 211860: 35, 212137: 75, 212574: 76, 213499: 79, 213597: 58, 215162: 38, 215467: 56, 215468: 25, 215520: 23, 217821: 31, 217920: 13, 218264: 60, 218323: 19, 218851: 85, 219470: 6, 219555: 1, 220375: 74, 222708: 92, 223013: 10, 224527: 48, 225121: 93, 226810: 21, 227746: 11, 228224: 73, 229419: 74, 229970: 25, 230898: 69, 231547: 27, 233281: 36, 233407: 66, 235232: 21, 235591: 24, 236842: 11, 237016: 93, 238850: 50, 239188: 63, 240959: 23, 241901: 81, 242259: 82, 242731: 17, 243797: 63, 244328: 71, 244444: 13, 246815: 47, 247046: 59, 251273: 38, 253241: 5, 253812: 98, 254007: 86, 254329: 21, 254558: 91, 254836: 78, 255904: 0, 256352: 84, 257847: 96, 259544: 88, 259730: 75, 260460: 24, 260734: 5, 260841: 30, 261291: 64, 262835: 29, 263129: 91, 263135: 62, 264519: 32, 265334: 11, 265964: 36, 267300: 14, 268299: 35, 268374: 69, 268378: 88, 269351: 44, 271285: 53, 272279: 38, 272312: 79, 272505: 58, 272802: 40, 274192: 58, 274976: 94, 275180: 35, 276230: 16, 277116: 86, 278461: 86, 278801: 58, 280557: 65, 281864: 49, 281897: 1, 284100: 27, 285043: 29, 285540: 10, 285603: 85, 285692: 75, 289577: 68, 289951: 27, 290206: 86, 290549: 92, 290794: 23, 292503: 48, 292534: 90, 292649: 54, 292834: 54, 293777: 51, 294479: 50, 295572: 88, 296910: 2, 297076: 75, 297607: 45, 298270: 40, 298369: 80, 299788: 85, 301595: 56, 302137: 62, 305017: 33, 305950: 64, 307050: 27, 309253: 87, 310201: 75, 310643: 51, 311338: 74, 311381: 34, 311393: 11, 312626: 23, 313446: 30, 315593: 29, 315840: 42, 315955: 57, 316623: 82, 317237: 63, 318486: 74, 318935: 11, 320798: 95, 320998: 39, 321500: 5, 321551: 4, 322131: 47, 322307: 80, 322378: 67, 324193: 16, 324528: 14, 324673: 28, 325470: 59, 325822: 57, 326041: 43, 326482: 3, 326944: 12, 327668: 53, 327670: 85, 327997: 62, 330062: 77, 331719: 3, 332239: 76, 332612: 29, 332637: 75, 333856: 30, 335154: 53, 336933: 5, 337048: 18, 338057: 61, 339028: 89, 339126: 89, 339236: 42, 339596: 51, 341602: 65, 342033: 21, 342357: 80, 342888: 66, 345712: 98, 346281: 64, 346570: 92, 348939: 21, 349966: 67, 350583: 63, 350779: 41, 351105: 16, 351193: 85, 352617: 37, 354108: 39, 354796: 62, 356457: 5, 356577: 61, 356915: 49, 357989: 8, 358100: 13, 358578: 86, 359316: 47, 360279: 9, 361356: 60, 362020: 24, 362160: 69, 362182: 81, 362353: 80, 365512: 84, 366869: 51, 368374: 94, 368694: 50, 369496: 17, 372742: 59, 373991: 9, 374548: 76, 374749: 64, 376269: 66, 378966: 69, 381412: 38, 381857: 45, 382394: 91, 383379: 0, 385967: 15, 386463: 55, 388003: 50, 388011: 97, 388156: 41, 388278: 62, 389413: 32, 391630: 70, 391727: 23, 392952: 97, 393129: 71, 393172: 70, 394574: 67, 396081: 53, 396953: 59, 397246: 41, 397304: 29, 398137: 92, 398373: 13, 399561: 70, 400361: 95, 400779: 76, 400900: 22, 402959: 45, 403034: 62, 403041: 23, 403245: 75, 404021: 24, 404278: 35, 404931: 84, 405267: 74, 405890: 21, 407132: 60, 407287: 67, 407486: 67, 407654: 91, 410245: 38, 411039: 7, 412097: 5, 412502: 33, 412688: 42, 413053: 74, 413993: 24, 415189: 98, 416075: 48, 417370: 12, 417553: 68, 418793: 68, 419302: 45, 419534: 59, 419635: 4, 419696: 16, 419735: 75, 421484: 70, 421533: 8, 421593: 68, 421605: 56, 422536: 39, 423147: 4, 423273: 20, 424640: 89, 425073: 68, 425748: 17, 425964: 99, 426844: 88, 430039: 53, 430111: 45, 431260: 13, 434235: 99, 434733: 15, 434755: 79, 435020: 97, 435490: 5, 436703: 13, 437143: 37, 437632: 85, 438116: 14, 439272: 76, 439848: 14, 440258: 25, 441121: 42, 441449: 62, 441834: 57, 442882: 17, 443134: 53, 446132: 11, 446979: 77, 447596: 14, 449099: 29, 450373: 81, 450511: 91, 450748: 16, 453386: 61, 453669: 82, 454338: 67, 455994: 67, 456374: 75, 456406: 72, 456664: 86, 456899: 7, 457678: 7, 458898: 18, 460968: 93, 461585: 70, 462348: 67, 462546: 18, 464225: 81, 464373: 45, 464421: 28, 464575: 10, 464922: 71, 465343: 6, 465777: 26, 466454: 31, 466486: 70, 467027: 82, 467585: 83, 467836: 31, 468744: 30, 468986: 84, 469018: 67, 470492: 89, 471112: 58, 471827: 19, 472170: 54, 474023: 10, 477903: 47, 480626: 41, 481202: 11, 481989: 88, 482826: 8, 483881: 15, 484175: 40, 484752: 28, 486260: 89, 488230: 74, 489081: 14, 489917: 8, 490068: 92, 490214: 52, 490328: 26, 490492: 47, 491333: 17, 492037: 29, 493145: 57, 493761: 95, 494154: 96, 494523: 67, 494821: 42, 499507: 95, 500700: 91, 501135: 77, 502330: 53, 502932: 0, 503410: 69, 505339: 70, 505395: 78, 505589: 47, 505829: 28, 507302: 17, 507454: 56, 508708: 28, 509445: 54, 509846: 66, 510248: 45, 510345: 36, 510572: 89, 512413: 28, 513039: 13, 516216: 97, 516233: 61, 516326: 16, 516724: 21, 516761: 89, 517149: 57, 517934: 7, 519423: 40, 519482: 60, 519911: 9, 520355: 16, 520668: 77, 522617: 79, 522908: 22, 523335: 37, 523786: 43, 526742: 34, 527536: 17, 528212: 70, 528319: 56, 529005: 6, 530893: 86, 530988: 0, 531013: 12, 531066: 9, 531632: 18, 532689: 69, 532806: 51, 533569: 80, 534605: 40, 534924: 16, 535152: 92, 535950: 3, 538389: 84, 539089: 54, 540366: 35, 541344: 95, 541471: 64, 541627: 59, 542197: 8, 545243: 18, 545591: 98, 546257: 57, 546646: 13, 547936: 20, 549036: 76, 550823: 29, 552155: 23, 553641: 45, 554501: 24, 554567: 56, 555194: 52, 555459: 59, 555911: 65, 557036: 63, 557618: 56, 557758: 70, 559420: 89, 561008: 91, 561452: 29, 563198: 80, 563336: 59, 563740: 92, 564130: 6, 564169: 32, 565522: 38, 565910: 5, 567262: 48, 567512: 79, 567676: 14, 568613: 79, 568708: 23, 569162: 16, 569839: 6, 571310: 4, 572283: 77, 573318: 16, 573472: 65, 574367: 68, 575459: 6, 576399: 4, 576845: 98, 578636: 11, 578639: 62, 580246: 48, 580462: 26, 581996: 77, 582174: 61, 583523: 61, 584354: 49, 587620: 19, 587688: 57, 589181: 10, 589723: 64, 589975: 2, 594555: 82, 595189: 38, 595224: 24, 596667: 76, 596753: 56, 596895: 14, 597770: 76, 599944: 66, 601345: 76, 602023: 71, 602315: 44, 603406: 81, 606255: 29, 606602: 43, 608658: 49, 610076: 32, 610747: 73, 611414: 51, 611834: 10, 612301: 42, 612352: 14, 612873: 32, 614161: 60, 614299: 21, 614656: 84, 616584: 10, 617872: 21, 618317: 89, 619326: 33, 619333: 24, 619396: 81, 619605: 33, 620714: 67, 621789: 78, 622409: 20, 623177: 32, 623193: 96, 623422: 63, 623605: 12, 624038: 97, 624832: 43, 624978: 4, 626804: 86, 628203: 20, 629225: 99, 629498: 72, 630169: 81, 631781: 51, 634228: 60, 634371: 26, 635602: 41, 636119: 67, 636562: 26, 637582: 40, 638412: 64, 638455: 19, 639345: 49, 639440: 82, 639933: 79, 640142: 58, 640529: 57, 641093: 44, 641239: 35, 641955: 55, 642772: 8, 642782: 66, 644135: 28, 644181: 46, 645450: 34, 646268: 77, 647198: 18, 647597: 90, 649414: 4, 649627: 78, 649881: 25, 651972: 66, 652786: 8, 653519: 78, 654669: 67, 655046: 52, 655101: 97, 656293: 62, 657529: 71, 658261: 13, 658638: 92, 659718: 86, 659974: 92, 662852: 38, 663837: 87, 663897: 87, 665097: 98, 666556: 68, 667869: 8, 671682: 80, 672130: 51, 672487: 56, 673662: 41, 673916: 35, 674828: 78, 675614: 50, 676502: 15, 678654: 23, 679041: 45, 680176: 52, 680651: 99, 680904: 31, 681123: 56, 681409: 36, 681982: 69, 682220: 41, 682734: 92, 682808: 66, 683541: 3, 684703: 34, 685853: 43, 688217: 93, 688664: 36, 689163: 31, 689384: 85, 690692: 19, 691668: 47, 692755: 65, 693139: 41, 693264: 17, 693366: 27, 694338: 94, 694785: 61, 695041: 58, 696759: 33, 697584: 82, 697708: 29, 698863: 43, 699099: 70, 699109: 81, 699201: 63, 699928: 25, 700105: 3, 700425: 82, 700689: 99, 701684: 95, 702396: 95, 702831: 66, 703673: 80, 704276: 1, 705404: 95, 709337: 5, 711208: 37, 711758: 45, 714525: 77, 715499: 57, 716304: 39, 716494: 91, 716774: 85, 717835: 33, 719079: 10, 719540: 30, 720951: 96, 721314: 28, 722218: 77, 723686: 77, 724112: 84, 724777: 62, 725241: 23, 725290: 55, 725741: 90, 726325: 15, 726829: 15, 727810: 51, 729546: 84, 730825: 29, 730966: 5, 731949: 72, 732562: 22, 732770: 93, 732878: 73, 733658: 35, 733980: 66, 734372: 97, 734623: 54, 735046: 35, 736578: 80, 737473: 64, 737537: 42, 737799: 0, 738770: 23, 739554: 42, 739823: 91, 741553: 74, 742877: 53, 743068: 0, 743180: 28, 743616: 34, 743977: 76, 744167: 26, 745194: 75, 745730: 52, 746048: 39, 746468: 9, 747215: 9, 747512: 91, 748707: 75, 748733: 96, 750313: 81, 751199: 48, 751425: 76, 752293: 56, 753933: 42, 757191: 7, 757198: 55, 757250: 37, 757608: 56, 757743: 70, 758266: 89, 758908: 76, 759178: 92, 760310: 67, 760356: 78, 760676: 46, 761155: 42, 761258: 51, 761947: 48, 762765: 6, 762896: 27, 764061: 95, 764863: 24, 764874: 91, 765842: 71, 767268: 95, 768188: 38, 769420: 18, 770298: 68, 772858: 96, 773030: 49, 773722: 37, 774748: 37, 775016: 99, 776155: 52, 776482: 30, 777149: 35, 778068: 24, 778394: 18, 779017: 3, 779767: 8, 782402: 69, 782982: 30, 783015: 50, 783190: 45, 783515: 24, 784613: 33, 785142: 14, 785942: 1, 786581: 25, 789013: 73, 790733: 92, 791028: 88, 793792: 97, 795835: 35, 796023: 91, 796557: 23, 797062: 22, 798368: 29, 799922: 19, 800652: 7, 800932: 65, 801364: 85, 801793: 50, 803244: 25, 803262: 26, 804484: 71, 804791: 90, 804889: 9, 805156: 35, 808558: 47, 809399: 93, 810560: 83, 811781: 35, 812213: 68, 813322: 20, 814386: 84, 814916: 47, 814981: 33, 815136: 72, 815845: 75, 816459: 84, 817457: 44, 817873: 59, 818608: 10, 819806: 88, 820027: 65, 820238: 45, 820337: 90, 820618: 17, 822494: 17, 823083: 64, 823150: 53, 823474: 0, 823624: 21, 823724: 68, 824164: 90, 825098: 92, 825656: 71, 825901: 44, 826118: 58, 826154: 68, 826919: 61, 828446: 72, 829119: 40, 834045: 45, 835706: 50, 836149: 68, 836188: 12, 836750: 62, 837104: 29, 837860: 76, 838458: 47, 839786: 55, 839972: 72, 841224: 61, 842648: 80, 843126: 68, 843433: 12, 844734: 64, 848521: 92, 848722: 13, 849458: 23, 850620: 21, 854392: 65, 854801: 7, 855932: 96, 856047: 77, 856269: 71, 858514: 81, 859065: 85, 859479: 46, 860578: 68, 861499: 68, 863310: 66, 863362: 34, 863461: 65, 864647: 34, 865683: 60, 865939: 25, 866937: 68, 867110: 98, 867766: 64, 867778: 13, 870161: 31, 870825: 41, 871352: 24, 871574: 30, 872575: 39, 872759: 1, 873158: 77, 874114: 29, 875697: 66, 876085: 50, 876376: 15, 876593: 33, 876633: 72, 876895: 32, 878373: 51, 879176: 23, 880457: 52, 880887: 12, 881027: 41, 882158: 81, 883385: 68, 884168: 24, 884189: 16, 884501: 21, 891255: 27, 892501: 97, 892944: 40, 893042: 51, 893903: 60, 896266: 55, 896361: 94, 896450: 95, 896495: 2, 896774: 29, 897398: 33, 898193: 41, 898820: 56, 899746: 76, 900025: 18, 900421: 15, 901295: 76, 901592: 95, 902227: 31, 902552: 11, 903323: 66, 903588: 51, 904286: 28, 905041: 78, 905044: 84, 905190: 21, 905513: 24, 906285: 58, 907209: 42, 907685: 25, 908311: 28, 908584: 95, 908939: 33, 909927: 44, 910198: 1, 911093: 54, 911536: 47, 912579: 5, 913264: 53, 913304: 22, 914605: 60, 914906: 59, 915069: 64, 916604: 2, 917270: 32, 917711: 54, 917981: 19, 918497: 39, 918577: 52, 920653: 75, 921860: 65, 922002: 81, 922030: 21, 922076: 20, 923455: 40, 923544: 94, 924122: 43, 924290: 63, 924338: 86, 924653: 1, 924714: 34, 926292: 46, 927924: 31, 928191: 86, 928725: 14, 931574: 33, 933043: 54, 933193: 66, 934600: 50, 934883: 5, 936354: 15, 936753: 79, 938462: 56, 938571: 59, 940540: 77, 940986: 17, 941000: 33, 942663: 55, 945161: 16, 950066: 28, 950836: 40, 952999: 87, 953027: 0, 953057: 33, 954014: 10, 954113: 47, 954581: 66, 954781: 71, 956347: 66, 956456: 31, 957029: 60, 959340: 30, 959738: 64, 959906: 88, 960761: 89, 961371: 78, 961372: 90, 961521: 61, 963510: 4, 965090: 16, 966196: 3, 966795: 39, 966811: 9, 967150: 8, 967620: 24, 967688: 45, 967952: 26, 968049: 29, 968545: 68, 968864: 68, 969776: 87, 970155: 86, 970987: 87, 971993: 92, 973445: 85, 973669: 21, 974070: 4, 974957: 29, 976781: 12, 978861: 14, 980579: 61, 980745: 53, 981386: 6, 981490: 5, 982540: 38, 982995: 51, 983214: 11, 983391: 53, 983466: 57, 983526: 54, 983531: 51, 984568: 53, 985179: 32, 985275: 20, 986659: 90, 988178: 39, 988495: 90, 990769: 30, 991667: 33, 991803: 15, 992541: 74, 993414: 10, 993617: 52, 993709: 9, 993794: 61, 994380: 57, 995043: 52, 995818: 79, 997145: 66, 997589: 61, 998346: 38, 999093: 86, 999132: 53, 999185: 83},
		{4229: 12, 6206: 18, 8635: 50, 9423: 40, 12358: 76, 16675: 56, 17542: 22, 20719: 97, 30818: 38, 42267: 88, 46510: 37, 47907: 60, 49098: 14, 53513: 30, 56007: 52, 58823: 9, 59690: 30, 66198: 11, 71995: 14, 79780: 99, 90060: 74, 91921: 18, 92086: 85, 93281: 76, 95138: 48, 98660: 44, 102567: 40, 103382: 54, 104234: 42, 104840: 30, 106330: 50, 106743: 54, 107772: 65, 114144: 49, 121077: 5, 121811: 84, 125995: 46, 130522: 84, 133101: 66, 133437: 2, 134095: 52, 140035: 49, 142433: 71, 143614: 80, 149240: 18, 166070: 82, 169362: 26, 172872: 76, 176818: 15, 177342: 26, 184897: 75, 193011: 74, 200518: 33, 201179: 5, 202361: 80, 207862: 36, 210160: 62, 212905: 3, 215011: 59, 215039: 26, 222794: 73, 225907: 53, 227925: 64, 231672: 81, 234381: 20, 237409: 23, 244708: 65, 250601: 70, 259430: 43, 261967: 63, 262369: 9, 265863: 66, 265998: 4, 268787: 5, 270216: 26, 270941: 63, 277835: 19, 284279: 13, 285323: 28, 286657: 41, 291833: 85, 296465: 7, 300420: 86, 314054: 43, 317790: 48, 318857: 98, 321623: 99, 327795: 15, 329243: 74, 333093: 18, 340684: 63, 344679: 91, 347510: 81, 348667: 13, 349472: 46, 349510: 33, 350960: 81, 352696: 39, 360183: 68, 360299: 16, 360732: 72, 364372: 70, 366146: 18, 368004: 54, 372954: 77, 374731: 36, 375405: 94, 382815: 47, 384012: 26, 389380: 38, 394762: 1, 395636: 97, 402573: 0, 403190: 48, 403908: 30, 405297: 17, 405844: 28, 406997: 75, 410408: 11, 412529: 5, 412921: 14, 415926: 37, 425567: 18, 427297: 28, 434853: 32, 448379: 96, 450795: 29, 456537: 95, 465784: 11, 467790: 74, 472421: 46, 480025: 74, 486206: 6, 490316: 53, 490889: 46, 499107: 99, 511051: 48, 514571: 82, 519877: 85, 521826: 82, 529286: 25, 529957: 73, 532833: 88, 534620: 60, 535834: 70, 537719: 50, 540404: 29, 543912: 73, 545381: 28, 546191: 25, 548339: 19, 557334: 90, 558598: 2, 559296: 26, 562539: 68, 579671: 58, 579800: 89, 583589: 83, 584741: 96, 590846: 4, 596052: 55, 606858: 82, 612954: 89, 614066: 16, 618890: 52, 618905: 34, 626006: 75, 627467: 28, 628986: 46, 630264: 64, 634275: 46, 639711: 62, 641345: 48, 645152: 97, 648322: 30, 652284: 54, 654445: 73, 656540: 98, 656689: 48, 658826: 84, 660427: 65, 665482: 11, 666259: 42, 667226: 59, 667896: 59, 670676: 17, 671471: 22, 674506: 3, 684574: 90, 687098: 39, 689153: 49, 689844: 92, 689940: 17, 690995: 84, 703010: 5, 706981: 45, 710281: 68, 717127: 7, 721158: 93, 723880: 5, 724587: 15, 738271: 72, 739947: 85, 742089: 85, 743022: 96, 747240: 51, 748171: 31, 748635: 63, 751687: 43, 751795: 57, 758700: 39, 759639: 98, 762843: 29, 766325: 51, 767685: 58, 779133: 7, 784948: 17, 787086: 56, 789910: 43, 791720: 58, 793295: 18, 794509: 48, 797648: 80, 802627: 21, 805190: 57, 814941: 6, 822306: 92, 826851: 19, 830610: 92, 830657: 77, 833943: 89, 836011: 7, 840596: 9, 845283: 28, 845843: 57, 851450: 54, 851941: 26, 865096: 79, 868696: 14, 870160: 61, 872677: 3, 872859: 33, 873340: 18, 875747: 51, 876401: 14, 886316: 15, 889998: 96, 890751: 14, 895769: 12, 896078: 84, 896250: 54, 900011: 53, 904209: 14, 912246: 23, 912704: 17, 922288: 69, 923661: 4, 927412: 83, 927709: 9, 930036: 29, 933384: 83, 936954: 1, 941459: 46, 944872: 54, 949616: 29, 951335: 70, 951955: 7, 963551: 10, 966015: 80, 967680: 61, 971223: 63, 971258: 9, 971550: 31, 972189: 77, 974358: 96, 977817: 66, 978426: 99, 983330: 8, 984897: 71, 987329: 79, 988411: 63, 991951: 35, 992909: 33},
		{103: 49, 3083: 77, 3817: 3, 11680: 54, 14133: 26, 15020: 83, 16925: 18, 17751: 64, 18749: 44, 22143: 54, 23281: 48, 24137: 86, 24908: 48, 29536: 82, 29695: 81, 30093: 55, 33387: 62, 36534: 25, 39913: 45, 42889: 74, 44035: 43, 44205: 65, 44642: 59, 44778: 73, 45216: 58, 46813: 25, 48181: 47, 49942: 94, 50202: 83, 51578: 43, 52055: 35, 52116: 46, 52940: 45, 53476: 7, 53593: 7, 53627: 90, 54475: 58, 58337: 84, 63548: 29, 63986: 59, 65935: 26, 68260: 29, 69076: 92, 69268: 97, 71096: 52, 73583: 62, 74926: 16, 75442: 76, 75539: 37, 76551: 24, 76592: 8, 76623: 69, 77114: 9, 78468: 22, 78998: 65, 79871: 46, 79919: 16, 82534: 78, 83744: 4, 85206: 25, 86979: 44, 87764: 39, 88181: 87, 89812: 70, 89986: 84, 90815: 66, 92773: 2, 92904: 78, 93296: 85, 95712: 77, 97002: 41, 97684: 46, 97757: 18, 97883: 17, 98469: 28, 99226: 79, 99935: 46, 103056: 23, 103103: 67, 104083: 74, 104568: 66, 105241: 86, 105925: 90, 107152: 2, 109166: 43, 109458: 73, 111426: 86, 112979: 70, 113783: 9, 114519: 72, 114598: 2, 114630: 1, 114726: 96, 116513: 44, 116731: 0, 118847: 75, 121069: 41, 121432: 16, 121494: 17, 121753: 16, 121806: 9, 122134: 88, 123172: 89, 125289: 32, 126383: 7, 130477: 18, 133323: 43, 133554: 85, 139522: 8, 139589: 73, 141433: 55, 141703: 89, 143681: 21, 144353: 74, 145218: 52, 145230: 30, 145416: 64, 145624: 43, 145970: 46, 147573: 56, 147987: 21, 148837: 59, 149502: 4, 152878: 30, 154941: 33, 155799: 25, 157396: 30, 158160: 55, 158614: 82, 159702: 85, 161012: 66, 163404: 18, 164011: 82, 166284: 2, 167170: 12, 170113: 1, 170761: 69, 172046: 28, 172675: 86, 173458: 55, 174960: 15, 175064: 56, 176549: 13, 176614: 79, 176780: 52, 177008: 52, 180739: 35, 181298: 3, 185879: 15, 191912: 89, 193990: 67, 196049: 36, 196289: 7, 197449: 99, 198221: 52, 198978: 61, 199194: 19, 199501: 82, 201032: 99, 203648: 90, 205158: 45, 205539: 43, 205701: 7, 205827: 64, 207848: 86, 209613: 23, 210655: 3, 211120: 8, 212164: 30, 213341: 26, 215087: 55, 218082: 61, 218851: 61, 220377: 51, 221914: 22, 226339: 9, 227314: 95, 228952: 28, 231167: 34, 233158: 40, 235212: 62, 237866: 36, 239355: 99, 240219: 68, 240563: 33, 240654: 41, 241250: 19, 242498: 2, 243545: 92, 246318: 54, 246476: 59, 246829: 36, 247457: 12, 248229: 42, 248801: 18, 249743: 63, 252436: 82, 254721: 68, 254910: 97, 256832: 3, 257473: 27, 259208: 73, 261544: 34, 262077: 93, 262653: 36, 262659: 62, 263182: 14, 264634: 85, 265605: 48, 271492: 49, 273526: 0, 274786: 89, 275930: 87, 279940: 23, 280843: 46, 281871: 88, 282106: 3, 282385: 71, 282711: 75, 283000: 0, 283032: 70, 283117: 67, 285782: 79, 289567: 56, 289906: 55, 291475: 69, 291687: 83, 291702: 90, 292281: 25, 293921: 54, 294512: 37, 296070: 88, 296206: 47, 298818: 27, 299963: 88, 300082: 44, 300630: 66, 300715: 38, 301765: 83, 303073: 82, 303889: 68, 304297: 33, 304532: 57, 305604: 95, 305965: 37, 307369: 97, 307754: 76, 308543: 9, 309017: 49, 311394: 65, 313100: 37, 314111: 91, 314222: 20, 315489: 6, 315739: 26, 316814: 17, 318821: 31, 323201: 73, 323752: 35, 324034: 50, 325129: 82, 326245: 31, 327530: 25, 331200: 98, 331837: 85, 332672: 28, 332912: 34, 333255: 11, 333388: 62, 336717: 79, 337696: 23, 337968: 67, 338500: 76, 339206: 12, 339513: 35, 340891: 56, 342188: 37, 342961: 23, 344578: 99, 344621: 83, 346942: 84, 347372: 62, 350083: 97, 350979: 60, 351769: 18, 353998: 93, 356297: 95, 356561: 25, 358687: 58, 360055: 59, 361446: 51, 361495: 0, 361841: 70, 364134: 33, 365030: 89, 365554: 79, 365879: 9, 366344: 73, 366569: 59, 366598: 60, 370884: 85, 371396: 32, 373207: 37, 375696: 51, 377580: 47, 377987: 70, 378151: 82, 381262: 20, 382745: 65, 383824: 31, 384231: 64, 384682: 38, 384905: 75, 385307: 9, 386122: 13, 386493: 12, 389613: 77, 389973: 72, 390899: 50, 391942: 67, 392371: 3, 393368: 33, 393556: 3, 393721: 53, 394015: 8, 394899: 21, 394917: 79, 397334: 85, 397396: 8, 400518: 16, 401180: 83, 403302: 34, 404284: 87, 404570: 47, 408574: 18, 409881: 98, 410229: 99, 410591: 49, 412570: 28, 412876: 76, 415329: 39, 415355: 40, 419413: 22, 421373: 63, 421385: 80, 426365: 43, 426930: 64, 428810: 96, 428857: 1, 429601: 11, 430882: 14, 432705: 17, 433570: 48, 438664: 42, 439683: 17, 442139: 98, 442698: 11, 443321: 67, 443624: 91, 445152: 8, 445358: 48, 445678: 90, 447971: 45, 448424: 75, 448443: 0, 449706: 97, 453888: 14, 454806: 60, 454990: 85, 455657: 61, 456523: 90, 457383: 69, 457732: 34, 457932: 90, 458599: 64, 458607: 61, 459047: 56, 459157: 21, 459457: 50, 460410: 51, 460487: 9, 461123: 24, 462881: 56, 463299: 88, 465605: 87, 465682: 9, 465998: 93, 467397: 80, 468276: 42, 469463: 38, 472735: 19, 473915: 49, 474178: 46, 474540: 33, 477150: 90, 480929: 44, 481678: 66, 481842: 82, 482966: 7, 484397: 56, 485258: 78, 485892: 53, 487211: 10, 488533: 73, 489956: 60, 490519: 22, 492482: 99, 492736: 90, 494020: 73, 494539: 3, 497567: 49, 498075: 77, 498645: 98, 499651: 92, 500484: 50, 500746: 95, 500955: 92, 501219: 61, 501319: 14, 501715: 64, 502654: 17, 503231: 77, 504064: 52, 504146: 99, 506497: 60, 508862: 20, 510462: 37, 511414: 4, 512183: 26, 512358: 66, 512982: 15, 513617: 69, 514384: 81, 514530: 40, 516251: 39, 516839: 27, 521897: 53, 526088: 98, 526644: 90, 529686: 47, 529835: 81, 530062: 8, 530263: 19, 531302: 2, 532154: 60, 532465: 36, 532481: 54, 534161: 47, 534363: 42, 534411: 69, 536970: 80, 537596: 63, 538819: 98, 540704: 67, 543783: 11, 544362: 29, 545639: 34, 545724: 46, 548103: 4, 549335: 78, 549845: 50, 551756: 29, 552075: 19, 552784: 49, 554356: 60, 554544: 69, 555105: 65, 555753: 35, 557014: 92, 557260: 52, 557959: 88, 559061: 72, 560474: 44, 563779: 64, 564044: 82, 565105: 43, 566453: 1, 568061: 82, 570151: 53, 571912: 83, 572735: 62, 572964: 94, 572998: 51, 574700: 0, 575077: 14, 575926: 72, 578003: 29, 579182: 74, 580783: 57, 581215: 96, 581264: 53, 582157: 25, 583274: 72, 583739: 95, 585264: 17, 586767: 94, 589301: 63, 591028: 62, 592555: 44, 593011: 8, 593254: 97, 594467: 54, 595143: 67, 597030: 20, 599662: 83, 601542: 55, 601634: 26, 601861: 74, 603111: 42, 606431: 53, 606812: 53, 607075: 22, 607284: 64, 608567: 92, 608899: 73, 609688: 44, 610139: 61, 611468: 85, 612981: 53, 612990: 4, 614312: 69, 614881: 10, 615038: 18, 616538: 66, 623444: 93, 623578: 46, 625201: 96, 625525: 81, 626405: 52, 627816: 65, 629806: 85, 629826: 52, 629905: 27, 630002: 62, 630076: 17, 632380: 80, 633353: 21, 634745: 67, 635525: 50, 635889: 57, 636013: 12, 636039: 48, 636084: 32, 637659: 93, 638485: 0, 641080: 21, 642251: 46, 645288: 77, 647067: 50, 648424: 89, 649299: 50, 649734: 93, 649856: 41, 653106: 38, 653510: 73, 654345: 64, 656448: 16, 656695: 69, 658362: 27, 658434: 51, 658692: 57, 659144: 98, 660292: 63, 660365: 6, 660399: 84, 660693: 4, 661582: 86, 663184: 47, 664950: 64, 665033: 3, 667212: 28, 668284: 70, 669784: 20, 670447: 15, 670493: 8, 674148: 39, 674450: 61, 675495: 69, 676404: 34, 676497: 89, 677150: 32, 677286: 80, 677874: 27, 681112: 28, 681298: 80, 681645: 60, 682876: 65, 682932: 65, 685135: 11, 685461: 56, 686352: 12, 687204: 25, 689718: 88, 689850: 3, 690324: 3, 690993: 47, 692002: 26, 695104: 52, 695228: 19, 695931: 72, 696084: 1, 696181: 78, 698237: 34, 698800: 27, 699011: 59, 701397: 44, 702537: 16, 703426: 89, 705464: 88, 705646: 15, 706324: 87, 707486: 53, 707507: 81, 708450: 87, 709155: 63, 709305: 4, 709923: 7, 710333: 12, 710819: 95, 711288: 79, 711435: 69, 712724: 14, 713514: 74, 717343: 27, 717635: 17, 718463: 69, 720213: 17, 722861: 76, 724056: 98, 724507: 33, 725246: 42, 725686: 48, 726051: 41, 727012: 15, 727250: 51, 731580: 32, 731874: 88, 733042: 61, 735597: 2, 736012: 40, 738519: 16, 739296: 96, 742115: 96, 745432: 64, 746096: 61, 748812: 78, 749822: 8, 751224: 76, 751708: 80, 752571: 56, 755503: 80, 757475: 81, 757648: 28, 757769: 44, 757968: 38, 759579: 83, 759762: 46, 760083: 49, 760348: 19, 760682: 95, 760715: 44, 762011: 70, 766219: 1, 767837: 7, 769003: 69, 769551: 86, 769790: 29, 771974: 11, 772056: 56, 772655: 23, 772788: 15, 773240: 3, 773441: 43, 774199: 1, 774498: 93, 776195: 7, 776485: 67, 776762: 62, 782013: 70, 782127: 11, 783510: 5, 783558: 90, 783562: 78, 783981: 59, 784214: 49, 785029: 73, 786462: 28, 787187: 46, 789123: 48, 790931: 82, 791654: 36, 792116: 99, 794380: 63, 795592: 18, 796357: 43, 796392: 27, 796766: 3, 797105: 72, 797574: 88, 798231: 81, 799360: 90, 801420: 35, 802004: 12, 802160: 18, 807010: 39, 807209: 33, 807571: 78, 808706: 34, 809319: 9, 809329: 55, 809609: 75, 811930: 26, 813054: 63, 814343: 94, 814707: 89, 815461: 39, 817306: 11, 817463: 1, 818559: 89, 820522: 16, 822949: 97, 822989: 92, 823170: 81, 825346: 9, 825427: 45, 829731: 78, 832215: 32, 833926: 96, 834127: 20, 835735: 36, 835915: 78, 836190: 8, 836754: 20, 836964: 62, 837387: 97, 842440: 14, 842793: 98, 844278: 81, 846626: 42, 847137: 26, 847451: 21, 847686: 9, 852136: 65, 852281: 36, 852614: 8, 854377: 77, 855482: 96, 857372: 8, 858274: 11, 859052: 3, 859262: 74, 859981: 67, 862070: 4, 863931: 10, 864539: 46, 864917: 39, 865643: 39, 865822: 46, 868272: 38, 869764: 70, 871544: 64, 873508: 48, 874683: 68, 876043: 50, 877150: 92, 879732: 31, 880929: 5, 881041: 71, 882236: 31, 882774: 42, 883861: 40, 885877: 80, 886113: 52, 887170: 14, 888096: 54, 888182: 75, 888322: 52, 888586: 39, 890386: 10, 890559: 9, 892535: 71, 894707: 59, 894843: 0, 895344: 44, 895515: 35, 896948: 92, 897796: 0, 899162: 44, 900856: 85, 901357: 4, 902363: 49, 905049: 37, 905052: 90, 905121: 54, 905189: 61, 906532: 43, 907469: 13, 908134: 13, 908768: 99, 910454: 13, 911001: 48, 911751: 30, 914223: 58, 915199: 80, 917777: 57, 917964: 8, 919198: 68, 922004: 15, 922367: 24, 922990: 31, 924756: 63, 924799: 46, 927563: 81, 928623: 24, 932090: 95, 932457: 51, 933748: 63, 934414: 84, 935264: 42, 936266: 91, 939121: 97, 939673: 72, 942751: 69, 943675: 93, 943968: 94, 945144: 70, 946940: 87, 948056: 68, 948125: 82, 948201: 92, 949042: 25, 950695: 80, 957932: 84, 958693: 27, 961440: 6, 962188: 37, 964513: 12, 966015: 94, 966213: 14, 966278: 81, 966446: 70, 966469: 14, 968284: 19, 969698: 43, 970712: 71, 971635: 44, 973613: 7, 976882: 29, 977558: 75, 978294: 80, 978383: 37, 979509: 88, 980710: 69, 980775: 0, 980822: 68, 982732: 89, 983159: 74, 983648: 6, 984225: 98, 984656: 9, 986209: 50, 986220: 70, 986466: 28, 987101: 55, 988122: 36, 989173: 94, 989966: 81, 991119: 68, 992898: 26, 993000: 78, 993120: 80, 993926: 78, 995517: 32, 996678: 95, 998944: 81, 999673: 75},
		{104: 93, 2826: 54, 3631: 11, 7704: 34, 12367: 12, 12513: 18, 23041: 18, 24057: 61, 24216: 27, 24943: 10, 27275: 65, 27879: 73, 28034: 63, 30170: 71, 30335: 26, 32421: 81, 32484: 85, 32846: 6, 33023: 2, 34921: 32, 35026: 36, 36088: 85, 36155: 42, 38219: 25, 38833: 44, 38992: 24, 40970: 47, 41312: 39, 41558: 5, 43308: 92, 43747: 38, 43850: 63, 44652: 62, 44714: 95, 44782: 66, 46377: 47, 48248: 21, 48286: 77, 50098: 40, 50321: 21, 52879: 29, 52986: 52, 56076: 13, 57466: 91, 57928: 35, 60117: 84, 60836: 79, 62872: 89, 65046: 89, 65132: 22, 65684: 11, 69645: 59, 69770: 64, 74550: 8, 78107: 0, 78135: 85, 78246: 60, 78424: 47, 82844: 82, 84166: 8, 85662: 65, 87554: 52, 89737: 26, 90189: 28, 90884: 18, 91824: 7, 91828: 63, 92032: 78, 95314: 60, 96274: 20, 97473: 85, 97600: 56, 98398: 9, 99438: 5, 100424: 13, 100438: 53, 107409: 39, 108193: 78, 108388: 80, 109066: 71, 111143: 18, 117828: 34, 124225: 69, 125877: 66, 129800: 43, 131298: 9, 131564: 55, 132562: 38, 132893: 82, 133243: 15, 134597: 51, 136111: 3, 136979: 0, 139415: 73, 139432: 57, 140118: 13, 140703: 72, 143901: 40, 143934: 39, 144081: 57, 144362: 86, 144866: 33, 145620: 45, 145790: 85, 148535: 69, 148801: 76, 148909: 58, 151180: 65, 151535: 82, 151659: 88, 154963: 61, 155962: 65, 156172: 61, 157064: 36, 158222: 99, 160644: 22, 160728: 51, 160831: 63, 163678: 59, 165372: 47, 168352: 83, 169144: 1, 171703: 4, 172211: 3, 172290: 52, 182220: 22, 182655: 87, 183971: 77, 186200: 14, 187796: 22, 188891: 2, 190231: 68, 191456: 36, 191871: 44, 192210: 24, 192664: 17, 194871: 74, 199159: 84, 202029: 76, 204165: 15, 206540: 33, 207198: 83, 209157: 3, 210933: 17, 211471: 91, 212751: 26, 214642: 23, 214672: 91, 214907: 79, 218575: 20, 219766: 57, 220428: 61, 220770: 15, 221064: 46, 222343: 97, 223672: 94, 225672: 37, 227264: 90, 227862: 85, 231654: 19, 232936: 38, 233519: 72, 233829: 4, 234629: 16, 235099: 46, 236451: 85, 237967: 80, 238751: 47, 238803: 75, 240491: 80, 241517: 82, 242327: 61, 247789: 15, 249900: 48, 252776: 80, 253257: 39, 253621: 33, 253744: 3, 256320: 53, 256404: 96, 257100: 56, 257882: 69, 258649: 82, 260877: 79, 261159: 87, 261175: 62, 262444: 65, 265527: 30, 266562: 56, 267706: 49, 272360: 73, 273155: 19, 273527: 5, 274722: 54, 275927: 41, 276006: 90, 276035: 14, 277041: 6, 277198: 62, 277435: 49, 282335: 16, 283611: 11, 283705: 3, 284132: 2, 285653: 77, 286067: 25, 289994: 58, 293118: 36, 295213: 32, 295276: 77, 299466: 16, 300986: 35, 301743: 27, 303354: 81, 305357: 19, 307592: 26, 308394: 41, 310203: 81, 313655: 29, 313725: 65, 314962: 93, 315135: 21, 315549: 8, 317702: 27, 318422: 69, 319548: 96, 321507: 77, 323339: 10, 323674: 18, 323754: 82, 324702: 27, 325100: 64, 325676: 85, 326379: 41, 329239: 80, 330725: 7, 331202: 8, 331414: 4, 333520: 24, 334511: 78, 335607: 37, 338063: 89, 338473: 37, 339454: 1, 339895: 30, 340711: 30, 343502: 24, 344778: 64, 345000: 52, 349750: 69, 349788: 22, 350173: 27, 351762: 14, 352137: 89, 353055: 69, 355067: 94, 355310: 86, 356668: 93, 357074: 37, 360194: 76, 360357: 88, 360443: 33, 360893: 22, 360997: 33, 363537: 92, 363737: 11, 364446: 21, 364703: 21, 364877: 74, 365316: 69, 366954: 79, 368789: 84, 368923: 30, 372683: 79, 372708: 21, 373198: 24, 373360: 46, 373939: 6, 377209: 14, 379860: 71, 381090: 34, 383102: 46, 384019: 83, 384600: 0, 387051: 79, 387724: 58, 387773: 43, 389410: 36, 391388: 38, 395601: 32, 396513: 71, 398585: 96, 399641: 16, 400226: 88, 401891: 55, 403747: 56, 404384: 90, 405884: 42, 406058: 53, 406063: 87, 406344: 35, 406462: 60, 406635: 96, 408058: 14, 411285: 87, 412712: 47, 414209: 20, 416090: 58, 417000: 32, 417392: 70, 417822: 39, 418565: 7, 419166: 40, 419371: 35, 421652: 52, 423376: 10, 423646: 58, 423831: 49, 424934: 7, 426394: 13, 428561: 4, 429278: 93, 429477: 87, 429778: 93, 431601: 9, 434780: 78, 435540: 59, 436603: 44, 436622: 57, 437855: 76, 443032: 79, 443617: 87, 443630: 34, 445532: 17, 447799: 87, 449343: 61, 456054: 99, 456609: 70, 456661: 70, 456727: 27, 456861: 65, 457181: 53, 457858: 56, 458771: 90, 460529: 5, 462884: 5, 467318: 61, 467454: 1, 468960: 93, 469475: 76, 470708: 25, 471906: 27, 472655: 62, 475096: 46, 477671: 22, 478692: 49, 479031: 36, 481755: 87, 484926: 83, 485820: 43, 486253: 61, 486731: 57, 486908: 92, 490991: 23, 491784: 17, 493124: 42, 496868: 88, 497266: 95, 497375: 11, 498040: 7, 499579: 79, 500497: 46, 502438: 17, 503415: 21, 507065: 41, 507953: 94, 511984: 84, 513439: 98, 517170: 41, 519371: 57, 520144: 14, 522876: 33, 523823: 73, 524317: 6, 524601: 91, 525485: 42, 527425: 14, 528540: 48, 529481: 11, 529815: 10, 530099: 94, 531328: 80, 532866: 81, 535400: 73, 536033: 48, 539799: 75, 541117: 84, 542554: 98, 543900: 18, 544220: 84, 544741: 27, 544793: 16, 545372: 94, 547229: 73, 549770: 22, 551063: 59, 553056: 85, 554469: 14, 554858: 29, 556429: 99, 557382: 43, 557928: 20, 558528: 10, 562122: 54, 562695: 24, 564534: 83, 564736: 33, 564784: 93, 565299: 0, 567121: 43, 567171: 95, 571399: 34, 571799: 49, 572382: 14, 574109: 23, 574184: 67, 574188: 33, 575234: 68, 576722: 77, 579351: 25, 581925: 63, 585773: 31, 587248: 88, 587641: 33, 588817: 33, 589837: 15, 591872: 29, 592301: 97, 593294: 11, 596138: 4, 598426: 93, 598939: 34, 599677: 5, 600256: 15, 600587: 16, 600603: 86, 601183: 21, 602690: 50, 605561: 18, 605584: 56, 607079: 61, 608789: 35, 609388: 13, 610718: 70, 614264: 53, 615172: 72, 615794: 37, 622331: 83, 622907: 45, 624060: 42, 625602: 90, 626057: 40, 626392: 34, 627030: 25, 628628: 22, 630110: 51, 630155: 92, 639485: 80, 640191: 28, 640745: 90, 641873: 51, 644961: 61, 646811: 61, 647914: 19, 648351: 17, 649081: 56, 650669: 26, 653242: 23, 654286: 35, 654348: 43, 656598: 88, 657473: 34, 660097: 87, 662188: 45, 662734: 67, 663292: 9, 664443: 19, 666713: 42, 667174: 64, 667431: 97, 668750: 28, 669219: 45, 669411: 2, 675492: 64, 677045: 9, 677665: 46, 677782: 8, 679390: 54, 681627: 54, 683656: 0, 686214: 37, 686373: 96, 687728: 68, 688464: 13, 688579: 51, 689222: 14, 691801: 42, 694145: 84, 694148: 61, 695066: 91, 695397: 95, 696479: 4, 697028: 73, 697352: 4, 697498: 17, 697912: 45, 698589: 63, 699746: 91, 700757: 75, 701252: 68, 701352: 99, 701764: 60, 704095: 47, 704956: 50, 706967: 58, 707167: 84, 707745: 75, 711861: 78, 712103: 84, 712395: 96, 712423: 43, 714530: 69, 715765: 79, 719517: 89, 721161: 70, 721391: 35, 722005: 29, 723684: 74, 724907: 14, 724995: 65, 726695: 5, 727756: 85, 728649: 25, 728782: 35, 729164: 0, 729196: 44, 731409: 10, 732665: 17, 733689: 67, 734338: 15, 734532: 56, 737976: 10, 738420: 27, 739803: 3, 740440: 97, 741909: 83, 742516: 60, 743000: 47, 745293: 54, 746294: 20, 748150: 19, 748402: 71, 750683: 28, 752984: 45, 755731: 34, 756632: 74, 756676: 82, 757953: 0, 762306: 34, 762561: 37, 762837: 26, 763080: 40, 763342: 54, 768276: 24, 769696: 84, 771040: 32, 772851: 52, 774483: 33, 775233: 25, 780129: 1, 780305: 25, 785015: 23, 787121: 2, 787531: 44, 787574: 2, 787657: 55, 787991: 70, 790158: 47, 790247: 29, 791641: 63, 794199: 97, 797421: 25, 797960: 46, 799704: 4, 799920: 42, 800396: 39, 801280: 36, 802421: 32, 805092: 17, 805576: 40, 806799: 13, 811563: 38, 812621: 67, 812888: 58, 816149: 24, 816931: 47, 818452: 13, 821387: 58, 822976: 40, 828679: 17, 834254: 39, 835090: 41, 835951: 4, 836173: 2, 836680: 14, 838158: 64, 838561: 52, 840792: 95, 841037: 60, 841057: 48, 842049: 7, 843617: 83, 844401: 52, 844760: 15, 844882: 82, 847858: 59, 848163: 72, 848787: 54, 849921: 80, 854512: 69, 855501: 89, 859521: 10, 860991: 2, 864694: 98, 865393: 21, 867804: 77, 867948: 97, 869059: 39, 869785: 58, 871541: 21, 872701: 88, 873297: 22, 877351: 40, 880464: 68, 882821: 96, 882996: 80, 887066: 34, 891660: 79, 895934: 78, 897747: 23, 898360: 97, 900316: 47, 902358: 27, 902437: 86, 903947: 68, 904936: 44, 906636: 41, 913986: 18, 914003: 31, 917383: 60, 919996: 24, 921602: 50, 922515: 39, 922586: 73, 924845: 99, 925509: 84, 926343: 76, 927182: 31, 927647: 96, 928755: 56, 930592: 77, 933736: 89, 933800: 15, 934284: 57, 935283: 87, 936070: 0, 936390: 83, 939136: 54, 942111: 92, 947293: 11, 947958: 23, 948698: 64, 951145: 67, 951867: 60, 951948: 58, 952282: 28, 954353: 85, 954684: 95, 959580: 98, 961752: 98, 965734: 5, 965755: 8, 966623: 44, 966777: 40, 968564: 56, 968989: 66, 969414: 84, 970077: 64, 970856: 77, 971791: 54, 972685: 80, 974642: 17, 975566: 58, 977047: 89, 977528: 2, 980054: 44, 982401: 38, 983626: 15, 987473: 74, 987734: 82, 988234: 57, 990838: 41, 993594: 58, 997452: 67, 997606: 96, 997751: 24, 998028: 91, 998522: 26},
		{5464: 9, 5466: 59, 10744: 86, 33789: 40, 40590: 63, 43748: 32, 45781: 31, 52143: 90, 53769: 77, 57337: 69, 64309: 20, 65910: 43, 73802: 56, 75645: 15, 79312: 93, 86296: 15, 98368: 66, 102766: 57, 107912: 59, 108635: 49, 110037: 73, 111177: 79, 122212: 47, 122577: 58, 132114: 52, 132232: 89, 136762: 77, 136789: 20, 144056: 17, 145000: 90, 151342: 49, 153973: 37, 164224: 96, 168626: 65, 173550: 80, 198630: 82, 200014: 19, 203048: 23, 203307: 63, 216591: 15, 221654: 77, 223182: 77, 228226: 50, 236397: 29, 237458: 17, 243300: 78, 243431: 9, 247475: 95, 265210: 30, 268400: 91, 276692: 69, 282066: 73, 284148: 89, 284904: 62, 285096: 27, 290710: 29, 295596: 10, 301811: 75, 303835: 89, 314781: 54, 316929: 40, 319724: 99, 321840: 58, 322439: 17, 336339: 45, 337269: 87, 345548: 50, 346041: 17, 350667: 26, 351727: 17, 353544: 42, 353822: 86, 355212: 69, 358785: 46, 370178: 1, 373634: 99, 381539: 93, 381733: 44, 384350: 15, 404770: 45, 406667: 34, 422187: 99, 427887: 16, 431568: 63, 439595: 83, 441495: 37, 445802: 35, 447818: 84, 452104: 7, 460747: 3, 473002: 9, 478921: 21, 482838: 16, 484879: 78, 487511: 48, 491848: 69, 498758: 33, 507332: 66, 507349: 50, 509188: 41, 510335: 95, 518977: 71, 524882: 46, 535200: 37, 541886: 88, 551429: 92, 552855: 93, 571353: 42, 571864: 71, 581200: 90, 586057: 77, 590847: 68, 598739: 0, 601610: 86, 609683: 74, 610139: 39, 618485: 4, 622121: 78, 638448: 60, 641846: 32, 644069: 29, 646643: 89, 649411: 38, 654548: 57, 660140: 37, 673395: 47, 697102: 72, 703581: 57, 704641: 95, 710989: 44, 721950: 2, 722650: 79, 733593: 85, 734462: 10, 734496: 65, 736375: 4, 737828: 85, 742891: 78, 748616: 10, 764220: 81, 767025: 90, 767918: 8, 773479: 23, 774990: 53, 777275: 50, 779054: 52, 782652: 37, 803761: 62, 816359: 46, 819933: 19, 839291: 17, 841075: 19, 848796: 93, 850774: 41, 851172: 53, 852455: 2, 855539: 53, 863706: 27, 865030: 58, 878005: 14, 884603: 62, 885344: 59, 887533: 45, 914275: 18, 922800: 48, 930393: 72, 935449: 94, 945993: 96, 948610: 41, 951164: 8, 964601: 91, 965264: 90, 967766: 49, 980057: 6, 989996: 33, 997735: 71},
		{250: 58, 2092: 83, 3940: 9, 7107: 61, 32285: 55, 42581: 1, 57607: 6, 62266: 16, 67052: 19, 77481: 62, 106045: 22, 113890: 5, 125438: 50, 143514: 32, 153795: 96, 154264: 23, 167487: 79, 169202: 18, 176888: 73, 178940: 16, 183970: 3, 186632: 31, 187112: 69, 199048: 31, 213101: 8, 223991: 61, 230544: 4, 238191: 72, 244427: 29, 253141: 41, 268256: 45, 274831: 14, 288758: 47, 289008: 24, 289387: 93, 304443: 27, 305341: 84, 307329: 67, 309102: 71, 316821: 14, 317610: 43, 321476: 39, 322274: 51, 324293: 61, 339922: 70, 354637: 29, 358274: 8, 365213: 18, 379290: 84, 392343: 30, 410604: 9, 419994: 80, 456353: 68, 475758: 84, 487407: 53, 500684: 16, 507992: 13, 509796: 74, 515128: 86, 534158: 61, 547640: 93, 556347: 25, 557018: 9, 557474: 16, 558278: 69, 560026: 30, 563514: 45, 591949: 41, 597566: 11, 607950: 11, 626681: 11, 640826: 22, 655216: 20, 660711: 17, 669985: 39, 696273: 35, 700605: 75, 728751: 79, 734298: 0, 742666: 45, 749679: 19, 756679: 91, 759332: 72, 762785: 75, 765996: 78, 780281: 15, 796088: 35, 806484: 84, 808197: 1, 812348: 90, 839001: 59, 866034: 44, 872543: 36, 879447: 14, 882504: 5, 886442: 94, 911425: 25, 935290: 77, 936060: 2, 957042: 28, 960506: 30, 961658: 97, 965351: 91, 979931: 13, 981213: 87, 989867: 75, 999341: 54},
		{598: 30, 1065: 56, 1260: 60, 1528: 42, 2568: 92, 3492: 84, 4907: 6, 5500: 25, 7881: 78, 8310: 26, 8860: 38, 9607: 66, 11542: 14, 11663: 40, 12332: 44, 12967: 69, 13598: 88, 16440: 62, 17023: 76, 18344: 34, 19958: 92, 21063: 45, 21865: 99, 22676: 95, 22831: 63, 23008: 2, 23126: 75, 24046: 32, 24416: 65, 26503: 89, 27011: 69, 28250: 64, 28535: 22, 29235: 80, 29317: 35, 30892: 46, 30929: 56, 31340: 97, 32405: 1, 32442: 97, 33080: 28, 37294: 31, 37360: 61, 39479: 94, 39961: 81, 40247: 87, 42230: 47, 42879: 46, 43886: 62, 44062: 66, 47884: 19, 47933: 48, 47965: 62, 49805: 7, 50131: 91, 54419: 5, 54849: 12, 55331: 74, 55669: 2, 58779: 32, 59713: 39, 59919: 19, 60441: 48, 62360: 71, 62430: 78, 64449: 64, 67446: 13, 71246: 46, 71571: 84, 72235: 30, 72862: 14, 73058: 91, 73426: 8, 73468: 63, 73638: 7, 75351: 11, 76679: 1, 77650: 4, 78053: 80, 78792: 94, 81534: 40, 81929: 6, 81972: 14, 83395: 82, 83682: 90, 84395: 89, 85489: 83, 86324: 79, 86396: 57, 90540: 61, 93393: 39, 94528: 90, 95998: 3, 97010: 22, 99350: 36, 103735: 38, 105939: 99, 108035: 84, 108204: 62, 109272: 80, 109552: 13, 110726: 21, 112876: 22, 113583: 15, 115370: 28, 115707: 2, 116004: 92, 119682: 52, 119814: 52, 119836: 65, 120060: 10, 121363: 25, 121952: 7, 122676: 65, 124357: 65, 124555: 60, 124964: 85, 125459: 97, 126161: 23, 128112: 84, 128529: 69, 129625: 65, 130594: 68, 130723: 33, 131046: 18, 133200: 89, 133380: 93, 134304: 54, 135053: 84, 135822: 9, 135920: 96, 135984: 41, 137195: 5, 139756: 36, 144124: 47, 145241: 91, 147100: 0, 149204: 81, 154456: 53, 155920: 84, 155939: 20, 156362: 92, 157193: 68, 157254: 38, 157651: 24, 159114: 48, 159372: 71, 159903: 49, 161065: 18, 161938: 20, 163562: 16, 164235: 68, 164639: 37, 165206: 13, 165361: 25, 166247: 11, 167366: 78, 168631: 3, 168636: 23, 168944: 48, 169686: 5, 172412: 7, 173044: 95, 173803: 95, 175243: 30, 175852: 66, 176558: 50, 176757: 14, 177388: 37, 177496: 57, 177578: 23, 178173: 29, 178432: 72, 179703: 32, 179737: 74, 181424: 10, 182332: 46, 184407: 94, 184790: 10, 185441: 92, 185661: 19, 187823: 90, 187833: 77, 190428: 43, 191140: 48, 192971: 70, 193540: 4, 193938: 2, 194420: 17, 195309: 94, 196049: 81, 196706: 10, 196956: 21, 197828: 42, 198371: 55, 199335: 92, 200870: 68, 202002: 38, 202187: 31, 202627: 10, 204296: 68, 205244: 32, 205476: 10, 205477: 93, 205823: 80, 206740: 55, 207139: 76, 207820: 37, 209834: 93, 210211: 83, 210938: 20, 215015: 91, 215305: 94, 215799: 34, 219994: 64, 221518: 52, 222281: 58, 223786: 1, 223875: 59, 224280: 29, 224744: 36, 227033: 31, 228806: 96, 229287: 55, 229527: 86, 229532: 15, 229971: 33, 230880: 9, 231223: 51, 232700: 12, 233447: 63, 237221: 32, 238055: 0, 239057: 75, 241881: 31, 243643: 21, 245080: 73, 245296: 78, 245627: 96, 245688: 28, 246240: 75, 248101: 49, 248696: 43, 249193: 96, 249339: 87, 252326: 82, 254218: 46, 254396: 87, 255569: 98, 255700: 97, 256014: 46, 256725: 96, 256891: 29, 257521: 14, 260766: 54, 260920: 64, 264355: 31, 264531: 56, 271097: 0, 275861: 1, 275899: 63, 275910: 44, 277184: 78, 277996: 94, 279929: 56, 279996: 44, 281399: 90, 282976: 48, 286796: 74, 289727: 64, 289909: 82, 290484: 15, 293486: 35, 293574: 45, 295304: 87, 295592: 71, 296079: 25, 298124: 73, 298683: 85, 298979: 7, 299644: 9, 299922: 10, 305100: 13, 305384: 66, 307657: 63, 310597: 54, 313366: 59, 314081: 34, 314465: 41, 315321: 9, 318991: 43, 319024: 65, 319508: 38, 319625: 26, 322611: 24, 322968: 43, 323122: 36, 323277: 36, 324539: 86, 327582: 72, 328477: 85, 330500: 44, 331024: 66, 332620: 66, 333257: 42, 333371: 71, 333611: 66, 336706: 48, 337010: 12, 338142: 78, 339253: 11, 342399: 75, 344618: 46, 345166: 71, 345174: 25, 347418: 58, 349423: 15, 349742: 62, 350038: 65, 351975: 30, 352047: 61, 352145: 80, 352598: 9, 353540: 65, 354574: 33, 356152: 21, 356204: 86, 357131: 18, 357720: 35, 359467: 47, 364289: 35, 364671: 58, 365910: 71, 367658: 63, 368307: 12, 368495: 55, 368670: 64, 370310: 68, 372814: 49, 374251: 80, 374321: 25, 375905: 92, 377468: 71, 377606: 55, 379007: 16, 380769: 76, 381148: 8, 382434: 88, 384851: 63, 384872: 24, 385269: 48, 385311: 90, 386620: 68, 387104: 31, 388609: 89, 388730: 87, 390076: 56, 390921: 13, 394675: 30, 395010: 94, 395290: 7, 395379: 26, 396213: 40, 396565: 41, 396681: 60, 396914: 30, 398869: 97, 400142: 54, 402125: 92, 402312: 66, 403700: 76, 404744: 23, 409204: 14, 409720: 91, 411157: 45, 419221: 96, 420847: 4, 421226: 49, 421331: 31, 421447: 73, 422699: 0, 424724: 84, 425529: 94, 426169: 81, 426711: 40, 429397: 5, 430871: 19, 434853: 23, 436437: 54, 436794: 57, 438325: 76, 438651: 75, 438696: 27, 440074: 32, 441110: 76, 442836: 19, 444264: 87, 448259: 26, 450246: 56, 450522: 75, 450871: 16, 451062: 18, 453327: 32, 456632: 74, 456974: 3, 457704: 96, 460905: 75, 463184: 71, 467842: 62, 468183: 25, 469544: 60, 470578: 43, 470664: 84, 471762: 13, 473319: 66, 473320: 29, 474081: 86, 477782: 30, 480220: 90, 480882: 5, 482489: 75, 486770: 98, 488554: 18, 488876: 38, 489473: 3, 491169: 45, 491810: 77, 491996: 10, 494645: 19, 495248: 40, 498022: 37, 498314: 3, 498742: 54, 499922: 91, 500130: 49, 501648: 44, 501883: 47, 502305: 77, 504125: 74, 505197: 51, 505910: 86, 507140: 35, 507790: 92, 507880: 93, 508357: 6, 509260: 48, 513255: 75, 514737: 19, 515806: 75, 517751: 22, 518219: 97, 519738: 7, 520853: 71, 520957: 36, 522035: 85, 523573: 77, 525078: 22, 527266: 70, 529194: 37, 530153: 27, 530771: 91, 532873: 12, 533181: 6, 534719: 41, 535627: 74, 535755: 59, 537949: 27, 541438: 81, 544094: 76, 544874: 36, 545366: 75, 545533: 76, 547846: 34, 548733: 84, 549367: 70, 549621: 86, 550352: 70, 552037: 26, 552876: 93, 554502: 14, 554798: 34, 554925: 39, 554983: 65, 556751: 61, 557611: 9, 557861: 40, 561941: 97, 561957: 61, 562895: 1, 565198: 66, 565326: 96, 565624: 17, 565889: 9, 566401: 30, 567563: 38, 567832: 4, 570220: 94, 570490: 35, 570736: 94, 570938: 24, 570982: 59, 571458: 24, 574516: 42, 578171: 29, 578619: 37, 579836: 31, 579983: 17, 581397: 15, 587659: 21, 587808: 44, 588589: 44, 589344: 95, 590032: 6, 594803: 53, 595217: 53, 596037: 57, 597372: 76, 597702: 88, 600239: 96, 601144: 66, 601611: 28, 602010: 65, 603106: 42, 603343: 33, 603879: 38, 604111: 78, 604510: 49, 605058: 6, 605713: 63, 605897: 57, 607922: 42, 611871: 76, 612412: 5, 612434: 20, 613322: 47, 613877: 27, 614292: 67, 616909: 4, 617750: 61, 618961: 52, 619737: 81, 619989: 71, 620473: 11, 620867: 62, 621870: 21, 622282: 54, 622662: 36, 625035: 73, 625779: 91, 627625: 27, 630444: 19, 630531: 27, 631927: 86, 632735: 81, 633705: 71, 633916: 3, 635463: 95, 635958: 95, 639717: 63, 645122: 13, 647962: 47, 649399: 51, 654583: 3, 655008: 21, 655222: 34, 655449: 42, 655595: 15, 657494: 58, 657638: 61, 660040: 45, 660166: 68, 661502: 5, 662615: 46, 662892: 84, 663487: 44, 663677: 12, 663840: 99, 664080: 97, 664266: 43, 664847: 74, 666728: 88, 666943: 18, 667684: 84, 669888: 34, 670917: 77, 671630: 92, 672762: 14, 673143: 10, 675022: 42, 675483: 64, 676154: 90, 676293: 23, 677035: 49, 677271: 30, 677689: 64, 678680: 5, 679682: 63, 680309: 54, 681992: 57, 685308: 68, 685992: 17, 688465: 74, 688724: 70, 690377: 52, 691076: 30, 691629: 27, 692122: 54, 692378: 83, 695828: 36, 696881: 2, 697002: 57, 699732: 48, 701181: 57, 701581: 51, 701584: 99, 701976: 27, 703467: 42, 705454: 82, 705870: 1, 707460: 22, 710095: 17, 710292: 15, 713012: 85, 713646: 55, 715366: 1, 716782: 72, 717224: 75, 717351: 69, 717670: 68, 720428: 94, 721118: 29, 721380: 98, 722030: 10, 722680: 42, 723690: 63, 723944: 54, 724781: 59, 725396: 13, 725712: 38, 725762: 96, 729442: 81, 731528: 15, 734001: 87, 734511: 16, 735447: 13, 735796: 38, 735862: 15, 737799: 93, 737878: 74, 738358: 68, 738810: 90, 739323: 80, 740880: 82, 741170: 81, 741197: 87, 741302: 7, 741607: 36, 741828: 59, 741929: 9, 742704: 50, 742789: 63, 743610: 63, 743613: 44, 745203: 6, 746988: 71, 746993: 84, 749003: 29, 750214: 14, 750228: 10, 750509: 47, 751160: 20, 755554: 30, 758176: 25, 758312: 8, 762047: 69, 762859: 54, 764339: 1, 765735: 60, 765920: 15, 766245: 14, 766492: 16, 766890: 73, 768066: 78, 769194: 88, 770806: 60, 774686: 3, 775493: 91, 775523: 11, 775928: 41, 777740: 98, 779042: 93, 779328: 15, 779996: 6, 780416: 52, 780866: 40, 781025: 16, 783483: 36, 783886: 75, 784203: 7, 784802: 93, 785296: 75, 785456: 45, 786484: 2, 786913: 58, 787111: 77, 787888: 56, 788831: 20, 789045: 54, 789660: 2, 791949: 57, 793190: 35, 793547: 97, 794103: 53, 794380: 25, 795421: 55, 797797: 32, 798305: 54, 798759: 4, 800283: 38, 801441: 82, 803000: 57, 803118: 84, 806603: 82, 807107: 63, 807407: 93, 809316: 30, 810005: 92, 814695: 90, 815390: 97, 817198: 78, 817370: 76, 818081: 6, 818272: 93, 820821: 88, 821408: 18, 821819: 78, 822189: 54, 823437: 86, 823730: 5, 824300: 30, 825043: 88, 826902: 5, 828710: 27, 829968: 48, 830199: 97, 831060: 70, 832456: 85, 833810: 16, 834008: 60, 834789: 54, 834821: 34, 835678: 21, 836942: 74, 838116: 69, 838187: 55, 838436: 49, 839333: 20, 840284: 46, 840494: 86, 841076: 50, 841586: 30, 843683: 53, 845396: 0, 846623: 26, 847088: 79, 847700: 72, 850570: 87, 850587: 32, 851580: 94, 852222: 76, 854323: 14, 854681: 28, 855302: 38, 856226: 64, 856888: 95, 857103: 3, 860591: 91, 860688: 82, 861383: 8, 861760: 11, 862928: 66, 863434: 29, 863696: 1, 864991: 93, 865889: 27, 866449: 39, 866890: 27, 867175: 31, 867329: 59, 872112: 38, 872277: 29, 872303: 40, 873381: 16, 874653: 47, 874975: 14, 875135: 0, 875415: 12, 876116: 83, 879364: 67, 879947: 86, 880820: 20, 881637: 70, 882377: 77, 882799: 23, 883313: 96, 884282: 79, 884501: 94, 884571: 40, 885608: 25, 888712: 42, 888794: 75, 888856: 70, 891943: 29, 892177: 41, 893242: 69, 894805: 78, 894822: 44, 897054: 35, 897192: 29, 901220: 78, 903448: 18, 904689: 21, 906708: 93, 907563: 62, 908042: 86, 909498: 2, 911010: 75, 911911: 64, 911997: 31, 912461: 59, 912488: 56, 912611: 91, 913277: 5, 914371: 41, 914471: 18, 916415: 58, 917079: 18, 918399: 9, 918450: 84, 919066: 93, 919595: 52, 921655: 42, 921673: 78, 921799: 28, 921890: 81, 922106: 8, 922310: 44, 923117: 12, 925494: 30, 927415: 38, 928358: 76, 928411: 35, 930233: 46, 931197: 87, 933015: 76, 934080: 57, 938310: 32, 938531: 67, 940712: 4, 941423: 33, 942391: 69, 945401: 21, 948004: 59, 949535: 95, 949881: 66, 951741: 7, 953246: 88, 954537: 59, 954612: 73, 955475: 43, 956007: 23, 956372: 37, 956819: 40, 958748: 84, 960924: 83, 962564: 98, 963201: 19, 963773: 20, 964219: 75, 964450: 52, 965019: 17, 965331: 77, 966723: 56, 967569: 75, 967725: 77, 968651: 63, 970250: 48, 970448: 60, 970804: 38, 970936: 34, 971039: 92, 973104: 89, 974201: 56, 975345: 14, 975603: 7, 975758: 13, 976440: 1, 977842: 17, 979602: 61, 980588: 88, 980738: 0, 983484: 70, 983586: 72, 985771: 31, 987256: 51, 987445: 49, 987713: 77, 987741: 8, 987971: 15, 989633: 39, 990015: 72, 990607: 24, 992791: 84, 994434: 24, 995823: 25, 996014: 97, 997730: 44, 998157: 14, 998370: 97},
		{147: 7, 2199: 23, 5106: 28, 5975: 90, 10160: 3, 10338: 6, 10574: 65, 13843: 45, 15303: 8, 15808: 25, 20949: 41, 21606: 51, 22107: 99, 23361: 87, 23972: 29, 24696: 53, 25908: 66, 26243: 52, 27025: 73, 28059: 1, 28840: 13, 32185: 1, 32405: 14, 33163: 87, 35330: 32, 35910: 23, 35995: 13, 36098: 82, 39606: 94, 40928: 18, 41592: 54, 41858: 78, 41961: 81, 42070: 28, 43433: 3, 44211: 38, 45015: 83, 45639: 60, 45777: 31, 45845: 94, 46292: 20, 46499: 58, 49818: 29, 50866: 68, 54336: 40, 56454: 37, 56662: 8, 57926: 66, 58945: 99, 59261: 92, 60386: 40, 60450: 92, 60908: 5, 61464: 58, 61771: 82, 63625: 99, 65172: 30, 71489: 15, 73061: 54, 73437: 17, 75413: 44, 77616: 60, 77830: 31, 78670: 63, 80254: 67, 81086: 77, 82766: 68, 84205: 39, 84773: 6, 85723: 2, 87523: 89, 88130: 22, 90189: 78, 90868: 27, 90976: 34, 91394: 60, 91494: 61, 91755: 57, 96371: 47, 96515: 99, 96763: 50, 97242: 79, 97918: 2, 98012: 35, 98202: 57, 99166: 53, 99216: 15, 100051: 10, 100631: 42, 100760: 71, 106731: 39, 107402: 37, 110373: 4, 110404: 96, 111412: 26, 112475: 2, 113689: 18, 114913: 72, 117522: 67, 119733: 67, 122959: 91, 123184: 93, 123492: 32, 124413: 60, 126355: 46, 126730: 42, 128215: 15, 128399: 2, 131232: 83, 131975: 86, 132384: 24, 133247: 70, 136144: 77, 137511: 55, 138987: 42, 145080: 30, 146185: 28, 147563: 61, 148759: 37, 149448: 47, 149643: 96, 155581: 83, 156669: 41, 157317: 45, 159943: 39, 161949: 88, 163625: 69, 163752: 59, 167366: 58, 167920: 77, 168465: 44, 169754: 78, 173970: 6, 174289: 81, 177043: 65, 178234: 68, 179300: 50, 179572: 47, 180042: 20, 181934: 25, 182043: 46, 184327: 28, 184541: 99, 185282: 63, 186074: 61, 186086: 98, 187178: 21, 188054: 59, 189177: 4, 189282: 64, 190944: 29, 192102: 38, 193876: 74, 194952: 91, 198934: 30, 203128: 96, 206900: 22, 210341: 16, 211327: 53, 212374: 53, 213589: 41, 213734: 74, 213812: 55, 215160: 90, 215713: 61, 215904: 36, 217190: 30, 220448: 70, 222290: 40, 222593: 11, 227252: 88, 228460: 78, 230625: 47, 231274: 85, 235816: 10, 236547: 64, 236799: 41, 239172: 75, 242665: 65, 244026: 81, 246243: 52, 247373: 62, 248664: 37, 248725: 70, 250119: 42, 250161: 36, 250395: 29, 251155: 74, 251204: 62, 254199: 15, 258472: 38, 259023: 98, 260863: 23, 260899: 22, 261482: 84, 269301: 17, 270972: 98, 271116: 19, 274363: 35, 275727: 94, 277276: 8, 277841: 95, 278565: 73, 279263: 25, 279369: 58, 281218: 18, 281858: 17, 284650: 60, 285168: 15, 285406: 39, 285700: 6, 286594: 95, 287719: 48, 287790: 38, 288074: 57, 288143: 99, 289472: 24, 290156: 26, 294949: 85, 298674: 29, 299078: 63, 301069: 31, 301227: 44, 301346: 1, 301701: 27, 301742: 66, 308613: 60, 308812: 34, 309107: 0, 309355: 48, 309474: 17, 311773: 22, 313844: 16, 314028: 80, 315870: 66, 319041: 32, 319650: 97, 321637: 95, 322372: 91, 324264: 68, 325019: 69, 330427: 4, 331966: 71, 333983: 21, 334988: 94, 336800: 17, 338496: 21, 339158: 73, 340143: 39, 341535: 26, 341559: 58, 342785: 61, 342865: 71, 342892: 75, 342993: 81, 344724: 7, 352915: 67, 356352: 63, 357189: 5, 361159: 94, 361439: 32, 361713: 85, 363802: 28, 365622: 6, 366020: 17, 366725: 44, 366929: 6, 367617: 44, 368254: 89, 373027: 92, 374201: 63, 374717: 8, 376957: 94, 379787: 74, 382410: 52, 385523: 23, 386109: 5, 386193: 62, 386905: 51, 387198: 74, 388500: 44, 389319: 57, 389883: 72, 390598: 38, 391376: 40, 391995: 8, 396975: 3, 398774: 80, 399497: 64, 399558: 17, 401604: 5, 401837: 3, 402405: 15, 403027: 82, 404133: 46, 406794: 23, 407163: 39, 408866: 1, 412062: 69, 412652: 13, 414219: 46, 416879: 92, 417493: 42, 418124: 63, 418910: 97, 422670: 71, 425469: 29, 425474: 85, 425856: 8, 426358: 13, 427145: 73, 428226: 71, 428283: 30, 428802: 25, 428991: 30, 430009: 55, 430867: 86, 433908: 3, 434681: 72, 435966: 91, 437315: 4, 439341: 79, 439458: 3, 439928: 92, 440001: 53, 440160: 98, 441703: 50, 442080: 3, 443419: 47, 443680: 33, 448640: 5, 450256: 69, 450658: 70, 452707: 55, 453768: 7, 454197: 93, 455422: 27, 456826: 95, 460039: 87, 460965: 70, 463509: 31, 465993: 63, 466633: 42, 466748: 39, 467663: 3, 468090: 88, 470406: 11, 470669: 25, 470695: 8, 471307: 62, 472775: 89, 473701: 97, 474013: 69, 475940: 29, 476967: 29, 477428: 60, 480143: 96, 480466: 90, 480516: 35, 481617: 86, 483613: 41, 486491: 49, 487065: 46, 488016: 24, 489835: 26, 490142: 72, 490402: 43, 490430: 75, 492119: 51, 493601: 64, 493900: 30, 495035: 11, 497126: 20, 499492: 57, 500198: 19, 500254: 21, 503180: 92, 503990: 80, 504130: 20, 504279: 22, 505181: 48, 507199: 28, 508238: 86, 508627: 65, 512221: 78, 512683: 78, 513576: 43, 515095: 96, 518028: 88, 518050: 14, 519387: 62, 522303: 98, 523025: 57, 524021: 34, 524093: 57, 527315: 2, 527822: 80, 527961: 44, 528976: 32, 531261: 79, 533556: 12, 534236: 78, 535209: 87, 537209: 51, 539147: 17, 540099: 68, 542667: 48, 543582: 40, 545682: 68, 545945: 20, 546728: 91, 549778: 17, 551243: 63, 551391: 0, 551946: 21, 552155: 99, 553428: 63, 554228: 84, 556468: 2, 558047: 79, 559680: 94, 562681: 62, 562938: 62, 563905: 62, 565023: 69, 569740: 78, 570634: 40, 571115: 64, 572553: 54, 572708: 47, 572749: 42, 572800: 56, 572885: 82, 573419: 23, 573599: 0, 575322: 81, 575517: 54, 576405: 8, 577124: 80, 582835: 84, 582852: 9, 585973: 28, 586336: 88, 587483: 8, 588553: 54, 588651: 90, 589731: 46, 591683: 87, 592164: 53, 594119: 31, 594604: 61, 595447: 55, 595897: 2, 599253: 85, 600817: 13, 602311: 50, 602548: 11, 604452: 31, 607313: 33, 607338: 86, 608152: 89, 608713: 64, 609461: 10, 610700: 58, 611191: 12, 612698: 25, 613328: 68, 614158: 59, 615815: 82, 617213: 50, 617548: 70, 617641: 97, 619496: 41, 620655: 69, 621160: 36, 622472: 40, 622924: 64, 626102: 95, 627040: 30, 627516: 82, 627857: 58, 631046: 15, 633085: 33, 633332: 30, 636285: 86, 636408: 88, 637370: 98, 638860: 60, 640035: 55, 642782: 0, 644346: 49, 644692: 34, 644961: 53, 645050: 84, 648080: 67, 648475: 43, 651306: 8, 654470: 59, 658345: 45, 659582: 95, 662231: 43, 662416: 79, 662601: 7, 671461: 33, 673217: 8, 674555: 28, 675840: 96, 676254: 53, 676884: 6, 677169: 40, 677592: 62, 682641: 70, 682726: 19, 684534: 87, 685025: 78, 686211: 96, 686824: 22, 687060: 28, 688144: 2, 690643: 62, 694800: 72, 695675: 63, 696377: 33, 704158: 51, 705011: 18, 706374: 72, 707760: 65, 709135: 60, 709888: 52, 710416: 16, 711383: 57, 713508: 47, 713706: 38, 714608: 79, 715038: 98, 715256: 48, 715537: 12, 716057: 92, 717981: 78, 720315: 68, 722991: 67, 724041: 90, 726089: 88, 726136: 95, 726195: 98, 727765: 20, 728574: 51, 730391: 9, 731518: 79, 732078: 36, 732811: 31, 739849: 22, 741364: 67, 742465: 51, 742552: 82, 745257: 47, 748011: 45, 754881: 57, 755209: 39, 756435: 94, 756610: 87, 757418: 75, 758243: 14, 759799: 54, 760249: 42, 761456: 19, 761867: 6, 766247: 35, 769750: 64, 775914: 72, 776428: 55, 781635: 23, 781920: 85, 782900: 66, 784438: 43, 786641: 93, 788649: 83, 788938: 97, 789582: 84, 789905: 1, 790749: 57, 791122: 78, 791398: 31, 791403: 22, 792402: 8, 792755: 78, 794490: 28, 796349: 29, 796368: 18, 796627: 8, 797027: 15, 802737: 81, 804078: 1, 808922: 79, 809196: 88, 810047: 80, 811425: 26, 813117: 64, 813903: 51, 814260: 53, 815074: 30, 816280: 41, 817274: 30, 820980: 58, 822346: 50, 823264: 38, 824041: 24, 824984: 56, 826239: 36, 826594: 54, 829789: 46, 830743: 41, 831805: 47, 833644: 1, 835202: 92, 836256: 58, 838153: 60, 838417: 13, 839633: 16, 841885: 73, 842344: 13, 843381: 47, 843886: 50, 845842: 37, 849030: 99, 849364: 66, 850885: 98, 854109: 75, 854114: 74, 855706: 78, 855719: 85, 857076: 62, 857322: 13, 863695: 64, 864206: 39, 868262: 40, 868309: 41, 869825: 45, 872481: 69, 873379: 78, 873781: 46, 874554: 81, 874930: 23, 875671: 75, 875802: 42, 878328: 22, 879488: 49, 879768: 2, 880312: 52, 880980: 53, 881885: 45, 885588: 72, 886922: 86, 886984: 57, 890303: 1, 890532: 35, 897089: 95, 897311: 89, 898681: 39, 898685: 17, 899685: 98, 900162: 31, 901218: 17, 901273: 79, 901311: 43, 901349: 57, 903128: 23, 905048: 67, 908057: 22, 908082: 99, 908436: 28, 908855: 33, 909444: 44, 909850: 10, 909980: 10, 911917: 11, 912479: 21, 912820: 56, 915204: 19, 916895: 19, 918139: 82, 921020: 69, 921028: 80, 923045: 78, 924218: 3, 925977: 81, 928706: 7, 932400: 53, 932675: 81, 934307: 50, 935870: 36, 936141: 39, 936335: 37, 937232: 99, 941758: 2, 942923: 28, 943465: 73, 944891: 98, 946555: 41, 949704: 24, 951474: 90, 951883: 92, 956507: 17, 956636: 89, 957469: 72, 957698: 82, 958091: 65, 958326: 68, 959118: 89, 959936: 28, 960276: 67, 962231: 89, 965850: 92, 966868: 80, 967721: 35, 968072: 22, 969135: 17, 970511: 72, 972960: 47, 975325: 48, 975604: 99, 975936: 96, 976987: 22, 980336: 28, 982406: 9, 984859: 27, 985111: 62, 985331: 99, 985945: 84, 986854: 92, 991951: 96, 994664: 38, 996100: 51, 996346: 59, 998167: 25, 998521: 27, 999206: 28, 999316: 9},
		{5368: 50, 18858: 50, 44410: 5, 59448: 39, 61358: 22, 61370: 31, 65112: 86, 74203: 89, 88000: 35, 91294: 14, 103377: 67, 114684: 64, 119815: 57, 137806: 93, 154170: 22, 156379: 78, 157804: 31, 161594: 91, 165892: 27, 173721: 39, 180360: 2, 199883: 21, 200162: 99, 201070: 50, 206870: 21, 236470: 44, 245668: 72, 256166: 1, 259372: 43, 269422: 3, 270728: 11, 270804: 81, 278837: 25, 283122: 85, 291451: 43, 307581: 86, 318715: 92, 321801: 87, 325557: 38, 327781: 2, 330765: 12, 367133: 49, 371153: 40, 379166: 88, 381214: 58, 392617: 87, 402077: 61, 407347: 68, 409697: 37, 415735: 3, 423765: 15, 456704: 97, 480124: 76, 491952: 77, 493505: 32, 523047: 4, 533350: 30, 540554: 99, 547830: 38, 548716: 8, 557527: 50, 564089: 25, 575755: 62, 591138: 51, 600218: 9, 648927: 2, 652127: 70, 655248: 50, 676320: 13, 682604: 47, 702287: 77, 702521: 70, 711850: 24, 732558: 47, 737515: 10, 740894: 52, 741030: 90, 741348: 97, 745002: 29, 760333: 73, 766074: 20, 779497: 66, 781565: 65, 791890: 53, 792179: 22, 799343: 67, 800464: 37, 802681: 0, 802685: 12, 805020: 59, 843787: 80, 851189: 19, 859197: 16, 867297: 50, 888002: 20, 890154: 47, 923002: 55, 924676: 13, 924766: 31, 931838: 12, 932148: 48, 933340: 16, 938085: 20, 959415: 36, 965995: 59, 967455: 61, 988119: 35, 997438: 0},
		{4002: 79, 19027: 22, 19652: 43, 21028: 30, 24069: 16, 27627: 37, 31043: 94, 38909: 27, 44445: 94, 44460: 19, 56587: 33, 59184: 20, 88128: 32, 89060: 29, 94969: 46, 103911: 79, 104775: 16, 109983: 75, 117347: 4, 125925: 70, 131876: 94, 137229: 39, 141330: 7, 153587: 97, 167310: 76, 173141: 78, 189296: 80, 214122: 31, 216278: 87, 231657: 57, 232115: 3, 234465: 1, 241733: 37, 253791: 97, 257558: 63, 260492: 17, 260930: 56, 267578: 35, 268255: 74, 268984: 26, 273215: 20, 287322: 82, 293754: 68, 315163: 22, 317005: 67, 321915: 90, 332211: 52, 337803: 68, 353408: 8, 354855: 21, 355831: 59, 360337: 27, 388584: 89, 401863: 71, 410265: 41, 410546: 76, 412377: 0, 418865: 15, 419085: 51, 425228: 85, 427575: 98, 440043: 28, 445282: 33, 449724: 42, 453916: 72, 454447: 62, 455435: 38, 460222: 91, 481890: 78, 492917: 14, 494637: 27, 498853: 78, 500600: 23, 505477: 78, 508134: 98, 509187: 73, 510080: 62, 510154: 26, 511664: 88, 525386: 69, 536028: 12, 539357: 81, 539621: 79, 541386: 15, 547578: 66, 547716: 8, 561166: 19, 566125: 71, 567745: 96, 572777: 63, 607149: 89, 607476: 60, 614829: 83, 620610: 18, 622562: 81, 629701: 67, 657826: 83, 661820: 93, 673443: 19, 686853: 95, 687024: 50, 700163: 34, 706137: 39, 710250: 16, 718780: 91, 721516: 17, 735123: 11, 736049: 30, 736146: 98, 743001: 57, 744528: 2, 745598: 70, 746546: 78, 750283: 11, 757689: 26, 765284: 38, 766763: 69, 767250: 63, 773339: 15, 777359: 3, 798062: 47, 824954: 27, 835039: 87, 838451: 54, 838780: 6, 841469: 22, 844898: 95, 847144: 21, 848184: 29, 855815: 80, 858906: 91, 861828: 67, 871641: 50, 882119: 42, 885610: 26, 887043: 80, 889087: 41, 889165: 46, 890248: 5, 890932: 63, 894805: 11, 913009: 85, 930628: 62, 932437: 12, 936520: 15, 971626: 10, 972110: 58, 982964: 80, 989070: 58},
		{653: 28, 1280: 83, 1713: 73, 3230: 15, 3464: 66, 3811: 65, 4083: 54, 4731: 63, 4752: 76, 5484: 20, 6108: 74, 6497: 23, 6516: 34, 7307: 40, 8841: 96, 9029: 82, 9286: 61, 9419: 54, 10067: 18, 10475: 58, 11699: 49, 11857: 95, 12559: 81, 12713: 1, 13193: 60, 13726: 39, 14863: 14, 14890: 58, 15020: 82, 15427: 78, 15697: 88, 15910: 71, 16113: 51, 16318: 32, 17141: 38, 17617: 23, 18685: 97, 18789: 23, 19032: 21, 19159: 73, 19373: 92, 19971: 66, 20464: 25, 20557: 67, 20618: 81, 20665: 81, 21255: 24, 22369: 49, 22412: 53, 22837: 64, 23597: 48, 25567: 46, 26078: 83, 26099: 91, 26164: 54, 26228: 73, 26248: 12, 27643: 29, 27840: 5, 28576: 67, 28883: 56, 28915: 77, 28947: 55, 29183: 94, 30202: 12, 30385: 99, 30927: 32, 31064: 77, 31860: 62, 32183: 6, 34224: 10, 34987: 80, 35083: 57, 35732: 44, 37061: 89, 37092: 36, 37354: 76, 38216: 53, 38445: 82, 38505: 7, 38565: 39, 39895: 18, 40135: 23, 41308: 15, 41520: 43, 42557: 6, 43115: 45, 43191: 25, 43222: 23, 44571: 93, 44704: 31, 45362: 54, 46245: 3, 46613: 83, 47077: 81, 47160: 69, 47431: 76, 47773: 36, 48951: 48, 49329: 35, 49355: 12, 49865: 26, 50630: 32, 50659: 3, 51406: 89, 51487: 15, 52040: 84, 52798: 64, 53136: 32, 54093: 53, 54670: 96, 54766: 21, 55284: 58, 57241: 0, 57956: 80, 58160: 54, 60577: 23, 62220: 70, 62478: 52, 63948: 46, 64693: 40, 64908: 25, 65279: 88, 65327: 61, 65404: 25, 65807: 70, 65982: 76, 66349: 64, 66391: 19, 66742: 96, 67617: 76, 67977: 78, 68017: 58, 68533: 86, 68910: 94, 69322: 15, 70175: 83, 70449: 78, 71090: 53, 71356: 98, 71895: 71, 72041: 0, 72869: 87, 73443: 14, 74009: 70, 75007: 15, 75337: 2, 75397: 14, 75818: 45, 75987: 28, 76712: 20, 76840: 50, 77158: 12, 77177: 35, 77682: 13, 77977: 40, 78423: 64, 78597: 36, 78941: 45, 80172: 2, 80346: 16, 80654: 33, 80711: 46, 80916: 98, 81233: 61, 81340: 40, 82450: 25, 82753: 96, 82904: 19, 84529: 45, 84646: 8, 85464: 32, 85691: 89, 85803: 86, 85831: 61, 85954: 92, 86578: 88, 86663: 8, 86947: 21, 87330: 24, 90040: 25, 90123: 11, 90181: 81, 90216: 38, 90794: 64, 91459: 17, 92783: 57, 93740: 8, 93993: 8, 94481: 54, 95577: 99, 96827: 33, 97044: 68, 97168: 89, 97727: 53, 99110: 90, 101277: 44, 101308: 95, 101582: 48, 101854: 20, 102556: 88, 102873: 48, 103050: 74, 103076: 72, 103320: 66, 104331: 9, 105242: 68, 105795: 20, 106583: 10, 107103: 66, 107122: 7, 108334: 37, 108966: 5, 109654: 63, 109919: 94, 109996: 37, 110149: 99, 110713: 23, 110921: 48, 111572: 75, 112580: 67, 112836: 74, 113272: 51, 113926: 31, 114648: 1, 115120: 68, 115409: 5, 116776: 56, 116859: 98, 116934: 20, 117022: 58, 117224: 14, 117886: 5, 119597: 51, 119938: 85, 119968: 20, 120546: 67, 120614: 19, 120846: 0, 120852: 11, 120901: 80, 121610: 99, 121616: 99, 122238: 25, 122259: 62, 122808: 88, 122925: 17, 124028: 4, 124406: 24, 124674: 84, 125860: 14, 126030: 74, 126194: 70, 126372: 38, 127721: 84, 127967: 39, 128969: 47, 130002: 20, 130400: 45, 130964: 51, 131505: 48, 131516: 18, 131589: 84, 132403: 28, 132530: 89, 132817: 65, 133184: 85, 133395: 75, 133793: 46, 134027: 18, 134100: 92, 134512: 62, 134735: 67, 135242: 65, 136137: 92, 137036: 17, 137141: 97, 137639: 13, 137917: 1, 138214: 43, 139786: 86, 140689: 25, 140929: 78, 141709: 31, 142110: 9, 142316: 16, 142750: 14, 142787: 52, 142969: 98, 143612: 2, 143934: 46, 144022: 74, 144774: 95, 144886: 53, 148922: 86, 149198: 34, 149471: 62, 149693: 85, 149892: 6, 149972: 78, 150147: 2, 150666: 65, 151095: 75, 151627: 0, 152111: 14, 152238: 58, 152348: 7, 152674: 92, 153250: 69, 153464: 53, 153579: 32, 154068: 50, 155343: 51, 156667: 79, 156751: 23, 157639: 66, 159738: 91, 160257: 4, 161090: 65, 161513: 36, 161976: 92, 161981: 20, 164175: 92, 164433: 52, 164505: 31, 164605: 71, 164685: 7, 164778: 55, 164865: 33, 165509: 41, 165618: 45, 165969: 39, 166566: 81, 167219: 7, 167478: 74, 168550: 11, 168589: 39, 169315: 73, 169548: 33, 169617: 45, 170511: 52, 170715: 83, 170850: 37, 171409: 86, 171465: 43, 171661: 83, 172447: 49, 172719: 59, 173057: 59, 173079: 81, 173796: 5, 174031: 52, 174045: 60, 175234: 28, 175821: 44, 176286: 81, 176413: 36, 177906: 8, 177909: 34, 178130: 37, 178378: 91, 178460: 77, 178783: 52, 179873: 22, 180090: 6, 180354: 43, 180380: 73, 180466: 95, 180820: 48, 181302: 66, 181706: 13, 181890: 42, 182312: 3, 182604: 99, 183513: 76, 184167: 44, 185076: 86, 185696: 89, 186674: 11, 187718: 48, 187809: 95, 187890: 5, 187909: 10, 189036: 79, 189178: 13, 189864: 40, 190083: 93, 190424: 62, 190488: 26, 190893: 9, 192263: 39, 192415: 44, 192941: 62, 194293: 39, 194443: 70, 194948: 21, 194976: 79, 195624: 93, 196338: 53, 196432: 81, 196954: 74, 197115: 3, 197135: 96, 197398: 36, 197426: 93, 198089: 84, 198215: 28, 199906: 23, 200092: 87, 200843: 43, 201178: 57, 201329: 8, 201583: 61, 201854: 30, 202025: 62, 202085: 53, 202123: 4, 202188: 88, 202256: 62, 202347: 49, 202526: 86, 202593: 58, 202632: 72, 203592: 68, 203705: 11, 203849: 8, 204296: 45, 205019: 40, 205422: 79, 205490: 13, 207062: 11, 207229: 36, 208095: 40, 208419: 95, 208746: 96, 209432: 50, 209948: 9, 210309: 53, 210334: 49, 210462: 14, 210540: 54, 210938: 37, 211077: 27, 211800: 82, 212013: 26, 212281: 27, 213145: 21, 213518: 70, 214176: 8, 214206: 56, 214587: 72, 216090: 68, 216292: 11, 216320: 75, 216956: 18, 217049: 83, 219247: 81, 219388: 72, 219595: 56, 219783: 96, 220140: 71, 220924: 10, 222295: 96, 222316: 51, 222360: 28, 222953: 30, 223387: 17, 223698: 2, 224368: 37, 227243: 72, 227542: 38, 227553: 53, 227705: 80, 227796: 99, 228224: 71, 228870: 37, 229450: 26, 230715: 64, 231774: 81, 231997: 60, 232564: 54, 232791: 67, 233542: 77, 233545: 62, 234606: 33, 236046: 78, 236145: 10, 236212: 48, 236430: 20, 236501: 32, 236637: 34, 236906: 25, 236992: 72, 237018: 37, 237380: 6, 238451: 31, 238769: 39, 238818: 48, 238968: 20, 239921: 78, 240204: 59, 240788: 16, 241045: 66, 241428: 93, 241764: 84, 241899: 54, 242690: 62, 242930: 39, 244816: 36, 245106: 69, 245238: 50, 245397: 12, 245948: 2, 246043: 17, 246164: 34, 246509: 23, 247016: 16, 248475: 46, 249393: 89, 249559: 24, 249654: 2, 249801: 1, 250006: 53, 251111: 67, 251566: 99, 252014: 59, 252413: 72, 252582: 37, 252773: 71, 252815: 21, 253104: 83, 253166: 5, 253541: 65, 253954: 12, 254040: 17, 254989: 80, 255152: 44, 255486: 1, 256334: 43, 258181: 91, 259442: 38, 259464: 12, 259873: 84, 260280: 38, 261102: 42, 261383: 53, 261440: 38, 262424: 22, 262640: 2, 263107: 78, 263161: 86, 263421: 98, 264132: 65, 264328: 68, 264647: 35, 264728: 67, 265539: 20, 266123: 44, 267481: 97, 267661: 69, 268210: 8, 268424: 84, 268444: 26, 269349: 47, 269654: 3, 270043: 34, 270559: 26, 271324: 94, 271449: 86, 272250: 12, 272267: 38, 272906: 85, 273717: 1, 273760: 10, 274380: 23, 274923: 43, 275418: 82, 275504: 21, 276330: 37, 278083: 43, 279333: 38, 280823: 56, 281233: 25, 281902: 15, 281995: 41, 282055: 2, 282205: 25, 283166: 24, 285745: 22, 286257: 96, 286546: 49, 287169: 74, 287635: 39, 288220: 29, 288343: 0, 289532: 5, 290349: 5, 290634: 23, 291685: 46, 292613: 43, 292704: 70, 292886: 3, 293880: 94, 294127: 50, 294262: 75, 294597: 54, 295915: 11, 296551: 35, 296983: 69, 298523: 55, 298731: 5, 298800: 5, 298880: 0, 299382: 3, 299397: 32, 299515: 30, 301261: 79, 301610: 4, 302121: 8, 303078: 50, 303238: 53, 303452: 38, 303487: 82, 303576: 11, 304769: 34, 306373: 17, 307384: 96, 307582: 40, 307938: 78, 308050: 50, 308127: 1, 309093: 61, 309703: 35, 310022: 17, 311273: 66, 311823: 87, 311869: 66, 311972: 10, 313967: 75, 314184: 85, 314650: 20, 314982: 33, 315203: 75, 315703: 93, 316325: 93, 316339: 65, 316700: 31, 317104: 23, 317301: 4, 318045: 47, 318315: 80, 318974: 25, 319691: 15, 319760: 35, 320669: 86, 320733: 30, 322590: 58, 322912: 95, 324895: 38, 325716: 95, 325779: 25, 326168: 74, 326395: 37, 326996: 70, 327146: 96, 329264: 98, 329396: 37, 329525: 80, 329538: 32, 329740: 99, 329782: 64, 330552: 34, 330617: 20, 332497: 95, 333337: 59, 333727: 36, 334150: 90, 335266: 68, 335363: 68, 335560: 99, 336351: 43, 336622: 71, 336931: 2, 338285: 58, 338653: 50, 338876: 41, 339008: 64, 339775: 0, 340458: 12, 340568: 24, 341583: 64, 342522: 84, 343295: 26, 343339: 86, 343424: 97, 343647: 8, 346186: 48, 346730: 31, 346770: 12, 346821: 66, 347236: 48, 347314: 30, 348071: 93, 348477: 2, 349772: 80, 349913: 70, 350464: 21, 350597: 13, 350677: 27, 351334: 41, 351984: 42, 352699: 85, 354381: 56, 354407: 60, 356894: 16, 357508: 45, 357578: 34, 358431: 37, 359346: 47, 360011: 21, 360535: 0, 360580: 91, 360632: 89, 361406: 40, 362400: 15, 363544: 69, 363714: 89, 364215: 50, 364450: 60, 365839: 31, 366189: 49, 366539: 86, 366611: 51, 369666: 2, 369678: 97, 369790: 91, 370256: 8, 370309: 87, 371338: 3, 373097: 47, 373171: 48, 373615: 33, 374292: 67, 374757: 19, 375075: 8, 375542: 21, 376107: 57, 376126: 2, 376272: 35, 376587: 60, 378186: 91, 378294: 49, 379198: 65, 380921: 74, 381645: 45, 382159: 8, 383454: 57, 383614: 61, 383615: 68, 384746: 94, 385457: 77, 385677: 81, 385790: 64, 385846: 79, 386204: 89, 386943: 57, 387316: 63, 388027: 49, 388030: 59, 388226: 45, 388326: 0, 388570: 76, 388839: 98, 388970: 85, 389808: 55, 390222: 96, 391197: 40, 391756: 49, 392118: 0, 392893: 48, 393887: 83, 394027: 96, 395414: 31, 395631: 96, 395821: 77, 396000: 38, 396143: 34, 398647: 95, 398931: 28, 399429: 57, 399577: 86, 399578: 47, 400470: 18, 401226: 42, 401489: 87, 401789: 89, 402948: 96, 402996: 42, 403514: 76, 404059: 72, 404487: 77, 404701: 81, 405334: 4, 408425: 91, 408965: 73, 409236: 55, 409339: 68, 409948: 49, 410817: 17, 411078: 90, 412566: 36, 412586: 83, 412649: 3, 412670: 80, 412773: 28, 412786: 56, 414054: 27, 414331: 48, 414436: 7, 414855: 1, 415456: 40, 415891: 19, 416153: 44, 417057: 66, 417506: 86, 418002: 33, 418005: 10, 418335: 14, 418918: 2, 419619: 81, 421095: 76, 421372: 63, 422424: 84, 422915: 6, 423195: 99, 423812: 70, 423950: 89, 424086: 47, 424130: 92, 425370: 66, 425434: 37, 426043: 5, 426168: 92, 426469: 10, 426705: 56, 428360: 11, 428453: 54, 428523: 96, 429329: 95, 430095: 19, 430585: 10, 430655: 42, 430913: 37, 432610: 44, 433878: 9, 434200: 59, 434639: 98, 434776: 33, 436026: 24, 436502: 92, 437338: 81, 437449: 79, 438645: 76, 438782: 90, 439287: 87, 440058: 87, 440243: 59, 440604: 30, 441110: 91, 441609: 5, 441685: 22, 442247: 90, 442429: 90, 442591: 11, 443269: 18, 443881: 76, 444139: 64, 444597: 24, 445845: 13, 446330: 65, 446592: 15, 446626: 38, 446768: 90, 447096: 77, 447373: 29, 447815: 84, 448337: 68, 449096: 44, 449262: 15, 449376: 94, 449691: 10, 449868: 76, 449921: 64, 450826: 84, 451055: 31, 451249: 37, 451394: 46, 451665: 7, 451757: 96, 453159: 99, 453349: 32, 453584: 54, 453722: 77, 453728: 63, 453837: 45, 453895: 52, 454505: 77, 454658: 3, 454801: 49, 455130: 86, 455190: 35, 456138: 85, 457321: 1, 457406: 44, 457469: 33, 458491: 72, 458663: 64, 459596: 1, 460195: 98, 460268: 29, 462672: 47, 462936: 8, 463124: 85, 463566: 11, 464054: 6, 465212: 44, 465457: 93, 466048: 92, 466190: 32, 466819: 47, 467051: 40, 467138: 79, 467139: 26, 467584: 78, 468203: 37, 468206: 40, 468817: 50, 469011: 1, 469071: 86, 469482: 2, 471416: 74, 471705: 41, 471769: 1, 472098: 96, 472327: 45, 474214: 84, 474381: 35, 474433: 86, 474484: 79, 475413: 6, 476750: 20, 477017: 78, 477297: 12, 477722: 59, 478436: 94, 478843: 33, 479297: 62, 479481: 70, 480629: 0, 480676: 85, 481302: 0, 481920: 21, 482500: 23, 482644: 99, 484333: 96, 485042: 73, 486116: 67, 486133: 50, 486580: 2, 487952: 64, 488225: 93, 488474: 67, 488494: 8, 488584: 6, 489717: 69, 491525: 85, 491789: 53, 492131: 33, 492269: 25, 492368: 5, 492444: 95, 492478: 15, 494533: 55, 494995: 79, 495017: 64, 495334: 74, 495625: 81, 495929: 48, 496356: 27, 496959: 43, 497329: 71, 498945: 53, 499065: 53, 499094: 44, 499626: 11, 499967: 11, 500588: 48, 501678: 19, 502185: 43, 502433: 46, 502900: 82, 503795: 65, 503951: 46, 503960: 53, 504053: 13, 504187: 48, 504653: 83, 504705: 69, 505687: 72, 506049: 86, 507134: 74, 507363: 20, 507681: 28, 507815: 6, 509081: 2, 510133: 97, 510863: 21, 512446: 52, 512953: 17, 512965: 4, 513216: 82, 513625: 59, 518501: 79, 519320: 8, 519545: 31, 520629: 37, 520647: 63, 521100: 20, 522201: 39, 522218: 88, 522244: 20, 524535: 51, 524628: 66, 524739: 63, 524830: 8, 525585: 37, 525811: 40, 526081: 8, 526527: 49, 526532: 13, 526799: 46, 527065: 70, 528512: 47, 528685: 79, 528789: 12, 528992: 1, 529602: 79, 529969: 61, 531288: 15, 531483: 38, 532155: 99, 532873: 51, 533017: 46, 533101: 32, 533994: 72, 534014: 69, 534081: 96, 534134: 11, 534854: 17, 535339: 64, 535556: 98, 535697: 51, 535959: 32, 536394: 10, 536441: 79, 537700: 36, 538812: 19, 539087: 31, 539135: 97, 539456: 92, 539672: 89, 539693: 31, 539947: 92, 540646: 40, 541023: 35, 541659: 74, 541925: 16, 542171: 12, 542393: 86, 544267: 32, 545798: 71, 547255: 35, 547390: 82, 547565: 32, 547603: 20, 548345: 97, 549639: 85, 549661: 81, 550014: 57, 550073: 20, 550307: 14, 550510: 6, 550772: 45, 551332: 70, 551654: 34, 552751: 76, 553648: 29, 554271: 1, 554289: 97, 554359: 56, 554878: 80, 555147: 3, 555799: 77, 556717: 26, 556757: 6, 557310: 30, 558512: 13, 558637: 51, 558800: 45, 559708: 13, 559811: 10, 560213: 86, 560897: 19, 560932: 46, 561549: 10, 562185: 15, 562285: 12, 562306: 24, 562388: 88, 562396: 44, 563336: 89, 563750: 21, 563784: 68, 563796: 51, 564069: 56, 564617: 16, 565290: 46, 567378: 81, 567881: 28, 568320: 14, 568658: 33, 569207: 76, 569585: 52, 569658: 31, 569963: 26, 570061: 85, 570443: 87, 571032: 69, 571998: 66, 572143: 4, 573272: 32, 573304: 20, 574104: 59, 574566: 19, 574760: 49, 574829: 38, 574903: 74, 574979: 26, 575233: 14, 575421: 56, 576744: 72, 577560: 30, 577782: 49, 577959: 69, 578033: 14, 578204: 69, 578574: 36, 578939: 42, 579917: 66, 580596: 88, 580674: 72, 580832: 20, 581452: 94, 581792: 0, 581801: 37, 581960: 42, 582130: 35, 582663: 39, 582749: 12, 583097: 43, 583544: 48, 583745: 33, 583939: 63, 584453: 54, 584741: 10, 584888: 85, 585515: 56, 585771: 49, 586042: 20, 586360: 27, 586370: 83, 586398: 33, 586442: 62, 586694: 44, 586932: 51, 586964: 92, 588329: 99, 589029: 33, 589047: 40, 589632: 38, 589661: 51, 589914: 21, 590455: 47, 590850: 75, 591077: 2, 591205: 14, 591634: 37, 592517: 76, 592729: 75, 593378: 39, 593449: 34, 593478: 61, 594541: 71, 594986: 9, 595962: 50, 597273: 90, 597462: 4, 598736: 86, 599117: 63, 599157: 83, 599482: 20, 601553: 42, 601738: 67, 601752: 24, 601857: 61, 602488: 60, 602506: 24, 602560: 55, 602763: 32, 602837: 8, 603023: 0, 603592: 31, 604328: 72, 604517: 2, 605571: 96, 605676: 99, 606893: 79, 607403: 42, 608063: 5, 608307: 33, 608993: 19, 609046: 50, 609171: 95, 609374: 4, 609696: 95, 609768: 90, 610234: 68, 611311: 91, 611313: 75, 611711: 82, 612164: 41, 612561: 47, 612663: 74, 613058: 44, 613233: 97, 613827: 57, 614161: 36, 614226: 40, 616020: 89, 616339: 12, 616492: 50, 616519: 85, 617576: 29, 617870: 21, 618728: 99, 619121: 38, 620862: 42, 621029: 97, 621172: 92, 621915: 8, 622086: 67, 623420: 47, 624713: 87, 624960: 36, 625132: 46, 625913: 89, 626006: 19, 626007: 73, 626044: 5, 626156: 50, 627618: 29, 628386: 49, 628736: 94, 628943: 51, 629589: 14, 629913: 30, 631554: 73, 631599: 41, 632685: 11, 633217: 70, 633708: 4, 633722: 67, 634205: 41, 634684: 91, 635013: 26, 635094: 91, 635189: 68, 637370: 48, 637626: 13, 638969: 85, 639888: 27, 640093: 81, 640174: 43, 640415: 63, 640829: 46, 640847: 53, 642042: 61, 642085: 74, 642118: 89, 642587: 28, 643154: 74, 643933: 15, 644343: 64, 644554: 76, 644994: 58, 645020: 59, 645813: 49, 645962: 83, 646212: 20, 646542: 42, 646607: 85, 646776: 79, 646923: 19, 647034: 29, 647080: 1, 647904: 85, 648641: 36, 649050: 41, 649247: 19, 649969: 68, 651463: 3, 652052: 96, 652518: 13, 653368: 22, 654091: 1, 654201: 80, 654408: 97, 654607: 12, 655524: 3, 655541: 25, 655959: 33, 656045: 89, 657025: 12, 657428: 47, 657928: 78, 658225: 0, 658598: 60, 658926: 48, 659430: 31, 659574: 45, 661646: 30, 661764: 48, 662135: 53, 662238: 7, 662412: 53, 662495: 89, 663690: 30, 664486: 22, 664622: 65, 664652: 40, 665414: 60, 665569: 27, 665588: 15, 666110: 80, 666661: 92, 667329: 76, 669684: 60, 670825: 30, 670897: 50, 671160: 53, 672151: 55, 672666: 84, 673049: 6, 673131: 54, 674383: 70, 674465: 49, 676864: 54, 677108: 56, 677480: 1, 677938: 49, 680422: 48, 680704: 50, 680722: 55, 680990: 13, 681685: 80, 681855: 32, 682291: 92, 683498: 54, 683638: 84, 684487: 27, 685000: 76, 686131: 60, 686726: 38, 686770: 47, 688290: 20, 688943: 3, 689074: 92, 689279: 23, 689281: 76, 689608: 1, 689695: 28, 690458: 6, 690610: 81, 690665: 54, 690795: 37, 691111: 95, 692329: 60, 692672: 89, 693303: 55, 693483: 52, 693535: 68, 694192: 26, 694700: 34, 694981: 5, 695206: 24, 695245: 5, 696088: 92, 696262: 72, 696503: 57, 696543: 69, 698007: 20, 698162: 9, 698296: 62, 698317: 41, 698675: 55, 699156: 35, 699219: 88, 699246: 37, 699585: 71, 700154: 72, 700363: 92, 700496: 26, 702210: 41, 702648: 35, 702785: 94, 702980: 31, 703455: 59, 703540: 23, 703745: 26, 704037: 35, 705858: 27, 706205: 99, 706247: 47, 706592: 43, 707868: 14, 708020: 2, 708500: 62, 709239: 40, 709531: 58, 709545: 34, 709986: 33, 710792: 26, 710994: 2, 711841: 50, 711855: 63, 712388: 40, 712521: 31, 713075: 55, 713918: 55, 714713: 54, 717065: 67, 717445: 14, 717531: 90, 717646: 84, 718101: 85, 718226: 28, 718413: 60, 719337: 72, 719348: 21, 719836: 72, 720359: 41, 720857: 64, 720981: 18, 721343: 98, 722285: 37, 722419: 65, 723438: 23, 724037: 69, 724062: 21, 724456: 38, 724719: 41, 724872: 20, 725082: 48, 725182: 97, 725332: 80, 725719: 7, 725742: 11, 725832: 4, 726387: 28, 728880: 10, 729552: 14, 729696: 20, 731211: 28, 731382: 13, 732042: 39, 732108: 9, 732176: 64, 732211: 95, 732393: 40, 732876: 91, 732928: 65, 733673: 47, 733989: 63, 734715: 51, 734979: 40, 735030: 35, 735101: 70, 735191: 3, 735219: 54, 736927: 58, 736963: 97, 737015: 10, 737183: 76, 737371: 78, 738234: 7, 738239: 30, 738629: 46, 738958: 80, 739877: 7, 739882: 21, 740041: 53, 740365: 33, 740801: 44, 741004: 14, 742492: 65, 742514: 64, 743235: 93, 743874: 71, 744672: 70, 745274: 35, 745346: 78, 745641: 12, 746033: 82, 746475: 2, 746509: 98, 748123: 92, 748872: 65, 750054: 80, 750693: 58, 750732: 18, 751048: 7, 751770: 42, 751779: 83, 753592: 74, 753871: 41, 754738: 78, 755360: 1, 755549: 53, 755584: 82, 755651: 48, 755924: 26, 756898: 28, 758287: 9, 758525: 73, 759110: 6, 759266: 34, 759538: 49, 760378: 71, 761018: 22, 761749: 24, 761776: 69, 761823: 85, 762930: 59, 763117: 54, 763409: 9, 763632: 86, 763768: 61, 764488: 71, 764562: 60, 765077: 76, 765342: 86, 765429: 73, 765707: 88, 765714: 22, 766079: 43, 766592: 91, 767403: 4, 768081: 8, 768484: 89, 768632: 9, 768769: 20, 769080: 25, 769218: 72, 769872: 91, 769969: 58, 770016: 58, 770831: 84, 771025: 58, 771438: 20, 772348: 98, 772545: 2, 773013: 45, 773239: 61, 773479: 69, 773514: 47, 773730: 58, 775282: 21, 775471: 29, 775683: 75, 776976: 26, 777295: 33, 777443: 91, 778328: 57, 779787: 97, 780331: 28, 781066: 45, 782513: 99, 782745: 62, 782848: 60, 783394: 42, 783753: 5, 784264: 35, 784382: 70, 784985: 19, 785436: 92, 785605: 23, 785843: 55, 786112: 27, 786332: 96, 786916: 4, 787298: 81, 788343: 32, 790625: 30, 790659: 66, 791536: 14, 792251: 75, 792418: 66, 793177: 94, 793298: 91, 793750: 18, 793941: 92, 794342: 61, 794794: 54, 794967: 87, 795069: 77, 795112: 97, 795413: 85, 796146: 61, 796217: 82, 796504: 66, 796713: 8, 797400: 90, 797880: 50, 798399: 36, 798555: 26, 799071: 92, 800962: 95, 801035: 90, 801274: 82, 801871: 93, 801886: 13, 802271: 3, 802710: 47, 803771: 22, 803850: 39, 804118: 44, 805504: 45, 805592: 90, 805660: 47, 805736: 13, 806134: 55, 806504: 13, 807892: 98, 808113: 60, 808613: 24, 808663: 8, 809303: 23, 809397: 88, 809541: 20, 809627: 63, 810041: 53, 810935: 86, 810950: 85, 811416: 67, 811551: 1, 811687: 86, 812941: 44, 813491: 44, 813523: 93, 814684: 63, 814960: 13, 816030: 71, 816287: 48, 816578: 15, 817093: 54, 817191: 47, 817274: 77, 817385: 11, 819330: 70, 819541: 3, 819736: 55, 819872: 73, 819953: 51, 820427: 75, 820628: 64, 821199: 97, 821274: 38, 821665: 76, 822157: 50, 823056: 49, 825124: 92, 825373: 41, 825532: 33, 825696: 50, 825775: 56, 826584: 54, 827127: 52, 827382: 75, 827924: 65, 829049: 65, 829553: 6, 830116: 69, 830945: 80, 832344: 10, 832403: 45, 832975: 90, 834404: 3, 834587: 18, 834901: 30, 834950: 84, 835412: 49, 835512: 98, 835606: 50, 835877: 67, 836109: 85, 836303: 9, 836654: 82, 836800: 26, 837088: 13, 837516: 90, 838517: 34, 839321: 53, 839396: 84, 839523: 79, 839744: 51, 839887: 54, 840081: 73, 841745: 65, 841992: 39, 842878: 57, 842913: 68, 843592: 57, 844107: 58, 844144: 43, 844573: 67, 845766: 74, 845903: 76, 846375: 18, 847106: 6, 847528: 86, 847778: 58, 847838: 68, 847915: 97, 848589: 46, 848921: 63, 849653: 88, 850644: 2, 850988: 98, 850998: 65, 851268: 37, 851356: 94, 851864: 2, 852188: 88, 852534: 67, 852891: 46, 853190: 80, 853594: 67, 853818: 42, 853989: 73, 854351: 37, 854517: 96, 855712: 77, 856208: 34, 856298: 24, 856334: 66, 856423: 17, 856571: 78, 857097: 63, 857239: 19, 858276: 27, 858689: 58, 859179: 29, 859829: 63, 859945: 0, 860792: 35, 861010: 1, 861590: 76, 861603: 9, 862660: 48, 863056: 77, 864294: 81, 865698: 28, 866179: 31, 866371: 0, 866463: 40, 866517: 47, 866688: 72, 866903: 65, 867050: 32, 867107: 37, 867111: 7, 867260: 97, 867289: 91, 867818: 54, 867890: 95, 868897: 88, 869632: 8, 869879: 4, 870554: 69, 870788: 69, 871230: 70, 871328: 89, 871380: 40, 871567: 57, 871787: 86, 871887: 96, 872655: 42, 873916: 84, 874052: 82, 874367: 82, 874566: 8, 874999: 97, 875355: 54, 876393: 44, 876889: 65, 877006: 25, 877116: 25, 877320: 47, 877356: 66, 877852: 14, 878372: 43, 878773: 81, 879667: 12, 879733: 15, 879995: 30, 881202: 61, 881535: 80, 881881: 66, 882012: 8, 883196: 28, 883324: 89, 884565: 55, 884594: 62, 884622: 32, 884977: 1, 885601: 51, 885949: 23, 886016: 91, 886497: 3, 887012: 61, 887377: 56, 887994: 4, 888812: 38, 890385: 60, 891253: 17, 892188: 6, 892319: 0, 892516: 66, 892692: 76, 892855: 5, 892911: 62, 893184: 9, 893551: 81, 894407: 89, 895009: 15, 898357: 68, 898564: 46, 901102: 60, 903164: 50, 903249: 16, 903533: 12, 903872: 59, 903974: 35, 904538: 70, 904597: 40, 905929: 7, 906460: 80, 906561: 25, 906751: 59, 907172: 65, 907602: 23, 908701: 16, 908708: 31, 909201: 11, 910288: 38, 910407: 77, 910723: 87, 910982: 34, 911304: 18, 911648: 86, 914196: 6, 915170: 25, 915608: 23, 916202: 6, 917150: 45, 917318: 17, 917650: 17, 918530: 38, 918980: 0, 919096: 15, 919248: 27, 919559: 66, 920467: 36, 921062: 49, 921267: 74, 921288: 31, 921590: 83, 921726: 21, 921854: 57, 922272: 70, 923001: 97, 923133: 85, 924462: 25, 924479: 7, 925722: 6, 926124: 20, 926349: 75, 926355: 9, 927664: 14, 928426: 47, 928504: 19, 928705: 92, 929482: 45, 930044: 64, 930833: 57, 931402: 32, 931782: 11, 931927: 19, 932452: 56, 933267: 15, 933412: 1, 933923: 18, 934436: 59, 934754: 60, 934945: 63, 935825: 38, 936402: 70, 936525: 0, 936762: 56, 937921: 59, 938472: 53, 939008: 28, 939367: 82, 939981: 18, 940232: 87, 940831: 71, 941125: 40, 941212: 82, 941344: 77, 941769: 53, 941911: 46, 942221: 31, 942308: 48, 943013: 25, 943397: 13, 943603: 37, 945024: 35, 945797: 28, 946457: 72, 946937: 91, 948326: 82, 948384: 63, 949373: 73, 950910: 86, 950968: 70, 950982: 70, 951778: 18, 951960: 21, 952202: 38, 952577: 3, 953852: 73, 955881: 3, 955921: 44, 956079: 63, 956820: 12, 957224: 92, 957308: 20, 957598: 72, 957802: 73, 957981: 34, 958068: 84, 958105: 87, 958199: 53, 958218: 75, 958949: 59, 959081: 56, 960025: 36, 960036: 17, 960136: 28, 960219: 31, 961008: 97, 961400: 71, 961573: 57, 961687: 15, 961705: 80, 961727: 97, 961786: 80, 961912: 15, 963104: 32, 963561: 10, 964612: 94, 964676: 91, 965027: 14, 965242: 53, 965753: 56, 966923: 13, 967133: 29, 967136: 82, 967458: 36, 967721: 80, 968382: 55, 968466: 3, 968607: 0, 969225: 44, 969767: 21, 970248: 19, 970251: 46, 970405: 0, 970615: 39, 970696: 94, 970960: 82, 971468: 14, 971724: 22, 972114: 8, 972199: 30, 973090: 2, 973104: 67, 973488: 18, 973680: 63, 973775: 36, 973976: 1, 974093: 29, 974645: 26, 975518: 82, 975592: 14, 975653: 14, 976142: 85, 976282: 70, 977219: 17, 978295: 39, 978567: 39, 980461: 37, 982196: 59, 982356: 51, 983101: 43, 983467: 76, 983872: 29, 984230: 24, 985009: 85, 985960: 55, 986134: 43, 986237: 17, 986667: 5, 987035: 36, 987341: 23, 988004: 66, 988066: 93, 988766: 34, 989664: 9, 989826: 37, 990237: 40, 990813: 9, 990860: 4, 992260: 41, 992547: 0, 992600: 14, 993084: 52, 993257: 96, 993770: 20, 993951: 51, 994549: 27, 994832: 79, 995166: 31, 995488: 82, 997500: 4, 998084: 86, 998800: 91, 999355: 30},
		{1060: 88, 5767: 44, 6015: 55, 6303: 77, 9905: 6, 10705: 78, 12523: 68, 13669: 12, 15534: 65, 17092: 52, 21072: 65, 21990: 58, 22096: 68, 24342: 82, 25407: 35, 25694: 95, 25824: 49, 27122: 32, 27215: 39, 28541: 64, 29084: 16, 33303: 70, 34230: 97, 34572: 57, 34850: 12, 36310: 36, 37586: 57, 37643: 31, 38047: 31, 38829: 3, 39312: 87, 39714: 45, 40073: 93, 43222: 36, 43669: 21, 44710: 92, 44952: 92, 46800: 9, 47080: 95, 51340: 17, 51351: 66, 51807: 73, 53156: 41, 55669: 84, 55884: 70, 59397: 99, 59827: 87, 60926: 57, 62644: 54, 63794: 97, 65417: 96, 66546: 4, 67160: 85, 67586: 18, 69670: 76, 72785: 78, 73060: 31, 73719: 53, 73929: 87, 74134: 5, 75771: 45, 76038: 72, 76607: 31, 79852: 37, 81148: 41, 81307: 88, 81488: 87, 82917: 33, 83532: 55, 85592: 75, 86789: 59, 87090: 88, 87682: 83, 88153: 92, 89394: 99, 92519: 90, 94628: 49, 95524: 70, 96813: 69, 98375: 89, 98585: 12, 101078: 15, 101999: 0, 102067: 60, 103924: 41, 107863: 61, 108757: 84, 111436: 52, 112912: 23, 113155: 85, 113227: 48, 114151: 63, 114372: 1, 115080: 61, 115349: 31, 116694: 81, 116862: 50, 117750: 10, 118350: 29, 118436: 18, 119512: 90, 120194: 23, 120216: 85, 120284: 56, 122926: 59, 125422: 51, 128434: 4, 128846: 1, 130765: 19, 133364: 10, 133924: 60, 135055: 67, 136187: 0, 137013: 89, 138896: 3, 140707: 26, 141348: 21, 142934: 80, 147327: 86, 148553: 51, 148831: 66, 150022: 64, 150061: 75, 150674: 27, 151968: 29, 153016: 92, 160761: 14, 163613: 79, 163679: 92, 163686: 31, 164414: 24, 165610: 72, 166174: 35, 166823: 23, 167550: 79, 169820: 42, 170269: 19, 171816: 83, 172444: 59, 174412: 6, 175384: 70, 177352: 41, 178237: 34, 179019: 98, 180092: 91, 183382: 19, 185273: 87, 186721: 94, 186858: 22, 187252: 36, 188195: 81, 188747: 83, 189718: 46, 192094: 46, 193537: 50, 193811: 25, 195363: 15, 196022: 96, 196863: 58, 197656: 7, 200508: 89, 201072: 18, 201140: 23, 206274: 32, 207846: 64, 207847: 76, 208484: 58, 211773: 34, 212316: 80, 213847: 3, 213952: 82, 216235: 57, 216636: 93, 220514: 32, 220876: 2, 226726: 3, 226932: 12, 229386: 15, 231216: 29, 231802: 12, 233483: 27, 236066: 33, 237869: 74, 238118: 23, 239410: 14, 240527: 67, 242800: 49, 242856: 96, 245335: 88, 246279: 80, 246714: 40, 247069: 78, 247359: 98, 252279: 59, 253730: 41, 253996: 5, 254683: 75, 254788: 49, 254943: 46, 256895: 4, 257083: 95, 257600: 6, 257660: 94, 257668: 97, 261702: 62, 263561: 20, 265301: 5, 265471: 56, 267570: 82, 268869: 76, 269825: 66, 269935: 14, 270046: 1, 271262: 55, 271422: 74, 274390: 82, 277421: 74, 277861: 8, 278233: 57, 278617: 78, 279305: 28, 280675: 24, 282166: 58, 282406: 97, 285371: 37, 287849: 79, 287865: 17, 288172: 87, 289938: 75, 292464: 19, 293254: 57, 293269: 98, 297959: 66, 298112: 0, 298736: 42, 300307: 85, 304850: 79, 305554: 2, 307580: 46, 307802: 94, 308827: 74, 313139: 17, 314620: 96, 314683: 32, 315281: 86, 316364: 93, 318225: 57, 319669: 74, 320094: 72, 320460: 48, 321574: 75, 323624: 18, 323710: 56, 325273: 80, 326318: 18, 327229: 62, 330470: 13, 330475: 0, 334580: 42, 336079: 34, 337130: 27, 337427: 55, 337512: 31, 338099: 60, 340468: 99, 340541: 67, 340745: 70, 340887: 26, 342005: 23, 343186: 75, 347266: 11, 348186: 74, 350681: 70, 351889: 57, 352628: 17, 352846: 94, 354053: 44, 356654: 52, 356928: 81, 357069: 63, 357548: 54, 359018: 53, 360920: 63, 361001: 68, 361472: 98, 362283: 78, 363210: 41, 365059: 77, 365561: 67, 367195: 74, 367644: 19, 367799: 14, 369031: 8, 370372: 51, 371774: 30, 372140: 83, 372142: 90, 374691: 27, 375297: 77, 375541: 80, 376008: 84, 377301: 49, 378554: 13, 380558: 84, 381382: 12, 381489: 48, 381549: 73, 383492: 49, 388947: 86, 389858: 94, 390252: 48, 391416: 12, 391459: 92, 392830: 79, 393341: 74, 393836: 22, 395130: 38, 395579: 67, 396588: 59, 397178: 28, 397885: 8, 399204: 65, 399544: 71, 400380: 4, 400721: 53, 403814: 82, 406200: 56, 406421: 83, 406656: 78, 408078: 92, 408197: 11, 413042: 70, 415022: 82, 415222: 58, 416060: 46, 416766: 9, 417160: 11, 418639: 82, 419121: 50, 420086: 83, 420781: 46, 426520: 61, 427273: 10, 427710: 48, 428136: 65, 428158: 13, 431939: 4, 432377: 13, 434048: 30, 434707: 42, 434963: 84, 435016: 23, 436601: 36, 438380: 79, 440722: 26, 441227: 85, 442074: 72, 442363: 73, 446344: 42, 447303: 17, 448190: 27, 448224: 51, 449687: 78, 451303: 39, 451515: 52, 452014: 49, 452088: 40, 452624: 29, 454137: 26, 454250: 97, 454324: 59, 454362: 2, 455244: 6, 455297: 18, 458251: 68, 461675: 32, 462768: 3, 462889: 2, 463417: 23, 465948: 78, 466148: 90, 467034: 4, 467547: 18, 467916: 68, 468407: 52, 469458: 73, 469567: 26, 471082: 88, 472476: 92, 473001: 76, 474153: 36, 475732: 62, 476853: 3, 477206: 48, 477788: 78, 478625: 14, 478752: 71, 479993: 90, 480243: 6, 480514: 5, 482263: 58, 487861: 83, 487913: 18, 489763: 13, 494077: 79, 495061: 10, 495218: 81, 495277: 44, 497620: 83, 504318: 48, 506161: 83, 508381: 64, 508468: 6, 508622: 51, 509387: 2, 510563: 89, 512903: 89, 514086: 22, 515229: 13, 516264: 62, 516325: 9, 516823: 12, 518092: 90, 518852: 31, 519567: 38, 522063: 18, 523235: 93, 523692: 50, 524477: 44, 524721: 46, 524922: 72, 525275: 72, 527381: 90, 529163: 61, 531458: 64, 532379: 31, 532672: 45, 533962: 88, 535201: 57, 536566: 52, 541858: 58, 542389: 72, 543252: 95, 543716: 51, 545529: 94, 546577: 25, 549140: 61, 550864: 96, 551455: 42, 551838: 92, 552151: 57, 553282: 65, 554324: 30, 554707: 38, 555215: 29, 556253: 2, 556915: 91, 557338: 78, 559803: 55, 560044: 15, 560926: 62, 561310: 11, 561625: 64, 561924: 50, 563155: 42, 569331: 10, 569960: 10, 570788: 0, 571030: 3, 572679: 37, 573096: 64, 574491: 7, 575152: 49, 575326: 72, 575511: 46, 575642: 45, 579027: 33, 581035: 21, 582044: 86, 583405: 81, 583702: 57, 584901: 41, 585216: 68, 585904: 99, 586643: 95, 587834: 66, 587945: 87, 589665: 45, 590478: 22, 595663: 33, 596037: 87, 596572: 63, 596994: 57, 597555: 16, 599138: 92, 599516: 72, 600059: 28, 600724: 63, 601396: 50, 602187: 25, 602287: 14, 604286: 46, 604977: 26, 610429: 60, 610857: 24, 611384: 97, 612685: 7, 613668: 91, 613733: 45, 614438: 93, 616916: 6, 621132: 28, 623381: 25, 624567: 94, 626670: 19, 628772: 47, 629628: 50, 630703: 18, 631304: 43, 637151: 40, 637268: 41, 637923: 54, 639403: 11, 639674: 82, 642231: 1, 645564: 61, 646394: 67, 647729: 75, 648762: 92, 650610: 44, 651102: 87, 653865: 37, 653900: 77, 656077: 82, 656204: 22, 657185: 43, 659756: 95, 660564: 95, 662908: 8, 663655: 85, 664042: 19, 665078: 15, 665867: 67, 665891: 50, 668093: 42, 668095: 40, 671953: 19, 672842: 63, 673173: 22, 675176: 34, 675620: 9, 676609: 95, 678712: 89, 680371: 9, 680915: 47, 680979: 94, 681486: 55, 681969: 30, 683149: 17, 684532: 11, 685131: 25, 685270: 49, 685839: 21, 687065: 63, 687588: 40, 689915: 41, 690218: 34, 691091: 77, 691758: 87, 692634: 81, 695927: 58, 696493: 10, 696808: 98, 697197: 77, 697756: 89, 697844: 44, 698134: 32, 701685: 53, 701730: 19, 704936: 23, 709034: 18, 709042: 67, 709769: 67, 714409: 30, 715669: 34, 716974: 63, 719369: 26, 719859: 33, 720169: 7, 725232: 54, 729296: 18, 731904: 79, 732729: 58, 733178: 62, 735327: 59, 736710: 33, 740600: 89, 744558: 24, 744987: 11, 746422: 94, 746598: 22, 746885: 27, 750078: 45, 751475: 30, 752596: 91, 753156: 74, 754303: 6, 755024: 91, 755083: 59, 755156: 2, 755416: 82, 755841: 64, 758128: 63, 758952: 43, 762002: 54, 764910: 31, 765486: 3, 767970: 50, 769297: 69, 771052: 82, 771866: 84, 771893: 54, 772957: 71, 776422: 89, 777118: 31, 777455: 23, 779623: 40, 780655: 36, 780972: 16, 781701: 50, 782142: 34, 786740: 16, 786783: 50, 787120: 87, 789945: 17, 792741: 92, 794690: 8, 796875: 78, 797325: 77, 798943: 46, 799635: 69, 799920: 21, 801758: 45, 802419: 15, 803085: 22, 803612: 62, 804023: 80, 804682: 76, 805046: 70, 805818: 2, 806605: 74, 806972: 12, 808945: 16, 810244: 42, 810683: 30, 810935: 84, 812700: 96, 815984: 29, 817484: 22, 818627: 42, 818858: 41, 820093: 67, 820781: 75, 822433: 46, 823235: 21, 824210: 12, 824479: 32, 824878: 9, 825596: 5, 825623: 83, 825818: 67, 825875: 3, 830397: 1, 833517: 97, 833621: 32, 833806: 51, 836786: 39, 837105: 79, 837133: 95, 837414: 44, 839871: 4, 840345: 35, 841120: 27, 841314: 45, 845102: 11, 850453: 89, 851081: 87, 853220: 41, 854976: 11, 856204: 78, 856271: 94, 857052: 85, 857372: 9, 858394: 0, 858768: 44, 860173: 99, 862931: 71, 863464: 23, 864285: 28, 866882: 59, 867312: 84, 870455: 12, 871297: 82, 874168: 90, 875774: 52, 878000: 74, 878845: 46, 879641: 48, 880744: 46, 881227: 38, 881352: 51, 881481: 52, 881932: 27, 881980: 13, 883299: 83, 883380: 91, 885103: 55, 885884: 73, 886293: 1, 890155: 35, 890515: 20, 891916: 91, 893079: 56, 893818: 94, 893839: 15, 894233: 15, 896948: 28, 897360: 79, 900791: 60, 901158: 79, 901805: 74, 901898: 7, 902878: 52, 904155: 72, 904496: 84, 904932: 43, 905019: 35, 905021: 4, 907251: 78, 907922: 16, 908768: 60, 909854: 85, 910036: 34, 911882: 85, 912167: 29, 913048: 67, 913188: 15, 913735: 44, 914355: 35, 915191: 48, 917412: 39, 917721: 71, 921550: 6, 921565: 62, 921828: 28, 922564: 22, 923576: 64, 927537: 52, 927568: 89, 931575: 92, 931729: 50, 932272: 1, 932634: 54, 934574: 53, 934946: 93, 938467: 93, 941107: 82, 943981: 66, 946700: 42, 946746: 45, 946965: 3, 947957: 91, 948723: 3, 949811: 23, 951249: 69, 953533: 30, 953828: 67, 955808: 29, 955858: 39, 957811: 15, 960256: 20, 960639: 41, 961361: 11, 961655: 52, 961711: 69, 964037: 64, 964714: 33, 967672: 0, 969120: 90, 969501: 46, 970467: 31, 971858: 58, 972594: 29, 973809: 76, 973858: 62, 974023: 78, 974826: 74, 975979: 37, 976587: 22, 976718: 55, 978113: 59, 978526: 51, 978660: 88, 980455: 36, 982015: 8, 986034: 73, 988656: 62, 991680: 85, 992862: 6, 993134: 15, 993227: 53, 994088: 39, 994112: 38, 994742: 20, 994926: 66, 995006: 27, 999193: 41},
		{4591: 18, 5879: 14, 6009: 92, 6276: 92, 10130: 37, 11676: 13, 18118: 94, 21817: 45, 23678: 63, 28422: 80, 34831: 86, 35091: 58, 35142: 87, 41438: 53, 42375: 86, 42775: 84, 47043: 46, 47638: 81, 49336: 32, 49419: 75, 50033: 51, 51204: 13, 51227: 74, 52888: 69, 53375: 50, 53503: 37, 54054: 29, 57570: 70, 60212: 38, 60428: 63, 62036: 32, 63878: 54, 64755: 86, 66975: 69, 67099: 86, 68192: 26, 70384: 90, 71863: 95, 73283: 93, 74048: 21, 76412: 18, 83168: 76, 85557: 25, 86061: 74, 86582: 79, 87289: 9, 92416: 11, 94889: 93, 96580: 84, 97776: 88, 98109: 60, 98787: 43, 99542: 69, 99563: 2, 99810: 9, 102882: 63, 103692: 65, 105528: 10, 108150: 73, 108287: 40, 110744: 58, 113090: 42, 116196: 1, 116842: 17, 117080: 43, 119678: 27, 122903: 28, 123779: 21, 124134: 64, 126452: 34, 127814: 72, 128245: 35, 133803: 62, 134684: 74, 135291: 43, 136660: 12, 137208: 38, 138055: 8, 139637: 54, 139799: 23, 142332: 0, 142454: 17, 145536: 10, 146898: 59, 150008: 68, 155344: 70, 157760: 77, 159003: 38, 164577: 91, 166829: 11, 168530: 69, 170045: 40, 171619: 12, 172727: 0, 174003: 12, 174727: 95, 175531: 89, 177797: 20, 178705: 88, 179161: 44, 179298: 70, 185309: 19, 187456: 60, 187867: 42, 187932: 20, 190296: 36, 191828: 40, 191847: 76, 193677: 20, 194766: 81, 205968: 68, 206485: 79, 207379: 68, 211656: 58, 212321: 28, 215526: 60, 216217: 12, 224013: 62, 224473: 73, 225481: 90, 226171: 14, 228461: 12, 229520: 98, 229697: 87, 229741: 51, 230466: 2, 233247: 32, 234027: 46, 234807: 31, 235003: 2, 235501: 35, 235866: 87, 239545: 67, 240046: 54, 240193: 17, 243621: 18, 244185: 40, 245709: 72, 245901: 73, 245920: 83, 249404: 82, 254807: 36, 256213: 98, 259858: 92, 260324: 92, 262468: 94, 269031: 17, 270516: 73, 271996: 94, 273200: 76, 273947: 49, 275320: 33, 280387: 9, 280613: 62, 280619: 21, 284764: 18, 285419: 62, 285659: 3, 286253: 93, 289076: 64, 292280: 60, 293957: 51, 295418: 76, 296044: 72, 296619: 37, 296622: 73, 297582: 57, 297997: 76, 298086: 48, 298234: 98, 299395: 70, 300352: 63, 300894: 52, 302299: 74, 304569: 12, 304839: 7, 305472: 92, 308107: 41, 314421: 42, 317099: 46, 321477: 2, 325384: 96, 326565: 61, 333665: 84, 334624: 2, 335333: 5, 335391: 35, 339639: 43, 342063: 92, 344626: 84, 348992: 98, 349075: 54, 350279: 58, 359179: 54, 367061: 21, 367408: 61, 368647: 22, 369067: 30, 372411: 4, 374727: 36, 374846: 61, 381555: 17, 384926: 22, 386185: 48, 386541: 4, 389358: 5, 394776: 84, 395528: 4, 396319: 41, 397741: 85, 401138: 9, 401440: 19, 405238: 50, 406822: 79, 406934: 5, 417249: 31, 417378: 19, 420239: 20, 424437: 42, 427017: 1, 431729: 47, 432099: 43, 438883: 72, 440549: 78, 448480: 80, 448607: 33, 450051: 28, 450939: 94, 452260: 26, 454563: 20, 456980: 70, 457418: 31, 458723: 49, 460194: 61, 461598: 78, 461723: 7, 462031: 58, 462073: 57, 462123: 69, 463505: 76, 465566: 53, 469143: 58, 472362: 44, 473285: 11, 474988: 51, 475277: 30, 476345: 58, 479850: 8, 482223: 71, 482754: 78, 483706: 30, 484803: 0, 485501: 2, 485739: 95, 485873: 73, 496315: 85, 498856: 67, 498920: 17, 500620: 37, 500758: 59, 501412: 62, 503668: 42, 504359: 75, 508889: 62, 510982: 23, 511678: 43, 511977: 52, 517043: 70, 517151: 92, 517892: 17, 518133: 85, 520871: 27, 523989: 92, 526211: 39, 526688: 29, 530194: 66, 538710: 1, 539969: 61, 540679: 10, 541121: 77, 543939: 6, 547547: 5, 547718: 78, 549263: 90, 555368: 24, 555846: 66, 557663: 49, 563464: 17, 565128: 37, 567956: 34, 568005: 54, 569151: 59, 571200: 44, 572868: 92, 576660: 7, 576718: 92, 576730: 29, 578625: 6, 579038: 31, 579477: 63, 584170: 14, 586633: 78, 586670: 97, 588044: 76, 588869: 36, 592670: 28, 594667: 63, 599186: 13, 601398: 7, 606925: 74, 608891: 76, 611602: 27, 614160: 57, 621027: 87, 621344: 82, 621933: 54, 624914: 78, 628835: 90, 631242: 24, 631306: 34, 631634: 81, 632251: 41, 632303: 12, 634855: 52, 637160: 67, 637701: 55, 639498: 95, 640218: 74, 640538: 53, 641119: 66, 644425: 4, 644960: 90, 647767: 51, 647976: 97, 648378: 5, 648555: 82, 649482: 26, 651596: 45, 655101: 26, 655601: 7, 656916: 94, 660750: 34, 665619: 90, 673095: 23, 674051: 1, 676243: 26, 678172: 38, 678293: 93, 685240: 63, 688862: 60, 689676: 91, 696223: 81, 697419: 78, 701692: 13, 703018: 72, 703590: 57, 704967: 52, 705229: 85, 705352: 61, 706180: 83, 707148: 30, 708784: 25, 710551: 71, 712923: 1, 714094: 48, 715264: 60, 718802: 33, 718946: 52, 720042: 83, 721522: 68, 723175: 43, 723186: 67, 724855: 72, 726893: 13, 730698: 71, 731250: 82, 732730: 32, 737016: 33, 737063: 74, 738266: 93, 738517: 57, 745126: 46, 748412: 31, 749994: 91, 750249: 83, 752271: 3, 756040: 31, 757247: 67, 758414: 44, 758654: 84, 759762: 11, 760207: 48, 763143: 3, 763341: 8, 764824: 73, 767125: 57, 770056: 16, 773766: 47, 775587: 3, 777544: 99, 781813: 7, 782464: 68, 783882: 33, 783930: 32, 785391: 98, 786639: 29, 786691: 39, 788546: 38, 792873: 12, 793419: 78, 795876: 12, 797049: 93, 797226: 63, 798076: 85, 799191: 13, 803889: 37, 804414: 24, 806779: 54, 809955: 81, 813310: 39, 817561: 88, 826117: 79, 827176: 6, 827408: 50, 831113: 48, 833142: 69, 837240: 84, 839035: 56, 840402: 10, 842750: 58, 844593: 22, 848244: 81, 850570: 94, 855640: 68, 857794: 69, 858563: 90, 858576: 92, 862713: 39, 863883: 14, 864082: 85, 866918: 59, 867943: 86, 869983: 74, 874864: 90, 876898: 40, 879031: 95, 880256: 39, 884440: 60, 884686: 93, 886224: 8, 894980: 86, 897323: 2, 901035: 50, 902073: 77, 904519: 35, 904775: 93, 905113: 33, 908658: 98, 910545: 61, 914125: 13, 914282: 34, 916474: 23, 920107: 28, 920388: 89, 921022: 3, 925047: 36, 925657: 4, 927615: 10, 928776: 68, 929727: 70, 935196: 94, 940166: 92, 942181: 86, 942651: 56, 945293: 73, 945706: 93, 952597: 46, 953794: 0, 956688: 5, 960163: 36, 961233: 21, 963916: 73, 965067: 65, 966030: 88, 968542: 67, 970117: 77, 970189: 2, 970803: 26, 974621: 57, 975099: 9, 977671: 85, 981660: 81, 982267: 28, 986247: 67, 993259: 25, 994943: 3, 997397: 66, 997627: 37, 997631: 45, 999575: 42},
		{891: 31, 1544: 65, 4765: 80, 5850: 28, 6768: 7, 9039: 99, 9577: 28, 10651: 80, 15746: 55, 17139: 45, 20769: 7, 20782: 50, 25088: 25, 25955: 15, 31832: 64, 32204: 81, 33318: 37, 33419: 67, 34359: 87, 41477: 22, 42337: 76, 43902: 3, 45155: 22, 46737: 91, 58179: 98, 60897: 89, 61752: 63, 62778: 60, 63762: 59, 67804: 60, 74091: 54, 74900: 53, 76120: 92, 76251: 42, 78076: 85, 78120: 14, 78161: 70, 80667: 89, 80850: 69, 81444: 38, 82657: 83, 84770: 77, 86718: 13, 91768: 74, 91853: 21, 93121: 3, 94776: 11, 97085: 57, 98401: 66, 99070: 52, 100433: 8, 100852: 92, 108097: 15, 109045: 20, 109546: 79, 109777: 60, 110972: 54, 112412: 42, 114111: 72, 117154: 89, 117271: 14, 117661: 5, 121278: 30, 126662: 56, 127878: 82, 133830: 72, 133957: 32, 139737: 68, 140589: 64, 149819: 52, 150289: 41, 150994: 68, 156056: 6, 157410: 47, 158434: 16, 159047: 91, 160966: 96, 163006: 34, 164276: 20, 165688: 71, 165967: 65, 169905: 59, 170819: 36, 178941: 49, 179209: 84, 180855: 37, 183217: 20, 185777: 46, 186080: 23, 186152: 11, 186863: 44, 189823: 71, 190530: 80, 190901: 25, 192065: 25, 192604: 97, 193194: 93, 196182: 6, 196898: 99, 197906: 31, 199016: 65, 200452: 8, 200646: 85, 208553: 34, 215151: 77, 217639: 59, 217890: 17, 218479: 93, 225138: 63, 232800: 36, 233270: 11, 233495: 57, 234500: 82, 235595: 13, 237050: 2, 238125: 12, 239679: 31, 240794: 18, 241021: 33, 244195: 77, 244857: 74, 251889: 31, 253870: 97, 258054: 93, 259720: 30, 261361: 59, 265101: 9, 265629: 90, 266720: 23, 267907: 12, 270227: 97, 270586: 43, 270824: 62, 271902: 56, 274176: 54, 275130: 66, 275785: 43, 275840: 21, 276541: 83, 277199: 90, 278706: 66, 279406: 75, 282700: 95, 285376: 76, 286959: 35, 287247: 75, 288014: 74, 289805: 29, 292360: 96, 293820: 64, 294667: 48, 297047: 69, 297497: 92, 298064: 14, 299389: 26, 300536: 79, 300594: 8, 302603: 46, 304587: 37, 307113: 83, 308536: 3, 310164: 57, 310232: 77, 311363: 48, 311546: 63, 311564: 89, 311803: 12, 312136: 85, 312166: 15, 313228: 90, 313279: 44, 314497: 90, 318123: 41, 320304: 32, 322636: 54, 326322: 61, 329689: 80, 330304: 51, 331597: 27, 333870: 53, 337397: 46, 341001: 70, 342067: 22, 343954: 95, 353737: 44, 354352: 0, 354359: 2, 356213: 90, 357307: 96, 359515: 68, 360132: 15, 360184: 31, 364029: 85, 369139: 78, 369326: 36, 369740: 98, 370055: 98, 370352: 0, 371499: 72, 374917: 84, 377371: 43, 378389: 75, 378529: 93, 379702: 70, 380727: 52, 386254: 52, 389384: 54, 394815: 32, 395006: 10, 395409: 51, 397130: 19, 402616: 1, 409460: 91, 409581: 66, 411845: 94, 412386: 72, 413062: 47, 415195: 83, 415260: 42, 415648: 22, 425514: 77, 425810: 18, 426461: 33, 427427: 60, 427776: 65, 431345: 81, 431874: 36, 433746: 78, 436602: 81, 437774: 83, 438600: 70, 446729: 98, 448834: 70, 455484: 45, 455585: 26, 459968: 28, 460156: 88, 460543: 94, 461261: 36, 463403: 29, 471250: 80, 472910: 50, 474616: 25, 480510: 25, 482376: 89, 490948: 83, 491084: 90, 494088: 29, 497140: 48, 500187: 42, 500723: 10, 501599: 3, 502629: 48, 505230: 26, 513688: 9, 514151: 39, 516909: 37, 517165: 96, 517249: 71, 520015: 90, 525269: 39, 526566: 61, 531875: 56, 533518: 13, 533916: 76, 534139: 47, 535735: 20, 544175: 14, 545254: 53, 546587: 50, 549404: 39, 549980: 16, 550988: 31, 552640: 44, 555461: 56, 555894: 58, 556219: 60, 556221: 74, 558969: 76, 560859: 64, 561940: 96, 565574: 86, 567649: 46, 567760: 49, 567946: 21, 568788: 10, 572299: 19, 572316: 27, 573235: 25, 574575: 70, 576695: 22, 577082: 88, 578945: 92, 579057: 28, 581335: 91, 581473: 66, 585764: 26, 589311: 46, 589538: 77, 589594: 3, 590202: 30, 591564: 8, 592218: 11, 592327: 60, 593456: 74, 593481: 8, 593785: 71, 594378: 60, 604138: 54, 607046: 25, 611878: 14, 624704: 30, 629141: 33, 634088: 80, 637616: 0, 638525: 59, 641007: 71, 642453: 56, 646457: 95, 654076: 47, 654098: 95, 654178: 35, 656013: 45, 656188: 57, 656773: 50, 660732: 8, 662252: 44, 675650: 39, 679371: 65, 682183: 51, 682236: 5, 684544: 66, 686292: 39, 686972: 95, 687732: 56, 689855: 82, 690499: 38, 690815: 45, 692217: 17, 692727: 11, 693127: 51, 696045: 40, 696344: 70, 697010: 7, 697120: 29, 698245: 91, 698349: 72, 698664: 17, 701986: 68, 702741: 81, 707582: 11, 708129: 35, 709967: 20, 711879: 74, 712550: 12, 713229: 53, 714379: 16, 716300: 65, 717011: 97, 718897: 92, 719006: 28, 719883: 79, 719967: 95, 720528: 91, 724988: 22, 726419: 91, 726638: 11, 726782: 93, 729852: 26, 732373: 81, 732577: 60, 734469: 64, 734680: 32, 736589: 71, 737004: 53, 737958: 5, 739425: 77, 739432: 78, 751662: 68, 755154: 73, 756804: 31, 757637: 67, 758906: 28, 761198: 22, 762571: 75, 764933: 7, 767826: 36, 768596: 75, 768760: 77, 769542: 53, 769982: 43, 774182: 27, 775738: 88, 777239: 12, 780054: 73, 781495: 84, 782893: 63, 785749: 2, 788053: 78, 788093: 67, 789550: 20, 794040: 66, 794249: 5, 796353: 63, 798621: 66, 800683: 40, 801522: 50, 804326: 25, 804514: 49, 805766: 75, 806473: 30, 809483: 11, 810992: 59, 815366: 6, 817354: 90, 818080: 29, 818145: 57, 818625: 36, 819371: 35, 823024: 81, 824470: 7, 825025: 92, 828106: 46, 829812: 33, 831517: 3, 834657: 22, 838806: 41, 844648: 77, 846493: 73, 848770: 14, 850843: 76, 851841: 69, 853062: 73, 853852: 54, 855376: 37, 855595: 95, 856571: 8, 856909: 55, 859595: 68, 860100: 67, 861458: 78, 862586: 97, 864225: 39, 867426: 73, 872073: 13, 873517: 12, 874097: 1, 875409: 80, 878605: 40, 879666: 44, 880574: 5, 881038: 90, 885471: 23, 887483: 89, 887701: 19, 888681: 57, 888849: 91, 891368: 19, 892762: 18, 894367: 77, 900262: 63, 900723: 42, 903007: 7, 907614: 81, 913303: 3, 914122: 44, 917228: 83, 918194: 59, 919442: 59, 920189: 67, 920637: 25, 929339: 27, 929695: 8, 930968: 80, 931341: 80, 932216: 55, 935001: 96, 939849: 98, 940316: 65, 942002: 35, 944394: 74, 946605: 30, 946923: 65, 947001: 56, 947361: 81, 948490: 26, 949268: 97, 957808: 20, 958338: 93, 958366: 83, 960459: 38, 960574: 94, 960706: 26, 962481: 82, 962881: 32, 974633: 62, 978642: 26, 982009: 18, 985011: 69, 986651: 65, 989052: 48, 991466: 2, 991785: 18, 998539: 25, 999169: 39},
		{626: 4, 1215: 97, 4282: 47, 4992: 16, 8673: 79, 16710: 62, 16754: 35, 17182: 13, 18255: 29, 19095: 17, 21460: 68, 21897: 80, 22677: 95, 24337: 21, 24678: 69, 27032: 5, 27105: 77, 28859: 48, 29641: 5, 31407: 76, 31958: 11, 32397: 38, 33477: 26, 34649: 98, 35625: 29, 37398: 24, 37660: 40, 40579: 8, 40737: 54, 41132: 64, 41742: 44, 46326: 79, 46758: 71, 47012: 72, 50947: 29, 51067: 44, 53108: 41, 54587: 9, 59717: 49, 60435: 86, 61016: 58, 61063: 48, 63297: 35, 63764: 51, 64678: 41, 67596: 49, 67798: 52, 70043: 47, 70124: 17, 72118: 24, 74100: 75, 76443: 99, 79540: 82, 80669: 35, 83400: 91, 87551: 48, 87931: 53, 88254: 58, 89937: 45, 90492: 34, 92074: 70, 95555: 57, 97925: 5, 98126: 4, 98693: 68, 98966: 64, 99397: 23, 102458: 6, 102706: 58, 105193: 90, 107330: 40, 112289: 91, 112474: 50, 114256: 30, 122125: 41, 123459: 2, 127672: 10, 131020: 73, 131986: 41, 132112: 75, 133285: 85, 133813: 30, 133868: 80, 135509: 30, 138191: 85, 138457: 43, 140546: 0, 141218: 61, 144896: 22, 145847: 75, 145920: 86, 146681: 16, 147062: 46, 147312: 9, 148461: 32, 149789: 39, 150083: 28, 150596: 95, 153651: 11, 154954: 64, 156735: 16, 157418: 47, 157798: 84, 158922: 69, 159154: 31, 162653: 0, 162881: 70, 164764: 61, 166894: 59, 167973: 82, 169756: 87, 170359: 60, 170974: 14, 171132: 74, 173961: 13, 175303: 40, 177682: 76, 178955: 56, 179311: 55, 181195: 80, 182698: 12, 182702: 34, 184548: 43, 185643: 66, 185874: 43, 186582: 39, 188016: 52, 191234: 41, 191425: 83, 193308: 38, 194667: 7, 195084: 16, 198861: 64, 198917: 39, 200500: 63, 203327: 2, 206104: 8, 208417: 34, 210463: 23, 210613: 48, 210957: 16, 211002: 70, 214479: 67, 215557: 33, 216479: 15, 218689: 30, 219908: 25, 220226: 29, 222605: 95, 223923: 69, 226020: 56, 227822: 60, 232163: 45, 232526: 9, 232992: 49, 233750: 22, 234050: 82, 240520: 4, 241323: 16, 244949: 26, 246464: 51, 248307: 73, 249528: 28, 250186: 47, 253073: 35, 253567: 12, 254223: 67, 254339: 65, 257071: 50, 260965: 38, 263213: 67, 267681: 56, 269120: 27, 272202: 66, 274700: 4, 278661: 40, 278855: 28, 279453: 84, 279914: 41, 282197: 49, 285297: 78, 290586: 30, 293730: 88, 295843: 96, 299952: 97, 300088: 35, 301501: 3, 302911: 70, 303074: 75, 305127: 97, 308955: 22, 310225: 30, 313627: 17, 313707: 22, 316223: 75, 316968: 96, 317253: 39, 319278: 19, 320705: 55, 321172: 78, 323177: 22, 327983: 29, 328605: 5, 330557: 29, 331869: 20, 332723: 13, 334571: 4, 334820: 79, 334981: 55, 335056: 63, 335991: 33, 337742: 20, 341333: 53, 341887: 89, 341915: 27, 343078: 26, 343458: 92, 344361: 94, 345990: 18, 346134: 1, 346809: 94, 348618: 66, 348929: 45, 350786: 28, 351353: 3, 351472: 94, 351573: 38, 354058: 20, 354817: 2, 358183: 7, 359330: 55, 359703: 10, 361334: 78, 361834: 32, 362474: 24, 363788: 42, 364265: 48, 366030: 28, 366112: 42, 367886: 90, 372350: 6, 372595: 3, 374984: 92, 375299: 18, 381014: 69, 382178: 82, 384471: 48, 385468: 41, 385854: 58, 387621: 11, 388685: 61, 389471: 11, 389717: 4, 389818: 34, 392016: 77, 392737: 2, 393551: 4, 396826: 22, 400407: 9, 401466: 14, 402241: 78, 402301: 96, 403336: 17, 404088: 66, 404756: 82, 406138: 93, 410272: 42, 410812: 70, 413961: 64, 414543: 73, 415612: 36, 416940: 33, 418690: 84, 418788: 67, 419159: 23, 422605: 34, 429026: 2, 429191: 71, 432953: 1, 433082: 33, 435652: 31, 439214: 85, 440654: 41, 440672: 51, 440916: 35, 443084: 86, 445788: 32, 446002: 27, 446476: 78, 448001: 69, 449216: 3, 450390: 84, 450482: 26, 455600: 79, 457238: 7, 457454: 64, 458114: 92, 458290: 72, 463324: 49, 463599: 82, 464053: 2, 464101: 62, 464140: 82, 465682: 0, 465721: 4, 466769: 2, 467268: 31, 468004: 42, 468018: 78, 469432: 89, 470699: 91, 472833: 84, 473668: 2, 474127: 59, 475801: 98, 479999: 18, 482647: 30, 482936: 10, 488119: 95, 488286: 12, 488841: 82, 489518: 63, 493008: 1, 494296: 63, 496669: 61, 497907: 33, 500054: 33, 506271: 94, 507429: 22, 507685: 55, 508815: 16, 510481: 3, 510597: 51, 511067: 42, 511380: 79, 511630: 71, 512138: 30, 512932: 46, 516573: 19, 518398: 1, 519970: 23, 520067: 60, 526069: 95, 526462: 15, 528428: 42, 530236: 99, 533122: 14, 533716: 86, 536229: 50, 538642: 75, 543387: 11, 545618: 67, 547254: 27, 549269: 78, 550523: 50, 552648: 47, 553543: 64, 553799: 83, 555796: 5, 562616: 99, 563844: 15, 564494: 72, 570264: 96, 570444: 18, 570558: 75, 575175: 64, 576404: 64, 576566: 77, 577338: 17, 582891: 9, 585850: 26, 586049: 29, 588927: 4, 589299: 2, 590058: 60, 591401: 34, 596586: 72, 598075: 44, 598492: 78, 599712: 32, 600551: 9, 601221: 17, 602588: 24, 603878: 59, 605047: 74, 607018: 99, 611930: 57, 612305: 35, 613268: 2, 613910: 59, 613916: 86, 615062: 45, 615178: 31, 616078: 2, 617673: 64, 618567: 76, 618752: 75, 618808: 90, 618879: 79, 618958: 73, 620672: 63, 621198: 0, 622181: 22, 623897: 1, 624263: 49, 624905: 14, 626232: 11, 627065: 22, 629632: 79, 630041: 61, 630177: 54, 630731: 76, 631520: 71, 633599: 75, 637302: 72, 637962: 14, 638163: 6, 640553: 92, 641160: 32, 641339: 71, 643097: 67, 647764: 71, 647909: 51, 652821: 70, 653069: 90, 653646: 32, 654768: 36, 654803: 25, 655100: 98, 657934: 37, 660397: 44, 660556: 66, 661149: 88, 661998: 34, 662464: 50, 665998: 98, 668396: 83, 668815: 35, 669281: 56, 671361: 90, 672142: 11, 675393: 60, 676746: 89, 677911: 19, 679743: 92, 680500: 27, 682534: 53, 683111: 4, 683130: 46, 683773: 3, 684168: 20, 687875: 19, 689655: 97, 689932: 82, 690712: 3, 692409: 44, 692641: 21, 693523: 45, 695441: 80, 696912: 54, 700356: 18, 700393: 52, 707969: 89, 709747: 40, 714389: 19, 715767: 11, 717267: 77, 718293: 78, 719453: 18, 720103: 60, 721830: 75, 723964: 56, 725530: 74, 727159: 67, 727845: 85, 728035: 25, 730178: 64, 731027: 74, 734417: 37, 736097: 76, 736714: 49, 737586: 70, 737631: 33, 740282: 97, 742273: 20, 742813: 1, 743057: 62, 743208: 59, 744490: 30, 746255: 57, 747929: 83, 748043: 23, 749269: 89, 749802: 57, 750700: 96, 751196: 89, 751870: 18, 753282: 6, 753377: 59, 754568: 25, 755985: 40, 756371: 53, 757721: 93, 757930: 1, 758147: 40, 759268: 70, 761419: 41, 761622: 99, 762093: 68, 762300: 86, 762776: 46, 764724: 55, 769993: 63, 770541: 99, 771074: 6, 771532: 99, 772563: 56, 776564: 51, 780088: 67, 780146: 25, 780386: 26, 780779: 71, 781521: 1, 781713: 40, 781939: 26, 782495: 68, 783661: 82, 785751: 9, 786950: 15, 789963: 91, 794227: 68, 794280: 22, 796429: 65, 798287: 22, 799662: 19, 802968: 82, 803245: 43, 804296: 19, 804368: 50, 806702: 94, 807619: 60, 808326: 0, 810441: 65, 811557: 40, 812486: 83, 812985: 89, 814157: 56, 815018: 22, 815938: 19, 817555: 3, 821683: 89, 821858: 50, 822501: 24, 823489: 99, 825220: 74, 830742: 22, 832073: 42, 834352: 15, 836171: 81, 836898: 4, 837696: 53, 841127: 54, 841884: 80, 845892: 31, 847361: 65, 848418: 88, 848609: 93, 848964: 15, 850121: 86, 850967: 3, 851852: 89, 852763: 61, 855836: 64, 860628: 7, 861953: 99, 864237: 80, 867419: 90, 873654: 53, 873732: 11, 874185: 70, 875193: 55, 876216: 76, 876319: 77, 878259: 92, 878964: 3, 880238: 77, 880850: 42, 881557: 62, 882027: 24, 883147: 39, 884923: 25, 885634: 29, 886317: 41, 891012: 92, 892873: 57, 894854: 73, 895071: 33, 896893: 97, 897192: 97, 898378: 94, 898628: 56, 899431: 38, 899561: 98, 901710: 84, 901942: 19, 902196: 21, 902253: 32, 903668: 74, 903980: 81, 906099: 90, 908488: 42, 914516: 35, 916587: 68, 917276: 79, 922940: 78, 924988: 83, 928679: 25, 931279: 72, 932462: 37, 932735: 39, 932744: 52, 933022: 78, 937473: 9, 940263: 76, 940690: 68, 942270: 36, 944449: 33, 946313: 71, 951278: 99, 955877: 96, 957950: 32, 960114: 44, 960502: 8, 966016: 44, 967150: 47, 969165: 71, 976663: 85, 977799: 24, 978247: 96, 978858: 14, 981123: 32, 982374: 91, 983105: 11, 986752: 27, 986966: 83, 990861: 37, 992341: 64, 993139: 69, 994726: 20, 996206: 32},
		{3196: 2, 3500: 50, 4025: 96, 4828: 59, 5771: 34, 5792: 91, 6045: 81, 6177: 63, 6258: 45, 9181: 8, 9930: 40, 11243: 58, 11893: 44, 11978: 39, 12327: 90, 13921: 91, 16258: 30, 17977: 29, 18730: 90, 18945: 3, 18958: 43, 19226: 22, 19352: 70, 19620: 83, 19739: 70, 19805: 65, 19941: 50, 20661: 4, 20803: 99, 21587: 77, 21921: 79, 22531: 83, 24524: 18, 26092: 52, 26294: 57, 27249: 52, 30717: 53, 31299: 57, 31537: 39, 32420: 47, 33758: 93, 35503: 29, 35538: 97, 38009: 93, 38614: 50, 38735: 56, 40714: 43, 40778: 58, 40988: 69, 42952: 48, 44535: 12, 45822: 63, 47412: 7, 47770: 90, 48446: 89, 48527: 90, 50501: 88, 50721: 17, 50806: 47, 51862: 35, 53816: 78, 54867: 78, 55295: 40, 55723: 3, 56192: 52, 56787: 46, 56906: 71, 58923: 37, 59614: 42, 62140: 63, 62684: 64, 63166: 29, 63183: 71, 63481: 5, 66314: 3, 68609: 11, 68841: 46, 70064: 11, 71883: 42, 71983: 22, 72523: 13, 72563: 1, 72827: 16, 73705: 55, 74270: 50, 74872: 59, 76087: 6, 76830: 17, 77794: 27, 77912: 54, 79147: 13, 79289: 20, 81308: 58, 81825: 84, 84443: 90, 84878: 15, 86218: 80, 88702: 74, 88730: 58, 91482: 14, 91941: 66, 92131: 48, 93826: 29, 94193: 64, 94431: 9, 96305: 46, 96890: 24, 97119: 23, 98156: 38, 99411: 55, 99605: 23, 100346: 73, 102358: 80, 102920: 22, 103603: 57, 103791: 32, 105122: 71, 107978: 82, 108439: 42, 109098: 73, 109209: 26, 110009: 9, 111365: 64, 111384: 45, 111458: 5, 111733: 90, 112366: 71, 113047: 76, 113989: 62, 114080: 14, 115645: 31, 116332: 56, 117110: 62, 117685: 9, 118125: 73, 118279: 14, 118844: 24, 119180: 34, 119378: 26, 120958: 85, 121005: 83, 121302: 24, 121316: 37, 121496: 53, 121692: 31, 122346: 46, 122769: 79, 124824: 53, 125306: 96, 126524: 66, 128509: 73, 129340: 58, 129413: 12, 130012: 5, 130745: 2, 131634: 35, 131998: 33, 132567: 68, 133759: 30, 135160: 56, 136473: 69, 137048: 74, 138009: 4, 138680: 66, 141057: 83, 141425: 92, 142411: 31, 142494: 7, 142682: 95, 144111: 15, 145930: 35, 147496: 91, 147922: 12, 148113: 84, 148542: 85, 149184: 40, 150896: 95, 150902: 9, 151292: 60, 151437: 71, 151865: 49, 152935: 80, 153586: 78, 153901: 33, 154843: 34, 155238: 66, 157165: 2, 157316: 61, 157377: 89, 158144: 59, 160573: 42, 162031: 33, 162331: 88, 163695: 97, 164507: 35, 164777: 51, 167449: 65, 167555: 14, 168721: 19, 168945: 72, 170018: 26, 171007: 13, 171289: 36, 171701: 76, 172454: 13, 172627: 27, 173001: 38, 173760: 40, 173983: 71, 174599: 9, 174830: 70, 176149: 25, 176894: 31, 178044: 35, 178933: 23, 179913: 72, 181405: 63, 182498: 45, 182536: 55, 183228: 68, 184389: 41, 184436: 95, 185878: 89, 187157: 37, 188809: 84, 189593: 0, 190116: 81, 190287: 4, 192293: 7, 193201: 63, 193260: 74, 193373: 64, 193422: 7, 193458: 72, 195138: 61, 195467: 51, 196085: 6, 196563: 30, 198751: 72, 201078: 57, 202053: 63, 202860: 47, 203433: 71, 204779: 32, 205170: 21, 205261: 49, 206042: 46, 206525: 17, 206621: 15, 206879: 84, 207020: 11, 207427: 83, 209791: 16, 210618: 47, 210934: 21, 212836: 20, 214057: 82, 214444: 32, 216239: 26, 216240: 99, 216825: 69, 216853: 45, 219461: 79, 219656: 11, 220208: 95, 221189: 10, 222235: 8, 223163: 63, 224058: 1, 224534: 38, 225984: 73, 226233: 33, 227155: 99, 228053: 61, 230238: 40, 230333: 64, 230375: 28, 230876: 94, 232630: 7, 234447: 38, 234697: 47, 235382: 86, 235432: 66, 236949: 10, 237970: 28, 238603: 4, 238711: 81, 239012: 66, 239300: 32, 240996: 4, 241736: 61, 243012: 69, 243360: 64, 244699: 97, 245018: 65, 245810: 72, 246812: 6, 247396: 66, 247617: 74, 247896: 86, 248105: 45, 250180: 64, 250370: 59, 250765: 4, 253052: 4, 253380: 15, 253562: 55, 253883: 0, 253951: 25, 254552: 50, 255452: 75, 255615: 99, 255673: 12, 255933: 76, 257173: 54, 257481: 46, 258078: 19, 258742: 64, 259739: 27, 261310: 6, 263000: 13, 264738: 60, 265241: 17, 267555: 54, 269652: 29, 269655: 47, 270503: 73, 271300: 53, 272901: 39, 273819: 60, 275563: 19, 275773: 76, 276344: 5, 277072: 62, 279041: 67, 280097: 30, 282515: 45, 283061: 15, 284653: 36, 284655: 98, 284704: 23, 285310: 97, 286033: 55, 286144: 13, 287504: 21, 287596: 13, 287716: 31, 287811: 21, 288778: 74, 290032: 2, 290883: 33, 291782: 54, 291919: 12, 292683: 24, 294140: 72, 294376: 67, 295202: 33, 297014: 39, 297183: 6, 301737: 69, 304423: 88, 305101: 20, 305919: 99, 306200: 84, 306219: 64, 306276: 45, 307179: 58, 308060: 33, 308841: 2, 309943: 94, 311902: 69, 311925: 54, 312457: 14, 313263: 24, 314436: 70, 317446: 98, 319647: 62, 320307: 21, 320598: 91, 320622: 49, 320681: 53, 321205: 37, 321248: 84, 321966: 23, 324167: 7, 324938: 58, 325599: 82, 325722: 15, 326081: 10, 326736: 4, 327178: 88, 327474: 23, 327885: 21, 327907: 99, 328624: 19, 329017: 51, 330069: 53, 330341: 87, 331515: 41, 331816: 40, 335162: 61, 336248: 92, 336496: 19, 338903: 24, 339906: 25, 341356: 67, 344464: 50, 345652: 39, 346582: 82, 347054: 70, 347075: 94, 349530: 27, 350586: 59, 350616: 33, 351040: 50, 351336: 64, 352428: 72, 352849: 66, 353556: 47, 353838: 79, 353882: 40, 354065: 6, 357263: 0, 359913: 90, 360646: 1, 360827: 49, 362515: 13, 362555: 71, 362565: 3, 362881: 33, 363215: 64, 363261: 38, 363432: 35, 364498: 3, 366541: 59, 367317: 0, 367822: 39, 368036: 0, 368098: 62, 369820: 25, 371266: 95, 372082: 24, 373088: 63, 373758: 59, 373841: 30, 374497: 68, 377627: 77, 378651: 9, 379095: 58, 379588: 35, 379996: 49, 380544: 49, 382564: 1, 382576: 44, 383065: 19, 383611: 22, 385211: 56, 386980: 7, 387673: 12, 388115: 68, 388263: 94, 389705: 74, 391188: 29, 391251: 55, 391572: 92, 392966: 68, 393019: 33, 393344: 87, 395517: 42, 395600: 71, 395964: 44, 396327: 6, 397115: 98, 398451: 70, 399886: 3, 400325: 61, 401504: 80, 401770: 65, 403112: 88, 403328: 69, 403821: 39, 404026: 45, 405161: 84, 406779: 85, 406962: 12, 407086: 87, 407890: 31, 408630: 77, 409604: 24, 412421: 48, 412893: 55, 414003: 92, 415826: 30, 417779: 9, 418999: 54, 420775: 50, 423245: 22, 423289: 17, 423712: 88, 424255: 95, 425507: 30, 425755: 18, 426858: 66, 427414: 86, 429301: 73, 429710: 19, 431244: 2, 433624: 22, 434443: 94, 434631: 83, 435803: 91, 436602: 79, 438881: 45, 439310: 26, 439380: 92, 439998: 35, 440513: 18, 441054: 94, 442435: 68, 443982: 42, 445907: 65, 446710: 98, 448936: 64, 449327: 25, 449508: 80, 449660: 91, 450394: 71, 450493: 82, 450628: 48, 451928: 19, 452904: 58, 453008: 70, 453264: 77, 453420: 73, 454258: 46, 459182: 32, 459968: 32, 461678: 54, 462062: 43, 462888: 42, 463711: 16, 465441: 74, 465699: 53, 466367: 15, 467406: 30, 467705: 55, 468791: 88, 468940: 44, 469995: 79, 471724: 75, 471996: 24, 472242: 19, 472964: 60, 474428: 95, 475411: 18, 476356: 81, 479528: 94, 480106: 87, 480361: 32, 481829: 49, 483272: 44, 483685: 39, 484804: 16, 485065: 69, 485130: 66, 485639: 54, 487430: 23, 487458: 65, 487705: 49, 489199: 26, 489512: 37, 490168: 15, 492921: 60, 495206: 15, 495816: 96, 500206: 58, 501569: 16, 504214: 39, 504231: 27, 504393: 52, 506010: 74, 508354: 83, 508463: 83, 509519: 94, 510306: 28, 511755: 90, 512454: 1, 514843: 84, 515021: 17, 515109: 96, 516162: 4, 516579: 10, 516888: 10, 517608: 24, 518272: 70, 520054: 25, 520494: 17, 521619: 14, 521643: 42, 522093: 93, 524125: 50, 526285: 24, 527009: 28, 528813: 92, 532155: 55, 533659: 15, 533674: 43, 535041: 83, 535043: 27, 535513: 55, 537803: 78, 539091: 14, 541962: 24, 542850: 26, 543429: 65, 543814: 71, 546125: 48, 546494: 9, 548402: 50, 548766: 37, 549661: 13, 550084: 85, 550600: 38, 551393: 44, 553448: 63, 554302: 19, 554664: 96, 554843: 61, 556240: 58, 556636: 44, 556761: 0, 557982: 53, 558153: 35, 558260: 7, 559718: 72, 560220: 67, 561672: 66, 561814: 9, 562767: 48, 563270: 41, 563808: 67, 565781: 46, 565796: 14, 566628: 21, 566830: 69, 568859: 57, 570892: 22, 571022: 83, 571498: 12, 572686: 46, 573747: 59, 576618: 38, 583046: 58, 584131: 98, 584490: 96, 584708: 25, 585665: 18, 587000: 81, 588040: 34, 588692: 67, 588713: 5, 589348: 52, 589410: 50, 589430: 1, 589496: 94, 590494: 12, 590771: 42, 592833: 98, 595858: 83, 596849: 17, 597571: 3, 598489: 83, 598707: 0, 600061: 63, 600159: 62, 600791: 19, 601806: 76, 603048: 46, 603823: 25, 605614: 67, 605636: 6, 607538: 74, 608079: 34, 608201: 96, 608422: 95, 610549: 91, 612199: 57, 612371: 70, 613011: 53, 614329: 24, 614447: 59, 615298: 78, 615317: 80, 616274: 85, 617466: 49, 619328: 24, 620058: 32, 620521: 37, 621474: 3, 621864: 45, 622458: 87, 622944: 81, 623921: 83, 624389: 73, 624491: 15, 626085: 35, 627279: 20, 627363: 60, 627737: 30, 629631: 64, 631681: 86, 632522: 7, 632631: 7, 632847: 27, 633451: 49, 633835: 61, 634929: 47, 635783: 37, 636857: 3, 637819: 56, 637862: 91, 639413: 80, 639638: 45, 640404: 32, 642957: 12, 643324: 3, 643972: 2, 645006: 8, 645478: 7, 645600: 98, 645784: 42, 646298: 80, 646832: 50, 647389: 32, 648498: 92, 650908: 42, 652456: 92, 653928: 49, 654318: 25, 656792: 6, 656938: 33, 657310: 34, 658275: 39, 658726: 45, 659744: 81, 661068: 35, 662418: 22, 663941: 78, 664825: 11, 666431: 82, 666998: 68, 667040: 30, 669435: 13, 671004: 89, 671603: 49, 672106: 20, 672217: 75, 672691: 24, 672838: 54, 673250: 95, 673418: 12, 673424: 59, 674730: 39, 674975: 70, 675510: 50, 676039: 70, 676279: 69, 676561: 19, 680801: 42, 680843: 82, 681554: 42, 681787: 76, 682182: 66, 684033: 45, 686126: 24, 687206: 83, 687225: 85, 688317: 46, 688767: 41, 689581: 48, 689747: 87, 691643: 61, 692099: 96, 692458: 2, 693936: 65, 694449: 57, 695024: 46, 695093: 59, 695335: 61, 695870: 28, 696362: 77, 696772: 36, 696871: 21, 696929: 87, 697292: 84, 697297: 16, 698016: 46, 698040: 96, 699253: 21, 699532: 35, 700752: 75, 700795: 99, 701665: 32, 704898: 72, 705250: 19, 706104: 48, 708118: 89, 708353: 47, 709176: 21, 709965: 12, 709968: 61, 710059: 20, 712106: 81, 712785: 79, 713546: 53, 713713: 65, 714116: 16, 715053: 57, 716185: 31, 716228: 85, 717083: 43, 718554: 19, 718782: 37, 720606: 92, 721080: 25, 721361: 70, 722309: 8, 723516: 57, 723545: 69, 723915: 42, 725507: 77, 726232: 99, 726574: 73, 727071: 25, 729286: 20, 729945: 19, 729981: 15, 730846: 73, 731931: 16, 734999: 68, 736330: 71, 737658: 69, 737724: 61, 739121: 64, 739650: 88, 740044: 27, 740050: 4, 740358: 89, 742924: 89, 743269: 38, 743791: 5, 746270: 64, 752742: 38, 753307: 64, 756725: 10, 758739: 23, 759048: 13, 759135: 41, 760341: 31, 760425: 89, 761078: 19, 761587: 21, 764351: 47, 765118: 86, 766570: 80, 766910: 82, 770947: 88, 770978: 88, 771944: 30, 772947: 27, 773219: 58, 774367: 73, 775969: 26, 776650: 8, 777389: 12, 777822: 2, 781316: 98, 781363: 96, 781636: 24, 782471: 3, 783454: 97, 783746: 38, 784695: 97, 785597: 54, 785695: 79, 785699: 12, 786888: 94, 787495: 29, 788007: 55, 788650: 16, 789150: 37, 789170: 84, 789618: 90, 789789: 54, 790289: 35, 792670: 13, 792794: 44, 792969: 10, 793460: 3, 795356: 99, 795893: 83, 796339: 13, 796390: 34, 796636: 72, 796669: 19, 797023: 91, 798715: 17, 799182: 28, 799803: 8, 801657: 7, 802376: 51, 804444: 36, 806527: 44, 807094: 95, 807636: 72, 808867: 64, 809011: 18, 809242: 88, 809383: 74, 811766: 90, 812207: 69, 812857: 24, 813891: 88, 814296: 41, 815121: 38, 815499: 93, 815617: 69, 816483: 69, 817010: 52, 818298: 94, 819372: 39, 819471: 82, 821008: 46, 821171: 80, 821793: 86, 822149: 96, 822250: 78, 823081: 71, 823436: 29, 824209: 99, 824285: 60, 824724: 70, 826736: 57, 828963: 60, 829074: 76, 829186: 79, 830311: 77, 830451: 11, 832395: 82, 833056: 3, 834124: 10, 836137: 80, 836943: 35, 837218: 18, 837873: 64, 838013: 39, 839829: 11, 841031: 11, 844084: 11, 844612: 14, 846249: 2, 846930: 1, 847466: 18, 847514: 22, 849132: 14, 850671: 48, 852570: 43, 853294: 48, 855456: 6, 856214: 20, 856463: 56, 857088: 30, 858120: 12, 858601: 27, 858833: 99, 859539: 56, 859817: 99, 860210: 48, 860930: 16, 861238: 64, 863113: 46, 863592: 2, 863611: 78, 864283: 83, 864341: 17, 866885: 82, 868534: 83, 870346: 65, 871209: 71, 871595: 22, 874467: 2, 875148: 2, 875614: 55, 877808: 80, 878682: 69, 878960: 35, 879137: 6, 880326: 50, 880702: 35, 881202: 43, 881210: 98, 881298: 0, 882895: 64, 883347: 38, 884521: 9, 885228: 39, 886412: 56, 886878: 53, 887199: 41, 889319: 11, 889769: 94, 890299: 23, 890971: 1, 892290: 14, 892898: 87, 894507: 43, 894745: 49, 895501: 24, 895736: 73, 895790: 39, 898191: 65, 898460: 71, 898494: 20, 899431: 96, 899853: 83, 900099: 32, 901015: 41, 901026: 69, 901092: 25, 902076: 48, 902464: 23, 903491: 25, 904523: 49, 904543: 58, 905080: 8, 905968: 8, 906382: 85, 906747: 74, 907923: 47, 909160: 56, 909197: 59, 910263: 85, 910813: 21, 910868: 55, 911081: 16, 911548: 42, 912112: 86, 913144: 70, 913201: 84, 915325: 9, 915335: 49, 915381: 52, 915604: 93, 915795: 17, 916449: 87, 917062: 9, 917495: 93, 918019: 33, 919594: 80, 919609: 36, 920876: 78, 921103: 74, 921277: 74, 922831: 38, 923770: 25, 924988: 62, 925464: 28, 925719: 9, 926693: 84, 928358: 50, 928495: 56, 929972: 64, 929981: 52, 932722: 55, 932723: 30, 933161: 15, 933983: 77, 934528: 4, 935592: 13, 937158: 33, 937825: 80, 938419: 46, 938808: 72, 939317: 67, 939333: 97, 939707: 48, 939834: 68, 940873: 98, 940951: 49, 942556: 27, 942634: 36, 942720: 62, 946001: 69, 946345: 57, 947218: 0, 948572: 78, 949488: 53, 950393: 15, 955674: 20, 956473: 87, 956840: 80, 957170: 94, 958539: 66, 958883: 69, 959060: 40, 960599: 65, 961679: 5, 961855: 18, 962650: 60, 963611: 22, 964189: 51, 964277: 19, 964418: 95, 964511: 66, 965853: 73, 966572: 8, 966605: 88, 966772: 7, 968259: 80, 968671: 63, 970289: 31, 970567: 7, 976469: 18, 979131: 4, 979666: 27, 982385: 9, 982537: 67, 982895: 54, 983250: 2, 984342: 9, 986437: 26, 986696: 66, 987249: 88, 987269: 94, 988214: 40, 988944: 91, 991525: 84, 991703: 83, 992033: 37, 992174: 99, 992884: 42, 993041: 80, 993310: 46, 994541: 37, 995171: 74, 995195: 36, 995800: 2, 996174: 72, 998344: 3, 998629: 36},
		{4463: 25, 8728: 67, 8822: 73, 10441: 77, 11210: 2, 22646: 14, 23393: 16, 23778: 34, 24469: 17, 28910: 3, 30667: 96, 32551: 50, 35208: 26, 39387: 73, 41225: 60, 44021: 74, 44218: 45, 48490: 32, 58798: 29, 59728: 39, 65470: 63, 66986: 78, 67916: 36, 73869: 73, 75073: 25, 79982: 64, 81732: 24, 82403: 81, 86292: 72, 87434: 52, 92467: 41, 92814: 87, 97831: 19, 98684: 50, 105952: 75, 107075: 46, 107130: 13, 109199: 81, 109567: 43, 116612: 83, 120843: 6, 121324: 20, 121637: 24, 124351: 93, 126743: 96, 128936: 80, 132183: 51, 134728: 99, 135536: 29, 135727: 88, 143362: 11, 145495: 72, 145542: 86, 145705: 35, 147157: 28, 147670: 80, 149794: 15, 155306: 13, 158739: 75, 159068: 74, 159735: 66, 161048: 66, 162105: 0, 164279: 33, 169814: 68, 175252: 13, 176420: 76, 179391: 74, 179498: 47, 179957: 82, 180973: 33, 183789: 86, 184226: 55, 186466: 41, 192195: 81, 192554: 89, 193667: 87, 194549: 18, 197578: 50, 202018: 50, 205540: 53, 208702: 99, 209162: 90, 209316: 88, 209720: 89, 214134: 91, 215018: 76, 215559: 35, 218498: 59, 226456: 70, 228432: 8, 229084: 95, 230587: 5, 233548: 38, 234780: 73, 234975: 85, 238547: 8, 245717: 15, 247044: 60, 248354: 24, 248598: 86, 250325: 44, 256550: 43, 261021: 52, 263690: 17, 268759: 53, 271943: 72, 274162: 72, 274445: 16, 274673: 50, 274967: 65, 276222: 84, 277653: 13, 280027: 25, 280864: 18, 283548: 5, 285036: 41, 287183: 29, 291069: 3, 296323: 34, 299554: 24, 300354: 8, 300945: 0, 302540: 13, 302817: 25, 303439: 10, 307934: 96, 310248: 6, 311349: 26, 311458: 25, 311510: 51, 315106: 76, 315160: 62, 317165: 22, 318056: 62, 320662: 75, 321391: 41, 322892: 17, 324065: 22, 324556: 37, 324607: 9, 325272: 23, 329228: 69, 330817: 84, 332307: 32, 332772: 79, 335418: 39, 336495: 35, 336804: 16, 339802: 93, 341736: 44, 343345: 90, 344515: 35, 344674: 70, 346924: 35, 349077: 75, 349320: 51, 349877: 66, 355516: 89, 355868: 13, 356726: 72, 357880: 30, 357958: 4, 359718: 37, 359933: 2, 361255: 5, 362240: 42, 366534: 35, 367097: 44, 375973: 73, 376953: 81, 379730: 30, 379844: 37, 382443: 18, 383146: 13, 383276: 38, 386130: 91, 390958: 29, 393220: 77, 394491: 40, 396914: 59, 397902: 10, 400433: 0, 400704: 82, 401796: 19, 407680: 98, 411813: 93, 419448: 8, 420231: 85, 420466: 21, 421094: 57, 421655: 83, 424351: 11, 430787: 31, 438613: 3, 442082: 10, 444463: 95, 446469: 73, 448309: 50, 448478: 8, 448645: 5, 449666: 55, 451752: 67, 451820: 71, 452419: 4, 456267: 76, 456413: 33, 458714: 85, 459974: 40, 460735: 44, 463042: 13, 465411: 2, 466338: 21, 469392: 99, 469913: 6, 471895: 18, 479697: 48, 481943: 0, 485915: 92, 488510: 48, 489810: 35, 491861: 99, 492798: 25, 493987: 14, 497878: 71, 498846: 47, 500406: 31, 501034: 23, 502198: 9, 507185: 80, 512670: 93, 514487: 96, 515368: 90, 516234: 74, 517250: 80, 517665: 81, 521326: 29, 522472: 83, 525473: 90, 525503: 23, 526587: 12, 528304: 37, 528352: 0, 530535: 74, 530864: 16, 531848: 43, 531871: 6, 532801: 97, 534974: 23, 538160: 37, 541552: 5, 542436: 22, 542712: 38, 548966: 76, 555057: 27, 570013: 77, 573292: 98, 574608: 26, 575009: 11, 578180: 97, 579345: 43, 581704: 44, 585168: 84, 586215: 91, 587862: 43, 591016: 15, 599232: 95, 601506: 73, 606349: 76, 610547: 61, 610622: 47, 610843: 19, 611257: 52, 611510: 71, 613505: 74, 616283: 33, 617078: 28, 623013: 95, 628573: 83, 629352: 4, 631988: 57, 632415: 84, 633522: 1, 633686: 33, 635837: 7, 636947: 91, 639590: 86, 639854: 43, 641703: 6, 642557: 52, 643398: 84, 647355: 42, 648707: 55, 653410: 46, 655850: 53, 656873: 77, 658934: 55, 660628: 38, 666912: 80, 666917: 15, 667559: 24, 667837: 51, 668041: 17, 672392: 93, 676340: 63, 677238: 89, 679430: 63, 679981: 49, 683124: 79, 690441: 52, 692611: 36, 695941: 95, 697618: 3, 701975: 80, 702574: 38, 704298: 32, 705294: 93, 707925: 37, 714736: 51, 717038: 30, 717209: 42, 724020: 36, 724545: 91, 726644: 79, 728321: 75, 730517: 61, 734790: 84, 735514: 34, 738238: 32, 738259: 75, 739419: 41, 744659: 36, 744725: 59, 745915: 61, 748392: 20, 748876: 19, 749037: 69, 749583: 68, 752833: 12, 756355: 98, 767693: 95, 769892: 18, 771909: 61, 772781: 25, 774624: 89, 785322: 18, 785984: 9, 790840: 68, 793505: 9, 793647: 37, 794220: 38, 795799: 30, 800655: 8, 802693: 12, 805801: 39, 809799: 83, 812749: 72, 823231: 98, 823911: 90, 828948: 67, 830379: 92, 831195: 80, 833136: 73, 833659: 87, 836201: 47, 838950: 79, 840764: 55, 841986: 0, 843606: 36, 844719: 41, 846505: 22, 858598: 31, 859293: 30, 863795: 69, 864847: 78, 865249: 32, 866680: 35, 867614: 73, 868004: 93, 869344: 27, 869623: 20, 873628: 49, 876331: 20, 879452: 43, 879462: 76, 882430: 63, 887383: 74, 887511: 52, 892797: 77, 893194: 4, 894587: 39, 897336: 35, 905897: 33, 905979: 26, 906768: 69, 909776: 82, 913035: 90, 914162: 27, 915750: 58, 917363: 68, 920725: 57, 921916: 52, 922191: 15, 927830: 17, 929848: 18, 930481: 1, 931565: 1, 931822: 29, 932291: 62, 934950: 97, 935858: 54, 936010: 74, 939671: 7, 939786: 19, 940419: 39, 944655: 15, 945416: 66, 948601: 19, 954225: 22, 955188: 75, 960172: 62, 963793: 47, 965822: 37, 969991: 59, 970465: 45, 974633: 74, 979331: 52, 980567: 27, 981201: 83, 985460: 40, 986763: 39, 986793: 75, 988881: 71, 989450: 96, 990099: 69, 990538: 29, 990901: 75, 992558: 53, 992728: 96, 994003: 63, 995089: 56, 995619: 8, 995763: 27, 996012: 85, 997513: 72},
		{1304: 85, 1346: 43, 1380: 93, 2425: 58, 2843: 7, 3202: 44, 4333: 95, 4699: 88, 5249: 46, 6258: 98, 6810: 63, 8799: 17, 10289: 62, 10393: 37, 11161: 20, 11392: 50, 15249: 41, 15667: 97, 16427: 80, 19823: 63, 20465: 20, 21826: 82, 22926: 63, 23021: 86, 24004: 31, 24614: 81, 26069: 73, 27654: 54, 30572: 96, 31192: 85, 32302: 57, 33873: 47, 35017: 98, 36148: 68, 37424: 17, 38349: 91, 38848: 40, 41433: 10, 42488: 21, 43925: 53, 45692: 92, 46465: 13, 47633: 36, 47925: 71, 48596: 43, 49838: 90, 50829: 67, 51853: 84, 57151: 94, 57332: 28, 57939: 9, 58871: 27, 60843: 11, 62540: 53, 63990: 4, 64500: 39, 65267: 14, 66426: 53, 68435: 43, 69572: 19, 69888: 64, 70579: 73, 70855: 61, 71446: 28, 71633: 22, 73019: 26, 78382: 57, 79052: 96, 81045: 91, 83994: 49, 84223: 49, 84267: 87, 85135: 35, 85918: 52, 88172: 12, 88560: 68, 93300: 38, 93801: 69, 97734: 91, 99439: 52, 99642: 28, 101397: 96, 111307: 77, 114223: 58, 115653: 74, 117997: 31, 118792: 96, 119010: 65, 120633: 42, 121213: 48, 124709: 46, 125439: 30, 126022: 29, 126806: 1, 127203: 84, 129813: 28, 130438: 95, 132123: 1, 137145: 4, 137959: 53, 138714: 55, 139464: 39, 141103: 1, 141754: 92, 142228: 58, 144393: 17, 145419: 22, 145999: 57, 148659: 32, 148762: 96, 151053: 40, 151896: 14, 152618: 22, 154508: 68, 154631: 14, 154657: 73, 155950: 84, 158963: 1, 160967: 74, 161061: 86, 165002: 83, 165496: 52, 166053: 84, 167768: 44, 168270: 58, 168872: 33, 172594: 72, 172936: 26, 173354: 46, 176629: 28, 177546: 88, 179933: 52, 179982: 52, 180236: 0, 180780: 20, 183160: 75, 184834: 95, 185091: 36, 190177: 40, 190462: 88, 190602: 45, 193828: 90, 194503: 19, 194581: 39, 200236: 45, 200523: 80, 203860: 39, 204283: 77, 206646: 25, 208862: 73, 211168: 33, 211249: 3, 212099: 71, 214477: 9, 214638: 87, 214953: 90, 217859: 5, 218155: 11, 218241: 28, 218570: 65, 218723: 12, 219328: 38, 219482: 23, 220355: 25, 222507: 36, 223103: 61, 226319: 1, 228637: 46, 231724: 98, 232038: 12, 232607: 66, 232638: 20, 235178: 21, 236315: 73, 237124: 56, 237856: 97, 240636: 29, 242349: 87, 242774: 41, 243746: 54, 246725: 43, 250211: 11, 250370: 7, 251999: 22, 252511: 50, 252933: 53, 253608: 99, 254126: 69, 254490: 34, 258255: 75, 260314: 56, 261248: 98, 261592: 47, 261682: 61, 262348: 47, 263594: 40, 265105: 61, 265624: 3, 265875: 78, 266786: 22, 267043: 64, 267427: 31, 271281: 34, 272217: 43, 272529: 91, 272699: 96, 278766: 35, 279667: 22, 279779: 47, 280385: 21, 280940: 6, 283446: 45, 283714: 79, 283839: 30, 287036: 95, 288250: 99, 288349: 91, 288702: 95, 291135: 55, 293883: 36, 293918: 34, 294182: 39, 294722: 24, 298055: 86, 298151: 56, 299127: 88, 300579: 41, 304935: 9, 305474: 59, 308105: 67, 310650: 81, 313250: 22, 314294: 44, 316142: 66, 317733: 91, 319896: 10, 320232: 82, 320682: 25, 321154: 34, 321163: 40, 323380: 77, 326535: 3, 327867: 0, 330621: 83, 332381: 42, 334983: 47, 336543: 13, 337058: 25, 337704: 43, 339663: 30, 339738: 84, 340021: 49, 340292: 97, 341100: 96, 342171: 40, 345359: 71, 348287: 62, 349376: 85, 349916: 35, 351424: 59, 353308: 85, 355850: 11, 356003: 81, 357265: 61, 358914: 19, 359191: 90, 361119: 0, 362767: 44, 363146: 7, 364314: 98, 366799: 8, 367738: 84, 369518: 51, 370240: 48, 373120: 84, 373216: 76, 373628: 71, 374827: 60, 376061: 47, 377119: 83, 377498: 47, 380303: 84, 380328: 5, 381182: 21, 387060: 0, 387877: 68, 389645: 80, 397672: 81, 397876: 20, 400002: 65, 400441: 51, 400505: 3, 401574: 15, 402742: 40, 403189: 71, 403396: 86, 403638: 15, 406215: 18, 406867: 47, 408849: 14, 409941: 59, 410184: 22, 410288: 41, 410893: 58, 410957: 34, 410976: 56, 412159: 78, 413596: 23, 413685: 71, 413785: 30, 416249: 91, 418336: 3, 418443: 79, 420439: 23, 421542: 39, 424914: 50, 425137: 79, 425745: 32, 426693: 72, 428276: 81, 428577: 59, 429064: 26, 429585: 28, 430235: 54, 431555: 11, 432019: 57, 432370: 56, 437095: 12, 439638: 17, 441500: 69, 444730: 88, 445083: 9, 450432: 56, 451166: 33, 451336: 31, 455436: 27, 456078: 66, 456690: 55, 457042: 65, 457066: 94, 458004: 88, 460787: 24, 462569: 46, 463999: 89, 464509: 75, 465613: 93, 466809: 36, 467848: 60, 468796: 47, 471150: 94, 471818: 81, 472923: 32, 474883: 43, 475696: 66, 476678: 69, 477806: 61, 478808: 98, 478897: 11, 480046: 11, 482081: 21, 482241: 99, 484083: 87, 484323: 50, 488483: 61, 493611: 84, 498792: 50, 499061: 70, 500093: 76, 501143: 54, 501403: 28, 501843: 29, 502549: 63, 503633: 72, 505241: 2, 505315: 35, 507440: 92, 508240: 43, 508975: 8, 509576: 91, 509711: 6, 510440: 92, 514759: 61, 515102: 76, 515673: 26, 515863: 13, 516853: 17, 517287: 85, 517468: 3, 519447: 74, 520048: 5, 521186: 51, 524555: 7, 525257: 13, 526463: 56, 527461: 49, 529642: 12, 530428: 97, 530950: 4, 536011: 71, 536443: 85, 537848: 95, 540989: 92, 544113: 21, 545217: 24, 545382: 62, 546716: 55, 546975: 29, 548471: 4, 548499: 94, 548553: 11, 549548: 74, 549685: 58, 550406: 8, 550619: 48, 550901: 66, 552497: 85, 554741: 36, 556581: 81, 559232: 65, 560673: 97, 567875: 94, 572495: 56, 572534: 87, 572944: 96, 573663: 8, 575246: 22, 575418: 35, 578043: 82, 580863: 75, 582135: 93, 582300: 29, 583753: 84, 588106: 28, 592246: 82, 596803: 24, 599280: 27, 600389: 87, 600461: 63, 604623: 47, 605295: 24, 605918: 26, 606650: 76, 608813: 40, 608824: 39, 611204: 44, 611459: 16, 612693: 93, 612819: 60, 614604: 10, 617213: 82, 617792: 4, 618135: 88, 623828: 74, 624700: 79, 626184: 92, 627484: 42, 628313: 37, 631080: 60, 632495: 61, 633402: 84, 634115: 88, 634666: 82, 637919: 49, 638316: 38, 638341: 49, 638595: 33, 638951: 56, 639226: 35, 642214: 30, 642942: 38, 644401: 27, 646165: 68, 649740: 61, 654153: 13, 654411: 17, 654510: 79, 654526: 17, 654865: 92, 656402: 5, 657430: 59, 658622: 93, 659908: 89, 666696: 68, 667481: 14, 667807: 69, 668617: 41, 670743: 39, 671897: 81, 673993: 38, 674062: 86, 677127: 83, 677155: 62, 677277: 98, 678332: 53, 679179: 70, 679853: 24, 680795: 10, 682279: 57, 683136: 57, 684407: 54, 687234: 23, 688990: 20, 689445: 94, 689727: 36, 690302: 35, 690375: 47, 691628: 53, 691636: 84, 692284: 86, 694604: 96, 695403: 0, 696193: 17, 697297: 53, 698628: 72, 699900: 12, 700310: 50, 700501: 13, 702035: 86, 703781: 64, 705544: 85, 705546: 97, 706974: 79, 710250: 86, 712243: 92, 712745: 61, 713879: 25, 714075: 70, 716519: 96, 716828: 50, 721119: 79, 722589: 19, 722770: 82, 722981: 77, 723510: 21, 725749: 19, 726280: 13, 728849: 26, 730201: 68, 731196: 29, 732257: 23, 738171: 74, 738177: 10, 741516: 76, 743013: 69, 744560: 79, 745267: 58, 747514: 37, 748277: 40, 750927: 58, 751007: 23, 752947: 83, 753211: 79, 755256: 60, 758246: 39, 759422: 22, 759475: 62, 760836: 34, 761842: 18, 762368: 94, 764538: 39, 764974: 91, 765611: 29, 768012: 97, 770083: 13, 770124: 6, 772349: 75, 774244: 91, 776684: 65, 778283: 25, 779022: 75, 779252: 88, 779279: 61, 782035: 77, 782707: 30, 782762: 79, 785994: 15, 786980: 74, 788477: 40, 788736: 26, 790020: 48, 790048: 38, 790594: 66, 792433: 19, 792911: 91, 794773: 28, 797219: 85, 799275: 8, 799319: 65, 799693: 70, 802270: 67, 805943: 89, 806099: 52, 806144: 50, 808246: 92, 808614: 58, 809540: 77, 809690: 83, 811757: 50, 811981: 0, 812279: 72, 812764: 24, 813538: 97, 813865: 62, 815445: 55, 817212: 86, 818173: 59, 818860: 22, 821381: 23, 821416: 6, 824121: 52, 824307: 54, 826237: 98, 826308: 42, 826765: 19, 829279: 65, 830037: 14, 831031: 29, 832237: 98, 832503: 80, 832795: 94, 833960: 46, 835217: 55, 836088: 40, 837858: 55, 838563: 56, 838831: 37, 838945: 86, 839036: 26, 840246: 43, 840384: 77, 841312: 94, 843417: 69, 843736: 22, 845179: 1, 847413: 19, 848072: 46, 848220: 83, 852602: 13, 856787: 33, 857907: 16, 859767: 3, 859796: 2, 862852: 56, 863260: 10, 864564: 7, 865665: 50, 867534: 92, 867597: 34, 868399: 72, 869613: 25, 871200: 99, 872249: 72, 875212: 23, 875788: 22, 876236: 84, 880647: 56, 881196: 90, 883051: 8, 885002: 60, 885958: 55, 888321: 8, 888415: 61, 889266: 84, 891745: 42, 892127: 72, 893983: 3, 894836: 23, 895423: 33, 896914: 44, 897229: 35, 898953: 50, 899371: 21, 899593: 56, 899709: 19, 901276: 10, 901930: 8, 902296: 23, 902706: 79, 903722: 58, 904112: 39, 904255: 27, 904278: 63, 906501: 74, 907620: 99, 913586: 0, 914671: 19, 915171: 99, 916155: 14, 917039: 51, 917353: 74, 919108: 97, 919506: 5, 919711: 27, 921264: 8, 921713: 98, 922696: 15, 922788: 32, 924248: 46, 924574: 71, 929686: 71, 930687: 84, 931983: 29, 935775: 65, 936321: 62, 938911: 91, 939426: 29, 944280: 22, 947023: 74, 947793: 27, 948431: 73, 949049: 2, 954278: 78, 957523: 53, 958360: 72, 959478: 48, 960766: 57, 961182: 83, 962679: 54, 963285: 48, 964179: 28, 970544: 92, 975061: 94, 981380: 70, 981765: 97, 982518: 42, 984764: 85, 984930: 29, 987013: 16, 987757: 22, 988670: 3, 989758: 53, 991787: 16, 992619: 86, 992993: 24, 996092: 9, 996555: 53, 998323: 96, 998461: 49, 999835: 1, 999936: 37},
	}

	deletions := []map[uint64]struct{}{
		{5158: {}, 22282: {}, 26625: {}, 30464: {}, 32276: {}, 39406: {}, 45829: {}, 49653: {}, 138857: {}, 139624: {}, 147712: {}, 169597: {}, 174727: {}, 185264: {}, 194404: {}, 206469: {}, 285942: {}, 290568: {}, 337242: {}, 348698: {}, 378784: {}, 386443: {}, 405301: {}, 410675: {}, 419614: {}, 425138: {}, 434550: {}, 485705: {}, 497978: {}, 549654: {}, 557748: {}, 564079: {}, 586240: {}, 589415: {}, 601790: {}, 608517: {}, 614523: {}, 620589: {}, 634586: {}, 645052: {}, 656207: {}, 664575: {}, 670473: {}, 674601: {}, 683011: {}, 686123: {}, 688524: {}, 689926: {}, 691224: {}, 691424: {}, 716584: {}, 737597: {}, 756233: {}, 817565: {}, 834273: {}, 835674: {}, 843958: {}, 855323: {}, 859622: {}, 875131: {}, 878883: {}, 905881: {}, 913565: {}, 915876: {}, 924544: {}, 938320: {}, 949593: {}, 957490: {}, 958340: {}, 960272: {}, 961744: {}, 981075: {}, 996546: {}},
		{1670: {}, 15244: {}, 19759: {}, 32442: {}, 32679: {}, 32986: {}, 46757: {}, 48853: {}, 53264: {}, 55916: {}, 58096: {}, 60847: {}, 73138: {}, 83351: {}, 89429: {}, 90182: {}, 112112: {}, 113801: {}, 122272: {}, 126406: {}, 134535: {}, 135558: {}, 138510: {}, 140540: {}, 141735: {}, 147340: {}, 153321: {}, 157112: {}, 160937: {}, 161243: {}, 164574: {}, 196972: {}, 198736: {}, 205425: {}, 218105: {}, 218263: {}, 220110: {}, 221218: {}, 226221: {}, 243603: {}, 246545: {}, 251669: {}, 258510: {}, 265861: {}, 268183: {}, 273011: {}, 282006: {}, 283042: {}, 286410: {}, 291378: {}, 291794: {}, 294225: {}, 298630: {}, 298923: {}, 310448: {}, 310808: {}, 317001: {}, 324642: {}, 326424: {}, 328942: {}, 333803: {}, 338863: {}, 344224: {}, 344283: {}, 350586: {}, 351893: {}, 368390: {}, 368863: {}, 373620: {}, 376862: {}, 380629: {}, 384258: {}, 387176: {}, 387309: {}, 390920: {}, 402819: {}, 403165: {}, 405780: {}, 407894: {}, 409206: {}, 409244: {}, 411759: {}, 413125: {}, 415512: {}, 417914: {}, 423447: {}, 428217: {}, 432413: {}, 434673: {}, 435388: {}, 439939: {}, 443231: {}, 443779: {}, 444281: {}, 447088: {}, 448414: {}, 452958: {}, 456867: {}, 467354: {}, 474480: {}, 493342: {}, 504752: {}, 507387: {}, 512063: {}, 513624: {}, 518528: {}, 521166: {}, 522076: {}, 525752: {}, 530582: {}, 536669: {}, 546914: {}, 549647: {}, 550482: {}, 558867: {}, 562064: {}, 562359: {}, 576889: {}, 584299: {}, 599101: {}, 615164: {}, 616858: {}, 623644: {}, 634673: {}, 639800: {}, 642274: {}, 646201: {}, 646649: {}, 656731: {}, 657747: {}, 670842: {}, 671493: {}, 674596: {}, 681633: {}, 681917: {}, 689703: {}, 698103: {}, 702740: {}, 711014: {}, 730100: {}, 731111: {}, 732358: {}, 736800: {}, 744204: {}, 745690: {}, 751993: {}, 756793: {}, 758521: {}, 768092: {}, 771774: {}, 774079: {}, 778012: {}, 781028: {}, 781855: {}, 783301: {}, 789312: {}, 790883: {}, 794225: {}, 795070: {}, 795602: {}, 796438: {}, 798460: {}, 803608: {}, 806721: {}, 809539: {}, 813844: {}, 817088: {}, 826185: {}, 830373: {}, 833175: {}, 844964: {}, 848515: {}, 854143: {}, 854210: {}, 856841: {}, 861722: {}, 870211: {}, 872311: {}, 882582: {}, 884939: {}, 888756: {}, 899099: {}, 905083: {}, 912337: {}, 914438: {}, 914645: {}, 918657: {}, 920789: {}, 923228: {}, 925633: {}, 925768: {}, 932217: {}, 945536: {}, 951565: {}, 953265: {}, 953344: {}, 959174: {}, 969560: {}, 971625: {}, 985901: {}, 988307: {}, 998830: {}},
		{230: {}, 9171: {}, 10017: {}, 11774: {}, 15921: {}, 17111: {}, 32945: {}, 37053: {}, 42004: {}, 52276: {}, 72718: {}, 78267: {}, 94048: {}, 100823: {}, 105501: {}, 117557: {}, 122303: {}, 123329: {}, 134817: {}, 138300: {}, 143103: {}, 144135: {}, 148593: {}, 157049: {}, 161434: {}, 161575: {}, 171081: {}, 179850: {}, 180682: {}, 181650: {}, 185019: {}, 192986: {}, 197098: {}, 202299: {}, 203830: {}, 208265: {}, 216550: {}, 221033: {}, 229841: {}, 230327: {}, 243084: {}, 244852: {}, 262904: {}, 269371: {}, 274193: {}, 281878: {}, 282131: {}, 282827: {}, 283436: {}, 288729: {}, 301048: {}, 321164: {}, 321878: {}, 324579: {}, 330616: {}, 331597: {}, 336592: {}, 339177: {}, 354393: {}, 354432: {}, 354610: {}, 358031: {}, 358892: {}, 363424: {}, 367029: {}, 367697: {}, 369444: {}, 371654: {}, 375169: {}, 380061: {}, 391525: {}, 394589: {}, 396845: {}, 398553: {}, 399230: {}, 402384: {}, 422035: {}, 425419: {}, 427800: {}, 429143: {}, 429150: {}, 430041: {}, 438716: {}, 442357: {}, 442385: {}, 446415: {}, 446822: {}, 456980: {}, 458424: {}, 461708: {}, 463685: {}, 465661: {}, 471348: {}, 473669: {}, 480251: {}, 492242: {}, 492279: {}, 497307: {}, 500225: {}, 501878: {}, 503117: {}, 503269: {}, 517587: {}, 523854: {}, 528094: {}, 547324: {}, 547557: {}, 549743: {}, 569258: {}, 594258: {}, 599347: {}, 602441: {}, 604071: {}, 604649: {}, 612989: {}, 613926: {}, 614762: {}, 615426: {}, 615450: {}, 623338: {}, 638153: {}, 638348: {}, 646079: {}, 650541: {}, 652613: {}, 655790: {}, 656460: {}, 658961: {}, 660481: {}, 666636: {}, 667998: {}, 668270: {}, 675631: {}, 689386: {}, 694256: {}, 698834: {}, 700520: {}, 704395: {}, 722188: {}, 723284: {}, 728834: {}, 730975: {}, 732586: {}, 740905: {}, 763342: {}, 771537: {}, 776940: {}, 781934: {}, 783064: {}, 785573: {}, 785968: {}, 803106: {}, 805313: {}, 807320: {}, 811875: {}, 829791: {}, 831010: {}, 833714: {}, 836578: {}, 840712: {}, 843070: {}, 862883: {}, 868463: {}, 870996: {}, 873847: {}, 876788: {}, 883263: {}, 884478: {}, 903996: {}, 907082: {}, 916467: {}, 928289: {}, 931349: {}, 944705: {}, 972898: {}, 973354: {}, 976583: {}, 980025: {}, 980040: {}, 986292: {}, 988765: {}, 998142: {}},
		{1310: {}, 4830: {}, 14416: {}, 23668: {}, 25886: {}, 36924: {}, 39926: {}, 56649: {}, 60048: {}, 65335: {}, 66706: {}, 72564: {}, 86292: {}, 98389: {}, 102878: {}, 117971: {}, 130251: {}, 134714: {}, 140994: {}, 168509: {}, 170831: {}, 171220: {}, 174936: {}, 182954: {}, 193947: {}, 198041: {}, 199200: {}, 210965: {}, 212914: {}, 224559: {}, 255060: {}, 271038: {}, 271866: {}, 275017: {}, 276357: {}, 281966: {}, 293453: {}, 301337: {}, 302076: {}, 305864: {}, 309570: {}, 330769: {}, 338014: {}, 349554: {}, 353582: {}, 361045: {}, 377268: {}, 393124: {}, 399013: {}, 401794: {}, 411217: {}, 422986: {}, 428447: {}, 437937: {}, 446697: {}, 447665: {}, 452386: {}, 453667: {}, 456186: {}, 461207: {}, 465564: {}, 476646: {}, 481083: {}, 482062: {}, 484883: {}, 503210: {}, 516330: {}, 518184: {}, 533574: {}, 541363: {}, 543742: {}, 544879: {}, 553121: {}, 554225: {}, 568910: {}, 570280: {}, 581300: {}, 581322: {}, 589655: {}, 599522: {}, 612893: {}, 615816: {}, 622272: {}, 634599: {}, 644164: {}, 644835: {}, 651861: {}, 652952: {}, 655278: {}, 671854: {}, 682906: {}, 685084: {}, 685862: {}, 705319: {}, 725610: {}, 729164: {}, 732593: {}, 739018: {}, 739929: {}, 751909: {}, 752658: {}, 755396: {}, 762356: {}, 764668: {}, 781960: {}, 792103: {}, 802875: {}, 808640: {}, 809937: {}, 827348: {}, 835279: {}, 839480: {}, 841321: {}, 860038: {}, 871288: {}, 891454: {}, 893765: {}, 894473: {}, 897727: {}, 901657: {}, 902069: {}, 906039: {}, 908168: {}, 911085: {}, 912309: {}, 913217: {}, 913647: {}, 915451: {}, 917587: {}, 926545: {}, 930857: {}, 931364: {}, 936420: {}, 939282: {}, 941380: {}, 950454: {}, 952362: {}, 954392: {}, 982918: {}, 985548: {}, 985651: {}, 987805: {}},
		{1441: {}, 3540: {}, 5357: {}, 8130: {}, 12519: {}, 14897: {}, 24822: {}, 25324: {}, 25681: {}, 26461: {}, 26721: {}, 28762: {}, 35617: {}, 37358: {}, 38551: {}, 42260: {}, 42645: {}, 43249: {}, 44226: {}, 48453: {}, 49347: {}, 53407: {}, 53463: {}, 62786: {}, 63290: {}, 64018: {}, 73518: {}, 75663: {}, 78876: {}, 82961: {}, 100810: {}, 108069: {}, 109159: {}, 118093: {}, 120407: {}, 127018: {}, 134112: {}, 136883: {}, 149287: {}, 149868: {}, 150394: {}, 152904: {}, 155702: {}, 161857: {}, 164435: {}, 166967: {}, 172536: {}, 175949: {}, 177306: {}, 184256: {}, 186810: {}, 187418: {}, 197877: {}, 201508: {}, 204326: {}, 206409: {}, 207903: {}, 212641: {}, 216701: {}, 229921: {}, 231483: {}, 231509: {}, 238334: {}, 243465: {}, 256977: {}, 257507: {}, 260531: {}, 263216: {}, 263944: {}, 274939: {}, 276775: {}, 278555: {}, 285829: {}, 290362: {}, 290474: {}, 291863: {}, 297294: {}, 298447: {}, 304352: {}, 305169: {}, 307837: {}, 313750: {}, 314031: {}, 319711: {}, 320989: {}, 322715: {}, 331780: {}, 335089: {}, 335679: {}, 348845: {}, 353341: {}, 354250: {}, 354983: {}, 358531: {}, 363474: {}, 364014: {}, 365599: {}, 366482: {}, 394768: {}, 401926: {}, 405562: {}, 406339: {}, 415014: {}, 416268: {}, 420249: {}, 429397: {}, 435623: {}, 437560: {}, 439822: {}, 440111: {}, 440234: {}, 441604: {}, 441794: {}, 442085: {}, 445008: {}, 451865: {}, 454879: {}, 455212: {}, 456680: {}, 460510: {}, 462417: {}, 469137: {}, 482030: {}, 482495: {}, 483131: {}, 493497: {}, 497625: {}, 499713: {}, 501621: {}, 505510: {}, 506075: {}, 507459: {}, 509359: {}, 510207: {}, 511126: {}, 513013: {}, 516132: {}, 516386: {}, 517093: {}, 518736: {}, 526286: {}, 532489: {}, 535153: {}, 536650: {}, 543543: {}, 544087: {}, 544658: {}, 545236: {}, 549066: {}, 549662: {}, 549698: {}, 549886: {}, 550597: {}, 552387: {}, 552494: {}, 555567: {}, 557116: {}, 559384: {}, 566830: {}, 571293: {}, 572630: {}, 577523: {}, 581414: {}, 584025: {}, 585884: {}, 587521: {}, 590423: {}, 590651: {}, 603067: {}, 605464: {}, 607641: {}, 614885: {}, 620596: {}, 620977: {}, 622243: {}, 629263: {}, 629333: {}, 629789: {}, 632609: {}, 634512: {}, 636206: {}, 642264: {}, 643384: {}, 645316: {}, 655582: {}, 658432: {}, 660365: {}, 676628: {}, 678642: {}, 686819: {}, 690768: {}, 694285: {}, 696507: {}, 702038: {}, 702242: {}, 709243: {}, 710754: {}, 711690: {}, 712914: {}, 717191: {}, 717695: {}, 718223: {}, 718915: {}, 719179: {}, 721726: {}, 722942: {}, 735566: {}, 738393: {}, 744404: {}, 744696: {}, 749720: {}, 755387: {}, 764240: {}, 765136: {}, 768853: {}, 772481: {}, 772710: {}, 783860: {}, 786447: {}, 787460: {}, 792089: {}, 796711: {}, 797142: {}, 802504: {}, 805807: {}, 807481: {}, 807706: {}, 808924: {}, 810981: {}, 811520: {}, 813840: {}, 815607: {}, 817332: {}, 817363: {}, 832584: {}, 839838: {}, 852521: {}, 857539: {}, 864707: {}, 864748: {}, 870258: {}, 885850: {}, 889916: {}, 891849: {}, 893310: {}, 896852: {}, 898862: {}, 900050: {}, 900732: {}, 904004: {}, 908024: {}, 912130: {}, 912705: {}, 916279: {}, 917032: {}, 923460: {}, 925252: {}, 928542: {}, 928585: {}, 937856: {}, 938757: {}, 941107: {}, 951266: {}, 952365: {}, 953912: {}, 959249: {}, 966553: {}, 968960: {}, 970018: {}, 977484: {}, 980796: {}, 991058: {}, 992151: {}, 997506: {}},
		{35277: {}, 44224: {}, 65905: {}, 71222: {}, 84923: {}, 101362: {}, 106323: {}, 107019: {}, 154290: {}, 167904: {}, 172701: {}, 179129: {}, 187133: {}, 202967: {}, 213598: {}, 228585: {}, 257131: {}, 259691: {}, 266969: {}, 294103: {}, 300660: {}, 310341: {}, 314430: {}, 332156: {}, 339779: {}, 342972: {}, 343067: {}, 350487: {}, 386587: {}, 426807: {}, 430301: {}, 430778: {}, 433214: {}, 460518: {}, 468912: {}, 470598: {}, 470606: {}, 472423: {}, 506719: {}, 510410: {}, 520683: {}, 530859: {}, 536257: {}, 543864: {}, 571389: {}, 575146: {}, 575640: {}, 581231: {}, 587696: {}, 599205: {}, 609032: {}, 627172: {}, 637921: {}, 650569: {}, 656569: {}, 680383: {}, 702382: {}, 703474: {}, 726705: {}, 734261: {}, 745267: {}, 779289: {}, 782663: {}, 788771: {}, 806065: {}, 814407: {}, 823806: {}, 841446: {}, 841496: {}, 855680: {}, 892547: {}, 904767: {}, 905616: {}, 921488: {}, 922278: {}, 947382: {}, 948985: {}, 963684: {}, 966861: {}, 974988: {}, 982871: {}},
		{935: {}, 1797: {}, 5265: {}, 5381: {}, 8363: {}, 10810: {}, 15241: {}, 21403: {}, 21663: {}, 23001: {}, 24223: {}, 26259: {}, 27071: {}, 27245: {}, 27685: {}, 29397: {}, 31192: {}, 34862: {}, 35972: {}, 36427: {}, 39805: {}, 42574: {}, 48731: {}, 49683: {}, 50798: {}, 51385: {}, 51897: {}, 55773: {}, 59104: {}, 61970: {}, 64567: {}, 65780: {}, 66218: {}, 72926: {}, 79767: {}, 83343: {}, 84192: {}, 85496: {}, 88961: {}, 89348: {}, 89911: {}, 90346: {}, 91860: {}, 91871: {}, 93799: {}, 96566: {}, 96762: {}, 98931: {}, 106663: {}, 107060: {}, 108198: {}, 109595: {}, 111651: {}, 116320: {}, 118359: {}, 126717: {}, 127103: {}, 129549: {}, 130295: {}, 132404: {}, 133248: {}, 135545: {}, 144992: {}, 150710: {}, 151872: {}, 153028: {}, 156486: {}, 158115: {}, 158991: {}, 159658: {}, 162917: {}, 163188: {}, 172209: {}, 179404: {}, 182991: {}, 191545: {}, 191884: {}, 197816: {}, 199345: {}, 199376: {}, 200521: {}, 204064: {}, 206193: {}, 211466: {}, 213857: {}, 214630: {}, 215354: {}, 220516: {}, 221103: {}, 227586: {}, 227748: {}, 228191: {}, 232441: {}, 239181: {}, 242116: {}, 245659: {}, 259454: {}, 263029: {}, 263506: {}, 264527: {}, 266456: {}, 268589: {}, 271629: {}, 273639: {}, 278633: {}, 285908: {}, 286667: {}, 288812: {}, 298395: {}, 299377: {}, 303372: {}, 304828: {}, 306851: {}, 309625: {}, 310088: {}, 311740: {}, 315493: {}, 317853: {}, 320400: {}, 322183: {}, 328074: {}, 329370: {}, 330077: {}, 336027: {}, 339254: {}, 342267: {}, 344419: {}, 344581: {}, 357055: {}, 358865: {}, 364153: {}, 366987: {}, 379081: {}, 381671: {}, 381726: {}, 383624: {}, 385468: {}, 387385: {}, 397102: {}, 403296: {}, 404329: {}, 407294: {}, 410896: {}, 411354: {}, 413124: {}, 419971: {}, 421210: {}, 422145: {}, 429913: {}, 441402: {}, 442420: {}, 448418: {}, 450814: {}, 451486: {}, 458271: {}, 464824: {}, 471710: {}, 474777: {}, 474971: {}, 480563: {}, 483460: {}, 485254: {}, 491638: {}, 493826: {}, 497464: {}, 498636: {}, 499291: {}, 504798: {}, 508038: {}, 508591: {}, 515547: {}, 516669: {}, 521261: {}, 525927: {}, 527852: {}, 528004: {}, 528712: {}, 534686: {}, 536653: {}, 539609: {}, 542195: {}, 542270: {}, 542921: {}, 548041: {}, 549579: {}, 551402: {}, 554311: {}, 555734: {}, 558630: {}, 562073: {}, 562549: {}, 575254: {}, 580103: {}, 582641: {}, 589427: {}, 598467: {}, 598506: {}, 598530: {}, 600859: {}, 601768: {}, 604918: {}, 605886: {}, 606162: {}, 606760: {}, 609330: {}, 612581: {}, 614379: {}, 615722: {}, 618392: {}, 623592: {}, 624012: {}, 626818: {}, 629553: {}, 630306: {}, 630845: {}, 639819: {}, 639920: {}, 640405: {}, 641226: {}, 644295: {}, 644534: {}, 646786: {}, 647594: {}, 648724: {}, 656095: {}, 665006: {}, 665637: {}, 669484: {}, 669624: {}, 672143: {}, 675118: {}, 678055: {}, 681093: {}, 684825: {}, 687568: {}, 688986: {}, 689687: {}, 703873: {}, 703893: {}, 707177: {}, 708767: {}, 708996: {}, 709529: {}, 711760: {}, 715488: {}, 716814: {}, 722392: {}, 726801: {}, 730433: {}, 731016: {}, 731515: {}, 731842: {}, 738537: {}, 738589: {}, 741694: {}, 743394: {}, 743784: {}, 746852: {}, 748482: {}, 750464: {}, 751759: {}, 755212: {}, 755416: {}, 757430: {}, 762081: {}, 762817: {}, 764267: {}, 764668: {}, 767888: {}, 769136: {}, 770754: {}, 770872: {}, 771432: {}, 777141: {}, 778148: {}, 782475: {}, 783653: {}, 787557: {}, 787734: {}, 790799: {}, 794457: {}, 798794: {}, 802398: {}, 810399: {}, 811732: {}, 818254: {}, 819373: {}, 822012: {}, 832062: {}, 833491: {}, 834379: {}, 838703: {}, 838795: {}, 846230: {}, 848905: {}, 849930: {}, 858420: {}, 859721: {}, 862787: {}, 867172: {}, 867420: {}, 871767: {}, 875434: {}, 880697: {}, 882179: {}, 883995: {}, 888280: {}, 888456: {}, 892665: {}, 893706: {}, 893830: {}, 896033: {}, 897148: {}, 897673: {}, 897728: {}, 897987: {}, 898046: {}, 898955: {}, 902130: {}, 905494: {}, 909287: {}, 910009: {}, 916363: {}, 918339: {}, 921229: {}, 925620: {}, 932543: {}, 940846: {}, 941160: {}, 947148: {}, 951694: {}, 957272: {}, 960696: {}, 965080: {}, 965836: {}, 968619: {}, 971501: {}, 977754: {}, 992816: {}, 993182: {}, 994686: {}, 996962: {}},
		{36727: {}, 63977: {}, 72993: {}, 90014: {}, 105370: {}, 106134: {}, 117051: {}, 132242: {}, 181476: {}, 211564: {}, 234585: {}, 243859: {}, 271716: {}, 274197: {}, 275464: {}, 278409: {}, 280966: {}, 291792: {}, 325050: {}, 339183: {}, 346230: {}, 349251: {}, 358273: {}, 371417: {}, 383867: {}, 396738: {}, 409562: {}, 424683: {}, 443002: {}, 462179: {}, 471853: {}, 477680: {}, 495461: {}, 499685: {}, 502087: {}, 546208: {}, 571231: {}, 573200: {}, 577997: {}, 600505: {}, 635713: {}, 644978: {}, 668823: {}, 682823: {}, 694011: {}, 702753: {}, 705444: {}, 731912: {}, 737054: {}, 748653: {}, 758586: {}, 760702: {}, 768510: {}, 772207: {}, 782158: {}, 806484: {}, 808616: {}, 854672: {}, 864974: {}, 881724: {}, 883754: {}, 901859: {}, 907161: {}, 915000: {}, 915082: {}, 930095: {}, 967507: {}, 990307: {}, 992895: {}},
		{9308: {}, 13282: {}, 17990: {}, 18646: {}, 19082: {}, 50132: {}, 51049: {}, 56162: {}, 58767: {}, 60340: {}, 61754: {}, 71206: {}, 77017: {}, 79145: {}, 85589: {}, 86455: {}, 90386: {}, 91054: {}, 103294: {}, 108307: {}, 109840: {}, 111117: {}, 112300: {}, 114691: {}, 115818: {}, 119583: {}, 140496: {}, 144512: {}, 152155: {}, 152808: {}, 155920: {}, 156626: {}, 159443: {}, 163732: {}, 165392: {}, 168715: {}, 168718: {}, 190887: {}, 194692: {}, 194780: {}, 195515: {}, 204633: {}, 204895: {}, 208933: {}, 217467: {}, 224074: {}, 224519: {}, 225522: {}, 226503: {}, 232077: {}, 235736: {}, 242518: {}, 266058: {}, 266669: {}, 273578: {}, 274896: {}, 280454: {}, 286259: {}, 290965: {}, 300481: {}, 311583: {}, 311643: {}, 312753: {}, 320514: {}, 322851: {}, 329193: {}, 330321: {}, 349609: {}, 352805: {}, 359143: {}, 366130: {}, 371283: {}, 372706: {}, 374537: {}, 379808: {}, 380258: {}, 383794: {}, 387772: {}, 388629: {}, 389407: {}, 399447: {}, 402593: {}, 403985: {}, 407544: {}, 418709: {}, 419151: {}, 421158: {}, 431309: {}, 433362: {}, 435071: {}, 444864: {}, 462667: {}, 464806: {}, 465058: {}, 465630: {}, 467096: {}, 468918: {}, 474922: {}, 475249: {}, 483780: {}, 484014: {}, 485677: {}, 488375: {}, 489894: {}, 493215: {}, 495346: {}, 498012: {}, 508671: {}, 514580: {}, 517112: {}, 519343: {}, 520414: {}, 521508: {}, 522117: {}, 523796: {}, 526074: {}, 526181: {}, 534914: {}, 536722: {}, 541767: {}, 550700: {}, 551441: {}, 555274: {}, 555571: {}, 558364: {}, 564239: {}, 568764: {}, 571194: {}, 572063: {}, 574445: {}, 584197: {}, 584704: {}, 586346: {}, 586884: {}, 595444: {}, 595810: {}, 596723: {}, 615149: {}, 615753: {}, 615826: {}, 620280: {}, 624966: {}, 630045: {}, 633956: {}, 643540: {}, 643941: {}, 653184: {}, 663478: {}, 666227: {}, 683459: {}, 684784: {}, 694578: {}, 698195: {}, 699610: {}, 704307: {}, 711125: {}, 711481: {}, 714524: {}, 715981: {}, 717167: {}, 717915: {}, 718924: {}, 724533: {}, 736034: {}, 744647: {}, 746360: {}, 748703: {}, 750065: {}, 750374: {}, 753126: {}, 774565: {}, 776289: {}, 778626: {}, 781610: {}, 781742: {}, 782240: {}, 784276: {}, 784338: {}, 793600: {}, 796424: {}, 801413: {}, 801668: {}, 812299: {}, 813959: {}, 815197: {}, 817234: {}, 819762: {}, 820026: {}, 822050: {}, 824647: {}, 826899: {}, 828441: {}, 831909: {}, 835653: {}, 836320: {}, 838251: {}, 839656: {}, 846261: {}, 846803: {}, 847503: {}, 849791: {}, 855441: {}, 858183: {}, 861126: {}, 872439: {}, 873167: {}, 875416: {}, 892861: {}, 893489: {}, 894839: {}, 897106: {}, 901009: {}, 903689: {}, 916087: {}, 919183: {}, 919540: {}, 919707: {}, 921360: {}, 921816: {}, 923034: {}, 926093: {}, 929991: {}, 932919: {}, 934418: {}, 934578: {}, 939799: {}, 941874: {}, 944978: {}, 959952: {}, 960546: {}, 961809: {}, 963425: {}, 966482: {}, 967623: {}, 981136: {}, 983194: {}, 983248: {}, 983761: {}, 995818: {}, 997527: {}},
		{4375: {}, 5000: {}, 5399: {}, 10825: {}, 16832: {}, 35121: {}, 36542: {}, 36800: {}, 38866: {}, 40663: {}, 47665: {}, 52811: {}, 57345: {}, 59858: {}, 62319: {}, 66158: {}, 78401: {}, 83724: {}, 84129: {}, 86368: {}, 90922: {}, 98816: {}, 118426: {}, 121855: {}, 124812: {}, 128128: {}, 130896: {}, 131680: {}, 134240: {}, 140790: {}, 143213: {}, 149100: {}, 151196: {}, 154101: {}, 160197: {}, 175142: {}, 178104: {}, 183559: {}, 185197: {}, 187654: {}, 187927: {}, 199626: {}, 203638: {}, 211789: {}, 218678: {}, 222410: {}, 223392: {}, 230039: {}, 230325: {}, 230556: {}, 234621: {}, 235477: {}, 253717: {}, 258475: {}, 261382: {}, 261515: {}, 267997: {}, 272732: {}, 275886: {}, 282192: {}, 288087: {}, 297418: {}, 299076: {}, 307638: {}, 318054: {}, 328895: {}, 332636: {}, 335216: {}, 347466: {}, 349911: {}, 350297: {}, 360529: {}, 368348: {}, 374846: {}, 384018: {}, 385729: {}, 391486: {}, 407928: {}, 419657: {}, 420951: {}, 423482: {}, 429609: {}, 431323: {}, 441870: {}, 453333: {}, 453648: {}, 456283: {}, 464049: {}, 466087: {}, 469889: {}, 476456: {}, 478507: {}, 489329: {}, 490452: {}, 513713: {}, 514009: {}, 516838: {}, 537002: {}, 541528: {}, 543640: {}, 548697: {}, 549336: {}, 553282: {}, 562591: {}, 562727: {}, 573556: {}, 615021: {}, 619158: {}, 625925: {}, 630194: {}, 633098: {}, 635919: {}, 637882: {}, 646702: {}, 656093: {}, 657027: {}, 661873: {}, 666175: {}, 673748: {}, 677374: {}, 700397: {}, 711075: {}, 714907: {}, 721857: {}, 722980: {}, 724696: {}, 730292: {}, 735942: {}, 744330: {}, 759087: {}, 771457: {}, 771830: {}, 774058: {}, 782998: {}, 785103: {}, 785585: {}, 803313: {}, 806450: {}, 810933: {}, 811261: {}, 815112: {}, 819662: {}, 822187: {}, 823721: {}, 829759: {}, 841131: {}, 844000: {}, 845705: {}, 854884: {}, 855104: {}, 858749: {}, 859536: {}, 862212: {}, 863541: {}, 863586: {}, 866551: {}, 869718: {}, 877527: {}, 881631: {}, 891139: {}, 899390: {}, 904078: {}, 913180: {}, 919207: {}, 922422: {}, 928295: {}, 933418: {}, 943856: {}, 954979: {}, 955257: {}, 959731: {}, 960288: {}, 960797: {}, 965184: {}, 971667: {}, 979949: {}, 985377: {}, 986551: {}, 989161: {}, 989481: {}},
		{1645: {}, 6322: {}, 24095: {}, 30207: {}, 104963: {}, 108641: {}, 130380: {}, 143066: {}, 170470: {}, 172603: {}, 173181: {}, 200570: {}, 203987: {}, 222124: {}, 239206: {}, 267446: {}, 339858: {}, 369716: {}, 372838: {}, 377757: {}, 387589: {}, 390921: {}, 400041: {}, 423613: {}, 430565: {}, 456879: {}, 466464: {}, 500206: {}, 507082: {}, 534506: {}, 560540: {}, 566495: {}, 618916: {}, 638246: {}, 654013: {}, 750914: {}, 812691: {}, 818103: {}, 863629: {}, 886273: {}, 889301: {}, 938088: {}, 966619: {}},
		{1677: {}, 34572: {}, 48924: {}, 62149: {}, 71900: {}, 81981: {}, 87476: {}, 139022: {}, 175158: {}, 217871: {}, 263746: {}, 401201: {}, 406575: {}, 426203: {}, 441456: {}, 458252: {}, 459712: {}, 468382: {}, 471072: {}, 517961: {}, 608381: {}, 617541: {}, 661548: {}, 685259: {}, 703165: {}, 706760: {}, 732019: {}, 736767: {}, 739255: {}, 799856: {}, 802157: {}, 806965: {}, 807618: {}, 815933: {}, 848166: {}, 868218: {}, 895859: {}, 906095: {}, 948367: {}, 989307: {}},
		{16: {}, 3166: {}, 9087: {}, 19854: {}, 21238: {}, 21327: {}, 23389: {}, 55777: {}, 66279: {}, 68842: {}, 81481: {}, 83509: {}, 86448: {}, 88590: {}, 96084: {}, 96642: {}, 101028: {}, 117472: {}, 118962: {}, 119839: {}, 126260: {}, 126974: {}, 129687: {}, 131114: {}, 133262: {}, 133763: {}, 134340: {}, 137956: {}, 139992: {}, 149256: {}, 153942: {}, 160613: {}, 161141: {}, 161414: {}, 163446: {}, 165151: {}, 167074: {}, 170699: {}, 173808: {}, 179750: {}, 180191: {}, 182053: {}, 182227: {}, 186835: {}, 190211: {}, 190887: {}, 193821: {}, 201561: {}, 203858: {}, 204234: {}, 204868: {}, 210492: {}, 211311: {}, 228019: {}, 228259: {}, 236797: {}, 238986: {}, 242949: {}, 248328: {}, 253549: {}, 262486: {}, 268999: {}, 275601: {}, 281448: {}, 289871: {}, 292676: {}, 293714: {}, 295674: {}, 306018: {}, 307410: {}, 307518: {}, 309439: {}, 320201: {}, 330635: {}, 338515: {}, 345848: {}, 351229: {}, 355525: {}, 358746: {}, 362203: {}, 362230: {}, 363631: {}, 367715: {}, 386426: {}, 392884: {}, 401825: {}, 404016: {}, 416394: {}, 417130: {}, 418514: {}, 420598: {}, 426747: {}, 430418: {}, 434572: {}, 436394: {}, 444197: {}, 446241: {}, 472763: {}, 477352: {}, 479288: {}, 479507: {}, 479860: {}, 480422: {}, 482561: {}, 488182: {}, 494436: {}, 495140: {}, 495768: {}, 497438: {}, 498897: {}, 498979: {}, 500352: {}, 508728: {}, 509648: {}, 512059: {}, 526398: {}, 530410: {}, 534226: {}, 544170: {}, 554050: {}, 558392: {}, 562415: {}, 570842: {}, 578602: {}, 579252: {}, 582600: {}, 584599: {}, 586809: {}, 587061: {}, 594939: {}, 595350: {}, 611219: {}, 618415: {}, 620645: {}, 628167: {}, 640718: {}, 645191: {}, 645267: {}, 651711: {}, 652772: {}, 653108: {}, 655223: {}, 656684: {}, 659736: {}, 660484: {}, 666568: {}, 669200: {}, 675504: {}, 678046: {}, 678614: {}, 680468: {}, 687067: {}, 692667: {}, 693485: {}, 697422: {}, 719529: {}, 727131: {}, 732916: {}, 736719: {}, 739774: {}, 740166: {}, 740633: {}, 745023: {}, 751435: {}, 755638: {}, 758499: {}, 778070: {}, 784320: {}, 784947: {}, 787912: {}, 788936: {}, 800857: {}, 801258: {}, 805499: {}, 805961: {}, 807047: {}, 836135: {}, 843851: {}, 847589: {}, 847950: {}, 860099: {}, 863373: {}, 864239: {}, 864315: {}, 865556: {}, 872350: {}, 892150: {}, 901407: {}, 901698: {}, 906316: {}, 910928: {}, 916348: {}, 916888: {}, 916972: {}, 918142: {}, 919413: {}, 939421: {}, 939551: {}, 948164: {}, 963522: {}, 969469: {}, 971729: {}, 975099: {}, 978332: {}, 980619: {}, 981129: {}, 981829: {}, 986886: {}, 994881: {}, 998918: {}},
		{9911: {}, 11546: {}, 17579: {}, 31468: {}, 34798: {}, 39903: {}, 42345: {}, 46022: {}, 62499: {}, 65854: {}, 69825: {}, 71824: {}, 82868: {}, 84868: {}, 91871: {}, 92172: {}, 92863: {}, 94790: {}, 95453: {}, 96871: {}, 100622: {}, 111860: {}, 112353: {}, 119770: {}, 120436: {}, 124471: {}, 129764: {}, 134186: {}, 142083: {}, 142105: {}, 145178: {}, 161027: {}, 165232: {}, 169254: {}, 177546: {}, 186302: {}, 195286: {}, 196585: {}, 199110: {}, 209688: {}, 216812: {}, 217144: {}, 224627: {}, 236535: {}, 240425: {}, 248230: {}, 254884: {}, 255666: {}, 257886: {}, 266663: {}, 270233: {}, 270330: {}, 273202: {}, 280912: {}, 285072: {}, 291668: {}, 296277: {}, 316010: {}, 316151: {}, 327098: {}, 333803: {}, 333996: {}, 346561: {}, 358868: {}, 361520: {}, 363869: {}, 367041: {}, 367276: {}, 374964: {}, 382168: {}, 382651: {}, 383480: {}, 394514: {}, 397417: {}, 402082: {}, 402941: {}, 411156: {}, 415445: {}, 418478: {}, 425634: {}, 463342: {}, 466375: {}, 467145: {}, 472818: {}, 475329: {}, 477370: {}, 479941: {}, 495455: {}, 507936: {}, 513941: {}, 514557: {}, 520157: {}, 524718: {}, 541790: {}, 548126: {}, 552890: {}, 557313: {}, 559468: {}, 568372: {}, 568956: {}, 572051: {}, 576027: {}, 578252: {}, 578511: {}, 586718: {}, 588213: {}, 592971: {}, 597652: {}, 599835: {}, 619295: {}, 619619: {}, 621571: {}, 624838: {}, 633361: {}, 635084: {}, 641477: {}, 647806: {}, 661048: {}, 663340: {}, 663618: {}, 672474: {}, 689921: {}, 690835: {}, 692450: {}, 695380: {}, 695807: {}, 702692: {}, 703025: {}, 704191: {}, 709468: {}, 709561: {}, 716491: {}, 731368: {}, 732643: {}, 744584: {}, 745922: {}, 753489: {}, 754410: {}, 757051: {}, 759848: {}, 760962: {}, 785201: {}, 796258: {}, 797462: {}, 821453: {}, 822717: {}, 834716: {}, 836607: {}, 838977: {}, 840095: {}, 848747: {}, 851099: {}, 851459: {}, 852033: {}, 852415: {}, 853403: {}, 855130: {}, 855575: {}, 857876: {}, 862611: {}, 866853: {}, 868938: {}, 871487: {}, 880106: {}, 886290: {}, 886357: {}, 887792: {}, 892041: {}, 893432: {}, 899034: {}, 901943: {}, 902632: {}, 902670: {}, 905320: {}, 905514: {}, 923394: {}, 930005: {}, 931595: {}, 933502: {}, 946736: {}, 947615: {}, 954874: {}, 984931: {}, 987958: {}, 997024: {}},
		{19193: {}, 47119: {}, 56979: {}, 61554: {}, 96132: {}, 277648: {}, 299398: {}, 344316: {}, 442966: {}, 475323: {}, 528237: {}, 547268: {}, 547940: {}, 626161: {}, 631251: {}, 669935: {}, 706305: {}, 712821: {}, 730871: {}, 802246: {}, 819042: {}, 847156: {}, 870746: {}, 886785: {}, 950542: {}, 961534: {}},
		{17909: {}, 54668: {}, 76579: {}, 107386: {}, 117899: {}, 121458: {}, 129708: {}, 133325: {}, 256731: {}, 258551: {}, 273517: {}, 308758: {}, 345442: {}, 363264: {}, 390851: {}, 410432: {}, 448472: {}, 460802: {}, 470534: {}, 491344: {}, 542972: {}, 559454: {}, 610637: {}, 613421: {}, 623550: {}, 653443: {}, 654854: {}, 666105: {}, 685176: {}, 712048: {}, 715522: {}, 736112: {}, 737071: {}, 785559: {}, 786868: {}, 794950: {}, 839713: {}, 851730: {}, 862608: {}, 863467: {}, 867116: {}, 870868: {}, 895345: {}, 918647: {}, 941267: {}, 942262: {}, 965118: {}, 970670: {}, 973720: {}, 981351: {}, 987995: {}, 998305: {}},
		{1439: {}, 5874: {}, 8852: {}, 10849: {}, 13322: {}, 14493: {}, 14865: {}, 18047: {}, 22700: {}, 22828: {}, 22829: {}, 22871: {}, 23039: {}, 24094: {}, 24411: {}, 31039: {}, 32219: {}, 32831: {}, 33088: {}, 37851: {}, 37927: {}, 38612: {}, 38978: {}, 41878: {}, 44057: {}, 44144: {}, 44578: {}, 45922: {}, 54139: {}, 56708: {}, 59940: {}, 62673: {}, 65432: {}, 66801: {}, 69079: {}, 73545: {}, 74136: {}, 75182: {}, 75444: {}, 76569: {}, 79071: {}, 80528: {}, 83452: {}, 83825: {}, 87715: {}, 93289: {}, 97004: {}, 98159: {}, 103402: {}, 103420: {}, 104857: {}, 106034: {}, 108090: {}, 108312: {}, 108467: {}, 108814: {}, 108914: {}, 112987: {}, 115621: {}, 118167: {}, 118663: {}, 121059: {}, 122008: {}, 123883: {}, 124160: {}, 125601: {}, 125994: {}, 126754: {}, 130136: {}, 130539: {}, 131876: {}, 135001: {}, 135193: {}, 135701: {}, 142372: {}, 147446: {}, 148971: {}, 149756: {}, 151278: {}, 153866: {}, 155575: {}, 155733: {}, 156623: {}, 157121: {}, 157895: {}, 163645: {}, 165152: {}, 171421: {}, 174207: {}, 174824: {}, 175654: {}, 175867: {}, 176797: {}, 180053: {}, 181908: {}, 185212: {}, 187682: {}, 189207: {}, 190194: {}, 190832: {}, 193604: {}, 199287: {}, 200102: {}, 203519: {}, 205643: {}, 206543: {}, 207741: {}, 207976: {}, 208017: {}, 208942: {}, 212809: {}, 214693: {}, 216885: {}, 217179: {}, 217318: {}, 218700: {}, 219199: {}, 222805: {}, 234805: {}, 242257: {}, 242393: {}, 242741: {}, 244642: {}, 245196: {}, 247512: {}, 247566: {}, 250673: {}, 256428: {}, 259531: {}, 259799: {}, 260054: {}, 261045: {}, 261934: {}, 262831: {}, 265637: {}, 276779: {}, 277433: {}, 278593: {}, 280613: {}, 281866: {}, 282041: {}, 282094: {}, 282945: {}, 283077: {}, 283843: {}, 288097: {}, 293379: {}, 294366: {}, 294401: {}, 296314: {}, 299041: {}, 301377: {}, 301806: {}, 302313: {}, 304060: {}, 305674: {}, 306940: {}, 310056: {}, 310612: {}, 310704: {}, 313355: {}, 317199: {}, 317481: {}, 318896: {}, 321003: {}, 324642: {}, 324800: {}, 324871: {}, 327133: {}, 331134: {}, 334998: {}, 335726: {}, 338984: {}, 340070: {}, 340541: {}, 341129: {}, 342237: {}, 344249: {}, 344865: {}, 346673: {}, 348817: {}, 348826: {}, 349396: {}, 350806: {}, 353619: {}, 355085: {}, 355277: {}, 360258: {}, 360278: {}, 360348: {}, 369288: {}, 373476: {}, 374403: {}, 377555: {}, 378789: {}, 378883: {}, 378933: {}, 380065: {}, 385639: {}, 386467: {}, 387753: {}, 387853: {}, 390282: {}, 400321: {}, 404735: {}, 410487: {}, 413046: {}, 413522: {}, 415322: {}, 416398: {}, 419379: {}, 422111: {}, 422217: {}, 423566: {}, 424542: {}, 425013: {}, 428546: {}, 428761: {}, 428943: {}, 429460: {}, 432486: {}, 438404: {}, 438692: {}, 439824: {}, 441686: {}, 445655: {}, 445869: {}, 452578: {}, 453862: {}, 457407: {}, 457716: {}, 459182: {}, 459970: {}, 462666: {}, 463047: {}, 463804: {}, 464514: {}, 465899: {}, 470690: {}, 471791: {}, 473846: {}, 474712: {}, 475333: {}, 476165: {}, 484617: {}, 485722: {}, 486072: {}, 486120: {}, 486279: {}, 486402: {}, 487505: {}, 491030: {}, 491107: {}, 494676: {}, 496470: {}, 497918: {}, 500639: {}, 502164: {}, 502604: {}, 503057: {}, 505639: {}, 507929: {}, 509051: {}, 509972: {}, 509986: {}, 510907: {}, 511029: {}, 511329: {}, 511461: {}, 511633: {}, 512193: {}, 512793: {}, 519476: {}, 520260: {}, 521312: {}, 523607: {}, 525819: {}, 528128: {}, 529811: {}, 530651: {}, 530778: {}, 531799: {}, 534093: {}, 534526: {}, 535086: {}, 535576: {}, 538652: {}, 549644: {}, 551073: {}, 551931: {}, 559987: {}, 561019: {}, 566451: {}, 566728: {}, 568369: {}, 570033: {}, 573724: {}, 574702: {}, 575362: {}, 580216: {}, 581516: {}, 582641: {}, 582659: {}, 583152: {}, 583545: {}, 584673: {}, 588157: {}, 591258: {}, 591571: {}, 592356: {}, 594982: {}, 597226: {}, 608859: {}, 610127: {}, 611018: {}, 612090: {}, 614895: {}, 615897: {}, 617791: {}, 623991: {}, 625699: {}, 626953: {}, 629191: {}, 632090: {}, 633704: {}, 634284: {}, 636008: {}, 636075: {}, 639278: {}, 644652: {}, 644788: {}, 645851: {}, 647320: {}, 652216: {}, 655508: {}, 656942: {}, 657423: {}, 658313: {}, 659922: {}, 661103: {}, 663750: {}, 664033: {}, 670384: {}, 671283: {}, 674403: {}, 676213: {}, 676792: {}, 679116: {}, 680516: {}, 684478: {}, 685742: {}, 686192: {}, 687236: {}, 690093: {}, 693770: {}, 695471: {}, 696466: {}, 697968: {}, 698646: {}, 698862: {}, 702494: {}, 702651: {}, 703387: {}, 706803: {}, 707297: {}, 709479: {}, 709580: {}, 709737: {}, 714740: {}, 721395: {}, 724097: {}, 724634: {}, 727060: {}, 727912: {}, 729715: {}, 732863: {}, 733032: {}, 735959: {}, 739636: {}, 741315: {}, 742481: {}, 745801: {}, 745998: {}, 746722: {}, 747682: {}, 752847: {}, 755337: {}, 757679: {}, 763247: {}, 765275: {}, 765400: {}, 767718: {}, 769304: {}, 771240: {}, 772430: {}, 773865: {}, 775840: {}, 780179: {}, 781594: {}, 783177: {}, 783248: {}, 783714: {}, 784328: {}, 786835: {}, 790628: {}, 793902: {}, 794601: {}, 794723: {}, 794976: {}, 799655: {}, 800657: {}, 801597: {}, 802851: {}, 803413: {}, 805708: {}, 808125: {}, 810121: {}, 812472: {}, 813132: {}, 818634: {}, 818882: {}, 823833: {}, 824901: {}, 826174: {}, 827011: {}, 829461: {}, 833531: {}, 833646: {}, 834217: {}, 834535: {}, 838270: {}, 842918: {}, 844001: {}, 846096: {}, 846294: {}, 846968: {}, 847237: {}, 849627: {}, 849917: {}, 850260: {}, 850654: {}, 850783: {}, 852059: {}, 853140: {}, 856452: {}, 858608: {}, 858655: {}, 859866: {}, 862581: {}, 863765: {}, 870669: {}, 871220: {}, 874904: {}, 875386: {}, 878663: {}, 880873: {}, 885351: {}, 885819: {}, 888527: {}, 890183: {}, 892148: {}, 896606: {}, 897015: {}, 899108: {}, 900578: {}, 901827: {}, 903214: {}, 903865: {}, 904435: {}, 907529: {}, 907921: {}, 909860: {}, 913723: {}, 914605: {}, 914948: {}, 915133: {}, 916944: {}, 917278: {}, 919798: {}, 920385: {}, 926312: {}, 928738: {}, 935791: {}, 936159: {}, 937301: {}, 938095: {}, 941790: {}, 942073: {}, 943694: {}, 945297: {}, 946306: {}, 948399: {}, 948876: {}, 950521: {}, 952628: {}, 954044: {}, 958379: {}, 958429: {}, 959388: {}, 960042: {}, 960713: {}, 962391: {}, 963079: {}, 963360: {}, 963959: {}, 965833: {}, 966707: {}, 969139: {}, 971526: {}, 971695: {}, 972925: {}, 977207: {}, 977922: {}, 978645: {}, 979548: {}, 982745: {}, 986582: {}, 991850: {}, 992866: {}, 992990: {}, 999291: {}},
		{6046: {}, 7217: {}, 10462: {}, 12763: {}, 15591: {}, 21994: {}, 23509: {}, 23863: {}, 47312: {}, 49265: {}, 49444: {}, 59742: {}, 77894: {}, 82724: {}, 98806: {}, 102215: {}, 102350: {}, 104411: {}, 107925: {}, 109795: {}, 117476: {}, 120153: {}, 120157: {}, 121790: {}, 128205: {}, 132062: {}, 133108: {}, 136362: {}, 147689: {}, 150995: {}, 158381: {}, 173363: {}, 175936: {}, 183874: {}, 185958: {}, 189225: {}, 193499: {}, 197787: {}, 201808: {}, 206812: {}, 213533: {}, 223850: {}, 224554: {}, 226659: {}, 232118: {}, 233334: {}, 235973: {}, 249864: {}, 254176: {}, 272810: {}, 274278: {}, 277120: {}, 279236: {}, 280117: {}, 281236: {}, 281381: {}, 285836: {}, 288820: {}, 289508: {}, 290441: {}, 296475: {}, 296910: {}, 301630: {}, 302367: {}, 303361: {}, 304020: {}, 305095: {}, 311760: {}, 314119: {}, 315869: {}, 329855: {}, 331000: {}, 334541: {}, 339480: {}, 345246: {}, 352455: {}, 353103: {}, 354147: {}, 354564: {}, 362591: {}, 365328: {}, 368383: {}, 369065: {}, 376118: {}, 379424: {}, 382196: {}, 382987: {}, 391534: {}, 392250: {}, 393704: {}, 394112: {}, 397246: {}, 399639: {}, 403487: {}, 417008: {}, 419810: {}, 428993: {}, 437155: {}, 452741: {}, 480133: {}, 480221: {}, 504438: {}, 504521: {}, 506408: {}, 511534: {}, 527015: {}, 530031: {}, 535252: {}, 537880: {}, 539204: {}, 541175: {}, 545832: {}, 546192: {}, 546609: {}, 548147: {}, 548284: {}, 548943: {}, 558572: {}, 560944: {}, 563900: {}, 567766: {}, 573993: {}, 577495: {}, 581182: {}, 584793: {}, 587448: {}, 589825: {}, 590100: {}, 593044: {}, 599186: {}, 604160: {}, 606511: {}, 616061: {}, 619031: {}, 620917: {}, 621540: {}, 625149: {}, 626791: {}, 629756: {}, 629939: {}, 632546: {}, 636577: {}, 637445: {}, 638132: {}, 651092: {}, 651413: {}, 655763: {}, 660988: {}, 663275: {}, 664520: {}, 666689: {}, 679932: {}, 695117: {}, 701901: {}, 703729: {}, 704227: {}, 711524: {}, 714539: {}, 717373: {}, 724182: {}, 727745: {}, 733532: {}, 737611: {}, 747065: {}, 750840: {}, 755202: {}, 756167: {}, 757137: {}, 757772: {}, 762202: {}, 772822: {}, 772910: {}, 784788: {}, 807400: {}, 822846: {}, 822944: {}, 823434: {}, 829593: {}, 840176: {}, 841259: {}, 842312: {}, 842352: {}, 843506: {}, 849235: {}, 849239: {}, 858623: {}, 860024: {}, 863984: {}, 866414: {}, 870277: {}, 877769: {}, 885696: {}, 886889: {}, 896973: {}, 904267: {}, 904946: {}, 913960: {}, 918113: {}, 918207: {}, 919342: {}, 924163: {}, 929898: {}, 930521: {}, 930798: {}, 937296: {}, 947772: {}, 947874: {}, 951341: {}, 955547: {}, 962297: {}, 963290: {}, 967519: {}, 972772: {}, 973315: {}, 990383: {}, 995103: {}},
		{15815: {}, 16299: {}, 37184: {}, 42449: {}, 43437: {}, 56201: {}, 69697: {}, 72566: {}, 72904: {}, 77761: {}, 109062: {}, 124996: {}, 126439: {}, 128830: {}, 140757: {}, 145701: {}, 154916: {}, 167381: {}, 178980: {}, 188957: {}, 197694: {}, 201649: {}, 202209: {}, 211543: {}, 232571: {}, 239488: {}, 248660: {}, 249397: {}, 258105: {}, 275137: {}, 278306: {}, 281311: {}, 291533: {}, 292270: {}, 301586: {}, 302600: {}, 304447: {}, 306932: {}, 323122: {}, 324693: {}, 328611: {}, 329414: {}, 333379: {}, 342494: {}, 343333: {}, 354432: {}, 354790: {}, 355382: {}, 358813: {}, 360467: {}, 373417: {}, 399215: {}, 401486: {}, 404786: {}, 406125: {}, 407176: {}, 411169: {}, 442435: {}, 443442: {}, 446315: {}, 451425: {}, 457785: {}, 458203: {}, 465578: {}, 472129: {}, 473301: {}, 479995: {}, 486890: {}, 495299: {}, 498064: {}, 547279: {}, 547966: {}, 549461: {}, 565637: {}, 574186: {}, 584930: {}, 586310: {}, 588696: {}, 591988: {}, 602039: {}, 603773: {}, 604229: {}, 620623: {}, 633591: {}, 644008: {}, 645968: {}, 678678: {}, 684015: {}, 697544: {}, 698047: {}, 700276: {}, 704835: {}, 707935: {}, 708926: {}, 736557: {}, 737944: {}, 741454: {}, 754894: {}, 760136: {}, 775223: {}, 778331: {}, 781641: {}, 794311: {}, 807657: {}, 811277: {}, 811341: {}, 814696: {}, 815223: {}, 815510: {}, 829259: {}, 841083: {}, 856271: {}, 861952: {}, 863706: {}, 871196: {}, 873081: {}, 884233: {}, 896550: {}, 908747: {}, 909490: {}, 915049: {}, 936063: {}, 954893: {}, 956229: {}, 959018: {}, 959801: {}, 962345: {}, 962592: {}, 970944: {}, 985044: {}, 985355: {}, 989228: {}, 992399: {}, 997871: {}},
		{260: {}, 5582: {}, 8087: {}, 16397: {}, 20770: {}, 33084: {}, 38342: {}, 40313: {}, 42508: {}, 45231: {}, 51219: {}, 67166: {}, 70029: {}, 72339: {}, 75162: {}, 83277: {}, 94824: {}, 99542: {}, 136331: {}, 145482: {}, 159949: {}, 160277: {}, 161513: {}, 162867: {}, 169031: {}, 180324: {}, 181409: {}, 185066: {}, 193026: {}, 196680: {}, 207214: {}, 207433: {}, 210875: {}, 215441: {}, 216768: {}, 235613: {}, 247370: {}, 257170: {}, 275910: {}, 277395: {}, 278561: {}, 279628: {}, 282181: {}, 298128: {}, 301304: {}, 314990: {}, 317459: {}, 321850: {}, 329663: {}, 333838: {}, 335411: {}, 343538: {}, 343951: {}, 346552: {}, 356005: {}, 358226: {}, 369608: {}, 370223: {}, 374883: {}, 378660: {}, 389997: {}, 405016: {}, 424953: {}, 430149: {}, 456418: {}, 460896: {}, 469159: {}, 500105: {}, 505548: {}, 507847: {}, 517590: {}, 521230: {}, 523621: {}, 528724: {}, 531988: {}, 538542: {}, 568105: {}, 576627: {}, 577085: {}, 583176: {}, 586967: {}, 592866: {}, 603159: {}, 608156: {}, 620117: {}, 645165: {}, 660340: {}, 670893: {}, 677419: {}, 677483: {}, 681523: {}, 685808: {}, 717432: {}, 717943: {}, 731485: {}, 731737: {}, 748218: {}, 749880: {}, 751795: {}, 756219: {}, 762362: {}, 767371: {}, 769279: {}, 790460: {}, 791034: {}, 791531: {}, 818144: {}, 829312: {}, 844668: {}, 845321: {}, 851251: {}, 864864: {}, 885479: {}, 908530: {}, 940087: {}, 963599: {}, 966577: {}, 969060: {}, 970427: {}, 984632: {}, 995312: {}},
		{7167: {}, 8466: {}, 10869: {}, 15822: {}, 18905: {}, 19996: {}, 21118: {}, 27702: {}, 31732: {}, 39982: {}, 56899: {}, 59451: {}, 67530: {}, 69936: {}, 72464: {}, 76877: {}, 79537: {}, 85400: {}, 92664: {}, 92941: {}, 104920: {}, 107184: {}, 119403: {}, 126761: {}, 136094: {}, 145481: {}, 160344: {}, 164738: {}, 166859: {}, 167169: {}, 167369: {}, 168933: {}, 178199: {}, 179016: {}, 194710: {}, 197676: {}, 216725: {}, 223815: {}, 229747: {}, 232908: {}, 233772: {}, 236948: {}, 260081: {}, 262300: {}, 263586: {}, 265103: {}, 277766: {}, 278479: {}, 299520: {}, 302915: {}, 310817: {}, 315762: {}, 317966: {}, 320768: {}, 327276: {}, 332259: {}, 347134: {}, 348428: {}, 354239: {}, 358070: {}, 365282: {}, 378677: {}, 386327: {}, 390697: {}, 394208: {}, 400014: {}, 401521: {}, 410061: {}, 410296: {}, 421791: {}, 423415: {}, 424233: {}, 427549: {}, 442586: {}, 445829: {}, 455136: {}, 456301: {}, 460942: {}, 464243: {}, 479470: {}, 489179: {}, 506300: {}, 521118: {}, 528877: {}, 536679: {}, 536929: {}, 541440: {}, 559259: {}, 572129: {}, 608506: {}, 615742: {}, 624153: {}, 631850: {}, 633546: {}, 637710: {}, 641040: {}, 665243: {}, 669051: {}, 671582: {}, 686940: {}, 690947: {}, 701032: {}, 702092: {}, 715946: {}, 725621: {}, 729823: {}, 735590: {}, 736767: {}, 742598: {}, 744612: {}, 750890: {}, 756090: {}, 770222: {}, 780676: {}, 780798: {}, 796987: {}, 797608: {}, 803542: {}, 806457: {}, 829013: {}, 846369: {}, 848881: {}, 854425: {}, 857279: {}, 860868: {}, 866109: {}, 866230: {}, 868331: {}, 868782: {}, 871077: {}, 871731: {}, 875412: {}, 880219: {}, 882106: {}, 885187: {}, 888093: {}, 891943: {}, 899525: {}, 899664: {}, 907656: {}, 914588: {}, 914900: {}, 915802: {}, 919145: {}, 922486: {}, 924227: {}, 941769: {}, 962944: {}, 966810: {}, 980708: {}, 984893: {}, 993741: {}, 996672: {}},
		{1780: {}, 2843: {}, 4835: {}, 17596: {}, 19010: {}, 19623: {}, 19958: {}, 21527: {}, 38993: {}, 39273: {}, 40365: {}, 42556: {}, 43362: {}, 45744: {}, 45766: {}, 48361: {}, 53828: {}, 60364: {}, 61498: {}, 63452: {}, 64915: {}, 70895: {}, 77403: {}, 95358: {}, 101692: {}, 103624: {}, 109953: {}, 110262: {}, 111147: {}, 115687: {}, 120044: {}, 123535: {}, 132719: {}, 133559: {}, 135903: {}, 136360: {}, 138776: {}, 146044: {}, 149399: {}, 159602: {}, 161101: {}, 165159: {}, 170830: {}, 174900: {}, 176376: {}, 179105: {}, 181498: {}, 183092: {}, 184205: {}, 185498: {}, 185811: {}, 187324: {}, 190302: {}, 195164: {}, 202392: {}, 203032: {}, 208671: {}, 213815: {}, 219547: {}, 220517: {}, 223121: {}, 223460: {}, 225898: {}, 230152: {}, 235755: {}, 241202: {}, 241960: {}, 242447: {}, 246068: {}, 264122: {}, 265303: {}, 266098: {}, 267278: {}, 274497: {}, 275825: {}, 276707: {}, 277253: {}, 293667: {}, 295064: {}, 297024: {}, 300927: {}, 301680: {}, 307223: {}, 307407: {}, 311114: {}, 318204: {}, 318741: {}, 323570: {}, 330779: {}, 344744: {}, 347845: {}, 353342: {}, 356969: {}, 360238: {}, 361896: {}, 373236: {}, 375434: {}, 385311: {}, 386461: {}, 392676: {}, 402777: {}, 403357: {}, 412964: {}, 414504: {}, 416033: {}, 417732: {}, 421237: {}, 431636: {}, 432118: {}, 433386: {}, 434015: {}, 437665: {}, 439507: {}, 443445: {}, 450655: {}, 455907: {}, 466576: {}, 467240: {}, 470073: {}, 470923: {}, 479849: {}, 485155: {}, 486776: {}, 491518: {}, 493966: {}, 495314: {}, 499893: {}, 502503: {}, 503824: {}, 506724: {}, 508274: {}, 508531: {}, 520378: {}, 524250: {}, 525824: {}, 526435: {}, 532319: {}, 535476: {}, 539188: {}, 540678: {}, 543139: {}, 545509: {}, 545846: {}, 554361: {}, 554523: {}, 554641: {}, 562503: {}, 569187: {}, 573851: {}, 582233: {}, 582588: {}, 599332: {}, 601598: {}, 608471: {}, 609359: {}, 623057: {}, 624986: {}, 627637: {}, 629617: {}, 631366: {}, 631691: {}, 632543: {}, 634443: {}, 639087: {}, 641398: {}, 641769: {}, 646057: {}, 650857: {}, 656797: {}, 657660: {}, 658303: {}, 658344: {}, 658441: {}, 660667: {}, 667611: {}, 670532: {}, 671110: {}, 671644: {}, 673798: {}, 674474: {}, 674955: {}, 675118: {}, 676480: {}, 676542: {}, 684142: {}, 690283: {}, 691819: {}, 695601: {}, 698044: {}, 699044: {}, 710355: {}, 713458: {}, 714289: {}, 716624: {}, 718720: {}, 720317: {}, 722880: {}, 724961: {}, 725113: {}, 729926: {}, 732993: {}, 736815: {}, 744136: {}, 744241: {}, 747265: {}, 749244: {}, 759322: {}, 759779: {}, 762423: {}, 769903: {}, 776406: {}, 778348: {}, 780747: {}, 785710: {}, 787129: {}, 788359: {}, 790332: {}, 793377: {}, 800938: {}, 801740: {}, 806492: {}, 807438: {}, 817949: {}, 819499: {}, 822844: {}, 824212: {}, 825791: {}, 828474: {}, 828558: {}, 830552: {}, 831784: {}, 833279: {}, 834099: {}, 837145: {}, 839895: {}, 843791: {}, 850486: {}, 855151: {}, 857995: {}, 859656: {}, 863372: {}, 866958: {}, 869486: {}, 875582: {}, 878999: {}, 880004: {}, 883148: {}, 885479: {}, 886787: {}, 886824: {}, 890161: {}, 890765: {}, 893536: {}, 900480: {}, 906712: {}, 907562: {}, 910859: {}, 911296: {}, 912164: {}, 915797: {}, 918976: {}, 922786: {}, 923742: {}, 927903: {}, 928354: {}, 933639: {}, 937221: {}, 938433: {}, 945610: {}, 954107: {}, 957992: {}, 966251: {}, 970887: {}, 972156: {}, 975277: {}, 978065: {}, 980595: {}, 980979: {}, 981295: {}, 984535: {}, 986395: {}, 990909: {}, 994690: {}, 995036: {}, 995158: {}, 995410: {}},
		{334: {}, 32655: {}, 55046: {}, 62022: {}, 73613: {}, 77343: {}, 101124: {}, 107068: {}, 117499: {}, 117730: {}, 120673: {}, 124794: {}, 132620: {}, 146974: {}, 147517: {}, 150501: {}, 156095: {}, 172428: {}, 173315: {}, 175385: {}, 175574: {}, 190869: {}, 198353: {}, 206146: {}, 216949: {}, 217954: {}, 242237: {}, 245294: {}, 264560: {}, 278889: {}, 291762: {}, 292801: {}, 300075: {}, 300481: {}, 302243: {}, 318737: {}, 324658: {}, 324864: {}, 329315: {}, 329700: {}, 337062: {}, 341116: {}, 342497: {}, 350780: {}, 354770: {}, 368156: {}, 375739: {}, 379571: {}, 382754: {}, 384423: {}, 392739: {}, 394348: {}, 398285: {}, 411003: {}, 412595: {}, 425690: {}, 432712: {}, 445636: {}, 465052: {}, 465365: {}, 468093: {}, 469866: {}, 475584: {}, 476164: {}, 477804: {}, 487777: {}, 488760: {}, 490033: {}, 510553: {}, 513808: {}, 520217: {}, 526976: {}, 529753: {}, 538410: {}, 546002: {}, 554632: {}, 564668: {}, 569539: {}, 581783: {}, 606930: {}, 607918: {}, 608347: {}, 611796: {}, 623435: {}, 642322: {}, 651086: {}, 651191: {}, 662426: {}, 663401: {}, 665029: {}, 678084: {}, 683432: {}, 683683: {}, 692495: {}, 694074: {}, 699463: {}, 707528: {}, 712394: {}, 717653: {}, 731394: {}, 734593: {}, 745234: {}, 751297: {}, 751837: {}, 755654: {}, 762967: {}, 768704: {}, 769565: {}, 773686: {}, 778473: {}, 804277: {}, 821133: {}, 832543: {}, 841831: {}, 851930: {}, 857658: {}, 867008: {}, 871904: {}, 872283: {}, 874563: {}, 886715: {}, 890451: {}, 897991: {}, 904759: {}, 907308: {}, 907359: {}, 910313: {}, 912964: {}, 915634: {}, 917989: {}, 932513: {}, 939019: {}, 940626: {}, 949595: {}, 961152: {}, 967112: {}, 992688: {}, 995334: {}, 997260: {}},
		{428: {}, 1889: {}, 6110: {}, 14464: {}, 30078: {}, 41633: {}, 43356: {}, 43435: {}, 47634: {}, 56164: {}, 67752: {}, 70826: {}, 84292: {}, 94432: {}, 99168: {}, 100153: {}, 107749: {}, 113178: {}, 113582: {}, 127241: {}, 134574: {}, 137433: {}, 139057: {}, 146335: {}, 152544: {}, 154286: {}, 168427: {}, 169087: {}, 170463: {}, 174853: {}, 183261: {}, 184904: {}, 188281: {}, 189127: {}, 190975: {}, 194390: {}, 201095: {}, 202816: {}, 205157: {}, 207939: {}, 211872: {}, 212590: {}, 223859: {}, 250815: {}, 256597: {}, 256750: {}, 269329: {}, 278881: {}, 280093: {}, 283752: {}, 292910: {}, 300938: {}, 306840: {}, 313670: {}, 314989: {}, 319535: {}, 321635: {}, 328144: {}, 340233: {}, 341936: {}, 344531: {}, 352863: {}, 372511: {}, 382752: {}, 395161: {}, 397062: {}, 399074: {}, 401217: {}, 403516: {}, 403845: {}, 406988: {}, 410969: {}, 412686: {}, 413753: {}, 428567: {}, 432535: {}, 443525: {}, 451282: {}, 467986: {}, 471367: {}, 473274: {}, 474429: {}, 478817: {}, 495301: {}, 503719: {}, 511865: {}, 512989: {}, 515783: {}, 528073: {}, 547924: {}, 547990: {}, 559159: {}, 567034: {}, 571787: {}, 577668: {}, 600386: {}, 601085: {}, 604438: {}, 606388: {}, 606810: {}, 619201: {}, 631228: {}, 644327: {}, 646549: {}, 649628: {}, 656910: {}, 658084: {}, 665166: {}, 666827: {}, 670414: {}, 672141: {}, 672434: {}, 672476: {}, 689902: {}, 693412: {}, 697202: {}, 701615: {}, 705986: {}, 711631: {}, 711975: {}, 727545: {}, 730273: {}, 734081: {}, 738142: {}, 740576: {}, 750787: {}, 752974: {}, 760598: {}, 761706: {}, 763263: {}, 775671: {}, 776423: {}, 779169: {}, 786672: {}, 795932: {}, 800366: {}, 804031: {}, 805465: {}, 812387: {}, 812916: {}, 822114: {}, 824099: {}, 825068: {}, 827134: {}, 828613: {}, 832090: {}, 847079: {}, 856342: {}, 857505: {}, 866123: {}, 866348: {}, 869166: {}, 880868: {}, 882259: {}, 884414: {}, 892521: {}, 905634: {}, 908987: {}, 909092: {}, 916607: {}, 922033: {}, 923039: {}, 926333: {}, 926473: {}, 940457: {}, 940780: {}, 943392: {}, 951710: {}, 957253: {}, 961014: {}, 961118: {}, 965514: {}, 991782: {}, 997101: {}},
	}

	for i := range additions {
		for v, k := range additions[i] {
			b.RoaringSetRangeAdd(k, v)
		}
		for v := range deletions[i] {
			b.RoaringSetRangeRemove(0, v)
		}
		require.Nil(t, b.FlushAndSwitch())
	}

	control := make([]*sroar.Bitmap, maxKey)
	for i := range control {
		control[i] = sroar.NewBitmap()
	}
	for v, k := range map[uint64]uint64{103: 49, 104: 93, 147: 7, 250: 58, 519: 67, 556: 42, 582: 31, 598: 30, 626: 4, 653: 28, 891: 31, 1060: 88, 1065: 56, 1215: 97, 1260: 60, 1280: 83, 1304: 85, 1335: 43, 1346: 43, 1380: 93, 1528: 42, 1544: 65, 1713: 73, 1788: 48, 1875: 0, 1929: 87, 2033: 3, 2090: 78, 2092: 83, 2199: 23, 2257: 63, 2425: 58, 2457: 1, 2522: 5, 2553: 4, 2568: 92, 2596: 84, 2617: 81, 2826: 54, 2843: 7, 2902: 22, 3083: 77, 3121: 5, 3196: 2, 3202: 44, 3230: 15, 3246: 49, 3464: 66, 3492: 84, 3500: 50, 3631: 11, 3770: 61, 3773: 63, 3811: 65, 3817: 3, 3892: 48, 3908: 29, 3940: 9, 3945: 65, 4002: 79, 4022: 99, 4025: 96, 4026: 0, 4083: 54, 4229: 12, 4261: 13, 4282: 47, 4333: 95, 4463: 25, 4591: 18, 4632: 82, 4699: 88, 4731: 63, 4752: 76, 4765: 80, 4828: 59, 4907: 6, 4992: 16, 5106: 28, 5249: 46, 5368: 50, 5464: 9, 5466: 59, 5484: 20, 5489: 37, 5500: 25, 5555: 68, 5692: 36, 5767: 44, 5771: 34, 5791: 32, 5792: 91, 5850: 28, 5879: 14, 5975: 90, 5997: 44, 6009: 92, 6015: 55, 6045: 81, 6108: 74, 6158: 42, 6177: 63, 6206: 18, 6258: 98, 6276: 92, 6303: 77, 6497: 23, 6516: 34, 6587: 51, 6768: 7, 6810: 63, 6834: 53, 7014: 84, 7107: 61, 7201: 4, 7277: 72, 7307: 40, 7337: 54, 7370: 2, 7573: 18, 7700: 23, 7704: 34, 7821: 39, 7838: 47, 7873: 8, 7881: 78, 7971: 7, 8310: 26, 8505: 18, 8590: 46, 8635: 50, 8673: 79, 8728: 67, 8770: 66, 8799: 17, 8822: 73, 8841: 96, 8860: 38, 8987: 80, 9008: 45, 9029: 82, 9039: 99, 9179: 44, 9181: 8, 9286: 61, 9337: 27, 9419: 54, 9423: 40, 9577: 28, 9607: 66, 9905: 6, 9930: 40, 10067: 18, 10074: 19, 10130: 37, 10160: 3, 10211: 83, 10289: 62, 10292: 63, 10338: 6, 10393: 37, 10441: 77, 10475: 58, 10574: 65, 10613: 24, 10651: 80, 10705: 78, 10744: 86, 11161: 20, 11187: 11, 11196: 33, 11210: 2, 11243: 58, 11276: 99, 11392: 50, 11542: 14, 11663: 40, 11676: 13, 11680: 54, 11699: 49, 11825: 58, 11857: 95, 11893: 44, 11927: 60, 11934: 29, 11978: 39, 12165: 48, 12273: 7, 12327: 90, 12332: 44, 12357: 66, 12358: 76, 12367: 12, 12513: 18, 12523: 68, 12559: 81, 12614: 6, 12713: 1, 12741: 44, 12779: 40, 12860: 33, 12967: 69, 13175: 57, 13193: 60, 13209: 25, 13301: 2, 13510: 97, 13598: 88, 13631: 65, 13669: 12, 13714: 80, 13726: 39, 13843: 45, 13921: 91, 14105: 31, 14133: 26, 14225: 45, 14251: 53, 14744: 67, 14764: 41, 14863: 14, 14890: 58, 15020: 82, 15249: 41, 15262: 89, 15303: 8, 15427: 78, 15534: 65, 15667: 97, 15697: 88, 15746: 55, 15808: 25, 15857: 21, 15910: 71, 16113: 51, 16143: 39, 16189: 16, 16246: 49, 16258: 30, 16318: 32, 16427: 80, 16436: 43, 16440: 62, 16503: 77, 16549: 3, 16675: 56, 16710: 62, 16754: 35, 16859: 8, 16925: 18, 17023: 76, 17092: 52, 17139: 45, 17141: 38, 17146: 62, 17182: 13, 17232: 55, 17375: 64, 17389: 59, 17542: 22, 17617: 23, 17716: 77, 17751: 64, 17918: 34, 17977: 29, 18118: 94, 18155: 86, 18255: 29, 18344: 34, 18436: 40, 18441: 32, 18538: 17, 18685: 97, 18730: 90, 18735: 94, 18749: 44, 18789: 23, 18858: 50, 18884: 25, 18945: 3, 18958: 43, 19027: 22, 19032: 21, 19055: 36, 19095: 17, 19159: 73, 19226: 22, 19352: 70, 19373: 92, 19620: 83, 19650: 57, 19652: 43, 19739: 70, 19781: 24, 19805: 65, 19816: 1, 19823: 63, 19843: 64, 19941: 50, 19971: 66, 20147: 33, 20464: 25, 20465: 20, 20540: 94, 20557: 67, 20618: 81, 20641: 39, 20661: 4, 20665: 81, 20719: 97, 20769: 7, 20782: 50, 20803: 99, 20949: 41, 21028: 30, 21063: 45, 21072: 65, 21137: 26, 21255: 24, 21389: 58, 21423: 72, 21460: 68, 21477: 53, 21587: 77, 21606: 51, 21803: 35, 21817: 45, 21826: 82, 21865: 99, 21897: 80, 21921: 79, 21927: 60, 21990: 58, 22096: 68, 22107: 99, 22124: 39, 22143: 54, 22369: 49, 22412: 53, 22433: 67, 22531: 83, 22646: 14, 22649: 60, 22654: 31, 22665: 25, 22676: 95, 22677: 95, 22831: 63, 22837: 64, 22926: 63, 23008: 2, 23021: 86, 23041: 18, 23126: 75, 23281: 48, 23294: 51, 23336: 58, 23361: 87, 23393: 16, 23597: 48, 23678: 63, 23761: 85, 23778: 34, 23792: 91, 23972: 29, 24004: 31, 24046: 32, 24057: 61, 24069: 16, 24137: 86, 24171: 65, 24216: 27, 24337: 21, 24338: 26, 24342: 82, 24416: 65, 24427: 16, 24469: 17, 24474: 29, 24524: 18, 24614: 81, 24678: 69, 24696: 53, 24821: 45, 24908: 48, 24943: 10, 25044: 37, 25088: 25, 25093: 26, 25407: 35, 25567: 46, 25665: 37, 25694: 95, 25731: 92, 25824: 49, 25908: 66, 25955: 15, 26069: 73, 26078: 83, 26092: 52, 26099: 91, 26164: 54, 26216: 43, 26228: 73, 26243: 52, 26248: 12, 26294: 57, 26394: 40, 26464: 21, 26503: 89, 26615: 68, 27005: 57, 27011: 69, 27025: 73, 27032: 5, 27105: 77, 27122: 32, 27215: 39, 27233: 56, 27249: 52, 27275: 65, 27369: 59, 27627: 37, 27643: 29, 27645: 53, 27654: 54, 27840: 5, 27879: 73, 27941: 12, 27990: 66, 28000: 12, 28034: 63, 28059: 1, 28062: 99, 28215: 72, 28250: 64, 28422: 80, 28448: 7, 28535: 22, 28541: 64, 28576: 67, 28640: 17, 28840: 13, 28859: 48, 28883: 56, 28910: 3, 28915: 77, 28947: 55, 29048: 40, 29084: 16, 29183: 94, 29209: 93, 29220: 82, 29235: 80, 29261: 9, 29317: 35, 29382: 37, 29536: 82, 29628: 77, 29641: 5, 29695: 81, 30022: 35, 30093: 55, 30170: 71, 30202: 12, 30335: 26, 30385: 99, 30444: 27, 30539: 63, 30556: 84, 30572: 96, 30667: 96, 30717: 53, 30755: 82, 30818: 38, 30892: 46, 30927: 32, 30929: 56, 31043: 94, 31064: 77, 31123: 44, 31192: 85, 31299: 57, 31340: 97, 31407: 76, 31420: 19, 31432: 23, 31537: 39, 31748: 98, 31832: 64, 31844: 11, 31860: 62, 31893: 39, 31958: 11, 32052: 42, 32183: 6, 32185: 1, 32204: 81, 32285: 55, 32302: 57, 32397: 38, 32405: 14, 32408: 21, 32420: 47, 32421: 81, 32442: 97, 32461: 89, 32484: 85, 32551: 50, 32846: 6, 32950: 26, 33023: 2, 33080: 28, 33163: 87, 33292: 75, 33294: 97, 33303: 70, 33318: 37, 33387: 62, 33419: 67, 33471: 13, 33477: 26, 33517: 30, 33612: 13, 33758: 93, 33789: 40, 33873: 47, 33896: 92, 34048: 64, 34224: 10, 34230: 97, 34246: 36, 34359: 87, 34572: 57, 34629: 88, 34649: 98, 34810: 34, 34831: 86, 34850: 12, 34871: 3, 34921: 32, 34952: 14, 34987: 80, 35017: 98, 35026: 36, 35083: 57, 35091: 58, 35142: 87, 35208: 26, 35330: 32, 35503: 29, 35538: 97, 35625: 29, 35661: 8, 35732: 44, 35835: 77, 35910: 23, 35922: 8, 35995: 13, 36088: 85, 36098: 82, 36122: 24, 36148: 68, 36155: 42, 36259: 41, 36310: 36, 36381: 39, 36534: 25, 36909: 67, 36932: 24, 37061: 89, 37092: 36, 37255: 33, 37294: 31, 37354: 76, 37360: 61, 37398: 24, 37424: 17, 37586: 57, 37643: 31, 37660: 40, 37815: 61, 37841: 58, 38009: 93, 38047: 31, 38216: 53, 38219: 25, 38349: 91, 38356: 40, 38445: 82, 38505: 7, 38565: 39, 38614: 50, 38735: 56, 38739: 58, 38829: 3, 38833: 44, 38848: 40, 38849: 21, 38877: 8, 38909: 27, 38992: 24, 39015: 98, 39153: 78, 39258: 75, 39312: 87, 39387: 73, 39479: 94, 39564: 42, 39569: 31, 39606: 94, 39657: 22, 39714: 45, 39828: 75, 39895: 18, 39913: 45, 39961: 81, 40073: 93, 40135: 23, 40247: 87, 40404: 44, 40579: 8, 40590: 63, 40714: 43, 40727: 64, 40737: 54, 40778: 58, 40928: 18, 40970: 47, 40988: 69, 41005: 68, 41094: 45, 41132: 64, 41225: 60, 41308: 15, 41312: 39, 41433: 10, 41438: 53, 41477: 22, 41510: 94, 41520: 43, 41558: 5, 41592: 54, 41742: 44, 41750: 67, 41858: 78, 41893: 73, 41907: 12, 41961: 81, 42070: 28, 42230: 47, 42267: 88, 42272: 72, 42337: 76, 42375: 86, 42468: 18, 42488: 21, 42490: 39, 42525: 8, 42557: 6, 42581: 1, 42775: 84, 42879: 46, 42889: 74, 42952: 48, 43098: 85, 43115: 45, 43120: 25, 43146: 76, 43191: 25, 43222: 36, 43308: 92, 43313: 82, 43389: 15, 43433: 3, 43488: 90, 43510: 35, 43669: 21, 43747: 38, 43748: 32, 43760: 28, 43850: 63, 43886: 62, 43902: 3, 43925: 53, 44021: 74, 44035: 43, 44062: 66, 44073: 68, 44099: 80, 44205: 65, 44211: 38, 44218: 45, 44320: 17, 44348: 55, 44379: 64, 44410: 5, 44445: 94, 44460: 19, 44508: 68, 44511: 64, 44535: 12, 44549: 50, 44571: 93, 44632: 64, 44642: 59, 44643: 78, 44652: 62, 44704: 31, 44710: 92, 44714: 95, 44754: 98, 44778: 73, 44782: 66, 44828: 16, 44917: 6, 44952: 92, 45015: 83, 45066: 15, 45155: 22, 45216: 58, 45362: 54, 45427: 19, 45504: 52, 45639: 60, 45692: 92, 45728: 61, 45772: 4, 45777: 31, 45781: 31, 45822: 63, 45845: 94, 46066: 98, 46140: 34, 46151: 82, 46188: 79, 46234: 59, 46245: 3, 46292: 20, 46303: 0, 46326: 79, 46377: 47, 46465: 13, 46499: 58, 46510: 37, 46613: 83, 46737: 91, 46758: 71, 46800: 9, 46813: 25, 47012: 72, 47022: 18, 47043: 46, 47077: 81, 47080: 95, 47160: 69, 47280: 8, 47389: 9, 47412: 7, 47430: 44, 47431: 76, 47527: 95, 47633: 36, 47638: 81, 47770: 90, 47773: 36, 47827: 82, 47884: 19, 47907: 60, 47925: 71, 47933: 48, 47965: 62, 48005: 66, 48181: 47, 48226: 20, 48248: 21, 48286: 77, 48325: 37, 48446: 89, 48490: 32, 48527: 90, 48596: 43, 48752: 33, 48847: 79, 48899: 63, 48951: 48, 49005: 32, 49015: 95, 49098: 14, 49201: 24, 49329: 35, 49336: 32, 49355: 12, 49361: 79, 49419: 75, 49483: 26, 49758: 65, 49805: 7, 49818: 29, 49838: 90, 49865: 26, 49939: 24, 49942: 94, 50000: 89, 50033: 51, 50098: 40, 50124: 12, 50131: 91, 50202: 83, 50321: 21, 50501: 88, 50607: 82, 50618: 87, 50630: 32, 50659: 3, 50680: 41, 50721: 17, 50806: 47, 50829: 67, 50866: 68, 50880: 24, 50943: 96, 50947: 29, 51031: 6, 51067: 44, 51143: 4, 51204: 13, 51227: 74, 51314: 4, 51340: 17, 51344: 48, 51351: 66, 51373: 25, 51406: 89, 51437: 91, 51487: 15, 51556: 59, 51578: 43, 51725: 6, 51807: 73, 51845: 65, 51853: 84, 51862: 35, 52040: 84, 52055: 35, 52078: 49, 52116: 46, 52143: 90, 52623: 26, 52673: 25, 52697: 10, 52712: 51, 52723: 32, 52733: 59, 52798: 64, 52828: 14, 52845: 57, 52879: 29, 52888: 69, 52940: 45, 52986: 52, 53108: 41, 53136: 32, 53156: 41, 53375: 50, 53476: 7, 53503: 37, 53513: 30, 53593: 7, 53627: 90, 53634: 7, 53645: 96, 53769: 77, 53816: 78, 53903: 59, 53967: 78, 54054: 29, 54093: 53, 54234: 67, 54255: 57, 54336: 40, 54419: 5, 54475: 58, 54587: 9, 54592: 33, 54670: 96, 54766: 21, 54849: 12, 54867: 78, 54997: 9, 55105: 54, 55139: 89, 55169: 12, 55206: 22, 55230: 91, 55284: 58, 55295: 40, 55331: 74, 55367: 52, 55669: 84, 55723: 3, 55731: 5, 55884: 70, 55934: 11, 56007: 52, 56076: 13, 56192: 52, 56247: 67, 56400: 39, 56454: 37, 56587: 33, 56602: 68, 56662: 8, 56787: 46, 56906: 71, 56965: 80, 57151: 94, 57241: 0, 57331: 11, 57332: 28, 57337: 69, 57351: 73, 57361: 5, 57402: 78, 57452: 24, 57466: 91, 57477: 48, 57492: 11, 57501: 26, 57570: 70, 57587: 84, 57607: 6, 57666: 40, 57841: 82, 57871: 66, 57926: 66, 57928: 35, 57939: 9, 57956: 80, 58160: 54, 58162: 89, 58179: 98, 58327: 17, 58337: 84, 58388: 13, 58406: 33, 58495: 81, 58643: 76, 58669: 56, 58779: 32, 58798: 29, 58823: 9, 58854: 92, 58871: 27, 58923: 37, 58945: 99, 58971: 52, 59184: 20, 59261: 92, 59288: 49, 59397: 99, 59448: 39, 59614: 42, 59624: 60, 59643: 71, 59647: 44, 59690: 30, 59713: 39, 59717: 49, 59728: 39, 59827: 87, 59919: 19, 60045: 3, 60117: 84, 60212: 38, 60386: 40, 60428: 63, 60431: 83, 60435: 86, 60441: 48, 60448: 2, 60450: 92, 60532: 92, 60577: 23, 60743: 91, 60836: 79, 60843: 11, 60870: 10, 60876: 18, 60897: 89, 60908: 5, 60926: 57, 61016: 58, 61063: 48, 61140: 92, 61230: 4, 61358: 22, 61370: 31, 61393: 38, 61464: 58, 61621: 19, 61752: 63, 61771: 82, 61789: 70, 62036: 32, 62140: 63, 62220: 70, 62266: 16, 62352: 45, 62360: 71, 62365: 18, 62430: 78, 62478: 52, 62540: 53, 62644: 54, 62666: 29, 62684: 64, 62778: 60, 62872: 89, 63166: 29, 63183: 71, 63196: 69, 63264: 99, 63297: 35, 63382: 10, 63481: 5, 63543: 87, 63548: 29, 63625: 99, 63668: 19, 63762: 59, 63764: 51, 63794: 97, 63860: 52, 63878: 54, 63948: 46, 63986: 59, 63990: 4, 64023: 7, 64309: 20, 64449: 64, 64500: 39, 64678: 41, 64693: 40, 64755: 86, 64908: 25, 64944: 45, 65046: 89, 65112: 86, 65132: 22, 65165: 0, 65172: 30, 65199: 18, 65267: 14, 65279: 88, 65327: 61, 65404: 25, 65417: 96, 65470: 63, 65684: 11, 65695: 48, 65807: 70, 65910: 43, 65935: 26, 65936: 73, 65982: 76, 66070: 1, 66198: 11, 66314: 3, 66349: 64, 66391: 19, 66426: 53, 66546: 4, 66742: 96, 66936: 83, 66975: 69, 66986: 78, 67052: 19, 67099: 86, 67152: 86, 67160: 85, 67178: 80, 67347: 38, 67396: 69, 67446: 13, 67542: 53, 67559: 9, 67560: 7, 67586: 18, 67596: 49, 67599: 85, 67617: 76, 67798: 52, 67804: 60, 67916: 36, 67977: 78, 68017: 58, 68192: 26, 68260: 29, 68359: 26, 68435: 43, 68533: 86, 68609: 11, 68719: 93, 68753: 28, 68841: 46, 68910: 94, 69056: 17, 69076: 92, 69172: 27, 69268: 97, 69322: 15, 69366: 18, 69376: 80, 69486: 5, 69572: 19, 69640: 84, 69645: 59, 69670: 76, 69770: 64, 69847: 66, 69888: 64, 69983: 73, 70043: 47, 70064: 11, 70124: 17, 70175: 83, 70338: 11, 70354: 53, 70384: 90, 70401: 69, 70449: 78, 70535: 48, 70547: 57, 70579: 73, 70654: 96, 70850: 62, 70855: 61, 70977: 64, 71090: 53, 71096: 52, 71166: 42, 71246: 46, 71356: 98, 71446: 28, 71489: 15, 71571: 84, 71633: 22, 71781: 29, 71863: 95, 71883: 42, 71895: 71, 71915: 24, 71983: 22, 71995: 14, 72041: 0, 72071: 51, 72118: 24, 72235: 30, 72267: 56, 72285: 87, 72523: 13, 72563: 1, 72785: 78, 72827: 16, 72862: 14, 72869: 87, 72877: 2, 73019: 26, 73058: 91, 73060: 31, 73061: 54, 73108: 40, 73283: 93, 73312: 87, 73375: 13, 73426: 8, 73437: 17, 73443: 14, 73468: 63, 73583: 62, 73617: 59, 73638: 7, 73705: 55, 73719: 53, 73765: 84, 73802: 56, 73869: 73, 73870: 98, 73929: 87, 73996: 16, 74009: 70, 74039: 58, 74048: 21, 74091: 54, 74100: 75, 74134: 5, 74203: 89, 74270: 50, 74550: 8, 74616: 27, 74839: 93, 74872: 59, 74900: 53, 74926: 16, 75007: 15, 75073: 25, 75077: 6, 75337: 2, 75351: 11, 75397: 14, 75413: 44, 75442: 76, 75539: 37, 75645: 15, 75771: 45, 75818: 45, 75987: 28, 76035: 64, 76038: 72, 76087: 6, 76108: 88, 76111: 86, 76120: 92, 76251: 42, 76279: 77, 76320: 8, 76412: 18, 76443: 99, 76551: 24, 76572: 79, 76592: 8, 76607: 31, 76623: 69, 76641: 4, 76661: 25, 76679: 1, 76712: 20, 76830: 17, 76840: 50, 76935: 34, 77114: 9, 77158: 12, 77177: 35, 77481: 62, 77616: 60, 77650: 4, 77682: 13, 77794: 27, 77830: 31, 77912: 54, 77975: 38, 77977: 40, 78053: 80, 78076: 85, 78107: 0, 78120: 14, 78135: 85, 78161: 70, 78246: 60, 78328: 58, 78382: 57, 78405: 94, 78423: 64, 78424: 47, 78468: 22, 78597: 36, 78656: 54, 78670: 63, 78792: 94, 78941: 45, 78998: 65, 79017: 7, 79052: 96, 79075: 87, 79147: 13, 79289: 20, 79312: 93, 79540: 82, 79632: 11, 79655: 33, 79750: 80, 79780: 99, 79852: 37, 79871: 46, 79919: 16, 79943: 38, 79954: 58, 79982: 64, 80134: 53, 80172: 2, 80196: 55, 80203: 1, 80254: 67, 80346: 16, 80372: 29, 80571: 63, 80654: 33, 80667: 89, 80669: 35, 80684: 61, 80711: 46, 80850: 69, 80867: 72, 80916: 98, 80951: 83, 81045: 91, 81086: 77, 81148: 41, 81233: 61, 81270: 6, 81307: 88, 81308: 58, 81340: 40, 81444: 38, 81488: 87, 81534: 40, 81610: 28, 81732: 24, 81758: 72, 81779: 12, 81825: 84, 81929: 6, 81972: 14, 82403: 81, 82450: 25, 82466: 0, 82534: 78, 82604: 0, 82657: 83, 82734: 86, 82753: 96, 82766: 68, 82844: 82, 82854: 27, 82904: 19, 82917: 33, 83003: 28, 83168: 76, 83395: 82, 83400: 91, 83532: 55, 83556: 81, 83682: 90, 83696: 63, 83744: 4, 83994: 49, 84166: 8, 84205: 39, 84223: 49, 84267: 87, 84395: 89, 84443: 90, 84529: 45, 84602: 7, 84646: 8, 84717: 42, 84770: 77, 84773: 6, 84775: 71, 84822: 62, 84844: 60, 84878: 15, 85008: 47, 85135: 35, 85206: 25, 85213: 3, 85266: 48, 85464: 32, 85489: 83, 85501: 10, 85520: 90, 85557: 25, 85592: 75, 85624: 73, 85662: 65, 85691: 89, 85723: 2, 85803: 86, 85831: 61, 85918: 52, 85954: 92, 86061: 74, 86107: 62, 86128: 86, 86211: 83, 86218: 80, 86277: 10, 86292: 72, 86296: 15, 86323: 56, 86324: 79, 86361: 47, 86396: 57, 86532: 82, 86578: 88, 86582: 79, 86663: 8, 86691: 8, 86718: 13, 86789: 59, 86916: 11, 86947: 21, 86979: 44, 86983: 42, 87090: 88, 87229: 0, 87289: 9, 87330: 24, 87434: 52, 87523: 89, 87551: 48, 87554: 52, 87682: 83, 87764: 39, 87906: 52, 87931: 53, 88000: 35, 88128: 32, 88130: 22, 88152: 22, 88153: 92, 88172: 12, 88181: 87, 88254: 58, 88293: 13, 88457: 91, 88560: 68, 88702: 74, 88730: 58, 89029: 31, 89060: 29, 89178: 53, 89344: 86, 89394: 99, 89401: 29, 89404: 34, 89618: 23, 89737: 26, 89812: 70, 89937: 45, 89986: 84, 90040: 25, 90060: 74, 90123: 11, 90181: 81, 90189: 78, 90195: 45, 90216: 38, 90228: 74, 90419: 58, 90492: 34, 90540: 61, 90566: 5, 90746: 15, 90794: 64, 90815: 66, 90868: 27, 90884: 18, 90976: 34, 91294: 14, 91304: 9, 91394: 60, 91459: 17, 91482: 14, 91494: 61, 91583: 40, 91755: 57, 91768: 74, 91824: 7, 91828: 63, 91853: 21, 91921: 18, 91939: 13, 91941: 66, 92032: 78, 92074: 70, 92086: 85, 92131: 48, 92340: 91, 92416: 11, 92467: 41, 92519: 90, 92677: 8, 92753: 29, 92773: 2, 92783: 57, 92814: 87, 92833: 34, 92904: 78, 92920: 99, 93121: 3, 93167: 65, 93281: 76, 93296: 85, 93300: 38, 93301: 61, 93356: 59, 93393: 39, 93740: 8, 93801: 69, 93826: 29, 93993: 8, 94193: 64, 94228: 81, 94325: 57, 94431: 9, 94481: 54, 94528: 90, 94620: 58, 94628: 49, 94776: 11, 94802: 51, 94889: 93, 94951: 36, 94969: 46, 95017: 99, 95138: 48, 95169: 42, 95256: 20, 95314: 60, 95442: 85, 95524: 70, 95555: 57, 95577: 99, 95669: 48, 95712: 77, 95998: 3, 96128: 13, 96196: 2, 96274: 20, 96305: 46, 96371: 47, 96515: 99, 96580: 84, 96763: 50, 96813: 69, 96827: 33, 96890: 24, 96947: 90, 96964: 74, 97002: 41, 97010: 22, 97022: 8, 97025: 17, 97044: 68, 97085: 57, 97094: 84, 97119: 23, 97141: 66, 97168: 89, 97242: 79, 97278: 16, 97302: 88, 97473: 85, 97529: 18, 97600: 56, 97641: 37, 97684: 46, 97727: 53, 97734: 91, 97757: 18, 97763: 75, 97776: 88, 97831: 19, 97835: 18, 97883: 17, 97918: 2, 97924: 66, 97925: 5, 98012: 35, 98109: 60, 98126: 4, 98156: 38, 98172: 69, 98202: 57, 98281: 15, 98368: 66, 98375: 89, 98398: 9, 98401: 66, 98469: 28, 98585: 12, 98660: 44, 98664: 48, 98684: 50, 98693: 68, 98778: 81, 98787: 43, 98805: 93, 98966: 64, 99070: 52, 99110: 90, 99166: 53, 99216: 15, 99226: 79, 99251: 83, 99350: 36, 99397: 23, 99411: 55, 99438: 5, 99439: 52, 99508: 87, 99563: 2, 99605: 23, 99622: 63, 99642: 28, 99667: 30, 99722: 32, 99810: 9, 99935: 46, 99942: 41, 100031: 93, 100051: 10, 100058: 2, 100112: 71, 100346: 73, 100424: 13, 100433: 8, 100438: 53, 100506: 27, 100631: 42, 100760: 71, 100852: 92, 100996: 89, 101078: 15, 101090: 69, 101277: 44, 101308: 95, 101372: 27, 101397: 96, 101582: 48, 101854: 20, 101886: 5, 101999: 0, 102067: 60, 102154: 70, 102323: 38, 102358: 80, 102447: 57, 102457: 41, 102458: 6, 102548: 63, 102556: 88, 102567: 40, 102572: 12, 102688: 57, 102706: 58, 102762: 41, 102766: 57, 102862: 55, 102873: 48, 102882: 63, 102908: 85, 102920: 22, 102995: 91, 103050: 74, 103056: 23, 103076: 72, 103103: 67, 103320: 66, 103377: 67, 103382: 54, 103603: 57, 103692: 65, 103735: 38, 103769: 46, 103791: 32, 103911: 79, 103924: 41, 103999: 19, 104083: 74, 104234: 42, 104331: 9, 104568: 66, 104775: 16, 104840: 30, 105041: 52, 105122: 71, 105193: 90, 105241: 86, 105242: 68, 105506: 92, 105528: 10, 105717: 27, 105795: 20, 105925: 90, 105939: 99, 105952: 75, 106045: 22, 106273: 7, 106330: 50, 106558: 38, 106583: 10, 106658: 22, 106691: 58, 106731: 39, 106743: 54, 106996: 27, 107041: 70, 107075: 46, 107103: 66, 107122: 7, 107130: 13, 107152: 2, 107330: 40, 107402: 37, 107409: 39, 107431: 49, 107440: 97, 107772: 65, 107863: 61, 107912: 59, 107978: 82, 108022: 65, 108035: 84, 108097: 15, 108150: 73, 108193: 78, 108204: 62, 108287: 40, 108334: 37, 108338: 65, 108388: 80, 108439: 42, 108449: 99, 108590: 15, 108635: 49, 108664: 59, 108757: 84, 108843: 3, 108966: 5, 109045: 20, 109066: 71, 109098: 73, 109166: 43, 109185: 91, 109199: 81, 109209: 26, 109272: 80, 109458: 73, 109546: 79, 109552: 13, 109567: 43, 109654: 63, 109777: 60, 109813: 11, 109858: 51, 109919: 94, 109983: 75, 109996: 37, 110000: 86, 110009: 9, 110037: 73, 110071: 7, 110149: 99, 110322: 30, 110373: 4, 110374: 16, 110404: 96, 110626: 32, 110713: 23, 110726: 21, 110744: 58, 110921: 48, 110972: 54, 111124: 10, 111143: 18, 111177: 79, 111249: 12, 111299: 17, 111307: 77, 111365: 64, 111384: 45, 111412: 26, 111426: 86, 111436: 52, 111458: 5, 111572: 75, 111615: 69, 111724: 66, 111733: 90, 111972: 60, 112046: 32, 112107: 56, 112271: 27, 112289: 91, 112341: 88, 112366: 71, 112393: 45, 112412: 42, 112474: 50, 112475: 2, 112580: 67, 112836: 74, 112876: 22, 112912: 23, 112979: 70, 113014: 65, 113047: 76, 113090: 42, 113155: 85, 113227: 48, 113272: 51, 113543: 43, 113583: 15, 113689: 18, 113783: 9, 113890: 5, 113926: 31, 113989: 62, 114080: 14, 114111: 72, 114144: 49, 114151: 63, 114223: 58, 114256: 30, 114337: 93, 114372: 1, 114506: 79, 114519: 72, 114594: 34, 114598: 2, 114630: 1, 114648: 1, 114684: 64, 114726: 96, 114913: 72, 115064: 94, 115080: 61, 115120: 68, 115164: 36, 115167: 62, 115283: 15, 115349: 31, 115370: 28, 115373: 78, 115409: 5, 115509: 61, 115562: 82, 115645: 31, 115653: 74, 115707: 2, 116004: 92, 116062: 41, 116196: 1, 116332: 56, 116513: 44, 116592: 4, 116612: 83, 116694: 81, 116731: 0, 116776: 56, 116808: 13, 116842: 17, 116859: 98, 116862: 50, 116934: 20, 117022: 58, 117080: 43, 117110: 62, 117154: 89, 117197: 27, 117224: 14, 117271: 14, 117347: 4, 117386: 66, 117398: 41, 117441: 66, 117448: 40, 117489: 31, 117522: 67, 117661: 5, 117685: 9, 117750: 10, 117775: 38, 117828: 34, 117886: 5, 117997: 31, 118122: 28, 118125: 73, 118174: 22, 118204: 32, 118269: 75, 118279: 14, 118350: 29, 118436: 18, 118465: 85, 118792: 96, 118800: 79, 118844: 24, 118847: 75, 119010: 65, 119027: 51, 119180: 34, 119378: 26, 119469: 51, 119512: 90, 119597: 51, 119678: 27, 119682: 52, 119733: 67, 119814: 52, 119815: 57, 119836: 65, 119873: 85, 119924: 59, 119938: 85, 119968: 20, 120060: 10, 120127: 19, 120194: 23, 120216: 85, 120284: 56, 120387: 96, 120467: 61, 120546: 67, 120568: 17, 120614: 19, 120633: 42, 120795: 3, 120843: 6, 120846: 0, 120852: 11, 120901: 80, 120913: 59, 120946: 56, 120958: 85, 120977: 4, 121005: 83, 121069: 41, 121077: 5, 121176: 57, 121213: 48, 121278: 30, 121302: 24, 121316: 37, 121324: 20, 121363: 25, 121432: 16, 121494: 17, 121496: 53, 121542: 81, 121603: 7, 121610: 99, 121616: 99, 121637: 24, 121670: 65, 121692: 31, 121753: 16, 121789: 90, 121806: 9, 121811: 84, 121952: 7, 122125: 41, 122134: 88, 122212: 47, 122238: 25, 122259: 62, 122346: 46, 122549: 32, 122577: 58, 122676: 65, 122769: 79, 122808: 88, 122903: 28, 122925: 17, 122926: 59, 122935: 23, 122959: 91, 123148: 80, 123172: 89, 123184: 93, 123319: 35, 123322: 22, 123459: 2, 123492: 32, 123578: 13, 123779: 21, 123819: 65, 123849: 2, 124028: 4, 124075: 92, 124134: 64, 124225: 69, 124351: 93, 124357: 65, 124406: 24, 124413: 60, 124555: 60, 124609: 9, 124674: 84, 124709: 46, 124723: 64, 124824: 53, 124929: 96, 124945: 39, 124964: 85, 125272: 72, 125289: 32, 125306: 96, 125422: 51, 125438: 50, 125439: 30, 125459: 97, 125637: 48, 125770: 78, 125773: 1, 125860: 14, 125877: 66, 125925: 70, 125933: 15, 125995: 46, 126022: 29, 126030: 74, 126087: 60, 126151: 19, 126161: 23, 126194: 70, 126355: 46, 126372: 38, 126383: 7, 126452: 34, 126524: 66, 126531: 8, 126626: 75, 126662: 56, 126730: 42, 126743: 96, 126806: 1, 126998: 46, 127003: 12, 127203: 84, 127533: 35, 127672: 10, 127721: 84, 127814: 72, 127860: 80, 127878: 82, 127967: 39, 128112: 84, 128215: 15, 128245: 35, 128399: 2, 128434: 4, 128509: 73, 128529: 69, 128846: 1, 128936: 80, 128969: 47, 128984: 48, 129277: 66, 129340: 58, 129413: 12, 129625: 65, 129760: 91, 129800: 43, 129813: 28, 129897: 18, 129980: 58, 130002: 20, 130012: 5, 130058: 44, 130237: 0, 130400: 45, 130438: 95, 130477: 18, 130522: 84, 130594: 68, 130694: 1, 130723: 33, 130734: 6, 130745: 2, 130765: 19, 130964: 51, 131020: 73, 131046: 18, 131232: 83, 131239: 73, 131241: 72, 131298: 9, 131505: 48, 131514: 22, 131516: 18, 131564: 55, 131588: 46, 131589: 84, 131634: 35, 131788: 80, 131975: 86, 131986: 41, 131998: 33, 132057: 4, 132112: 75, 132114: 52, 132123: 1, 132183: 51, 132232: 89, 132257: 70, 132384: 24, 132403: 28, 132530: 89, 132562: 38, 132567: 68, 132668: 30, 132679: 94, 132680: 4, 132817: 65, 132843: 55, 132865: 90, 132893: 82, 133085: 23, 133101: 66, 133124: 10, 133184: 85, 133200: 89, 133243: 15, 133247: 70, 133285: 85, 133323: 43, 133364: 10, 133380: 93, 133395: 75, 133437: 2, 133554: 85, 133645: 37, 133759: 30, 133793: 46, 133803: 62, 133813: 30, 133830: 72, 133868: 80, 133924: 60, 133957: 32, 134027: 18, 134095: 52, 134100: 92, 134265: 37, 134304: 54, 134313: 34, 134512: 62, 134597: 51, 134651: 40, 134684: 74, 134728: 99, 134735: 67, 134753: 42, 135053: 84, 135055: 67, 135084: 66, 135160: 56, 135242: 65, 135290: 24, 135291: 43, 135356: 55, 135509: 30, 135522: 87, 135536: 29, 135671: 41, 135727: 88, 135822: 9, 135920: 96, 135984: 41, 136027: 56, 136111: 3, 136137: 92, 136144: 77, 136187: 0, 136445: 49, 136473: 69, 136496: 52, 136647: 52, 136660: 12, 136762: 77, 136789: 20, 136805: 96, 136979: 0, 137013: 89, 137036: 17, 137048: 74, 137141: 97, 137145: 4, 137195: 5, 137208: 38, 137229: 39, 137241: 65, 137340: 86, 137511: 55, 137566: 57, 137639: 13, 137806: 93, 137856: 51, 137917: 1, 137959: 53, 138009: 4, 138055: 8, 138191: 85, 138214: 43, 138286: 35, 138360: 11, 138452: 93, 138457: 43, 138480: 77, 138574: 19, 138680: 66, 138714: 55, 138784: 57, 138794: 86, 138885: 30, 138896: 3, 138987: 42, 139415: 73, 139432: 57, 139464: 39, 139522: 8, 139589: 73, 139637: 54, 139737: 68, 139756: 36, 139786: 86, 139799: 23, 139825: 42, 139868: 94, 139901: 52, 139949: 15, 140035: 49, 140118: 13, 140546: 0, 140589: 64, 140689: 25, 140703: 72, 140707: 26, 140929: 78, 141056: 47, 141057: 83, 141103: 1, 141196: 6, 141218: 61, 141330: 7, 141348: 21, 141425: 92, 141433: 55, 141451: 12, 141590: 73, 141703: 89, 141709: 31, 141737: 1, 141754: 92, 141771: 72, 141856: 6, 141896: 0, 141946: 95, 142100: 28, 142110: 9, 142152: 45, 142228: 58, 142316: 16, 142319: 74, 142332: 0, 142343: 26, 142411: 31, 142433: 71, 142454: 17, 142494: 7, 142632: 51, 142669: 22, 142682: 95, 142750: 14, 142787: 52, 142934: 80, 142969: 98, 143121: 71, 143202: 20, 143362: 11, 143428: 20, 143514: 32, 143582: 15, 143612: 2, 143614: 80, 143681: 21, 143782: 82, 143802: 81, 143901: 40, 143934: 46, 144018: 58, 144022: 74, 144056: 17, 144081: 57, 144101: 41, 144111: 15, 144124: 47, 144170: 32, 144203: 76, 144353: 74, 144362: 86, 144393: 17, 144774: 95, 144813: 12, 144866: 33, 144886: 53, 144896: 22, 144988: 33, 145000: 90, 145080: 30, 145218: 52, 145230: 30, 145235: 79, 145241: 91, 145416: 64, 145419: 22, 145495: 72, 145536: 10, 145542: 86, 145620: 45, 145624: 43, 145639: 59, 145705: 35, 145783: 22, 145790: 85, 145844: 88, 145847: 75, 145920: 86, 145930: 35, 145970: 46, 145982: 0, 145999: 57, 146185: 28, 146215: 26, 146361: 79, 146681: 16, 146854: 11, 146898: 59, 147062: 46, 147081: 0, 147100: 0, 147151: 62, 147157: 28, 147312: 9, 147327: 86, 147496: 91, 147504: 30, 147526: 44, 147563: 61, 147573: 56, 147670: 80, 147726: 19, 147765: 96, 147895: 1, 147922: 12, 147987: 21, 148036: 46, 148113: 84, 148114: 14, 148285: 74, 148461: 32, 148535: 69, 148542: 85, 148553: 51, 148605: 8, 148659: 32, 148759: 37, 148762: 96, 148801: 76, 148831: 66, 148837: 59, 148909: 58, 148922: 86, 148967: 94, 149051: 24, 149184: 40, 149198: 34, 149204: 81, 149240: 18, 149329: 32, 149448: 47, 149471: 62, 149502: 4, 149643: 96, 149693: 85, 149789: 39, 149794: 15, 149819: 52, 149892: 6, 149972: 78, 150008: 68, 150022: 64, 150061: 75, 150083: 28, 150147: 2, 150173: 33, 150289: 41, 150596: 95, 150666: 65, 150674: 27, 150797: 36, 150896: 95, 150902: 9, 150994: 68, 151005: 12, 151053: 40, 151095: 75, 151180: 65, 151292: 60, 151324: 2, 151342: 49, 151344: 29, 151437: 71, 151535: 82, 151627: 0, 151659: 88, 151752: 14, 151865: 49, 151896: 14, 151968: 29, 152096: 28, 152111: 14, 152123: 12, 152150: 16, 152238: 58, 152348: 7, 152366: 39, 152613: 41, 152618: 22, 152674: 92, 152700: 58, 152834: 36, 152878: 30, 152935: 80, 153016: 92, 153043: 38, 153209: 23, 153250: 69, 153275: 44, 153464: 53, 153467: 98, 153551: 69, 153576: 36, 153579: 32, 153586: 78, 153587: 97, 153651: 11, 153768: 87, 153795: 96, 153901: 33, 153959: 12, 153973: 37, 154068: 50, 154170: 22, 154264: 23, 154340: 83, 154456: 53, 154508: 68, 154563: 77, 154631: 14, 154657: 73, 154843: 34, 154941: 33, 154954: 64, 154963: 61, 154996: 55, 155007: 7, 155105: 63, 155150: 9, 155238: 66, 155306: 13, 155335: 20, 155343: 51, 155344: 70, 155399: 85, 155431: 59, 155581: 83, 155799: 25, 155920: 84, 155939: 20, 155950: 84, 155962: 65, 156056: 6, 156172: 61, 156272: 13, 156362: 92, 156379: 78, 156522: 35, 156572: 26, 156601: 81, 156605: 67, 156667: 79, 156669: 41, 156735: 16, 156751: 23, 156829: 46, 157064: 36, 157097: 21, 157165: 2, 157193: 68, 157254: 38, 157316: 61, 157317: 45, 157377: 89, 157396: 30, 157410: 47, 157418: 47, 157621: 7, 157639: 66, 157651: 24, 157760: 77, 157798: 84, 157804: 31, 157825: 0, 157984: 71, 158066: 37, 158144: 59, 158160: 55, 158222: 99, 158380: 48, 158434: 16, 158576: 46, 158614: 82, 158739: 75, 158741: 28, 158922: 69, 158963: 1, 159003: 38, 159047: 91, 159068: 74, 159114: 48, 159123: 6, 159154: 31, 159220: 96, 159372: 71, 159374: 71, 159431: 97, 159571: 50, 159597: 72, 159702: 85, 159735: 66, 159738: 91, 159903: 49, 159943: 39, 160124: 64, 160136: 46, 160158: 35, 160247: 10, 160257: 4, 160326: 73, 160410: 35, 160538: 34, 160573: 42, 160598: 82, 160628: 55, 160644: 22, 160728: 51, 160761: 14, 160831: 63, 160894: 63, 160941: 1, 160966: 96, 160967: 74, 161012: 66, 161048: 66, 161061: 86, 161065: 18, 161090: 65, 161321: 48, 161594: 91, 161938: 20, 161949: 88, 161976: 92, 161981: 20, 162031: 33, 162087: 32, 162105: 0, 162331: 88, 162502: 85, 162653: 0, 162746: 81, 162785: 97, 162881: 70, 162887: 38, 163006: 34, 163035: 90, 163061: 32, 163083: 8, 163404: 18, 163482: 85, 163562: 16, 163613: 79, 163625: 69, 163678: 59, 163679: 92, 163686: 31, 163695: 97, 163752: 59, 164011: 82, 164175: 92, 164205: 64, 164224: 96, 164235: 68, 164276: 20, 164279: 33, 164414: 24, 164433: 52, 164505: 31, 164507: 35, 164551: 8, 164577: 91, 164605: 71, 164639: 37, 164685: 7, 164764: 61, 164777: 51, 164778: 55, 164865: 33, 164899: 47, 165002: 83, 165114: 97, 165206: 13, 165361: 25, 165372: 47, 165496: 52, 165509: 41, 165610: 72, 165618: 45, 165688: 71, 165892: 27, 165967: 65, 165969: 39, 166001: 27, 166053: 84, 166070: 82, 166166: 50, 166174: 35, 166207: 38, 166218: 49, 166246: 35, 166247: 11, 166254: 57, 166284: 2, 166427: 73, 166442: 76, 166483: 21, 166566: 81, 166823: 23, 166829: 11, 166877: 72, 166894: 59, 166913: 74, 167170: 12, 167219: 7, 167310: 76, 167366: 58, 167370: 77, 167449: 65, 167478: 74, 167487: 79, 167550: 79, 167555: 14, 167768: 44, 167823: 42, 167920: 77, 167973: 82, 168107: 22, 168270: 58, 168352: 83, 168465: 44, 168530: 69, 168550: 11, 168589: 39, 168626: 65, 168631: 3, 168636: 23, 168721: 19, 168872: 33, 168903: 5, 168944: 48, 168945: 72, 169047: 52, 169144: 1, 169194: 97, 169202: 18, 169315: 73, 169362: 26, 169548: 33, 169617: 45, 169648: 52, 169686: 5, 169754: 78, 169756: 87, 169814: 68, 169820: 42, 169905: 59, 170018: 26, 170045: 40, 170113: 1, 170269: 19, 170359: 60, 170410: 24, 170412: 26, 170511: 52, 170519: 51, 170715: 83, 170761: 69, 170819: 36, 170850: 37, 170974: 14, 170977: 74, 170989: 77, 171007: 13, 171132: 74, 171289: 36, 171316: 24, 171358: 27, 171409: 86, 171465: 43, 171601: 17, 171619: 12, 171661: 83, 171701: 76, 171703: 4, 171816: 83, 172046: 28, 172186: 89, 172211: 3, 172227: 63, 172290: 52, 172412: 7, 172444: 59, 172447: 49, 172454: 13, 172472: 51, 172496: 65, 172584: 35, 172594: 72, 172627: 27, 172675: 86, 172695: 78, 172716: 79, 172719: 59, 172727: 0, 172872: 76, 172936: 26, 173001: 38, 173044: 95, 173057: 59, 173079: 81, 173141: 78, 173354: 46, 173458: 55, 173550: 80, 173721: 39, 173760: 40, 173796: 5, 173803: 95, 173961: 13, 173970: 6, 173983: 71, 173991: 47, 174002: 34, 174003: 12, 174031: 52, 174045: 60, 174061: 12, 174229: 10, 174231: 19, 174289: 81, 174291: 68, 174412: 6, 174463: 52, 174599: 9, 174727: 95, 174830: 70, 174960: 15, 175064: 56, 175072: 33, 175151: 56, 175234: 28, 175243: 30, 175252: 13, 175303: 40, 175384: 70, 175531: 89, 175775: 61, 175821: 44, 175825: 31, 175852: 66, 175938: 32, 176039: 96, 176149: 25, 176274: 55, 176286: 81, 176404: 83, 176413: 36, 176420: 76, 176461: 28, 176517: 36, 176543: 44, 176549: 13, 176558: 50, 176614: 79, 176629: 28, 176757: 14, 176780: 52, 176818: 15, 176888: 73, 176894: 31, 176899: 73, 176974: 11, 177008: 52, 177043: 65, 177262: 95, 177342: 26, 177352: 41, 177358: 9, 177359: 75, 177388: 37, 177496: 57, 177546: 88, 177578: 23, 177617: 64, 177682: 76, 177727: 17, 177797: 20, 177906: 8, 177909: 34, 178044: 35, 178130: 37, 178173: 29, 178234: 68, 178237: 34, 178378: 91, 178432: 72, 178460: 77, 178481: 43, 178539: 73, 178705: 88, 178783: 52, 178933: 23, 178940: 16, 178941: 49, 178955: 56, 179004: 34, 179019: 98, 179026: 4, 179161: 44, 179190: 17, 179209: 84, 179238: 40, 179298: 70, 179300: 50, 179311: 55, 179391: 74, 179498: 47, 179572: 47, 179579: 18, 179591: 32, 179595: 96, 179703: 32, 179737: 74, 179873: 22, 179885: 19, 179913: 72, 179933: 52, 179957: 82, 179982: 52, 180042: 20, 180090: 6, 180092: 91, 180236: 0, 180301: 65, 180354: 43, 180357: 22, 180360: 2, 180380: 73, 180457: 27, 180466: 95, 180739: 35, 180780: 20, 180820: 48, 180827: 37, 180855: 37, 180859: 99, 180894: 27, 180896: 6, 180973: 33, 181144: 37, 181195: 80, 181298: 3, 181302: 66, 181316: 41, 181405: 63, 181424: 10, 181706: 13, 181890: 42, 181934: 25, 181998: 53, 182043: 46, 182054: 87, 182220: 22, 182312: 3, 182332: 46, 182376: 91, 182498: 45, 182536: 55, 182562: 7, 182604: 99, 182655: 87, 182698: 12, 182702: 34, 182985: 80, 183160: 75, 183217: 20, 183228: 68, 183370: 55, 183382: 19, 183386: 85, 183513: 76, 183789: 86, 183970: 3, 183971: 77, 184103: 66, 184167: 44, 184206: 28, 184226: 55, 184327: 28, 184346: 64, 184389: 41, 184407: 94, 184415: 16, 184436: 95, 184541: 99, 184543: 45, 184548: 43, 184790: 10, 184825: 34, 184834: 95, 184855: 41, 184875: 89, 184897: 75, 185037: 92, 185051: 15, 185076: 86, 185091: 36, 185244: 59, 185273: 87, 185282: 63, 185309: 19, 185381: 13, 185441: 92, 185551: 84, 185630: 49, 185643: 66, 185661: 19, 185696: 89, 185777: 46, 185874: 43, 185878: 89, 185879: 15, 185935: 96, 185953: 99, 186074: 61, 186080: 23, 186086: 98, 186152: 11, 186200: 14, 186352: 58, 186466: 41, 186485: 35, 186582: 39, 186632: 31, 186639: 25, 186674: 11, 186721: 94, 186858: 22, 186863: 44, 187112: 69, 187157: 37, 187178: 21, 187252: 36, 187374: 79, 187389: 76, 187456: 60, 187557: 49, 187619: 30, 187718: 48, 187796: 22, 187809: 95, 187823: 90, 187833: 77, 187842: 72, 187867: 42, 187890: 5, 187909: 10, 187932: 20, 188016: 52, 188054: 59, 188195: 81, 188285: 46, 188539: 43, 188634: 43, 188747: 83, 188809: 84, 188827: 7, 188891: 2, 189017: 33, 189035: 21, 189036: 79, 189177: 4, 189178: 13, 189236: 83, 189282: 64, 189296: 80, 189408: 76, 189593: 0, 189628: 33, 189718: 46, 189823: 71, 189864: 40, 190083: 93, 190116: 81, 190140: 31, 190177: 40, 190231: 68, 190287: 4, 190296: 36, 190318: 72, 190375: 62, 190424: 62, 190428: 43, 190462: 88, 190488: 26, 190530: 80, 190602: 45, 190679: 83, 190795: 28, 190872: 21, 190893: 9, 190901: 25, 190944: 29, 190976: 65, 190993: 24, 191140: 48, 191234: 41, 191361: 53, 191425: 83, 191456: 36, 191723: 23, 191741: 10, 191828: 40, 191847: 76, 191871: 44, 191909: 54, 191912: 89, 191950: 67, 192065: 25, 192094: 46, 192102: 38, 192195: 81, 192210: 24, 192263: 39, 192271: 73, 192293: 7, 192333: 80, 192415: 44, 192554: 89, 192570: 46, 192604: 97, 192654: 58, 192664: 17, 192941: 62, 192971: 70, 193011: 74, 193194: 93, 193201: 63, 193260: 74, 193308: 38, 193373: 64, 193422: 7, 193458: 72, 193537: 50, 193540: 4, 193643: 98, 193654: 51, 193667: 87, 193677: 20, 193811: 25, 193828: 90, 193876: 74, 193880: 9, 193891: 16, 193938: 2, 193990: 67, 194245: 81, 194293: 39, 194384: 39, 194401: 94, 194420: 17, 194443: 70, 194503: 19, 194549: 18, 194581: 39, 194667: 7, 194760: 47, 194766: 81, 194842: 57, 194871: 74, 194948: 21, 194952: 91, 194976: 79, 195006: 3, 195084: 16, 195138: 61, 195157: 3, 195186: 56, 195309: 94, 195337: 81, 195363: 15, 195467: 51, 195549: 44, 195624: 93, 195751: 26, 196022: 96, 196043: 80, 196049: 81, 196083: 18, 196085: 6, 196114: 67, 196157: 37, 196182: 6, 196286: 2, 196289: 7, 196338: 53, 196388: 16, 196432: 81, 196451: 1, 196539: 13, 196563: 30, 196706: 10, 196832: 89, 196854: 70, 196863: 58, 196898: 99, 196954: 74, 196956: 21, 197115: 3, 197135: 96, 197191: 98, 197332: 11, 197398: 36, 197426: 93, 197438: 67, 197449: 99, 197578: 50, 197614: 42, 197656: 7, 197701: 88, 197767: 56, 197828: 42, 197889: 91, 197904: 45, 197906: 31, 197977: 82, 198089: 84, 198215: 28, 198221: 52, 198371: 55, 198578: 93, 198630: 82, 198751: 72, 198799: 15, 198846: 86, 198861: 64, 198917: 39, 198934: 30, 198978: 61, 199016: 65, 199048: 31, 199159: 84, 199187: 15, 199194: 19, 199253: 5, 199254: 66, 199335: 92, 199440: 84, 199501: 82, 199883: 21, 199906: 23, 200014: 19, 200092: 87, 200162: 99, 200236: 45, 200296: 72, 200452: 8, 200500: 63, 200508: 89, 200518: 33, 200523: 80, 200646: 85, 200843: 43, 200870: 68, 200954: 9, 201017: 33, 201032: 99, 201070: 50, 201072: 18, 201078: 57, 201140: 23, 201178: 57, 201179: 5, 201299: 76, 201329: 8, 201510: 16, 201583: 61, 201854: 30, 201959: 39, 202002: 38, 202018: 50, 202025: 62, 202029: 76, 202053: 63, 202081: 62, 202085: 53, 202123: 4, 202187: 31, 202188: 88, 202256: 62, 202347: 49, 202361: 80, 202459: 46, 202526: 86, 202593: 58, 202627: 10, 202632: 72, 202705: 9, 202814: 6, 202860: 47, 203048: 23, 203107: 79, 203128: 96, 203307: 63, 203327: 2, 203395: 42, 203433: 71, 203442: 91, 203592: 68, 203648: 90, 203705: 11, 203738: 49, 203799: 95, 203849: 8, 203860: 39, 204045: 65, 204165: 15, 204283: 77, 204296: 45, 204343: 87, 204600: 69, 204779: 32, 204808: 30, 204852: 47, 204931: 85, 204959: 80, 205000: 11, 205019: 40, 205063: 58, 205091: 84, 205158: 45, 205170: 21, 205244: 32, 205261: 49, 205422: 79, 205476: 10, 205477: 93, 205490: 13, 205539: 43, 205540: 53, 205701: 7, 205823: 80, 205827: 64, 205859: 53, 205864: 52, 205968: 68, 206042: 46, 206104: 8, 206265: 55, 206274: 32, 206320: 91, 206485: 79, 206525: 17, 206540: 33, 206606: 28, 206621: 15, 206646: 25, 206664: 91, 206740: 55, 206870: 21, 206879: 84, 206900: 22, 207020: 11, 207051: 3, 207062: 11, 207139: 76, 207198: 83, 207201: 81, 207229: 36, 207265: 47, 207379: 68, 207407: 37, 207427: 83, 207556: 56, 207820: 37, 207846: 64, 207847: 76, 207848: 86, 207862: 36, 208095: 40, 208402: 0, 208417: 34, 208419: 95, 208430: 41, 208457: 48, 208484: 58, 208553: 34, 208619: 10, 208640: 76, 208673: 1, 208702: 99, 208746: 96, 208862: 73, 209079: 8, 209157: 3, 209162: 90, 209316: 88, 209432: 50, 209497: 60, 209613: 23, 209720: 89, 209791: 16, 209812: 46, 209834: 93, 209944: 17, 209948: 9, 210160: 62, 210209: 36, 210211: 83, 210309: 53, 210334: 49, 210341: 16, 210462: 14, 210463: 23, 210540: 54, 210613: 48, 210618: 47, 210655: 3, 210933: 17, 210934: 21, 210938: 37, 210957: 16, 210970: 61, 211002: 70, 211006: 91, 211077: 27, 211120: 8, 211168: 33, 211249: 3, 211321: 65, 211327: 53, 211342: 99, 211379: 43, 211471: 91, 211656: 58, 211680: 33, 211722: 74, 211773: 34, 211800: 82, 211842: 43, 211860: 35, 211887: 28, 212009: 92, 212013: 26, 212099: 71, 212137: 75, 212164: 30, 212247: 30, 212281: 27, 212316: 80, 212321: 28, 212374: 53, 212574: 76, 212751: 26, 212781: 46, 212836: 20, 212905: 3, 213101: 8, 213145: 21, 213217: 78, 213341: 26, 213499: 79, 213518: 70, 213575: 99, 213579: 86, 213589: 41, 213597: 58, 213734: 74, 213758: 85, 213812: 55, 213824: 83, 213847: 3, 213952: 82, 214057: 82, 214122: 31, 214134: 91, 214176: 8, 214206: 56, 214305: 56, 214444: 32, 214477: 9, 214479: 67, 214587: 72, 214638: 87, 214642: 23, 214648: 7, 214672: 91, 214882: 49, 214907: 79, 214953: 90, 215011: 59, 215015: 91, 215018: 76, 215039: 26, 215087: 55, 215151: 77, 215160: 90, 215162: 38, 215286: 1, 215305: 94, 215467: 56, 215468: 25, 215520: 23, 215526: 60, 215557: 33, 215559: 35, 215713: 61, 215799: 34, 215904: 36, 216090: 68, 216157: 64, 216217: 12, 216235: 57, 216239: 26, 216240: 99, 216278: 87, 216292: 11, 216320: 75, 216479: 15, 216531: 48, 216591: 15, 216636: 93, 216825: 69, 216853: 45, 216956: 18, 217049: 83, 217166: 92, 217190: 30, 217456: 2, 217546: 98, 217639: 59, 217693: 24, 217821: 31, 217859: 5, 217890: 17, 217920: 13, 218082: 61, 218155: 11, 218238: 77, 218241: 28, 218264: 60, 218277: 73, 218311: 75, 218323: 19, 218479: 93, 218498: 59, 218514: 85, 218570: 65, 218575: 20, 218660: 89, 218689: 30, 218723: 12, 218731: 98, 218851: 61, 219108: 43, 219247: 81, 219290: 14, 219328: 38, 219388: 72, 219461: 79, 219470: 6, 219482: 23, 219555: 1, 219577: 39, 219595: 56, 219656: 11, 219766: 57, 219783: 96, 219908: 25, 219994: 64, 219998: 20, 220140: 71, 220208: 95, 220226: 29, 220355: 25, 220375: 74, 220377: 51, 220428: 61, 220448: 70, 220514: 32, 220557: 95, 220589: 91, 220770: 15, 220876: 2, 220924: 10, 220975: 24, 221064: 46, 221189: 10, 221277: 63, 221518: 52, 221590: 77, 221654: 77, 221914: 22, 222235: 8, 222281: 58, 222290: 40, 222295: 96, 222306: 35, 222316: 51, 222343: 97, 222360: 28, 222505: 18, 222507: 36, 222593: 11, 222605: 95, 222708: 92, 222794: 73, 222953: 30, 223013: 10, 223103: 61, 223144: 13, 223163: 63, 223182: 77, 223191: 40, 223387: 17, 223672: 94, 223698: 2, 223766: 49, 223786: 1, 223875: 59, 223923: 69, 223991: 61, 224013: 62, 224054: 13, 224058: 1, 224150: 67, 224280: 29, 224368: 37, 224405: 89, 224473: 73, 224527: 48, 224534: 38, 224744: 36, 224892: 61, 225121: 93, 225138: 63, 225481: 90, 225549: 19, 225672: 37, 225907: 53, 225984: 73, 225989: 19, 226020: 56, 226171: 14, 226233: 33, 226319: 1, 226339: 9, 226389: 3, 226438: 44, 226456: 70, 226505: 0, 226583: 33, 226726: 3, 226810: 21, 226932: 12, 227033: 31, 227049: 2, 227155: 99, 227231: 70, 227241: 86, 227243: 72, 227252: 88, 227264: 90, 227287: 53, 227297: 12, 227298: 45, 227314: 95, 227542: 38, 227553: 53, 227573: 48, 227705: 80, 227746: 11, 227751: 21, 227796: 99, 227822: 60, 227850: 46, 227853: 19, 227862: 85, 227925: 64, 228022: 39, 228053: 61, 228064: 40, 228067: 98, 228224: 71, 228226: 50, 228310: 94, 228316: 67, 228432: 8, 228460: 78, 228461: 12, 228637: 46, 228665: 43, 228806: 96, 228870: 37, 228952: 28, 229084: 95, 229132: 58, 229287: 55, 229386: 15, 229419: 74, 229450: 26, 229520: 98, 229527: 86, 229532: 15, 229697: 87, 229741: 51, 229970: 25, 229971: 33, 229984: 59, 230083: 97, 230238: 40, 230333: 64, 230375: 28, 230450: 73, 230466: 2, 230544: 4, 230587: 5, 230625: 47, 230681: 15, 230715: 64, 230876: 94, 230880: 9, 230898: 69, 230991: 2, 231167: 34, 231216: 29, 231223: 51, 231274: 85, 231401: 28, 231535: 6, 231547: 27, 231654: 19, 231657: 57, 231672: 81, 231724: 98, 231774: 81, 231797: 10, 231802: 12, 231880: 77, 231997: 60, 232038: 12, 232095: 4, 232115: 3, 232163: 45, 232316: 95, 232526: 9, 232564: 54, 232607: 66, 232630: 7, 232638: 20, 232700: 12, 232791: 67, 232800: 36, 232827: 95, 232933: 12, 232936: 38, 232992: 49, 233015: 88, 233158: 40, 233247: 32, 233270: 11, 233281: 36, 233336: 60, 233407: 66, 233447: 63, 233483: 27, 233495: 57, 233519: 72, 233542: 77, 233545: 62, 233548: 38, 233635: 56, 233750: 22, 233829: 4, 234027: 46, 234050: 82, 234381: 20, 234447: 38, 234465: 1, 234500: 82, 234559: 54, 234603: 7, 234606: 33, 234629: 16, 234697: 47, 234732: 5, 234780: 73, 234807: 31, 234904: 82, 234975: 85, 235003: 2, 235099: 46, 235156: 86, 235178: 21, 235212: 62, 235232: 21, 235318: 78, 235382: 86, 235432: 66, 235501: 35, 235591: 24, 235595: 13, 235641: 6, 235663: 77, 235816: 10, 235866: 87, 235972: 44, 235984: 27, 236046: 78, 236066: 33, 236145: 10, 236212: 48, 236315: 73, 236333: 53, 236397: 29, 236430: 20, 236451: 85, 236470: 44, 236501: 32, 236547: 64, 236561: 88, 236637: 34, 236799: 41, 236842: 11, 236906: 25, 236949: 10, 236992: 72, 237016: 93, 237018: 37, 237050: 2, 237124: 56, 237221: 32, 237380: 6, 237409: 23, 237458: 17, 237759: 73, 237856: 97, 237866: 36, 237869: 74, 237967: 80, 237970: 28, 238055: 0, 238094: 64, 238118: 23, 238125: 12, 238162: 92, 238191: 72, 238363: 99, 238451: 31, 238547: 8, 238603: 4, 238711: 81, 238751: 47, 238769: 39, 238803: 75, 238818: 48, 238846: 28, 238850: 50, 238950: 29, 238968: 20, 239012: 66, 239016: 45, 239057: 75, 239066: 96, 239172: 75, 239188: 63, 239202: 69, 239300: 32, 239328: 4, 239339: 55, 239355: 99, 239410: 14, 239545: 67, 239679: 31, 239921: 78, 240046: 54, 240193: 17, 240204: 59, 240219: 68, 240491: 80, 240520: 4, 240527: 67, 240563: 33, 240636: 29, 240654: 41, 240667: 96, 240788: 16, 240794: 18, 240803: 43, 240904: 90, 240952: 0, 240959: 23, 240996: 4, 241021: 33, 241029: 77, 241045: 66, 241069: 79, 241250: 19, 241323: 16, 241361: 86, 241428: 93, 241517: 82, 241733: 37, 241736: 61, 241764: 84, 241881: 31, 241899: 54, 241901: 81, 241999: 23, 242083: 23, 242153: 3, 242259: 82, 242327: 61, 242349: 87, 242498: 2, 242665: 65, 242690: 62, 242731: 17, 242774: 41, 242800: 49, 242828: 78, 242856: 96, 242930: 39, 243012: 69, 243300: 78, 243321: 72, 243360: 64, 243431: 9, 243545: 92, 243559: 4, 243621: 18, 243643: 21, 243746: 54, 243797: 63, 243922: 8, 243936: 84, 244026: 81, 244185: 40, 244195: 77, 244243: 63, 244328: 71, 244427: 29, 244444: 13, 244699: 97, 244708: 65, 244757: 59, 244816: 36, 244857: 74, 244949: 26, 245018: 65, 245080: 73, 245082: 7, 245106: 69, 245115: 54, 245238: 50, 245296: 78, 245335: 88, 245397: 12, 245627: 96, 245655: 97, 245668: 72, 245688: 28, 245709: 72, 245717: 15, 245740: 96, 245748: 19, 245810: 72, 245901: 73, 245920: 83, 245948: 2, 246043: 17, 246164: 34, 246240: 75, 246243: 52, 246279: 80, 246318: 54, 246464: 51, 246476: 59, 246485: 85, 246509: 23, 246572: 5, 246714: 40, 246725: 43, 246812: 6, 246815: 47, 246829: 36, 246895: 77, 247016: 16, 247044: 60, 247046: 59, 247069: 78, 247151: 32, 247359: 98, 247373: 62, 247385: 86, 247396: 66, 247399: 64, 247457: 12, 247475: 95, 247527: 10, 247617: 74, 247745: 2, 247789: 15, 247896: 86, 248016: 19, 248101: 49, 248105: 45, 248229: 42, 248307: 73, 248354: 24, 248387: 50, 248475: 46, 248598: 86, 248664: 37, 248696: 43, 248711: 36, 248725: 70, 248801: 18, 248906: 11, 248917: 25, 249193: 96, 249339: 87, 249393: 89, 249404: 82, 249462: 47, 249511: 93, 249528: 28, 249559: 24, 249654: 2, 249743: 63, 249801: 1, 249900: 48, 250006: 53, 250119: 42, 250161: 36, 250180: 64, 250186: 47, 250211: 11, 250268: 64, 250325: 44, 250370: 7, 250395: 29, 250601: 70, 250702: 55, 250765: 4, 250964: 30, 251111: 67, 251155: 74, 251204: 62, 251273: 38, 251566: 99, 251889: 31, 251999: 22, 252014: 59, 252101: 8, 252279: 59, 252326: 82, 252353: 15, 252413: 72, 252436: 82, 252484: 89, 252511: 50, 252537: 19, 252582: 37, 252773: 71, 252776: 80, 252815: 21, 252926: 98, 252933: 53, 253052: 4, 253073: 35, 253091: 45, 253104: 83, 253141: 41, 253166: 5, 253241: 5, 253257: 39, 253359: 12, 253380: 15, 253541: 65, 253562: 55, 253567: 12, 253608: 99, 253621: 33, 253730: 41, 253744: 3, 253791: 97, 253812: 98, 253870: 97, 253883: 0, 253951: 25, 253954: 12, 253996: 5, 254007: 86, 254040: 17, 254126: 69, 254199: 15, 254212: 84, 254218: 46, 254223: 67, 254329: 21, 254339: 65, 254342: 74, 254396: 87, 254488: 15, 254490: 34, 254552: 50, 254558: 91, 254683: 75, 254721: 68, 254788: 49, 254807: 36, 254836: 78, 254894: 75, 254910: 97, 254943: 46, 254975: 29, 254989: 80, 255152: 44, 255416: 85, 255452: 75, 255486: 1, 255569: 98, 255615: 99, 255673: 12, 255700: 97, 255904: 0, 255933: 76, 256014: 46, 256166: 1, 256213: 98, 256320: 53, 256334: 43, 256352: 84, 256404: 96, 256550: 43, 256725: 96, 256832: 3, 256891: 29, 256895: 4, 257002: 69, 257071: 50, 257083: 95, 257100: 56, 257169: 47, 257173: 54, 257230: 49, 257473: 27, 257481: 46, 257521: 14, 257558: 63, 257600: 6, 257660: 94, 257668: 97, 257847: 96, 257882: 69, 258054: 93, 258078: 19, 258181: 91, 258255: 75, 258277: 10, 258316: 64, 258317: 60, 258472: 38, 258566: 98, 258599: 23, 258649: 82, 258742: 64, 259023: 98, 259079: 12, 259170: 30, 259208: 73, 259359: 73, 259372: 43, 259430: 43, 259442: 38, 259464: 12, 259544: 88, 259720: 30, 259730: 75, 259739: 27, 259850: 7, 259858: 92, 259873: 84, 260037: 83, 260280: 38, 260314: 56, 260324: 92, 260358: 67, 260460: 24, 260492: 17, 260734: 5, 260766: 54, 260775: 77, 260841: 30, 260863: 23, 260877: 79, 260899: 22, 260920: 64, 260930: 56, 260965: 38, 261004: 26, 261021: 52, 261102: 42, 261159: 87, 261175: 62, 261248: 98, 261291: 64, 261310: 6, 261360: 22, 261361: 59, 261383: 53, 261440: 38, 261458: 82, 261482: 84, 261544: 34, 261589: 67, 261592: 47, 261682: 61, 261702: 62, 261967: 63, 262077: 93, 262222: 85, 262348: 47, 262369: 9, 262424: 22, 262444: 65, 262468: 94, 262588: 74, 262640: 2, 262653: 36, 262659: 62, 262835: 29, 263000: 13, 263107: 78, 263129: 91, 263135: 62, 263161: 86, 263182: 14, 263213: 67, 263345: 53, 263421: 98, 263561: 20, 263594: 40, 263690: 17, 263957: 55, 264132: 65, 264328: 68, 264355: 31, 264519: 32, 264531: 56, 264626: 60, 264634: 85, 264647: 35, 264728: 67, 264738: 60, 264914: 92, 265101: 9, 265105: 61, 265194: 47, 265210: 30, 265241: 17, 265301: 5, 265334: 11, 265471: 56, 265527: 30, 265539: 20, 265581: 79, 265605: 48, 265624: 3, 265629: 90, 265863: 66, 265875: 78, 265899: 27, 265964: 36, 265998: 4, 266123: 44, 266419: 63, 266490: 35, 266562: 56, 266630: 21, 266632: 1, 266720: 23, 266758: 82, 266786: 22, 266799: 49, 267043: 64, 267162: 11, 267300: 14, 267427: 31, 267481: 97, 267553: 18, 267555: 54, 267570: 82, 267578: 35, 267661: 69, 267681: 56, 267706: 49, 267753: 49, 267907: 12, 268088: 51, 268210: 8, 268255: 74, 268256: 45, 268299: 35, 268315: 54, 268374: 69, 268378: 88, 268400: 91, 268424: 84, 268444: 26, 268759: 53, 268787: 5, 268869: 76, 268984: 26, 269031: 17, 269120: 27, 269301: 17, 269349: 47, 269351: 44, 269422: 3, 269429: 17, 269475: 19, 269627: 94, 269652: 29, 269654: 3, 269655: 47, 269762: 18, 269825: 66, 269935: 14, 270043: 34, 270046: 1, 270216: 26, 270227: 97, 270503: 73, 270516: 73, 270559: 26, 270586: 43, 270728: 11, 270804: 81, 270824: 62, 270941: 63, 270972: 98, 271097: 0, 271116: 19, 271262: 55, 271281: 34, 271285: 53, 271300: 53, 271324: 94, 271370: 60, 271422: 74, 271449: 86, 271492: 49, 271572: 43, 271902: 56, 271940: 38, 271943: 72, 271996: 94, 272175: 26, 272202: 66, 272217: 43, 272250: 12, 272267: 38, 272279: 38, 272312: 79, 272360: 73, 272505: 58, 272529: 91, 272556: 85, 272640: 55, 272648: 98, 272687: 44, 272699: 96, 272802: 40, 272901: 39, 272906: 85, 273068: 2, 273155: 19, 273200: 76, 273215: 20, 273473: 12, 273526: 0, 273527: 5, 273717: 1, 273727: 29, 273760: 10, 273819: 60, 273947: 49, 274162: 72, 274176: 54, 274192: 58, 274363: 35, 274380: 23, 274390: 82, 274445: 16, 274646: 30, 274673: 50, 274690: 36, 274700: 4, 274722: 54, 274786: 89, 274831: 14, 274845: 47, 274923: 43, 274967: 65, 274976: 94, 275130: 66, 275153: 70, 275180: 35, 275268: 48, 275320: 33, 275418: 82, 275504: 21, 275563: 19, 275727: 94, 275773: 76, 275785: 43, 275840: 21, 275861: 1, 275899: 63, 275927: 41, 275930: 87, 276006: 90, 276035: 14, 276222: 84, 276230: 16, 276271: 15, 276330: 37, 276344: 5, 276540: 27, 276541: 83, 276566: 68, 276583: 39, 276692: 69, 276859: 5, 277041: 6, 277072: 62, 277116: 86, 277184: 78, 277198: 62, 277199: 90, 277248: 89, 277276: 8, 277355: 86, 277421: 74, 277435: 49, 277489: 45, 277653: 13, 277723: 92, 277729: 61, 277736: 33, 277798: 76, 277835: 19, 277841: 95, 277861: 8, 277917: 40, 277993: 62, 277996: 94, 278083: 43, 278233: 57, 278461: 86, 278565: 73, 278617: 78, 278661: 40, 278706: 66, 278766: 35, 278801: 58, 278837: 25, 278855: 28, 278953: 89, 279041: 67, 279190: 47, 279263: 25, 279305: 28, 279333: 38, 279369: 58, 279406: 75, 279453: 84, 279463: 26, 279501: 56, 279667: 22, 279779: 47, 279914: 41, 279929: 56, 279940: 23, 279996: 44, 280027: 25, 280097: 30, 280206: 82, 280221: 51, 280385: 21, 280387: 9, 280557: 65, 280613: 62, 280619: 21, 280675: 24, 280823: 56, 280843: 46, 280864: 18, 280940: 6, 281001: 36, 281080: 46, 281123: 32, 281218: 18, 281233: 25, 281342: 89, 281375: 64, 281399: 90, 281640: 34, 281647: 85, 281858: 17, 281864: 49, 281871: 88, 281897: 1, 281902: 15, 281995: 41, 282055: 2, 282066: 73, 282106: 3, 282166: 58, 282197: 49, 282205: 25, 282335: 16, 282380: 10, 282385: 71, 282406: 97, 282438: 5, 282515: 45, 282700: 95, 282711: 75, 282745: 88, 282810: 99, 282832: 83, 282976: 48, 283000: 0, 283032: 70, 283061: 15, 283117: 67, 283122: 85, 283166: 24, 283319: 2, 283446: 45, 283548: 5, 283611: 11, 283705: 3, 283714: 79, 283839: 30, 284100: 27, 284132: 2, 284148: 89, 284149: 27, 284233: 13, 284279: 13, 284327: 72, 284383: 87, 284650: 60, 284653: 36, 284655: 98, 284704: 23, 284764: 18, 284798: 76, 284866: 53, 284904: 62, 285036: 41, 285043: 29, 285073: 93, 285096: 27, 285168: 15, 285297: 78, 285310: 97, 285323: 28, 285371: 37, 285376: 76, 285406: 39, 285419: 62, 285540: 10, 285603: 85, 285653: 77, 285659: 3, 285692: 75, 285700: 6, 285745: 22, 285757: 94, 285782: 79, 285883: 96, 286033: 55, 286057: 39, 286067: 25, 286144: 13, 286173: 98, 286224: 17, 286253: 93, 286257: 96, 286546: 49, 286563: 24, 286594: 95, 286657: 41, 286796: 74, 286924: 69, 286959: 35, 287036: 95, 287169: 74, 287183: 29, 287247: 75, 287322: 82, 287504: 21, 287596: 13, 287635: 39, 287716: 31, 287719: 48, 287790: 38, 287811: 21, 287849: 79, 287865: 17, 288014: 74, 288024: 94, 288066: 3, 288074: 57, 288143: 99, 288172: 87, 288220: 29, 288250: 99, 288343: 0, 288349: 91, 288491: 14, 288702: 95, 288758: 47, 288778: 74, 289008: 24, 289076: 64, 289336: 97, 289387: 93, 289472: 24, 289532: 5, 289567: 56, 289577: 68, 289713: 51, 289727: 64, 289805: 29, 289898: 69, 289906: 55, 289909: 82, 289938: 75, 289951: 27, 289994: 58, 290032: 2, 290156: 26, 290206: 86, 290349: 5, 290484: 15, 290494: 41, 290549: 92, 290586: 30, 290634: 23, 290710: 29, 290794: 23, 290883: 33, 291069: 3, 291135: 55, 291174: 70, 291417: 82, 291451: 43, 291475: 69, 291594: 69, 291685: 46, 291687: 83, 291702: 90, 291782: 54, 291833: 85, 291919: 12, 292280: 60, 292281: 25, 292360: 96, 292385: 91, 292441: 50, 292464: 19, 292499: 16, 292503: 48, 292534: 90, 292565: 18, 292613: 43, 292649: 54, 292681: 59, 292683: 24, 292704: 70, 292834: 54, 292886: 3, 292994: 6, 293118: 36, 293254: 57, 293269: 98, 293486: 35, 293574: 45, 293730: 88, 293754: 68, 293777: 51, 293820: 64, 293880: 94, 293883: 36, 293918: 34, 293921: 54, 293957: 51, 294003: 97, 294127: 50, 294140: 72, 294141: 53, 294182: 39, 294262: 75, 294376: 67, 294479: 50, 294512: 37, 294597: 54, 294667: 48, 294722: 24, 294744: 45, 294949: 85, 295196: 5, 295202: 33, 295213: 32, 295276: 77, 295304: 87, 295418: 76, 295572: 88, 295592: 71, 295596: 10, 295843: 96, 295915: 11, 296044: 72, 296070: 88, 296079: 25, 296206: 47, 296264: 85, 296323: 34, 296465: 7, 296551: 35, 296619: 37, 296622: 73, 296698: 15, 296700: 55, 296733: 19, 296983: 69, 297014: 39, 297047: 69, 297076: 75, 297086: 52, 297120: 7, 297183: 6, 297262: 95, 297497: 92, 297582: 57, 297607: 45, 297789: 31, 297824: 13, 297959: 66, 297997: 76, 298055: 86, 298064: 14, 298086: 48, 298112: 0, 298124: 73, 298151: 56, 298234: 98, 298270: 40, 298369: 80, 298523: 55, 298604: 57, 298674: 29, 298683: 85, 298731: 5, 298735: 51, 298736: 42, 298800: 5, 298818: 27, 298832: 40, 298880: 0, 298979: 7, 299078: 63, 299127: 88, 299382: 3, 299389: 26, 299395: 70, 299397: 32, 299466: 16, 299480: 47, 299515: 30, 299554: 24, 299644: 9, 299788: 85, 299922: 10, 299952: 97, 299963: 88, 300082: 44, 300088: 35, 300270: 26, 300307: 85, 300352: 63, 300354: 8, 300420: 86, 300536: 79, 300579: 41, 300594: 8, 300630: 66, 300715: 38, 300792: 43, 300894: 52, 300945: 0, 300986: 35, 301069: 31, 301227: 44, 301261: 79, 301292: 26, 301325: 91, 301338: 98, 301346: 1, 301501: 3, 301595: 56, 301610: 4, 301611: 63, 301637: 99, 301701: 27, 301737: 69, 301742: 66, 301743: 27, 301744: 70, 301765: 83, 301811: 75, 302121: 8, 302137: 62, 302299: 74, 302540: 13, 302603: 46, 302817: 25, 302911: 70, 303073: 82, 303074: 75, 303078: 50, 303096: 43, 303220: 60, 303238: 53, 303354: 81, 303439: 10, 303452: 38, 303487: 82, 303507: 4, 303576: 11, 303626: 26, 303835: 89, 303889: 68, 303998: 2, 304297: 33, 304423: 88, 304443: 27, 304532: 57, 304569: 12, 304587: 37, 304590: 57, 304756: 30, 304769: 34, 304796: 89, 304839: 7, 304850: 79, 304935: 9, 304949: 90, 304959: 6, 305017: 33, 305099: 74, 305100: 13, 305101: 20, 305127: 97, 305341: 84, 305357: 19, 305384: 66, 305472: 92, 305474: 59, 305554: 2, 305604: 95, 305919: 99, 305950: 64, 305965: 37, 306200: 84, 306219: 64, 306243: 10, 306276: 45, 306373: 17, 306590: 15, 306655: 85, 307050: 27, 307113: 83, 307179: 58, 307188: 8, 307329: 67, 307353: 63, 307369: 97, 307384: 96, 307496: 56, 307580: 46, 307581: 86, 307582: 40, 307592: 26, 307657: 63, 307659: 62, 307674: 83, 307754: 76, 307802: 94, 307891: 77, 307934: 96, 307938: 78, 308050: 50, 308060: 33, 308105: 67, 308107: 41, 308127: 1, 308377: 83, 308394: 41, 308459: 46, 308536: 3, 308543: 9, 308613: 60, 308616: 33, 308812: 34, 308827: 74, 308841: 2, 308899: 20, 308955: 22, 309017: 49, 309093: 61, 309102: 71, 309107: 0, 309253: 87, 309270: 42, 309355: 48, 309474: 17, 309703: 35, 309754: 77, 309795: 0, 309938: 42, 309943: 94, 309955: 82, 310022: 17, 310164: 57, 310201: 75, 310203: 81, 310225: 30, 310232: 77, 310248: 6, 310597: 54, 310603: 38, 310643: 51, 310650: 81, 310988: 58, 311273: 66, 311338: 74, 311349: 26, 311363: 48, 311381: 34, 311393: 11, 311394: 65, 311458: 25, 311510: 51, 311546: 63, 311564: 89, 311773: 22, 311803: 12, 311823: 87, 311869: 66, 311884: 86, 311902: 69, 311925: 54, 311972: 10, 312136: 85, 312166: 15, 312405: 36, 312457: 14, 312626: 23, 313100: 37, 313139: 17, 313228: 90, 313250: 22, 313263: 24, 313279: 44, 313366: 59, 313446: 30, 313627: 17, 313655: 29, 313707: 22, 313725: 65, 313844: 16, 313967: 75, 314028: 80, 314054: 43, 314081: 34, 314104: 15, 314111: 91, 314184: 85, 314222: 20, 314278: 79, 314294: 44, 314421: 42, 314436: 70, 314464: 98, 314465: 41, 314497: 90, 314620: 96, 314650: 20, 314683: 32, 314781: 54, 314962: 93, 314982: 33, 315106: 76, 315135: 21, 315160: 62, 315163: 22, 315203: 75, 315281: 86, 315321: 9, 315489: 6, 315549: 8, 315593: 29, 315617: 44, 315703: 93, 315722: 93, 315739: 26, 315784: 88, 315840: 42, 315870: 66, 315955: 57, 316103: 50, 316142: 66, 316223: 75, 316279: 28, 316325: 93, 316339: 65, 316364: 93, 316623: 82, 316700: 31, 316814: 17, 316821: 14, 316929: 40, 316948: 6, 316968: 96, 317005: 67, 317017: 62, 317099: 46, 317104: 23, 317147: 26, 317165: 22, 317180: 30, 317237: 63, 317253: 39, 317286: 32, 317301: 4, 317446: 98, 317559: 45, 317599: 68, 317610: 43, 317627: 55, 317702: 27, 317733: 91, 317752: 57, 317790: 48, 317848: 70, 317903: 79, 318045: 47, 318056: 62, 318123: 41, 318225: 57, 318315: 80, 318422: 69, 318486: 74, 318715: 92, 318821: 31, 318857: 98, 318935: 11, 318974: 25, 318991: 43, 319024: 65, 319041: 32, 319155: 57, 319278: 19, 319441: 65, 319508: 38, 319518: 52, 319548: 96, 319625: 26, 319647: 62, 319650: 97, 319669: 74, 319691: 15, 319724: 99, 319760: 35, 319896: 10, 320094: 72, 320232: 82, 320304: 32, 320307: 21, 320379: 2, 320460: 48, 320478: 51, 320570: 40, 320598: 91, 320608: 40, 320622: 49, 320662: 75, 320669: 86, 320681: 53, 320682: 25, 320705: 55, 320733: 30, 320798: 95, 320998: 39, 321154: 34, 321163: 40, 321172: 78, 321205: 37, 321248: 84, 321335: 67, 321391: 41, 321476: 39, 321477: 2, 321500: 5, 321507: 77, 321521: 6, 321551: 4, 321574: 75, 321623: 99, 321628: 55, 321637: 95, 321801: 87, 321840: 58, 321915: 90, 321966: 23, 322131: 47, 322274: 51, 322307: 80, 322372: 91, 322378: 67, 322439: 17, 322590: 58, 322611: 24, 322636: 54, 322892: 17, 322912: 95, 322968: 43, 323039: 14, 323177: 22, 323201: 73, 323209: 96, 323277: 36, 323339: 10, 323380: 77, 323406: 17, 323624: 18, 323674: 18, 323710: 56, 323752: 35, 323754: 82, 323959: 6, 324034: 50, 324065: 22, 324067: 50, 324167: 7, 324193: 16, 324264: 68, 324293: 61, 324528: 14, 324539: 86, 324556: 37, 324607: 9, 324673: 28, 324702: 27, 324895: 38, 324938: 58, 325019: 69, 325100: 64, 325129: 82, 325272: 23, 325273: 80, 325384: 96, 325470: 59, 325557: 38, 325599: 82, 325676: 85, 325695: 25, 325716: 95, 325722: 15, 325723: 38, 325779: 25, 325822: 57, 325914: 25, 326041: 43, 326081: 10, 326168: 74, 326245: 31, 326318: 18, 326322: 61, 326379: 41, 326395: 37, 326482: 3, 326535: 3, 326565: 61, 326719: 40, 326736: 4, 326795: 65, 326944: 12, 326996: 70, 327146: 96, 327178: 88, 327229: 62, 327474: 23, 327530: 25, 327582: 72, 327601: 0, 327668: 53, 327670: 85, 327678: 67, 327693: 55, 327781: 2, 327795: 15, 327867: 0, 327885: 21, 327907: 99, 327983: 29, 327997: 62, 328231: 53, 328295: 85, 328477: 85, 328502: 76, 328547: 9, 328557: 7, 328605: 5, 328624: 19, 328915: 86, 329017: 51, 329043: 97, 329180: 0, 329228: 69, 329239: 80, 329243: 74, 329264: 98, 329396: 37, 329456: 70, 329504: 29, 329525: 80, 329538: 32, 329600: 51, 329689: 80, 329740: 99, 329782: 64, 330062: 77, 330069: 53, 330179: 49, 330304: 51, 330341: 87, 330403: 71, 330427: 4, 330433: 25, 330470: 13, 330475: 0, 330500: 44, 330531: 92, 330552: 34, 330557: 29, 330617: 20, 330621: 83, 330690: 5, 330725: 7, 330765: 12, 330817: 84, 331024: 66, 331200: 98, 331202: 8, 331414: 4, 331479: 15, 331515: 41, 331597: 27, 331719: 3, 331781: 25, 331816: 40, 331837: 85, 331869: 20, 331966: 71, 332211: 52, 332239: 76, 332307: 32, 332381: 42, 332497: 95, 332612: 29, 332620: 66, 332637: 75, 332672: 28, 332723: 13, 332772: 79, 332912: 34, 333093: 18, 333255: 11, 333257: 42, 333337: 59, 333371: 71, 333388: 62, 333520: 24, 333527: 51, 333611: 66, 333665: 84, 333674: 34, 333727: 36, 333856: 30, 333870: 53, 333983: 21, 334150: 90, 334363: 35, 334511: 78, 334571: 4, 334580: 42, 334604: 83, 334607: 3, 334624: 2, 334787: 10, 334820: 79, 334869: 89, 334981: 55, 334983: 47, 334988: 94, 335056: 63, 335154: 53, 335162: 61, 335266: 68, 335333: 5, 335363: 68, 335391: 35, 335418: 39, 335557: 16, 335560: 99, 335607: 37, 335637: 83, 335674: 41, 335991: 33, 336079: 34, 336248: 92, 336316: 58, 336339: 45, 336351: 43, 336495: 35, 336496: 19, 336543: 13, 336622: 71, 336706: 48, 336717: 79, 336766: 57, 336800: 17, 336804: 16, 336816: 50, 336837: 80, 336901: 32, 336931: 2, 336933: 5, 337010: 12, 337048: 18, 337058: 25, 337130: 27, 337185: 36, 337269: 87, 337397: 46, 337427: 55, 337512: 31, 337696: 23, 337704: 43, 337742: 20, 337803: 68, 337968: 67, 337980: 54, 338057: 61, 338063: 89, 338099: 60, 338142: 78, 338150: 83, 338278: 96, 338285: 58, 338473: 37, 338496: 21, 338500: 76, 338653: 50, 338780: 15, 338826: 40, 338876: 41, 338903: 24, 339008: 64, 339028: 89, 339109: 89, 339126: 89, 339135: 77, 339158: 73, 339206: 12, 339221: 73, 339236: 42, 339253: 11, 339267: 59, 339454: 1, 339513: 35, 339596: 51, 339639: 43, 339663: 30, 339677: 2, 339738: 84, 339775: 0, 339802: 93, 339840: 28, 339846: 3, 339895: 30, 339906: 25, 339922: 70, 340021: 49, 340143: 39, 340179: 23, 340292: 97, 340445: 19, 340458: 12, 340468: 99, 340541: 67, 340558: 49, 340568: 24, 340616: 18, 340637: 33, 340684: 63, 340711: 30, 340745: 70, 340854: 86, 340887: 26, 340891: 56, 341001: 70, 341100: 96, 341186: 63, 341333: 53, 341356: 67, 341535: 26, 341559: 58, 341583: 64, 341602: 65, 341736: 44, 341887: 89, 341915: 27, 342005: 23, 342033: 21, 342045: 41, 342063: 92, 342067: 22, 342142: 18, 342171: 40, 342188: 37, 342232: 1, 342357: 80, 342396: 26, 342399: 75, 342522: 84, 342785: 61, 342865: 71, 342888: 66, 342892: 75, 342961: 23, 342993: 81, 343078: 26, 343186: 75, 343295: 26, 343339: 86, 343345: 90, 343424: 97, 343451: 17, 343458: 92, 343502: 24, 343647: 8, 343954: 95, 343974: 74, 344219: 38, 344361: 94, 344464: 50, 344515: 35, 344578: 99, 344618: 46, 344621: 83, 344626: 84, 344674: 70, 344679: 91, 344723: 14, 344724: 7, 344778: 64, 345000: 52, 345166: 71, 345174: 25, 345359: 71, 345548: 50, 345652: 39, 345712: 98, 345990: 18, 346041: 17, 346134: 1, 346186: 48, 346281: 64, 346446: 84, 346570: 92, 346573: 24, 346582: 82, 346632: 84, 346730: 31, 346770: 12, 346798: 77, 346809: 94, 346821: 66, 346924: 35, 346942: 84, 347054: 70, 347075: 94, 347182: 53, 347236: 48, 347266: 11, 347314: 30, 347372: 62, 347382: 86, 347418: 58, 347510: 81, 347894: 70, 348071: 93, 348186: 74, 348287: 62, 348364: 49, 348477: 2, 348618: 66, 348667: 13, 348929: 45, 348939: 21, 348963: 8, 348992: 98, 349073: 66, 349075: 54, 349077: 75, 349320: 51, 349340: 73, 349376: 85, 349423: 15, 349472: 46, 349510: 33, 349530: 27, 349703: 41, 349742: 62, 349750: 69, 349772: 80, 349788: 22, 349842: 98, 349877: 66, 349908: 40, 349913: 70, 349916: 35, 349966: 67, 350038: 65, 350083: 97, 350146: 88, 350173: 27, 350279: 58, 350337: 33, 350352: 80, 350464: 21, 350583: 63, 350586: 59, 350597: 13, 350616: 33, 350667: 26, 350677: 27, 350681: 70, 350779: 41, 350786: 28, 350960: 81, 350970: 28, 350979: 60, 351040: 50, 351105: 16, 351193: 85, 351334: 41, 351336: 64, 351353: 3, 351389: 33, 351406: 56, 351424: 59, 351472: 94, 351484: 71, 351501: 84, 351552: 12, 351573: 38, 351727: 17, 351762: 14, 351769: 18, 351889: 57, 351975: 30, 351984: 42, 352038: 38, 352047: 61, 352137: 89, 352145: 80, 352253: 54, 352428: 72, 352598: 9, 352617: 37, 352628: 17, 352675: 80, 352696: 39, 352699: 85, 352803: 72, 352846: 94, 352849: 66, 352860: 56, 352915: 67, 352964: 70, 353026: 93, 353055: 69, 353155: 69, 353308: 85, 353408: 8, 353540: 65, 353544: 42, 353556: 47, 353737: 44, 353822: 86, 353838: 79, 353882: 40, 353980: 67, 353998: 93, 354053: 44, 354058: 20, 354065: 6, 354102: 52, 354108: 39, 354352: 0, 354359: 2, 354381: 56, 354407: 60, 354547: 64, 354574: 33, 354637: 29, 354711: 27, 354796: 62, 354817: 2, 354855: 21, 355067: 94, 355200: 79, 355212: 69, 355310: 86, 355388: 93, 355420: 10, 355434: 90, 355516: 89, 355629: 62, 355772: 60, 355789: 83, 355831: 59, 355850: 11, 355868: 13, 356003: 81, 356031: 93, 356152: 21, 356204: 86, 356213: 90, 356297: 95, 356352: 63, 356394: 17, 356457: 5, 356486: 59, 356561: 25, 356577: 61, 356654: 52, 356668: 93, 356726: 72, 356783: 27, 356856: 34, 356894: 16, 356915: 49, 356928: 81, 357069: 63, 357074: 37, 357131: 18, 357189: 5, 357243: 10, 357263: 0, 357265: 61, 357307: 96, 357412: 2, 357508: 45, 357548: 54, 357578: 34, 357584: 12, 357720: 35, 357880: 30, 357958: 4, 357989: 8, 358019: 78, 358100: 13, 358183: 7, 358238: 18, 358274: 8, 358414: 41, 358431: 37, 358443: 54, 358550: 79, 358578: 86, 358588: 83, 358687: 58, 358785: 46, 358888: 76, 358914: 19, 359018: 53, 359179: 54, 359191: 90, 359316: 47, 359330: 55, 359346: 47, 359467: 47, 359508: 97, 359515: 68, 359701: 81, 359703: 10, 359718: 37, 359841: 15, 359913: 90, 359933: 2, 360011: 21, 360053: 85, 360055: 59, 360132: 15, 360183: 68, 360184: 31, 360194: 76, 360206: 86, 360279: 9, 360299: 16, 360316: 57, 360337: 27, 360357: 88, 360443: 33, 360522: 80, 360535: 0, 360580: 91, 360583: 72, 360632: 89, 360646: 1, 360732: 72, 360783: 83, 360827: 49, 360893: 22, 360920: 63, 360997: 33, 361001: 68, 361119: 0, 361159: 94, 361255: 5, 361274: 56, 361330: 60, 361334: 78, 361335: 57, 361356: 60, 361365: 24, 361406: 40, 361411: 44, 361439: 32, 361446: 51, 361472: 98, 361495: 0, 361713: 85, 361800: 79, 361834: 32, 361841: 70, 361887: 75, 362020: 24, 362160: 69, 362182: 81, 362240: 42, 362283: 78, 362339: 33, 362353: 80, 362400: 15, 362473: 88, 362474: 24, 362515: 13, 362536: 86, 362555: 71, 362565: 3, 362767: 44, 362881: 33, 362885: 72, 362935: 32, 362986: 91, 363146: 7, 363210: 41, 363215: 64, 363261: 38, 363302: 99, 363307: 28, 363432: 35, 363537: 92, 363544: 69, 363714: 89, 363737: 11, 363788: 42, 363802: 28, 363821: 86, 363979: 37, 364029: 85, 364134: 33, 364215: 50, 364253: 42, 364265: 48, 364289: 35, 364290: 27, 364314: 98, 364372: 70, 364446: 21, 364450: 60, 364498: 3, 364522: 57, 364671: 58, 364703: 21, 364711: 24, 364877: 74, 365030: 89, 365059: 77, 365116: 55, 365213: 18, 365316: 69, 365512: 84, 365554: 79, 365561: 67, 365622: 6, 365839: 31, 365867: 22, 365879: 9, 365910: 71, 366020: 17, 366030: 28, 366103: 12, 366112: 42, 366146: 18, 366189: 49, 366344: 73, 366534: 35, 366539: 86, 366541: 59, 366569: 59, 366598: 60, 366611: 51, 366725: 44, 366779: 22, 366799: 8, 366869: 51, 366929: 6, 366954: 79, 367002: 96, 367061: 21, 367097: 44, 367110: 18, 367133: 49, 367195: 74, 367317: 0, 367408: 61, 367617: 44, 367644: 19, 367658: 63, 367738: 84, 367799: 14, 367822: 39, 367886: 90, 367910: 80, 368004: 54, 368036: 0, 368098: 62, 368254: 89, 368307: 12, 368374: 94, 368431: 5, 368495: 55, 368496: 84, 368647: 22, 368670: 64, 368694: 50, 368789: 84, 368831: 8, 368895: 62, 368923: 30, 369031: 8, 369067: 30, 369139: 78, 369157: 2, 369262: 92, 369326: 36, 369496: 17, 369518: 51, 369664: 59, 369666: 2, 369678: 97, 369740: 98, 369790: 91, 369820: 25, 369854: 15, 370055: 98, 370178: 1, 370214: 7, 370240: 48, 370256: 8, 370309: 87, 370310: 68, 370341: 86, 370352: 0, 370372: 51, 370394: 27, 370677: 98, 370884: 85, 371153: 40, 371188: 61, 371266: 95, 371338: 3, 371396: 32, 371463: 48, 371499: 72, 371774: 30, 372050: 71, 372082: 24, 372140: 83, 372142: 90, 372350: 6, 372393: 71, 372411: 4, 372484: 83, 372578: 84, 372595: 3, 372683: 79, 372708: 21, 372742: 59, 372814: 49, 372863: 55, 372883: 5, 372954: 77, 373027: 92, 373088: 63, 373097: 47, 373120: 84, 373171: 48, 373198: 24, 373207: 37, 373216: 76, 373360: 46, 373615: 33, 373628: 71, 373634: 99, 373758: 59, 373841: 30, 373853: 96, 373939: 6, 373975: 38, 373991: 9, 374062: 76, 374149: 17, 374201: 63, 374251: 80, 374292: 67, 374321: 25, 374391: 56, 374497: 68, 374548: 76, 374691: 27, 374717: 8, 374727: 36, 374731: 36, 374749: 64, 374757: 19, 374759: 99, 374827: 60, 374846: 61, 374862: 84, 374917: 84, 374984: 92, 375075: 8, 375111: 39, 375297: 77, 375299: 18, 375405: 94, 375541: 80, 375542: 21, 375696: 51, 375905: 92, 375955: 41, 375973: 73, 376008: 84, 376061: 47, 376107: 57, 376126: 2, 376269: 66, 376272: 35, 376373: 55, 376419: 73, 376517: 94, 376587: 60, 376649: 90, 376953: 81, 376957: 94, 377119: 83, 377158: 71, 377209: 14, 377301: 49, 377371: 43, 377380: 75, 377399: 0, 377468: 71, 377498: 47, 377580: 47, 377606: 55, 377627: 77, 377987: 70, 378151: 82, 378186: 91, 378294: 49, 378383: 27, 378389: 75, 378504: 29, 378529: 93, 378554: 13, 378622: 27, 378651: 9, 378862: 63, 378931: 98, 378966: 69, 379003: 42, 379007: 16, 379021: 13, 379095: 58, 379166: 88, 379198: 65, 379234: 70, 379290: 84, 379588: 35, 379702: 70, 379730: 30, 379787: 74, 379796: 60, 379844: 37, 379860: 71, 379996: 49, 380303: 84, 380328: 5, 380544: 49, 380553: 14, 380558: 84, 380727: 52, 380732: 99, 380769: 76, 380921: 74, 381014: 69, 381090: 34, 381148: 8, 381182: 21, 381214: 58, 381257: 34, 381262: 20, 381382: 12, 381412: 38, 381456: 4, 381489: 48, 381519: 87, 381539: 93, 381549: 73, 381555: 17, 381645: 45, 381733: 44, 381838: 55, 381857: 45, 381958: 65, 382021: 9, 382042: 66, 382128: 5, 382159: 8, 382178: 82, 382207: 84, 382309: 39, 382394: 91, 382410: 52, 382434: 88, 382443: 18, 382473: 51, 382564: 1, 382576: 44, 382605: 35, 382745: 65, 382815: 47, 383065: 19, 383102: 46, 383146: 13, 383227: 89, 383276: 38, 383379: 0, 383454: 57, 383492: 49, 383503: 49, 383567: 13, 383584: 15, 383611: 22, 383614: 61, 383615: 68, 383824: 31, 384012: 26, 384019: 83, 384040: 1, 384231: 64, 384350: 15, 384363: 45, 384471: 48, 384600: 0, 384635: 79, 384682: 38, 384698: 84, 384746: 94, 384851: 63, 384872: 24, 384905: 75, 384926: 22, 385074: 4, 385211: 56, 385269: 48, 385307: 9, 385361: 12, 385457: 77, 385468: 41, 385523: 23, 385677: 81, 385690: 27, 385740: 81, 385790: 64, 385846: 79, 385854: 58, 385967: 15, 386109: 5, 386122: 13, 386130: 91, 386141: 68, 386185: 48, 386193: 62, 386204: 89, 386254: 52, 386388: 22, 386463: 55, 386493: 12, 386541: 4, 386558: 70, 386617: 25, 386620: 68, 386905: 51, 386943: 57, 386980: 7, 387051: 79, 387060: 0, 387104: 31, 387198: 74, 387244: 90, 387309: 27, 387316: 63, 387621: 11, 387673: 12, 387724: 58, 387773: 43, 387823: 82, 387877: 68, 388000: 10, 388003: 50, 388011: 97, 388027: 49, 388030: 59, 388115: 68, 388156: 41, 388226: 45, 388263: 94, 388278: 62, 388326: 0, 388500: 44, 388570: 76, 388584: 89, 388590: 2, 388609: 89, 388685: 61, 388730: 87, 388822: 13, 388839: 98, 388947: 86, 388970: 85, 389319: 57, 389358: 5, 389380: 38, 389384: 54, 389410: 36, 389413: 32, 389471: 11, 389613: 77, 389645: 80, 389705: 74, 389717: 4, 389808: 55, 389818: 34, 389858: 94, 389883: 72, 389973: 72, 390026: 46, 390076: 56, 390222: 96, 390252: 48, 390373: 90, 390598: 38, 390899: 50, 390921: 13, 390948: 67, 390958: 29, 390968: 41, 391106: 18, 391188: 29, 391197: 40, 391214: 37, 391251: 55, 391376: 40, 391384: 34, 391388: 38, 391416: 12, 391459: 92, 391555: 78, 391572: 92, 391630: 70, 391647: 32, 391727: 23, 391756: 49, 391942: 67, 391995: 8, 392016: 77, 392118: 0, 392343: 30, 392371: 3, 392404: 30, 392482: 0, 392617: 87, 392737: 2, 392830: 79, 392893: 48, 392952: 97, 392966: 68, 393019: 33, 393110: 15, 393129: 71, 393172: 70, 393220: 77, 393235: 18, 393249: 63, 393341: 74, 393344: 87, 393368: 33, 393551: 4, 393556: 3, 393721: 53, 393763: 7, 393836: 22, 393887: 83, 393968: 29, 394015: 8, 394027: 96, 394091: 26, 394175: 13, 394240: 65, 394491: 40, 394537: 4, 394574: 67, 394675: 30, 394762: 1, 394776: 84, 394815: 32, 394899: 21, 394917: 79, 395006: 10, 395010: 94, 395130: 38, 395290: 7, 395323: 83, 395379: 26, 395409: 51, 395414: 31, 395517: 42, 395528: 4, 395579: 67, 395600: 71, 395601: 32, 395611: 71, 395631: 96, 395636: 97, 395784: 10, 395821: 77, 395964: 44, 396000: 38, 396081: 53, 396143: 34, 396213: 40, 396231: 75, 396319: 41, 396327: 6, 396467: 95, 396513: 71, 396565: 41, 396588: 59, 396596: 57, 396681: 60, 396826: 22, 396914: 59, 396953: 59, 396975: 3, 397115: 98, 397130: 19, 397165: 37, 397178: 28, 397304: 29, 397334: 85, 397396: 8, 397672: 81, 397741: 85, 397876: 20, 397885: 8, 397902: 10, 398056: 30, 398137: 92, 398373: 13, 398451: 70, 398463: 72, 398517: 87, 398585: 96, 398607: 83, 398647: 95, 398704: 11, 398774: 80, 398869: 97, 398907: 36, 398910: 59, 398931: 28, 399204: 65, 399363: 10, 399405: 49, 399424: 6, 399429: 57, 399497: 64, 399544: 71, 399558: 17, 399561: 70, 399577: 86, 399578: 47, 399580: 97, 399620: 1, 399641: 16, 399886: 3, 400002: 65, 400142: 54, 400226: 88, 400325: 61, 400361: 95, 400380: 4, 400407: 9, 400433: 0, 400441: 51, 400470: 18, 400505: 3, 400518: 16, 400551: 12, 400704: 82, 400721: 53, 400729: 61, 400779: 76, 400900: 22, 400916: 61, 401084: 39, 401138: 9, 401180: 83, 401226: 42, 401312: 83, 401314: 33, 401440: 19, 401457: 42, 401466: 14, 401489: 87, 401504: 80, 401574: 15, 401604: 5, 401626: 1, 401770: 65, 401789: 89, 401796: 19, 401837: 3, 401863: 71, 401891: 55, 402077: 61, 402125: 92, 402241: 78, 402301: 96, 402312: 66, 402405: 15, 402421: 99, 402517: 69, 402526: 49, 402573: 0, 402616: 1, 402742: 40, 402948: 96, 402959: 45, 402996: 42, 403027: 82, 403034: 62, 403041: 23, 403112: 88, 403189: 71, 403190: 48, 403245: 75, 403302: 34, 403328: 69, 403336: 17, 403396: 86, 403514: 76, 403638: 15, 403700: 76, 403740: 68, 403747: 56, 403814: 82, 403821: 39, 403907: 97, 403908: 30, 403968: 12, 404021: 24, 404026: 45, 404059: 72, 404088: 66, 404133: 46, 404271: 27, 404278: 35, 404284: 87, 404296: 40, 404384: 90, 404468: 8, 404487: 77, 404570: 47, 404701: 81, 404744: 23, 404756: 82, 404770: 45, 404931: 84, 405161: 84, 405238: 50, 405267: 74, 405297: 17, 405334: 4, 405639: 40, 405844: 28, 405884: 42, 405890: 21, 405951: 99, 406058: 53, 406063: 87, 406138: 93, 406169: 89, 406200: 56, 406215: 18, 406280: 92, 406344: 35, 406345: 12, 406353: 76, 406421: 83, 406462: 60, 406635: 96, 406656: 78, 406667: 34, 406779: 85, 406794: 23, 406822: 79, 406867: 47, 406934: 5, 406962: 12, 406997: 75, 407086: 87, 407132: 60, 407163: 39, 407287: 67, 407347: 68, 407378: 64, 407486: 67, 407654: 91, 407680: 98, 407890: 31, 408058: 14, 408078: 92, 408197: 11, 408294: 0, 408425: 91, 408574: 18, 408630: 77, 408673: 55, 408849: 14, 408866: 1, 408965: 73, 409000: 92, 409204: 14, 409230: 64, 409236: 55, 409339: 68, 409460: 91, 409581: 66, 409604: 24, 409633: 86, 409697: 37, 409720: 91, 409881: 98, 409941: 59, 409948: 49, 410175: 45, 410184: 22, 410229: 99, 410245: 38, 410265: 41, 410272: 42, 410288: 41, 410408: 11, 410546: 76, 410591: 49, 410604: 9, 410812: 70, 410817: 17, 410873: 88, 410876: 86, 410893: 58, 410957: 34, 410976: 56, 411039: 7, 411078: 90, 411083: 61, 411157: 45, 411285: 87, 411309: 74, 411599: 39, 411813: 93, 411845: 94, 412059: 22, 412062: 69, 412097: 5, 412159: 78, 412308: 91, 412377: 0, 412386: 72, 412421: 48, 412471: 13, 412502: 33, 412506: 69, 412529: 5, 412566: 36, 412570: 28, 412586: 83, 412649: 3, 412652: 13, 412670: 80, 412688: 42, 412712: 47, 412773: 28, 412786: 56, 412802: 69, 412846: 45, 412876: 76, 412893: 55, 412921: 14, 412937: 16, 413042: 70, 413053: 74, 413062: 47, 413596: 23, 413685: 71, 413755: 70, 413785: 30, 413814: 68, 413961: 64, 413993: 24, 414003: 92, 414054: 27, 414209: 20, 414219: 46, 414324: 81, 414331: 48, 414436: 7, 414521: 70, 414543: 73, 414819: 4, 414855: 1, 415022: 82, 415162: 72, 415189: 98, 415195: 83, 415222: 58, 415260: 42, 415329: 39, 415355: 40, 415396: 31, 415456: 40, 415612: 36, 415648: 22, 415735: 3, 415826: 30, 415891: 19, 415926: 37, 416060: 46, 416075: 48, 416090: 58, 416153: 44, 416249: 91, 416766: 9, 416879: 92, 416916: 55, 416940: 33, 416955: 95, 417000: 32, 417057: 66, 417160: 11, 417249: 31, 417293: 22, 417370: 12, 417378: 19, 417392: 70, 417493: 42, 417506: 86, 417553: 68, 417779: 9, 417822: 39, 417836: 90, 417949: 74, 418002: 33, 418005: 10, 418014: 67, 418124: 63, 418229: 30, 418335: 14, 418336: 3, 418443: 79, 418565: 7, 418588: 36, 418639: 82, 418690: 84, 418788: 67, 418793: 68, 418865: 15, 418892: 42, 418910: 97, 418918: 2, 418999: 54, 419085: 51, 419121: 50, 419159: 23, 419166: 40, 419184: 30, 419221: 96, 419302: 45, 419371: 35, 419413: 22, 419448: 8, 419534: 59, 419619: 81, 419635: 4, 419678: 50, 419696: 16, 419731: 22, 419735: 75, 419861: 47, 419994: 80, 420008: 66, 420070: 50, 420078: 57, 420086: 83, 420214: 54, 420231: 85, 420239: 20, 420400: 64, 420439: 23, 420447: 29, 420466: 21, 420775: 50, 420781: 46, 420815: 44, 420847: 4, 420955: 17, 421009: 32, 421094: 57, 421095: 76, 421226: 49, 421253: 92, 421308: 57, 421331: 31, 421345: 44, 421372: 63, 421373: 63, 421385: 80, 421389: 65, 421447: 73, 421484: 70, 421533: 8, 421542: 39, 421593: 68, 421605: 56, 421652: 52, 421655: 83, 421770: 18, 421818: 73, 422050: 40, 422090: 25, 422187: 99, 422354: 13, 422424: 84, 422439: 87, 422533: 87, 422536: 39, 422573: 2, 422605: 34, 422670: 71, 422699: 0, 422701: 67, 422884: 14, 422915: 6, 423147: 4, 423195: 99, 423245: 22, 423273: 20, 423289: 17, 423376: 10, 423646: 58, 423712: 88, 423765: 15, 423812: 70, 423831: 49, 423950: 89, 424069: 11, 424078: 26, 424086: 47, 424130: 92, 424169: 17, 424255: 95, 424307: 47, 424351: 11, 424437: 42, 424640: 89, 424724: 84, 424914: 50, 424934: 7, 425070: 47, 425073: 68, 425137: 79, 425228: 85, 425370: 66, 425434: 37, 425469: 29, 425474: 85, 425507: 30, 425514: 77, 425529: 94, 425567: 18, 425745: 32, 425748: 17, 425755: 18, 425810: 18, 425856: 8, 425964: 99, 426043: 5, 426124: 10, 426168: 92, 426169: 81, 426358: 13, 426365: 43, 426394: 13, 426461: 33, 426469: 10, 426520: 61, 426693: 72, 426705: 56, 426710: 80, 426711: 40, 426844: 88, 426858: 66, 426930: 64, 427017: 1, 427145: 73, 427273: 10, 427297: 28, 427414: 86, 427427: 60, 427575: 98, 427587: 78, 427675: 1, 427698: 87, 427710: 48, 427776: 65, 427887: 16, 428125: 85, 428136: 65, 428158: 13, 428226: 71, 428276: 81, 428283: 30, 428339: 23, 428360: 11, 428397: 51, 428453: 54, 428478: 35, 428523: 96, 428561: 4, 428577: 59, 428638: 70, 428802: 25, 428810: 96, 428857: 1, 428991: 30, 429026: 2, 429064: 26, 429081: 71, 429191: 71, 429278: 93, 429301: 73, 429329: 95, 429382: 89, 429397: 5, 429477: 87, 429534: 57, 429585: 28, 429601: 11, 429710: 19, 429778: 93, 429818: 37, 429944: 27, 429955: 28, 429980: 46, 430009: 55, 430039: 53, 430095: 19, 430111: 45, 430235: 54, 430255: 63, 430261: 11, 430585: 10, 430655: 42, 430787: 31, 430867: 86, 430871: 19, 430882: 14, 430913: 37, 431244: 2, 431260: 13, 431345: 81, 431555: 11, 431568: 63, 431601: 9, 431729: 47, 431829: 73, 431874: 36, 431939: 4, 432019: 57, 432099: 43, 432348: 89, 432360: 84, 432370: 56, 432377: 13, 432610: 44, 432705: 17, 432953: 1, 432974: 94, 432976: 66, 433082: 33, 433570: 48, 433624: 22, 433669: 98, 433746: 78, 433854: 11, 433878: 9, 433908: 3, 434048: 30, 434200: 59, 434235: 99, 434443: 94, 434631: 83, 434639: 98, 434675: 33, 434681: 72, 434707: 42, 434733: 15, 434755: 79, 434776: 33, 434780: 78, 434850: 53, 434853: 23, 434872: 40, 434963: 84, 435016: 23, 435020: 97, 435184: 93, 435490: 5, 435540: 59, 435652: 31, 435803: 91, 435962: 41, 435966: 91, 436026: 24, 436351: 68, 436437: 54, 436499: 85, 436502: 92, 436601: 36, 436602: 79, 436603: 44, 436613: 95, 436622: 57, 436703: 13, 436794: 57, 437095: 12, 437143: 37, 437315: 4, 437338: 81, 437449: 79, 437632: 85, 437774: 83, 437855: 76, 437892: 61, 437932: 28, 438116: 14, 438325: 76, 438380: 79, 438506: 67, 438600: 70, 438613: 3, 438645: 76, 438651: 75, 438663: 53, 438664: 42, 438696: 27, 438782: 90, 438862: 97, 438881: 45, 438883: 72, 438906: 78, 439168: 40, 439214: 85, 439272: 76, 439287: 87, 439310: 26, 439341: 79, 439380: 92, 439446: 94, 439458: 3, 439522: 59, 439583: 48, 439595: 83, 439624: 74, 439638: 17, 439683: 17, 439697: 45, 439848: 14, 439924: 42, 439928: 92, 439937: 57, 439998: 35, 440001: 53, 440043: 28, 440058: 87, 440074: 32, 440160: 98, 440243: 59, 440258: 25, 440331: 44, 440371: 4, 440513: 18, 440549: 78, 440604: 30, 440654: 41, 440672: 51, 440722: 26, 440916: 35, 441054: 94, 441110: 91, 441121: 42, 441227: 85, 441344: 13, 441368: 99, 441449: 62, 441495: 37, 441500: 69, 441609: 5, 441617: 74, 441685: 22, 441703: 50, 441834: 57, 441895: 72, 442074: 72, 442080: 3, 442082: 10, 442139: 98, 442247: 90, 442363: 73, 442381: 86, 442429: 90, 442435: 68, 442591: 11, 442653: 20, 442698: 11, 442721: 93, 442820: 81, 442836: 19, 442882: 17, 442916: 87, 443032: 79, 443084: 86, 443093: 0, 443134: 53, 443200: 23, 443207: 71, 443269: 18, 443321: 67, 443419: 47, 443571: 1, 443617: 87, 443624: 91, 443630: 34, 443680: 33, 443881: 76, 443982: 42, 444139: 64, 444243: 72, 444264: 87, 444463: 95, 444597: 24, 444679: 82, 444730: 88, 444907: 91, 445083: 9, 445152: 8, 445216: 39, 445237: 51, 445249: 61, 445251: 69, 445265: 1, 445282: 33, 445358: 48, 445532: 17, 445543: 93, 445595: 74, 445678: 90, 445788: 32, 445802: 35, 445838: 51, 445845: 13, 445907: 65, 446002: 27, 446112: 40, 446132: 11, 446330: 65, 446344: 42, 446469: 73, 446476: 78, 446592: 15, 446626: 38, 446710: 98, 446729: 98, 446736: 74, 446768: 90, 446979: 77, 447096: 77, 447303: 17, 447373: 29, 447596: 14, 447677: 40, 447799: 87, 447815: 84, 447818: 84, 447971: 45, 448001: 69, 448140: 47, 448190: 27, 448224: 51, 448226: 54, 448259: 26, 448309: 50, 448337: 68, 448379: 96, 448424: 75, 448443: 0, 448478: 8, 448480: 80, 448512: 34, 448607: 33, 448640: 5, 448645: 5, 448677: 30, 448834: 70, 448936: 64, 449082: 37, 449096: 44, 449099: 29, 449216: 3, 449262: 15, 449327: 25, 449343: 61, 449376: 94, 449473: 7, 449508: 80, 449660: 91, 449666: 55, 449687: 78, 449691: 10, 449706: 97, 449724: 42, 449868: 76, 449921: 64, 450051: 28, 450246: 56, 450256: 69, 450373: 81, 450390: 84, 450394: 71, 450432: 56, 450461: 23, 450482: 26, 450493: 82, 450511: 91, 450522: 75, 450628: 48, 450658: 70, 450659: 61, 450661: 24, 450748: 16, 450795: 29, 450807: 3, 450826: 84, 450871: 16, 450939: 94, 451021: 40, 451055: 31, 451062: 18, 451087: 45, 451164: 66, 451166: 33, 451249: 37, 451303: 39, 451336: 31, 451394: 46, 451515: 52, 451665: 7, 451752: 67, 451757: 96, 451820: 71, 451909: 74, 451928: 19, 451938: 20, 452014: 49, 452088: 40, 452104: 7, 452260: 26, 452299: 80, 452419: 4, 452421: 7, 452559: 99, 452624: 29, 452707: 55, 452794: 66, 452904: 58, 453008: 70, 453159: 99, 453183: 76, 453225: 23, 453257: 39, 453264: 77, 453327: 32, 453349: 32, 453386: 61, 453420: 73, 453564: 49, 453584: 54, 453591: 43, 453669: 82, 453676: 35, 453722: 77, 453728: 63, 453768: 7, 453837: 45, 453888: 14, 453895: 52, 453916: 72, 453988: 81, 454137: 26, 454197: 93, 454250: 97, 454258: 46, 454324: 59, 454338: 67, 454362: 2, 454447: 62, 454505: 77, 454563: 20, 454658: 3, 454801: 49, 454806: 60, 454836: 90, 454908: 35, 454990: 85, 455130: 86, 455190: 35, 455244: 6, 455258: 90, 455297: 18, 455388: 44, 455422: 27, 455435: 38, 455436: 27, 455484: 45, 455585: 26, 455600: 79, 455657: 61, 455994: 67, 456040: 51, 456054: 99, 456078: 66, 456138: 85, 456242: 48, 456267: 76, 456353: 68, 456374: 75, 456406: 72, 456413: 33, 456523: 90, 456537: 95, 456541: 71, 456609: 70, 456632: 74, 456661: 70, 456664: 86, 456690: 55, 456702: 55, 456704: 97, 456727: 27, 456763: 96, 456826: 95, 456861: 65, 456899: 7, 456974: 3, 456980: 70, 457018: 74, 457042: 65, 457066: 94, 457181: 53, 457238: 7, 457321: 1, 457383: 69, 457406: 44, 457418: 31, 457454: 64, 457469: 33, 457675: 9, 457678: 7, 457704: 96, 457732: 34, 457738: 32, 457858: 56, 457932: 90, 458004: 88, 458114: 92, 458251: 68, 458290: 72, 458491: 72, 458599: 64, 458607: 61, 458663: 64, 458714: 85, 458723: 49, 458746: 91, 458771: 90, 458898: 18, 459047: 56, 459157: 21, 459182: 32, 459221: 12, 459230: 60, 459457: 50, 459536: 10, 459596: 1, 459968: 32, 459974: 40, 460039: 87, 460156: 88, 460194: 61, 460195: 98, 460222: 91, 460268: 29, 460410: 51, 460487: 9, 460529: 5, 460543: 94, 460730: 74, 460735: 44, 460747: 3, 460787: 24, 460827: 68, 460905: 75, 460965: 70, 460968: 93, 461123: 24, 461261: 36, 461585: 70, 461598: 78, 461675: 32, 461678: 54, 461723: 7, 462031: 58, 462062: 43, 462073: 57, 462123: 69, 462348: 67, 462546: 18, 462569: 46, 462672: 47, 462768: 3, 462881: 56, 462884: 5, 462888: 42, 462889: 2, 462936: 8, 463042: 13, 463124: 85, 463184: 71, 463299: 88, 463317: 69, 463324: 49, 463403: 29, 463417: 23, 463505: 76, 463509: 31, 463566: 11, 463599: 82, 463711: 16, 463895: 36, 463999: 89, 464053: 2, 464054: 6, 464085: 51, 464101: 62, 464140: 82, 464225: 81, 464373: 45, 464397: 34, 464421: 28, 464509: 75, 464575: 10, 464650: 27, 464853: 39, 464922: 71, 464994: 48, 465212: 44, 465343: 6, 465411: 2, 465441: 74, 465457: 93, 465566: 53, 465605: 87, 465613: 93, 465682: 0, 465699: 53, 465721: 4, 465777: 26, 465784: 11, 465803: 60, 465904: 53, 465948: 78, 465993: 63, 465998: 93, 466048: 92, 466128: 46, 466148: 90, 466190: 32, 466338: 21, 466367: 15, 466454: 31, 466486: 70, 466633: 42, 466748: 39, 466762: 10, 466769: 2, 466809: 36, 466819: 47, 467027: 82, 467034: 4, 467051: 40, 467079: 12, 467138: 79, 467139: 26, 467268: 31, 467318: 61, 467397: 80, 467406: 30, 467425: 40, 467454: 1, 467547: 18, 467584: 78, 467585: 83, 467663: 3, 467705: 55, 467790: 74, 467836: 31, 467842: 62, 467848: 60, 467916: 68, 468004: 42, 468018: 78, 468090: 88, 468100: 0, 468127: 11, 468183: 25, 468203: 37, 468206: 40, 468276: 42, 468377: 90, 468394: 20, 468407: 52, 468608: 34, 468680: 23, 468744: 30, 468791: 88, 468796: 47, 468817: 50, 468940: 44, 468960: 93, 468986: 84, 469011: 1, 469018: 67, 469071: 86, 469143: 58, 469392: 99, 469432: 89, 469458: 73, 469463: 38, 469470: 47, 469475: 76, 469482: 2, 469508: 4, 469544: 60, 469567: 26, 469734: 59, 469741: 82, 469913: 6, 469995: 79, 470363: 37, 470406: 11, 470492: 89, 470576: 78, 470578: 43, 470664: 84, 470669: 25, 470695: 8, 470699: 91, 470708: 25, 470784: 41, 471082: 88, 471112: 58, 471150: 94, 471250: 80, 471307: 62, 471398: 21, 471416: 74, 471515: 94, 471705: 41, 471724: 75, 471762: 13, 471769: 1, 471818: 81, 471827: 19, 471871: 57, 471895: 18, 471906: 27, 471996: 24, 472075: 87, 472098: 96, 472170: 54, 472242: 19, 472322: 23, 472327: 45, 472362: 44, 472421: 46, 472476: 92, 472543: 42, 472561: 35, 472655: 62, 472735: 19, 472775: 89, 472784: 3, 472833: 84, 472910: 50, 472923: 32, 472964: 60, 473001: 76, 473002: 9, 473019: 90, 473059: 68, 473214: 80, 473285: 11, 473319: 66, 473320: 29, 473668: 2, 473701: 97, 473915: 49, 474013: 69, 474023: 10, 474081: 86, 474127: 59, 474153: 36, 474177: 10, 474178: 46, 474214: 84, 474381: 35, 474428: 95, 474433: 86, 474484: 79, 474540: 33, 474603: 7, 474616: 25, 474768: 66, 474784: 21, 474883: 43, 474988: 51, 475096: 46, 475277: 30, 475411: 18, 475413: 6, 475502: 72, 475696: 66, 475732: 62, 475758: 84, 475790: 3, 475801: 98, 475940: 29, 476002: 67, 476176: 76, 476345: 58, 476356: 81, 476675: 3, 476678: 69, 476750: 20, 476853: 3, 476897: 87, 476944: 16, 476967: 29, 477017: 78, 477111: 19, 477150: 90, 477206: 48, 477297: 12, 477304: 57, 477428: 60, 477441: 26, 477671: 22, 477722: 59, 477782: 30, 477788: 78, 477806: 61, 477845: 76, 477903: 47, 477969: 14, 478153: 12, 478416: 63, 478436: 94, 478591: 91, 478625: 14, 478692: 49, 478752: 71, 478808: 98, 478843: 33, 478897: 11, 478921: 21, 479031: 36, 479297: 62, 479481: 70, 479519: 64, 479528: 94, 479592: 28, 479697: 48, 479745: 70, 479850: 8, 479993: 90, 479999: 18, 480025: 74, 480046: 11, 480106: 87, 480124: 76, 480143: 96, 480220: 90, 480243: 6, 480361: 32, 480466: 90, 480510: 25, 480514: 5, 480516: 35, 480626: 41, 480629: 0, 480644: 83, 480676: 85, 480764: 47, 480882: 5, 480929: 44, 481103: 41, 481162: 97, 481174: 66, 481202: 11, 481302: 0, 481450: 85, 481520: 40, 481617: 86, 481653: 29, 481678: 66, 481755: 87, 481829: 49, 481842: 82, 481890: 78, 481920: 21, 481943: 0, 481989: 88, 482070: 67, 482081: 21, 482221: 84, 482223: 71, 482241: 99, 482263: 58, 482376: 89, 482385: 16, 482443: 9, 482489: 75, 482500: 23, 482644: 99, 482647: 30, 482743: 1, 482754: 78, 482826: 8, 482838: 16, 482923: 9, 482936: 10, 482966: 7, 483272: 44, 483536: 60, 483613: 41, 483685: 39, 483706: 30, 483784: 53, 483865: 68, 483881: 15, 483918: 19, 483948: 5, 483971: 22, 484083: 87, 484175: 40, 484323: 50, 484333: 96, 484397: 56, 484443: 0, 484752: 28, 484803: 0, 484804: 16, 484879: 78, 484913: 91, 484926: 83, 485042: 73, 485065: 69, 485130: 66, 485258: 78, 485270: 45, 485352: 45, 485501: 2, 485639: 54, 485650: 55, 485739: 95, 485820: 43, 485873: 73, 485892: 53, 485915: 92, 486116: 67, 486133: 50, 486170: 29, 486206: 6, 486253: 61, 486260: 89, 486291: 33, 486491: 49, 486580: 2, 486591: 11, 486715: 13, 486731: 57, 486770: 98, 486889: 23, 486908: 92, 486924: 34, 487065: 46, 487211: 10, 487407: 53, 487430: 23, 487458: 65, 487511: 48, 487667: 75, 487705: 49, 487812: 70, 487861: 83, 487869: 76, 487913: 18, 487918: 71, 487933: 88, 487952: 64, 488016: 24, 488106: 10, 488119: 95, 488225: 93, 488230: 74, 488286: 12, 488389: 40, 488401: 57, 488474: 67, 488483: 61, 488494: 8, 488510: 48, 488533: 73, 488554: 18, 488584: 6, 488841: 82, 488876: 38, 489081: 14, 489147: 92, 489199: 26, 489244: 78, 489352: 18, 489473: 3, 489512: 37, 489518: 63, 489717: 69, 489741: 73, 489763: 13, 489810: 35, 489835: 26, 489917: 8, 489956: 60, 490044: 20, 490068: 92, 490142: 72, 490168: 15, 490196: 51, 490212: 40, 490214: 52, 490302: 88, 490316: 53, 490328: 26, 490402: 43, 490430: 75, 490492: 47, 490519: 22, 490564: 20, 490800: 62, 490889: 46, 490913: 22, 490935: 24, 490948: 83, 490991: 23, 491002: 38, 491084: 90, 491169: 45, 491258: 52, 491333: 17, 491525: 85, 491784: 17, 491789: 53, 491810: 77, 491848: 69, 491861: 99, 491952: 77, 491996: 10, 492037: 29, 492119: 51, 492131: 33, 492244: 18, 492269: 25, 492358: 44, 492368: 5, 492386: 87, 492444: 95, 492478: 15, 492482: 99, 492505: 17, 492736: 90, 492798: 25, 492833: 77, 492917: 14, 492921: 60, 493008: 1, 493079: 7, 493107: 49, 493124: 42, 493125: 91, 493145: 57, 493289: 93, 493505: 32, 493601: 64, 493611: 84, 493761: 95, 493900: 30, 493987: 14, 494013: 55, 494020: 73, 494077: 79, 494088: 29, 494154: 96, 494296: 63, 494366: 73, 494523: 67, 494533: 55, 494539: 3, 494637: 27, 494645: 19, 494687: 64, 494821: 42, 494995: 79, 495017: 64, 495035: 11, 495061: 10, 495206: 15, 495218: 81, 495248: 40, 495277: 44, 495334: 74, 495390: 19, 495412: 60, 495625: 81, 495780: 38, 495816: 96, 495929: 48, 496119: 41, 496235: 32, 496315: 85, 496337: 96, 496356: 27, 496619: 7, 496669: 61, 496681: 97, 496800: 47, 496804: 0, 496834: 91, 496868: 88, 496959: 43, 497126: 20, 497140: 48, 497266: 95, 497329: 71, 497330: 6, 497375: 11, 497452: 1, 497567: 49, 497620: 83, 497878: 71, 497907: 33, 498022: 37, 498040: 7, 498075: 77, 498314: 3, 498337: 27, 498345: 1, 498346: 58, 498644: 70, 498645: 98, 498742: 54, 498758: 33, 498792: 50, 498846: 47, 498853: 78, 498856: 67, 498920: 17, 498945: 53, 499061: 70, 499065: 53, 499094: 44, 499107: 99, 499295: 23, 499312: 67, 499492: 57, 499507: 95, 499565: 5, 499579: 79, 499626: 11, 499651: 92, 499922: 91, 499967: 11, 500054: 33, 500093: 76, 500130: 49, 500187: 42, 500198: 19, 500206: 58, 500254: 21, 500302: 4, 500406: 31, 500484: 50, 500497: 46, 500588: 48, 500600: 23, 500620: 37, 500684: 16, 500700: 91, 500723: 10, 500746: 95, 500750: 53, 500758: 59, 500821: 28, 500895: 52, 500955: 92, 501034: 23, 501135: 77, 501143: 54, 501219: 61, 501319: 14, 501343: 93, 501403: 28, 501412: 62, 501420: 97, 501569: 16, 501599: 3, 501648: 44, 501678: 19, 501715: 64, 501843: 29, 501883: 47, 501955: 1, 502185: 43, 502198: 9, 502305: 77, 502330: 53, 502433: 46, 502438: 17, 502460: 51, 502549: 63, 502629: 48, 502641: 97, 502654: 17, 502827: 0, 502900: 82, 502910: 16, 502932: 0, 503108: 91, 503180: 92, 503220: 45, 503231: 77, 503234: 70, 503410: 69, 503415: 21, 503614: 52, 503633: 72, 503668: 42, 503795: 65, 503951: 46, 503960: 53, 503990: 80, 504053: 13, 504064: 52, 504125: 74, 504130: 20, 504146: 99, 504187: 48, 504214: 39, 504231: 27, 504261: 8, 504279: 22, 504318: 48, 504359: 75, 504393: 52, 504653: 83, 504655: 32, 504705: 69, 504717: 40, 504901: 55, 505042: 63, 505181: 48, 505197: 51, 505230: 26, 505241: 2, 505315: 35, 505339: 70, 505345: 9, 505395: 78, 505477: 78, 505589: 47, 505644: 24, 505687: 72, 505829: 28, 505910: 86, 505962: 73, 506010: 74, 506049: 86, 506161: 83, 506271: 94, 506304: 65, 506351: 15, 506497: 60, 506701: 96, 507051: 14, 507065: 41, 507134: 74, 507140: 35, 507185: 80, 507199: 28, 507302: 17, 507332: 66, 507349: 50, 507363: 20, 507419: 16, 507429: 22, 507440: 92, 507454: 56, 507678: 73, 507681: 28, 507685: 55, 507760: 12, 507790: 92, 507815: 6, 507880: 93, 507931: 23, 507953: 94, 507992: 13, 508056: 84, 508134: 98, 508238: 86, 508240: 43, 508263: 35, 508354: 83, 508357: 6, 508381: 64, 508455: 25, 508463: 83, 508468: 6, 508622: 51, 508627: 65, 508708: 28, 508815: 16, 508862: 20, 508889: 62, 508892: 25, 508975: 8, 509081: 2, 509187: 73, 509188: 41, 509210: 49, 509260: 48, 509365: 25, 509387: 2, 509439: 83, 509445: 54, 509519: 94, 509573: 99, 509576: 91, 509711: 6, 509732: 97, 509796: 74, 509810: 42, 509846: 66, 509858: 42, 510080: 62, 510133: 97, 510154: 26, 510248: 45, 510306: 28, 510335: 95, 510345: 36, 510440: 92, 510462: 37, 510476: 89, 510481: 3, 510563: 89, 510572: 89, 510597: 51, 510863: 21, 510982: 23, 511051: 48, 511067: 42, 511330: 81, 511380: 79, 511414: 4, 511630: 71, 511646: 3, 511664: 88, 511678: 43, 511755: 90, 511977: 52, 511984: 84, 512138: 30, 512183: 26, 512221: 78, 512358: 66, 512413: 28, 512446: 52, 512454: 1, 512530: 81, 512532: 15, 512670: 93, 512683: 78, 512726: 20, 512856: 3, 512903: 89, 512932: 46, 512953: 17, 512965: 4, 512982: 15, 513039: 13, 513216: 82, 513242: 81, 513255: 75, 513390: 63, 513439: 98, 513576: 43, 513617: 69, 513625: 59, 513672: 59, 513688: 9, 514086: 22, 514131: 26, 514151: 39, 514357: 1, 514384: 81, 514487: 96, 514530: 40, 514571: 82, 514572: 9, 514656: 37, 514689: 96, 514737: 19, 514759: 61, 514820: 15, 514843: 84, 514853: 80, 515021: 17, 515052: 63, 515095: 96, 515102: 76, 515109: 96, 515128: 86, 515196: 94, 515229: 13, 515368: 90, 515508: 37, 515673: 26, 515806: 75, 515863: 13, 515891: 49, 516162: 4, 516216: 97, 516233: 61, 516234: 74, 516251: 39, 516264: 62, 516325: 9, 516326: 16, 516400: 55, 516573: 19, 516579: 10, 516656: 49, 516724: 21, 516736: 29, 516761: 89, 516823: 12, 516839: 27, 516853: 17, 516888: 10, 516909: 37, 517043: 70, 517149: 57, 517151: 92, 517165: 96, 517170: 41, 517249: 71, 517250: 80, 517287: 85, 517463: 10, 517468: 3, 517608: 24, 517665: 81, 517751: 22, 517870: 4, 517892: 17, 517934: 7, 518028: 88, 518050: 14, 518060: 56, 518092: 90, 518133: 85, 518219: 97, 518272: 70, 518398: 1, 518501: 79, 518513: 13, 518852: 31, 518977: 71, 519320: 8, 519371: 57, 519387: 62, 519423: 40, 519447: 74, 519482: 60, 519545: 31, 519567: 38, 519738: 7, 519761: 13, 519877: 85, 519911: 9, 519929: 75, 519956: 34, 519970: 23, 520015: 90, 520048: 5, 520054: 25, 520067: 60, 520144: 14, 520223: 27, 520355: 16, 520455: 22, 520494: 17, 520629: 37, 520647: 63, 520668: 77, 520693: 34, 520853: 71, 520871: 27, 520957: 36, 521100: 20, 521125: 11, 521186: 51, 521326: 29, 521475: 1, 521619: 14, 521635: 56, 521643: 42, 521667: 88, 521702: 7, 521826: 82, 521897: 53, 522035: 85, 522063: 18, 522093: 93, 522201: 39, 522218: 88, 522244: 20, 522303: 98, 522472: 83, 522617: 79, 522876: 33, 522908: 22, 523025: 57, 523047: 4, 523235: 93, 523335: 37, 523573: 77, 523686: 11, 523692: 50, 523752: 94, 523786: 43, 523820: 92, 523823: 73, 523911: 67, 523934: 75, 523958: 24, 523989: 92, 524021: 34, 524093: 57, 524119: 23, 524125: 50, 524276: 28, 524317: 6, 524477: 44, 524535: 51, 524555: 7, 524601: 91, 524628: 66, 524721: 46, 524739: 63, 524830: 8, 524882: 46, 524886: 70, 524922: 72, 524980: 96, 525010: 13, 525078: 22, 525257: 13, 525267: 24, 525269: 39, 525275: 72, 525325: 44, 525386: 69, 525473: 90, 525485: 42, 525503: 23, 525585: 37, 525796: 55, 525811: 40, 525902: 82, 525907: 96, 526069: 95, 526081: 8, 526088: 98, 526211: 39, 526285: 24, 526462: 15, 526463: 56, 526527: 49, 526532: 13, 526566: 61, 526587: 12, 526612: 61, 526641: 25, 526644: 90, 526673: 53, 526688: 29, 526742: 34, 526780: 82, 526799: 46, 527009: 28, 527065: 70, 527213: 67, 527266: 70, 527315: 2, 527381: 90, 527425: 14, 527461: 49, 527536: 17, 527671: 18, 527822: 80, 527961: 44, 528050: 42, 528171: 9, 528212: 70, 528283: 38, 528304: 37, 528319: 56, 528352: 0, 528428: 42, 528452: 35, 528512: 47, 528540: 48, 528685: 79, 528706: 43, 528742: 29, 528789: 12, 528813: 92, 528976: 32, 528992: 1, 529005: 6, 529104: 71, 529163: 61, 529194: 37, 529286: 25, 529481: 11, 529602: 79, 529642: 12, 529686: 47, 529815: 10, 529835: 81, 529957: 73, 529969: 61, 530003: 36, 530062: 8, 530099: 94, 530153: 27, 530186: 53, 530194: 66, 530236: 99, 530263: 19, 530428: 97, 530479: 17, 530535: 74, 530558: 58, 530590: 28, 530726: 0, 530771: 91, 530864: 16, 530893: 86, 530950: 4, 530957: 6, 530988: 0, 531013: 12, 531066: 9, 531081: 75, 531115: 96, 531261: 79, 531288: 15, 531302: 2, 531328: 80, 531458: 64, 531483: 38, 531632: 18, 531652: 97, 531661: 39, 531848: 43, 531871: 6, 531875: 56, 532076: 46, 532154: 60, 532155: 55, 532379: 31, 532465: 36, 532481: 54, 532566: 71, 532599: 61, 532609: 92, 532672: 45, 532689: 69, 532801: 97, 532806: 51, 532833: 88, 532857: 37, 532866: 81, 532873: 51, 533017: 46, 533066: 73, 533101: 32, 533122: 14, 533181: 6, 533281: 11, 533350: 30, 533518: 13, 533556: 12, 533569: 80, 533627: 77, 533659: 15, 533674: 43, 533676: 96, 533716: 86, 533867: 81, 533916: 76, 533957: 3, 533962: 88, 533994: 72, 533996: 86, 534005: 73, 534014: 69, 534024: 83, 534081: 96, 534134: 11, 534139: 47, 534158: 61, 534161: 47, 534236: 78, 534363: 42, 534411: 69, 534416: 42, 534605: 40, 534620: 60, 534719: 41, 534800: 90, 534854: 17, 534924: 16, 534974: 23, 535041: 83, 535043: 27, 535152: 92, 535200: 37, 535201: 57, 535209: 87, 535260: 52, 535339: 64, 535400: 73, 535513: 55, 535519: 41, 535556: 98, 535627: 74, 535697: 51, 535704: 91, 535733: 37, 535735: 20, 535755: 59, 535765: 32, 535834: 70, 535950: 3, 535959: 32, 535964: 83, 536011: 71, 536028: 12, 536033: 48, 536229: 50, 536267: 36, 536291: 19, 536384: 96, 536394: 10, 536441: 79, 536443: 85, 536458: 3, 536566: 52, 536610: 76, 536687: 32, 536705: 99, 536762: 26, 536789: 86, 536970: 80, 537209: 51, 537596: 63, 537599: 30, 537652: 48, 537700: 36, 537719: 50, 537732: 63, 537753: 58, 537770: 40, 537771: 11, 537803: 78, 537848: 95, 537917: 38, 537949: 27, 538026: 21, 538160: 37, 538389: 84, 538642: 75, 538710: 1, 538757: 90, 538812: 19, 538819: 98, 538831: 69, 538835: 39, 539087: 31, 539089: 54, 539091: 14, 539100: 38, 539135: 97, 539147: 17, 539178: 4, 539357: 81, 539376: 94, 539456: 92, 539621: 79, 539672: 89, 539693: 31, 539783: 19, 539799: 75, 539947: 92, 539969: 61, 540099: 68, 540366: 35, 540404: 29, 540411: 41, 540554: 99, 540646: 40, 540648: 93, 540679: 10, 540704: 67, 540749: 37, 540989: 92, 541023: 35, 541117: 84, 541121: 77, 541344: 95, 541386: 15, 541438: 81, 541471: 64, 541552: 5, 541627: 59, 541659: 74, 541858: 58, 541886: 88, 541909: 79, 541925: 16, 541962: 24, 542171: 12, 542197: 8, 542389: 72, 542393: 86, 542436: 22, 542554: 98, 542566: 18, 542667: 48, 542698: 70, 542712: 38, 542715: 33, 542850: 26, 542948: 82, 543252: 95, 543322: 44, 543387: 11, 543429: 65, 543582: 40, 543716: 51, 543765: 44, 543783: 11, 543814: 71, 543900: 18, 543912: 73, 543939: 6, 543972: 69, 543979: 2, 544025: 11, 544029: 35, 544094: 76, 544113: 21, 544175: 14, 544220: 84, 544267: 32, 544362: 29, 544528: 81, 544741: 27, 544793: 16, 544874: 36, 544941: 20, 544952: 16, 544998: 42, 545217: 24, 545243: 18, 545254: 53, 545366: 75, 545372: 94, 545381: 28, 545382: 62, 545529: 94, 545533: 76, 545591: 98, 545618: 67, 545639: 34, 545682: 68, 545724: 46, 545798: 71, 545799: 56, 545807: 3, 545920: 62, 545945: 20, 546042: 57, 546096: 89, 546125: 48, 546191: 25, 546217: 22, 546257: 57, 546409: 59, 546494: 9, 546577: 25, 546587: 50, 546646: 13, 546716: 55, 546728: 91, 546975: 29, 547229: 73, 547254: 27, 547255: 35, 547390: 82, 547547: 5, 547565: 32, 547578: 66, 547603: 20, 547640: 93, 547716: 8, 547718: 78, 547830: 38, 547846: 34, 547891: 73, 547936: 20, 548084: 4, 548103: 4, 548263: 83, 548335: 55, 548339: 19, 548345: 97, 548402: 50, 548471: 4, 548495: 31, 548499: 94, 548553: 11, 548716: 8, 548733: 84, 548766: 37, 548966: 76, 549036: 76, 549059: 47, 549140: 61, 549263: 90, 549269: 78, 549317: 68, 549335: 78, 549367: 70, 549384: 25, 549404: 39, 549548: 74, 549590: 16, 549621: 86, 549639: 85, 549661: 13, 549685: 58, 549770: 22, 549778: 17, 549824: 20, 549845: 50, 549980: 16, 550005: 54, 550014: 57, 550073: 20, 550084: 85, 550243: 80, 550307: 14, 550352: 70, 550379: 59, 550403: 24, 550406: 8, 550446: 64, 550510: 6, 550523: 50, 550551: 73, 550600: 38, 550619: 48, 550764: 38, 550772: 45, 550823: 29, 550864: 96, 550901: 66, 550988: 31, 551063: 59, 551121: 59, 551191: 41, 551243: 63, 551332: 70, 551391: 0, 551393: 44, 551411: 54, 551429: 92, 551455: 42, 551654: 34, 551756: 29, 551805: 69, 551807: 23, 551838: 92, 551892: 80, 551946: 21, 552037: 26, 552075: 19, 552121: 79, 552151: 57, 552155: 99, 552193: 23, 552378: 94, 552497: 85, 552640: 44, 552648: 47, 552751: 76, 552784: 49, 552855: 93, 552876: 93, 553015: 52, 553056: 85, 553210: 66, 553261: 0, 553282: 65, 553428: 63, 553448: 63, 553461: 80, 553480: 39, 553543: 64, 553641: 45, 553648: 29, 553799: 83, 553885: 38, 553952: 85, 554228: 84, 554271: 1, 554289: 97, 554302: 19, 554313: 12, 554324: 30, 554356: 60, 554359: 56, 554469: 14, 554501: 24, 554502: 14, 554544: 69, 554567: 56, 554664: 96, 554707: 38, 554741: 36, 554798: 34, 554843: 61, 554858: 29, 554878: 80, 554925: 39, 554983: 65, 555057: 27, 555105: 65, 555147: 3, 555194: 52, 555215: 29, 555277: 72, 555368: 24, 555459: 59, 555461: 56, 555753: 35, 555754: 92, 555796: 5, 555799: 77, 555846: 66, 555894: 58, 555911: 65, 556123: 94, 556178: 56, 556219: 60, 556221: 74, 556240: 58, 556253: 2, 556327: 57, 556347: 25, 556408: 40, 556429: 99, 556468: 2, 556581: 81, 556601: 46, 556636: 44, 556717: 26, 556751: 61, 556757: 6, 556761: 0, 556808: 95, 556854: 98, 556855: 50, 556915: 91, 557014: 92, 557018: 9, 557036: 63, 557260: 52, 557310: 30, 557334: 90, 557338: 78, 557382: 43, 557398: 38, 557474: 16, 557527: 50, 557611: 9, 557618: 56, 557663: 49, 557758: 70, 557853: 97, 557861: 40, 557928: 20, 557959: 88, 557982: 53, 558047: 79, 558153: 35, 558260: 7, 558278: 69, 558512: 13, 558528: 10, 558598: 2, 558637: 51, 558651: 88, 558712: 70, 558800: 45, 558847: 17, 558969: 76, 559061: 72, 559232: 65, 559296: 26, 559420: 89, 559680: 94, 559708: 13, 559718: 72, 559803: 55, 559811: 10, 559903: 51, 560026: 30, 560044: 15, 560213: 86, 560219: 49, 560220: 67, 560474: 44, 560673: 97, 560726: 43, 560827: 73, 560845: 11, 560859: 64, 560897: 19, 560926: 62, 560932: 46, 561008: 91, 561166: 19, 561289: 44, 561310: 11, 561452: 29, 561549: 10, 561625: 64, 561647: 34, 561672: 66, 561814: 9, 561924: 50, 561940: 96, 561941: 97, 561957: 61, 562122: 54, 562144: 57, 562185: 15, 562285: 12, 562306: 24, 562388: 88, 562396: 44, 562539: 68, 562616: 99, 562681: 62, 562695: 24, 562767: 48, 562872: 77, 562895: 1, 562938: 62, 563155: 42, 563198: 80, 563270: 41, 563336: 89, 563464: 17, 563514: 45, 563527: 52, 563626: 46, 563740: 92, 563750: 21, 563779: 64, 563784: 68, 563796: 51, 563808: 67, 563844: 15, 563869: 4, 563905: 62, 564044: 82, 564069: 56, 564089: 25, 564130: 6, 564169: 32, 564474: 46, 564494: 72, 564509: 3, 564534: 83, 564617: 16, 564736: 33, 564784: 93, 565023: 69, 565061: 71, 565105: 43, 565128: 37, 565198: 66, 565290: 46, 565299: 0, 565326: 96, 565495: 73, 565522: 38, 565574: 86, 565588: 50, 565624: 17, 565781: 46, 565796: 14, 565798: 68, 565813: 87, 565859: 15, 565889: 9, 565910: 5, 566125: 71, 566205: 85, 566401: 30, 566453: 1, 566533: 71, 566573: 29, 566623: 37, 566628: 21, 566812: 36, 566830: 69, 566944: 60, 567121: 43, 567171: 95, 567185: 35, 567262: 48, 567378: 81, 567389: 18, 567512: 79, 567563: 38, 567623: 4, 567649: 46, 567676: 14, 567745: 96, 567760: 49, 567792: 38, 567832: 4, 567875: 94, 567881: 28, 567946: 21, 567956: 34, 568005: 54, 568061: 82, 568320: 14, 568596: 70, 568613: 79, 568658: 33, 568708: 23, 568788: 10, 568859: 57, 569040: 10, 569151: 59, 569162: 16, 569200: 95, 569207: 76, 569331: 10, 569585: 52, 569658: 31, 569665: 37, 569720: 42, 569740: 78, 569802: 5, 569839: 6, 569960: 10, 569963: 26, 570013: 77, 570061: 85, 570099: 63, 570151: 53, 570220: 94, 570264: 96, 570322: 1, 570443: 87, 570444: 18, 570487: 26, 570490: 35, 570558: 75, 570634: 40, 570736: 94, 570788: 0, 570892: 22, 570938: 24, 570982: 59, 571022: 83, 571030: 3, 571032: 69, 571115: 64, 571160: 84, 571200: 44, 571249: 78, 571310: 4, 571353: 42, 571399: 34, 571458: 24, 571498: 12, 571633: 40, 571799: 49, 571864: 71, 571912: 83, 571998: 66, 572143: 4, 572228: 61, 572283: 77, 572299: 19, 572316: 27, 572382: 14, 572495: 56, 572534: 87, 572553: 54, 572679: 37, 572686: 46, 572708: 47, 572735: 62, 572749: 42, 572777: 63, 572800: 56, 572868: 92, 572885: 82, 572944: 96, 572964: 94, 572998: 51, 573096: 64, 573210: 9, 573235: 25, 573272: 32, 573292: 98, 573304: 20, 573318: 16, 573419: 23, 573472: 65, 573599: 0, 573626: 73, 573663: 8, 573747: 59, 573821: 13, 573967: 96, 574102: 77, 574104: 59, 574109: 23, 574156: 16, 574184: 67, 574188: 33, 574367: 68, 574491: 7, 574516: 42, 574566: 19, 574575: 70, 574608: 26, 574700: 0, 574760: 49, 574829: 38, 574903: 74, 574908: 56, 574921: 68, 574962: 87, 574979: 26, 575009: 11, 575077: 14, 575152: 49, 575163: 58, 575175: 64, 575233: 14, 575234: 68, 575246: 22, 575271: 94, 575322: 81, 575326: 72, 575369: 76, 575418: 35, 575421: 56, 575459: 6, 575511: 46, 575517: 54, 575642: 45, 575755: 62, 575926: 72, 576038: 92, 576115: 38, 576278: 98, 576399: 4, 576404: 64, 576405: 8, 576566: 77, 576618: 38, 576660: 7, 576665: 40, 576695: 22, 576718: 92, 576722: 77, 576730: 29, 576744: 72, 576815: 4, 576845: 98, 577082: 88, 577124: 80, 577167: 60, 577239: 71, 577338: 17, 577422: 97, 577560: 30, 577782: 49, 577959: 69, 578003: 29, 578011: 81, 578033: 14, 578043: 82, 578171: 29, 578180: 97, 578196: 99, 578204: 69, 578502: 56, 578574: 36, 578615: 99, 578619: 37, 578625: 6, 578636: 11, 578639: 62, 578939: 42, 578945: 92, 579027: 33, 579038: 31, 579057: 28, 579182: 74, 579292: 32, 579318: 93, 579345: 43, 579351: 25, 579477: 63, 579487: 43, 579671: 58, 579800: 89, 579836: 31, 579917: 66, 579965: 45, 579983: 17, 580222: 76, 580246: 48, 580343: 11, 580462: 26, 580573: 21, 580596: 88, 580674: 72, 580783: 57, 580832: 20, 580863: 75, 581035: 21, 581200: 90, 581215: 96, 581264: 53, 581335: 91, 581397: 15, 581452: 94, 581473: 66, 581671: 48, 581704: 44, 581792: 0, 581801: 37, 581892: 4, 581925: 63, 581960: 42, 581996: 77, 582005: 25, 582044: 86, 582130: 35, 582135: 93, 582157: 25, 582174: 61, 582263: 1, 582300: 29, 582462: 11, 582533: 6, 582663: 39, 582749: 12, 582835: 84, 582852: 9, 582891: 9, 583046: 58, 583069: 2, 583097: 43, 583274: 72, 583405: 81, 583523: 61, 583544: 48, 583589: 83, 583702: 57, 583739: 95, 583745: 33, 583753: 84, 583939: 63, 584131: 98, 584170: 14, 584246: 31, 584354: 49, 584453: 54, 584490: 96, 584708: 25, 584741: 10, 584828: 79, 584888: 85, 584901: 41, 584918: 73, 585168: 84, 585216: 68, 585264: 17, 585515: 56, 585518: 36, 585575: 83, 585665: 18, 585759: 96, 585764: 26, 585771: 49, 585773: 31, 585850: 26, 585904: 99, 585973: 28, 586042: 20, 586049: 29, 586057: 77, 586066: 20, 586119: 72, 586173: 37, 586215: 91, 586336: 88, 586360: 27, 586370: 83, 586398: 33, 586442: 62, 586633: 78, 586643: 95, 586670: 97, 586692: 69, 586694: 44, 586767: 94, 586932: 51, 586964: 92, 587000: 81, 587063: 11, 587248: 88, 587483: 8, 587601: 57, 587620: 19, 587641: 33, 587654: 65, 587659: 21, 587688: 57, 587706: 0, 587808: 44, 587834: 66, 587862: 43, 587945: 87, 588035: 18, 588040: 34, 588044: 76, 588052: 15, 588106: 28, 588329: 99, 588553: 54, 588589: 44, 588651: 90, 588692: 67, 588713: 5, 588817: 33, 588869: 36, 588927: 4, 589029: 33, 589047: 40, 589181: 10, 589299: 2, 589301: 63, 589311: 46, 589344: 95, 589348: 52, 589410: 50, 589430: 1, 589496: 94, 589538: 77, 589574: 79, 589594: 3, 589632: 38, 589661: 51, 589665: 45, 589723: 64, 589731: 46, 589762: 92, 589837: 15, 589914: 21, 589975: 2, 590032: 6, 590058: 60, 590202: 30, 590225: 86, 590455: 47, 590478: 22, 590494: 12, 590645: 18, 590771: 42, 590846: 4, 590847: 68, 590850: 75, 591016: 15, 591028: 62, 591077: 2, 591138: 51, 591205: 14, 591339: 8, 591401: 34, 591513: 22, 591564: 8, 591634: 37, 591637: 20, 591683: 87, 591854: 60, 591872: 29, 591917: 97, 591949: 41, 592104: 16, 592164: 53, 592218: 11, 592246: 82, 592301: 97, 592327: 60, 592517: 76, 592555: 44, 592670: 28, 592729: 75, 592833: 98, 593011: 8, 593246: 31, 593254: 97, 593294: 11, 593365: 83, 593378: 39, 593449: 34, 593456: 74, 593478: 61, 593481: 8, 593785: 71, 593852: 25, 593964: 20, 594093: 10, 594119: 31, 594239: 45, 594378: 60, 594414: 26, 594467: 54, 594541: 71, 594555: 82, 594604: 61, 594667: 63, 594780: 25, 594803: 53, 594986: 9, 595143: 67, 595189: 38, 595217: 53, 595224: 24, 595447: 55, 595663: 33, 595858: 83, 595897: 2, 595905: 52, 595962: 50, 596037: 87, 596052: 55, 596138: 4, 596572: 63, 596586: 72, 596667: 76, 596753: 56, 596803: 24, 596849: 17, 596895: 14, 596943: 21, 596992: 20, 596994: 57, 597030: 20, 597223: 72, 597250: 81, 597273: 90, 597283: 35, 597372: 76, 597431: 54, 597462: 4, 597555: 16, 597566: 11, 597571: 3, 597667: 3, 597702: 88, 597770: 76, 598059: 55, 598075: 44, 598426: 93, 598453: 73, 598489: 83, 598492: 78, 598707: 0, 598736: 86, 598739: 0, 598939: 34, 599002: 94, 599117: 63, 599138: 92, 599157: 83, 599186: 13, 599232: 95, 599253: 85, 599264: 24, 599280: 27, 599322: 44, 599482: 20, 599516: 72, 599546: 91, 599662: 83, 599677: 5, 599712: 32, 599858: 68, 599935: 7, 599944: 66, 599964: 80, 600059: 28, 600061: 63, 600159: 62, 600218: 9, 600239: 96, 600256: 15, 600261: 90, 600389: 87, 600461: 63, 600511: 6, 600551: 9, 600572: 46, 600587: 16, 600603: 86, 600724: 63, 600791: 19, 600793: 23, 600817: 13, 601109: 36, 601144: 66, 601183: 21, 601191: 96, 601221: 17, 601345: 76, 601396: 50, 601398: 7, 601506: 73, 601542: 55, 601553: 42, 601610: 86, 601611: 28, 601634: 26, 601635: 37, 601711: 84, 601738: 67, 601752: 24, 601799: 79, 601806: 76, 601857: 61, 601861: 74, 602010: 65, 602023: 71, 602187: 25, 602287: 14, 602311: 50, 602315: 44, 602488: 60, 602506: 24, 602548: 11, 602560: 55, 602588: 24, 602690: 50, 602763: 32, 602837: 8, 603023: 0, 603048: 46, 603053: 4, 603106: 42, 603111: 42, 603132: 35, 603257: 33, 603343: 33, 603406: 81, 603568: 39, 603592: 31, 603619: 50, 603651: 38, 603823: 25, 603878: 59, 603879: 38, 604060: 25, 604111: 78, 604138: 54, 604286: 46, 604328: 72, 604353: 60, 604442: 87, 604452: 31, 604497: 23, 604498: 72, 604510: 49, 604517: 2, 604532: 72, 604579: 0, 604623: 47, 604866: 58, 604977: 26, 605047: 74, 605048: 71, 605058: 6, 605212: 78, 605295: 24, 605524: 6, 605561: 18, 605571: 96, 605584: 56, 605614: 67, 605636: 6, 605673: 90, 605676: 99, 605713: 63, 605897: 57, 605918: 26, 606255: 29, 606349: 76, 606431: 53, 606602: 43, 606611: 17, 606623: 68, 606650: 76, 606812: 53, 606858: 82, 606866: 42, 606893: 79, 606925: 74, 606929: 53, 607018: 99, 607046: 25, 607075: 22, 607079: 61, 607149: 89, 607284: 64, 607313: 33, 607338: 86, 607403: 42, 607466: 30, 607476: 60, 607538: 74, 607922: 42, 607950: 11, 608063: 5, 608079: 34, 608152: 89, 608201: 96, 608307: 33, 608422: 95, 608567: 92, 608658: 49, 608713: 64, 608789: 35, 608813: 40, 608824: 39, 608864: 77, 608891: 76, 608899: 73, 608910: 89, 608993: 19, 609046: 50, 609171: 95, 609214: 26, 609374: 4, 609388: 13, 609461: 10, 609675: 98, 609683: 74, 609688: 44, 609696: 95, 609768: 90, 609981: 11, 610055: 96, 610063: 73, 610076: 32, 610139: 39, 610234: 68, 610426: 84, 610429: 60, 610547: 61, 610549: 91, 610596: 27, 610622: 47, 610625: 13, 610700: 58, 610718: 70, 610747: 73, 610843: 19, 610857: 24, 610876: 12, 611144: 73, 611191: 12, 611204: 44, 611257: 52, 611311: 91, 611313: 75, 611384: 97, 611414: 51, 611459: 16, 611468: 85, 611510: 71, 611569: 14, 611602: 27, 611711: 82, 611803: 5, 611834: 10, 611841: 10, 611871: 76, 611878: 14, 611930: 57, 612164: 41, 612174: 37, 612197: 95, 612199: 57, 612301: 42, 612305: 35, 612352: 14, 612371: 70, 612412: 5, 612434: 20, 612486: 8, 612561: 47, 612663: 74, 612685: 7, 612693: 93, 612698: 25, 612704: 53, 612709: 66, 612819: 60, 612873: 32, 612893: 3, 612954: 89, 612981: 53, 612990: 4, 613011: 53, 613058: 44, 613233: 97, 613268: 2, 613296: 3, 613322: 47, 613328: 68, 613505: 74, 613636: 86, 613668: 91, 613733: 45, 613827: 57, 613877: 27, 613895: 19, 613910: 59, 613916: 86, 614065: 49, 614066: 16, 614144: 12, 614158: 59, 614160: 57, 614161: 36, 614226: 40, 614264: 53, 614292: 67, 614299: 21, 614312: 69, 614329: 24, 614438: 93, 614447: 59, 614541: 22, 614548: 41, 614604: 10, 614656: 84, 614829: 83, 614872: 42, 614881: 10, 614891: 6, 615038: 18, 615062: 45, 615172: 72, 615178: 31, 615298: 78, 615317: 80, 615794: 37, 615815: 82, 616020: 89, 616078: 2, 616274: 85, 616283: 33, 616339: 12, 616492: 50, 616519: 85, 616538: 66, 616584: 10, 616909: 4, 616916: 6, 617031: 63, 617078: 28, 617213: 82, 617377: 6, 617466: 49, 617504: 2, 617548: 70, 617576: 29, 617641: 97, 617673: 64, 617750: 61, 617792: 4, 617870: 21, 617872: 21, 618084: 71, 618135: 88, 618317: 89, 618347: 34, 618485: 4, 618567: 76, 618728: 99, 618752: 75, 618808: 90, 618879: 79, 618890: 52, 618905: 34, 618936: 23, 618938: 35, 618958: 73, 618961: 52, 619121: 38, 619194: 95, 619222: 33, 619259: 45, 619326: 33, 619328: 24, 619333: 24, 619335: 1, 619374: 24, 619376: 93, 619396: 81, 619496: 41, 619605: 33, 619671: 38, 619737: 81, 619914: 79, 619989: 71, 620058: 32, 620060: 8, 620108: 11, 620473: 11, 620521: 37, 620610: 18, 620655: 69, 620672: 63, 620704: 83, 620714: 67, 620862: 42, 620867: 62, 621027: 87, 621029: 97, 621132: 28, 621160: 36, 621172: 92, 621198: 0, 621344: 82, 621474: 3, 621697: 25, 621725: 24, 621789: 78, 621864: 45, 621866: 72, 621870: 21, 621915: 8, 621933: 54, 622086: 67, 622121: 78, 622181: 22, 622241: 40, 622282: 54, 622331: 83, 622409: 20, 622458: 87, 622472: 40, 622534: 38, 622562: 81, 622662: 36, 622681: 78, 622683: 15, 622746: 8, 622907: 45, 622924: 64, 622944: 81, 623013: 95, 623177: 32, 623193: 96, 623381: 25, 623420: 47, 623422: 63, 623444: 93, 623578: 46, 623605: 12, 623630: 21, 623828: 74, 623897: 1, 623921: 83, 623938: 67, 624038: 97, 624060: 42, 624263: 49, 624293: 42, 624356: 16, 624383: 10, 624389: 73, 624442: 48, 624491: 15, 624525: 24, 624549: 78, 624567: 94, 624700: 79, 624704: 30, 624713: 87, 624802: 72, 624814: 9, 624832: 43, 624905: 14, 624914: 78, 624957: 48, 624960: 36, 624978: 4, 625035: 73, 625132: 46, 625201: 96, 625326: 16, 625449: 75, 625525: 81, 625527: 60, 625602: 90, 625779: 91, 625906: 57, 625913: 89, 626004: 87, 626006: 19, 626007: 73, 626044: 5, 626057: 40, 626085: 35, 626102: 95, 626156: 50, 626184: 92, 626232: 11, 626392: 34, 626405: 52, 626452: 74, 626628: 68, 626670: 19, 626681: 11, 626797: 61, 626804: 86, 626845: 54, 627015: 42, 627030: 25, 627040: 30, 627065: 22, 627279: 20, 627339: 12, 627363: 60, 627393: 6, 627467: 28, 627484: 42, 627516: 82, 627526: 73, 627618: 29, 627625: 27, 627737: 30, 627816: 65, 627857: 58, 628203: 20, 628313: 37, 628378: 74, 628386: 49, 628573: 83, 628628: 22, 628736: 94, 628772: 47, 628835: 90, 628867: 44, 628923: 92, 628943: 51, 628986: 46, 629141: 33, 629225: 99, 629352: 4, 629498: 72, 629556: 37, 629589: 14, 629628: 50, 629631: 64, 629632: 79, 629657: 73, 629701: 67, 629806: 85, 629826: 52, 629905: 27, 629913: 30, 630002: 62, 630023: 68, 630041: 61, 630076: 17, 630091: 8, 630110: 51, 630133: 42, 630155: 92, 630169: 81, 630177: 54, 630264: 64, 630444: 19, 630531: 27, 630628: 86, 630703: 18, 630731: 76, 631046: 15, 631080: 60, 631242: 24, 631304: 43, 631306: 34, 631520: 71, 631554: 73, 631582: 76, 631599: 41, 631634: 81, 631681: 86, 631730: 8, 631781: 51, 631927: 86, 631950: 18, 631988: 57, 632141: 50, 632251: 41, 632303: 12, 632380: 80, 632415: 84, 632458: 46, 632495: 61, 632522: 7, 632631: 7, 632685: 11, 632731: 12, 632735: 81, 632847: 27, 633085: 33, 633217: 70, 633332: 30, 633350: 30, 633353: 21, 633402: 84, 633421: 39, 633451: 49, 633522: 1, 633590: 38, 633599: 75, 633686: 33, 633705: 71, 633708: 4, 633722: 67, 633835: 61, 633916: 3, 634088: 80, 634115: 88, 634205: 41, 634228: 60, 634273: 45, 634275: 46, 634371: 26, 634510: 97, 634661: 46, 634666: 82, 634684: 91, 634745: 67, 634790: 22, 634821: 40, 634855: 52, 634929: 47, 635013: 26, 635020: 54, 635094: 91, 635189: 68, 635463: 95, 635525: 50, 635556: 32, 635602: 41, 635726: 22, 635783: 37, 635817: 27, 635837: 7, 635889: 57, 635958: 95, 636013: 12, 636039: 48, 636084: 32, 636119: 67, 636133: 77, 636159: 87, 636193: 47, 636219: 72, 636285: 86, 636408: 88, 636409: 66, 636431: 30, 636562: 26, 636572: 23, 636758: 74, 636857: 3, 636892: 0, 636896: 75, 636947: 91, 637044: 27, 637151: 40, 637160: 67, 637268: 41, 637302: 72, 637370: 48, 637582: 40, 637604: 15, 637616: 0, 637626: 13, 637659: 93, 637701: 55, 637743: 59, 637819: 56, 637862: 91, 637919: 49, 637923: 54, 637962: 14, 638163: 6, 638316: 38, 638341: 49, 638412: 64, 638448: 60, 638455: 19, 638485: 0, 638507: 67, 638525: 59, 638595: 33, 638669: 20, 638860: 60, 638951: 56, 638969: 85, 639172: 35, 639226: 35, 639345: 49, 639403: 11, 639413: 80, 639440: 82, 639467: 50, 639485: 80, 639498: 95, 639590: 86, 639638: 45, 639661: 13, 639674: 82, 639711: 62, 639717: 63, 639755: 48, 639832: 46, 639854: 43, 639888: 27, 639933: 79, 640035: 55, 640093: 81, 640142: 58, 640174: 43, 640191: 28, 640214: 37, 640218: 74, 640322: 89, 640395: 26, 640404: 32, 640415: 63, 640529: 57, 640538: 53, 640551: 64, 640553: 92, 640650: 77, 640745: 90, 640826: 22, 640829: 46, 640835: 17, 640847: 53, 640887: 79, 641007: 71, 641080: 21, 641093: 44, 641119: 66, 641160: 32, 641218: 86, 641239: 35, 641287: 87, 641339: 71, 641345: 48, 641703: 6, 641846: 32, 641873: 51, 641955: 55, 642042: 61, 642080: 90, 642085: 74, 642118: 89, 642150: 56, 642214: 30, 642231: 1, 642251: 46, 642314: 18, 642453: 56, 642557: 52, 642587: 28, 642616: 55, 642696: 57, 642772: 8, 642782: 0, 642834: 66, 642942: 38, 642957: 12, 643097: 67, 643154: 74, 643324: 3, 643398: 84, 643865: 15, 643932: 5, 643933: 15, 643972: 2, 644052: 91, 644069: 29, 644135: 28, 644181: 46, 644272: 43, 644343: 64, 644346: 49, 644401: 27, 644425: 4, 644554: 76, 644633: 11, 644686: 36, 644692: 34, 644929: 29, 644960: 90, 644961: 53, 644994: 58, 645006: 8, 645020: 59, 645050: 84, 645102: 76, 645122: 13, 645152: 97, 645249: 57, 645288: 77, 645423: 38, 645450: 34, 645478: 7, 645564: 61, 645600: 98, 645644: 42, 645666: 84, 645784: 42, 645813: 49, 645908: 64, 645962: 83, 646006: 1, 646106: 78, 646165: 68, 646185: 75, 646212: 20, 646223: 78, 646268: 77, 646298: 80, 646394: 67, 646457: 95, 646472: 33, 646542: 42, 646607: 85, 646643: 89, 646776: 79, 646811: 61, 646832: 50, 646923: 19, 646978: 70, 647034: 29, 647067: 50, 647080: 1, 647145: 59, 647198: 18, 647232: 50, 647235: 90, 647355: 42, 647389: 32, 647541: 37, 647597: 90, 647620: 19, 647729: 75, 647764: 71, 647767: 51, 647814: 59, 647904: 85, 647909: 51, 647914: 19, 647962: 47, 647976: 97, 648061: 32, 648080: 67, 648214: 20, 648322: 30, 648351: 17, 648378: 5, 648424: 89, 648475: 43, 648498: 92, 648555: 82, 648623: 87, 648641: 36, 648707: 55, 648762: 92, 648927: 2, 649050: 41, 649081: 56, 649247: 19, 649299: 50, 649390: 60, 649399: 51, 649411: 38, 649414: 4, 649482: 26, 649627: 78, 649734: 93, 649740: 61, 649760: 23, 649856: 41, 649881: 25, 649956: 97, 649969: 68, 650156: 40, 650610: 44, 650669: 26, 650879: 38, 650908: 42, 651049: 29, 651102: 87, 651204: 93, 651263: 51, 651306: 8, 651463: 3, 651581: 60, 651596: 45, 651947: 6, 651972: 66, 651989: 44, 651991: 75, 652052: 96, 652127: 70, 652144: 88, 652284: 54, 652456: 92, 652518: 13, 652786: 8, 652821: 70, 652957: 77, 653069: 90, 653106: 38, 653242: 23, 653294: 75, 653368: 22, 653383: 29, 653410: 46, 653510: 73, 653519: 78, 653623: 89, 653646: 32, 653865: 37, 653900: 77, 653928: 49, 654050: 19, 654076: 47, 654091: 1, 654098: 95, 654153: 13, 654178: 35, 654201: 80, 654276: 2, 654286: 35, 654318: 25, 654345: 64, 654348: 43, 654408: 97, 654411: 17, 654445: 73, 654470: 59, 654510: 79, 654526: 17, 654548: 57, 654583: 3, 654607: 12, 654669: 67, 654768: 36, 654769: 93, 654803: 25, 654865: 92, 655008: 21, 655046: 52, 655100: 98, 655101: 26, 655216: 20, 655222: 34, 655248: 50, 655449: 42, 655524: 3, 655541: 25, 655584: 16, 655595: 15, 655601: 7, 655724: 47, 655850: 53, 655881: 71, 655959: 33, 656013: 45, 656045: 89, 656077: 82, 656188: 57, 656204: 22, 656293: 62, 656345: 18, 656402: 5, 656448: 16, 656540: 98, 656598: 88, 656689: 48, 656695: 69, 656773: 50, 656792: 6, 656873: 77, 656916: 94, 656938: 33, 657025: 12, 657039: 35, 657181: 43, 657185: 43, 657298: 12, 657310: 34, 657428: 47, 657430: 59, 657473: 34, 657494: 58, 657529: 71, 657638: 61, 657750: 56, 657826: 83, 657928: 78, 657934: 37, 658225: 0, 658261: 13, 658275: 39, 658345: 45, 658362: 27, 658434: 51, 658598: 60, 658622: 93, 658638: 92, 658645: 48, 658692: 57, 658726: 45, 658826: 84, 658926: 48, 658934: 55, 659020: 45, 659135: 94, 659144: 98, 659293: 78, 659302: 83, 659430: 31, 659503: 41, 659574: 45, 659582: 95, 659718: 86, 659744: 81, 659750: 73, 659756: 95, 659761: 25, 659808: 50, 659908: 89, 659957: 89, 659974: 92, 660040: 45, 660097: 87, 660140: 37, 660166: 68, 660292: 63, 660365: 6, 660397: 44, 660399: 84, 660420: 68, 660427: 65, 660556: 66, 660564: 95, 660628: 38, 660693: 4, 660711: 17, 660732: 8, 660750: 34, 660808: 29, 660870: 59, 660928: 87, 661068: 35, 661149: 88, 661502: 5, 661582: 86, 661646: 30, 661709: 24, 661764: 48, 661793: 99, 661820: 93, 661926: 7, 661998: 34, 662031: 40, 662134: 92, 662135: 53, 662188: 45, 662216: 68, 662231: 43, 662238: 7, 662252: 44, 662412: 53, 662416: 79, 662418: 22, 662464: 50, 662495: 89, 662600: 82, 662601: 7, 662615: 46, 662734: 67, 662852: 38, 662863: 46, 662870: 51, 662892: 84, 662908: 8, 663113: 8, 663184: 47, 663292: 9, 663429: 86, 663487: 44, 663655: 85, 663677: 12, 663690: 30, 663837: 87, 663840: 99, 663897: 87, 663941: 78, 664021: 70, 664042: 19, 664080: 97, 664241: 70, 664266: 43, 664443: 19, 664486: 22, 664622: 65, 664652: 40, 664747: 17, 664780: 64, 664825: 11, 664847: 74, 664947: 53, 664950: 64, 665033: 3, 665078: 15, 665097: 98, 665133: 0, 665414: 60, 665482: 11, 665569: 27, 665588: 15, 665619: 90, 665859: 46, 665867: 67, 665891: 50, 665998: 98, 666110: 80, 666124: 45, 666143: 79, 666259: 42, 666337: 64, 666431: 82, 666556: 68, 666661: 92, 666696: 68, 666713: 42, 666728: 88, 666912: 80, 666917: 15, 666943: 18, 666998: 68, 667040: 30, 667174: 64, 667212: 28, 667226: 59, 667329: 76, 667431: 97, 667481: 14, 667559: 24, 667684: 84, 667807: 69, 667837: 51, 667869: 8, 667896: 59, 668041: 17, 668093: 42, 668095: 40, 668283: 16, 668284: 70, 668285: 62, 668289: 86, 668396: 83, 668518: 19, 668582: 17, 668617: 41, 668750: 28, 668815: 35, 668895: 29, 668942: 94, 669110: 5, 669219: 45, 669281: 56, 669411: 2, 669425: 85, 669435: 13, 669684: 60, 669784: 20, 669888: 34, 669985: 39, 670001: 24, 670184: 35, 670447: 15, 670493: 8, 670669: 45, 670676: 17, 670743: 39, 670825: 30, 670854: 73, 670897: 50, 670917: 77, 671004: 89, 671160: 53, 671217: 90, 671274: 18, 671355: 29, 671361: 90, 671394: 38, 671461: 33, 671471: 22, 671530: 53, 671603: 49, 671630: 92, 671678: 74, 671682: 80, 671749: 21, 671897: 81, 671953: 19, 671997: 58, 672106: 20, 672130: 51, 672142: 11, 672151: 55, 672217: 75, 672392: 93, 672487: 56, 672666: 84, 672691: 24, 672762: 14, 672838: 54, 672842: 63, 672848: 78, 673049: 6, 673095: 23, 673131: 54, 673143: 10, 673173: 22, 673217: 8, 673250: 95, 673295: 82, 673385: 98, 673395: 47, 673418: 12, 673424: 59, 673443: 19, 673590: 48, 673638: 12, 673662: 41, 673803: 58, 673813: 46, 673873: 89, 673877: 44, 673916: 35, 673987: 56, 673993: 38, 674051: 1, 674062: 86, 674080: 14, 674148: 39, 674308: 42, 674383: 70, 674450: 61, 674465: 49, 674506: 3, 674536: 20, 674555: 28, 674603: 72, 674626: 56, 674730: 39, 674763: 76, 674828: 78, 674975: 70, 675010: 99, 675022: 42, 675176: 34, 675393: 60, 675483: 64, 675492: 64, 675495: 69, 675510: 50, 675614: 50, 675620: 9, 675650: 39, 675840: 96, 675925: 33, 675934: 91, 676039: 70, 676154: 90, 676228: 4, 676243: 26, 676254: 53, 676279: 69, 676293: 23, 676320: 13, 676340: 63, 676404: 34, 676435: 2, 676487: 13, 676497: 89, 676502: 15, 676561: 19, 676609: 95, 676746: 89, 676864: 54, 676884: 6, 677035: 49, 677045: 9, 677108: 56, 677127: 83, 677150: 32, 677155: 62, 677169: 40, 677238: 89, 677240: 84, 677271: 30, 677277: 98, 677286: 80, 677480: 1, 677534: 56, 677592: 62, 677665: 46, 677689: 64, 677782: 8, 677841: 72, 677874: 27, 677911: 19, 677913: 72, 677921: 31, 677938: 49, 678172: 38, 678293: 93, 678332: 53, 678539: 62, 678569: 15, 678654: 23, 678680: 5, 678712: 89, 678751: 20, 678769: 88, 679041: 45, 679179: 70, 679371: 65, 679390: 54, 679430: 63, 679661: 87, 679682: 63, 679743: 92, 679847: 80, 679853: 24, 679981: 49, 680176: 52, 680304: 69, 680309: 54, 680358: 8, 680371: 9, 680422: 48, 680423: 34, 680500: 27, 680514: 43, 680651: 99, 680704: 50, 680722: 55, 680769: 91, 680795: 10, 680801: 42, 680843: 82, 680904: 31, 680915: 47, 680979: 94, 680990: 13, 681112: 28, 681123: 56, 681298: 80, 681355: 25, 681409: 36, 681486: 55, 681499: 55, 681554: 42, 681627: 54, 681645: 60, 681685: 80, 681787: 76, 681855: 32, 681942: 81, 681969: 30, 681982: 69, 681992: 57, 682182: 66, 682183: 51, 682187: 32, 682220: 41, 682236: 5, 682279: 57, 682291: 92, 682494: 16, 682534: 53, 682552: 4, 682604: 47, 682641: 70, 682690: 23, 682724: 36, 682726: 19, 682734: 92, 682808: 66, 682809: 5, 682876: 65, 682932: 65, 682984: 22, 683111: 4, 683124: 79, 683130: 46, 683136: 57, 683149: 17, 683188: 97, 683498: 54, 683541: 3, 683638: 84, 683656: 0, 683698: 0, 683743: 7, 683755: 8, 683773: 3, 683991: 53, 684016: 6, 684033: 45, 684166: 69, 684168: 20, 684407: 54, 684487: 27, 684532: 11, 684534: 87, 684544: 66, 684574: 90, 684662: 63, 684703: 34, 685000: 76, 685025: 78, 685131: 25, 685135: 11, 685148: 85, 685240: 63, 685253: 38, 685270: 49, 685308: 68, 685387: 65, 685461: 56, 685839: 21, 685853: 43, 685992: 17, 686126: 24, 686131: 60, 686211: 96, 686214: 37, 686292: 39, 686352: 12, 686373: 96, 686379: 86, 686414: 30, 686556: 31, 686726: 38, 686770: 47, 686824: 22, 686853: 95, 686972: 95, 687024: 50, 687054: 25, 687060: 28, 687065: 63, 687098: 39, 687204: 25, 687206: 83, 687225: 85, 687234: 23, 687588: 40, 687728: 68, 687732: 56, 687875: 19, 688048: 19, 688144: 2, 688169: 27, 688217: 93, 688290: 20, 688317: 46, 688464: 13, 688465: 74, 688579: 51, 688609: 32, 688664: 36, 688724: 70, 688767: 41, 688862: 60, 688943: 3, 688990: 20, 689074: 92, 689153: 49, 689163: 31, 689193: 88, 689222: 14, 689279: 23, 689281: 76, 689358: 30, 689384: 85, 689445: 94, 689474: 16, 689581: 48, 689608: 1, 689655: 97, 689676: 91, 689695: 28, 689718: 88, 689727: 36, 689747: 87, 689834: 16, 689841: 62, 689844: 92, 689850: 3, 689855: 82, 689910: 79, 689915: 41, 689932: 82, 689940: 17, 690218: 34, 690262: 92, 690270: 15, 690302: 35, 690324: 3, 690375: 47, 690377: 52, 690403: 76, 690439: 56, 690441: 52, 690458: 6, 690499: 38, 690545: 65, 690610: 81, 690643: 62, 690665: 54, 690692: 19, 690712: 3, 690795: 37, 690815: 45, 690984: 79, 690993: 47, 690995: 84, 691076: 30, 691091: 77, 691111: 95, 691628: 53, 691629: 27, 691636: 84, 691643: 61, 691668: 47, 691758: 87, 691801: 42, 692002: 26, 692099: 96, 692122: 54, 692217: 17, 692284: 86, 692329: 60, 692378: 83, 692409: 44, 692458: 2, 692611: 36, 692634: 81, 692641: 21, 692672: 89, 692727: 11, 692733: 86, 692755: 65, 692825: 76, 692923: 93, 692925: 18, 693127: 51, 693139: 41, 693202: 19, 693264: 17, 693283: 66, 693291: 18, 693303: 55, 693366: 27, 693473: 90, 693483: 52, 693523: 45, 693535: 68, 693574: 95, 693683: 74, 693936: 65, 694080: 96, 694118: 59, 694145: 84, 694148: 61, 694154: 64, 694192: 26, 694231: 89, 694338: 94, 694449: 57, 694604: 96, 694700: 34, 694757: 90, 694785: 61, 694800: 72, 694808: 68, 694936: 17, 694981: 5, 695024: 46, 695041: 58, 695066: 91, 695067: 76, 695093: 59, 695104: 52, 695206: 24, 695228: 19, 695245: 5, 695335: 61, 695394: 1, 695397: 95, 695403: 0, 695441: 80, 695675: 63, 695685: 82, 695828: 36, 695870: 28, 695927: 58, 695931: 72, 695941: 95, 695987: 14, 696045: 40, 696084: 1, 696088: 92, 696181: 78, 696193: 17, 696223: 81, 696262: 72, 696273: 35, 696344: 70, 696362: 77, 696377: 33, 696479: 4, 696493: 10, 696503: 57, 696543: 69, 696759: 33, 696772: 36, 696800: 51, 696808: 98, 696871: 21, 696881: 2, 696912: 54, 696929: 87, 697002: 57, 697010: 7, 697028: 73, 697102: 72, 697120: 29, 697129: 62, 697197: 77, 697292: 84, 697297: 53, 697352: 4, 697419: 78, 697498: 17, 697584: 82, 697618: 3, 697627: 38, 697708: 29, 697756: 89, 697817: 37, 697844: 44, 697875: 80, 697890: 59, 697912: 45, 697935: 42, 698007: 20, 698016: 46, 698040: 96, 698134: 32, 698162: 9, 698212: 98, 698237: 34, 698245: 91, 698296: 62, 698317: 41, 698349: 72, 698589: 63, 698628: 72, 698664: 17, 698675: 55, 698800: 27, 698863: 43, 698919: 70, 699011: 59, 699088: 51, 699099: 70, 699109: 81, 699156: 35, 699201: 63, 699204: 5, 699219: 88, 699246: 37, 699253: 21, 699400: 10, 699450: 59, 699532: 35, 699585: 71, 699669: 78, 699706: 26, 699732: 48, 699737: 99, 699746: 91, 699830: 65, 699900: 12, 699928: 25, 700105: 3, 700154: 72, 700163: 34, 700310: 50, 700356: 18, 700363: 92, 700393: 52, 700425: 82, 700462: 60, 700496: 26, 700501: 13, 700605: 75, 700689: 99, 700752: 75, 700757: 75, 700795: 99, 701181: 57, 701252: 68, 701321: 53, 701352: 99, 701385: 5, 701397: 44, 701424: 29, 701470: 73, 701581: 51, 701584: 99, 701665: 32, 701684: 95, 701685: 53, 701692: 13, 701730: 19, 701764: 60, 701847: 88, 701888: 35, 701975: 80, 701976: 27, 701986: 68, 702025: 44, 702035: 86, 702210: 41, 702287: 77, 702341: 93, 702396: 95, 702521: 70, 702537: 16, 702574: 38, 702619: 54, 702648: 35, 702712: 20, 702741: 81, 702759: 35, 702785: 94, 702831: 66, 702920: 15, 702980: 31, 703010: 5, 703018: 72, 703049: 21, 703058: 90, 703331: 50, 703426: 89, 703455: 59, 703467: 42, 703540: 23, 703581: 57, 703590: 57, 703673: 80, 703745: 26, 703781: 64, 704037: 35, 704095: 47, 704158: 51, 704276: 1, 704298: 32, 704641: 95, 704690: 1, 704898: 72, 704936: 23, 704956: 50, 704967: 52, 705011: 18, 705229: 85, 705250: 19, 705294: 93, 705352: 61, 705404: 95, 705454: 82, 705464: 88, 705544: 85, 705546: 97, 705646: 15, 705858: 27, 705870: 1, 706104: 48, 706135: 78, 706137: 39, 706156: 92, 706180: 83, 706205: 99, 706247: 47, 706324: 87, 706374: 72, 706408: 71, 706592: 43, 706647: 3, 706799: 83, 706800: 32, 706967: 58, 706974: 79, 706981: 45, 707148: 30, 707167: 84, 707425: 23, 707460: 22, 707461: 80, 707486: 53, 707507: 81, 707582: 11, 707682: 35, 707745: 75, 707760: 65, 707782: 96, 707868: 14, 707925: 37, 707969: 89, 708020: 2, 708118: 89, 708129: 35, 708353: 47, 708450: 87, 708500: 62, 708738: 29, 708784: 25, 708965: 9, 709034: 18, 709042: 67, 709135: 60, 709155: 63, 709176: 21, 709239: 40, 709248: 66, 709276: 97, 709305: 4, 709337: 5, 709394: 65, 709528: 21, 709531: 58, 709545: 34, 709674: 68, 709747: 40, 709752: 29, 709766: 8, 709769: 67, 709888: 52, 709923: 7, 709937: 89, 709965: 12, 709967: 20, 709968: 61, 709986: 33, 710059: 20, 710095: 17, 710134: 39, 710250: 86, 710281: 68, 710292: 15, 710333: 12, 710416: 16, 710551: 71, 710792: 26, 710819: 95, 710989: 44, 710994: 2, 711145: 80, 711208: 37, 711288: 79, 711321: 2, 711383: 57, 711435: 69, 711539: 79, 711617: 48, 711697: 54, 711758: 45, 711841: 50, 711850: 24, 711855: 63, 711861: 78, 711865: 27, 711879: 74, 711943: 44, 712103: 84, 712106: 81, 712135: 67, 712243: 92, 712280: 95, 712284: 9, 712388: 40, 712395: 96, 712423: 43, 712424: 29, 712521: 31, 712550: 12, 712674: 93, 712724: 14, 712745: 61, 712785: 79, 712923: 1, 713012: 85, 713051: 27, 713056: 62, 713075: 55, 713229: 53, 713508: 47, 713514: 74, 713546: 53, 713646: 55, 713706: 38, 713713: 65, 713879: 25, 713918: 55, 714032: 32, 714075: 70, 714094: 48, 714116: 16, 714227: 79, 714379: 16, 714389: 19, 714395: 49, 714409: 30, 714525: 77, 714530: 69, 714608: 79, 714713: 54, 714736: 51, 714847: 74, 715038: 98, 715053: 57, 715200: 32, 715256: 48, 715264: 60, 715366: 1, 715499: 57, 715537: 12, 715669: 34, 715752: 65, 715765: 79, 715767: 11, 716057: 92, 716185: 31, 716228: 85, 716270: 36, 716300: 65, 716304: 39, 716494: 91, 716519: 96, 716594: 77, 716614: 58, 716774: 85, 716782: 72, 716828: 50, 716974: 63, 717011: 97, 717015: 61, 717038: 30, 717065: 67, 717083: 43, 717118: 31, 717127: 7, 717199: 11, 717209: 42, 717224: 75, 717267: 77, 717343: 27, 717351: 69, 717445: 14, 717531: 90, 717635: 17, 717646: 84, 717670: 68, 717786: 72, 717835: 33, 717981: 78, 718101: 85, 718226: 28, 718293: 78, 718413: 60, 718463: 69, 718554: 19, 718628: 6, 718712: 58, 718780: 91, 718782: 37, 718802: 33, 718897: 92, 718946: 52, 719006: 28, 719079: 10, 719310: 71, 719337: 72, 719341: 54, 719348: 21, 719369: 26, 719453: 18, 719517: 89, 719540: 30, 719717: 92, 719836: 72, 719859: 33, 719883: 79, 719967: 95, 720005: 2, 720042: 83, 720103: 60, 720169: 7, 720213: 17, 720315: 68, 720359: 41, 720428: 94, 720528: 91, 720606: 92, 720857: 64, 720951: 96, 720965: 57, 720981: 18, 721080: 25, 721118: 29, 721119: 79, 721158: 93, 721161: 70, 721314: 28, 721337: 90, 721343: 98, 721361: 70, 721380: 98, 721391: 35, 721516: 17, 721522: 68, 721770: 64, 721830: 75, 721835: 89, 721950: 2, 722005: 29, 722030: 10, 722130: 87, 722172: 3, 722218: 77, 722285: 37, 722309: 8, 722326: 90, 722419: 65, 722589: 19, 722650: 79, 722680: 42, 722770: 82, 722861: 76, 722981: 77, 722991: 67, 723175: 43, 723186: 67, 723375: 34, 723390: 7, 723438: 23, 723510: 21, 723516: 57, 723545: 69, 723684: 74, 723686: 77, 723690: 63, 723880: 5, 723915: 42, 723944: 54, 723964: 56, 724020: 36, 724037: 69, 724041: 90, 724056: 98, 724062: 21, 724112: 84, 724456: 38, 724507: 33, 724545: 91, 724587: 15, 724719: 41, 724777: 62, 724781: 59, 724823: 28, 724855: 72, 724872: 20, 724907: 14, 724988: 22, 724995: 65, 725034: 41, 725082: 48, 725182: 97, 725232: 54, 725241: 23, 725246: 42, 725290: 55, 725329: 56, 725332: 80, 725396: 13, 725507: 77, 725530: 74, 725604: 43, 725686: 48, 725712: 38, 725719: 7, 725741: 90, 725742: 11, 725749: 19, 725762: 96, 725801: 77, 725832: 4, 725949: 99, 726051: 41, 726089: 88, 726136: 95, 726195: 98, 726232: 99, 726280: 13, 726325: 15, 726355: 51, 726387: 28, 726419: 91, 726574: 73, 726638: 11, 726644: 79, 726695: 5, 726782: 93, 726829: 15, 726893: 13, 727008: 25, 727012: 15, 727071: 25, 727132: 40, 727159: 67, 727250: 51, 727355: 99, 727429: 77, 727486: 39, 727756: 85, 727765: 20, 727810: 51, 727845: 85, 727995: 17, 728035: 25, 728321: 75, 728574: 51, 728649: 25, 728725: 81, 728727: 96, 728751: 79, 728782: 35, 728849: 26, 728880: 10, 729164: 0, 729196: 44, 729286: 20, 729296: 18, 729442: 81, 729486: 54, 729546: 84, 729552: 14, 729653: 5, 729696: 20, 729852: 26, 729945: 19, 729981: 15, 730024: 69, 730178: 64, 730201: 68, 730320: 97, 730391: 9, 730409: 22, 730517: 61, 730596: 77, 730698: 71, 730825: 29, 730846: 73, 730966: 5, 731027: 74, 731196: 29, 731211: 28, 731250: 82, 731382: 13, 731409: 10, 731518: 79, 731528: 15, 731580: 32, 731874: 88, 731904: 79, 731931: 16, 731949: 72, 732042: 39, 732078: 36, 732108: 9, 732176: 64, 732183: 77, 732211: 95, 732257: 23, 732373: 81, 732393: 40, 732511: 34, 732558: 47, 732562: 22, 732577: 60, 732662: 31, 732665: 17, 732716: 45, 732729: 58, 732730: 32, 732770: 93, 732811: 31, 732876: 91, 732878: 73, 732928: 65, 732987: 54, 733042: 61, 733045: 30, 733178: 62, 733566: 77, 733593: 85, 733658: 35, 733673: 47, 733689: 67, 733980: 66, 733989: 63, 734001: 87, 734077: 45, 734298: 0, 734338: 15, 734372: 97, 734417: 37, 734462: 10, 734469: 64, 734496: 65, 734511: 16, 734532: 56, 734559: 80, 734579: 17, 734623: 54, 734636: 39, 734666: 82, 734680: 32, 734715: 51, 734790: 84, 734947: 41, 734979: 40, 734999: 68, 735030: 35, 735046: 35, 735101: 70, 735123: 11, 735191: 3, 735219: 54, 735287: 23, 735327: 59, 735382: 7, 735447: 13, 735514: 34, 735582: 54, 735597: 2, 735716: 12, 735796: 38, 735799: 81, 735862: 15, 736012: 40, 736049: 30, 736097: 76, 736146: 98, 736172: 35, 736330: 71, 736333: 21, 736375: 4, 736578: 80, 736589: 71, 736710: 33, 736714: 49, 736927: 58, 736963: 97, 737004: 53, 737015: 10, 737016: 33, 737063: 74, 737066: 76, 737183: 76, 737321: 51, 737342: 5, 737371: 78, 737473: 64, 737515: 10, 737537: 42, 737586: 70, 737631: 33, 737658: 69, 737724: 61, 737799: 93, 737828: 85, 737878: 74, 737958: 5, 737976: 10, 738171: 74, 738177: 10, 738234: 7, 738238: 32, 738239: 30, 738259: 75, 738266: 93, 738271: 72, 738272: 81, 738337: 38, 738358: 68, 738420: 27, 738517: 57, 738519: 16, 738629: 46, 738770: 23, 738810: 90, 738840: 32, 738904: 32, 738958: 80, 739121: 64, 739296: 96, 739323: 80, 739335: 67, 739419: 41, 739425: 77, 739432: 78, 739548: 15, 739554: 42, 739650: 88, 739803: 3, 739823: 91, 739849: 22, 739877: 7, 739882: 21, 739947: 85, 740041: 53, 740044: 27, 740050: 4, 740282: 97, 740358: 89, 740365: 33, 740440: 97, 740600: 89, 740760: 13, 740801: 44, 740880: 82, 740894: 52, 740944: 78, 740950: 60, 740955: 20, 741004: 14, 741030: 90, 741170: 81, 741197: 87, 741302: 7, 741348: 97, 741364: 67, 741516: 76, 741553: 74, 741607: 36, 741828: 59, 741856: 86, 741909: 83, 741929: 9, 742041: 67, 742089: 85, 742095: 12, 742115: 96, 742273: 20, 742378: 39, 742465: 51, 742492: 65, 742499: 85, 742514: 64, 742516: 60, 742519: 63, 742532: 98, 742552: 82, 742666: 45, 742704: 50, 742789: 63, 742813: 1, 742877: 53, 742891: 78, 742918: 51, 742924: 89, 743000: 47, 743001: 57, 743013: 69, 743022: 96, 743057: 62, 743068: 0, 743124: 17, 743180: 28, 743208: 59, 743235: 93, 743269: 38, 743610: 63, 743613: 44, 743616: 34, 743791: 5, 743830: 91, 743874: 71, 743977: 76, 744167: 26, 744237: 4, 744490: 30, 744528: 2, 744558: 24, 744560: 79, 744609: 95, 744659: 36, 744672: 70, 744725: 59, 744781: 56, 744987: 11, 745002: 29, 745078: 90, 745126: 46, 745194: 75, 745203: 6, 745257: 47, 745267: 58, 745274: 35, 745293: 54, 745346: 78, 745432: 64, 745522: 38, 745598: 70, 745641: 12, 745730: 52, 745915: 61, 746033: 82, 746048: 39, 746096: 61, 746255: 57, 746270: 64, 746294: 20, 746422: 94, 746468: 9, 746475: 2, 746509: 98, 746546: 78, 746598: 22, 746716: 39, 746859: 74, 746885: 27, 746988: 71, 746993: 84, 747058: 57, 747104: 45, 747215: 9, 747240: 51, 747279: 37, 747367: 68, 747512: 91, 747514: 37, 747629: 22, 747929: 83, 747992: 29, 748011: 45, 748043: 23, 748052: 89, 748123: 92, 748150: 19, 748171: 31, 748277: 40, 748390: 2, 748392: 20, 748399: 78, 748402: 71, 748412: 31, 748616: 10, 748635: 63, 748677: 75, 748707: 75, 748733: 96, 748812: 78, 748872: 65, 748876: 19, 748973: 0, 749003: 29, 749012: 38, 749037: 69, 749269: 89, 749334: 70, 749404: 57, 749583: 68, 749679: 19, 749754: 30, 749802: 57, 749822: 8, 749878: 87, 749952: 24, 749994: 91, 750054: 80, 750078: 45, 750214: 14, 750228: 10, 750249: 83, 750283: 11, 750313: 81, 750509: 47, 750608: 17, 750683: 28, 750693: 58, 750700: 96, 750732: 18, 750927: 58, 751007: 23, 751048: 7, 751160: 20, 751196: 89, 751199: 48, 751224: 76, 751425: 76, 751475: 30, 751529: 47, 751662: 68, 751687: 43, 751708: 80, 751770: 42, 751771: 80, 751779: 83, 751870: 18, 752170: 60, 752190: 64, 752271: 3, 752293: 56, 752399: 65, 752571: 56, 752596: 91, 752683: 48, 752742: 38, 752757: 56, 752833: 12, 752947: 83, 752984: 45, 753156: 74, 753201: 76, 753211: 79, 753282: 6, 753307: 64, 753377: 59, 753592: 74, 753792: 7, 753871: 41, 753933: 42, 754298: 35, 754303: 6, 754568: 25, 754624: 18, 754738: 78, 754881: 57, 755024: 91, 755083: 59, 755154: 73, 755156: 2, 755209: 39, 755256: 60, 755360: 1, 755416: 82, 755503: 80, 755549: 53, 755554: 30, 755567: 96, 755584: 82, 755651: 48, 755731: 34, 755841: 64, 755924: 26, 755985: 40, 756040: 31, 756355: 98, 756371: 53, 756435: 94, 756605: 81, 756610: 87, 756632: 74, 756641: 51, 756676: 82, 756679: 91, 756725: 10, 756804: 31, 756898: 28, 757131: 11, 757191: 7, 757198: 55, 757247: 67, 757250: 37, 757418: 75, 757454: 47, 757475: 81, 757528: 94, 757608: 56, 757619: 55, 757620: 69, 757637: 67, 757648: 28, 757689: 26, 757721: 93, 757743: 70, 757769: 44, 757930: 1, 757953: 0, 757968: 38, 758128: 63, 758147: 40, 758176: 25, 758243: 14, 758246: 39, 758266: 89, 758287: 9, 758312: 8, 758402: 39, 758414: 44, 758525: 73, 758607: 83, 758654: 84, 758700: 39, 758739: 23, 758782: 35, 758906: 28, 758908: 76, 758952: 43, 759048: 13, 759085: 50, 759110: 6, 759135: 41, 759161: 34, 759178: 92, 759266: 34, 759268: 70, 759332: 72, 759355: 4, 759422: 22, 759475: 62, 759538: 49, 759562: 43, 759579: 83, 759588: 26, 759639: 98, 759656: 10, 759762: 11, 759799: 54, 760083: 49, 760156: 24, 760207: 48, 760249: 42, 760310: 67, 760333: 73, 760341: 31, 760348: 19, 760356: 78, 760378: 71, 760425: 89, 760489: 70, 760676: 46, 760682: 95, 760715: 44, 760836: 34, 760944: 34, 761018: 22, 761078: 19, 761155: 42, 761198: 22, 761258: 51, 761286: 62, 761419: 41, 761456: 19, 761499: 66, 761570: 94, 761572: 34, 761587: 21, 761622: 99, 761640: 60, 761749: 24, 761776: 69, 761823: 85, 761842: 18, 761867: 6, 761908: 16, 761947: 48, 762002: 54, 762011: 70, 762047: 69, 762093: 68, 762165: 77, 762300: 86, 762306: 34, 762324: 58, 762368: 94, 762561: 37, 762571: 75, 762682: 91, 762701: 38, 762765: 6, 762776: 46, 762785: 75, 762837: 26, 762843: 29, 762859: 54, 762896: 27, 762930: 59, 763080: 40, 763117: 54, 763143: 3, 763172: 95, 763341: 8, 763342: 54, 763409: 9, 763597: 56, 763632: 86, 763768: 61, 763884: 21, 764017: 31, 764061: 95, 764220: 81, 764339: 1, 764351: 47, 764488: 71, 764538: 39, 764562: 60, 764642: 8, 764724: 55, 764744: 42, 764824: 73, 764863: 24, 764874: 91, 764910: 31, 764933: 7, 764974: 91, 765077: 76, 765118: 86, 765284: 38, 765342: 86, 765429: 73, 765486: 3, 765611: 29, 765666: 25, 765707: 88, 765714: 22, 765735: 60, 765842: 71, 765920: 15, 765996: 78, 766026: 48, 766074: 20, 766079: 43, 766219: 1, 766245: 14, 766247: 35, 766325: 51, 766449: 48, 766492: 16, 766511: 7, 766570: 80, 766592: 91, 766708: 52, 766751: 74, 766763: 69, 766890: 73, 766910: 82, 767025: 90, 767125: 57, 767250: 63, 767268: 95, 767403: 4, 767654: 32, 767685: 58, 767693: 95, 767723: 30, 767826: 36, 767837: 7, 767890: 50, 767918: 8, 767970: 50, 768012: 97, 768066: 78, 768081: 8, 768188: 38, 768276: 24, 768455: 99, 768484: 89, 768596: 75, 768632: 9, 768760: 77, 768769: 20, 768802: 6, 768895: 8, 769003: 69, 769080: 25, 769194: 88, 769218: 72, 769249: 6, 769297: 69, 769420: 18, 769542: 53, 769551: 86, 769599: 69, 769620: 54, 769622: 87, 769696: 84, 769750: 64, 769790: 29, 769872: 91, 769892: 18, 769895: 29, 769969: 58, 769982: 43, 769993: 63, 769998: 22, 770016: 58, 770056: 16, 770083: 13, 770124: 6, 770298: 68, 770483: 66, 770515: 18, 770541: 99, 770716: 63, 770806: 60, 770831: 84, 770947: 88, 770978: 88, 771025: 58, 771040: 32, 771052: 82, 771074: 6, 771175: 9, 771285: 70, 771438: 20, 771532: 99, 771866: 84, 771893: 54, 771903: 93, 771909: 61, 771944: 30, 771974: 11, 772056: 56, 772348: 98, 772349: 75, 772545: 2, 772563: 56, 772596: 89, 772655: 23, 772781: 25, 772788: 15, 772790: 70, 772851: 52, 772858: 96, 772917: 93, 772947: 27, 772957: 71, 773013: 45, 773030: 49, 773219: 58, 773239: 61, 773240: 3, 773339: 15, 773412: 45, 773441: 43, 773479: 69, 773492: 9, 773514: 47, 773722: 37, 773730: 58, 773766: 47, 773959: 45, 774182: 27, 774199: 1, 774244: 91, 774248: 89, 774327: 64, 774367: 73, 774425: 62, 774483: 33, 774498: 93, 774624: 89, 774686: 3, 774748: 37, 774912: 17, 774990: 53, 775016: 99, 775198: 58, 775233: 25, 775273: 92, 775282: 21, 775471: 29, 775493: 91, 775523: 11, 775528: 20, 775587: 3, 775683: 75, 775724: 23, 775738: 88, 775907: 68, 775912: 60, 775914: 72, 775928: 41, 775969: 26, 776155: 52, 776185: 48, 776195: 7, 776262: 85, 776422: 89, 776428: 55, 776457: 34, 776482: 30, 776485: 67, 776564: 51, 776650: 8, 776684: 65, 776762: 62, 776820: 69, 776976: 26, 777096: 88, 777118: 31, 777149: 35, 777200: 90, 777239: 12, 777257: 2, 777275: 50, 777295: 33, 777359: 3, 777389: 12, 777443: 91, 777448: 77, 777455: 23, 777544: 99, 777740: 98, 777822: 2, 778068: 24, 778283: 25, 778328: 57, 778394: 18, 778798: 70, 778885: 78, 778908: 96, 778932: 59, 779017: 3, 779022: 75, 779042: 93, 779043: 44, 779054: 52, 779133: 7, 779252: 88, 779279: 61, 779328: 15, 779400: 27, 779497: 66, 779623: 40, 779753: 65, 779767: 8, 779787: 97, 779971: 85, 779996: 6, 780054: 73, 780088: 67, 780129: 1, 780146: 25, 780281: 15, 780305: 25, 780331: 28, 780386: 26, 780416: 52, 780655: 36, 780779: 71, 780866: 40, 780896: 63, 780972: 16, 781025: 16, 781041: 41, 781066: 45, 781148: 73, 781176: 87, 781316: 98, 781363: 96, 781495: 84, 781521: 1, 781565: 65, 781635: 23, 781636: 24, 781701: 50, 781713: 40, 781794: 43, 781813: 7, 781920: 85, 781939: 26, 782013: 70, 782035: 77, 782127: 11, 782142: 34, 782147: 99, 782402: 69, 782464: 68, 782471: 3, 782495: 68, 782513: 99, 782652: 37, 782707: 30, 782745: 62, 782762: 79, 782848: 60, 782893: 63, 782900: 66, 782982: 30, 783015: 50, 783190: 45, 783394: 42, 783438: 72, 783454: 97, 783483: 36, 783510: 5, 783515: 24, 783558: 90, 783562: 78, 783661: 82, 783725: 32, 783746: 38, 783753: 5, 783882: 33, 783886: 75, 783930: 32, 783981: 59, 784133: 10, 784200: 58, 784203: 7, 784214: 49, 784264: 35, 784382: 70, 784438: 43, 784613: 33, 784695: 97, 784745: 73, 784802: 93, 784893: 26, 784948: 17, 784985: 19, 785015: 23, 785029: 73, 785135: 46, 785142: 14, 785296: 75, 785322: 18, 785391: 98, 785436: 92, 785456: 45, 785597: 54, 785605: 23, 785654: 23, 785695: 79, 785699: 12, 785749: 2, 785751: 9, 785835: 17, 785843: 55, 785942: 1, 785951: 99, 785984: 9, 785994: 15, 786015: 91, 786112: 27, 786208: 90, 786216: 36, 786223: 70, 786332: 96, 786462: 28, 786484: 2, 786581: 25, 786639: 29, 786641: 93, 786691: 39, 786740: 16, 786774: 93, 786783: 50, 786888: 94, 786913: 58, 786916: 4, 786950: 15, 786955: 15, 786980: 74, 787086: 56, 787111: 77, 787120: 87, 787121: 2, 787187: 46, 787272: 65, 787298: 81, 787467: 33, 787495: 29, 787531: 44, 787556: 39, 787574: 2, 787604: 34, 787657: 55, 787723: 15, 787875: 25, 787888: 56, 787991: 70, 788007: 55, 788053: 78, 788093: 67, 788116: 69, 788198: 61, 788343: 32, 788477: 40, 788546: 38, 788649: 83, 788650: 16, 788736: 26, 788831: 20, 788938: 97, 789013: 73, 789045: 54, 789123: 48, 789150: 37, 789170: 84, 789550: 20, 789582: 84, 789618: 90, 789660: 2, 789681: 81, 789789: 54, 789814: 43, 789905: 1, 789910: 43, 789945: 17, 789963: 91, 790020: 48, 790048: 38, 790083: 43, 790158: 47, 790247: 29, 790257: 15, 790289: 35, 790506: 98, 790529: 14, 790594: 66, 790625: 30, 790659: 66, 790706: 12, 790733: 92, 790749: 57, 790821: 90, 790828: 29, 790840: 68, 790931: 82, 791028: 88, 791122: 78, 791190: 16, 791311: 9, 791374: 94, 791398: 31, 791403: 22, 791536: 14, 791638: 17, 791641: 63, 791654: 36, 791720: 58, 791858: 67, 791890: 53, 791949: 57, 792019: 40, 792032: 53, 792116: 99, 792179: 22, 792251: 75, 792402: 8, 792418: 66, 792433: 19, 792670: 13, 792741: 92, 792755: 78, 792794: 44, 792873: 12, 792911: 91, 792969: 10, 793090: 44, 793177: 94, 793190: 35, 793295: 18, 793298: 91, 793308: 38, 793419: 78, 793460: 3, 793505: 9, 793547: 97, 793647: 37, 793675: 74, 793750: 18, 793792: 97, 793837: 77, 793941: 92, 794040: 66, 794093: 0, 794103: 53, 794199: 97, 794220: 38, 794227: 68, 794249: 5, 794280: 22, 794342: 61, 794380: 25, 794490: 28, 794509: 48, 794609: 19, 794690: 8, 794773: 28, 794785: 58, 794794: 54, 794894: 52, 794967: 87, 795069: 77, 795112: 97, 795356: 99, 795413: 85, 795421: 55, 795433: 71, 795589: 60, 795592: 18, 795799: 30, 795835: 35, 795876: 12, 795893: 83, 795960: 71, 796023: 91, 796088: 35, 796146: 61, 796217: 82, 796339: 13, 796349: 29, 796353: 63, 796357: 43, 796368: 18, 796390: 34, 796392: 27, 796429: 65, 796504: 66, 796557: 23, 796627: 8, 796636: 72, 796669: 19, 796713: 8, 796760: 98, 796766: 3, 796803: 90, 796819: 27, 796875: 78, 796973: 62, 797023: 91, 797027: 15, 797049: 93, 797062: 22, 797105: 72, 797219: 85, 797226: 63, 797281: 96, 797325: 77, 797400: 90, 797421: 25, 797566: 35, 797574: 88, 797648: 80, 797673: 18, 797684: 45, 797797: 32, 797807: 77, 797880: 50, 797960: 46, 797982: 30, 798062: 47, 798076: 85, 798148: 82, 798231: 81, 798273: 71, 798287: 22, 798305: 54, 798368: 29, 798399: 36, 798470: 92, 798555: 26, 798621: 66, 798702: 85, 798715: 17, 798759: 4, 798943: 46, 799071: 92, 799182: 28, 799191: 13, 799275: 8, 799319: 65, 799325: 13, 799343: 67, 799360: 90, 799635: 69, 799642: 94, 799662: 19, 799693: 70, 799704: 4, 799803: 8, 799920: 21, 799922: 19, 800026: 41, 800283: 38, 800396: 39, 800404: 82, 800464: 37, 800652: 7, 800655: 8, 800683: 40, 800687: 30, 800932: 65, 800962: 95, 800972: 16, 801035: 90, 801274: 82, 801280: 36, 801364: 85, 801420: 35, 801428: 58, 801441: 82, 801522: 50, 801580: 62, 801657: 7, 801758: 45, 801793: 50, 801871: 93, 801886: 13, 802004: 12, 802160: 18, 802270: 67, 802271: 3, 802376: 51, 802419: 15, 802421: 32, 802470: 22, 802627: 21, 802681: 0, 802685: 12, 802693: 12, 802710: 47, 802737: 81, 802913: 93, 802968: 82, 803000: 57, 803042: 50, 803085: 22, 803118: 84, 803244: 25, 803245: 43, 803262: 26, 803358: 24, 803485: 99, 803527: 8, 803564: 97, 803612: 62, 803761: 62, 803771: 22, 803850: 39, 803889: 37, 804023: 80, 804078: 1, 804118: 44, 804296: 19, 804312: 96, 804326: 25, 804368: 50, 804414: 24, 804444: 36, 804484: 71, 804514: 49, 804682: 76, 804791: 90, 804884: 18, 804889: 9, 805020: 59, 805046: 70, 805092: 17, 805156: 35, 805190: 57, 805504: 45, 805576: 40, 805592: 90, 805660: 47, 805736: 13, 805766: 75, 805801: 39, 805815: 94, 805818: 2, 805943: 89, 806099: 52, 806134: 55, 806144: 50, 806164: 83, 806263: 73, 806473: 30, 806484: 84, 806504: 13, 806507: 48, 806527: 44, 806603: 82, 806605: 74, 806677: 45, 806702: 94, 806779: 54, 806799: 13, 806972: 12, 807010: 39, 807094: 95, 807107: 63, 807203: 64, 807209: 33, 807407: 93, 807571: 78, 807619: 60, 807636: 72, 807703: 23, 807892: 98, 807959: 77, 808066: 80, 808095: 53, 808113: 60, 808158: 8, 808197: 1, 808246: 92, 808318: 94, 808326: 0, 808337: 69, 808558: 47, 808613: 24, 808614: 58, 808663: 8, 808706: 34, 808829: 67, 808838: 79, 808850: 59, 808867: 64, 808900: 75, 808922: 79, 808945: 16, 809011: 18, 809153: 7, 809196: 88, 809242: 88, 809303: 23, 809316: 30, 809319: 9, 809329: 55, 809383: 74, 809397: 88, 809399: 93, 809483: 11, 809540: 77, 809541: 20, 809609: 75, 809627: 63, 809690: 83, 809799: 83, 809955: 81, 810005: 92, 810041: 53, 810047: 80, 810244: 42, 810389: 8, 810441: 65, 810560: 83, 810683: 30, 810935: 84, 810941: 46, 810950: 85, 810992: 59, 811237: 7, 811400: 18, 811416: 67, 811425: 26, 811551: 1, 811557: 40, 811563: 38, 811566: 19, 811687: 86, 811757: 50, 811766: 90, 811781: 35, 811782: 14, 811930: 26, 811981: 0, 812016: 40, 812207: 69, 812213: 68, 812279: 72, 812314: 7, 812348: 90, 812434: 64, 812486: 83, 812621: 67, 812700: 96, 812733: 81, 812749: 72, 812764: 24, 812857: 24, 812888: 58, 812941: 44, 812985: 89, 813054: 63, 813117: 64, 813198: 31, 813275: 25, 813310: 39, 813322: 20, 813491: 44, 813523: 93, 813538: 97, 813638: 25, 813668: 32, 813865: 62, 813891: 88, 813903: 51, 814028: 83, 814157: 56, 814260: 53, 814296: 41, 814343: 94, 814386: 84, 814466: 39, 814647: 68, 814684: 63, 814695: 90, 814707: 89, 814916: 47, 814941: 6, 814960: 13, 814981: 33, 815018: 22, 815074: 30, 815121: 38, 815136: 72, 815366: 6, 815390: 97, 815445: 55, 815461: 39, 815499: 93, 815617: 69, 815622: 79, 815668: 2, 815845: 75, 815938: 19, 815984: 29, 815993: 83, 816030: 71, 816149: 24, 816280: 41, 816287: 48, 816348: 63, 816359: 46, 816386: 13, 816459: 84, 816483: 69, 816578: 15, 816672: 74, 816779: 23, 816853: 51, 816931: 47, 817010: 52, 817044: 27, 817093: 54, 817191: 47, 817198: 78, 817212: 86, 817274: 77, 817306: 11, 817345: 31, 817354: 90, 817370: 76, 817385: 11, 817454: 86, 817457: 44, 817463: 1, 817484: 22, 817555: 3, 817561: 88, 817702: 59, 817767: 16, 817873: 59, 818080: 29, 818081: 6, 818114: 74, 818145: 57, 818152: 21, 818173: 59, 818272: 93, 818298: 94, 818452: 13, 818559: 89, 818608: 10, 818621: 18, 818625: 36, 818627: 42, 818658: 35, 818764: 63, 818858: 41, 818860: 22, 819169: 70, 819330: 70, 819371: 35, 819372: 39, 819471: 82, 819541: 3, 819703: 0, 819731: 39, 819736: 55, 819782: 14, 819806: 88, 819868: 8, 819872: 73, 819890: 97, 819912: 79, 819933: 19, 819953: 51, 820027: 65, 820093: 67, 820113: 48, 820238: 45, 820302: 57, 820337: 90, 820378: 98, 820427: 75, 820522: 16, 820601: 12, 820618: 17, 820628: 64, 820781: 75, 820821: 88, 820980: 58, 820983: 7, 821008: 46, 821034: 27, 821171: 80, 821199: 97, 821274: 38, 821381: 23, 821387: 58, 821408: 18, 821416: 6, 821469: 34, 821591: 89, 821665: 76, 821683: 89, 821727: 31, 821793: 86, 821819: 78, 821858: 50, 821990: 46, 822009: 3, 822103: 40, 822149: 96, 822157: 50, 822189: 54, 822250: 78, 822306: 92, 822346: 50, 822433: 46, 822494: 17, 822501: 24, 822724: 70, 822949: 97, 822976: 40, 822989: 92, 823024: 81, 823056: 49, 823081: 71, 823083: 64, 823150: 53, 823170: 81, 823231: 98, 823235: 21, 823264: 38, 823436: 29, 823437: 86, 823474: 0, 823489: 99, 823624: 21, 823724: 68, 823730: 5, 823911: 90, 824041: 24, 824106: 27, 824115: 20, 824121: 52, 824164: 90, 824209: 99, 824210: 12, 824262: 51, 824285: 60, 824300: 30, 824307: 54, 824470: 7, 824479: 32, 824528: 84, 824702: 16, 824724: 70, 824878: 9, 824954: 27, 824984: 56, 825025: 92, 825043: 88, 825098: 92, 825124: 92, 825128: 35, 825220: 74, 825346: 9, 825373: 41, 825427: 45, 825532: 33, 825596: 5, 825623: 83, 825656: 71, 825696: 50, 825726: 77, 825775: 56, 825818: 67, 825875: 3, 825901: 44, 825967: 50, 826117: 79, 826118: 58, 826149: 68, 826154: 68, 826237: 98, 826239: 36, 826308: 42, 826584: 54, 826594: 54, 826680: 5, 826736: 57, 826754: 94, 826765: 19, 826851: 19, 826902: 5, 826919: 61, 827085: 16, 827127: 52, 827176: 6, 827382: 75, 827387: 98, 827408: 50, 827834: 13, 827924: 65, 828106: 46, 828432: 36, 828446: 72, 828662: 83, 828679: 17, 828710: 27, 828948: 67, 828963: 60, 828964: 1, 829049: 65, 829074: 76, 829119: 40, 829186: 79, 829279: 65, 829335: 62, 829540: 30, 829553: 6, 829731: 78, 829776: 94, 829789: 46, 829812: 33, 829968: 48, 830037: 14, 830116: 69, 830199: 97, 830311: 77, 830378: 51, 830379: 92, 830397: 1, 830440: 38, 830451: 11, 830510: 0, 830610: 92, 830657: 77, 830742: 22, 830743: 41, 830945: 80, 830985: 46, 831031: 29, 831060: 70, 831103: 35, 831113: 48, 831195: 80, 831517: 3, 831805: 47, 831816: 1, 832073: 42, 832127: 62, 832215: 32, 832237: 98, 832344: 10, 832365: 82, 832395: 82, 832403: 45, 832456: 85, 832470: 83, 832503: 80, 832795: 94, 832975: 90, 833056: 3, 833136: 73, 833142: 69, 833326: 71, 833399: 15, 833517: 97, 833621: 32, 833644: 1, 833659: 87, 833806: 51, 833810: 16, 833926: 96, 833943: 89, 833960: 46, 833992: 70, 834008: 60, 834045: 45, 834124: 10, 834127: 20, 834175: 15, 834254: 39, 834286: 71, 834352: 15, 834404: 3, 834476: 5, 834587: 18, 834657: 22, 834773: 84, 834789: 54, 834821: 34, 834901: 30, 834922: 62, 834950: 84, 835039: 87, 835090: 41, 835202: 92, 835217: 55, 835259: 14, 835412: 49, 835512: 98, 835606: 50, 835678: 21, 835706: 50, 835735: 36, 835764: 94, 835877: 67, 835904: 79, 835915: 78, 835951: 4, 835961: 28, 836011: 7, 836088: 40, 836109: 85, 836137: 80, 836149: 68, 836171: 81, 836173: 2, 836188: 12, 836190: 8, 836201: 47, 836256: 58, 836303: 9, 836598: 99, 836654: 82, 836680: 14, 836750: 62, 836754: 20, 836786: 39, 836800: 26, 836818: 20, 836898: 4, 836942: 74, 836943: 35, 836964: 62, 837031: 16, 837088: 13, 837104: 29, 837105: 79, 837133: 95, 837218: 18, 837240: 84, 837256: 6, 837387: 97, 837414: 44, 837479: 32, 837516: 90, 837696: 53, 837843: 97, 837858: 55, 837860: 76, 837873: 64, 838013: 39, 838071: 41, 838079: 46, 838116: 69, 838153: 60, 838158: 64, 838187: 55, 838343: 83, 838417: 13, 838436: 49, 838451: 54, 838458: 47, 838517: 34, 838561: 52, 838563: 56, 838780: 6, 838806: 41, 838831: 37, 838894: 46, 838945: 86, 838950: 79, 838991: 10, 839001: 59, 839035: 56, 839036: 26, 839229: 17, 839291: 17, 839321: 53, 839333: 20, 839396: 84, 839436: 44, 839523: 79, 839530: 63, 839633: 16, 839644: 35, 839708: 48, 839744: 51, 839786: 55, 839829: 11, 839871: 4, 839887: 54, 839955: 11, 839970: 34, 839972: 72, 840081: 73, 840122: 36, 840246: 43, 840284: 46, 840345: 35, 840384: 77, 840402: 10, 840469: 66, 840494: 86, 840596: 9, 840678: 71, 840764: 55, 840792: 95, 840984: 73, 841031: 11, 841037: 60, 841057: 48, 841075: 19, 841076: 50, 841120: 27, 841127: 54, 841224: 61, 841312: 94, 841314: 45, 841469: 22, 841495: 6, 841586: 30, 841745: 65, 841770: 10, 841884: 80, 841885: 73, 841986: 0, 841992: 39, 842049: 7, 842344: 13, 842440: 14, 842648: 80, 842703: 13, 842750: 58, 842793: 98, 842878: 57, 842899: 76, 842913: 68, 843028: 92, 843126: 68, 843215: 48, 843381: 47, 843417: 69, 843433: 12, 843548: 46, 843592: 57, 843606: 36, 843617: 83, 843683: 53, 843736: 22, 843787: 80, 843805: 65, 843875: 99, 843886: 50, 844050: 90, 844084: 11, 844107: 58, 844144: 43, 844278: 81, 844327: 1, 844401: 52, 844573: 67, 844590: 11, 844593: 22, 844604: 64, 844612: 14, 844648: 77, 844719: 41, 844734: 64, 844760: 15, 844882: 82, 844898: 95, 844975: 82, 845102: 11, 845179: 1, 845241: 99, 845283: 28, 845396: 0, 845534: 76, 845704: 9, 845720: 11, 845728: 14, 845743: 1, 845766: 74, 845842: 37, 845843: 57, 845892: 31, 845903: 76, 845963: 17, 845995: 25, 846107: 9, 846249: 2, 846255: 46, 846375: 18, 846493: 73, 846505: 22, 846595: 71, 846623: 26, 846626: 42, 846930: 1, 846976: 28, 847088: 79, 847106: 6, 847120: 90, 847137: 26, 847144: 21, 847236: 38, 847361: 65, 847364: 86, 847407: 52, 847413: 19, 847431: 13, 847451: 21, 847466: 18, 847514: 22, 847528: 86, 847685: 39, 847686: 9, 847700: 72, 847778: 58, 847838: 68, 847858: 59, 847915: 97, 848072: 46, 848163: 72, 848184: 29, 848220: 83, 848227: 83, 848244: 81, 848418: 88, 848451: 34, 848521: 92, 848589: 46, 848609: 93, 848722: 13, 848770: 14, 848787: 54, 848796: 93, 848886: 18, 848921: 63, 848951: 33, 848964: 15, 849030: 99, 849132: 14, 849303: 96, 849364: 66, 849458: 23, 849653: 88, 849921: 80, 850121: 86, 850295: 45, 850453: 89, 850570: 94, 850587: 32, 850620: 21, 850644: 2, 850671: 48, 850774: 41, 850843: 76, 850885: 98, 850932: 97, 850967: 3, 850988: 98, 850998: 65, 851081: 87, 851105: 80, 851172: 53, 851189: 19, 851268: 37, 851356: 94, 851360: 37, 851450: 54, 851493: 28, 851580: 94, 851841: 69, 851852: 89, 851864: 2, 851941: 26, 852136: 65, 852188: 88, 852222: 76, 852281: 36, 852455: 2, 852534: 67, 852570: 43, 852602: 13, 852614: 8, 852700: 2, 852763: 61, 852891: 46, 852952: 87, 853062: 73, 853190: 80, 853220: 41, 853294: 48, 853398: 36, 853515: 35, 853594: 67, 853818: 42, 853852: 54, 853887: 52, 853989: 73, 854049: 29, 854109: 75, 854114: 74, 854323: 14, 854351: 37, 854374: 56, 854377: 77, 854392: 65, 854512: 69, 854517: 96, 854681: 28, 854801: 7, 854976: 11, 855235: 61, 855241: 51, 855302: 38, 855376: 37, 855452: 73, 855456: 6, 855482: 96, 855487: 10, 855501: 89, 855539: 53, 855595: 95, 855640: 68, 855706: 78, 855712: 77, 855719: 85, 855815: 80, 855836: 64, 855932: 96, 856047: 77, 856196: 71, 856204: 78, 856208: 34, 856214: 20, 856226: 64, 856269: 71, 856298: 24, 856334: 66, 856423: 17, 856463: 56, 856571: 8, 856775: 89, 856787: 33, 856888: 95, 856909: 55, 857052: 85, 857068: 19, 857076: 62, 857088: 30, 857097: 63, 857103: 3, 857239: 19, 857322: 13, 857372: 9, 857547: 9, 857609: 67, 857794: 69, 857907: 16, 858067: 71, 858120: 12, 858274: 11, 858276: 27, 858394: 0, 858514: 81, 858563: 90, 858576: 92, 858598: 31, 858601: 27, 858689: 58, 858700: 53, 858768: 44, 858833: 99, 858851: 93, 858901: 23, 858906: 91, 859052: 3, 859065: 85, 859179: 29, 859197: 16, 859262: 74, 859293: 30, 859333: 86, 859479: 46, 859521: 10, 859539: 56, 859595: 68, 859671: 8, 859767: 3, 859796: 2, 859817: 99, 859829: 63, 859942: 88, 859945: 0, 859981: 67, 860100: 67, 860173: 99, 860210: 48, 860218: 53, 860578: 68, 860591: 91, 860628: 7, 860688: 82, 860792: 35, 860930: 16, 860991: 2, 861010: 1, 861237: 75, 861238: 64, 861383: 8, 861390: 6, 861415: 62, 861450: 17, 861458: 78, 861499: 68, 861590: 76, 861603: 9, 861714: 24, 861760: 11, 861828: 67, 861953: 99, 862042: 50, 862070: 4, 862302: 59, 862586: 97, 862660: 48, 862713: 39, 862852: 56, 862928: 66, 862931: 71, 863056: 77, 863113: 46, 863260: 10, 863310: 66, 863362: 34, 863434: 29, 863461: 65, 863464: 23, 863567: 22, 863592: 2, 863611: 78, 863695: 64, 863696: 1, 863795: 69, 863883: 14, 863931: 10, 864082: 85, 864199: 62, 864206: 39, 864225: 39, 864237: 80, 864283: 83, 864285: 28, 864294: 81, 864341: 17, 864539: 46, 864564: 7, 864647: 34, 864694: 98, 864728: 9, 864809: 81, 864847: 78, 864917: 39, 864991: 93, 865030: 58, 865096: 79, 865190: 2, 865249: 32, 865393: 21, 865576: 16, 865642: 7, 865643: 39, 865658: 28, 865665: 50, 865683: 60, 865698: 28, 865822: 46, 865889: 27, 865939: 25, 866034: 44, 866179: 31, 866371: 0, 866449: 39, 866463: 40, 866517: 47, 866680: 35, 866688: 72, 866702: 42, 866882: 59, 866885: 82, 866890: 27, 866903: 65, 866918: 59, 866937: 68, 867050: 32, 867107: 37, 867110: 98, 867111: 7, 867147: 28, 867175: 31, 867204: 6, 867235: 90, 867260: 97, 867289: 91, 867297: 50, 867312: 84, 867329: 59, 867419: 90, 867426: 73, 867472: 6, 867534: 92, 867597: 34, 867601: 7, 867614: 73, 867766: 64, 867778: 13, 867804: 77, 867818: 54, 867890: 95, 867943: 86, 867948: 97, 868004: 93, 868031: 93, 868262: 40, 868272: 38, 868309: 41, 868399: 72, 868465: 39, 868534: 83, 868631: 22, 868696: 14, 868704: 83, 868731: 82, 868897: 88, 869059: 39, 869344: 27, 869485: 84, 869613: 25, 869623: 20, 869632: 8, 869764: 70, 869785: 58, 869825: 45, 869879: 4, 869983: 74, 870160: 61, 870161: 31, 870271: 20, 870346: 65, 870455: 12, 870554: 69, 870558: 20, 870725: 13, 870788: 69, 870825: 41, 871057: 0, 871200: 99, 871209: 71, 871229: 27, 871230: 70, 871297: 82, 871328: 89, 871352: 24, 871380: 40, 871497: 68, 871541: 21, 871544: 64, 871567: 57, 871574: 30, 871595: 22, 871641: 50, 871645: 15, 871787: 86, 871887: 96, 872073: 13, 872112: 38, 872238: 18, 872249: 72, 872277: 29, 872303: 40, 872481: 69, 872543: 36, 872575: 39, 872621: 15, 872655: 42, 872663: 82, 872677: 3, 872701: 88, 872726: 3, 872759: 1, 872859: 33, 873005: 32, 873153: 91, 873158: 77, 873297: 22, 873340: 18, 873379: 78, 873381: 16, 873508: 48, 873517: 12, 873628: 49, 873654: 53, 873732: 11, 873781: 46, 873787: 45, 873916: 84, 874052: 82, 874097: 1, 874114: 29, 874168: 90, 874185: 70, 874232: 86, 874253: 8, 874367: 82, 874467: 2, 874554: 81, 874555: 73, 874566: 8, 874653: 47, 874683: 68, 874864: 90, 874930: 23, 874975: 14, 874999: 97, 875135: 0, 875148: 2, 875160: 96, 875193: 55, 875212: 23, 875293: 80, 875347: 11, 875355: 54, 875409: 80, 875415: 12, 875515: 56, 875614: 55, 875671: 75, 875682: 40, 875697: 66, 875747: 51, 875763: 96, 875774: 52, 875788: 22, 875802: 42, 875925: 66, 876043: 50, 876085: 50, 876116: 83, 876187: 54, 876216: 76, 876236: 84, 876319: 77, 876331: 20, 876376: 15, 876393: 44, 876394: 44, 876401: 14, 876593: 33, 876633: 72, 876889: 65, 876895: 32, 876898: 40, 876961: 43, 877006: 25, 877116: 25, 877150: 92, 877320: 47, 877351: 40, 877356: 66, 877680: 8, 877808: 80, 877852: 14, 878000: 74, 878005: 14, 878259: 92, 878328: 22, 878372: 43, 878373: 51, 878497: 85, 878559: 24, 878579: 53, 878605: 40, 878682: 69, 878731: 61, 878773: 81, 878845: 46, 878917: 36, 878960: 35, 878964: 3, 879031: 95, 879137: 6, 879176: 23, 879199: 0, 879259: 15, 879364: 67, 879447: 14, 879452: 43, 879457: 53, 879462: 76, 879488: 49, 879641: 48, 879666: 44, 879667: 12, 879732: 31, 879733: 15, 879768: 2, 879940: 48, 879947: 86, 879995: 30, 880223: 49, 880238: 77, 880256: 39, 880312: 52, 880326: 50, 880396: 56, 880457: 52, 880464: 68, 880574: 5, 880647: 56, 880702: 35, 880744: 46, 880820: 20, 880850: 42, 880860: 67, 880861: 94, 880887: 12, 880929: 5, 880980: 53, 881027: 41, 881038: 90, 881041: 71, 881185: 8, 881196: 90, 881202: 43, 881210: 98, 881227: 38, 881298: 0, 881327: 16, 881352: 51, 881481: 52, 881535: 80, 881557: 62, 881637: 70, 881881: 66, 881885: 45, 881932: 27, 881980: 13, 882012: 8, 882027: 24, 882119: 42, 882158: 81, 882191: 62, 882236: 31, 882377: 77, 882430: 63, 882504: 5, 882713: 42, 882774: 42, 882799: 23, 882821: 96, 882895: 64, 882996: 80, 883051: 8, 883147: 39, 883196: 28, 883257: 66, 883297: 55, 883299: 83, 883313: 96, 883324: 89, 883347: 38, 883380: 91, 883385: 68, 883684: 58, 883751: 3, 883853: 72, 883861: 40, 884168: 24, 884189: 16, 884222: 81, 884282: 79, 884440: 60, 884501: 94, 884521: 9, 884565: 55, 884571: 40, 884594: 62, 884603: 62, 884622: 32, 884686: 93, 884923: 25, 884977: 1, 885002: 60, 885097: 16, 885103: 55, 885228: 39, 885307: 53, 885344: 59, 885471: 23, 885588: 72, 885601: 51, 885608: 25, 885610: 26, 885633: 64, 885634: 29, 885877: 80, 885884: 73, 885940: 54, 885949: 23, 885958: 55, 886016: 91, 886034: 70, 886113: 52, 886129: 0, 886224: 8, 886293: 1, 886316: 15, 886317: 41, 886412: 56, 886442: 94, 886497: 3, 886703: 3, 886877: 49, 886878: 53, 886922: 86, 886984: 57, 887012: 61, 887043: 80, 887066: 34, 887159: 71, 887170: 14, 887199: 41, 887346: 78, 887377: 56, 887383: 74, 887483: 89, 887511: 52, 887533: 45, 887701: 19, 887840: 44, 887994: 4, 888002: 20, 888096: 54, 888182: 75, 888321: 8, 888322: 52, 888415: 61, 888507: 33, 888586: 39, 888681: 57, 888712: 42, 888794: 75, 888812: 38, 888849: 91, 888856: 70, 888904: 73, 889087: 41, 889165: 46, 889248: 28, 889266: 84, 889284: 47, 889319: 11, 889525: 68, 889769: 94, 889860: 36, 889997: 78, 889998: 96, 890154: 47, 890155: 35, 890248: 5, 890299: 23, 890303: 1, 890385: 60, 890386: 10, 890515: 20, 890532: 35, 890559: 9, 890576: 50, 890730: 11, 890751: 14, 890932: 63, 890971: 1, 891012: 92, 891253: 17, 891255: 27, 891307: 95, 891368: 19, 891502: 55, 891545: 46, 891660: 79, 891745: 42, 891916: 91, 892127: 72, 892177: 41, 892188: 6, 892247: 48, 892290: 14, 892319: 0, 892335: 94, 892443: 87, 892501: 97, 892516: 66, 892535: 71, 892664: 95, 892692: 76, 892706: 41, 892762: 18, 892797: 77, 892855: 5, 892873: 57, 892894: 55, 892898: 87, 892911: 62, 892944: 40, 893042: 51, 893079: 56, 893184: 9, 893194: 4, 893242: 69, 893551: 81, 893667: 40, 893818: 94, 893839: 15, 893903: 60, 893905: 7, 893982: 74, 893983: 3, 894195: 96, 894233: 15, 894367: 77, 894407: 89, 894507: 43, 894557: 19, 894587: 39, 894707: 59, 894745: 49, 894805: 11, 894822: 44, 894836: 23, 894843: 0, 894854: 73, 894980: 86, 895009: 15, 895071: 33, 895266: 64, 895344: 44, 895423: 33, 895431: 8, 895452: 78, 895501: 24, 895515: 35, 895537: 69, 895727: 96, 895735: 62, 895736: 73, 895769: 12, 895790: 39, 895920: 52, 895924: 43, 895934: 78, 896078: 84, 896250: 54, 896266: 55, 896361: 94, 896450: 95, 896495: 2, 896582: 56, 896774: 29, 896893: 97, 896914: 44, 896929: 61, 896948: 28, 897054: 35, 897089: 95, 897192: 97, 897229: 35, 897299: 96, 897311: 89, 897323: 2, 897336: 35, 897351: 27, 897360: 79, 897398: 33, 897747: 23, 897796: 0, 898093: 65, 898136: 23, 898191: 65, 898193: 41, 898281: 67, 898357: 68, 898360: 97, 898378: 94, 898460: 71, 898494: 20, 898564: 46, 898628: 56, 898681: 39, 898685: 17, 898820: 56, 898953: 50, 899063: 67, 899162: 44, 899371: 21, 899431: 96, 899561: 98, 899593: 56, 899680: 6, 899685: 98, 899709: 19, 899746: 76, 899851: 95, 899853: 83, 900011: 53, 900025: 18, 900056: 53, 900057: 98, 900099: 32, 900162: 31, 900237: 51, 900262: 63, 900316: 47, 900421: 15, 900630: 45, 900723: 42, 900791: 60, 900856: 85, 900925: 12, 901015: 41, 901024: 43, 901026: 69, 901035: 50, 901092: 25, 901102: 60, 901158: 79, 901218: 17, 901220: 78, 901273: 79, 901276: 10, 901295: 76, 901311: 43, 901349: 57, 901357: 4, 901592: 95, 901710: 84, 901805: 74, 901898: 7, 901930: 8, 901942: 19, 902073: 77, 902076: 48, 902196: 21, 902227: 31, 902253: 32, 902296: 23, 902358: 27, 902363: 49, 902396: 34, 902437: 86, 902464: 23, 902552: 11, 902706: 79, 902878: 52, 902885: 51, 903007: 7, 903128: 23, 903164: 50, 903249: 16, 903323: 66, 903448: 18, 903491: 25, 903533: 12, 903588: 51, 903668: 74, 903669: 26, 903722: 58, 903872: 59, 903907: 74, 903947: 68, 903974: 35, 903980: 81, 904112: 39, 904155: 72, 904209: 14, 904255: 27, 904278: 63, 904286: 28, 904496: 84, 904519: 35, 904523: 49, 904538: 70, 904543: 58, 904597: 40, 904604: 6, 904689: 21, 904724: 48, 904775: 93, 904932: 43, 904936: 44, 905019: 35, 905021: 4, 905041: 78, 905044: 84, 905048: 67, 905049: 37, 905052: 90, 905080: 8, 905113: 33, 905121: 54, 905189: 61, 905190: 21, 905196: 53, 905422: 92, 905513: 24, 905590: 37, 905897: 33, 905929: 7, 905968: 8, 905979: 26, 906099: 90, 906285: 58, 906382: 85, 906460: 80, 906474: 55, 906498: 22, 906501: 74, 906532: 43, 906540: 17, 906561: 25, 906636: 41, 906708: 93, 906747: 74, 906751: 59, 906768: 69, 907013: 29, 907172: 65, 907209: 42, 907251: 78, 907443: 54, 907469: 13, 907538: 30, 907563: 62, 907602: 23, 907614: 81, 907620: 99, 907685: 25, 907922: 16, 907923: 47, 908042: 86, 908057: 22, 908082: 99, 908134: 13, 908311: 28, 908436: 28, 908488: 42, 908584: 95, 908658: 98, 908701: 16, 908708: 31, 908768: 60, 908843: 48, 908855: 33, 908939: 33, 909146: 12, 909160: 56, 909197: 59, 909201: 11, 909424: 13, 909444: 44, 909498: 2, 909758: 47, 909775: 26, 909776: 82, 909850: 10, 909854: 85, 909927: 44, 909980: 10, 910036: 34, 910198: 1, 910263: 85, 910288: 38, 910407: 77, 910454: 13, 910456: 60, 910545: 61, 910630: 38, 910723: 87, 910813: 21, 910868: 55, 910885: 76, 910982: 34, 911001: 48, 911010: 75, 911081: 16, 911093: 54, 911304: 18, 911306: 6, 911398: 21, 911425: 25, 911536: 47, 911548: 42, 911648: 86, 911751: 30, 911882: 85, 911911: 64, 911917: 11, 911997: 31, 912112: 86, 912167: 29, 912246: 23, 912249: 88, 912461: 59, 912479: 21, 912488: 56, 912491: 31, 912579: 5, 912611: 91, 912704: 17, 912748: 75, 912820: 56, 912934: 85, 913009: 85, 913035: 90, 913048: 67, 913144: 70, 913188: 15, 913201: 84, 913264: 53, 913277: 5, 913303: 3, 913304: 22, 913586: 0, 913735: 44, 913986: 18, 914003: 31, 914122: 44, 914125: 13, 914162: 27, 914196: 6, 914223: 58, 914275: 18, 914282: 34, 914355: 35, 914371: 41, 914471: 18, 914516: 35, 914586: 1, 914630: 55, 914671: 19, 914906: 59, 915067: 32, 915069: 64, 915170: 25, 915171: 99, 915191: 48, 915199: 80, 915204: 19, 915325: 9, 915335: 49, 915381: 52, 915477: 65, 915604: 93, 915608: 23, 915750: 58, 915795: 17, 915909: 0, 916155: 14, 916202: 6, 916384: 5, 916410: 78, 916415: 58, 916449: 87, 916474: 23, 916587: 68, 916604: 2, 916743: 75, 916895: 19, 916915: 90, 917039: 51, 917062: 9, 917079: 18, 917104: 74, 917150: 45, 917228: 83, 917270: 32, 917276: 79, 917318: 17, 917353: 74, 917363: 68, 917383: 60, 917412: 39, 917495: 93, 917506: 28, 917650: 17, 917711: 54, 917721: 71, 917777: 57, 917797: 29, 917964: 8, 917981: 19, 918019: 33, 918139: 82, 918175: 7, 918194: 59, 918399: 9, 918450: 84, 918497: 39, 918530: 38, 918577: 52, 918980: 0, 919066: 93, 919096: 15, 919108: 97, 919198: 68, 919245: 22, 919248: 27, 919442: 59, 919506: 5, 919559: 66, 919594: 80, 919595: 52, 919609: 36, 919668: 53, 919674: 48, 919711: 27, 919996: 24, 920107: 28, 920164: 19, 920189: 67, 920388: 89, 920467: 36, 920575: 35, 920637: 25, 920653: 75, 920725: 57, 920876: 78, 920898: 25, 921020: 69, 921022: 3, 921028: 80, 921062: 49, 921103: 74, 921264: 8, 921267: 74, 921277: 74, 921288: 31, 921550: 6, 921565: 62, 921590: 83, 921602: 50, 921655: 42, 921673: 78, 921713: 98, 921726: 21, 921799: 28, 921828: 28, 921854: 57, 921860: 65, 921890: 81, 921916: 52, 922002: 81, 922004: 15, 922030: 21, 922076: 20, 922106: 8, 922142: 27, 922146: 28, 922157: 63, 922191: 15, 922272: 70, 922288: 69, 922310: 44, 922367: 24, 922463: 30, 922515: 39, 922564: 22, 922586: 73, 922661: 71, 922696: 15, 922788: 32, 922800: 48, 922831: 38, 922940: 78, 922990: 31, 923001: 97, 923002: 55, 923045: 78, 923060: 74, 923117: 12, 923127: 68, 923133: 85, 923239: 81, 923455: 40, 923544: 94, 923576: 64, 923639: 83, 923661: 4, 923685: 12, 923770: 25, 924012: 23, 924040: 59, 924122: 43, 924165: 94, 924218: 3, 924248: 46, 924290: 63, 924297: 88, 924338: 86, 924462: 25, 924479: 7, 924574: 71, 924653: 1, 924676: 13, 924702: 44, 924714: 34, 924756: 63, 924766: 31, 924799: 46, 924845: 99, 924937: 93, 924988: 62, 925047: 36, 925071: 83, 925155: 75, 925452: 91, 925464: 28, 925494: 30, 925509: 84, 925563: 28, 925657: 4, 925719: 9, 925722: 6, 925762: 71, 925915: 22, 925977: 81, 926124: 20, 926292: 46, 926343: 76, 926349: 75, 926355: 9, 926503: 57, 926614: 16, 926626: 70, 926693: 84, 927182: 31, 927412: 83, 927415: 38, 927537: 52, 927563: 81, 927568: 89, 927615: 10, 927647: 96, 927664: 14, 927709: 9, 927830: 17, 927924: 31, 928021: 32, 928191: 86, 928358: 50, 928411: 35, 928426: 47, 928495: 56, 928497: 92, 928504: 19, 928623: 24, 928679: 25, 928705: 92, 928706: 7, 928713: 73, 928725: 14, 928755: 56, 928776: 68, 929160: 75, 929311: 11, 929339: 27, 929482: 45, 929686: 71, 929695: 8, 929727: 70, 929848: 18, 929972: 64, 929981: 52, 930004: 53, 930036: 29, 930044: 64, 930233: 46, 930393: 72, 930481: 1, 930592: 77, 930628: 62, 930687: 84, 930689: 10, 930741: 65, 930833: 57, 930968: 80, 931197: 87, 931279: 72, 931341: 80, 931402: 32, 931411: 32, 931459: 33, 931565: 1, 931574: 33, 931575: 92, 931729: 50, 931782: 11, 931822: 29, 931838: 12, 931927: 19, 931963: 18, 931983: 29, 932040: 19, 932090: 95, 932110: 49, 932114: 10, 932148: 48, 932216: 55, 932272: 1, 932291: 62, 932380: 76, 932400: 53, 932405: 8, 932433: 16, 932437: 12, 932452: 56, 932457: 51, 932462: 37, 932634: 54, 932675: 81, 932712: 51, 932722: 55, 932723: 30, 932735: 39, 932744: 52, 932798: 95, 933015: 76, 933022: 78, 933043: 54, 933161: 15, 933193: 66, 933267: 15, 933340: 16, 933384: 83, 933412: 1, 933736: 89, 933748: 63, 933776: 72, 933800: 15, 933923: 18, 933983: 77, 934080: 57, 934284: 57, 934307: 50, 934368: 62, 934414: 84, 934436: 59, 934528: 4, 934574: 53, 934600: 50, 934707: 62, 934754: 60, 934793: 46, 934883: 5, 934945: 63, 934946: 93, 934950: 97, 935001: 96, 935055: 61, 935187: 42, 935196: 94, 935264: 42, 935283: 87, 935290: 77, 935449: 94, 935592: 13, 935775: 65, 935825: 38, 935858: 54, 935870: 36, 936010: 74, 936060: 2, 936070: 0, 936141: 39, 936266: 91, 936301: 3, 936321: 62, 936335: 37, 936354: 15, 936364: 86, 936390: 83, 936402: 70, 936520: 15, 936525: 0, 936601: 44, 936611: 46, 936672: 39, 936697: 92, 936753: 79, 936762: 56, 936883: 2, 936954: 1, 937158: 33, 937207: 35, 937232: 99, 937413: 44, 937473: 9, 937757: 91, 937825: 80, 937921: 59, 938085: 20, 938310: 32, 938419: 46, 938462: 56, 938467: 93, 938472: 53, 938531: 67, 938571: 59, 938703: 16, 938808: 72, 938911: 91, 938983: 34, 939008: 28, 939121: 97, 939136: 54, 939189: 56, 939308: 18, 939317: 67, 939328: 41, 939333: 97, 939367: 82, 939426: 29, 939594: 84, 939671: 7, 939673: 72, 939707: 48, 939786: 19, 939834: 68, 939849: 98, 939981: 18, 940166: 92, 940232: 87, 940263: 76, 940271: 31, 940316: 65, 940419: 39, 940540: 77, 940607: 80, 940690: 68, 940712: 4, 940777: 49, 940831: 71, 940873: 98, 940951: 49, 940986: 17, 940995: 97, 941000: 33, 941107: 82, 941125: 40, 941212: 82, 941344: 77, 941423: 33, 941459: 46, 941758: 2, 941832: 51, 941911: 46, 942002: 35, 942111: 92, 942181: 86, 942221: 31, 942270: 36, 942308: 48, 942391: 69, 942556: 27, 942629: 44, 942634: 36, 942651: 56, 942663: 55, 942720: 62, 942751: 69, 942923: 28, 942928: 15, 942942: 90, 942998: 46, 943013: 25, 943374: 60, 943397: 13, 943465: 73, 943552: 69, 943603: 37, 943672: 17, 943675: 93, 943928: 2, 943968: 94, 943981: 66, 944015: 30, 944280: 22, 944394: 74, 944449: 33, 944655: 15, 944872: 54, 944891: 98, 945024: 35, 945144: 70, 945161: 16, 945293: 73, 945307: 67, 945335: 32, 945401: 21, 945416: 66, 945558: 91, 945594: 74, 945688: 21, 945706: 93, 945786: 63, 945797: 28, 945984: 51, 945993: 96, 946001: 69, 946313: 71, 946345: 57, 946457: 72, 946555: 41, 946605: 30, 946700: 42, 946746: 45, 946908: 30, 946923: 65, 946937: 91, 946940: 87, 946965: 3, 946973: 77, 947001: 56, 947023: 74, 947218: 0, 947293: 11, 947360: 8, 947361: 81, 947403: 71, 947440: 72, 947698: 95, 947793: 27, 947957: 91, 947958: 23, 947963: 32, 948004: 59, 948056: 68, 948125: 82, 948201: 92, 948269: 50, 948277: 82, 948319: 37, 948326: 82, 948336: 71, 948383: 97, 948384: 63, 948431: 73, 948490: 26, 948572: 78, 948601: 19, 948610: 41, 948698: 64, 948723: 3, 949002: 70, 949042: 25, 949049: 2, 949268: 97, 949369: 64, 949373: 73, 949488: 53, 949535: 95, 949561: 33, 949580: 81, 949584: 40, 949616: 29, 949643: 84, 949689: 4, 949704: 24, 949811: 23, 949881: 66, 950066: 28, 950393: 15, 950691: 15, 950695: 80, 950817: 67, 950836: 40, 950910: 86, 950968: 70, 950982: 70, 951145: 67, 951164: 8, 951249: 69, 951278: 99, 951335: 70, 951474: 90, 951741: 7, 951778: 18, 951834: 35, 951867: 60, 951883: 92, 951948: 58, 951955: 7, 951960: 21, 952132: 29, 952202: 38, 952282: 28, 952541: 10, 952577: 3, 952597: 46, 952809: 34, 952999: 87, 953027: 0, 953057: 33, 953073: 86, 953102: 33, 953246: 88, 953454: 12, 953533: 30, 953577: 48, 953790: 64, 953794: 0, 953828: 67, 953852: 73, 953986: 38, 954014: 10, 954113: 47, 954225: 22, 954249: 9, 954278: 78, 954353: 85, 954415: 10, 954537: 59, 954581: 66, 954612: 73, 954684: 95, 954717: 19, 954781: 71, 954961: 76, 955029: 46, 955032: 95, 955188: 75, 955475: 43, 955500: 45, 955674: 20, 955808: 29, 955858: 39, 955877: 96, 955881: 3, 955921: 44, 955944: 9, 956007: 23, 956079: 63, 956264: 26, 956347: 66, 956372: 37, 956412: 42, 956430: 11, 956456: 31, 956473: 87, 956507: 17, 956636: 89, 956688: 5, 956736: 7, 956819: 40, 956820: 12, 956840: 80, 956852: 53, 957029: 60, 957042: 28, 957170: 94, 957192: 96, 957224: 92, 957308: 20, 957404: 84, 957469: 72, 957523: 53, 957598: 72, 957698: 82, 957733: 96, 957802: 73, 957808: 20, 957811: 15, 957932: 84, 957950: 32, 957981: 34, 958068: 84, 958091: 65, 958105: 87, 958199: 53, 958218: 75, 958326: 68, 958338: 93, 958360: 72, 958366: 83, 958498: 9, 958539: 66, 958658: 41, 958693: 27, 958723: 19, 958748: 84, 958883: 69, 958949: 59, 958966: 7, 959004: 76, 959060: 40, 959081: 56, 959118: 89, 959340: 30, 959415: 36, 959478: 48, 959564: 79, 959580: 98, 959630: 34, 959738: 64, 959884: 93, 959906: 88, 959936: 28, 960025: 36, 960036: 17, 960114: 44, 960136: 28, 960163: 36, 960172: 62, 960219: 31, 960256: 20, 960276: 67, 960331: 24, 960459: 38, 960502: 8, 960506: 30, 960542: 8, 960574: 94, 960599: 65, 960639: 41, 960706: 26, 960761: 89, 960766: 57, 960924: 83, 961008: 97, 961082: 58, 961126: 14, 961129: 67, 961182: 83, 961233: 21, 961302: 65, 961361: 11, 961371: 78, 961372: 90, 961400: 71, 961426: 62, 961440: 6, 961479: 60, 961521: 61, 961573: 57, 961620: 63, 961655: 52, 961658: 97, 961679: 5, 961687: 15, 961705: 80, 961711: 69, 961727: 97, 961752: 98, 961786: 80, 961855: 18, 961912: 15, 962008: 57, 962153: 77, 962188: 37, 962231: 89, 962250: 60, 962481: 82, 962564: 98, 962650: 60, 962679: 54, 962881: 32, 962991: 25, 963001: 64, 963013: 30, 963104: 32, 963180: 9, 963201: 19, 963285: 48, 963510: 4, 963551: 10, 963561: 10, 963611: 22, 963659: 11, 963666: 1, 963773: 20, 963793: 47, 963839: 2, 963916: 73, 964030: 98, 964037: 64, 964179: 28, 964189: 51, 964219: 75, 964229: 15, 964277: 19, 964418: 95, 964450: 52, 964511: 66, 964513: 12, 964601: 91, 964612: 94, 964676: 91, 964714: 33, 964758: 37, 964929: 36, 965019: 17, 965027: 14, 965067: 65, 965090: 16, 965242: 53, 965264: 90, 965331: 77, 965351: 91, 965353: 74, 965592: 64, 965734: 5, 965753: 56, 965755: 8, 965822: 37, 965831: 67, 965850: 92, 965853: 73, 965886: 56, 965995: 59, 966015: 94, 966016: 44, 966030: 88, 966073: 63, 966196: 3, 966213: 14, 966278: 81, 966409: 49, 966446: 70, 966469: 14, 966547: 17, 966572: 8, 966605: 88, 966623: 44, 966723: 56, 966772: 7, 966777: 40, 966795: 39, 966811: 9, 966868: 80, 966876: 45, 966923: 13, 967133: 29, 967136: 82, 967150: 47, 967357: 65, 967455: 61, 967458: 36, 967530: 68, 967569: 75, 967620: 24, 967672: 0, 967680: 61, 967688: 45, 967721: 80, 967725: 77, 967766: 49, 967952: 26, 968048: 49, 968049: 29, 968072: 22, 968175: 42, 968259: 80, 968284: 19, 968382: 55, 968411: 29, 968466: 3, 968484: 26, 968542: 67, 968545: 68, 968564: 56, 968607: 0, 968651: 63, 968671: 63, 968864: 68, 968878: 60, 968916: 27, 968989: 66, 969120: 90, 969135: 17, 969165: 71, 969225: 44, 969289: 38, 969414: 84, 969501: 46, 969698: 43, 969767: 21, 969776: 87, 969878: 4, 969960: 39, 969991: 59, 970077: 64, 970117: 77, 970155: 86, 970189: 2, 970248: 19, 970250: 48, 970251: 46, 970289: 31, 970330: 62, 970405: 0, 970448: 60, 970465: 45, 970467: 31, 970511: 72, 970544: 92, 970567: 7, 970570: 13, 970615: 39, 970620: 69, 970696: 94, 970712: 71, 970803: 26, 970804: 38, 970821: 19, 970823: 12, 970856: 77, 970936: 34, 970960: 82, 970987: 87, 971039: 92, 971050: 46, 971168: 26, 971187: 71, 971223: 63, 971258: 9, 971468: 14, 971550: 31, 971607: 1, 971626: 10, 971635: 44, 971684: 59, 971724: 22, 971791: 54, 971858: 58, 971906: 55, 971993: 92, 972110: 58, 972114: 8, 972189: 77, 972199: 30, 972506: 50, 972594: 29, 972668: 81, 972685: 80, 972960: 47, 973090: 2, 973104: 67, 973445: 85, 973488: 18, 973528: 69, 973613: 7, 973669: 21, 973680: 63, 973775: 36, 973809: 76, 973858: 62, 973976: 1, 974023: 78, 974027: 10, 974056: 59, 974070: 4, 974091: 64, 974093: 29, 974098: 63, 974201: 56, 974279: 7, 974358: 96, 974621: 57, 974633: 74, 974642: 17, 974645: 26, 974723: 2, 974826: 74, 974834: 34, 974955: 49, 974957: 29, 975061: 94, 975099: 9, 975105: 15, 975286: 59, 975325: 48, 975345: 14, 975518: 82, 975566: 58, 975592: 14, 975603: 7, 975604: 99, 975615: 95, 975653: 14, 975728: 60, 975758: 13, 975936: 96, 975979: 37, 976142: 85, 976220: 50, 976282: 70, 976309: 23, 976428: 15, 976440: 1, 976469: 18, 976587: 22, 976663: 85, 976718: 55, 976781: 12, 976882: 29, 976987: 22, 977038: 77, 977047: 89, 977219: 17, 977510: 48, 977528: 2, 977558: 75, 977616: 66, 977671: 85, 977799: 24, 977817: 66, 977842: 17, 978113: 59, 978133: 0, 978247: 96, 978294: 80, 978295: 39, 978301: 65, 978383: 37, 978426: 99, 978526: 51, 978559: 5, 978567: 39, 978642: 26, 978660: 88, 978734: 14, 978769: 49, 978858: 14, 978861: 14, 978883: 55, 978885: 67, 979035: 44, 979131: 4, 979331: 52, 979509: 88, 979530: 38, 979602: 61, 979666: 27, 979931: 13, 980054: 44, 980057: 6, 980336: 28, 980455: 36, 980461: 37, 980567: 27, 980579: 61, 980588: 88, 980687: 66, 980710: 69, 980738: 0, 980745: 53, 980775: 0, 980822: 68, 981069: 63, 981089: 95, 981123: 32, 981159: 75, 981201: 83, 981213: 87, 981272: 44, 981380: 70, 981386: 6, 981490: 5, 981544: 67, 981660: 81, 981695: 22, 981765: 97, 982009: 18, 982015: 8, 982196: 59, 982267: 28, 982356: 51, 982374: 91, 982385: 9, 982401: 38, 982406: 9, 982415: 0, 982518: 42, 982537: 67, 982540: 38, 982690: 95, 982732: 89, 982895: 54, 982964: 80, 982995: 51, 983101: 43, 983105: 11, 983159: 74, 983214: 11, 983250: 2, 983330: 8, 983348: 14, 983391: 53, 983466: 57, 983467: 76, 983484: 70, 983526: 54, 983531: 51, 983586: 72, 983626: 15, 983648: 6, 983849: 37, 983872: 29, 983977: 86, 984204: 72, 984208: 57, 984225: 98, 984230: 24, 984342: 9, 984439: 97, 984568: 53, 984656: 9, 984756: 54, 984764: 85, 984859: 27, 984897: 71, 984930: 29, 985009: 85, 985011: 69, 985111: 62, 985179: 32, 985182: 88, 985275: 20, 985331: 99, 985454: 5, 985460: 40, 985771: 31, 985883: 53, 985945: 84, 985960: 55, 986034: 73, 986089: 78, 986134: 43, 986156: 34, 986209: 50, 986220: 70, 986237: 17, 986240: 53, 986247: 67, 986252: 29, 986313: 99, 986384: 8, 986437: 26, 986466: 28, 986651: 65, 986659: 90, 986667: 5, 986696: 66, 986752: 27, 986763: 39, 986793: 75, 986834: 24, 986854: 92, 986966: 83, 987013: 16, 987035: 36, 987101: 55, 987209: 22, 987231: 99, 987249: 88, 987256: 51, 987269: 94, 987329: 79, 987341: 23, 987386: 98, 987445: 49, 987473: 74, 987713: 77, 987734: 82, 987741: 8, 987757: 22, 987971: 15, 988004: 66, 988066: 93, 988119: 35, 988121: 15, 988122: 36, 988178: 39, 988214: 40, 988234: 57, 988354: 81, 988360: 81, 988400: 99, 988411: 63, 988495: 90, 988620: 92, 988656: 62, 988670: 3, 988766: 34, 988881: 71, 988944: 91, 989027: 50, 989052: 48, 989070: 58, 989173: 94, 989328: 36, 989450: 96, 989468: 27, 989569: 39, 989633: 39, 989664: 9, 989758: 53, 989771: 32, 989826: 37, 989867: 75, 989966: 81, 989996: 33, 990015: 72, 990023: 82, 990099: 69, 990139: 1, 990237: 40, 990370: 67, 990538: 29, 990607: 24, 990613: 81, 990769: 30, 990813: 9, 990838: 41, 990860: 4, 990861: 37, 990901: 75, 990965: 59, 990979: 11, 991082: 43, 991119: 68, 991264: 50, 991282: 2, 991466: 2, 991475: 40, 991525: 84, 991667: 33, 991680: 85, 991703: 83, 991785: 18, 991787: 16, 991803: 15, 991951: 96, 992033: 37, 992174: 99, 992260: 41, 992341: 64, 992366: 10, 992541: 74, 992547: 0, 992558: 53, 992600: 14, 992612: 48, 992619: 86, 992728: 96, 992791: 84, 992862: 6, 992884: 42, 992898: 26, 992907: 95, 992909: 33, 992993: 24, 993000: 78, 993041: 80, 993084: 52, 993120: 80, 993134: 15, 993139: 69, 993168: 82, 993227: 53, 993257: 96, 993259: 25, 993310: 46, 993414: 10, 993443: 62, 993516: 82, 993594: 58, 993617: 52, 993709: 9, 993770: 20, 993794: 61, 993858: 86, 993926: 78, 993951: 51, 994003: 63, 994088: 39, 994112: 38, 994317: 50, 994380: 57, 994434: 24, 994541: 37, 994549: 27, 994664: 38, 994726: 20, 994742: 20, 994768: 21, 994832: 79, 994925: 5, 994926: 66, 994943: 3, 995006: 27, 995029: 76, 995043: 52, 995089: 56, 995166: 31, 995171: 74, 995180: 46, 995195: 36, 995279: 89, 995488: 82, 995517: 32, 995619: 8, 995716: 70, 995763: 27, 995800: 2, 995823: 25, 995960: 83, 996012: 85, 996014: 97, 996092: 9, 996100: 51, 996174: 72, 996206: 32, 996346: 59, 996406: 61, 996555: 53, 996635: 49, 996678: 95, 997045: 88, 997145: 66, 997225: 86, 997397: 66, 997438: 0, 997452: 67, 997500: 4, 997513: 72, 997589: 61, 997606: 96, 997627: 37, 997631: 45, 997730: 44, 997735: 71, 997751: 24, 997768: 95, 998028: 91, 998084: 86, 998131: 21, 998157: 14, 998167: 25, 998323: 96, 998344: 3, 998346: 38, 998370: 97, 998461: 49, 998521: 27, 998522: 26, 998539: 25, 998625: 52, 998629: 36, 998675: 21, 998800: 91, 998944: 81, 999093: 86, 999132: 53, 999169: 39, 999185: 83, 999193: 41, 999206: 28, 999298: 11, 999316: 9, 999341: 54, 999355: 30, 999575: 42, 999673: 75, 999821: 65, 999835: 1, 999918: 99, 999936: 37} {
		control[k].Set(v)
	}

	var compacted bool
	for compacted, err = b.disk.compactOnce(); err == nil && compacted; compacted, err = b.disk.compactOnce() {
	}
	require.Nil(t, err)

	verifyBucketRangeAgainstControl(t, b, control)
}
