/* Edge Impulse inferencing library
 * Copyright (c) 2023 EdgeImpulse Inc.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#ifndef __EI_DSP_HANDLE__H__
#define __EI_DSP_HANDLE__H__

#include "edge-impulse-sdk/dsp/config.hpp"
#include "edge-impulse-sdk/dsp/numpy_types.h"

class DspHandle {
public:
    /**
     * @brief Override and call ei_printf to print debug information, especially the current state
     *
     * @return int
     */
    virtual int print() = 0;

    /**
     * @brief Override and convert raw data into processed features. Any state should live inside your custom class.
     * Provide a constructor to initialize your state.
     *
     * @param signal Callback object to get raw data from
     * @param output_matrix Output matrix to write features to
     * @param config Configuration object, generated by Studio based on your DSP block parameters
     * @param frequency Sampling frequency, as set in your project
     * @return int 0 on success, anything else for failure
     */
    virtual int extract(ei::signal_t *signal, ei::matrix_t *output_matrix, void *config, const float frequency) = 0;

    // Must declare so user can override
    /**
     * @brief If you call new or ei_malloc anywhere in your class, you must override this function and delete your objects
     *
     */
    virtual ~DspHandle() {};
};

#endif  //!__EI_DSP_HANDLE__H__