#!/usr/bin/env python3
# ===================================================================================
# Project:   rvprog - Programming Tool for WCH RISC-V Microcontrollers with WCH-LinkE
# Version:   v1.8
# Year:      2023
# Author:    Stefan Wagner
# Github:    https://github.com/wagiminator
# License:   MIT License
# ===================================================================================
#
# Description:
# ------------
# Simple Python tool for flashing WCH RISC-V microcontrollers using the WCH-LinkE,
# WCH-LinkW, or compatible programmers/debuggers.
# Currently supports: CH32V003, CH32V103, CH32V203, CH32V208, CH32V303, CH32V305, 
#                     CH32V307, CH32X033, CH32X035, CH32L103,
#                     CH571, CH573, CH581, CH582, CH583, CH591, CH592.
#
# References:
# -----------
# - CNLohr minichlink: https://github.com/cnlohr/ch32v003fun/tree/master/minichlink
# - My mighty USB sniffer
#
# Dependencies:
# -------------
# - PyUSB
#
# Operating Instructions:
# -----------------------
# You need to install PyUSB to use rvprog. Install it via "pip install pyusb".
#
# Linux users need permission to access the device. Run:
# echo 'SUBSYSTEM=="usb", ATTR{idVendor}=="1a86", ATTR{idProduct}=="8010", MODE="666"' | sudo tee /etc/udev/rules.d/99-WCH-LinkE.rules
# echo 'SUBSYSTEM=="usb", ATTR{idVendor}=="1a86", ATTR{idProduct}=="8012", MODE="666"' | sudo tee -a /etc/udev/rules.d/99-WCH-LinkE.rules
# sudo udevadm control --reload-rules
#
# Connect the WCH-Link to your PC and to your WCH RISC-V MCU board. The WCH-Link
# must be in LinkRV mode (blue LED off)! If not, run: python rvprog.py -v
# Run:
# - rvprog.py [-h] [-a] [-v] [-b] [-u] [-l] [-e] [-G] [-R] [-f FLASH]
#   -h, --help                show help message and exit
#   -a, --armmode             switch WCH-Link to ARM mode
#   -v, --rvmode              switch WCH-Link to RISC-V mode
#   -b, --unbrick             unbrick chip (power cycle erase)
#   -u, --unlock              unlock chip (remove read protection)
#   -l, --lock                lock chip (set read protection)
#   -e, --erase               perform a whole chip erase
#   -G, --pingpio             make nRST pin a GPIO pin (CH32V003 only)
#   -R, --pinreset            make nRST pin a reset pin (CH32V003 only)
#   -f FLASH, --flash FLASH   write BIN file to flash
#
# - Example:
#   python3 rvprog.py -f firmware.bin


import usb.core
import usb.util
import sys
import time
import argparse

# ===================================================================================
# Main Function
# ===================================================================================

def _main():
    # Parse command line arguments
    parser = argparse.ArgumentParser(description='Minimal command line interface for WCH-Link')
    parser.add_argument('-a', '--armmode',  action='store_true', help='switch WCH-Link to ARM mode')
    parser.add_argument('-v', '--rvmode',   action='store_true', help='switch WCH-Link to RISC-V mode')
    parser.add_argument('-b', '--unbrick',  action='store_true', help='unbrick chip (power cycle erase)')
    parser.add_argument('-u', '--unlock',   action='store_true', help='unlock chip (remove read protection)')
    parser.add_argument('-l', '--lock',     action='store_true', help='lock chip (set read protection)')
    parser.add_argument('-e', '--erase',    action='store_true', help='perform a whole chip erase')
    parser.add_argument('-G', '--pingpio',  action='store_true', help='make nRST pin a GPIO pin (CH32V003 only)')
    parser.add_argument('-R', '--pinreset', action='store_true', help='make nRST pin a reset pin (CH32V003 only)')
    parser.add_argument('-f', '--flash',    help='write BIN file to flash and verify')
    args = parser.parse_args(sys.argv[1:])

    # Check arguments
    if not any( (args.armmode, args.rvmode, args.unbrick, args.unlock, args.lock, args.erase, args.pingpio, args.pinreset, args.flash) ):
        print('No arguments - no action!')
        sys.exit(0)

    # Switch WCH-Link to RISC-V mode
    try:
        if args.rvmode:
            print('Searching for WCH-Link in ARM mode ...')
            armlink = usb.core.find(idVendor = CH_VENDOR_ID, idProduct = CH_ARM_ID)
            if armlink is None:
                raise Exception('No WCH-Link in ARM mode found!')
            print('SUCCESS: Found WCH-Link in ARM mode.')
            print('Switching WCH-Link to RISC-V mode ...')
            armlink.write(0x02, b'\x81\xff\x01\x52')
            time.sleep(2)
            print('DONE.')
    except Exception as ex:
        sys.stderr.write('ERROR: %s!\n' % str(ex))

    # Establish connection to WCH-Link
    try:
        print('Searching for WCH-Link in RISC-V mode ...')
        isp = Programmer()
        print('SUCCESS: Found %s v%s in RISC-V mode.' % (isp.linkname, isp.linkversion))
        if isp.linkvercode < 211:
            print('WARNING: WCH-Link firmware needs to be updated!')
    except Exception as ex:
        sys.stderr.write('ERROR: %s!\n' % str(ex))
        sys.exit(1)

    # Performing actions
    try:
        # Unbrick chip
        if args.unbrick:
            print('Unbricking chip ...')
            isp.unbrick()
            print('SUCCESS: Chip is unbricked.')

        # Establish connection to target MCU
        if any( (args.unlock, args.lock, args.erase, args.flash, args.pingpio, args.pinreset) ):
            print('Connecting to MCU ...')
            isp.connect()
            print('SUCCESS: Connected to %s with %d bytes of flash.' % (isp.chipname, isp.flashsize))

        # Unlock chip
        if args.unlock:
            print('Unlocking chip ...')
            isp.unlock()
            print('SUCCESS: Chip is unlocked.')

        # Perform chip erase
        if args.erase:
            print('Performing whole chip erase ...')
            isp.erase()
            print('SUCCESS: Chip is erased.')

        # Flash binary file
        if args.flash is not None:
            print('Flashing %s to %s ...' % (args.flash, isp.chipname))
            with open(args.flash, 'rb') as f: data = f.read()
            isp.flash_data(data)
            print('SUCCESS: %d bytes written and verified.' % len(data))

        # Make nRST pin a normal GPIO pin
        if args.pingpio:
            print('Configuring nRST pin as GPIO ...')
            isp.setnrstasgpio(1)
            print('SUCCESS: nRST pin is now a GPIO pin.')

        # Make nRST pin a reset pin
        if args.pinreset:
            print('Configuring nRST pin as reset ...')
            isp.setnrstasgpio(0)
            print('SUCCESS: nRST pin is now a reset pin.')

        # Lock chip
        if args.lock:
            print('Locking chip ...')
            isp.lock()
            print('SUCCESS: Chip is locked.')

        # Switch device to ARM mode
        if args.armmode:
            print('Switching WCH-Link to ARM mode ...')
            isp.exit()
            isp.dev.write(CH_EP_OUT, b'\x81\xff\x01\x41')
            print('DONE: Check if blue LED lights up!')
            sys.exit(0)

        isp.exit()

    except Exception as ex:
        sys.stderr.write('ERROR: %s!\n' % str(ex))
        isp.exit()
        sys.exit(1)

    print('DONE.')
    sys.exit(0)

# ===================================================================================
# Programmer Class
# ===================================================================================

class Programmer:
    # Init programmer
    def __init__(self):
        # Find programmer
        self.dev = usb.core.find(idVendor = CH_VENDOR_ID, idProduct = CH_PRODUCT_ID)
        if self.dev is None:
            raise Exception('WCH-Link not found. Check if device is in RISC-V mode')

        # Clear receive buffers
        self.clearreply()

        # Get programmer info
        reply = self.sendcommand(b'\x81\x0d\x01\x01')
        if reply[5]   == 0x05:
            self.linkname = 'WCH-LinkW'
        elif reply[5] == 0x12:
            self.linkname = 'WCH-LinkE'
        else:
            raise Exception('Unsupported programmer (code: %d)' % reply[5])
        self.linkdevice  = reply[5]
        self.linkvercode = (reply[3] * 100) + reply[4]
        self.linkversion = '%d.%d' % (reply[3], reply[4])

    # Connect programmer to MCU
    def connect(self):
        # Connect to target MCU and get type
        self.chipid = None
        success = 0
        for i in range(3):
            if i > 0:
                print('Failed to connect, trying to unbrick MCU ...')
                self.sendcommand(b'\x81\x0d\x01\x0f\x09')
            reply = self.sendcommand(b'\x81\x0d\x01\x02')
            if len(reply) < 8 or set(reply[:4]) == set((0x81, 0x55, 0x01, 0x01)):
                time.sleep(0.2)
                continue

            # Get chip identification data
            self.chipmark   = reply[3]
            self.chipseries = reply[4]
            self.chiptype   = reply[5]
            self.chipid     = (reply[4] << 8) + reply[5]
            self.device     = None

            # Find device in dictionary
            for d in DEVICES:
                if d['id'] == self.chipid:
                    self.device = d
            if self.device is None:
                continue

            # Success
            self.chipname = self.device['name']
            for f in FAMILIES:
                if f['id'] == self.chipseries:
                    self.family = f
            success = 1
            break

        # Unsuccessful
        if success == 0:
            if self.chipid is None:
                raise Exception('Failed to identify chip')
            else:
                raise Exception('Unsupported chip (ID: 0x%04x)' % self.chipid)

        # Read some chip data
        if self.family['optbytes'] is not None:
            reply = self.sendcommand((0x81, 0x11, 0x01, self.chipmark))
            self.flashsize = int.from_bytes(reply[2:4], byteorder='big') * 1024
        else:
            self.flashsize = self.device['code_size']

    # Send command to programmer
    def sendcommand(self, stream):
        self.dev.write(CH_EP_OUT, stream)
        return self.dev.read(CH_EP_IN, CH_PACKET_SIZE, CH_TIMEOUT)

    # Clear USB receive buffers
    def clearreply(self):
        try:    self.dev.read(CH_EP_IN, CH_PACKET_SIZE, 1)
        except: None
        try:    self.dev.read(CH_EP_IN_RAW, CH_PACKET_SIZE, 1)
        except: None

    # Write to MCU register
    def writereg(self, addr, data):
        stream = bytes((0x81, 0x08, 0x06, addr)) + data.to_bytes(4, byteorder='big') + b'\x02'
        reply = self.sendcommand(stream)
        if (len(reply) != 9) or (reply[3] != addr):
            raise Exception('Failed to write register')

    # Read from MCU register
    def readreg(self, addr):
        stream = (0x81, 0x08, 0x06, addr, 0, 0, 0, 0, 1)
        reply = self.sendcommand(stream)
        if (len(reply) != 9) or (reply[3] != addr):
            raise Exception('Failed to read register')
        return int.from_bytes(reply[4:8], byteorder='big')

    # Unbrick MCU
    def unbrick(self):
        self.sendcommand(b'\x81\x0d\x01\x0f\x09')

    # Lock MCU (set read protection)
    def lock(self):
        if self.family['optbytes'] is not None:
            self.sendcommand(b'\x81\x06\x08\x03' + self.family['optbytes'])

    # Unlock MCU (remove read protection -> erase chip!!!)
    def unlock(self):
        if self.family['optbytes'] is not None:
            reply = self.sendcommand(b'\x81\x06\x01\x01')
            if reply[3] == 0x01:
                self.sendcommand(b'\x81\x06\x08\x02' + self.family['optbytes'])

    # Perform a whole chip erase
    def erase(self):
        self.sendcommand(b'\x81\x02\x01\x01')
        self.sendcommand(b'\x81\x0d\x01\x02')

    # Make reset pin a normal GPIO pin (0=RESET, 1=GPIO)
    def setnrstasgpio(self, state):
        if self.chipseries == 0x00:
            if state:
                self.sendcommand(b'\x81\x06\x08\x02\xff\xff\xff\xff\xff\xff\xff')
            else:
                self.sendcommand(b'\x81\x06\x08\x02\xf7\xff\xff\xff\xff\xff\xff')
        else:
            raise Exception('RST pin option not available for this MCU')

    # Enable/disable programmer's 3V3 output
    def poweron3v3(self):
        self.sendcommand(b'\x81\x0d\x01\x09')
    def poweroff3v3(self):
        self.sendcommand(b'\x81\x0d\x01\x0a')

    # Enable/disable programmer's 5V output
    def poweron5v(self):
        self.sendcommand(b'\x81\x0d\x01\x0b')
    def poweroff5v(self):
        self.sendcommand(b'\x81\x0d\x01\x0c')

    # Disconnect from MCU
    def exit(self):
        self.sendcommand(b'\x81\x0b\x01\x01')
        self.sendcommand(b'\x81\x0d\x01\xff')

    #--------------------------------------------------------------

    # Get padded data length
    def padlen(self, data, pagesize):
        if (len(data) % pagesize) == 0:
            return len(data)
        else:
            return (len(data) + (pagesize - (len(data) % pagesize)))

    # Pad data and divide into pages
    def page_data(self, data, pagesize):
        if (len(data) % pagesize) > 0:
            data += b'\xff' * (pagesize - (len(data) % pagesize))
        result = list()
        while len(data):
            result.append(data[:pagesize])
            data = data[pagesize:]
        return result

    # Write data blob to flash
    def writebinaryblob(self, addr, blocksize, bootloader, data):
        if addr & (blocksize - 1):
            raise Exception('Blob is not %d byte aligned' % blocksize)
        self.unlock()
        stream = b'\x81\x01\x08' \
               + addr.to_bytes(4, byteorder='big') \
               + self.padlen(data, blocksize).to_bytes(4, byteorder='big')
        self.sendcommand(stream)
        self.sendcommand(b'\x81\x02\x01\x05')
        pages = self.page_data(bootloader, 128)
        for page in pages:
            self.dev.write(CH_EP_OUT_RAW, page)
        self.sendcommand(b'\x81\x02\x01\x07')
        self.sendcommand(b'\x81\x02\x01\x04')
        pages = self.page_data(data, blocksize)
        for page in pages:
            self.dev.write(CH_EP_OUT_RAW, page)
        reply = self.dev.read(CH_EP_IN_RAW, CH_PACKET_SIZE, CH_TIMEOUT)
        if set(reply) != set((0x41, 0x01, 0x01, 0x04)):
            raise Exception('Failed writing/verifying data blob')

    # Write data to code flash
    def flash_data(self, data):
        if len(data) > self.flashsize:
            raise Exception('Not enough memory')
        self.writebinaryblob(self.family['code_base'], self.family['block_size'], self.family['loader'], data)

# ===================================================================================
# Device Constants
# ===================================================================================

# USB device settings
CH_VENDOR_ID    = 0x1A86    # VID
CH_PRODUCT_ID   = 0x8010    # PID in RISC-V mode
CH_ARM_ID       = 0x8012    # PID in ARM mode
CH_PACKET_SIZE  = 1024      # packet size
CH_INTERFACE    = 0         # interface number
CH_EP_OUT       = 0x01      # endpoint for command transfer out
CH_EP_IN        = 0x81      # endpoint for reply transfer in
CH_EP_OUT_RAW   = 0x02      # endpoint for raw data transfer out
CH_EP_IN_RAW    = 0x82      # endpoint for raw data transfer in
CH_TIMEOUT      = 5000      # timeout for USB operations

# ===================================================================================
# Flash Loader
# ===================================================================================

LOADER_V003 = \
     b"\x21\x11\x22\xca\x26\xc8\x93\x77\x15\x00\x99\xcf\xb7\x06\x67\x45" \
    +b"\xb7\x27\x02\x40\x93\x86\x36\x12\x37\x97\xef\xcd\xd4\xc3\x13\x07" \
    +b"\xb7\x9a\xd8\xc3\xd4\xd3\xd8\xd3\x93\x77\x25\x00\x9d\xc7\xb7\x27" \
    +b"\x02\x40\x98\x4b\xad\x66\x37\x33\x00\x40\x13\x67\x47\x00\x98\xcb" \
    +b"\x98\x4b\x93\x86\xa6\xaa\x13\x67\x07\x04\x98\xcb\xd8\x47\x05\x8b" \
    +b"\x63\x16\x07\x10\x98\x4b\x6d\x9b\x98\xcb\x93\x77\x45\x00\xa9\xcb" \
    +b"\x93\x07\xf6\x03\x99\x83\x2e\xc0\x2d\x63\x81\x76\x3e\xc4\xb7\x32" \
    +b"\x00\x40\xb7\x27\x02\x40\x13\x03\xa3\xaa\xfd\x16\x98\x4b\xb7\x03" \
    +b"\x02\x00\x33\x67\x77\x00\x98\xcb\x02\x47\xd8\xcb\x98\x4b\x13\x67" \
    +b"\x07\x04\x98\xcb\xd8\x47\x05\x8b\x69\xe7\x98\x4b\x75\x8f\x98\xcb" \
    +b"\x02\x47\x13\x07\x07\x04\x3a\xc0\x22\x47\x7d\x17\x3a\xc4\x79\xf7" \
    +b"\x93\x77\x85\x00\xf1\xcf\x93\x07\xf6\x03\x2e\xc0\x99\x83\x37\x27" \
    +b"\x02\x40\x3e\xc4\x1c\x4b\xc1\x66\x2d\x63\xd5\x8f\x1c\xcb\x37\x07" \
    +b"\x00\x20\x13\x07\x07\x20\xb7\x27\x02\x40\xb7\x03\x08\x00\xb7\x32" \
    +b"\x00\x40\x13\x03\xa3\xaa\x94\x4b\xb3\xe6\x76\x00\x94\xcb\xd4\x47" \
    +b"\x85\x8a\xf5\xfe\x82\x46\xba\x84\x37\x04\x04\x00\x36\xc2\xc1\x46" \
    +b"\x36\xc6\x92\x46\x84\x40\x11\x07\x84\xc2\x94\x4b\xc1\x8e\x94\xcb" \
    +b"\xd4\x47\x85\x8a\xb1\xea\x92\x46\xba\x84\x91\x06\x36\xc2\xb2\x46" \
    +b"\xfd\x16\x36\xc6\xf9\xfe\x82\x46\xd4\xcb\x94\x4b\x93\xe6\x06\x04" \
    +b"\x94\xcb\xd4\x47\x85\x8a\x85\xee\xd4\x47\xc1\x8a\x85\xce\xd8\x47" \
    +b"\xb7\x06\xf3\xff\xfd\x16\x13\x67\x07\x01\xd8\xc7\x98\x4b\x21\x45" \
    +b"\x75\x8f\x98\xcb\x52\x44\xc2\x44\x61\x01\x02\x90\x23\x20\xd3\x00" \
    +b"\xf5\xb5\x23\xa0\x62\x00\x3d\xb7\x23\xa0\x62\x00\x55\xb7\x23\xa0" \
    +b"\x62\x00\xc1\xb7\x82\x46\x93\x86\x06\x04\x36\xc0\xa2\x46\xfd\x16" \
    +b"\x36\xc4\xb5\xf2\x98\x4b\xb7\x06\xf3\xff\xfd\x16\x75\x8f\x98\xcb" \
    +b"\x41\x89\x05\xcd\x2e\xc0\x0d\x06\x02\xc4\x09\x82\xb7\x07\x00\x20" \
    +b"\x32\xc6\x93\x87\x07\x20\x98\x43\x13\x86\x47\x00\xa2\x47\x82\x46" \
    +b"\x8a\x07\xb6\x97\x9c\x43\x63\x1c\xf7\x00\xa2\x47\x85\x07\x3e\xc4" \
    +b"\xa2\x46\x32\x47\xb2\x87\xe3\xe0\xe6\xfe\x01\x45\x61\xb7\x41\x45" \
    +b"\x51\xb7"

LOADER_V103 = \
     b"\x93\x77\x15\x00\x41\x11\x99\xcf\xb7\x06\x67\x45\xb7\x27\x02\x40" \
    +b"\x93\x86\x36\x12\x37\x97\xef\xcd\xd4\xc3\x13\x07\xb7\x9a\xd8\xc3" \
    +b"\xd4\xd3\xd8\xd3\x93\x77\x25\x00\x9d\xc7\xb7\x27\x02\x40\x98\x4b" \
    +b"\xad\x66\x37\x38\x00\x40\x13\x67\x47\x00\x98\xcb\x98\x4b\x93\x86" \
    +b"\xa6\xaa\x13\x67\x07\x04\x98\xcb\xd8\x47\x05\x8b\x63\x1f\x07\x10" \
    +b"\x98\x4b\x6d\x9b\x98\xcb\x93\x77\x45\x00\xa9\xcb\x93\x07\xf6\x07" \
    +b"\x9d\x83\x2e\xc0\x2d\x68\x81\x76\x3e\xc4\xb7\x08\x02\x00\xb7\x27" \
    +b"\x02\x40\x37\x33\x00\x40\x13\x08\xa8\xaa\xfd\x16\x98\x4b\x33\x67" \
    +b"\x17\x01\x98\xcb\x02\x47\xd8\xcb\x98\x4b\x13\x67\x07\x04\x98\xcb" \
    +b"\xd8\x47\x05\x8b\x71\xef\x98\x4b\x75\x8f\x98\xcb\x02\x47\x13\x07" \
    +b"\x07\x08\x3a\xc0\x22\x47\x7d\x17\x3a\xc4\x69\xfb\x93\x77\x85\x00" \
    +b"\xed\xc3\x93\x07\xf6\x07\x2e\xc0\x9d\x83\x37\x27\x02\x40\x3e\xc4" \
    +b"\x1c\x4b\xc1\x66\x37\x08\x08\x00\xd5\x8f\x1c\xcb\xa1\x48\x37\x17" \
    +b"\x00\x20\xb7\x27\x02\x40\x37\x03\x04\x00\x94\x4b\xb3\xe6\x06\x01" \
    +b"\x94\xcb\xd4\x47\x85\x8a\xf5\xfe\x82\x46\x3a\x8e\x36\xc2\x46\xc6" \
    +b"\x92\x46\x83\x2e\x07\x00\x41\x07\x23\xa0\xd6\x01\x92\x46\x83\x2e" \
    +b"\x47\xff\x23\xa2\xd6\x01\x92\x46\x83\x2e\x87\xff\x23\xa4\xd6\x01" \
    +b"\x92\x46\x03\x2e\xce\x00\x23\xa6\xc6\x01\x94\x4b\xb3\xe6\x66\x00" \
    +b"\x94\xcb\xd4\x47\x85\x8a\xf5\xfe\x92\x46\x3a\x8e\xc1\x06\x36\xc2" \
    +b"\xb2\x46\xfd\x16\x36\xc6\xcd\xfe\x82\x46\xd4\xcb\x94\x4b\x93\xe6" \
    +b"\x06\x04\x94\xcb\xd4\x47\x85\x8a\xf5\xfe\xd4\x47\xd1\x8a\x85\xc6" \
    +b"\xd8\x47\xb7\x06\xf3\xff\xfd\x16\x13\x67\x47\x01\xd8\xc7\x98\x4b" \
    +b"\x21\x45\x75\x8f\x98\xcb\x41\x01\x02\x90\x23\x20\xd8\x00\xe9\xbd" \
    +b"\x23\x20\x03\x01\x31\xbf\x82\x46\x93\x86\x06\x08\x36\xc0\xa2\x46" \
    +b"\xfd\x16\x36\xc4\xb9\xfa\x98\x4b\xb7\x06\xf3\xff\xfd\x16\x75\x8f" \
    +b"\x98\xcb\x41\x89\x15\xc9\x2e\xc0\x0d\x06\x02\xc4\x09\x82\x32\xc6" \
    +b"\xb7\x17\x00\x20\x98\x43\x13\x86\x47\x00\xa2\x47\x82\x46\x8a\x07" \
    +b"\xb6\x97\x9c\x43\x63\x1c\xf7\x00\xa2\x47\x85\x07\x3e\xc4\xa2\x46" \
    +b"\x32\x47\xb2\x87\xe3\xe0\xe6\xfe\x01\x45\x71\xbf\x41\x45\x61\xbf"

LOADER_V203 = \
     b"\x93\x77\x15\x00\x41\x11\x99\xcf\xb7\x06\x67\x45\xb7\x27\x02\x40" \
    +b"\x93\x86\x36\x12\x37\x97\xef\xcd\xd4\xc3\x13\x07\xb7\x9a\xd8\xc3" \
    +b"\xd4\xd3\xd8\xd3\x93\x77\x25\x00\x95\xc7\xb7\x27\x02\x40\x98\x4b" \
    +b"\xad\x66\x37\x38\x00\x40\x13\x67\x47\x00\x98\xcb\x98\x4b\x93\x86" \
    +b"\xa6\xaa\x13\x67\x07\x04\x98\xcb\xd8\x47\x05\x8b\x61\xeb\x98\x4b" \
    +b"\x6d\x9b\x98\xcb\x93\x77\x45\x00\xa9\xcb\x93\x07\xf6\x0f\xa1\x83" \
    +b"\x2e\xc0\x2d\x68\x81\x76\x3e\xc4\xb7\x08\x02\x00\xb7\x27\x02\x40" \
    +b"\x37\x33\x00\x40\x13\x08\xa8\xaa\xfd\x16\x98\x4b\x33\x67\x17\x01" \
    +b"\x98\xcb\x02\x47\xd8\xcb\x98\x4b\x13\x67\x07\x04\x98\xcb\xd8\x47" \
    +b"\x05\x8b\x41\xeb\x98\x4b\x75\x8f\x98\xcb\x02\x47\x13\x07\x07\x10" \
    +b"\x3a\xc0\x22\x47\x7d\x17\x3a\xc4\x69\xfb\x93\x77\x85\x00\xd5\xcb" \
    +b"\x93\x07\xf6\x0f\x2e\xc0\xa1\x83\x3e\xc4\x37\x27\x02\x40\x1c\x4b" \
    +b"\xc1\x66\x41\x68\xd5\x8f\x1c\xcb\xb7\x16\x00\x20\xb7\x27\x02\x40" \
    +b"\x93\x08\x00\x04\x37\x03\x20\x00\x98\x4b\x33\x67\x07\x01\x98\xcb" \
    +b"\xd8\x47\x05\x8b\x75\xff\x02\x47\x3a\xc2\x46\xc6\x32\x47\x0d\xef" \
    +b"\x98\x4b\x33\x67\x67\x00\x98\xcb\xd8\x47\x05\x8b\x75\xff\xd8\x47" \
    +b"\x41\x8b\x39\xc3\xd8\x47\xc1\x76\xfd\x16\x13\x67\x07\x01\xd8\xc7" \
    +b"\x98\x4b\x21\x45\x75\x8f\x98\xcb\x41\x01\x02\x90\x23\x20\xd8\x00" \
    +b"\x25\xb7\x23\x20\x03\x01\xa5\xb7\x12\x47\x13\x8e\x46\x00\x94\x42" \
    +b"\x14\xc3\x12\x47\x11\x07\x3a\xc2\x32\x47\x7d\x17\x3a\xc6\xd8\x47" \
    +b"\x09\x8b\x75\xff\xf2\x86\x5d\xb7\x02\x47\x13\x07\x07\x10\x3a\xc0" \
    +b"\x22\x47\x7d\x17\x3a\xc4\x49\xf3\x98\x4b\xc1\x76\xfd\x16\x75\x8f" \
    +b"\x98\xcb\x41\x89\x15\xc9\x2e\xc0\x0d\x06\x02\xc4\x09\x82\x32\xc6" \
    +b"\xb7\x17\x00\x20\x98\x43\x13\x86\x47\x00\xa2\x47\x82\x46\x8a\x07" \
    +b"\xb6\x97\x9c\x43\x63\x1c\xf7\x00\xa2\x47\x85\x07\x3e\xc4\xa2\x46" \
    +b"\x32\x47\xb2\x87\xe3\xe0\xe6\xfe\x01\x45\xbd\xbf\x41\x45\xad\xbf"

LOADER_X035 = \
     b"\x01\x11\x02\xce\x93\x77\x15\x00\x99\xcf\xb7\x06\x67\x45\xb7\x27" \
    +b"\x02\x40\x93\x86\x36\x12\x37\x97\xef\xcd\xd4\xc3\x13\x07\xb7\x9a" \
    +b"\xd8\xc3\xd4\xd3\xd8\xd3\x93\x77\x25\x00\x9d\xc7\xb7\x27\x02\x40" \
    +b"\x98\x4b\xad\x66\x37\x38\x00\x40\x13\x67\x47\x00\x98\xcb\x98\x4b" \
    +b"\x93\x86\xa6\xaa\x13\x67\x07\x04\x98\xcb\xd8\x47\x05\x8b\x63\x16" \
    +b"\x07\x10\x98\x4b\x6d\x9b\x98\xcb\x93\x77\x45\x00\xa9\xcb\x93\x07" \
    +b"\xf6\x0f\xa1\x83\x2e\xc6\x2d\x68\x81\x76\x3e\xca\xb7\x08\x02\x00" \
    +b"\xb7\x27\x02\x40\x37\x33\x00\x40\x13\x08\xa8\xaa\xfd\x16\x98\x4b" \
    +b"\x33\x67\x17\x01\x98\xcb\x32\x47\xd8\xcb\x98\x4b\x13\x67\x07\x04" \
    +b"\x98\xcb\xd8\x47\x05\x8b\x69\xe7\x98\x4b\x75\x8f\x98\xcb\x32\x47" \
    +b"\x13\x07\x07\x10\x3a\xc6\x52\x47\x7d\x17\x3a\xca\x69\xfb\x93\x77" \
    +b"\x85\x00\xf1\xcf\x93\x07\xf6\x0f\x2e\xc6\xa1\x83\x3e\xca\x37\x27" \
    +b"\x02\x40\x1c\x4b\xc1\x66\x2d\x68\xd5\x8f\x1c\xcb\xb7\x16\x00\x20" \
    +b"\xb7\x27\x02\x40\x37\x03\x08\x00\x13\x0e\x00\x04\xb7\x0e\x04\x00" \
    +b"\xb7\x38\x00\x40\x13\x08\xa8\xaa\x98\x4b\x33\x67\x67\x00\x98\xcb" \
    +b"\xd8\x47\x05\x8b\x75\xff\x32\x47\x36\x8f\x3a\xc8\x72\xcc\x42\x47" \
    +b"\x03\x2f\x0f\x00\x91\x06\x23\x20\xe7\x01\x98\x4b\x33\x67\xd7\x01" \
    +b"\x98\xcb\xd8\x47\x05\x8b\x21\xeb\x42\x47\x36\x8f\x11\x07\x3a\xc8" \
    +b"\x62\x47\x7d\x17\x3a\xcc\x61\xff\x32\x47\xd8\xcb\x98\x4b\x13\x67" \
    +b"\x07\x04\x98\xcb\xd8\x47\x05\x8b\x15\xeb\xd8\x47\x41\x8b\x15\xcb" \
    +b"\xd8\x47\xb7\x06\xf3\xff\xfd\x16\x13\x67\x07\x01\xd8\xc7\x98\x4b" \
    +b"\x21\x45\x75\x8f\x98\xcb\x05\x61\x02\x90\x23\x20\xd8\x00\xf5\xb5" \
    +b"\x23\x20\x03\x01\x3d\xb7\x23\xa0\x08\x01\x65\xb7\x23\xa0\x08\x01" \
    +b"\xd1\xb7\x32\x47\x13\x07\x07\x10\x3a\xc6\x52\x47\x7d\x17\x3a\xca" \
    +b"\x25\xf7\x98\x4b\xb7\x06\xf3\xff\xfd\x16\x75\x8f\x98\xcb\x41\x89" \
    +b"\x19\xe1\x01\x45\xc9\xb7\x2e\xc6\x0d\x06\x02\xca\x09\x82\x32\xcc" \
    +b"\xb7\x17\x00\x20\x98\x43\x13\x86\x47\x00\xd2\x47\xb2\x46\x8a\x07" \
    +b"\xb6\x97\x9c\x43\x63\x18\xf7\x02\xd2\x47\x32\x47\x8a\x07\xba\x97" \
    +b"\x98\x43\xf2\x47\xba\x97\x3e\xce\xd2\x47\x85\x07\x3e\xca\xd2\x46" \
    +b"\x62\x47\xb2\x87\xe3\xe8\xe6\xfc\xb7\x27\x00\x20\x98\x4b\xf2\x47" \
    +b"\xe3\x09\xf7\xfa\x41\x45\x85\xbf"

LOADER_L103 = \
     b"\x01\x11\x02\xce\x93\x77\x15\x00\x99\xcf\xb7\x06\x67\x45\xb7\x27" \
    +b"\x02\x40\x93\x86\x36\x12\x37\x97\xef\xcd\xd4\xc3\x13\x07\xb7\x9a" \
    +b"\xd8\xc3\xd4\xd3\xd8\xd3\x93\x77\x25\x00\x9d\xc7\xb7\x27\x02\x40" \
    +b"\x98\x4b\xad\x66\x37\x38\x00\x40\x13\x67\x47\x00\x98\xcb\x98\x4b" \
    +b"\x93\x86\xa6\xaa\x13\x67\x07\x04\x98\xcb\xd8\x47\x05\x8b\x63\x16" \
    +b"\x07\x10\x98\x4b\x6d\x9b\x98\xcb\x93\x77\x45\x00\xa9\xcb\x93\x07" \
    +b"\xf6\x0f\xa1\x83\x2e\xc6\x2d\x68\x81\x76\x3e\xca\xb7\x08\x02\x00" \
    +b"\xb7\x27\x02\x40\x37\x33\x00\x40\x13\x08\xa8\xaa\xfd\x16\x98\x4b" \
    +b"\x33\x67\x17\x01\x98\xcb\x32\x47\xd8\xcb\x98\x4b\x13\x67\x07\x04" \
    +b"\x98\xcb\xd8\x47\x05\x8b\x69\xe7\x98\x4b\x75\x8f\x98\xcb\x32\x47" \
    +b"\x13\x07\x07\x10\x3a\xc6\x52\x47\x7d\x17\x3a\xca\x69\xfb\x93\x77" \
    +b"\x85\x00\xf1\xcf\x93\x07\xf6\x0f\x2e\xc6\xa1\x83\x3e\xca\x37\x27" \
    +b"\x02\x40\x1c\x4b\xc1\x66\x2d\x68\xd5\x8f\x1c\xcb\xb7\x16\x00\x20" \
    +b"\xb7\x27\x02\x40\x37\x03\x08\x00\x13\x0e\x00\x04\xb7\x0e\x04\x00" \
    +b"\xb7\x38\x00\x40\x13\x08\xa8\xaa\x98\x4b\x33\x67\x67\x00\x98\xcb" \
    +b"\xd8\x47\x05\x8b\x75\xff\x32\x47\x36\x8f\x3a\xc8\x72\xcc\x42\x47" \
    +b"\x03\x2f\x0f\x00\x91\x06\x23\x20\xe7\x01\x98\x4b\x33\x67\xd7\x01" \
    +b"\x98\xcb\xd8\x47\x05\x8b\x21\xeb\x42\x47\x36\x8f\x11\x07\x3a\xc8" \
    +b"\x62\x47\x7d\x17\x3a\xcc\x61\xff\x32\x47\xd8\xcb\x98\x4b\x13\x67" \
    +b"\x07\x04\x98\xcb\xd8\x47\x05\x8b\x15\xeb\xd8\x47\x41\x8b\x15\xcb" \
    +b"\xd8\x47\xb7\x06\xf3\xff\xfd\x16\x13\x67\x07\x01\xd8\xc7\x98\x4b" \
    +b"\x21\x45\x75\x8f\x98\xcb\x05\x61\x02\x90\x23\x20\xd8\x00\xf5\xb5" \
    +b"\x23\x20\x03\x01\x3d\xb7\x23\xa0\x08\x01\x65\xb7\x23\xa0\x08\x01" \
    +b"\xd1\xb7\x32\x47\x13\x07\x07\x10\x3a\xc6\x52\x47\x7d\x17\x3a\xca" \
    +b"\x25\xf7\x98\x4b\xb7\x06\xf3\xff\xfd\x16\x75\x8f\x98\xcb\x41\x89" \
    +b"\x19\xe1\x01\x45\xc9\xb7\x2e\xc6\x0d\x06\x02\xca\x09\x82\x32\xcc" \
    +b"\xb7\x17\x00\x20\x98\x43\x13\x86\x47\x00\xd2\x47\xb2\x46\x8a\x07" \
    +b"\xb6\x97\x9c\x43\x63\x18\xf7\x02\xd2\x47\x32\x47\x8a\x07\xba\x97" \
    +b"\x98\x43\xf2\x47\xba\x97\x3e\xce\xd2\x47\x85\x07\x3e\xca\xd2\x46" \
    +b"\x62\x47\xb2\x87\xe3\xe8\xe6\xfc\xb7\x27\x00\x20\x98\x4b\xf2\x47" \
    +b"\xe3\x09\xf7\xfa\x41\x45\x85\xbf"

LOADER_573 = \
     b"\x79\x71\x22\xd4\x4a\xd0\x52\xcc\x06\xd6\x26\xd2\x4e\xce\x56\xca" \
    +b"\x5a\xc8\x5e\xc6\x93\x77\x15\x00\x2a\x84\x2e\x8a\x32\x89\x9d\xef" \
    +b"\x93\x77\x24\x00\x99\xcb\xb7\x86\x07\x00\x01\x46\x81\x45\x05\x45" \
    +b"\x69\x22\x93\x77\xf5\x0f\x09\x45\x9d\xeb\x93\x77\x44\x00\x91\xcb" \
    +b"\x85\x66\x01\x46\xd2\x85\x05\x45\x8d\x2a\x93\x77\xf5\x0f\x11\x45" \
    +b"\x99\xef\x93\x77\x84\x01\x9d\xe7\x01\x45\x11\xa8\x81\x46\x01\x46" \
    +b"\x81\x45\x21\x45\x99\x2a\x93\x77\xf5\x0f\x05\x45\xd5\xdb\xb2\x50" \
    +b"\x22\x54\x92\x54\x02\x59\xf2\x49\x62\x4a\xd2\x4a\x42\x4b\xb2\x4b" \
    +b"\x45\x61\x02\x90\xb7\x54\x00\x20\x13\x09\xf9\x0f\x93\x84\x04\x10" \
    +b"\x93\x7b\x84\x00\x13\x59\x89\x00\x81\x4a\x33\x0a\x9a\x40\x41\x88" \
    +b"\x33\x0b\x9a\x00\x63\x90\x0b\x02\x21\xc0\x93\x89\x04\xf0\x93\x06" \
    +b"\x00\x10\x4e\x86\xda\x85\x0d\x45\x09\x22\x13\x75\xf5\x0f\x19\xcd" \
    +b"\x41\x45\x75\xb7\x93\x06\x00\x10\x13\x86\x04\xf0\xda\x85\x09\x45" \
    +b"\xed\x20\x13\x75\xf5\x0f\x69\xd9\x21\x45\x51\xbf\x83\xa7\x09\x00" \
    +b"\x91\x09\xbe\x9a\xe3\x9c\x99\xfe\x7d\x19\x93\x84\x04\x10\xe3\x19" \
    +b"\x09\xfa\x3d\xd0\xb7\x67\x00\x20\x9c\x4b\xe3\x8f\x57\xf5\xc9\xb7" \
    +b"\x23\x03\x04\x80\x95\x47\x23\x03\xf4\x80\x23\x02\xa4\x80\x82\x80" \
    +b"\x83\x07\x64\x80\xe3\xce\x07\xfe\x23\x03\x04\x80\x82\x80\x83\x07" \
    +b"\x64\x80\xe3\xce\x07\xfe\x03\x45\x44\x80\x82\x80\x83\x07\x64\x80" \
    +b"\xe3\xce\x07\xfe\x23\x02\xa4\x80\x82\x80\x41\x11\x26\xc4\x4a\xc2" \
    +b"\x4e\xc0\x06\xc6\x13\x77\xf5\x0b\xad\x47\xaa\x89\x2e\x89\x95\x44" \
    +b"\x63\x06\xf7\x00\x19\x45\x6d\x37\x65\x3f\x8d\x44\x4e\x85\x4d\x37" \
    +b"\xfd\x59\xfd\x14\x63\x98\x34\x01\xb2\x40\xa2\x44\x12\x49\x82\x49" \
    +b"\x41\x01\x82\x80\x13\x55\x09\x01\x13\x75\xf5\x0f\x45\x3f\x22\x09" \
    +b"\xcd\xb7\x01\x11\x26\xcc\x06\xce\xb7\x04\x08\x00\x51\x37\x15\x45" \
    +b"\x85\x3f\x71\x37\x69\x37\x2a\xc6\xa5\x3f\x32\x45\x93\x77\x15\x00" \
    +b"\x89\xeb\x13\x65\x15\x00\x13\x75\xf5\x0f\xf2\x40\xe2\x44\x05\x61" \
    +b"\x82\x80\xfd\x14\xe9\xfc\x01\x45\xcd\xbf\x39\x71\x26\xdc\x4a\xda" \
    +b"\x4e\xd8\x52\xd6\x56\xd4\x5a\xd2\x5e\xd0\x06\xde\x62\xce\x66\xcc" \
    +b"\xb7\xe7\x00\xe0\x7d\x57\x83\xaa\x07\x00\x22\xc6\x03\xaa\x47\x00" \
    +b"\x23\xa0\xe7\x18\x23\xa2\xe7\x18\xb7\x17\x00\x40\x13\x07\x70\x05" \
    +b"\x23\x80\xe7\x04\x13\x07\x80\xfa\x23\x80\xe7\x04\x83\xc7\x47\x04" \
    +b"\x13\x09\x75\xff\xb6\x84\xe2\x07\x13\x79\xf9\x0f\x85\x46\xaa\x89" \
    +b"\xae\x8b\x32\x8b\x37\x24\x00\x40\xe1\x87\x01\x57\x63\xfa\x26\x01" \
    +b"\x63\x08\xd5\x00\x89\x46\x13\x07\x00\x02\x63\x13\xd5\x00\x01\x57" \
    +b"\xd9\x8f\x93\xf7\xf7\x0f\xb7\x1c\x00\x40\x23\x82\xfc\x04\x11\x47" \
    +b"\x23\x03\xe4\x80\x13\x05\xf0\x0f\x65\x3d\x09\x4c\xd1\x35\x63\x69" \
    +b"\x2c\x11\xb7\x07\x07\x00\xbe\x9b\x37\x87\x07\x00\x79\x55\x63\xfe" \
    +b"\xeb\x02\xb3\x87\x9b\x00\x63\x6a\xf7\x02\xa9\x47\x63\x99\xf9\x06" \
    +b"\x89\xe4\x81\x44\x71\x3d\x26\x85\x0d\xa0\xde\x85\x09\x45\x75\x3d" \
    +b"\x05\x0b\x03\x45\xfb\xff\xfd\x14\x85\x0b\x4d\x35\x81\xc4\x93\xf7" \
    +b"\xfb\x0f\xfd\xf7\xfd\x35\x69\xfd\x7d\x55\xb7\x17\x00\x40\x13\x07" \
    +b"\x70\x05\x23\x80\xe7\x04\x13\x07\x80\xfa\x23\x80\xe7\x04\x03\xc7" \
    +b"\x47\x04\x41\x8b\x23\x82\xe7\x04\xf2\x50\xb7\xe7\x00\xe0\x23\xa0" \
    +b"\x57\x11\x23\xa2\x47\x11\xe2\x54\x32\x44\x52\x59\xc2\x59\x32\x5a" \
    +b"\xa2\x5a\x12\x5b\x82\x5b\x72\x4c\xe2\x4c\x21\x61\x82\x80\xa5\x47" \
    +b"\x63\x95\xf9\x06\x85\x69\x13\x09\xf0\x0f\xb3\x06\x99\x00\xb3\xf4" \
    +b"\x2b\x01\xb6\x94\x13\x49\xf9\xff\xb3\x74\x99\x00\x41\x6b\x33\x79" \
    +b"\x79\x01\x85\x6b\x93\x87\xf9\xff\xb3\xf7\x27\x01\x99\xe3\x63\xfc" \
    +b"\x34\x01\x93\xd9\x49\x00\xc1\x47\xe3\xe6\x37\xff\x99\xbf\x05\x69" \
    +b"\xc1\x69\x7d\x19\xd9\xb7\x13\x05\x80\x0d\x63\x88\x69\x01\x13\x05" \
    +b"\x00\x02\x63\x84\x79\x01\x13\x05\x10\x08\xca\x85\xfd\x3b\x91\x35" \
    +b"\x21\xdd\x4e\x99\xb3\x84\x34\x41\xd9\xb7\xde\x85\x2d\x45\xf5\x33" \
    +b"\xda\x94\xe3\x00\x9b\xf2\x05\x0b\xd9\x33\xa3\x0f\xab\xfe\xd5\xbf" \
    +b"\x93\x87\xf9\xff\x93\xf7\xf7\x0f\x63\x6c\xfc\x08\x03\xc7\x5c\x04" \
    +b"\xb7\x07\x08\x00\x13\x77\x07\x02\x19\xe3\xb7\x87\x07\x00\x79\x55" \
    +b"\xe3\xfd\xfb\xf0\x33\x87\x9b\x00\xe3\xe9\xe7\xf0\x89\x47\x63\x90" \
    +b"\xf9\x04\x89\x80\x55\x49\xe3\x8e\x04\xec\xde\x85\x09\x45\x71\x3b" \
    +b"\x11\x0b\x03\x27\xcb\xff\x91\x47\x23\x20\xe4\x80\x03\x07\x64\x80" \
    +b"\xe3\x4e\x07\xfe\x23\x03\x24\x81\xfd\x17\xed\xfb\xfd\x14\x91\x0b" \
    +b"\x81\xc4\x93\xf7\xfb\x0f\xe9\xff\x6d\x3b\x71\xf5\xf1\xb5\x85\x47" \
    +b"\xe3\x87\xf9\xf4\xde\x85\x2d\x45\x8d\x33\x13\x89\xf4\xff\xe3\x8a" \
    +b"\x04\xe8\x35\x3b\x93\x77\x39\x00\x91\xeb\x83\x26\x04\x80\x03\x27" \
    +b"\x0b\x00\x93\x07\x4b\x00\xe3\x9f\xe6\xe6\x3e\x8b\xca\x84\xf1\xbf" \
    +b"\xa1\x47\x63\x92\xf9\x04\xb5\x3b\x81\x44\x63\x8d\x0b\x00\x8d\x47" \
    +b"\x93\x04\xc0\x03\x63\x88\xfb\x00\x93\x04\x00\x05\x63\x84\x8b\x01" \
    +b"\x93\x04\x40\x04\x13\x75\xc5\x07\xe3\x05\x95\xe4\x19\x45\xc9\x39" \
    +b"\xc5\x31\x05\x45\xf1\x31\x26\x85\xd5\x39\x09\x45\xc5\x39\x91\x33" \
    +b"\xe3\x19\x05\xe2\x91\xbd\xe3\x86\x09\xe2\xf1\x54\x25\xb5"

LOADER_583 = \
     b"\x79\x71\x22\xd4\x4a\xd0\x56\xca\x06\xd6\x26\xd2\x4e\xce\x52\xcc" \
    +b"\x5a\xc8\x5e\xc6\x62\xc4\x93\x77\x15\x00\x2a\x84\xae\x8a\x32\x89" \
    +b"\xc1\xe3\x93\x77\x24\x00\x99\xcb\xb7\x86\x07\x00\x01\x46\x81\x45" \
    +b"\x05\x45\x49\x2a\x93\x77\xf5\x0f\x09\x45\xa5\xef\x93\x77\x44\x00" \
    +b"\x91\xcb\x85\x66\x01\x46\xd6\x85\x05\x45\xad\x2a\x93\x77\xf5\x0f" \
    +b"\x11\x45\xa5\xe3\x93\x77\x84\x01\x01\x4a\xd9\xcf\xb7\x54\x00\x20" \
    +b"\x13\x09\xf9\x0f\x93\x84\x04\x10\x13\x59\x89\x00\x01\x4a\xb3\x8a" \
    +b"\x9a\x40\x93\x7b\x84\x00\x13\x7c\x04\x01\x33\x8b\x9a\x00\x63\x96" \
    +b"\x0b\x04\x63\x06\x0c\x06\x93\x89\x04\xf0\x93\x06\x00\x10\x4e\x86" \
    +b"\xda\x85\x0d\x45\x05\x2a\x13\x75\xf5\x0f\x21\xc5\x41\x45\x11\xa8" \
    +b"\x81\x46\x01\x46\x81\x45\x21\x45\x31\x2a\x93\x77\xf5\x0f\x05\x45" \
    +b"\xad\xdb\xb2\x50\x22\x54\x92\x54\x02\x59\xf2\x49\x62\x4a\xd2\x4a" \
    +b"\x42\x4b\xb2\x4b\x22\x4c\x45\x61\x02\x90\x93\x06\x00\x10\x13\x86" \
    +b"\x04\xf0\xda\x85\x09\x45\xfd\x20\x13\x75\xf5\x0f\x5d\xd1\x21\x45" \
    +b"\xc9\xbf\x83\xa7\x09\x00\x91\x09\x3e\x9a\xe3\x9c\x34\xff\x7d\x19" \
    +b"\x93\x84\x04\x10\xe3\x13\x09\xf8\x41\x88\x01\x45\x5d\xd8\xb7\x67" \
    +b"\x00\x20\x9c\x4b\xe3\x87\x47\xfb\x51\xbf\x23\x03\x04\x80\x95\x47" \
    +b"\x23\x03\xf4\x80\x23\x02\xa4\x80\x82\x80\x83\x07\x64\x80\xe3\xce" \
    +b"\x07\xfe\x23\x03\x04\x80\x82\x80\x83\x07\x64\x80\xe3\xce\x07\xfe" \
    +b"\x03\x45\x44\x80\x82\x80\x83\x07\x64\x80\xe3\xce\x07\xfe\x23\x02" \
    +b"\xa4\x80\x82\x80\x41\x11\x26\xc4\x4a\xc2\x4e\xc0\x06\xc6\x13\x77" \
    +b"\xf5\x0b\xad\x47\xaa\x89\x2e\x89\x95\x44\x63\x06\xf7\x00\x19\x45" \
    +b"\x6d\x37\x65\x3f\x8d\x44\x4e\x85\x4d\x37\xfd\x59\xfd\x14\x63\x98" \
    +b"\x34\x01\xb2\x40\xa2\x44\x12\x49\x82\x49\x41\x01\x82\x80\x13\x55" \
    +b"\x09\x01\x13\x75\xf5\x0f\x45\x3f\x22\x09\xcd\xb7\x01\x11\x26\xcc" \
    +b"\x06\xce\xb7\x04\x08\x00\x51\x37\x15\x45\x85\x3f\x71\x37\x69\x37" \
    +b"\x2a\xc6\xa5\x3f\x32\x45\x93\x77\x15\x00\x89\xeb\x13\x65\x15\x00" \
    +b"\x13\x75\xf5\x0f\xf2\x40\xe2\x44\x05\x61\x82\x80\xfd\x14\xe9\xfc" \
    +b"\x01\x45\xcd\xbf\x39\x71\x26\xdc\x4a\xda\x4e\xd8\x52\xd6\x56\xd4" \
    +b"\x5a\xd2\x5e\xd0\x06\xde\x62\xce\x66\xcc\xb7\xe7\x00\xe0\x7d\x57" \
    +b"\x83\xaa\x07\x00\x22\xc6\x03\xaa\x47\x00\x23\xa0\xe7\x18\x23\xa2" \
    +b"\xe7\x18\xb7\x17\x00\x40\x13\x07\x70\x05\x23\x80\xe7\x04\x13\x07" \
    +b"\x80\xfa\x23\x80\xe7\x04\x83\xc7\x47\x04\x93\x0b\x75\xff\xb6\x84" \
    +b"\xe2\x07\x93\xfb\xfb\x0f\x85\x46\xaa\x89\x2e\x8b\x32\x89\x37\x24" \
    +b"\x00\x40\xe1\x87\x01\x57\x63\xfa\x76\x01\x63\x08\xd5\x00\x89\x46" \
    +b"\x13\x07\x00\x02\x63\x13\xd5\x00\x01\x57\xd9\x8f\x93\xf7\xf7\x0f" \
    +b"\xb7\x1c\x00\x40\x23\x82\xfc\x04\x11\x47\x23\x03\xe4\x80\x13\x05" \
    +b"\xf0\x0f\x65\x3d\x09\x4c\xd1\x35\x63\x6d\x7c\x11\xb7\x05\x07\x00" \
    +b"\xda\x95\x37\x87\x07\x00\x79\x55\x63\xf2\xe5\x04\xb3\x87\x95\x00" \
    +b"\x63\x6e\xf7\x02\x37\x0b\x08\x00\xa9\x47\x33\xeb\x65\x01\x63\x99" \
    +b"\xf9\x06\x89\xe4\x81\x44\x51\x3d\x26\x85\x0d\xa0\xda\x85\x09\x45" \
    +b"\x55\x3d\x05\x09\x03\x45\xf9\xff\xfd\x14\x05\x0b\x69\x3d\x81\xc4" \
    +b"\x93\x77\xfb\x0f\xfd\xf7\xdd\x35\x69\xfd\x7d\x55\xb7\x17\x00\x40" \
    +b"\x13\x07\x70\x05\x23\x80\xe7\x04\x13\x07\x80\xfa\x23\x80\xe7\x04" \
    +b"\x03\xc7\x47\x04\x41\x8b\x23\x82\xe7\x04\xf2\x50\xb7\xe7\x00\xe0" \
    +b"\x23\xa0\x57\x11\x23\xa2\x47\x11\xe2\x54\x32\x44\x52\x59\xc2\x59" \
    +b"\x32\x5a\xa2\x5a\x12\x5b\x82\x5b\x72\x4c\xe2\x4c\x21\x61\x82\x80" \
    +b"\xa5\x47\x63\x95\xf9\x06\x85\x69\x13\x09\xf0\x0f\xb3\x06\x99\x00" \
    +b"\xb3\x74\x2b\x01\xb6\x94\x13\x49\xf9\xff\xb3\x74\x99\x00\x85\x6b" \
    +b"\x33\x79\x69\x01\x41\x6b\x93\x87\xf9\xff\xb3\xf7\x27\x01\x99\xe3" \
    +b"\x63\xfc\x34\x01\x93\xd9\x49\x00\xc1\x47\xe3\xe6\x37\xff\x99\xbf" \
    +b"\x05\x69\xc1\x69\x7d\x19\xd9\xb7\x13\x05\x80\x0d\x63\x88\x69\x01" \
    +b"\x13\x05\x00\x02\x63\x84\x79\x01\x13\x05\x10\x08\xca\x85\xdd\x3b" \
    +b"\x35\x3d\x21\xdd\x4e\x99\xb3\x84\x34\x41\xd9\xb7\xda\x85\x2d\x45" \
    +b"\xd5\x33\xca\x94\xe3\x00\x99\xf2\x05\x09\x7d\x3b\xa3\x0f\xa9\xfe" \
    +b"\xd5\xbf\x93\x87\xf9\xff\x93\xf7\xf7\x0f\x63\x61\xfc\x0c\x83\xc7" \
    +b"\x1c\x04\x13\x07\x30\x08\x63\x1f\xf7\x04\x37\x07\x08\x00\x63\x6b" \
    +b"\xeb\x04\xb3\x07\x9b\x00\xb7\x06\x10\x00\x63\xf5\xd7\x04\x33\x4b" \
    +b"\xeb\x00\x89\x47\x63\x93\xf9\x06\x89\x80\xd5\x49\xe3\x8c\x04\xec" \
    +b"\xda\x85\x09\x45\x41\x3b\x11\x09\x03\x27\xc9\xff\x91\x47\x23\x20" \
    +b"\xe4\x80\x03\x07\x64\x80\xe3\x4e\x07\xfe\x23\x03\x34\x81\xfd\x17" \
    +b"\xed\xfb\xfd\x14\x11\x0b\x81\xc4\x93\x77\xfb\x0f\xe9\xff\x7d\x33" \
    +b"\x71\xf5\xe1\xb5\xb7\x17\x00\x40\x03\xc7\x57\x04\xb7\x07\x08\x00" \
    +b"\x13\x77\x07\x02\x19\xe3\xb7\x87\x07\x00\x79\x55\xe3\x78\xfb\xea" \
    +b"\x33\x07\x9b\x00\xe3\xff\xe7\xf8\x55\xb5\x85\x47\xe3\x82\xf9\xf2" \
    +b"\xda\x85\x2d\x45\x05\x3b\x93\x89\xf4\xff\xe3\x85\x04\xe6\x29\x33" \
    +b"\x93\xf7\x39\x00\x91\xeb\x83\x26\x04\x80\x03\x27\x09\x00\x93\x07" \
    +b"\x49\x00\xe3\x9a\xe6\xe4\x3e\x89\xce\x84\xf1\xbf\xb5\x47\x63\x96" \
    +b"\xf9\x00\x13\x05\x90\x0b\xd1\x31\x35\xbd\xb1\x47\x13\x05\xb0\x0a" \
    +b"\xe3\x8b\xf9\xfe\x99\x47\x63\x91\xf9\x04\xb7\x05\x08\x00\xb3\x65" \
    +b"\xbb\x00\x2d\x45\xc5\x31\x81\x44\x8d\x4b\xa1\x49\x75\x39\x63\x96" \
    +b"\x74\x01\x83\x27\x04\x80\x23\x20\xf9\x00\x85\x04\xe3\x98\x34\xff" \
    +b"\x83\x27\x04\x80\x13\x17\x2b\x01\x63\x55\x07\x00\x23\x12\xf9\x00" \
    +b"\xd5\xbb\x23\x22\xf9\x00\xfd\xb3\x9d\x47\x63\x99\xf9\x02\x81\x45" \
    +b"\x13\x05\xb0\x04\x45\x31\xbd\x44\x23\x20\x09\x00\x23\x22\x09\x00" \
    +b"\xfd\x59\x9d\x39\x93\xf7\x74\x00\xca\x97\x03\xc7\x07\x00\xfd\x14" \
    +b"\x39\x8d\x23\x80\xa7\x00\xe3\x96\x34\xff\x6d\xbb\xa1\x47\x63\x92" \
    +b"\xf9\x04\x6d\x39\x81\x44\x63\x0d\x0b\x00\x8d\x47\x93\x04\xc0\x03" \
    +b"\x63\x08\xfb\x00\x93\x04\x00\x05\x63\x04\x8b\x01\x93\x04\x40\x04" \
    +b"\x13\x75\xc5\x07\xe3\x08\x95\xd8\x19\x45\x01\x39\x39\x39\x05\x45" \
    +b"\x29\x31\x26\x85\x0d\x39\x09\x45\x3d\x31\x49\x31\xe3\x1c\x05\xd6" \
    +b"\x69\xbb\x91\x47\x63\x99\xf9\x00\x13\x05\x60\x06\xfd\x36\xf5\x3e" \
    +b"\x13\x05\x90\x09\x0d\xb7\xe3\x8f\x09\xd4\xf1\x54\xa9\xbb"

# ===================================================================================
# Device definitions
# ===================================================================================

DEVICES = [
    {'name': 'CH32V003F4P6', 'id': 0x0030},
    {'name': 'CH32V003F4U6', 'id': 0x0031},
    {'name': 'CH32V003A4M6', 'id': 0x0032},
    {'name': 'CH32V003J4M6', 'id': 0x0033},

    {'name': 'CH32V103',     'id': 0x2500},

    {'name': 'CH32V203C8U6', 'id': 0x2030},
    {'name': 'CH32V203C8T6', 'id': 0x2031},
    {'name': 'CH32V203K8T6', 'id': 0x2032},
    {'name': 'CH32V203C6T6', 'id': 0x2033},
    {'name': 'CH32V203RBT6', 'id': 0x2034},
    {'name': 'CH32V203K6T6', 'id': 0x2035},
    {'name': 'CH32V203G6U6', 'id': 0x2036},
    {'name': 'CH32V203F6P6', 'id': 0x2037},
    {'name': 'CH32V203F8P6', 'id': 0x203a},
    {'name': 'CH32V203G8R6', 'id': 0x203b},
    {'name': 'CH32V203F8U6', 'id': 0x203e},
    {'name': 'CH32V208WBU6', 'id': 0x2080},
    {'name': 'CH32V208RBT6', 'id': 0x2081},
    {'name': 'CH32V208CBU6', 'id': 0x2082},
    {'name': 'CH32V208GBU6', 'id': 0x2083},

    {'name': 'CH32V303VCT6', 'id': 0x3030},
    {'name': 'CH32V303RCT6', 'id': 0x3031},
    {'name': 'CH32V303RBT6', 'id': 0x3032},
    {'name': 'CH32V303CBT6', 'id': 0x3033},
    {'name': 'CH32V305RBT6', 'id': 0x3050},
    {'name': 'CH32V305FBP6', 'id': 0x3052},
    {'name': 'CH32V305GBU6', 'id': 0x305b},
    {'name': 'CH32V307VCT6', 'id': 0x3070},
    {'name': 'CH32V307RCT6', 'id': 0x3071},
    {'name': 'CH32V307WCU6', 'id': 0x3073},

    {'name': 'CH32X033F8P6', 'id': 0x035a},
    {'name': 'CH32X035R8T6', 'id': 0x0350},
    {'name': 'CH32X035C8T6', 'id': 0x0351},
    {'name': 'CH32X035G8U6', 'id': 0x0356},
    {'name': 'CH32X035F7P6', 'id': 0x0357},
    {'name': 'CH32X035G8R6', 'id': 0x035b},
    {'name': 'CH32X035F8U6', 'id': 0x035e},

    {'name': 'CH32L103C8U6', 'id': 0x1030},
    {'name': 'CH32L103C8T6', 'id': 0x1031},
    {'name': 'CH32L103K8U6', 'id': 0x1032},
    {'name': 'CH32L103F7P6', 'id': 0x1037},
    {'name': 'CH32L103F8P6', 'id': 0x103a},
    {'name': 'CH32L103G8R6', 'id': 0x103b},
    {'name': 'CH32L103F8U6', 'id': 0x103d},

    {'name': 'CH571', 'id': 0x7100, 'code_size': 196608},
    {'name': 'CH573', 'id': 0x7300, 'code_size': 458752},
    {'name': 'CH581', 'id': 0x8100, 'code_size': 196608},
    {'name': 'CH582', 'id': 0x8200, 'code_size': 458752},
    {'name': 'CH583', 'id': 0x8300, 'code_size': 458752},
    {'name': 'CH591', 'id': 0x9100, 'code_size': 196608},
    {'name': 'CH592', 'id': 0x9200, 'code_size': 458752}
]

FAMILIES = [
    {'id': 0x00, 'block_size':  64, 'loader': LOADER_V003, 'code_base' : 0x08000000, 'optbytes': b'\xf7\xff\xff\xff\xff\xff\xff'},
    {'id': 0x25, 'block_size': 128, 'loader': LOADER_V103, 'code_base' : 0x08000000, 'optbytes': b'\xff\xff\xff\xff\xff\xff\xff'},
    {'id': 0x20, 'block_size': 256, 'loader': LOADER_V203, 'code_base' : 0x08000000, 'optbytes': b'\x3f\xff\xff\xff\xff\xff\xff'},
    {'id': 0x30, 'block_size': 256, 'loader': LOADER_V203, 'code_base' : 0x08000000, 'optbytes': b'\x3f\xff\xff\xff\xff\xff\xff'},
    {'id': 0x03, 'block_size': 256, 'loader': LOADER_X035, 'code_base' : 0x08000000, 'optbytes': b'\xff\xff\xff\xff\xff\xff\xff'},
    {'id': 0x10, 'block_size': 256, 'loader': LOADER_L103, 'code_base' : 0x08000000, 'optbytes': b'\xff\xff\xff\xff\xff\xff\xff'},

    {'id': 0x71, 'block_size': 256, 'loader': LOADER_573,  'code_base' : 0x00000000, 'optbytes': None},
    {'id': 0x73, 'block_size': 256, 'loader': LOADER_573,  'code_base' : 0x00000000, 'optbytes': None},
    {'id': 0x81, 'block_size': 256, 'loader': LOADER_583,  'code_base' : 0x00000000, 'optbytes': None},
    {'id': 0x82, 'block_size': 256, 'loader': LOADER_583,  'code_base' : 0x00000000, 'optbytes': None},
    {'id': 0x83, 'block_size': 256, 'loader': LOADER_583,  'code_base' : 0x00000000, 'optbytes': None},
    {'id': 0x91, 'block_size': 256, 'loader': LOADER_583,  'code_base' : 0x00000000, 'optbytes': None},
    {'id': 0x92, 'block_size': 256, 'loader': LOADER_583,  'code_base' : 0x00000000, 'optbytes': None}
]

# ===================================================================================

if __name__ == "__main__":
    _main()
