#!/usr/bin/env bash
# Copyright 2019-2024 VMware, Inc.
# SPDX-License-Identifier: Apache-2.0

##############################################################################################################
# Extract key results from the reports generated by ...
#   "SLSCAN" - https://github.com/ShiftLeftSecurity/sast-scan & https://slscan.io/en/latest/
##############################################################################################################

# ----- Please adjust

# ------ Do not modify
VERSION=${SLSCAN_VERSION}
STEP=$(get_step)
SEPARATOR=","
APP_DIR_OUT="${REPORTS_DIR}/${STEP}__SLSCAN"
RESULT_FILE="${APP_DIR_OUT}/_results_extracted.csv"
export LOG_FILE="${APP_DIR_OUT}.log"

function generate_csv() {
	echo "Applications${SEPARATOR}SLScan SAST vulns" >"${RESULT_FILE}"

	while read -r FILE; do
		APP="$(basename "${FILE}")"
		log_extract_message "app '${APP}'"
		TXT_IN="${APP_DIR_OUT}/${APP}.txt"
		SLSCAN_OUTPUT_STATS="${APP_DIR_OUT}/${APP}.stats"
		VULNS="n/a"
		if [ -f "${TXT_IN}" ]; then
			VULNS=0
			COUNT_VULNS_LOW=0
			COUNT_VULNS_MEDIUM=0
			COUNT_VULNS_HIGH=0
			COUNT_VULNS_CRITICAL=0

			EXTRACTED_VALUES_TMP="${TXT_IN}.tmp"
			sed -n '/.*Tool.*Critical.*$/,$p' "${TXT_IN}" | tail -n +3 | sed 's/[^0-9 ]*//g' | sed 's/ \+/ /g' | sed 's/^ *//;s/ *$//' | tr -s ' ' ',' >"${EXTRACTED_VALUES_TMP}"

			COUNT_VULNS=$(sed -n '/.*Tool.*Critical.*$/,$p' "${TXT_IN}" | tail -n +3 | sed 's/[^0-9 ]*//g' | xargs | tr ' ' '\n' | awk '{n += $1}; END{print n}')
			[ -n "${COUNT_VULNS}" ] && VULNS=${COUNT_VULNS}

			LOW=$(cut -d',' -f 4 "${EXTRACTED_VALUES_TMP}" | xargs | tr ' ' '\n' | awk '{n += $1}; END{print n}')
			MEDIUM=$(cut -d',' -f 3 "${EXTRACTED_VALUES_TMP}" | xargs | tr ' ' '\n' | awk '{n += $1}; END{print n}')
			HIGH=$(cut -d',' -f 2 "${EXTRACTED_VALUES_TMP}" | xargs | tr ' ' '\n' | awk '{n += $1}; END{print n}')
			CRITICAL=$(cut -d',' -f 1 "${EXTRACTED_VALUES_TMP}" | xargs | tr ' ' '\n' | awk '{n += $1}; END{print n}')

			[ -n "${LOW}" ] && COUNT_VULNS_LOW=${LOW}
			[ -n "${MEDIUM}" ] && COUNT_VULNS_MEDIUM=${MEDIUM}
			[ -n "${HIGH}" ] && COUNT_VULNS_HIGH=${HIGH}
			[ -n "${CRITICAL}" ] && COUNT_VULNS_CRITICAL=${CRITICAL}

			COUNT_VULNS_ALL=$((COUNT_VULNS_LOW + COUNT_VULNS_MEDIUM + COUNT_VULNS_HIGH + COUNT_VULNS_CRITICAL))

			rm -f "${EXTRACTED_VALUES_TMP}"

			HAS_FINDINGS=false
			[[ ${COUNT_VULNS_ALL} -gt 0 ]] && {
				HAS_FINDINGS=true
			}

			if [[ "${OWASP_ACTIVE}" == "true" ||
				"${SCANCODE_ACTIVE}" == "true" ||
				"${FSB_ACTIVE}" == "true" ||
				"${TRIVY_ACTIVE}" == "true" ||
				"${INSIDER_ACTIVE}" == "true" ||
				"${GRYPE_ACTIVE}" == "true" ||
				"${OSV_ACTIVE}" == "true" ||
				"${BEARER_ACTIVE}" == "true" ]]; then
				HAS_ANOTHER_SECURITY_REPORT='TRUE'
			else
				HAS_ANOTHER_SECURITY_REPORT=''
			fi
			{
				echo "SLSCAN__VULNS_ALL=${COUNT_VULNS_ALL}"
				echo "SLSCAN__VULNS_LOW=${COUNT_VULNS_LOW}"
				echo "SLSCAN__VULNS_MEDIUM=${COUNT_VULNS_MEDIUM}"
				echo "SLSCAN__VULNS_HIGH=${COUNT_VULNS_HIGH}"
				echo "SLSCAN__VULNS_CRITICAL=${COUNT_VULNS_CRITICAL}"
				echo "HAS_ANOTHER_SECURITY_REPORT=${HAS_ANOTHER_SECURITY_REPORT}"
				echo "HAS_FINDINGS=${HAS_FINDINGS}"
			} >"${SLSCAN_OUTPUT_STATS}"
		fi
		echo "${APP}${SEPARATOR}${VULNS}" >>"${RESULT_FILE}"

	done <"${REPORTS_DIR}/00__Weave/list__all_apps.txt"

	log_console_success "Results: ${RESULT_FILE}"
}

function main() {
	if [[ -n $(find "${REPORTS_DIR}" -mindepth 1 -maxdepth 1 -type d -iname "${STEP}"'__SLSCAN*') ]]; then
		if [[ -d "${APP_DIR_OUT}" ]]; then
			generate_csv
		else
			LOG_FILE=/dev/null
			log_console_error "SLSCAN result directory does not exist: ${APP_DIR_OUT}"
		fi
	elif [[ "${ARCH}" == "arm64" ]]; then
		exit
	else
		log_console_error "No SLSCAN result directory found in ${REPORTS_DIR}."
	fi
}

main
