#!/usr/bin/env bash
# Copyright 2019-2024 VMware, Inc.
# SPDX-License-Identifier: Apache-2.0

##############################################################################################################
# Extract key results from the HTML files generated by ...
#   "IBM WebSphere Application Server Migration Toolkit" - https://www.ibm.com/developerworks/library/mw-1701-was-migration/index.html
##############################################################################################################

# ----- Please adjust

# ------ Do not modify
VERSION=${WAMT_VERSION}
STEP=$(get_step)
SEPARATOR=","
APP_DIR_OUT="${REPORTS_DIR}/${STEP}__WAMT"
export LOG_FILE="${APP_DIR_OUT}.log"
RESULT_FILE="${APP_DIR_OUT}/_results_extracted.csv"

JAVA_BIN_LIST="${REPORTS_DIR}/00__Weave/list__java-bin.txt"

function generate_csv() {
	if ! grep -q '.*\.[ejw]ar$' "${JAVA_BIN_LIST}"; then
		log_console_warning "No EAR/WAR Java application found. Skipping WAMT result extraction."
		return
	fi

	#echo "Applications${SEPARATOR}WAMT criticals${SEPARATOR}WAMT warns${SEPARATOR}WAMT findings" >"${RESULT_FILE}"
	echo "Applications${SEPARATOR}WAMT findings" >"${RESULT_FILE}"

	while read -r FILE; do
		APP="$(basename "${FILE}")"
		log_extract_message "app '${APP}'"

		WAMT_FILE="${APP_DIR_OUT}/${APP}.html"
		if [[ -f "${WAMT_FILE}" ]]; then
			TEMP_TABLE_FILE="${RESULT_FILE}.tmp"
			sed -n '/<table summary="This table summarizes how many rules and rule results are included in the report for each rule severity. A description for each rule severity is also provid*/,/<\/table>/p' "${WAMT_FILE}" >"${TEMP_TABLE_FILE}"
			if [ -s "${TEMP_TABLE_FILE}" ]; then
				# Very ugly but works to make sure that we capture the right value
				VALUES=$(grep "<td class=\"center\">" "${TEMP_TABLE_FILE}" |
					sed -n '1d;p;n' |
					sed 's:^<td class="center">\([0-9]*\)</td>:\1:g' |
					tr '\n' ',' |
					awk -F "," '{printf("%s,%s,%s", $1, $3, $5)}')

				KEYS=$(grep "<td class=\"severity-label\">" "${TEMP_TABLE_FILE}" |
					sed 's:^<td class="severity-label"><a[^>]*>\([^<]*\)<.*:\1:g' |
					tr '\n' ',')

				FIRST=$(echo "${KEYS}" | cut -d',' -f1)
				SECOND=$(echo "${KEYS}" | cut -d',' -f2)
				THIRD=$(echo "${KEYS}" | cut -d',' -f3)
				FIRST_VALUE=$(echo "${VALUES}" | cut -d',' -f1)
				SECOND_VALUE=$(echo "${VALUES}" | cut -d',' -f2)
				THIRD_VALUE=$(echo "${VALUES}" | cut -d',' -f3)

				CRITICALS="0"
				INFOS="0"
				WARNINGS="0"

				if [[ "${FIRST}" == "Warning" ]]; then
					WARNINGS="${FIRST_VALUE}"
				elif [[ "${FIRST}" == "Critical" ]]; then
					CRITICALS="${FIRST_VALUE}"
				elif [[ "${FIRST}" == "Information" ]]; then
					INFOS="${FIRST_VALUE}"
				fi

				if [[ "${SECOND}" == "Warning" ]]; then
					WARNINGS="${SECOND_VALUE}"
				elif [[ "${SECOND}" == "Critical" ]]; then
					CRITICALS="${SECOND_VALUE}"
				elif [[ "${SECOND}" == "Information" ]]; then
					INFOS="${SECOND_VALUE}"
				fi

				if [[ "${THIRD}" == "Warning" ]]; then
					WARNINGS="${THIRD_VALUE}"
				elif [[ "${THIRD}" == "Critical" ]]; then
					CRITICALS="${THIRD_VALUE}"
				elif [[ "${THIRD}" == "Information" ]]; then
					INFOS="${THIRD_VALUE}"
				fi

				TOTAL=$((CRITICALS + WARNINGS + INFOS))
				#echo "${APP}${SEPARATOR}${CRITICALS}${SEPARATOR}${WARNINGS}${SEPARATOR}${TOTAL}" >>"${RESULT_FILE}"
				echo "${APP}${SEPARATOR}${TOTAL}" >>"${RESULT_FILE}"
			else
				#echo "${APP}${SEPARATOR}0${SEPARATOR}0${SEPARATOR}0" >>"${RESULT_FILE}"
				echo "${APP}${SEPARATOR}0" >>"${RESULT_FILE}"
			fi
			rm "${TEMP_TABLE_FILE}"
		else
			echo "${APP}${SEPARATOR}n/a" >>"${RESULT_FILE}"
		fi

	done <"${REPORTS_DIR}/00__Weave/list__all_apps.txt"

	log_console_success "Results: ${RESULT_FILE}"
}

function main() {
	if [[ -d "${APP_DIR_OUT}" ]]; then
		generate_csv
	else
		LOG_FILE=/dev/null
		log_console_error "WAMT result directory does not exist: ${APP_DIR_OUT}"
	fi
}

main
