package com.varabyte.kobweb.gradle.application.ksp

import com.varabyte.kobweb.gradle.core.kmp.JsTarget
import com.varabyte.kobweb.gradle.core.kmp.JvmTarget
import com.varabyte.kobweb.ksp.KOBWEB_APP_METADATA_BACKEND
import com.varabyte.kobweb.ksp.KOBWEB_APP_METADATA_FRONTEND
import org.gradle.api.Project
import org.gradle.api.file.RegularFile
import org.gradle.api.provider.Provider

/**
 * A reference to the frontend metadata file generated by KSP.
 *
 * The underlying value is guaranteed to exist.
 *
 * The file is associated as an output of the KSP task that creates it, allowing gradle to properly generate task
 * dependencies when it is used as a task input.
 *
 * For example, registering a task like:
 *
 * ```
 * tasks.register("printFrontendMetadata") {
 *   inputs.file(project.kspFrontendFile(jsTarget))
 *   /* ... task action here ... */
 * }
 * ```
 *
 * will ensure that the `printFrontendMetadata` task runs after the frontend metadata file is created by KSP.
 */
fun Project.kspFrontendFile(jsTarget: JsTarget): Provider<RegularFile> {
    return project.files(tasks.named(jsTarget.kspKotlin)).asFileTree
        .matching { include(KOBWEB_APP_METADATA_FRONTEND) }
        .elements
        .map {
            RegularFile { it.single().asFile }
        }
}

/**
 * A reference to the backend metadata file generated by KSP.
 *
 * This represents an OPTIONAL value; it will not be present if no JVM sources are detected.
 *
 * The file is associated as an output of the KSP task that creates it, allowing gradle to properly generate task
 * dependencies when it is used as a task input.
 *
 * See [kspFrontendFile] for an example of how this can be used for task dependency inference.
 */
fun Project.kspBackendFile(jvmTarget: JvmTarget): Provider<RegularFile> {
    // Ideally we'd return null directly in a map {} function, but that's unsupported
    // See: https://github.com/gradle/gradle/issues/12388
    // So, instead, we zip with a dummy provider and return null from there (which is allowed :/)
    return project.files(tasks.named(jvmTarget.kspKotlin)).asFileTree
        .matching { include(KOBWEB_APP_METADATA_BACKEND) }
        .elements
        .zip(provider {}) { potentialFile, _ ->
            potentialFile.singleOrNull()?.let { RegularFile { it.asFile } }
        }
}
