/* SPDX-License-Identifier: BSD-3-Clause */
/* Copyright (c) 2022, Unikraft GmbH and The Unikraft Authors.
 * Licensed under the BSD-3-Clause License (the "License").
 * You may not use this file except in compliance with the License.
 */

#ifndef __UK_RANDOM_DRIVER_H__
#define __UK_RANDOM_DRIVER_H__

#include <uk/arch/types.h>
#include <uk/essentials.h>
#include <uk/prio.h>

#ifdef __cplusplus
extern "C" {
#endif

#define UK_RANDOM_EARLY_DRIVER_PRIO	UK_PRIO_AFTER(3)

struct uk_random_driver_ops {
	/**
	 * Get random bytes
	 *
	 * @param [out] buf  Buffer to store the generated bytes
	 * @param       size Buffer size
	 * @return 0 on success, negative value on failure
	 */
	int __check_result (*random_bytes)(__u8 *buf, __sz size);

	/**
	 * Get random bytes suitable for seeding
	 *
	 * Some random number generators, most notably those provided by CPUs,
	 * provide a separate operation for random numbers suitable for seeding.
	 * These normally originate directly from the TRNG instead of a seeded
	 * DRBG, so they have higher multiplicative prediction resistance.
	 *
	 * Devices that don't support this operation return an error.
	 *
	 * @param [out] buf  Buffer to store the generated bytes
	 * @param       size Buffer size
	 * @return 0 on success, negative value on failure
	 */
	int __check_result (*seed_bytes)(__u8 *buf, __sz size);

	/**
	 * seed_bytes() with fallback.
	 *
	 * This function implementes the same functionality as @seed_bytes
	 * but falls back to random_bytes() if the device does not provide
	 * a seed operation, or if the seed operation fails due to exhaustion.
	 *
	 * @param [out] buf  Buffer to store the generated bytes
	 * @param       size Buffer size
	 * @return 0 on success, negative value on failure
	 */
	int __check_result (*seed_bytes_fb)(__u8 *buf, __sz size);
};

struct uk_random_driver {
	const char *name;
	struct uk_random_driver_ops *ops;
};

/**
 * Initialize libukrandom
 *
 * Registers the calling driver with libukrandom and initializes the
 * software CSPRNG. Only one source of entropy is supported. If more
 * than one driver is enabled, only the first registration takes
 * effect.
 *
 * @param drv driver representation (@uk_random_driver)
 * @return zero on success, negative value on error
 */
int __check_result uk_random_init(struct uk_random_driver *drv);

#ifdef __cplusplus
}
#endif

#endif /* __UK_RANDOM_DRIVER_H__*/
