/* SPDX-License-Identifier: BSD-3-Clause */
/*
 ****************************************************************************
 *
 *        File: printf.c
 *      Author: Juergen Gross <jgross@suse.com>
 *
 *        Date: Jun 2016
 *
 * Environment: Xen Minimal OS
 * Description: Library functions for printing
 *              (FreeBSD port)
 *
 ****************************************************************************
 */

/*-
 * Copyright (c) 1990, 1993
 * The Regents of the University of California.  All rights reserved.
 *
 * This code is derived from software contributed to Berkeley by
 * Chris Torek.
 *
 * Copyright (c) 2011 The FreeBSD Foundation
 * All rights reserved.
 * Portions of this software were developed by David Chisnall
 * under sponsorship from the FreeBSD Foundation.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <errno.h>
#include <stdint.h>
#include <stdlib.h>
#include <limits.h>
#include <ctype.h>
#include <uk/print.h>
#include <uk/plat/bootstrap.h>

#define __DECONST(type, var) ((type)(uintptr_t)(const void *)(var))


#if CONFIG_LIBUKALLOC
/* Allocate size bytes of memory. Returns pointer to start of allocated memory,
 * or NULL on failure.
 */
void *malloc(size_t size)
{
	return uk_malloc(uk_alloc_get_default(), size);
}

/* Release memory previously allocated by malloc(). ptr must be a pointer
 * previously returned by malloc(), otherwise undefined behavior will occur.
 */
void free(void *ptr)
{
	return uk_free(uk_alloc_get_default(), ptr);
}

/* Allocate memory for an array of nmemb elements of size bytes. Returns
 * pointer to start of allocated memory, or NULL on failure.
 */
void *calloc(size_t nmemb, size_t size)
{
	return uk_calloc(uk_alloc_get_default(), nmemb, size);
}

/* Change the size of the memory block pointed to by ptr to size bytes.
 * Returns a pointer to the resized memory area. If the area pointed to was
 * moved, a free(ptr) is done. If the call fails, the memory remains unchanged
 */
void *realloc(void *ptr, size_t size)
{
	return uk_realloc(uk_alloc_get_default(), ptr, size);
}

/* Allocate size bytes of memory, aligned to align bytes, and return the
 * pointer to that area in *memptr. Returns 0 on success, and a non-zero error
 * value on failure.
 */
int posix_memalign(void **memptr, size_t align, size_t size)
{
	return uk_posix_memalign(uk_alloc_get_default(),
				 memptr, align, size);
}

/* Allocate size bytes of memory, aligned to align bytes. Returns pointer to
 * start of allocated memory, or NULL on failure.
 */
void *memalign(size_t align, size_t size)
{
	return uk_memalign(uk_alloc_get_default(), align, size);
}
#endif /* CONFIG_LIBUKALLOC */

/*
 * Convert a string to an unsigned long integer.
 *
 * Ignores `locale' stuff.  Assumes that the upper and lower case
 * alphabets and digits are each contiguous.
 */
unsigned long strtoul(const char *nptr, char **endptr, int base)
{
	const char *s = nptr;
	unsigned long acc;
	unsigned char c;
	unsigned long cutoff;
	int neg = 0, any, cutlim;

	if (base < 0 || base == 1 || base > 36) {
		errno = -EINVAL;
		any = 0;
		acc = 0;
		goto exit;
	}

	/*
	 * See strtol for comments as to the logic used.
	 */
	do {
		c = *s++;
	} while (isspace(c));
	if (c == '-') {
		neg = 1;
		c = *s++;
	} else if (c == '+')
		c = *s++;
	if ((base == 0 || base == 16) && c == '0' && (*s == 'x' || *s == 'X')) {
		c = s[1];
		s += 2;
		base = 16;
	}
	if (base == 0)
		base = c == '0' ? 8 : 10;
	cutoff = (unsigned long)ULONG_MAX / (unsigned long)base;
	cutlim = (unsigned long)ULONG_MAX % (unsigned long)base;
	for (acc = 0, any = 0;; c = *s++) {
		if (!isascii(c))
			break;
		if (isdigit(c))
			c -= '0';
		else if (isalpha(c))
			c -= isupper(c) ? 'A' - 10 : 'a' - 10;
		else
			break;
		if (c >= base)
			break;
		if (any < 0 || acc > cutoff || (acc == cutoff && c > cutlim))
			any = -1;
		else {
			any = 1;
			acc *= base;
			acc += c;
		}
	}
	if (any < 0) {
		acc = ULONG_MAX;
		errno = ERANGE;
	} else if (neg)
		acc = -acc;
exit:
	if (endptr != 0)
		*endptr = __DECONST(char *, any ? s - 1 : nptr);
	return acc;
}

long strtol(const char *nptr, char **endptr, int base)
{
	const char *s;
	unsigned long acc;
	unsigned char c;
	unsigned long qbase, cutoff;
	int neg, any, cutlim;

	s = nptr;
	if (base < 0 || base == 1 || base > 36) {
		errno = -EINVAL;
		any = 0;
		acc = 0;
		goto exit;
	}

	/*
	 * Skip white space and pick up leading +/- sign if any.
	 * If base is 0, allow 0x for hex and 0 for octal, else
	 * assume decimal; if base is already 16, allow 0x.
	 */
	do {
		c = *s++;
	} while (isspace(c));
	if (c == '-') {
		neg = 1;
		c = *s++;
	} else {
		neg = 0;
		if (c == '+')
			c = *s++;
	}
	if ((base == 0 || base == 16) && c == '0' && (*s == 'x' || *s == 'X')) {
		c = s[1];
		s += 2;
		base = 16;
	}
	if (base == 0)
		base = c == '0' ? 8 : 10;

	/*
	 * Compute the cutoff value between legal numbers and illegal
	 * numbers.  That is the largest legal value, divided by the
	 * base.  An input number that is greater than this value, if
	 * followed by a legal input character, is too big.  One that
	 * is equal to this value may be valid or not; the limit
	 * between valid and invalid numbers is then based on the last
	 * digit.  For instance, if the range for quads is
	 * [-2147483648..2147483647] and the input base
	 * is 10, cutoff will be set to 2147483647 and cutlim to
	 * either 7 (neg==0) or 8 (neg==1), meaning that if we have
	 * accumulated a value > 2147483647, or equal but the
	 * next digit is > 7 (or 8), the number is too big, and we will
	 * return a range error.
	 *
	 * Set any if any `digits' consumed; make it negative to indicate
	 * overflow.
	 */
	qbase = (unsigned int)base;
	cutoff = neg
		     ? (unsigned long)LONG_MAX
			   - (unsigned long)(LONG_MIN + LONG_MAX)
		     : LONG_MAX;
	cutlim = cutoff % qbase;
	cutoff /= qbase;
	for (acc = 0, any = 0;; c = *s++) {
		if (!isascii(c))
			break;
		if (isdigit(c))
			c -= '0';
		else if (isalpha(c))
			c -= isupper(c) ? 'A' - 10 : 'a' - 10;
		else
			break;
		if (c >= base)
			break;
		if (any < 0 || acc > cutoff || (acc == cutoff && c > cutlim))
			any = -1;
		else {
			any = 1;
			acc *= qbase;
			acc += c;
		}
	}
	if (any < 0) {
		acc = neg ? LONG_MIN : LONG_MAX;
		errno = ERANGE;
	} else if (neg)
		acc = -acc;

exit:
	if (endptr != 0)
		*endptr = __DECONST(char *, any ? s - 1 : nptr);
	return acc;
}

/*
 * Convert a string to a long long integer.
 *
 * Ignores `locale' stuff.  Assumes that the upper and lower case
 * alphabets and digits are each contiguous.
 */
long long strtoll(const char *nptr, char **endptr, int base)
{
	const char *s;
	unsigned long long acc;
	unsigned char c;
	unsigned long long qbase, cutoff;
	int neg, any, cutlim;

	s = nptr;
	if (base < 0 || base == 1 || base > 36) {
		errno = -EINVAL;
		any = 0;
		acc = 0;
		goto exit;
	}
	/*
	 * Skip white space and pick up leading +/- sign if any.
	 * If base is 0, allow 0x for hex and 0 for octal, else
	 * assume decimal; if base is already 16, allow 0x.
	 */
	do {
		c = *s++;
	} while (isspace(c));
	if (c == '-') {
		neg = 1;
		c = *s++;
	} else {
		neg = 0;
		if (c == '+')
			c = *s++;
	}
	if ((base == 0 || base == 16) && c == '0' && (*s == 'x' || *s == 'X')) {
		c = s[1];
		s += 2;
		base = 16;
	}
	if (base == 0)
		base = c == '0' ? 8 : 10;

	/*
	 * Compute the cutoff value between legal numbers and illegal
	 * numbers.  That is the largest legal value, divided by the
	 * base.  An input number that is greater than this value, if
	 * followed by a legal input character, is too big.  One that
	 * is equal to this value may be valid or not; the limit
	 * between valid and invalid numbers is then based on the last
	 * digit.  For instance, if the range for quads is
	 * [-9223372036854775808..9223372036854775807] and the input base
	 * is 10, cutoff will be set to 922337203685477580 and cutlim to
	 * either 7 (neg==0) or 8 (neg==1), meaning that if we have
	 * accumulated a value > 922337203685477580, or equal but the
	 * next digit is > 7 (or 8), the number is too big, and we will
	 * return a range error.
	 *
	 * Set any if any `digits' consumed; make it negative to indicate
	 * overflow.
	 */
	qbase = (unsigned int)base;
	cutoff = neg
		     ? (unsigned long long)LLONG_MAX
			   - (unsigned long long)(LLONG_MIN + LLONG_MAX)
		     : LLONG_MAX;
	cutlim = cutoff % qbase;
	cutoff /= qbase;
	for (acc = 0, any = 0;; c = *s++) {
		if (!isascii(c))
			break;
		if (isdigit(c))
			c -= '0';
		else if (isalpha(c))
			c -= isupper(c) ? 'A' - 10 : 'a' - 10;
		else
			break;
		if (c >= base)
			break;
		if (any < 0 || acc > cutoff || (acc == cutoff && c > cutlim))
			any = -1;
		else {
			any = 1;
			acc *= qbase;
			acc += c;
		}
	}
	if (any < 0) {
		errno = ERANGE;
		acc = neg ? LLONG_MIN : LLONG_MAX;
	} else if (neg)
		acc = -acc;

exit:
	if (endptr != 0)
		*endptr = __DECONST(char *, any ? s - 1 : nptr);
	return acc;
}

/*
 * Convert a string to an unsigned long long integer.
 *
 * Ignores `locale' stuff.  Assumes that the upper and lower case
 * alphabets and digits are each contiguous.
 */
unsigned long long strtoull(const char *nptr, char **endptr, int base)
{
	const char *s = nptr;
	unsigned long long acc;
	unsigned char c;
	unsigned long long qbase, cutoff;
	int neg, any, cutlim;

	if (base < 0 || base == 1 || base > 36) {
		errno = -EINVAL;
		any = 0;
		acc = 0;
		goto exit;
	}
	/*
	 * See strtoq for comments as to the logic used.
	 */
	do {
		c = *s++;
	} while (isspace(c));
	if (c == '-') {
		neg = 1;
		c = *s++;
	} else {
		neg = 0;
		if (c == '+')
			c = *s++;
	}
	if ((base == 0 || base == 16) && c == '0' && (*s == 'x' || *s == 'X')) {
		c = s[1];
		s += 2;
		base = 16;
	}
	if (base == 0)
		base = c == '0' ? 8 : 10;
	qbase = (unsigned int)base;
	cutoff = (unsigned long long)ULLONG_MAX / qbase;
	cutlim = (unsigned long long)ULLONG_MAX % qbase;
	for (acc = 0, any = 0;; c = *s++) {
		if (!isascii(c))
			break;
		if (isdigit(c))
			c -= '0';
		else if (isalpha(c))
			c -= isupper(c) ? 'A' - 10 : 'a' - 10;
		else
			break;
		if (c >= base)
			break;
		if (any < 0 || acc > cutoff || (acc == cutoff && c > cutlim))
			any = -1;
		else {
			any = 1;
			acc *= qbase;
			acc += c;
		}
	}
	if (any < 0) {
		errno = ERANGE;
		acc = ULLONG_MAX;
	} else if (neg)
		acc = -acc;

exit:
	if (endptr != 0)
		*endptr = __DECONST(char *, any ? s - 1 : nptr);
	return acc;
}

int atoi(const char *s)
{
	long long atoll;

	atoll = strtoll(s, NULL, 10);
	atoll = (atoll > __I_MAX) ? __I_MAX : atoll;
	atoll = (atoll < __I_MIN) ? __I_MIN : atoll;

	return (int) atoll;
}

long atol(const char *s)
{
	long long atoll;

	atoll = strtoll(s, NULL, 10);
	atoll = (atoll > __L_MAX) ? __L_MAX : atoll;
	atoll = (atoll < __L_MIN) ? __L_MIN : atoll;

	return (long) atoll;
}

void abort(void)
{
	uk_pr_crit("Abnormal termination!\n");
	ukplat_crash();
}

#ifndef CONFIG_LIBPOSIX_PROCESS
void exit(int status)
{
	uk_pr_info("exit called with status %d, halting system\n", status);
	ukplat_terminate(status);
}
#endif /* !CONFIG_LIBPOSIX_PROCESS */
