import { describe, it, expect } from 'vitest';
import { ScriptingDefinition } from '@/domain/ScriptingDefinition';
import { ScriptingLanguage } from '@/domain/ScriptingLanguage';
import { getEnumValues } from '@/application/Common/Enum';
import { itEachAbsentStringValue } from '@tests/unit/shared/TestCases/AbsentTests';

describe('ScriptingDefinition', () => {
  describe('language', () => {
    describe('sets as expected', () => {
      // arrange
      const expectedValues = getEnumValues(ScriptingLanguage);
      expectedValues.forEach((expected) => {
        it(ScriptingLanguage[expected], () => {
          // act
          const sut = new ScriptingDefinitionBuilder()
            .withLanguage(expected)
            .build();
          // assert
          expect(sut.language).to.equal(expected);
        });
      });
    });
    it('throws if unknown', () => {
      // arrange
      const unknownValue: ScriptingLanguage = 666 as never;
      const errorMessage = `unsupported language: ${unknownValue}`;
      // act
      const act = () => new ScriptingDefinitionBuilder()
        .withLanguage(unknownValue)
        .build();
      // assert
      expect(act).to.throw(errorMessage);
    });
  });
  describe('fileExtension', () => {
    describe('returns expected for each language', () => {
      // arrange
      const testCases = new Map<ScriptingLanguage, string>([
        [ScriptingLanguage.batchfile, 'bat'],
        [ScriptingLanguage.shellscript, 'sh'],
      ]);
      for (const test of testCases.entries()) {
        const language = test[0];
        const expectedExtension = test[1];
        it(`${ScriptingLanguage[language]} has ${expectedExtension}`, () => {
          // act
          const sut = new ScriptingDefinitionBuilder()
            .withLanguage(language)
            .build();
          // assert
          expect(sut.fileExtension, expectedExtension);
        });
      }
    });
  });
  describe('startCode', () => {
    it('sets as expected', () => {
      // arrange
      const expected = 'REM start-code';
      // act
      const sut = new ScriptingDefinitionBuilder()
        .withStartCode(expected)
        .build();
      // assert
      expect(sut.startCode).to.equal(expected);
    });
    describe('throws when absent', () => {
      itEachAbsentStringValue((absentValue) => {
        // arrange
        const expectedError = 'missing start code';
        // act
        const act = () => new ScriptingDefinitionBuilder()
          .withStartCode(absentValue)
          .build();
        // assert
        expect(act).to.throw(expectedError);
      }, { excludeNull: true, excludeUndefined: true });
    });
  });
  describe('endCode', () => {
    it('sets as expected', () => {
      // arrange
      const expected = 'REM end-code';
      // act
      const sut = new ScriptingDefinitionBuilder()
        .withEndCode(expected)
        .build();
      // assert
      expect(sut.endCode).to.equal(expected);
    });
    describe('throws when absent', () => {
      itEachAbsentStringValue((absentValue) => {
        // arrange
        const expectedError = 'missing end code';
        // act
        const act = () => new ScriptingDefinitionBuilder()
          .withEndCode(absentValue)
          .build();
        // assert
        expect(act).to.throw(expectedError);
      }, { excludeNull: true, excludeUndefined: true });
    });
  });
});

class ScriptingDefinitionBuilder {
  private language = ScriptingLanguage.shellscript;

  private startCode = `# [${ScriptingDefinitionBuilder.name}]: start-code`;

  private endCode = `# [${ScriptingDefinitionBuilder.name}]: end-code`;

  public withLanguage(language: ScriptingLanguage): this {
    this.language = language;
    return this;
  }

  public withStartCode(startCode: string): this {
    this.startCode = startCode;
    return this;
  }

  public withEndCode(endCode: string): this {
    this.endCode = endCode;
    return this;
  }

  public build(): ScriptingDefinition {
    return new ScriptingDefinition(this.language, this.startCode, this.endCode);
  }
}
