#undef UFBXT_TEST_GROUP
#define UFBXT_TEST_GROUP "deflate"

#if UFBXT_IMPL

static uint32_t ufbxt_fnv1a(const void *data, size_t size)
{
	const char *src = (const char*)data;
	uint32_t hash = 0x811c9dc5u;
	for (size_t i = 0; i < size; i++) {
		hash = (hash ^ (uint32_t)(uint8_t)src[i]) * 0x01000193u;
	}
	return hash;
}

static ptrdiff_t ufbxt_inflate_no_fuzz(void *dst, size_t dst_size, const void *src, size_t src_size, const ufbxt_inflate_opts *opts)
{
	ufbx_inflate_retain retain;
	retain.initialized = false;

	ufbx_inflate_input input = { 0 };
	input.data = src;
	input.data_size = src_size;
	input.total_size = src_size;
	input.internal_fast_bits = opts->fast_bits;

	if (opts->force_fast) {
		size_t padding = 512;
		char *large_src = (char*)malloc(src_size + padding);
		char *large_dst = (char*)malloc(dst_size + padding);
		ufbxt_assert(large_src);
		ufbxt_assert(large_dst);

		memcpy(large_src, src, src_size);
		memset(large_src + src_size, 0, padding);

		input.data = large_src;
		input.data_size += padding;
		input.total_size += padding;
		input.internal_fast_bits = 10;
		ptrdiff_t result = ufbx_inflate(large_dst, dst_size + padding, &input, &retain);

		if (result >= 0) {
			memcpy(dst, large_dst, (size_t)result);
		}

		free(large_src);
		free(large_dst);

		return result;
	} else {
		return ufbx_inflate(dst, dst_size, &input, &retain);
	}
}

static ptrdiff_t ufbxt_inflate(void *dst, size_t dst_size, const void *src, size_t src_size, const ufbxt_inflate_opts *opts)
{
	if (opts->primary && ufbxt_begin_fuzz()) {
		ufbx_inflate_input input = { 0 };
		input.data = src;
		input.data_size = src_size;
		input.total_size = src_size;

		int i;

		uint8_t *data_copy[256] = { 0 };

		#pragma omp parallel for schedule(static, 16)
		for (i = 0; i < (int)src_size; i++) {
			ufbx_inflate_retain retain;
			retain.initialized = false;

			if (omp_get_thread_num() == 0) {
				if (i % 16 == 0) {
					fprintf(stderr, "\rFuzzing %d/%d", i, (int)src_size);
					fflush(stderr);
				}
			}

			uint8_t **p_data_copy = &data_copy[omp_get_thread_num()];
			if (*p_data_copy == NULL) {
				*p_data_copy = malloc(src_size);
				ufbxt_assert(*p_data_copy);
				memcpy(*p_data_copy, src, src_size);
			}
			uint8_t *data_u8 = *p_data_copy;

			size_t step = i * 10;

			uint8_t original = data_u8[i];

			if (src_size < 256) {
				// Small input: Try all possible byte values

				for (uint32_t byte = 0; byte < 0x100; byte++) {
					data_u8[i] = (uint8_t)byte;
					ufbx_inflate(dst, dst_size, &input, &retain);
				}

			} else {
				// Large input: Try +1, -1, 0, 0xff

				data_u8[i] = original + 1;
				ufbx_inflate(dst, dst_size, &input, &retain);

				data_u8[i] = original - 1;
				ufbx_inflate(dst, dst_size, &input, &retain);

				if (original != 0) {
					data_u8[i] = 0;
					ufbx_inflate(dst, dst_size, &input, &retain);
				}

				if (original != 0xff) {
					data_u8[i] = 0xff;
					ufbx_inflate(dst, dst_size, &input, &retain);
				}
			}

			data_u8[i] = original;
		}

		for (size_t i = 0; i < ufbxt_arraycount(data_copy); i++) {
			free(data_copy[i]);
		}

		fprintf(stderr, "\rFuzzing %d/%d\n", (int)src_size, (int)src_size);
	}

	return ufbxt_inflate_no_fuzz(dst, dst_size, src, src_size, opts);
}

#endif

UFBXT_DEFLATE_TEST(deflate_empty)
#if UFBXT_IMPL
{
	char src[1], dst[1];
	ptrdiff_t res = ufbxt_inflate(dst, 1, src, 0, opts);
	ufbxt_hintf("res = %d", (int)res);
	ufbxt_assert(res != 0);
}
#endif

UFBXT_DEFLATE_TEST(deflate_simple)
#if UFBXT_IMPL
{
	char src[] = "\x78\x9c\x01\x06\x00\xf9\xffHello!\x07\xa2\x02\x16";
	char dst[6];
	ptrdiff_t res = ufbxt_inflate(dst, sizeof(dst), src, sizeof(src) - 1, opts);
	ufbxt_hintf("res = %d", (int)res);
	ufbxt_assert(res == 6);
	ufbxt_assert(!memcmp(dst, "Hello!", 6));
}
#endif

UFBXT_DEFLATE_TEST(deflate_simple_chunks)
#if UFBXT_IMPL
{
	char src[] = "\x78\x9c\x00\x06\x00\xf9\xffHello \x01\x06\x00\xf9\xffworld!\x1d\x09\x04\x5e";
	char dst[12];
	ptrdiff_t res = ufbxt_inflate(dst, sizeof(dst), src, sizeof(src) - 1, opts);
	ufbxt_hintf("res = %d", (int)res);
	ufbxt_assert(res == 12);
	ufbxt_assert(!memcmp(dst, "Hello world!", 12));
}
#endif

UFBXT_DEFLATE_TEST(deflate_static)
#if UFBXT_IMPL
{
	char src[] = "x\xda\xf3H\xcd\xc9\xc9W(\xcf/\xcaIQ\x04\x00\x1d\t\x04^";
	char dst[12];
	ptrdiff_t res = ufbxt_inflate(dst, sizeof(dst), src, sizeof(src) - 1, opts);
	ufbxt_hintf("res = %d", (int)res);
	ufbxt_assert(res == 12);
	ufbxt_assert(!memcmp(dst, "Hello world!", 12));
}
#endif

UFBXT_DEFLATE_TEST(deflate_static_match)
#if UFBXT_IMPL
{
	char src[] = "x\xda\xf3H\xcd\xc9\xc9W\xf0\x00\x91\x8a\x00\x1b\xbb\x04*";
	char dst[12];
	ptrdiff_t res = ufbxt_inflate(dst, sizeof(dst), src, sizeof(src) - 1, opts);
	ufbxt_hintf("res = %d", (int)res);
	ufbxt_assert(res == 12);
	ufbxt_assert(!memcmp(dst, "Hello Hello!", 12));
}
#endif

UFBXT_DEFLATE_TEST(deflate_static_rle)
#if UFBXT_IMPL
{
	char src[] = "x\xdastD\x00\x00\x13\xda\x03\r";
	char dst[12];
	ptrdiff_t res = ufbxt_inflate(dst, sizeof(dst), src, sizeof(src) - 1, opts);
	ufbxt_hintf("res = %d", (int)res);
	ufbxt_assert(res == 12);
	ufbxt_assert(!memcmp(dst, "AAAAAAAAAAAA", 12));
}
#endif

UFBXT_DEFLATE_TEST(deflate_dynamic)
#if UFBXT_IMPL
{
	char src[] = "\x78\x9c\x1d\xc4\x31\x0d\x00\x00\x0c\x02\x41\x2b\xad"
		"\x1b\x8c\xb0\x7d\x82\xff\x8d\x84\xe5\x64\xc8\xcd\x2f\x1b\xbb\x04\x2a";
	char dst[64];
	ptrdiff_t res = ufbxt_inflate(dst, sizeof(dst), src, sizeof(src) - 1, opts);
	ufbxt_hintf("res = %d", (int)res);
	ufbxt_assert(res == 12);
	ufbxt_assert(!memcmp(dst, "Hello Hello!", 12));
}
#endif

UFBXT_DEFLATE_TEST(deflate_dynamic_no_match)
#if UFBXT_IMPL
{
	char src[] =
		"\x78\x9c\x05\x80\x41\x09\x00\x00\x08\x03\xab\x68\x1b\x1b\x58\x40\x7f\x07\x83\xf5"
		"\x7f\x8c\x79\x50\xad\xcc\x75\x00\x1c\x49\x04\x3e";
	char dst[64];
	ptrdiff_t res = ufbxt_inflate(dst, sizeof(dst), src, sizeof(src) - 1, opts);
	ufbxt_hintf("res = %d", (int)res);
	ufbxt_assert(res == 12);
	ufbxt_assert(!memcmp(dst, "Hello World!", 12));
}
#endif

UFBXT_DEFLATE_TEST(deflate_dynamic_empty)
#if UFBXT_IMPL
{
	char src[] =
		"\x78\x9c\x05\xc0\x81\x08\x00\x00\x00\x00\x20\x7f\xeb\x03\x00\x00\x00\x01";
	char dst[64];
	ptrdiff_t res = ufbxt_inflate(dst, sizeof(dst), src, sizeof(src) - 1, opts);
	ufbxt_hintf("res = %d", (int)res);
	ufbxt_assert(res == 0);
}
#endif

UFBXT_DEFLATE_TEST(deflate_dynamic_rle)
#if UFBXT_IMPL
{
	char src[] =
		"\x78\x9c\x5d\xc0\xb1\x00\x00\x00\x00\x80\x30\xb6\xfc\xa5\xfa\xb7\x34\x26\xea\x04\x52";
	char dst[17];
	ptrdiff_t res = ufbxt_inflate(dst, sizeof(dst), src, sizeof(src) - 1, opts);
	ufbxt_hintf("res = %d", (int)res);
	ufbxt_assert(res == 17);
	ufbxt_assert(!memcmp(dst, "AAAAAAAAAAAAAAAAA", 17));
}
#endif

UFBXT_DEFLATE_TEST(deflate_dynamic_rle_boundary)
#if UFBXT_IMPL
{
	char src[] =
		"\x78\x9c\x75\xc0\x81\x00\x00\x00\x00\x80\x20\xb6\xfd\xa5\x26\xa8\x64\x52\x9e\x06\x5a";
	char dst[25];
	ptrdiff_t res = ufbxt_inflate(dst, sizeof(dst), src, sizeof(src) - 1, opts);
	ufbxt_hintf("res = %d", (int)res);
	ufbxt_assert(res == 25);
	ufbxt_assert(!memcmp(dst, "AAAAAAAAAAAAAAAAAAAAAAAAA", 25));
}
#endif

UFBXT_DEFLATE_TEST(deflate_repeat_length)
#if UFBXT_IMPL
{
	char src[] = "\x78\x9c\x05\x00\x05\x0d\x00\x20\x2c\x1b\xee\x0e\xb7"
		"\xfe\x41\x98\xd2\xc6\x3a\x1f\x62\xca\xa5\xb6\x3e\xe6\xda\xe7\x3e\x40"
		"\x62\x11\x26\x84\x77\xcf\x5e\x73\xf4\x56\x4b\x4e\x31\x78\x67\x8d\x56\x1f\xa1\x6e\x0f\xbf";
	char dst[64];
	ptrdiff_t res = ufbxt_inflate(dst, sizeof(dst), src, sizeof(src) - 1, opts);
	ufbxt_hintf("res = %d", (int)res);
	ufbxt_assert(res == 52);
	ufbxt_assert(!memcmp(dst, "ABCDEFGHIJKLMNOPQRSTUVWXYZZYXWVUTSRQPONMLKJIHGFEDCBA", 52));
}
#endif

UFBXT_DEFLATE_TEST(deflate_huff_lengths)
#if UFBXT_IMPL
{
	char src[] = "\x78\x9c\x05\xe0\xc1\x95\x65\x59\x96\x65\xd9\xb1\x84"
		"\xca\x70\x53\xf9\xaf\x79\xcf\x5e\x93\x7f\x96\x30\xfe\x7f\xff\xdf\xff"
		"\xfb\xbf\xff\xfd\xf7\xef\xef\xf7\xbd\x5b\xfe\xff\x19\x28\x03\x5d";
	char dst[64];
	ptrdiff_t res = ufbxt_inflate(dst, sizeof(dst), src, sizeof(src) - 1, opts);
	ufbxt_hintf("res = %d", (int)res);
	ufbxt_assert(res == 15);
	ufbxt_assert(!memcmp(dst, "0123456789ABCDE", 15));
}
#endif

UFBXT_DEFLATE_TEST(deflate_multi_part_matches)
#if UFBXT_IMPL
{
	char src[] = "\x78\x9c\x00\x04\x00\xfb\xff\x54\x65\x73\x74\x52\x08"
		"\x48\x2c\x02\x10\x00\x06\x32\x00\x00\x00\x0c\x52\x39\xcc\x45\x72\xc8"
		"\x7f\xcd\x9d\x00\x08\x00\xf7\xff\x74\x61\x20\x44\x61\x74\x61\x20\x02"
		"\x8b\x01\x38\x8c\x43\x12\x00\x00\x00\x00\x40\xff\x5f\x0b\x36\x8b\xc0"
		"\x12\x80\xf9\xa5\x96\x23\x84\x00\x8e\x36\x10\x41";
	char dst[64];
	ptrdiff_t res = ufbxt_inflate(dst, sizeof(dst), src, sizeof(src) - 1, opts);
	ufbxt_hintf("res = %d", (int)res);
	ufbxt_assert(res == 48);
	ufbxt_assert(!memcmp(dst, "Test Part Data Data Test Data Part New Test Data", 48));
}
#endif

UFBXT_DEFLATE_TEST(deflate_uncompressed_bounds)
#if UFBXT_IMPL
{
	char src[] = "\x78\x9c\x01\x06\x00\xf9\xffHello!";

	char dst[64];
	ptrdiff_t res = ufbxt_inflate(dst, sizeof(dst), src, sizeof(src) - 1, opts);
	ufbxt_hintf("res = %d", (int)res);
	ufbxt_assert(res == -9);
}
#endif

UFBXT_DEFLATE_TEST(deflate_fail_cfm)
#if UFBXT_IMPL
{
	char src[] = "\x79\x9c";
	char dst[4];
	ptrdiff_t res = ufbxt_inflate(dst, sizeof(dst), src, sizeof(src) - 1, opts);
	ufbxt_hintf("res = %d", (int)res);
	ufbxt_assert(res == -1);
}
#endif

UFBXT_DEFLATE_TEST(deflate_fail_fdict)
#if UFBXT_IMPL
{
	char src[] = "\x78\xbc";
	char dst[4];
	ptrdiff_t res = ufbxt_inflate(dst, sizeof(dst), src, sizeof(src) - 1, opts);
	ufbxt_hintf("res = %d", (int)res);
	ufbxt_assert(res == -2);
}
#endif

UFBXT_DEFLATE_TEST(deflate_fail_fcheck)
#if UFBXT_IMPL
{
	char src[] = "\x78\0x9d";
	char dst[4];
	ptrdiff_t res = ufbxt_inflate(dst, sizeof(dst), src, sizeof(src) - 1, opts);
	ufbxt_hintf("res = %d", (int)res);
	ufbxt_assert(res == -3);
}
#endif

UFBXT_DEFLATE_TEST(deflate_fail_nlen)
#if UFBXT_IMPL
{
	char src[] = "\x78\x9c\x01\x06\x00\xf8\xffHello!\x07\xa2\x02\x16";
	char dst[64];
	ptrdiff_t res = ufbxt_inflate(dst, sizeof(dst), src, sizeof(src) - 1, opts);
	ufbxt_hintf("res = %d", (int)res);
	ufbxt_assert(res == -4);
}
#endif

UFBXT_DEFLATE_TEST(deflate_fail_dst_overflow)
#if UFBXT_IMPL
{
	if (opts->force_fast) return;

	char src[] = "\x78\x9c\x01\x06\x00\xf9\xffHello!\x07\xa2\x02\x16";
	char dst[5];
	ptrdiff_t res = ufbxt_inflate(dst, sizeof(dst), src, sizeof(src) - 1, opts);
	ufbxt_hintf("res = %d", (int)res);
	ufbxt_assert(res == -6);
}
#endif

UFBXT_DEFLATE_TEST(deflate_fail_src_overflow)
#if UFBXT_IMPL
{
	if (opts->force_fast) return;

	char src[] = "\x78\x9c\x01\x06\x00\xf9\xffHello";
	char dst[64];
	ptrdiff_t res = ufbxt_inflate(dst, sizeof(dst), src, sizeof(src) - 1, opts);
	ufbxt_hintf("res = %d", (int)res);
	ufbxt_assert(res < 0);
}
#endif

UFBXT_DEFLATE_TEST(deflate_fail_bad_block)
#if UFBXT_IMPL
{
	char src[] = "\x78\x9c\x07\x08\x00\xf8\xff";
	char dst[64];
	ptrdiff_t res = ufbxt_inflate(dst, sizeof(dst), src, sizeof(src) - 1, opts);
	ufbxt_hintf("res = %d", (int)res);
	ufbxt_assert(res == -7);
}
#endif

UFBXT_DEFLATE_TEST(deflate_fail_bad_truncated_checksum)
#if UFBXT_IMPL
{
	char src[] = "\x78\x9c\x01\x06\x00\xf9\xffHello!\x07\xa2\x02";
	char dst[64];
	ptrdiff_t res = ufbxt_inflate(dst, sizeof(dst), src, sizeof(src) - 1, opts);
	ufbxt_hintf("res = %d", (int)res);
	ufbxt_assert(res == -9);
}
#endif

UFBXT_DEFLATE_TEST(deflate_fail_bad_checksum)
#if UFBXT_IMPL
{
	char src[] = "\x78\x9c\x01\x06\x00\xf9\xffHello!\x07\xa2\x02\xff";
	char dst[64];
	ptrdiff_t res = ufbxt_inflate(dst, sizeof(dst), src, sizeof(src) - 1, opts);
	ufbxt_hintf("res = %d", (int)res);
	ufbxt_assert(res == -9);
}
#endif

UFBXT_DEFLATE_TEST(deflate_fail_codelen_16_overflow)
#if UFBXT_IMPL
{
	char src[] = "\x78\x9c\x05\x80\x85\x0c\x00\x00\x00\xc0\xfc\xa1\x5f\xc3\x06\x05\xf5\x02\xfb";
	char dst[64];
	ptrdiff_t res = ufbxt_inflate(dst, sizeof(dst), src, sizeof(src) - 1, opts);
	ufbxt_hintf("res = %d", (int)res);
	ufbxt_assert(res == -18);
}
#endif

UFBXT_DEFLATE_TEST(deflate_fail_codelen_17_overflow)
#if UFBXT_IMPL
{
	char src[] = "\x78\x9c\x05\xc0\xb1\x0c\x00\x00\x00\x00\x20\x7f\xe7\xae\x26\x00\xfd\x00\xfd";
	char dst[64];
	ptrdiff_t res = ufbxt_inflate(dst, sizeof(dst), src, sizeof(src) - 1, opts);
	ufbxt_hintf("res = %d", (int)res);
	ufbxt_assert(res == -19);
}
#endif

UFBXT_DEFLATE_TEST(deflate_fail_codelen_18_overflow)
#if UFBXT_IMPL
{
	char src[] = "\x78\x9c\x05\xc0\x81\x08\x00\x00\x00\x00\x20\x7f\xdf\x09\x4e\x00\xf5\x00\xf5";
	char dst[64];
	ptrdiff_t res = ufbxt_inflate(dst, sizeof(dst), src, sizeof(src) - 1, opts);
	ufbxt_hintf("res = %d", (int)res);
	ufbxt_assert(res == -20);
}
#endif

UFBXT_DEFLATE_TEST(deflate_fail_codelen_overfull)
#if UFBXT_IMPL
{
	char src[] = "\x78\x9c\x05\x80\x31\x11\x01\x00\x00\x01\xc3\xa9\xe2\x37\x47\xff\xcd\x69\x26\xf4\x0a\x7a\x02\xbb";
	char dst[64];
	ptrdiff_t res = ufbxt_inflate(dst, sizeof(dst), src, sizeof(src) - 1, opts);
	ufbxt_hintf("res = %d", (int)res);
	ufbxt_assert(res == -14);
}
#endif

UFBXT_DEFLATE_TEST(deflate_fail_codelen_underfull)
#if UFBXT_IMPL
{
	char src[] = "\x78\x9c\x05\x80\x31\x11\x00\x00\x00\x41\xc3\xa9\xe2\x37\x47\xff\xcd\x69\x26\xf4\x0a\x7a\x02\xbb";
	char dst[64];
	ptrdiff_t res = ufbxt_inflate(dst, sizeof(dst), src, sizeof(src) - 1, opts);
	ufbxt_hintf("res = %d", (int)res);
	ufbxt_assert(res == -15);
}
#endif

UFBXT_DEFLATE_TEST(deflate_fail_litlen_bad_huffman)
#if UFBXT_IMPL
{
	char src[] = "\x78\x9c\x05\x40\x81\x09\x00\x20\x08\x7b\xa5\x0f\x7a\xa4\x27\xa2"
		"\x46\x0a\xa2\xa0\xfb\x1f\x11\x23\xea\xf8\x16\xc4\xa7\xae\x9b\x0f\x3d\x4e\xe4\x07\x8d";
	char dst[64];
	ptrdiff_t res = ufbxt_inflate(dst, sizeof(dst), src, sizeof(src) - 1, opts);
	ufbxt_hintf("res = %d", (int)res);
	ufbxt_assert(res == -17);
}
#endif

UFBXT_DEFLATE_TEST(deflate_fail_distance_bad_huffman)
#if UFBXT_IMPL
{
	char src[] = "\x78\x9c\x1d\xc5\x31\x0d\x00\x00\x0c\x02\x41\x2b\x55\x80\x8a\x9a"
		"\x61\x06\xff\x21\xf9\xe5\xfe\x9d\x1e\x48\x3c\x31\xba\x05\x79";
	char dst[64];
	ptrdiff_t res = ufbxt_inflate(dst, sizeof(dst), src, sizeof(src) - 1, opts);
	ufbxt_hintf("res = %d", (int)res);
	ufbxt_assert(res == -23);
}
#endif

UFBXT_DEFLATE_TEST(deflate_fail_bad_distance)
#if UFBXT_IMPL
{
	{
		char src[] = "\x78\x9c\x73\xc9\x2c\x2e\x51\x00\x3d\x00\x0f\xd7\x03\x49";
		char dst[64];
		ptrdiff_t res = ufbxt_inflate(dst, sizeof(dst), src, sizeof(src) - 1, opts);
		ufbxt_hintf("res = %d", (int)res);
		ufbxt_assert(res == -11);
	}

	{
		char src[] = "\x78\x9c\x73\xc9\x2c\x2e\x51\x00\x7d\x00\x0f\xd7\x03\x49";
		char dst[64];
		ptrdiff_t res = ufbxt_inflate(dst, sizeof(dst), src, sizeof(src) - 1, opts);
		ufbxt_hintf("res = %d", (int)res);
		ufbxt_assert(res == -11);
	}
}
#endif

UFBXT_DEFLATE_TEST(deflate_fail_bad_litlen)
#if UFBXT_IMPL
{
	{
		char src[] = "\x78\x9c\x1b\x03\x00\x00\x42\x00\x42";
		char dst[64];
		ptrdiff_t res = ufbxt_inflate(dst, sizeof(dst), src, sizeof(src) - 1, opts);
		ufbxt_hintf("res = %d", (int)res);
		ufbxt_assert(res == -13);
	}

	{
		char src[] = "\x78\x9c\x1b\x07\x00\x00\x42\x00\x42";
		char dst[64];
		ptrdiff_t res = ufbxt_inflate(dst, sizeof(dst), src, sizeof(src) - 1, opts);
		ufbxt_hintf("res = %d", (int)res);
		ufbxt_assert(res == -13);
	}
}
#endif

UFBXT_DEFLATE_TEST(deflate_fail_distance_too_far)
#if UFBXT_IMPL
{
	char src[] = "\x78\x9c\x73\x04\x41\x00\x00\x00\x00\x01";
	char dst[64];
	ptrdiff_t res = ufbxt_inflate(dst, sizeof(dst), src, sizeof(src) - 1, opts);
	ufbxt_hintf("res = %d", (int)res);
	ufbxt_assert(res == -12);
}
#endif

UFBXT_DEFLATE_TEST(deflate_fail_literal_overflow)
#if UFBXT_IMPL
{
	if (opts->force_fast) return;

	char src[] = "x\xda\xf3H\xcd\xc9\xc9W(\xcf/\xcaIQ\x04\x00\x1d\t\x04^";
	char dst[8];
	ptrdiff_t res = ufbxt_inflate(dst, sizeof(dst), src, sizeof(src) - 1, opts);
	ufbxt_hintf("res = %d", (int)res);
	ufbxt_assert(res == -10);
}
#endif

UFBXT_DEFLATE_TEST(deflate_fail_match_overflow)
#if UFBXT_IMPL
{
	if (opts->force_fast) return;

	char src[] = "x\xda\xf3H\xcd\xc9\xc9W\xf0\x00\x91\x8a\x00\x1b\xbb\x04*";
	char dst[8];
	ptrdiff_t res = ufbxt_inflate(dst, sizeof(dst), src, sizeof(src) - 1, opts);
	ufbxt_hintf("res = %d", (int)res);
	ufbxt_assert(res == -12);
}
#endif

UFBXT_DEFLATE_TEST(deflate_fail_bad_distance_bit)
#if UFBXT_IMPL
{
	char src[] = "\x78\x9c\x0d\xc3\x41\x09\x00\x00\x00\xc2\xc0\x2a\x56\x13"
		"\x6c\x60\x7f\xd8\x1e\xd7\x2f\x06\x0a\x41\x02\x91";
	char dst[8];
	ptrdiff_t res = ufbxt_inflate(dst, sizeof(dst), src, sizeof(src) - 1, opts);
	ufbxt_hintf("res = %d", (int)res);
	ufbxt_assert(res == -11);
}
#endif

UFBXT_DEFLATE_TEST(deflate_fail_bad_distance_empty)
#if UFBXT_IMPL
{
	char src[] =
		"\x78\x9c\x0d\xc4\x41\x09\x00\x00\x00\xc2\xc0\x2a\x56\x13\x6c\x60\x7f\xd8\x1e\xd0"
		"\x2f\x02\x0a\x41\x02\x91";
	char dst[8];
	ptrdiff_t res = ufbxt_inflate(dst, sizeof(dst), src, sizeof(src) - 1, opts);
	ufbxt_hintf("res = %d", (int)res);
	ufbxt_assert(res == -11);
}
#endif

UFBXT_DEFLATE_TEST(deflate_fail_bad_lit_length)
#if UFBXT_IMPL
{
	char src[] =
		"\x78\x9c\x05\xc0\x81\x08\x00\x00\x00\x00\x20\x7f\xeb\x0b\x00\x00\x00\x01";
	char dst[8];
	ptrdiff_t res = ufbxt_inflate(dst, sizeof(dst), src, sizeof(src) - 1, opts);
	ufbxt_hintf("res = %d", (int)res);
	ufbxt_assert(res == -13);
}
#endif

UFBXT_DEFLATE_TEST(deflate_fail_no_litlen_codes)
#if UFBXT_IMPL
{
	char src[] =
		"\x78\x9c\x05\x00\x80\xe4\x3f\x1b\x00\x00\x00\x01";
	char dst[8];
	ptrdiff_t res = ufbxt_inflate(dst, sizeof(dst), src, sizeof(src) - 1, opts);
	ufbxt_hintf("res = %d", (int)res);
	ufbxt_assert(res == -13);
}
#endif

UFBXT_DEFLATE_TEST(deflate_fail_no_dist_codes)
#if UFBXT_IMPL
{
	char src[] =
		"\x78\x9c\x15\xc0\x81\x0c\x00\x00\x00\x80\x30\xb6\xfc\xa5\xda\x17"
		"\x03\x03\xd4\x01\x46";
	char dst[8];
	ptrdiff_t res = ufbxt_inflate(dst, sizeof(dst), src, sizeof(src) - 1, opts);
	ufbxt_hintf("res = %d", (int)res);
	ufbxt_assert(res == -11);
}
#endif

UFBXT_DEFLATE_TEST(deflate_bit_flip)
#if UFBXT_IMPL
{
	char src[] = "\x78\x9c\x00\x04\x00\xfb\xff\x54\x65\x73\x74\x52\x08"
		"\x48\x2c\x02\x10\x00\x06\x32\x00\x00\x00\x0c\x52\x39\xcc\x45\x72\xc8"
		"\x7f\xcd\x9d\x00\x08\x00\xf7\xff\x74\x61\x20\x44\x61\x74\x61\x20\x02"
		"\x8b\x01\x38\x8c\x43\x12\x00\x00\x00\x00\x40\xff\x5f\x0b\x36\x8b\xc0"
		"\x12\x80\xf9\xa5\x96\x23\x84\x00\x8e\x36\x10\x41";

	char dst[64];
	int num_res[64] = { 0 };

	for (size_t byte_ix = 0; byte_ix < sizeof(src) - 1; byte_ix++) {
		for (size_t bit_ix = 0; bit_ix < 8; bit_ix++) {
			size_t bit = (size_t)1 << bit_ix;

			ufbxt_hintf("byte_ix==%u && bit_ix==%u", (unsigned)byte_ix, (unsigned)bit_ix);

			src[byte_ix] ^= bit;
			ptrdiff_t res = ufbxt_inflate_no_fuzz(dst, sizeof(dst), src, sizeof(src) - 1, opts);
			src[byte_ix] ^= bit;

			res = -res;
			if (res < 0) res = 0;
			if (res > ufbxt_arraycount(num_res)) res = ufbxt_arraycount(num_res);
			num_res[res]++;
		}
	}

	char line[128], *ptr = line, *end = line + sizeof(line);
	for (size_t i = 0; i < ufbxt_arraycount(num_res); i++) {
		if (num_res[i] > 0) {
			ptr += snprintf(ptr, end - ptr, "%3d:%3d    ", -(int)i, num_res[i]);
			if (ptr - line > 70) {
				ufbxt_logf("%s", line);
				ptr = line;
			}
		}
	}
}
#endif

UFBXT_DEFLATE_TEST(deflate_static_distances_and_lengths)
#if UFBXT_IMPL
{
	char src[] =
		"\x78\x9c\x63\x60\x04\x02\x26\x66\x10\x62\x61\x05\x53\x6c\xec\x10\x36\x07\x27\x54"
		"\x80\x8b\x1b\x26\xc1\xc3\x0b\x57\xc0\xc7\x8f\x50\x27\x20\x88\xa4\x43\x48\x18\x59"
		"\x87\x88\x28\x8a\x51\x62\xe2\xa8\x46\x49\x48\xa2\x59\x20\x25\x8d\x6e\x9b\x8c\x2c"
		"\x86\x03\xe4\xe4\x31\x1d\xa0\xa0\x88\xc5\x7d\x4a\xca\xd8\xdc\xa7\xa2\x8a\xd5\x03"
		"\x6a\xea\xd8\x3d\xa0\xa1\x89\xc3\x97\x5a\xda\xb8\x02\x40\x47\x17\x67\x00\xe8\xe9"
		"\xe3\x0e\x00\x03\x43\x3c\x21\x69\x64\x8c\x2f\x24\x4d\x4c\xf1\xc6\x87\x99\x39\xfe"
		"\xf8\xb0\xb0\x24\x10\x1f\x56\xd6\x84\xe2\xc3\xc6\x96\x60\x02\xb0\xb3\x27\x9c\x00"
		"\x1c\x1c\x89\x48\x00\x4e\xce\x44\xa6\x2b\x17\x57\xe2\xd3\x95\x9b\x3b\x09\xe9\xca"
		"\xc3\x93\xd4\x74\xe5\xe5\x4d\x7a\x06\xf0\xf1\x25\x23\x03\xf8\xf9\x93\x99\x07\x03"
		"\x02\xc9\xcf\x83\x41\xc1\x14\xe4\xc1\x90\x50\x0a\x0b\x80\xb0\x70\xca\x0b\x80\x88"
		"\x48\x2a\x14\x00\x51\xd1\xd4\x29\xe1\x62\x62\xa9\x5f\xc2\xc5\xc5\xd3\xa0\x84\x4b"
		"\x48\xa4\x5d\x09\x97\x94\x4c\xfb\x0a\x20\x25\x95\x0e\x15\x40\x5a\x3a\xdd\x2a\x80"
		"\x8c\x4c\xfa\xd7\x83\x59\xd9\x03\x50\x0f\xe6\xe4\x0e\x54\x03\x20\x2f\x7f\xe0\x1b"
		"\x00\x05\x85\x83\xa0\x01\x50\x54\x3c\x98\xda\x70\x25\xa5\x83\xb3\x0d\x57\x56\x3e"
		"\xf8\xdb\x70\x15\x95\x43\xa0\x0d\x57\x55\x3d\x74\x3a\x00\x35\xb5\x43\xb1\x03\x50"
		"\x57\x3f\xf4\x3b\x00\x0d\x8d\xc3\xa0\xaf\xd6\xd4\x3c\x5c\xfa\x6a\x2d\xad\xc3\xaf"
		"\xaf\xd6\xd6\x3e\xfc\x07\x00\x3a\x3a\x47\xc0\x00\x40\x57\xf7\xc8\x18\x00\xe8\xe9"
		"\x1d\x69\x03\x00\x7d\xfd\x23\x77\x1c\x6e\xc2\xc4\x11\x3f\x0e\x37\x69\xf2\xe8\x38"
		"\xdc\x94\xa9\xa3\xe3\x70\xd3\xa6\x8f\x4e\x00\xcc\x98\x39\x3a\x01\x30\x6b\xf6\xe8"
		"\x04\xc0\x9c\xb9\xa3\x13\x00\xf3\xe6\x8f\xce\xab\x2d\x58\x38\x3a\xaf\xb6\x68\xf1"
		"\xe8\xbc\xda\x92\xa5\xa3\xf3\x6a\xcb\x96\x8f\x2e\x00\x58\xb1\x72\x74\x01\xc0\xaa"
		"\xd5\xa3\x0b\x00\xd6\xac\x1d\x5d\x00\xb0\x6e\xfd\xe8\x02\x80\x0d\x1b\x47\x17\x00"
		"\x6c\xda\x3c\xba\x00\x60\xcb\xd6\xd1\xf5\x70\xdb\xb6\x8f\xae\x87\xdb\xb1\x73\x74"
		"\x3d\xdc\xae\xdd\xa3\xeb\xe1\xf6\xec\x1d\x5d\x0f\xb7\x6f\xff\xe8\x7a\xb8\x03\x07"
		"\x47\xd7\xc3\x1d\x3a\x3c\xba\x1e\xee\xc8\xd1\xd1\xf5\x70\xc7\x8e\x8f\x6e\x00\x38"
		"\x71\x72\x74\x03\xc0\xa9\xd3\xa3\x1b\x00\xce\x9c\x1d\xdd\x00\x70\xee\xfc\xe8\x06"
		"\x80\x0b\x17\x47\x37\x00\x5c\xba\x3c\xba\x01\xe0\xca\xd5\xd1\xfd\x6a\xd7\xae\x8f"
		"\xee\x57\xbb\x71\x73\x74\xbf\xda\xad\xdb\xa3\xfb\xd5\xee\xdc\x1d\xdd\xaf\x76\xef"
		"\xfe\xe8\x7e\xb5\x07\x0f\x47\xf7\xab\x3d\x7a\x3c\xba\x5f\xed\xc9\xd3\xd1\xfd\x6a"
		"\xcf\x9e\x8f\x1e\x00\xf0\xe2\xe5\xe8\x01\x00\xaf\x5e\x8f\x1e\x00\xf0\xe6\xed\xe8"
		"\x01\x00\xef\xde\x8f\x1e\x00\xf0\xe1\xe3\xe8\x01\x00\x9f\x3e\x8f\x1e\x00\xf0\xe5"
		"\xeb\xe8\x01\x00\xdf\xbe\x8f\x1e\x00\xf0\xe3\xe7\xe8\x01\x00\xbf\x7e\x8f\x1e\x00"
		"\xf0\xe7\xef\xe8\x01\x00\xff\xfe\x8f\x1e\x00\xc0\xc0\x38\x7a\x00\x00\x13\xf3\xe8"
		"\xf9\x70\x2c\xac\xa3\xe7\xc3\xb1\xb1\x8f\x9e\x0f\xc7\xc1\x39\x7a\x3e\x1c\x17\xf7"
		"\xe8\xf9\x70\x3c\xbc\xa3\xe7\xc3\xf1\xf1\x8f\x9e\x0f\x27\x20\x38\x7a\x3e\x9c\x90"
		"\xf0\xe8\xf9\x70\x22\xa2\xa3\xe7\xc3\x89\x89\x8f\x9e\x0f\x27\x21\x39\x7a\x3e\x9c"
		"\x94\xf4\xe8\xf9\x70\x32\xb2\xa3\xe7\xc3\xc9\xc9\x8f\x9e\x0f\xa7\xa0\x38\x7a\x3e"
		"\x9c\x92\xf2\xe8\xf9\x70\x2a\xaa\xa3\xe7\xc3\xa9\xa9\x8f\x5e\x00\xa0\xa1\x39\x7a"
		"\x4f\x83\x96\xf6\xe8\x3d\x0d\x3a\xba\xa3\xf7\x34\xe8\xe9\x8f\xde\xd3\x60\x60\x38"
		"\x7a\x1f\x8b\x91\xf1\xe8\x7d\x2c\x26\xa6\xa3\xf7\xb1\x98\x99\x8f\xde\xc7\x62\x61"
		"\x39\x7a\xef\x92\x95\xf5\xe8\xbd\x4b\x36\xb6\xa3\xf7\x2e\xd9\xd9\x8f\xde\xbb\xe4"
		"\xe0\x38\x7a\xbf\x9a\x93\xf3\xe8\xfd\x6a\x2e\xae\xa3\xf7\xab\xb9\xb9\x8f\xde\xaf"
		"\xe6\xe1\x39\x7a\x8f\xa2\x97\xf7\xe8\x3d\x8a\x3e\xbe\xa3\xf7\x28\xfa\xf9\x8f\xde"
		"\xa3\x18\x10\x38\x7a\x5f\x6a\x50\xf0\xe8\x7d\xa9\x21\xa1\xa3\xf7\xa5\x86\x85\x8f"
		"\xde\x97\x1a\x11\x39\x7a\x2f\x72\x54\xf4\xe8\xbd\xc8\x31\xb1\xa3\xf7\x22\xc7\xc5"
		"\x8f\xde\x8b\x9c\x90\x38\x7a\x2f\x72\x52\xf2\xe8\xbd\xc8\x29\xa9\xa3\xf7\xff\xa7"
		"\xa5\x8f\xde\xff\x0f\x00\x5e\x3b\xcf\x7c";

	size_t dst_size = 33665;
	char *dst = malloc(dst_size);
	ptrdiff_t res = ufbxt_inflate(dst, dst_size, src, sizeof(src) - 1, opts);
	ufbxt_hintf("res = %d", (int)res);
	ufbxt_assert(res == dst_size);
	ufbxt_assert(ufbxt_fnv1a(dst, dst_size) == 0x88398917);
	free(dst);
}
#endif

UFBXT_DEFLATE_TEST(deflate_dynamic_distances_and_lengths)
#if UFBXT_IMPL
{
	char src[] =
		"\x78\x9c\xed\x9d\x03\x70\x34\x41\x14\x84\x63\xdb\xb6\x6d\xdb\xb6\xed\xfc\xb6\x6d"
		"\xdb\xb6\x6d\xdb\xb6\x6d\xdb\x36\x82\xb3\xb1\x9d\x4a\x55\x52\xc9\xdd\xed\xee\x78"
		"\xe6\xbd\xee\x4f\x44\xf4\xe7\x97\x98\xf8\xaf\x6f\x09\xc9\xdf\x3f\xa4\xa4\xff\xfc"
		"\x2e\x23\xfb\xf7\x0f\x72\xf2\xff\xfe\xa1\xa0\xf8\xff\x05\x4a\xca\x35\xaf\x53\x51"
		"\xad\xf5\x0e\x35\xf5\xda\xef\xd0\xd0\xac\xf3\x51\x5a\xda\x75\x3f\x4a\x47\xb7\xde"
		"\x05\xf4\xf4\xeb\x5f\xcd\xc0\x90\xe4\x06\x8c\x8c\x49\x6f\xc0\xc4\x94\xcc\xfd\x99"
		"\x99\x93\xbb\x3f\x0b\x4b\xb2\x0f\x60\x65\x4d\xfe\x01\x6c\x6c\x29\x3c\xa5\x9d\x3d"
		"\xa5\x02\x70\x70\xa4\x58\x00\x4e\xce\x94\x0b\xc0\xc5\x95\x4a\x49\xba\xb9\x53\x2b"
		"\x49\x0f\x4f\xaa\xf5\xe1\xe5\x4d\xbd\x3e\x7c\x7c\x69\xd4\x87\x9f\x3f\xad\xfa\x08"
		"\x08\xa4\xd9\x00\x82\x82\x69\x37\x80\x90\x50\x3a\x1a\x40\x58\x38\x9d\xed\x2a\x22"
		"\x92\xfe\x76\x15\x15\xcd\x40\xbb\x8a\x89\x65\xb4\x5d\xc5\xc5\x33\xde\x01\x12\x12"
		"\x99\xe8\x00\x49\xc9\x4c\xf6\xc1\x94\x54\xe6\xfb\x60\x5a\x3a\x0b\x7d\x30\x23\x93"
		"\xc5\x01\x20\x2b\x9b\xf5\x01\x20\x27\x97\x0d\x03\x40\x5e\x3e\x7b\x46\xb8\x82\x42"
		"\xf6\x8f\x70\x45\xc5\x1c\x18\xe1\x4a\x4a\x39\x37\xc2\x95\x95\x73\x7e\x02\xa8\xa8"
		"\xe4\xc2\x04\x50\x55\xcd\xb5\x09\xa0\x41\x43\xee\xcf\x83\x8d\x1a\xf3\x60\x1e\x6c"
		"\xd2\x94\x57\x0b\x80\x66\xcd\x79\xbf\x00\x68\xd1\x92\x0f\x16\x00\xad\x5a\xf3\xd3"
		"\x1a\xae\x4d\x5b\xfe\x5c\xc3\xb5\x6b\xcf\xff\x6b\xb8\x0e\x1d\x05\x60\x0d\xd7\xa9"
		"\xb3\xe0\x6c\x00\xba\x74\x15\xc4\x0d\x40\xb7\xee\x82\xbf\x01\xe8\xd1\x53\x08\xf6"
		"\x6a\xbd\x7a\x0b\xcb\x5e\xad\x4f\x5f\xe1\xdb\xab\xf5\xeb\x2f\xfc\x07\x00\x03\x06"
		"\x12\xe0\x00\x60\xd0\x60\x62\x1c\x00\x0c\x19\x4a\xb4\x03\x80\x61\xc3\x89\x7b\x0e"
		"\x37\x62\x24\xe1\xcf\xe1\x46\x8d\xc6\x39\xdc\x98\xb1\x38\x87\x1b\x37\x1e\x01\x80"
		"\x09\x13\x11\x00\x98\x34\x19\x01\x80\x29\x53\x11\x00\x98\x36\x1d\x71\xb5\x19\x33"
		"\x11\x57\x9b\x35\x1b\x71\xb5\x39\x73\x11\x57\x9b\x37\x1f\x09\x00\x0b\x16\x22\x01"
		"\x60\xd1\x62\x24\x00\x2c\x59\x8a\x04\x80\x65\xcb\x91\x00\xb0\x62\x25\x12\x00\x56"
		"\xad\x46\x02\xc0\x9a\xb5\xc8\x87\x5b\xb7\x1e\xf9\x70\x1b\x36\x22\x1f\x6e\xd3\x66"
		"\xe4\xc3\x6d\xd9\x8a\x7c\xb8\x6d\xdb\x91\x0f\xb7\x63\x27\xf2\xe1\x76\xed\x46\x3e"
		"\xdc\x9e\xbd\xc8\x87\xdb\xb7\x1f\x02\x80\x03\x07\x21\x00\x38\x74\x18\x02\x80\x23"
		"\x47\x21\x00\x38\x76\x1c\x02\x80\x13\x27\x21\x00\x38\x75\x1a\x02\x80\x33\x67\xa1"
		"\x57\x3b\x77\x1e\x7a\xb5\x0b\x17\xa1\x57\xbb\x74\x19\x7a\xb5\x2b\x57\xa1\x57\xbb"
		"\x76\x1d\x7a\xb5\x1b\x37\xa1\x57\xbb\x75\x1b\x7a\xb5\x3b\x77\xa1\x57\xbb\x77\x1f"
		"\x06\x00\x0f\x1e\xc2\x00\xe0\xd1\x63\x18\x00\x3c\x79\x0a\x03\x80\x67\xcf\x61\x00"
		"\xf0\xe2\x25\x0c\x00\x5e\xbd\x86\x01\xc0\x9b\xb7\x30\x00\x78\xf7\x1e\x06\x00\x1f"
		"\x3e\xc2\x00\xe0\xd3\x67\x18\x00\x7c\xf9\x0a\x03\x80\x6f\xdf\x61\x00\x20\x22\x0a"
		"\x03\x00\x31\x71\xf8\xc3\x49\x48\xc2\x1f\x4e\x4a\x1a\xfe\x70\x32\xb2\xf0\x87\x93"
		"\x93\x87\x3f\x9c\x82\x22\xfc\xe1\x94\x94\xe1\x0f\xa7\xa2\x0a\x7f\x38\x35\x75\xf8"
		"\xc3\x69\x68\xc2\x1f\x4e\x4b\x1b\xfe\x70\x3a\xba\xf0\x87\xd3\xd3\x87\x3f\x9c\x81"
		"\x21\xfc\xe1\x8c\x8c\xe1\x0f\x67\x62\x0a\x7f\x38\x33\x73\xf8\xc3\x59\x58\xc2\x1f"
		"\xce\xca\x1a\x00\x00\x1b\x5b\x70\x1a\xec\xec\xc1\x69\x70\x70\x04\xa7\xc1\xc9\x19"
		"\x9c\x06\x17\x57\xf0\x58\xdc\xdc\xc1\x63\xf1\xf0\x04\x8f\xc5\xcb\x1b\x3c\x16\x1f"
		"\x5f\x70\x97\xfc\xfc\xc1\x5d\x0a\x08\x04\x77\x29\x28\x18\xdc\xa5\x90\x50\xf0\xd5"
		"\xc2\xc2\xc1\x57\x8b\x88\x04\x5f\x2d\x2a\x1a\x7c\xb5\x98\x58\x70\x14\xe3\xe2\xc1"
		"\x51\x4c\x48\x04\x47\x31\x29\x19\x1c\xc5\x94\x54\xf0\x52\xd3\xd2\xc1\x4b\xcd\xc8"
		"\x04\x2f\x35\x2b\x1b\xbc\xd4\x9c\x5c\x70\x91\xf3\xf2\xc1\x45\x2e\x28\x04\x17\xb9"
		"\xa8\x18\x5c\xe4\x92\x52\x70\x91\xcb\xca\xc1\x45\xae\xa8\x04\xff\xbf\xaa\x1a\xfc"
		"\xff\x1f\x5e\x3b\xcf\x7c";

	size_t dst_size = 33665;
	char *dst = malloc(dst_size);
	ptrdiff_t res = ufbxt_inflate(dst, dst_size, src, sizeof(src) - 1, opts);
	ufbxt_hintf("res = %d", (int)res);
	ufbxt_assert(res == dst_size);
	ufbxt_assert(ufbxt_fnv1a(dst, dst_size) == 0x88398917);
	free(dst);
}
#endif

UFBXT_DEFLATE_TEST(deflate_long_codes)
#if UFBXT_IMPL
{
	char src[] =
		"\x78\x9c\xed\xfd\xc7\xb9\x65\x5d\xb6\x65\xd9\xc9\x06\x40\x85\xae\xc2\x90\x60\x2e"
		"\xfd\x3f\x14\xf6\xb9\xe6\xfe\x32\xc1\x39\x69\x85\x56\xeb\x63\xae\x7d\xae\xfd\x1e"
		"\x01\x8e\xb7\x7b\x00\x00\x00\x00\x00\x00\x00\x00\x00\xc0\xff\x27\xfd\x7f\x1e\xee"
		"\xff\xa3\xfe\x3f\x0f\xf7\xbf\xf9\xff\xac\xff\xcf\xc3\xfd\x6f\xfe\xb7\xff\x1f\xf6"
		"\xff\x79\xb8\xff\xcd\xff\xf6\x7f\xf7\xff\x69\xff\x9f\x87\x03\x00\x00\xe0\xff\x1b"
		"\xff\xbf\xaf\xf6\xff\x95\xff\xdf\x57\xfb\xdf\xfc\x7f\xe7\xff\xf7\xd5\xfe\x37\xff"
		"\xdb\xff\x2f\xfd\xff\xbe\xda\xff\xe6\x7f\xfb\xbf\xfb\xff\xd6\xff\xef\xab\x01\x00"
		"\x00\x00\x00\xfc\xff\xde\xff\xef\xc3\xfd\xff\xe0\xff\xef\xc3\xfd\x6f\xfe\x7f\xf1"
		"\xff\xf7\xe1\xfe\x37\xff\xdb\xff\x9f\xfc\xff\x7d\xb8\xff\xcd\xff\xf6\x7f\xf7\xff"
		"\x9b\xff\xbf\x0f\x07\x00\x00\x00\x00\x00\x00\x00\x00\xff\xff\xf1\xff\x7f\xa9\xff"
		"\x7f\xf2\xff\x7f\xa9\xff\x9b\xff\x7f\xf9\xff\xbf\xd4\xff\xcd\xff\xf6\xff\x6f\xfe"
		"\xff\x2f\xf5\x7f\xf3\xbf\xfd\xdf\xfd\xff\xcf\xff\xff\xa5\xfe\xef\x01\x7e\xa8\x57"
		"\xe0";

	size_t dst_size = 31216;
	char *dst = malloc(dst_size);
	ptrdiff_t res = ufbxt_inflate(dst, dst_size, src, sizeof(src) - 1, opts);
	ufbxt_hintf("res = %d", (int)res);
	ufbxt_assert(res == dst_size);
	ufbxt_assert(ufbxt_fnv1a(dst, dst_size) == 0x9e9ed1e5);
	free(dst);
}
#endif

UFBXT_DEFLATE_TEST(deflate_long_code_sequences)
#if UFBXT_IMPL
{
	char src[] =
		"\x78\x9c\xdc\x93\x01\x0d\x00\x00\x0c\x82\x2a\x09\xf4\xef\xf6\x1e\x4f\xe0\x86\xb0"
		"\x02\x6c\x8b\x86\xab\xcd\x18\x1b\x6c\x83\xa5\xab\xa1\x91\x58\xba\xc5\x54\x30\x61"
		"\x33\x99\x65\x33\xc1\x56\x99\xb2\x9c\x80\xca\x36\xab\xc8\x6d\x6d\xb2\x39\x04\x06"
		"\x5b\xce\xb6\x39\x75\xe5\x44\x1c\x45\x24\xac\x95\x2b\x85\xc9\x26\x50\x1b\xc5\x2c"
		"\x92\xe6\xd6\x32\x20\x6b\x6e\xab\x30\xdc\xc8\xa0\xe6\xd6\x16\xdb\x5c\x14\x30\xb7"
		"\x10\x56\x6d\xa6\xe2\x97\x8d\xd1\x9a\xac\x2f\xa4\xa0\x40\xdd\x17\x26\x41\x6e\xc1"
		"\x97\x73\x16\xba\x2d\xbe\xb4\x6a\xb4\x46\x7d\xa9\x26\x1d\x53\xf6\x45\xe4\x34\x80"
		"\xf9\xc5\xd4\xa6\x2e\xdc\x17\xb4\x9b\x8a\x8e\x2f\xa6\x27\x6a\xab\x2f\x4d\xb6\xc2"
		"\xd0\x2f\x7d\x6d\xca\xc8\xbe\x98\xaf\x38\x57\x7e\x11\xc2\x95\xb9\xf6\x45\xc3\x22"
		"\x64\xf8\x85\xbb\x60\xba\xf6\xc5\x84\x21\x62\xf5\xa5\x16\xd8\x5a\xcd\x2f\x30\xac"
		"\xa6\xc8\x97\x38\x98\x88\xae\x2f\x1a\x6e\x58\x46\x5f\x0c\xcd\xc4\xa9\x5f\xec\xc4"
		"\x6c\x2d\xbe\x40\x41\xab\x92\x2f\x36\x84\xc0\x64\x5f\xda\x02\x67\xa2\x5f\x4e\x75"
		"\xcc\xc4\x7d\xf1\x45\xc3\xb9\xfa\x22\x94\x38\x8b\xfa\xf2\x7e\x73\x91\xf2\x25\x8d"
		"\x15\xb0\xb9\x2f\x47\xc3\x36\xdb\xfc\x12\x79\xd2\x1c\xf0\xe5\xab\x86\x00\xdb\x17"
		"\x9e\x2b\x30\xea\xcb\x35\x8d\x94\xc9\x17\x1a\x5b\x63\x9b\x7e\xc9\xb7\x01\x38\xfb"
		"\x72\x19\xea\x60\xf1\xa5\xd2\x61\xdb\x36\xbe\xc0\xda\x82\x26\xfb\xe2\x64\x21\x40"
		"\xfb\xa2\x02\xab\x60\xf2\xc5\x2c\x89\xa5\xf9\xa5\x09\x8b\xe1\xf0\x0b\x2b\x12\xd0"
		"\xed\x8b\x43\xc1\xa2\xea\x0b\xcf\x9a\x96\xf2\xa5\x48\x10\xc7\xb6\x2f\x8e\x66\xe9"
		"\xa6\x5f\x48\xb6\xc4\x68\x5f\xce\xb4\xd0\xb2\x2f\x16\x30\x69\x2a\x5f\x3c\x8b\xe1"
		"\x46\x7e\x69\xc5\xa0\x54\xbe\xbc\xa0\x36\x53\xbe\xa8\x89\x6d\x1b\xf4\x25\x45\x81"
		"\xc1\xc6\x97\x07\x24\x54\xe2\x4b\xdb\x05\xdb\xdc\xbe\x40\xc6\x66\x4a\x5f\x40\x6b"
		"\xe6\x72\x5f\xae\x75\x88\xad\x7d\x51\x51\x87\x99\x7d\xb9\x36\x5b\x92\xfb\xd2\x1f"
		"\x35\x64\xf3\x4b\x63\x2d\x03\xe0\x4b\x64\xe0\xd2\xe1\x97\x26\x6b\xb2\x9c\x5f\x72"
		"\xde\xa6\x42\x7d\xa9\x6e\x62\xc8\xf8\xc2\x9c\xdc\x1c\xec\x0b\x9e\x85\x8e\xd5\x17"
		"\xe2\x64\x86\xfa\x25\x20\x62\xcc\xad\x2f\x62\x08\xc2\xd0\x2f\xb1\x98\xca\xa6\x5f"
		"\xce\xa5\x60\x45\x5f\x3e\xc1\x5c\x89\x5f\x30\x93\x84\x63\x5f\x82\xb6\x34\xb3\x2f"
		"\xae\xd5\x9a\xac\xbe\x08\x84\x58\x42\x5f\xe4\x42\x9d\x4d\xbe\xc8\xd5\x64\xc6\xf6"
		"\xc5\x27\x03\x72\xf8\x25\x24\xd8\xc8\xe0\x4b\x01\xb1\x6a\xe8\x97\xa3\x65\x95\xac"
		"\x2f\xe9\x34\xa5\xd4\x2f\x71\x36\x70\x5b\xfb\x52\xc4\x16\x60\xf2\x25\x11\x4b\x0d"
		"\xfa\x22\x82\xa5\x0c\xf8\x62\xf7\x6a\x53\xdd\x97\x37\x64\xae\x89\x5f\xe2\x52\x6b"
		"\x13\xbf\xbc\x36\x71\x0d\xfa\x52\xd7\xc0\xb5\xd5\x17\xe4\xb9\xb9\x8c\x2f\x41\x4c"
		"\xb6\x05\x7e\x31\x62\xd3\x81\xf0\x25\xd6\x26\x5b\xd6\x17\xbe\xb3\x36\x83\x2f\x6f"
		"\x53\x2d\xda\xbe\x38\x82\xcb\xa1\x7c\x31\x22\x32\x90\x7d\x49\xcb\x66\xe0\xfc\x02"
		"\xc9\x05\x2a\x7e\xe9\x04\xdb\x58\xdb\x17\x87\x02\xc0\xd6\x17\x2b\xd5\x2d\x86\x5f"
		"\x64\xd8\x9c\x35\x3d\xf2\xee\x44\x00\x00\x00\x00\x61\x20\x5b\xaf\x3f\x52\x20\x1d"
		"\xc8\xf6\x16\xf8\xfa\x4c\xd5\xe0\x2e\xfc\x1d\xbc\x35\xfc\x52\xbc\xe6\x8d\x5d\xf5"
		"\x29\xed\x04\x18\x7c\x0a\x6d\xa3\x6a\xf8\x51\xb6\x28\x68\x04\x47\x19\x8e\x91\xf7"
		"\x27\x84\xb6\x6d\xdb\xae\x65\x87\x4d\x29\x7f\x48\x02\xa2\x52\xd5\xfa\x98\x6b\x9f"
		"\xfb\x7e\x30\xb0\x3f\xff\x7f\xcf\xe8\xe9\xff\x37\x09\x79\xf9\xff\xbb\x94\xd2\xcb"
		"\xff\xcf\x14\x99\x7e\xfd\xff\x29\x40\x4b\x7d\xfd\xff\xa9\x5e\x37\xd6\xe1\xff\xcf"
		"\x14\x25\x8d\xfd\xf9\xff\x7b\x8d\x4e\x63\xf0\xf3\xff\xd7\xc0\x46\xde\x7d\x08\x00"
		"\x00\x00\x20\x10\x64\x6b\xfa\x23\x05\xd2\xbd\x47\xbd\x2e\x5c\xff\xcd\xba\xfa\x6b"
		"\x54\x0c\x79\xdc\x42\x2c\x75\x39\x92\x16\x4b\x37\x6b\x6e\x2d\x8d\xbf\x85\x0b\xb2"
		"\x41\xde\xd6\x63\x12\x84\xd2\xc7\x06\xee\xc8\xfb\x13\x42\xdb\xb6\x6d\xd7\xb2\xc3"
		"\xa6\x94\x3f\x24\x01\x51\xa9\x6a\x7d\xcc\xb5\xcf\x7d\x3f\x18\xd8\x47\xff\x3f\x4a"
		"\xfd\xf9\xff\x2b\xd3\xbe\xfb\xff\x55\x30\xea\xf3\xff\xb7\x12\xd3\x1e\xfe\xff\xa0"
		"\x93\x6d\xfe\xfd\xff\x14\x1d\xd1\xf4\xe3\xff\x57\x93\x43\x05\x3a\xfd\xff\x5d\xb5"
		"\x9a\x36\xfd\xf5\xff\x2b\x81\x23\xef\x4e\x04\x00\x00\x40\x20\x08\x8e\x5a\xaf\x3f"
		"\x52\x20\xcd\x1e\x47\xfd\x38\xac\xfe\xec\xf2\x74\xbc\x3c\x0a\xd3\xfb\xeb\xc2\x60"
		"\x65\xfc\xbc\x6d\xd9\xb1\xfb\xe5\xc4\x52\xc4\x6c\xfc\x38\xc2\xd1\x19\x91\xf9\xec"
		"\x08\x6a\x67\xe6\x66\x7f\x3c\xc4\x71\xe4\xfd\x09\xa1\x6d\xdb\xae\x6d\xcb\x05\x35"
		"\xa5\xfc\x21\x09\x88\xa2\x54\xb5\x3e\xe6\xda\xe7\xbe\xcf\xc0\xfe\xf6\xff\xdb\x66"
		"\x0f\xff\x7f\x3b\x26\x1e\xfd\xff\x28\x65\xcf\xa7\xff\x5f\x66\xca\x8d\x6f\xff\x3f"
		"\xd3\x1c\x13\xf1\xec\xff\xb7\xd6\x22\x32\xf3\xc9\xff\x0f\x1c\xc5\xc4\xcc\xfe\xfa"
		"\xff\x15\xcd\xca\xc8\xac\x7a\xfb\xff\x23\x75\x47\xde\x7d\x10\x00\x00\x02\x41\x0c"
		"\x4b\xa5\x31\xfd\x4b\x42\x08\xe9\xbd\x0d\xf8\xef\x0d\x95\x0f\x5f\x8e\x3a\x5f\x3e"
		"\xb3\x76\xf7\x37\x4f\x42\x8d\x73\x3f\x7d\x1f\xec\xde\xb3\x7e\x7c\x0f\x6e\xad\x35"
		"\xc6\x47\x4f\x82\xed\x7b\xc6\xda\xff\x3c\xc7\x33\xf6\xdc\xf3\xce\x0f\x1f\xd5\x79"
		"\xe4\xfd\x0b\xa1\x6d\xbb\xb6\x2d\xcb\x45\x42\xd3\x97\x3f\x24\x01\x51\x64\xa9\x6a"
		"\x7d\xcc\xb5\xcf\x7d\x0c\xec\x4f\xff\x7f\xb4\xed\xc7\xff\xdf\x75\xb9\xbf\xfd\xff"
		"\x3a\xc9\x59\x9f\xfc\xff\xcc\xc7\x99\xeb\xbb\xff\x9f\xa9\xbc\xeb\xee\xef\xfe\x7f"
		"\x59\x6d\xcf\x75\xef\x4f\xff\x3f\x33\x32\xef\x5c\x6b\x7c\xf6\xff\xd3\xc5\x7b\xde"
		"\xb9\xe7\xf8\xe5\xff\xe7\xc5\x1e\x79\x77\x42\x00\x20\x10\xc4\x30\x70\x52\x67\xbc"
		"\xfe\x25\x21\x84\x49\xf7\x64\xc0\x5f\x3f\x5f\xf9\xcf\x4f\x75\xde\x3f\xfd\x50\xf7"
		"\xf5\xfe\xf5\x3b\x96\xd7\x7d\xfd\xf0\x2b\xe1\xc7\xf9\x1c\x7f\xfa\x28\xc6\x71\x1d"
		"\xd7\xfd\xa7\x0f\xa6\x5c\xf7\x7d\x3c\xef\x3f\xbe\xae\x74\x5e\xc7\x71\xdd\xc7\x5f"
		"\x3e\x61\xf9\x91\xf7\x27\x84\xb6\x6b\xdb\xb2\x2c\x69\x1e\xcc\x54\xf2\x87\x24\x20"
		"\x32\x8f\xd6\x35\xe6\xda\xe7\xbe\xcf\x20\xf3\x6f\xff\x3f\xd4\xfb\xd7\xff\x5f\xf5"
		"\x7d\x7f\xfb\xff\x93\x64\xef\xfd\xcb\xff\xdf\x4d\xbb\x8e\xe3\x6f\xff\xbf\x81\xf1"
		"\xbe\xe7\xf9\xaf\xff\xdf\x0c\x7b\xaf\xe3\x38\x7f\xf3\xff\xab\x81\xd7\x7d\x3e\xcf"
		"\xf5\xb3\xff\x1f\x78\x3d\xcf\xe3\xb8\x9e\xe7\x4f\xff\xbf\x88\xf7\x91\x77\x27\x04"
		"\x08\x04\x31\x10\x04\xab\x27\xca\x78\xfd\x4b\x42\x08\xd5\x93\x3d\x17\xf0\x8f\x3f"
		"\x9a\xf3\x97\xbf\x5d\xfc\xfc\xab\x9f\x3b\xf9\x3c\xfe\xf3\xe7\x17\xdf\x9f\xef\x1f"
		"\xfd\xd6\xe8\xe7\xf5\xfa\xfc\xe3\x2f\xf6\xbc\x5f\xdf\xd7\xeb\xaf\x7e\x68\xf8\xfd"
		"\xfe\xbe\xbe\xcf\xff\xf9\x35\xc2\xef\xe3\xf5\xfa\xbc\x5f\xff\xfa\x1b\x4f\x3f\xf2"
		"\xfe\x85\xd0\xb5\x6d\x59\x96\x25\xcd\x23\x91\xe9\xcb\x1f\x92\x80\xc8\x3c\xb2\x75"
		"\x8d\xb9\xf6\xb9\x8f\x41\xd5\xdf\xfe\xff\x9a\xc8\x3f\xfe\xff\xb0\xee\xe3\x9f\xfe"
		"\x7f\x36\xef\xfb\xf8\xaf\xff\x1f\xb5\xf9\x3e\x9f\x7f\xfb\xff\xf3\x9a\xcf\xf7\xeb"
		"\xf5\xa7\xff\xbf\x2d\xf6\x7c\x7f\xdf\x9f\x3f\xfb\xff\xd1\xdf\xc7\xe3\xfb\x78\xbd"
		"\xff\xe8\xff\xe7\x62\xbf\xaf\xf7\xeb\xf9\x78\xfc\xe9\xff\xaf\xd9\xfd\xc8\xbb\x13"
		"\x82\x08\x82\x20\x08\x82\x91\xd5\x63\x8c\xd7\xbf\x24\x84\x10\x59\x3d\x7b\x2a\xe0"
		"\xff\xfc\x5d\xc2\xff\xf7\x97\x83\xbe\xfe\xe7\x5f\x6c\xff\xfe\xfe\xaf\x7f\xcd\xf0"
		"\xeb\xf3\xeb\x1f\xfd\x5d\xf5\xef\xcf\x9f\xef\xff\xf2\x17\x40\x3f\x7f\xbf\x7e\x3f"
		"\xfe\xc9\xdf\x93\xfe\xfa\xfd\xfa\xfe\xf8\xfe\x87\x7f\x09\xf6\xe7\xeb\xe3\xe3\xf7"
		"\xfb\xf3\x3f\xfd\xb9\xc3\x3f\xf2\xfe\x85\xb0\xb6\x65\x59\x96\x2d\x9b\x79\x24\x31"
		"\x7d\xf9\x43\x12\x10\x35\xf3\xc8\x3e\x34\xd7\x3e\xf7\x31\xa8\xfa\x3f\xff\x3f\xdb"
		"\xf2\x6f\xfe\x7f\x02\xf0\xe3\xff\xfc\xff\xc6\xdc\xf7\xef\xff\xfb\xff\x6d\x8b\xaf"
		"\x9f\xcf\x7f\xf6\xff\xfb\xb7\xdf\x9f\x3f\x9f\xff\xf2\xff\x77\x82\xbe\xbe\xbe\xbf"
		"\x3f\xff\xe5\xff\xef\x01\x7f\x7f\xff\x7c\x7e\x7f\xfd\xcb\xff\xdf\x20\xfe\xfa\xfe"
		"\xfc\xfc\xfc\xfe\xfc\x6f\xff\xbf\x1d\xfb\x47\xde\x9d\x10\x5a\x11\x04\x41\x10\x8c"
		"\xac\x9e\x15\xc6\xe9\x5f\x12\x42\x88\xac\x9e\x7d\xdf\x04\xfc\x27\xff\xd0\xd9\x7f"
		"\xf8\x4f\x6b\xfc\xfa\xef\xfe\x71\xd6\xdf\xbf\xff\xc3\x7f\x18\xe9\xcf\xdf\x9f\xff"
		"\xd5\xbf\x37\xf6\xeb\xcf\x9f\x9f\xff\xdf\x3f\x9f\xfb\xf7\xef\xef\x3f\x7f\xff\xab"
		"\x7f\xe3\xe9\xc7\xef\x5f\x7f\x7e\xfe\xfa\xef\xfe\x89\x95\x3f\x7f\x7f\xfc\xf9\xf3"
		"\xf3\xd7\xff\xf7\x4f\x0c\xfe\x23\xef\x5f\x0c\xed\x58\x96\x65\x59\xce\x3c\xb2\x5a"
		"\x30\x7c\xf5\x17\x09\x82\xc0\x3c\xb2\x7a\x4c\xac\x7d\xee\xd3\x80\xfc\xdf\xfe\x7f"
		"\xc0\xf3\xff\xfc\xff\x15\xad\xbf\xff\xd3\xff\x6f\x13\xfd\xf8\xfb\x1f\xfd\xff\x30"
		"\xce\x8f\xdf\x3f\xfe\x9f\xff\xdf\x65\xff\xf5\xe7\xcf\xcf\xff\xe7\xff\x07\x31\x7e"
		"\xfd\xfc\xf9\xeb\xd7\xff\xf2\xff\x6b\x35\xbf\x7f\xff\xf8\xf5\xe3\xef\x7f\xf7\xff"
		"\xd3\x23\xbf\x7e\xfc\xfa\xf9\xf3\xcf\x9f\xff\xec\xff\x07\x3c\xfe\x91\x77\x2f\x84"
		"\xb2\x14\x41\x10\x44\x4f\x64\xf5\xac\x2f\xbe\xfe\x25\x21\x84\x13\x59\x3d\x7b\x9f"
		"\x08\xf8\x7f\xfd\x17\xca\xff\x9f\xff\xd9\xa1\x3f\xfe\x37\xff\xb9\xf1\x7f\xfe\xfd"
		"\xff\xfd\x57\xd0\xfe\xf8\xe7\xdf\xff\xe7\x7f\x1f\xf9\xef\x7f\xfe\xf8\xeb\x7f\xf7"
		"\x1f\x9b\xfe\xeb\xaf\x7f\xff\xfd\xf3\xff\xf3\x1f\x81\xfa\xf3\xcf\xbf\xfe\xf9\xeb"
		"\x8f\xff\xd1\x7f\x7c\xfc\x9f\xbf\xff\xf8\xf7\xcf\x3f\xfe\xfc\xdf\xfd\xc7\xfa\xfe"
		"\x23\xef\x4f\x0c\x65\x39\x96\x65\x59\x52\xd4\x3c\xb2\xf8\xc2\xc8\x3f\x49\x20\x04"
		"\xa2\xe6\x91\xb5\x36\xce\x7d\x9f\x83\xee\xff\xdb\xff\xef\xa3\xfe\x4f\xfe\x7f\x9a"
		"\xf2\xaf\xff\xe7\xff\xaf\xc1\xfd\xfd\xd7\xff\xea\xff\xe7\x6b\xff\xfe\xeb\x8f\xff"
		"\xc7\xff\x0f\xc3\xfd\xf1\xe7\x5f\x7f\xfc\x5f\xfe\x7f\x68\xfa\x9f\x7f\xff\xfa\xeb"
		"\xef\xff\xcd\xff\xaf\xd0\xfc\xfd\xcf\xbf\x7f\xff\xfd\xd7\xff\xf3\xff\xe7\xe8\xfe"
		"\xfa\xe7\xaf\x3f\xfe\xfe\xf7\xaf\xff\xcd\xff\x4f\xb8\xfd\x47\xde\x9d\x10\xc8\xb2"
		"\x1c\x41\x10\x34\x8f\xac\x9e\xc5\xa5\x93\x3f\x24\x01\x91\x79\x64\xf5\xec\xfb\x1c"
		"\xa4\xff\x9b\xff\xad\xa2\xff\x9b\xff\xe1\xe1\x7f\xff\xbf\xfc\x8f\x50\xfc\xfb\xbf"
		"\xff\x57\xff\xa3\x2a\xff\xfc\xf7\xbf\xff\x4f\xff\x27\xbf\xfe\xf5\x8f\x7f\xff\xf3"
		"\xff\xf6\x7f\xb5\xf1\x5f\xff\xfc\xef\x3f\xff\xfb\x7f\xf9\x3f\x44\xf8\xef\x7f\xfe"
		"\xf3\x3f\xff\xfa\xf7\xff\xd1\xff\x58\xd9\xbf\xff\xfd\x9f\x7f\xff\xeb\x3f\xff\xfe"
		"\xbf\xfc\xdf\x9d\xfe\x1f\x79\xff\x62\x20\xcb\x72\x2c\xcb\x92\xa2\xe6\x91\xd5\x74"
		"\xe1\xcb\x3f\x49\x20\x04\xa2\xe6\x91\xd5\x6b\xe3\xdc\xc7\xc1\xcc\xff\xd5\xff\x8f"
		"\x86\xfd\x7f\xfc\xff\xd8\xc8\x7f\xff\x2f\xff\x7f\xaf\xfc\xbf\xff\xf9\xff\xfa\xff"
		"\xd1\xf5\xff\xfe\xe7\xdf\xff\x67\xff\xbf\x6a\xf3\x8f\xff\xfe\xf7\xbf\xff\x8f\xff"
		"\x3f\xb7\xf8\x9f\xff\xfe\xfb\x1f\xff\xfd\x3f\xfd\xff\x65\xa3\xff\xfc\xfb\x1f\xff"
		"\xfa\xc7\x3f\xff\x3f\xff\x7f\x87\xf8\x1f\xff\xfa\xc7\x3f\xff\xfd\x9f\x7f\xff\x3f"
		"\xfe\xff\xa8\xca\xff\xc8\xbb\x17\x02\x49\x96\xe5\x08\xa2\xc7\x3c\xb2\x7a\x96\x96"
		"\xbe\xfc\x21\x09\x88\x8e\x79\x64\xf5\xec\x7d\x18\xa4\xff\xb7\xff\x97\x25\xff\x7f"
		"\xfe\x9f\x59\xfb\xef\xff\x27\xff\xa7\x86\xfe\xf3\xbf\xff\x9f\xfc\xdf\x89\xf9\xef"
		"\xff\xfc\x9f\xff\x47\xff\x77\x37\xff\xe3\x7f\xff\xe3\xbf\xfe\xbf\xfe\x5f\x15\xfc"
		"\xcf\xff\xf8\xef\xff\xfc\xdf\xff\x5f\xff\x87\x43\xfe\xf7\x3f\xfe\xfb\x3f\xff\xeb"
		"\xbf\xfe\xbf\xfc\x1f\x15\xfa\xcf\xff\xf9\xdf\xff\xfd\x9f\xff\xfd\xaf\xff\x77\xff"
		"\x67\x2c\xff\x8f\xbc\x7f\x31\x90\x64\x59\x8e\x65\xcb\x25\x6a\x1e\x59\xcd\x16\xbe"
		"\xfc\x93\x04\x42\xb0\x44\xcd\x23\xab\xf7\xc1\x7d\x1c\xcc\xfc\xff\xf9\xff\x81\xa0"
		"\xff\x57\xff\x3f\xdf\xf4\x3f\xff\x4f\xfe\x7f\x19\xee\x3f\xfe\xeb\xff\xd3\xff\x2f"
		"\x2d\xfd\xe7\xff\xfc\xef\xff\xcf\xff\xdf\xa6\xfb\x8f\xff\xf9\xdf\xff\xf9\xff\xf1"
		"\xff\x13\x8c\xff\xf9\x3f\xff\xf3\xbf\xff\xfb\xff\xe7\xff\x77\x87\xff\xfb\xbf\xff"
		"\xe7\xbf\xfe\xfb\x3f\xfe\x5f\xfd\xff\x5e\xe4\x7f\xff\xef\xff\xfc\xd7\x7f\xff\xc7"
		"\x7f\xfe\x7f\xfd\xff\x71\x98\xff\x23\xef\x4e\x0c\x24\x49\x96\xe5\x08\x8a\x9a\x47"
		"\x56\xcf\x92\x85\x93\x7f\x92\x40\x08\x44\xcd\x23\xab\x67\xdf\x67\x01\xf8\xff\xe8"
		"\xff\x29\xc9\xff\xef\xfe\x5f\x41\xfe\xdf\xff\xdf\xfe\x3f\x5a\xfb\x7f\xff\xd7\xff"
		"\xa7\xff\xcf\xfc\xfd\xdf\xff\xf1\xbf\xfe\xff\xf9\x7f\x06\xf0\x7f\xfc\xcf\xff\xfd"
		"\xbf\xff\x7f\xfa\x7f\x8b\xf5\x7f\xfe\xef\xff\xf9\x7f\xfe\xc7\xff\x3f\xff\x8f\x8b"
		"\xfe\x9f\xff\xf3\x7f\xff\xef\xff\xf8\xdf\xff\xdf\xfc\xbf\x86\xfd\xbf\xfe\xe7\xff"
		"\xf8\x1f\xff\xe3\xff\xfc\xaf\xff\x2f\xff\x1f\x43\xfd\x7f\xe4\xfd\x8b\x81\x24\xc9"
		"\xb2\x5c\x5b\xae\x2d\x6a\x1e\x59\x4d\x16\xbe\xfc\x93\x04\x42\xb0\xb6\xa8\x79\x64"
		"\xf5\xb9\x78\x1c\xcc\xfc\xff\xf3\xff\xa3\xd9\xff\x9f\xfe\xff\xda\xdd\xff\xfa\xff"
		"\xea\xff\x37\x2b\xff\xc7\xff\xf9\xff\xf0\xff\x07\x5a\xff\xc7\xff\xf9\xbf\xff\xbf"
		"\xfc\xff\x98\xd7\xff\xfe\x5f\xff\xe3\x7f\xfc\x7f\xf2\xff\xe3\xa8\xff\xf3\x7f\xfe"
		"\xef\xff\xfd\x7f\xff\x3f\xfa\xff\xf1\xd9\xff\xf5\xbf\xff\xe7\xff\xfc\x9f\xff\xe7"
		"\xff\xa7\xff\x1f\x16\xfb\xbf\xff\xf3\x7f\xff\x9f\xff\xfd\xbf\xff\xe7\xff\x57\xff"
		"\xbf\x8f\xf3\xff\xc8\xbb\x17\x03\x49\x92\x64\x59\xae\x47\xd4\x3c\xb2\x7a\x96\x2b"
		"\x7c\xf9\x27\x09\x84\xe0\x88\x9a\x47\x56\xcf\xde\x47\x02\xf0\xff\xab\xff\xef\xf1"
		"\xfc\xff\xe5\xff\xab\x55\xff\xd7\xff\x7f\xfe\x3f\xd7\xf3\xff\xfc\xbf\xff\xbf\xfd"
		"\xff\x8a\xc9\xff\xf9\xff\xfe\x9f\xff\x3f\xf9\xff\x54\xe1\xff\xf9\xff\xfc\x3f\xff"
		"\xe7\xff\xaf\xfe\x3f\xbe\xfc\x7f\xff\xdf\xff\xcf\xff\xf5\x7f\xfc\xff\xe5\xff\xfb"
		"\xe5\xff\xe7\xff\xfd\xff\xfc\xdf\xff\xc7\xff\xfb\xff\xc7\xff\xef\x07\xfe\xdf\xff"
		"\xe7\xff\xf9\x7f\xfe\xdf\xff\xef\xff\xf9\xff\xbb\xff\xef\x5c\xfd\x7f\xe4\xfd\x8b"
		"\x81\x24\x49\xb2\x6d\x57\xae\x7d\x44\xcd\x23\xab\xb9\xc2\x97\x7f\x92\x40\x08\xd6"
		"\x3e\xa2\xe6\x91\xd5\xf7\x81\x83\x99\xff\x9f\xfd\xff\xd8\xc7\xff\x3f\xfe\x7f\xbd"
		"\xe2\xff\xfe\xff\xe3\xff\x4f\x78\xfc\x9f\xff\xe7\xff\xcf\xfe\x7f\x0f\xe6\xff\xfc"
		"\x7f\xff\xcf\xff\x3f\xfd\xff\x88\xed\xff\xfb\x7f\xff\x3f\xff\xcf\xff\x3f\xfe\x7f"
		"\x7d\xf0\xff\xfd\x3f\xfe\xcf\xff\xeb\xff\xfd\xff\xc7\xff\x2f\x32\xfd\x1f\xff\xc7"
		"\xff\xf1\xff\xfe\x1f\xff\xf7\xff\x9f\xfe\x7f\x26\xef\xff\xf8\x3f\xff\xcf\xff\xfb"
		"\xff\xfd\xbf\xff\xef\xff\x7f\xfd\xff\x78\xca\xff\x47\xde\xbf\x18\x48\x92\x24\xd9"
		"\xb6\xdd\xda\x47\xd4\x3c\xb2\x7a\xa8\xc2\x97\x7f\x92\x40\x08\xd6\x3e\xa2\xe6\x91"
		"\xd5\x73\x1f\x07\xc0\xff\x1f\xfd\xff\x6c\xf5\xff\x5f\xfe\xff\xd2\xe8\xff\xeb\xff"
		"\x6f\xff\x7f\x11\xe9\xff\xfd\xff\xfe\xff\xb7\xff\x1f\xc2\xf9\x7f\xfe\xbf\xfe\xbf"
		"\xff\xff\xfa\xff\xc3\x8c\xff\x9f\xff\xef\xff\xe3\xff\xf3\xff\x3f\xff\x3f\x16\xfa"
		"\xff\xfa\x7f\xfd\x3f\xff\x1f\xff\xcf\xff\xff\xfb\xff\xf5\xaf\xff\xef\xff\xe3\xff"
		"\xfd\xff\xfc\x7f\xff\xbf\xff\xff\xe8\xff\x27\x9c\xfe\x1f\xff\xaf\xff\xc7\xff\xf7"
		"\xff\xfb\xff\xf8\xff\xfc\xff\xc9\xff\xef\x21\xff\xff\xca\xfb\x17\x03\x49\x92\x24"
		"\xdb\xb6\x5b\xfb\x88\x9a\x47\x56\x0f\x55\xf8\xf2\x4f\x12\x08\xc1\xda\x47\xd4\x3c"
		"\xb2\x7a\xee\xe3\x00\xf8\xff\xd7\xff\x9f\x0a\xfc\xff\xcb\xff\x9f\x00\xfa\xff\xfe"
		"\xff\xd3\xff\xcf\x90\xfe\xbf\xfe\xdf\xff\x7f\xf3\xff\xdb\xbc\xff\xaf\xff\xe7\xff"
		"\xe3\xff\x3f\xff\x7f\x98\xf1\xff\xf5\xff\xfe\x7f\xfd\x7f\xff\xff\xea\xff\xe7\xd7"
		"\xff\xbf\xff\xcf\xff\xd7\xff\xeb\xff\xf3\xff\x57\xff\xbf\x3c\xf7\xff\xfd\xff\xfe"
		"\x3f\xff\xbf\xff\xef\xff\xd7\xff\x7f\xfc\xff\xa6\xdf\xff\xf7\xff\xf5\xff\xfd\x7f"
		"\xfd\x3f\xfe\xdf\xff\xdf\xff\xbf\xfa\xff\xcd\xc3\xff\x1f\x77\x74\xe4\x9b";

	size_t dst_size = 85130;
	char *dst = malloc(dst_size);
	ptrdiff_t res = ufbxt_inflate(dst, dst_size, src, sizeof(src) - 1, opts);
	ufbxt_hintf("res = %d", (int)res);
	ufbxt_assert(res == dst_size);
	ufbxt_assert(ufbxt_fnv1a(dst, dst_size) == 0x138da12f);
	free(dst);
}
#endif

UFBXT_DEFLATE_TEST(deflate_two_symbol_bits)
#if UFBXT_IMPL
{
	char src[] =
		"\x78\x9c\x04\xc0\x81\x10\x00\x30\x0c\x04\xc1\xaa\x35\x24\xf7\xfe\x12\x3b\xff\x10"
		"\x00\x07\x42\x00\x03\x41\x10\x04\x13\xb4\x7b\x92\x59\x7f\x89\xae\x7b\x08\x80\x03"
		"\x22\x06\x01\x18\x08\x82\x2d\x28\x0b\x4a\xee\xfd\x9b\xd8\xb9\x0f\x01\x70\x42\xd4"
		"\x00\x00\x04\x41\x10\xf2\xea\xba\x28\x99\xf5\x6f\xa2\xeb\x7e\x08\x80\x0f\xaa\x30"
		"\x00\x20\x08\xa2\x09\xbd\xa3\xe9\x50\x32\xeb\xdf\xc4\x7f\xf7\x87\x00\xb8\xa0\x02"
		"\xc3\x00\x82\x20\xda\x86\x99\x19\xb5\x5c\x94\xcc\xfa\x37\xf1\xdf\xfd\x43\x00\x1c"
		"\x08\x01\x0c\x04\x41\x10\x8c\x5a\x9e\x64\xf6\xfc\x21\xba\xfe\x87\x00\x38\x20\x42"
		"\x18\x08\x82\x20\x08\xd2\x3e\x4a\x66\xcf\xbf\x88\xae\xf7\x21\x00\x0e\x88\x13\x00"
		"\x82\x20\x08\x26\xa0\xec\x51\x32\x7b\xfe\x45\x74\xbd\x1f\x02\xe0\x83\x38\x0c\x20"
		"\x08\x82\xa0\xad\x88\xeb\x85\x64\xf6\xf8\x83\xe8\x7a\x7f\x08\x80\x0f\x22\x30\x8c"
		"\x20\x08\x82\xb6\x72\x16\xa6\x17\x92\xd9\xe3\x0f\xa2\xeb\xfd\x43\x00\x5c\x50\x01"
		"\x60\x1c\x51\x14\x6d\xc3\xcc\x8c\x5a\x36\x4a\xde\x1f\xff\x22\xee\x79\xff\x21\x00"
		"\x0e\x88\x10\x06\x82\x20\x08\x62\x8d\x28\x99\xbd\xf7\xef\xa1\xeb\xff\x21\x00\x0e"
		"\x88\x22\x08\x82\x20\x08\x82\xb4\x7b\x25\xd5\xb3\xfe\x3d\x64\x7c\x3f\x04\xc0\x01"
		"\x55\x18\x41\x10\x04\xd1\x04\x94\x1d\x4a\xaa\x67\xfd\x7b\xf8\xef\xfb\x43\x00\x9c"
		"\x10\x81\x71\x04\x41\x10\xb4\xf5\xe2\x3a\x21\xa9\x9e\xe5\xcf\x21\xe3\xfb\x87\x00"
		"\xf8\xa0\x02\xe0\x38\x82\x20\x6a\x2b\x67\x61\x3a\x21\xa9\x9e\xe5\xcf\xe1\xbf\xef"
		"\x3f\x04\xc0\x05\x15\x00\xd1\x75\x45\xd1\x36\xcc\xcc\xa8\x65\xa2\xe4\xdc\xf7\xfd"
		"\x7b\xd8\xeb\xfb\x1f\x02\xe0\x80\x28\x82\x20\x08\x82\x20\xd6\x78\x25\xd5\xb3\xe7"
		"\x5f\x43\xc6\xff\x0f\x01\x70\x40\x1c\x47\x10\x04\x41\xd0\x86\xf6\x42\x52\x3d\x7b"
		"\xfc\x31\x64\xfc\xfe\x10\x00\x07\x44\x70\x1c\x41\x10\x04\x6d\x09\xd9\x0b\x49\xf5"
		"\xec\xf1\xc7\x90\xf1\xfb\x87\x00\x38\xa1\x82\xe0\x38\x82\x20\x6a\xeb\xc4\xb5\x42"
		"\x12\x59\x3d\xfc\x31\xfc\xf7\xfb\x0f\x01\xf0\x41\x05\x41\x70\x5c\x51\xd4\x56\xce"
		"\xc2\xb4\x42\xf2\x7e\xf5\xf0\xc7\x70\xcf\xef\x7f\x08\x80\x0b\x2a\x8a\xa2\xe8\xba"
		"\xa2\x6d\x98\x99\x51\xcb\x8f\x92\xb5\xcf\x7d\xfe\x35\xcc\xf1\xfb\x3f\x04\xc0\x01"
		"\x71\x1c\x41\x10\x04\x41\x53\xb3\x90\x54\xcf\xde\xf3\xa7\x90\xf1\xff\x0f\x01\x70"
		"\x40\x24\xc9\x11\x04\x41\x50\x82\x76\x8f\x24\xb2\x7a\x96\x3f\x05\xb7\xbf\x7f\x08"
		"\x80\x03\x2a\x09\x8e\x23\x08\xa2\xb6\x84\xec\x84\x24\xb2\x7a\x96\x3f\x85\xff\xfe"
		"\xfe\x43\x00\x9c\x50\x49\x10\x1c\x57\x14\xb5\xb5\xe2\x1a\x21\x79\x3f\xb2\x9a\x3f"
		"\x85\x7b\xfe\xfe\x87\x00\xf8\xa0\x92\x20\x8a\xae\x2b\x6a\x2b\x67\x61\x1a\x21\x39"
		"\xf7\xfd\x6a\xfe\x14\xf6\xfa\xfb\x3f\x04\xc0\x05\x95\x45\x51\x14\x65\xd7\xa4\x99"
		"\x99\x51\x4b\xb5\x92\xb5\xcf\x7d\xdf\xbf\x85\x39\xfe\xfe\x1f\x02\xe0\x80\x48\x92"
		"\x23\x08\x82\xa0\xa8\xe9\x91\x44\x56\xcf\x1e\x7f\x08\x6e\xff\xff\x43\x00\x1c\x50"
		"\x49\x92\x1c\x41\x10\x95\xa0\xed\x21\x31\x8f\xac\x1e\xfe\x10\xfe\xfb\xf7\x1f\x02"
		"\xe0\x80\x4a\x92\xe0\xb8\xa2\xa8\x2d\x21\x5b\x21\x79\x3f\xb2\x7a\xf8\x43\xb8\xe7"
		"\xdf\xff\x10\x00\x27\x54\x92\x44\xd1\x75\x45\x6d\x8d\xb8\x5a\x48\xce\x7d\x3f\xb2"
		"\xf8\x43\xd8\xeb\xdf\xff\x21\x00\x3e\xa8\x2c\x8b\xa2\xe8\xba\xda\xca\x59\x98\x5a"
		"\x48\xd6\x3e\xf7\xfd\xe2\x0f\x61\x8e\x7f\xff\x0f\x01\x78\x41\x6d\x5b\x14\x45\x51"
		"\x76\x49\x31\x33\xa3\x96\x5f\x4a\xc6\x5c\xfb\xdc\xe7\x5f\x42\x6f\xff\xfe\x3f\x04"
		"\xc0\x01\x95\x24\xc9\x11\x04\x51\x51\xd3\x21\x31\x8f\xac\x9e\xe5\xcf\xe0\xbf\xff"
		"\xff\x87\x00\x38\xa0\x92\x24\x49\xae\x28\x2a\x41\x9b\x45\xf2\xbe\x79\x64\x35\x7f"
		"\x06\xf7\xfc\xf7\x3f\x04\xc0\x01\x95\x24\x59\x74\x5d\x51\x5b\x42\x36\x42\x72\xee"
		"\xfb\x91\xd5\xfc\x19\xec\xf5\xdf\xff\x21\x00\x4e\xa8\x2c\xcb\xa2\xe8\xba\xda\x6a"
		"\x71\x95\x90\xac\x7d\xee\xfb\x91\xfc\x19\xcc\xf1\xdf\xff\x43\x00\x7e\x50\xdb\x96"
		"\x45\x51\x74\x9d\xad\x9c\x85\xa9\x84\x64\xcc\xb5\xcf\x7d\x9f\x3f\x83\xde\xfe\xfb"
		"\xff\x10\x80\x17\xd4\xb6\x45\x5b\x14\x5d\xfb\x99\x31\xcc\xd1\xf2\xa2\xa4\xf5\x31"
		"\xd7\x3e\xd7\xbf\x83\x5a\xfe\xf3\xbf\x21\x00\x0e\xa8\x24\x49\x92\x2b\x8a\x8a\x9a"
		"\x16\xc9\xfb\xe6\x91\xd5\xc3\x1f\xc1\x3d\xff\xff\x3f\x04\xc0\x01\x95\x24\x59\x96"
		"\x5d\x51\x09\x5a\x0f\x92\x73\xdf\x37\x8f\x2c\xfe\x08\xf6\xfa\xdf\xff\x21\x00\x0e"
		"\xa8\x2c\xcb\xb2\xe8\xba\xda\x12\xb2\x16\x92\xb5\xcf\x7d\x3f\xb2\xf8\x23\x98\xe3"
		"\x7f\xff\x0f\x01\x78\x42\x6d\x5b\x96\x45\xd1\x75\xb6\x4a\x5c\x29\x24\x63\xae\x7d"
		"\xee\xfb\xc1\x1f\x41\x6f\xff\xfb\xff\x10\x80\x1f\xd4\xb6\x65\x51\x16\x5d\x59\xde"
		"\xc8\x62\x4a\x21\x19\x73\xed\x73\xdf\x0f\xfe\x08\x7a\xfb\x9f\xff\x0b\x01\x78\x41"
		"\x6d\x5b\xb4\x6d\x51\x96\xc9\xcc\x9c\x81\x5a\x7e\x28\x69\x7d\xcc\xb5\xcf\x7d\xfe"
		"\x15\xd4\xf2\x5f\xff\x13\x02\xe0\x80\x4a\x92\x2c\xcb\xae\xa8\xa8\x69\x90\x9c\xfb"
		"\xbe\x79\x64\x35\x7f\x02\x7b\xfd\xff\xff\x43\x00\x1c\x50\x59\x96\x65\x59\x76\x95"
		"\xa0\x55\x23\x59\xfb\xdc\xf7\xcd\x23\xf9\x13\x98\xe3\xff\xfe\x1f\x02\xf0\x80\xda"
		"\xb6\x2c\xcb\xa2\xeb\x6c\x09\x59\x09\xc9\x98\x6b\x9f\xfb\x7e\x24\x7f\x02\xbd\xfd"
		"\xdf\xff\x87\x00\x3c\xa1\xb6\x2d\xcb\xa2\xec\xca\x2c\x25\x5c\x21\x24\x63\xae\x7d"
		"\xee\xfb\xe6\xfc\x09\xf4\xf6\x7f\xfe\x1f\x04\xe0\x07\xb5\x6d\x5b\xb4\x65\x57\xa6"
		"\xf7\x29\x61\x0a\x21\x69\x7d\xcc\xb5\xcf\x7d\x9f\x3f\x81\x5a\xfe\xd7\xff\x81\x00"
		"\xbc\xb0\xb6\x2d\xda\xb6\xec\xca\x64\x66\x46\xc9\x96\x90\x25\xad\x8f\xb9\xf6\xb9"
		"\xef\xfb\xef\x40\x2d\xff\xed\x7f\x40\x00\x1c\x50\x59\x96\x65\x59\x76\x15\x35\x35"
		"\x92\xb5\xcf\x7d\xdf\x3c\xb2\xf8\x03\x98\xe3\xff\xff\x7f\x08\xc0\x03\x6a\xdb\xb2"
		"\x2c\xcb\xb2\x93\xa0\x65\x21\x19\x73\xed\x73\xdf\x37\x0f\xfe\x00\x7a\xfb\x7f\xff"
		"\x1f\x02\xf0\x84\xda\xb6\x2c\xcb\xa2\x2b\xb3\x40\x26\x21\x19\x73\xed\x73\xdf\x37"
		"\x0f\xfe\x00\x7a\xfb\x7f\xfe\x5f\x04\xe0\x09\xb5\x6d\xdb\x16\x65\xd7\x39\x37\x2e"
		"\x09\x49\xeb\x63\xae\x7d\xee\xfb\xc1\x1f\x40\x2d\xff\xd7\xff\x8d\x00\xfc\xa0\xb6"
		"\x6d\x8b\xb6\xec\xca\xf4\xde\x60\x92\x90\xb4\x3e\xe6\xda\xe7\xbe\x1f\xfc\x01\xd4"
		"\xf2\xbf\xfd\xef\x08\xc0\x83\x2b\xcb\xb2\x2c\xcb\xb2\x03\x6b\x28\x4a\xd6\x3e\xf7"
		"\x7d\x51\xf3\x48\xfe\xfb\x73\xfc\xff\xff\x3f\x04\xe0\xc1\xb5\x6d\x59\x96\x65\x51"
		"\x81\x68\x20\xc9\x98\x6b\x9f\xfb\xbe\x79\xa4\xfe\xf5\xde\xfe\x7f\xff\x7f\x04\xe0"
		"\xc9\x95\x65\x59\x96\x45\xd9\xa1\xa0\x0c\xa0\x64\xed\x73\xdf\x17\x35\x8f\xe4\xbf"
		"\x3f\xc7\xff\xef\xff\x8f\x00\x3c\xb9\xb6\x6d\xdb\xa2\x2c\x2b\x62\xe4\x02\x49\x5a"
		"\x1f\x73\xed\x73\xdf\x37\xd7\x3f\x5f\xcb\xff\xef\xff\x8f\x00\x3c\xb8\xb6\x2d\xcb"
		"\xb2\x2c\x2a\x54\x03\x49\xc6\x5c\xfb\xdc\xf7\xcd\x23\xd1\xbf\xdc\xdb\xff\xf7\xff"
		"\x87\x00\x3c\xb8\xb6\x6d\xdb\xb2\xe8\x2a\x30\x0d\x2c\x69\x7d\xcc\xb5\xcf\x7d\x3f"
		"\x24\xff\xd3\xb5\xfc\xff\xfe\xff\x08\xc0\x93\x6b\xdb\xb2\x2c\x8b\xa2\x22\x91\x0c"
		"\x24\x19\x73\xed\x73\xdf\x37\x0f\xd0\xbf\xdc\xdb\xff\xe7\xff\x8b\x00\x3c\xb8\xb6"
		"\x6d\xdb\xb2\xe8\x2a\x5c\x03\x4b\x5a\x1f\x73\xed\x73\xdf\x0f\x09\xff\xb3\xb5\xfc"
		"\x7f\xff\x7f\x08\xc0\x83\x6b\xdb\xb2\x2c\x8b\xb2\x02\xd1\x40\x92\x31\xd7\x3e\xf7"
		"\x7d\xf3\x48\xd0\xbf\xda\xdb\xff\xe7\xff\xab\x00\x3c\xb8\xb6\x2d\xcb\xb2\x28\x2b"
		"\x54\x03\x49\xc6\x5c\xfb\xdc\xf7\xcd\x23\x01\xfd\x8b\xbd\xfd\xbf\xff\x3f\x00\x96"
		"\xe1\x23\x50";

	char dst[256];
	ptrdiff_t res = ufbxt_inflate(dst, sizeof(dst), src, sizeof(src) - 1, opts);
	ufbxt_hintf("res = %d", (int)res);
	ufbxt_assert(res == 138);
	for (size_t i = 0; i < 138; i += 2) {
		ufbxt_assert(dst[i + 0] == 'A');
		ufbxt_assert(dst[i + 1] == 'B');
	}
}
#endif

UFBXT_DEFLATE_TEST(deflate_fuzz_1)
#if UFBXT_IMPL
{
	char src[] =
		"\x78\x9c\x30\x04\x00\xfb\xff\x30\x30\x30\x30\x52\x30\x30\x30\x02\x10\x00\x06\x32"
		"\x00\x00\x00\x0c\x52\x39\xcc\x45\x72\xc8\x7f\xcd\x9d\x30\x08\x00\xf7\xff\x30\x30"
		"\x30\x30\x30\x30\x30\x30\x02\x8b\x01\x38\x8c\x43\x12\x00\x00\x00\x00\x40\xff\x5f"
		"\x0b";

	char dst[4096];
	ufbxt_inflate(dst, sizeof(dst), src, sizeof(src) - 1, opts);
}
#endif

UFBXT_DEFLATE_TEST(deflate_fuzz_2)
#if UFBXT_IMPL
{
	char src[] =
		"\x78\x9c\x00\x04\x00\xfb\xff\x54\x65\x73\x74\x52\x08\x48\x2c\x02\x10\x00\x06\x32"
		"\x00\x00\x00\x0c\x52\x39\xcc\x45\x72\xc8\x7f\xcd\x9d\x00\x08\x00\xf7\xff\x74\x61"
		"\x20\x44\x61\x74\x61\x20\x02\x8b\x01\x38\x8c\x43\x12\x00\x00\x00\x00\x40\xff\x5f"
		"\x0b\x36\x8b\xc0\x12\x80\xf9\xa5\x92\x23\x84\x00\x8e\x36\x10\x41";

	char dst[4096];
	ufbxt_inflate(dst, sizeof(dst), src, sizeof(src) - 1, opts);
}
#endif

UFBXT_DEFLATE_TEST(deflate_benchmark)
#if UFBXT_IMPL
{
	if (opts->force_fast) return;

	char src[] =
		"\x78\x9c\x4d\x9c\x79\x7c\x56\xc5\xd5\xc7\x6f\xf6\xa0\x40\x40\xa5\x40\x15\x12\xc1"
		"\x95\x2a\x22\x6e\x08\x79\x66\xa2\xa0\xe0\x4a\x44\xa2\x68\xb5\xc4\x56\x6d\x4b\x6b"
		"\xc5\xb6\x56\x5f\xdf\xb6\xcf\xd3\xba\xd6\x0d\xd4\x52\xf7\x26\xd6\x05\x4d\x5d\xb0"
		"\xe2\xce\x9d\x93\x2e\x16\xb5\x8a\x68\xeb\x52\x6d\x25\x6a\x95\xaa\xaf\x06\x44\x21"
		"\xfb\x7d\x7f\xbf\x79\xf8\x05\xff\xf0\xe3\xd7\xeb\xef\xcc\x33\x77\x96\x33\x67\x66"
		"\xce\xcd\xc8\xa4\xcf\x0d\xe4\xa7\x59\x75\xe2\xdc\xc8\xa4\x2f\x88\xcb\x93\x1a\x9f"
		"\x24\x75\xd6\x95\xed\x4a\x36\x71\x92\x74\xb8\xca\xc2\x1e\x76\x51\x56\x9b\x03\x07"
		"\x71\x57\xf6\xb2\xeb\xcb\xea\x6d\x42\xf2\x06\x39\x88\xcb\x93\x1b\xa1\x3f\x5c\x1c"
		"\xc4\x49\x72\x05\x9e\x4f\xde\x5a\xce\x15\x41\xdc\x99\xed\x88\xe7\x53\xe3\x6f\x81"
		"\x83\x78\x64\x72\x1a\xca\x9f\xb5\xb5\x9e\xa7\x85\x2f\x71\x7c\xde\x95\xdd\x3c\xf8"
		"\x9c\xac\x72\x92\xe4\xde\xc1\x72\xc8\xfa\xdd\xea\xe4\x03\xa7\xdf\x25\xab\x9e\x9d"
		"\xd9\x9f\x9d\xea\x49\xd6\x7b\x6d\xe5\x20\x56\x3b\x14\xcb\x29\xb6\x43\xb1\x9c\x62"
		"\xbb\xf1\xb7\xd4\x6e\xe4\x91\x5b\xdb\xb9\x58\xcf\x62\x3b\x93\xbb\xb2\x0f\x63\xfd"
		"\x37\x65\xd7\x93\x83\x78\x53\xf6\x3c\xf4\x73\xc0\x7f\x20\x07\x71\x79\xd2\x02\xfd"
		"\xb1\xe2\x20\xee\xcb\x0a\xf8\xef\xb9\xd1\x16\x1c\xbe\xc4\xf1\x79\x75\x72\xfc\xe0"
		"\x73\xb2\xca\xe9\xca\x46\x0e\x96\x43\xd6\xef\x6e\xe5\x20\x56\x3d\x69\xab\x7a\x92"
		"\x1b\x9b\xfe\xe9\x16\x34\x1d\x63\x7f\x6c\x3b\x95\x1c\xc4\x6c\x23\xfc\x7f\xbc\xfb"
		"\x14\x72\x10\x9f\xd1\x74\x37\x74\x27\x88\x83\x38\x49\x96\xba\x15\x6d\xf3\xa2\x2d"
		"\x38\x7c\x89\xe3\xf3\x91\xc9\xd5\x83\xcf\xc9\x2a\xa7\xa5\xed\x77\x83\xe5\x90\xf5"
		"\xbb\x5b\x39\x88\x55\x4f\xda\xaa\x9e\xe4\x99\xc9\x0a\x8c\xd7\xb1\xd6\x97\xfd\xd8"
		"\xc0\xa9\xf8\xed\xec\xdb\x28\xe7\x2b\x36\x90\x5f\x4c\x0e\xe2\xbe\xec\x26\xf4\x3b"
		"\xf9\x4c\x72\x10\x57\x27\xbf\x45\xfb\xec\x62\xe5\xc9\x6c\x72\x10\xb3\xfe\x95\x85"
		"\x09\x6c\x43\x72\x10\x0f\x4b\x96\x41\xbf\x9b\x4d\x4d\x3e\x4f\xc1\x41\xdc\x93\xaf"
		"\xf0\xa5\x85\xd1\xa8\xcb\x14\xb2\x89\x4b\x0b\x07\xf8\x9e\xfc\x30\xd4\x79\x55\x0e"
		"\x6c\xe2\xae\xec\x28\x5f\x9e\xec\x88\x31\x79\x03\xfa\xe5\x28\x13\x27\x49\x83\xe7"
		"\x98\xec\xca\xfe\x49\x36\x71\x5f\xb6\x13\xca\x39\x08\xba\x32\x0f\x36\x71\x92\x3c"
		"\xeb\x4a\x0b\x33\xc1\x13\xc9\x41\x5c\x9d\xfc\x1c\xcf\x8f\x01\xef\x47\x0e\xe2\xf5"
		"\x59\x45\x1c\x3f\x03\xf9\x11\xe4\x20\x3e\x2b\xd9\x8b\x75\x8a\xbf\x05\x4e\xc5\x13"
		"\x92\x69\x68\x87\xe3\x61\xbb\x85\x1c\xc4\xc5\x7e\x3d\x16\x6d\x39\xd4\x83\x83\xb8"
		"\x2f\x6b\x47\xfb\x1f\x89\x7a\x57\x92\x83\xb8\x2b\x1b\x82\x7f\x0e\xc5\xd8\x7e\x17"
		"\xef\x3b\xc4\xc4\xe5\xc9\xd7\xfc\x40\x7e\x1f\xf0\x53\x64\xdb\xc6\xfb\xa1\x3d\x27"
		"\x1a\xe7\x3d\xd8\xc4\x3d\xf9\x71\x68\xb7\x89\xb0\x2f\x21\x9b\xb8\x2f\xdb\x10\xed"
		"\x0f\xc9\x1e\xaa\x07\x07\xf1\xb0\xe4\x1a\xb4\xc3\x0c\xbb\xa1\xa4\x6e\x15\x38\x88"
		"\x31\x97\xf1\xbe\x87\xc1\xa7\x4d\xcb\x81\x53\xf1\xdb\xd9\xea\x1c\xdb\x24\x49\xda"
		"\x1d\x38\x15\x77\x66\x17\xe7\x38\x2e\xba\xb2\x5f\x91\x53\xf1\x84\xe4\xa6\x1c\xfb"
		"\x69\x64\xb2\x0b\x39\x15\x9f\x95\x7c\x07\xe5\xef\x8d\x7e\xfa\xae\x81\x53\x31\xfd"
		"\x1b\xeb\x96\x24\x67\x92\x83\x78\x6c\x32\x1d\xf5\xdc\x0f\x6d\x35\x8f\x1c\xc4\x6f"
		"\x67\x7b\x6e\xf5\x3f\x23\xc9\x41\x4c\x1f\x3a\x90\x3f\xd0\xf8\x3e\xe0\x20\x66\x3f"
		"\x96\x27\xde\xd8\xf6\xec\x47\xf1\xdb\xd9\xec\x1c\x74\xec\x6b\x94\x33\x3b\x15\xb3"
		"\x9e\x49\x52\x8f\xdf\x3a\x38\xd6\x53\x0c\xdf\x98\x1b\xc8\xd3\x27\xff\x02\x7d\xdb"
		"\x91\x8a\x67\x26\x1b\xe2\xf3\x91\x49\x3d\x39\x15\x27\x49\x01\xbf\x3b\x13\x3c\x94"
		"\x9c\x8a\xd1\xd6\xf5\xd4\xac\xcf\x76\x27\xaf\x12\xb3\xad\xa9\xd9\x94\xdd\x42\x4e"
		"\xc5\x98\x33\x78\x7e\x0c\xea\xbc\x94\x9c\x8a\xd9\x97\xf4\x09\x63\x93\x49\xe4\x55"
		"\xe2\xa9\xc9\xaf\x73\x1c\x7b\x49\x32\x82\x9c\x8a\x3b\xb3\xf7\x72\xd4\xb0\x6d\xc0"
		"\xa9\x78\x6a\xf2\x8f\x5c\x4f\xfe\x38\xd8\x5e\x44\x4e\xc5\xac\x7f\x57\x76\x82\xde"
		"\x25\xdd\xc6\x0d\x18\x0f\x8d\x18\x1b\x17\x92\x53\x31\xdb\xad\x27\x7f\x58\xf4\x3d"
		"\x6c\x37\x71\x67\x56\x86\xb1\xd8\xc0\xb1\x84\xdf\x2d\x0b\xe2\xea\x64\x38\xe6\xd1"
		"\x74\xbc\xcb\x15\xe4\xf0\x25\x8e\xcf\xcb\x13\x1b\x7c\x4e\x5e\x9f\x9d\xef\x38\xb6"
		"\xd9\x77\xe0\xb0\x8d\x9b\x30\x7f\x76\x85\x66\x09\x39\x88\x39\x7e\xfa\xb2\xbd\xd8"
		"\xb7\xe4\x20\x4e\xb3\xb9\x39\x96\xc9\x71\x09\x4e\xc5\x6c\x07\x8e\xa5\xd2\xc2\x6c"
		"\x72\x2a\xee\xcc\xee\x8c\x7e\xaa\x2b\x3b\x85\x9c\x8a\x2f\x4b\x3a\xd0\x8f\x53\x50"
		"\x87\x66\xf2\x2a\xf1\xd0\x92\xf6\x7a\xfe\x56\x5f\x76\x1a\x79\x95\x38\xcd\x96\xe4"
		"\x38\xb6\x2b\x0b\x27\x91\x53\x31\xe6\x52\x8e\x63\x3b\x49\xe6\x90\xd3\x6d\xcc\x76"
		"\x9e\x62\x1c\x07\xc5\x76\x2e\x72\x9a\xd5\xa0\x9d\x77\x47\xdd\xe6\x92\x53\x71\x92"
		"\xb4\xe6\x1a\x9b\x76\xb7\xe2\xfa\xd4\x9a\x8a\x39\x5f\x92\x64\x56\x9c\x4f\x9c\x2f"
		"\x62\xce\xc9\x9e\xfc\x11\x78\xa7\x8b\xc9\x41\x5c\x9d\x4c\x46\xfb\x1f\x89\x77\x7c"
		"\x98\x1c\xc4\x68\x3b\x57\xac\x4b\x42\x0e\x62\xc6\x1b\x9c\x3b\xc5\xf7\xbb\x31\x88"
		"\x47\x26\xff\x72\xf4\xc9\x18\x1b\xe4\x20\x2e\x4f\xb6\x87\xff\x9d\xcc\xb9\x00\xdb"
		"\xed\x4d\x5c\x5a\xd8\x19\x7e\x7e\x12\x35\x64\x13\xf7\x65\x63\xa2\x3f\x64\xec\x01"
		"\xb6\x6d\x9c\xf8\xe2\xbc\x7c\x96\x6c\xe2\xae\xec\x8f\xb0\xf7\x68\xab\x75\xe4\x20"
		"\x4e\x92\x9b\x1c\x7d\x5a\x31\xce\xb9\x29\x88\xd7\x67\x67\xc5\x76\x28\x4f\x5e\x27"
		"\x07\x31\xc6\x7c\x8c\x13\xca\x93\xd5\xe4\x20\x1e\x96\xdc\x1e\xdb\xa1\x33\xfb\x3b"
		"\x39\x88\xe9\xe7\x59\x66\x92\xac\x25\x07\x31\xd7\x86\x9e\xfc\xa1\x58\x27\x53\x72"
		"\xd8\xc6\x35\x9e\x7e\xbc\x2f\x6b\x8d\xf1\x95\xb8\x2f\x1b\x81\x75\x6d\x5f\xce\x41"
		"\xb2\x89\x93\xa4\x1c\xcf\xf7\x67\x1c\x46\xb6\x6d\xfc\x06\x74\xd3\xe2\x1a\x0a\x0e"
		"\xe2\x91\xc9\x6d\xf0\x87\x0d\x8c\xf1\xc8\x61\x1b\x9f\x16\xc7\x03\x7c\x42\x8c\x33"
		"\xc5\x98\xdf\x5b\xe3\xcf\xfb\xc8\x41\x5c\xf4\xab\xb3\x50\xe7\x4b\x5d\xd1\xaf\x16"
		"\x19\xef\x17\xd7\xd9\xf5\xd9\x11\xe4\x20\xa6\xbf\xea\xcb\xf6\x81\xe6\x7d\x72\x2a"
		"\xe6\x3a\xcb\x77\xec\xc9\x8f\x8a\xeb\xac\x98\x71\x1d\x35\x7d\xd9\x81\xe4\xb0\x8d"
		"\xd7\xe2\x77\x77\xe3\x7a\x4f\x0e\x62\xc4\x11\x68\xb7\xd1\xe8\xbb\xca\x18\x5f\x88"
		"\xd1\x0f\x9e\xe3\xa9\xb8\x86\x72\xcd\x2f\xf2\x40\xbe\x11\x71\xc0\x86\xc0\x18\x12"
		"\x6c\x62\x8c\x47\x8f\xff\x1f\xd6\x64\x57\xe5\x38\x36\xc5\xe5\x49\x15\xd6\xe5\x6e"
		"\x32\xe6\x68\x95\x89\xd1\x86\x5b\xdb\x31\x41\xfb\xde\x96\x8a\xdf\xce\x2e\x87\x9f"
		"\x7c\x38\xae\x5b\xe0\x54\x4c\x7d\x75\xb2\x06\xbc\x30\xea\xc5\xc5\x38\xad\x04\xf5"
		"\xb8\x70\x6b\x9c\x56\x64\x8e\xb7\x61\xc9\x17\x18\x1f\xe7\x91\xc3\x36\xbe\x17\xf1"
		"\xeb\x47\xe0\xd3\xc9\x41\xcc\xb1\xc2\x77\x29\x4f\x0e\x20\x07\x31\x63\x59\xf8\x11"
		"\xd4\xad\x8e\x1c\xc4\x23\x93\xef\xc7\x38\xb8\x2f\x9b\x4f\x0e\xe2\xb1\xc9\xd1\xf8"
		"\xdd\x47\x61\xbb\x1e\x7e\xf2\xe8\x20\x66\x1b\x25\x49\x17\xe6\xf3\x00\x39\x88\x59"
		"\x0e\xdf\x9b\xbe\x9a\xe5\x88\x37\x65\xbf\x44\xf9\x3d\x28\x73\x05\x39\x88\xc7\x26"
		"\xa7\x3a\xc6\x7f\xf4\xcd\xe0\x20\x86\x2f\x8d\xe3\x78\x42\x32\x86\x1c\xc4\xf4\x3f"
		"\x18\x03\xfe\xb2\xa4\x39\xa5\xff\x11\x77\x66\x13\x63\xec\x8e\xb1\x07\xfd\xc4\x20"
		"\x66\xac\xc3\x3e\xa8\x4e\xfe\x4a\x0e\x62\xc6\x2d\x9d\xd9\xb7\xb8\x56\x93\x53\xf1"
		"\x84\x64\x81\x2b\xb6\xd7\x13\xe4\x20\xee\xc9\xe7\xfc\xd8\x64\x11\xd6\xc0\x95\x18"
		"\x0f\x39\x13\x8f\x4c\x8e\xe5\x78\xf2\x8c\x7f\xc1\x41\xcc\xb8\x9a\xfe\xba\xb2\xb0"
		"\xc8\x33\xae\xfe\x12\x3b\xc6\x41\xa5\x85\xef\xc5\xe7\x62\x94\x87\x72\x6a\x31\x96"
		"\xa7\xc4\xdf\x11\x73\x1c\xf6\x65\xcf\x06\xf9\x4b\x31\xfa\xd2\xf3\xf7\x92\xa4\xda"
		"\xb3\x5f\xc5\x49\x32\x0d\xe5\x56\x61\xad\xda\x8b\x6c\x5f\x62\xe8\x96\xa0\x8c\xdd"
		"\xe3\x73\x31\xe7\x12\xdb\x2a\x49\xfe\x8b\xf7\x3e\xd0\xc4\xf4\xc3\xa5\x85\x2a\x68"
		"\x77\x8a\xf3\x46\x9c\x24\x31\x7e\x65\x8c\x4b\xb6\x2f\xb1\x7f\xab\xcf\x9e\xc6\xbe"
		"\x2b\x3e\xff\x12\x7b\x8e\x57\xac\xbb\xf1\xb9\x98\x31\x74\x5f\xd6\xc1\x31\xe3\x18"
		"\x57\x89\xd9\x0e\x9b\xb2\x3f\x81\x0f\x8e\xed\x20\x66\x9b\x26\xc9\x18\xee\x13\xc8"
		"\x41\x5c\x9e\x5c\x8e\xdf\xdf\xce\x38\x2f\xc1\x41\xdc\x37\x7f\x54\x8e\x9a\x33\x9a"
		"\xbe\x70\xe0\x54\xdc\x93\x3f\xca\xd3\x2f\x23\x76\x22\x9b\x98\x75\x65\x7c\x86\xdf"
		"\x8b\xfb\x48\x71\x4f\xfe\x04\xcf\x75\x89\xed\x02\x36\x31\xe2\xa6\xd8\x9e\xdc\x1f"
		"\x33\x86\x12\x8f\x4c\xee\xe7\x7e\x20\x57\x1c\xff\xf7\x07\x31\xea\xe5\xb9\xef\x1f"
		"\x9b\x6c\x17\xeb\x28\x2e\x2d\x6c\xef\x19\x67\x73\x4d\x06\x9b\x78\x53\xf6\xbb\xb8"
		"\x3e\xc1\x67\xa2\x7e\xbf\x0b\xe2\x15\x6d\xd7\x62\x1f\xf8\xb3\x50\xdc\x23\x5e\x1b"
		"\xc4\x8d\x4d\xd7\xe1\xf9\x17\x81\x6b\x38\x38\x88\xfb\xb2\x3a\xb4\x57\xe2\xb9\x9e"
		"\x83\x4d\xbc\x29\xfb\x0c\xf5\xad\xf3\x1c\xef\xe0\x20\xee\xc9\x0f\x41\x7d\x1e\x70"
		"\xf4\x49\x60\x13\x17\xd7\x39\xae\x9f\xbb\x62\x6c\xae\x0e\xe2\x15\x6d\x7f\xc5\x3e"
		"\x73\x94\x67\x2c\x03\x0e\xe2\xca\xc2\x4f\x62\xdf\x70\xed\x03\x9b\xb8\xdf\xae\x45"
		"\xf9\x8f\xd2\x67\x91\x4d\x5c\xd6\x70\x87\xe7\x19\x04\x7f\x07\x6c\xe2\xd2\xc2\x9d"
		"\xd0\xbf\x1a\xd7\x19\xb0\x89\x93\xe4\x7a\xbc\xc7\x2a\xb7\xa6\x37\x7b\x0a\x6c\xe2"
		"\xf2\xe4\x7b\x1e\x73\xd9\x61\x6e\xa2\xce\xdf\x33\x71\x57\x76\x1e\xca\xff\x05\xe6"
		"\x65\x15\xd9\xc4\xe8\x37\xcf\x7e\xe6\x3c\x01\x9b\xb8\xdf\x6e\xf3\xdc\xcf\xad\xcf"
		"\x0e\x42\x3d\x6f\x33\x71\x69\xe1\x56\xcf\xf5\x7f\x53\xd6\x46\x36\x71\xbf\x5d\xed"
		"\x8b\xeb\xf6\x6a\xb2\x89\xbb\xb2\xff\xf1\xdc\xa7\x56\x27\x8f\x91\x4d\x8c\xf5\xd9"
		"\xaf\xc9\x7e\x9f\xe3\xde\x82\x6b\xb5\x18\xf1\x26\xfa\x61\x0c\xf6\x46\x4f\xc4\xd8"
		"\x53\x8c\xb8\xde\xf3\xfd\xe8\xcf\x18\xe3\x8b\x2b\x0b\xdf\x40\xbc\xf4\x0d\xf4\xed"
		"\x0a\xb2\x89\x11\xf3\x79\xfa\x6e\xbe\x23\xe3\x3f\x31\xf6\x97\xd1\x07\xad\xe9\xcd"
		"\xcf\xe0\x5e\x53\xdc\x97\xcd\x89\x6b\xed\xcc\x64\x31\xc6\xeb\x1c\x13\x63\xdf\x86"
		"\x76\x38\xcf\x15\x7f\x97\x7b\xb8\x22\x97\x16\x4e\xf6\x8c\xb9\x69\x0f\x36\x71\x65"
		"\xe1\x6c\xf4\xcb\x4f\x5d\xf1\xfc\xe0\x6c\x13\x0f\xe4\xe7\xe3\xf9\x3e\xb0\xbd\x8e"
		"\x6c\x62\xc4\x8f\x9e\x7b\x00\xc6\x18\x8c\x25\xc5\xf7\xb6\x4d\xf3\x2d\x6d\x3b\xb8"
		"\x75\xcb\xdb\x66\x80\x4d\x4c\x5f\xc9\xd8\x05\x73\x8a\x7e\xd3\xc4\x58\xf3\x3c\x63"
		"\x4d\xc6\x29\x5c\xff\xc4\xd8\x8f\x7a\xc6\x76\x78\x77\x57\xdc\x9b\x16\x99\xed\x49"
		"\xa6\x0f\x67\x7b\x8a\xab\x93\x93\xb6\xea\xb9\xff\x3b\xc9\xc4\xac\x3f\x19\xf3\x27"
		"\x61\xfd\xc5\x7d\xd9\x02\xcf\xdf\x47\x4c\x8b\x58\x7e\x81\x89\x7b\xf2\x3f\xc3\x7b"
		"\x3d\x82\x3a\xdc\x81\xfa\xfc\xcc\xc4\xa5\x85\x6b\xa0\x7f\x03\xb6\xec\xa3\x6b\x4c"
		"\x5c\xd6\x70\x2b\xfa\xf1\xd3\xd8\x56\x60\x13\x0f\xe4\x6f\xc3\x7b\x75\x3a\xb6\x17"
		"\xd8\xc4\xa5\x85\x25\x9e\xeb\x00\xf7\x85\x60\x13\x73\xde\xf1\x77\xb8\x77\xe4\xbc"
		"\x13\x0f\xe4\x9b\x51\xce\xf2\xb8\x16\x82\x4d\x5c\x59\xf8\xa1\xe7\x78\xe1\x3e\x15"
		"\x6c\x62\xf4\x7f\x1c\xcf\x3c\xcf\x00\x9b\x78\x20\x7f\x05\x9e\x3f\x13\xfb\x1a\x6c"
		"\xe2\xf2\xe4\xe2\x38\x8f\xe8\xa7\xc1\x26\x1e\xc8\xff\xd8\x17\xf7\x0a\x3f\x20\x9b"
		"\xb8\x27\x7f\x16\x7e\x97\xe7\x94\x53\xc8\x26\x46\xcc\x81\xf2\xef\x46\x99\xeb\x73"
		"\x8c\x3f\xc4\x88\x8d\x3c\xcf\xf2\x86\x96\x34\xd4\x33\x4e\x12\x97\x16\x2e\x88\xe5"
		"\x33\xae\x04\x9b\x38\x49\xae\xf4\xf4\x37\x6b\xb2\xbf\x91\x4d\x9c\xf9\x5f\x7b\xfa"
		"\x69\xee\x71\xc0\x26\x1e\xc8\xff\x9a\xeb\x51\x1c\x4b\x60\x13\x2f\x68\xba\x26\xb6"
		"\x73\x63\xd3\x1d\x64\x13\xf7\x65\xdf\xf1\xf4\xd1\xa8\x1b\x62\xd8\xef\x98\xb8\xb4"
		"\xb0\x0c\xbf\xfb\x9e\xbb\x28\x6b\x59\x05\x36\x71\x92\xb4\x79\x9e\xb9\xf0\xb7\xc0"
		"\x26\x1e\xc8\xdf\x83\xfa\x6f\x44\x9f\xbe\x40\x36\x31\xf6\xf0\x78\xfe\x8e\xe3\x7f"
		"\x83\x4d\x5c\x9e\xfc\x28\xfa\x4f\xfa\x23\xb0\x89\x7b\xf2\xe7\x78\xc6\xfa\xc5\x7d"
		"\xdb\x39\x26\x4e\x92\x5f\x63\x1d\xe7\xb9\x57\x33\xd9\xc4\xf7\xb7\x2d\xf7\x2b\xda"
		"\x2a\xe3\x5a\x0e\x36\x31\xc7\x7f\x71\x9f\xb1\x82\x6c\x62\xec\x03\x50\x9f\xe7\x5c"
		"\x71\x7e\x32\xf6\x2f\x32\xe3\x90\xea\xe4\x32\xb6\x6d\x5a\x8c\x2f\x8b\xcc\xfd\x57"
		"\xb7\x1d\x62\x35\x85\x0b\x51\x9f\x4f\x5c\x49\x47\x03\xda\xe8\x68\x8c\xdb\x8f\x5d"
		"\xcf\xc2\xe9\xd8\xa7\x0d\x75\xbd\xbe\xca\x57\xd7\xed\xcb\x3d\x1e\xd9\xc4\xd0\x04"
		"\x69\x60\x1b\x64\xcb\x7d\x9c\xca\x2c\x2d\xec\x88\xf1\x50\x1b\xb9\xdf\x4a\x7d\x69"
		"\xeb\x04\x1b\x97\x3c\x8c\x39\xfb\x15\xcf\xf3\x1e\xfa\x4d\x9e\x65\x8a\xa1\x31\x69"
		"\x60\x6b\xb2\xdd\xae\xf0\x4f\xd7\xb5\x6e\x2a\xfc\xe3\x95\x39\xce\xd1\x7e\x9b\x64"
		"\xc5\xb3\x93\x0f\xe2\x7e\x12\xf1\x0a\x39\x88\xa1\x09\xd2\xc0\x36\xc8\xb6\xaa\xe1"
		"\x05\x57\xd6\x3c\xcb\xde\xc9\x6f\xef\xaa\x1a\x56\xba\x8a\x0e\x1f\xf7\xee\xe0\xb0"
		"\x8d\x5f\x08\xd2\xf4\xe4\x6f\x47\x9b\xcc\x89\x3c\x90\xff\x69\xb4\xe5\xfb\x0e\x6f"
		"\xbf\xd6\x95\xb4\x1f\x1a\xcb\x04\x07\x31\x34\x41\x1a\xd8\x06\xd9\x0e\x4b\x8e\x80"
		"\xbf\xd8\x3f\xbe\x17\xd7\x34\x9e\x6f\xb0\x9e\x3c\x7b\x12\x43\x13\xa4\xe9\xcc\x0e"
		"\xc0\xef\xd6\xc7\x76\xa6\x1f\xef\xb7\xa9\xb1\x1d\x38\xe7\xcb\x93\x3d\x63\x5b\x31"
		"\xf6\x11\x33\x66\x96\x06\xb6\x41\xb6\xef\xe4\xff\x37\xae\x43\xec\x97\x24\x69\x8c"
		"\x7b\x43\x6a\xc0\x41\x0c\x4d\x90\x46\xfa\xaa\x86\x5f\x0f\xbe\x6f\x69\xe1\x11\xa7"
		"\xfa\x6c\x57\xb8\x63\xb0\x7c\xb2\xde\x97\x1a\x95\x43\x5b\xbd\x6f\x4f\xfe\xc5\xc1"
		"\x3a\x57\x27\x0f\x0e\xd6\x99\xac\xf7\xa5\x46\xed\xb9\x39\xff\x9a\x53\xff\xd2\x87"
		"\xaa\xad\x3a\xb3\x4d\x4e\x6d\x45\x56\xff\x52\xa3\xf6\xa7\xad\xfa\x6b\x83\x3d\x3a"
		"\xd8\xbf\x9c\x03\xea\x5f\xb2\xfa\xa5\xa8\x29\x8e\x07\xb2\xc6\x3c\xdf\x45\xe3\x8d"
		"\x65\x6a\xfc\x90\x35\xe6\xa9\xd1\x38\xa1\xad\xc6\x36\xdf\x45\x63\x92\xf5\xd4\x98"
		"\x24\x6b\x6c\x53\xa3\x39\xc5\x36\xd4\x5c\x60\x9b\x68\x2e\x90\x35\xd7\xa8\xd1\xdc"
		"\x64\xdb\x6a\xae\x91\xb9\xfe\x70\x2f\xdf\x97\x5d\x08\x3f\xf9\x36\xea\x7f\x24\xea"
		"\x79\x8f\x3b\xea\xd5\x0f\xdc\xdc\x65\xb3\xad\xa4\xfd\x3a\x72\x10\x43\x13\xa4\x81"
		"\x6d\x90\xed\x40\x7e\x25\xde\xf7\x68\xdb\x98\x7f\x1c\xfd\xf0\x9c\xcb\x6a\x8f\xb1"
		"\x87\x17\xad\x20\x07\x31\x34\x41\x1a\x9e\x3f\xf3\xcc\x9c\xe5\x9c\x3e\xaa\x15\xbf"
		"\x35\x37\x6a\xc0\x41\xcc\x18\x59\x1a\xfe\x06\xe3\x75\xfe\x7b\x68\xc3\xc5\xee\x81"
		"\x45\x27\xc4\xfa\x80\x83\x18\xff\x2f\x48\xa3\xf2\xc7\x26\x93\x06\xf5\x4f\x4f\x6a"
		"\x1c\xd4\x93\x55\x3e\x35\xaa\x7f\x5f\x56\x3d\x58\x9f\xd1\x85\xaf\x0c\xd6\x87\xac"
		"\xfa\x53\xa3\xb6\xa2\xad\xde\x97\x1a\xbd\x2f\x59\x6d\x45\x8d\xda\x93\xbf\xab\xf6"
		"\x24\xaf\x68\x7b\xcd\xb5\x4d\x3a\xd6\x9e\x6b\xbb\x10\x71\xfd\x8b\x58\x73\x8e\x03"
		"\xcf\xe1\xf3\xf0\xa5\xe7\x41\xcf\xcf\x6a\x5a\xe9\xe6\x37\x35\x1a\xe7\x13\x38\x6c"
		"\xe3\xdb\x1c\xcf\xc5\x8b\x9a\xdb\x82\x98\xb1\xd7\xa9\xaf\xce\x8b\xe5\x80\x83\x58"
		"\xfa\x95\x6d\xcb\x06\xf5\x45\x2e\x96\x3f\xbf\xa9\x75\xb0\x7c\xb2\xea\x46\x8d\xea"
		"\x43\x66\x2c\x55\x5a\xa8\xc1\x58\xba\x70\xeb\x19\xe3\x18\xeb\x5a\xf7\x43\xfb\x38"
		"\xbb\x2d\xc7\xbb\xa1\xfe\x96\x73\xb9\x17\x49\xca\x93\x9d\x31\x26\x7f\xc2\xe7\xa9"
		"\x9e\x17\xcf\x21\x8b\x7a\xc6\xee\x2a\x87\xb1\x64\x75\xdd\x10\xe3\xba\x37\xb4\xe1"
		"\x2a\xf0\x28\x3c\xff\xbe\x15\x7d\xc2\xae\x96\xd5\x7e\xcf\x8a\x73\xbf\xc8\xd0\x04"
		"\x69\x60\x1b\x64\xfb\x99\xbf\x0f\xef\x58\x89\xdf\x3d\xdb\x86\xb7\xff\xd6\x0d\x2c"
		"\x1c\x6b\x65\xcd\x0b\xe0\xa7\x2e\x8a\x67\x4a\x15\xed\xa7\x91\x83\x18\x9a\x20\x0d"
		"\x6c\x83\x6c\xbb\xed\xf7\xae\xdb\xb6\x37\xae\x87\x35\x05\xc4\x6d\xcd\x75\x78\xb6"
		"\x1f\x34\xf0\x2d\xeb\xe0\x43\xea\x0e\x27\x07\x31\x34\x41\x1a\xd8\x06\xd9\x6e\xb0"
		"\xdb\x5d\x77\xcb\x08\xfc\xd6\x50\xf0\xa5\xae\xaa\x79\xa2\x8d\x68\xf8\x28\x30\xae"
		"\xaa\x6a\x98\x62\x95\x85\x1d\xe2\xdd\x9c\x18\x9a\x20\x0d\x6c\x83\x6c\x3f\xce\x8e"
		"\x85\x1f\xe1\xb9\xf1\x57\xc3\x16\xdf\x0a\xdb\x11\xf8\xcd\x31\xd8\x3f\xbe\xe2\x7a"
		"\x6b\x77\xb1\x57\xb2\x3f\xa0\x2d\xaf\xc2\x98\x3c\x04\xf5\xbf\x96\x1c\xc4\xd0\x04"
		"\x69\x60\x1b\x64\x8b\x32\x83\xca\xec\x6f\xa9\xf2\x95\xad\xdc\x13\x2e\x4b\x33\x3f"
		"\x1e\xeb\xfb\x8e\xf6\xdd\xec\xe2\xfa\x81\x7c\xe2\xfb\x5b\x26\xda\x65\x49\xf3\xd3"
		"\x3c\xef\x11\x43\x63\xd2\xc0\xd6\x64\x8b\x35\xdc\xf7\xe4\xfb\x03\xd7\xb5\xae\x75"
		"\x87\xfb\xac\x76\x7b\xac\x65\x33\x30\x2f\x26\xc1\xd7\x7d\x95\xf7\x83\x88\xd3\x26"
		"\x99\x18\x1a\x93\x86\xeb\xbf\x6c\x4b\x5b\xe7\xf9\xb2\x86\x0a\xfc\xc6\x0d\x8e\xfb"
		"\x97\x9e\x85\xe3\x51\xcf\x3f\xe1\xbd\xeb\x7d\x79\xdd\x78\xde\x83\x90\x4d\xcc\x7d"
		"\x8d\x34\xb0\x35\xd9\x96\x35\x1c\x01\xdb\xaf\x62\x9c\xfc\x07\xbe\x74\x72\x3c\x5f"
		"\xe5\x9e\x34\xf3\x07\xa3\xfc\x7d\xd1\xce\xcf\x93\x4d\x0c\x8d\x49\x03\x5b\x93\x6d"
		"\x6f\xed\x18\xbc\xcb\x9e\x28\xb3\x0a\x31\xd7\x26\xfc\xbf\x1c\xef\x29\xd1\x4e\x35"
		"\x7e\x60\x61\x0e\xf6\x5f\xa0\xfe\x35\x26\x86\x26\x48\x03\x5b\x93\x6d\xe6\xd7\xc4"
		"\xb3\x98\x81\x85\x7b\xfb\x7e\xfb\x1d\xde\x61\x0e\xd7\x5c\x3f\xa2\xe1\x4f\xd1\x17"
		"\xf5\x2c\xfc\x0a\x39\x88\xa1\x09\xd2\xc0\x36\xc8\x96\xe7\xc2\x95\xad\x39\xac\x29"
		"\x87\xf8\x8f\xb3\x7a\xcc\x97\xe3\xf0\x7c\x77\xff\xb9\x5d\xe2\xfa\xd6\x9d\x80\xe7"
		"\xb5\xe4\x20\x86\x26\x48\xc3\x3b\x1e\xd9\xae\xcf\xbe\xc8\x15\xcf\xc9\x47\xc7\x3d"
		"\x6b\xb7\x1d\x63\x9f\xdb\xe7\x6e\xe7\xc2\x38\x8c\xd5\x13\xe3\xf9\x2a\x38\x88\xa1"
		"\x49\xa5\x81\x6d\x2a\xdb\xf5\xd9\xa8\x1c\xdf\x77\x83\xbd\xbf\x75\xef\xc4\x7d\xe6"
		"\x5a\x77\x4e\xf2\x13\xc4\x46\x8d\xe8\x97\xe7\xc9\xa9\x18\xfa\x54\xfa\x4d\xd9\x76"
		"\xf0\xab\xc7\xc3\x9f\xff\x13\xeb\xe6\x09\x71\x2f\x3a\x90\xff\x4f\x71\xcf\xd9\x7a"
		"\x9c\x71\x6c\x72\x8e\x88\xa1\x09\xd2\xc0\x36\xc8\x96\xe7\x96\x3c\x8b\xee\xcb\xba"
		"\xd0\x3f\x77\xc7\x33\xa0\xb2\xe6\xa1\xe8\xe7\xbb\x83\x98\xe7\x99\xd2\x94\xb4\xff"
		"\x11\x6d\xee\x78\x2e\xec\xd8\xc7\xbc\x9f\xea\xca\xba\xc9\x41\x0c\x4d\x90\xa6\xa2"
		"\xbd\xdc\x77\xb7\x1c\x68\xfd\xf6\x0a\xfa\xba\x16\xe3\x79\x2a\xea\xf0\x02\xd9\xc4"
		"\xd0\x98\x34\x03\xf9\x3a\xcf\xb1\x3e\xbc\xfd\xc1\xad\x77\xb8\x7b\x82\xef\xd9\x7a"
		"\x87\x5b\x64\x9e\xcf\x4a\x83\xb9\xe5\x8b\x39\x18\x3f\xc7\x7b\xef\x8d\xf9\xb8\x2b"
		"\xe6\xe9\x1c\xb2\x89\x39\xff\xb6\x69\x46\xfa\x8a\xf6\xbd\x18\xf7\xa3\x9c\xa1\xd1"
		"\xb6\xb8\x07\x1c\x6a\x62\x68\x4c\x9a\x11\x0d\xef\x62\x1c\x4f\xc1\xda\x31\x31\xc7"
		"\x79\x52\xd9\xba\xbf\xdd\x50\xd2\xfa\x34\x38\x88\xa1\x09\xd2\xc4\x7c\x8e\x75\xd3"
		"\xed\x97\xd9\xbe\x39\xee\x67\x38\x57\x2f\x4b\x3a\x66\xf0\x4e\x4d\xcc\xf6\x94\x66"
		"\xc7\xc2\x7e\x78\xff\x06\x7b\x2f\x5b\x83\xfd\xdd\x13\xf1\xee\xf5\xb9\xec\x2f\x3c"
		"\x3b\xcf\x55\x35\xcf\xb4\xf3\x93\xbf\xac\xe2\x79\x93\x18\x9a\x54\x1a\xd8\x06\xd9"
		"\x6e\xce\x0f\xc7\xd8\x9b\x8d\xf7\x7d\xc2\xbd\x92\x5d\x8f\xba\x1e\x1d\xe3\x01\x70"
		"\x2a\x86\x26\x6c\xd3\x7c\x96\xeb\x6e\x99\x85\x79\x7b\xe9\xe0\xd8\x63\x2c\xb0\x29"
		"\xbb\x32\xd7\xeb\x67\x5a\xf1\x1c\xec\xca\x54\x0c\x7d\xfa\x65\x3d\xc7\x30\xcf\x25"
		"\xb8\xc6\xf3\xee\x92\xef\x01\x0e\xe2\x43\x92\xb7\xa2\xcf\xaa\x6a\xfe\xb6\x71\x9c"
		"\x67\xb5\xfb\x59\x6f\xed\x99\x71\x8d\xe3\x5d\x76\xe6\x17\xf1\x79\xaa\xe7\xd0\xa7"
		"\xd2\xef\x5c\xd8\x17\xfe\x7f\x1f\xf4\xdf\xe9\xf1\xdd\xd8\xce\x3c\xc3\x00\xa7\x62"
		"\x68\x82\x34\x3c\xff\x62\x9e\x0f\xcf\x6f\x46\x25\x65\x8e\x77\x5b\xcc\x3f\x00\x07"
		"\x31\xef\x62\xa4\xd9\x98\x1f\x0d\xff\x35\x1d\xe3\xed\xa3\x30\x2e\x79\x19\xed\x80"
		"\x18\xbe\x6e\x24\x9e\x1f\x52\xf4\x6b\xcd\xfb\x90\x83\x18\x9a\x74\x9b\x66\x74\x90"
		"\x6d\x8c\xb7\xeb\xa6\xc7\xbb\xb4\x18\x6f\x0f\xf2\x8a\x5c\xdf\xba\x1c\x7e\x2b\x84"
		"\xe2\x3e\xdc\x83\xef\x09\xc5\xf5\xdd\x63\x5c\xfd\x39\xac\xc9\xbe\x86\x76\xf6\xd0"
		"\x7f\x44\x4e\xc5\xc5\x9c\x8f\xa2\x9e\xf7\x1d\x2a\xa7\x33\xfb\x16\xc6\x68\x3d\xfc"
		"\xe4\xde\xb1\x0d\x79\x6f\xc8\x35\x09\xcf\xd3\x2f\x3f\xe7\xfd\x3b\xd7\x84\x67\xb2"
		"\xd3\x73\xfd\x2d\x33\x8c\xf5\x06\xa7\x62\xf8\xe8\xf8\xbb\xf4\x73\x13\x92\xae\x1c"
		"\xfd\x35\xcf\xfa\xdf\xcc\x3a\x72\x9c\xef\x3b\x16\xf2\x58\x1f\xdb\xe2\xd8\xeb\xf5"
		"\x4b\xc8\xa9\x18\x9a\x54\x1a\xd8\xa6\xb2\x45\x99\xa9\xca\xc4\x9e\x3c\xc7\x98\x99"
		"\xbe\x93\x77\x0a\xb4\x1d\x9b\x7c\x15\xed\xbc\x11\xfd\x78\x14\xde\x71\x3a\x39\x15"
		"\xf3\xae\x41\x1a\xd8\xa6\xb2\x9d\x91\x7c\x33\xd7\x5b\xdb\x10\xfb\xb0\x33\x7b\xaf"
		"\x9e\x9a\x4f\xf2\x23\x60\x7b\x1d\xfc\xed\x6c\xc6\x96\xe4\x54\x0c\xcd\x2a\x69\x60"
		"\x9b\xca\x96\x2d\x52\xd2\x7e\x18\x34\x7b\x45\x66\x9d\x87\x25\x07\x85\x34\x5b\x52"
		"\xcf\x3a\xf0\x6e\x84\xe7\x53\xe2\x62\x1f\xcd\xc4\x1c\x69\x0d\xaf\x65\x87\xe6\x78"
		"\xa7\xcd\x7b\x5a\x70\x2a\x2e\x6a\x8e\xc1\x9c\xfa\x4d\x18\x95\xdc\x89\x76\x26\x5f"
		"\x16\xfb\x94\x77\xc8\x5b\xfc\x15\xb1\x4f\xc5\xd0\xa4\xd2\x14\x6d\x19\x7b\x4f\x0b"
		"\x93\x92\xca\x98\x1b\x50\x5a\x38\x20\xe6\x33\x0c\xe4\x8f\xb5\xda\xc2\x4e\x31\x9f"
		"\x41\x0c\x4d\x2a\xcd\xe4\xe4\x9a\x1c\xcf\x2e\xc7\x25\xbb\xe2\xf9\x5f\xa3\x9e\xeb"
		"\x0c\x38\x15\x43\x93\x4a\xf3\x66\xd6\x1e\xcb\xff\x24\xdf\x80\xf8\xe0\x23\xc4\x0d"
		"\xc7\x63\x2e\x9c\x41\x4e\xc5\xd0\xa4\xd2\x4c\x4e\x1e\x40\x9b\x37\x22\x26\xff\x09"
		"\x39\x15\x23\x76\xc9\xf5\xdb\x3c\x8c\x9d\x53\xc9\xa9\xb8\xe8\x37\x1a\xb7\xe6\x3f"
		"\x24\x49\xbf\xb9\x78\xa7\xf7\x4a\xf6\x0a\xca\x99\xc5\xb3\xcb\x00\x4e\xc5\xf4\x51"
		"\x65\xcd\xcc\x0f\x59\x14\x3e\xcc\x8f\x75\x1c\xab\xcc\x6d\x00\x07\x31\x7d\x94\x34"
		"\x5c\xe3\x38\xd7\x98\xa7\x36\x2e\xd9\xdf\x95\xd7\x4d\x8d\xf7\x01\xe0\x20\xe6\x5a"
		"\x26\x0d\xe3\xde\x9e\x85\x53\x63\x9e\x03\xe3\x5e\x71\x4d\xe1\x0a\xf8\xf0\x11\x58"
		"\x6b\x56\xa1\x2f\xbe\x85\x32\x27\x30\x37\x81\x1c\xc4\xd0\x04\x69\x36\xe7\x7f\x80"
		"\x35\xa3\x86\x71\x20\xef\xf8\xb0\x56\x4d\xc4\xbc\xbf\x20\xde\xf7\x89\xa1\x09\xd2"
		"\x0c\x4b\x9a\x10\x43\xee\x88\xe7\x27\x93\x83\xf8\x8f\xd9\x63\x39\xd6\xad\xaa\xd9"
		"\xc7\xb9\x49\x66\x7e\xe1\xf9\xc9\x2c\xcc\x8b\x43\x8c\xe3\x12\x9c\x8a\xa1\x4f\xa5"
		"\x1f\x9b\x3c\x8f\xb1\x74\x30\xe2\x99\x79\xd1\xd7\x93\x2b\x3a\x66\x93\x53\x31\x34"
		"\xa9\x34\x9c\xef\x3c\x37\xef\xb7\x6f\x30\xa6\xcc\x55\xd7\xed\x0f\xbf\xba\x20\xc6"
		"\x97\x62\xfa\x01\x69\x58\x1f\xc6\xcf\x3d\xf9\x33\xd0\x67\xab\xeb\x7b\x16\x4e\x36"
		"\x9e\x23\x82\x57\x89\x8b\xfd\xbb\x97\x15\xef\x52\x1b\x50\xfe\xde\xd8\x83\x9c\x6a"
		"\xb7\x66\xaf\xd7\x77\xb7\xec\x06\x7f\xb2\x80\xbc\x4a\xcc\xbc\x0b\x69\xe2\xd9\x31"
		"\xf6\xd4\x1c\x43\xcc\xb1\x39\x79\xd9\xee\x88\x43\x4e\x24\xa7\xdb\x78\x65\x2a\xcd"
		"\xdb\xd9\xf2\x58\xe7\xcc\x33\x4f\xf4\x89\xdc\xc9\x8f\xf2\x77\x8f\x22\xa7\x62\x68"
		"\x52\x69\x8a\xe7\xce\xcc\x21\x99\x65\xf7\x67\x07\xc0\x0f\x33\x3e\x3e\x92\x9c\x8a"
		"\x8b\xf3\x7d\x77\xac\xe5\x8d\x36\x23\xf9\x65\x6e\xde\xa3\xbb\x5b\xdf\xba\xe3\xc8"
		"\xa9\xf8\xbd\x6c\x3c\xf6\xfb\xb3\x30\x37\xbf\x11\xef\x33\xb8\xde\x21\xb6\x77\xa3"
		"\x0b\x07\x63\x1f\x74\x04\xdf\x89\x1c\xc4\xd0\x04\x69\x60\x1b\x64\xfb\x49\xbe\x6e"
		"\x6b\xde\xda\x2d\xe0\x29\xf1\xec\xbe\xaa\xe1\x12\x72\xd8\xc6\x75\x41\x1a\x8e\x55"
		"\x3e\xef\xf5\xab\x5d\x67\x36\x13\x71\xc2\x91\xf1\x6e\x09\x1c\xc4\x1c\xc3\xd2\xbc"
		"\x93\xff\x9e\xe3\x18\xe1\x99\xed\xa8\xe4\xc4\x18\x67\xf3\xdc\x1e\x1c\xc4\xd0\x04"
		"\x69\x3e\xb7\x55\xa8\xff\xa1\xbc\x3b\xce\x95\xb4\xdf\x8c\x58\xf7\x30\xe3\xfc\x06"
		"\x07\x31\x34\x41\x9a\x9a\x42\x17\x62\xe0\xfd\x8d\x31\x08\xcf\x4f\xab\xeb\x0e\x45"
		"\x3b\xfd\x8e\x1c\xc4\xd0\x04\x69\x4a\x5b\xbf\x82\xd8\xfe\x6b\xf0\xb1\x33\xe3\x19"
		"\xcb\xc0\xc2\xfd\x19\xd3\xc6\xb3\x14\x31\x34\x26\x4d\x7f\xcb\xce\xbe\xd7\x7f\x2d"
		"\xde\xf7\x54\xb6\x8e\x8a\x7b\x0a\x9e\x89\x81\x4d\x0c\x8d\x49\xc3\xbc\x0e\xee\x0b"
		"\xf1\x5b\xf1\x6c\xb3\xaa\x79\x3f\xb4\xe1\x5d\x64\x13\x33\xdf\x43\x9a\xea\xe4\xed"
		"\x98\xaf\xfb\xb9\xfd\x03\xb1\x68\x9f\x63\x7e\xe0\x67\xfe\x8f\xe4\x20\x86\x26\x48"
		"\x53\x5a\xf8\x3d\x7c\x51\x03\xd6\xcd\xf7\x63\x4c\xdb\xb3\xf0\x70\xc4\xd5\xaf\xc6"
		"\xd8\x55\x0c\x4d\x90\x86\xfb\xd0\x8a\x8e\x99\x28\xe7\x1d\x37\xb4\xe1\xd6\x38\x4e"
		"\x4a\x0b\x6f\x91\x83\x98\xfb\x50\x69\x7a\xf2\xe7\xc3\x5f\x1d\x19\xcf\xa0\xc0\x41"
		"\x9c\xf9\x6b\xe3\x39\xd2\x90\xf6\x97\x30\xae\xce\x8e\xe7\x84\x23\x1a\x9e\x62\x4e"
		"\x60\xcc\xc3\x60\x3e\x0e\x38\x88\xa1\x09\xd2\xc0\x36\xc8\xb6\x27\xff\x40\x1c\x0f"
		"\x3c\xc7\x67\x0e\x6a\x31\x3f\x67\x0d\x39\x88\xa1\x09\xd2\xf0\x5e\x85\xef\xb2\xc5"
		"\x3f\xcb\x7b\x78\x57\xcc\x33\x7c\x21\xe6\x3e\x88\xa1\x09\xd2\x74\xb7\x54\x7b\xfa"
		"\xd8\x8d\xf9\x87\xb6\x9e\xb3\x1f\x82\xfa\x3c\x49\x0e\x62\x68\x4c\x1a\xf6\x57\x79"
		"\xdd\x64\xf4\xe3\xd2\x78\x57\xce\xbc\x14\xd6\x8b\x77\xe5\x62\xf6\x97\x34\x65\x0d"
		"\xc3\x10\xe7\x4f\xc6\xdc\x61\x7e\x32\xef\xc4\xf7\xd9\x7a\x37\xc6\x3b\xf1\x22\x43"
		"\x63\xd2\x70\x2f\x51\xd9\x7a\x10\xe6\xeb\xa7\x39\x9e\xa5\x27\xc9\x94\x78\x8e\xc2"
		"\x33\x70\x31\x73\x3a\xa4\xe1\xfd\x0c\xd7\x11\xee\xcf\xaa\x93\xd7\x62\xfd\xd3\xec"
		"\x25\x72\x10\x43\x13\xa4\xf9\x38\xfb\x29\xc6\xdf\x4c\x63\x9c\xc2\xba\x32\xbe\x62"
		"\xbe\x3a\xdb\x53\x0c\x4d\x90\xe6\x9d\xfc\x31\x8e\xf1\xc6\x87\xf9\xbd\x1d\xef\x54"
		"\x8b\x79\x7a\x63\xc8\x41\x0c\x4d\xd8\xa6\x99\x19\xfd\xcc\x90\xf6\x1b\x63\xee\x7a"
		"\x31\xdf\xe9\xde\x98\x53\x24\xe6\xdc\x97\xa6\x78\xa7\xcb\x5c\xc7\xb3\x62\x2e\x19"
		"\xcb\xe4\x5d\x02\x73\xc9\xc4\x8c\x69\xa5\x61\xde\x45\xb1\x1f\x8f\x75\xc5\xdc\x87"
		"\x22\x17\xfd\xe4\x3e\x18\x7b\xff\x76\xc5\xfb\x00\xe6\xbd\x0f\xb8\xd7\x9a\x0e\xc3"
		"\x9a\x3e\xde\x4e\x5e\xf6\x3e\x39\x15\x43\x93\x4a\x33\x32\xa9\x8f\xfb\x2e\xcc\x61"
		"\xcf\xd8\xb8\xaa\xa1\x0e\xf3\x73\x47\x72\x10\x33\xc7\x51\x9a\x9a\xc2\xbd\xf0\x39"
		"\x7b\x63\x7e\x1e\xe0\x11\xf7\x38\x96\xc9\x3d\x19\x63\x20\x31\x34\x41\x9a\x61\x49"
		"\xbf\xe3\xf9\x43\x75\xdd\x28\xcf\xf9\xd9\xdd\x32\x86\xf7\x10\xe4\x20\x86\x26\x48"
		"\xd3\xdf\xb2\x1f\xfa\x9d\x73\x7e\x13\xe6\xd0\x4e\xbe\xa4\xa3\x1c\xeb\x66\x89\x07"
		"\x9b\x18\x1a\x93\x06\x31\x8e\xaf\x6a\x28\xc7\x78\xe3\x78\xae\xf7\x25\xed\x1b\xc3"
		"\x06\xfb\x37\xd9\xc4\x8c\x83\xa4\x49\x92\x39\x1e\xcf\x10\x53\x4c\x8a\xf7\xa1\x3c"
		"\x3b\x18\xc8\xff\x3a\x9e\xd3\x8b\x99\x6b\x38\xa8\xa9\x1b\x8f\xba\x6e\x8c\x79\xa9"
		"\x15\xed\x07\x62\x9f\xfc\x14\x73\x73\x72\x60\x13\xf3\xac\x45\x9a\x81\xfc\xbf\xd0"
		"\x86\x09\xde\x6b\x3d\xf6\x3d\x55\x1e\xfb\xdd\xc0\xfd\x3d\xd8\xc4\xd0\x04\x69\x7c"
		"\xf2\x2b\xec\xe3\xda\xe0\x73\x47\xc6\xb5\x8c\xf9\xad\xdd\x56\xc6\x3b\xae\x1c\xe3"
		"\x95\xa1\x0d\x1f\x07\xe6\x03\x0f\x6d\xb8\x13\xff\xde\x44\x0e\x62\x68\x52\x69\x50"
		"\x4e\xaa\x72\x18\x3f\xb3\x6e\x5c\x57\x8a\xfb\x88\x07\x43\xd1\xff\x56\xbb\xed\x0a"
		"\x4f\x42\xb3\x3b\x39\x88\x19\x63\x4b\x5f\x9e\x2c\xcf\x6d\xcc\xaf\xc7\x9e\x73\xf1"
		"\x56\xdb\x35\x81\xf1\xc3\x40\x7e\x17\xcc\xd5\x17\x03\xf3\x3c\xc1\x41\x0c\x7d\x2a"
		"\x3d\xef\x48\xca\x1a\xfa\x31\xbe\x7f\x6c\xc5\xfd\x63\x19\xc6\xe4\xff\xf0\x79\xd0"
		"\xf3\xe2\xbc\xfb\x14\x63\x65\x51\x3c\x9b\xde\x94\xbd\x89\x71\xf0\x1d\x72\x10\x17"
		"\xe7\x63\x51\x93\xf9\x7b\x10\x9b\x7d\x1c\x4a\x5b\xb9\xbf\xfd\x0d\xea\xff\x32\x34"
		"\xf3\x63\x5c\x2e\x86\x26\x48\xc3\xef\x2e\x98\x27\xc6\x7b\x02\xe6\x5f\x94\x27\x2f"
		"\x61\x6c\x4d\x20\x07\x31\x34\x41\x9a\x81\xfc\x69\x18\xc3\x7f\xc2\x1c\x9e\x11\x73"
		"\xf5\xca\x1a\xd2\x50\x5a\xa8\x22\x07\x31\x34\x41\x9a\xf7\xf3\xf3\x1c\xdb\x99\x67"
		"\xdb\xcc\xc5\xfc\xcc\xbf\x8c\xf8\xf2\x03\x72\x10\x43\x13\xa4\xe1\xd9\x63\x49\x7b"
		"\x37\xda\x67\x2d\x39\x88\x11\x73\x3b\x8e\x7b\xc4\x09\xf8\xe7\x22\xbc\xe3\x7b\xb1"
		"\x8c\xd2\xc2\xb7\x50\xe7\x32\xf8\xf3\x1f\x91\x83\x18\x9a\x20\x0d\xe3\x75\xd9\x32"
		"\x66\xe0\xb7\x2b\x43\x1b\x7e\x4f\x0e\xe2\x4f\xf2\x8b\x31\xef\x2a\xe3\x1e\xf7\xf1"
		"\x49\xdf\x75\xc7\x8d\xea\x63\x1e\x27\x39\x88\xa1\x09\xd2\x2c\x68\x6a\x71\xb3\x9b"
		"\x7a\xe3\xf8\x02\x07\x31\xfb\x11\xfb\x5d\x3f\x33\x59\x9c\xd6\x16\x0e\x64\x9e\xbc"
		"\x67\xdc\xf7\xd4\xa2\xdf\xba\x93\x97\x8d\xf3\xdc\xcb\xc2\xf7\xc4\x7b\x97\x9a\xc2"
		"\x5e\x98\x3f\xfb\x04\x31\x34\x41\x1a\xd8\x06\xd9\x16\xf7\xb0\x5d\x98\xdf\x57\x87"
		"\x2d\xfe\x26\x94\xf9\x77\xf8\xa4\x33\x50\xa7\x51\xf1\x8c\x9e\xe7\xc4\xe0\x20\x86"
		"\x26\x48\x43\x4b\xe6\xd3\xb3\x2f\x19\x4f\x30\xae\xd9\x60\x0f\x84\x3d\x92\x75\xcc"
		"\x21\x44\xcc\xfe\x1a\x39\x15\x43\x13\xa4\xa1\x2d\xf3\xab\xfa\xb2\xff\x62\x5f\x70"
		"\x40\x5c\xb3\x87\x25\x2f\x92\x83\xf8\xbb\x4d\xdf\x73\x8f\x7c\x74\x43\x60\x5e\x21"
		"\x38\x88\x4b\xda\xf9\x2d\xc6\x05\x5b\xf7\x0a\x7b\x7a\xae\x13\x4b\x06\x46\xaf\xea"
		"\x6f\x99\xed\xd7\x67\x8f\x32\x97\x11\x7b\xc4\xd9\x26\x86\xc6\xa4\x81\xad\xc9\xb6"
		"\x18\x6f\x1f\xeb\x99\xe7\xc0\xbe\xc6\x9e\x08\xbe\xad\x0f\x3e\x63\x15\xcf\xbd\x63"
		"\x1e\x09\x38\x88\x39\x06\xa4\xa1\x2d\xef\x99\x4b\xda\xcf\x41\xfb\x7e\xc7\x71\x8f"
		"\x8e\x98\x3f\xe6\xe6\x60\x7c\xa7\x65\xcd\xcd\x31\xcf\x65\x52\x72\x0f\xda\x78\x71"
		"\x3c\x1b\x14\x43\x13\xa4\x81\x6d\x90\x2d\xcb\xe4\xf7\x2a\x99\xff\x41\x2c\xe7\xbd"
		"\xec\xe7\x68\x97\x33\x62\xf9\x68\x27\x47\x0d\xf5\x62\x96\x23\xcd\xd6\xbd\x83\x67"
		"\xae\xda\x90\xf6\xd7\xb1\x3f\xaa\xf1\x15\xed\xbb\x91\x83\x98\xf9\x76\x88\x89\x30"
		"\x56\x79\x6e\x7c\x18\xd6\x15\x83\x4f\x7e\x06\xb1\xe7\xfe\xf0\x8d\xb7\xc4\x72\x19"
		"\xef\x8b\xa1\x31\x69\x98\xc7\xbd\xcd\x76\x57\xb4\xd9\x9c\xc0\x35\x01\x6b\x29\xea"
		"\x31\x1b\xf3\x68\x03\xe6\xee\xc1\x3e\xae\x67\x75\xe3\x50\xde\xc1\x26\xe6\x7a\x2b"
		"\x0d\xef\x46\x64\xcb\xef\x5e\xb8\x87\xef\x5a\x37\xdd\x97\x35\xcf\xf1\xbc\xd3\xee"
		"\x5a\xb7\x43\xcc\x15\x64\xee\x30\xbf\x35\x61\xae\xa0\x18\x1a\x93\x86\xf9\xf5\xb2"
		"\x65\x39\x15\xed\xcc\x55\x61\x8e\xe1\x6c\xf4\xcf\x25\x88\x15\x6a\x10\xff\x1e\xe2"
		"\xcb\x1a\x9e\x41\x5c\xb2\x13\xd9\xc4\xc5\x73\xd9\xa2\x86\xe5\xc8\xb6\xa2\x7d\x16"
		"\xd6\xaf\xbf\xc4\xd8\xad\xb2\xc0\x73\xf8\x7e\xf8\xb8\x9b\xe3\x19\x69\x75\x5d\x89"
		"\xaf\x2c\x94\xfa\x98\xb7\xba\x95\xf9\x4d\x9b\x34\xb0\x35\xd9\x96\xd7\x55\xa2\xae"
		"\xbb\x78\xe6\x5f\x75\xdb\xf6\x28\x77\x2d\x62\x9d\x19\x9e\xf7\x2d\x62\x68\x4c\x1a"
		"\xe6\x28\x72\x3d\xe8\x5a\x37\x2f\xe6\x28\x7e\x89\xfd\x3b\xf9\xed\xe1\x6b\x7c\x7c"
		"\x2e\xde\xe2\xdf\x40\xec\x7b\x37\x73\x51\x51\xaf\x61\x7e\x68\x03\xef\x8c\x2b\xc9"
		"\x26\x86\x26\x48\xc3\x3c\xe3\xd2\x42\x2f\xef\x8d\x62\x4e\xb1\x98\x63\x66\x63\xfe"
		"\x25\xe6\x64\x7b\xf8\x35\xf8\x8d\xcd\xf0\xb3\xa3\xc8\x41\x1c\xef\x56\xb1\xbf\x66"
		"\x7c\x0a\x0e\xe2\x5f\xb6\xbd\x9c\xeb\xad\xdd\xc9\xee\x5b\x54\x4a\x4e\xc5\xc5\xf3"
		"\x96\xd1\x76\x7f\xdb\x16\xde\x3d\xfa\x4f\xf2\x8d\x58\x1b\xee\x8b\xf7\x90\xe2\x24"
		"\x99\xef\x1f\xcd\x4a\xd2\xf8\xad\x50\xdd\x3c\x7f\x43\x36\x24\xad\x6a\xf8\x17\xd9"
		"\xc4\xd0\x98\x34\xc5\x9c\x9c\x2f\x98\xc7\x1c\x63\x0c\x8c\xf3\x78\xa7\xcd\x58\x42"
		"\x5c\xcc\xd5\x29\x6a\x98\x0f\xca\xb5\x6d\x8b\x5f\x11\xef\x5a\xc5\x2b\xdb\x6e\x77"
		"\xf3\x5e\x2d\x0f\x8c\xeb\x4b\xda\x9f\x74\x9c\x57\xef\x65\x3f\x80\x8f\x7e\xcb\xfd"
		"\x3c\xd9\x21\xc7\xfd\x1f\x38\x88\xa1\x09\xd2\xc0\x36\xc8\x36\x49\x76\xf3\xdc\xff"
		"\xd3\xbf\x54\x16\x86\xa3\x73\x3e\x4c\x3f\xce\xfe\x95\xb2\x4d\x3f\xc9\x37\x30\xbf"
		"\x2b\xde\xd1\x6d\xe3\xe1\x26\x0d\x6c\x6d\x9b\xed\x9b\x8e\x77\x27\x1f\xe6\x87\x40"
		"\xf3\x66\x10\xf3\x9e\xf7\xf8\x57\xdf\x0f\x3c\x73\xbd\x63\xd1\x6d\xee\x0f\x93\x6e"
		"\x01\x9f\x43\x0e\x62\xde\xff\x4a\xd3\x6d\x07\xf8\xed\x0a\xef\xb8\xb1\xc9\x76\xd8"
		"\x03\xee\x80\xbe\xff\xaf\xbb\x2a\x7b\xba\x9e\x63\x83\x79\xae\x88\xbf\x1d\xc7\x86"
		"\x18\x1a\x93\x06\xb6\x26\xdb\xc7\x16\xbd\xee\x4e\x7d\xf5\xab\x9e\x67\xf2\xe0\x20"
		"\x66\xdc\xc5\x39\xf2\x4c\x76\x7b\xca\xb6\xce\x7c\x70\x88\xe7\xc9\x41\xcc\xb8\x4b"
		"\x9a\xdb\x16\xbd\xe8\x4e\x7a\xb5\xcc\xf3\x4e\x15\x1c\xc4\x25\xed\x17\xc7\xdc\xb9"
		"\x92\xf6\x77\x10\x9b\xff\xd8\xd3\xd7\x31\x46\x2b\x6b\xf8\xae\x47\x1b\x23\xae\xfa"
		"\x33\xfc\xda\x85\xf0\x59\x97\xc3\xdf\x3c\x43\x36\x31\x34\x26\x0d\x63\x17\xd9\xa2"
		"\x4c\x1b\x2c\x73\xca\x52\xcc\xb1\x3f\x3b\xde\x6d\x0d\x2c\xbc\xc6\x2b\xdf\xa1\xb4"
		"\x70\xb3\xe7\x5d\x78\x4f\xfe\x7d\xb2\x89\xa1\x19\xcc\x89\x80\xad\xc9\x36\x7b\xe8"
		"\x2e\xd4\xed\xdf\x6e\x78\xfb\x1a\x57\xb6\xe2\x76\xb4\xe7\x5a\xc6\x6a\x6e\xa0\xf3"
		"\x2e\x94\xff\x77\xb7\x31\x7f\x23\xd9\xc4\xd0\x98\x34\xb0\x35\xd9\xf6\xb7\x2c\xf7"
		"\x9b\xf3\x6f\xc7\xbd\x45\x49\x63\x4b\xb4\xdd\x94\x4d\xc7\x5c\xb8\x1d\xfe\xe8\x25"
		"\xc7\x3d\x3e\xd8\xc4\xd0\x98\x34\xb0\x35\xd9\x96\xd7\x31\x37\xec\x59\x37\x3f\xb9"
		"\x7a\x55\x92\x2c\x85\x6f\x7d\x8a\x39\x7e\xf5\xe5\x75\x97\xa0\x7f\x6f\x73\x6b\xb2"
		"\xef\xa1\xfd\x2f\x31\x31\x34\xb6\x4d\xc3\x5c\xb2\xa2\x2d\x73\xc6\x36\xe7\x7f\xee"
		"\x46\x25\xef\xa5\x5c\x63\x98\xeb\x4c\x7d\x45\xc7\x5c\xf8\x6d\x7e\x7f\xbc\x9a\x6c"
		"\x62\x68\x4c\x1a\xe6\x98\x6d\xb3\xbd\x04\xf5\x5f\x1e\xf3\xfa\xaa\xeb\xce\xe7\x7d"
		"\x95\xfb\x38\xb9\x62\x55\xdf\xba\xd3\x3d\xf7\x69\x9d\xd9\xc4\x14\x6c\x62\x68\x4c"
		"\x1a\xd8\x9a\x6c\xcb\x56\xdc\x08\x7f\xfb\x0f\x7e\x3f\x9a\x2b\xe9\xb8\xd6\x73\x9f"
		"\xcd\xf3\x7b\xb0\x89\xa1\x31\x69\x06\x6a\x7e\x8b\xf8\xff\x9f\x6e\x73\xfe\x32\x97"
		"\xbd\x74\xab\xe7\x7d\x18\xcf\x7e\xc0\x26\x86\xc6\xa4\x49\x96\x2e\xc3\x78\x7f\xc9"
		"\xf1\x1e\x29\x7b\xe8\x16\xcf\xbb\x25\xe6\xed\x81\x4d\x0c\x8d\x49\x53\xb6\xe2\xe2"
		"\xf8\x5e\x5b\x7c\xbb\x2b\x5b\x7b\x35\xc6\xcf\xdf\xdc\x90\xf6\xc7\xc9\x36\xc8\x2b"
		"\x2e\x36\x69\xfa\x5b\xce\xf5\xbc\x9f\xa9\x68\xbf\xd7\x55\x74\xfc\x6f\xcc\x1b\x64"
		"\x1e\x21\xd8\xc4\xcc\x27\x90\x86\x7b\xa2\xb3\x92\x7b\xeb\xd7\xf4\x66\x33\x78\xd7"
		"\xb6\x5d\x61\x64\xbc\x2f\x7a\x64\xbb\x43\xfc\xd4\xa6\x0d\xb9\x7b\xe7\xff\x6a\x15"
		"\xd8\xc4\xdc\xe7\x49\xc3\x3d\x94\x6c\xbb\xd6\x4d\xf5\x8c\x99\xe8\x53\x10\x3b\xfa"
		"\xd1\x85\x3d\x5c\xf1\xac\xf1\x08\x13\x33\xaf\x47\x9a\xbe\x75\x47\xf9\xd7\xb2\xff"
		"\xc9\x31\xf7\x03\x6c\xe2\xd2\xd6\x53\x99\x5b\x8e\xb6\x7a\xcc\xf5\xfa\x93\x31\x96"
		"\x8e\xc5\x3b\xa2\x7d\xfc\x37\x7d\x6d\xe1\x4c\xcc\x91\xbb\xc9\x26\x86\xc6\xa4\x81"
		"\xad\xc9\xb6\xb4\xf5\x24\xd4\xe7\x47\xcc\x13\x5e\x55\xd5\x3c\xdb\xf3\x7c\x8e\xe3"
		"\x8d\x77\x5f\x62\x68\x4c\x9a\xee\x96\xc6\x98\xb3\xfa\xe0\xf0\xc2\xf4\x6e\x9b\xed"
		"\x19\x1f\xcf\x4c\x36\xd4\xdf\xdf\x36\xc3\xff\xb1\x6d\xba\xbb\xf2\xf7\xcf\xcf\x00"
		"\x9b\x98\x77\x0e\xd2\xc0\xd6\x64\xcb\x6f\x72\xab\x1a\xae\x76\xdc\x27\x55\x16\x0e"
		"\x8f\x75\xe0\xfd\x40\xe6\xe7\xc5\xe7\x8c\x35\xc1\x26\xe6\x59\xb0\x34\xbc\xa3\x92"
		"\x2d\xf7\x9e\x23\x1a\xae\x74\xa3\x0b\x95\xf0\x39\x8d\x31\xaf\xb8\xb2\x30\x9d\xf9"
		"\x53\x18\x03\x33\xdd\xfb\xf9\xd2\xad\xf9\x62\x45\xe6\xf9\xa9\x34\xdc\xab\xca\x96"
		"\xb9\xd0\xdc\x4b\xf4\xfa\xcb\xe3\x1a\x57\x5b\x98\x8f\x35\xa6\x80\x18\xe0\xeb\xe8"
		"\xeb\x5f\x81\x17\x90\x4d\xcc\x35\x4e\x1a\xe6\x4e\xcb\xb6\x2b\xfb\x2e\xc6\xd5\x35"
		"\x68\xdb\x6b\xc9\x26\xae\x68\x6f\xf4\xfc\xfe\x90\xef\xc0\x33\x56\xf1\xdd\x47\xd7"
		"\xfb\xfb\xdb\xaa\xdc\x19\x4d\xb9\x1c\xd8\xc4\xdd\xf6\xa3\x38\x26\x87\xb7\xff\x06"
		"\x71\xec\x99\x9e\xe7\x1c\xd4\x57\x27\x3f\x88\x67\x07\xfc\x26\x06\x6c\x62\x68\x4c"
		"\x1a\xd8\x9a\x6c\xcb\xeb\x4e\x8a\xb6\x9b\xf3\x93\xf1\x7c\x61\xcc\xef\xe5\x3d\x2f"
		"\xd8\xc4\xd0\x98\x34\xa5\x85\xd9\x31\x57\x99\x77\x7d\xcc\x03\xff\xcc\xdf\xe1\x78"
		"\x4f\xc8\x7b\x92\x6d\x3c\xdb\xa4\x29\x69\x3f\x21\xea\x5f\xcb\x6e\xe6\x7d\x4e\xd4"
		"\xef\x91\x58\xbc\xdb\x11\x43\x63\xd2\x64\xb5\x27\x47\x7d\x5f\x72\x17\xd6\xb8\x53"
		"\xe0\xdb\xef\x88\xfd\x0b\x36\x31\x34\x26\x4d\x55\xf3\x89\x9e\x39\x65\x6f\x67\xb3"
		"\x31\x26\x4f\xf6\x1c\xbf\xae\xa2\x61\x06\xd8\xb6\xf1\x89\x26\x4d\x49\xc7\x99\x71"
		"\x0d\xa2\xbf\xaa\x2c\x9c\x1a\xd7\x29\xee\x1d\xc0\x26\x86\xc6\xa4\x29\x1d\x71\xa9"
		"\xcf\xfc\x5f\xb9\x17\xc5\x3a\x52\xf0\xc3\xdb\x03\x73\xf3\xc9\x26\x86\xc6\xa4\x29"
		"\x5b\xb1\x0c\xe5\xaf\x47\x7b\xdf\xef\x4e\xfd\xe2\x1a\xac\x2f\x1d\xee\xd8\xa6\x36"
		"\xb2\x89\xa1\x31\x69\xb8\x06\xf5\xfa\xce\x78\xc6\x59\x32\xe5\x16\xcf\x38\xba\xb4"
		"\x70\x03\xd9\xc4\x5c\x83\x06\x35\xed\x37\xa2\x4d\x3e\xc4\xdc\x7f\x2e\xd7\xbf\xdf"
		"\xad\x51\xcf\x3b\x5e\xb0\x89\xa1\x31\x69\xfa\xce\xbd\x14\x73\xff\x2f\x58\x2f\x6a"
		"\xea\x4b\x5b\xaf\x8d\xf9\xc9\xf0\xf9\x39\xb0\x89\xa1\x31\x69\xb0\x7f\x8a\xdf\x68"
		"\x74\x96\x2d\x85\x1f\xbb\xc0\xc4\x55\xcd\x5f\xc7\xf8\xbc\xc7\xf1\xfb\xbe\x9e\x85"
		"\xdf\xf4\xf4\x19\xfc\xce\xa4\xa4\xfd\xbb\x9e\xdf\x28\x20\xe6\x21\x9b\x18\x1a\x93"
		"\x06\xb6\x26\xdb\x81\xfc\xf7\x3d\xcf\x5b\xf9\xf7\x10\x98\x4f\xde\x6f\x7f\x00\x3f"
		"\x8c\xfa\x5c\xe4\x99\xa3\xc7\xb3\x3e\xb0\x89\x99\x67\x2e\x0d\x6c\x4d\xb6\xbd\xb5"
		"\x3f\xc5\xbb\x3f\x11\xf3\x35\x10\xef\xc7\xd8\x9e\x77\xd1\xa5\xad\x97\x60\x3e\xf2"
		"\xdc\xf7\x44\xb2\x89\xa1\x31\x69\x60\x6b\xb2\x2d\xaf\xbb\xcc\x17\xff\x2e\xc5\x2f"
		"\x5c\xb2\xf8\x6a\xcf\xdc\x4a\x9e\x91\x32\x97\x9e\x67\xcc\x3c\x33\x60\x2e\xbd\x18"
		"\x1a\x93\x06\xb6\x26\xdb\xa6\x51\x57\x78\xc6\x05\x8d\x4d\x37\x63\x2d\xf8\x29\xc6"
		"\xf9\x93\x8e\xb9\x30\x60\x13\x43\x63\xd2\xf4\x65\xdf\xf7\x9f\xdb\xfd\xd8\x5b\xce"
		"\x25\x9b\xb8\xb2\xf5\x74\xcf\xfc\x82\x71\xc9\x48\xb2\x89\xcb\xeb\x16\xc5\xb9\xf3"
		"\x5c\xb2\x16\xeb\xfe\x22\x13\x0f\x74\xfe\x12\xfa\xbf\x3a\x9e\x6f\x81\x6d\x90\x6b"
		"\x96\xc6\xdc\x72\xfa\x11\xb0\x89\x4b\x37\x2c\xf3\x3c\x53\x2f\x4f\xbe\x4d\x36\xf1"
		"\xfc\x83\xae\xc3\x38\x5c\xe7\x8e\x5c\x76\x33\xd9\xc4\x7d\xd9\x25\xe8\xa3\xe7\x1d"
		"\xbf\x3f\xa6\x2f\xe5\xdd\x0b\x62\xf1\xb4\xb4\x75\x11\xc6\xf9\xea\xf8\x9d\x25\xd8"
		"\xc4\xc5\x5c\xda\xa2\x06\xb6\x26\xdb\x92\x8e\xdf\x61\x2e\x6c\x88\x39\x11\xfd\x2d"
		"\xcb\xfc\x06\xeb\x77\xdc\xe3\x81\x4d\x0c\x8d\x49\x93\xd5\xfe\x1e\xed\xf9\xa9\xfb"
		"\xcc\xdf\xec\xda\x26\xb5\xf9\x15\xcf\x97\x78\xb6\x39\xd8\xc4\xd0\x98\x34\x03\x0b"
		"\x6f\x8f\x39\xf0\x03\xf9\xff\x73\xf7\x1d\xbd\xdc\x2f\xff\x88\xfb\x95\x27\xc8\x26"
		"\x86\xc6\xa4\x61\x7c\xd5\x6d\x7f\x8b\xdf\xa3\xf1\x7b\x9f\xcc\xf3\x5b\x31\x9e\xe1"
		"\x5f\x6f\x62\xc6\x5d\xd2\x70\xdd\x1c\xc8\xdf\xe5\xf8\x9e\xdd\x2d\x8b\xb1\xa6\xbc"
		"\xe2\xf8\x37\x43\xc0\x26\xe6\x7a\x2a\x4d\xf6\xd0\x25\x9e\x39\x4a\xd8\x7f\x61\x5d"
		"\x6e\xc2\x5a\xf3\xe7\x18\x87\x83\x4d\x0c\x8d\x49\xb3\xfc\xa3\x56\xff\xf0\x95\x55"
		"\xf1\x3b\x22\xb0\x89\xf9\xed\x18\x63\xd7\x21\xed\x4b\x62\x2e\xaa\xb8\xac\xe1\xd0"
		"\x98\xab\x3f\x35\xf9\xc7\x2a\xb0\x89\xab\x57\x74\xb8\xfe\xb9\x87\xdb\xce\xad\xbb"
		"\x39\x70\x10\x6f\x57\xe8\x8e\x7f\xd3\xe0\x8d\xfc\xbf\x73\xe0\x20\xae\x6a\xa8\xc6"
		"\x3c\xda\xc9\xf8\xfd\x29\xd8\xc4\xcc\x67\x2e\x69\x1f\x67\x6b\x7a\xf3\x4f\x33\x6f"
		"\x59\x3c\x7c\xc4\x4a\x57\x71\xc5\xc1\xf1\x9e\x0b\x1c\xc4\xc9\xe2\x95\x78\x9f\x23"
		"\xed\x53\x5f\x8a\x39\xb2\x32\x88\x87\x35\x5e\xe9\xaa\x2e\x3c\x3a\xd6\x01\x1c\x06"
		"\xb9\xe3\x74\x97\x55\xd5\xc7\x3a\x80\x83\xf8\xbd\x8c\x67\x9b\xbb\xc7\x3a\x80\x83"
		"\x98\xf9\x0f\xcc\x59\xe2\x59\x26\xf3\x1c\xc4\xb5\x85\xdd\x10\xaf\xd7\xdb\x98\x86"
		"\xc8\x41\xfc\xa9\xff\x61\x5c\xbf\x4b\x3a\x22\x07\xb1\xea\xc3\x98\x4d\xf5\x89\xbc"
		"\xb5\x3e\xf4\x17\xaa\x0f\x59\xf5\xa9\x29\xbc\xe1\x54\x1f\xb2\xda\x87\x67\xfb\x6a"
		"\x1f\xb2\xda\x87\xb9\xba\x6a\x1f\xb2\xda\x67\xfb\xe6\xa7\x06\xdb\x87\xac\xfe\x62"
		"\x1d\xd4\x5f\x64\xf5\x17\xeb\xa0\xfe\x22\xab\xbf\x58\x07\xf5\x17\xb9\x3c\xd9\xd9"
		"\xf3\x4e\xa8\x98\xb3\xbc\xb3\x89\xb3\xda\xed\x7d\xf7\x7e\xfb\xc4\x36\x61\x9e\xa3"
		"\x98\xf7\x50\x15\x8d\x0d\xb1\x4d\x78\x0f\x25\x7e\x7c\xd2\x87\xee\xee\x45\xb3\xad"
		"\xfd\xa3\xc8\x41\xbc\xe0\xfd\x37\xdd\x49\xfb\x1e\x65\xf7\x1d\x7d\x27\x39\x88\x9b"
		"\xcf\x5f\xe9\xe6\x3f\x70\x9c\x9d\x79\xfe\x23\xe4\x20\x3e\xe2\xd5\x25\x8e\xf9\x0c"
		"\xd4\x80\x83\xb8\xb7\xb6\xe0\x4e\x3b\xff\x84\x58\x26\x38\x88\xa5\xbf\xaa\x6d\xbf"
		"\x41\x3d\x79\xce\x41\x2b\xdd\x5d\x57\xce\xb3\xed\x5a\x2f\x24\x07\xb1\x7e\x77\x8b"
		"\x1f\x31\xf8\xbb\x64\xd5\x93\xb6\xaa\x27\x99\x67\x17\x8c\xb3\xbf\x68\xf9\x77\xe4"
		"\xda\xd9\x73\xed\xb5\xf3\x56\x6e\xbd\xd3\xd9\x96\xf3\xc3\xd8\xe5\x73\xbb\xad\x98"
		"\x5b\xb2\xf4\xe8\x98\x2b\x12\xef\xf7\x37\x30\x4f\x60\x66\x31\xaf\x18\xfb\xe1\x3b"
		"\x9f\xef\x70\xc5\xf3\xd5\xa1\xf6\xb3\x21\x23\xe3\xd9\xc8\xd0\x86\x5e\xc4\x02\x5f"
		"\xf7\xc5\x3c\x99\x23\x7c\x77\x4b\x79\xe4\xca\xd6\x23\x3c\xe3\x1c\x32\xf7\xb3\x65"
		"\xcd\x2b\xc3\x8e\xcd\xdf\x71\x95\x6f\xec\xc8\xbf\x1d\x43\x0e\xe2\xcd\x2f\x61\xbd"
		"\x5f\x3b\x22\x7e\x13\x0e\x0e\xe2\x61\x1d\x77\xbb\x92\x31\x35\x56\x32\x65\x92\x81"
		"\x83\x98\xeb\x61\x45\x63\x35\xfc\xce\x02\x72\x18\xe4\x97\x6e\x71\xc9\xbd\x95\xc6"
		"\x7d\x1e\x38\x88\x99\x4f\xd6\x7f\xf9\x50\xcb\x5e\x3a\x9f\x1c\xc4\xb1\xce\x1d\x23"
		"\x6c\x60\xe1\xff\x58\x69\xeb\x9b\xae\xe7\xf1\xed\x6d\x6d\xbe\x13\xbe\xfd\xcd\x20"
		"\xee\x1b\x5d\xe7\xab\x2e\xac\xb4\xbe\x81\xe3\x56\x81\x4d\x5c\x31\xe6\x68\xf8\xea"
		"\xcd\xa1\xb6\xf5\x6b\x0e\x6c\xe2\xae\xd1\x27\xf8\xbe\xd5\x3b\x60\xdc\x3e\x47\x36"
		"\x71\xef\x05\xfb\xfb\x8a\x8e\x5a\xdb\xf4\xd0\x17\x64\x13\xf7\xb7\xf4\xa2\xce\x93"
		"\xad\x6f\xf9\x68\x0f\x0e\xe2\xed\x46\xdc\xed\xaa\xaa\xa7\x5b\x32\xed\x60\x72\x10"
		"\x8f\x6f\x9f\xea\x4a\xae\xf0\xfc\xde\x80\x1c\xc4\xfb\x36\xdc\x9a\xeb\x3a\xd7\x59"
		"\xe9\x86\xcc\x81\x53\x31\xef\x99\x2a\xdf\x60\xce\xde\x5a\x72\x10\x1f\x92\xf4\xe6"
		"\x78\xa7\xcc\x7c\x6d\x70\x2a\x1e\xdf\x71\x8a\xeb\x5b\xee\xed\xfd\xfc\xf5\x39\x70"
		"\x10\x57\x8e\x08\xae\xa4\xe3\x40\x6b\x4e\xde\xa9\x07\x07\x71\xdf\xb9\x25\xbe\xeb"
		"\x90\x7d\xec\x45\x7f\x0b\x62\xaa\x12\x13\x97\x4c\x19\xef\xcb\x97\xee\x61\x7d\xeb"
		"\x0e\x43\x5c\x37\xde\xc4\xd5\xcf\x4e\xf4\xa5\x1b\xf6\xb4\xb2\x15\x37\x93\x4d\x5c"
		"\x3a\x62\xb4\x4f\xa6\x4d\xc6\x7c\x6c\x27\x9b\xb8\x6c\x6d\x07\xe6\xf6\x0c\xdb\xdc"
		"\xf9\x2e\x39\x0c\x72\xf3\x4f\x63\x7e\x45\x69\xeb\x67\xe4\x20\x8e\xc3\x73\xe9\x0c"
		"\x2b\x1d\xe1\xed\xec\xb3\x3f\xc8\x85\x2f\x0e\xb5\x53\x9f\xa8\x27\xa7\xe2\xcf\x7c"
		"\x77\xae\xfa\xd9\x03\x8c\xeb\x12\x38\x15\x6f\x57\xb8\x0e\x31\xc4\x14\xeb\x5d\x7f"
		"\x06\x39\x15\xb3\xc8\x9e\x97\x26\x5b\xf5\x8a\x62\xfe\x21\xcf\x46\xcb\x93\x3d\xe3"
		"\x1c\x59\xd0\xe4\xed\xae\xb6\x21\xe1\xef\xf6\x97\xdc\xd6\x7b\x5b\x72\x2a\x7e\xd7"
		"\xf7\xe7\x7a\xf6\x6c\xb0\x1d\x5a\xcf\x26\xa7\xe2\xaa\xd3\x1e\xcf\xad\x3a\xdf\xd9"
		"\x9e\x97\xde\x4c\x4e\xc5\x95\xad\xcf\xb9\xaa\x0d\x93\xd0\xe6\x53\x53\x70\x10\x97"
		"\x6e\x18\xed\xab\x36\x8c\xb7\xb6\x7c\x63\x0e\x6c\x83\xdc\x7a\xb0\xef\x3b\x77\xac"
		"\x55\x16\x8e\x46\xec\x77\xb0\x89\x2b\x3a\x66\xf8\xd2\x27\x76\xb7\xca\x11\x8f\x91"
		"\x4d\xcc\x6f\xd7\x7b\xab\x0e\xb2\xae\x75\x6f\x92\x4d\xdc\xd3\xf9\xa1\xeb\xae\x39"
		"\x02\x73\x61\x88\x07\x07\x71\x65\xe1\x77\xe8\x8f\xb9\x36\xf0\xf8\x78\x72\x10\x63"
		"\xff\xe2\xba\xe6\x9e\x68\xfd\x2d\xa3\xc9\x41\x0c\xbf\x9c\x1b\x98\x3d\x0f\x63\xf2"
		"\x7d\xec\xa3\xde\x48\xc5\xb9\x57\x47\xba\x53\x5f\x3d\xd2\x3a\xa7\x3d\x93\x82\x83"
		"\xb8\xac\xa1\x09\xe3\x7c\xaa\xf1\x9e\x11\x1c\xc4\xf4\x77\x15\xf7\xee\x8b\xf9\x33"
		"\x99\xbe\x2f\x88\x7b\x3a\x7f\x81\xf1\xbc\x27\xc6\x59\x23\x39\x88\x99\x0b\xd1\xd3"
		"\xb9\x2b\xc6\xcd\xd9\xe4\x20\xae\x6a\xde\x11\x7b\x89\x5d\xad\x7a\xe9\x39\xe4\x20"
		"\x2e\x9e\xeb\x4e\xb4\xbe\x25\xc5\xbb\xc8\xeb\x27\xe6\xec\xc9\x69\x93\xa2\xbf\xa2"
		"\xcf\x1b\x5d\x38\x09\x6b\xec\xbd\x2e\x59\xc0\xbb\xf5\x2e\x72\x10\x6f\xa9\x1d\xeb"
		"\xfa\x96\x34\x60\x1d\xaa\x26\x07\x31\xfb\xba\xac\x7a\xa6\x75\xd6\x7e\x9d\x9c\x8a"
		"\x47\x34\x7c\x94\xab\xda\x30\xdd\xaa\xaa\x27\xf1\x1b\x83\x54\xec\x1a\x1e\x8f\x7f"
		"\x13\xe9\x8b\x9a\x8f\x02\x38\x15\xb3\x0e\xcc\x87\xe4\xdf\x9e\x50\xae\x2c\xef\xfc"
		"\x0e\x2b\xdc\x9f\xe3\xfd\x71\xf9\xe2\xab\xc8\xa9\xb8\xa6\xd0\x95\xab\xb8\xe2\x68"
		"\x8c\x93\x6f\x92\x53\xf1\xce\x85\x17\x73\x5d\x87\xcc\xb1\xcd\x0b\x47\x93\x53\xf1"
		"\x5e\x85\x09\xf1\xf9\x0e\xed\xc3\xc8\xa9\x38\xde\x7f\xad\x9e\x67\xcf\x4c\x9b\x1b"
		"\x26\x5d\x3b\x33\xf7\x93\x5d\xe6\xda\xd4\x6b\x77\x25\xa7\xe2\xae\xf3\x56\xc6\xe7"
		"\x4b\x76\xd9\x9b\x9c\x8a\x3f\xc9\xb7\xe7\x92\x67\x4f\xc0\x7a\xd4\x4c\x4e\xc5\x5d"
		"\xd7\x2c\xc9\xad\xdd\x78\x9c\xcd\x6e\xba\x98\x9c\x8a\x8b\xf7\xbf\xc7\x5b\x59\xf3"
		"\xd5\x81\x73\xb9\x7c\x71\x8d\x55\xb6\xde\x41\x0e\xe2\xed\x5a\xf7\x72\x5d\xcb\x0f"
		"\x44\x9b\x2f\x26\x07\x71\x4d\x61\x5a\xcc\xe5\xe3\xf9\x32\x38\x88\xbf\xda\xf1\xcd"
		"\xb8\x8e\x74\xad\xbb\x88\x1c\xc4\x31\xb7\x70\xee\x64\xac\x6b\x0b\xb1\xc7\x9a\x1d"
		"\x73\x89\xcb\x9f\x3d\x8d\x9c\x8a\x3f\xcc\x3f\x99\xeb\x5b\x7e\x00\xc6\xea\x09\xe4"
		"\x54\xfc\xba\xfd\x3e\x57\x76\xe1\x21\x56\xb5\xd7\x61\xe4\x54\x1c\x73\xc3\x56\x60"
		"\xfd\x5c\x3c\xc7\x0e\x2d\xfc\x36\xd7\x78\xd0\x24\xeb\xbe\xfc\x08\x72\x2a\xbe\xe1"
		"\xbc\xfb\x73\x1b\xee\xde\xcb\x1e\x9c\x34\x97\x9c\x8a\x6f\xcc\x1a\x73\xc7\x7f\xb2"
		"\x9b\x65\x0f\x9d\x44\x4e\xc5\xf1\xce\x6b\xc9\x6e\x56\xd2\x78\x8a\xfd\xc7\xa6\x60"
		"\x5d\x39\xd2\x36\xd5\x7e\xd3\x81\x83\x98\x3e\x34\x5b\x3f\x2b\xfe\x6d\x43\x70\x10"
		"\x73\xcf\x5a\x55\x7d\xb8\x6d\xac\x79\x8b\x1c\xc4\xd8\x57\xb8\xfe\x2d\xb3\xb6\x7e"
		"\x57\x7f\x65\x10\x33\xdf\xa8\xf4\x8d\x06\xac\xfb\x2f\x90\x83\x18\x6b\xa3\x2f\xbb"
		"\x09\xed\xd9\xf2\x48\x5c\x33\xc5\x15\xf7\x8e\xf2\xbd\x27\x4f\x46\x1d\xae\x27\x9b"
		"\xb8\xf7\x82\x91\xbe\x62\xcc\x64\xdb\xb1\x79\x0e\xd9\xc4\x43\xeb\x3e\x8f\xf9\x36"
		"\x7d\xd9\x1b\x39\x70\x10\xd7\x6c\x58\x85\x98\xaf\xde\xf6\x6c\x5f\x4a\x0e\xe2\xea"
		"\xba\xf3\x5d\xcf\x8d\xb3\x6c\x97\x86\x75\xe4\x20\xde\xa1\xf5\x60\x37\xf0\xf8\x91"
		"\x36\x74\xe9\x4d\x0e\x1c\xc4\x25\x1d\x0b\x5c\xbf\x1d\x89\xf1\x6c\xe4\x20\x1e\x97"
		"\x9c\xe7\x98\xdf\xc7\x73\x3c\x70\x10\xf3\x5b\x1e\xe6\x32\xf1\x7c\x9b\xdf\xe0\x88"
		"\xb9\xdf\x60\x0e\xfc\xb8\xa4\x24\x7e\xb7\x28\x66\xfe\x0f\x73\x29\x67\x24\x7f\x8a"
		"\x79\x3e\xe2\x91\xc9\xff\x39\xde\x31\xa0\xab\xb0\xb7\xff\xbf\x20\xe6\xdf\x79\x28"
		"\xfe\xed\x9d\x26\xb2\x89\x99\x9b\x54\xd5\xb0\x4f\xfc\x7b\x16\xcc\x4d\xda\xc6\xe5"
		"\x31\x07\x89\x7f\xdf\x02\x6c\x62\x7e\xbb\x58\xd2\x9e\x33\xde\x11\xf0\x8e\x59\xcc"
		"\x9c\x27\xe6\x89\xf1\xbb\x48\x70\x10\x33\x16\xe2\xdf\x1a\xda\x9c\x2f\xe6\x6b\x89"
		"\xb9\xff\xaf\x6a\xe0\xbb\xec\x4f\x0e\xe2\xbe\x6c\x4e\x7c\xce\x73\x70\xde\x5b\x8a"
		"\x7f\x7d\x45\xe2\xce\xf9\x60\x17\x9b\xf7\xf3\x1e\x72\x10\xb7\xb4\xcd\x73\xbc\xe3"
		"\x59\xd0\x34\xd1\x83\x83\x78\xdc\xea\xe5\xae\x76\x76\xad\x9d\xd6\x34\x89\x1c\xc4"
		"\xa7\x2d\xf9\xc8\x9d\x9b\x0e\xb7\x71\x0f\x0e\x23\x07\x71\x59\xf5\xd7\xfc\xf6\xcd"
		"\x3d\xa1\xa4\xf1\x13\x07\x36\x71\xf5\xb3\xb3\x3d\xf3\x1a\x4a\x1a\x9f\x21\x9b\x18"
		"\xeb\x8f\x1f\xd6\xf8\x54\xd8\xb5\xe1\xe0\xb8\x2e\x89\xcb\x6e\x1a\xef\xbb\xce\xfd"
		"\x6b\x58\xf1\x9e\x3d\x09\x36\xf1\x19\x4d\xaf\xba\xf3\xca\x3e\x08\x7b\xbc\xf3\x6e"
		"\x0a\x0e\xe2\x98\x37\x5b\x7b\x79\xe8\xdf\xaf\x18\xa3\x32\xb7\xbe\xd7\xff\x28\x72"
		"\xcf\xc2\xf1\x7e\x68\x43\x31\xd6\xad\x2d\x94\x87\x8d\xf9\x75\x5b\xf3\xbd\x3f\xc4"
		"\x78\xfa\x61\x9c\x8f\x35\x85\x27\xb0\x2e\x1d\x15\x79\x78\xfb\x3d\x88\x35\x77\xb2"
		"\x62\xde\x02\xe6\x52\xcb\x26\xf8\xfe\x46\x3f\xb1\xfd\xc3\x74\x63\xfe\xdb\x58\x13"
		"\x1a\x4d\x7c\xd3\xd8\x93\xfc\xa1\x7b\x1f\x9b\xbb\x7a\xa7\x5b\xc8\x26\x7e\x6c\xf6"
		"\x4c\x7f\xf6\xd9\xaf\xb9\xd1\xaf\x2f\x25\x9b\xf8\x33\xff\xb2\xeb\x3d\x79\x12\xda"
		"\xe8\xdd\x98\xf7\x21\xbe\x67\xf6\x01\xfe\xfc\xf1\xcb\xd2\x27\xc6\x57\xa4\x60\x13"
		"\x9f\xf4\xc4\xb8\xf8\xf7\x1f\x9e\x3f\xa1\x83\x6c\xe2\xde\x93\xf7\x40\xbc\xf8\xa2"
		"\x3b\x2b\xb9\x77\x06\xd8\xc4\x2b\xda\xee\x71\xf3\x0f\xda\xdb\x3f\x93\xdd\x9e\x03"
		"\x07\xf1\x87\x0b\x7f\xe8\x36\xcc\x7d\x39\x8c\xa8\xfb\x73\x00\x07\xf1\x06\xdb\xdd"
		"\xf1\xef\xba\xf1\x1e\x0f\x1c\xc4\x59\xed\x12\xd7\xb3\x70\x4d\xcc\x7b\x03\x07\x31"
		"\xff\x9e\x52\xd9\xda\xb5\xf1\x9b\x40\xfe\x4d\x1a\x71\x7a\xde\xf5\xee\xac\xa6\x7f"
		"\x85\x9f\xec\xf2\x2d\x72\x10\x97\xb4\x8f\x74\x9f\x9f\x6b\xf0\x47\x0d\xe4\x20\xce"
		"\x6a\xab\xdc\xe6\xce\x3f\xc0\x1f\x0d\x25\x07\xf1\x0e\xad\x7b\xb8\x4d\xb5\xef\x86"
		"\xca\x27\xbe\x4f\x0e\xe2\x83\x7e\x7e\x9e\x3b\x7e\xd6\x4b\xe1\x9f\x2f\x5c\x17\xc0"
		"\x41\x7c\xed\x47\x3f\x76\x0b\x1f\xfd\x0c\x75\x3b\x8d\x1c\xc4\x7f\xb9\xf1\x17\xee"
		"\xda\x9f\x24\x36\x6e\xf5\x7e\xe4\x20\x7e\xf0\xe9\x53\xdc\x8a\xeb\x6b\x43\x7e\xa7"
		"\x94\x1c\xc4\x9f\xaf\xbb\xca\x0d\x5d\x5a\x70\x95\x23\x7e\x43\x0e\xe2\xdb\x27\xb5"
		"\xb8\x53\x46\x0d\xf1\xbd\xbe\x8a\x1c\xc4\xfd\xfb\xbd\x0e\xff\xb6\x16\x9a\xe3\x3d"
		"\x38\x88\xe9\x73\xff\x66\x8f\xe5\x7a\x2f\x38\x2b\xe6\x39\x88\xbb\xe6\xbe\x0e\x7f"
		"\x72\x62\xa8\xe8\x3a\x9e\x1c\xc4\xc9\xd2\xb7\x60\xf7\x5a\xe0\xb7\x6e\xe0\x20\xee"
		"\x9f\xeb\xf8\xf7\x4f\xe0\x4f\x5f\xc6\x1e\xda\x99\xb8\xef\xdc\x5d\x3d\xf3\xbb\x2a"
		"\x5b\xbb\xc9\x26\x2e\x5f\x3a\x81\x7f\xa7\x22\xed\xde\xb2\x2f\xe2\xf1\x09\x26\x2e"
		"\x19\x33\xdb\x73\x9d\xab\x9c\x32\x84\x6c\xe2\x9e\xc7\x39\x1f\x87\x63\xce\x8d\x22"
		"\x9b\x98\x39\x09\xe5\x75\x93\x5d\xd5\x4d\xb9\x98\x63\x20\xee\x59\x38\xc1\x17\xff"
		"\x26\xcd\x1e\x64\x13\x5f\xb4\xcb\xe1\x9e\x7f\x87\xe3\x0f\x93\x2a\xc9\x26\xee\xfe"
		"\xf6\x57\x7d\xf5\x7f\x87\xfb\x2f\x5a\x1e\x70\x60\x13\x9f\xf6\xdc\x66\x77\xfc\xab"
		"\x15\xfe\xa2\xac\xa5\x1e\x1c\xc4\xff\x9e\x78\xa2\x9f\x9b\x9b\xc7\xb3\x50\x07\x36"
		"\x71\xad\xbd\xe4\x06\x6e\x9b\x10\xff\xd6\x20\x38\x88\x79\xb6\xf0\x95\xf6\xf9\xee"
		"\x53\x3f\x95\x1c\xc4\x7d\xeb\xce\x8a\x79\x14\xfc\x0e\x0e\x6c\xe2\x92\x31\xc7\xf8"
		"\xf1\x1d\x87\x3a\x7e\x8f\x02\x36\x71\xd9\x4d\x97\xa0\xbd\xda\xdc\x8e\x85\x16\xb2"
		"\x89\x4b\x5b\x5b\x31\x2f\xdf\x74\x3b\x35\x5c\x87\xd8\xbb\xd5\xc4\xe5\xe5\xf7\xf9"
		"\x0d\xeb\xfe\xe5\x7a\x1f\xba\xd5\x81\x4d\x5c\xd2\x75\x2b\xc7\x81\xdb\x98\xef\x24"
		"\x9b\xb8\x6f\xf4\x25\x7e\x73\x67\x9b\x1b\x3e\xe5\x63\xb2\x89\xcb\x7e\x7a\xb1\xdf"
		"\xfc\xd2\xad\xcc\x45\x23\x9b\xb8\xbc\xfc\x46\xff\xf9\xb9\xab\xe1\xeb\xff\x41\x36"
		"\x71\xf6\x50\x2b\xf3\x00\x51\xe6\x12\xb2\x89\xcb\x56\xde\x8c\x76\x65\xee\xf0\xa3"
		"\x39\xb0\x89\xcb\xff\x7b\xb1\xdf\xf2\xd0\x6d\xee\x98\x64\x60\x15\xd8\xc4\x55\x1b"
		"\x4e\xc2\xde\x7d\x1f\xf7\x81\xbf\x04\xf1\xe9\x49\x26\xce\x2e\xf8\xbe\x7f\x3f\xff"
		"\xc3\x78\x2f\x06\x36\x71\x32\xed\x87\x18\x0f\xbf\x71\xd5\x75\xb7\x93\x4d\x5c\x3a"
		"\xe2\x5b\xfe\x3f\xb6\xd8\xf1\x5b\x33\xb0\x89\x4b\xfe\x75\x09\xfc\xf6\x63\xee\xc9"
		"\xbe\xcb\x67\x80\x4d\x5c\x3a\xf4\x46\xcf\xbf\x17\xf8\x91\xff\x5b\x0e\x6c\x83\x3c"
		"\xe2\x76\xff\xf9\xba\xf7\xdc\xe8\xd6\x9f\xa1\xcc\xdb\x4d\x9c\xd5\x2e\xf3\xe5\x8b"
		"\xdf\x42\xdb\x3e\x42\xb6\x41\xf6\x97\xf8\xe1\xad\x8f\x63\x0f\xf0\x30\xd9\xc4\x3d"
		"\x9d\xa7\xf8\x77\xfd\xe9\x6e\x68\xc3\x0d\x64\x13\x77\xef\x37\xdf\x0f\x59\x7c\x25"
		"\xd6\xda\xfd\xc9\x26\xe6\x19\xc6\xf6\xcd\x57\x3a\x7e\x1f\x03\x36\xf1\xb8\xd5\x73"
		"\xfd\x79\xbb\x2c\x75\x0b\x56\x5f\x47\x36\x71\xf7\xeb\xf3\xfc\x17\x35\x57\x62\x2d"
		"\x7d\xa2\x1e\x6c\xe2\x8a\xc6\x46\x8c\x87\x4b\xdc\x65\x49\xf3\x2a\xb0\x89\xef\x9c"
		"\x9e\xf3\xe7\x37\xb5\xe6\xae\x6c\x2a\xcb\x81\x4d\xbc\xbb\x9f\xee\x0f\x9d\xb5\x8f"
		"\xbb\xe3\xcd\x8e\x55\x60\x13\xaf\x28\xc9\xf9\xdd\x7f\x33\xcb\xdd\x7e\xf4\x91\x39"
		"\xb0\x89\x4b\x9f\x38\x2e\xde\x4d\xf7\x2c\x3c\xde\x81\x4d\x9c\x3d\xf4\x4d\xec\xc7"
		"\x7e\xe5\x46\x75\xfc\x2f\xd9\xc4\xe5\x4b\xcf\x46\x7b\xde\xe9\xc6\xd5\x7d\x87\x6c"
		"\xe2\xdd\x66\x9f\xe4\x79\x36\x7f\xd1\xba\xae\xa7\xc1\x26\xee\x59\x78\xaa\xaf\x69"
		"\xbe\xc3\x3d\x99\x3d\x5b\x0f\x36\x71\xd7\xf2\xf9\x9e\x7f\xe7\x00\x73\x21\x07\x36"
		"\xf1\x73\xe7\x1d\xe3\x27\xac\xfe\xad\xeb\x9b\xf6\x26\xd9\xc4\xc9\xe2\x05\xbe\x7a"
		"\xe9\x1d\x6e\x48\xc7\x1e\x0e\x6c\xe2\x93\x8e\xba\x2c\xe6\x0c\xcc\x39\xe8\x2e\xb2"
		"\x89\x4f\x19\xbb\xcc\x77\xdb\xc7\x6e\xde\xa3\x77\x93\x4d\x9c\x5d\x70\x9b\xaf\x6c"
		"\xdd\xe8\x86\xb7\x9e\x4b\x36\x71\xff\xc5\xbf\xf1\xbd\x0f\x7d\xec\x5e\xf4\x6f\xe6"
		"\xc0\x26\x2e\x19\x73\x69\x3c\xeb\x1e\x9b\x85\x7a\xb0\x89\x33\x7f\x76\xbc\xff\xfa"
		"\x63\xe9\x4d\x4f\x83\x4d\x5c\x75\xe1\x4f\x7c\x77\xcd\xc3\xee\xb3\x85\x4b\x1d\xd8"
		"\xc4\x25\x5d\xdf\xf2\x25\x1d\xf7\x3a\x7e\xff\x04\x36\x71\xcf\x8d\xdf\xc7\x7b\xdd"
		"\xc7\xbf\x85\x40\x36\x31\xd6\x3e\x5f\x8c\x1d\x3f\x25\x9b\xb8\x6b\xf9\x8f\xfd\xc6"
		"\x9a\x47\xf8\x8d\x04\xf6\x34\x3f\x36\x71\xff\x5b\xbf\xc0\xbc\x08\x6e\x58\xc7\x39"
		"\x64\x13\x27\xf7\x5f\xe1\xb7\xd4\x3e\x17\xbf\x29\x01\x9b\x78\xfa\x9a\x25\x3e\x3d"
		"\xef\x15\x77\xd1\x7b\x67\x92\x4d\x7c\xf2\xee\x57\xf9\x2d\x4b\x5e\x74\xa7\xdc\x7d"
		"\x35\xd9\xc4\x25\x07\x62\xde\xad\xfb\x08\xef\xd2\x4e\x36\xf1\x43\x7f\xf8\xad\xbf"
		"\xe7\xf0\x72\x5f\x5d\xf7\x22\xd9\xc4\xed\xe5\x6d\xfe\xeb\x87\x0f\xb8\xd7\xa6\xdd"
		"\x40\x36\xf1\x03\x53\xef\xf4\xf7\x3f\x5f\xca\x58\x27\x07\x36\x71\xb2\xcb\xa5\xbe"
		"\x6f\xdd\x87\xee\xce\x92\xfa\x55\x60\x13\x77\xbf\x8e\xf5\xec\xdc\x07\xdd\x0b\xf9"
		"\x39\x29\xd8\xc4\xa5\x73\x4e\x8f\x7f\xa3\x79\xf8\x94\x3b\x1d\xd8\xc4\xff\x0f\x0d"
		"\xa9\x7f\xa1";

	size_t dst_size = 24144;
	char *dst = malloc(dst_size);

	ufbxt_bechmark_begin();
	ptrdiff_t res = ufbxt_inflate(dst, dst_size, src, sizeof(src) - 1, opts);
	double sec = ufbxt_bechmark_end();
	ufbxt_logf("-> %.2f MB/s", (double)dst_size / sec * 1e-6);

	ufbxt_hintf("res = %d", (int)res);
	ufbxt_assert(res == dst_size);
	ufbxt_assert(ufbxt_fnv1a(dst, dst_size) == 0xbaccc7ea);

	free(dst);
}
#endif

#if UFBXT_IMPL
typedef struct {
	const char *prefix;
	size_t prefix_end;

	size_t content_end;

	const char *suffix;
	size_t suffix_end;

	size_t total_size;

	size_t position;
} ufbxt_deflate_byte_stream;

static void ufbxt_deflate_byte_stream_init(ufbxt_deflate_byte_stream *s,
	const char *prefix, size_t prefix_len,
	size_t content_len,
	const char *suffix, size_t suffix_len)
{
	s->position = 0;
	s->prefix = prefix;
	s->prefix_end = prefix_len;

	s->content_end = s->prefix_end + content_len;

	s->suffix = suffix;
	s->suffix_end = s->content_end + suffix_len;

	s->total_size = s->suffix_end;
}

static size_t ufbxt_deflate_byte_stream_read(void *user, void *data, size_t size)
{
	ufbxt_deflate_byte_stream *s = (ufbxt_deflate_byte_stream*)user;
	char *dst = (char*)data;

	for (size_t len = 0; len < size; len++) {
		if (s->position < s->prefix_end) {
			size_t pos = s->position;
			dst[len] = s->prefix[pos];
		} else if (s->position < s->content_end) {
			size_t pos = s->position - s->prefix_end;
			dst[len] = (char)(unsigned char)pos;
		} else if (s->position < s->suffix_end) {
			size_t pos = s->position - s->content_end;
			dst[len] = s->suffix[pos];
		} else {
			return len;
		}

		s->position++;
	}

	return size;
}

static void ufbxt_check_deflate_byte_result(const char *data, size_t length)
{
	for (size_t i = 0; i < length; i++) {
		ufbxt_assert(data[i] == (char)(unsigned char)i);
	}
}

#endif

UFBXT_TEST(deflate_byte_stream)
#if UFBXT_IMPL
{
	const char prefix[] = "\x78\x01\x01\x00\x80\xff\x7f";
	const char suffix[] = "\x3f\xdc\xc3\xb2";
	ufbxt_deflate_byte_stream stream;
	ufbxt_deflate_byte_stream_init(&stream, prefix, sizeof(prefix) - 1, 0x8000, suffix, sizeof(suffix) - 1);

	ufbx_inflate_input input = { 0 };
	input.total_size = stream.total_size;
	input.read_fn = &ufbxt_deflate_byte_stream_read;
	input.read_user = &stream;

	ufbx_inflate_retain retain;
	retain.initialized = false;

	size_t result_len = 0x8000;
	char *result = (char*)malloc(0x8000);
	ufbxt_assert(result);
	ptrdiff_t ret = ufbx_inflate(result, result_len, &input, &retain);
	ufbxt_assert(ret >= 0);
	ufbxt_assert(ret == result_len);
	ufbxt_check_deflate_byte_result(result, result_len);
	free(result);
}
#endif

UFBXT_TEST(deflate_byte_stream_prefix)
#if UFBXT_IMPL
{
	const char prefix[] = "\x78\x01\x01\x00\x80\xff\x7f";
	const char suffix[] = "\x3f\xdc\xc3\xb2";
	ufbxt_deflate_byte_stream stream;
	ufbxt_deflate_byte_stream_init(&stream, NULL, 0, 0x8000, suffix, sizeof(suffix) - 1);

	ufbx_inflate_input input = { 0 };
	input.total_size = stream.total_size + (sizeof(prefix) - 1);
	input.data = prefix;
	input.data_size = sizeof(prefix) - 1;
	input.read_fn = &ufbxt_deflate_byte_stream_read;
	input.read_user = &stream;

	ufbx_inflate_retain retain;
	retain.initialized = false;

	size_t result_len = 0x8000;
	char *result = (char*)malloc(0x8000);
	ufbxt_assert(result);
	ptrdiff_t ret = ufbx_inflate(result, result_len, &input, &retain);
	ufbxt_assert(ret >= 0);
	ufbxt_assert(ret == result_len);
	ufbxt_check_deflate_byte_result(result, result_len);
	free(result);
}
#endif

UFBXT_TEST(deflate_byte_stream_truncated)
#if UFBXT_IMPL
{
	const char prefix[] = "\x78\x01\x01\x00\x80\xff\x7f";
	const char suffix[] = "\x3f\xdc\xc3\xb2";
	ufbxt_deflate_byte_stream stream;
	ufbxt_deflate_byte_stream_init(&stream, prefix, sizeof(prefix) - 1, 0x4000, suffix, sizeof(suffix) - 1);

	ufbx_inflate_input input = { 0 };
	input.total_size = stream.total_size;
	input.read_fn = &ufbxt_deflate_byte_stream_read;
	input.read_user = &stream;

	ufbx_inflate_retain retain;
	retain.initialized = false;

	size_t result_len = 0x8000;
	char *result = (char*)malloc(0x8000);
	ufbxt_assert(result);
	ptrdiff_t ret = ufbx_inflate(result, result_len, &input, &retain);
	ufbxt_assert(ret == -5);
	free(result);
}
#endif

UFBXT_TEST(deflate_byte_stream_no_adler)
#if UFBXT_IMPL
{
	const char prefix[] = "\x78\x01\x01\x00\x80\xff\x7f";
	ufbxt_deflate_byte_stream stream;
	ufbxt_deflate_byte_stream_init(&stream, prefix, sizeof(prefix) - 1, 0x8000, NULL, 0);

	ufbx_inflate_input input = { 0 };
	input.total_size = stream.total_size;
	input.read_fn = &ufbxt_deflate_byte_stream_read;
	input.read_user = &stream;

	ufbx_inflate_retain retain;
	retain.initialized = false;

	size_t result_len = 0x8000;
	char *result = (char*)malloc(0x8000);
	ufbxt_assert(result);
	ptrdiff_t ret = ufbx_inflate(result, result_len, &input, &retain);
	ufbxt_assert(ret == -9);
	free(result);
}
#endif

