/*
 * This file is part of the Trezor project, https://trezor.io/
 *
 * Copyright (c) SatoshiLabs
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "py/mpz.h"
#include "py/objint.h"
#include "py/objstr.h"

#include "embed/upymod/trezorobj.h"

#include "bignum.h"
#include "memzero.h"
#include "monero/monero.h"

/// package: trezorcrypto.monero

typedef struct _mp_obj_hasher_t {
  mp_obj_base_t base;
  Hasher h;
} mp_obj_hasher_t;

typedef struct _mp_obj_ge25519_t {
  mp_obj_base_t base;
  ge25519 p;
} mp_obj_ge25519_t;

typedef struct _mp_obj_bignum256modm_t {
  mp_obj_base_t base;
  bignum256modm p;
} mp_obj_bignum256modm_t;

//
// Helpers
//

STATIC const mp_obj_type_t mod_trezorcrypto_monero_ge25519_type;
STATIC const mp_obj_type_t mod_trezorcrypto_monero_bignum256modm_type;
STATIC const mp_obj_type_t mod_trezorcrypto_monero_hasher_type;

#define MP_OBJ_IS_GE25519(o) \
  MP_OBJ_IS_TYPE((o), &mod_trezorcrypto_monero_ge25519_type)
#define MP_OBJ_IS_SCALAR(o) \
  MP_OBJ_IS_TYPE((o), &mod_trezorcrypto_monero_bignum256modm_type)
#define MP_OBJ_PTR_MPC_GE25519(o) ((const mp_obj_ge25519_t *)(o))
#define MP_OBJ_PTR_MPC_SCALAR(o) ((const mp_obj_bignum256modm_t *)(o))
#define MP_OBJ_PTR_MP_GE25519(o) ((mp_obj_ge25519_t *)(o))
#define MP_OBJ_PTR_MP_SCALAR(o) ((mp_obj_bignum256modm_t *)(o))
#define MP_OBJ_C_GE25519(o) (MP_OBJ_PTR_MPC_GE25519(o)->p)
#define MP_OBJ_GE25519(o) (MP_OBJ_PTR_MP_GE25519(o)->p)
#define MP_OBJ_C_SCALAR(o) (MP_OBJ_PTR_MPC_SCALAR(o)->p)
#define MP_OBJ_SCALAR(o) (MP_OBJ_PTR_MP_SCALAR(o)->p)

static inline void assert_ge25519(const mp_obj_t o) {
  if (!MP_OBJ_IS_GE25519(o)) {
    mp_raise_ValueError("ge25519 expected");
  }
}

static inline void assert_scalar(const mp_obj_t o) {
  if (!MP_OBJ_IS_SCALAR(o)) {
    mp_raise_ValueError("scalar expected");
  }
}

STATIC mp_obj_t mp_obj_new_scalar() {
  mp_obj_bignum256modm_t *o = m_new_obj_with_finaliser(mp_obj_bignum256modm_t);
  o->base.type = &mod_trezorcrypto_monero_bignum256modm_type;
  set256_modm(o->p, 0);
  return MP_OBJ_FROM_PTR(o);
}

STATIC mp_obj_t mp_obj_new_scalar_r(mp_obj_t r) {
  if (r == mp_const_none) {
    return mp_obj_new_scalar();
  }

  assert_scalar(r);
  return r;
}

STATIC mp_obj_t mp_obj_new_ge25519() {
  mp_obj_ge25519_t *o = m_new_obj_with_finaliser(mp_obj_ge25519_t);
  o->base.type = &mod_trezorcrypto_monero_ge25519_type;
  ge25519_set_neutral(&o->p);
  return MP_OBJ_FROM_PTR(o);
}

STATIC mp_obj_t mp_obj_new_ge25519_r(mp_obj_t r) {
  if (r == mp_const_none) {
    return mp_obj_new_ge25519();
  }

  assert_ge25519(r);
  return r;
}

STATIC void mp_unpack_ge25519(ge25519 *r, const mp_obj_t arg, mp_int_t offset) {
  mp_buffer_info_t buff = {0};
  mp_get_buffer_raise(arg, &buff, MP_BUFFER_READ);
  if (buff.len < 32 + offset) {
    mp_raise_ValueError("Invalid length of the EC point");
  }

  const int res = ge25519_unpack_vartime(r, ((uint8_t *)buff.buf) + offset);
  if (res != 1) {
    mp_raise_ValueError("Point decoding error");
  }
}

STATIC void mp_unpack_scalar(bignum256modm r, const mp_obj_t arg,
                             mp_int_t offset) {
  mp_buffer_info_t buff = {0};
  mp_get_buffer_raise(arg, &buff, MP_BUFFER_READ);
  if (buff.len < 32 + offset) {
    mp_raise_ValueError("Invalid length of secret key");
  }
  expand256_modm(r, ((uint8_t *)buff.buf) + offset, 32);
}

//
// Constructors
//

/// class Point:
///     """
///     EC point on ED25519
///     """

///     def __init__(self, x: Point | bytes | None = None):
///         """
///         Constructor
///         """

STATIC mp_obj_t mod_trezorcrypto_monero_ge25519_make_new(
    const mp_obj_type_t *type, size_t n_args, size_t n_kw,
    const mp_obj_t *args) {
  mp_arg_check_num(n_args, n_kw, 0, 1, false);
  mp_obj_ge25519_t *o = m_new_obj_with_finaliser(mp_obj_ge25519_t);
  o->base.type = type;

  if (n_args == 0 || args[0] == mp_const_none) {
    ge25519_set_neutral(&o->p);
  } else if (n_args == 1 && MP_OBJ_IS_GE25519(args[0])) {
    ge25519_copy(&o->p, &MP_OBJ_C_GE25519(args[0]));
  } else if (n_args == 1 && MP_OBJ_IS_STR_OR_BYTES(args[0])) {
    mp_unpack_ge25519(&o->p, args[0], 0);
  } else {
    m_del_obj(mp_obj_ge25519_t, o);
    mp_raise_ValueError("Invalid ge25519 constructor");
  }

  return MP_OBJ_FROM_PTR(o);
}

STATIC mp_obj_t mod_trezorcrypto_monero_ge25519___del__(mp_obj_t self) {
  mp_obj_ge25519_t *o = MP_OBJ_TO_PTR(self);
  memzero(&(o->p), sizeof(ge25519));
  return mp_const_none;
}
STATIC MP_DEFINE_CONST_FUN_OBJ_1(mod_trezorcrypto_monero_ge25519___del___obj,
                                 mod_trezorcrypto_monero_ge25519___del__);

/// class Scalar:
///     """
///     EC scalar on SC25519
///     """

///     def __init__(self, x: Scalar | bytes | int | None = None):
///         """
///         Constructor
///         """

STATIC mp_obj_t mod_trezorcrypto_monero_bignum256modm_make_new(
    const mp_obj_type_t *type, size_t n_args, size_t n_kw,
    const mp_obj_t *args) {
  mp_arg_check_num(n_args, n_kw, 0, 1, false);
  mp_obj_bignum256modm_t *o = m_new_obj_with_finaliser(mp_obj_bignum256modm_t);
  o->base.type = type;

  if (n_args == 0 || args[0] == mp_const_none) {
    set256_modm(o->p, 0);
  } else if (n_args == 1 && MP_OBJ_IS_SCALAR(args[0])) {
    copy256_modm(o->p, MP_OBJ_C_SCALAR(args[0]));
  } else if (n_args == 1 && MP_OBJ_IS_STR_OR_BYTES(args[0])) {
    mp_unpack_scalar(o->p, args[0], 0);
  } else if (n_args == 1 && mp_obj_is_integer(args[0])) {
    uint64_t v = trezor_obj_get_uint64(args[0]);
    set256_modm(o->p, v);
  } else {
    m_del_obj(mp_obj_bignum256modm_t, o);
    mp_raise_ValueError("Invalid scalar constructor");
  }

  return MP_OBJ_FROM_PTR(o);
}

STATIC mp_obj_t mod_trezorcrypto_monero_bignum256modm___del__(mp_obj_t self) {
  mp_obj_bignum256modm_t *o = MP_OBJ_TO_PTR(self);
  memzero(o->p, sizeof(bignum256modm));
  return mp_const_none;
}
STATIC MP_DEFINE_CONST_FUN_OBJ_1(
    mod_trezorcrypto_monero_bignum256modm___del___obj,
    mod_trezorcrypto_monero_bignum256modm___del__);

/// class Hasher:
///     """
///     XMR hasher
///     """

///     def __init__(self, x: bytes | None = None):
///         """
///         Constructor
///         """

///     def update(self, buffer: bytes) -> None:
///         """
///         Update hasher
///         """

///     def digest(self) -> bytes:
///         """
///         Computes digest
///         """

///     def copy(self) -> Hasher:
///         """
///         Creates copy of the hasher, preserving the state
///         """

STATIC mp_obj_t mod_trezorcrypto_monero_hasher_make_new(
    const mp_obj_type_t *type, size_t n_args, size_t n_kw,
    const mp_obj_t *args) {
  mp_arg_check_num(n_args, n_kw, 0, 1, false);
  mp_obj_hasher_t *o = m_new_obj_with_finaliser(mp_obj_hasher_t);
  o->base.type = type;
  xmr_hasher_init(&(o->h));

  if (n_args == 1 && MP_OBJ_IS_STR_OR_BYTES(args[0])) {
    mp_buffer_info_t buff = {0};
    mp_get_buffer_raise(args[0], &buff, MP_BUFFER_READ);
    xmr_hasher_update(&o->h, buff.buf, buff.len);
  }

  return MP_OBJ_FROM_PTR(o);
}

STATIC mp_obj_t mod_trezorcrypto_monero_hasher___del__(mp_obj_t self) {
  mp_obj_hasher_t *o = MP_OBJ_TO_PTR(self);
  memzero(&(o->h), sizeof(Hasher));
  return mp_const_none;
}
STATIC MP_DEFINE_CONST_FUN_OBJ_1(mod_trezorcrypto_monero_hasher___del___obj,
                                 mod_trezorcrypto_monero_hasher___del__);

//
// Scalar defs
//

/// mock:global

/// def sc_copy(
///     dst: Scalar | None, val: int | bytes | Scalar
/// ) -> Scalar:
///     """
///     Initializes a scalar
///     """
STATIC mp_obj_t mod_trezorcrypto_monero_sc_copy(const mp_obj_t dest,
                                                const mp_obj_t src) {
  mp_obj_t res = mp_obj_new_scalar_r(dest);

  if (MP_OBJ_IS_SCALAR(src)) {
    copy256_modm(MP_OBJ_SCALAR(res), MP_OBJ_C_SCALAR(src));
  } else if (MP_OBJ_IS_STR_OR_BYTES(src)) {
    mp_unpack_scalar(MP_OBJ_SCALAR(res), src, 0);
  } else if (mp_obj_is_integer(src)) {
    uint64_t v = trezor_obj_get_uint64(src);
    set256_modm(MP_OBJ_SCALAR(res), v);
  } else {
    mp_raise_ValueError("Invalid scalar def");
  }
  return res;
}
STATIC MP_DEFINE_CONST_FUN_OBJ_2(mod_trezorcrypto_monero_sc_copy_obj,
                                 mod_trezorcrypto_monero_sc_copy);

/// def sc_check(val: Scalar) -> None:
///     """
///     Throws exception if scalar is invalid
///     """
STATIC mp_obj_t mod_trezorcrypto_monero_sc_check(const mp_obj_t arg) {
  assert_scalar(arg);
  if (check256_modm(MP_OBJ_C_SCALAR(arg)) != 1) {
    mp_raise_ValueError("Ed25519 scalar invalid");
  }
  return mp_const_none;
}
STATIC MP_DEFINE_CONST_FUN_OBJ_1(mod_trezorcrypto_monero_sc_check_obj,
                                 mod_trezorcrypto_monero_sc_check);

/// def sc_iszero(val: Scalar) -> bool:
///     """
///     Returns False if the scalar is zero
///     """
STATIC mp_obj_t mod_trezorcrypto_monero_sc_iszero(const mp_obj_t arg) {
  assert_scalar(arg);
  const int r = iszero256_modm(MP_OBJ_C_SCALAR(arg));
  return mp_obj_new_int(r);
}
STATIC MP_DEFINE_CONST_FUN_OBJ_1(mod_trezorcrypto_monero_sc_iszero_obj,
                                 mod_trezorcrypto_monero_sc_iszero);

/// def sc_eq(a: Scalar, b: Scalar) -> int:
///     """
///     Compares scalars, returns 1 on the same value
///     """
STATIC mp_obj_t mod_trezorcrypto_monero_sc_eq(const mp_obj_t a,
                                              const mp_obj_t b) {
  assert_scalar(a);
  assert_scalar(b);
  int r = eq256_modm(MP_OBJ_C_SCALAR(a), MP_OBJ_C_SCALAR(b));
  return MP_OBJ_NEW_SMALL_INT(r);
}
STATIC MP_DEFINE_CONST_FUN_OBJ_2(mod_trezorcrypto_monero_sc_eq_obj,
                                 mod_trezorcrypto_monero_sc_eq);

/// def sc_add_into(r: Scalar | None, a: Scalar, b: Scalar) -> Scalar:
///     """
///     Scalar addition
///     """
STATIC mp_obj_t mod_trezorcrypto_monero_sc_add_into(const mp_obj_t dest,
                                                    const mp_obj_t a,
                                                    const mp_obj_t b) {
  mp_obj_t res = mp_obj_new_scalar_r(dest);

  assert_scalar(a);
  assert_scalar(b);
  add256_modm(MP_OBJ_SCALAR(res), MP_OBJ_C_SCALAR(a), MP_OBJ_C_SCALAR(b));
  return res;
}
STATIC MP_DEFINE_CONST_FUN_OBJ_3(mod_trezorcrypto_monero_sc_add_into_obj,
                                 mod_trezorcrypto_monero_sc_add_into);

/// def sc_sub_into(r: Scalar | None, a: Scalar, b: Scalar) -> Scalar:
///     """
///     Scalar subtraction
///     """
STATIC mp_obj_t mod_trezorcrypto_monero_sc_sub_into(const mp_obj_t dest,
                                                    const mp_obj_t a,
                                                    const mp_obj_t b) {
  mp_obj_t res = mp_obj_new_scalar_r(dest);

  assert_scalar(a);
  assert_scalar(b);
  sub256_modm(MP_OBJ_SCALAR(res), MP_OBJ_C_SCALAR(a), MP_OBJ_C_SCALAR(b));
  return res;
}
STATIC MP_DEFINE_CONST_FUN_OBJ_3(mod_trezorcrypto_monero_sc_sub_into_obj,
                                 mod_trezorcrypto_monero_sc_sub_into);

/// def sc_mul_into(r: Scalar | None, a: Scalar, b: Scalar) -> Scalar:
///     """
///     Scalar multiplication
///     """
STATIC mp_obj_t mod_trezorcrypto_monero_sc_mul_into(const mp_obj_t dest,
                                                    const mp_obj_t a,
                                                    const mp_obj_t b) {
  mp_obj_t res = mp_obj_new_scalar_r(dest);

  assert_scalar(a);
  assert_scalar(b);
  mul256_modm(MP_OBJ_SCALAR(res), MP_OBJ_C_SCALAR(a), MP_OBJ_C_SCALAR(b));
  return res;
}
STATIC MP_DEFINE_CONST_FUN_OBJ_3(mod_trezorcrypto_monero_sc_mul_into_obj,
                                 mod_trezorcrypto_monero_sc_mul_into);

/// def sc_mulsub_into(
///     r: Scalar | None, a: Scalar, b: Scalar, c: Scalar
/// ) -> Scalar:
///     """
///     c - a*b
///     """
STATIC mp_obj_t mod_trezorcrypto_monero_sc_mulsub_into(size_t n_args,
                                                       const mp_obj_t *args) {
  mp_arg_check_num(n_args, 0, 4, 4, false);
  mp_obj_t res = mp_obj_new_scalar_r(args[0]);

  assert_scalar(args[1]);
  assert_scalar(args[2]);
  assert_scalar(args[3]);
  mulsub256_modm(MP_OBJ_SCALAR(res), MP_OBJ_C_SCALAR(args[1]),
                 MP_OBJ_C_SCALAR(args[2]), MP_OBJ_C_SCALAR(args[3]));
  return res;
}
STATIC MP_DEFINE_CONST_FUN_OBJ_VAR_BETWEEN(
    mod_trezorcrypto_monero_sc_mulsub_into_obj, 4, 4,
    mod_trezorcrypto_monero_sc_mulsub_into);

/// def sc_muladd_into(
///     r: Scalar | None, a: Scalar, b: Scalar, c: Scalar
/// ) -> Scalar:
///     """
///     c + a*b
///     """
STATIC mp_obj_t mod_trezorcrypto_monero_sc_muladd_into(size_t n_args,
                                                       const mp_obj_t *args) {
  mp_arg_check_num(n_args, 0, 4, 4, false);
  mp_obj_t res = mp_obj_new_scalar_r(args[0]);

  assert_scalar(args[1]);
  assert_scalar(args[2]);
  assert_scalar(args[3]);
  muladd256_modm(MP_OBJ_SCALAR(res), MP_OBJ_C_SCALAR(args[1]),
                 MP_OBJ_C_SCALAR(args[2]), MP_OBJ_C_SCALAR(args[3]));
  return res;
}
STATIC MP_DEFINE_CONST_FUN_OBJ_VAR_BETWEEN(
    mod_trezorcrypto_monero_sc_muladd_into_obj, 4, 4,
    mod_trezorcrypto_monero_sc_muladd_into);

/// def sc_inv_into(r: Scalar | None, a: Scalar) -> Scalar:
///     """
///     Scalar modular inversion
///     """
STATIC mp_obj_t mod_trezorcrypto_monero_sc_inv_into(const mp_obj_t dest,
                                                    const mp_obj_t arg) {
  mp_obj_t res = mp_obj_new_scalar_r(dest);
  assert_scalar(arg);

  // bn_prime = curve order, little endian encoded
  bignum256 bn_prime = {.val = {0x1cf5d3ed, 0x9318d2, 0x1de73596, 0x1df3bd45,
                                0x14d, 0x0, 0x0, 0x0, 0x100000}};
  bignum256 bn_x = {0};
  bignum256modm bm_x = {0};
  uint8_t raw_x[32] = {0};

  memcpy(bm_x, MP_OBJ_C_SCALAR(arg), sizeof(bignum256modm));
  contract256_modm(raw_x, bm_x);
  bn_read_le(raw_x, &bn_x);

  bn_inverse(&bn_x, &bn_prime);

  bn_write_le(&bn_x, raw_x);
  expand_raw256_modm(bm_x, raw_x);
  memcpy(MP_OBJ_SCALAR(res), bm_x, sizeof(bignum256modm));

  return res;
}
STATIC MP_DEFINE_CONST_FUN_OBJ_2(mod_trezorcrypto_monero_sc_inv_into_obj,
                                 mod_trezorcrypto_monero_sc_inv_into);

/// def encodeint_into(
///     r: bytes | None, a: Scalar, offset: int | None = 0
/// ) -> bytes:
///     """
///     Scalar compression
///     """
STATIC mp_obj_t mod_trezorcrypto_monero_encodeint_into(size_t n_args,
                                                       const mp_obj_t *args) {
  mp_arg_check_num(n_args, 0, 2, 3, false);
  if (args[0] == mp_const_none) {
    assert_scalar(args[1]);
    vstr_t out = {0};
    vstr_init_len(&out, 32);
    contract256_modm((uint8_t *)out.buf, MP_OBJ_C_SCALAR(args[1]));
    return mp_obj_new_str_from_vstr(&mp_type_bytes, &out);
  } else {
    mp_buffer_info_t bufm = {0};
    mp_get_buffer_raise(args[0], &bufm, MP_BUFFER_WRITE);
    const mp_int_t offset = n_args >= 3 ? mp_obj_get_int(args[2]) : 0;
    if (bufm.len < 32 + offset) {
      mp_raise_ValueError("Buffer too small");
    }

    contract256_modm(((uint8_t *)bufm.buf) + offset, MP_OBJ_C_SCALAR(args[1]));
    return args[0];
  }
}
STATIC MP_DEFINE_CONST_FUN_OBJ_VAR_BETWEEN(
    mod_trezorcrypto_monero_encodeint_into_obj, 2, 3,
    mod_trezorcrypto_monero_encodeint_into);

/// def decodeint_into(
///     r: Scalar | None, a: bytes, offset: int = 0
/// ) -> Scalar:
///     """
///     Scalar decompression
///     """
STATIC mp_obj_t mod_trezorcrypto_monero_decodeint_into(size_t n_args,
                                                       const mp_obj_t *args) {
  mp_arg_check_num(n_args, 0, 2, 3, false);
  mp_obj_t res = mp_obj_new_scalar_r(args[0]);
  const mp_int_t offset = n_args >= 3 ? mp_obj_get_int(args[2]) : 0;
  mp_unpack_scalar(MP_OBJ_SCALAR(res), args[1], offset);
  return res;
}
STATIC MP_DEFINE_CONST_FUN_OBJ_VAR_BETWEEN(
    mod_trezorcrypto_monero_decodeint_into_obj, 2, 3,
    mod_trezorcrypto_monero_decodeint_into);

/// def decodeint_into_noreduce(
///     r: Scalar | None, a: bytes, offset: int = 0
/// ) -> Scalar:
///     """
///     Scalar decompression, raw, without modular reduction
///     """
STATIC mp_obj_t mod_trezorcrypto_monero_decodeint_into_noreduce(
    size_t n_args, const mp_obj_t *args) {
  mp_arg_check_num(n_args, 0, 2, 3, false);
  mp_obj_t res = mp_obj_new_scalar_r(args[0]);
  const mp_int_t offset = n_args >= 3 ? mp_obj_get_int(args[2]) : 0;

  mp_buffer_info_t buff = {0};
  mp_get_buffer_raise(args[1], &buff, MP_BUFFER_READ);
  if (buff.len != 32 + offset) {
    mp_raise_ValueError("Invalid length of secret key");
  }

  expand_raw256_modm(MP_OBJ_SCALAR(res), ((uint8_t *)buff.buf) + offset);
  return res;
}
STATIC MP_DEFINE_CONST_FUN_OBJ_VAR_BETWEEN(
    mod_trezorcrypto_monero_decodeint_into_noreduce_obj, 2, 3,
    mod_trezorcrypto_monero_decodeint_into_noreduce);

//
// GE25519 Defs
//

/// def identity_into(r: Point | None = None) -> Point:
///     """
///     Sets neutral point
///     """
STATIC mp_obj_t mod_trezorcrypto_monero_identity_into(size_t n_args,
                                                      const mp_obj_t *args) {
  mp_obj_t res = mp_obj_new_ge25519_r(n_args == 1 ? args[0] : mp_const_none);
  ge25519_set_neutral(&MP_OBJ_GE25519(res));
  return res;
}
STATIC MP_DEFINE_CONST_FUN_OBJ_VAR_BETWEEN(
    mod_trezorcrypto_monero_identity_into_obj, 0, 1,
    mod_trezorcrypto_monero_identity_into);

/// def xmr_H(r: Point | None = None) -> Point:
///     """
///     Sets H point
///     """
STATIC mp_obj_t mod_trezorcrypto_monero_xmr_H(size_t n_args,
                                              const mp_obj_t *args) {
  mp_obj_t res = mp_obj_new_ge25519_r(n_args == 1 ? args[0] : mp_const_none);
  ge25519_set_xmr_h(&MP_OBJ_GE25519(res));
  return res;
}
STATIC MP_DEFINE_CONST_FUN_OBJ_VAR_BETWEEN(mod_trezorcrypto_monero_xmr_H_obj, 0,
                                           1, mod_trezorcrypto_monero_xmr_H);

/// def ge25519_check(r: Point) -> None:
///     """
///     Checks point, throws if not on curve
///     """
STATIC mp_obj_t mod_trezorcrypto_monero_ge25519_check(const mp_obj_t arg) {
  assert_ge25519(arg);
  if (ge25519_check(&MP_OBJ_C_GE25519(arg)) != 1) {
    mp_raise_ValueError("Ed25519 point not on curve");
  }
  return mp_const_none;
}
STATIC MP_DEFINE_CONST_FUN_OBJ_1(mod_trezorcrypto_monero_ge25519_check_obj,
                                 mod_trezorcrypto_monero_ge25519_check);

/// def point_eq(a: Point, b: Point) -> bool:
///     """
///     Compares EC points
///     """
STATIC mp_obj_t mod_trezorcrypto_monero_point_eq(const mp_obj_t a,
                                                 const mp_obj_t b) {
  assert_ge25519(a);
  assert_ge25519(b);
  int r = ge25519_eq(&MP_OBJ_C_GE25519(a), &MP_OBJ_C_GE25519(b));
  return MP_OBJ_NEW_SMALL_INT(r);
}
STATIC MP_DEFINE_CONST_FUN_OBJ_2(mod_trezorcrypto_monero_point_eq_obj,
                                 mod_trezorcrypto_monero_point_eq);

/// def point_add_into(r: Point | None, a: Point, b: Point) -> Point:
///     """
///     Adds EC points
///     """
STATIC mp_obj_t mod_trezorcrypto_monero_point_add_into(const mp_obj_t dest,
                                                       const mp_obj_t a,
                                                       const mp_obj_t b) {
  mp_obj_t res = mp_obj_new_ge25519_r(dest);

  assert_ge25519(a);
  assert_ge25519(b);
  ge25519_add(&MP_OBJ_GE25519(res), &MP_OBJ_C_GE25519(a), &MP_OBJ_C_GE25519(b),
              0);
  return res;
}
STATIC MP_DEFINE_CONST_FUN_OBJ_3(mod_trezorcrypto_monero_point_add_into_obj,
                                 mod_trezorcrypto_monero_point_add_into);

/// def point_sub_into(r: Point | None, a: Point, b: Point) -> Point:
///     """
///     Subtracts EC points
///     """
STATIC mp_obj_t mod_trezorcrypto_monero_point_sub_into(const mp_obj_t dest,
                                                       const mp_obj_t a,
                                                       const mp_obj_t b) {
  mp_obj_t res = mp_obj_new_ge25519_r(dest);

  assert_ge25519(a);
  assert_ge25519(b);
  ge25519_add(&MP_OBJ_GE25519(res), &MP_OBJ_C_GE25519(a), &MP_OBJ_C_GE25519(b),
              1);
  return res;
}
STATIC MP_DEFINE_CONST_FUN_OBJ_3(mod_trezorcrypto_monero_point_sub_into_obj,
                                 mod_trezorcrypto_monero_point_sub_into);

/// def ge25519_mul8(r: Point | None, p: Point) -> Point:
///     """
///     EC point * 8
///     """
STATIC mp_obj_t mod_trezorcrypto_monero_ge25519_mul8(const mp_obj_t dest,
                                                     const mp_obj_t src) {
  mp_obj_t res = mp_obj_new_ge25519_r(dest);
  assert_ge25519(src);
  ge25519_mul8(&MP_OBJ_GE25519(res), &MP_OBJ_C_GE25519(src));
  return res;
}
STATIC MP_DEFINE_CONST_FUN_OBJ_2(mod_trezorcrypto_monero_ge25519_mul8_obj,
                                 mod_trezorcrypto_monero_ge25519_mul8);

/// def ge25519_double_scalarmult_vartime_into(
///     r: Point | None, p1: Point, s1: Scalar, s2: Scalar
/// ) -> Point:
///     """
///     s1 * G + s2 * p1
///     """
STATIC mp_obj_t mod_trezorcrypto_monero_ge25519_double_scalarmult_vartime_into(
    size_t n_args, const mp_obj_t *args) {
  mp_arg_check_num(n_args, 0, 4, 4, false);
  mp_obj_t res = mp_obj_new_ge25519_r(args[0]);
  assert_ge25519(args[1]);
  assert_scalar(args[2]);
  assert_scalar(args[3]);

  ge25519_double_scalarmult_vartime(
      &MP_OBJ_GE25519(res), &MP_OBJ_C_GE25519(args[1]),
      MP_OBJ_C_SCALAR(args[2]), MP_OBJ_C_SCALAR(args[3]));
  return res;
}
STATIC MP_DEFINE_CONST_FUN_OBJ_VAR_BETWEEN(
    mod_trezorcrypto_monero_ge25519_double_scalarmult_vartime_into_obj, 4, 4,
    mod_trezorcrypto_monero_ge25519_double_scalarmult_vartime_into);

/// def scalarmult_base_into(
///     r: Point | None, s: Scalar | int
/// ) -> Point:
///     """
///     s * G
///     """
STATIC mp_obj_t mod_trezorcrypto_monero_scalarmult_base_into(
    const mp_obj_t dest, const mp_obj_t src) {
  mp_obj_t res = mp_obj_new_ge25519_r(dest);

  if (MP_OBJ_IS_SCALAR(src)) {
    ge25519_scalarmult_base_wrapper(&MP_OBJ_GE25519(res), MP_OBJ_C_SCALAR(src));
  } else if (mp_obj_is_integer(src)) {
    bignum256modm mlt = {0};
    set256_modm(mlt, mp_obj_get_int(src));
    ge25519_scalarmult_base_wrapper(&MP_OBJ_GE25519(res), mlt);
  } else {
    mp_raise_ValueError("unknown base mult type");
  }

  return res;
}
STATIC MP_DEFINE_CONST_FUN_OBJ_2(
    mod_trezorcrypto_monero_scalarmult_base_into_obj,
    mod_trezorcrypto_monero_scalarmult_base_into);

/// def scalarmult_into(
///     r: Point | None, p: Point, s: Scalar | int
/// ) -> Point:
///     """
///     s * p
///     """
STATIC mp_obj_t mod_trezorcrypto_monero_scalarmult_into(const mp_obj_t dest,
                                                        const mp_obj_t p,
                                                        const mp_obj_t s) {
  mp_obj_t res = mp_obj_new_ge25519_r(dest);
  assert_ge25519(p);

  if (MP_OBJ_IS_SCALAR(s)) {
    ge25519_scalarmult(&MP_OBJ_GE25519(res), &MP_OBJ_C_GE25519(p),
                       MP_OBJ_C_SCALAR(s));
  } else if (mp_obj_is_integer(s)) {
    bignum256modm mlt = {0};
    set256_modm(mlt, mp_obj_get_int(s));
    ge25519_scalarmult(&MP_OBJ_GE25519(res), &MP_OBJ_C_GE25519(p), mlt);
  } else {
    mp_raise_ValueError("unknown mult type");
  }

  return res;
}
STATIC MP_DEFINE_CONST_FUN_OBJ_3(mod_trezorcrypto_monero_scalarmult_into_obj,
                                 mod_trezorcrypto_monero_scalarmult_into);

/// def encodepoint_into(r: bytes | None, p: Point, offset: int = 0) -> bytes:
///     """
///     Point compression
///     """
STATIC mp_obj_t mod_trezorcrypto_monero_encodepoint_into(size_t n_args,
                                                         const mp_obj_t *args) {
  mp_arg_check_num(n_args, 0, 2, 3, false);
  if (args[0] == mp_const_none) {
    assert_ge25519(args[1]);
    vstr_t out = {0};
    vstr_init_len(&out, 32);
    ge25519_pack((uint8_t *)out.buf, &MP_OBJ_C_GE25519(args[1]));
    return mp_obj_new_str_from_vstr(&mp_type_bytes, &out);
  } else {
    mp_buffer_info_t bufm = {0};
    mp_get_buffer_raise(args[0], &bufm, MP_BUFFER_WRITE);
    const mp_int_t offset = n_args >= 3 ? mp_obj_get_int(args[2]) : 0;
    if (bufm.len < 32 + offset) {
      mp_raise_ValueError("Buffer too small");
    }

    ge25519_pack(((uint8_t *)bufm.buf) + offset, &MP_OBJ_C_GE25519(args[1]));
    return args[0];
  }
}
STATIC MP_DEFINE_CONST_FUN_OBJ_VAR_BETWEEN(
    mod_trezorcrypto_monero_encodepoint_into_obj, 2, 3,
    mod_trezorcrypto_monero_encodepoint_into);

/// def decodepoint_into(
///     r: Point | None, buff: bytes, offset: int = 0
/// ) -> Point:
///     """
///     Point decompression
///     """
STATIC mp_obj_t mod_trezorcrypto_monero_decodepoint_into(size_t n_args,
                                                         const mp_obj_t *args) {
  mp_arg_check_num(n_args, 0, 2, 3, false);
  mp_obj_t res = mp_obj_new_ge25519_r(args[0]);
  const mp_int_t offset = n_args >= 3 ? mp_obj_get_int(args[2]) : 0;
  mp_unpack_ge25519(&MP_OBJ_GE25519(res), args[1], offset);
  return res;
}
STATIC MP_DEFINE_CONST_FUN_OBJ_VAR_BETWEEN(
    mod_trezorcrypto_monero_decodepoint_into_obj, 2, 3,
    mod_trezorcrypto_monero_decodepoint_into);

//
// XMR defs
//

/// def xmr_base58_addr_encode_check(tag: int, buff: bytes) -> str:
///     """
///     Monero block base 58 encoding
///     """
STATIC mp_obj_t mod_trezorcrypto_monero_xmr_base58_addr_encode_check(
    const mp_obj_t tag, const mp_obj_t buff) {
  vstr_t out = {0};
  vstr_init_len(&out, 128);
  mp_buffer_info_t data = {0};
  mp_get_buffer_raise(buff, &data, MP_BUFFER_READ);

  int sz = xmr_base58_addr_encode_check(mp_obj_get_int(tag), data.buf, data.len,
                                        out.buf, out.alloc);
  if (sz <= 0) {
    vstr_clear(&out);
    mp_raise_ValueError("b58 encoding error");
  }
  out.len = sz;

  return mp_obj_new_str_from_vstr(&mp_type_str, &out);
}
STATIC MP_DEFINE_CONST_FUN_OBJ_2(
    mod_trezorcrypto_monero_xmr_base58_addr_encode_check_obj,
    mod_trezorcrypto_monero_xmr_base58_addr_encode_check);

/// def xmr_base58_addr_decode_check(buff: bytes) -> tuple[bytes, int]:
///     """
///     Monero block base 58 decoding, returning (decoded, tag) or raising on
///     error.
///     """
STATIC mp_obj_t
mod_trezorcrypto_monero_xmr_base58_addr_decode_check(const mp_obj_t buff) {
  uint64_t tag = 0;
  vstr_t out = {0};
  vstr_init_len(&out, 128);

  mp_buffer_info_t data = {0};
  mp_get_buffer_raise(buff, &data, MP_BUFFER_READ);

  int sz = xmr_base58_addr_decode_check(data.buf, data.len, &tag, out.buf,
                                        out.alloc);
  if (sz <= 0) {
    vstr_clear(&out);
    mp_raise_ValueError("b58 decoding error");
  }
  out.len = sz;

  mp_obj_tuple_t *tuple = MP_OBJ_TO_PTR(mp_obj_new_tuple(2, NULL));
  tuple->items[0] = mp_obj_new_str_from_vstr(&mp_type_bytes, &out);
  tuple->items[1] = mp_obj_new_int_from_ull(tag);
  return MP_OBJ_FROM_PTR(tuple);
}
STATIC MP_DEFINE_CONST_FUN_OBJ_1(
    mod_trezorcrypto_monero_xmr_base58_addr_decode_check_obj,
    mod_trezorcrypto_monero_xmr_base58_addr_decode_check);

/// def random_scalar(r: Scalar | None = None) -> Scalar:
///     """
///     Generates a random scalar
///     """
STATIC mp_obj_t mod_trezorcrypto_monero_random_scalar(size_t n_args,
                                                      const mp_obj_t *args) {
  mp_obj_t res = mp_obj_new_scalar_r(n_args == 1 ? args[0] : mp_const_none);
  xmr_random_scalar(MP_OBJ_SCALAR(res));
  return res;
}
STATIC MP_DEFINE_CONST_FUN_OBJ_VAR_BETWEEN(
    mod_trezorcrypto_monero_random_scalar_obj, 0, 1,
    mod_trezorcrypto_monero_random_scalar);

/// def fast_hash_into(
///    r: bytes | None,
///    buff: bytes,
///    length: int | None = None,
///    offset: int = 0,
/// ) -> bytes:
///     """
///     XMR fast hash
///     """
STATIC mp_obj_t mod_trezorcrypto_monero_fast_hash_into(size_t n_args,
                                                       const mp_obj_t *args) {
  mp_arg_check_num(n_args, 0, 2, 4, false);
  vstr_t out = {0};
  uint8_t *buff_use = NULL;
  if (args[0] != mp_const_none) {
    mp_buffer_info_t odata = {0};
    mp_get_buffer_raise(args[0], &odata, MP_BUFFER_WRITE);
    if (odata.len < HASHER_DIGEST_LENGTH) {
      vstr_clear(&out);
      mp_raise_ValueError("Output buffer too small");
    }
    buff_use = odata.buf;
  } else {
    vstr_init_len(&out, HASHER_DIGEST_LENGTH);
    buff_use = (uint8_t *)out.buf;
  }

  mp_buffer_info_t data = {0};
  mp_get_buffer_raise(args[1], &data, MP_BUFFER_READ);
  mp_int_t length = n_args >= 3 ? mp_obj_get_int(args[2]) : data.len;
  mp_int_t offset = n_args >= 4 ? mp_obj_get_int(args[3]) : 0;
  if (length < 0) length += data.len;
  if (offset < 0) offset += data.len;
  if (length < 0 || offset < 0 || offset + length > data.len) {
    mp_raise_ValueError("Illegal offset/length");
  }
  xmr_fast_hash(buff_use, (const char *)data.buf + offset, length);
  return args[0] != mp_const_none
             ? args[0]
             : mp_obj_new_str_from_vstr(&mp_type_bytes, &out);
}
STATIC MP_DEFINE_CONST_FUN_OBJ_VAR_BETWEEN(
    mod_trezorcrypto_monero_fast_hash_into_obj, 2, 4,
    mod_trezorcrypto_monero_fast_hash_into);

/// def hash_to_point_into(
///     r: Point | None,
///     buff: bytes,
///     length: int | None = None,
///     offset: int = 0,
/// ) -> Point:
///     """
///     XMR hashing to EC point
///     """
STATIC mp_obj_t mod_trezorcrypto_monero_hash_to_point_into(
    size_t n_args, const mp_obj_t *args) {
  mp_arg_check_num(n_args, 0, 2, 4, false);
  mp_obj_t res = mp_obj_new_ge25519_r(args[0]);
  mp_buffer_info_t data = {0};
  mp_get_buffer_raise(args[1], &data, MP_BUFFER_READ);
  mp_int_t length = n_args >= 3 ? mp_obj_get_int(args[2]) : data.len;
  mp_int_t offset = n_args >= 4 ? mp_obj_get_int(args[3]) : 0;
  if (length < 0) length += data.len;
  if (offset < 0) offset += data.len;
  if (length < 0 || offset < 0 || offset + length > data.len) {
    mp_raise_ValueError("Illegal offset/length");
  }

  xmr_hash_to_ec(&MP_OBJ_GE25519(res), (const char *)data.buf + offset, length);
  return res;
}
STATIC MP_DEFINE_CONST_FUN_OBJ_VAR_BETWEEN(
    mod_trezorcrypto_monero_hash_to_point_into_obj, 2, 4,
    mod_trezorcrypto_monero_hash_to_point_into);

/// def hash_to_scalar_into(
///    r: Scalar | None,
///    buff: bytes,
///    length: int | None = None,
///    offset: int = 0,
/// ) -> Scalar:
///     """
///     XMR hashing to EC scalar
///     """
STATIC mp_obj_t mod_trezorcrypto_monero_hash_to_scalar_into(
    size_t n_args, const mp_obj_t *args) {
  mp_arg_check_num(n_args, 0, 2, 4, false);
  mp_obj_t res = mp_obj_new_scalar_r(args[0]);
  mp_buffer_info_t data = {0};
  mp_get_buffer_raise(args[1], &data, MP_BUFFER_READ);
  mp_int_t length = n_args >= 3 ? mp_obj_get_int(args[2]) : data.len;
  mp_int_t offset = n_args >= 4 ? mp_obj_get_int(args[3]) : 0;
  if (length < 0) length += data.len;
  if (offset < 0) offset += data.len;
  if (length < 0 || offset < 0 || offset + length > data.len) {
    mp_raise_ValueError("Illegal offset/length");
  }
  xmr_hash_to_scalar(MP_OBJ_SCALAR(res), (const char *)data.buf + offset,
                     length);
  return res;
}
STATIC MP_DEFINE_CONST_FUN_OBJ_VAR_BETWEEN(
    mod_trezorcrypto_monero_hash_to_scalar_into_obj, 2, 4,
    mod_trezorcrypto_monero_hash_to_scalar_into);

/// def xmr_derivation_to_scalar(
///     r: Scalar | None, p: Point, output_index: int
/// ) -> Scalar:
///     """
///     H_s(derivation || varint(output_index))
///     """
STATIC mp_obj_t mod_trezorcrypto_monero_xmr_derivation_to_scalar(
    const mp_obj_t dest, const mp_obj_t p, const mp_obj_t output_index) {
  mp_obj_t res = mp_obj_new_scalar_r(dest);
  assert_ge25519(p);
  xmr_derivation_to_scalar(MP_OBJ_SCALAR(res), &MP_OBJ_C_GE25519(p),
                           mp_obj_get_int(output_index));
  return res;
}
STATIC MP_DEFINE_CONST_FUN_OBJ_3(
    mod_trezorcrypto_monero_xmr_derivation_to_scalar_obj,
    mod_trezorcrypto_monero_xmr_derivation_to_scalar);

/// def xmr_generate_key_derivation(
///     r: Point | None, A: Point, b: Scalar
/// ) -> Point:
///     """
///     8*(key2*key1)
///     """
STATIC mp_obj_t mod_trezorcrypto_monero_xmr_generate_key_derivation(
    const mp_obj_t dest, const mp_obj_t A, const mp_obj_t b) {
  mp_obj_t res = mp_obj_new_ge25519_r(dest);
  assert_ge25519(A);
  assert_scalar(b);
  xmr_generate_key_derivation(&MP_OBJ_GE25519(res), &MP_OBJ_C_GE25519(A),
                              MP_OBJ_C_SCALAR(b));
  return res;
}
STATIC MP_DEFINE_CONST_FUN_OBJ_3(
    mod_trezorcrypto_monero_xmr_generate_key_derivation_obj,
    mod_trezorcrypto_monero_xmr_generate_key_derivation);

/// def xmr_derive_private_key(
///     r: Scalar | None, deriv: Point, idx: int, base: Scalar
/// ) -> Scalar:
///     """
///     base + H_s(derivation || varint(output_index))
///     """
STATIC mp_obj_t mod_trezorcrypto_monero_xmr_derive_private_key(
    size_t n_args, const mp_obj_t *args) {
  mp_arg_check_num(n_args, 0, 4, 4, false);
  mp_obj_t res = mp_obj_new_scalar_r(args[0]);
  assert_ge25519(args[1]);
  assert_scalar(args[3]);
  xmr_derive_private_key(MP_OBJ_SCALAR(res), &MP_OBJ_C_GE25519(args[1]),
                         mp_obj_get_int(args[2]), MP_OBJ_C_SCALAR(args[3]));
  return res;
}
STATIC MP_DEFINE_CONST_FUN_OBJ_VAR_BETWEEN(
    mod_trezorcrypto_monero_xmr_derive_private_key_obj, 4, 4,
    mod_trezorcrypto_monero_xmr_derive_private_key);

/// def xmr_derive_public_key(
///     r: Point | None, deriv: Point, idx: int, base: Point
/// ) -> Point:
///     """
///     H_s(derivation || varint(output_index))G + base
///     """
STATIC mp_obj_t mod_trezorcrypto_monero_xmr_derive_public_key(
    size_t n_args, const mp_obj_t *args) {
  mp_arg_check_num(n_args, 0, 4, 4, false);
  mp_obj_t res = mp_obj_new_ge25519_r(args[0]);
  assert_ge25519(args[1]);
  assert_ge25519(args[3]);
  xmr_derive_public_key(&MP_OBJ_GE25519(res), &MP_OBJ_C_GE25519(args[1]),
                        mp_obj_get_int(args[2]), &MP_OBJ_C_GE25519(args[3]));
  return res;
}
STATIC MP_DEFINE_CONST_FUN_OBJ_VAR_BETWEEN(
    mod_trezorcrypto_monero_xmr_derive_public_key_obj, 4, 4,
    mod_trezorcrypto_monero_xmr_derive_public_key);

/// def add_keys2_into(
///     r: Point | None, a: Scalar, b: Scalar, B: Point
/// ) -> Point:
///     """
///     aG + bB, G is basepoint
///     """
STATIC mp_obj_t mod_trezorcrypto_monero_add_keys2_into(size_t n_args,
                                                       const mp_obj_t *args) {
  mp_arg_check_num(n_args, 0, 4, 4, false);
  mp_obj_t res = mp_obj_new_ge25519_r(args[0]);
  assert_scalar(args[1]);
  assert_scalar(args[2]);
  assert_ge25519(args[3]);
  xmr_add_keys2_vartime(&MP_OBJ_GE25519(res), MP_OBJ_SCALAR(args[1]),
                        MP_OBJ_SCALAR(args[2]), &MP_OBJ_C_GE25519(args[3]));
  return res;
}
STATIC MP_DEFINE_CONST_FUN_OBJ_VAR_BETWEEN(
    mod_trezorcrypto_monero_add_keys2_into_obj, 4, 4,
    mod_trezorcrypto_monero_add_keys2_into);

/// def add_keys3_into(
///     r: Point | None, a: Scalar, A: Point, b: Scalar, B: Point
/// ) -> Point:
///     """
///     aA + bB
///     """
STATIC mp_obj_t mod_trezorcrypto_monero_add_keys3_into(size_t n_args,
                                                       const mp_obj_t *args) {
  mp_arg_check_num(n_args, 0, 5, 5, false);
  mp_obj_t res = mp_obj_new_ge25519_r(args[0]);
  assert_scalar(args[1]);
  assert_ge25519(args[2]);
  assert_scalar(args[3]);
  assert_ge25519(args[4]);
  xmr_add_keys3_vartime(&MP_OBJ_GE25519(res), MP_OBJ_SCALAR(args[1]),
                        &MP_OBJ_C_GE25519(args[2]), MP_OBJ_SCALAR(args[3]),
                        &MP_OBJ_C_GE25519(args[4]));
  return res;
}
STATIC MP_DEFINE_CONST_FUN_OBJ_VAR_BETWEEN(
    mod_trezorcrypto_monero_add_keys3_into_obj, 5, 5,
    mod_trezorcrypto_monero_add_keys3_into);

/// def xmr_get_subaddress_secret_key(
///     r: Scalar | None, major: int, minor: int, m: Scalar
/// ) -> Scalar:
///     """
///     Hs(SubAddr || a || index_major || index_minor)
///     """
STATIC mp_obj_t mod_trezorcrypto_monero_xmr_get_subaddress_secret_key(
    size_t n_args, const mp_obj_t *args) {
  mp_arg_check_num(n_args, 0, 4, 4, false);
  mp_obj_t res = mp_obj_new_scalar_r(args[0]);
  assert_scalar(args[3]);
  xmr_get_subaddress_secret_key(MP_OBJ_SCALAR(res), mp_obj_get_int(args[1]),
                                mp_obj_get_int(args[2]),
                                MP_OBJ_C_SCALAR(args[3]));
  return res;
}
STATIC MP_DEFINE_CONST_FUN_OBJ_VAR_BETWEEN(
    mod_trezorcrypto_monero_xmr_get_subaddress_secret_key_obj, 4, 4,
    mod_trezorcrypto_monero_xmr_get_subaddress_secret_key);

/// def gen_commitment_into(r: Point | None, a: Scalar, amount: int) -> Point:
///     """
///     aG + amount * H
///     """
STATIC mp_obj_t mod_trezorcrypto_monero_gen_commitment_into(
    const mp_obj_t dest, const mp_obj_t a, const mp_obj_t amount) {
  mp_obj_t res = mp_obj_new_ge25519_r(dest);
  assert_scalar(a);
  xmr_gen_c(&MP_OBJ_GE25519(res), MP_OBJ_C_SCALAR(a),
            trezor_obj_get_uint64(amount));
  return res;
}
STATIC MP_DEFINE_CONST_FUN_OBJ_3(
    mod_trezorcrypto_monero_gen_commitment_into_obj,
    mod_trezorcrypto_monero_gen_commitment_into);

/// def ct_equals(a: bytes, b: bytes) -> bool:
///     """
///     Constant time buffer comparison
///     """
STATIC mp_obj_t mod_trezorcrypto_ct_equals(const mp_obj_t a, const mp_obj_t b) {
  mp_buffer_info_t buff_a = {0}, buff_b = {0};
  mp_get_buffer_raise(a, &buff_a, MP_BUFFER_READ);
  mp_get_buffer_raise(b, &buff_b, MP_BUFFER_READ);

  if (buff_a.len != buff_b.len) {
    return MP_OBJ_NEW_SMALL_INT(0);
  }

  int r = ed25519_verify(buff_a.buf, buff_b.buf, buff_a.len);
  return MP_OBJ_NEW_SMALL_INT(r);
}
STATIC MP_DEFINE_CONST_FUN_OBJ_2(mod_trezorcrypto_ct_equals_obj,
                                 mod_trezorcrypto_ct_equals);

// Hasher
STATIC mp_obj_t mod_trezorcrypto_monero_hasher_update(mp_obj_t self,
                                                      const mp_obj_t arg) {
  mp_obj_hasher_t *o = MP_OBJ_TO_PTR(self);
  mp_buffer_info_t buff = {0};
  mp_get_buffer_raise(arg, &buff, MP_BUFFER_READ);
  if (buff.len > 0) {
    xmr_hasher_update(&o->h, buff.buf, buff.len);
  }
  return mp_const_none;
}
STATIC MP_DEFINE_CONST_FUN_OBJ_2(mod_trezorcrypto_monero_hasher_update_obj,
                                 mod_trezorcrypto_monero_hasher_update);

STATIC mp_obj_t mod_trezorcrypto_monero_hasher_digest(size_t n_args,
                                                      const mp_obj_t *args) {
  mp_obj_hasher_t *o = MP_OBJ_TO_PTR(args[0]);

  Hasher ctx = {0};
  memcpy(&ctx, &(o->h), sizeof(Hasher));

  if (n_args == 1 || args[1] == mp_const_none) {
    vstr_t hash = {0};
    vstr_init_len(&hash, SHA3_256_DIGEST_LENGTH);
    xmr_hasher_final(&ctx, (uint8_t *)hash.buf);
    memzero(&ctx, sizeof(SHA3_CTX));
    return mp_obj_new_str_from_vstr(&mp_type_bytes, &hash);
  } else {
    mp_buffer_info_t bufm = {0};
    mp_get_buffer_raise(args[1], &bufm, MP_BUFFER_WRITE);
    const mp_int_t offset = n_args >= 3 ? mp_obj_get_int(args[2]) : 0;
    if (bufm.len < SHA3_256_DIGEST_LENGTH + offset) {
      mp_raise_ValueError("Buffer too small");
    }

    xmr_hasher_final(&ctx, (uint8_t *)bufm.buf + offset);
    memzero(&ctx, sizeof(SHA3_CTX));
    return args[1];
  }
}
STATIC MP_DEFINE_CONST_FUN_OBJ_VAR_BETWEEN(
    mod_trezorcrypto_monero_hasher_digest_obj, 1, 3,
    mod_trezorcrypto_monero_hasher_digest);

STATIC mp_obj_t mod_trezorcrypto_monero_hasher_copy(mp_obj_t self) {
  mp_obj_hasher_t *o = MP_OBJ_TO_PTR(self);
  mp_obj_hasher_t *cp = m_new_obj_with_finaliser(mp_obj_hasher_t);
  cp->base.type = o->base.type;
  memcpy(&(cp->h), &(o->h), sizeof(Hasher));
  return MP_OBJ_FROM_PTR(o);
}
STATIC MP_DEFINE_CONST_FUN_OBJ_1(mod_trezorcrypto_monero_hasher_copy_obj,
                                 mod_trezorcrypto_monero_hasher_copy);

//
// Type defs
//

STATIC const mp_rom_map_elem_t
    mod_trezorcrypto_monero_ge25519_locals_dict_table[] = {
        {MP_ROM_QSTR(MP_QSTR___del__),
         MP_ROM_PTR(&mod_trezorcrypto_monero_ge25519___del___obj)},
};
STATIC MP_DEFINE_CONST_DICT(mod_trezorcrypto_monero_ge25519_locals_dict,
                            mod_trezorcrypto_monero_ge25519_locals_dict_table);

STATIC const mp_obj_type_t mod_trezorcrypto_monero_ge25519_type = {
    {&mp_type_type},
    .name = MP_QSTR_Point,
    .make_new = mod_trezorcrypto_monero_ge25519_make_new,
    .locals_dict = (void *)&mod_trezorcrypto_monero_ge25519_locals_dict,
};

STATIC const mp_rom_map_elem_t
    mod_trezorcrypto_monero_bignum256modm_locals_dict_table[] = {
        {MP_ROM_QSTR(MP_QSTR___del__),
         MP_ROM_PTR(&mod_trezorcrypto_monero_bignum256modm___del___obj)},
};
STATIC MP_DEFINE_CONST_DICT(
    mod_trezorcrypto_monero_bignum256modm_locals_dict,
    mod_trezorcrypto_monero_bignum256modm_locals_dict_table);

STATIC const mp_obj_type_t mod_trezorcrypto_monero_bignum256modm_type = {
    {&mp_type_type},
    .name = MP_QSTR_Scalar,
    .make_new = mod_trezorcrypto_monero_bignum256modm_make_new,
    .locals_dict = (void *)&mod_trezorcrypto_monero_bignum256modm_locals_dict,
};

STATIC const mp_rom_map_elem_t
    mod_trezorcrypto_monero_hasher_locals_dict_table[] = {
        {MP_ROM_QSTR(MP_QSTR_update),
         MP_ROM_PTR(&mod_trezorcrypto_monero_hasher_update_obj)},
        {MP_ROM_QSTR(MP_QSTR_digest),
         MP_ROM_PTR(&mod_trezorcrypto_monero_hasher_digest_obj)},
        {MP_ROM_QSTR(MP_QSTR_copy),
         MP_ROM_PTR(&mod_trezorcrypto_monero_hasher_copy_obj)},
        {MP_ROM_QSTR(MP_QSTR___del__),
         MP_ROM_PTR(&mod_trezorcrypto_monero_hasher___del___obj)},
        {MP_ROM_QSTR(MP_QSTR_block_size), MP_ROM_INT(SHA3_256_BLOCK_LENGTH)},
        {MP_ROM_QSTR(MP_QSTR_digest_size), MP_ROM_INT(SHA3_256_DIGEST_LENGTH)},
};
STATIC MP_DEFINE_CONST_DICT(mod_trezorcrypto_monero_hasher_locals_dict,
                            mod_trezorcrypto_monero_hasher_locals_dict_table);

STATIC const mp_obj_type_t mod_trezorcrypto_monero_hasher_type = {
    {&mp_type_type},
    .name = MP_QSTR_hasher,
    .make_new = mod_trezorcrypto_monero_hasher_make_new,
    .locals_dict = (void *)&mod_trezorcrypto_monero_hasher_locals_dict,
};

/// BP_GI_PLUS_PRE: bytes
STATIC const mp_obj_str_t mod_trezorcrypto_monero_BP_PLUS_GI_PRE_obj = {{&mp_type_bytes}, 0, 8192, (const byte*)""
"\x38\xc5\xd4\xdb\x53\xae\xb8\x6f\x5a\x80\xde\xf9\xbe\x49\x53\xf2"
"\x28\x8e\xd5\xa4\x4c\x66\xaf\x72\x3f\x46\x3d\x01\x70\x82\x90\x10"
"\x8a\x6c\x81\x7d\xab\xe9\x0f\xdb\x50\xcc\x38\x67\x7b\x23\xff\xa7"
"\xd6\x4e\xfe\xb0\x0b\xbd\x53\xfe\xbe\x62\xe0\x77\xde\x0d\xb5\x93"
"\xf8\x00\x40\x4a\xe0\x9e\xa0\xb9\x00\xbf\x69\x7f\x82\xe3\xbd\x80"
"\xc7\x19\x29\x50\xe9\xa8\x8d\xab\x84\xff\x00\x0e\x39\x73\x63\x90"
"\x38\xf8\x17\x9c\xbf\x4a\xd0\x96\xef\x42\x2a\xa6\x89\x89\x91\x5d"
"\xcb\x36\x1a\x8a\x6a\x51\x00\xf5\x6c\xd0\xed\xf0\xc6\x43\x1d\x98"
"\xba\x16\x97\x26\xa6\x49\x29\xaf\xca\x1f\xfe\xbd\xf7\x0e\xa1\xcc"
"\x17\xae\x52\x6d\x95\xf0\x7d\xe7\x9d\x2b\x51\xea\xdb\xa0\x04\xeb"
"\x7a\x07\xaf\x94\xf0\x3c\x01\xa3\xd9\x94\x0b\xf4\x4a\x24\xbf\x47"
"\x59\xf1\x87\xea\x69\x79\x48\xad\x6f\xb5\x91\xa5\x5c\x9c\x12\xea"
"\x64\xdd\x15\x51\x8e\x34\x50\xdc\x60\x8e\x38\xa9\x3e\xc0\xa4\x57"
"\x3f\x15\x6b\x22\xd0\x02\x67\x02\xc7\x1c\x12\x27\x5a\x6c\x66\x3e"
"\xbf\xb6\xc4\x13\x91\xcb\x92\x85\xf0\x2c\x2f\xe8\x70\x6a\xcc\x6a"
"\xb6\xc1\x68\x92\xbb\xf5\x9e\x4f\x1a\x5a\xa8\x67\x3d\x08\xec\xb9"
"\xfc\xd6\x63\xe5\x1e\x2b\x96\xbd\x23\x91\x74\x31\x8e\x33\x4b\x1b"
"\xbc\x40\x72\x65\xe6\x3d\x55\x1a\x35\x8c\x4c\x6c\xae\x1f\x37\x44"
"\xb3\xb5\x65\xbf\xd1\xfc\x81\x1d\x24\xfa\x54\xca\x1a\xdb\x50\x8f"
"\xfc\x34\xa6\x26\x89\x78\x01\xfc\x8d\x44\x51\x84\xa4\x9b\x9e\xca"
"\xc4\x55\x62\x86\x36\xb1\x67\xad\x6a\x9f\x27\x18\x33\xc5\x30\xc3"
"\x01\xa3\x56\x36\x0e\xef\xba\x79\xfd\xe5\x41\xec\x6d\x23\x58\x49"
"\xa2\x31\x5c\xe7\x0d\x83\xea\x45\x62\x13\x91\x55\xfe\xc5\x5a\xe3"
"\x8b\xa6\xdc\xf1\x46\x93\x86\x2d\x6f\x9b\x29\x8e\x47\xb7\xd6\x93"
"\xeb\x68\x1b\x26\x46\xb5\xff\xc1\x7b\xf5\x60\x21\x2b\xf1\xfb\x91"
"\x42\xb5\xe1\x6a\xdd\xd1\xbb\x1b\x9e\x19\x49\x7d\x45\xc8\x16\x8f"
"\x01\xd0\xb1\xc8\x2b\xc4\x48\x5b\xc4\x41\x06\x0e\x77\xac\xb2\x73"
"\x6e\x7f\x97\xbe\x03\xfa\xee\xe4\x37\x93\x87\x4e\xe5\x7a\x09\xb8"
"\xd9\x64\xf1\x9a\x06\xaa\xfc\x12\xe6\xdb\xa4\x94\x46\x94\x0b\x7c"
"\x28\x47\x96\x01\x2c\xc4\xc7\xbd\x7b\x9d\xed\x48\xb3\xdb\xa3\x1d"
"\xb0\x74\x60\x3e\xc2\x96\xd7\x36\xb9\x65\x38\x53\x15\xb6\xb9\x6a"
"\xba\x3c\x3e\x96\xa2\xcb\x44\x95\xa1\xea\x0d\x99\x8e\xe9\xc0\xd2"
"\x3c\x40\x52\x68\x2e\x88\x85\x96\xe2\xad\x67\xb5\x7e\x7c\x18\x24"
"\x0f\x97\xa8\xf0\xa7\xb0\x60\x11\xe8\xc2\xd4\x62\x90\xe7\x06\xd4"
"\xd7\xc2\x6e\xa0\xc7\x13\x42\x4c\xbc\x4e\x97\xef\xc2\xd5\x0c\xe2"
"\x0c\xd4\x43\x20\x07\x72\x45\xc5\x66\x9f\xee\x46\xc0\x64\x26\x95"
"\xf8\x33\x39\x20\x0f\xd6\xbd\x73\x6e\x4d\x7d\x21\x5e\x8d\xc0\x46"
"\x8a\xbd\x0a\xc2\xbe\x0a\xf9\x05\x30\x93\x79\xab\xb8\x39\x28\x4a"
"\xa8\x44\x53\x19\xb8\xf4\xaf\x30\xb9\x2c\x9d\xdb\x0f\x66\x6b\x74"
"\x63\xd0\x38\x90\x56\x88\x64\x36\x9d\x5d\xf6\x40\x6d\x8f\x02\xeb"
"\x6b\x6f\xc6\xf4\x70\x1b\xd4\x92\xd2\x4f\x6b\x50\x0a\x20\x0e\xc6"
"\xc6\x1f\x14\xec\x6f\x16\x99\x8c\x5c\xd3\xcc\x7e\x05\x53\x23\xd7"
"\xd2\x34\xfd\x45\xf1\xf7\xb9\x04\x4a\xde\xc1\x52\x24\xb2\x26\xb0"
"\x85\xe6\x31\xa8\x7f\xc8\xfc\xbd\x8d\x87\x06\x06\x35\x05\x3e\x59"
"\xf2\x83\xd6\x23\x0c\xc0\x59\x80\xd6\x3c\xf2\x60\x99\x96\xa8\x2c"
"\x17\x35\x79\xad\x70\xaf\xf4\x53\xd2\x9e\x5e\xbb\x22\x82\xee\x13"
"\xec\x0d\x36\xf8\xda\xe5\xd3\xe4\xf7\xa1\xd0\x24\xa3\xa2\x26\x3c"
"\xf0\x37\x84\x0c\x4a\x5e\x63\x5f\xb4\xf4\x5e\xb4\xc9\xa5\x14\xa7"
"\xcb\x3e\x5a\x0b\xed\x90\xb4\xc7\x89\x7d\xd2\xe5\x5e\x30\x9d\xfb"
"\x62\x59\x0c\xaf\x9e\x44\x48\x4c\x6c\xab\x03\x57\x75\xf9\xb3\x6e"
"\x85\xe3\xae\x64\xbe\x45\xc8\xe7\x19\x0a\x30\xb0\xb2\xf0\xd3\x51"
"\x99\x72\xcd\x4d\xf7\x98\x36\x2f\x88\x2c\xa5\x6d\xc7\xc8\x65\x8b"
"\x27\x6a\x88\xc8\x22\x22\xef\xd4\xae\x25\xb6\x52\x8b\x28\x75\x38"
"\x25\xb6\x07\xe4\xb7\x91\xc6\x8c\x09\x62\x2d\xa6\x40\xcc\x76\x8b"
"\x25\xb1\xec\x81\xab\x7a\xa1\x7f\x25\x1c\xe1\xf4\xb1\x02\xf8\x1d"
"\x3c\x4d\x06\x66\x8f\x8e\x92\x9a\x24\x0e\x6b\x41\x53\x0e\x6b\xa1"
"\x72\x47\x2c\x67\x7b\xfa\xc2\x00\x83\xf5\x46\x46\x8d\xb5\xbf\x5f"
"\x38\xcc\x58\xa3\x55\x65\x91\xfd\x9e\x10\x96\xdd\x20\xf5\xa9\x0c"
"\xdc\x38\x58\xa6\x4c\x84\xcc\xa1\x59\xd8\xdb\x5d\xe6\xbe\xc0\x18"
"\x16\x4e\xa3\x8b\xd6\xe8\xbb\x38\x04\xdc\xdc\xe6\x35\x23\x74\xce"
"\xb0\x54\xbf\xd3\x97\x94\xc0\x72\xdb\x13\xad\xdf\xd2\x64\x47\x13"
"\xbd\xf3\xb9\xd5\x39\x3b\x3d\x3c\xf3\x8c\x91\xf7\xe5\xa8\x1e\x7a"
"\x97\x40\x06\xa0\x83\x40\x66\x2e\x29\x27\x1e\xfd\xe6\x6f\x7f\xa5"
"\x3d\x25\xd5\x4e\xb2\x00\x07\x54\xd2\x3d\xcf\x16\xb5\xc9\xad\x78"
"\x85\x0b\xc4\x0b\x20\x8d\x0b\xd9\xf0\x40\xc6\x6a\x2e\x6b\x74\xaa"
"\x28\x78\xe8\xa8\x41\xda\xbe\xfd\x12\xee\xc3\x10\x1a\x12\x0f\xa7"
"\x7f\x4e\x2a\x77\xa9\xb9\xf0\x98\xc9\x97\x04\xdc\x6b\x55\xfb\x15"
"\x30\xa7\x91\xf0\x18\x37\xe5\x76\xa2\x6e\x60\xf7\x6b\x51\x7c\x7d"
"\x34\x66\x73\xca\x59\xb7\x69\x69\xec\xd9\x27\xaa\x3e\x7a\xa1\xfc"
"\xc6\xb4\x79\xd7\x9e\xda\x52\x40\x3d\x9e\xa3\x5a\x94\xf0\xcb\x1f"
"\x81\x34\xbb\xb5\x09\xaf\xb4\xe7\x09\x9b\xca\x32\x15\xf9\xb4\xc1"
"\xbc\x2e\x40\xdd\x62\x5b\x73\xd9\x69\xbc\x79\x93\x13\x9d\x51\x45"
"\x80\xaf\x04\x49\x5d\x4b\xef\xae\x21\x2a\x35\x96\x2e\xc0\x2c\xf8"
"\x87\xc1\x11\xb4\xec\x82\x67\x27\xd0\x6a\x5a\x6f\x6d\x8f\x71\x9d"
"\x49\x49\x9a\xd6\x77\x1a\x08\xa8\x36\x2e\xc2\xff\x0b\xa2\xe5\xe9"
"\xd6\x8c\x61\x03\x2a\xf9\x4d\x8f\xf3\x9e\xf3\x9b\x03\x88\x50\xf5"
"\x86\x53\xcd\xbd\xc6\xeb\xf4\x1b\x39\x68\x63\x73\x37\x81\xfe\xf8"
"\x06\xe4\x69\x71\xfd\xca\xf4\x7b\x36\xae\xe1\x2d\xa9\x6c\xb2\x45"
"\xc7\x2f\x45\xbb\x59\x8a\x7d\xc2\x63\x79\xb3\xab\xb7\x21\xfb\xe4"
"\x29\x38\x7c\x36\xcf\x72\x87\x74\x42\xef\x83\x48\x95\xe4\xb4\xc4"
"\x9a\x53\xa6\xb3\xe0\x41\x9a\xf8\xe1\xb3\xc4\xef\x7d\xdb\xaa\x7a"
"\x3f\x05\xeb\x6d\xa7\xd7\x53\xd6\xd5\x8d\xd5\x8e\x90\x20\xba\x52"
"\x59\x13\xac\x99\x93\xaa\x90\x2e\xc4\xc6\x0b\xb2\x92\x4c\x2b\x36"
"\x43\xc3\xaf\xc2\xd2\x10\xbf\x49\x36\x3d\xe0\x78\xa0\x63\x4d\xe5"
"\x98\x62\x7a\xc3\x50\x49\xeb\x1f\xbf\x4f\xf4\x67\x35\x90\xa1\x40"
"\x65\xf3\x2b\xc3\xc0\x8f\x25\xb3\x8b\xbc\x59\xa5\x84\xc9\x1e\x13"
"\xcf\x6d\x6e\x33\x97\x09\x3b\xab\xea\x9f\x57\xeb\xf1\x97\x92\x2c"
"\x46\xe9\xa8\x60\xe2\xcb\xe8\x42\xff\xc0\x4c\xe6\x5f\xe0\xd5\xc5"
"\x38\xd5\xb0\x37\x1c\x24\xaf\xf6\xd3\x4a\xf0\x53\xfb\x4e\x41\x95"
"\x18\x4d\x44\xce\x15\x1d\x81\xdb\xc0\x02\xba\x98\x75\xe6\x94\xb2"
"\x19\x05\x13\x67\xe3\xe8\x31\x0c\x4d\xc4\xd7\xab\xa5\x40\x39\xb6"
"\x0e\xc1\x39\x71\x86\xcd\x16\xef\x78\x14\x9e\x64\xdb\x20\xef\x7b"
"\xd1\xe8\xb2\xe0\x2f\xd0\x2d\x5d\xf1\x96\xda\xa0\x4e\xe8\xd0\x82"
"\x3f\x7e\x60\x20\x6b\x68\x26\xc0\x06\xc1\x2f\xc5\xb9\xfc\x87\xac"
"\x22\xde\x9d\xa9\x1e\xd7\xc4\xcd\xc3\x45\x72\xc0\x59\x78\x26\x3b"
"\x0b\x5d\x83\xf8\x82\x85\xc3\xc7\xa5\xa2\x77\x3a\x19\xb9\x7c\x60"
"\x1a\xf5\x45\x72\xe2\x6a\x7b\xda\x09\x8f\x26\xbf\xc7\xde\x63\x91"
"\xf6\xbe\xe2\xe9\x43\xf2\x65\xe7\xa0\xc0\x89\x14\x24\x27\xea\xd2"
"\xb5\x88\x0c\x8e\xc2\xe8\xec\x5c\x76\x50\xb4\x10\x7d\x0d\xc2\x15"
"\x7c\x40\x79\xf7\xd9\x20\x08\x2e\x84\x4c\x72\xff\xc8\x45\x99\xa8"
"\x11\x56\x12\x15\x59\x53\xf4\xd8\x5f\x51\xd6\x3a\x87\xad\xea\x19"
"\x73\xe6\xa3\x87\x7e\x54\xd7\xad\xba\x0b\x14\xd6\x7f\x25\x28\x0d"
"\x48\x09\xd2\x9c\x2e\x06\x8e\x5f\x0c\x48\xea\x29\xfc\x85\xe1\x01"
"\xb8\x13\x95\x4f\x99\x32\x16\xb9\x3d\x3d\xc0\x5e\x3c\x99\xf3\x9d"
"\xa1\x52\x6c\xb5\xf5\x5f\x76\x38\xd7\xa3\x4c\x92\x85\xaa\x68\xb5"
"\x67\x08\x83\x0d\x7c\xb2\x42\x1f\xed\x63\x00\xc6\x3c\xe0\x26\x51"
"\x51\x94\x95\x83\x3f\x68\xff\xbb\x95\x71\x9d\x67\x8f\x58\xd7\x1a"
"\xdf\xaa\x69\xc6\xf3\xc2\x22\x75\x35\x83\x87\x26\x0e\x6e\x86\x11"
"\xc5\x32\x90\x40\x11\x6f\x56\xe2\xe3\x5d\xe6\xaa\x5a\x1b\x67\x62"
"\x8a\x87\x6e\xa0\xd5\xef\x2a\xcc\x0f\xe4\xe8\xb2\x90\xd1\xca\x7c"
"\x60\x32\x04\x8f\x3e\x2c\xde\x35\x2b\x75\x9c\xc5\xff\x68\x7a\xda"
"\xf8\x13\x0e\xec\x41\x7c\xc5\x2e\xf3\x83\x95\xfb\x04\x6e\xd4\xa7"
"\xc7\xe6\x4c\x8b\xd7\x7a\x90\x2b\x74\x87\x85\x7b\x74\xb7\x25\xea"
"\x4f\xe5\x31\xdb\x0f\xf2\xca\xed\x60\x27\x64\xb9\xdf\x5b\x1a\xae"
"\xbf\x87\x89\x2a\x34\xd4\x26\x87\x69\x33\x6f\xef\x73\x5a\xbf\x1e"
"\x43\x92\x08\x3a\xa7\x0b\xe0\x62\xce\x40\x3c\x72\xfb\x40\x52\x78"
"\x20\xfa\x7a\x39\x59\x38\xe0\xbd\x78\x54\xf3\xe5\x87\x0a\x85\xec"
"\x80\x09\x85\x56\x48\x70\xb0\x38\xc9\x14\x72\x84\x5e\x5b\x71\xc9"
"\x02\x6f\xe1\xec\xcb\x14\x06\x47\x14\x20\xf1\xa8\x8b\x41\x87\x7d"
"\x47\xb2\x1b\x06\xe8\x27\x24\x7f\xfb\x97\x2d\xde\x5d\x62\x16\x2f"
"\x60\xb2\x05\x70\xe6\x4a\x70\x5d\x63\x78\xf9\x4d\x5e\xda\x5a\x68"
"\xdf\x11\xc2\xe8\xbd\x3c\xa4\xc9\x14\xd9\x84\x28\xaf\xd1\x5e\xec"
"\x28\xc7\x76\x82\x61\xe6\xdd\x6a\xa0\x8e\x31\xca\x18\x06\x00\x79"
"\x7d\x34\x60\x4f\x41\x19\x8b\xc4\x79\x86\xd1\x00\x3d\xc0\x3e\xcc"
"\x3a\xce\xf6\xb1\x08\xaa\x71\xe9\x45\x3d\x09\xfa\x3d\x37\xb5\x3f"
"\x95\x2b\xc9\x9d\xdd\x2a\xe3\x95\xe6\xf6\xd1\x6b\x1e\xc2\xd0\x6f"
"\x46\xe9\xe2\xd3\x58\x6d\xfd\x89\x37\x45\xc0\x95\x7b\xfa\xb3\xcc"
"\x00\x5a\x1a\x6c\x51\xce\x25\xf0\x65\x81\x56\x03\xea\xb1\x16\x0c"
"\xa0\x4b\x4d\x22\x98\xff\x60\x90\x5e\x2c\xb0\xcb\x80\xcb\x57\xf6"
"\xae\x36\x57\xe9\x6d\x4f\xe1\xe0\x5d\x4c\xeb\xbc\xe9\xf9\xb5\x6c"
"\xf0\x39\xec\xe8\xca\x37\xa8\x90\xcb\xde\xbb\x2f\x4e\x7f\xae\xbd"
"\xf8\x7a\xfa\x2b\xd2\x10\x72\xb4\x1b\xe5\x39\x40\x6b\x05\x55\xda"
"\xcf\x00\x18\xed\x2d\x53\xea\x85\x0a\x47\xf2\x09\xd6\x5d\x32\x03"
"\xa0\x4c\xb7\xf3\x2b\xb9\x58\xb4\x7f\x55\x25\x6c\x1d\xc7\xd0\x6f"
"\xd8\x8f\x81\x92\x1e\x5d\x95\x18\x4f\x4b\x66\x88\xbb\x4e\x24\xf9"
"\xaf\xcb\x78\x12\x02\x8e\x54\x2b\x69\xf7\xcf\x62\x81\x61\xfd\x17"
"\x14\x34\x89\x64\x11\x09\x2b\x79\x79\xad\x74\x23\x75\x66\xfd\xeb"
"\xda\x44\x49\x0c\xca\x43\x1b\x98\x23\xd6\x5f\xf6\xf7\xbe\x72\xc3"
"\x18\xd2\x0b\x70\x03\x1c\x67\x6f\xc7\x06\xe3\x47\x5b\x34\x11\xac"
"\x13\xe1\x43\xcc\x1a\x68\x16\xc0\x2c\x21\x1a\xd6\x9f\xb5\x4a\x75"
"\x3c\x2d\xb5\xfa\x1b\xed\xc5\xf5\xd0\x23\xd8\x1f\x59\xb1\xa1\xb0"
"\x46\xa0\x80\x2a\xee\x7a\x37\xa9\x98\x5f\x3c\x9f\xe8\x6a\xc8\x5e"
"\x15\x15\xed\x32\x3f\xeb\x44\xa2\xdf\x64\x53\x8d\x05\x9a\x66\xee"
"\x58\x67\xd3\x7b\x14\x92\x68\xbb\xad\xde\x04\xa5\xfc\x53\x63\xec"
"\xe7\xd2\x9b\xeb\x31\xf3\x6b\xa8\xec\x27\x44\x52\xdb\xc5\x6b\x58"
"\x38\x45\x92\x79\x03\x50\x3d\x70\x59\x44\x5c\x72\x20\xc6\xd6\x6c"
"\x28\xe4\xbc\x49\x94\x40\x31\x27\xaf\xee\x08\xcb\xec\x38\x2c\x79"
"\x58\x94\x70\x31\x84\xf2\x2c\x6a\x45\xad\x48\x4a\x89\x80\x1e\xad"
"\xd2\xba\xd6\xa7\x5a\x80\x9f\x31\x2c\x05\x6d\x23\x44\xa3\x6a\xc1"
"\x5a\x3d\xcc\x63\xa6\x3e\xba\x5a\x46\x46\xda\x93\x3e\x5a\x41\x63"
"\x4a\x15\x5f\xb7\x89\x59\x84\x39\xb7\x61\xac\x09\x4b\x93\xac\x8b"
"\x33\x3e\x16\x2f\x39\x41\x9c\xfd\x9e\xe3\x87\x91\x77\x45\xbf\x11"
"\x78\x1c\x12\xde\x1f\x1b\x4e\x67\xcc\x63\xd0\xb9\x9a\xf8\xd4\xf0"
"\x12\x1b\x08\x31\xf9\x4f\x15\xac\x37\x9c\x6c\xb4\xaa\x34\x0c\xd0"
"\xc9\x4e\x78\xc6\xd1\x05\x50\x9d\x75\x51\xcf\xdd\x23\xaf\x18\xae"
"\x7b\xc5\x79\x62\xab\x18\x4c\x26\x6d\x3f\xce\x17\x88\x46\xaf\x54"
"\xcc\x52\x10\xae\xff\x86\x46\x68\xd8\xfc\x47\x64\x24\x50\x76\xb2"
"\x24\xff\xc5\xc7\x5d\xf7\xe5\x1a\x21\x54\x7f\x7e\x30\x23\x9c\x1c"
"\x23\x7b\xc2\x45\xec\xb7\xa8\xe2\xea\xc9\xcd\x36\xc2\xc7\x5f\xa8"
"\xcf\x6f\xce\x87\x77\xe3\x12\xec\xa8\x8c\x30\xf4\xfe\xe6\x0a\x55"
"\x92\xc9\xbd\xc9\xae\x8c\x0d\xeb\xea\xa4\x1b\x4b\xcc\x32\x47\x94"
"\x65\x9d\xd5\x8d\x18\xea\x00\x15\x97\x5d\xa6\x39\xff\x9c\x89\x52"
"\x66\x80\x8c\xf2\x85\x9b\xaf\x7c\x17\xd1\xff\x26\x88\xfb\x68\x82"
"\xe4\x3b\x45\xb9\x21\xd0\x4a\x14\x73\x34\x2e\x49\x15\x5c\xa7\x8d"
"\x1a\xa2\x00\xa8\x16\x78\x34\x76\x97\xf5\xb4\x49\x65\x79\x57\x37"
"\xdb\x9a\x98\x89\xf4\xee\x73\x26\xc8\x25\x96\x90\xf2\x51\x15\xdd"
"\x98\x55\xd6\xc4\x92\x30\x71\x6a\xba\x7f\x6c\xd0\x2f\xa4\x44\x90"
"\xd4\x0a\x8c\x03\x1f\x05\x90\xc4\x55\xd6\x11\x8d\x12\x57\x7b\x06"
"\x37\xbd\x05\xdb\xad\x75\x22\x28\x99\x27\xc1\x13\xdd\x27\x31\x77"
"\xf4\xa1\x7a\xf2\xc2\x4b\x8e\x8c\x20\x81\xf2\x92\x44\xb0\x81\xea"
"\x35\x26\xa7\xd3\xd5\x9e\x3d\x06\x0f\x9c\x16\x90\x13\x8e\xc8\x35"
"\xab\x3f\xe0\x4d\xdd\x65\x23\x5b\x7f\x52\xe5\x7a\xec\x21\x83\x89"
"\xe3\x27\x53\xec\x47\x47\xdc\x33\x1e\x8c\x68\xd0\xca\x41\x1d\x09"
"\xf7\x43\xa6\x62\x84\x53\xd2\xa4\xf2\x08\x2d\x25\xd7\x67\x18\x08"
"\x4e\x24\x5f\x89\x72\x70\x19\x75\x79\x92\xdf\x00\x5b\xd9\x29\x2c"
"\xa3\xf4\x05\xcc\x5d\xf8\xeb\x7e\xde\xb0\xc3\xac\xa4\xc7\x52\x55"
"\xaf\x44\xaf\x8d\x51\x80\xbf\x75\x07\x07\x1d\x4e\xd5\xaf\xee\xaa"
"\xf5\x0d\x24\x6a\x40\xa2\xef\xa7\xf7\xa9\x71\x14\x75\x27\xa2\xb9"
"\x38\x29\x9d\xa6\x40\x87\xd7\x9e\x08\x95\xb6\xa8\xbd\x32\xe8\x51"
"\xcf\x0d\xb7\xcb\x89\xa9\x78\x30\x6c\x96\x01\x4b\x2a\x2a\x95\x2a"
"\x93\xf9\xe3\x2a\x5f\x5f\xb8\x7e\x42\xb8\x1d\xff\x68\x9e\x1a\xec"
"\x7d\xe9\xa4\xd7\x4a\x73\x1e\xb4\x76\x66\xdb\x67\xef\x7b\x47\xa2"
"\x99\x33\xa2\x29\x57\x07\x93\x13\xe8\x9c\x69\x62\x06\x3d\xc9\x49"
"\xd6\x2d\xc6\xf2\x94\xec\x2c\x1e\x2c\xc3\x12\x30\x70\xeb\x87\x52"
"\xdc\x4a\x24\x50\x53\xbb\x04\x82\x72\xea\x27\xc5\x1e\x98\x8c\x03"
"\x05\x79\x9f\xfe\x62\x10\x1c\x89\xfa\xb3\x1c\x28\xc9\x2e\x8d\xeb"
"\x09\x27\x86\xf4\xa3\xc1\xaa\xed\xaf\x5f\x94\x8c\xd4\x7a\xae\x2f"
"\x7d\x00\x0f\x22\x14\x1d\x3b\x1e\xeb\xf1\xb3\xc9\xc2\x38\x5b\x34"
"\xa1\x0c\x6e\xa4\x05\x3b\xc4\xdf\xed\x35\x18\x24\xa6\x9d\xb3\xef"
"\xdd\xc6\x11\x29\x7a\x4f\xf5\xdb\x74\xd4\xc8\x53\xa7\x26\x62\x73"
"\x8d\x00\xa5\x03\x9f\xdc\x27\x9d\xec\xa0\x6c\xb9\x3e\x85\x0f\x64"
"\x7e\xf6\xb0\xb4\x9d\xe7\x5b\xee\xb9\x9d\xc9\x84\xc1\xda\x99\x54"
"\x33\xbc\x0e\x0c\xb6\xa5\xa0\xba\xbb\x89\x74\x8e\xf4\xc3\x20\x46"
"\x49\xa7\xe6\x37\x75\x82\xc8\x39\x2b\x8e\xf4\x3a\x1f\xc9\x96\x3c"
"\xe0\x74\xcc\x46\x23\x2c\x3b\x29\x0d\x92\x65\x8d\x57\x69\x08\x67"
"\xed\x2a\x62\x5c\xfe\x51\xfe\x2e\x11\x5d\x4f\x44\xc4\xa5\x45\x97"
"\xa6\xb0\x04\xa0\xdf\xd0\xaf\x8b\x17\xa1\x6b\xe3\x61\x5f\x46\xf5"
"\x9e\x98\xa4\x48\xf6\xa1\x6e\x2e\x33\xce\x07\x13\xc1\x37\xb7\x67"
"\x38\xe6\xfa\xdf\xcd\xca\xe7\x03\x28\x48\x5c\x4c\xbe\xa6\xee\x39"
"\x11\x1a\xaa\x78\xcf\x0d\x82\x83\x7b\x23\x65\x41\x4d\x2c\x9c\x3a"
"\x04\x6e\xb5\xae\x77\x38\xac\xc9\x08\x96\x9c\x2e\xd1\x8b\x28\x73"
"\x62\xcd\x76\x8f\x01\x77\x7e\x92\x01\x0d\x65\xd7\x6c\xb1\x87\xe3"
"\x70\xd0\xc3\x2d\x32\xe9\x7f\xe2\xe9\xfe\xe9\x32\x9a\x2a\xe2\x00"
"\x5b\x1e\xf5\x45\xda\xae\xe3\x6e\x84\xb6\x4a\xa6\x1b\x09\x10\x3d"
"\x0b\x55\x43\x72\x47\xd4\x4e\x8e\x01\xc4\x5e\xc7\x47\xad\xf3\x64"
"\xc2\x0e\xf9\x91\x69\xa6\x51\x55\x20\xeb\x1a\x15\x94\x2d\x53\xd5"
"\x1d\xe8\x75\xaa\x71\x58\xfd\xae\x59\xec\xc1\x2b\x95\x26\x9e\xd5"
"\xcf\xab\x92\xa0\x1d\xd1\x14\x17\xf1\xfc\x1f\x61\xe1\xf0\x40\x10"
"\xca\x66\x01\x8c\x54\xed\x80\x43\x00\xc3\xe7\x1d\xbf\x91\xb3\x19"
"\x39\x9d\x79\x06\xc0\x0d\xb2\xc8\x85\xf4\xe8\xe0\x6e\x42\xec\x59"
"\x24\xa4\x51\x8a\x36\xb6\x2e\xb4\x17\x2d\x6a\x57\xb4\xbb\xff\x37"
"\x9e\x3d\x2a\xd7\x3d\xe2\xe9\x0d\x07\x18\xd9\x09\x45\xc7\x1e\x49"
"\x57\x2a\xdc\x95\x64\x2d\xed\xa6\x88\x31\xa5\x2e\xba\x96\x03\xb6"
"\x59\xa9\xc7\xef\x88\xe1\x0c\xf7\xd4\xd9\x43\x96\xf0\x14\xc2\x47"
"\x15\x19\xd3\xb7\x3d\xf7\xf4\x65\x4f\xe9\xef\x62\x77\xd4\x44\x45"
"\xb8\xfc\x01\x37\x8a\x2f\xb8\xfc\x60\x0b\xdb\x44\x16\xa4\x30\xdd"
"\x92\x8c\xe5\xac\xc9\xb4\x3d\xe7\xca\x8c\x0d\xe9\xa8\x3d\xd6\xbc"
"\x84\x3e\x2a\x08\x23\xaf\x9a\x1a\x99\xca\x7e\xb6\xad\x5f\x32\xd4"
"\xb2\x69\x95\xc2\x0c\xb0\x70\x88\x70\x09\x96\xfd\x62\x96\xf7\x83"
"\x11\xc2\xcd\x0a\x1f\x0f\x86\x37\xb7\x00\xd0\xa9\x6a\xdd\xae\xc9"
"\x2d\x93\x56\x03\xd6\x48\xc3\x6d\x16\x06\xa2\xcf\xcc\x1e\x03\x89"
"\xe2\x14\xfb\xe5\x1f\x7f\xc6\xca\x10\x8e\x52\x9d\xeb\xd3\x6e\xf4"
"\x36\x20\x8b\xc5\xed\x49\x3f\x42\x3b\x17\xbf\x5f\x31\x0f\x4e\xd6"
"\x76\x39\x05\x3b\x4c\x1e\x62\x82\x9a\x5b\x48\x05\x8b\x6f\x36\x60"
"\xd4\xea\xdd\x30\xbd\x1e\x75\x00\x14\x9a\xd8\x90\x38\xfe\xee\x08"
"\x7d\xb5\x59\x8c\x19\xaa\xd4\x5b\x50\x2b\x27\x5f\xe6\x3c\xd1\x4b"
"\x7d\xbb\x3e\x67\x50\x88\x36\xd2\xad\xd5\xfa\xfb\x83\xe1\x87\xa6"
"\x9a\xe8\x8d\xad\x4e\x7b\x16\x49\x54\xc3\x7b\x03\x05\xe9\xe0\x61"
"\x12\xb5\x97\x62\xfe\x35\x6b\xb2\x54\x37\xa2\x88\x52\x6f\x47\xab"
"\x66\x83\xf8\x41\x28\xa1\x58\x45\x4c\x0b\x0c\x08\xde\xae\x15\x64"
"\xdf\x2a\x60\xf0\x19\x22\xc0\x48\x07\x35\x82\xf2\x06\x63\xed\xc0"
"\x6a\x04\xb1\xdd\x64\xfd\x78\x53\x57\x4f\x53\xe2\x19\xae\x50\x20"
"\x3b\x75\x63\x58\xb4\x65\xc5\x21\x3e\xee\x04\xeb\xfe\x8d\x3a\xf2"
"\x34\xb1\xbc\x25\xaa\x7d\x87\x14\x85\x1c\x6f\x11\xfc\x77\x1a\xa4"
"\xcc\xb0\x23\x90\xa8\xee\xe4\x38\x4c\x5a\x59\xaf\xb6\x52\x9d\x95"
"\x90\x40\x79\x8d\x19\x6d\x3d\x42\x44\x13\x8c\xac\x68\x7c\xe9\xdc"
"\x6e\xc7\x54\xc2\x8a\x1a\x47\x04\xcc\xaf\x08\x60\xfc\xd3\x08\xdd"
"\x88\x63\x49\xdf\x26\x20\x4e\xb6\x89\x7b\x28\xbc\x9d\xae\x7a\xa2"
"\x2f\x94\xde\x00\xb1\x3b\x5a\x73\x8c\x90\x51\x08\xd2\x49\x20\xbb"
"\x63\xbc\x7b\xe5\x2a\x89\xb2\xd3\xab\x35\x16\xe0\x72\xcb\x77\xb5"
"\x79\xde\x3e\xb3\x49\xd1\xcf\x78\xf0\x13\x58\x64\x97\x3d\x94\x95"
"\x04\x73\xcd\xd1\xc6\x3f\x4f\x3d\xf1\x9b\x7d\x2c\x32\x00\xf4\xda"
"\x47\x71\x44\xc7\x8a\xb4\x38\x4b\x4b\x3e\xfb\x1c\x2e\x19\xc2\x8b"
"\xab\xf0\x6d\xbc\xac\x5c\x9e\xa1\x69\xd1\x03\x20\x3d\xa2\x9d\x4b"
"\xf8\x72\x85\xf9\x1a\x06\x10\x99\x64\xc0\x84\x8a\x34\xf7\x00\x09"
"\x4d\xf0\x83\x8f\x6c\x45\x02\xbd\x45\xde\x4f\xeb\x18\x7d\x36\xd0"
"\x9c\x20\xbe\x22\x37\x30\xa3\x04\x93\x4e\xde\x1f\x3b\x65\xc2\xda"
"\x58\x02\xa1\x26\x44\xbd\x29\x06\x17\x0d\x5e\x20\x19\x8f\x2a\x50"
"\xba\x4b\xb1\xeb\x32\x17\xfa\x57\xae\xec\xec\xff\x07\xc4\x76\xb7"
"\xc0\x47\x2d\x3e\xb2\xf9\xdc\x4e\x21\x16\x10\xc8\x52\xf3\x33\x38"
"\xaa\xa2\x6e\x87\xae\x69\x16\x25\x63\x85\xd8\xaa\xe2\x82\x21\xff"
"\x87\xbe\xaf\x91\xf7\x9c\xab\xa7\xd1\xf5\x7c\x76\xbf\x8a\x0e\x86"
"\x13\x32\xaa\xcf\xb4\x3a\x29\xc4\x41\x39\xa0\x93\xfc\x7e\xb0\x7b"
"\x3c\x2a\x30\xa1\xbe\x71\xa7\x5e\x60\x8d\x4d\x94\xb8\x56\x88\x8d"
"\x90\xdb\x58\xb3\x7d\x1c\xc8\xa1\xb4\x10\x7e\xcd\xcd\xcd\x65\xc8"
"\x49\x81\x67\x48\x72\x50\x2f\x07\xe2\x28\x61\xb1\xa5\xc7\xd5\x0b"
"\xc3\x33\x8e\xfc\xb5\x3c\x76\xf4\xe7\x94\x53\x9a\xc8\x3c\xe4\xb8"
"\xf6\x3c\x85\x35\xbc\x31\xc8\x81\x48\x70\xbb\x37\x9a\x0c\xfa\xe9"
"\xd7\x9f\xe1\x24\x73\x14\x5f\x38\x58\x16\x89\x21\x9a\x8b\xba\x03"
"\xd0\x7a\xfc\xb1\x6d\xc3\x31\x2f\xa7\x95\xb8\x71\x23\x19\xbb\x97"
"\xca\x92\x6f\xb0\x22\xcd\xf6\x62\x77\x95\xef\x87\xf0\xd5\x32\x2f"
"\xb9\xd9\xc6\x32\xb3\x38\x4e\xdb\xcb\x59\xee\x6d\x44\xa7\xaa\xa1"
"\xaa\xdf\x73\x3f\x50\xab\xde\x2d\x85\x8a\x8e\x69\x31\xed\x4e\xca"
"\xaa\xec\x20\xab\x12\xf3\x37\xce\xe0\x14\x53\xe7\x75\x9c\x5e\x97"
"\x42\x66\x51\x9f\xfc\xbc\x33\x81\x4f\x9e\xdb\x1e\x21\xb8\x03\x8e"
"\xfa\x05\xba\x75\xba\xdb\x0e\xe1\xae\xe5\x3c\x52\x1e\xe3\x65\xc1"
"\xaf\x39\x0d\x39\xc8\xb7\xbe\xd0\xf8\x4e\x1b\xb2\xb0\x9d\xcb\x57"
"\xa4\xc0\x53\x94\x2b\xc6\xb1\xbb\x68\x49\x5b\xce\x03\xfe\x60\x38"
"\xe1\x3c\x25\x13\xd7\x0b\x1c\x62\xdd\xd6\x42\xe0\x30\x41\x5f\xeb"
"\xcd\x5d\x75\x93\xe4\x94\x03\x09\x4c\xa1\xa3\x71\x82\xb5\x0c\xac"
"\xb5\xc5\xaf\xad\xf6\x4b\xe5\x17\x5c\xb6\x12\x26\x0c\xc4\x3f\xae"
"\xce\x10\xc3\xf4\x59\x14\x54\xd6\xc9\x49\xce\x55\x8b\x9d\x9d\x74"
"\x18\xe1\x37\x4c\x9b\xa6\x3f\xd3\xd4\xd5\x12\xf4\xce\x3d\x3a\x8c"
"\x1c\xab\x8b\xb7\xed\xe3\xb3\xfd\x81\xa7\x79\x1a\x90\xb4\x97\x80"
"\xfb\xb8\x49\xfe\xed\xc3\x43\x28\xd9\x92\xce\xa9\xdb\x38\x10\xa7"
"\x6c\xb3\xff\x13\xb3\x5a\x89\x8c\xbe\x62\x1a\xd8\xe6\x64\xba\x59"
"\x13\x7e\x2f\x36\x77\x37\xd4\xbe\xce\x60\x17\x30\xdb\x59\xbe\xd6"
"\xa5\x1c\x93\x48\xd3\x95\x43\x0f\x14\x4d\x4c\x9e\x99\x69\x9a\xed"
"\xb4\x5e\x0d\x67\xda\xf5\xc3\xf6\x85\x4e\xc5\xc0\xbc\x28\x95\xf9"
"\x7c\x0b\x7a\xd3\x0b\x3b\x5a\xcc\xbe\x16\x4a\x48\xa7\xbe\x80\x39"
"\x59\xe8\x8d\x64\x20\x00\x0a\x3b\xa9\x5d\x85\xc1\x9d\x32\x9e\x60"
"\x94\xed\x8a\x41\x39\x86\x5a\xc0\xf8\xaa\xc0\x0e\x49\xc2\x48\xcd"
"\x93\x13\x4e\x45\x21\x4a\xb9\x3e\xf3\x42\x78\xc5\xac\xcd\x16\x2e"
"\xad\xfa\x65\xbb\x88\x2e\xfb\x2d\xb1\x73\xf2\xfd\x03\x2d\xe4\xb6"
"\xb9\xcb\x8b\x06\xb1\xe7\x9d\x1d\x8c\xdf\xc9\xe8\x99\xdb\x3e\x88"
"\x02\x27\x3e\xd6\x51\x5b\x56\x21\xb5\xa1\xc1\xb3\x99\x2b\x09\xe2"
"\x8a\x06\xc3\x6b\x2d\xf1\xc9\x27\x1f\xb4\xa3\x4e\xdf\xbe\x77\x71"
"\xe9\x0b\x81\xdb\x54\xfc\xd0\xd1\x17\xad\x9b\x01\x60\x00\x4f\x25"
"\x2e\x6c\x7f\xd1\xd4\x25\x6b\x4c\x56\x8d\x4f\x86\x02\x33\xc1\x9c"
"\x97\xa6\x51\x9e\xfd\xe4\xa4\x61\x60\xe9\x2d\xee\x14\xcd\x71\x5d"
"\x13\x7e\x04\x39\xe8\xb9\xb8\x92\x48\x72\x87\xd3\x3e\x6c\x80\x5e"
"\x2b\x52\xf9\xdd\x79\xd8\x74\x75\xb3\x4c\xdc\xf6\x0a\x6e\xf7\x91"
"\xea\x6c\x76\x7c\xd7\x23\xfe\xda\x9f\x10\xc6\x59\x38\x13\x33\xc8"
"\xdb\xc6\xaf\xf8\x0f\x15\x0a\x26\x1a\x42\xee\x93\x1d\x1c\x00\xe3"
"\x49\x18\x6c\xc3\x1b\x49\xd4\xd2\x90\x09\xe4\xc9\xa9\x7e\xb5\x64"
"\x81\x57\x9a\x0b\x1c\x41\x57\x7a\xb5\x1d\x9d\x01\xed\x77\x8c\xe7"
"\xe4\xb6\x81\x0c\x68\x70\x20\xd8\xe7\xdf\x75\xec\xae\x03\xb9\x22"
"\x82\x57\xda\x2c\x32\xa5\xd0\x2c\x51\xc8\xf5\x7d\xc4\x1c\xdf\x3f"
"\x50\x33\xc3\xbe\xd4\xad\xde\x6c\x5e\x84\xa2\x1b\x07\x0c\xd2\xb0"
"\x97\x9d\xcc\x1f\x2d\xff\x54\xb0\x42\xd3\x31\x00\x9e\xd7\xc6\x84"
"\xf1\x21\xbe\x64\x5a\xd9\xfa\x95\x70\x37\xdf\xdd\xdb\xad\xe3\xe4"
"\xf5\x98\x30\x87\xbc\xfc\xd9\x13\xdc\xb0\xa5\xcb\x66\x40\x8d\x98"
"\x87\x96\xe6\x58\x18\xbf\x94\xdd\xaf\x97\x9d\xe8\xae\x93\x87\x4e"
"\xee\x6e\x36\xf2\xc6\xf9\x54\x84\xf7\x90\x5f\x80\x1f\x37\x63\x44"
"\xcc\x9d\x8f\x77\x90\x70\x0c\x04\x14\x8a\x3b\x89\x41\x9d\x6b\xa5"
"\x12\x3b\x33\x3c\x64\xa9\x9c\xaf\xd1\x75\x76\x24\xc7\x11\xd7\x39"
"\x99\x73\x99\xb7\x35\x0c\x92\x84\xec\x7a\x61\x54\xa2\x41\x1d\x40"
"\xc9\x1d\x63\xd7\x80\xf7\xa6\x9e\xbd\x50\x72\x83\x97\x5e\x02\x1f"
"\xa2\x35\x7b\x99\x8a\x8c\x61\xa7\xa7\xb9\x75\x6e\xc6\xc3\x7d\x2c"
"\x55\x3b\xfc\x71\x46\xa9\xa1\x41\x03\xa6\x8a\xeb\x9a\x35\xbd\x5e"
"\x65\x94\x93\xb2\xf8\x87\xbc\x06\x17\x1a\xc6\x17\x4a\x0e\x79\x03"
"\x21\xb7\x57\xf0\xbd\x14\xb6\x85\x33\x70\xb6\x26\x5d\x75\x98\xd9"
"\xe4\xfc\x0f\x7d\xcb\xea\xe0\x2c\xfd\x45\xbb\x09\xeb\xe6\x03\x43"
"\xf7\xb6\xc3\x96\x6b\x29\x91\xbd\x4c\x73\x1b\x3c\xb9\x7d\x26\x48"
"\x51\x02\xac\x15\xc7\x17\xb0\xac\x62\xc1\x91\xa5\x2d\xdc\x04\xd9"
"\x02\x31\x4e\x7c\x05\x69\x9e\x85\xbb\xd8\x34\xe8\xab\x5a\x75\xeb"
"\x17\xb8\x98\x34\x1c\x25\x83\xbf\xef\xcb\x93\xf5\xd7\x73\xbe\xca"
"\x07\x70\x95\xf6\xd0\x58\x7b\xab\x6f\x5e\xf0\xd5\x47\x59\x74\x82"
"\xd7\x8a\xaf\xf1\x4a\xcb\x31\x64\x58\xf0\xba\xff\x85\xfd\xdf\x07"
"\x48\x87\x96\x51\xdd\x36\x55\x56\xc7\xbe\xa1\x5d\xfc\x06\xa2\x69"
"\xc9\x15\xea\xdc\x03\xb8\x43\x8d\x70\xc4\x0a\xb4\xd8\x84\xe1\x09"
"\x5f\x03\x63\xa9\xa4\x15\x9c\xf3\x3c\xf7\xdf\xc3\x97\x61\xd0\x4a"
"\xed\x78\x43\x3f\x08\x3c\x93\x3b\x30\xf1\x23\x32\x4e\x77\x24\x74"
"\xe5\xc0\xec\x73\x31\xdf\xad\xd9\x52\x73\xcb\x11\x5b\xea\xfb\x4f"
"\x5b\xc4\xd2\x22\x9f\xa7\x07\x37\x95\x24\xb7\x2a\xa5\xd3\x98\x7b"
"\x9c\x13\xfb\x9b\xdf\x40\x34\x16\xf3\xac\x5d\xee\x02\x25\x62\xe8"
"\xe1\x8d\x07\xcd\x5c\x0e\x81\xc1\x99\x9a\xc0\xc6\x0b\xa4\xa3\x84"
"\x2c\xd3\x45\x57\xf8\x6c\x3d\x2b\x1a\x0d\x8f\x79\x21\x4c\x0c\x9b"
"\xbb\x5d\xb8\x9f\xdd\x51\x93\x52\x07\x78\xdd\x34\x09\xd8\x3f\x2b"
"\x47\x07\x47\x13\xd8\x03\xd3\x8c\xe3\xc7\xd7\x66\x1f\xf5\x76\x55"
"\xd8\x80\x77\x87\x0e\x6a\x92\xc9\x14\xc1\xac\x81\xf4\x03\x39\x65"
"\x11\xba\xcc\xa1\x3f\x98\x00\x7c\xc3\xbc\xab\xa9\x2d\xcc\x09\xc2"
"\xbf\x20\xa0\xb1\x24\x3b\x1f\x12\x0e\x65\x26\xeb\x65\x40\xb8\x9d"
"\x1b\x67\x60\x00\xff\x3b\xeb\xa6\x58\xdc\x3f\xb2\x6d\x92\xdd\x40"
"\x39\xe6\x28\xd1\x0d\x6f\xae\x6f\x02\xc9\x1c\xf5\xc8\xa9\xc9\xef"
"\x2d\xd0\x25\xe3\x86\x3c\x44\xec\xfd\x07\x13\x6a\xb2\x65\x4f\xd2"
"\x81\x11\x2e\x3c\xc5\x0d\x05\x77\x6e\xb8\xfd\xfd\x73\xfd\x8c\xaa"
"\xee\xc9\x17\x5e\xac\x2e\xe0\x6e\xd5\xda\xf2\x4b\x60\x0f\x6f\x23"
"\x89\xee\x44\x92\x1a\x7c\xec\x9d\x03\xa2\x68\x2a\x6d\x44\x36\x5f"
"\xef\x60\xf5\x0c\x2e\xb5\xa8\x6c\x3c\x2f\xac\xea\x4a\x4c\xb9\x9a"
"\x42\x88\x91\xd1\x7e\xbf\xcc\x09\x90\x8b\xe7\x28\x70\x41\x35\x10"
"\x69\xe0\xd7\x83\x7d\x25\xa0\x75\xdc\x54\x20\x31\x6f\xcc\x4b\x0c"
"\x5b\x8d\x22\x02\x05\x19\xa9\x86\x58\xf3\x00\xf5\x9b\x8d\x6b\xd9"
"\x81\xf5\x4a\xa2\x7e\x53\x00\x46\xd1\x35\xf5\x4e\xba\xfe\xbc\x98"
"\xc4\x8f\xb9\xbf\xc9\xae\xb4\x1e\x63\x5a\xbe\x82\x16\xbf\x0b\xae"
"\x83\xac\xcb\x4f\xb2\xb5\xff\x06\x51\x5d\x34\x6e\xfa\x4a\x4f\x1a"
"\x94\xf4\xd1\xd4\xdc\x9e\xb4\x45\xa9\x3f\xb6\x78\xa0\x32\xde\xc3"
"\x7a\x31\x97\x78\x91\x4a\xb3\x2c\xd2\x10\x3c\x9a\xec\x7b\x72\xa8"
"\xa2\x30\x9d\x9e\xa2\x11\xfd\xac\x6a\xb2\x7a\x33\x36\x94\x4b\xc7"
"\xa1\xc0\x06\xde\x04\x29\x93\x1f\xf4\xe9\x0a\xe3\xbb\x9e\xf3\xea"
"\x61\x87\x4c\xe3\x75\x05\xb8\xde\xf2\x15\x18\x6e\xbc\xc4\x2a\x26"
"\xc8\xa7\x18\x78\x65\x18\x0e\xa0\x2a\xfa\x25\x77\x27\xc2\x3d\xc7"
"\xcd\x55\xe9\x30\x0d\x76\x5d\x4c\xb6\xd4\x87\x41\x8d\xae\xcb\xb3"
"\x81\x86\xee\xe2\x8a\xd7\x28\x25\xc5\x90\x65\x0a\x23\x0b\x7b\xd4"
"\xf5\xe6\x0e\x56\xab\xa5\x3b\x7c\xbc\xfe\x5f\x49\xda\x69\xb2\x43"
"\x79\xbc\xdf\x7a\x6a\x02\xb5\x6d\x6d\x73\x70\x86\xe3\x3a\x1a\x47"
"\x2b\x72\x67\xaa\xfd\xe6\x67\x4c\x57\xca\xc8\x2a\x9b\xf2\xad\x69"
"\xce\x09\x92\xf6\xb4\xfa\x2f\x9f\xc7\x5e\x62\x7b\xc8\xd2\x4b\x48"
"\x85\x65\xb5\x2d\x62\xb1\x68\xa9\xab\x9f\x79\xe9\x66\xc9\x95\x4f"
"\x3c\x99\x54\x91\xc3\x7a\x25\x6d\x09\xc8\x22\x49\x09\xdb\x67\x7b"
"\x90\x29\xd6\x70\xe9\x64\x2e\x20\xab\x3a\x10\xb7\x63\xc9\x87\x9c"
"\x18\xa8\x30\xbe\xb6\x43\x06\xe8\xff\xdf\xb5\xb6\x3a\x82\xa2\x71"
"\x85\x15\x61\x99\xaf\x1a\x8d\x4e\x5f\x58\x17\x1c\x10\xa2\x0a\x64"
"\x4f\x5f\x91\xbe\x1a\x62\x08\x76\xa5\x85\xd2\xa2\x67\x4c\x0d\x75"
"\x53\x48\x76\x9d\x8a\xd2\x41\xb8\xa0\xbc\x75\xa9\x83\x41\x88\xda"
"\x00\xdc\x51\xd2\x72\x33\xaf\xb5\x83\x01\x8e\x46\x96\x0a\x0e\x85"
"\x5d\xed\x7c\x39\x69\x00\xc1\x2a\x76\x95\x25\x15\x4d\x2e\xf2\x81"
"\xf6\x3f\xc4\x7d\x1a\x66\xe0\x77\x75\x08\x36\xa4\xa5\x8d\xf2\x9d"
"\x7f\x1f\xeb\xa7\x9d\x7c\xa7\xdd\x72\xd4\xb7\x84\x35\x54\x65\xec"
"\xf4\x87\xf6\x78\x74\xb8\x8b\xf1\xaf\xd4\x52\x02\xeb\x53\x7f\x0b"
"\xa1\x84\xa0\xaf\x74\xbc\x0c\xda\x82\x3e\x8e\xde\x38\x85\x08\x6f"
"\x0d\xdc\x26\x99\x55\xe8\x37\x2b\x02\xea\x7b\xf7\x56\x71\xd0\xcc"
"\x12\x6a\x44\x6f\xde\x81\xb7\x49\x91\x21\x4c\x3b\x2f\xf4\x97\xb2"
"\x8e\x26\x14\x00\xf0\xf7\x73\x0c\xcf\x88\x7f\x61\xa8\x97\x6b\xb4"
"\x96\xc1\xc9\x7b\xdf\x21\x50\x13\xa9\xb0\x56\xa8\xb4\x7f\x4e\xfb"
"\xba\x49\xa3\x70\x7e\x7d\x06\x66\x98\x1a\xec\xdb\x32\x7d\x9f\xb2"
"\x55\xd3\x8c\x2a\x7b\x8f\x77\x4b\xbe\xdf\x60\x0f\x90\xdd\xa0\x56"
"\xef\xa1\x2c\xc9\xb6\x08\x94\xf5\x47\x36\x7b\xa3\x1a\x58\xf3\x44"
"\x49\x1b\xa4\x35\x41\x3a\x71\xc2\x1c\x88\x07\x1b\xb0\x19\x0f\x90"
"\x5f\xe5\xc0\x97\x02\xb3\xbc\x5a\x16\xc7\x63\x01\xe4\xc3\x82\x87"
"\x59\xdb\x28\x07\x27\xa9\x92\x98\x25\x6f\xf3\x62\x58\xbf\xf2\x46"
"\x68\x92\x9a\xf4\x2e\xf5\xd0\x3e\x93\x27\x62\x8a\xed\x39\x02\x51"
"\xec\xd6\x31\x10\xcf\xfb\x2f\x25\xa3\x63\xb9\x71\x05\x2d\x06\xfc"
"\x97\x65\xe8\x61\x7c\x50\x44\xcc\x32\x55\x94\xa1\x58\xca\xf9\x99"
"\x83\x62\x53\x4e\x40\xd7\xec\x69\x61\xa1\x4b\xa9\x20\x0a\xc5\x26"
"\x70\xa8\x1b\x0c\x01\x99\x9b\x53\x8d\x41\x2d\x65\xb8\xa1\xc7\x88"
"\x71\xfa\x5b\x06\x92\xae\x4c\xf0\xd5\xeb\x91\x87\x86\x2a\x99\x01"
"\x77\x31\x9e\x9c\xb7\xda\xa5\x23\x46\x77\x67\x19\x44\xf4\x16\x52"
"\x7a\x05\x5e\x29\x91\xd7\x90\x34\x38\x4c\xc9\x6d\xf2\x1c\x1d\xa5"
"\x0c\xa5\x36\x53\x6c\x0e\x46\x5f\x07\x0b\x43\xbb\xb2\x2c\xdb\xcf"
"\xd1\x25\xac\x4a\x64\xda\x9e\x72\xe4\xb2\xfe\xd9\x80\xbe\x03\xa0"
"\xe6\x32\x09\x1b\x35\x51\xb5\x67\xa5\xf4\xc5\x7a\x14\xee\xff\xee"
"\x58\x17\xb3\xc0\x91\x1e\xb9\x4e\x5b\xc0\x90\x92\x1d\x60\x99\xa7"
"\x0f\xcc\xfd\x6d\xce\x94\x35\x2f\x44\x26\x5a\x74\xa3\x87\xe1\x64"
"\xa7\x9f\x5b\x3d\xf5\xd0\x60\x4a\x60\xb7\x98\x83\x54\xf5\xee\xb1"
"\x57\x83\xd4\xdb\xf8\xbd\x7d\xf6\x5b\xac\xdc\x5b\xc4\x16\x9a\xa9"
"\x31\x7c\x75\x6e\xae\x0d\x7c\xcc\xe3\xab\x15\x4b\xb0\x67\x77\x2f"
"\x29\x88\xc9\x55\x83\x3f\x3a\x59\x4d\x51\x85\x89\xe8\x3c\x51\x88"
"\xe2\x82\x29\x84\x1d\xa8\xdc\xf4\x3e\xce\x4a\xa7\xaf\x4a\x30\xdf"
"\xe9\x4b\x4d\xeb\x4a\x33\x5c\xf3\xce\xfc\x05\x68\x26\xe6\xfe\xe2"
"\x3d\xf3\xb8\xc7\x5c\x79\xb1\xb1\xe7\x01\x95\xbf\x16\x6e\xcd\x74"
"\x64\xad\xa2\x93\xb1\x67\x22\x2f\x0d\x55\xd7\xf2\x21\x5a\xb2\x6c"
"\xdf\xa0\x2b\x37\xa2\x0c\x82\x42\x90\x2c\x04\x92\xe3\xfc\xa8\x14"
"\x65\xe4\x5c\x6f\x4f\xe2\xf5\x2a\xff\xa8\x6e\x3d\xd7\xfc\x36\x94"
"\x61\xe8\xab\x2e\x03\x04\x77\xd2\x4c\x91\x1b\x5b\x55\x7e\x65\xa3"
"\x81\x43\x1a\x66\x4f\x99\x67\x24\x6f\x07\x9b\xb5\x67\x4c\xe8\x0b"
"\x60\xbf\x61\x70\x9a\xed\xa8\x55\x09\x50\xc0\x05\xad\xde\xc8\x38"
"\x57\x8b\x9e\x08\x22\x1d\x82\xda\x8a\xa0\xf8\x01\x62\x7b\xae\x94"
"\x86\xea\xee\x84\x82\x4a\x71\x79\x70\x62\x87\xe3\xdc\x9f\x23\xb4"
"\x81\x36\x69\x54\x15\xa9\x6c\xf1\x9e\xed\x5a\x35\xd5\xba\x19\xd1"
"\xb8\x31\x0c\x73\xc1\x8a\x9f\xf5\x46\xa4\x25\xea\xcb\x5f\xa4\x83"
"\xe2\xd8\x1c\xcb\x5f\x2f\x61\x72\xf8\xf5\x1d\xe7\x05\xb9\x10\x0a"
"\xb5\x0c\x2a\x4f\xce\x2b\x3c\x2a\xfd\x45\xcc\x68\xa9\x72\x04\x96"
"\x47\x86\xc4\x06\xac\x98\x89\x09\x63\x82\xa5\xf6\xde\x77\x09\x05"
"\x33\x4a\xd8\xa4\x14\xa1\x55\x54\xe1\x2f\xf6\x93\x76\x77\x55\x61"
"\x30\x2c\xc2\xf2\x4f\x70\x41\xb0\xd4\x31\xdb\x82\x38\x53\xec\xaa"
"\x35\x8d\x6d\x93\xa2\x8f\x83\xfe\xfe\xe2\xf9\x0a\x1f\x16\x64\x3d"
"\xa9\x11\x35\x63\xd0\x44\x08\x48\x3c\x56\xd4\x51\x84\xe4\xcf\x28"
"\x03\x06\x3c\x89\x9d\x94\xa7\x88\xf2\x0d\x1d\x22\x08\x7e\xc7\x8e"
"\xd9\xb3\x3c\x72\xb2\x0b\x43\x57\x0b\xa0\xd4\x68\xc9\xaa\x86\xba"
"\xaa\xf0\x91\x86\x27\xa5\xbd\xaf\x2a\x25\x97\x9f\x44\xe0\x4d\x68"
"\xaf\x5d\x16\x5c\x3b\xcb\xa7\xd0\xb2\x6b\x0c\x65\x9b\xde\xcc\xe3"
"\x38\x63\x00\x8f\x27\x9c\xb5\x1e\xc9\xb6\x6d\xf8\x96\xfa\xfa\xe6"
"\x7c\x1c\x92\xeb\x7c\x3f\x90\x0a\x6d\xd9\x33\xa6\x92\x3a\x8b\x5c"
"\xb3\x35\x29\x00\x51\xf7\xed\x92\x65\xeb\x6f\xaf\x2f\x20\x10\x40"
"\x04\x67\x54\xa8\xe0\x44\x8b\x81\xb0\xb2\x96\xd9\x20\x30\xc9\xbe"
"\xe1\x91\xa6\x40\xa5\x2e\x20\x7b\xbf\x35\xf9\x02\xfd\xa3\x62\x56"
"\x1e\xe4\xbb\x8d\x5d\x9f\x9b\xc8\x95\x82\x05\x78\x09\x90\x8d\x62"
"\x08\xc2\xc0\xc1\x4a\xe4\x46\x0e\x9f\x67\xc5\x3a\xb7\xbf\xce\xff"
"\x10\xbe\x7b\x67\xbb\xfa\x62\x12\xe6\x25\xc0\xc6\x59\xf9\x36\xbf"
"\x19\xf6\x6f\x6e\x57\x65\x40\xb9\x79\x14\x8c\x07\x90\xab\x4c\x21"
"\x9c\xd8\x3f\x7b\x57\xc2\x68\x1d\x4a\x2b\x21\xdc\xee\xce\x77\xc4"
"\x30\x82\x1a\x6a\xb6\xea\xf0\xe7\xfa\xde\xcc\xd9\x3f\x65\x8e\xe9"
"\xa4\x71\x1c\xc1\xc9\x88\x72\x0a\x28\x73\xf1\x6c\x5f\x5c\xb9\xc6"
"\x00\xd5\xdb\x9e\xce\x89\x34\x34\x83\x65\x5f\x4f\xf5\x6c\xd7\xb9"
"\x78\x3b\xc7\xe0\x44\x52\xc1\x1b\x9a\x11\xdd\x9d\x9f\x4c\xc8\x35"
"\x29\x73\x53\x23\x32\x13\x1d\x12\x73\xaf\x50\x2e\x03\xf2\xa6\x79"
"\xc3\xa6\xe3\xcb\x1b\x9f\xb0\xd9\xd4\xf1\xa9\xf1\x37\x5b\x21\x7e"
"\x62\x23\x4e\x16\x01\x7d\xfd\xe8\xd3\x60\xaa\xaa\xd2\x69\xa8\x10"
"\x13\xe3\x94\xfb\x0a\xf1\xe9\x5e\x06\x40\x75\xb5\x14\x43\x73\xcf"
"\xd2\x7a\xb0\x29\x9d\x60\x82\x2b\x83\xb4\xdd\x28\x18\xe1\xaf\xb5"
"\xf1\x90\x83\xc2\x7c\x10\x91\x38\xc1\xec\x57\x01\x63\x78\xea\x86"
"\x85\xf6\xa7\x0a\x3c\xdf\x37\xb7\x70\xb0\xa9\x51\x86\x58\xc8\x8f"
"\xd3\x1f\x83\x72\xfb\x58\xcd\x12\x01\x92\x6e\xf5\x5d\xdd\xfc\x41"
"\x46\xe2\x25\xbb\x1d\xd6\xea\x4b\xe1\x23\xb8\xb2\x0a\x4a\x48\x11"
"\xd8\xf6\x51\xb1\x08\x90\x7c\x18\x10\xe4\x87\x6d\x1c\xa4\x0c\x0a"
"\x59\xa2\x07\x3a\x78\xf1\xa4\xe1\x86\xb4\xe6\xc1\x23\x00\xb7\xc9"
"\xcb\x8a\x33\x71\x18\xd7\xff\x97\x66\x68\xfc\x29\xb7\xc3\x72\x3f"
"\xab\xfd\xe8\x79\xf3\xe9\x25\x86\x67\x71\xe0\x67\x77\x25\x5a\xdb"
"\xf1\x37\x54\xd8\xda\x3c\xc9\x34\xfa\x2d\xc2\x4b\xbc\x3c\x93\x61"
"\x22\x79\x25\xc9\x68\xc3\xc7\x77\x5a\x29\x81\x0e\x83\x28\x63\x9c"
"\x4b\xa0\x7e\xc5\x56\x8a\xf4\x8d\xd5\xe4\xe7\xee\x77\x2a\x40\x31"
"\x8e\xd1\x40\x9f\x09\x4a\x9d\xab\x4b\xfe\x12\xd1\x03\xbe\x36\x5c"
"\x24\x91\x41\x17\x2a\x52\xe6\x01\x1e\x48\x66\x47\xb8\x8c\x0c\x0d"
"\x51\x5d\xef\x76\xf8\x4a\xb5\xad\xcd\x64\xd6\x83\x32\xe2\x79\xfa"
"\x01\x6d\x42\x38\x24\x60\x3d\x7f\x17\x37\x49\x59\x2d\xb0\x3a\xaa"
"\xe1\x86\x44\x12\xf1\x16\xb1\x3a\xf8\x6d\x60\xf9\x88\x86\x7d\x04"
"\x3b\x98\x06\xeb\xc5\x20\x09\x4a\xb0\xdb\x16\x17\x9b\x88\xf4\xa7"
"\x75\xd2\xcc\x68\x06\x43\x58\x39\x37\x69\x7c\xc9\x4c\xcf\x0d\x21"
"\x43\xe8\x55\x2e\x93\xae\x54\x68\xea\xcc\xc3\xb3\xa7\x6e\x32\xdb"
"\xa3\x0b\x84\x42\x88\x31\x5d\x89\xac\xf1\xdc\x8e\xbf\xb3\x82\xf0"
"\x87\x6c\x0b\xb7\x68\x60\xf2\x7d\xaa\xad\xee\x17\x8c\xfd\x35\xc6"
"\x2c\x12\x00\x06\x5e\x1c\x6c\xbd\x3e\x3c\xa0\x6e\xca\x57\x1a\x45"
"\xea\x26\x4f\xe9\x55\x27\xdc\x66\x58\x0a\xee\x28\x98\x85\xfe\xd3"
"\xa6\xd2\x10\xdd\x58\x0f\xb5\x45\x0b\x49\x95\x07\x37\x1a\x8d\xb8"
"\xe9\x9d\x1d\x7c\x74\xa5\x5d\x34\x1c\x28\x56\xe5\xa0\xd8\xae\xe5"
"\x43\xd9\x14\xb3\xe3\x13\x5a\xd2\x90\xc9\x69\xdf\x07\x28\xa5\xa8"
"\x90\xd5\x0a\xe9\x56\x54\x38\xc0\xba\xd3\x6f\xdb\x10\xd9\xe2\x15"
"\xae\x95\x3b\x5d\x10\xc4\xff\x13\x19\xaf\xd5\x38\x5e\x8d\x23\x92"
"\xa9\xa1\x44\x02\xf9\x29\x65\xdf\x5f\x84\x5f\x4f\xf0\x93\x75\x27"
"\xed\x53\x3b\x72\xb1\xa7\x5c\x5e\xd0\x4a\xc5\xdb\x59\xe6\xd6\x70"
"\x0f\x1d\xfc\x82\xf5\xae\xa6\x6b\xd6\x42\x01\x26\x79\xc5\xb8\x68"
"\xa3\x99\xd2\xe2\xda\xde\x2e\xe2\x2d\x74\xfe\x21\x08\x8f\xcb\x9d"
"\x85\xc5\x88\x75\x6c\x3a\xee\x6c\x02\xe9\x32\xce\x99\x52\x80\xa6"
"\xbe\xc6\x12\x22\xdd\xf3\xc2\x69\x82\xe4\x9e\x27\x38\x14\xed\x92"
"\xe4\xd7\xb3\xba\x6e\xf2\x71\x41\xef\xf7\xb0\x8f\x4c\x4c\xec\xa8"
"\xef\xd1\x77\x86\xe7\x27\x7e\x21\x37\x2d\xfc\x33\xe6\x72\x9b\x90"
"\xba\x3a\xdc\xd6\x7a\xf6\x61\x91\x58\x6b\xe7\x91\x38\x5d\x83\xf4"
"\xe4\xd0\xdb\xc3\x8d\x23\xd9\xb6\xbd\x0b\x63\xa6\xa8\xe7\x26\xbc"
"\x7d\x70\x56\x73\xfb\xc3\x45\x97\x8d\xdb\x30\x21\x6f\x1c\x5a\x2b"
"\x88\x28\x34\xc2\x32\xef\x2a\xe1\x53\x7d\x03\x48\x26\xbf\x26\x7a"
"\xf2\xbb\x0f\x21\xd4\xa3\xf0\xdc\x14\x3d\x4c\x0e\x41\x9f\x80\xcd"
"\x4f\xd1\x76\x5b\xf9\x53\xee\xa8\x86\xe4\x70\x65\xb8\xa7\xb4\xa6"
"\x14\x5a\x13\xff\x78\x1c\xca\xec\x8d\x04\x75\xde\x67\x89\x8c\xa7"
"\x56\x7c\xfa\x13\xf3\xc9\x25\xa8\x33\xb3\xa3\x7a\xd9\xa4\x64\xec"
"\x98\x78\xba\x69\x1b\xe1\xb8\x93\x07\x13\x75\xe2\x4f\xa5\x7b\xa6"
"\xf9\x3f\xd1\x27\x77\x74\xb3\x8f\xcc\xac\xbe\xc7\xbd\xf5\x5f\x28"
"\xbb\x80\x65\x4f\xa7\xc9\xae\xbf\xd9\xe7\x0d\x63\x5a\xf5\x04\xff"
"\x91\xc1\xd9\xcb\xdd\xc2\x6a\x54\x00\xb8\x37\x7b\xf1\x94\x1c\x81"
"\xf0\x54\x56\x5a\x62\x84\x3c\xed\x45\xda\xb8\x90\xe4\x8c\xcc\x2d"
"\x5e\xff\x16\x77\x26\x16\x42\x9e\xdb\x38\xe2\x9f\x07\x6d\x2e\xc1"
"\x2e\x2d\x3a\x7d\x10\x4f\x4b\x90\x10\xf9\xf0\x47\xe1\x5d\xe3\xb7"
"\x68\xa0\x80\x16\x41\xe3\x8d\xbc\x8d\x8d\xc7\x9f\x4b\x6d\xa8\x80"
"\x59\x45\x35\xf3\x67\xc9\x6f\x20\x19\x87\x26\xc9\x58\xd9\x1d\xc5"
"\xc1\x80\x50\xec\x35\x14\x21\xa7\x5b\x4d\x88\xd4\xf0\x37\xf4\x3a"
"\xc6\x33\xbc\xf8\xaa\x57\x11\xe9\xed\x8c\x3c\x69\x9d\xe3\x09\xde"
"\x53\x78\x37\x16\x03\x1e\x65\x87\xaf\x4a\xd2\xb7\x25\x19\xf7\x1d"
"\xfe\xfe\x19\x48\x4d\xa1\x27\xd0\xf3\x65\xea\x5f\xda\xcb\xef\x2f"
"\x11\x91\x75\x71\x7e\x66\xb9\x7e\x68\x2a\x78\x92\xe4\xa1\x54\xd4"
"\x76\x53\x90\x70\x7b\x62\xdc\x79\x56\xdf\x2f\xce\x1a\x5f\xde\x48"
"\x3b\x1d\xd7\xc5\xaa\x46\xcd\x0c\x46\xe8\xf3\x4a\xbf\xf0\x06\x44"
"\xae\x3b\x8e\xbc\x5d\xec\x8a\x49\xf9\x55\xdf\xab\xce\x9a\xa5\xa7"
"\x71\x85\x4c\x94\x49\xf6\x8f\x2b\x0f\x40\x39\xf7\x9a\x15\x9c\x03"
"\x10\xf5\x0c\x9a\xed\x5c\x5c\x7e\xf1\xfd\xbe\xb3\x78\x05\xf3\x03"
"\xc4\x81\x9c\xd0\x12\x1f\x92\x13\x75\x31\xe2\x53\xbf\xfa\x70\x98"
"\x01\xf5\xc1\xe7\xfe\x36\xe6\x24\x38\x7e\xaa\x7d\xdd\xb9\x05\xb6"
"\x79\x18\x9d\xf6\x62\x85\x68\x14\x16\x62\xfa\x72\x07\xb1\x8d\xea"
"\x47\x11\xa2\xeb\x03\x0f\x7c\xaa\x08\x80\x74\xb5\xe7\xf9\x43\xf3"
"\xcb\x24\xac\x9b\x47\x75\xa9\xb2\x9e\xf0\xbd\x4d\x3f\x51\x59\xc3"
"\x68\x6b\xe5\xdc\x5d\x32\x34\x9a\x9d\xef\x5c\x57\x24\x2a\x06\xaa"
"\xf7\xd3\x1b\x7f\xe2\x51\x50\x7b\xfa\x6c\xc0\xf3\xf2\x6f\xc0\x75"
"\x22\x12\xc6\x7f\xc7\xe8\x8c\xb8\xb4\x54\x7b\xe5\x37\x33\xab\x23"
};

/// BP_HI_PLUS_PRE: bytes
STATIC const mp_obj_str_t mod_trezorcrypto_monero_BP_PLUS_HI_PRE_obj = {{&mp_type_bytes}, 0, 8192, (const byte*)""
"\x48\x62\x8d\xf3\x80\xa5\x01\x6d\x25\x45\x1a\xaa\x50\x17\x31\xa1"
"\x1b\x72\xbf\x66\xdc\x41\xd8\x1f\x71\x9a\xbd\x35\xce\x92\xb0\xed"
"\x11\x0d\x2b\x61\xf8\xc7\xc1\x08\x61\xc3\xe4\xff\xe7\x77\x4f\xab"
"\xa6\x32\xaf\x94\x85\x4a\xa2\x95\x38\x51\x7a\xef\xe6\xa3\x9e\x48"
"\x3c\x5c\x3f\x66\x25\xc4\x0d\xc3\x76\x32\x19\x70\xb2\xef\x1f\x96"
"\x4c\x39\xaf\x0d\x7a\xae\xd8\xa2\xd6\xd7\xb6\x6c\x6c\xf4\xf9\xb1"
"\x01\xe3\xab\xd2\xe0\x87\x97\x02\x2d\x8a\x0c\xa7\xe8\x01\x42\xa0"
"\xb6\x2a\x25\x09\x9b\x15\xf4\xc6\xad\x30\xd2\xb4\xf3\xf2\xa1\x90"
"\x90\x9d\xf9\xbc\x55\xa0\x5c\x84\x3d\x22\x76\x14\xf4\x26\x36\x25"
"\xf9\x19\x8d\x23\x98\xcd\xe8\xd1\x1b\x90\x33\x99\x5a\xbf\x7c\xa3"
"\xf6\xf6\xcd\x40\xbb\x85\xf1\xb8\x30\x8a\x3a\xd4\xfc\x7b\x3f\x0c"
"\xce\x62\x10\xe3\xaa\xc6\x1d\x3c\xcc\x83\xa0\xe5\x18\x14\x18\x71"
"\x85\x9d\xfb\xde\x93\xdf\x8d\x63\xc7\x7b\x4b\xa5\xc8\xde\x66\x05"
"\xa7\xd6\xfc\xd8\x35\x01\xc8\xa7\x38\xd5\xfe\xd5\x78\x9e\x9a\xcf"
"\xbd\x89\xc5\x9d\xfd\xd1\xea\xae\xa8\xf4\x00\x11\x7c\xf4\xaf\x42"
"\x01\xd2\xe1\x24\x85\x84\x62\xd8\xf3\xb0\x9a\xd4\x61\x93\x8e\x21"
"\x0b\xee\xdd\xd9\x99\x50\x51\xac\xf8\x75\xfc\x7b\x9d\x2b\xeb\x0f"
"\xd1\xab\xa7\xef\xa2\xa3\xbe\x2a\xd9\x2a\x2f\xb1\x2f\xf6\x8a\xdc"
"\x31\x17\x85\x79\xce\xc1\x67\x65\x01\xad\xe7\x80\xf0\x03\xf5\xbb"
"\x44\xe1\x3c\x78\xdf\xd8\xd1\x4c\xed\x75\xca\xea\x2a\x8d\x0d\xb3"
"\xa2\x0d\x29\x9b\x17\x18\x31\xba\x03\x75\xac\xa1\x63\x86\xa6\xbd"
"\x6e\x67\xfe\x89\x5d\x6e\x70\x03\xe6\x9e\xeb\xe3\x62\xd1\x80\xbc"
"\xd4\x67\x8b\x83\x14\x22\x4b\x6a\x2b\xa1\x56\xf0\x07\x44\x8c\xf1"
"\x73\x0f\xa1\x0c\x36\xd8\x0c\xcb\x79\x57\x83\xe8\x57\x7a\xba\x22"
"\x45\x97\x68\x96\x03\xc3\x7e\x71\x5a\x6d\xee\xee\x5f\xaf\x50\xb6"
"\xb6\x3c\x73\x08\x5c\xf9\xe9\x42\x3c\xdd\x06\x77\xdf\xb1\x34\x0e"
"\xb7\x79\xf2\x85\x7a\xa1\x0b\x49\x3a\x45\x1c\x92\xd8\x00\xa8\x3f"
"\x19\xa2\xce\x43\x04\xef\x8a\x01\x01\x47\x6a\x95\x69\x5a\x63\x3c"
"\x54\x24\x77\xfb\xa2\xef\xe1\xc1\xbd\x32\xbb\x12\x07\xab\x84\xca"
"\x97\x0c\x6b\x66\x20\x34\x2f\xc1\xd0\x79\x25\x92\x0f\x31\x3b\x31"
"\x15\xf0\x0f\x0f\xd2\x43\xb3\x38\xde\x6c\xdc\x70\xa0\xdd\x9c\x6e"
"\x50\x5e\xda\x84\xef\xf7\x55\x2d\x22\xcd\xcf\x46\xf1\x05\x91\xb5"
"\xaf\x35\x32\x55\xe8\xd9\x6b\x1e\xb7\x75\xd9\xfd\xda\x79\x59\x61"
"\xe2\x1b\xeb\xd4\x8e\xff\xe1\x7a\xb3\x2b\x63\xd9\x5b\xab\xd0\xa7"
"\x09\xb4\xfa\x9d\xdc\xee\x1b\x48\x1b\xa8\xbf\x0e\x8d\x73\x68\x2a"
"\xf7\x61\x0a\x62\xd3\xb0\xcc\x5e\x41\x38\x33\xba\xcb\xe0\xc6\x70"
"\x29\xa3\xe2\xc6\xb5\x17\x9f\x9c\x03\xaf\xf9\xf3\x24\x94\x5c\x1b"
"\x77\xb4\xbe\x11\x54\xb8\xa4\x90\x05\x73\xa7\xcf\xa9\x5f\x3d\x05"
"\x8e\x16\x92\x53\xf9\xb0\x04\x28\xe8\xe2\xd0\xbb\x43\xa2\x44\x74"
"\xcb\x4f\xe1\x2e\xfe\x42\x82\xe0\x88\x6b\xe2\x19\x63\x15\x36\x16"
"\xc0\xbd\x85\x43\x15\xec\x57\x0a\xa8\x76\x90\xd4\x46\xa4\x66\x46"
"\x79\x42\x04\x67\x86\x12\x3b\x08\xd3\x84\x6f\xf8\x05\x51\xa9\x4c"
"\x9c\xca\xff\x3e\xb3\x1f\x45\xf1\x24\x7c\x10\xab\xa3\xc4\xc5\xb4"
"\x43\x41\x21\x71\xdd\x95\x94\x91\xb5\xa3\xa2\x05\xbc\x29\x65\xa9"
"\x34\x60\x00\x31\x34\x42\x51\x44\xc7\x63\xe0\x3a\x0d\xe2\xa4\xfa"
"\x8c\xb4\x7a\xf5\xd7\xa9\x67\xcc\x4d\x8a\x1e\x6c\x7b\x0e\x8d\x5b"
"\xf8\x25\x91\x63\x12\xb7\x03\xf3\x90\xa6\x92\xde\x3f\x65\x94\xe1"
"\x50\xde\xa3\xae\x9e\xf6\xa5\x58\x9b\x55\x9f\x2a\x65\xae\x9a\xdc"
"\x35\xb9\x14\xf1\xec\x64\x29\x20\xdf\x51\x30\x76\x38\xf4\x73\x93"
"\x31\x5b\xaf\xeb\xc3\xa4\xe3\x4c\xd3\x5c\xe2\xa5\x41\x98\xcd\x55"
"\x34\x39\x72\x4d\x05\x99\x60\x2e\x36\xa5\x93\x54\x66\x35\xa2\x7a"
"\x23\xc6\x3b\x31\x8b\x8f\x75\x29\x53\xb5\xf4\xa9\x32\x1c\xf6\x73"
"\x31\x10\xf7\xe2\x2b\xd7\xd6\xd4\x7d\xea\xf7\xe0\x53\xc4\x18\x34"
"\x86\xa0\xcc\x96\xff\x9c\x56\x67\x05\x3c\xe4\x96\x36\x7c\xff\x9e"
"\xfa\x4c\x0c\x6d\x62\x4a\x5f\xd4\xe9\x48\xe7\x5e\x83\x8a\x53\x46"
"\xc1\x21\xab\x8a\x4a\x1c\x19\x0b\x95\x5e\x74\x00\x97\xda\x32\x82"
"\xd8\xf5\xb7\x33\xd4\x1e\x5a\xd6\xc4\xce\xfc\xa1\xe8\xdf\xed\x54"
"\xe9\xbf\x2d\x88\xdf\x36\x74\x49\x42\x32\x36\xfb\x9c\xee\xd1\xef"
"\xff\x90\xbb\x0b\xb7\x0e\xc9\xc3\x32\x82\x8a\x15\xdc\xa0\x3f\x89"
"\x52\x92\xe1\x95\x73\xef\x33\x4f\xa2\xb7\x71\x17\xad\xbe\x96\x6c"
"\xf0\x81\x31\x6b\xea\xb8\xf0\xf0\x84\xd0\xe8\xc8\xad\xef\x9d\xff"
"\x7c\xbd\x52\x86\x9b\xb6\x6a\x90\x14\x6c\x4c\x73\x38\xb5\x0d\x77"
"\x2c\x78\xa0\x6c\xe2\xcc\x68\x65\x47\x24\x4e\x96\xfc\x8c\x01\x2c"
"\xff\xaf\x96\xb3\x2c\x34\xdf\x8a\xb1\x99\x1f\x18\xd6\xfc\x36\xd1"
"\x6c\x8f\x1e\x53\xaf\x4d\x56\xc2\x0b\xca\x34\x92\xf0\xcd\xd0\x23"
"\x29\xfe\xeb\x70\x7e\x77\xa8\x40\x3c\xa1\x39\xda\x3d\x74\xb5\x77"
"\xe7\x6b\x8e\x0f\x24\x54\xb7\x59\x48\x4f\x80\x13\x15\x12\x0c\xcb"
"\xe1\x36\xc5\xb3\x7e\x84\x07\x5e\xbb\xe3\xf8\xce\x2c\x66\xca\x7a"
"\x03\xd3\xe4\xc6\x4e\x22\x1e\xbd\x3d\xc9\xaf\x19\xe5\xb1\x40\xdb"
"\x73\xe1\x1d\x7e\xdb\xbd\x5f\x5d\xf3\x9a\x1c\x75\xdc\xfc\x3d\x40"
"\x2f\xe7\xbc\x61\x92\xe4\x75\xf3\xb6\x59\xdf\xa5\xcf\xba\xe5\x1c"
"\x7e\x61\x4f\x28\xcd\x62\x2a\x58\x68\x8a\x15\xa6\xcd\xe3\xbe\xda"
"\x11\x0d\x69\x46\x53\x61\x0f\x36\x5c\x7e\xe0\xe4\x35\x95\x74\x2d"
"\xf5\x41\x30\x70\x69\xf4\xd5\x52\xec\x3a\xbb\xc7\x80\xc1\xcd\x85"
"\x49\xbd\x8e\xd2\x0f\xf9\x7c\xa8\xcb\x1b\x96\x49\x89\x92\xf1\x05"
"\x21\x1a\x13\x1e\x43\x9e\xae\xe9\xde\xa8\xca\x1b\xd9\x43\x66\x1b"
"\x09\x31\xb9\xa7\x41\x57\xc1\x9b\x72\xed\x83\xfb\xba\x59\x04\x62"
"\x3e\xc2\x8e\x9b\x6a\x79\x1d\x69\xe3\x11\xcd\xd6\x6b\x01\x15\xf6"
"\xc2\x0b\xd0\x24\xdd\x0c\xfd\x79\x64\xcf\x04\xdf\x95\xe0\x53\x06"
"\xfe\x8f\x7e\xf7\x96\x22\xb9\xe6\x93\x5f\x35\x8a\x14\x2e\xd7\xa9"
"\xc5\x5e\x13\xcd\x6e\xc9\x37\x3e\x12\x6b\xc8\x22\xc3\xc3\xcd\x27"
"\x23\xc8\x3c\x21\xe9\x15\x61\x0b\x71\x10\xf0\x1b\x4f\x87\x4c\x2e"
"\x73\x03\x28\xe5\xbd\xd4\x48\xed\xea\x46\xd4\x42\x9b\x3e\x16\x81"
"\xc3\x43\x0d\xa5\x03\x06\x56\xbf\xe9\x37\xc3\xa0\xbf\x48\xcb\xf2"
"\xd1\xec\x6b\x4d\xed\x5e\xd3\x1c\x44\x98\x7e\x6c\x6e\x1b\x8c\x67"
"\xd9\xe5\xdb\xae\xcc\xaa\xe5\x0d\x30\xb0\xfa\x02\x3f\xa9\x43\x86"
"\xe6\x34\x30\xd3\xbd\x16\x88\xa8\xfb\x93\x1d\xfc\x54\xb3\x5c\x56"
"\x7c\xe4\xfa\xef\xd0\x32\xcd\x90\xbe\x61\xa7\xe3\x64\xd9\x68\xda"
"\xd5\x6c\x91\x03\xb4\xf2\x18\x38\x77\x92\x67\xd4\x6b\x38\x30\xf2"
"\xa9\x0a\xd4\x35\xb8\xfb\x53\x64\x0d\x75\x67\x94\x1e\x6e\xc3\x19"
"\x4a\x99\x46\x4d\xc5\x82\xa3\x64\x66\x21\xd4\x0f\x55\xcd\x0d\xd9"
"\xde\x9f\x00\xcc\x82\x33\x56\x34\x0c\x6f\x0c\x25\xdc\xb6\x91\x41"
"\xc0\x03\x98\xc8\xf6\xf5\xfc\x83\xcf\x58\x89\xa8\xe6\xd0\xef\xd0"
"\x92\x11\x26\xf5\xb1\x4e\x13\xe3\x80\x75\x44\xa4\x79\x4f\x8a\xc7"
"\x9b\x07\x30\x05\xdb\xfe\xc3\xd8\x50\xd2\x94\xe9\xc2\xb8\xc8\x60"
"\xe4\x3f\xee\x90\x18\x94\xcb\xeb\xf5\x3c\xc6\x17\x17\xfd\xbd\xfa"
"\xf4\xfd\xa8\x86\x1f\x16\x84\xc4\x6e\xbb\x64\xb0\xa2\x4e\xf4\x00"
"\xad\xf1\xdc\x8b\xdb\xae\xfe\x8f\x15\x62\x4e\x92\x0b\xf3\x40\xef"
"\x40\x22\xa5\xfe\x1a\x70\x08\xab\xed\xd9\x86\xf2\xf0\x91\x8a\x2a"
"\xcd\x8b\xe1\xc8\x2d\xda\x8c\xce\x67\x54\x3d\xc8\xef\xe2\xa6\x7a"
"\x6d\x7c\x39\x03\x19\xf1\x1e\xe4\xd8\x3e\x62\xef\xe6\x0f\x9b\x16"
"\x28\x7b\x15\x1f\xbe\x76\xe9\xc2\x83\x1d\x6f\xbd\x10\xb5\xc2\x48"
"\xa9\x28\x2f\x0a\xef\x7d\x8c\x56\x98\x73\x71\xe1\xa4\x9d\x17\x2c"
"\x4f\x08\xad\x35\x8a\x5d\x7e\xdb\xcc\x39\x1b\xf2\xe1\xa2\x5c\xcc"
"\x51\x0a\x20\x92\xf2\x6f\xe3\x9e\x32\x8e\x21\x2f\x63\xae\x54\xfa"
"\x3e\xa3\x71\x38\x1d\x75\x34\xbc\x40\x52\xb9\x9a\x1c\xd7\xf1\x6a"
"\x93\x0f\x12\xfa\x3a\x15\x27\xed\x96\x05\x49\x88\x96\xc2\x0f\x4b"
"\x8b\xec\xaf\xae\x2a\x42\x7a\x04\xb8\xf1\x38\x3e\x11\x01\xc4\xe1"
"\x60\x33\x66\xa6\xa9\x83\xcf\x8c\xd2\x4c\xb5\x45\x79\xe6\x05\x14"
"\x91\xcb\x4d\x29\x8b\xde\x46\x4e\xbc\xa2\x4f\x8c\x2e\x87\x3c\xaa"
"\x08\xb1\xbb\x94\xb2\x8b\xc7\x04\x4a\xd3\xbe\x2d\xb5\x7c\x40\x65"
"\x0a\xa5\x1f\xda\xee\xe8\x17\x3b\xbd\x87\x3e\xae\x55\x62\x84\xd8"
"\x80\x76\x46\x59\xad\xff\x68\x9d\x98\x31\xcf\xb4\xc2\xeb\x96\x49"
"\xca\x00\xff\x13\x05\x52\x84\xca\x60\xed\xd8\xb7\x85\xc5\x2b\x36"
"\xd9\x8c\x5d\x14\xb3\xd0\xc7\x57\x67\xf9\x28\x54\x6d\x70\x21\x5c"
"\xd4\x9e\xcc\xd6\x7f\x60\xda\x16\x5f\xfa\x0c\xb2\x5d\x72\x46\xcc"
"\x2c\xae\xc0\x98\x97\xe3\xad\x21\xd2\xaf\xcf\x24\x0d\x7d\x9b\x64"
"\xa6\x39\x84\x14\xa3\xc3\x17\x3e\x98\xb1\x84\x19\x72\x28\xf3\xa4"
"\x92\x2a\x06\x21\xdd\x7f\x09\x47\xcc\xcd\x12\xa4\xfa\x71\xdf\x8d"
"\x82\x14\x64\xb8\xf8\x05\x99\x3e\xce\xb6\x2a\x10\xfb\x8c\x2d\x1b"
"\x3b\xd6\xc8\x56\xa2\xdb\x97\xe9\x5f\xa9\xe5\xb0\x83\x7d\x5c\x83"
"\xe0\x75\xfd\x02\xee\xfd\x44\x48\x0d\x1b\x68\x48\x42\xa9\xd3\x93"
"\x13\x16\x87\x0d\x08\xeb\x6a\xcb\x17\xd5\x02\x3f\x0d\x92\xe0\xc3"
"\x84\xe2\xc6\xd4\xd9\x8a\xf1\x84\x2f\x36\xec\xda\x99\x3d\x77\x8a"
"\xfa\x2d\x51\x92\xea\x29\x55\x5b\xb3\x53\x9b\xc0\x73\xe4\x21\x6a"
"\x82\x9c\x20\xc8\x87\x3b\x4e\x33\x7a\x81\xd0\x7d\xa7\xd4\xd3\x87"
"\xe5\x2e\x92\x32\xa8\xbe\x13\x26\xee\x8a\x0f\xc4\x2f\xce\xe9\x28"
"\x7f\xbf\x1c\x70\x22\xfb\x8f\x33\x56\xe3\xd9\xb5\xf9\x6c\x06\xd5"
"\x27\xe9\x47\xe7\xf5\xa9\x75\xed\xb6\xf3\x5f\x63\xc1\x9d\xf3\xa8"
"\xa3\x4d\x9c\x81\x12\xfa\x9f\x0f\xfe\x20\x10\x9b\x8e\xb4\x25\x85"
"\x09\x0a\x79\x5f\x1d\x61\xb0\x9e\xee\xc0\x26\xbe\x0c\x09\xba\x9b"
"\xe0\x5b\x4e\xd3\x2e\x82\xa6\x8e\x72\x8c\x71\x8c\x98\xb0\xb8\x4d"
"\x6a\x2c\xc3\xac\xe3\xdd\x8e\x3e\x41\x49\xe2\x1e\x52\x9f\x25\x67"
"\xfd\xc0\x9e\xc6\x4a\x6c\xb8\xb5\xb0\x28\x30\xea\x2d\xc7\x4b\x2b"
"\x74\x78\xca\xa9\x70\x94\x1a\xec\xc7\xcb\x9c\xda\x9f\x54\xd9\x8a"
"\x4c\x23\xbe\x9e\x98\x9b\x00\x4f\xb8\x8a\x1e\xa4\x2c\x54\x2a\xac"
"\x47\x44\xf4\x72\x74\xf2\xe8\x2a\x98\xeb\xd9\x5d\x02\x71\xe4\x77"
"\xd9\xde\xb6\xa8\x97\x6b\x9d\x61\x2e\x3f\x15\x6d\xb9\x18\x04\xcf"
"\xff\xe7\xd4\xe0\x41\x21\x26\xfd\x2d\x2f\x19\x79\xf1\x69\xc1\xf6"
"\x92\xcc\x0b\x00\xfd\x44\x85\x0b\x55\x8e\x61\xe0\xa4\xa5\x54\xe7"
"\x77\x50\x37\x2f\xff\x40\x6b\xd7\x30\x87\x5a\x03\x6c\xcf\xa6\x1e"
"\xdd\xd5\xaf\xc3\x55\x69\x5e\x9c\x8e\x1a\x08\x59\xce\x6f\xa7\x99"
"\xbe\xc8\x08\x19\xa7\xe6\x01\x5e\x7d\xb1\x1d\x29\xfe\x7d\x06\xe2"
"\x8e\x23\xb1\x46\x98\x46\x93\xd4\x34\x31\xf7\x0f\x7d\xaa\x69\xfe"
"\x46\xa8\x97\xf0\x93\xf6\x23\x5f\x51\xc1\x05\x18\x38\xe7\x87\x70"
"\xb8\x5d\xd6\x31\xa8\x78\x04\xf4\xd4\x8b\xf6\x20\xa4\x3b\x4e\xd8"
"\x05\x35\x15\x82\x65\xc9\x47\x9b\x99\x69\xad\x2c\x9a\x93\x6e\xa5"
"\x81\x65\x5a\x02\x3a\x14\xdb\xb3\x16\x28\x47\xb9\x37\x89\xbe\xf3"
"\x6f\x79\xee\x58\x74\xbd\x2c\x1c\xc6\x2c\xa4\x0d\x2e\x1d\xe0\x10"
"\x31\x5d\xbc\xab\xd8\x18\xb1\xbf\x7a\x1b\xbd\x19\x00\xd8\x64\x7b"
"\x79\xf6\x78\x5f\x21\xfe\x20\x00\x3d\x5c\xc8\xe6\x9c\x72\x6a\x88"
"\x4b\xeb\xc1\xc5\x5f\xc7\xf1\x72\xe6\x3f\xd3\xe6\xbf\xba\x49\xcb"
"\x89\x4a\xd8\xed\xb8\xf0\x9b\xd7\x26\x20\xab\xe4\xa8\xd3\xf8\x2b"
"\x06\xd5\x32\x42\xd2\x63\xd2\xd9\xf9\x01\xee\x5a\x13\xeb\x21\x95"
"\xdb\x2b\xc9\xde\xb7\x5d\x36\x79\xda\xe1\x36\xef\xfc\x0d\xbb\x25"
"\xd4\xea\x55\xd5\x45\xed\x47\x10\xdd\x2e\x94\x94\x24\xf1\x8f\x75"
"\x55\x0b\xdf\x91\xc9\x59\x2f\xe7\xbc\x00\x9c\xeb\x19\xe9\x3d\x97"
"\x0b\xdc\xac\x36\x94\xcd\x4e\x5b\x7e\x97\x51\x4c\x40\x5d\xb9\x74"
"\x21\xfb\xa5\xf6\x6d\xb1\x81\x16\x80\x0e\xf4\xe5\xbc\x5a\xce\xb0"
"\x49\xe9\xab\x43\x85\x5e\x34\xaf\xb3\xb6\x97\xca\x1b\x33\xcc\x05"
"\x8e\x97\x4e\xb9\x14\x85\x75\xda\x04\x74\x6f\xcf\xe3\xdc\x09\x99"
"\x88\x19\x90\xb8\x21\xd5\x73\x27\xb3\x80\x63\x5d\x8f\xf1\xf3\xad"
"\x42\x87\x3f\xf1\xfa\xe7\x26\x44\x39\x88\x85\xe9\xfc\xc2\x63\x54"
"\xc8\xbe\x60\xfa\xd1\x80\x20\x98\xf9\x3d\xc9\xde\xb1\xee\x80\xe9"
"\x85\xac\xcf\x39\x40\x3d\x4d\xf1\xb2\x1b\x24\xd1\x57\x55\x8d\xf1"
"\x40\xc6\x99\x4b\x52\x27\x6f\x64\x38\x1b\x29\xab\x86\x63\x1a\x7d"
"\xcf\x43\xf7\x9d\xc1\x63\x83\xd7\x0f\xb9\x2d\x52\x04\x4d\x30\x84"
"\xdb\x1d\x5e\x61\x24\x9b\xbf\x2f\x7e\xa5\xd9\x81\xcb\xab\x8d\xbe"
"\x79\xf2\x48\x89\x41\x27\x78\x6e\x31\xc2\x53\xed\x6c\x57\x23\xc6"
"\x3d\x15\x59\xbf\xae\x92\x2c\xa7\xc5\x00\xee\xb5\xfb\x98\xa0\x94"
"\xad\xa0\xce\xb9\x9e\x87\x01\x61\x4f\x27\xf0\xbf\xa0\x84\x0e\x36"
"\x75\xca\xcc\xdd\xd2\x62\x1d\xab\x84\x50\xdb\x34\xb5\x5a\xdc\xbe"
"\x5d\x60\x8a\x24\xf2\x3d\xd4\xfe\xc5\xdc\xe6\xd2\xbf\x99\x1c\x39"
"\x1e\x33\x6c\xb3\x01\x3e\xd4\xca\x73\x7c\xea\x8d\x16\x5c\x14\x5a"
"\xa4\xa6\x10\xa6\xb6\x84\x45\x91\xb6\x03\x5d\xf2\xb4\x7d\x3a\x56"
"\xf7\x8b\x7b\x8c\x0a\xbc\xbf\x29\x3f\xa9\x83\x50\x1a\xa1\x7d\x53"
"\x56\x11\xe9\x2a\xeb\x31\x72\x0c\xed\x92\x81\xe2\xd6\xf8\x29\xbd"
"\x96\xf1\x4d\x6e\x0f\x9a\xdd\x0e\x01\x8f\x4b\x07\xf7\x60\x58\xd4"
"\xf3\xac\x80\xd6\x8e\x95\x66\x44\xa8\x4a\xbb\x35\x72\x38\x8b\xb2"
"\xef\x9a\x37\x87\x56\xd4\x92\x31\x7b\xa2\xf6\x1d\xdb\x6f\x2d\xbf"
"\x0d\x38\x78\x9d\xe4\x99\x2e\x31\x83\x16\x96\x16\x54\x11\x6b\x99"
"\x52\xec\x74\x24\x39\x93\xbf\xeb\x8b\xc6\xfb\x31\x3c\xd8\xcd\x4d"
"\x93\x48\x98\x5d\x4d\x90\x70\x36\x18\xbb\x6e\x8a\x7e\x41\xad\xfd"
"\x80\x37\xa6\x36\xc2\x13\x30\x7f\x18\x66\x7a\xc2\x77\xd5\x4d\xad"
"\xce\x38\x75\xfa\x59\x96\xe6\x08\x80\xef\xb7\xda\x17\xd1\x5a\x23"
"\xd3\x81\x0f\xab\x62\xcf\x63\x86\x67\x17\x3d\x3c\x0a\xa3\x0c\xc7"
"\xf7\x65\xfd\x92\xda\x06\x94\xb9\x6e\x63\x49\xe3\xa6\x4b\xeb\x8d"
"\x2c\xb3\x80\x59\xab\x54\x08\x83\x51\xa7\x58\xbb\xe6\x58\x4f\xfe"
"\x19\x72\x98\xe9\xdf\x1b\x63\x67\x83\x7c\x42\xd8\xcf\xf7\xf4\x56"
"\xd7\xaa\xc3\xa6\xf9\xf6\xb2\x9f\x63\x45\xf7\xdb\x23\xbc\x9c\x28"
"\x45\x86\x2d\x98\x41\x31\x3d\xc3\xff\xf8\x07\x61\x81\x52\x27\xa7"
"\x12\x14\xf8\xcb\xea\xcd\xe8\x67\x39\x14\x14\x3b\xcb\x05\xd5\x5a"
"\x52\x3e\x15\x28\x5f\x32\x83\x23\x50\x7f\x97\x0f\x6b\x54\xbf\x17"
"\xfc\x71\x01\x4f\x6f\xe4\x6f\xd0\x9a\x40\xa6\x8e\xff\xf6\xb9\xc0"
"\x81\xa1\xf5\x40\x32\xbd\x9a\xd8\x69\x33\x93\x5a\x36\x2a\xda\x5f"
"\x16\xe8\x89\x58\x9b\x27\x59\xca\x60\xf8\xf5\xfc\xa0\x9c\xe8\xc7"
"\xfd\xbf\x24\xb8\x76\x45\xfb\x6f\xd3\xd7\xd4\x1c\x0a\xb8\xf9\x6b"
"\x81\x8f\x18\x36\x53\xa0\xd4\x13\x7d\xc4\x82\x3b\x14\xe8\xde\xf7"
"\xd2\x94\x2c\x21\xd0\x28\xe0\x87\x35\xf6\xe0\x1f\x30\x4b\x69\x68"
"\xf4\xcb\x50\x49\xba\x04\xf7\xf0\x89\x01\x27\x3a\x58\x68\xb4\xe9"
"\x5e\x61\x68\xb6\xff\x99\xbf\x98\xd8\xa6\x4e\x52\x82\xa9\x51\x88"
"\x46\x36\x4d\xda\x39\x2f\x59\x4b\xe3\x6a\x3c\xee\x00\x5a\xd2\x15"
"\x49\x94\x32\x86\xe7\xca\x29\x96\x0d\x17\x36\xf0\x71\x64\xa0\x19"
"\x60\x29\xda\xd1\xbb\xa0\x80\x26\x51\x27\x70\x7b\x08\x7a\xea\xf1"
"\x07\x4c\x4f\xa4\x09\x09\x8e\x6a\x8f\x3b\xfc\xa9\xe4\x31\x1a\x9f"
"\x91\x21\x46\x9d\x21\xa9\xc9\xce\x9a\xaa\x33\xea\x2a\x60\x28\x14"
"\x3b\x30\xc8\xd6\xeb\x40\xb7\xfd\x02\xd1\x7c\x58\xa3\x16\x54\xa9"
"\x40\xa8\xd3\xab\x4a\xe6\x9c\x51\xc7\x40\x2d\xd7\xc5\xe1\xc3\xe6"
"\x87\x0c\x0f\x1d\x52\x9c\x3b\x47\x76\xba\xbf\x82\xcd\x75\x03\x3b"
"\xcb\x7a\x23\xf5\xfa\xaa\xeb\xe7\x33\x28\x5f\x29\x9b\xcc\x9c\x14"
"\x2d\x3a\x42\xe3\x83\xfe\x5f\xe8\xab\x68\x96\xd0\xf9\x33\x6e\x00"
"\xfb\xa2\x48\x92\x72\x5d\x25\xb1\x05\xb6\xe0\x82\xd9\x5f\xd0\x0f"
"\x54\xf8\x4c\x01\x87\xc1\xef\xee\x80\x90\xea\x8d\x49\xc4\xb0\xca"
"\x74\x63\xb4\xf1\xb9\xa0\x34\x03\xb4\xb9\xa4\xd1\x73\x53\x60\x96"
"\xd6\x29\x4d\x8e\x88\x72\xc1\x9b\xac\x9c\x97\x16\xf0\x44\x6f\xc5"
"\x80\x12\x9b\xdd\x43\xbe\x8a\x76\x12\x28\x6f\x15\x40\x6f\x8e\xda"
"\xf1\x21\x35\x5e\xc7\x85\x27\x21\x07\x7e\x80\xad\x16\x46\x29\x8b"
"\xc4\xc4\x8a\xbd\xa0\xab\x7c\x83\x52\xc0\xd5\x66\x4e\x0f\x20\x8d"
"\xe2\xa3\x5b\x70\x09\x35\xf5\x83\xfa\x90\x9b\x00\x64\x6b\x50\xd8"
"\xc9\xf5\x9b\x38\x79\xe7\x8d\xd8\xe4\x9e\xd8\x98\xb1\x08\xc4\xb8"
"\x27\x33\x7e\xbb\x0d\x34\x5a\x55\x58\xc9\x1b\xa7\x2a\x01\x30\xd5"
"\x83\x01\xb7\x2d\xd0\xc6\xdd\xae\xfc\x12\x50\xc1\x93\x52\xcd\xef"
"\x7b\x8c\x26\x72\xf6\xd6\x2a\xad\x9e\x25\x8e\x5f\x5e\x91\x39\xa6"
"\xaa\xc7\xc7\xab\x14\xa0\x8b\x87\x9d\x66\x51\xd8\x3e\xf2\x95\x32"
"\xb1\xaa\xd6\x94\x69\x10\xb7\x5e\xeb\xa0\x87\x85\xde\x16\x5a\xd9"
"\xd2\xc9\xbd\x2b\x9f\x69\x4b\xe6\xf2\xc0\xf8\xae\xf3\x62\x78\xa8"
"\xcf\x45\x16\xb3\x6b\xec\x14\x46\x2e\xd9\xeb\xbc\x3b\x0a\xbb\xcc"
"\x7c\xb0\xdf\x72\x62\x4d\xef\x46\xe6\x84\x64\x9e\x4a\xc9\x8b\x03"
"\x83\x6e\x3e\xba\x67\x6b\x96\x0c\xc1\x03\x3c\xf6\xc8\x20\x9a\x1d"
"\xc5\xf5\x13\xbf\xe7\x8e\x63\x25\x69\x76\x5b\x42\xaf\xf4\x9f\xd9"
"\x17\x0d\xd7\x51\x35\xcf\x90\xc8\xe2\x6e\xa6\xb1\x37\xcf\x33\x6a"
"\xc7\xb0\x03\x0c\xf2\x0c\x06\xeb\x92\x32\x30\x28\xda\xf3\x0c\x25"
"\x5e\x32\x58\x42\x3e\x92\x3e\xd7\x07\x70\xc5\x3a\xe1\x58\xc7\x8e"
"\x35\xc1\xee\x0e\xba\xce\x59\xf8\x69\x38\xce\xa4\x61\x30\x61\xe8"
"\xc0\x14\x7d\x30\x13\xae\xe0\x10\x8b\x62\xbd\x1b\xa1\xa8\x82\x7c"
"\xd6\x47\xeb\x3f\x8c\xb6\xbe\x2d\x13\x3d\xe6\x1c\x66\x28\xfb\x1d"
"\xa7\xf5\x0c\xb8\xd7\xd4\x6d\x49\xa3\xf9\x03\xbb\x3d\xfc\x71\x36"
"\x12\x4d\x97\xd8\xea\x42\xc1\x98\xd8\x8e\xe5\x43\x5d\xdb\xd1\x86"
"\x50\x7e\x22\xf8\x5e\x2e\x43\x8f\x1e\x96\x0a\x7e\xb8\x12\xc4\xd9"
"\x7f\x9b\x81\xc0\x75\xe5\x22\xbc\x35\xd8\x9c\xa4\xfd\x03\x2f\xc1"
"\x31\x0b\x89\x8e\x68\x2c\x28\xe4\x02\xb3\xfd\x71\xe8\x75\x38\xe4"
"\x3d\x64\x2f\xff\x04\x33\x3a\x52\x19\x75\x7b\x3d\xee\x0b\x83\xd9"
"\x4a\xf2\xf4\xb4\x16\x93\xea\x29\xf9\xdf\xa2\x44\x70\xe8\xef\xe4"
"\x46\xb7\xb0\x18\x84\x7f\x88\x17\x75\xb7\xc0\x23\x1b\x1c\xf4\x5f"
"\x11\x70\xcf\xa2\x0f\x4c\xf4\xc1\xef\x38\x5d\xb3\x3e\xfb\x50\x48"
"\x24\x8f\x79\xfe\xde\x71\x56\x61\x1c\xde\xe0\x38\x49\x8b\xdf\xae"
"\xa0\xc7\x51\xad\xc0\xdf\xd1\x38\xfa\xe1\x27\x2d\x13\x72\x7f\x28"
"\xdd\x7a\x65\xe4\xeb\x6b\x0a\x70\x58\x10\xc0\x20\x6f\x7c\x57\x0f"
"\x55\x2f\x3d\xcf\x4b\x3e\xac\x30\xc9\x57\x1f\x69\xca\x68\x28\x2e"
"\x78\x68\x59\xfb\x50\x78\x76\x6f\xcd\x47\xaa\x38\x9f\x05\x82\xf1"
"\x77\x24\x4e\xaf\x12\x4b\x63\x36\x63\x55\x29\xf1\x94\x68\xf1\x50"
"\x53\xdd\xdc\x6c\xbf\xe2\x3b\x23\x05\xc8\xfb\xb7\xc6\x47\x3a\x56"
"\x59\x59\x43\x0a\x78\x4c\x9c\xfd\xfe\xb8\xcb\x8a\x16\xc2\xd5\xeb"
"\x3e\xe5\x83\xf6\x02\x2c\x35\xb8\x6b\xa5\x76\x11\x3d\xa8\x34\x25"
"\x2f\x21\x73\x59\x62\xc1\x11\x68\xc8\xc8\x3d\xfd\xaa\x12\x20\x91"
"\x59\xd2\x84\x57\x8a\x46\x03\xf8\xb9\x4d\x5e\xc7\xf4\x0b\x8e\x66"
"\x8b\x30\xd3\xfb\x2b\x59\xb7\x50\xf0\xfe\x24\xa3\x69\xcb\x2e\x2c"
"\x18\x44\x09\x3d\x81\x77\x58\x2b\xf9\xfd\x57\x96\x97\xe4\x29\x9d"
"\x80\x09\x16\x05\x19\x7f\x2c\x5e\x66\x94\x3b\x09\xf3\x77\xf8\x9c"
"\x21\x1d\x91\x54\x76\xf3\xbf\x9f\xc3\x4f\x59\x62\x26\x36\xe0\x0f"
"\xc5\xbc\x74\xba\xf5\xe2\x20\x4c\xa3\x5a\xf6\xde\x23\x9e\x0b\xde"
"\x91\x61\xe8\x74\x12\x04\x3c\xdb\x18\x99\xd2\xb1\x80\xe4\x65\xe0"
"\x4c\x18\xd9\x7c\xb0\xae\x15\x4a\x31\xc1\x91\xcc\x38\xab\xe5\x7e"
"\x71\x9c\xea\x4b\x00\xee\xe1\x56\x98\x2d\x95\x51\x0e\xe6\x10\x52"
"\xfb\x2f\xe6\xbe\x1b\xce\x31\x7c\xd9\xe9\xed\xcf\xa2\x16\x0f\x68"
"\x56\xcc\x26\xb7\x47\x07\x8f\x30\xd8\xf5\xe7\x9f\x5d\x61\xed\x4a"
"\x3b\x35\x06\xf4\x06\xf8\xd8\x10\x99\xf4\x16\xbe\x3c\x61\x85\x91"
"\x14\x0f\x2e\xb9\x82\xd8\xfa\xca\xbb\xcb\xa6\x74\x8a\x1c\xf2\xc3"
"\xae\xd9\xc4\xa1\xda\x72\x59\x30\x62\x5c\xf9\x4d\x74\xc7\xe0\xf7"
"\x7a\x4c\x53\x28\x52\xed\xbb\xf7\xd3\x91\x02\xd6\x61\x14\xf2\x66"
"\x5c\xd4\x87\xfc\x09\x86\x86\x2b\x3a\x2a\x93\x42\x22\xb8\xa6\xbf"
"\xe5\x94\x7c\xac\x6a\xd7\xf4\x27\xc6\xf4\xa1\xf0\xe4\x27\xb9\x1e"
"\xd3\x70\x63\x04\x01\x48\x10\x5a\x81\x36\xb8\xb1\xcf\xeb\x82\xe9"
"\x15\xf4\xe2\xe5\xd9\xec\xee\x4f\x91\xac\xf1\x56\xcc\x23\x88\x2d"
"\xde\xa0\x99\xf0\x8f\x3a\x06\x95\xc6\xa8\x3c\xae\x31\x94\x73\x88"
"\xc5\x4f\x3a\x34\x6f\x5c\x51\x3e\x02\x41\x37\xbe\x62\xf0\xe2\x2a"
"\x36\x2c\x5c\x37\xd2\x2f\xe0\x3e\x6d\x16\x05\xac\xec\xe7\x23\xec"
"\xa0\x24\x2c\xe4\xc0\x8a\x11\xdc\x99\x0d\x4f\x22\x14\x39\x41\x3c"
"\x33\x28\x09\x03\x20\x88\xdb\xaf\x4c\x71\xe4\xbb\x60\xec\x4a\xcd"
"\xe5\x88\x39\x24\x35\x61\xa5\x3b\x24\xb0\x87\xd2\x55\x86\x93\x0e"
"\xa2\x6f\xb9\x00\x9b\x20\xb7\xca\x00\x00\x29\x82\xbd\x44\x64\x97"
"\xc3\xfd\x52\x01\x20\xc3\x94\x47\xd9\x2d\xc0\x4d\xbd\x69\x30\x4d"
"\xb6\x5d\x0c\xb8\x23\x17\x21\x4a\x2d\x21\x96\xe2\xa8\xc3\x98\x58"
"\x01\x2b\x8f\x9e\xe1\x09\x06\xa7\x7d\x87\x33\x5b\xae\xb7\x26\xe6"
"\x26\xd6\xff\x7d\xc6\x6b\x02\x2d\x1c\x9a\x92\x56\x91\x8e\xfc\xcd"
"\x5f\x8a\xc9\xdf\xbe\x28\xdc\x9c\xfb\x3c\xd7\x59\x2e\x5b\xc4\x5c"
"\xfe\x6d\xc5\xde\x9a\x40\x92\x31\xe0\x31\xe1\x0e\xec\xca\xec\xde"
"\x4d\x43\xae\xb2\xd3\xdd\x5f\xa1\xf8\x03\x65\x50\x38\x04\x92\x42"
"\xc3\x9a\xb7\x45\x94\xe6\x31\xb2\x89\x1e\x76\x2f\xdf\xf6\x7e\x6f"
"\x87\xde\x92\x53\x15\x73\x33\x18\x98\x24\x56\xf7\x80\xd5\x87\xfc"
"\x35\x4d\x76\x86\xa8\x7e\x27\xdf\xd7\x75\x05\xa4\xee\xad\x1b\x84"
"\xab\x84\x0c\x1f\x10\xf5\x5e\x67\x24\xfb\xa5\xce\x60\xe7\x5d\x65"
"\x05\xe0\x10\xac\x42\x45\xd9\x11\xcc\xfd\x0f\xd3\xd2\x0e\xa4\x3f"
"\x85\xe1\x9a\x8f\x01\x95\x1c\x32\x7b\x4a\x80\xb2\xb5\x93\x95\x9b"
"\xdc\x54\x0b\x94\xfd\xb6\x6a\xe7\xa9\xf7\xf3\x0b\x77\xfb\xcd\xfd"
"\x3d\x79\x39\xf1\x32\x9d\x47\xaa\xc4\x9b\x8b\x38\xaa\xe4\xaa\x2b"
"\xaa\x9a\x43\xff\x17\xff\x1d\xb4\xa8\x05\x0d\x92\x5f\x11\x89\x9d"
"\xea\xb5\x59\xa7\xf0\x17\x78\x6d\x78\x96\x9d\xf4\xfa\x1d\x81\x4e"
"\x05\x88\x79\x38\xe3\x2f\x4e\xfb\x1c\x75\x7f\x22\xf1\x22\xf6\x1e"
"\x48\x4a\x0a\x4e\x41\x63\x6f\xe1\x54\x87\x6b\x24\x82\x3c\xa7\x26"
"\x9e\xf9\xa8\x21\x8a\xfd\x6b\x53\xcf\x14\xf6\x60\x45\x20\xc2\x35"
"\x1a\xf9\xb9\xcd\x51\x9e\x94\xf1\x3a\xb0\x54\x56\x37\x69\xbb\x87"
"\xec\xfb\x81\x3e\x20\xf2\xef\xce\x24\x21\x79\xe6\xad\xac\x7f\x09"
"\xb6\x45\xff\xe0\x42\xeb\xb8\xa7\x27\xcd\xd5\x2b\xae\x8f\x51\xd7"
"\x14\xe8\x9b\x90\xe5\x4e\x7e\x79\x4a\x39\x0f\xf4\xee\x04\xe0\x3e"
"\x7a\x9e\xa4\x4b\xef\xf0\x36\xb5\x37\x09\x11\x69\xbd\x8c\xc8\xaf"
"\x77\x9e\x6f\x8b\x3d\x87\x8b\x94\x39\x85\xb6\xed\x16\x51\x99\x54"
"\x59\xc8\x81\xcf\x89\x55\x82\xc4\x4e\xbe\xcd\x98\xb6\x2a\x52\x1b"
"\x93\x18\x33\xfe\x92\x43\x63\x63\x82\xd7\x7c\x97\x98\x43\x12\x27"
"\xd4\xa5\x70\x53\x77\xec\xa8\x8d\xd0\x96\xf6\x4f\x1b\xa9\xb2\x7a"
"\x4f\xc0\x4f\x97\xff\xa3\x6a\xde\xf1\x2e\x52\xf9\xae\xc7\x1c\x48"
"\xd0\x17\xcc\x4b\x81\xee\xcc\xe5\xd9\x1c\x43\x6d\xff\x4a\xb8\x08"
"\xc9\x41\xed\xad\x2f\x68\x9c\xf1\xfa\xf6\x65\x70\xb8\xcf\xab\xee"
"\x8a\xe6\xde\x6c\x39\x8a\x1c\x83\xb1\x9a\xbb\x74\x63\x85\x76\xf7"
"\x83\xf8\x47\x7f\x0a\x79\xa3\x3e\x8b\x08\x22\x5f\xb4\xfe\x91\x51"
"\x45\x48\x5f\x1a\xa4\x05\x58\x6a\x1d\x8d\x88\xba\x20\x0e\xfd\x50"
"\x8c\xac\xb0\xf2\x09\xce\x59\x9a\x7d\xaf\x30\x71\xd5\x89\xc1\x8e"
"\x77\xcd\xce\xe6\x13\x8f\xe2\xd1\x0f\x66\x00\x39\x47\x40\xe1\x6e"
"\x30\x6f\x4b\x10\x5d\x0a\x05\x03\x9d\xa2\x95\x7f\x79\x5d\x63\xae"
"\x00\xcc\x7c\xd5\x40\xb0\x35\x02\x91\x34\x53\x95\xd2\x82\xa2\xe4"
"\x84\xf4\xb5\x4d\x63\x28\x36\xd0\x79\x38\xdf\x1e\xfc\x2f\xf2\x55"
"\x58\xb6\x82\xa8\xf1\x93\xa1\x8a\xa5\x8e\x8b\xfc\x27\x9e\x37\x31"
"\x8e\xcc\x0b\x3f\x22\x40\x86\x41\x5b\xfe\xbb\xf9\x0b\x87\x86\x07"
"\x4a\xfb\x46\x89\x58\xfa\x40\xfa\xea\x49\xa0\xdc\xeb\x63\x5f\xde"
"\x00\xd2\xc5\x33\xf1\x5a\xb4\xd2\xa3\xb2\x12\x18\x75\xff\x76\xdc"
"\x86\x14\x66\xe5\xda\xfc\x08\x73\x48\xb4\x71\x1a\x64\x71\xdb\x24"
"\x70\x6a\x01\xcd\xa1\xee\x8a\xa1\x02\xfd\x78\x66\x0e\x64\x62\x60"
"\xcb\x02\x9f\x1c\x39\xf6\x1d\x54\x2a\x71\x62\x79\x71\xc3\xc6\x39"
"\x7e\xf2\xa9\x95\x3d\x1f\xf6\xf1\xb9\x3d\x8d\x69\x50\x0d\x12\xfd"
"\x79\xd0\x7f\xb4\xe7\x67\x48\xa1\xcd\x4e\x66\x1d\xca\x80\x59\x74"
"\x92\x16\xea\x28\x1e\xfb\xee\x62\x9c\xc3\x43\xcf\xa7\xd4\x81\x33"
"\x1a\x88\xd4\xf5\x2b\xce\x47\xfc\x24\x4a\x01\x4d\x4f\x9a\xd9\x77"
"\x9b\x74\x89\x72\x10\x78\x0f\xaa\x1f\x2b\x23\xf7\xca\xaf\xfc\x13"
"\x0a\xfc\xc2\xdc\x12\xbe\xf4\x4e\xf9\xdd\x6d\x8c\xf0\xe9\x3c\xb1"
"\x65\x4a\xa0\x50\x29\x6d\x66\x83\x6b\xdd\x96\xb7\x38\x3e\x4e\x66"
"\x36\x31\x97\xf3\xcc\x0b\xb3\x77\x98\x95\x00\x26\xf0\xe4\x4c\xed"
"\xb6\xe7\x12\xb7\xb5\x99\x01\x5d\x0e\xe0\xf7\x92\x40\x04\xfd\x12"
"\x5b\x27\xc0\x6e\x62\xcb\x71\x94\xce\xf7\x32\x7e\xf3\x42\xa9\x17"
"\x0a\x85\x99\x92\x18\xbe\xfe\xa6\x37\xe7\xee\xba\x78\x82\xe7\x92"
"\x90\xe7\x31\x9b\xdd\xdb\x9a\xed\x48\x05\x43\x21\x9a\x4e\x13\xbc"
"\x88\xbd\xc1\x3b\x3d\x3c\x17\x1c\x6c\x82\x9d\x83\x67\x98\x57\x08"
"\x1c\x70\xd9\x0c\x97\xaa\x7b\x4a\xb2\xc3\x50\x64\x82\xc3\xde\x29"
"\x98\x42\xfd\x27\xd0\x17\x24\x2e\x14\xbf\x37\x81\x35\xc6\x57\x96"
"\x8c\xe8\x26\x9a\x5b\xed\x07\x6d\xc2\x8d\x07\x61\x20\x09\x97\x73"
"\xe6\xb4\xe3\xa7\x44\xff\x2a\x86\xf7\xd3\x56\x48\x4a\x3c\xee\x6b"
"\xa5\x34\xa1\x2c\xb7\xe1\xb3\x8e\xd9\xd1\xc3\xf3\x30\xdf\x4b\x35"
"\xfc\x7d\x02\x05\x7f\x81\x43\x6a\xf0\xa3\x40\x0b\x7b\xf4\x6e\xa3"
"\xd4\x14\xc2\x67\x87\x60\x56\x1d\xa7\xf3\x9b\x17\xa9\x63\x05\x14"
"\xd6\x53\x76\xd3\x08\x09\x7e\x55\xc4\x83\x25\x05\x3e\xfd\x9f\x46"
"\x0e\x79\xd8\xcf\x73\x0a\xaf\x8a\x7d\xb1\xd2\xd0\x89\x10\xf4\x83"
"\xa2\x18\x7b\xe7\xc7\x4f\x35\x08\x12\x7f\x06\xa6\xb0\xf9\xe0\x55"
"\x5d\xd6\x42\x46\x75\x0d\x0f\xef\x98\x1d\x85\x13\x8e\x27\x9c\xf4"
"\xac\x81\x77\x72\x68\x64\x6c\x80\x04\x0e\xf3\xf8\x36\x42\x2c\x63"
"\x9e\x65\x92\x89\xce\x11\xbf\x81\x69\xc1\x4c\x10\xdd\x47\x7c\xe7"
"\x2c\x14\xeb\xa8\x44\x75\xb2\x0b\xae\xb9\xa3\x45\xe2\x35\xe1\xf9"
"\xce\x65\xe3\x08\xba\xc6\x17\x0d\xc0\xe6\xb6\x9a\x9a\x8c\xd7\x60"
"\x69\x62\xf7\xfe\x2b\x20\x18\x58\x7b\x9e\x27\xa9\x34\xd9\x53\x09"
"\x2c\x31\x94\xae\x38\xf3\x5b\x90\x14\xc7\x22\x0f\xde\xa0\xd3\x5f"
"\xb5\x21\x75\xaa\x68\xa7\xd1\x52\x31\xf8\x69\x3a\x00\x01\xf1\x78"
"\xd2\x21\xc7\x92\x90\x30\x9a\x59\x61\x45\x64\x8b\xb2\x64\xc6\x5c"
"\x24\x35\x7e\x9c\x39\x47\xdc\x47\xae\xdc\x66\xe9\x94\xcd\xc3\xb0"
"\x1b\x80\xc2\xcf\x89\xbd\x22\x19\x8a\xb9\x67\x0a\x3b\xcf\x1e\x6f"
"\xf2\x8a\xb6\x97\x14\xae\x0b\x99\xe5\xba\x1a\x7e\xca\x7c\x03\xc5"
"\x31\x53\x7c\xf4\x48\x4f\xdb\xee\xfd\x11\x6c\x98\x84\xef\x61\xef"
"\xb9\x71\x43\x00\x4f\x57\xad\x23\x8e\x3c\x98\x39\x8d\x0f\x90\xbd"
"\x3a\x7d\xb2\xd5\x90\xc2\x2f\xf7\xa2\x13\x8b\x7e\xbb\x40\xf3\xf3"
"\xf8\xaf\xb6\x88\xfe\x5b\x20\x07\x0e\xf2\x90\x1e\x83\x23\xac\x21"
"\x63\xb7\x04\xe0\x81\xa6\xaa\xbb\x45\x23\xc1\x7a\xf0\x67\x8d\xf7"
"\xf9\xf4\x09\xe0\xe9\x5c\xa4\x80\xf9\xb6\x58\xc3\x0c\xe7\xf7\x15"
"\x53\xee\x92\xa4\xc8\xc8\x79\x8f\x69\xb1\x6d\x3e\x56\xb1\xa7\x67"
"\x45\x4d\x28\xc2\x98\x41\xdb\xad\x41\xec\x5f\x04\x42\x08\x06\x21"
"\x7a\x3b\x90\x5f\xbb\x26\x9b\xf5\xb9\xbe\xe5\x80\xdc\x7d\xbf\x4f"
"\x81\x16\x09\xf6\x4d\xb3\x32\xf8\xd6\x6c\x64\xd0\x71\x74\x34\x80"
"\xce\xaa\x51\x8f\xb1\x22\x8a\x0e\x25\x02\xe5\x5e\x61\x27\x13\x25"
"\x63\x8c\xc1\x49\xed\xcb\x10\x9b\x81\x81\xae\x97\x58\x5b\x70\x1c"
"\x4f\x7d\x81\x35\x76\xbc\x0a\x6f\x2e\xfd\xa0\x87\x8a\x6e\xae\x15"
"\xdc\xb0\xd8\x05\x74\x9e\xda\xe9\xc1\xf8\x27\xff\x9a\xbb\xa6\x62"
"\x9c\xad\xde\x35\x50\x7a\x9f\x4f\x41\xe7\x85\x4e\x90\xc6\x7e\xdf"
"\x70\xb0\x5c\x51\x7f\xe0\x26\x3a\xff\xea\x05\x12\xe9\x82\xfe\x07"
"\xc3\x97\x58\xd3\x53\xe7\x97\x4f\x9f\xa4\x24\x9b\xa6\xfb\x69\xe2"
"\x7b\xd6\x69\x05\x46\x57\xc2\xf6\x49\x13\x64\x26\x0f\x77\x11\xd9"
"\x09\xd2\x31\x90\x39\xde\xc1\x55\xf7\xdc\x49\x87\x1a\x5a\x9f\xe1"
"\x53\xef\xd8\x3b\x70\xdf\xd8\x4a\x9f\x62\xda\x48\xf6\xee\x2d\x01"
"\xf0\x0b\xc4\xf3\x98\x47\x3b\xe5\x95\xca\x06\x85\x07\x16\x8b\x33"
"\x29\x9f\xee\x28\x05\xe9\x14\x89\xc6\x9c\x27\x75\xc2\xa1\x34\x5e"
"\x13\x75\x24\x4d\x93\x49\x1c\x13\x5a\x4d\xb7\x3f\x98\xbd\x60\x65"
"\x1d\xaa\x6c\x46\xfc\x71\xc5\x70\x86\x6c\xd6\x38\xab\x4f\xc0\x55"
"\xa6\xb3\xa7\x81\x79\xcd\x74\x8e\x7e\xe3\x56\x87\x78\xab\xf8\xe5"
"\x6d\xb6\x8b\xba\x23\x68\x94\xdd\x4d\xd2\xb0\x83\x95\xff\x35\xff"
"\x4e\xa5\xa5\xe4\x0a\x66\x33\x76\xa4\x36\xd0\x1c\x4f\x9c\x7c\xe6"
"\xef\x40\x8f\xd3\x4e\x4b\x26\x13\x18\x22\x40\x0d\x73\x9e\x27\x9c"
"\x93\xe4\xaf\x2f\x18\xc6\x18\xbd\x80\xb1\x80\x75\xbc\x70\xea\x93"
"\xd0\x2e\x27\x08\x49\xb0\x1d\x05\x55\x17\x2f\xda\xff\x78\x7a\x91"
"\x80\xa1\x6a\xca\x3b\xfd\x7f\xfc\xeb\x7e\x34\xf7\xde\x3c\xfe\xad"
"\xcc\x2d\x00\x98\xca\xc4\x69\xc5\x1e\xe2\x62\x72\x8c\x8b\x54\x99"
"\xc7\xfd\x65\xf6\x00\x54\xdc\x75\xea\x48\x90\xb6\x97\xec\x81\xc9"
"\x6c\x2d\x17\x7d\xa2\x89\x08\xbe\x84\xe2\xa7\x4c\xc1\x1e\x27\x43"
"\x27\x77\xb9\x29\x1e\xfb\xd3\x1f\xe9\x52\x62\x01\x9c\xab\x20\xd8"
"\xe5\xf5\xe2\x9b\xb5\xfe\xd3\xf5\x58\xd9\x9e\x7f\x8d\xd6\xcc\xbe"
"\xfb\x05\xca\xcc\x5d\xc9\x89\xb6\x70\xea\xad\x69\x95\x35\xf0\x06"
"\x49\x52\xde\x0b\x55\xec\x93\xfc\xe0\x28\x04\xdf\x2e\xee\xa8\x12"
"\xf6\xd5\x4d\xa0\x89\x29\xc9\x2a\x61\xc0\xd6\x4a\x76\xc9\x5c\xfb"
"\x9b\xfa\xd4\x96\xb0\x9f\xa5\x02\xee\xa6\x0c\x8a\x4d\x27\xfd\x4a"
"\xd3\x60\x59\xd6\xe8\x19\x6f\xd3\xdb\x58\x04\x82\x20\xce\x46\x14"
"\x41\x83\xd3\x90\x74\xef\x57\x7f\x39\x07\xa2\x52\xfa\x34\xfa\x5f"
"\x46\xbe\x55\x26\xcd\xc3\xff\x5e\xf0\x19\x6d\x5a\xb7\x11\xe9\x22"
"\x21\x33\x87\x62\x7e\xde\xb3\x1b\x62\xba\x37\x00\x1c\xdd\x17\x7a"
"\x56\x7c\xc1\xbb\xe9\x15\x42\x86\x6c\x42\xe3\xd9\x59\xaa\x1c\xd3"
"\xa7\xfd\x7f\x4b\x86\xb7\x21\x96\x10\x8f\x5f\xd9\xc8\x1e\xf6\xf2"
"\x66\x54\x9c\xd8\x51\x54\x74\xad\xf1\x8f\x6d\x72\x83\xc2\x9e\x40"
"\x0b\x28\xbe\x38\x6f\x32\x2c\x29\x3a\x10\x3a\x5f\x45\x15\xa0\x5c"
"\x20\xae\x8d\xb9\xce\x90\x8a\x5c\x23\xd8\xff\x63\x76\xf0\xde\xa3"
"\x98\x8d\xce\x78\x9d\x83\x38\x06\x07\x58\xe0\x25\x98\x42\xcb\xd9"
"\x2a\xed\x97\x37\xa7\x2f\xc1\x49\xce\x81\x5f\x41\x95\xae\x46\xcd"
"\x8e\xe8\x2a\xb7\x4e\xb8\x28\x93\x32\xc0\x8b\x83\xac\x64\xf8\x4e"
"\x05\x48\x99\x18\x5a\xdc\x2c\x3e\xd5\x89\x42\xdb\x51\xee\xef\x51"
"\xb2\x54\xa8\x72\x5b\x33\x7a\x0d\x3a\x20\x71\xc7\x9f\xcc\x40\xe9"
"\x0b\xa9\x15\x93\x66\xb7\x63\x50\xae\xca\xbe\x3a\x68\x1e\x4c\x29"
"\xcd\xeb\xbe\x4d\xeb\x89\x79\x5f\x01\x4b\x71\x0b\xdb\xfc\x34\x3d"
"\xc0\xbd\x3d\xd6\x38\x89\xf7\x4b\x9d\x44\x18\xcb\x23\x7c\x1e\x4f"
"\xb0\xb9\xb1\xa3\x4f\xad\x3c\x88\x4a\xa2\xbe\x4f\x2d\x16\x09\x08"
"\x65\x2a\x61\xb1\x53\xc4\x90\x6e\xbc\x11\x1a\xb5\xa5\xce\x14\x17"
"\x21\xc8\x63\x59\x9c\x7a\x87\xcb\xff\x94\xf0\x7f\xb8\x86\x95\x04"
"\xa6\x16\x61\x8b\x3d\x1c\x04\x62\x07\xdd\x97\x19\xf6\x38\x54\xfa"
"\x37\xae\xdf\x41\x47\x19\x82\x87\x0d\x3b\x52\x96\x07\xb2\x89\x6b"
"\xb8\xd1\x46\x9c\xf3\xfc\x4b\xe8\x22\x9a\x4a\xe2\xa3\x94\x94\x4b"
"\x2f\x9f\x64\x6c\x85\xcd\xb5\xc1\x30\xee\x8b\xf9\x12\x98\x32\x6f"
"\x8b\x71\x36\x4c\x26\xe8\x43\x1c\x54\x9a\xeb\xd8\x3d\xd3\x87\x98"
"\xdd\xd7\xda\x19\x57\x73\x64\x8c\x6d\xc3\xb2\xfb\xf8\xaf\xea\xf7"
"\xe2\xa4\x26\xf8\x5c\x13\x5b\xdc\x09\xee\xb4\x15\x31\x0d\x02\x98"
"\x45\x5a\x08\xbe\xc3\xa6\x76\xa6\x30\xfe\x7d\xa1\x5d\xcd\xe8\xcc"
"\xce\x36\xcc\xb6\x3e\x4d\x5f\xc2\x91\xc7\xc3\xbc\xf1\x9d\x25\x80"
"\xca\x75\xab\xea\x47\xe5\xd8\x45\x31\x88\x2f\xe2\x78\x66\x1c\xab"
"\x43\x06\xad\x2e\x80\x0e\xba\xbb\xce\x23\xb0\xef\xd5\x87\xe6\x11"
"\xef\xd1\xd2\x41\x52\xe5\x58\x3c\xe8\xc9\x21\x2b\xfe\x07\xad\x06"
"\x3a\xef\x8f\x82\x4a\x32\x05\xee\x7e\x41\x55\xb2\xd3\xaf\x23\x92"
"\xd3\x2d\xb2\xfc\xd3\xcf\x08\x6f\xb6\xab\x61\x5c\x2d\xe4\x3f\xa0"
"\x97\xad\x45\x42\xdc\xd7\x5e\xbf\xca\xab\xe8\xc3\xd7\xff\x67\xf7"
"\xaf\x9d\x77\xec\x0c\xd1\xb0\xa9\xf1\x74\x7c\x3d\x5e\xf9\xb0\xcb"
"\x48\xed\x3e\x4d\xed\x9d\xd3\xd1\xc7\x8e\x93\xd1\xf0\x4a\x6a\x8a"
"\x62\xf3\x92\x1a\x30\x82\xc3\x20\x42\x8e\xb4\x6f\x8e\x07\x6a\x35"
"\x25\xdd\x59\xaa\x36\x88\x6d\xe3\x38\x1d\x7b\xa5\x1b\xb8\x94\xac"
"\x4a\xde\x14\x09\xf2\x28\xe7\x07\x5a\x2c\xc3\x3e\xe4\x61\xa3\x7f"
"\x56\xbe\x77\x36\xc4\x07\x36\x05\x05\xd1\x26\xe6\xfa\x56\xa1\x07"
"\x32\xf3\xe4\x6a\xc6\x29\x2d\xd9\x47\x37\x9b\x65\x9c\xa8\x64\x09"
"\x79\x70\x23\x05\x39\x43\xa5\x73\x48\xdb\x9e\xfa\x5e\x48\xac\x5e"
"\xe9\xe5\xb4\x04\x8a\xa0\xc4\x13\xfd\xf1\x8e\xdc\xdc\xc8\x71\xe3"
"\xcf\x66\xaa\xf6\x1a\x5c\x67\x80\x58\x6e\x52\x00\x79\xb3\xc8\x5d"
"\x92\x9b\x95\x51\x09\xb9\xea\xb8\x04\x7c\xac\x7f\xac\x29\xd4\xc0"
"\x7f\x4a\xa0\x7f\x70\x18\x95\x1d\xbb\xc1\x4d\x3e\x52\x30\x59\x50"
"\x95\xcc\x7e\x7a\xda\x3c\xdc\x4d\x63\x1b\xfe\x37\x6e\x5d\x31\x10"
"\xa0\x20\xa6\x1c\xa3\xfc\x63\x2f\x9e\x71\x85\x5c\x5b\x5f\x2d\x8e"
"\x8f\x96\xc2\xa1\xe3\xa1\xc2\x09\x0c\xbb\xaf\xba\x9d\x33\xf3\x3d"
"\xef\x4f\x64\x08\x68\xa8\x59\x71\x57\xdb\xd7\x9d\xc6\xd6\xc9\x92"
"\x78\xcd\x5b\x57\x44\x1a\xa5\x18\x23\x70\xfe\xb0\xa3\xc3\xbf\x35"
"\xf8\xf1\x44\x02\xac\x72\xe0\x00\x1a\x08\xb7\x03\x8b\x66\x09\xbf"
"\x97\xa2\x92\x35\x38\x16\x60\x78\xcd\xac\x9a\xd0\xf7\x2a\xcd\x50"
"\x7f\xa8\x4b\xc7\x3a\x58\xe2\x4e\x2b\x64\x7a\x94\x7f\x87\xdd\x91"
"\xfc\x0a\x5a\xac\xf7\xbe\xbd\x46\x88\x6c\x17\xa6\xa9\x0b\xd3\xaf"
"\x4f\xf7\x65\x21\x11\x00\xdc\x0f\x8d\x09\xeb\xa7\x24\x4b\x34\xa3"
"\x11\xef\xc1\x55\x64\x29\xa3\xb1\x21\x4b\x1f\xb8\xd6\x78\x3a\x58"
"\xff\x35\x4b\xc6\x7d\x6c\xaa\x6a\xe9\xb1\x0c\x31\xc5\x97\x2d\x80"
"\x23\x29\x78\x7e\x86\x46\x12\x1c\x11\x89\x73\x9a\xd3\x47\xa0\x90"
"\x2a\x4b\xe5\x59\xef\x53\x4b\x49\xb4\x30\x30\x65\x94\x5b\xe6\x7a"
"\x07\xd5\x9c\x24\x47\xe7\x09\xbb\xa8\x10\xa5\xfd\xbb\xfa\xce\x40"
"\x5d\x8f\x14\xe9\x9f\xa3\x19\x1e\x83\x37\x04\xe0\x43\x5a\x1c\x8d"
"\x84\xa2\x6a\x7c\xfa\xec\x77\x6b\x09\x39\xb9\x81\xa8\x42\xad\x86"
"\x52\x32\xb2\x69\x8e\x07\xd7\x6d\xe8\xa4\xd3\xea\x26\x08\x1e\x81"
"\x08\x6d\xff\x93\xbe\x23\x45\xec\xfe\xc2\x91\x09\xab\xde\x82\x4f"
"\x1c\x88\xdd\xb9\xab\x1c\x73\x76\xc0\xb0\x7b\x66\x7e\x0e\x4a\xeb"
"\x1b\x8d\xab\x89\xaf\x34\x98\xc7\x7d\xd1\x91\x62\xd2\x6f\xaf\xc0"
"\x55\xa2\x3d\x42\x56\x97\xe6\xc6\xfa\x99\x58\x22\x30\x96\xc9\xa0"
"\x9c\xc6\x72\x5f\xcf\xf0\x80\x37\xd2\x65\x69\x6b\xa4\x1e\xeb\x2e"
"\xdb\xb3\xec\xe9\x00\xfa\xdf\x3d\x8b\x97\x3b\x2f\x1f\xfe\x14\xf2"
"\xef\x95\x3e\x12\xa6\x81\x38\x2a\x30\xa1\x3e\x0b\xcf\xb1\xc8\xbd"
"\xbc\x20\x91\x78\x06\x41\x93\x9b\xd9\x5b\x0b\xb0\x96\xd4\xa4\x88"
"\x16\xd6\xa7\x97\x1a\x12\x66\x61\x77\x72\x99\x88\x70\xcf\xc9\xca"
"\xb8\x1f\x42\x4c\x88\x62\xb8\x52\xb9\x97\x8e\xe8\x6d\x50\xe6\xbb"
"\x09\x7c\x61\x71\xad\xc4\x73\xc2\x87\x0a\x10\xfc\x6f\x80\x57\x14"
"\x5f\x75\xb8\x2e\xf5\xef\x00\x8c\x8b\x41\x17\x9d\x30\xc9\xae\xd3"
"\xb3\x61\x05\x67\xef\x28\x99\x59\x9d\x1c\x55\x4f\xf6\x39\xd5\xf1"
"\xb3\xb9\xd0\x25\xe0\x8a\xc5\xc8\xab\x71\xfc\xaf\xc3\xf9\xb3\xb9"
"\xa2\x6e\xd7\x3f\x0a\x51\x2c\x4f\x37\x61\x50\x3e\xe7\xd1\x15\x3f"
"\x26\xf5\xbd\x29\xfa\x18\x44\xea\x68\xbc\x83\xd4\xb8\x06\xe7\x59"
"\xbc\x30\x05\x67\x72\xe8\x62\x1e\xdf\xed\x43\x14\x40\x9d\xae\xff"
"\x3a\xb3\xe8\x0f\x52\x12\x6b\x9a\xac\x64\x2c\xab\xb8\x6e\xa4\xbb"
"\x68\xb3\x01\xed\xcf\x67\xd3\x05\x79\x41\x46\x21\x77\x8e\x1c\x1a"
"\x15\x36\x92\x76\x79\x08\x6c\xda\xe9\x8b\xc1\x89\xd5\xcb\xfc\x3f"
"\xe0\x0c\x09\x5d\x30\xe4\xdd\x9e\x71\xa5\xf2\xae\x09\xd4\x27\xf1"
"\xc1\xe2\x60\x2d\x5e\x30\x03\xb9\xa5\xc7\x69\xdd\xf3\x27\x9b\xea"
"\x0f\x5c\x53\xaf\x58\xfd\xd0\x15\xc2\xc7\x76\x97\x34\x1d\x43\x91"
"\x7b\x62\x97\x59\x82\x5a\xbd\xc0\xd6\xde\x02\xe6\x7a\x11\x7a\xd1"
"\xea\x0c\x5b\xd3\x94\x0a\xeb\x80\xe1\x0f\x8e\xc6\xcc\xad\x39\x73"
"\xf5\xec\x87\xd5\x2e\x50\xee\x08\xcb\xdc\xe5\xd5\xc3\xe8\xaa\x90"
"\xbc\xa0\xec\x6f\xa3\xda\xac\xb5\xe7\x78\xd8\xde\x45\xe9\x04\x51"
"\x28\x4c\xc2\x1b\xb7\x28\xfd\xab\xa6\xe8\x7f\x01\x90\x85\x55\x65"
"\xb5\x3f\x92\x33\x0e\xe2\x39\x00\x01\x3b\x6b\x71\x48\x98\x60\x67"
"\xc4\xdb\x63\x5a\xf0\xa8\x04\xd9\x2c\x98\x71\x17\xd4\x04\x1b\xc4"
"\x93\x3f\x2e\x7c\xcb\xf2\x9f\x8f\x56\x93\x2b\x9c\x63\x27\x3e\x44"
"\xf5\x7e\x31\x13\x09\x6c\x8f\xb7\x85\x61\x5f\x4e\x5a\x40\xfd\xd1"
"\xff\x42\xfa\x84\xaf\x63\x96\x0c\xef\xd2\x38\x83\x1c\xa6\x64\x2f"
"\x52\x9d\xa4\x04\xca\x6e\x9b\xef\xda\xbb\xd8\x82\xa2\xad\x96\x31"
"\xd2\x38\x7c\xcf\x81\x60\x53\x8e\x4f\xc9\xa0\x30\x6f\xc2\x49\xe2"
"\xf0\x5c\x4a\x40\x14\x43\x1b\xfc\x64\xf7\x94\xe4\xdb\xa2\x4a\xc5"
"\xe2\xaa\x49\x5e\xf0\x0f\x00\x3f\xc6\x5b\xc5\xc2\x9b\x26\x8b\x6f"
"\xdc\xdd\x47\x71\x64\xac\x17\x43\x14\x5d\x69\xfe\x02\xf4\x03\x44"
"\xaf\xee\xf6\x63\x93\x15\x09\x8f\xf9\xe8\xe2\x82\x05\x0d\xce\x75"
"\x2d\xcd\xdf\xbc\xea\x77\x7f\x3c\x55\x6c\x9c\x42\xac\x69\x4b\x98"
"\x96\x20\x67\x5d\x4a\x31\xeb\xad\xfe\x84\x59\x77\xe7\xd3\x0c\x38"
"\x37\xa6\x87\x9f\x64\x2c\x54\xd5\x82\xe1\x57\xea\xb9\x4f\xa5\x56"
"\x80\xbd\x4c\x14\x76\xd0\x58\xc7\x7a\x8c\xcc\xae\x21\x39\x82\x89"
"\xd3\x3a\xb9\xb1\x12\x23\xee\x67\xea\x78\xb1\xd3\x8a\x6a\xe3\xeb"
};

STATIC const mp_rom_map_elem_t mod_trezorcrypto_monero_globals_table[] = {
    {MP_ROM_QSTR(MP_QSTR___name__), MP_ROM_QSTR(MP_QSTR_monero)},
    // types
    {MP_ROM_QSTR(MP_QSTR_Point),
     MP_ROM_PTR(&mod_trezorcrypto_monero_ge25519_type)},
    {MP_ROM_QSTR(MP_QSTR_Scalar),
     MP_ROM_PTR(&mod_trezorcrypto_monero_bignum256modm_type)},
    // functions
    {MP_ROM_QSTR(MP_QSTR_sc_copy),
     MP_ROM_PTR(&mod_trezorcrypto_monero_sc_copy_obj)},
    {MP_ROM_QSTR(MP_QSTR_sc_check),
     MP_ROM_PTR(&mod_trezorcrypto_monero_sc_check_obj)},
    {MP_ROM_QSTR(MP_QSTR_sc_iszero),
     MP_ROM_PTR(&mod_trezorcrypto_monero_sc_iszero_obj)},
    {MP_ROM_QSTR(MP_QSTR_sc_eq),
     MP_ROM_PTR(&mod_trezorcrypto_monero_sc_eq_obj)},
    {MP_ROM_QSTR(MP_QSTR_sc_add_into),
     MP_ROM_PTR(&mod_trezorcrypto_monero_sc_add_into_obj)},
    {MP_ROM_QSTR(MP_QSTR_sc_sub_into),
     MP_ROM_PTR(&mod_trezorcrypto_monero_sc_sub_into_obj)},
    {MP_ROM_QSTR(MP_QSTR_sc_mul_into),
     MP_ROM_PTR(&mod_trezorcrypto_monero_sc_mul_into_obj)},
    {MP_ROM_QSTR(MP_QSTR_sc_mulsub_into),
     MP_ROM_PTR(&mod_trezorcrypto_monero_sc_mulsub_into_obj)},
    {MP_ROM_QSTR(MP_QSTR_sc_muladd_into),
     MP_ROM_PTR(&mod_trezorcrypto_monero_sc_muladd_into_obj)},
    {MP_ROM_QSTR(MP_QSTR_sc_inv_into),
     MP_ROM_PTR(&mod_trezorcrypto_monero_sc_inv_into_obj)},
    {MP_ROM_QSTR(MP_QSTR_encodeint_into),
     MP_ROM_PTR(&mod_trezorcrypto_monero_encodeint_into_obj)},
    {MP_ROM_QSTR(MP_QSTR_decodeint_into),
     MP_ROM_PTR(&mod_trezorcrypto_monero_decodeint_into_obj)},
    {MP_ROM_QSTR(MP_QSTR_decodeint_into_noreduce),
     MP_ROM_PTR(&mod_trezorcrypto_monero_decodeint_into_noreduce_obj)},
    {MP_ROM_QSTR(MP_QSTR_identity_into),
     MP_ROM_PTR(&mod_trezorcrypto_monero_identity_into_obj)},
    {MP_ROM_QSTR(MP_QSTR_xmr_H),
     MP_ROM_PTR(&mod_trezorcrypto_monero_xmr_H_obj)},
    {MP_ROM_QSTR(MP_QSTR_encodepoint_into),
     MP_ROM_PTR(&mod_trezorcrypto_monero_encodepoint_into_obj)},
    {MP_ROM_QSTR(MP_QSTR_decodepoint_into),
     MP_ROM_PTR(&mod_trezorcrypto_monero_decodepoint_into_obj)},
    {MP_ROM_QSTR(MP_QSTR_ge25519_check),
     MP_ROM_PTR(&mod_trezorcrypto_monero_ge25519_check_obj)},
    {MP_ROM_QSTR(MP_QSTR_point_eq),
     MP_ROM_PTR(&mod_trezorcrypto_monero_point_eq_obj)},
    {MP_ROM_QSTR(MP_QSTR_point_add_into),
     MP_ROM_PTR(&mod_trezorcrypto_monero_point_add_into_obj)},
    {MP_ROM_QSTR(MP_QSTR_point_sub_into),
     MP_ROM_PTR(&mod_trezorcrypto_monero_point_sub_into_obj)},
    {MP_ROM_QSTR(MP_QSTR_ge25519_mul8),
     MP_ROM_PTR(&mod_trezorcrypto_monero_ge25519_mul8_obj)},
    {MP_ROM_QSTR(MP_QSTR_ge25519_double_scalarmult_vartime_into),
     MP_ROM_PTR(
         &mod_trezorcrypto_monero_ge25519_double_scalarmult_vartime_into_obj)},
    {MP_ROM_QSTR(MP_QSTR_scalarmult_base_into),
     MP_ROM_PTR(&mod_trezorcrypto_monero_scalarmult_base_into_obj)},
    {MP_ROM_QSTR(MP_QSTR_scalarmult_into),
     MP_ROM_PTR(&mod_trezorcrypto_monero_scalarmult_into_obj)},
    {MP_ROM_QSTR(MP_QSTR_xmr_base58_addr_encode_check),
     MP_ROM_PTR(&mod_trezorcrypto_monero_xmr_base58_addr_encode_check_obj)},
    {MP_ROM_QSTR(MP_QSTR_xmr_base58_addr_decode_check),
     MP_ROM_PTR(&mod_trezorcrypto_monero_xmr_base58_addr_decode_check_obj)},
    {MP_ROM_QSTR(MP_QSTR_random_scalar),
     MP_ROM_PTR(&mod_trezorcrypto_monero_random_scalar_obj)},
    {MP_ROM_QSTR(MP_QSTR_fast_hash_into),
     MP_ROM_PTR(&mod_trezorcrypto_monero_fast_hash_into_obj)},
    {MP_ROM_QSTR(MP_QSTR_hash_to_point_into),
     MP_ROM_PTR(&mod_trezorcrypto_monero_hash_to_point_into_obj)},
    {MP_ROM_QSTR(MP_QSTR_hash_to_scalar_into),
     MP_ROM_PTR(&mod_trezorcrypto_monero_hash_to_scalar_into_obj)},
    {MP_ROM_QSTR(MP_QSTR_xmr_derivation_to_scalar),
     MP_ROM_PTR(&mod_trezorcrypto_monero_xmr_derivation_to_scalar_obj)},
    {MP_ROM_QSTR(MP_QSTR_xmr_generate_key_derivation),
     MP_ROM_PTR(&mod_trezorcrypto_monero_xmr_generate_key_derivation_obj)},
    {MP_ROM_QSTR(MP_QSTR_xmr_derive_private_key),
     MP_ROM_PTR(&mod_trezorcrypto_monero_xmr_derive_private_key_obj)},
    {MP_ROM_QSTR(MP_QSTR_xmr_derive_public_key),
     MP_ROM_PTR(&mod_trezorcrypto_monero_xmr_derive_public_key_obj)},
    {MP_ROM_QSTR(MP_QSTR_add_keys2_into),
     MP_ROM_PTR(&mod_trezorcrypto_monero_add_keys2_into_obj)},
    {MP_ROM_QSTR(MP_QSTR_add_keys3_into),
     MP_ROM_PTR(&mod_trezorcrypto_monero_add_keys3_into_obj)},
    {MP_ROM_QSTR(MP_QSTR_xmr_get_subaddress_secret_key),
     MP_ROM_PTR(&mod_trezorcrypto_monero_xmr_get_subaddress_secret_key_obj)},
    {MP_ROM_QSTR(MP_QSTR_gen_commitment_into),
     MP_ROM_PTR(&mod_trezorcrypto_monero_gen_commitment_into_obj)},
    {MP_ROM_QSTR(MP_QSTR_ct_equals),
     MP_ROM_PTR(&mod_trezorcrypto_ct_equals_obj)},
    // bulletproof constants
    {MP_ROM_QSTR(MP_QSTR_BP_PLUS_GI_PRE),
     MP_ROM_PTR(&mod_trezorcrypto_monero_BP_PLUS_GI_PRE_obj)},
    {MP_ROM_QSTR(MP_QSTR_BP_PLUS_HI_PRE),
     MP_ROM_PTR(&mod_trezorcrypto_monero_BP_PLUS_HI_PRE_obj)},
};
STATIC MP_DEFINE_CONST_DICT(mod_trezorcrypto_monero_globals,
                            mod_trezorcrypto_monero_globals_table);

STATIC const mp_obj_module_t mod_trezorcrypto_monero_module = {
    .base = {&mp_type_module},
    .globals = (mp_obj_dict_t *)&mod_trezorcrypto_monero_globals,
};
