import unittest
from bdfparser import Font, Glyph
from .info import unifont_path, glyph_a_meta, missing_glyph_meta


# Test all `Font` attributes and methods, with Unifont


class TestFontLoading(unittest.TestCase):

    def setUp(self):
        self.maxDiff = None
        self.font = Font()

    def test_load_file_path(self):
        self.assertIsInstance(self.font.load_file_path(unifont_path), Font)

    def test_load_file_obj(self):
        self.assertIsInstance(
            self.font.load_file_obj(open(unifont_path)), Font)


class TestFont(unittest.TestCase):

    def setUp(self):
        self.maxDiff = None
        self.font = Font(unifont_path)

    def test_init(self):
        self.assertIsInstance(self.font, Font)

    def test_headers(self):
        self.assertEqual(self.font.headers, {'bdfversion': 2.1,
                                             'fontname': '-gnu-Unifont-Medium-R-Normal-Sans-16-160-75-75-c-80-iso10646-1',
                                             'pointsize': 16,
                                             'xres': 75,
                                             'yres': 75,
                                             'fbbx': 16,
                                             'fbby': 16,
                                             'fbbxoff': 0,
                                             'fbbyoff': -2,
                                             'comment': ['Generated by fontforge, http://fontforge.sourceforge.net',
                                                         '(C)Copyright'],
                                             'metricsset': 0})

    def test_props(self):
        self.assertEqual(self.font.props, {'add_style_name': 'Sans Serif',
                                           'average_width': '80',
                                           'cap_height': '10',
                                           'charset_encoding': '1',
                                           'charset_registry': 'ISO10646',
                                           'copyright': 'Copyright (C) 1998-2020 Roman Czyborra, Paul Hardy, Qianqian '
                                           'Fang, Andrew Miller, Johnnie Weaver, David Corbett, Rebecca '
                                           'Bettencourt, et al. License: SIL Open Font License version 1.1 '
                                           'and GPLv2+: GNU GPL version 2 or later '
                                           '<http://gnu.org/licenses/gpl.html> with the GNU Font Embedding '
                                           'Exception.',
                                           'default_char': '65533',
                                           'family_name': 'Unifont',
                                           'font_ascent': '14',
                                           'font_descent': '2',
                                           'font_type': 'Bitmap',
                                           'font_version': '13.0.04',
                                           'foundry': 'GNU',
                                           'pixel_size': '16',
                                           'point_size': '160',
                                           'resolution_x': '75',
                                           'resolution_y': '75',
                                           'setwidth_name': 'Normal',
                                           'slant': 'R',
                                           'spacing': 'C',
                                           'underline_position': '-2',
                                           'underline_thickness': '1',
                                           'weight_name': 'Medium',
                                           'x_height': '8'})

    def test_glyphs_a(self):
        self.assertEqual(self.font.glyphs[97], ['U+0061',
                                                97,
                                                8,
                                                16,
                                                0,
                                                -2,
                                                500,
                                                0,
                                                8,
                                                0,
                                                None,
                                                None,
                                                None,
                                                None,
                                                None,
                                                None,
                                                ['00',
                                                 '00',
                                                 '00',
                                                 '00',
                                                 '00',
                                                 '00',
                                                 '3C',
                                                 '42',
                                                 '02',
                                                 '3E',
                                                 '42',
                                                 '42',
                                                 '46',
                                                 '3A',
                                                 '00',
                                                 '00']])

    def test_glyphs_len(self):
        self.assertEqual(len(self.font.glyphs), len(self.font))

    def test_length(self):
        self.assertEqual(self.font.length(), 849)


class TestFontIter(unittest.TestCase):

    def setUp(self):
        self.maxDiff = None
        self.font = Font(unifont_path)

    def test_itercps_list_len(self):
        self.assertEqual(len(list(self.font.itercps())), len(self.font))

    def test_itercps_cp_first(self):
        self.assertEqual(next(self.font.itercps()), 0)

    def test_itercps_cp_reversed_first(self):
        self.assertEqual(next(self.font.itercps(order=2)), 30340)

    def test_itercps_file_first(self):
        self.assertEqual(next(self.font.itercps(order=0)), 1)

    def test_itercps_file_reversed_first(self):
        self.assertEqual(next(self.font.itercps(order=-1)), 1790)

    def test_itercps_range(self):
        r128 = list(self.font.itercps(r=128))
        r0x100 = list(self.font.itercps(r=0x100))

        self.assertEqual(len(r128), 128)
        self.assertEqual(len(r0x100), 256)
        self.assertEqual(list(self.font.itercps(r=(0, 127))), r128)
        self.assertEqual(list(self.font.itercps(r=(0, 0xff))), r0x100)

    def test_itercps_range2(self):
        r_numbers = list(self.font.itercps(r=(48, 57)))
        r_reversed_uppers = list(self.font.itercps(order=2, r=(65, 90)))
        r_letters_with_nonexistent_range = list(
            self.font.itercps(r=[(65, 90), (97, 122), (0x20000, 0x3134F)]))

        self.assertEqual(r_numbers, [48, 49, 50, 51, 52, 53, 54, 55, 56, 57])
        self.assertEqual(r_reversed_uppers, [90, 89, 88, 87, 86, 85, 84, 83, 82,
                                             81, 80, 79, 78, 77, 76, 75, 74, 73, 72, 71, 70, 69, 68, 67, 66, 65])
        self.assertEqual(r_letters_with_nonexistent_range, [65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77, 78, 79, 80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 90,
                                                            97, 98, 99, 100, 101, 102, 103, 104, 105, 106, 107, 108, 109, 110, 111, 112, 113, 114, 115, 116, 117, 118, 119, 120, 121, 122])

    def test_iterglyphs_list_len(self):
        self.assertEqual(len(list(self.font.iterglyphs())), len(self.font))

    def test_iterglyphs_cp_first(self):
        first_cp_glyph = next(self.font.iterglyphs())
        self.assertIsInstance(first_cp_glyph, Glyph)
        self.assertEqual(first_cp_glyph.meta['glyphname'], 'U+0000')


class TestFontGetGlyph(unittest.TestCase):

    def setUp(self):
        self.maxDiff = None
        self.font = Font(unifont_path)

    def test_glyphbycp_a(self):
        self.assertEqual(self.font.glyphbycp(97).meta, glyph_a_meta)

    def test_glyph_a(self):
        self.assertEqual(self.font.glyph('a').meta, glyph_a_meta)

    def test_glyphbycp_nonexistent(self):
        self.assertEqual(self.font.glyphbycp(22909), None)

    def test_glyph_nonexistent(self):
        self.assertEqual(self.font.glyph('好'), None)

    def test_lacksglyphs(self):
        self.assertEqual(self.font.lacksglyphs('Bé H好Δi的'), ['好', 'Δ'])

    def test_lacksglyphs_none(self):
        self.assertEqual(self.font.lacksglyphs('Bé Hi的'), None)


class TestFontDraw(unittest.TestCase):

    def setUp(self):
        self.maxDiff = None
        self.font = Font(unifont_path)

    def test_drawcps(self):
        self.assertEqual(self.font.drawcps([66, 100, 102, 32, 72, 105]).bindata, ['00000000000000000000000000000000000000000000000000000000',
                                                                                  '00000000000000000000000000000000000000000000000000000000',
                                                                                  '00000000000000000000000000000000000000000000000000000000',
                                                                                  '00000000000000100000110000000000000000000000100000000000',
                                                                                  '01111100000000100001000000000000010000100000100000000000',
                                                                                  '01000010000000100001000000000000010000100000000000000000',
                                                                                  '01000010001110100001000000000000010000100001100000000000',
                                                                                  '01000010010001100111110000000000010000100000100000000000',
                                                                                  '01111100010000100001000000000000011111100000100000000000',
                                                                                  '01000010010000100001000000000000010000100000100000000000',
                                                                                  '01000010010000100001000000000000010000100000100000000000',
                                                                                  '01000010010000100001000000000000010000100000100000000000',
                                                                                  '01000010010001100001000000000000010000100000100000000000',
                                                                                  '01111100001110100001000000000000010000100011111000000000',
                                                                                  '00000000000000000000000000000000000000000000000000000000',
                                                                                  '00000000000000000000000000000000000000000000000000000000'])

    def test_draw_default(self):
        self.assertEqual(self.font.draw('Bdf Hi').bindata, ['00000000000000000000000000000000000000000000000000000000',
                                                            '00000000000000000000000000000000000000000000000000000000',
                                                            '00000000000000000000000000000000000000000000000000000000',
                                                            '00000000000000100000110000000000000000000000100000000000',
                                                            '01111100000000100001000000000000010000100000100000000000',
                                                            '01000010000000100001000000000000010000100000000000000000',
                                                            '01000010001110100001000000000000010000100001100000000000',
                                                            '01000010010001100111110000000000010000100000100000000000',
                                                            '01111100010000100001000000000000011111100000100000000000',
                                                            '01000010010000100001000000000000010000100000100000000000',
                                                            '01000010010000100001000000000000010000100000100000000000',
                                                            '01000010010000100001000000000000010000100000100000000000',
                                                            '01000010010001100001000000000000010000100000100000000000',
                                                            '01111100001110100001000000000000010000100011111000000000',
                                                            '00000000000000000000000000000000000000000000000000000000',
                                                            '00000000000000000000000000000000000000000000000000000000'])

    def test_draw_mode0(self):
        self.assertEqual(self.font.draw('Bdf Hi', mode=0).bindata, ['000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000',
                                                                    '000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000',
                                                                    '000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000',
                                                                    '000000000000000000000010000000000000110000000000000000000000000000000000000000000000100000000000',
                                                                    '011111000000000000000010000000000001000000000000000000000000000001000010000000000000100000000000',
                                                                    '010000100000000000000010000000000001000000000000000000000000000001000010000000000000000000000000',
                                                                    '010000100000000000111010000000000001000000000000000000000000000001000010000000000001100000000000',
                                                                    '010000100000000001000110000000000111110000000000000000000000000001000010000000000000100000000000',
                                                                    '011111000000000001000010000000000001000000000000000000000000000001111110000000000000100000000000',
                                                                    '010000100000000001000010000000000001000000000000000000000000000001000010000000000000100000000000',
                                                                    '010000100000000001000010000000000001000000000000000000000000000001000010000000000000100000000000',
                                                                    '010000100000000001000010000000000001000000000000000000000000000001000010000000000000100000000000',
                                                                    '010000100000000001000110000000000001000000000000000000000000000001000010000000000000100000000000',
                                                                    '011111000000000000111010000000000001000000000000000000000000000001000010000000000011111000000000',
                                                                    '000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000',
                                                                    '000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000'])

    def test_draw_rl(self):
        self.assertEqual(self.font.draw('مرحبا', direction='rl').bindata, ['000000000000000000000000000000000000000000000000',
                                                                           '000000000000000000000000000000000000000000000000',
                                                                           '000000000000000000000000000000000000000000000000',
                                                                           '000010000000000000000000000000000000000000000000',
                                                                           '000010000000000000000000000000000000000000000000',
                                                                           '000010000000000000000000000000000000000000000000',
                                                                           '000010000000000000000000000000000000000000000000',
                                                                           '000010000000010000000000000000000000000000000000',
                                                                           '000010000010001001111110000001000001110000000000',
                                                                           '000010000100001000010000000000100000011000000000',
                                                                           '000010000011110000100000000000100001111000000000',
                                                                           '000000000000000001000000000000100010000000000000',
                                                                           '000000000000000001000000000001000010000000000000',
                                                                           '000000000000100001000000010010000010000000000000',
                                                                           '000000000000000000100010001100000010000000000000',
                                                                           '000000000000000000011100000000000010000000000000'])

    def test_draw_lrbt_linelimit30(self):
        self.assertEqual(self.font.draw(
            'Bdf Hi', direction='lrbt', linelimit=30).bindata, ['00000000000000000000000000000000',
                                                                '00000000000000000000000000000000',
                                                                '00000000000000000000000000000000',
                                                                '00000000000000000000100000000000',
                                                                '00000000010000100000100000000000',
                                                                '00000000010000100000000000000000',
                                                                '00000000010000100001100000000000',
                                                                '00000000010000100000100000000000',
                                                                '00000000011111100000100000000000',
                                                                '00000000010000100000100000000000',
                                                                '00000000010000100000100000000000',
                                                                '00000000010000100000100000000000',
                                                                '00000000010000100000100000000000',
                                                                '00000000010000100011111000000000',
                                                                '00000000000000000000000000000000',
                                                                '00000000000000000000000000000000',
                                                                '00000000000000000000000000000000',
                                                                '00000000000000000000000000000000',
                                                                '00000000000000000000000000000000',
                                                                '00000000000000100000110000000000',
                                                                '01111100000000100001000000000000',
                                                                '01000010000000100001000000000000',
                                                                '01000010001110100001000000000000',
                                                                '01000010010001100111110000000000',
                                                                '01111100010000100001000000000000',
                                                                '01000010010000100001000000000000',
                                                                '01000010010000100001000000000000',
                                                                '01000010010000100001000000000000',
                                                                '01000010010001100001000000000000',
                                                                '01111100001110100001000000000000',
                                                                '00000000000000000000000000000000',
                                                                '00000000000000000000000000000000'])

    def test_draw_tbrl_linelimit40(self):
        self.assertEqual(self.font.draw(
            'Bdf Hi', direction='tbrl', linelimit=40).bindata, ['000000000000000000000000000000000000000000000000',
                                                                '000000000000000000000000000000000000000000000000',
                                                                '000000000000000000000000000000000000000000000000',
                                                                '000000000000000000001100000000000000000000000000',
                                                                '010000100000000000010000000000000111110000000000',
                                                                '010000100000000000010000000000000100001000000000',
                                                                '010000100000000000010000000000000100001000000000',
                                                                '010000100000000001111100000000000100001000000000',
                                                                '011111100000000000010000000000000111110000000000',
                                                                '010000100000000000010000000000000100001000000000',
                                                                '010000100000000000010000000000000100001000000000',
                                                                '010000100000000000010000000000000100001000000000',
                                                                '010000100000000000010000000000000100001000000000',
                                                                '010000100000000000010000000000000111110000000000',
                                                                '000000000000000000000000000000000000000000000000',
                                                                '000000000000000000000000000000000000000000000000',
                                                                '000000000000000000000000000000000000000000000000',
                                                                '000000000000000000000000000000000000000000000000',
                                                                '000000000000000000000000000000000000000000000000',
                                                                '000010000000000000000000000000000000001000000000',
                                                                '000010000000000000000000000000000000001000000000',
                                                                '000000000000000000000000000000000000001000000000',
                                                                '000110000000000000000000000000000011101000000000',
                                                                '000010000000000000000000000000000100011000000000',
                                                                '000010000000000000000000000000000100001000000000',
                                                                '000010000000000000000000000000000100001000000000',
                                                                '000010000000000000000000000000000100001000000000',
                                                                '000010000000000000000000000000000100001000000000',
                                                                '000010000000000000000000000000000100011000000000',
                                                                '001111100000000000000000000000000011101000000000',
                                                                '000000000000000000000000000000000000000000000000',
                                                                '000000000000000000000000000000000000000000000000'])

    def test_draw_rl_without_ucgs(self):
        self.assertEqual(self.font.draw('a的', direction='rl').bindata, ['000100000100000000000000',
                                                                        '000100000100000000000000',
                                                                        '001000000100000000000000',
                                                                        '011111100111110000000000',
                                                                        '010000101000010000000000',
                                                                        '010000101000010000000000',
                                                                        '010000110011110000000000',
                                                                        '010000100100011000000000',
                                                                        '011111100010011000000000',
                                                                        '010000100011111000000000',
                                                                        '010000100100011000000000',
                                                                        '010000100100011000000000',
                                                                        '010000100100011000000000',
                                                                        '011111100011111000000000',
                                                                        '010000100010100000000000',
                                                                        '000000000001000000000000'])

    def test_draw_rl_with_ucgs(self):
        self.assertEqual(self.font.draw('a的', direction='rl',
                                        usecurrentglyphspacing=True).bindata, ['00010000010000000000000000000000',
                                                                               '00010000010000000000000000000000',
                                                                               '00100000010000000000000000000000',
                                                                               '01111110011111000000000000000000',
                                                                               '01000010100001000000000000000000',
                                                                               '01000010100001000000000000000000',
                                                                               '01000011000001000011110000000000',
                                                                               '01000010010001000100001000000000',
                                                                               '01111110001001000000001000000000',
                                                                               '01000010001001000011111000000000',
                                                                               '01000010000001000100001000000000',
                                                                               '01000010000001000100001000000000',
                                                                               '01000010000001000100011000000000',
                                                                               '01111110000001000011101000000000',
                                                                               '01000010001010000000000000000000',
                                                                               '00000000000100000000000000000000'])

    def test_draw_btlr_linelimit40(self):
        self.assertEqual(self.font.draw(
            'Bdf的 Hi', direction='btlr', linelimit=40).bindata, ['0000000000000000000100000100000000000000000000000000000000000000',
                                                                 '0000000000000000000100000100000000000000000000000000000000000000',
                                                                 '0000000000000000001000000100000000000000000000000000000000000000',
                                                                 '0000001000000000011111100111110000000000000000000000000000000000',
                                                                 '0000001000000000010000101000010001000010000000000000000000000000',
                                                                 '0000001000000000010000101000010001000010000000000000000000000000',
                                                                 '0011101000000000010000110000010001000010000000000000000000000000',
                                                                 '0100011000000000010000100100010001000010000000000000000000000000',
                                                                 '0100001000000000011111100010010001111110000000000000000000000000',
                                                                 '0100001000000000010000100010010001000010000000000000000000000000',
                                                                 '0100001000000000010000100000010001000010000000000000000000000000',
                                                                 '0100001000000000010000100000010001000010000000000000000000000000',
                                                                 '0100011000000000010000100000010001000010000000000000000000000000',
                                                                 '0011101000000000011111100000010001000010000000000000000000000000',
                                                                 '0000000000000000010000100010100000000000000000000000000000000000',
                                                                 '0000000000000000000000000001000000000000000000000000000000000000',
                                                                 '0000000000000000000000000000000000000000000000000000000000000000',
                                                                 '0000000000000000000000000000000000000000000000000000000000000000',
                                                                 '0000000000000000000000000000000000000000000000000000000000000000',
                                                                 '0000000000000000000011000000000000000000000000000000100000000000',
                                                                 '0111110000000000000100000000000000000000000000000000100000000000',
                                                                 '0100001000000000000100000000000000000000000000000000000000000000',
                                                                 '0100001000000000000100000000000000000000000000000001100000000000',
                                                                 '0100001000000000011111000000000000000000000000000000100000000000',
                                                                 '0111110000000000000100000000000000000000000000000000100000000000',
                                                                 '0100001000000000000100000000000000000000000000000000100000000000',
                                                                 '0100001000000000000100000000000000000000000000000000100000000000',
                                                                 '0100001000000000000100000000000000000000000000000000100000000000',
                                                                 '0100001000000000000100000000000000000000000000000000100000000000',
                                                                 '0111110000000000000100000000000000000000000000000011111000000000',
                                                                 '0000000000000000000000000000000000000000000000000000000000000000',
                                                                 '0000000000000000000000000000000000000000000000000000000000000000'])

    def test_draw_nonexistent_default(self):
        self.assertEqual(self.font.draw('Bé H好Δi的').bindata, ['00000000000000000000000000000000000000000001000001000000',
                                                              '00000000000000000000000000000000000000000001000001000000',
                                                              '00000000000011000000000000000000000000000010000001000000',
                                                              '00000000001100000000000000000000000010000111111001111100',
                                                              '01111100000000000000000001000010000010000100001010000100',
                                                              '01000010000000000000000001000010000000000100001010000100',
                                                              '01000010001111000000000001000010000110000100001100000100',
                                                              '01000010010000100000000001000010000010000100001001000100',
                                                              '01111100010000100000000001111110000010000111111000100100',
                                                              '01000010011111100000000001000010000010000100001000100100',
                                                              '01000010010000000000000001000010000010000100001000000100',
                                                              '01000010010000000000000001000010000010000100001000000100',
                                                              '01000010010000100000000001000010000010000100001000000100',
                                                              '01111100001111000000000001000010001111100111111000000100',
                                                              '00000000000000000000000000000000000000000100001000101000',
                                                              '00000000000000000000000000000000000000000000000000010000'])

    def test_draw_nonexistent_mode0_linelimit80(self):
        self.assertEqual(self.font.draw('Bé H好Δi的', mode=0, linelimit=80).bindata, ['00000000000000000000000000000000000000000000000000000000000000000000000000000000',
                                                                                    '00000000000000000000000000000000000000000000000000000000000000000000000000000000',
                                                                                    '00000000000000000000110000000000000000000000000000000000000000000000000000000000',
                                                                                    '00000000000000000011000000000000000000000000000000000000000000000000000000000000',
                                                                                    '01111100000000000000000000000000000000000000000001000010000000000000000000000000',
                                                                                    '01000010000000000000000000000000000000000000000001000010000000000000000000000000',
                                                                                    '01000010000000000011110000000000000000000000000001000010000000000000000000000000',
                                                                                    '01000010000000000100001000000000000000000000000001000010000000000000000000000000',
                                                                                    '01111100000000000100001000000000000000000000000001111110000000000000000000000000',
                                                                                    '01000010000000000111111000000000000000000000000001000010000000000000000000000000',
                                                                                    '01000010000000000100000000000000000000000000000001000010000000000000000000000000',
                                                                                    '01000010000000000100000000000000000000000000000001000010000000000000000000000000',
                                                                                    '01000010000000000100001000000000000000000000000001000010000000000000000000000000',
                                                                                    '01111100000000000011110000000000000000000000000001000010000000000000000000000000',
                                                                                    '00000000000000000000000000000000000000000000000000000000000000000000000000000000',
                                                                                    '00000000000000000000000000000000000000000000000000000000000000000000000000000000',
                                                                                    '00000000000000000000000000000000000100000100000000000000000000000000000000000000',
                                                                                    '00000000000000000000000000000000000100000100000000000000000000000000000000000000',
                                                                                    '00000000000000000000000000000000001000000100000000000000000000000000000000000000',
                                                                                    '00000000000000000000100000000000011111100111110000000000000000000000000000000000',
                                                                                    '00000000000000000000100000000000010000101000010000000000000000000000000000000000',
                                                                                    '00000000000000000000000000000000010000101000010000000000000000000000000000000000',
                                                                                    '00000000000000000001100000000000010000110000010000000000000000000000000000000000',
                                                                                    '00000000000000000000100000000000010000100100010000000000000000000000000000000000',
                                                                                    '00000000000000000000100000000000011111100010010000000000000000000000000000000000',
                                                                                    '00000000000000000000100000000000010000100010010000000000000000000000000000000000',
                                                                                    '00000000000000000000100000000000010000100000010000000000000000000000000000000000',
                                                                                    '00000000000000000000100000000000010000100000010000000000000000000000000000000000',
                                                                                    '00000000000000000000100000000000010000100000010000000000000000000000000000000000',
                                                                                    '00000000000000000011111000000000011111100000010000000000000000000000000000000000',
                                                                                    '00000000000000000000000000000000010000100010100000000000000000000000000000000000',
                                                                                    '00000000000000000000000000000000000000000001000000000000000000000000000000000000'])

    def test_draw_nonexistent_tb_linelimit40(self):
        self.assertEqual(self.font.draw('Bé H好Δi的', direction='tb', linelimit=40).bindata, ['000000000000000000000000000000000000000000000000',
                                                                                            '000000000000000000000000000000000000000000000000',
                                                                                            '000000000000000000000000000000000000000000000000',
                                                                                            '000010000000000000000000000000000000000000000000',
                                                                                            '000010000000000000000000000000000111110000000000',
                                                                                            '000000000000000000000000000000000100001000000000',
                                                                                            '000110000000000000000000000000000100001000000000',
                                                                                            '000010000000000000000000000000000100001000000000',
                                                                                            '000010000000000000000000000000000111110000000000',
                                                                                            '000010000000000000000000000000000100001000000000',
                                                                                            '000010000000000000000000000000000100001000000000',
                                                                                            '000010000000000000000000000000000100001000000000',
                                                                                            '000010000000000000000000000000000100001000000000',
                                                                                            '001111100000000000000000000000000111110000000000',
                                                                                            '000000000000000000000000000000000000000000000000',
                                                                                            '000000000000000000000000000000000000000000000000',
                                                                                            '000100000100000000000000000000000000000000000000',
                                                                                            '000100000100000000000000000000000000000000000000',
                                                                                            '001000000100000000000000000000000000110000000000',
                                                                                            '011111100111110000000000000000000011000000000000',
                                                                                            '010000101000010001000010000000000000000000000000',
                                                                                            '010000101000010001000010000000000000000000000000',
                                                                                            '010000110000010001000010000000000011110000000000',
                                                                                            '010000100100010001000010000000000100001000000000',
                                                                                            '011111100010010001111110000000000100001000000000',
                                                                                            '010000100010010001000010000000000111111000000000',
                                                                                            '010000100000010001000010000000000100000000000000',
                                                                                            '010000100000010001000010000000000100000000000000',
                                                                                            '010000100000010001000010000000000100001000000000',
                                                                                            '011111100000010001000010000000000011110000000000',
                                                                                            '010000100010100000000000000000000000000000000000',
                                                                                            '000000000001000000000000000000000000000000000000',
                                                                                            '000000000000000000000000000000000000000000000000',
                                                                                            '000000000000000000000000000000000000000000000000',
                                                                                            '000000000000000000000000000000000000000000000000',
                                                                                            '000000000000000000000000000000000000000000000000',
                                                                                            '000000000000000000000000000000000000000000000000',
                                                                                            '000000000000000000000000000000000000000000000000',
                                                                                            '000000000000000000000000000000000000000000000000',
                                                                                            '000000000000000000000000000000000000000000000000',
                                                                                            '000000000000000000000000000000000000000000000000',
                                                                                            '000000000000000000000000000000000000000000000000',
                                                                                            '000000000000000000000000000000000000000000000000',
                                                                                            '000000000000000000000000000000000000000000000000',
                                                                                            '000000000000000000000000000000000000000000000000',
                                                                                            '000000000000000000000000000000000000000000000000',
                                                                                            '000000000000000000000000000000000000000000000000',
                                                                                            '000000000000000000000000000000000000000000000000'])

    def test_draw_nonexistent_default_withmissingglyphobj(self):
        self.assertEqual(self.font.draw('Bé H好Δi的', missing=Glyph(missing_glyph_meta, self.font)).bindata,
                         ['0000000000000000000000000000000000000000000000000000000000000000000000000001000001000000',
                          '0000000000000000000000000000000000000000000000000000000000000000000000000001000001000000',
                          '0000000000001100000000000000000000000000000000000000000000000000000000000010000001000000',
                          '0000000000110000000000000000000000111111111110000011111111111000000010000111111001111100',
                          '0111110000000000000000000100001000110000000110000011000000011000000010000100001010000100',
                          '0100001000000000000000000100001000101000001010000010100000101000000000000100001010000100',
                          '0100001000111100000000000100001000100100010010000010010001001000000110000100001100000100',
                          '0100001001000010000000000100001000100010100010000010001010001000000010000100001001000100',
                          '0111110001000010000000000111111000100001000010000010000100001000000010000111111000100100',
                          '0100001001111110000000000100001000100010100010000010001010001000000010000100001000100100',
                          '0100001001000000000000000100001000100100010010000010010001001000000010000100001000000100',
                          '0100001001000000000000000100001000101000001010000010100000101000000010000100001000000100',
                          '0100001001000010000000000100001000110000000110000011000000011000000010000100001000000100',
                          '0111110000111100000000000100001000111111111110000011111111111000001111100111111000000100',
                          '0000000000000000000000000000000000000000000000000000000000000000000000000100001000101000',
                          '0000000000000000000000000000000000000000000000000000000000000000000000000000000000010000'])

    def test_draw_nonexistent_default_withmissingglyphmeta(self):
        self.assertEqual(self.font.draw('Bé H好Δi的', missing=missing_glyph_meta).bindata,
                         ['0000000000000000000000000000000000000000000000000000000000000000000000000001000001000000',
                          '0000000000000000000000000000000000000000000000000000000000000000000000000001000001000000',
                          '0000000000001100000000000000000000000000000000000000000000000000000000000010000001000000',
                          '0000000000110000000000000000000000111111111110000011111111111000000010000111111001111100',
                          '0111110000000000000000000100001000110000000110000011000000011000000010000100001010000100',
                          '0100001000000000000000000100001000101000001010000010100000101000000000000100001010000100',
                          '0100001000111100000000000100001000100100010010000010010001001000000110000100001100000100',
                          '0100001001000010000000000100001000100010100010000010001010001000000010000100001001000100',
                          '0111110001000010000000000111111000100001000010000010000100001000000010000111111000100100',
                          '0100001001111110000000000100001000100010100010000010001010001000000010000100001000100100',
                          '0100001001000000000000000100001000100100010010000010010001001000000010000100001000000100',
                          '0100001001000000000000000100001000101000001010000010100000101000000010000100001000000100',
                          '0100001001000010000000000100001000110000000110000011000000011000000010000100001000000100',
                          '0111110000111100000000000100001000111111111110000011111111111000001111100111111000000100',
                          '0000000000000000000000000000000000000000000000000000000000000000000000000100001000101000',
                          '0000000000000000000000000000000000000000000000000000000000000000000000000000000000010000'])

    def test_drawall(self):
        drawall_bitmap_bindata = self.font.drawall(linelimit=700).bindata
        self.assertEqual(len(drawall_bitmap_bindata[0]), 688)
        self.assertEqual(len(drawall_bitmap_bindata), 320)


# if __name__ == '__main__':
#     unittest.main()
