import unittest
from bdfparser import Font, Bitmap
from .info import specfont_path, bitmap_qr2_bindata, bitmap_qr3_bindata


# Test all `Bitmap` attributes and methods, with "a" and other glyphs in Unifont


class TestBitmap(unittest.TestCase):

    def setUp(self):
        self.maxDiff = None
        self.font = Font(specfont_path)
        self.bitmap_qr = self.font.glyph("'").draw(mode=2)
        self.bitmap_qr2 = Bitmap(bitmap_qr2_bindata)

    def test_bindata(self):
        self.assertEqual(self.bitmap_qr.bindata, ['01110000',
                                                  '01110000',
                                                  '01110000',
                                                  '01100000',
                                                  '11100000',
                                                  '11000000'])
        self.assertEqual(self.bitmap_qr2.bindata, ['01110',
                                                   '02112',
                                                   '01102',
                                                   '10200',
                                                   '01000'])

    def test_width(self):
        self.assertEqual(self.bitmap_qr.width(), 8)
        self.assertEqual(self.bitmap_qr2.width(), 5)

    def test_height(self):
        self.assertEqual(self.bitmap_qr.height(), 6)
        self.assertEqual(self.bitmap_qr2.height(), 5)

    def test_clone(self):
        self.assertNotEqual(self.bitmap_qr.clone(),
                            self.bitmap_qr)
        self.assertEqual(self.bitmap_qr.clone().bindata,
                         self.bitmap_qr.bindata)


class TestBitmapAlter(unittest.TestCase):

    def setUp(self):
        self.maxDiff = None
        self.font = Font(specfont_path)
        self.bitmap_qr = self.font.glyph("'").draw(mode=2)
        self.bitmap_qr2 = Bitmap(bitmap_qr2_bindata)

    def test_crop_default(self):
        self.assertEqual(self.bitmap_qr2.crop(6, 10).bindata, ['000000',
                                                               '000000',
                                                               '000000',
                                                               '000000',
                                                               '000000',
                                                               '011100',
                                                               '021120',
                                                               '011020',
                                                               '102000',
                                                               '010000'])

    def test_crop(self):
        self.assertEqual(self.bitmap_qr.crop(6, 10, -1, -2).bindata, ['000000',
                                                                      '000000',
                                                                      '001110',
                                                                      '001110',
                                                                      '001110',
                                                                      '001100',
                                                                      '011100',
                                                                      '011000',
                                                                      '000000',
                                                                      '000000'])

    def test_replace(self):
        self.assertEqual(self.bitmap_qr2.replace('2', '3').bindata, ['01110',
                                                                     '03113',
                                                                     '01103',
                                                                     '10300',
                                                                     '01000'])


class TestBitmapOverlay(unittest.TestCase):

    def setUp(self):
        self.maxDiff = None
        self.font = Font(specfont_path)
        self.bitmap_qr = self.font.glyph("'").draw(mode=2)
        self.bitmap_qr2 = Bitmap(bitmap_qr2_bindata)

    def test_overlay(self):
        self.assertEqual(
            self.bitmap_qr.overlay(
                self.bitmap_qr2.crop(self.bitmap_qr.width(),
                                     self.bitmap_qr.height())
            ).bindata, ['01110000',
                        '01110000',
                        '02112000',
                        '01102000',
                        '11200000',
                        '11000000'])


class TestBitmapConcat(unittest.TestCase):

    def setUp(self):
        self.maxDiff = None
        self.font = Font(specfont_path)
        self.bitmap_qr = self.font.glyph("'").draw(mode=2)
        self.bitmap_qr2 = Bitmap(bitmap_qr2_bindata)
        self.bitmap_j = self.font.glyph("j").draw(mode=2)

    def test_concatall_onlyone(self):
        self.assertEqual(Bitmap.concatall(
            [self.bitmap_qr]).bindata, self.bitmap_qr.bindata)

    def test_concatall(self):
        w = self.bitmap_qr.width()
        self.assertEqual(Bitmap.concatall(
            [self.bitmap_qr, self.bitmap_j, self.bitmap_qr2]).bindata, ['00000000000000111000000000000',
                                                                        '00000000000000111000000000000',
                                                                        '00000000000000111000000000000',
                                                                        '00000000000000111000000000000',
                                                                        '00000000000000000000000000000',
                                                                        '00000000000001110000000000000',
                                                                        '00000000000001110000000000000',
                                                                        '00000000000001110000000000000',
                                                                        '00000000000001110000000000000',
                                                                        '00000000000011100000000000000',
                                                                        '00000000000011100000000000000',
                                                                        '00000000000011100000000000000',
                                                                        '00000000000011100000000000000',
                                                                        '00000000000011100000000000000',
                                                                        '00000000000111000000000000000',
                                                                        '00000000000111000000000000000',
                                                                        '01110000000111000000000000000',
                                                                        '01110000000111000000000001110',
                                                                        '01110000001011000000000002112',
                                                                        '01100000011110000000000001102',
                                                                        '11100000111100000000000010200',
                                                                        '11000000111000000000000001000'])
        self.assertEqual(self.bitmap_qr.width(), w)

    def test_concatall_offsetlist(self):
        self.assertEqual(Bitmap.concatall([self.bitmap_qr, self.bitmap_j, self.bitmap_qr2], offsetlist=[-5, 4]).bindata, ['0000000001110000000000000000',
                                                                                                                          '0000000001110000000000000000',
                                                                                                                          '0000000001110000000000000000',
                                                                                                                          '0000000001110000000000000000',
                                                                                                                          '0000000000000000000000000000',
                                                                                                                          '0000000011100000000000000000',
                                                                                                                          '0000000011100000000000000000',
                                                                                                                          '0000000011100000000000000000',
                                                                                                                          '0000000011100000000000000000',
                                                                                                                          '0000000111000000000000000000',
                                                                                                                          '0000000111000000000000000000',
                                                                                                                          '0000000111000000000000000000',
                                                                                                                          '0000000111000000000000000000',
                                                                                                                          '0000000111000000000000000000',
                                                                                                                          '0000001110000000000000000000',
                                                                                                                          '0000001110000000000000000000',
                                                                                                                          '0111001110000000000000000000',
                                                                                                                          '0111001110000000000000001110',
                                                                                                                          '0111010110000000000000002112',
                                                                                                                          '0110111100000000000000001102',
                                                                                                                          '1111111000000000000000010200',
                                                                                                                          '1101110000000000000000001000'])

    def test_concatall_direction_2(self):
        self.assertEqual(Bitmap.concatall([self.bitmap_qr, self.bitmap_j, self.bitmap_qr2], direction=2).bindata, ['00000000000111000000000000000',
                                                                                                                   '00000000000111000000000000000',
                                                                                                                   '00000000000111000000000000000',
                                                                                                                   '00000000000111000000000000000',
                                                                                                                   '00000000000000000000000000000',
                                                                                                                   '00000000001110000000000000000',
                                                                                                                   '00000000001110000000000000000',
                                                                                                                   '00000000001110000000000000000',
                                                                                                                   '00000000001110000000000000000',
                                                                                                                   '00000000011100000000000000000',
                                                                                                                   '00000000011100000000000000000',
                                                                                                                   '00000000011100000000000000000',
                                                                                                                   '00000000011100000000000000000',
                                                                                                                   '00000000011100000000000000000',
                                                                                                                   '00000000111000000000000000000',
                                                                                                                   '00000000111000000000000000000',
                                                                                                                   '00000000111000000000001110000',
                                                                                                                   '01110000111000000000001110000',
                                                                                                                   '02112001011000000000001110000',
                                                                                                                   '01102011110000000000001100000',
                                                                                                                   '10200111100000000000011100000',
                                                                                                                   '01000111000000000000011000000'])

    def test_concatall_direction_0(self):
        self.assertEqual(Bitmap.concatall([self.bitmap_qr, self.bitmap_j, self.bitmap_qr2], direction=0).bindata, ['0111000000000000',
                                                                                                                   '0111000000000000',
                                                                                                                   '0111000000000000',
                                                                                                                   '0110000000000000',
                                                                                                                   '1110000000000000',
                                                                                                                   '1100000000000000',
                                                                                                                   '0000001110000000',
                                                                                                                   '0000001110000000',
                                                                                                                   '0000001110000000',
                                                                                                                   '0000001110000000',
                                                                                                                   '0000000000000000',
                                                                                                                   '0000011100000000',
                                                                                                                   '0000011100000000',
                                                                                                                   '0000011100000000',
                                                                                                                   '0000011100000000',
                                                                                                                   '0000111000000000',
                                                                                                                   '0000111000000000',
                                                                                                                   '0000111000000000',
                                                                                                                   '0000111000000000',
                                                                                                                   '0000111000000000',
                                                                                                                   '0001110000000000',
                                                                                                                   '0001110000000000',
                                                                                                                   '0001110000000000',
                                                                                                                   '0001110000000000',
                                                                                                                   '0010110000000000',
                                                                                                                   '0111100000000000',
                                                                                                                   '1111000000000000',
                                                                                                                   '1110000000000000',
                                                                                                                   '0111000000000000',
                                                                                                                   '0211200000000000',
                                                                                                                   '0110200000000000',
                                                                                                                   '1020000000000000',
                                                                                                                   '0100000000000000'])

    def test_concatall_direction_m1(self):
        self.assertEqual(Bitmap.concatall([self.bitmap_qr, self.bitmap_j, self.bitmap_qr2], direction=-1).bindata, ['0111000000000000',
                                                                                                                    '0211200000000000',
                                                                                                                    '0110200000000000',
                                                                                                                    '1020000000000000',
                                                                                                                    '0100000000000000',
                                                                                                                    '0000001110000000',
                                                                                                                    '0000001110000000',
                                                                                                                    '0000001110000000',
                                                                                                                    '0000001110000000',
                                                                                                                    '0000000000000000',
                                                                                                                    '0000011100000000',
                                                                                                                    '0000011100000000',
                                                                                                                    '0000011100000000',
                                                                                                                    '0000011100000000',
                                                                                                                    '0000111000000000',
                                                                                                                    '0000111000000000',
                                                                                                                    '0000111000000000',
                                                                                                                    '0000111000000000',
                                                                                                                    '0000111000000000',
                                                                                                                    '0001110000000000',
                                                                                                                    '0001110000000000',
                                                                                                                    '0001110000000000',
                                                                                                                    '0001110000000000',
                                                                                                                    '0010110000000000',
                                                                                                                    '0111100000000000',
                                                                                                                    '1111000000000000',
                                                                                                                    '1110000000000000',
                                                                                                                    '0111000000000000',
                                                                                                                    '0111000000000000',
                                                                                                                    '0111000000000000',
                                                                                                                    '0110000000000000',
                                                                                                                    '1110000000000000',
                                                                                                                    '1100000000000000'])

    def test_concatall_align_0(self):
        self.assertEqual(Bitmap.concatall([self.bitmap_qr, self.bitmap_j, self.bitmap_qr2], align=0).bindata, ['01110000000000111000000001110',
                                                                                                               '01110000000000111000000002112',
                                                                                                               '01110000000000111000000001102',
                                                                                                               '01100000000000111000000010200',
                                                                                                               '11100000000000000000000001000',
                                                                                                               '11000000000001110000000000000',
                                                                                                               '00000000000001110000000000000',
                                                                                                               '00000000000001110000000000000',
                                                                                                               '00000000000001110000000000000',
                                                                                                               '00000000000011100000000000000',
                                                                                                               '00000000000011100000000000000',
                                                                                                               '00000000000011100000000000000',
                                                                                                               '00000000000011100000000000000',
                                                                                                               '00000000000011100000000000000',
                                                                                                               '00000000000111000000000000000',
                                                                                                               '00000000000111000000000000000',
                                                                                                               '00000000000111000000000000000',
                                                                                                               '00000000000111000000000000000',
                                                                                                               '00000000001011000000000000000',
                                                                                                               '00000000011110000000000000000',
                                                                                                               '00000000111100000000000000000',
                                                                                                               '00000000111000000000000000000'])

    def test_concatall_direction_2_align_0(self):
        self.assertEqual(Bitmap.concatall([self.bitmap_qr, self.bitmap_j, self.bitmap_qr2], direction=2, align=0).bindata, ['01110000000111000000001110000',
                                                                                                                            '02112000000111000000001110000',
                                                                                                                            '01102000000111000000001110000',
                                                                                                                            '10200000000111000000001100000',
                                                                                                                            '01000000000000000000011100000',
                                                                                                                            '00000000001110000000011000000',
                                                                                                                            '00000000001110000000000000000',
                                                                                                                            '00000000001110000000000000000',
                                                                                                                            '00000000001110000000000000000',
                                                                                                                            '00000000011100000000000000000',
                                                                                                                            '00000000011100000000000000000',
                                                                                                                            '00000000011100000000000000000',
                                                                                                                            '00000000011100000000000000000',
                                                                                                                            '00000000011100000000000000000',
                                                                                                                            '00000000111000000000000000000',
                                                                                                                            '00000000111000000000000000000',
                                                                                                                            '00000000111000000000000000000',
                                                                                                                            '00000000111000000000000000000',
                                                                                                                            '00000001011000000000000000000',
                                                                                                                            '00000011110000000000000000000',
                                                                                                                            '00000111100000000000000000000',
                                                                                                                            '00000111000000000000000000000'])

    def test_concatall_direction_0_align_0(self):
        self.assertEqual(Bitmap.concatall([self.bitmap_qr, self.bitmap_j, self.bitmap_qr2], direction=0, align=0).bindata, ['0000000001110000',
                                                                                                                            '0000000001110000',
                                                                                                                            '0000000001110000',
                                                                                                                            '0000000001100000',
                                                                                                                            '0000000011100000',
                                                                                                                            '0000000011000000',
                                                                                                                            '0000001110000000',
                                                                                                                            '0000001110000000',
                                                                                                                            '0000001110000000',
                                                                                                                            '0000001110000000',
                                                                                                                            '0000000000000000',
                                                                                                                            '0000011100000000',
                                                                                                                            '0000011100000000',
                                                                                                                            '0000011100000000',
                                                                                                                            '0000011100000000',
                                                                                                                            '0000111000000000',
                                                                                                                            '0000111000000000',
                                                                                                                            '0000111000000000',
                                                                                                                            '0000111000000000',
                                                                                                                            '0000111000000000',
                                                                                                                            '0001110000000000',
                                                                                                                            '0001110000000000',
                                                                                                                            '0001110000000000',
                                                                                                                            '0001110000000000',
                                                                                                                            '0010110000000000',
                                                                                                                            '0111100000000000',
                                                                                                                            '1111000000000000',
                                                                                                                            '1110000000000000',
                                                                                                                            '0000000000001110',
                                                                                                                            '0000000000002112',
                                                                                                                            '0000000000001102',
                                                                                                                            '0000000000010200',
                                                                                                                            '0000000000001000'])

    def test_concatall_direction_m1_align_0(self):
        self.assertEqual(Bitmap.concatall([self.bitmap_qr, self.bitmap_j, self.bitmap_qr2], direction=-1, align=0).bindata, ['0000000000001110',
                                                                                                                             '0000000000002112',
                                                                                                                             '0000000000001102',
                                                                                                                             '0000000000010200',
                                                                                                                             '0000000000001000',
                                                                                                                             '0000001110000000',
                                                                                                                             '0000001110000000',
                                                                                                                             '0000001110000000',
                                                                                                                             '0000001110000000',
                                                                                                                             '0000000000000000',
                                                                                                                             '0000011100000000',
                                                                                                                             '0000011100000000',
                                                                                                                             '0000011100000000',
                                                                                                                             '0000011100000000',
                                                                                                                             '0000111000000000',
                                                                                                                             '0000111000000000',
                                                                                                                             '0000111000000000',
                                                                                                                             '0000111000000000',
                                                                                                                             '0000111000000000',
                                                                                                                             '0001110000000000',
                                                                                                                             '0001110000000000',
                                                                                                                             '0001110000000000',
                                                                                                                             '0001110000000000',
                                                                                                                             '0010110000000000',
                                                                                                                             '0111100000000000',
                                                                                                                             '1111000000000000',
                                                                                                                             '1110000000000000',
                                                                                                                             '0000000001110000',
                                                                                                                             '0000000001110000',
                                                                                                                             '0000000001110000',
                                                                                                                             '0000000001100000',
                                                                                                                             '0000000011100000',
                                                                                                                             '0000000011000000'])

    def test_plus(self):
        w = self.bitmap_qr.width()
        self.assertEqual((self.bitmap_qr + self.bitmap_j).bindata, ['000000000000001110000000',
                                                                    '000000000000001110000000',
                                                                    '000000000000001110000000',
                                                                    '000000000000001110000000',
                                                                    '000000000000000000000000',
                                                                    '000000000000011100000000',
                                                                    '000000000000011100000000',
                                                                    '000000000000011100000000',
                                                                    '000000000000011100000000',
                                                                    '000000000000111000000000',
                                                                    '000000000000111000000000',
                                                                    '000000000000111000000000',
                                                                    '000000000000111000000000',
                                                                    '000000000000111000000000',
                                                                    '000000000001110000000000',
                                                                    '000000000001110000000000',
                                                                    '011100000001110000000000',
                                                                    '011100000001110000000000',
                                                                    '011100000010110000000000',
                                                                    '011000000111100000000000',
                                                                    '111000001111000000000000',
                                                                    '110000001110000000000000'])
        self.assertEqual(self.bitmap_qr.width(), w)

    def test_concat(self):
        w = self.bitmap_qr.width()
        w2 = self.bitmap_j.width()
        self.assertEqual(self.bitmap_qr.concat(self.bitmap_j).bindata, ['000000000000001110000000',
                                                                        '000000000000001110000000',
                                                                        '000000000000001110000000',
                                                                        '000000000000001110000000',
                                                                        '000000000000000000000000',
                                                                        '000000000000011100000000',
                                                                        '000000000000011100000000',
                                                                        '000000000000011100000000',
                                                                        '000000000000011100000000',
                                                                        '000000000000111000000000',
                                                                        '000000000000111000000000',
                                                                        '000000000000111000000000',
                                                                        '000000000000111000000000',
                                                                        '000000000000111000000000',
                                                                        '000000000001110000000000',
                                                                        '000000000001110000000000',
                                                                        '011100000001110000000000',
                                                                        '011100000001110000000000',
                                                                        '011100000010110000000000',
                                                                        '011000000111100000000000',
                                                                        '111000001111000000000000',
                                                                        '110000001110000000000000'])
        self.assertEqual(self.bitmap_qr.width(), w + w2)


class TestBitmapEnlarge(unittest.TestCase):

    def setUp(self):
        self.maxDiff = None
        self.font = Font(specfont_path)
        self.bitmap_qr = self.font.glyph("'").draw(mode=2)
        self.bitmap_qr2 = Bitmap(bitmap_qr2_bindata)

    def test_enlarge(self):
        self.assertEqual(self.bitmap_qr2.enlarge(2, 3).bindata, ['0011111100',
                                                                 '0011111100',
                                                                 '0011111100',
                                                                 '0022111122',
                                                                 '0022111122',
                                                                 '0022111122',
                                                                 '0011110022',
                                                                 '0011110022',
                                                                 '0011110022',
                                                                 '1100220000',
                                                                 '1100220000',
                                                                 '1100220000',
                                                                 '0011000000',
                                                                 '0011000000',
                                                                 '0011000000'])

    def test_asterisk(self):
        w = self.bitmap_qr.width()
        self.assertEqual((self.bitmap_qr * 3).bindata,
                         self.bitmap_qr.enlarge(3, 3).bindata)
        self.assertEqual(self.bitmap_qr.width(), w * 3)

    def test_asterisk2(self):
        self.assertEqual((self.bitmap_qr * (2, 3)).bindata,
                         self.bitmap_qr.enlarge(2, 3).bindata)


class TestBitmapEffect(unittest.TestCase):

    def setUp(self):
        self.maxDiff = None
        self.font = Font(specfont_path)
        self.bitmap_qr = self.font.glyph("'").draw(mode=2)

    def test_shadow(self):
        self.assertEqual(self.bitmap_qr.shadow(2, -3).bindata, ['0111000000',
                                                                '0111000000',
                                                                '0111000000',
                                                                '0112220000',
                                                                '1112220000',
                                                                '1102220000',
                                                                '0002200000',
                                                                '0022200000',
                                                                '0022000000'])

    def test_glow0(self):
        self.assertEqual(self.bitmap_qr.glow().bindata, ['0022200000',
                                                         '0211120000',
                                                         '0211120000',
                                                         '0211120000',
                                                         '0211200000',
                                                         '2111200000',
                                                         '2112000000',
                                                         '0220000000'])

    def test_glow1(self):
        self.assertEqual(self.bitmap_qr.glow(1).bindata, ['0222220000',
                                                          '0211120000',
                                                          '0211120000',
                                                          '0211120000',
                                                          '2211220000',
                                                          '2111200000',
                                                          '2112200000',
                                                          '2222000000'])


class TestBitmapPad(unittest.TestCase):

    def setUp(self):
        self.maxDiff = None
        self.bitmap_qr2 = Bitmap(bitmap_qr2_bindata)
        self.bitmap_qr3 = Bitmap(bitmap_qr3_bindata)

    def test_bytepad(self):
        self.assertEqual(self.bitmap_qr2.bytepad().bindata, ['01110000',
                                                             '02112000',
                                                             '01102000',
                                                             '10200000',
                                                             '01000000'])

    def test_bytepad4(self):
        self.assertEqual(self.bitmap_qr3.bytepad(4).bindata, ['011100000000',
                                                              '021120000000',
                                                              '011020003000',
                                                              '102000002100',
                                                              '010000000000'])


class TestBitmapTodata(unittest.TestCase):

    def setUp(self):
        self.maxDiff = None
        self.font = Font(specfont_path)
        self.bitmap_todata_test = Bitmap(['00010',
                                          '11010',
                                          '00201'])
        self.bitmap_todata_test2 = Bitmap(['00010',
                                           '11010'])

    def test_todata0(self):
        self.assertEqual(self.bitmap_todata_test.todata(0), '00010\n'
                                                            '11010\n'
                                                            '00201')

    def test_todata1_default(self):
        self.assertEqual(self.bitmap_todata_test.todata(), ['00010',
                                                            '11010',
                                                            '00201'])

    def test_todata2(self):
        self.assertEqual(self.bitmap_todata_test.todata(2), [[0, 0, 0, 1, 0],
                                                             [1, 1, 0, 1, 0],
                                                             [0, 0, 2, 0, 1]])

    def test_todata3(self):
        self.assertEqual(self.bitmap_todata_test.todata(3),
                         [0, 0, 0, 1, 0, 1, 1, 0, 1, 0, 0, 0, 2, 0, 1])

    def test_todata4(self):
        self.assertEqual(self.bitmap_todata_test2.todata(4), ['02', '1a'])

    def test_todata4_error(self):
        def todata4_error():
            self.bitmap_todata_test.todata(4)
        self.assertRaises(ValueError, todata4_error)

    def test_todata5(self):
        self.assertEqual(self.bitmap_todata_test2.todata(5), [2, 26])

    def test_todata5_error(self):
        def todata5_error():
            self.bitmap_todata_test.todata(5)
        self.assertRaises(ValueError, todata5_error)


class TestBitmapTobytes(unittest.TestCase):

    def setUp(self):
        self.maxDiff = None
        self.font = Font(specfont_path)
        self.bitmap_qr = self.font.glyph("'").draw(mode=2)
        self.bitmap_qr2 = Bitmap(bitmap_qr2_bindata)

    def test_tobytes_1(self):
        self.assertEqual(self.bitmap_qr.tobytes(
            mode='1'), b'\x8f\x8f\x8f\x9f\x1f?')
        self.assertEqual(self.bitmap_qr2.tobytes(
            mode='1'), b'\x8f\x87\x97_\xbf')

    def test_tobytes_L(self):
        self.assertEqual(self.bitmap_qr.tobytes(
            mode='L'), b'\xff\x00\x00\x00\xff\xff\xff\xff\xff\x00\x00\x00\xff\xff\xff\xff\xff\x00\x00\x00\xff\xff\xff\xff\xff\x00\x00\xff\xff\xff\xff\xff\x00\x00\x00\xff\xff\xff\xff\xff\x00\x00\xff\xff\xff\xff\xff\xff')
        self.assertEqual(self.bitmap_qr2.tobytes(
            mode='L'), b'\xff\x00\x00\x00\xff\xff\x7f\x00\x00\x7f\xff\x00\x00\xff\x7f\x00\xff\x7f\xff\xff\xff\x00\xff\xff\xff')

    def test_tobytes_RGB_default(self):
        self.assertEqual(self.bitmap_qr.tobytes(), b'\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x00\x00\x00\x00\x00\x00\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x00\x00\x00\x00\x00\x00\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff')
        self.assertEqual(self.bitmap_qr2.tobytes(), b'\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\xff\xff\xff\xff\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\xff\x00\x00\xff\xff\xff\x00\x00\x00\x00\x00\x00\xff\xff\xff\xff\x00\x00\x00\x00\x00\xff\xff\xff\xff\x00\x00\xff\xff\xff\xff\xff\xff\xff\xff\xff\x00\x00\x00\xff\xff\xff\xff\xff\xff\xff\xff\xff')

    def test_tobytes_RGBA(self):
        self.assertEqual(self.bitmap_qr.tobytes('RGBA'), b'\xff\xff\xff\x00\x00\x00\x00\xff\x00\x00\x00\xff\x00\x00\x00\xff\xff\xff\xff\x00\xff\xff\xff\x00\xff\xff\xff\x00\xff\xff\xff\x00\xff\xff\xff\x00\x00\x00\x00\xff\x00\x00\x00\xff\x00\x00\x00\xff\xff\xff\xff\x00\xff\xff\xff\x00\xff\xff\xff\x00\xff\xff\xff\x00\xff\xff\xff\x00\x00\x00\x00\xff\x00\x00\x00\xff\x00\x00\x00\xff\xff\xff\xff\x00\xff\xff\xff\x00\xff\xff\xff\x00\xff\xff\xff\x00\xff\xff\xff\x00\x00\x00\x00\xff\x00\x00\x00\xff\xff\xff\xff\x00\xff\xff\xff\x00\xff\xff\xff\x00\xff\xff\xff\x00\xff\xff\xff\x00\x00\x00\x00\xff\x00\x00\x00\xff\x00\x00\x00\xff\xff\xff\xff\x00\xff\xff\xff\x00\xff\xff\xff\x00\xff\xff\xff\x00\xff\xff\xff\x00\x00\x00\x00\xff\x00\x00\x00\xff\xff\xff\xff\x00\xff\xff\xff\x00\xff\xff\xff\x00\xff\xff\xff\x00\xff\xff\xff\x00\xff\xff\xff\x00')
        self.assertEqual(self.bitmap_qr2.tobytes('RGBA'), b'\xff\xff\xff\x00\x00\x00\x00\xff\x00\x00\x00\xff\x00\x00\x00\xff\xff\xff\xff\x00\xff\xff\xff\x00\xff\x00\x00\xff\x00\x00\x00\xff\x00\x00\x00\xff\xff\x00\x00\xff\xff\xff\xff\x00\x00\x00\x00\xff\x00\x00\x00\xff\xff\xff\xff\x00\xff\x00\x00\xff\x00\x00\x00\xff\xff\xff\xff\x00\xff\x00\x00\xff\xff\xff\xff\x00\xff\xff\xff\x00\xff\xff\xff\x00\x00\x00\x00\xff\xff\xff\xff\x00\xff\xff\xff\x00\xff\xff\xff\x00')


class TestBitmapStrRepr(unittest.TestCase):

    def setUp(self):
        self.maxDiff = None
        self.font = Font(specfont_path)
        self.bitmap_qr = self.font.glyph("'").draw(mode=2)

    def test_str(self):
        self.assertEqual(str(self.bitmap_qr),
                         '.###....\n'
                         '.###....\n'
                         '.###....\n'
                         '.##.....\n'
                         '###.....\n'
                         '##......')

    def test_repr(self):
        self.assertEqual(repr(self.bitmap_qr),
                         "Bitmap(['01110000',\n"
                         "        '01110000',\n"
                         "        '01110000',\n"
                         "        '01100000',\n"
                         "        '11100000',\n"
                         "        '11000000'])")


# if __name__ == '__main__':
#     unittest.main()
