(* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
                         -****************+.
               .-=++*+::*###############*-
           :=+##*+=-::*##**############=
        .=*#*=:    .+#*=+############+.
       =##+.      +#+-=############*-:
     :*#+.      =+-.=#############- =#*.
    -##-      -=..+#############*:.  ---.
   -##:     .: .+################= :+#*-:.
  .##-        :==========*#####= :*#*- -#*
  =##                   -####*:-*#*-   .##:
  +#+                  -###*--*#*-      *#=
  *#+                 +###==*#*:        *#=
  +#*                *##+=*#*-          ##-
  :##.             .*##*##*:           -##.
   +#*            -#####*:             *#=
    *#+          =####+:             .*#+
     +#*.       +###+.              :##=
      -##+.    *##+:              :+#*:
        =*#+..*#+:             .-*#*-
          :-:#+::.        .:-=*##+:
           -+..+*###****####*+-.
          :.      ..:::::..
        ____                   _
       / ___| _ __   __ _ _ __| | __
       \___ \| '_ \ / _` | '__| |/ /
        ___) | |_) | (_| | |  |   <
       |____/| .__/ \__,_|_|  |_|\_\
             |_|   Game Toolkit

Copyright  2024-present tinyBigGAMES LLC
         All Rights Reserved.

Website: https://tinybiggames.com
Email  : support@tinybiggames.com

See LICENSE for license information
* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *)

unit UVideo02;

interface

uses
  System.SysUtils,
  SGT.Deps,
  SGT.Deps.Ext,
  SGT.OGL,
  SGT.Core,
  SGT.Lua,
  SGT.LuaDebugger,
  UCommon;

procedure Demo();

implementation

// The 'Demo' procedure demonstrates sequential video playback within a window using a zip file resource.
// It initializes the window, audio, and video playback, and cycles through multiple videos until the window is closed.

procedure Demo();
const
  // Array of video filenames to be played sequentially from the zip file.
  CVideoFilename: array[0..2] of string = ('tinyBigGAMES.mpg', 'SGT.mpg', 'sample01.mpg');
var
  LWindow: TWindow;              // Represents the window where the video and text will be displayed.
  LFont: TFont;                  // Represents the font used to render text on the window.
  LPos: TPoint;                  // Represents the position on the window where the text will be drawn.
  LVideoNum: Integer;            // Index of the currently playing video in the CVideoFilename array.

  // Procedure to play the current video based on LVideoNum index.
  procedure PlayVideo();
  begin
    // Start playing the video with specified filename and settings (volume: 0.8, looping: False).
    Video.Play(TZipFileIO.Open(CZipFilename, 'res/videos/' + CVideoFilename[LVideoNum]), 0.8, False);
  end;

begin
  // Initialize the window with the title 'SGT: Video #02'.
  LWindow := TWindow.Init('SGT: Video #02');

  // Load the default font with a size of 10, associated with the window.
  LFont := TFont.LoadDefault(LWindow, 10);

  // Initialize the audio system.
  Audio.Open();

  // Start with the first video.
  LVideoNum := 0;
  PlayVideo();

  // Main loop to keep the window running until the user closes it.
  while not LWindow.ShouldClose() do
  begin
    // Start processing a new frame.
    LWindow.StartFrame();

    // Check if the ESC key was pressed. If so, set the window to close.
    if LWindow.GetKey(KEY_ESCAPE, isWasPressed) then
      LWindow.SetShouldClose(True);

    // Check if the current video has stopped playing.
    if Video.GetStatus() = vsStopped then
    begin
      // Move to the next video in the array.
      Inc(LVideoNum);
      // Loop back to the first video if the end of the array is reached.
      if LVideoNum > 2 then
        LVideoNum := 0;
      // Play the new video.
      PlayVideo();
    end;

    // Start drawing on the window.
    LWindow.StartDrawing();

      // Clear the window with a dark slate brown background color.
      LWindow.Clear(DARKSLATEBROWN);

      // Draw the video on the window at position (0, 0) with a scale factor of 0.5.
      Video.Draw(0, 0, 0.5);

      // Set the initial position for text rendering.
      LPos := Math.Point(3, 3);

      // Draw the current frame rate in the top-left corner of the window in white color.
      LFont.DrawText(LWindow, LPos.X, LPos.Y, 0, WHITE, haLeft, 'fps %d', [FrameLimitTimer.FrameRate()]);

      // Draw instructions for quitting the application.
      LFont.DrawText(LWindow, LPos.X, LPos.Y, 0, GREEN, haLeft, Utils.HudTextItem('Quit', 'ESC'), [FrameLimitTimer.FrameRate()]);

    // Finish drawing on the window.
    LWindow.EndDrawing();

    // End processing the frame.
    LWindow.EndFrame();
  end;

  // Stop video playback when the loop ends.
  Video.Stop();

  // Close the audio system after the loop ends.
  Audio.Close();

  // Free the font resources after the loop ends.
  LFont.Free();

  // Free the window resources after the loop ends.
  LWindow.Free();
end;


end.
