(* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
                         -****************+.
               .-=++*+::*###############*-
           :=+##*+=-::*##**############=
        .=*#*=:    .+#*=+############+.
       =##+.      +#+-=############*-:
     :*#+.      =+-.=#############- =#*.
    -##-      -=..+#############*:.  ---.
   -##:     .: .+################= :+#*-:.
  .##-        :==========*#####= :*#*- -#*
  =##                   -####*:-*#*-   .##:
  +#+                  -###*--*#*-      *#=
  *#+                 +###==*#*:        *#=
  +#*                *##+=*#*-          ##-
  :##.             .*##*##*:           -##.
   +#*            -#####*:             *#=
    *#+          =####+:             .*#+
     +#*.       +###+.              :##=
      -##+.    *##+:              :+#*:
        =*#+..*#+:             .-*#*-
          :-:#+::.        .:-=*##+:
           -+..+*###****####*+-.
          :.      ..:::::..
        ____                   _
       / ___| _ __   __ _ _ __| | __
       \___ \| '_ \ / _` | '__| |/ /
        ___) | |_) | (_| | |  |   <
       |____/| .__/ \__,_|_|  |_|\_\
             |_|   Game Toolkit

Copyright  2024-present tinyBigGAMES LLC
         All Rights Reserved.

Website: https://tinybiggames.com
Email  : support@tinybiggames.com

See LICENSE for license information
* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *)

unit UTimer01;

interface

uses
  SysUtils,
  SGT.Deps,
  SGT.Deps.Ext,
  SGT.OGL,
  SGT.Core,
  SGT.Lua,
  SGT.LuaDebugger,
  UCommon;

procedure Demo();

implementation

// The 'Demo' procedure demonstrates a window with a timer and color-changing square.
procedure Demo();
var
  LWindow: TWindow;      // Manages the window for rendering.
  LFont: TFont;          // Manages the font used for rendering text.
  LPos: TPoint;          // Position for rendering text.
  LTimer: TTimer;        // Timer used to control the frame rate.
  LCount: Integer;       // Counter to track color changes.
begin
  // Initialize the window with the title 'SGT: Timer #01'.
  LWindow := TWindow.Init('SGT: Timer #01');

  // Load the default font with size 10.
  LFont := TFont.LoadDefault(LWindow, 10);

  // Initialize the timer with a frame rate of 8 FPS.
  LTimer.InitFPS(8);
  LCount := 0;

  // Main loop to keep the window running until the user closes it.
  while not LWindow.ShouldClose() do
  begin
    // Start processing a new frame.
    LWindow.StartFrame();

      // Check if the ESC key was pressed. If so, set the window to close.
      if Lwindow.GetKey(KEY_ESCAPE, isWasPressed) then
        LWindow.SetShouldClose(True);

      // Check if it's time to update the counter based on the timer.
      if LTimer.Check() then
      begin
        Inc(LCount); // Increment the counter.
        LCount := Math.ClipValue(LCount, 0, 3, True); // Clip the value to the range [0, 3].
      end;

      // Start drawing on the window.
      LWindow.StartDrawing();

        // Clear the window with a DARKSLATEBROWN color.
        LWindow.Clear(DARKSLATEBROWN);

        // Draw a square with a color based on the counter value.
        case LCount of
          0: LWindow.DrawFilledRect(LWindow.DEFAULT_CENTER_WIDTH, LWindow.DEFAULT_CENTER_HEIGHT, 50, 50, DARKSEAGREEN, 0);
          1: LWindow.DrawFilledRect(LWindow.DEFAULT_CENTER_WIDTH, LWindow.DEFAULT_CENTER_HEIGHT, 50, 50, FORESTGREEN, 0);
          2: LWindow.DrawFilledRect(LWindow.DEFAULT_CENTER_WIDTH, LWindow.DEFAULT_CENTER_HEIGHT, 50, 50, GREEN, 0);
          3: LWindow.DrawFilledRect(LWindow.DEFAULT_CENTER_WIDTH, LWindow.DEFAULT_CENTER_HEIGHT, 50, 50, DARKGREEN, 0);
        end;

        // Set the initial position for text rendering.
        LPos := Math.Point(3, 3);

        // Draw the current frame rate and quit instruction using the default font.
        LFont.DrawText(LWindow, LPos.X, LPos.Y, 0, WHITE, haLeft, 'fps %d', [FrameLimitTimer.FrameRate()]);
        LFont.DrawText(LWindow, LPos.X, LPos.Y, 0, GREEN, haLeft, Utils.HudTextItem('Quit', 'ESC'), [FrameLimitTimer.FrameRate()]);

      // Finish drawing on the window.
      LWindow.EndDrawing();

    // End processing the frame.
    LWindow.EndFrame();
  end;

  // Free the font resources after the loop ends.
  LFont.Free();

  // Free the window resources after the loop ends.
  LWindow.Free();
end;


end.
