(* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
                         -****************+.
               .-=++*+::*###############*-
           :=+##*+=-::*##**############=
        .=*#*=:    .+#*=+############+.
       =##+.      +#+-=############*-:
     :*#+.      =+-.=#############- =#*.
    -##-      -=..+#############*:.  ---.
   -##:     .: .+################= :+#*-:.
  .##-        :==========*#####= :*#*- -#*
  =##                   -####*:-*#*-   .##:
  +#+                  -###*--*#*-      *#=
  *#+                 +###==*#*:        *#=
  +#*                *##+=*#*-          ##-
  :##.             .*##*##*:           -##.
   +#*            -#####*:             *#=
    *#+          =####+:             .*#+
     +#*.       +###+.              :##=
      -##+.    *##+:              :+#*:
        =*#+..*#+:             .-*#*-
          :-:#+::.        .:-=*##+:
           -+..+*###****####*+-.
          :.      ..:::::..
        ____                   _
       / ___| _ __   __ _ _ __| | __
       \___ \| '_ \ / _` | '__| |/ /
        ___) | |_) | (_| | |  |   <
       |____/| .__/ \__,_|_|  |_|\_\
             |_|   Game Toolkit

Copyright  2024-present tinyBigGAMES LLC
         All Rights Reserved.

Website: https://tinybiggames.com
Email  : support@tinybiggames.com

See LICENSE for license information
* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *)

unit UTexture02;


interface

uses
  System.SysUtils,
  SGT.Deps,
  SGT.Deps.Ext,
  SGT.OGL,
  SGT.Core,
  SGT.Lua,
  SGT.LuaDebugger,
  UCommon;

procedure Demo();

implementation

// The 'Demo' procedure demonstrates rendering and animating multiple textures in a window.

procedure Demo();
var
  LZipFile: TZipFile;           // Represents the ZIP file containing texture resources.
  LWindow: TWindow;             // Represents the window for rendering textures.
  LFont: TFont;                 // Represents the font used for displaying text.
  LPos: TPoint;                 // Represents the position on the window for text rendering.
  FTexture: array[0..3] of TTexture; // Array of textures loaded from the ZIP file.
  FTexPos: array[0..3] of TPoint;    // Array of positions for each texture.
  FSpeed: array[0..3] of Single;     // Array of speeds for texture movement.
  I: Integer;                   // Loop variable for iterating through textures.
begin
  // Initialize the ZIP file with the given filename.
  LZipFile := TZipFile.Init(CZipFilename);

  // Initialize the window with the title 'SGT: Texture #02'.
  LWindow := TWindow.Init('SGT: Texture #02');

  // Load the default font with a size of 10, associated with the window.
  LFont := TFont.LoadDefault(LWindow, 10);

  // Load textures from the ZIP file with different settings.
  FTexture[0] := TTexture.LoadFromZipFile(LZipFile, 'res/backgrounds/space.png', nil);
  FTexture[1] := TTexture.LoadFromZipFile(LZipFile, 'res/backgrounds/nebula1.png', @BLACK);
  FTexture[2] := TTexture.LoadFromZipFile(LZipFile, 'res/backgrounds/spacelayer1.png', @BLACK);
  FTexture[3] := TTexture.LoadFromZipFile(LZipFile, 'res/backgrounds/spacelayer2.png', @BLACK);

  // Set color and blend mode for each texture.
  FTexture[0].SetColor(0.3, 0.3, 0.3, 0.3);
  FTexture[0].SetBlend(tbNone);
  FTexture[1].SetBlend(tbAdditiveAlpha);
  FTexture[2].SetBlend(tbAlpha);
  FTexture[3].SetBlend(tbAlpha);

  // Initialize positions for each texture.
  FTexPos[0] := Math.Point(0, 0);
  FTexPos[1] := Math.Point(0, 0);
  FTexPos[2] := Math.Point(0, 0);
  FTexPos[3] := Math.Point(0, 0);

  // Initialize movement speeds for each texture.
  FSpeed[0] := 0.1;
  FSpeed[1] := 0.15;
  FSpeed[2] := 0.2;
  FSpeed[3] := 0.3;

  // Main loop to keep the window running until the user closes it.
  while not LWindow.ShouldClose() do
  begin
    // Start processing a new frame.
    LWindow.StartFrame();

    // Check if the ESC key was pressed. If so, set the window to close.
    if LWindow.GetKey(KEY_ESCAPE, isWasPressed) then
      LWindow.SetShouldClose(True);

    // Update texture positions based on their speeds.
    for I := 0 to 3 do
    begin
      FTexPos[I].Y := FTexPos[I].Y + FSpeed[I];
    end;

    // Start drawing on the window.
    LWindow.StartDrawing();

      // Clear the window with a black background color.
      LWindow.Clear(BLACK);

      // Draw each texture tiled across the window with updated positions.
      for I := 0 to 3 do
      begin
        FTexture[I].DrawTiled(LWindow, FTexPos[I].X, FTexPos[I].Y);
      end;

      // Set the initial position for text rendering.
      LPos := Math.Point(3, 3);

      // Draw the current frame rate in the top-left corner of the window using the default font.
      LFont.DrawText(LWindow, LPos.X, LPos.Y, 0, WHITE, haLeft, 'fps %d', [FrameLimitTimer.FrameRate()]);

      // Draw the quit instruction in the top-left corner of the window using the default font.
      LFont.DrawText(LWindow, LPos.X, LPos.Y, 0, GREEN, haLeft, Utils.HudTextItem('Quit', 'ESC'), [FrameLimitTimer.FrameRate()]);

    // Finish drawing on the window.
    LWindow.EndDrawing();

    // End processing the frame.
    LWindow.EndFrame();
  end;

  // Free texture resources in reverse order of initialization.
  FTexture[3].Free();
  FTexture[2].Free();
  FTexture[1].Free();
  FTexture[0].Free();

  // Free the font resources after the loop ends.
  LFont.Free();

  // Free the window resources after the loop ends.
  LWindow.Free();

  // Free the ZIP file resources after the loop ends.
  LZipFile.Free();
end;


end.
