(* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
                         -****************+.
               .-=++*+::*###############*-
           :=+##*+=-::*##**############=
        .=*#*=:    .+#*=+############+.
       =##+.      +#+-=############*-:
     :*#+.      =+-.=#############- =#*.
    -##-      -=..+#############*:.  ---.
   -##:     .: .+################= :+#*-:.
  .##-        :==========*#####= :*#*- -#*
  =##                   -####*:-*#*-   .##:
  +#+                  -###*--*#*-      *#=
  *#+                 +###==*#*:        *#=
  +#*                *##+=*#*-          ##-
  :##.             .*##*##*:           -##.
   +#*            -#####*:             *#=
    *#+          =####+:             .*#+
     +#*.       +###+.              :##=
      -##+.    *##+:              :+#*:
        =*#+..*#+:             .-*#*-
          :-:#+::.        .:-=*##+:
           -+..+*###****####*+-.
          :.      ..:::::..
        ____                   _
       / ___| _ __   __ _ _ __| | __
       \___ \| '_ \ / _` | '__| |/ /
        ___) | |_) | (_| | |  |   <
       |____/| .__/ \__,_|_|  |_|\_\
             |_|   Game Toolkit

Copyright  2024-present tinyBigGAMES LLC
         All Rights Reserved.

Website: https://tinybiggames.com
Email  : support@tinybiggames.com

See LICENSE for license information
* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *)

unit UTexture01;

interface

uses
  System.SysUtils,
  SGT.Deps,
  SGT.Deps.Ext,
  SGT.OGL,
  SGT.Core,
  SGT.Lua,
  SGT.LuaDebugger,
  UCommon;

procedure Demo();

implementation

// The 'Demo' procedure demonstrates loading a texture from a zip file, rotating it, and rendering it within a window.
// It also displays the current frame rate and a quit instruction on the window.

procedure Demo();
var
  LZipFile: TZipFile;    // Represents the zip file from which resources will be loaded.
  LWindow: TWindow;      // Represents the window where the content will be displayed.
  LFont: TFont;          // Represents the font used to render text.
  LPos: TPoint;          // Represents the position on the window where the text will be drawn.
  LTexture: TTexture;    // Represents the texture loaded from the zip file.
  LAngle: Single;        // Represents the current angle of the texture, used for rotation.
begin
  // Initialize the zip file with the specified filename.
  LZipFile := TZipFile.Init(CZipFilename);

  // Initialize the window with the title 'SGT: Texture #01'.
  LWindow := TWindow.Init('SGT: Texture #01');

  // Load the default font with a size of 10, associated with the window.
  LFont := TFont.LoadDefault(LWindow, 10);

  // Load the texture from the zip file, using the specified path within the zip archive.
  LTexture := TTexture.LoadFromZipFile(LZipFile, 'res/images/SGT.png');

  // Set the initial position of the texture to the center of the window.
  LTexture.SetPos(LWindow.DEFAULT_CENTER_WIDTH, LWindow.DEFAULT_CENTER_HEIGHT);

  // Initialize the rotation angle to 0.
  LAngle := 0;

  // Main loop to keep the window running until the user closes it.
  while not LWindow.ShouldClose() do
  begin
    // Increment the angle for rotation.
    LAngle := LAngle + 0.1;

    // Clip the angle value to ensure it stays within the range [0, 360] degrees.
    Math.ClipValuef(LAngle, 0, 360, True);

    // Apply the updated angle to the texture for rotation.
    LTexture.SetAngle(LAngle);

    // Start processing a new frame.
    LWindow.StartFrame();

    // Check if the ESC key was pressed. If so, set the window to close.
    if LWindow.GetKey(KEY_ESCAPE, isWasPressed) then
      LWindow.SetShouldClose(True);

    // Start drawing on the window.
    LWindow.StartDrawing();

    // Clear the window with a dark slate brown background color.
    LWindow.Clear(DARKSLATEBROWN);

    // Draw the texture on the window.
    LTexture.Draw();

    // Set the initial position for text rendering.
    LPos := Math.Point(3, 3);

    // Draw the current frame rate in the top-left corner of the window in white color.
    LFont.DrawText(LWindow, LPos.X, LPos.Y, 0, WHITE, haLeft, 'fps %d', [FrameLimitTimer.FrameRate()]);

    // Draw a HUD text item, indicating how to quit, in green color.
    LFont.DrawText(LWindow, LPos.X, LPos.Y, 0, GREEN, haLeft, Utils.HudTextItem('Quit', 'ESC'), [FrameLimitTimer.FrameRate()]);

    // Finish drawing on the window.
    LWindow.EndDrawing();

    // End processing the frame.
    LWindow.EndFrame();
  end;

  // Free the texture resources after the loop ends.
  LTexture.Free();

  // Free the font resources after the loop ends.
  LFont.Free();

  // Free the window resources after the loop ends.
  LWindow.Free();

  // Free the zip file resources after the loop ends.
  LZipFile.Free();
end;

end.
