(* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
                         -****************+.
               .-=++*+::*###############*-
           :=+##*+=-::*##**############=
        .=*#*=:    .+#*=+############+.
       =##+.      +#+-=############*-:
     :*#+.      =+-.=#############- =#*.
    -##-      -=..+#############*:.  ---.
   -##:     .: .+################= :+#*-:.
  .##-        :==========*#####= :*#*- -#*
  =##                   -####*:-*#*-   .##:
  +#+                  -###*--*#*-      *#=
  *#+                 +###==*#*:        *#=
  +#*                *##+=*#*-          ##-
  :##.             .*##*##*:           -##.
   +#*            -#####*:             *#=
    *#+          =####+:             .*#+
     +#*.       +###+.              :##=
      -##+.    *##+:              :+#*:
        =*#+..*#+:             .-*#*-
          :-:#+::.        .:-=*##+:
           -+..+*###****####*+-.
          :.      ..:::::..
        ____                   _
       / ___| _ __   __ _ _ __| | __
       \___ \| '_ \ / _` | '__| |/ /
        ___) | |_) | (_| | |  |   <
       |____/| .__/ \__,_|_|  |_|\_\
             |_|   Game Toolkit

Copyright  2024-present tinyBigGAMES LLC
         All Rights Reserved.

Website: https://tinybiggames.com
Email  : support@tinybiggames.com

See LICENSE for license information
* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *)

unit UStarfield01;

interface

uses
  System.SysUtils,
  SGT.Deps,
  SGT.Deps.Ext,
  SGT.OGL,
  SGT.Core,
  SGT.Lua,
  SGT.LuaDebugger,
  UCommon;

procedure Demo();

implementation

// The 'Demo' procedure demonstrates interactive starfield rendering within a window.
// It allows the user to control the starfield's movement and parameters using keyboard inputs.

procedure Demo();
var
  LWindow: TWindow;              // Represents the window where the starfield and text will be displayed.
  LFont: TFont;                  // Represents the font used to render text on the window.
  LPos: TPoint;                  // Represents the position on the window where the text will be drawn.
  LStarfield: TStarfield;        // Represents the starfield object that handles starfield rendering and behavior.
begin
  // Initialize the window with the title 'SGT: Starfield #01'.
  LWindow := TWindow.Init('SGT: Starfield #01');

  // Load the default font with a size of 10, associated with the window.
  LFont := TFont.LoadDefault(LWindow, 10);

  // Create a new starfield instance for rendering.
  LStarfield := TStarfield.New(LWindow);

  // Main loop to keep the window running until the user closes it.
  while not LWindow.ShouldClose() do
  begin
    // Start processing a new frame.
    LWindow.StartFrame();

    // Check if the ESC key was pressed. If so, set the window to close.
    if LWindow.GetKey(KEY_ESCAPE, isWasPressed) then
      LWindow.SetShouldClose(True);

    // Update the starfield based on keyboard input.
    if LWindow.GetKey(KEY_1, isWasPressed) then
    begin
      // Set starfield movement parameters.
      LStarfield.SetXSpeed(6);
      LStarfield.SetYSpeed(0);
      LStarfield.SetZSpeed(-5);
      LStarfield.SetVirtualPos(0, 0);
    end;

    if LWindow.GetKey(KEY_2, isWasPressed) then
    begin
      // Set starfield movement parameters.
      LStarfield.SetXSpeed(0);
      LStarfield.SetYSpeed(-6);
      LStarfield.SetZSpeed(-6);
      LStarfield.SetVirtualPos(0, 0);
    end;

    if LWindow.GetKey(KEY_3, isWasPressed) then
    begin
      // Set starfield movement parameters.
      LStarfield.SetXSpeed(-6);
      LStarfield.SetYSpeed(0);
      LStarfield.SetZSpeed(-6);
      LStarfield.SetVirtualPos(0, 0);
    end;

    if LWindow.GetKey(KEY_4, isWasPressed) then
    begin
      // Set starfield movement parameters.
      LStarfield.SetXSpeed(0);
      LStarfield.SetYSpeed(6);
      LStarfield.SetZSpeed(-6);
      LStarfield.SetVirtualPos(0, 0);
    end;

    if LWindow.GetKey(KEY_5, isWasPressed) then
    begin
      // Set starfield movement parameters.
      LStarfield.SetXSpeed(0);
      LStarfield.SetYSpeed(0);
      LStarfield.SetZSpeed(6);
      LStarfield.SetVirtualPos(0, 0);
    end;

    if LWindow.GetKey(KEY_6, isWasPressed) then
    begin
      // Reinitialize the starfield with new parameters.
      LStarfield.Init(LWindow, 250, -1000, -1000, 10, 1000, 1000, 1000, 160);
      LStarfield.SetZSpeed(0);
      LStarfield.SetYSpeed(6);
    end;

    if LWindow.GetKey(KEY_7, isWasPressed) then
    begin
      // Reinitialize the starfield with new parameters.
      LStarfield.Init(LWindow, 250, -1000, -1000, 10, 1000, 1000, 1000, 80);
      LStarfield.SetXSpeed(0);
      LStarfield.SetYSpeed(0);
      LStarfield.SetZSpeed(-3);
      LStarfield.SetVirtualPos(0, 0);
    end;

    // Update the starfield based on the current settings.
    LStarfield.Update();

    // Start drawing on the window.
    LWindow.StartDrawing();

      // Clear the window with a black background color.
      LWindow.Clear(BLACK);

      // Render the starfield onto the window.
      LStarfield.Render(LWindow);

      // Set the initial position for text rendering.
      LPos := Math.Point(3, 3);

      // Draw the current frame rate in the top-left corner of the window in white color.
      LFont.DrawText(LWindow, LPos.X, LPos.Y, 0, WHITE, haLeft, 'fps %d', [FrameLimitTimer.FrameRate()]);

      // Draw instructions for quitting the application.
      LFont.DrawText(LWindow, LPos.X, LPos.Y, 0, GREEN, haLeft, Utils.HudTextItem('Quit', 'ESC'), [FrameLimitTimer.FrameRate()]);

      // Draw instructions for changing the starfield settings.
      LFont.DrawText(LWindow, LPos.X, LPos.Y, 0, GREEN, haLeft, Utils.HudTextItem('1-7', 'Change starfield'), []);

    // Finish drawing on the window.
    LWindow.EndDrawing();

    // End processing the frame.
    LWindow.EndFrame();
  end;

  // Free the starfield resources after the loop ends.
  LStarfield.Free();

  // Free the font resources after the loop ends.
  LFont.Free();

  // Free the window resources after the loop ends.
  LWindow.Free();
end;


end.
