(* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
                         -****************+.
               .-=++*+::*###############*-
           :=+##*+=-::*##**############=
        .=*#*=:    .+#*=+############+.
       =##+.      +#+-=############*-:
     :*#+.      =+-.=#############- =#*.
    -##-      -=..+#############*:.  ---.
   -##:     .: .+################= :+#*-:.
  .##-        :==========*#####= :*#*- -#*
  =##                   -####*:-*#*-   .##:
  +#+                  -###*--*#*-      *#=
  *#+                 +###==*#*:        *#=
  +#*                *##+=*#*-          ##-
  :##.             .*##*##*:           -##.
   +#*            -#####*:             *#=
    *#+          =####+:             .*#+
     +#*.       +###+.              :##=
      -##+.    *##+:              :+#*:
        =*#+..*#+:             .-*#*-
          :-:#+::.        .:-=*##+:
           -+..+*###****####*+-.
          :.      ..:::::..
        ____                   _
       / ___| _ __   __ _ _ __| | __
       \___ \| '_ \ / _` | '__| |/ /
        ___) | |_) | (_| | |  |   <
       |____/| .__/ \__,_|_|  |_|\_\
             |_|   Game Toolkit

Copyright  2024-present tinyBigGAMES LLC
         All Rights Reserved.

Website: https://tinybiggames.com
Email  : support@tinybiggames.com

See LICENSE for license information
* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *)

unit UPolygon01;

interface

uses
  System.SysUtils,
  SGT.Deps,
  SGT.Deps.Ext,
  SGT.OGL,
  SGT.Core,
  SGT.Lua,
  SGT.LuaDebugger,
  UCommon;

procedure Demo();

implementation

// The 'Demo' procedure demonstrates interactive rendering of a polygon within a window.
// It allows the user to scale and rotate the polygon using keyboard inputs.

procedure Demo();
var
  LWindow: TWindow;              // Represents the window where the polygon and text will be displayed.
  LFont: TFont;                  // Represents the font used to render text on the window.
  LPos: TPoint;                  // Represents the position on the window where the text will be drawn.
  LPolygon: TPolygon;            // Represents the polygon object for rendering.
  LAngle: Single;                // Represents the current angle of rotation for the polygon.
  LScale: Single;                // Represents the current scale factor for the polygon.
begin
  // Initialize the window with the title 'SGT: Polygon #01'.
  LWindow := TWindow.Init('SGT: Polygon #01');

  // Load the default font with a size of 10, associated with the window.
  LFont := TFont.LoadDefault(LWindow, 10);

  // Create a new polygon instance and define its vertices.
  LPolygon := TPolygon.Create();
  LPolygon.AddLocalPoint(-1, -1, True);
  LPolygon.AddLocalPoint(1, -1, True);
  LPolygon.AddLocalPoint(1, 1, True);
  LPolygon.AddLocalPoint(-1, 1, True);
  LPolygon.AddLocalPoint(-1, -1, True);

  // Initialize angle and scale.
  LAngle := 0.0;
  LScale := 150.0;

  // Main loop to keep the window running until the user closes it.
  while not LWindow.ShouldClose() do
  begin
    // Start processing a new frame.
    LWindow.StartFrame();

    // Check if the ESC key was pressed. If so, set the window to close.
    if LWindow.GetKey(KEY_ESCAPE, isWasPressed) then
      LWindow.SetShouldClose(True);

    // Adjust the scale based on UP and DOWN arrow key presses.
    if LWindow.GetKey(KEY_UP, isPressed) then
    begin
      LScale := LScale + 1.0;
      Math.ClipValuef(LScale, 10, 150, False);  // Clip the scale value between 10 and 150.
    end
    else if LWindow.GetKey(KEY_DOWN, isPressed) then
    begin
      LScale := LScale - 1.0;
      Math.ClipValuef(LScale, 10, 150, False);  // Clip the scale value between 10 and 150.
    end;

    // Increment the angle for rotation and clip it between 0 and 360 degrees.
    LAngle := LAngle + 0.6;
    Math.ClipValuef(LAngle, 0, 360, True);

    // Start drawing on the window.
    LWindow.StartDrawing();

      // Clear the window with a dark slate brown background color.
      LWindow.Clear(DARKSLATEBROWN);

      // Render the polygon with current scale, angle, and center position.
      LPolygon.Render(LWindow, LWindow.DEFAULT_CENTER_WIDTH, LWindow.DEFAULT_CENTER_HEIGHT, LScale, LAngle, 2, YELLOW, nil, False, False);

      // Set the initial position for text rendering.
      LPos := Math.Point(3, 3);

      // Draw the current frame rate in the top-left corner of the window in white color.
      LFont.DrawText(LWindow, LPos.X, LPos.Y, 0, WHITE, haLeft, 'fps %d', [FrameLimitTimer.FrameRate()]);

      // Draw instructions for quitting the application and scaling the polygon.
      LFont.DrawText(LWindow, LPos.X, LPos.Y, 0, GREEN, haLeft, Utils.HudTextItem('Quit', 'ESC'), [FrameLimitTimer.FrameRate()]);
      LFont.DrawText(LWindow, LPos.X, LPos.Y, 0, GREEN, haLeft, Utils.HudTextItem('UP', 'Scale up'), []);
      LFont.DrawText(LWindow, LPos.X, LPos.Y, 0, GREEN, haLeft, Utils.HudTextItem('DOWN', 'Scale down'), []);

    // Finish drawing on the window.
    LWindow.EndDrawing();

    // End processing the frame.
    LWindow.EndFrame();
  end;

  // Free the polygon resources after the loop ends.
  LPolygon.Free();

  // Free the font resources after the loop ends.
  LFont.Free();

  // Free the window resources after the loop ends.
  LWindow.Free();
end;


end.
