﻿(* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
                         -****************+.
               .-=++*+::*###############*-
           :=+##*+=-::*##**############=
        .=*#*=:    .+#*=+############+.
       =##+.      +#+-=############*-:
     :*#+.      =+-.=#############- =#*.
    -##-      -=..+#############*:.  ---.
   -##:     .: .+################= :+#*-:.
  .##-        :==========*#####= :*#*- -#*
  =##                   -####*:-*#*-   .##:
  +#+                  -###*--*#*-      *#=
  *#+                 +###==*#*:        *#=
  +#*                *##+=*#*-          ##-
  :##.             .*##*##*:           -##.
   +#*            -#####*:             *#=
    *#+          =####+:             .*#+
     +#*.       +###+.              :##=
      -##+.    *##+:              :+#*:
        =*#+..*#+:             .-*#*-
          :-:#+::.        .:-=*##+:
           -+..+*###****####*+-.
          :.      ..:::::..
        ____                   _
       / ___| _ __   __ _ _ __| | __
       \___ \| '_ \ / _` | '__| |/ /
        ___) | |_) | (_| | |  |   <
       |____/| .__/ \__,_|_|  |_|\_\
             |_|   Game Toolkit™

Copyright © 2024-present tinyBigGAMES™ LLC
         All Rights Reserved.

Website: https://tinybiggames.com
Email  : support@tinybiggames.com

See LICENSE for license information
* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *)

unit UFont01;

interface

uses
  System.SysUtils,
  SGT.Deps,
  SGT.Deps.Ext,
  SGT.OGL,
  SGT.Core,
  SGT.Lua,
  SGT.LuaDebugger,
  UCommon;

procedure Demo();

implementation

// The 'Demo' procedure demonstrates rendering text using different fonts from a ZIP file and a default font within a window.

procedure Demo();
var
  LZipFile: TZipFile;           // Represents the ZIP file containing font resources.
  LWindow: TWindow;             // Represents the window for rendering text.
  LPos: TPoint;                 // Represents the position on the window for text rendering.
  LFont: array[0..1] of TFont; // Array of fonts: default font and TrueType font loaded from ZIP file.
begin
  // Initialize the ZIP file with the given filename.
  LZipFile := TZipFile.Init(CZipFilename);

  // Initialize the window with the title 'SGT: Font #01'.
  LWindow := TWindow.Init('SGT: Font #01');

  // Load the default font with size 10.
  LFont[0] := TFont.LoadDefault(LWindow, 10);

  // Load a TrueType font from the ZIP file with size 16 and specific characters.
  LFont[1] := TFont.LoadFromZipFile(LWindow, LZipFile, 'res/fonts/unifont.ttf', 16, '你好こんにちは안녕하세요');

  // Main loop to keep the window running until the user closes it.
  while not LWindow.ShouldClose() do
  begin
    // Start processing a new frame.
    LWindow.StartFrame();

    // Check if the ESC key was pressed. If so, set the window to close.
    if LWindow.GetKey(KEY_ESCAPE, isWasPressed) then
      LWindow.SetShouldClose(True);

    // Start drawing on the window.
    LWindow.StartDrawing();

      // Clear the window with a dark slate brown background color.
      LWindow.Clear(DARKSLATEBROWN);

      // Draw text using the TrueType font with multiple languages and align it in the center.
      LFont[1].DrawText(LWindow, LWindow.DEFAULT_CENTER_WIDTH, LWindow.DEFAULT_CENTER_HEIGHT, YELLOW, haCenter, ' en   zh      ja       ko        de   es   pt     fr      vi    id', []);

      // Draw multilingual greetings using the TrueType font and align it in the center.
      LFont[1].DrawText(LWindow, LWindow.DEFAULT_CENTER_WIDTH, LWindow.DEFAULT_CENTER_HEIGHT+LFont[1].TextHeight()+3, DARKGREEN, haCenter, 'Hello|你好|こんにちは|안녕하세요|Hallo|Hola|Olá|Bonjour|Xin chào|Halo', []);

      // Draw a descriptive text using the default font and align it in the center above the multilingual text.
      LFont[0].DrawText(LWindow, LWindow.DEFAULT_CENTER_WIDTH, LWindow.DEFAULT_CENTER_HEIGHT-LFont[1].TextHeight()-10, GREENYELLOW, haCenter, 'these are truetype fonts', []);

      // Set the initial position for text rendering.
      LPos := Math.Point(3, 3);

      // Draw the current frame rate in the top-left corner of the window using the default font.
      LFont[0].DrawText(LWindow, LPos.X, LPos.Y, 0, WHITE, haLeft, 'fps %d', [FrameLimitTimer.FrameRate()]);

      // Draw the quit instruction in the top-left corner of the window using the default font.
      LFont[0].DrawText(LWindow, LPos.X, LPos.Y, 0, GREEN, haLeft, Utils.HudTextItem('Quit', 'ESC'), [FrameLimitTimer.FrameRate()]);

    // Finish drawing on the window.
    LWindow.EndDrawing();

    // End processing the frame.
    LWindow.EndFrame();
  end;

  // Free the TrueType font resources after the loop ends.
  LFont[1].Free();

  // Free the default font resources after the loop ends.
  LFont[0].Free();

  // Free the window resources after the loop ends.
  LWindow.Free();

  // Free the ZIP file resources after the loop ends.
  LZipFile.Free();
end;


end.
