(* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
                         -****************+.
               .-=++*+::*###############*-
           :=+##*+=-::*##**############=
        .=*#*=:    .+#*=+############+.
       =##+.      +#+-=############*-:
     :*#+.      =+-.=#############- =#*.
    -##-      -=..+#############*:.  ---.
   -##:     .: .+################= :+#*-:.
  .##-        :==========*#####= :*#*- -#*
  =##                   -####*:-*#*-   .##:
  +#+                  -###*--*#*-      *#=
  *#+                 +###==*#*:        *#=
  +#*                *##+=*#*-          ##-
  :##.             .*##*##*:           -##.
   +#*            -#####*:             *#=
    *#+          =####+:             .*#+
     +#*.       +###+.              :##=
      -##+.    *##+:              :+#*:
        =*#+..*#+:             .-*#*-
          :-:#+::.        .:-=*##+:
           -+..+*###****####*+-.
          :.      ..:::::..
        ____                   _
       / ___| _ __   __ _ _ __| | __
       \___ \| '_ \ / _` | '__| |/ /
        ___) | |_) | (_| | |  |   <
       |____/| .__/ \__,_|_|  |_|\_\
             |_|   Game Toolkit

Copyright  2024-present tinyBigGAMES LLC
         All Rights Reserved.

Website: https://tinybiggames.com
Email  : support@tinybiggames.com

See LICENSE for license information
* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *)

unit UEntity01;

interface

uses
  SysUtils,
  SGT.Deps,
  SGT.Deps.Ext,
  SGT.OGL,
  SGT.Core,
  SGT.Lua,
  SGT.LuaDebugger,
  UCommon;

procedure Demo();

implementation

// The 'Demo' procedure demonstrates a window with entities (sprites) and collision detection.
procedure Demo();
var
  LZipFile: TZipFile;      // Manages the ZIP file containing sprite resources.
  LWindow: TWindow;        // Manages the window for rendering.
  LFont: TFont;            // Manages the font used for rendering text.
  LPos: TPoint;            // Position for rendering text.
  LSprite: TSprite;        // Manages sprite images and animation.
  LBoss: TEntity;          // Represents the boss entity.
  LPlayer: TEntity;        // Represents the player entity.
  LCollide: Boolean;       // Boolean to check for collisions between entities.
  LMousePos: TPoint;       // Position of the mouse cursor.
begin
  // Initialize the ZIP file with the specified filename.
  LZipFile := TZipFile.Init(CZipFilename);

  // Initialize the window with the title 'SGT: Entity #01'.
  LWindow := TWindow.Init('SGT: Entity #01');

  // Load the default font with size 10.
  LFont := TFont.LoadDefault(LWindow, 10);

  // Create a new sprite instance.
  LSprite := TSprite.Create();

  // Load the boss sprite page from the ZIP file and add images to it.
  LSprite.LoadPageFromZipFile(LZipFile, 'res/sprites/boss.png', nil); // page #0
  LSprite.AddGroup(); // group #0
  LSprite.AddImageFromGrid(0, 0, 0, 0, 128, 128);
  LSprite.AddImageFromGrid(0, 0, 1, 0, 128, 128);
  LSprite.AddImageFromGrid(0, 0, 0, 1, 128, 128);

  // Load the ship sprite page from the ZIP file and add images to it.
  LSprite.LoadPageFromZipFile(LZipFile, 'res/sprites/ship.png', nil); // page #1
  LSprite.AddGroup(); // group #1
  LSprite.AddImageFromGrid(1, 1, 1, 0, 64, 64);
  LSprite.AddImageFromGrid(1, 1, 2, 0, 64, 64);
  LSprite.AddImageFromGrid(1, 1, 3, 0, 64, 64);

  // Create the boss entity with sprite group 0 and set its position and frame speed.
  LBoss := TEntity.New(LSprite, 0);
  LBoss.SetPosAbs(LWindow.DEFAULT_CENTER_WIDTH, LWindow.DEFAULT_CENTER_HEIGHT);
  LBoss.SetFrameSpeed(24);

  // Create the player entity with sprite group 1 and set its position and frame speed.
  LPlayer := TEntity.New(LSprite, 1);
  LPlayer.SetPosAbs(0, 0);
  LPlayer.SetFrameSpeed(24);

  // Main loop to keep the window running until the user closes it.
  while not LWindow.ShouldClose() do
  begin
    // Start processing a new frame.
    LWindow.StartFrame();

      // Get the current position of the mouse cursor.
      LMousePos := LWindow.GetMousePos();

      // Check if the ESC key was pressed. If so, set the window to close.
      if Lwindow.GetKey(KEY_ESCAPE, isWasPressed) then
        LWindow.SetShouldClose(True);

      // Update the boss entity to the next animation frame.
      LBoss.NextFrame();

      // Update the player entity to the next animation frame and move towards the mouse position.
      LPlayer.NextFrame();
      LPlayer.ThrustToPos(40, 40, LMousePos.x, LMousePos.y, 128, 32, 1, 0.001);

      // Check for collision between the player and the boss using an OBB (Oriented Bounding Box) collision method.
      LCollide := LPlayer.Collide(LBoss, eoOBB);

      // Start drawing on the window.
      LWindow.StartDrawing();

        // Clear the window with a DARKSLATEBROWN color.
        LWindow.Clear(DARKSLATEBROWN);

        // Render the boss entity.
        LBoss.Render();

        // If a collision is detected, draw a red rectangle.
        if LCollide then
          LWindow.DrawFilledRect(LWindow.DEFAULT_CENTER_WIDTH, LWindow.DEFAULT_CENTER_HEIGHT-64, 64, 10, RED, 0);

        // Render the player entity.
        LPlayer.Render();

        // Set the initial position for text rendering.
        LPos := Math.Point(3, 3);

        // Draw the current frame rate and quit instruction using the default font.
        LFont.DrawText(LWindow, LPos.X, LPos.Y, 0, WHITE, haLeft, 'fps %d', [FrameLimitTimer.FrameRate()]);
        LFont.DrawText(LWindow, LPos.X, LPos.Y, 0, GREEN, haLeft, Utils.HudTextItem('Quit', 'ESC'), [FrameLimitTimer.FrameRate()]);
        LFont.DrawText(LWindow, LWindow.DEFAULT_CENTER_WIDTH, 150, YELLOW, haCenter, 'move blue ship over green ship', []);

      // Finish drawing on the window.
      LWindow.EndDrawing();

    // End processing the frame.
    LWindow.EndFrame();
  end;

  // Free the resources used by the player and boss entities, sprite, font, window, and ZIP file.
  LPlayer.Free();
  LBoss.Free();
  LSprite.Free();

  LFont.Free();
  LWindow.Free();
  LZipFile.Free();
end;


end.
