(* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
                         -****************+.
               .-=++*+::*###############*-
           :=+##*+=-::*##**############=
        .=*#*=:    .+#*=+############+.
       =##+.      +#+-=############*-:
     :*#+.      =+-.=#############- =#*.
    -##-      -=..+#############*:.  ---.
   -##:     .: .+################= :+#*-:.
  .##-        :==========*#####= :*#*- -#*
  =##                   -####*:-*#*-   .##:
  +#+                  -###*--*#*-      *#=
  *#+                 +###==*#*:        *#=
  +#*                *##+=*#*-          ##-
  :##.             .*##*##*:           -##.
   +#*            -#####*:             *#=
    *#+          =####+:             .*#+
     +#*.       +###+.              :##=
      -##+.    *##+:              :+#*:
        =*#+..*#+:             .-*#*-
          :-:#+::.        .:-=*##+:
           -+..+*###****####*+-.
          :.      ..:::::..
        ____                   _
       / ___| _ __   __ _ _ __| | __
       \___ \| '_ \ / _` | '__| |/ /
        ___) | |_) | (_| | |  |   <
       |____/| .__/ \__,_|_|  |_|\_\
             |_|   Game Toolkit

Copyright  2024-present tinyBigGAMES LLC
         All Rights Reserved.

Website: https://tinybiggames.com
Email  : support@tinybiggames.com

See LICENSE for license information
* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *)

unit UCloudDB01;

interface

uses
  System.SysUtils,
  SGT.Deps,
  SGT.Deps.Ext,
  SGT.OGL,
  SGT.Core,
  SGT.Lua,
  SGT.LuaDebugger,
  SGT.CloudDB,
  UCommon;

procedure Demo();

implementation

{
  CloudDB - PHP Interface for Remote MySQL Database

  1. If you are using cPanel, ensure that both php_mysql and pdo_mysql
     extensions are enabled.
  2. Update the config.php file to set your MySQL configuration.
  3. In the Config class within index.php, adjust the path to correctly
     reference your config.php script.
  4. Ensure that config.php is stored outside of the publicly accessible HTTP
     directory for security reasons.

-----------------------------------------------------------------------------

  Explanation of SQL Static Macros (&text) and Dynamic Parameters (:text):

  1. SQL Static Macros (&text):
     - Purpose: Static macros are placeholders in your SQL query that are
       replaced with fixed values or strings at the time the SQL text is
       prepared.
     - How it works: When you use &text in your SQL statement, it acts as a
       macro that is replaced with a specific value or table name before the
       query is executed. This is typically used for SQL elements that don't
       change per execution, like table names or field names.
     - Example: If you have 'SELECT * FROM &table;' in your SQL text, and you
       set &table to 'users', the final SQL executed would be
       'SELECT * FROM users;'.
     - Analogy: Think of it like a "find and replace" that happens before the
       query runs.

  2. SQL Dynamic Parameters (:text):
     - Purpose: Dynamic parameters are used to securely insert variable data
       into SQL queries at runtime. They are typically used for values that
       can change, such as user input or variable data, and are often used to
       prevent SQL injection.
     - How it works: When you use :text in your SQL statement, it acts as a
       placeholder that will be dynamically replaced with an actual value at
       runtime. The value is passed separately from the SQL query, allowing
       for secure and flexible data handling.
     - Example: If you have 'SELECT * FROM users WHERE id = :userId;' in your
       SQL text, and you bind :userId to the value '42', the final SQL
       executed would be 'SELECT * FROM users WHERE id = 42;'.
     - Analogy: Think of it as a variable that gets its value just before the
       SQL query is run, making it possible to execute the same query with
       different data multiple times.
}

procedure Demo();
var
  LCloudDB: TCloudDb;                   // Instance of TCloudDb to handle database operations
  LCount: Integer;                      // Variable to store the number of records retrieved
  I: Integer;                           // Loop counter for iterating through records
  LName, LLevel, LScore, LSkill,        // Variables to hold individual field values from each record
  LDuration, LLocation: string;
begin
  // Set the console title for the application window
  Console.SetTitle('SGT: CloudDB01 #01');

  // Create an instance of TCloudDb to manage the connection to the database
  LCloudDB := TCloudDB.Create();
  try
    // Setup the CloudDB connection with the specified URL, API key, and database name
    LCloudDB.Setup(CDbURL, CDbApiKey, CDbName);

    // Set the SQL query text to select all records from the specified table
    LCloudDb.SetSQLText('SELECT * FROM &table;');

    // Set a macro to replace the placeholder "&table" with the actual table name
    LCloudDb.SetMacro('table', CDbTable);

    // Execute the SQL query and check if it returns any records
    if LCloudDB.Execute() then
    begin
      // Retrieve the count of records returned by the query
      LCount := LCloudDB.RecordCount();
      Console.PrintLn(); // Print a blank line for spacing

      // Print the title of the table
      Console.PrintLn();
      Console.PrintLn('                          --= H I G H S C O R E S =--');

      // Print the table header with column names
      Console.PrintLn('-------------------------------------------------------------------------------');
      Console.PrintLn('| %-20s | %-5s | %-6s | %-10s | %-8s | %-10s |', ['Name', 'Level', 'Score', 'Skill', 'Duration', 'Location']);
      Console.PrintLn('-------------------------------------------------------------------------------');

      // Iterate through each record and print the fields in a formatted table row
      for I := 0 to LCount-1 do
      begin
        // Retrieve and store each field value from the current record
        LName := LCloudDB.GetField(I, 'name');
        LLevel := LCloudDB.GetField(I, 'level');
        LScore := LCloudDB.GetField(I, 'score');
        LSkill := LCloudDB.GetField(I, 'skill');
        LDuration := LCloudDB.GetField(I, 'duration');
        LLocation := LCloudDB.GetField(I, 'location');

        // Print the field values in a formatted table row
        Console.PrintLn('| %-20s | %-5s | %-6s | %-10s | %-8s | %-10s |', [LName, LLevel, LScore, LSkill, LDuration, LLocation]);
      end;

      // Print the table footer to close off the table
      Console.PrintLn('-------------------------------------------------------------------------------');
    end;
  finally
    // Free the TCloudDb instance to release resources
    LCloudDB.Free();
  end;
end;

end.
