import { Command, Option } from 'clipanion'
import { logger } from '../../../logger'
import { ConfigManager } from '../../config-manager'
import fs from 'fs-extra'
import path from 'path'

export class CodemodCommand extends Command {
  static paths = [['codemod'], ['codemod', 'move-tina-folder']]
  rootPath = Option.String('--rootPath', {
    description: 'Specify the root directory to run the CLI from',
  })
  verbose = Option.Boolean('-v,--verbose', false, {
    description: 'increase verbosity of logged output',
  })
  static usage = Command.Usage({
    category: `Commands`,
    description: `Use codemods for various Tina tasks`,
  })

  async catch(error: any): Promise<void> {
    console.log(error)
    // logger.error('Error occured during tinacms codemod')
    // process.exit(1)
  }

  async execute(): Promise<number | void> {
    const mod = this.path[1]
    if (!mod) {
      logger.error(
        "Must specify an additional argument (eg. 'move-tina-folder')"
      )
      process.exit(1)
    }
    const mods = { 'move-tina-folder': () => moveTinaFolder(this.rootPath) }
    const command = mods[mod]
    if (!command) {
      logger.error(`Mod not found for ${mod}`)
      process.exit(1)
    }
    await command()
  }
}
const moveTinaFolder = async (rootPath: string = process.cwd()) => {
  const configManager = new ConfigManager({ rootPath })
  try {
    await configManager.processConfig()
  } catch (e) {
    logger.error(e.message)
    process.exit(1)
  }

  const tinaDestination = path.join(configManager.rootPath, 'tina')
  if (await fs.existsSync(tinaDestination)) {
    logger.info(
      `Folder already exists at ${tinaDestination}. Either delete this folder to complete the codemod, or ensure you have properly copied your config from the ".tina" folder.`
    )
  } else {
    await fs.moveSync(configManager.tinaFolderPath, tinaDestination)
    await writeGitignore(configManager.rootPath)
    logger.info(
      "Move to 'tina' folder complete. Be sure to update any imports of the autogenerated client!"
    )
  }
}

export const writeGitignore = async (rootPath: string) => {
  await fs.outputFileSync(
    path.join(rootPath, 'tina', '.gitignore'),
    '__generated__'
  )
}
