<ol class="breadcrumb">
  <li><a href="/">Home</a></li>
  <li><a href="/book/">Book</a></li>
  <li><a href="/book/1-0-0-overview/">Part One: Grokking Lisp</a></li>
  <li class="active">Numbers and Math</li>
</ol>

## Chapter 1.6

# Numbers and Math

> "The 3-legged stool of understanding is held up by history, languages, and mathematics. Equipped with these three you can learn anything you want to learn."
> <footer>Robert A. Heinlein, <em>The Happy Days Ahead, Expanded Universe</em></footer>

Common Lisp comes complete with a rigorous collection of mathematical tools, and excellent handling of Big Integers and Long-Floats---enabling Lisp software to leverage powerful number crunching facilities, perform statistical analysis, scientific computing, efficient cryptography, and more. All the functions, macros, and types presented in this chapter are a part of the language itself, and do not have to be specially imported from a separate standard library to be used.

In this chapter, we will review the numeric types available in Common Lisp, their built in notations, and cover the core mathematical functionality in the Common Lisp standard.

## Exercise 1.6.1

**Integers**

### In the REPL

```lisp
5

-5

1

1234567890123456789012345678901234567890

(expt 1234567890123456789012345678901234567890 2)

(expt 1234567890123456789012345678901234567890 147)

(expt -1234567890123456789012345678901234567890 147)
```

### What You Should See

An integer can be positive or negative.

```lisp
* 5
5

* -5
-5
```

It can also be any size.

```lisp
* 1
1

* 1234567890123456789012345678901234567890
1234567890123456789012345678901234567890

* (expt 1234567890123456789012345678901234567890 2)
1524157875323883675049535156256668194500533455762536198787501905199875019052100

* (expt 1234567890123456789012345678901234567890 147)
28359701166822119002645842775796164394603824493192308992511430860386097198464925090842243230790533477940244570728295851459491385176585144170964621679502069824147038958221291813459132607163593914354377069746203130422037905686708413876733975489712473525613257892039681023664998852971258229495008391165107722464983140113500795964750718853126179945626874681627025844850717262723634971570643204319421793412180932982739374448881083466936332022732961808690588764034922252761924188111222363747422636060610421980123564882414670579229034052634931500921834454800331387940044587060537251654210178980941254596059948798585595192790855651343261793334910977777931623460005142267493877721408197703442690087370167204737022958768502191070578783760037157882946001731487599646212608732215841855127446561704865232688622032806245252109842811741839793155868753108531079690267914644282259537817752050851343847989750626261911318014138527937766662092487813019520860694214448313380208847963544022911831684240156176614179046207559676217319304969968054489835783842628886747847826767654222767674696004889599047813847192848815177663084206054871917268109330747249478813642854846246124571862847081040739279546241365773288801165145755512895607490138217700274175688233752129949654806928425862281190382391697944069642222797081219350664418394503232590015742635652649806236926819570061650726617821167970628130742149569690029212859092151125608737771298199780798162612280161120749731449293965860353396325959994680450109487573973071775094410084052614102169032319628784048991436837872628705317762406953765296999298739688533946783745213440902633009726551159723781731964362479665246396838507533767331335766440451725423529739139289806428556389096277133397086365351269525200272987043473869264746071294888167233021676077692910008375436070995941036019743951164878043306551098847191327994826362562290782846430130375486827405572675198389935568181296323622724441962363726463941550383817611092252242267017085113473255281162257557985190260345061789058759665667817860288662501019564419853078838693230231740928375475389888235059008843513668386286232359751185486897832317467628547177596771041015266752659426230454029370702262933403181270275567366299560850033072077059058274484316373365803186811991504207017335709425845461253020869727810361334471806874647535720009169877232591340905594693087432998961315811261144570023458258316289907143068229756641300891568326347983970318449351418271690689710764128991734894443411441248904652701224255213756650040781539363600124291449465323208876408293350100597218284255692732934501129284945155709284421798134275575483129940104697089170880618649383833154407463439997790456214679118878742252108020551352326392770396009084713433801699131853757645058745619318707021897143432987187338162151001517911699998696598048830121188917022149431401577346720723325671706299540906201970059782158541958766312073727083868573518957527371514894088233453354056581453133278077146623293366638424538760288712270162344774665941494402083907424236004702268988155919994806919018029966601324947482333580141274385131965066194207611041036401897761829204954356309124916682717596989486997566236545137192771945638859656459213269080410642621960535397332035470632135441704667916520348915264240722565913172190028926509251165275224590368810091672766631664347035543113648447137211815088467787084406247603897570460783465958343363620704551603506914250730715293567596072453172765068848294192369319419703508398374289741520130196610761155904195617577372006415845023221759023343012838885366541908092395883542124949958735786219607354998781970856304412970819891648681135001785576221394774290518565096355638813217012552194636271320150344145550861080634221019456365656062376697178924823410764907881626390566077762774644049769541236374795290900433489882524990384371578800417686025308925948839836723828544576460573765136525103087439211788588810570675710515803970922946223958495381648015371711414922532153161351543087530518644056621956908207466043253306836638000849089921337896091882493820510061707559361051534036393060590897885321218885861893520599730730294711914247628578872507259936596000540951540997596919984233895110345657486527733330297551443420766499822063704011161611814597067563044555583463707613064181492432992156053170021032738460285483604706855666327743351347141719958634114410696124141825420909824331040397741727046333680687920385673059855671462653729137373035148015607420936557164095742493880200535027937446366311361034201082740445509414212637730131744482190887409943545154190951624878059279079152167432853589055145864839367542956286693078771346027621339773757538994276088043863489998356231241588855039236490726236171116769454403714225550282162689418471595574745804555680370982907963124287214017015974025342345009266209112212638038243745829004816018169951031744559877732354470788161109213350736420914117988971491915132331466741058619100066370282130489684306255237541821635913980786044908855266269818002157461072683476343771466461090981411498674576706331939128297740132529721498509956303606482487090605490542701602956259789819287213527009031268732824050545225973021066351988152056145280051007424702089231838999896821339955775313293088345275789169985241504255813562563413686198958712670412472393543096017607148572738600372032391506579650469430310333003554557076208444000334322864000778743171951064938709219207793926759939347275539625162728206818883772800727194889700355787333584547738786373586191571981349943227351058704862427425567985946169217956106019452824994243579438816006572796546699558194267858948619086276877310702728733870778764190322547666438235561186029000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000

* (expt -1234567890123456789012345678901234567890 147)
-28359701166822119002645842775796164394603824493192308992511430860386097198464925090842243230790533477940244570728295851459491385176585144170964621679502069824147038958221291813459132607163593914354377069746203130422037905686708413876733975489712473525613257892039681023664998852971258229495008391165107722464983140113500795964750718853126179945626874681627025844850717262723634971570643204319421793412180932982739374448881083466936332022732961808690588764034922252761924188111222363747422636060610421980123564882414670579229034052634931500921834454800331387940044587060537251654210178980941254596059948798585595192790855651343261793334910977777931623460005142267493877721408197703442690087370167204737022958768502191070578783760037157882946001731487599646212608732215841855127446561704865232688622032806245252109842811741839793155868753108531079690267914644282259537817752050851343847989750626261911318014138527937766662092487813019520860694214448313380208847963544022911831684240156176614179046207559676217319304969968054489835783842628886747847826767654222767674696004889599047813847192848815177663084206054871917268109330747249478813642854846246124571862847081040739279546241365773288801165145755512895607490138217700274175688233752129949654806928425862281190382391697944069642222797081219350664418394503232590015742635652649806236926819570061650726617821167970628130742149569690029212859092151125608737771298199780798162612280161120749731449293965860353396325959994680450109487573973071775094410084052614102169032319628784048991436837872628705317762406953765296999298739688533946783745213440902633009726551159723781731964362479665246396838507533767331335766440451725423529739139289806428556389096277133397086365351269525200272987043473869264746071294888167233021676077692910008375436070995941036019743951164878043306551098847191327994826362562290782846430130375486827405572675198389935568181296323622724441962363726463941550383817611092252242267017085113473255281162257557985190260345061789058759665667817860288662501019564419853078838693230231740928375475389888235059008843513668386286232359751185486897832317467628547177596771041015266752659426230454029370702262933403181270275567366299560850033072077059058274484316373365803186811991504207017335709425845461253020869727810361334471806874647535720009169877232591340905594693087432998961315811261144570023458258316289907143068229756641300891568326347983970318449351418271690689710764128991734894443411441248904652701224255213756650040781539363600124291449465323208876408293350100597218284255692732934501129284945155709284421798134275575483129940104697089170880618649383833154407463439997790456214679118878742252108020551352326392770396009084713433801699131853757645058745619318707021897143432987187338162151001517911699998696598048830121188917022149431401577346720723325671706299540906201970059782158541958766312073727083868573518957527371514894088233453354056581453133278077146623293366638424538760288712270162344774665941494402083907424236004702268988155919994806919018029966601324947482333580141274385131965066194207611041036401897761829204954356309124916682717596989486997566236545137192771945638859656459213269080410642621960535397332035470632135441704667916520348915264240722565913172190028926509251165275224590368810091672766631664347035543113648447137211815088467787084406247603897570460783465958343363620704551603506914250730715293567596072453172765068848294192369319419703508398374289741520130196610761155904195617577372006415845023221759023343012838885366541908092395883542124949958735786219607354998781970856304412970819891648681135001785576221394774290518565096355638813217012552194636271320150344145550861080634221019456365656062376697178924823410764907881626390566077762774644049769541236374795290900433489882524990384371578800417686025308925948839836723828544576460573765136525103087439211788588810570675710515803970922946223958495381648015371711414922532153161351543087530518644056621956908207466043253306836638000849089921337896091882493820510061707559361051534036393060590897885321218885861893520599730730294711914247628578872507259936596000540951540997596919984233895110345657486527733330297551443420766499822063704011161611814597067563044555583463707613064181492432992156053170021032738460285483604706855666327743351347141719958634114410696124141825420909824331040397741727046333680687920385673059855671462653729137373035148015607420936557164095742493880200535027937446366311361034201082740445509414212637730131744482190887409943545154190951624878059279079152167432853589055145864839367542956286693078771346027621339773757538994276088043863489998356231241588855039236490726236171116769454403714225550282162689418471595574745804555680370982907963124287214017015974025342345009266209112212638038243745829004816018169951031744559877732354470788161109213350736420914117988971491915132331466741058619100066370282130489684306255237541821635913980786044908855266269818002157461072683476343771466461090981411498674576706331939128297740132529721498509956303606482487090605490542701602956259789819287213527009031268732824050545225973021066351988152056145280051007424702089231838999896821339955775313293088345275789169985241504255813562563413686198958712670412472393543096017607148572738600372032391506579650469430310333003554557076208444000334322864000778743171951064938709219207793926759939347275539625162728206818883772800727194889700355787333584547738786373586191571981349943227351058704862427425567985946169217956106019452824994243579438816006572796546699558194267858948619086276877310702728733870778764190322547666438235561186029000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
```

## Exercise 1.6.2

**More Integers**

Of course, this doesn't mean they're always the same type internally. Various integers have different types based on their size to conserve space, though they are all usable interchangeably.

### In the REPL

```lisp
(type-of 1)

(type-of 1234567890123456789012345678901234567890)

(type-of (expt -1234567890123456789012345678901234567890 147))

(typep 1 'integer)

(typep 1234567890123456789012345678901234567890 'integer)

(typep (expt -1234567890123456789012345678901234567890 147) 'integer)
```

### What You Should See

```lisp
* (type-of 1)
BIT

* (type-of 1234567890123456789012345678901234567890)
(INTEGER 4611686018427387904)

* (type-of (expt -1234567890123456789012345678901234567890 147))
BIGNUM

* (typep 1 'integer)
T

* (typep 1234567890123456789012345678901234567890 'integer)
T

* (typep (expt -1234567890123456789012345678901234567890 147) 'integer)
T
```

## Exercise 1.6.3

**Hexadecimal Integer Notation**

### In the REPL

```lisp
#xFF

#x00

#x0123456789ABCDEF

(parse-integer "FF" :radix 16)

(parse-integer "0123456789ABCDEF" :radix 16)

(format nil "~x" 255)

(format nil "~x" 81985529216486895)

(type-of #xFF)

(+ #xDEADBEEF #xFF)

(format nil "~x" #x0123456789ABCDEF)

(format nil "~x" (expt -1234567890123456789012345678901234567890 147))
```

### What You Should See

There exists a built-in reader macro to allow for hexadecimal literals.

```lisp
* #xFF
255

* #x00
0

* #x0123456789ABCDEF
81985529216486895
```

You also have access to appropriate options in `parse-integer` and `format` to read or write hexadecimal numbers.

```lisp
* (parse-integer "FF" :radix 16)
255
2

* (parse-integer "0123456789ABCDEF" :radix 16)
81985529216486895
16

* (format nil "~x" 255)
"FF"

* (format nil "~x" 81985529216486895)
"123456789ABCDEF"
```

Of course, the reader macro expands into a number...

```lisp
* (type-of #xFF)
(INTEGER 0 4611686018427387903)
```

...which means you can use it anywhere you can use numbers.

```lisp
* (+ #xDEADBEEF #xFF)
3735928814

* (format nil "~x" #x0123456789ABCDEF)
"123456789ABCDEF"

* (format nil "~x" (expt -1234567890123456789012345678901234567890 147))
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
```

## Exercise 1.6.4

**Octal Integer Notation**

### In the REPL

```lisp
#o77

#o00

#o01234567

(parse-integer "77" :radix 8)

(parse-integer "01234567" :radix 8)

(format nil "~o" 63)

(format nil "~o" 342391)

(type-of #o77)

(+ #o123 #o456)

(format nil "~o" #o01234567)

(format nil "~o" (expt -1234567890123456789012345678901234567890 147))
```

### What You Should See

There is also a built-in syntax for octal literals.

```lisp
* #o77
63

* #o0
0

* #o01234567
342391
```

There also exist appropriate supporting options in `parse-integer` and `format` to read or write octal numbers.

```lisp
* (parse-integer "77" :radix 8)
63
2

* (parse-integer "01234567" :radix 8)
342391
8

* (format nil "~o" 63)
"77"

* (format nil "~o" 342391)
"1234567"
```

And again, the `#o` reader form expands into a number...

```lisp
* (type-of #o77)
(INTEGER 0 4611686018427387903)
```

... which means you can use anywhere you could use numbers.

```lisp
* (+ #o123 #o456)
385

* (format nil "~o" #o01234567)
"1234567"

* (format nil "~o" (expt -1234567890123456789012345678901234567890 147))
"-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"
```

## Exercise 1.6.5

**Binary Integer Notation**

### In the REPL

```lisp
#b1

#b1011

#b101010101

(parse-integer "1011" :radix 2)

(parse-integer "101010101" :radix 2)

(format nil "~b" 11)

(format nil "~b" 341)

(type-of #b1)

(type-of #b101)

(+ #b10101 #b101)

(format nil "~b" #b101010101)

(format nil "~b" (expt -1234567890123456789012345678901234567890 147))
```

### What You Should See

Similarly, there is a binary literal notation.

```lisp
* #b1
1

* #b1011
11

* #b101010101
341
```

`format` and `parse-integer` work nicely to encode and decode binary numbers.

```lisp
* (parse-integer "1011" :radix 2)
11
4

* (parse-integer "101010101" :radix 2)
341
9

* (format nil "~b" 11)
"1011"

* (format nil "~b" 341)
"101010101"
```

And once again, the reader macro expands into a number.

```lisp
* (type-of #b1)
BIT

* (type-of #b101)
(INTEGER 0 4611686018427387903)
```
So all the usual number-related tricks are available.

```lisp
* (+ #b10101 #b101)
26

* (format nil "~b" #b101010101)
"101010101"

* (format nil "~b" (expt -1234567890123456789012345678901234567890 147))
"-1001110111100001000001101011001010101100101111101101100011000111000110101110110001110010110000001110011001110110011100010001111100100011111101010111001010011001110110100111000110010110111000000001011001010111011011100000010011000110000001001100010011110101010010010101001100100111000111011010010100001111100011000111010000110001110110000001011000011001111100010001001110010101111011001010100011010010111011010111011111101011111000111011111101110011110100110110111000101101101000011101001110110100011001110001011001010000001011110011101000001001100111110011001001101100111111100100110101000010100110110010101110100110111011101001100100000100001011000101011110010100011111010111011110111000001100100001101011000001010111111001001000011010100011100101111000110011010110111111100011101001110000111100011000011000001101011101100001011100000001000100011000110100011100011001011000000010001011100000000110110011000101101100000011101010010101001110000010010100011110011100111011010010111110011010010001110000100111111011100101110001001100010001100011101100101001011010010011010110101100110110011011110110110001101001000100100101101101111010100100111100110111110010101000011011100001001010011001011110111000001111110110111100010011010000111111000111101010011100100100011001110110011100010011111100010100100100001101001010001011011001011011001111110010100011110010101010000101111000110110100100111000100011110110001110101110000100001110010100000111111110000101001100000100010001111010000010110000110011110111000110101000010010101101100010101111000100001111011110011111110001100101000110101011101010110111100111001001101100111011010101000010100101111011010101111010010101000011010010101100001011110001110001111101111101010010011000001101000010100110011010010110110000110010010111110010100010010101010010110101110110111010011010110001011010100100010101100011011001010011001101100010110111010101001101010101011101010111101010101000110010111100110101101101110110001001011100111110110001001000001001101010001101100001000111101010011101000101000000001010101010100011101100101101101111111110111100100010101000010011110111111000111111001111110001111101111110011001011111101001101011010101111001000000010101110111000101010111001101000011001001011110101011001001111111001110000011000011110100011011101001000010111000101101000011101001101111011000101001010101010011010100000111110100000110110011100100000011001111101111001010011111111010010010110101000000101010110100111011111101011010001100100011000101101101000000000111011001011001101111001110110000001111010100111010101011110110000011000111100110000100101110101000111111010111000010001000011100011110110110101010000011010000100111101010000001010100101010010100110110101010010101011110111110000111010100011011111011111001100111001100011100100000000000010101101011110110100101010101001101001000110010110100111110000101110010110010000111001000100000101000001101000100010000011101001100011010101000000001110010010100110111100101100001010100001100010000101110101101000110010011010100001010110010110111110001110011011000000101011111010100100000000011011100010110000010011100111010000010001110011111010000001100101000110100001101111101000111010011011110010110110101011111110100101110001001010110110000010001111010010101001100100011001011101111100011110000000100100010000011011111010100110010110000101001101100010010010000101010010101110101100010110100000100111111111100011010111010100100010010111010010100111011011101111000110010111110011011010010110100001100110010010001111100111110010111010011100101100110000100111110010011101001111110100101001101001001011110101010001111101000100100010100010010101011000110100110010110011101101001100111110101011011111111011110011101101111000101110011010110101101100000001011000100101001011101101010001101001000100000110100100001000010001100111100110010001000110100100011110101110110101010101100000110111101100111000010110001100110011110111000110101101000011110000011000110110100011011001000110101100011010110010101111100111101100000110000101111100010001001000100000100110110110100100110100111000110111101111100011010000000101110110011101111001111001001001001011000001111111101000101001010011000000001011110100001101011011000001000100010100011110100100100100110011111110001001100101011001111010101100110001011010111001100001000001111110000111010010110110101011000110011110011011111010101100101110101011010100010111001000011010011011110001000100100101100011100011100010100111101011110010101110110111100101110110011100111011100111101011001111010010101001101000110001111000101010010110011011101110100100111011100101001100000111111100010100100010011010100110100000001010001010010001001011011011101001101110011010111100110001111001100000110111001011101011001001101111000100011111101110000101001100110111110101110011001010100000101100000010100100111101100110100010010100000001100011111000000100101011001010001110000110011101101111010100000010111010110110011111110001100010100011110000111000011001001010100011110110011000101011001001110001000100101100111001011111010000011111111001011010010001001100111010111011010111011100011000010110111110000111101100001110101001101101101111010000011000000101010000110100010010011001001110100100111000110001100101101010101100111100001000100100111110011001110011000111110110000000010000001001101010100010111110000110000000000110110101100000001111001100100011101011101101000111111001000010000011110010111110100000101111110001100100100010001010100011111011010101011110010100000001101001111110011110000010001101110001000011011000101111000101011100001111101010111110010010101110000010001010110010100000011101111110111111010110001100001001010111111101001111101110001110001011100110111000100000011111011000100110001111111111110001011011110100011000011010100001110000101101001100111001110011001011111111000110010010010101011101001110001100011001111111110101010111111010110000100101001100010010100110100011011001000100101011011011110110011011001111101100010000100111000001110011000000000001010110100101101110110010011110010101001101010001001010110101101101101100011101110000000100000111100011100100010001011000001110110001100111100101000010111000100000110101011001001000001110011111011111110011110110111100001010001011000011111101101000111001100000010010001011111111010111001110101101100111100110101011110010111010010100111110000011010011011101010110011100100111011001101111111110000011101011001010010111100101111101111000101011011111001111001110001101001111011011100100110100111110100010011100100011000011010000000110001101011011011110011100010110110100001010000100011100101111110110110011110010111101001110111110001000010110000100110101100111101011110110111011101101001110001010010111111001110010000000110001111011010100000000111101111001100000111011001010010110000100110101101110110000011010011011100111111010101010001110011010100100110101010011110101011110010001010001001000000010101100001110110000001001110100001110011001100010101100101100010101101100000010010010000110100101000110001000001101101100101101110101101011110010110000101110000101010110100110100111100010110001111110101000011100100111101000011110001010111110101101100100011110000000000111100000101100111100001101001011110010110111101010000111001011001001101101101100101001100111101110000011111001101011000010111100001101010111000111010100111010100111000011000100111101000100110100010110111101010110001110101010010101010001101101001101010000001011111110010110011001001000011001011111001110010101101010011011100100011001001001010101000110111011100010101011011101000101111001100000010000001001010100111100010000010010101101011001001000000001111101000000011100000110111000110100000110001010100000111101000011100000001010000100101011110010011101111000111010111000010111101101001100010110000111100111001001100101000011110100010001111111001011011101100101101001010001101010111011100101110111010010110001011010111110111101110100100100111010111001011011001001101101111100100011010000100110111100111010000001101110111101010000110111111010000100011110000101101001111011010100111010000100110001010001110111111011111100011010101111000101001001000110011001110101101100101001110100010101001111111011011001101011001010010110011000101101010110110111001011011110011111010000110000011111111000101001000011010110110011010000100010110111110110111001000100101011000111101001101101000100110001111100011111101011001101111101111001111111000010001110000100111011010010100010010110001110100101000101000000111101001011111100111111000100100010011101000011000001100011101011000000010111110111011010110101111110100111110000110101111010101101010000001100101001000000110010000100011000000000110101001111000100000111000111111011100100001010010011100101100110000110001110111111010110100110011000001001011010011001001101011010000010100001100110000001101011011011010101100111001110011101100111100111101110000010011000001000110100100100011110100101000111000101110011111010000011110111110011110001010110010100100011100010011101011110000000011011011011111100110011001001001100111111100000010001111111101110011001110011101010100110001011111000111110011110000101101110001011101010110001100010010011010100101111101000010100010110000111011111011010111100011101011001100000000010010101111110111111110110001101000100010001001010000111000110111101000011101101001000100010010001001110011010011101001010001111011110010010111011100101111010111000110001111011111101111101111111110010011000110110000100111010111111100110110100011000001011000111011100001000100100010100110111100011111100001000111000000110011001000010010110111110101100111001111100101110100110101111011010100101110011110101101111001010010101000101011010100001000011001000000101101100010010111100100111111001100111000011111001110001111001001110001111100110001111100110100100010100111010001100010111010001010101011111010001000110111000010001000011110010011110011100100100101001110000100110001011111000101101101001111000111010111011000100100110101000101100110110010101111101100011110101011111101010001011100101110010110110111101110001011001011001101011011000000101011101010001010111011011001001001111101010011000101111111100110011100100000100111100111011001100001100010100100100011000101010111001010110100100011110000011110100010100110100100010110011010000111010001110011110000100111111111110010100010110000011101001100111001010001111100101001100110110011111000000111001001000010111100100010011000110011111011110101000001100010010011011101000001001001111100110011110000101110001101000100011010000100101001011110100101010000111010110001000011001111000010111001110101101110001110101111010111110011100110101110000100111100110000011000100110110010101010010001011000101001001000010101111001111101101111000011101011111001010111011001011011111111000111010001100110100011110001011110011011110110010100001001011010010101001101100001100111000000100101100101111110011110010101011000111110100011011000011100011100100000101001100010100000101111110010100111011110011010110011101100100011111010011100111010100111110000101100111111100110101010011011001011110000101010010100000101110100101010001000001001010000101101110000100010011110101100111010101100101011001010100011000001110011100000101010000000111000000000100011111111100110000111010110001110010011000010101111100111010001001111000111001000010010100011010101110101000111010110011101000101101001110110110001101101110111110010111110111110101100101010110000011111000011001101111011010110011000011001100100010011111100011000001000100101100001010010100101000110010100111110000010111101010110100100000010100111101000101010110101110000011110000010001101101110000101000111000011010110010110110110000111110010001111100000100110100010000001011001100100101001010110011000101010110110110000010001010001110100110110000110100100101010111000001110011010101110101111011100110100000000011010001000111110011101101110000011110110000010101101110001101100101011001010100101010111100101111011101011001011001000001011101000011010100101100101000000111101111110011111010100100111111010011111101110011000111101100111111010011011100010010101011001100011100001010010001001111100111011000010101010001011011011110011110101000111101111100110111011011011011001001110100011101000001111011001000111010100010111100001011101111000101011001101110011110110001110101100110101101100001111111100000110100010011001010101100000001101011011000000101100110001011000000111011110101100010000001001101101111000000011010101100110111110110010011111001001100000111000101100000111001011111010001101001101110100101001110111101000101011110000011111010101001011000100000100101100110011100111101100001011110101110110101010001001110011010010100101000111110011010100111111110111011010100001010101010111100000111000110110110101110001111010111100101010001010101110000010011001001111000101001111111000010110110100110001110001101101101100010000011011110111000101001110010011001100000111001011000001000101010000101010101000010110101011000111000001000100011111111010110110111001011011111111110111111110111010100110001101101100101011000111110000101111110000100011000001100101011000110100100110011111100101001100111100100101110000110010100000100001110001001011000010101000011011100000000000111100011101101011001101000110000101000000010011010011100001011110101111010000001101010100001110000001110101110100111001100100000101101111100001001100001111010100010111001100000010011100100111101001001101010101100101111000000111000000000101000111000011110000011001001011011111001010001011011111000100011100101001110100100101110111110110001111110110000001010001110110001100101001100011111100110001010101100000001001000011101111110001100001100101000100010101001111010110011011101110000101010100000001101100100100010001101101001100111111111100010101100110110101011001101100001001001111111010000000001000101101001101000000110100001110101111101000010110010111111011100000110101111000111111000001011000101110111001011100010011001000110011110001011100010000100111010101100010111110011100111111110001111111000111110011000000111111101110110001011011101111100110110000111001101111111110110000111011001010100000001010101000011101110000010101010100100111100010111110110011000100010000010000101111101010101000001101111100001111000111100101011010011011100010011111110111001010100011011001010000110010110111100011110011011110100100110101110010001111001001100011100111111110011110100101111111001011000100111010101100011011000100011011000111100100111000011110110111111001100101010111010000110011100101111011110000001010111010100000011001100110111110110101011010111000011111000110110110101010110111110111001011011110000111001001110000111110101101000010110001110110110101110101110000100001011000111001001000000001110001110111101101001111100010000000111000100110001100000001100101101011010110110111001011011111000001111111011010000101100111000001011001010000010100100110000110111111101100011000011011110010000100101001001111011100000110010101100101100000010001100000010000000101010110100100001010101100010111000011011100011110000100011010101010110111111101011100110110011101000000001101110110111001100111101101111011011110010111111010010100101010001011000000000001101001001001000111011111110000001010000111110011101100100101111101100100110110100111010011011100000111111001111110101111100011011101010110100100001011110011101000101001100010001100101111001111100000001001100100110011110100110100011011000101111011010111000101101100100000110001110110000000100110101100110011011100110100110100000001100110111010101100101101001000111101110001111010111010001111111010001010101001101001100010101011001111111001010000101001001011110011111110001100010000011101101010101101000110111011011000111010111101000111011101110010111001100001111010100110011010110011101010001110101000010101010110101010011000011111110110011001000101010110010100010001010000100001111101111100010000100110010100010010010110011001010100000011010000101001110110000110001100101100110011100101111001100010100101101111101100000001110100010011110110001110101100111000010000000111001001101100110011000010000110110110101100101010110111111001001000100011100110100001100110111110111001001101111001010001010011110110111010101010010010011110000001110011010001101011100111110101101011110011011111010111100100001101001010000110011000000101011001110011111010010100101011001110000000101000111011101101000011111100100100111001100100110000100110110010111000110001010010101000000110111000101010001000001111110001000110110100000111001101000100000010000101011101000010101000000110010011001010101010110000101110010101100111000010100011110010100101010110011100100101010001001001010010110001001010010011100111000011111001010111101110101111010000011010001001000101100000101111100000010110110100111110110010101101100110010010111011100010100001110011000001111110110011101000001101100110111110000101010110110010001000110011011010010010001110110101010011010101111101100111010010100111000010010100111001100100110101010000110001110111111101101011001001100100000100001111010010110100010001111000001110101001111001111000001100101111111100111111110001101100101101101011000010011101110100011011011111001110011110110010111111110010110101110000010110010101001110001011111011111010100111000001000101101000000010110011100110010100100110111011101111000011110111001100111011100101010101111010001100111101100110000111110010110001111010110101000000100010111110000110101110010010000110000100111111101111011001011100100101000011011111110100111011101011110111101011010010111100011110111101000110111110100010000100110001100001011111100101011001010001000000001101000111010011010000101111111010011010011101110010010100001011000101101100000110011000100001000100001011001101101100011001111111110001001000100101111111000010101101011111011110001010111001010110000011010110100000110110101000111100110010011111111000001101001110000000110010110011110000111110011000111100101000101000111011010111100001111100011110001000101101010101111100001111011110001100010101110100110010110100110011000011101000110010010000001011000111110001011010001101000011111110110101100110000001111010100100001110101110110001010010000010001111010011110100101111000011101111100000111010110110000111000010010111011101011000110100110111010110110110111100001010011001100011110100000001001100000111011010101100000100101110110100110011100000010101011101011111011001000000001000110011000110010011110101011100100000011101011010001001110001011001010111011010000010111100100111001010000111000001011110111001111010011010011111010001111110001011101100100110010101110011110111011111110010110001010010000111111000101101000100101000110000010011111111011110000100010010010000010010101001010010111101100001101010001000111011001101111011110101011001000000011010011111110010101011001000111101110001110101100101000000011111110100011111100101011000100011111001101011100110011011000000111000111010100101011101000111111110000011100001000101100000101010111001011111000110000000110000000100011110100100101100111001101101100100101111000110001101101111101010010111000111111101100110101111001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
```

## Exercise 1.6.6

**Ratios**

### In the REPL

```lisp
3/17

1/2

(typep 1/2 'integer)

(typep 1/2 'number)

(type-of 1/2)

.5

(type-of .5)

1/2

4/8

32/64
```

### What You Should See

Ratios are a primitive numeric type in Common Lisp.

```lisp
* 3/17
3/17

* 1/2
1/2
```

Ratios are not integers, but they are still numbers.

```lisp
* (typep 1/2 'integer)
NIL

* (typep 1/2 'number)
T
```

These are not the same things as their float equivalents.

```lisp
* (type-of 1/2)
RATIO

* .5
0.5

* (type-of .5)
SINGLE-FLOAT
```

They are always reduced to lowest terms.

```lisp
* 1/2
1/2

* 4/8
1/2

* 32/64
1/2
```

## Exercise 1.6.7

**Floating-Point Numbers**

### In the REPL

```lisp
.1

1.5

1.1234

(typep 1.1234 'number)

(typep 1.1234 'integer)

(type-of 1.1234)

(/ 6.44444444444 2)

(+ .5000001 .4000009)
```

### What You Should See

Floating point numbers are also among the Common Lisp numeric types.

```lisp
* .1
0.1

* 1.5
1.5

* 1.1234
1.1234

* (typep 1.1234 'number)
T

* (typep 1.1234 'integer)
NIL

* (type-of 1.1234)
SINGLE-FLOAT
```

They suffer all the same precision issues here as in every other language you've seen them at work.

```lisp
* (/ 6.44444444444 2)
3.2222223

* (+ .5000001 .4000009)
0.90000105
```

So, as always, careful.

## Exercise 1.6.8

**Numeric Constants**

### In the REPL

```lisp
pi

(= pi pi)

(eq pi pi)

(eql pi pi)

(equal pi pi)

(equalp pi pi)
```

### What You Should See

The constant `pi` is defined for you.

```lisp
* pi
3.141592653589793d0
```

It's always equal to itself, obviously, but its exact representation depends on the float precision of a particular Lisp implementation.

```lisp
* (= pi pi)
T

* (eq pi pi)
T

* (eql pi pi)
T

* (equal pi pi)
T

* (equalp pi pi)
T
```

## Exercise 1.6.9

**Complex Numbers**

Common Lisp has built-in support for Complex Numbers. You can perform arithmetic on them just like any other number type.

### In the REPL

```lisp
(sqrt -1)

(+ #C(0.0 1.0) #C(0.0 1.0))

(* #C(0.0 1.0) #C(0.0 1.0))
```

### What You Should See

```lisp
* (sqrt -1)
#C(0.0 1.0)

* (+ #C(0.0 1.0) #C(0.0 1.0))
#C(0.0 2.0)

* (* #C(0.0 1.0) #C(0.0 1.0))
#C(-1.0 0.0)
```

## Exercise 1.6.10

**Arithmetic**

Now that you've had a chance to explore some more number types built in to Common Lisp, let's take another look at what you can do with them.

### In the REPL

```lisp
(+ 1 1)

(1+ 1)

(- 1 1)

(- 1)

(* 2 2)

(/ 2 2)

(/ 10 2)

(/ 9 2)
```

### What You Should See

```lisp
* (+ 1 1)
2

* (1+ 1)
2

* (- 1 1)
0

* (1- 1)
0

* (- 1)
-1

* (* 2 2)
4

* (/ 2 2)
1

* (/ 10 2)
5

* (/ 9 2)
9/2
```

You can non-destructively increment or decrement a value with the built-in functions `1+` and `1-`, respectively.

You can negate a number by passing it as the only argument to the subtraction function.  This may seem verbose compared to the much simpler `-n` syntax for negative integers, however, this allows *any* number that can be negated to be, and also allows for user-supplied values to be negated.

Division normally returns a *rational* number, the canonical representation of which could be either an integer *or* a ratio.

## Exercise 1.6.11

**More Arithmetic**

### In the REPL

```lisp
(+ 1 1.0)

(+ 1 #C(0.0 1.0))

(/ 9 2.0)

(floor 9/2)

(floor 9 2)

(ceiling 9/2)

(ceiling 9 2)
```

### What You Should See

```lisp
* (+ 1 1.0)
2.0

* (+ 1 #C(0.0 1.0))
#C(1.0 1.0)

* (/ 9 2.0)
4.5

* (floor 9/2)
4
1/2

* (floor 9 2)
4
1

* (ceiling 9/2)
5
-1/2

* (ceiling 9 2)
5
-1
```

You can perform arithmetic across multiple number types, as Common Lisp takes into consideration the law of propagation of floats; while normally division returns a rational number, as stated in the previous exercise, if you supply a Real number represented as a Float, you will get a float back instead of a rational.

Common Lisp also supports Floor and Ceiling division, which return the quotient and remainder as multiple values---although sometimes that remainder can be deceptive.  These functions also can take a single rational number, rounding down or up to the nearest integer and returning the remainder as a ratio.

## Exercise 1.6.12

**Even More Arithmetic**

### In the REPL

```lisp
(mod (+ 10 2) 2)

(mod (+ 55 20) 60)
```

### What You Should See

```lisp
* (mod (+ 10 2) 2)
0

* (mod (+ 55 20) 60)
15
```

## Exercise 1.6.13

**Exponents**

### In the REPL

```lisp
(expt 2 2)

(expt 10 28)

(exp 28)
```

### What You Should See

```lisp
* (expt 2 2)
4

* (expt 10 28)
10000000000000000000000000000

* (exp 28)
1.4462571e12
```

## Exercise 1.6.14

**Logarithms**

### In the REPL

```lisp
(log 2)

(log 2 10)

(log 2 2)
```

### What You Should See

```lisp
* (log 2)
0.6931472

* (log 2 10)
0.30103

* (log 2 2)
1.0
```

## Exercise 1.6.15

**Trigonometry**

### In the REPL

```lisp
pi

(sin pi)

(cos pi)

(tan pi)
```

### What You Should See

```lisp
* pi
3.141592653589793d0

* (sin pi)
1.2246467991473532d-16

* (cos pi)
-1.0d0

* (tan pi)
-1.2246467991473532d-16
```

## Exercise 1.6.16

**Pseudo-Random Numbers**

Pseudo-Random number generation is available with the `random` function; you pass it a limit, which can either be a positive integer or float, and it returns to you a number of the same type as the limit.

### In the REPL

```lisp
(random 10)

(random 10)

(random 10)

(random 10)

(random 10)

(random 10)
```

### What You Should See

```lisp
* (random 10)
3

* (random 10)
1

* (random 10)
5

* (random 10)
9

* (random 10)
6

* (random 10)
2
```

<ul class="pager">
  <li class="previous"><a href="/book/1-05-0-lookups-trees/">&laquo; Previous</a></li>
  <li><a href="/book/">Table of Contents</a></li>
  <li class="next"><a href="/book/1-07-0-arrays/">Next &raquo;</a><li>
</ul>
