use std::collections::HashMap;
use std::path::{Path, PathBuf};
use path_calculate::Calculate;
use fontpm_api::font::{DefinedFontStyle, DefinedFontVariantSpec, DefinedFontWeight, FontDescription};

#[derive(Clone)]
pub struct Generate {
    target_file: PathBuf,
    fonts: HashMap<(FontDescription, DefinedFontVariantSpec), PathBuf>
}

#[derive(thiserror::Error, Debug)]
pub enum GenerateError {
    #[error("relative_path error: {0}")]
    PathError(std::io::Error)
}

impl Generate {
    pub fn from_font(target_file: impl AsRef<Path>, desc: impl AsRef<FontDescription>, map: HashMap<DefinedFontVariantSpec, impl AsRef<Path>>) -> Self {
        let target_file = target_file.as_ref().to_path_buf();
        let desc = desc.as_ref();
        let mut fonts = HashMap::new();
        for (spec, path) in map {
            fonts.insert((desc.clone(), spec), path.as_ref().to_path_buf());
        }
        Self {
            target_file,
            fonts
        }
    }
    fn sorted_entries(&self) -> Vec<(&(FontDescription, DefinedFontVariantSpec), &PathBuf)> {
        let mut vec: Vec<(&(FontDescription, DefinedFontVariantSpec), &PathBuf)> = self.fonts.iter().collect();
        vec.sort_by(|((adesc, aspec), _), ((bdesc, bspec), _)| {
            adesc.name.cmp(&bdesc.name)
                .then_with(|| aspec.cmp(bspec))
        });
        vec
    }
    pub fn generate_css(&self) -> Result<String, GenerateError> {
        let mut stylesheet = String::from("/* Generated by FontPM -- If there is an error, please report it at https://github.com/tecc/fontpm */\n");

        let target_parent = self.target_file.parent().unwrap();
        for ((desc, spec), path) in self.sorted_entries() {
            let relation = path.related_to(&target_parent).map_err(GenerateError::PathError)?.to_path_buf();

            let rule = format!(r#"
@font-face {{
    /* {} */
    font-family: "{}";
    src: url("{}");
    font-style: {};{}
}}
            "#, format_args!("{} v{} - {} {}", desc.name, desc.version, spec.weight, spec.style), desc.name, relation.display(), match spec.style {
                DefinedFontStyle::Regular => "normal",
                DefinedFontStyle::Italic => "italic"
            }, match spec.weight {
                DefinedFontWeight::Fixed(w) => format!("\n    font-weight: {};", w.clone()),
                DefinedFontWeight::Variable => String::new()
            });
            stylesheet.push_str(rule.as_str());
        }

        Ok(stylesheet)
    }
}