use rstest::fixture;
use crate::{Resource, EmbeddedResource};

#[fixture]
pub fn content_fixture() -> [&'static str; 6] {
    CONTENT
}

#[fixture]
pub fn embedding_fixture() -> [[f32; 768]; 6] {
    EMBEDDING
}

#[fixture]
pub fn question_fixture() -> [f32; 768] {
    QUESTION
}

#[fixture]
pub fn resource_fixture() -> Resource {
    let content = content_fixture();
    let embeddings = embedding_fixture()
        .iter()
        .enumerate()
        .map(|(i, x)| EmbeddedResource {
            id: i.to_string(),
            title: content.get(i).unwrap().to_string(),
            url: "".to_owned(),
            embeddings: x.to_vec(),
        })
        .collect();
    Resource { embeddings }
}

pub static CONTENT: [&str; 6] = [
    "The Amazon rainforest,[a] also called Amazon jungle or Amazonia, is a moist broadleaf tropical rainforest in the",
    "in the Amazon biome that covers most of the Amazon basin of South America. This basin encompasses 7,000,000 km2",
    "(2,700,000 sq mi), of which 5,500,000 km2 (2,100,000 sq mi) are covered by the rainforest. This region includes",
    "includes territory belonging to nine nations and 3,344 formally acknowledged indigenous territories.",
    "The majority of the forest, 60%, is in Brazil, followed by Peru with 13%, Colombia with 10%, and with minor amounts in",
    "amounts in Bolivia, Ecuador, French Guiana, Guyana, Suriname, and Venezuela. Four nations have \"Amazonas\" as the",  
];

pub static EMBEDDING: [[f32; 768]; 6] = [
    [
        0.01960003957247733,
        -0.03651725347725505,
        0.03361894761373059,
        0.013358129772869822,
        -0.05039944845462328,
        0.011364077497303499,
        0.027246834147945893,
        0.0337673762995586,
        0.011207834391821972,
        -0.04518483954588946,
        -0.007357094491310686,
        -0.01916516226651759,
        0.03960826265216422,
        0.028729191200402637,
        0.03444573262203905,
        0.0099097155685966,
        0.03338002508318172,
        -0.018434076402928646,
        -0.014041041657085597,
        0.045625574662820696,
        0.010017783770468452,
        -0.06431224305421371,
        -0.06274850882592578,
        -0.03136774409101856,
        -0.010889489445650923,
        0.042478578937357854,
        0.004337046808934617,
        -0.04804668982469325,
        -0.015289684251701633,
        -0.050598007728506485,
        0.07534821104666616,
        0.023637620352219314,
        -0.007066742820416473,
        0.06672620092858811,
        -0.02463106553566214,
        0.015878200164670564,
        0.02704371838809086,
        -0.038414308055811704,
        -0.004208145854111844,
        0.029319007514134725,
        -0.007353839330338519,
        -0.004308401946921444,
        0.04891709509911256,
        0.024114812902344226,
        -0.039898617095767946,
        -0.019501735929285474,
        -0.013623091742723031,
        0.02373722532041083,
        -0.04156520775627354,
        -0.032497904186819415,
        -0.009351146766877011,
        0.03179220525156196,
        -0.0008196250058655802,
        -0.06670211162831026,
        -0.009119386145008805,
        0.04392187405814858,
        0.01912024252388008,
        0.0390275611588488,
        0.04165504724154855,
        0.02008504376954624,
        -0.03778152028999558,
        0.03249204452737491,
        -0.04024625571797899,
        0.009314038671183501,
        -0.0539780663250737,
        0.021868167714077028,
        -0.03459220932036757,
        -0.01180741711270707,
        0.01892168324999688,
        -0.009295159292086833,
        0.0438131542695404,
        -0.050098683411022256,
        -0.0017408080071882393,
        -0.05022042014657309,
        -0.01580919297022822,
        -0.07604544767247978,
        -0.024539272214414622,
        0.030599550131736104,
        0.08075877288233783,
        -0.035568077373949895,
        0.010700695654684225,
        -0.031023358781306666,
        -0.006665718911296324,
        0.004434698403271881,
        0.04253847315985654,
        0.02222296916920525,
        -0.08208684158952206,
        0.012481866687479014,
        0.052346629924289116,
        0.04408723105981028,
        -0.009515852181802365,
        -0.02905079038330935,
        -0.027707750893209964,
        0.03756408440972523,
        -0.04495502998728425,
        0.03275049432681077,
        0.013416069235159494,
        -0.02101403824451855,
        0.003775873739801818,
        -0.005909242422429871,
        -0.05098406131682973,
        0.015490196437320834,
        0.007772440831437422,
        -0.030925707649087655,
        0.05956115354074328,
        -0.003704913171505222,
        -0.07353839330338519,
        -0.06914471344046134,
        0.007750957415986668,
        0.031210849398800693,
        0.006740585303064919,
        -0.03668651815086175,
        -0.02183496544185552,
        0.050432650726844795,
        -0.03188985638378097,
        -0.01671214615742319,
        -0.013003978980241433,
        -0.008137658431981977,
        0.01869903837278185,
        -0.040523585820748026,
        0.03214049823321886,
        -0.03231040987237937,
        -0.015613238194817355,
        -0.02943423762468725,
        0.0011138826735261377,
        0.010112831328451631,
        -0.014796216820952387,
        -0.01841910192306747,
        0.003679523701523343,
        0.06725286785884935,
        -0.0014185566598973744,
        0.04007373773187313,
        0.0023866124888878844,
        -0.006771183076814092,
        0.004066225179636905,
        -0.022258123428926253,
        -0.009284092484170666,
        0.08101787704121956,
        -0.071166758067068,
        -0.00005598709209849991,
        0.02578596189729466,
        -0.026028139588815703,
        0.0031046795570611675,
        -0.03583825018922079,
        -0.06459412779448157,
        -0.009501530212914031,
        -0.007533518763006797,
        -0.07251565536813888,
        0.03836157481775719,
        0.03105851304102767,
        0.005917054531492122,
        0.015154925023789113,
        0.05310766474760041,
        -0.006029679681454056,
        0.018947724537774224,
        0.01070915888836456,
        -0.022896116494741024,
        -0.020613015721753163,
        -0.044393859459801836,
        -0.05721099756081257,
        -0.06425430451616054,
        -0.006598013503971896,
        0.06876126619908345,
        0.01496222448511391,
        0.060586497822934945,
        0.00220497991342448,
        0.00970985404547723,
        0.013918651486325411,
        -0.04916447993910527,
        0.007362302564018854,
        0.07079111877568807,
        0.02266956394558099,
        0.009446844987360021,
        -0.042163488227922465,
        0.023491141805417785,
        -0.04832076661496312,
        -0.01681305429908738,
        -0.017396362139348143,
        0.037217092753068,
        -0.026650507512269467,
        0.016259042907576127,
        0.023616788061386645,
        0.024881705536627014,
        -0.023870030712350845,
        -0.0035740597670647018,
        -0.012065869684852447,
        -0.07286784862784873,
        0.05409003942618105,
        -0.007017916792188715,
        -0.010755380880238235,
        0.05458871677811147,
        0.005933980536734539,
        -0.022465146860726294,
        -0.0027134208997983958,
        0.023165636799039074,
        0.06733164238468121,
        -0.006553744885952474,
        -0.05440382881061677,
        -0.05141307711487196,
        0.013660199838416539,
        -0.004484826680735808,
        -0.031204339076856356,
        0.0010253449753690402,
        -0.006754907734071506,
        0.010963704712801434,
        -0.0033416482516375394,
        0.009639544601798718,
        0.006533563257619636,
        -0.027817121344317984,
        -0.030992761007557493,
        -0.019303829166375117,
        -0.02688812686934567,
        0.036292656612540525,
        0.001794841992594602,
        0.0502516704312951,
        0.010679863363851556,
        0.035154684869795665,
        0.05757686582385696,
        0.05886196492743016,
        0.0520816661059629,
        -0.02014363481857225,
        -0.004870225909613201,
        -0.04500320489089392,
        -0.021745776619080337,
        -0.017431516399069147,
        -0.05829102706855824,
        -0.05005636816685708,
        -0.05155044084125682,
        -0.03068222863256695,
        0.0519241207512452,
        0.06819032464326549,
        0.03343731295873505,
        0.02327370407667442,
        0.0013196027007943791,
        -0.03244256829876557,
        0.0035538783697909903,
        -0.03656998671530956,
        -0.017627471174480012,
        0.021188510066596916,
        0.04759488605137283,
        -0.00436048313612137,
        0.028184943443334876,
        0.009935755932137436,
        -0.01322011445974863,
        -0.017051324318663415,
        -0.019678812249836176,
        0.06250958999232292,
        0.024371310714280593,
        -0.03857250407302923,
        0.011644663685281208,
        -0.01315891983648679,
        -0.05048212695545413,
        -0.0056546961358747445,
        -0.024473520180944446,
        -0.01236403300269066,
        -0.05140786811792729,
        0.011114739745574791,
        -0.018001803595441238,
        -0.00695997686778079,
        0.01821403417571294,
        -0.036028343456714136,
        0.05443833240783794,
        -0.007809548002894424,
        0.060941948092089984,
        -0.005504963352337554,
        0.02348072566000145,
        0.05163702701403267,
        0.05453468221505729,
        -0.03207800136072085,
        0.050676131591838515,
        0.01740482444879197,
        -0.017346233399765966,
        -0.04352736185732753,
        0.002206933056219606,
        0.029334632656495732,
        0.007633774855816397,
        0.058765615120210814,
        -0.009740451819226403,
        0.04446937697618852,
        -0.06105262356514369,
        -0.0034842200507305622,
        0.05624684512917323,
        0.02071457267744417,
        -0.012975334118228259,
        -0.020078533447601903,
        -0.014584637827417021,
        0.020490624626756472,
        0.029102871110391025,
        -0.009067956080426964,
        0.011440897262926346,
        0.023039990543070217,
        -0.027149833678226713,
        -0.0662743971552677,
        0.04169736248571372,
        0.042099039367924966,
        0.003863760313340831,
        0.009064050256954965,
        0.027130302712393705,
        -0.017059788476580254,
        -0.022465146860726294,
        0.0036098653824629154,
        -0.022790001204605174,
        -0.01810336055113225,
        0.03129483292157722,
        0.058192070914393546,
        -0.03792539433832477,
        -0.03233189282571187,
        0.020006271092187392,
        -0.01112385364175496,
        0.07995802916180672,
        -0.03468465515258793,
        0.042968139620398596,
        -0.04161273199738338,
        -0.0369365093377998,
        0.0015123025462922025,
        0.006670926984004491,
        0.005456137324109796,
        -0.054733241488940494,
        -0.045742756760872715,
        0.043972651611757584,
        0.002925000124392889,
        -0.02768952310084963,
        0.029252604818164718,
        0.042896529775956926,
        -0.02802154027764569,
        -0.016978411300749074,
        0.040733864413520234,
        -0.06525295315112901,
        0.01820491935529627,
        0.019059699487354576,
        -0.025712398216880483,
        -0.06928923445434435,
        -0.04531048395338531,
        0.03723923006584635,
        -0.08940292031319423,
        -0.02803390841056152,
        0.052458603025396455,
        -0.0033019363506490737,
        0.047944480358029395,
        0.02947590220635259,
        -0.02164617165088882,
        -0.03460522996425624,
        -0.055054191857820386,
        -0.00016861229260461836,
        -0.001859943248035383,
        0.014144552448749112,
        0.003681476844318469,
        -0.05373133214758083,
        -0.011608858763060373,
        0.008748959547519577,
        -0.05819858123633788,
        -0.002634648453498675,
        0.015774036862034206,
        -0.06568522965556244,
        -0.011675913045766718,
        0.011599093280143871,
        -0.02501451277704004,
        0.09051354944316209,
        -0.08640630895800493,
        -0.012287213237067812,
        0.024539922876914453,
        -0.054995598960321364,
        0.020214595848987098,
        -0.026257294787975064,
        0.048424928069126474,
        0.00952171184124687,
        0.10366726120694071,
        0.021564143812557807,
        0.007634425980434481,
        -0.037933860344714625,
        0.013605514612862529,
        -0.006443723772344623,
        0.036796539264469595,
        0.05948433100241092,
        0.04690611543194906,
        0.018412591601123133,
        -0.024461152048028616,
        0.051954069710967554,
        -0.06160077344873743,
        0.03330711021679436,
        0.017363810529626465,
        0.00863438102370339,
        -0.03951582051688988,
        0.022073237309904563,
        0.03934460385578368,
        0.02447612467941678,
        0.03594696997782897,
        -0.018508290745842648,
        -0.06172381520623395,
        -0.09199850914561816,
        0.02576643093146165,
        -0.01173320184555656,
        0.010322457410250998,
        0.015828722087588214,
        -0.04466532990312637,
        -0.08096384247816538,
        0.020204179703570763,
        -0.03465731253981093,
        -0.046587120747514685,
        0.019201619699711264,
        -0.021023153064935223,
        0.08894070593987652,
        -0.01728113202879562,
        0.014199889261039458,
        -0.03506289339702116,
        -0.004277153510672441,
        -0.03337221343623772,
        0.010802905121348077,
        0.03467879549314343,
        -0.014307305876174972,
        -0.041732516745434725,
        0.024558803180247626,
        -0.021250356276595092,
        -0.03624969070587552,
        0.029615217920236938,
        0.026916120144622507,
        -0.005180108084222169,
        -0.09799498516849595,
        0.03963951293688623,
        0.006625356116748901,
        0.017025935541858914,
        -0.026083475476869542,
        -0.012295676470748147,
        0.016274666201464123,
        -0.021396184160896788,
        0.014240903180204966,
        0.00799964404309729,
        -0.08530089252192775,
        0.09515527220247053,
        -0.0046605998278138345,
        0.02644413659144227,
        -0.007023124864896882,
        -0.08503006904415704,
        -0.025210467552450905,
        -0.03357272469762041,
        -0.0320832066607195,
        0.003706866314300348,
        0.03320425008763068,
        0.08662700184772046,
        0.01746146351031849,
        -0.05894594475326067,
        -0.0005195081408787713,
        0.0169439077035279,
        0.08109274759205243,
        0.027449949907800923,
        0.061932141811506664,
        -0.020133869335655745,
        -0.059549434221854276,
        0.044502575551464,
        -0.002270732177953782,
        -0.058507161623828935,
        0.025663570802297968,
        -0.0588684752493745,
        0.04905641081299691,
        0.055411597811420944,
        0.038200772453594316,
        0.001103466528109803,
        0.0334692139059569,
        -0.005504963352337554,
        -0.01696994714283223,
        0.04471806314118088,
        0.07001511501793463,
        -0.07900429842100275,
        0.0012987702944321467,
        0.04215762856847796,
        -0.014665363416511867,
        0.02507896163703754,
        -0.05773376051607482,
        0.00426217995504777,
        0.009774955416447574,
        -0.02233233962031327,
        0.03423740971371236,
        -0.004435349527889965,
        0.03003381942133584,
        -0.044044915815645104,
        -0.003916492396099715,
        0.03114705120130302,
        -0.01345643249182517,
        0.03778868127443975,
        -0.0011216948981179519,
        0.041922613709874094,
        0.03438453707454071,
        0.0038227466252344503,
        0.0033383930906653717,
        -0.05146255334348129,
        0.04140180274211134,
        0.01662686500659302,
        -0.0036840808806725524,
        -0.007568673484846053,
        -0.019500434604285813,
        0.004906682880688625,
        -0.020333729934538608,
        -0.06146340972235257,
        0.004393684484106374,
        -0.013902376605701077,
        -0.1367328499401329,
        0.006340212518562855,
        -0.04331317559261031,
        -0.023929924934849526,
        0.03816496753137348,
        -0.06324718248091019,
        -0.012709069899138883,
        -0.005308357452308605,
        0.04672383381139971,
        0.0303795079045204,
        -0.019910571947467873,
        0.042184971181254964,
        -0.028572296508184776,
        -0.020627988353141325,
        -0.028709660234569633,
        0.02314220000973407,
        -0.015270804872604963,
        0.021517272082420807,
        0.039603057352165565,
        0.046758988071120715,
        -0.03923458274217583,
        -0.03264828855709294,
        -0.01066879655593539,
        0.01458659073915302,
        -0.04811439569413593,
        -0.06163592770845843,
        -0.03268474414181361,
        0.01685992602922438,
        0.01018834884483831,
        0.001481704888072593,
        0.034838292835360614,
        -0.07001121104293564,
        0.006586295571437646,
        -0.01683323592742022,
        0.05467204594144215,
        -0.016780502689365703,
        0.05231212632706794,
        0.02235903157059044,
        0.031388576381851226,
        0.06707579153829865,
        0.07331770041366965,
        0.03276547065514496,
        0.04208992269903528,
        -0.028949884090118172,
        0.00463130430330083,
        -0.0386675488583029,
        -0.03480248791313978,
        0.02719540408336405,
        -0.0006028377663277014,
        -0.0603697089082184,
        -0.015565713953707512,
        -0.017748559096004028,
        0.012222112790333974,
        0.014325534592771812,
        0.0005663810263114037,
        -0.04323765807622363,
        0.052827728297886026,
        -0.012292422234012486,
        -0.0041645283607105055,
        0.046849478218895554,
        -0.009925990449220934,
        0.02609128712381354,
        0.011773565102222234,
        0.01989169349260771,
        0.007656560520503319,
        0.019175578411933923,
        -0.03497695788674513,
        -0.033560354716231575,
        0.025918118475207853,
        -0.008063443164831466,
        0.02094633422354888,
        -0.003046739401594116,
        0.03662857961280858,
        0.01164271077354521,
        0.07726609052216345,
        -0.028525422929574763,
        -0.00520484666064509,
        -0.020521873063005475,
        -0.000014322279374034861,
        0.009208574967783987,
        -0.019473091991508807,
        -0.016064389457164926,
        -0.016224539310354953,
        -0.041298942612947656,
        0.05592850110723869,
        -0.03265024054459244,
        -0.025320488666058755,
        -0.008337519955101347,
        -0.002024649587197244,
        0.022149405488791074,
        -0.03445289360648322,
        0.0520426078712429,
        -0.012302186792692483,
        -0.020634498675085662,
        0.03234556413210037,
        0.01293822602253475,
        0.005501708191365386,
        0.03627703331865253,
        -0.02378279572554817,
        0.013633507888139365,
        -0.01696864581783257,
        -0.00169133096987196,
        -0.02031355015467878,
        -0.01761835635406334,
        0.02771491187765413,
        0.0695554959007242,
        0.029578762335516273,
        -0.024707884377048482,
        -0.05350803660786597,
        0.0628715505834223,
        0.05447478799255861,
        -0.013767616453552053,
        -0.04602789851058575,
        -0.011316554180430162,
        -0.025987776332150027,
        0.023532806387083126,
        -0.03076295329742529,
        0.0043513687778229514,
        0.018906710618608713,
        0.004021956561617483,
        -0.030061162034112847,
        0.0165389775088175,
        0.04246099995902434,
        -0.0003020698922523716,
        -0.011085444221061789,
        0.03589683938977378,
        0.028403684345550916,
        -0.03507265703146465,
        0.004156716251648255,
        0.07907265495294526,
        -0.027173270467531718,
        -0.0033240708907179115,
        0.01273120443920772,
        -0.05533998427003325,
        0.009172769121326647,
        -0.003831860752473743,
        -0.011241687326543313,
        0.031646379215733274,
        0.020107829896351414,
        0.012985750263644595,
        -0.024565964164691794,
        0.0239390397552662,
        0.07338931025811132,
        0.04350197123205002,
        0.07467246107113105,
        0.025296401214253916,
        -0.013459036990297506,
        -0.012901118851077748,
        -0.012854896859204074,
        0.04016227589214848,
        0.0010559426335886497,
        -0.0568282009819345,
        -0.017940608047942893,
        0.0283040793773594,
        0.0010240429571919984,
        0.0071604885912817376,
        0.011899210433954588,
        0.03695148196918797,
        0.03746838526500571,
        -0.017106009544217423,
        -0.04060431233407938,
        -0.026970154707676688,
        -0.039746277041048904,
        -0.0547176181950525,
        0.03690525905307779,
        0.012601652359766864,
        0.0008313432272237383,
        -0.027442138260856924,
        0.06937516626767434,
        0.004943139389645797,
        -0.03001233646800334,
        0.018506338758343156,
        0.0813616116884316,
        -0.03941621370022535,
        0.02020352719259792,
        0.05161098572625533,
        -0.016953022523944573,
        -0.050262739087684284,
        0.024557501855247965,
        0.02689073136781801,
        -0.04517181890200079,
        -0.06001686128906268,
        0.052385037496509274,
        0.007745748881160248,
        0.031408758010184064,
        0.0021561541162558482,
        0.0345635653825909,
        -0.01123192184362681,
        0.025547691877718624,
        -0.023793862533464334,
        0.03787331545971611,
        0.06973973690266509,
        0.007083018163159059,
        0.03407205271205067,
        -0.0586913989288238,
        -0.0020337637144365365,
        0.05537513852975425,
        0.028852883620398992,
        0.0020044681899235323,
        0.05543959108669778,
        0.016318935281601797,
        -0.033248521016241366,
        0.03460913763620125,
        0.008809503508281585,
        -0.018170415758075098,
        -0.008246378220590165,
        0.05392598744646504,
        -0.0597427844987928,
        0.005523842731434224,
        -0.02095414587049288,
        -0.005476318952442634,
        -0.011888142701801916,
        0.004830514239683862,
        -0.017274621706851283,
        0.0597427844987928,
        0.06196534038678215,
        0.04226634835708615,
        -0.024402559150529596,
        -0.056084094474456885,
        -0.01804542016460607,
        -0.009044519291121957,
        -0.021106482228265902,
        -0.02773118768251497,
        -0.054202666886734245,
        -0.021740569470608675,
        -0.023993723132347197,
        0.0576907909124638,
        0.07018893063209414,
        0.044559216461463526,
        0.0009817271353790124,
        0.04977317470769752,
        0.057050195196649695,
        0.025433112429665932,
        -0.02950259415662976,
        0.030617127261596603,
        -0.01356580294293319,
        -0.006293339864189349,
        -0.035518601145340556,
        -0.007708641247584993,
        -0.003863760313340831,
        -0.02320079105876008,
        -0.026502075129495435,
        -0.011343245206470828,
        -0.08257119327561814,
        0.03792214102582562,
        0.004689895352326839,
        0.0008391553940507709,
        -0.044996043906449755,
        -0.03211510760794135,
        -0.035740595360055756,
        0.019172323250961753,
        -0.0011601046653998122,
        -0.008530870232039875,
        -0.04285486519429159,
        -0.029311846529690558,
        0.003419118679760218,
        0.01741524059420831,
        0.06064899469543295,
        -0.062051272200112154,
        -0.036096699988656646,
        -0.024012603435680374,
        -0.03501276650591199,
        -0.051107099377380234,
        -0.011454568569314849,
        -0.024954618554541355,
        -0.05029528700045994,
        -0.013419324396131662,
        0.007840797363379934,
        0.018817521795833535,
        0.04739828246193515,
        -0.03742216234889553,
        0.019366326038873123,
        0.008409782310515857,
    ],
    [
        -0.025333738636002502,
        0.010855743756584628,
        0.004185596536018679,
        -0.03496242321042482,
        0.003319209286538382,
        -0.011383476366718328,
        -0.05674384482866805,
        0.0216525630200104,
        0.03021283261496375,
        -0.06804120085655785,
        0.027054211056771083,
        0.004607135039640819,
        0.022881564279953977,
        -0.01968603166094824,
        0.06388279920096143,
        0.06753677800603645,
        -0.01180048158589725,
        -0.05314878858625707,
        0.02757676318407571,
        0.034997388332466996,
        -0.005613387603526837,
        -0.021342398140481483,
        -0.03343685740690568,
        0.005913838715783462,
        0.0297174772985762,
        -0.0026580869655353024,
        0.05494437269918648,
        -0.015236244979357933,
        -0.031577167352740944,
        -0.028079888742083163,
        0.06655707505468167,
        0.002902203584735755,
        -0.026558207094126568,
        0.03385127286729377,
        -0.024459579697031192,
        0.036732107804951085,
        -0.034490378618897,
        0.02257334340870115,
        0.040024767478958244,
        0.034760396928307256,
        -0.03159205918972379,
        0.016932111313247674,
        0.0077340293982154085,
        0.04549893648384151,
        -0.01475901990500103,
        -0.0057668508407673945,
        0.02341900824424145,
        -0.03683635838581026,
        0.00925376896839073,
        -0.008121249012933826,
        -0.016289119475587092,
        0.010671846751250302,
        -0.03145608092662283,
        -0.025665918878372017,
        -0.01872186919283326,
        0.01934867190099706,
        -0.0017250046193084644,
        0.043584854738337425,
        0.03990303433707798,
        0.023075173604689483,
        -0.013618082026914662,
        0.016775411118830425,
        -0.0350776814727043,
        -0.0028834254203836973,
        -0.045147980248926606,
        0.029963536236869776,
        -0.037406178678596476,
        -0.040399685015484646,
        0.03312086629403295,
        -0.0008832230344112664,
        0.025514398201536434,
        -0.028578481498271104,
        -0.0031845242136498823,
        -0.02283623819225366,
        -0.025090268491252354,
        -0.025025517764749674,
        0.051186143313252275,
        -0.03820134176984442,
        0.07800271625010147,
        0.014874279132527913,
        0.039917277527803864,
        -0.05191525645498879,
        -0.007419332682114387,
        -0.035608003841628835,
        0.05370048153275486,
        -0.01670418358223215,
        -0.0908567151870004,
        0.015620227154275957,
        0.019716465911665714,
        0.01818183257401273,
        -0.027350775600346645,
        -0.01637847821946342,
        0.02168882350407169,
        0.01977150431876721,
        -0.06598466017686758,
        0.06687759896937391,
        0.012646148351932366,
        -0.021883081475064164,
        0.041196141538256896,
        -0.029425443626325338,
        -0.01364333579480306,
        0.0006870232269493722,
        -0.05090252659838347,
        -0.0038547118969801362,
        0.06586681505153949,
        0.01724162754651962,
        -0.037153646791196926,
        -0.06278330132451802,
        -0.0009945972014561582,
        0.008824460705772386,
        -0.019729415670867285,
        -0.06781197390253357,
        -0.043676804206251994,
        0.06567579645084222,
        -0.05194115597339194,
        -0.03697233857940603,
        -0.024943280616236286,
        0.025439933225137764,
        -0.014926081065076424,
        -0.03525639896045696,
        0.01281320951200776,
        -0.04345340734656117,
        -0.05881527348435154,
        0.00957947336591205,
        0.0548181067554867,
        0.04073056796534023,
        -0.04991440668964661,
        -0.019704162868226294,
        -0.009883162400592772,
        0.029347741210126274,
        -0.022458729931689014,
        0.02454181684554458,
        0.003674052572758055,
        -0.029677979374714513,
        0.03514761557777827,
        0.03622704016916235,
        -0.08064461384482813,
        0.04689435218809833,
        -0.04519784106894125,
        -0.02490507805439372,
        0.04153868081578974,
        0.021382544710600134,
        0.016715837979414602,
        -0.005837431178979815,
        -0.013974868356163014,
        -0.0009253121649348947,
        -0.031835529334473936,
        -0.057943705269418465,
        0.017724033103705596,
        -0.00492765943695677,
        -0.00959371945238015,
        0.009150165103293707,
        -0.007671866982632454,
        0.020820494659433822,
        -0.005099253398851676,
        0.013089702701018809,
        -0.04240377415112203,
        0.009336651867418867,
        -0.05388567390010831,
        -0.060930088080448884,
        -0.06250357262620139,
        0.006561363355263557,
        0.028136222511203783,
        -0.01865323044977841,
        0.05091612500384201,
        0.000007122766573782287,
        0.014003359563851805,
        0.004774843880725773,
        -0.010671846751250302,
        0.01193710946050258,
        0.032249298079099875,
        0.031531841265040626,
        0.01603658565769273,
        -0.015245958264006518,
        0.02591845269626638,
        0.01252635632258597,
        -0.009633866022498801,
        0.005730589391458696,
        -0.07090390072594747,
        -0.024831906509519355,
        0.03335979977597396,
        0.02088783610997474,
        -0.009553572882261498,
        0.03633582355184104,
        -0.004805924847205398,
        -0.0283518491292747,
        -0.037827717664842324,
        0.031092819717773692,
        -0.015236244979357933,
        -0.04765454629631447,
        0.029284930630910058,
        0.0034383536348752174,
        -0.0075715010399595275,
        -0.008946842493909615,
        0.05905291739575787,
        0.051432848967308,
        -0.02312438577844716,
        -0.008191828868522544,
        -0.012471316950237066,
        0.02059968852378124,
        -0.03583075205506269,
        0.005332362095319978,
        -0.008558975198181636,
        -0.01219806120102641,
        0.0017865193538818586,
        0.0210995766419883,
        0.022687306308961507,
        0.0206035746098386,
        -0.0317053773047168,
        -0.032938264650717745,
        0.04082899231285558,
        0.010974888104921464,
        0.018185070013813125,
        -0.03506797108379793,
        0.04902405862117852,
        0.009747182206997005,
        0.0639825169800011,
        -0.01911038320432339,
        0.0011040288684240366,
        0.02263032389358392,
        0.020588034126598785,
        0.05577191018843835,
        -0.021044538234886803,
        -0.009433133171905543,
        -0.0017411926629019112,
        -0.03843703974247985,
        -0.051482709787322646,
        -0.05228110838787617,
        -0.050965985823856655,
        0.0563100047294776,
        0.06339974692801331,
        0.024439506411971866,
        0.01613954184178019,
        -0.02330698742176237,
        -0.0686246195548026,
        0.06943920342386328,
        -0.037881462640419514,
        -0.015599506188207072,
        -0.005410064994142745,
        -0.033500961417646205,
        0.0034726725237789395,
        0.01711212223252464,
        -0.016632956045633873,
        0.002051356818495271,
        0.02060163060156251,
        -0.016396609426741475,
        0.04588227097774998,
        0.030641495127067344,
        -0.009711568438697865,
        0.009753658051845197,
        -0.017766770397862487,
        -0.0040489691440370535,
        -0.006343795142035069,
        0.00734875282652567,
        0.007705538673150319,
        -0.06299892987308375,
        0.0063664581858852275,
        -0.020343917266086482,
        -0.0363299934575076,
        0.011648960909061671,
        -0.02726854038232807,
        0.05103203191237845,
        -0.0020312835334359448,
        0.03432849890689808,
        -0.016638782278977693,
        -0.014905360099007538,
        0.03345110059763157,
        0.08357077293969095,
        0.004901758470682515,
        -0.018569053153978558,
        0.02378486017712883,
        -0.0022080577368480733,
        -0.011831562552376877,
        0.015307473481203615,
        0.042440681350945475,
        0.011795948784077739,
        0.04512143208426649,
        0.00103474377157481,
        0.05121593084820759,
        -0.047813839145264776,
        -0.01263708274829334,
        0.03069718218042581,
        -0.00004856431930522474,
        0.0028109027630781516,
        -0.02023254315936955,
        0.012891559609216383,
        0.02387162819696692,
        -0.009098363170745196,
        -0.039952246510835655,
        0.052425505538854,
        0.02534733704146104,
        -0.038372935731739326,
        -0.04115858183118686,
        0.01062198786173047,
        0.027485460431923292,
        0.0016006799087984816,
        -0.007574090798750361,
        0.05025241316535995,
        -0.07572990609756775,
        -0.0018007649577262547,
        -0.04525806092411923,
        0.005517553980049721,
        0.0062809850454425544,
        0.011204759844213074,
        0.03394451576673265,
        -0.013844068645396319,
        0.0024217405879547564,
        0.007052186352455294,
        0.04296517748652619,
        0.059194723822697456,
        0.0000628098515101649,
        0.026845707964557916,
        -0.013910763380175084,
        -0.0643101644205511,
        0.004000404651288936,
        0.04524769802796627,
        0.010023027714998493,
        -0.01958825595968985,
        -0.08177972162858106,
        0.03289876286586643,
        -0.014852911450696874,
        -0.025526701244975854,
        0.04539468590298234,
        0.038903258100663855,
        -0.025277402936387073,
        -0.038734251001817564,
        0.008459904134904127,
        -0.014578360339467097,
        0.003008397691247314,
        -0.002923571990502349,
        0.03515473910363602,
        -0.040219674096207685,
        -0.025594042695516774,
        0.05390380124639674,
        -0.11654205677367997,
        -0.022303325099290887,
        0.05131370654946598,
        0.021229081955983282,
        0.057632246958365246,
        0.004177179096012915,
        -0.045012000055330834,
        -0.005350492819974327,
        -0.11747449349004799,
        -0.02075444664041721,
        -0.029462352756643596,
        -0.03312734117363374,
        0.016553956336920875,
        -0.024618870615486677,
        -0.05442959081350175,
        0.04253716362067956,
        -0.031458023004404104,
        0.02156579500017231,
        -0.002923571990502349,
        -0.023412533364640664,
        -0.006141119731036833,
        0.014085594781870383,
        -0.03272393339466594,
        0.11559861623688128,
        -0.024001133510961902,
        -0.038396893172361196,
        0.01030470042159121,
        -0.05793269758799816,
        0.04811493456016504,
        -0.055633987917061305,
        0.010792286461606257,
        0.025217183081209096,
        0.10717691755956194,
        0.014053866134381197,
        0.007249681280624457,
        0.0011707238445146523,
        -0.008840648870021759,
        0.03165940257075952,
        0.03226030576052018,
        0.02635488351949507,
        0.04783714793962968,
        0.013759242703339503,
        -0.017411279430633256,
        0.0578219721275382,
        -0.031171817495991877,
        0.02939306729782659,
        0.045412817110260385,
        -0.04785203977661253,
        -0.05106441017137201,
        0.004306683927384194,
        0.019814888328686255,
        -0.019753375041983972,
        0.03716983205970408,
        -0.030600699910691725,
        -0.08817531966841205,
        -0.16854861405032637,
        0.03876079916647768,
        -0.019669195815689308,
        0.02089301755805122,
        0.00198725184250734,
        -0.006456464128147414,
        -0.11472187043486134,
        0.025748800812152747,
        -0.05578939274945943,
        -0.02352649819539583,
        0.02460656950254207,
        -0.010625225301530863,
        0.022006111909458358,
        -0.072606241939438,
        0.02616645371234123,
        -0.044703777253583196,
        0.022311742056672947,
        -0.002768166192856815,
        0.004151925328124516,
        0.03853999689181472,
        -0.034579737362773325,
        -0.021511403308832965,
        -0.030692649378606295,
        -0.04433728053542848,
        -0.07178518705176617,
        0.03498508721952239,
        -0.0173238627645382,
        0.017883972668418048,
        -0.08465991178097103,
        0.004710738904737842,
        0.020120520406395656,
        0.02999008440152989,
        -0.01686153049241155,
        0.017533661218770483,
        0.0035050486109658332,
        0.008470911816324428,
        0.03727602857933415,
        -0.012996458836332525,
        -0.09239070180889122,
        0.09644096486707629,
        -0.03428770369052246,
        0.010689330277518795,
        -0.02810190410492376,
        -0.0030045125704373605,
        0.04130233419689735,
        -0.020277869247069872,
        -0.006021975382699998,
        -0.005072057553182004,
        0.006705113307855528,
        0.06976685472540291,
        0.06170776474968611,
        -0.015920031068146726,
        -0.029855399569953242,
        0.015255023867645545,
        0.03875432428687689,
        0.04681146832382279,
        0.03464643209755209,
        0.005603027120492394,
        -0.00855250031858085,
        0.03470924267676831,
        -0.019032680788124326,
        -0.04425828082671548,
        -0.013399218934290757,
        -0.05928472735184113,
        0.06063740576194106,
        0.049903399008226317,
        0.013963212993733154,
        -0.018560634266101685,
        0.03758942896816865,
        -0.00803707075188657,
        -0.00350181117116544,
        0.004181064216822869,
        0.022030717996337195,
        -0.05199101679340658,
        0.06027544184560585,
        0.03035464097239815,
        -0.029676035366438428,
        -0.0026639146467502322,
        -0.045966452134539454,
        -0.02136247142554081,
        0.01124555313009388,
        -0.025954713180327674,
        0.03125081527421006,
        -0.0351126504557361,
        0.05307885834217273,
        -0.04072150622269083,
        -0.011099212936087371,
        -0.024534693319686825,
        -0.03623610577280138,
        0.04655958122169058,
        -0.010849268876983842,
        0.022180943311153656,
        0.04879419074287654,
        -0.03781606133716506,
        -0.014201501690406826,
        -0.0158526886523584,
        0.06560132568295912,
        0.044527005704601434,
        -0.08285720028119424,
        -0.05760958294926768,
        -0.04464291261313788,
        -0.00021821566800818732,
        -0.009985470903670675,
        -0.06742345796054863,
        0.015275097152704871,
        -0.02143693833243429,
        -0.06159120919489271,
        0.005252068955082674,
        -0.045586353149936566,
        -0.03420481982624692,
        0.01286695448758495,
        -0.05429620134394423,
        -0.003713551461867147,
        -0.014663186281523915,
        0.05723207854969266,
        -0.013765717582940288,
        -0.009446731577364082,
        0.01724098083075747,
        -0.038686336120573823,
        -0.021473198816495585,
        0.010197859116693794,
        0.0025674335835754077,
        0.05052113804324591,
        0.033214753978739166,
        0.03071531145720905,
        0.04499451749430975,
        -0.004785851562146072,
        0.02800412840366537,
        0.02013606088963547,
        0.004367550980948028,
        -0.016550072181358327,
        -0.08564349888788837,
        0.008533722395540644,
        0.028215868935678932,
        0.01132390419254991,
        -0.03828357698786299,
        0.032977762574579426,
        -0.061495375571415586,
        0.07144458599102498,
        0.004585119194176516,
        0.021392905676258282,
        0.008833525344164006,
        -0.003453894359426882,
        0.03422619040382018,
        -0.01666468372787565,
        -0.0232843234126648,
        0.07355292570752153,
        0.029691575849678242,
        -0.010967117863301557,
        -0.07797098663736488,
        0.053569034140978605,
        -0.008215139593382263,
        -0.03266630426352621,
        0.022413403843988697,
        0.004248406632611192,
        -0.06447916765840776,
        -0.0010787753418474893,
        0.006616402727612461,
        0.038780872451537,
        0.047768511127069646,
        0.005234585911437885,
        -0.013044375648071083,
        0.02687225612921803,
        0.014042858452960898,
        0.02312244370066589,
        -0.019581781080089063,
        0.000028491066295129148,
        -0.010430320614776237,
        0.11623254440139764,
        0.0033056111223916944,
        0.007883607514646012,
        0.004368198661957588,
        -0.03949897791185322,
        -0.03658382263742108,
        0.02993245527039015,
        0.00788684543707011,
        -0.050131324808747026,
        0.0073999070780646214,
        0.015556769859297586,
        0.021214188188505623,
        0.09598834877534045,
        -0.04148040303839304,
        0.02260636645296205,
        -0.02328238133488353,
        -0.05753058710154431,
        -0.0016375887978048885,
        -0.040587468106876336,
        0.004201784700268051,
        -0.022245047321894182,
        0.004524899338998539,
        0.04479248735120256,
        -0.046648939965566905,
        -0.03353204238412583,
        0.001909548943684573,
        0.008741576841496844,
        0.003517351654405253,
        -0.05036119654803864,
        0.01585333633336796,
        -0.01450713280286882,
        -0.05576090250701805,
        -0.011713065885049601,
        -0.0407234483004721,
        -0.01770914126672275,
        0.028374511207877453,
        -0.08001715669892293,
        0.039875837526160904,
        -0.017181407691341643,
        -0.004671887214014607,
        0.04721746682682882,
        0.00517889885807942,
        0.03366543185368336,
        0.018308747163969474,
        0.002719601700108697,
        0.019050163349145412,
        -0.07807264649418581,
        0.021460895773056165,
        0.011101802694878204,
        0.014426192946869363,
        0.02606673593330157,
        -0.013601893862665288,
        -0.06672931380184391,
        0.009793803656716442,
        -0.050003116787265975,
        0.0025603107816532097,
        -0.006099678281522765,
        0.03747611278367045,
        -0.002972136241938615,
        -0.014867156571917567,
        0.0280131940073044,
        0.021730265436209463,
        -0.057566200869843445,
        -0.009648111143719494,
        0.022051437997158675,
        0.006747202438379157,
        0.032623566969369314,
        0.026965499028656907,
        -0.044757522229160386,
        -0.010663428828620836,
        0.009089297567106169,
        -0.05943236387311416,
        0.007806552296832805,
        -0.0490279466377307,
        -0.040588112892143674,
        0.03103454194037699,
        0.025577855496514805,
        0.014512313285697893,
        -0.04871713311194482,
        -0.009255711046172004,
        0.04904801992279002,
        0.029436451307745637,
        0.06764685482023945,
        0.010023027714998493,
        -0.008634087855589867,
        -0.05878484116412888,
        -0.02059321364418045,
        0.02368967519990868,
        0.001956818195069497,
        -0.05492300598260284,
        -0.01834112349246822,
        0.06580918592039975,
        0.006594386882148159,
        0.011654789072900304,
        0.029495375800904496,
        0.007032113067395969,
        -0.005055221707923072,
        -0.007472429494058315,
        -0.02191027828598124,
        -0.032642342961914705,
        -0.07566126542401809,
        -0.06232809450874394,
        0.042825309276378246,
        -0.01626257131092698,
        -0.013611607147313874,
        -0.000997187322214769,
        0.04262069227022244,
        -0.023629455344730703,
        -0.0238878173264637,
        0.031088933631716335,
        0.05476177105587127,
        -0.016469132325358873,
        -0.026118536900602674,
        0.09259143465948448,
        -0.006071187073833972,
        -0.048111699050859456,
        0.013160930237617085,
        0.013237338257044436,
        0.019908133158619945,
        -0.06038034107272199,
        -0.0025933345498496633,
        -0.03501487475447771,
        0.022956029256352647,
        0.029561423819921107,
        0.021653858382029517,
        0.002319431843565002,
        -0.010552054721903906,
        -0.015108035992629477,
        0.004713976344538235,
        0.05066294447018549,
        0.01772144431016217,
        0.060875049673347385,
        -0.05276221858304302,
        -0.05742633652068513,
        0.00753459190964127,
        0.0034331736346698474,
        0.0017735688707447308,
        0.02151658282641463,
        -0.00880309205869394,
        0.004445901560780355,
        0.05156300292755995,
        0.007182986545845693,
        -0.03753309326855322,
        -0.0025959247912642,
        0.04050458424260079,
        -0.025598575497336285,
        0.03678520413427131,
        -0.01617450792906977,
        -0.02244513345672529,
        -0.04544713744703522,
        -0.005445678762441884,
        -0.031568101749101914,
        0.04825544755558032,
        0.04243032231578214,
        0.008801796696674821,
        0.008737045004924737,
        -0.01715615488870065,
        0.003570448707661032,
        -0.009501123268703426,
        -0.00016447115997270865,
        -0.02685024076637743,
        -0.06413015350127413,
        -0.08555543743652597,
        -0.04759367779487953,
        0.043949413239700495,
        0.08158157564103641,
        0.01638689517684548,
        -0.00595722320832621,
        0.01655719570721608,
        0.012353466998671944,
        0.01365304811420424,
        -0.02838228337999217,
        -0.009429248051095589,
        -0.026844412602538798,
        0.011085614530628833,
        -0.09612044095238405,
        -0.004762540837286353,
        -0.014033792849321871,
        -0.001027620969652612,
        0.0005497480936302246,
        -0.04784038730992489,
        0.0005393876709237446,
        0.025342155593384562,
        0.03142629339166751,
        -0.04059976921982094,
        -0.009159877422694886,
        -0.045392095178944096,
        -0.0013682187726838148,
        0.040252048494211615,
        0.006768571085457602,
        0.0036513892875960443,
        0.015492017202300095,
        -0.0032324410253884404,
        0.030555373822991408,
        -0.012721908690350156,
        0.061800359002868024,
        -0.050456385386248415,
        -0.038953115059688875,
        -0.02483255515577632,
        -0.025772111537012466,
        -0.048116880498935935,
        -0.02294825901473274,
        -0.04178409303832117,
        -0.050960804375780176,
        0.02615803675495917,
        0.02484097211315838,
        0.03302179330026063,
        0.027292499753444752,
        0.013069629415959483,
        -0.001706873894654115,
        0.017134139525860053,
    ],
    [
        0.005018911861136282,
        0.011653518782652434,
        -0.016296763516431327,
        0.01327836016600071,
        0.0388387467969106,
        0.0025088025048579238,
        -0.023611490839267304,
        0.05445734882500722,
        0.06168257073982786,
        -0.013324746570698475,
        0.02279612904982442,
        0.007721754519018092,
        0.02116279504204707,
        -0.004554391114499132,
        0.0735967646380037,
        0.07407958048603962,
        -0.007155966974410799,
        -0.05329572106404656,
        -0.0014811081892834562,
        0.06592597132214222,
        -0.0058989527012137345,
        0.004152590778340541,
        -0.03199376571658977,
        0.041787243691024906,
        -0.016415016381746948,
        -0.005298538777717948,
        0.0461188444669088,
        -0.020729634091405536,
        0.008779502051996851,
        0.014862695157555852,
        0.05241567913265267,
        0.010105116475748004,
        0.0030974564025444376,
        0.06793432094137049,
        -0.007204967081949434,
        0.023740851123169298,
        -0.007009620350334219,
        -0.02457973078159787,
        0.034377781371859,
        0.04932933151284015,
        0.0030491097207160202,
        -0.010056770066748695,
        0.019941059754961396,
        -0.0009381876116208737,
        -0.04087649648066205,
        -0.022920264110677532,
        0.029254336675517915,
        0.040557666204100194,
        0.023309650176829313,
        -0.008040581521630667,
        -0.014372041474946608,
        -0.030821030537042726,
        -0.04808799400010617,
        -0.08459564525788925,
        -0.019927339724850578,
        0.03843694591509379,
        -0.007565607873433786,
        0.03557272920594594,
        0.05426200318470843,
        -0.0003906937360595363,
        -0.033240981059595526,
        0.005743459208510538,
        -0.02444971789047298,
        0.021198727726697782,
        -0.08185360360529297,
        0.06142450059661391,
        -0.009033648912658428,
        -0.00599891292059255,
        0.0352898359793005,
        0.0286232152904203,
        0.0028439629675930784,
        -0.06988387043155239,
        -0.011927918293552362,
        -0.004085950904917104,
        -0.06716273403641736,
        -0.01830184464296298,
        0.012051398564549723,
        0.03537411616426696,
        0.09725856516452114,
        -0.018319485772993315,
        -0.01621052332716333,
        -0.016513016596824217,
        -0.04371849835263716,
        -0.02226627528118427,
        0.06269262337971378,
        0.019278578301039052,
        -0.028041094012861322,
        0.024021784345513185,
        0.016793297211945207,
        0.003309136594282233,
        -0.05828000327234507,
        -0.0248867988497176,
        -0.00007578674205005279,
        0.03260659227311772,
        0.0019998553578245534,
        0.07344061908373582,
        -0.017788978305813836,
        -0.033834860180330925,
        0.05734704559007517,
        -0.014292333905806144,
        -0.0008232008517471898,
        -0.04324483210151892,
        -0.03873421177907294,
        -0.008985955656540229,
        0.01541214896921146,
        -0.0020122686729294897,
        -0.034349033914558715,
        -0.043056671688570566,
        -0.007811261018349856,
        0.010480783603105396,
        -0.03210025855609611,
        -0.06072216863207062,
        -0.06964017510830502,
        0.0784608405245836,
        -0.008090234782050412,
        -0.023806183872343192,
        -0.020726366690025345,
        -0.0012060545706005445,
        0.02591841371955337,
        -0.00745519393400792,
        -0.02902566877946,
        -0.020669526565280528,
        -0.0631512643862754,
        0.005059418252929411,
        0.03037676435044767,
        0.04027673516438916,
        -0.04876419548413932,
        0.00007513341012484085,
        -0.005696419105273449,
        0.06484405331295487,
        -0.011341878098706718,
        0.03627179595174189,
        0.011159597698662998,
        -0.02043498083895083,
        0.01779028461157606,
        0.01185670553147383,
        -0.018644846487049848,
        0.034835767588564874,
        -0.03517223572120778,
        -0.017788324607274513,
        0.05305333532051068,
        -0.001475881465674484,
        0.05114037112220238,
        -0.0032549098995491793,
        -0.0345685543963318,
        -0.03906545250603411,
        -0.022802009062729056,
        -0.018327325790199495,
        0.031944113001828235,
        0.019285765711022338,
        -0.01521876224189779,
        0.04891381060091252,
        0.005435085380286802,
        0.009906503434069023,
        -0.03411644819253057,
        -0.05798273740836592,
        -0.021634501288863756,
        0.029020443556411115,
        -0.01916294022988073,
        -0.08365353361343597,
        -0.06465719811445231,
        -0.004839244618275217,
        0.05088949057160457,
        0.017909844873970328,
        0.0643351026191431,
        0.03531858125396793,
        0.002482669296056723,
        -0.04073602659554083,
        -0.021923929318269576,
        0.011745638984825068,
        0.06579792061795171,
        0.004311351126765717,
        0.02522587959388495,
        0.030834097959930647,
        0.027445254887824375,
        -0.003107909849762382,
        0.03276862002292309,
        0.007621141146758167,
        -0.003027549673399021,
        -0.008343074791291554,
        0.026502492818781045,
        0.03563610195081829,
        -0.01795557758012877,
        0.010416103461154397,
        0.03897267751400573,
        -0.02731915982266972,
        -0.032645139751925734,
        0.014790174998398674,
        -0.004274764197917466,
        -0.04416210974937646,
        0.04724258172155006,
        0.022262355272581177,
        -0.022424383022386547,
        0.01832536578589795,
        0.012934053107468044,
        0.06627028165462416,
        -0.013456066858902015,
        -0.02916678908917127,
        -0.020274260486224112,
        -0.01584661622561521,
        -0.005604952055981926,
        0.004731444927348434,
        -0.03856630619899579,
        0.012021998500026542,
        0.055121135493375356,
        -0.019113285332486345,
        -0.015428482702163146,
        0.034138660847070464,
        -0.0530415752947014,
        -0.03294632562450797,
        0.047141311922459746,
        0.008076514751939594,
        -0.00981699693473726,
        0.00795891449384687,
        0.019944327156341587,
        -0.009359009626714963,
        0.03429938119979718,
        -0.0008323475157519761,
        0.06265211807923707,
        0.014217854833663847,
        0.023422677819096047,
        0.04846954004905177,
        -0.04628544483254016,
        -0.01453864074926154,
        0.053998707818445596,
        -0.015734895980427124,
        -0.024844331362306498,
        0.022514543220257636,
        0.011513051080164064,
        0.04819514381210112,
        0.037441918519764486,
        -0.0036273104440136964,
        0.03880346671948278,
        -0.03433662128152655,
        -0.023576210761839484,
        -0.001394214710719795,
        -0.010864290747668967,
        -0.05520541567834181,
        0.004413271077616971,
        0.017475377617566575,
        -0.0005474938755613373,
        0.053171586003193445,
        0.011417664567927664,
        0.004446591150743242,
        0.014120507225809474,
        -0.038608116713918283,
        0.040291760408945776,
        -0.0022063088259533766,
        -0.0288780158496212,
        0.02042322081314156,
        0.01403818704514457,
        0.02002076732410186,
        0.022039569572060755,
        0.03559232924896139,
        0.009711156702453809,
        -0.04135342795052763,
        -0.017197057006747132,
        -0.03216101868944402,
        -0.001008747698269231,
        -0.018177711764742723,
        -0.020679326586788255,
        0.06020864750506573,
        0.007921021259236398,
        0.030814497916915193,
        -0.05145658660460694,
        -0.0015229215689115734,
        0.028022801366924515,
        0.06403195674225932,
        0.005257378687385496,
        0.009072195845808225,
        0.04821474385511657,
        -0.01244470573062102,
        -0.0376888790617592,
        0.004404777634700572,
        0.029453602324419795,
        0.0007715874506113457,
        0.024716931082706048,
        -0.04619463421007903,
        0.007868101143094672,
        -0.03462082190525063,
        -0.006509819799098359,
        0.05203935830675596,
        0.01987703440286615,
        -0.05496564036369752,
        -0.041328602684463284,
        0.060864594156227685,
        0.04161018851403007,
        0.03828994559247789,
        0.0365220238951168,
        0.021503181000660215,
        0.05064971962222599,
        -0.02338282294320939,
        -0.03002004356756641,
        0.028430482261645955,
        0.021207220351126863,
        0.04221713505765339,
        -0.018600418995337202,
        0.05554384163265341,
        -0.027746440760406613,
        -0.031827165350958406,
        -0.03522450104749376,
        -0.010102503864223561,
        0.027261014483479108,
        0.022514543220257636,
        0.08189280369132387,
        0.0017319884670521592,
        -0.020951114577480168,
        0.001935828641583773,
        0.029984110882915697,
        0.06866278775168962,
        -0.004330950896952064,
        0.009078075858712861,
        -0.05862169881066971,
        -0.08313348204893639,
        0.025381372540929936,
        0.005633698967623997,
        0.040892176515074416,
        -0.0107440778780518,
        -0.06906720342766372,
        -0.006909660130955404,
        0.038919106973273956,
        -0.0012295746222190892,
        0.01507372192358343,
        0.022587715986637714,
        -0.03158281742048813,
        -0.05494016030777743,
        0.042270709963650864,
        -0.0500930784005871,
        -0.006875033752066911,
        0.028660455372149663,
        0.037895329938011514,
        -0.003412363396553921,
        -0.04391384399293594,
        0.0351219282165905,
        -0.07142508423715711,
        -0.0013125479557651062,
        0.08504775582695619,
        -0.060130247333003914,
        0.0518133073874882,
        -0.02405053180281347,
        -0.035674649429626296,
        -0.014000946963415207,
        -0.03796850488702445,
        -0.0006840406827520205,
        -0.04297565672235146,
        -0.01415709470031594,
        0.0020220686944372166,
        -0.019987447250975586,
        0.0019181886028698644,
        -0.0019606552717936457,
        -0.03293783300007889,
        -0.0016829883595135245,
        0.01737411109242554,
        0.008259448304864426,
        0.03369373987061966,
        0.021875581817953838,
        -0.04597707373260749,
        0.1261529136557776,
        -0.032381191778440004,
        -0.019354365861576427,
        -0.00003920003913840506,
        -0.052701839760678305,
        0.0,
        -0.02931444201901007,
        0.03547799639224886,
        0.008830462163837031,
        0.12772026230715816,
        0.014588948253878823,
        0.03294828562880951,
        -0.016890643728483153,
        0.04118878758919781,
        0.00651765981630454,
        -0.002551269173781705,
        0.06867846778610198,
        0.05557389648703234,
        0.003989257268431162,
        -0.04143770813549408,
        0.06667469296533254,
        0.015354655145927319,
        0.0027237492794885926,
        0.024829303935117033,
        -0.043299712221962196,
        -0.03620385059104355,
        0.006662700134618899,
        0.01608834936192819,
        -0.01590214895328138,
        0.022001022093252744,
        0.0027877762685584793,
        0.008722661927252035,
        -0.14944165857514566,
        0.013893800425369534,
        -0.036385476201231524,
        -0.023606916040808463,
        0.038380760580204724,
        -0.043528377935387254,
        -0.08059397126398932,
        0.013857867740718819,
        -0.020681286591089802,
        -0.03785220984337752,
        0.05044260959085222,
        0.002879242772191789,
        0.014609854602656498,
        -0.05305137531620913,
        -0.014534720740658449,
        -0.031953258233480206,
        0.06233263737808518,
        -0.00968959665513681,
        0.02300519690286688,
        0.05148925188787746,
        -0.019201485526055887,
        -0.029232121838345167,
        0.009201555584052008,
        -0.038847237238706825,
        -0.07015435102516565,
        0.02483910395662476,
        0.004835978308211451,
        -0.0330162331721407,
        -0.06498647883711192,
        0.0036606302443108613,
        0.013376360381077979,
        0.002548656016599049,
        -0.03389431509923304,
        -0.01785627105928928,
        0.03945287856788435,
        0.01875329933085774,
        0.048683835307775966,
        -0.00227948227440622,
        -0.05305529532481222,
        0.10617723079587699,
        -0.02781700091526225,
        0.021100074904397616,
        -0.013799720218895355,
        -0.06872354788503753,
        0.094952945316048,
        -0.018828432101539363,
        -0.03233153906367847,
        0.009418462362984222,
        0.04735626197103969,
        0.0959597327371866,
        0.006931220178272404,
        -0.024971731641906948,
        0.026038626589170537,
        0.01502145550598103,
        0.07011253614497745,
        0.0275412950986001,
        0.007933434437926781,
        0.02520954476961684,
        -0.021662596138941143,
        0.040809856334409506,
        -0.0009185875686054198,
        -0.04249284087431554,
        -0.01912374014384982,
        -0.016315709860907458,
        -0.007660341232789075,
        0.04871062057814185,
        -0.018708871839145096,
        -0.007900768063339834,
        0.016776309780454197,
        0.027185226922941737,
        -0.025800813461460646,
        0.027029081368673855,
        0.056448058405521584,
        -0.033500354234622415,
        -0.0011413744968568234,
        0.0401140548073609,
        -0.04879294075880675,
        0.020683246595391346,
        -0.03168735243832579,
        -0.001296214632057079,
        -0.024375237727014252,
        -0.038907997372054734,
        0.021901714481096828,
        -0.02085572697392734,
        0.04301616202282816,
        -0.025602198237148814,
        0.03455352915177519,
        -0.02434714505956972,
        -0.009057822662816297,
        0.027567427761743087,
        0.02463591829911979,
        0.029609097454097632,
        0.028607535256007937,
        -0.016839029918103647,
        -0.0398579446684485,
        -0.08180198870359703,
        0.020042327371418856,
        0.07677523682525457,
        -0.03855911660637965,
        -0.0565048985302664,
        -0.038680636873075463,
        -0.01578389608796576,
        0.0015686549571427813,
        -0.06022890233793693,
        0.047612367744686385,
        0.0057447655142727596,
        -0.10577346990975864,
        0.0002632935928736394,
        -0.028840123160668942,
        0.019998552486929107,
        0.0036632436743226243,
        -0.052884120160722026,
        0.01621574964152864,
        0.02691213371780403,
        0.055364176026766986,
        -0.023188782517356395,
        -0.00138310811062074,
        0.0003887337317579909,
        -0.028928975961461382,
        -0.021903021878175475,
        0.012149398779626992,
        -0.015359228853069734,
        0.037929959590849285,
        0.01741592378998089,
        0.022531528469115793,
        0.0283684147312194,
        -0.024633958294818244,
        -0.022981022061392586,
        -0.007689087598772932,
        0.031192778747113452,
        -0.012394399317320165,
        -0.07839812038693415,
        0.006019819269370225,
        0.03346964677302059,
        0.047006075990918814,
        -0.0416820538833315,
        0.03378912747417249,
        -0.027181961704194395,
        -0.017275457178808948,
        0.0000738267377485074,
        0.06787029558927524,
        -0.01730943095047454,
        0.048952355897087685,
        0.020777981591721274,
        -0.025413385216977558,
        -0.008051034696019504,
        0.04716940677253713,
        0.014215241130822977,
        -0.03701006278356875,
        -0.05382949265865695,
        0.03338863398943433,
        -0.07796038900309948,
        -0.010304383215966311,
        0.06197134179674507,
        -0.009116622246204445,
        -0.03247984678337302,
        -0.042986111533714934,
        -0.00006664006525640217,
        0.06518378448171226,
        0.06429917211712524,
        0.01020507669512682,
        -0.04348786826964485,
        0.04089152172521866,
        0.026418867423670344,
        0.012918373073055682,
        0.020892314448433806,
        -0.01640129635163613,
        0.002524482539270287,
        0.10804119052138095,
        -0.017626950556008467,
        -0.009384489682635052,
        -0.007003740337429583,
        0.022335528038961255,
        -0.05579929698171006,
        0.006029619290877952,
        -0.010220756729539184,
        0.01000907626497228,
        0.01399506695051057,
        0.039721399157196066,
        0.029839723171824237,
        0.05894183430167736,
        -0.025296439748740584,
        0.011726691549032511,
        -0.03786854466764563,
        -0.060635273652946874,
        -0.011993252134042684,
        -0.03130384420244579,
        -0.015444815343798407,
        0.02179914165019357,
        -0.022197022523407283,
        0.029093616322791192,
        -0.009935903498592203,
        -0.038823066762498235,
        -0.03857937143925086,
        0.020425180817443104,
        -0.017721684461021972,
        -0.06964801512551119,
        0.018391351142294742,
        -0.021098114900096072,
        -0.08725013714150605,
        0.032163631300968464,
        0.0031366564885753454,
        -0.02765562795531263,
        -0.010180904036285377,
        -0.0893943818473967,
        0.004753004974665433,
        -0.027248599667814087,
        -0.01249435953669898,
        0.0489013957852475,
        0.0394724786108998,
        0.0025996158556101222,
        0.03779275928974111,
        -0.02336583769435123,
        -0.03365911458304759,
        -0.03920591802588963,
        0.02865653536354657,
        0.016927883810212516,
        -0.029742375563969865,
        0.026943493786628758,
        -0.0025368959907897766,
        -0.05895097953332934,
        -0.006481073433114502,
        -0.03535843612985459,
        0.024631345683293802,
        0.03160437746780513,
        0.0031085630026434926,
        -0.030176191304467144,
        -0.02632217242303887,
        0.02840173480434567,
        0.033842047590314214,
        -0.02916678908917127,
        -0.017666150642039375,
        0.00783086106136531,
        -0.013782733878720772,
        0.014054520778096255,
        0.0166900695911862,
        -0.027818960919563793,
        -0.0812571075077674,
        0.02110595491730225,
        -0.06830084174575947,
        -0.010802878007098162,
        -0.027360319913002175,
        -0.025898813676537916,
        0.05375828207921127,
        -0.018535086246163308,
        0.011053104859156648,
        0.022103594924156002,
        0.05009438361503289,
        0.03832261087574841,
        0.013677547344976644,
        0.032698712475290354,
        0.04018722539110812,
        0.008810208967940467,
        -0.05771879161751304,
        -0.021942221964206383,
        0.0649740640214469,
        0.026578280379318416,
        -0.006289646164444164,
        -0.00331893661578996,
        0.039671748625067386,
        0.0202781804948272,
        0.017715804448117333,
        0.019400098567734867,
        0.031415564447633874,
        0.012258505321974209,
        -0.0014190414773442212,
        -0.08496608170561562,
        -0.012176838839848627,
        -0.06025307281414552,
        -0.053468851867172595,
        0.062252277201721815,
        0.000784000765716282,
        -0.04736932721129476,
        -0.04341796290464496,
        0.033386019195277034,
        -0.02142543561845415,
        0.02890872331122303,
        0.030953003432469164,
        0.04719815204720456,
        -0.00857435529887391,
        0.01660644310475907,
        0.08082917178017476,
        -0.06840341675929558,
        -0.007641394342654732,
        0.05765280626111625,
        0.01308627974444926,
        -0.037880959483310654,
        -0.057546966028832795,
        -0.0044544311679494235,
        0.025777293409842104,
        0.034922007777832866,
        0.05538900129283132,
        0.03966129381370391,
        -0.008358754825703918,
        0.012523759601222161,
        -0.007342167383057611,
        0.051684597528176245,
        0.011183117750281541,
        0.01680570984497738,
        0.02224471523386727,
        -0.05703409532641795,
        -0.025091944511524116,
        0.018190778096314215,
        0.0003423470201274806,
        -0.005370405238335804,
        0.0280920541247015,
        0.029453602324419795,
        0.03393351518526395,
        0.07167727000220071,
        0.007584554217909915,
        0.0018456685801272385,
        -0.05881051401347382,
        0.007491780862856172,
        -0.04740068728011949,
        -0.032684992445179535,
        0.015672829541316994,
        0.00022932023045170376,
        -0.041233867688133356,
        -0.008312368421006152,
        -0.0437407088245442,
        0.02088120702984743,
        0.04685123565109673,
        0.005890458985468229,
        0.031178403927146884,
        -0.03583014237667128,
        0.021554795902356146,
        -0.060361527840586265,
        0.014922147893825113,
        -0.01635817625700213,
        -0.019860046971375136,
        -0.06342174497988866,
        -0.02356249073172867,
        -0.003217016664938706,
        0.014981601721410798,
        -0.005175058506720589,
        -0.024300757563555528,
        -0.004590977770518277,
        0.03202643318249314,
        -0.025643358327481266,
        -0.0013282279901774693,
        0.0004900004956244954,
        -0.01591782898769374,
        -0.00282632292887917,
        -0.017582524155612247,
        0.05079933037373348,
        0.012301625416608208,
        0.03388712768924976,
        -0.023975396849498995,
        -0.04390012396282513,
        -0.01870299182624046,
        0.026999026514294925,
        0.0012223878943085716,
        -0.054405083498721245,
        0.012830826578025463,
        -0.058223163147600246,
        -0.02110660752452515,
        0.02208856967959939,
        0.0010812677210118571,
        -0.03828406557957325,
        -0.02342659782769914,
        -0.01518413477169287,
        0.030036376209201673,
        -0.019941059754961396,
        0.050232889130586864,
        -0.03786592987348834,
        -0.020936087150290703,
        0.013167293619351945,
        -0.017067696722845135,
        0.00446031118085406,
        -0.015106388298170379,
        -0.001296214632057079,
        -0.08617932873353792,
        -0.023840156552692362,
        0.01747080391042416,
        0.0021644954463252595,
        0.048683835307775966,
        0.04158274845380844,
        -0.03336249914365849,
        0.027042146608928924,
    ],
    [
        -0.018073327672876845,
        0.010703699657335138,
        0.03129403971429953,
        0.03700175294864097,
        0.022677974086194225,
        -0.017775617318628242,
        -0.03469394352440145,
        0.05509556233764214,
        0.02553878007514475,
        -0.03554026220209483,
        0.008819415360492057,
        -0.016688642751131567,
        0.03828841997369229,
        -0.021400083543675362,
        0.07454552758416985,
        0.045767771119519816,
        0.017544470709787617,
        -0.07508536064641289,
        -0.0007878004483837329,
        0.0295611692824329,
        0.06590239971003695,
        0.0401595377147406,
        -0.0015353696522836678,
        0.018512945664466016,
        -0.0011191594562172775,
        -0.0011462240685731367,
        0.019468985029783777,
        0.019201996028390945,
        0.02563021377794328,
        0.06422438901131715,
        0.1004134733861799,
        -0.006211700850705464,
        -0.015529251821430981,
        0.07189538648793563,
        -0.00675811383218641,
        0.021846284507829238,
        -0.013925123081775502,
        -0.08258737969034818,
        -0.014523471320001391,
        0.004573192872696761,
        -0.006389449659267523,
        -0.014614174667031148,
        0.003657383835864758,
        0.008900609426559144,
        -0.015967406658821168,
        -0.028284011538344245,
        0.02988740870090023,
        0.09028251554725773,
        -0.009852260550610677,
        0.034457677097195105,
        -0.01599008188491325,
        -0.033846161692509084,
        -0.014748034956277311,
        -0.06542401423882832,
        -0.03636682951943266,
        -0.0001938413483388214,
        -0.038333770425876455,
        0.03247756780296047,
        0.029171293648286273,
        0.021330594711200135,
        -0.041397927979209805,
        0.02388417984360867,
        -0.04654606167627552,
        -0.07789130445955515,
        -0.08478181786945017,
        0.065096311666162,
        -0.020276536227857703,
        -0.03298155480865914,
        0.060853745842533453,
        -0.030133917207499458,
        0.0555285973566666,
        -0.026096166102452707,
        0.00020335053388502973,
        0.002707926832446342,
        -0.012715992237293808,
        -0.060472646690919285,
        -0.047516731362252394,
        -0.00817132720748506,
        0.05481540372712774,
        -0.05217623995137804,
        0.02679545610310908,
        -0.01161950838883446,
        0.0064779582749988065,
        -0.01869727744559278,
        -0.026418013615661658,
        0.0005486075601130797,
        -0.00877918411600826,
        0.016823964973245994,
        -0.0015617029165395453,
        0.023285100028283292,
        -0.05154058616640096,
        -0.0321476717203264,
        -0.0040787145372954,
        0.05834624640340072,
        -0.017341850722504284,
        0.055752428194512316,
        0.01974548331213928,
        -0.036993706821877276,
        0.024781701590282146,
        0.011272788475573522,
        0.04030875745908393,
        -0.017104852718198445,
        -0.04808728333468976,
        0.047202194734715486,
        -0.008477084299162708,
        0.005401956142663783,
        -0.033921503141780246,
        -0.011273520052673014,
        -0.039038182687559986,
        0.036941770954466976,
        -0.02860220483004789,
        -0.03290109354102226,
        -0.05382279405615043,
        0.025655084956664555,
        -0.013103674361472677,
        -0.0460596300783032,
        -0.03248122446712721,
        -0.008071846156592122,
        0.030995595340290005,
        0.010185813908076845,
        0.025798454526873387,
        -0.05838867054718692,
        -0.03138181675293133,
        -0.05238251293883596,
        0.05408027011258844,
        0.027069759654166112,
        0.008542185501037062,
        -0.031698544447774556,
        0.0007073380357493069,
        0.020111953364425054,
        -0.034554230618359355,
        0.021272806226993905,
        0.002881287094409491,
        -0.048778527208582445,
        -0.016268774898567227,
        0.018003837619070904,
        -0.005473640317102841,
        0.06588118763814385,
        -0.054255824189852024,
        -0.010153628790356735,
        0.020833921033834944,
        0.0033955152635030054,
        0.04660311736205154,
        -0.012918611003246422,
        -0.05384034702121535,
        0.05260927008507964,
        0.0012478993927655289,
        -0.04875438882829138,
        0.04800535647019247,
        0.028490289411125032,
        -0.06047776650928501,
        0.024026816615387296,
        0.006251932095189262,
        -0.030294109387004442,
        -0.0028110651581713776,
        0.013988761740116231,
        -0.004306203565971248,
        0.06140820556144344,
        -0.03468736055183674,
        -0.019335124740537616,
        -0.05151644778610989,
        0.010049759498197567,
        0.014882626822622966,
        0.04381984633233993,
        0.025474408618373814,
        0.034451091681968964,
        0.021203317394518677,
        -0.031124335811170446,
        -0.06549496866816397,
        -0.005191290639282122,
        0.042561704707515184,
        -0.04850715240858482,
        0.0029185923358280024,
        0.0001711655688312593,
        0.006829067040191335,
        -0.029038163714808884,
        0.04941564659041994,
        -0.03889188680895318,
        -0.045353021863990484,
        0.036897881213820365,
        0.005117411428211911,
        0.04041555305964107,
        0.0013737134025816641,
        0.02858391662389132,
        0.01426745475377021,
        0.001841858473727148,
        -0.03290913966778595,
        -0.002171022979261728,
        0.038873601045458055,
        -0.03662869992379049,
        0.09610141384475325,
        0.02850638166465241,
        0.011253769913648182,
        -0.0028315465689630224,
        -0.03782173729341409,
        0.022156431672769192,
        -0.0005127652057274563,
        -0.014660256696314803,
        0.015896453450816243,
        0.010264813242845458,
        -0.043315861769302695,
        -0.03880045310615465,
        -0.034225797689924266,
        0.004509554214356032,
        0.03299106408962181,
        -0.006393838511199112,
        -0.033496514249519466,
        0.0392349512793781,
        -0.03417898408354112,
        -0.0209034123089716,
        0.01888599868931649,
        0.030910014254287407,
        0.04825405848675945,
        0.031357678372640266,
        0.04330488933414104,
        -0.0213181591218395,
        0.020041000156420127,
        0.020242156989504476,
        0.04163273125221717,
        -0.0021885785396544262,
        -0.0357962775469953,
        0.03613860921899,
        -0.003748086877561837,
        -0.0254978166428961,
        -0.032383938147532744,
        -0.02618174718845531,
        -0.04630540578647222,
        -0.028024337086615607,
        -0.008992043740023034,
        0.07616282349028546,
        0.002356818067253814,
        -0.03133280719391899,
        0.04238834902554223,
        -0.057783740557727045,
        0.036504348915506996,
        0.014948458990931451,
        -0.0869104116669365,
        -0.028262799466451147,
        -0.026926390083957277,
        -0.031092151304115696,
        0.0823313692307706,
        0.007405471018759274,
        -0.03539835578608498,
        -0.008610213011309379,
        0.05154204687793851,
        -0.051232630068767326,
        0.006254857792921869,
        0.00031892387647192746,
        0.007622719494705639,
        0.0525353942326689,
        -0.04197067346161493,
        0.02348479248583794,
        0.005193484759915238,
        0.002787658049647126,
        -0.007463257670969431,
        -0.024988709818831705,
        0.03945731896302483,
        0.0039558263778782105,
        0.019280265007390784,
        -0.01493456146870255,
        -0.05169346257491103,
        0.06420464253628447,
        -0.032781862351104495,
        0.04065840490207356,
        0.025655084956664555,
        0.0006451625316075608,
        0.060567739500545996,
        0.012293198076331505,
        0.011961108483729676,
        -0.01747132277048422,
        0.003579115772862957,
        -0.03538445704252536,
        -0.02030213776234775,
        0.0059615356799394975,
        -0.027880967516406775,
        0.018021393026797262,
        0.056426847016447415,
        0.005848888073251791,
        -0.0016253414222139344,
        -0.03135987188260802,
        -0.03797534650035437,
        -0.019083497636903382,
        0.03056402591812596,
        -0.0001733599947970541,
        -0.005495585187426147,
        -0.005886193620002983,
        0.03712171449432751,
        0.053970550646294774,
        -0.04038629486098428,
        -0.004216963678473152,
        0.0009838362417716416,
        -0.03211695036747063,
        0.03342263595444729,
        0.007681968690449421,
        0.01236415128433643,
        -0.021582223036165086,
        -0.009007405027116278,
        0.040918809709570965,
        -0.015335410759341545,
        -0.03148056622672477,
        -0.021558815011642796,
        0.009350467665545118,
        0.051438907941548116,
        0.022912047003432814,
        0.060164695478608525,
        0.047437730806153065,
        0.011364954976802262,
        0.03232688490441815,
        0.027515960618319988,
        0.035587075808477975,
        0.02216813446369962,
        -0.019369505200221557,
        -0.06249298661897237,
        -0.08480156434448285,
        -0.032668483777982656,
        -0.0002026190712852931,
        0.020136091744716117,
        0.01779756096762083,
        -0.04836670426145108,
        -0.006352875689615823,
        0.08910557287382295,
        -0.003977039060436668,
        0.07427415278417222,
        0.06633177562223325,
        -0.0004527841275515052,
        -0.07335395336874524,
        0.03647728422681797,
        -0.03668209650273834,
        0.027052204246439754,
        0.03193188700924437,
        0.025683612799552565,
        -0.011722646103894138,
        0.02602374851891808,
        0.07268977418354158,
        -0.0505435797048424,
        0.0070199824045481625,
        0.012718918545691774,
        -0.056385155671091425,
        -0.024146047805305066,
        -0.03365963518008385,
        0.007720737391399594,
        -0.07405543687936937,
        -0.015157661950779487,
        -0.005443649930681205,
        0.0458306745367689,
        0.010972883390026446,
        0.004338388683691357,
        0.00004023122062968236,
        -0.024777312127685195,
        0.03638365457139025,
        -0.05416000346711798,
        -0.005205188772176383,
        0.009317550970725516,
        0.012635529748326213,
        -0.018363724698791967,
        -0.009675974896247599,
        -0.055275503434841236,
        0.05493317420550796,
        -0.03886116545609742,
        -0.0630876769717008,
        -0.027447934940043744,
        -0.0443779651580781,
        0.007486664474161003,
        -0.037979005607182544,
        -0.0343903793320262,
        0.005762574188818982,
        0.015596547143938449,
        0.04127869678929203,
        0.05089176643629447,
        -0.07528505310396753,
        0.020543525229250535,
        0.02046233116318345,
        0.0111074752563721,
        0.024677832298122974,
        0.002005709302061286,
        -0.0017789515451522504,
        -0.003001980522528205,
        0.01660232764536804,
        0.04646632832174598,
        0.03605010182458943,
        0.0482130950545108,
        -0.021826535590135122,
        -0.0346463971195881,
        0.03127063168977724,
        0.0452813395215475,
        -0.032051848554930915,
        0.0225353373144156,
        -0.05831625540631373,
        -0.013103674361472677,
        -0.11519515111838809,
        0.03642754186937543,
        -0.03306567518312419,
        0.01968550253929601,
        0.03263190980833095,
        -0.07134239236588606,
        -0.05936665356149727,
        0.0013546949933226624,
        -0.014171630367044013,
        -0.03677865246656403,
        0.02680716133670094,
        0.0009267810903278067,
        -0.0016443598314729358,
        -0.05997670336932287,
        -0.013786874551263109,
        -0.03224861714674903,
        0.012298319116027944,
        -0.04995766827730786,
        0.03121796546659817,
        -0.012992490519649312,
        -0.02826060595648339,
        -0.07901338984250454,
        -0.0018791638678119994,
        -0.036630163077989476,
        -0.02467271003709582,
        0.004077251383096417,
        -0.01748156362854638,
        -0.025566575119602555,
        -0.08684457705596657,
        0.0006451625316075608,
        0.064643525286782,
        0.038948212138960436,
        -0.058740511500144296,
        0.04857737403949025,
        0.09843336409194527,
        0.027194110662449603,
        0.06036000091622766,
        -0.06762283455254295,
        -0.03736968371246429,
        0.06970315158944715,
        -0.05717075955618061,
        0.04394200383065566,
        0.052376660322040025,
        0.005489733181295575,
        0.058817313660953004,
        0.01687370610935782,
        -0.007524701598011685,
        -0.006270950046449245,
        -0.012947870423233926,
        0.07855694924161923,
        0.020122925799586706,
        -0.011231826264655591,
        -0.030805412163698028,
        0.01779390430345409,
        0.058699548067895645,
        0.035898686127616916,
        0.005745749747526756,
        0.04383155156593179,
        0.020474035175444594,
        0.016268774898567227,
        -0.02455567479980724,
        -0.054454055935869126,
        0.014023872555568948,
        0.011171844270481603,
        0.02106872430684231,
        0.0024811692282036457,
        0.012082533183615202,
        -0.016287061883393076,
        0.021257446771896737,
        0.00871993430959912,
        -0.005122531857242992,
        -0.014846052852971265,
        0.028267188929048094,
        -0.02729432329044203,
        -0.0034693941692405376,
        0.0367581707504397,
        -0.025380048607177314,
        -0.0001989616819534401,
        -0.04545030818358495,
        0.04646340201334802,
        -0.015170827895908897,
        -0.0016560633857350617,
        0.021490056534936346,
        0.0002070079232168827,
        0.05236130086694286,
        -0.08974123154412289,
        0.0038219660886320483,
        0.011810424363856646,
        0.007950420846041237,
        0.03984719703983289,
        -0.02113894593774774,
        0.024603221204620593,
        0.09252741911158674,
        -0.006527698800445275,
        -0.059368847071465024,
        -0.05006592703206398,
        0.031152133298289683,
        0.04610936755042288,
        0.03159540795404559,
        -0.024598101386254873,
        -0.023565256196136253,
        -0.03071836548083501,
        -0.022250061328196916,
        -0.030354821736947205,
        -0.013106600669870643,
        -0.014575405966080975,
        -0.08082305999251845,
        -0.0007314767977062195,
        -0.033873956736966894,
        -0.0017730996916880176,
        -0.025070634240667567,
        -0.06964902221206909,
        -0.006495513682725165,
        0.010767338315675867,
        0.03567412249134099,
        -0.007957004429271301,
        -0.002461419241845153,
        0.02789340310576741,
        -0.04401953878989458,
        0.0037853921189803484,
        0.019047655244351175,
        0.013326774843549615,
        0.038251113205610386,
        0.022012329304130152,
        0.042811872320942586,
        -0.024097771044722938,
        -0.029905696907056797,
        -0.03420751192642914,
        0.031865322042505674,
        0.0067625026841179985,
        -0.09104617944734648,
        0.015132790772058214,
        0.0215134621167972,
        -0.013848318478305364,
        0.012411697689149783,
        -0.017063158930181024,
        0.02523741183539869,
        -0.05837038478369178,
        0.04202553563742319,
        0.030524525640076294,
        0.012094968772975839,
        -0.061500372062672184,
        -0.004713636134507628,
        0.049587541560855354,
        -0.02801848691248111,
        0.020253859780434903,
        0.04025243457173812,
        0.06201606185930129,
        0.035992313340383206,
        -0.016191969073766372,
        0.06896435886807967,
        -0.042799439174243385,
        -0.007504951458986853,
        0.09658565681808068,
        -0.008713351337034415,
        -0.03695420654382761,
        -0.03615470147251738,
        -0.028176485582018337,
        -0.023625968546079016,
        0.04622201821043738,
        0.023116128923584413,
        -0.08317549439849622,
        0.04749112738510091,
        0.09048513919853322,
        0.025162800741896307,
        -0.03420019615543422,
        -0.02693370585495219,
        -0.03149373217185418,
        0.04418265972045896,
        0.013282154747134227,
        -0.006152451044296325,
        0.01141396331448388,
        0.002113967904151063,
        0.011539045899866865,
        0.020503294595432098,
        -0.006338245979622071,
        -0.03169781409200578,
        0.053424137054148474,
        0.006148062192364735,
        -0.013486968244385316,
        0.014546146546093472,
        -0.040619634979792665,
        -0.04705589885610869,
        0.013538172534696125,
        -0.05587238729753742,
        0.02195746957098332,
        0.004690229025983377,
        0.05745384447509297,
        -0.00225880017055986,
        -0.0046251275187763445,
        0.018146475612180243,
        0.07620232132567369,
        -0.05091736552812308,
        -0.07360631449214039,
        -0.02771492272010586,
        0.022897417904104423,
        -0.03248927303655233,
        -0.018594139730533102,
        0.01542757603923957,
        -0.026371200009278513,
        0.04396468149940918,
        0.0023860771819086382,
        -0.011940626767605352,
        -0.013832956580546762,
        0.021754850805030704,
        0.022447557833122373,
        -0.05576705485117927,
        0.013849050055404856,
        0.05160129607368229,
        0.08246669267421575,
        0.02191284947456793,
        -0.012076681788149989,
        -0.013707873995163778,
        -0.0015821841746648502,
        -0.04988817944483264,
        0.0335089498388801,
        0.020658366956571358,
        0.01813550317701859,
        0.05588409253112928,
        -0.003579115772862957,
        -0.04180535902108279,
        -0.009333643224252893,
        -0.028625612854570177,
        -0.05794978291136652,
        -0.004072131259398015,
        -0.046501439137198695,
        -0.0043347314088592595,
        -0.03347018235926064,
        0.04464787924653818,
        0.024471554425342188,
        -0.05765573044261538,
        0.008657758805457374,
        -0.018576584322806743,
        0.00897229421166356,
        -0.014649285482483865,
        -0.002879092668443696,
        -0.037758098635073366,
        -0.02552561413001534,
        0.012072292325553041,
        -0.08429245507775703,
        -0.05885754429477144,
        -0.012376586873697066,
        -0.038729501119480446,
        0.05538742129642552,
        0.01822254863855089,
        0.00003364793080524015,
        0.0361232473212314,
        0.0679388318916174,
        -0.02745817457677519,
        -0.021251594155100806,
        0.03649703314451209,
        0.0328469641636442,
        0.023712282430511825,
        -0.016153201594146915,
        0.01926051486836595,
        0.0806460452037173,
        0.020992652501802378,
        -0.0164889490722462,
        0.03464347081119013,
        0.035646325004221754,
        0.04162395232702327,
        0.02554463269194068,
        0.016323635853044777,
        0.03143301982191142,
        0.03185288645314504,
        0.012497279996483101,
        -0.031061429951259927,
        0.017984087480046068,
        -0.046523384007522005,
        -0.056995205478917024,
        0.06632080318707159,
        0.03162759246110035,
        -0.06338612134571033,
        -0.004464202540841153,
        -0.00661693899327605,
        0.023922216967459352,
        0.0005917646931624091,
        0.019907871444273455,
        -0.013522810636937523,
        -0.03878143454422931,
        0.03489070967355813,
        0.017962875408152973,
        0.01688687205448723,
        0.01824961332723992,
        0.057683527929734614,
        0.012041570972697273,
        -0.01010535141910925,
        -0.03717072405333985,
        -0.0064033477921617826,
        -0.0039682613565734885,
        0.04782102346773498,
        -0.010062194476892845,
        0.015187651726535764,
        -0.013587912449477236,
        0.012881305456495231,
        0.0554122924751468,
        -0.003282136385048487,
        0.023297535617643927,
        0.026987104876561474,
        0.041958968228023065,
        -0.012147634996154915,
        -0.003451107489747367,
        0.02068616200102916,
        -0.02923932176922395,
        -0.0031073132742190355,
        -0.035578299325945514,
        -0.0360435164093633,
        -0.03176657256871223,
        0.06840551212923417,
        0.03868414822463485,
        0.03050404392395197,
        -0.0318770248334361,
        0.018706055149455957,
        0.012473141616192038,
        -0.0011279371600804568,
        0.003413070671229364,
        -0.03252292069780606,
        -0.0041811212859209435,
        -0.016868585069661383,
        -0.07489590782558969,
        0.029717707240432576,
        0.07483738654295324,
        0.04135915805692891,
        -0.004156982294964522,
        -0.0060207854863486366,
        -0.016444328975830813,
        -0.03736529669252877,
        0.014311343273086106,
        -0.017149472814613834,
        -0.07061383928125003,
        -0.01708217627077565,
        -0.007912384332855913,
        -0.021311576149274796,
        -0.004603914836217888,
        0.0017182388898768076,
        0.008869887463038018,
        0.04735800233694639,
        0.05256538522975589,
        -0.008785767699238323,
        -0.014748034956277311,
        0.018659240321742095,
        0.033272683411673755,
        0.005625056014075362,
        -0.0025243263230863905,
        0.06806171730304048,
        0.020381868674215854,
        0.0164019036107139,
        -0.04536180078918438,
        -0.032570464659957984,
        -0.015303225030956078,
        0.055917742635044454,
        -0.005796953427172207,
        -0.05252442179750724,
        -0.02294203800051981,
        -0.058101198608339046,
        -0.015870851916326197,
        0.028420798135988374,
        -0.04275335592362901,
        -0.03600621208394282,
        0.018311057254282177,
        0.013531588340800703,
        0.021002161782765046,
        -0.046905215957566374,
        0.03032922020245716,
        -0.02386077181908638,
        0.005920572858356208,
        0.0038014846778404035,
        -0.06269999240486049,
        0.019190293237460517,
        0.001695563205785708,
        0.0005317836149864579,
        -0.026287812433243667,
        -0.002461419241845153,
        -0.018979627123413497,
        0.034040001533267805,
        0.04315712785867382,
        0.011764341113242276,
        0.021762164133364185,
        0.0295553191082984,
    ],
    [
        0.01107290578709138,
        -0.0051766791694271615,
        -0.03214277908255849,
        0.010856364562097555,
        0.033938263780024396,
        -0.011810498080039556,
        -0.04034802157927873,
        0.041356288601867276,
        0.015196203248435187,
        -0.05478031885204003,
        0.04648334702205587,
        -0.010803733078698925,
        0.03286082142108721,
        0.013666132603038122,
        0.06688254569636094,
        0.003754126324301142,
        -0.011472904649126243,
        -0.007545094378135283,
        0.016811237439816308,
        0.05716752991608323,
        -0.020789423588617007,
        -0.050193110958344674,
        -0.057626928107770196,
        0.013320268275136613,
        0.012528541448055085,
        0.014182672459814694,
        0.06901035902922141,
        -0.05492768782914564,
        0.023502951085115877,
        -0.029013462455415726,
        0.02882850073815316,
        -0.012212752547663307,
        -0.008786444419694053,
        0.09640427402310463,
        -0.04967581854671394,
        0.07787424453481827,
        0.026260085583684204,
        -0.01436312296589948,
        0.017674392120374947,
        0.04552470126344929,
        -0.0331089421538123,
        -0.02489392365039553,
        0.022540547699672484,
        0.04986529353412789,
        -0.020765362423019778,
        0.022208968530671672,
        0.06534795972183888,
        -0.027577375719384668,
        0.006434571004962325,
        -0.05940887917170277,
        -0.030754812618748278,
        0.01807514456675078,
        -0.04107508634016301,
        -0.055356254950008295,
        -0.0030278120592974155,
        0.01985784715605071,
        -0.001229320359561108,
        0.06184571561767515,
        0.02518189375835327,
        -0.012115760477854245,
        -0.04910589465168457,
        0.04624349718631898,
        -0.015563872106858232,
        0.02041122925710368,
        -0.04433748472653707,
        0.055593098684275734,
        -0.033429993790749214,
        -0.040205165872324505,
        0.03756908255018886,
        -0.01716010889867407,
        0.029056319991091435,
        -0.010192456757702183,
        -0.050876566658185886,
        -0.058344970446298924,
        -0.026039785702366765,
        -0.04074275976374206,
        -0.029839777980158376,
        -0.0078044921093180135,
        0.07689755804944778,
        -0.017018755212967772,
        0.032359319278065506,
        -0.020600702185544027,
        -0.006219533860190027,
        -0.06899907997179015,
        0.004690966265527685,
        -0.058797602850706024,
        -0.06698855812955196,
        0.009148849512085151,
        0.039348023394704765,
        -0.009007496855865656,
        0.025126254114511585,
        -0.03572096431035072,
        -0.018434542528185625,
        0.043345759497924914,
        -0.031078120890760893,
        0.06645999077843719,
        0.021086414059956697,
        -0.045290867748598515,
        0.029525492130501322,
        -0.01712777786557545,
        0.009449600698722062,
        0.016619508905787075,
        -0.0022849567724323983,
        -0.03438863749038219,
        0.052062280144363385,
        0.04228410740759676,
        -0.07113591572700619,
        -0.058481060365973274,
        -0.031627743306003445,
        -0.011202980415366427,
        -0.0133307947777137,
        -0.07455245299571293,
        -0.04215177717373283,
        0.05613896141370787,
        -0.02712775353439423,
        -0.02132476084495886,
        -0.06535773572853541,
        0.029902934936647287,
        0.009608999229139478,
        0.03181119991355768,
        -0.018753337530046852,
        -0.07313290599468467,
        -0.03466307190583298,
        0.03974727073137226,
        0.02089994929195941,
        0.03475404971372954,
        -0.002055634082701849,
        -0.008367648672837309,
        -0.026307454330537692,
        0.08791332217144956,
        -0.03824050972620507,
        0.03665028616681493,
        -0.015715751265654814,
        0.005296227809868523,
        0.006192466078379948,
        -0.011122529110046955,
        -0.07158930171575426,
        0.0762005625391524,
        -0.01523304549271159,
        0.004338335120859507,
        0.03143300558204435,
        0.00266615900701708,
        0.009337570915158135,
        0.014323273590666825,
        -0.011783429783486074,
        0.004811266431336408,
        -0.038954792378923374,
        -0.03624126694139812,
        0.04803447067260711,
        0.07227952732082255,
        -0.004066907321621564,
        0.024693173723497542,
        0.02618189194292723,
        0.008679678402162069,
        0.032875860164328884,
        -0.007299230280485892,
        -0.0012857111427126034,
        -0.02853000515710514,
        -0.0463186826666329,
        0.0034105181167319244,
        -0.06690810991269289,
        -0.006759381812966259,
        0.04753372005495277,
        -0.04462620635986217,
        0.06355699035349797,
        0.030415716137100243,
        0.030578121798473912,
        0.006424796542496002,
        -0.02894353765267334,
        -0.02937662010266099,
        0.022939792977366376,
        -0.011324784146653276,
        0.026536776634210692,
        -0.03059165543200725,
        0.0027556324773204102,
        -0.00869621916665166,
        0.018933788036131636,
        0.015798458176563175,
        -0.011636813361234635,
        -0.02521497528733245,
        0.01242703713758144,
        0.04309764082417344,
        0.005801489638700644,
        0.022808967853210772,
        0.02349543171349504,
        -0.010214261288223718,
        -0.025791667028615288,
        -0.054108891676023826,
        -0.008435317869990805,
        -0.08890504740006172,
        0.01332853917212481,
        0.02804128512224941,
        -0.018140558158315386,
        0.05423821580841831,
        0.031248043864781787,
        0.04951792409651805,
        0.029403687369727666,
        -0.05581264704022581,
        -0.005440588111787671,
        0.02396159569246187,
        -0.03729539966010543,
        0.004523297352865475,
        -0.04546079072261941,
        -0.004899236078789912,
        0.06277728999126783,
        0.013693199870104799,
        0.04041493822157805,
        0.02003528950169244,
        -0.04560665664899029,
        -0.03696757811794143,
        0.0010233057658302197,
        -0.024742797046453116,
        -0.001946611816151724,
        -0.04735251802350062,
        0.026000687852501474,
        0.006866900385352407,
        0.017918753167289615,
        0.004185704179323863,
        0.029888649777746586,
        0.01779243719533818,
        -0.03718186373734637,
        0.027356324312699867,
        -0.01691048511521426,
        -0.025161592278566458,
        0.02604580202325288,
        -0.017641310591395766,
        -0.00965260829018255,
        -0.012999968163567056,
        -0.027033769625028223,
        0.059024666460329715,
        0.04551116557094234,
        0.0501374733734766,
        0.0099217799690882,
        0.0005090213216137635,
        -0.03862095863382048,
        0.018318752029324478,
        0.008052611812556294,
        -0.07141335830290003,
        0.012886435329268406,
        0.026211213786095994,
        -0.019032285215649033,
        0.03138413378445614,
        0.01588266813820626,
        -0.061420904064675694,
        0.027956323635238964,
        -0.04816304122066063,
        0.044859290408319194,
        -0.020940550192559422,
        -0.024111970771036925,
        0.025772117074195838,
        -0.051913408116523056,
        0.03515179399966232,
        -0.017990934605107693,
        0.018286422025712658,
        0.02824203710812101,
        -0.04601718304308904,
        -0.045124700342440816,
        -0.039209680957901935,
        -0.03934501729323532,
        0.0005646602576832708,
        -0.06407277559644677,
        -0.009322534230890073,
        0.05464347534802471,
        0.05134348422149369,
        0.03465028979766701,
        0.009222534000637954,
        0.04681943328428725,
        0.03381570749388338,
        -0.01047892175595159,
        -0.01695935691280247,
        0.003063902263463053,
        -0.00800073236926843,
        -0.021982652328468123,
        -0.008698474772240549,
        -0.02282250148674411,
        0.024743548571820477,
        0.01607514922708966,
        0.02156836830753256,
        -0.023083402268661565,
        -0.027332265206076244,
        -0.011493205099426251,
        0.049856271111772335,
        0.037176601000801224,
        -0.027366098260422786,
        -0.008345092616948411,
        -0.015106730550246961,
        0.011590949724089479,
        0.008055619458255949,
        -0.02306234926350739,
        0.03548036500822008,
        0.029481131544090888,
        -0.048457025590530875,
        0.00244134739977846,
        0.02039694203922937,
        0.0016721763712141278,
        0.03951043111505204,
        -0.018242812964669588,
        0.04519913635636098,
        -0.04076456429426359,
        0.004938333928655203,
        -0.0012661623464658084,
        -0.008561633327198835,
        -0.0024511218622447826,
        0.07189682177813145,
        0.021696938855586077,
        -0.03695780622919212,
        -0.029873611034504914,
        -0.02376836307821111,
        0.021267618150382455,
        0.051849497575693175,
        0.03582096454060284,
        -0.04699838279861092,
        -0.026485650260520393,
        -0.05519460081400199,
        -0.003509765792129876,
        0.04929837574056797,
        -0.007483440472381095,
        -0.025411216062026266,
        -0.0637855611318036,
        0.04135027639892838,
        0.03332773898439501,
        -0.04920514541554294,
        0.06029609501785863,
        0.015866878899084034,
        -0.016935296776692046,
        -0.09342759116858622,
        0.008931556761723964,
        -0.008514265095088748,
        -0.02130370783980469,
        -0.0023150319425704296,
        0.05732993763643051,
        -0.05778331950723136,
        -0.01158869411850059,
        0.0376314859223368,
        -0.06245999804014134,
        0.012827788554470467,
        0.048404394107132244,
        -0.02172851939280414,
        0.04997957892328071,
        0.042238241711478,
        0.01200222661406879,
        -0.03419841303657448,
        -0.11427415539280603,
        0.006374420664686262,
        -0.019017997997774722,
        -0.013090945971463616,
        -0.004469162303988719,
        -0.03271946876486772,
        -0.015303721820821336,
        0.013107486735953206,
        -0.0824576927427032,
        0.04480139618837543,
        -0.02973752111483056,
        -0.013449591378044298,
        -0.005404497907622033,
        0.011831550055706925,
        -0.01465410020481347,
        0.06308105036783461,
        -0.0630419504589957,
        -0.0026556327618116947,
        0.06273368093022477,
        -0.06426675973606488,
        -0.02495106634497194,
        -0.04880439091019349,
        0.0442367350298912,
        0.042412677955650284,
        0.10184862851236694,
        0.032476612827661386,
        0.03482397657544554,
        -0.014287182871757786,
        0.014457859430119652,
        0.013132298397430991,
        0.04456004330190381,
        0.011175160593445584,
        0.05221265522293844,
        0.04571191961578755,
        -0.05220663890205233,
        0.039098401670218566,
        -0.028736019879521877,
        0.0011601475969294864,
        0.042072832007608495,
        -0.018298452608511272,
        -0.04727732842523949,
        0.06263969702085877,
        0.013048088435787907,
        -0.03543149321063186,
        0.0012969892993429026,
        -0.007843589959183305,
        -0.03326382844356513,
        -0.11128920370027304,
        -0.03394578315164523,
        -0.040886364937090025,
        -0.030396166182680793,
        0.013099215838965008,
        0.02884353948139483,
        -0.04984950120654525,
        -0.013045081304831655,
        -0.0097007285624034,
        -0.02602700359420079,
        0.022556335909307907,
        -0.011556363085401967,
        0.0027022489538110167,
        -0.01744958102787973,
        0.020096191882079267,
        -0.0396728347174521,
        0.009597721201195031,
        -0.056603622341939976,
        0.04778033567796952,
        0.027784895551509738,
        -0.0074398314113380245,
        -0.03970817494048058,
        0.014607482983327343,
        -0.0510073938413151,
        -0.02419317463121056,
        0.0349698383838692,
        -0.01981423809500764,
        0.0018225519645325833,
        -0.0669464521192436,
        0.0158368034715743,
        -0.019883409313409053,
        0.01978641724359999,
        -0.041177343205490824,
        0.03466156885509826,
        0.03914125714692066,
        0.03532171697368321,
        0.04640214110290863,
        0.007178178074566405,
        -0.07049982465607124,
        0.03501871018145741,
        0.011427791507861645,
        0.016872139820203136,
        0.011789445074885382,
        0.016533042309068297,
        0.01369094426451591,
        0.002563903428547774,
        -0.011052604307304568,
        0.04271117559567191,
        0.01705860561768723,
        0.048747250274590696,
        0.04442245033149474,
        -0.035942769301376486,
        -0.013206734411351157,
        0.022728515518404494,
        0.03374127147996322,
        0.0600652676044773,
        0.020797694485605204,
        0.008360129815959875,
        -0.015648834623355486,
        0.006131564212736524,
        -0.014790189095001018,
        -0.034111946439855705,
        -0.012731548010028768,
        -0.01776837808871456,
        -0.02009468883134454,
        0.041909670702920246,
        0.021066114639143495,
        -0.006836073432475313,
        0.0009263135673353073,
        0.02132325573525053,
        -0.008199228234807735,
        -0.03803148684334527,
        0.04643221858939197,
        -0.014900715827830223,
        -0.01551500030927002,
        0.03111045089437271,
        -0.029802934706395168,
        0.05685023796495673,
        -0.02580595218751599,
        0.01020975007704594,
        0.013356358994045654,
        -0.01612778071048829,
        0.017848076839179865,
        -0.025611968047897865,
        0.028537524528725974,
        -0.07273140614088869,
        0.030955564604619876,
        -0.024875878805684412,
        -0.053144233714478344,
        0.020715739100064204,
        0.003519540254596199,
        0.01272778832421835,
        0.04503147001741578,
        0.028760081045119106,
        -0.013305983116235914,
        -0.0353179572878728,
        0.0340548037452793,
        0.012694705765752367,
        -0.04651718007640242,
        -0.0067007355529117235,
        -0.03283601078909623,
        0.004462394972478648,
        -0.04061343563134757,
        -0.06952915037363964,
        0.06383367934505084,
        -0.01759093368409922,
        -0.06750209085537227,
        0.0034969839413356006,
        -0.02944729643077074,
        -0.06098932672439637,
        -0.03099165429404211,
        -0.03200368100244108,
        -0.024929261814450404,
        0.010257117794412622,
        0.012305984931661951,
        0.0045345753808099225,
        -0.06441337924677674,
        0.006302992296465751,
        0.012543578132323147,
        -0.028023240277538292,
        -0.014045078459918812,
        0.02608565139848553,
        0.012607487643666225,
        0.014980414578295529,
        0.04574575267013409,
        0.016060111513334797,
        -0.015311241192442171,
        0.0228615993366094,
        -0.016337554089228642,
        0.0333751077312485,
        -0.02773752680465625,
        -0.03727434459597765,
        -0.005221791795948358,
        0.056894596492393555,
        0.039111933244778296,
        -0.07498177370091656,
        0.020527017696991223,
        -0.016432290553448817,
        0.02741121243117419,
        0.021897690841457675,
        0.07604492884197943,
        0.030069100283831373,
        -0.011573656404745723,
        -0.006899983458561793,
        0.02235032118689117,
        -0.0032473604152474926,
        0.06510660683536726,
        0.020716490625431568,
        0.02343753749355127,
        -0.028316473122041174,
        0.02647361967772178,
        -0.03975854978880352,
        0.013178163064062953,
        0.01630146439980641,
        -0.02866534355141213,
        -0.04360816333057709,
        0.013731545165115924,
        -0.0182916847622578,
        0.009472909309465125,
        0.0848388874858603,
        0.04391794002802998,
        0.0035165328662682465,
        0.029778124074404187,
        -0.0035857056288998677,
        0.02366836284795899,
        -0.0038796897413984083,
        -0.02474956489270659,
        -0.026702190456027414,
        0.10919823364117577,
        0.0001315786280679172,
        0.0014082673001677678,
        0.03212924544902515,
        -0.010792455050754476,
        -0.0848591889656471,
        0.018895443770607315,
        -0.009858623012599288,
        0.013196960463628237,
        0.0034105181167319244,
        0.008733061410928061,
        0.024548812906834992,
        0.09163887843532101,
        -0.04615402242915715,
        0.03010218181281055,
        -0.0037180361201355045,
        -0.026271364641115458,
        -0.010337569099732095,
        -0.007921033618803123,
        0.002239092363172139,
        -0.00010977416993217281,
        0.028889403118539988,
        0.0556795632220209,
        -0.06500811377379709,
        -0.027230759866115794,
        -0.03886381457102682,
        0.037872089342414664,
        0.04512019119023665,
        0.005680437432781157,
        0.05559159563354101,
        -0.04775702809671326,
        -0.06466074433618725,
        0.024756330679986456,
        -0.019913484740918787,
        0.007563891262957165,
        0.00023458589276189863,
        -0.028128503244335554,
        -0.0028007451038416066,
        -0.034478112247544025,
        -0.025202946763507445,
        -0.019504463456528363,
        0.04575026594028548,
        -0.013528538603142242,
        0.02443227088260648,
        0.004587958904319316,
        -0.004712770281305819,
        -0.019801455986841663,
        0.04037433937995165,
        0.0891306120768979,
        0.020342055979728657,
        -0.014571393293905107,
        -0.02622850710543975,
        -0.03469916571320244,
        0.03304803977342547,
        -0.013194704858039346,
        -0.033040520401804635,
        0.005281942136224421,
        -0.0242209934236446,
        -0.0012902224825762335,
        -0.05462016776676845,
        0.04240967185418083,
        0.02127964667420746,
        -0.016753343219872537,
        0.010187945546524404,
        0.012911998516113554,
        0.025772117074195838,
        -0.010557117455682172,
        0.08855843154679284,
        -0.0320751109148918,
        -0.010118020743782019,
        -0.008148100316887232,
        -0.0828531803935603,
        -0.03838035933168984,
        -0.09648172437438868,
        -0.021454834443747104,
        0.06958027880630356,
        -0.01910822428030392,
        0.03544953805534298,
        -0.017175146612428936,
        0.034393148701559975,
        0.0036518707458318357,
        0.0074112600640498195,
        0.03793148661309315,
        0.055328438216547864,
        -0.0074969741059144335,
        -0.04236756584387249,
        -0.014749588194401003,
        0.04418410354649257,
        0.009166142831428907,
        -0.03128338202883666,
        -0.02256761496673916,
        0.05367430411632784,
        0.018969877725553873,
        0.02203378076113203,
        -0.01752777672761031,
        0.022562350171220408,
        0.022230021535825848,
        0.04359463175601736,
        -0.05238633788276975,
        -0.050980326574248426,
        -0.030051806964487615,
        -0.09995238794133433,
        0.009716517801525628,
        0.03735629998151865,
        0.007820281863183644,
        -0.03956757175065484,
        0.004264651404421807,
        -0.009692457665415202,
        0.006499232556416166,
        0.023432274757006127,
        0.033142777267132446,
        -0.018701457572015582,
        -0.014075905412795906,
        0.09254413500824077,
        -0.001216538380080982,
        -0.034741267605563564,
        0.08075243432776649,
        0.031731501163092375,
        -0.0005428557915046608,
        -0.06249533826316982,
        -0.004406756358123766,
        0.0024781893866831603,
        0.06909456693189088,
        0.01421199533247026,
        0.05751564882008683,
        -0.009817269557145108,
        0.04784198958372371,
        -0.004526304998565128,
        0.038023967471724436,
        0.00777140955085203,
        -0.017951836755242402,
        0.053373553959177734,
        -0.0749584661196603,
        -0.043663051449051415,
        0.021069120740612943,
        0.004557131951442222,
        0.015705224763077726,
        0.03553675617742912,
        -0.006306751467532766,
        -0.015138309028491417,
        -0.016345073460849478,
        0.012615758540654422,
        -0.024057084711536208,
        0.00894885008106772,
        0.048190108487727305,
        -0.0061586314798032,
        -0.02816534445912515,
        -0.028614215118748225,
        -0.023669867957667325,
        -0.05410588145660716,
        0.01630898274194044,
        -0.04940514175809996,
        0.06611487797590303,
        0.01476011469697809,
        0.016736801425896144,
        -0.022845809068000367,
        -0.029533011502122158,
        0.0253164806272929,
        0.020025515553969517,
        0.05281114943576921,
        -0.005880436863798588,
        -0.03686833147203028,
        -0.058808877790190055,
        0.028162338357655702,
        0.00674810378502181,
        0.031766087801779884,
        0.013251847552615756,
        0.01028644066706819,
        0.026548055691641943,
        0.021949570799488942,
        -0.028975118189891406,
        -0.0022195434382394933,
        0.03490668142738029,
        -0.057475799444854175,
        0.041593132336134715,
        -0.05466377682781152,
        0.02936684615493807,
        0.010920274073440633,
        0.018670630619138488,
        -0.05329084910724298,
        -0.02413076920008901,
        -0.05952767371305975,
        0.010996965692949688,
        0.02352400203129644,
        -0.057386324687692336,
        -0.0066383296070467715,
        -0.03158413424496038,
        -0.020505213166469687,
        0.01022704236690289,
        -0.020369123246795334,
        -0.02129017420627135,
        0.04856604824313855,
        -0.06607502860067038,
        0.021987917123986872,
        -0.0037796900258896928,
        0.08279829639303318,
        -0.011927790600148626,
        -0.044594629940591324,
        -0.046473571015359344,
        -0.04748785435883401,
        -0.00488043919396803,
        -0.019854838995607655,
        -0.03283149957791845,
        0.01337440383875677,
        -0.033608941246099276,
        0.006928554805849998,
        0.03638788233416275,
        0.004381192141791813,
        0.02948639428063603,
        0.027164594749183826,
        0.024569865911989168,
    ],
    [
        0.004538903136153924,
        -0.005777526710814441,
        0.033351715080514856,
        0.03886921952606484,
        0.005796787209329025,
        0.013144076557657586,
        -0.019627589224445765,
        -0.01243883185290745,
        -0.05093023948095775,
        -0.04604167803465924,
        0.032938347190365556,
        -0.003072852856122948,
        0.05273113339529847,
        0.026152586930493594,
        0.06949107385357453,
        0.01306925542927828,
        -0.015924311588622078,
        -0.05293929984282032,
        -0.040738266195764226,
        0.051124331781870425,
        -0.018820853749889458,
        -0.04721437212629994,
        -0.054210516811381004,
        0.040973102284712945,
        0.037419466905647875,
        -0.024704314887195978,
        0.015246476813858486,
        -0.02372719559891516,
        -0.011515790210712185,
        0.009465246734917812,
        0.09745640701667507,
        0.002679486234379869,
        -0.006119037033482957,
        0.05644107844147887,
        -0.017836325800028318,
        -0.0011660247559561287,
        -0.020261717131606752,
        -0.038428442668592924,
        -0.03946556672475634,
        -0.00931190014736904,
        0.005404902127369224,
        -0.03257164924738542,
        -0.030547773487690055,
        0.020941032483489698,
        -0.03664532723165155,
        -0.02129439753108426,
        0.04717140265926523,
        -0.03473331109491746,
        -0.04229098920044279,
        -0.027071923279234772,
        -0.03288574723340757,
        0.019997250246992462,
        -0.03829954196880021,
        -0.03638234284948446,
        0.0025068791483165128,
        -0.021142531520654838,
        -0.027634933426142688,
        0.061367422521574726,
        0.015868011151529644,
        -0.020312091409566076,
        0.00021853703046369538,
        0.05054131940389236,
        0.01325075223537327,
        -0.05841828232867721,
        -0.038657348747752826,
        0.02946471881025323,
        -0.027814208403894723,
        -0.0005222664792048576,
        0.04361406680006571,
        -0.012426978571977295,
        0.03259905821470803,
        -0.0013645599915567664,
        -0.0210425242159598,
        -0.031039665874344896,
        -0.03586155706773195,
        -0.05949911517777109,
        -0.019623143642432003,
        -0.026125917289066727,
        0.09257302855295822,
        0.010355692576553097,
        -0.015389451660260383,
        -0.023624964540549312,
        0.014341955862950092,
        -0.04255249493081838,
        -0.02399092123230585,
        -0.02664670114549139,
        -0.0563484778730363,
        0.007088748622847384,
        -0.012624773278352276,
        0.017347396492940034,
        -0.0027994964921430037,
        -0.026645220568372038,
        -0.005481205276863772,
        0.029344707349160185,
        -0.07166755687492292,
        0.05393271895670899,
        -0.014882001661885228,
        0.007112454222043766,
        0.019070505236670966,
        -0.05917019986858829,
        0.006362761398467423,
        0.018816408167875696,
        -0.061547436825222505,
        0.007672501770717087,
        0.019680926581971642,
        -0.008626656711025128,
        -0.050528723909074674,
        -0.03640234315522676,
        -0.006963552735176792,
        0.010368286146042929,
        0.02591552901320192,
        -0.07157717428416759,
        -0.044802313690107086,
        0.02838314470993575,
        0.007887334665931733,
        -0.01385450713298116,
        -0.042393963435360095,
        0.005712335591026445,
        0.01087869922398464,
        -0.03885514537945566,
        -0.00722283423054961,
        -0.04071085722844161,
        -0.05973172558772115,
        0.015701329743267437,
        -0.002306862132266615,
        0.01751111289630783,
        -0.020909919185376925,
        0.029474349300176502,
        -0.02157516039065069,
        0.06823170920328007,
        -0.017546671776434365,
        0.04641652396511545,
        0.004826334850072958,
        -0.00823180854949877,
        -0.00465224574289436,
        -0.002495766877943979,
        -0.019085320634503748,
        0.08006974923814619,
        -0.05546840280988396,
        -0.005869386028033413,
        0.005399716738127749,
        0.045848328910298025,
        0.016433984378004116,
        -0.02000910256525869,
        -0.046038716880420535,
        0.03721500478891618,
        -0.010751281026639132,
        -0.04388964282639477,
        0.030088474976267342,
        0.03715573934692933,
        0.004716696092790717,
        -0.003438809788545467,
        0.04913971923309176,
        -0.008534056142582552,
        -0.012263261205945574,
        0.00256614338697345,
        -0.011122423588969105,
        0.03057444120378907,
        -0.03764985548725498,
        -0.026428164882150513,
        -0.03795284433156237,
        -0.0473240079955904,
        0.06389874731698131,
        0.01058978645161429,
        0.042505081807097764,
        -0.012837383382560042,
        0.007416183836242791,
        -0.012022499920527665,
        -0.04750624797823685,
        -0.04937159031714607,
        0.006219045300841924,
        -0.020487662056527953,
        0.0002148330155476368,
        -0.037699490439318556,
        -0.019409051035777034,
        -0.0317804704218855,
        0.025460676083792967,
        -0.024781358806582165,
        -0.041509440864866146,
        0.01805412273747345,
        -0.011023896861393417,
        0.017376286037382,
        0.023097513273767932,
        0.0014756805292882673,
        -0.005139694472863293,
        0.04759736604423222,
        -0.04061529165510463,
        0.034492552697491324,
        -0.040342676783014265,
        -0.05932947068994233,
        0.03346875961138564,
        0.05807454969633378,
        -0.010189011168290891,
        0.011815074724229411,
        0.03250645764626545,
        0.016514732628180463,
        0.015952462769832223,
        -0.024360582254852544,
        -0.005239703221554224,
        -0.01061275079958707,
        -0.05135027670679162,
        -0.03585414840615163,
        -0.009631927180516092,
        -0.0061975620113204595,
        0.031994563028540465,
        0.01082684340624204,
        -0.0044766755772644475,
        0.04247248600653778,
        -0.04718992431321603,
        -0.04374815048244008,
        0.0006274605569707674,
        0.015810969463213607,
        0.023330123683717993,
        -0.008569615022709089,
        0.030053657347364406,
        -0.007250984449095827,
        0.10655791696285145,
        -0.02050840361349385,
        -0.022583394902372135,
        0.020138742590947153,
        0.013315942873829303,
        0.09118031569552944,
        -0.03273462439952961,
        -0.04678100007975263,
        0.009756382298295042,
        -0.01813338848520259,
        -0.011120942049185827,
        -0.0513399068909726,
        -0.024791730547729044,
        0.026097025819296907,
        0.06954885679311416,
        0.022943425434995557,
        0.044068917804146804,
        -0.04728622921244876,
        -0.05357268649875771,
        0.027176376165943577,
        -0.07797178878797549,
        -0.027669751055045624,
        -0.02646964992141016,
        -0.04056269554880235,
        0.026388901671233814,
        -0.003077297716138765,
        -0.020597299851146264,
        0.020309128329999517,
        0.008064386852676889,
        -0.04730622951819106,
        0.04067677892543443,
        -0.004895970107878827,
        0.006562778414517321,
        0.012218071835895765,
        -0.031050037615491775,
        0.030532956164529418,
        0.002626148395522027,
        0.002126846865954903,
        0.02140329407447897,
        -0.05094061314743248,
        -0.02774605372320821,
        0.028857999870414854,
        -0.010507556661654665,
        0.06272827890433927,
        -0.057495980975041626,
        -0.02311380924872007,
        0.007954007325503009,
        0.04862856394126252,
        0.02029801533762904,
        -0.004107014554717751,
        0.018986052655704456,
        0.03343542448492991,
        0.029262478521864487,
        -0.023730158678481716,
        0.004630762453372896,
        -0.019614254403732333,
        0.01876084898200686,
        0.0012275114246209748,
        -0.033784343950510706,
        -0.009782310688498306,
        0.030457394747590436,
        0.018339331179053637,
        0.0038106931207670816,
        -0.055585451191410444,
        -0.03280870716467709,
        0.009987513093789679,
        0.028944674279724317,
        -0.04672692147100315,
        -0.018532680303414855,
        -0.06120073918798467,
        0.020615821505097064,
        -0.022641919093135378,
        -0.04597204277685337,
        0.011350591304897186,
        0.03699720785147105,
        -0.03118782755398416,
        -0.019454240405826842,
        0.026614107270259264,
        -0.008449604524279973,
        0.03534669744511255,
        0.001274922863679721,
        0.06423803629263884,
        -0.03985893045850764,
        -0.013264087056086702,
        -0.011901749133538872,
        0.018990498237718218,
        0.012405495763787794,
        0.08670290416169299,
        0.08239290373537522,
        -0.048370023215781335,
        0.0011393558365272078,
        0.01391747498043031,
        -0.006909474126427311,
        0.033193183585056574,
        -0.0027750498823890008,
        0.0365001305569067,
        -0.04434671950947453,
        -0.11329182044231688,
        0.0168192020496072,
        0.003663273414349368,
        0.013817466713071344,
        -0.020295793509286085,
        -0.06398838288052948,
        0.005193032311721134,
        0.03479554154179871,
        -0.015444271520233466,
        0.055049108760601546,
        0.01486644501282884,
        -0.004374444518898599,
        -0.058349386396767094,
        0.021524044861467764,
        -0.029951426288998557,
        -0.019306819977411186,
        -0.010607564929013632,
        0.014227872486318016,
        -0.015062757216756614,
        -0.0021379588956614553,
        0.09392202876680084,
        -0.061868204146929236,
        -0.012878868421819692,
        0.011040194761673408,
        -0.03123079317036316,
        0.06573001520353906,
        -0.0032513865826513813,
        0.021623312840267056,
        -0.07008149489313291,
        -0.13080589641819126,
        -0.005861237077893418,
        -0.03387324018816312,
        -0.0029795117584547155,
        0.04386964252065247,
        -0.022668586809234392,
        -0.0248161764354851,
        0.07961859871419955,
        -0.03005884225527392,
        0.019717966039217532,
        -0.03716018685427095,
        -0.011864708713629057,
        -0.04177687275437485,
        0.005191550771937856,
        -0.018068938135306235,
        0.07743544635716659,
        -0.016734751393968548,
        -0.04385038154080593,
        0.03261164985887001,
        -0.054707597956601066,
        -0.020709161399435388,
        -0.04952419565347124,
        0.010601638769880519,
        0.02766086181634595,
        0.03955668479075171,
        0.03861734428561253,
        0.0034877027673874917,
        0.014727914785776779,
        -0.032400522257109446,
        -0.0024002032299348473,
        0.04534680410839206,
        0.031619715172756406,
        0.040150066984548795,
        0.02096696087369296,
        0.01833488752236773,
        0.01964610895306871,
        0.0032565722125588374,
        0.030278861021062004,
        0.03947890154546977,
        -0.06623598366213093,
        -0.06331276993410884,
        0.007662871280793814,
        0.051076918658149804,
        -0.04435783250184501,
        0.024266501109290618,
        -0.0082962579367312,
        -0.06882731322542011,
        -0.1315844835995292,
        0.05395568041668999,
        -0.003657346773884291,
        0.0038032851811847074,
        -0.0090585452924613,
        -0.031554525496964296,
        -0.05251037530361464,
        0.033591734152045236,
        -0.02009577697456815,
        -0.028024596679759536,
        0.05376529629722318,
        -0.034145855060253485,
        0.08021790706712975,
        -0.06679677005536803,
        -0.006830208378698169,
        -0.024657644940026816,
        -0.001666807825306531,
        -0.0021409222158939936,
        0.015079796368260207,
        -0.005460462757233947,
        -0.025814780457283276,
        -0.07072080385754771,
        0.005507133185735072,
        -0.045531265919381454,
        -0.052108116555180106,
        -0.0252688075365511,
        -0.009343013445481814,
        -0.007898446695638285,
        -0.06963034436918626,
        0.03557412487248095,
        -0.02591182468241176,
        0.02585107866330556,
        -0.021978158464980968,
        0.0724468756808452,
        0.09360645212898719,
        -0.0066724171717995615,
        0.0066620459119846485,
        -0.02235819122867454,
        -0.037855800106433884,
        0.025019156049769592,
        -0.07978972185381981,
        0.03012996001552699,
        -0.00957784857176661,
        -0.017680757384136592,
        0.04943603874171457,
        0.03661199210519582,
        0.0062879402700881165,
        0.002514287087898887,
        0.012269928616302292,
        0.03607638900028266,
        0.04039082923263063,
        0.029029126860690823,
        -0.02238485894477355,
        0.013814503633504787,
        0.021577384144321494,
        0.010367544894819326,
        0.05841531732378281,
        0.016333235822085474,
        -0.02887578027314205,
        0.05331118461903783,
        -0.03943963640922522,
        -0.0399826481757187,
        -0.03887588501109371,
        -0.015467235868206246,
        0.010231237458774146,
        0.015676142604287777,
        0.013290755734849643,
        -0.02002984412222459,
        0.06677010041394117,
        -0.023939805703122926,
        0.01903272260287362,
        0.012126212518676792,
        0.042754731368551416,
        -0.062254163069755925,
        -0.009471172894050927,
        0.03502519079751007,
        -0.028716507526460162,
        0.026761525773347068,
        -0.020357280779615884,
        0.0026254076256303875,
        0.029432123009693247,
        -0.02165442806370768,
        0.02770530993517216,
        -0.027126001887984256,
        0.038478073770000794,
        -0.04192133010322395,
        -0.05293929984282032,
        -0.0001659399765391166,
        -0.053400820182586,
        0.04772107991078756,
        -0.012373641214451418,
        0.034072515471657454,
        0.0643921202807555,
        0.029892902098235315,
        0.03402954985527845,
        -0.023773865546084322,
        0.009791940215757654,
        0.019026796443740505,
        -0.03203975047326242,
        -0.04526087287563405,
        -0.04348812725451169,
        -0.03862179179295414,
        -0.04647060257386493,
        -0.05959986373368973,
        0.03913590823902209,
        -0.009174851460100258,
        -0.06616634840432506,
        -0.0005000423596252583,
        -0.05529950149860665,
        0.00019260891100966138,
        0.013239640205666718,
        -0.04352294488341463,
        0.012874423802469856,
        -0.02190629945350429,
        0.004531494955905568,
        -0.010082335490575208,
        -0.01912828625088275,
        0.047659592640457764,
        -0.03483184167314885,
        -0.004886340099287517,
        0.033923614466122444,
        -0.0291646930455124,
        0.0378002389952372,
        -0.0005637514582980129,
        0.07500043031164845,
        0.014629387095537164,
        -0.04292956268961754,
        -0.018122275492832112,
        -0.007423592016491147,
        -0.03489851192606031,
        -0.05668628434624662,
        -0.028737249083426064,
        0.04066789161206261,
        0.059797660365392566,
        0.02817498018774175,
        0.010687571927966377,
        0.014324917674110427,
        -0.0771665396665221,
        0.0679731684777989,
        0.01915569714353322,
        0.05156140430855218,
        -0.04388519916970886,
        -0.00879630119885389,
        -0.009070397610727528,
        -0.007453224256152606,
        0.007960674254527761,
        0.03773430806822149,
        0.04618094855027098,
        0.016895504717769787,
        -0.023968697172892746,
        0.0717060711333046,
        0.008849638556379769,
        -0.011340219563750309,
        0.05631439957002911,
        -0.002033505527620689,
        -0.06589965969136782,
        0.0061160734725844365,
        -0.02753122179065749,
        0.03091669325901315,
        0.039575206444702506,
        0.02203668265574421,
        -0.03271906775047322,
        0.00931190014736904,
        0.022435974473956476,
        0.04178353823940371,
        -0.026207404865138825,
        -0.007242095210396157,
        0.02485247656683524,
        0.10929518637743695,
        0.017305170202456782,
        0.008748148749237522,
        0.03045443166802388,
        0.03457922710680079,
        -0.02121290802968431,
        -0.03558597719074717,
        0.0025824410465874587,
        -0.028592793659904813,
        -0.028450558139398667,
        0.021235873340321017,
        -0.008880011565932867,
        0.04756254841532928,
        -0.06279124482646058,
        -0.02952027799612206,
        -0.04807740803794868,
        -0.04307179821012369,
        0.04176575976200437,
        0.012501799700356602,
        0.031537486345460705,
        0.03811137582702065,
        0.018737142420146555,
        0.07323583652865787,
        -0.03831435544130514,
        0.009278564058249384,
        -0.03246497068167795,
        -0.012908501142813115,
        0.010214198307270553,
        -0.03398287990810929,
        0.03964409852595692,
        0.0022653770930069644,
        -0.010666829408336551,
        0.06704049345768857,
        -0.015264256253921754,
        -0.03474664591563089,
        0.0336598869074039,
        -0.0548453859697656,
        0.026903761293853218,
        -0.02290268164695951,
        -0.034013993206222067,
        0.025060641089029244,
        0.06018806294490516,
        0.04193169991904297,
        0.04105607236323225,
        0.0260525777004707,
        0.010763874596128962,
        -0.06175486009619291,
        0.02665410980707171,
        0.07175867109026257,
        -0.009021504872551486,
        0.016125069374563615,
        0.0036210476051980784,
        -0.028363144404193456,
        -0.016605850694175833,
        -0.07652944867979455,
        -0.05351416423332232,
        -0.013335203853675849,
        -0.00463965217340453,
        -0.01763038310617727,
        -0.023436059072874,
        0.03870772302571215,
        0.042156905518068416,
        -0.07494486534979605,
        -0.00642128462656674,
        0.013221860765603449,
        0.01159950154045509,
        0.025889600622998655,
        0.04235321579666834,
        -0.055229123064249326,
        -0.005976802475640736,
        0.04223617126579756,
        -0.10151823268267497,
        -0.029208399913115007,
        -0.020853618748284487,
        -0.027443064878900823,
        0.03241089207292847,
        0.029979576507544774,
        -0.03294797768028883,
        -0.011354295635687345,
        -0.007014668264359716,
        0.026963024810512208,
        0.021870742498705607,
        0.04060566116518135,
        0.017836325800028318,
        0.02532659047609074,
        -0.01729702028965286,
        -0.05236369612642258,
        0.0010615713879153627,
        0.004077382315056291,
        -0.03299538695335374,
        0.020369133097882113,
        0.07439371329713793,
        -0.021247725658587242,
        0.0651655244795118,
        0.025258435795404224,
        0.0557950962907238,
        0.043174768594385286,
        -0.011969162563001786,
        -0.02632889690335122,
        0.007832515287290613,
        -0.05073318602580637,
        -0.07064154196047427,
        0.04214283137145924,
        -0.016576959224406013,
        -0.012284744014135075,
        -0.03150489054490072,
        0.025375482251602858,
        0.013559667238813768,
        0.008890382344415816,
        -0.022668586809234392,
        -0.018082272956019666,
        0.0009445244503847662,
        -0.002549845727359442,
        0.06193932190718231,
        0.01792003712977122,
        -0.042556199261608536,
        0.05950207633200979,
        0.010729056967226027,
        -0.030951510887916086,
        -0.06116221722829157,
        0.01983575374663977,
        -0.046352814866442694,
        0.012501799700356602,
        -0.003815878750674538,
        -0.009395610514448018,
        0.030864097152710875,
        0.0386669792376761,
        0.013604115357639977,
        -0.00558121354422274,
        0.020358020105511635,
        -0.025756997517743633,
        0.03401325002967061,
        -0.022174470668908743,
        -0.03997968317082429,
        0.009712674468028511,
        0.01271366951600469,
        -0.011390594804373558,
        0.014297507744123885,
        -0.036332707897420886,
        0.016562143826573228,
        -0.035368923429853504,
        0.021701098010876843,
        -0.005744190140362822,
        0.00957784857176661,
        0.01563317602524485,
        0.006675380251366119,
        0.016005800127358104,
        -0.012758858886054502,
        -0.017158491025264726,
        -0.031514521034824,
        0.04951160015865356,
        -0.04838187553404757,
        0.05956208495054809,
        0.08767705844507939,
        0.03641123431925428,
        -0.049686431479719684,
        -0.02691339178377649,
        -0.0004504085206418512,
        -0.04412521824123924,
        0.03384805304918346,
        0.0183430355098438,
        -0.09132107256424409,
        -0.05348379026110529,
        0.009997883872272628,
        0.01378487187517529,
        0.014442705381532661,
        0.01830747662971726,
        -0.008934090174682347,
        0.06302608091540929,
        0.031505631796124325,
        -0.008028087683990675,
        -0.0393662968206292,
        -0.00842589892508359,
        -0.026327416326231867,
        0.014261208575437672,
        -0.023629410122563074,
        0.03697053821004418,
        -0.013974517631410277,
        0.013442620782615137,
        -0.0008474792024258605,
        -0.04167760670090341,
        -0.03457181844522047,
        0.04035823343207065,
        0.02766827047792627,
        -0.0225693188304351,
        0.005910130297401425,
        -0.0400700619109239,
        -0.01054459708156448,
        0.03334282391648733,
        -0.04603649120142188,
        -0.0001681623915054013,
        0.05368380872115108,
        -0.012875905342253136,
        0.0338643490241356,
        -0.006490179595812932,
        0.05231628300270196,
        -0.02301454319524863,
        -0.033396163199341064,
        -0.03396954508739586,
        -0.06731014717554024,
        -0.0451601204690597,
        -0.025877007053508823,
        0.0002933581738846257,
        0.0019194219103226314,
        -0.006314608948851056,
        0.002063137767282149,
        0.033636182270871444,
        0.058422725985363126,
        0.02284563995864347,
        0.014575308486787682,
        -0.004084049725413008,
    ],
];

pub static QUESTION: [f32; 768] = [
    -0.005659515628217071,
    -0.012769670390692456,
    -0.0005048823767471018,
    0.03581399221949009,
    -0.018508217640733154,
    0.004000523189982066,
    -0.013612882885594077,
    0.020239054539890836,
    0.04583652763764491,
    -0.014990368549003756,
    -0.01629078433495134,
    0.0006544529537362676,
    0.04130173009880319,
    0.024682412600874972,
    0.045339481578293546,
    0.022868625312577,
    0.017889033519074875,
    -0.03333660653947619,
    -0.009984000479423775,
    0.033804911037333366,
    0.03216485970124609,
    -0.06867249262141105,
    -0.035706875311662646,
    -0.028259043202354308,
    0.007532740701755361,
    0.028409920465452217,
    -0.003469515169091394,
    -0.05294472301646196,
    -0.04944450837765755,
    -0.04831848599976265,
    0.0744123441599713,
    0.031352345790793795,
    0.0015087687948363742,
    0.05139415177457396,
    -0.010161002464515564,
    0.0017341044074782699,
    -0.012426768807671684,
    -0.017581402022255666,
    -0.032770979691380035,
    0.029399437541347904,
    -0.007057902957784417,
    -0.024013589972971233,
    0.03791516443358838,
    0.052290269988953725,
    -0.050409861295569285,
    0.040598292479836665,
    0.009795893746465972,
    0.027479187260009665,
    -0.010944125620092894,
    -0.022362436249721153,
    0.006280005218746505,
    0.010106791276166266,
    -0.030736428882368202,
    -0.07722414469499188,
    0.0001841873570223452,
    0.026368837590678214,
    0.034267339744918746,
    0.030606453285427965,
    0.03240587097333044,
    -0.020525133996820003,
    -0.06715000902623006,
    0.0485764778099848,
    -0.014603052695925342,
    0.02738839933623605,
    -0.05434114978201938,
    0.001506809411178043,
    -0.02046569857228289,
    -0.04053950624868032,
    0.05833579482102645,
    0.07787076018786679,
    0.03652722675674827,
    -0.0273583546665739,
    -0.009457563665197441,
    -0.07111330392670384,
    -0.03465139074546767,
    -0.07688777281703008,
    -0.049934368456459564,
    -0.02073283574671272,
    0.07209236780881749,
    -0.03878580829970664,
    0.014766340175760416,
    -0.011520200940564686,
    -0.013213156815778499,
    -0.007448484843743917,
    0.017970024524556833,
    0.018431799317355037,
    -0.0774514379207647,
    0.011404593763668341,
    0.03527514518852659,
    -0.003176252092316081,
    -0.011538489308276844,
    -0.00779987728600939,
    -0.0712282560084614,
    0.01787466509903098,
    -0.020309594712293964,
    0.029112052615547584,
    -0.000617223594534336,
    -0.023406816068050118,
    0.02532902809374989,
    -0.027427587009402673,
    -0.02799386541158281,
    0.011565267945057904,
    0.005905752316840834,
    0.0046569356011486405,
    0.030045399119435642,
    -0.008040888558093736,
    -0.060850411151251844,
    -0.012482285464892136,
    0.029264889262303823,
    0.04090330949785876,
    -0.009822672973422833,
    -0.04682212594622394,
    -0.04360929859392304,
    0.07379054910057072,
    -0.006647726645065806,
    -0.0041167833964018885,
    0.014288236398908387,
    0.01933510115158095,
    0.06132655082303914,
    -0.04002874450372064,
    -0.011516282173248023,
    0.0031853962761721603,
    -0.05230921227145306,
    -0.0323928092025093,
    0.01809673408861599,
    0.031405251510271945,
    -0.023111593607616474,
    -0.013784660634498052,
    -0.009081351379633437,
    0.06212143264126966,
    -0.0035936129101445404,
    0.03707913555853291,
    0.024679799302429462,
    -0.01608896191638522,
    -0.022050230890446505,
    -0.03425884829549824,
    -0.06392738475563751,
    0.04360342044294805,
    -0.05376050177944375,
    0.006266288942962386,
    0.02295092178693011,
    -0.0005460306139236584,
    0.01509683154204402,
    -0.023499563375482063,
    -0.0516351606859551,
    -0.03135430517445213,
    -0.043959381436086756,
    -0.05322034927960909,
    0.05255152901240856,
    0.04260997987832929,
    -0.008132329216302928,
    0.005541296333226818,
    0.055277763499140126,
    0.00003200418703218202,
    0.046257153012914956,
    -0.031075411698072312,
    0.00011168807760579461,
    -0.016211099978692134,
    -0.03113027680120879,
    -0.025774473902494262,
    -0.034446956208807646,
    0.037896876065876224,
    0.08770191811877633,
    -0.02284772364641933,
    0.0667953511412593,
    0.0010091115764267766,
    0.023849650939052183,
    -0.007625487419011507,
    -0.03425035684607774,
    -0.0025008985790017715,
    0.07156985448331422,
    0.004389145692283235,
    0.0321877207510621,
    -0.03147905653520455,
    -0.0016080471352228419,
    -0.07560041998225524,
    -0.0199921651168315,
    -0.015991643107201033,
    0.020509458927553355,
    -0.02090265293160676,
    -0.007975573802230027,
    0.011938214197740821,
    0.00021684469806821212,
    0.02716633034665104,
    -0.026261069128766797,
    0.01302701064683063,
    -0.03506221920244606,
    0.056982474495952326,
    0.007519677750582619,
    -0.012769670390692456,
    0.05048562570416023,
    0.018770782133059143,
    0.014725191938583859,
    0.019241701109713426,
    0.004038405689365437,
    0.07571210957218652,
    0.023821565653048368,
    -0.07257308724481823,
    -0.044367598955322836,
    0.013609616852712992,
    0.00019398455195890774,
    0.012413051941711765,
    0.035883878477106036,
    -0.03336599729435116,
    0.00047157189551979536,
    0.0071081953788170536,
    0.003931942991413072,
    0.0017243071940987135,
    -0.02048333302520787,
    -0.015120345326295597,
    -0.002248130857423488,
    -0.009283826532635134,
    0.026223184563768123,
    -0.0033382326278404988,
    0.037253525425530794,
    -0.009189120431720658,
    0.04198034728429538,
    0.08941251198796991,
    0.022592996688726667,
    0.04819242444873267,
    -0.023855529090027177,
    -0.019928157010190545,
    0.0065882906303528915,
    -0.026998470184712128,
    -0.03397930326503445,
    -0.04192352515820377,
    -0.021492443937686873,
    -0.02520950096918528,
    -0.02747983881409364,
    0.08393522730243885,
    0.03854022434551846,
    0.01669442917208358,
    0.017390683171203958,
    -0.037608185671204754,
    -0.004958036212557278,
    -0.002963979430846927,
    -0.04026192060187486,
    -0.03003560220114399,
    -0.00008099018476873709,
    0.017880543250005973,
    0.008628067446079938,
    0.06024559426928586,
    0.0011547632754413174,
    0.046886785606948866,
    0.0021462399563112627,
    -0.026286541116325102,
    0.035979236722280296,
    0.009071553871165981,
    -0.028600639316503924,
    0.04314229484353062,
    -0.011424188780603254,
    -0.022360476866062823,
    -0.020789657872804325,
    -0.04455374512427071,
    -0.030603839986982455,
    -0.04704484885059613,
    0.050384384586604575,
    -0.036568376174276426,
    -0.029024531905006656,
    -0.0092962367493723,
    -0.07209955378936683,
    0.06287647051084318,
    0.0023741881296789166,
    0.043844429354329195,
    -0.02324940909989324,
    0.0043009707718672276,
    0.049938287223776225,
    0.07628361221055446,
    -0.03303811686580986,
    0.012027042737856106,
    0.04017701082907624,
    0.00743868733527646,
    -0.03913654741736074,
    0.043041058447381375,
    -0.012164857049781272,
    0.00977825870336519,
    0.048047430058016156,
    0.022027372201333707,
    -0.011268738835401505,
    -0.07260051861603486,
    -0.00308677111285312,
    0.052522138257533595,
    0.005739199984476276,
    -0.027633329375637058,
    -0.004074982129701855,
    0.010885995664426933,
    0.007874989550340555,
    0.005741812102570184,
    -0.010861829145739781,
    0.014772871061170987,
    0.020277590658973485,
    -0.047360970616484234,
    -0.031865066919411815,
    0.022458448409682585,
    0.021099249933633466,
    -0.029703803005286026,
    -0.03412821878377082,
    0.0711884167812108,
    -0.03036740139700195,
    -0.04881160857039095,
    -0.032719381801476245,
    -0.02698605996797496,
    -0.027285855110512448,
    0.04716894629656137,
    0.03706150110560793,
    -0.027575853334758274,
    0.008929821382099953,
    -0.02507103392282454,
    -0.002815715171106628,
    0.05875119359940548,
    -0.008237485560120439,
    0.046347289382604594,
    -0.014130175515964332,
    -0.07114334623566278,
    0.011761212212649032,
    -0.013021785230291212,
    -0.008396200357851671,
    -0.07283956814375768,
    -0.08376932652415828,
    0.01745730457629042,
    0.008814214205203608,
    0.006346626033657168,
    0.02588159081032171,
    0.012177266086166837,
    -0.015960944522751708,
    -0.025793413824290398,
    0.046204249654140016,
    -0.05721238338087386,
    0.0024865292736941757,
    0.04102087723876504,
    -0.005072990064842246,
    -0.05059665901860113,
    -0.00646745803691713,
    0.04930930618382629,
    -0.10071390643077588,
    0.006191177268807021,
    0.05121845407800172,
    -0.018570919098151357,
    0.050906902633514245,
    0.040817748170976165,
    -0.0072120456635876105,
    -0.05330068873100708,
    -0.0604924860530484,
    -0.009357632147743546,
    0.022391173089808945,
    -0.018741390197832573,
    0.015008004182280339,
    -0.05697986355821002,
    0.004892068427170092,
    0.017708765501101998,
    -0.055555349145945586,
    0.004103720445229147,
    -0.02160347961283098,
    -0.04771366911779666,
    -0.007726071670900979,
    0.028982730933394523,
    -0.03832729835943793,
    0.08979852473288039,
    -0.0594147919911214,
    -0.011792563531533932,
    -0.010472553909003037,
    -0.031873558368832315,
    -0.0034597179557118375,
    -0.013171355844166366,
    0.04987558694670963,
    0.015843379142548634,
    0.07147253213307524,
    -0.0043786951542923004,
    0.03773685579927384,
    -0.035491989941923784,
    -0.004379348478903678,
    -0.02128017186639352,
    0.02435126731980419,
    0.045873755927153204,
    0.035956378033167494,
    0.017857029465754397,
    -0.009579702317680155,
    0.04038209928052345,
    -0.062664852354337,
    0.038862226623084764,
    0.006167010750119869,
    0.013824502222451853,
    -0.03693348489232602,
    0.008686197401745898,
    0.03259593827029818,
    -0.048447153767128534,
    0.004622318839558454,
    0.012818656162502337,
    -0.035409039552783496,
    -0.10597108859248987,
    0.049645678061788094,
    -0.028742371215394153,
    0.0456059671986394,
    0.041220739093321225,
    0.0018144414981730518,
    -0.09110808234198084,
    0.030323641041731485,
    -0.03279318682640885,
    -0.043745152341838275,
    0.006485746404629288,
    0.003095915296709199,
    0.06945822907543389,
    -0.02944711666393503,
    0.02734724991870789,
    -0.0030410507837485235,
    0.01544495972195581,
    -0.019710659522357774,
    0.02202018622078436,
    0.02767251704880368,
    -0.010484310210953023,
    -0.058940606981586034,
    0.02320434091504842,
    -0.048227698075989034,
    -0.04582019629288788,
    0.02590118464690502,
    0.014042653625071802,
    0.01177492789825735,
    -0.09430131113629202,
    0.07043403046572126,
    0.01932334366927936,
    0.036005362624625774,
    -0.009643057099709734,
    0.006931845537985039,
    0.048568640275351474,
    -0.010204110085350453,
    0.008310638440793274,
    0.02518990713260197,
    -0.07830510124874515,
    0.10703832473922842,
    -0.026521020322647276,
    0.05748735808993702,
    0.022288628864085343,
    -0.04429771269770689,
    0.0032363417267282733,
    -0.00907873867136373,
    -0.028556225046446282,
    -0.005325757638876579,
    0.046553029388135776,
    0.09960094818440532,
    0.014386861857315327,
    -0.039606159744792264,
    -0.005942981786700728,
    0.012947326290571423,
    0.07932074186628313,
    0.0199608149782982,
    0.05546390966808799,
    0.009645669807979441,
    -0.057027540320093946,
    0.03298064082493108,
    -0.007262337494444446,
    -0.0425100513117544,
    -0.030793904923246986,
    -0.018615986102644574,
    0.02239444030304163,
    0.008855362442380164,
    0.0399444910064124,
    0.015046540301362987,
    0.05430457304659506,
    0.014637670047691329,
    -0.03836452900964942,
    0.042414036791089765,
    0.05038700024575329,
    -0.08734856806337991,
    0.00766532900696531,
    0.005651024768972369,
    0.012537150568028612,
    0.054362051448177046,
    -0.027207477403475996,
    -0.0009398780532464052,
    -0.014950526961049954,
    0.004223899418965631,
    0.029609750228982927,
    -0.01654420464342125,
    0.045611189074124016,
    -0.05731492760659747,
    0.0002057412043257766,
    0.02934653182186976,
    -0.011152478924069585,
    0.0441605511202173,
    -0.020981030638643212,
    0.06303453375449043,
    -0.00950916214527703,
    0.011042751078499835,
    -0.015847950644300873,
    -0.05610137979675523,
    0.04217694664702529,
    0.023611904519497323,
    -0.03555403866490641,
    -0.010249831004630849,
    -0.018113716987456993,
    -0.026223184563768123,
    -0.017961533075136333,
    -0.04496653296490743,
    0.00277717964219978,
    -0.013313088923408194,
    -0.141173734224955,
    -0.019408253442077982,
    -0.030121817442813763,
    -0.004593580524031163,
    0.022909772369401952,
    -0.06644265263994688,
    -0.04175828065541357,
    -0.003809804634018257,
    0.0317056982068934,
    0.013102122320985994,
    -0.026995856886266618,
    0.0082642647870773,
    -0.015753897867997774,
    -0.050271391888505344,
    -0.008529442577848798,
    0.012435912991527765,
    -0.01758466805513675,
    0.01812743267306531,
    0.015401852101120923,
    0.03408380451371318,
    -0.027407339258032182,
    0.009349141288498843,
    -0.031563312393216,
    0.005385194243765294,
    -0.006750270870789409,
    -0.050632584199941474,
    -0.05577154234525881,
    0.00473335392452676,
    0.015142552461324418,
    -0.008296922165009154,
    0.01207080309312657,
    -0.06997160175813588,
    0.008978807153909835,
    0.003058032797325828,
    0.06376409491509924,
    -0.019813856482516955,
    0.03184351369917017,
    0.008819439621743025,
    0.020023517616068,
    0.037124855297461706,
    0.07722806346230854,
    0.019370371237782513,
    0.01645211125077648,
    -0.05466314969888248,
    0.024745115238644772,
    -0.04957252459093946,
    -0.01036609091596277,
    0.05047974755318523,
    0.017219558156735558,
    -0.056252908613937114,
    0.02902845067232332,
    -0.031217797511749718,
    0.02994808149060306,
    0.03195454701361107,
    -0.020446756289783555,
    0.020767450737775504,
    0.035328050908004734,
    -0.016350220939840055,
    -0.012938182106715344,
    0.0740080501294583,
    -0.07007871630526012,
    0.023607334198096686,
    0.05651351608330798,
    0.0065602053443490765,
    0.011554164377543493,
    -0.01754939796893519,
    -0.03330917280755635,
    -0.052461392642718915,
    0.01638091834393778,
    -0.02527285752174226,
    -0.006429575832621658,
    0.016966790582701227,
    0.022184779169490586,
    0.0466294453508107,
    0.06395024108404711,
    -0.035354828364434196,
    0.0498279054634193,
    -0.06317364999423195,
    -0.03343784293562544,
    0.006083408216719802,
    -0.013265408620469465,
    0.007469385329549983,
    -0.04514614942879633,
    -0.04280984645429189,
    0.045633398569856036,
    -0.016162114206882253,
    -0.0011835017385125593,
    -0.021664874421026423,
    0.00552496734917299,
    0.023497603991823733,
    -0.04877764749411534,
    0.01642533261399542,
    -0.022148202434066268,
    -0.034215741855014956,
    0.03807257140174526,
    0.04142713537434279,
    0.023387222231466803,
    0.008473924740276745,
    -0.029632611278798925,
    0.006301559029163949,
    -0.03447765243255377,
    -0.05039026745898598,
    -0.03206362330509685,
    0.007084028860129901,
    0.004389145692283235,
    0.07211131009131683,
    0.016213059362350464,
    -0.020659028361076907,
    -0.011078021164701397,
    0.041267764301121175,
    0.058309015003893794,
    0.0004728781758826745,
    -0.02513373656059434,
    -0.013294146640908857,
    -0.040649236454953275,
    -0.011492768388996448,
    -0.022572095022569,
    0.02675354014531112,
    -0.00854250552902154,
    0.04456027955073608,
    0.001991444220984593,
    -0.0008353745914084211,
    0.05656446005842459,
    0.033766377278953916,
    -0.0217367200623007,
    0.03705692842350409,
    -0.00000457202680121794,
    -0.002682473246197402,
    -0.012204045903299498,
    0.018117635754773657,
    -0.03751935713108947,
    -0.030236117970487354,
    -0.0031187754612614976,
    -0.06144411856394542,
    0.019576112423665293,
    -0.0007792039456288165,
    0.02103066914488867,
    0.031043407644751834,
    -0.012390192072247367,
    0.009388330142017069,
    -0.015167371714447148,
    -0.004764051918800283,
    0.04744326945154055,
    0.0303517263277353,
    0.030027764666510665,
    0.016045201561114754,
    -0.0077424006549548065,
    -0.038181647693231034,
    -0.021572127113594474,
    0.012277198193796532,
    -0.040538854694596345,
    -0.034501167397156945,
    0.01812285999096147,
    0.04221482649061755,
    0.014819897449322536,
    0.0013265409505733166,
    -0.00559681358062307,
    0.051550255634562885,
    0.012979330343891902,
    -0.012669739463414362,
    -0.010716830033616866,
    -0.05742530936695439,
    -0.0628927971341938,
    -0.06280527642365287,
    0.02360863966696784,
    0.038631012269292074,
    0.015730384083746198,
    0.0043767357706339695,
    0.034956410124192976,
    -0.014147809968889313,
    -0.01592502170211457,
    0.031162934769316444,
    0.11513016376437257,
    -0.015098790925702352,
    -0.0071362806648208685,
    0.04349107929893279,
    -0.022327818897955168,
    -0.06534863195466925,
    0.016343036139642306,
    0.007445872135474208,
    -0.031456195485388555,
    -0.044310776829231235,
    0.02387512292661049,
    0.046772485079275276,
    0.012313773748869248,
    -0.009802425222052344,
    0.04783188841278691,
    -0.04939682689436722,
    -0.011833057853923312,
    -0.01041311730411432,
    0.05626597274546146,
    0.06167533527844131,
    -0.020073810037100636,
    0.05535352790773107,
    -0.07494661850883097,
    0.004760786181007098,
    0.037604918457972064,
    0.014083801862248358,
    0.0035550773812376923,
    0.012099542293917563,
    0.03785115632694743,
    -0.05039091901306995,
    -0.015987724339884372,
    0.007694067617580501,
    -0.018316192140458692,
    -0.014008690188092993,
    0.0451043508178874,
    -0.03350381042592473,
    0.0023670034770251185,
    -0.0416263409534086,
    -0.01250645198357929,
    -0.03699292031686313,
    0.0260128718761331,
    -0.02998661760968571,
    0.09599427527396705,
    0.002163874851868094,
    0.001344829023197575,
    0.019555210757507626,
    -0.042254018885190585,
    -0.007948795165448967,
    0.0093902895256754,
    -0.001456516990145407,
    -0.03446981489792044,
    -0.05253127890033487,
    -0.07195390312315995,
    0.0005623592660035981,
    0.09258745949667854,
    0.06879006036231733,
    0.03832664680535395,
    -0.0013768332240620025,
    0.04615983538408237,
    0.035824440691865723,
    0.029178674020634048,
    -0.015668335360763574,
    0.015963557821197218,
    -0.02072499585137619,
    -0.007544497003705348,
    -0.07246335703854527,
    0.017806084310286185,
    0.022087461540658,
    -0.013402569017607453,
    -0.004860717403373092,
    -0.01743966894301384,
    -0.06325137850788762,
    0.06279417167578685,
    0.025739202635941098,
    -0.046551070004477446,
    -0.01685836820600263,
    -0.035232036387340106,
    -0.047345295547217586,
    0.022930020120772443,
    -0.014911992022318906,
    0.015464553558539122,
    -0.01521374536816312,
    -0.012899647167984297,
    0.007508574183068209,
    0.02313706795587798,
    0.05869567930288823,
    -0.03273571078553007,
    -0.007288463396789929,
    -0.0016211100863955837,
    -0.022675947077866958,
    -0.0002174978382496517,
    -0.016103984251216293,
    -0.05708632655125028,
    -0.033595906179272696,
    -0.01906992336080945,
    -0.014999512732859835,
    0.007244703041519465,
    0.0017406358830646407,
    -0.045756844461737306,
    -0.038345586727150084,
    -0.004995919007028549,
];
