/**
 * Copyright © 2021-present Sveltin contributors <github@sveltin.io>
 *
 * Use of this source code is governed by Apache 2.0 license
 * that can be found in the LICENSE file.
 */

// Package composer implements the Composite design pattern used to manage files and folder structure for sveltin projects and artifacts.
package composer

import (
	"errors"
	"path/filepath"

	"github.com/sveltinio/sveltin/config"
	"github.com/sveltinio/sveltin/helpers"
	"github.com/sveltinio/sveltin/helpers/factory"
	sveltinerr "github.com/sveltinio/sveltin/internal/errors"
)

// File is the main struct representing a file to be generated by sveltin cmds.
type File struct {
	Name         string
	path         string
	TemplateID   string
	TemplateData *config.TemplateData
}

// GetName returns a string representing the file name.
func (f *File) GetName() string {
	return f.Name
}

// SetName set the file name.
func (f *File) SetName(name string) {
	f.Name = name
}

// GetPath returns a string representing the file path.
func (f *File) GetPath() string {
	return f.path
}

// SetPath set the file path.
func (f *File) SetPath(path string) {
	f.path = path
}

// GetTemplateID returns a string representing the template id for the file.
func (f *File) GetTemplateID() string {
	return f.TemplateID
}

// GetTemplateData returns a TemplateData struct the file.
func (f *File) GetTemplateData() *config.TemplateData {
	return f.TemplateData
}

// Create is resposibile to execute the file template providing the info needed by the builder
// get the content and save the newly creted file on the file system.
func (f *File) Create(sf *factory.Artifact) error {
	preparedContent := helpers.PrepareContent(sf.GetBuilder(), sf.GetResources(), f.GetTemplateID(), f.GetTemplateData())
	fileContent := helpers.MakeFileContent(sf.GetEFS(), preparedContent)
	saveAs := filepath.Join(f.GetPath(), f.GetName())

	if err := helpers.WriteContentToDisk(sf.GetFS(), saveAs, fileContent); err != nil {
		nErr := errors.New("something went wrong: " + err.Error())
		return sveltinerr.NewDefaultError(nErr)
	}
	return nil
}
