# 💻 가비지 컬렉션(GC)

- 원시값, 객체, 함수 등 우리가 만드는 모든 것을 메모리를 차지한다. 그렇다면 더는 쓸모 없어지게되는 것들은 어떻게 처리할까?
- 자바스크립트는 자동으로 메모리를 할당하고 더 이상 필요하지 않을 때 자동으로 해제한다 이를 가비지 컬렉션이라고 한다.
- 이러한 메모리 관리는 잠재적 혼란의 원인이기도 하는데, 개발자가 메모리 관리에 대해 고민할 필요가 없다는 잘못된 인상을 줄 수 있기 때문이다.

<br />

## 👨🏻‍💻 가비지 컬렉션 기준

- 자바스크립트는 도달 가능성(reachability)이라는 개념을 사용해 메모리 관리를 수행한다.
- `도달 가능한 값`은 쉽게 말해 어떻게든 접근하거나 사용할 수 있는 값을 의미한다. 도달 가능한 값은 메모리에서 삭제되지 않는다.
- 아래의 값들은 태생부터 도달 가능하기 때문에, 명백한 이유 없이는 삭제되지 않으며, 이러한 값을 `루트(root)`라고 부른다.
  - 현재 함수의 지역 변수와 매개 변수
  - 중첩 함수의 체인에 있는 함수에서 사용되는 변수와 매개변수
  - 전역 변수
- 루트가 `참조하는 값`이나 `체이닝`으로 루트에서 참조할 수 있는 값은 `도달 가능한 값`이 된다.
- 쉽게 말해, 전역 변수에 객체가 저장되어 있다고 가정해보자. 이 객체의 프로퍼티가 또 다른 객체를 참조하고 있다면, 프로퍼티가 참조하는 객체는 도달 가능한 값이 된다. 해당 전역 변수로 관리하는 객체가 참조하는 다른 모든 것들도 도달 가능하다고 여겨진다.

<br />

### 🏃‍♂️ 간단한 예시(1) - 참조 1개

![스크린샷 2022-10-09 오후 11 28 32](https://user-images.githubusercontent.com/64779472/194762374-104df100-b076-4f1e-a9f2-f7cb0406be97.png)

```js
// user엔 객체 참조 값이 저장됩니다.
let user = {
  name: "John",
};
```

- 위 예제에서 화살표는 `객체 참조`를 의미한다. 전역 변수 user는 객체 `{ name: "John "}`을 참조한다. 이때 name은 원시값인 `"John"`을 저장한다.

<br />

![스크린샷 2022-10-09 오후 11 30 25](https://user-images.githubusercontent.com/64779472/194762494-370d5e43-30a9-4313-9ea8-9a121364b885.png)

```js
user = null;
```

- 이때, user에 null로 덮어쓰면 `참조(화살표)가 사라진다.`
- 이제 `{ name: "John" }` 객체는 접근할 방법도, 참조하는 것도 모두 사라졌다. 가비지 컬렉터는 이제 해당 객체를 메모리에서 삭제한다.

<br />

### 🏃‍♂️ 간단한 예시(2) - 참조 2개

![스크린샷 2022-10-09 오후 11 32 49](https://user-images.githubusercontent.com/64779472/194762586-bbb98fd2-1341-44a8-a58d-3a96a9a6e99d.png)

```js
// user엔 객체 참조 값이 저장됩니다.
let user = {
  name: "John",
};

let admin = user;

user = null;
```

- 만약 전역 변수 user를 전역 변수 admin에 `복사`하고, 기존처럼 user를 null로 덮어씌웠다고 가정하자.
- 전역 변수 admin을 통하면 여전히 `{ name: "John" }` 객체에 접근할 수 있기 때문에 해당 객체는 메모리에서 삭제되지 않는다.
- 하지만 이 상태에서 admin마저 다른 값(null)로 덮어쓰면 해당 객체는 메모리에서 삭제된다.

<br />

### 🏃‍♂️ 간단한 예시(3) - 연결된 객체

![스크린샷 2022-10-09 오후 11 35 35](https://user-images.githubusercontent.com/64779472/194762718-56b95521-8295-46d4-8168-830a643eaa56.png)

```js
function marry(man, woman) {
  woman.husband = man;
  man.wife = woman;

  return {
    father: man,
    mother: woman,
  };
}

let family = marry(
  {
    name: "John",
  },
  {
    name: "Ann",
  }
);
```

- 좀 더 복잡한 예시를 보자.
- 함수 `marry`는 매개변수로 받은 두 객체를 서로 참조하고, 두 객체를 포함하는 새로운 객체를 반환한다. 현재로서는 모든 객체가 도달 가능한 상태이다.

<br />

![스크린샷 2022-10-09 오후 11 36 53](https://user-images.githubusercontent.com/64779472/194762810-6b0ff909-2e16-45cf-bc3c-e0e87aec56c2.png)

```js
delete family.father;
delete family.mother.husband;
```

- 이때 참조 두 개를 지운다면, `{ name: "John" }`로 들어오는 참조(화살표)는 `모두 사라져 해당 객체는 도달 가능한 상태에서 벗어난다.`
- 만약, 위에 삭제한 두 개의 참조중 한 개만 지웠다면, 모든 객체가 여전히 도달 가능한 상태였을 것이다.
- 짚고 넘어가야 되는 사항이 외부로 나가는 참조는 도달 가능한 상태에 영향을 주지 않는다. 즉, 외부에서 들어오는 참조만이 도달 가능한 상태에 영향을 준다.
  - 예를 들어 `B -> A` 에서 A는 B한테 참조되기 때문에 도달 가능한 상태지만, B는 도달 가능한 상태가 아니다.

<br />

![스크린샷 2022-10-09 오후 11 40 26](https://user-images.githubusercontent.com/64779472/194762981-20b9406a-385b-495d-ae93-33aa0c344214.png)

- 위 예제에서 가비지 컬렉션 후 메모리 구조는 위 이미지와 같다.

<br />

![스크린샷 2022-10-09 오후 11 43 07](https://user-images.githubusercontent.com/64779472/194763126-6982afde-3f94-4652-9aef-c53817a3039c.png)

- 추가적으로 여기서 family에 `null`을 할당해 아무것도 참조하지 않도록 한다면, 기존에 family가 참조하고 있던 객체들은 더 이상 도달할 수 없는 상태가되며 이를 `도달할 수 없는 섬`이라고 한다.
  - 객체들이 연결되어 섬 같은 구조를 만드는데, 이 섬에 도달할 방법이 없으면, 섬을 구성하는 객체 전부가 메모리에서 삭제된다.
- 여기서 도달 가능성이라는 개념이 정말 중요하다. 위 그림에서 여전히 `{ name: "John" }`객체와 `{ name: "Ann" }` 객체는 서로를 참조하고 있다. family 객체와 루트의 연결이 사라지면 루트 객체를 참조하는 것이 아무것도 없게 된다.
- 즉, 섬 전체가 도달할 수 없는 상태가되고, 섬을 구성하는 객체 전부가 메모리에서 제거된다.

<br />

## 👨🏻‍💻 mark-and-sweep

- `mark-and-sweep`이라 불리는 가비지 컬렉션 기본 알고리즘이 있다. 해당 알고리즘을 통해 가비지 컬렉션은 대개 다음 단계를 거쳐 수행된다.
  - 가비지 컬렉터는 `루트(root)` 정보를 수집하고 이를 `mark(기억)`한다.
  - 루트가 참조하고 있는 모든 객체를 방문하고 이것들을 `mark` 한다.
  - mark 된 모든 객체에 방문하고, 그 객체들이 참조하는 객체도 mark 한다. 한번 방문한 객체는 전부 mark 하기 때문에 같은 객체를 다시 방문하는 일은 없다.
  - 루트에서 도달 가능한 모든 객체를 방문할 때까지 위 과정을 반복한다.
  - mark 되지 않은 모든 객체를 메모리에서 삭제한다.

<br />

### 🏃‍♂️ mark-and-sweep 동작 과정

![스크린샷 2022-10-09 오후 11 47 35](https://user-images.githubusercontent.com/64779472/194763345-81c49077-696f-457a-a74c-7407e254a3f9.png)

- 다음과 같은 객체 구조가 있다고 가정하자.

<br />

![스크린샷 2022-10-09 오후 11 48 03](https://user-images.githubusercontent.com/64779472/194763364-5abe64fd-66cd-4916-a737-f39d75980352.png)

- 첫 단계에서는 루트를 mark 한다.

<br />

![스크린샷 2022-10-09 오후 11 48 27](https://user-images.githubusercontent.com/64779472/194763380-c6226df1-91a5-4b15-9a8a-09f701e37204.png)
![스크린샷 2022-10-09 오후 11 49 02](https://user-images.githubusercontent.com/64779472/194763421-fe254b9c-91e0-49e6-969f-5d56db3de518.png)

- 그리고 루트가 참조하고 있던 것들을 mark하고, 그 이후에 도달 가능한 모든 객체를 방문할 때까지 mark 한 객체가 참조하는 객체를 계속해서 mark한다.

<br />

![스크린샷 2022-10-09 오후 11 51 44](https://user-images.githubusercontent.com/64779472/194763582-25d210c6-cbf4-485a-95ac-54e24fd64eae.png)

- 위와 같은 과정을 모두 끝나면, 방문할 수 없었던(도달할 수 없는 섬) 우측에 있는 객체를 메모리에서 삭제한다.

<br />

## 👨🏻‍💻 최적화 기법

- 지금까지 가비지 컬렉션이 어떻게 동작하는지에 대한 개념을 살펴봤다.
- 자바스크립트 엔진은 실행에 영향을 미치지 않으면서 가비지 컬렉션을 더 빠르게 하는 다양한 최적화 기법을 사용하고, 종류는 아래와 같다.
  - generational collection(세대별 수집)
    - 객체를 `새로운 객체`와 `오래된 객체`로 나눈다. 객체 상당수는 생성 이후 제 역할을 빠르게 수행해 금방 쓸모가 없어지는데, 이런 객체를 `새로운 객체`로 구분합니다. 가비지 컬렉터는 이런 객체를 공격적으로 메모리에서 제거합니다.
    - 일정 시간 이상 동안 살아남은 객체는 `오래된 객체`로 분류하고, 가비지 컬렉터가 `덜 감시`한다.
  - incremental collection(점진적 수집)
    - 방문해야 할 객체가 많다면 모든 객체를 한 번에 방문하고 mark 하는데 상당한 시간이 소모된다. 가비지 컬렉션에 많은 리소스가 사용되어 실행 속도도 눈에 띄게 느려질 것이다.
    - 자바스크립트 엔진은 이런 현상을 개선하기 위해 가비지 컬렉션을 `여러 부분으로 분리`한 다음, 각 부분을 `별도로 수행`한다.
    - 작업을 분리하고, 변경 사항을 추적하는 데 추가 작업이 필요하긴 하지만, 긴 지연을 짧은 지연 여러 개로 분산시킬 수 있다는 장점이 있다.
  - idle-time collection(유휴 시간 수집)
    - 가비지 컬렉터는 실행에 주는 영향을 최소화하기 위해 `CPU가 유휴 상태`일 때에만 가비지 컬렉션을 실행합니다.

<br />

## 👨🏻‍💻 요약

- 가비지 컬렉션은 엔진이 자동으로 수행하므로 개발자는 이를 억지로 실행하거나 막을 수 없습니다.
- 객체는 도달 가능한 상태일 때 메모리에 남습니다.
- 참조된다고 해서 도달 가능한 것은 아닙니다. 서로 연결된 객체들도 도달 불가능할 수 있습니다.

<br />

## 참고

https://ko.javascript.info/garbage-collection

<br />
