# 💻 자바스크립트 부동소수점 계산 문제

- 자바스크립트 소수점 계산을 하다보면 우리의 상식과 다른 결과를 맞이할 때가 있다.
- 아래 예제를 보자.

```js
let a = 0.1;
let b = 0.2;

console.log(a + b); // 0.30000000000000004
```

- 분명히 상식대로라면 0.1과 0.2를 더하면 0.3이나와야되는데, 실제 콘솔에 찍히는 값을 보면 `0.30000000000000004`가 나온다.

<br />

### 자바스크립트만 그럴까?

- 사실 위와 같은 문제는 자바스크립트만의 문제는 아니다. java, python 등의 굵직한 프로그래밍 언어들도 이러한 부동소수점 문제가 발생한다.

<br />

### 이런 문제는 왜 발생할까?

- 컴퓨터는 우리가 흔히 사용하는 10진수가 아닌, 0과 1로만 이루어진 2진수을 사용해 계산한다.
- 이때 `몇몇 소수 경우에 10진법을 2진법으로 변환할 때 무한 소수가 되어버리는데`, 한정적인 컴퓨터 자원을 활용하는 프로그램은 저장공간의 한계로 이런 무한 소수를 유한 소수로 바꾸게 된다.
  - 이러한 과정에서 아주 미세한 오차가 발생하는 것이다.
- 근본적으로 해당 문제를 이해해보자면, 자바스크립트에서 숫자는 내부적으로 `64비트 형식(IEEE-754)`으로 표현되기 때문에 숫자를 저장하려면 정확히 64비트가 필요하다.
  - 64비트 중 `52비트는 숫자를 저장`하는데 사용되고, `11비트는 소수점 위치를(정수는 0)`, `1비트는 부호`를 저장하는데 사용된다.
  - 하지만, 이때 너무 큰 수는 표현하는데 64비트를 초과하기 때문에 `Infinity`로 처리된다.

**feat. chatGPT**
![스크린샷 2023-04-03 오후 5 14 38](https://user-images.githubusercontent.com/64779472/229451161-b373001c-7d9b-43da-bb72-856a880ca60e.png)

- 정리하자면 다음과 같다.

```
1. 컴퓨터로 숫자를 표현하는 데에는 한계가 있다.

2. 10진수로 표현되는 소수를 2진수로 표현하려면 몇몇의 경우 무한소수가 발생한다.

3. 무한 소수를 유한하게 표현하려다 보니 미세한 값들이 초과되거나 손실된다.

4. 0.1 + 0.2 == 0.3 //false 와 같은 계산 오류가 일어난다.
```

<br />

### 해결책은?

1. toFixed()사용하기

```js
let a = 0.1;
let b = 0.2;
let sum = a + b;

console.log(sum.toFixed(1)); // '0.3'
```

- 단, toFixed()는 문자열로 반환하기 때문에 `숫자로 형변환`이 필요하다.

```js
let a = 0.1;
let b = 0.2;
let sum = a + b;

console.log(+sum.toFixed(1)); // 0.3
console.log(Number(sum.toFixed(1))); // 0.3
```

<br />

2. Math 메서드 활용하기.

- Math 객체에는 `Math.floor`, `Math.ceil`, `Math.trunc`, `Math.round` 등의 메서드 들이 존재한다. 이를 활용해서 원하는 계산을 해줄 수 있다.

```js
let a = 0.1;
let b = 0.2;
let sum = a + b;

console.log(Math.round(sum * 10) / 10); // 0.3
```

<br />

3. 라이브러리 활용

- 수학 관련 라이브러리를 활용하는 방법도 있다.
- Big.js, BigNumber.js, Decimal.js math.js 등이 있다.

<br />
