# 💻 객체 프로퍼티 접근법(마침표 표기법, 대괄호 표기법)

- 자바스크립트에서 프로퍼티에 접근하는 방법은 2가지가 있다.
  1. 마침표 프로퍼티 접근 연산자`(.)`를 사용하는 마침표 표기법
  2. 대괄호 프로퍼티 접근 연산자`([ ... ])`를 사용하는 대괄호 표기법

```js
const person = { name: "Lee" };

person.name; // 마침표 표기법
person["name"]; // 대괄표 표기법
```

- 마침표 프로퍼티 접근 연산자, 대괄호 프로퍼티 접근 연산자의 좌측에는 `객체`로 평가되는 표현식을 기술한다.

```js
obj.name; // . 좌측의 obj
obj["name"]; // [] 좌측의 obj
```

- 마침표 프로퍼티 접근 연산자 우측 또는 대괄호 프로퍼티 접근 연산자의 내부에는 `프로퍼티 키`를 지정한다.

```js
obj.name; // . 우측의 name
obj["name"]; // [] 내부에 있는 name
```

- 대괄호 표기법을 사용하는 경우 대괄호 프로퍼티 접근 연산자([]) 내부에 지정하는 프로퍼티 키는 반드시 따옴표로 감싼 `문자열`이어야 한다.
- 대괄호 프로퍼티 접근 연산자([]) 내에 따옴표로 감싸지 않은 이름을 프로퍼티 키로 사용한다면, 자바스크립트는 이를 `식별자로 해석`한다.

```js
const person = { name: "Lee" };

person[name]; // ReferenceError
```

- 식별자로 해석하기 때문에 미리 선언하지 않고 접근하려고 하면, 위 예제처럼 `식별자 name`을 찾지 못했기 때문에 `ReferenceError`가 발생한다.

```js
const person = { name: "Lee" };
const name = "name";

person[name]; // "Lee"
```

- 당연히 위에 처럼 name을 미리 선언해주면, 에러없이 정상적으로 값을 가져오는 것을 확인할 수 있다.

<br />

### 존재하지 않는 프로퍼티 접근

- 객체에 존재하지 않는 프로퍼티에 접근하면 undefined가 반환된다.
  - 이때 ReferenceError가 발생하지 않는 것을 주의하자.

```js
const person = { name: "Lee" };

person.age; // undefined
```

<br />

### 언제 점표기법? 대괄호 표기법?

- 프로퍼티 키가 식별자 네이밍 규칙을 `준수하지 않는 이름을 사용`한다면 반드시 `대괄호 표기법`을 사용해야 한다.
- 프로퍼티 키가 식별자 네이밍 규칙을 준수하는 이름이면 마침표 표기법과 대괄표 표기법을 모두 사용할 수 있다.
  - **식별자 네이밍 규칙은 문서 하단에 표기**

```js
const person = { "last-name": "Lee", age: 8 };

person.age; // 8
person.'last-name'; // SyntaxError: Unexpected String
person.last-name; // 브라우저 -> NaN
                  // Node -> ReferenceError: last is not defined
person[last-name]; // ReferenceError: last is not defined
person["last-name"]; // "Lee"
```

- 첫 번째는 네이밍 규칙을 준수했기 때문에 점 표기법으로 정상적으로 프로퍼티 값을 가져온다.
- 두 번째는 점 표기법을 사용해서 식별자 네이밍 규칙을 준수하지 않은 프로퍼티 키를 접근하려 하기 때문에 `SyntaxError`가 발생한다.
- 세 번째는 런타임 환경에 따라 결과가 다르다. 브라우저에서는 `NaN`, Node에서는 `ReferenceError`가 발생하는 이유는 다음과 같다.
  <br />

  1. person.last-name을 실행하면 자바스크립트 엔진은 `person.last`를 먼저 평가한다.
  2. person 객체는 프로퍼티 키가 last인 프로퍼티가 없기 때문에 undefined로 평가된다.
  3. 따라서 `undefined-name`과 같다.
  4. Node 환경에서는 name이라는 식별자가 어디에도 없다. 따라서 ReferenceError가 반환된다.
  5. 브라우저에서는 name이라는 암묵적인 전역 변수가 존재한다. 그리고 해당 값은 `빈 문자열`이다. 그래서 `undefined - ""` 이 되면서 `NaN`을 반환하는 것이다.

<br />

- 네 번째 person[last-name]는 last를 먼저 평가를 할거고 last라는 식별자는 존재하지 않기 때문에 여기서 ReferenceError가 발생한다.
- 다섯 번째는 대괄호 프로퍼티 접근 연산자 내에 `문자열`로 감싸줌으로써 정상적으로 프로퍼티 값을 가져올 수 있다.

<br />

### 정리

- 프로퍼티 키가 식별자 네이밍 규칙을 준수하는 이름이면 `마침표 표기법`과 `대괄표 표기법`을 모두 사용할 수 있다.
- 프로퍼티 키가 식별자 네이밍 규칙을 준수하지 않으면 `대괄호 표기법`만을 사용해야 한다.
- 대괄호 프로퍼티 접근 연산자 내의 문자열로 감싸지 않음 프로퍼티 키를 넣으면 이를 `식별자로 인식`한다. 만약 식별자로 선언된게 없다면 `ReferenceError`가 호출한다.
- 참고로, 숫자로 이뤄진 키는 대괄호 표기법에서 문자열을 생략할 수 있다.(마치 배열 인덱스 접근) 당연히 마침표 표기법으로는 접근 할 수 없다.

```js
const person = { 0: 8 };

person[0]; // 8
person["0"]; // 8
```

<br />

### 식별자 네이밍 규칙

```
1. 변수, 함수, 매개변수의 이름은 영문자, 숫자, 언더스코어(_), 달러 기호($)로 구성될 수 있다.
2. 식별자는 숫자로 시작할 수 없다.
  - 반드시 영문자, 언더스코어(_), 달러 기호($)로 시작해야 한다.
3. 식별자는 대소문자를 구별한다.
  - 즉, "myVar"와 "myvar"는 서로 다른 식별자로 간주된다.
4. JavaScript 예약어(Reserved Words)는 식별자로 사용할 수 없다.
  - 예를 들면, "if", "else", "while", "function" 등이 있다.
```
