# -*- coding: utf-8 -*-
# @Time:  22:39
# @Author: tk
# @File：ia3
import re
import warnings
from dataclasses import asdict
from enum import Enum
import torch
from transformers.pytorch_utils import Conv1D
from ..petl_model_base import PetlModelBase
from ....layers.petl.ia3.layer import IA3Layer, Linear, is_bnb_4bit_available,Conv2d
from ....layers.petl.petl_layer import check_target_module_exists
from ....layers.petl.utils import is_bnb_available, _get_submodules, ModulesToSaveWrapper, \
    TRANSFORMERS_MODELS_TO_IA3_TARGET_MODULES_MAPPING, \
    TRANSFORMERS_MODELS_TO_IA3_FEEDFORWARD_MODULES_MAPPING



if is_bnb_available():
    import bitsandbytes as bnb

    from ....layers.petl.ia3.bnb import Linear8bitLt

if is_bnb_4bit_available():
    from ....layers.petl.ia3.bnb import Linear4bit

class IA3Module(PetlModelBase):
    """
        Creates a Infused Adapter by Inhibiting and Amplifying Inner Activations ((IA)^3) model from a pretrained
        transformers model. The method is described in detail in https://arxiv.org/abs/2205.05638

        Args:
            model ([`~transformers.PreTrainedModel`]): The model to be adapted.
            config ([`IA3Config`]): The configuration of the (IA)^3 model.
            adapter_name (`str`): The name of the adapter, defaults to `"default"`.

        Returns:
            `torch.nn.Module`: The (IA)^3 model.

        Example:

            ```py
            >>> from transformers import AutoModelForSeq2SeqLM, ia3Config
            >>> from peft import IA3Model, IA3Config

            >>> config = IA3Config(
            ...     peft_type="IA3",
            ...     task_type="SEQ_2_SEQ_LM",
            ...     target_modules=["k", "v", "w0"],
            ...     feedforward_modules=["w0"],
            ... )

            >>> model = AutoModelForSeq2SeqLM.from_pretrained("t5-base")
            >>> ia3_model = IA3Model(config, model)
            ```

        **Attributes**:
            - **model** ([`~transformers.PreTrainedModel`]) -- The model to be adapted.
            - **petl_config** ([`ia3Config`]): The configuration of the (IA)^3 model.
        """

    def __init__(self, model, config, adapter_name,**kwargs):
        super().__init__(model, config, adapter_name,**kwargs)

    @staticmethod
    def _create_new_module(ia3_config, adapter_name, target, **kwargs):
        bias = hasattr(target, "bias") and target.bias is not None
        loaded_in_8bit = kwargs.pop("loaded_in_8bit", False)
        loaded_in_4bit = kwargs.pop("loaded_in_4bit", False)
        is_feedforward = kwargs.pop("is_feedforward", False)

        if loaded_in_8bit and isinstance(target, bnb.nn.Linear8bitLt):
            eightbit_kwargs = kwargs.copy()
            eightbit_kwargs.update(
                {
                    "has_fp16_weights": target.state.has_fp16_weights,
                    "memory_efficient_backward": target.state.memory_efficient_backward,
                    "threshold": target.state.threshold,
                    "index": target.index,
                }
            )
            new_module = Linear8bitLt(
                adapter_name,
                target.in_features,
                target.out_features,
                is_feedforward,
                bias=bias,
                **eightbit_kwargs,
            )
        elif loaded_in_4bit and isinstance(target, bnb.nn.Linear4bit):
            fourbit_kwargs = kwargs.copy()
            fourbit_kwargs.update(
                {
                    "compute_dtype": target.compute_dtype,
                    "compress_statistics": target.weight.compress_statistics,
                    "quant_type": target.weight.quant_type,
                }
            )
            new_module = Linear4bit(
                adapter_name,
                target.in_features,
                target.out_features,
                is_feedforward,
                bias=bias,
                **fourbit_kwargs,
            )
        elif isinstance(target, torch.nn.Conv2d):
            out_channels, in_channels = target.weight.size()[ :2 ]
            kernel_size = target.weight.size()[ 2: ]
            stride = target.stride
            padding = target.padding
            new_module = Conv2d(
                adapter_name=adapter_name,
                in_channels=in_channels,
                out_channels=out_channels,
                kernel_size=kernel_size,
                stride=stride,
                padding=padding,
                is_feedforward=is_feedforward,
                **kwargs,
            )
        else:
            if isinstance(target, torch.nn.Linear):
                in_features, out_features = target.in_features, target.out_features
                if kwargs[ "fan_in_fan_out" ]:
                    warnings.warn(
                        "fan_in_fan_out is set to True but the target module is `torch.nn.Linear`. "
                        "Setting fan_in_fan_out to False."
                    )
                    kwargs[ "fan_in_fan_out" ] = ia3_config.fan_in_fan_out = False
            elif isinstance(target, Conv1D):
                in_features, out_features = (
                    target.weight.ds_shape if hasattr(target.weight, "ds_shape") else target.weight.shape
                )
                kwargs[ "is_target_conv_1d_layer" ] = True  # useful for unloading later
                if not kwargs[ "fan_in_fan_out" ]:
                    warnings.warn(
                        "fan_in_fan_out is set to False but the target module is `Conv1D`. "
                        "Setting fan_in_fan_out to True."
                    )
                    kwargs[ "fan_in_fan_out" ] = ia3_config.fan_in_fan_out = True
            else:
                raise ValueError(
                    f"Target module {target} is not supported. "
                    f"Currently, only `torch.nn.Linear`, `torch.nn.Conv2d`, and `Conv1D` are supported."
                )
            new_module = Linear(
                adapter_name, in_features, out_features, is_feedforward=is_feedforward, bias=bias, **kwargs
            )
        return new_module

    @staticmethod
    def _check_target_module_exists(ia3_config, key):
        return check_target_module_exists(ia3_config, key)

    def _mark_only_adapters_as_trainable(self) -> None:
        for n, p in self.model.named_parameters():
            if "ia3_" not in n:
                p.requires_grad = False

    def _create_and_replace(
            self,
            ia3_config,
            adapter_name,
            target,
            target_name,
            parent,
            **optional_kwargs,
    ):
        loaded_in_8bit = optional_kwargs[ "loaded_in_8bit" ]
        loaded_in_4bit = optional_kwargs[ "loaded_in_4bit" ]
        current_key = optional_kwargs[ "current_key" ]

        # check if target module is in feedforward_modules
        is_feedforward = self._check_target_module_feedforward(ia3_config, current_key)

        kwargs = {
            "fan_in_fan_out": ia3_config.fan_in_fan_out,
            "init_ia3_weights": ia3_config.init_ia3_weights,
            "loaded_in_8bit": loaded_in_8bit,
            "loaded_in_4bit": loaded_in_4bit,
            "is_feedforward": is_feedforward,
        }

        if isinstance(target, IA3Layer):
            if target.is_feedforward != is_feedforward:
                raise ValueError(
                    "New adapter should have the same value for `is_feedforward` as previously added adapter."
                )
            if isinstance(target, torch.nn.Conv2d):
                target.update_layer(
                    adapter_name,
                    ia3_config.init_ia3_weights,
                )
            else:  # Linear
                target.update_layer(
                    adapter_name,
                    ia3_config.init_ia3_weights,
                )
        else:
            new_module = self._create_new_module(ia3_config, adapter_name, target, **kwargs)
            if adapter_name != self.active_adapter:
                # adding an additional adapter: it is not automatically trainable
                new_module.requires_grad_(False)
            self._replace_module(parent, target_name, new_module, target)

    @staticmethod
    def _check_target_module_feedforward(ia3_config, key) -> bool:
        """
        A helper private method that checks if the target module `key` matches with a feedforward module specified in
        `ia3_config`
        """
        if isinstance(ia3_config.feedforward_modules, str):
            is_feedforward = bool(re.fullmatch(ia3_config.feedforward_modules, key))
        else:
            is_feedforward = any(key.endswith(target_key) for target_key in ia3_config.feedforward_modules)
        return is_feedforward

    @staticmethod
    def _replace_module(parent, child_name, new_module, child):
        setattr(parent, child_name, new_module)
        new_module.weight = child.weight
        if child.bias is not None:
            new_module.bias = child.bias
        if getattr(child, "state", None) is not None:
            new_module.state = child.state
            new_module.to(child.weight.device)

        # dispatch to correct device
        for name, module in new_module.named_modules():
            if "ia3_" in name:
                module.to(child.weight.device)


    def get_petl_config_as_dict(self, inference: bool = False):
        config_dict = {}
        for key, value in self.petl_config.items():
            config = {k: v.value if isinstance(v, Enum) else v for k, v in asdict(value).items()}
            if inference:
                config[ "inference_mode" ] = True
        config_dict[ key ] = config
        return config

    def _set_adapter_layers(self, enabled=True):
        for module in self.model.modules():
            if isinstance(module, (IA3Layer, ModulesToSaveWrapper)):
                module.enable_adapters(enabled)

    def enable_adapter_layers(self):
        self._set_adapter_layers(enabled=True)

    def disable_adapter_layers(self):
        self._set_adapter_layers(enabled=False)

    def set_adapter(self, adapter_name):
        for module in self.model.modules():
            if isinstance(module, IA3Layer):
                if module.merged:
                    warnings.warn("Adapter cannot be set when the model is merged. Unmerging the model first.")
                    module.unmerge()
                module.set_adapter(adapter_name)

    def _prepare_adapter_config(self, petl_config, model_config):
        if petl_config.target_modules is None:
            if model_config[ "model_type" ] not in TRANSFORMERS_MODELS_TO_IA3_TARGET_MODULES_MAPPING:
                raise ValueError("Please specify `target_modules` in `petl_config`")
            petl_config.target_modules = TRANSFORMERS_MODELS_TO_IA3_TARGET_MODULES_MAPPING[
                model_config[ "model_type" ] ]
        if petl_config.feedforward_modules is None:
            if model_config[ "model_type" ] not in TRANSFORMERS_MODELS_TO_IA3_FEEDFORWARD_MODULES_MAPPING:
                raise ValueError("Please specify `feedforward_modules` in `petl_config`")
            petl_config.feedforward_modules = TRANSFORMERS_MODELS_TO_IA3_FEEDFORWARD_MODULES_MAPPING[
                model_config[ "model_type" ]
            ]
        return petl_config

    def merge_and_unload(self, safe_merge: bool = False):
        r"""
        This method merges the (IA)^3 layers into the base model. This is needed if someone wants to use the base model
        as a standalone model.

        Args:
            safe_merge (`bool`, `optional`, defaults to `False`):
                If True, the merge operation will be performed in a copy of the original weights and check for NaNs
                before merging the weights. This is useful if you want to check if the merge operation will produce
                NaNs. Defaults to `False`.
        """
        if getattr(self.model, "is_loaded_in_8bit", False):
            raise ValueError("Cannot merge ia3 layers when the model is loaded in 8-bit mode")

        if getattr(self.model, "is_loaded_in_4bit", False):
            raise ValueError("Cannot merge ia3 layers when the model is loaded in 4-bit mode")

        key_list = [ key for key, _ in self.model.named_modules() if "ia3" not in key ]
        for key in key_list:
            try:
                parent, target, target_name = _get_submodules(self.model, key)
            except AttributeError:
                continue

            # save any additional trainable modules part of `modules_to_save`
            if isinstance(target, ModulesToSaveWrapper):
                setattr(parent, target_name, target.modules_to_save[ target.active_adapter ])
                continue

            if not isinstance(target, IA3Layer):
                continue

            if isinstance(target, torch.nn.Conv2d):
                new_module = torch.nn.Conv2d(
                    target.in_channels,
                    target.out_channels,
                    kernel_size=target.kernel_size,
                    stride=target.stride,
                    padding=target.padding,
                    dilation=target.dilation,
                )
            else:
                bias = target.bias is not None
                if getattr(target, "is_target_conv_1d_layer", False):
                    new_module = Conv1D(target.out_features, target.in_features)
                else:
                    new_module = torch.nn.Linear(target.in_features, target.out_features, bias=bias)

            target.merge(safe_merge=safe_merge)
            self._replace_module(parent, target_name, new_module, target)

        return self.model

    def __getattr__(self, name: str):
        """Forward missing attributes to the wrapped module."""
        try:
            return super().__getattr__(name)  # defer to nn.Module's logic
        except AttributeError:
            if hasattr(self.model, name):
                return getattr(self.model, name)
            return getattr(self.model.model, name)

