// <autogenerated/>
// Define the symbol DESKBAND_WINFORMS for winforms or DESKBAND_WPF for wpf
// VERSION 3.1
// LICENSE: https://raw.githubusercontent.com/dsafa/CSDeskBand/master/LICENSE

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Reflection;
using System.Runtime.CompilerServices;
using System.Runtime.InteropServices;
using System.Windows;
using System.Windows.Documents;
using System.Windows.Interop;
using System.Windows.Media;
using CSDeskBand.ContextMenu;
using CSDeskBand.Interop;
using EverythingToolbar.Helpers;
using Microsoft.Win32;
using NLog;
using MSG = CSDeskBand.Interop.MSG;

#pragma warning disable 1591
namespace CSDeskBand
{
    using static DESKBANDINFO.DBIF;
    using static DESKBANDINFO.DBIM;
    using static DESKBANDINFO.DBIMF;

    /// <summary>
    /// Default implementation for icsdeskband
    /// </summary>
    internal sealed class CSDeskBandImpl : ICSDeskBand
    {
        private readonly IDeskBandProvider _provider;
        private readonly Dictionary<uint, DeskBandMenuAction> _contextMenuActions = new Dictionary<uint, DeskBandMenuAction>();
        private IntPtr _parentWindowHandle;
        private object _parentSite; // Has these interfaces: IInputObjectSite, IOleWindow, IOleCommandTarget, IBandSite
        private uint _id;
        private uint _menutStartId;
        private Guid _deskbandCommandGroupId = new Guid("EB0FE172-1A3A-11D0-89B3-00A0C90A90AC"); // Command group id for deskband. Used for IOleCommandTarge.Exec

        /// <summary>
        /// Initializes a new instance of the <see cref="CSDeskBandImpl"/> class
        /// with the handle to the window and the options.
        /// </summary>
        public CSDeskBandImpl(IDeskBandProvider provider)
        {
            _provider = provider;
            Options = provider.Options;
            Options.PropertyChanged += Options_PropertyChanged;
        }

        /// <summary>
        /// Occurs when the deskband is closed.
        /// </summary>
        internal event EventHandler Closed;

        /// <summary>
        /// Gets the <see cref="CSDeskBandOptions"/>.
        /// </summary>
        internal CSDeskBandOptions Options { get; }

        /// <summary>
        /// Gets the <see cref="TaskbarInfo"/>.
        /// </summary>
        internal TaskbarInfo TaskbarInfo { get; } = new TaskbarInfo();

        /// <inheritdoc/>
        public int GetWindow(out IntPtr phwnd)
        {
            phwnd = _provider.Handle;
            return HRESULT.S_OK;
        }

        /// <inheritdoc/>
        public int ContextSensitiveHelp(bool fEnterMode)
        {
            return HRESULT.E_NOTIMPL;
        }

        /// <inheritdoc/>
        public int ShowDW([In] bool fShow)
        {
            return HRESULT.S_OK;
        }

        /// <inheritdoc/>
        public int CloseDW([In] uint dwReserved)
        {
            Closed?.Invoke(this, null);
            return HRESULT.S_OK;
        }

        /// <inheritdoc/>
        public int ResizeBorderDW(RECT prcBorder, [In, MarshalAs(UnmanagedType.IUnknown)] IntPtr punkToolbarSite, bool fReserved)
        {
            // Must return notimpl
            return HRESULT.E_NOTIMPL;
        }

        /// <inheritdoc/>
        public int GetBandInfo(uint dwBandID, DESKBANDINFO.DBIF dwViewMode, ref DESKBANDINFO pdbi)
        {
            // Sizing information is requested whenever the taskbar changes size/orientation
            _id = dwBandID;

            if (pdbi.dwMask.HasFlag(DBIM_MINSIZE))
            {
                if (dwViewMode.HasFlag(DBIF_VIEWMODE_VERTICAL))
                {
                    pdbi.ptMinSize.Y = Options.MinVerticalSize.Width;
                    pdbi.ptMinSize.X = Options.MinVerticalSize.Height;
                }
                else
                {
                    pdbi.ptMinSize.X = Options.MinHorizontalSize.Width;
                    pdbi.ptMinSize.Y = Options.MinHorizontalSize.Height;
                }
            }

            // X is ignored
            if (pdbi.dwMask.HasFlag(DBIM_MAXSIZE))
            {
                if (dwViewMode.HasFlag(DBIF_VIEWMODE_VERTICAL))
                {
                    pdbi.ptMaxSize.Y = Options.MaxVerticalWidth;
                    pdbi.ptMaxSize.X = 0;
                }
                else
                {
                    pdbi.ptMaxSize.X = 0;
                    pdbi.ptMaxSize.Y = Options.MaxHorizontalHeight;
                }
            }

            // x member is ignored
            if (pdbi.dwMask.HasFlag(DBIM_INTEGRAL))
            {
                pdbi.ptIntegral.Y = Options.HeightIncrement;
                pdbi.ptIntegral.X = 0;
            }

            if (pdbi.dwMask.HasFlag(DBIM_ACTUAL))
            {
                if (dwViewMode.HasFlag(DBIF_VIEWMODE_VERTICAL))
                {
                    pdbi.ptActual.Y = Options.VerticalSize.Width;
                    pdbi.ptActual.X = Options.VerticalSize.Height;
                }
                else
                {
                    pdbi.ptActual.X = Options.HorizontalSize.Width;
                    pdbi.ptActual.Y = Options.HorizontalSize.Height;
                }
            }

            if (pdbi.dwMask.HasFlag(DBIM_TITLE))
            {
                pdbi.wszTitle = Options.Title;
                if (!Options.ShowTitle)
                {
                    pdbi.dwMask &= ~DBIM_TITLE;
                }
            }

            if (pdbi.dwMask.HasFlag(DBIM_MODEFLAGS))
            {
                pdbi.dwModeFlags = DBIMF_NORMAL;
                pdbi.dwModeFlags |= Options.IsFixed ? DBIMF_FIXED | DBIMF_NOGRIPPER : 0;
                pdbi.dwModeFlags |= Options.HeightCanChange ? DBIMF_VARIABLEHEIGHT : 0;
                pdbi.dwModeFlags &= ~DBIMF_BKCOLOR; // Don't use background color
            }

            TaskbarInfo.UpdateInfo();

            return HRESULT.S_OK;
        }

        /// <inheritdoc/>
        public int CanRenderComposited(out bool pfCanRenderComposited)
        {
            pfCanRenderComposited = true;
            return HRESULT.S_OK;
        }

        /// <inheritdoc/>
        public int SetCompositionState(bool fCompositionEnabled)
        {
            return HRESULT.S_OK;
        }

        /// <inheritdoc/>
        public int GetCompositionState(out bool pfCompositionEnabled)
        {
            pfCompositionEnabled = true;
            return HRESULT.S_OK;
        }

        /// <inheritdoc/>
        public int SetSite([In, MarshalAs(UnmanagedType.IUnknown)] object pUnkSite)
        {
            // Let gc release old site
            _parentSite = null;

            // pUnkSite null means deskband was closed
            if (pUnkSite == null)
            {
                Closed?.Invoke(this, null);
                return HRESULT.S_OK;
            }

            try
            {
                var oleWindow = (IOleWindow)pUnkSite;
                oleWindow.GetWindow(out _parentWindowHandle);
                User32.SetParent(_provider.Handle, _parentWindowHandle);

                _parentSite = (IInputObjectSite)pUnkSite;
                return HRESULT.S_OK;
            }
            catch
            {
                return HRESULT.E_FAIL;
            }
        }

        /// <inheritdoc/>
        public int GetSite(ref Guid riid, [MarshalAs(UnmanagedType.IUnknown)] out IntPtr ppvSite)
        {
            if (_parentSite == null)
            {
                ppvSite = IntPtr.Zero;
                return HRESULT.E_FAIL;
            }

            return Marshal.QueryInterface(Marshal.GetIUnknownForObject(_parentSite), ref riid, out ppvSite);
        }

        /// <inheritdoc/>
        public int QueryContextMenu(IntPtr hMenu, uint indexMenu, uint idCmdFirst, uint idCmdLast, QueryContextMenuFlags uFlags)
        {
            if (uFlags.HasFlag(QueryContextMenuFlags.CMF_DEFAULTONLY))
            {
                return HRESULT.MakeHResult((uint)HRESULT.S_OK, 0, 0);
            }

            _menutStartId = idCmdFirst;
            foreach (var item in Options.ContextMenuItems)
            {
                item.AddToMenu(hMenu, indexMenu++, ref idCmdFirst, _contextMenuActions);
            }

            return HRESULT.MakeHResult((uint)HRESULT.S_OK, 0, idCmdFirst + 1); // #id of last command + 1
        }

        /// <inheritdoc/>
        public int InvokeCommand(IntPtr pici)
        {
            var commandInfo = Marshal.PtrToStructure<CMINVOKECOMMANDINFO>(pici);
#pragma warning disable CS0219 // Variable is assigned but its value is never used
            var isUnicode = false;
            var isExtended = false;
#pragma warning restore CS0219 // Variable is assigned but its value is never used
            var verbPtr = commandInfo.lpVerb;

            if (commandInfo.cbSize == Marshal.SizeOf<CMINVOKECOMMANDINFOEX>())
            {
                isExtended = true;

                var extended = Marshal.PtrToStructure<CMINVOKECOMMANDINFOEX>(pici);
                if (extended.fMask.HasFlag(CMINVOKECOMMANDINFOEX.CMIC.CMIC_MASK_UNICODE))
                {
                    isUnicode = true;
                    verbPtr = extended.lpVerbW;
                }
            }

            if (User32.HiWord(commandInfo.lpVerb.ToInt32()) != 0)
            {
                // TODO verbs
                return HRESULT.E_FAIL;
            }

            var cmdIndex = User32.LoWord(verbPtr.ToInt32());

            if (!_contextMenuActions.TryGetValue((uint)cmdIndex + _menutStartId, out var action))
            {
                return HRESULT.E_FAIL;
            }

            action.DoAction();
            return HRESULT.S_OK;
        }

        /// <inheritdoc/>
        public int GetCommandString(ref uint idcmd, uint uflags, ref uint pwReserved, out string pcszName, uint cchMax)
        {
            pcszName = "";
            return HRESULT.E_NOTIMPL;
        }

        /// <inheritdoc/>
        public int HandleMenuMsg(uint uMsg, IntPtr wParam, IntPtr lParam)
        {
            return HandleMenuMsg2(uMsg, wParam, lParam, out var i);
        }

        /// <inheritdoc/>
        public int HandleMenuMsg2(uint uMsg, IntPtr wParam, IntPtr lParam, out IntPtr plResult)
        {
            plResult = IntPtr.Zero;
            return HRESULT.S_OK;
        }

        /// <inheritdoc/>
        public int GetClassID(out Guid pClassID)
        {
            pClassID = _provider.Guid;
            return HRESULT.S_OK;
        }

        /// <inheritdoc/>
        public int GetSizeMax(out ulong pcbSize)
        {
            pcbSize = 0;
            return HRESULT.S_OK;
        }

        /// <inheritdoc/>
        public int IsDirty()
        {
            return HRESULT.S_OK;
        }

        /// <inheritdoc/>
        public int Load(object pStm)
        {
            return HRESULT.S_OK;
        }

        /// <inheritdoc/>
        public int Save(IntPtr pStm, bool fClearDirty)
        {
            return HRESULT.S_OK;
        }

        /// <summary>
        /// Closes the deskband.
        /// </summary>
        public void CloseDeskBand()
        {
            var bandSite = (IBandSite)_parentSite;
            bandSite.RemoveBand(_id);
        }

        /// <inheritdoc/>
        public int UIActivateIO(int fActivate, ref MSG msg)
        {
            _provider.HasFocus = fActivate != 0;
            UpdateFocus(_provider.HasFocus);
            return HRESULT.S_OK;
        }

        /// <inheritdoc/>
        public int HasFocusIO()
        {
            return _provider.HasFocus ? HRESULT.S_OK : HRESULT.S_FALSE;
        }

        /// <inheritdoc/>
        public int TranslateAcceleratorIO(ref MSG msg)
        {
            return HRESULT.S_OK;
        }

        /// <summary>
        /// Updates the focus on the deskband. Explorer will call <see cref="UIActivateIO(int, ref MSG)"/> for example if tabbing when the taskbar is focused. 
        /// But if focus is acquired without in other ways, then explorer isn't aware of it and <see cref="IInputObjectSite.OnFocusChangeIS(object, int)"/> needs to be called.
        /// </summary>
        /// <param name="focused">True if focused.</param>
        public void UpdateFocus(bool focused)
        {
            (_parentSite as IInputObjectSite)?.OnFocusChangeIS(this, focused ? 1 : 0);
        }

        private void Options_PropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            if (_parentSite == null)
            {
                return;
            }

            var parent = (IOleCommandTarget)_parentSite;

            // Set pvaln to the id that was passed in SetSite
            // When int is marshalled to variant, it is marshalled as VT_i4. See default marshalling for objects
            parent.Exec(ref _deskbandCommandGroupId, (uint)tagDESKBANDCID.DBID_BANDINFOCHANGED, 0, IntPtr.Zero, IntPtr.Zero);
        }
    }
}
namespace CSDeskBand
{
    /// <summary>
    /// Options to configure the deskband
    /// </summary>
    public sealed class CSDeskBandOptions : INotifyPropertyChanged
    {
        /// <summary>
        /// Height for a default horizontal taskbar.
        /// </summary>
        public static readonly int TaskbarHorizontalHeightLarge = 40;

        /// <summary>
        /// Height for a default horizontal taskbar with small icons.
        /// </summary>
        public static readonly int TaskbarHorizontalHeightSmall = 30;

        /// <summary>
        /// Width for a default vertical taskbar. There is no small vertical taskbar.
        /// </summary>
        public static readonly int TaskbarVerticalWidth = 62;

        /// <summary>
        /// Value that represents no limit for deskband size.
        /// </summary>
        /// <seealso cref="MaxHorizontalHeight"/>
        /// <seealso cref="MaxVerticalWidth"/>
        public static readonly int NoLimit = -1;

        private DeskBandSize _horizontalSize;
        private int _maxHorizontalHeight;
        private DeskBandSize _minHorizontalSize;
        private DeskBandSize _verticalSize;
        private int _maxVerticalWidth;
        private DeskBandSize _minVerticalSize;
        private string _title = "";
        private bool _showTitle;
        private bool _isFixed;
        private int _heightIncrement = 1;
        private bool _heightCanChange = true;
        private ICollection<DeskBandMenuItem> _contextMenuItems = new List<DeskBandMenuItem>();

        /// <summary>
        /// Initializes a new instance of the <see cref="CSDeskBandOptions"/> class.
        /// </summary>
        public CSDeskBandOptions()
        {
            // Initialize in constructor to hook up property change events
            HorizontalSize = new DeskBandSize(200, TaskbarHorizontalHeightLarge);
            MaxHorizontalHeight = NoLimit;
            MinHorizontalSize = new DeskBandSize(NoLimit, NoLimit);

            VerticalSize = new DeskBandSize(TaskbarVerticalWidth, 200);
            MaxVerticalWidth = NoLimit;
            MinVerticalSize = new DeskBandSize(NoLimit, NoLimit);
        }

        /// <summary>
        /// Occurs when a property has change.
        /// </summary>
        public event PropertyChangedEventHandler PropertyChanged;

        /// <summary>
        /// Gets or sets a value indicating whether the height of the horizontal deskband is allowed to change.
        /// <para/>
        /// Or for a deskband in the vertical orientation, if the width can change.
        /// Works alongside with the property <see cref="HeightIncrement"/>.
        /// </summary>
        /// <value>
        /// <see langword="true"/> if the height / width of the deskband can be changed. <see langword="false"/> to prevent changes.
        /// The default value is <see langword="true"/>.
        /// </value>
        public bool HeightCanChange
        {
            get => _heightCanChange;
            set
            {
                if (value == _heightCanChange)
                {
                    return;
                }

                _heightCanChange = value;
                OnPropertyChanged();
            }
        }

        /// <summary>
        /// Gets or sets the height step size of a horizontal deskband when it is being resized.
        /// For a deskband in the vertical orientation, it will be the step size of the width.
        /// <para/>
        /// The deskband will only be resized to multiples of this value.
        /// </summary>
        /// <example>
        /// If increment is 50, then the height of the deskband can only be resized to 50, 100 ...
        /// </example>
        /// <value>
        /// The step size for resizing. This value is only used if <see cref="HeightCanChange"/> is true. If the value is less than 0, the height / width can be any size.
        /// The default value is 1.
        /// </value>
        public int HeightIncrement
        {
            get => _heightIncrement;
            set
            {
                if (value == _heightIncrement)
                {
                    return;
                }

                _heightIncrement = value;
                OnPropertyChanged();
            }
        }

        /// <summary>
        /// Gets or sets a value indicating whether the deskband has a fixed position and size.
        /// </summary>
        /// <value>
        /// <see langword="true"/> if the deskband is fixed. <see langword="false"/> if the deskband can be adjusted.
        /// The default value is <see langword="false"/>.
        /// </value>
        public bool IsFixed
        {
            get => _isFixed;
            set
            {
                if (value == _isFixed)
                {
                    return;
                }

                _isFixed = value;
                OnPropertyChanged();
            }
        }

        /// <summary>
        /// Gets or sets a value indicating whether the value of <see cref="Title"/> is shown next to the deskband.
        /// </summary>
        /// <value>
        /// <see langword="true"/> if the title should be shown. <see langword="false"/> if the title is hidden.
        /// The default value is <see langword="false"/>.
        /// </value>
        public bool ShowTitle
        {
            get => _showTitle;
            set
            {
                if (value == _showTitle)
                {
                    return;
                }

                _showTitle = value;
                OnPropertyChanged();
            }
        }

        /// <summary>
        /// Gets or sets the title of the deskband. This will be shown if <see cref="ShowTitle"/> is <see langword="true"/>.
        /// </summary>
        /// <value>
        /// The title to display. If the title is null, it will be converted to an empty string.
        /// The default value is an empty string.
        /// </value>
        public string Title
        {
            get => _title;
            set
            {
                if (value == _title)
                {
                    return;
                }

                _title = value ?? "";
                OnPropertyChanged();
            }
        }

        /// <summary>
        /// Gets or sets the minimum <see cref="DeskBandSize"/> of the deskband in the vertical orientation.
        /// </summary>
        /// <seealso cref="TaskbarOrientation"/>
        /// <value>
        /// The default value is <see cref="NoLimit"/> for the width and height.
        /// </value>
        public DeskBandSize MinVerticalSize
        {
            get => _minVerticalSize;
            set
            {
                if (value.Equals(_minVerticalSize))
                {
                    return;
                }

                _minVerticalSize = value;
                _minVerticalSize.PropertyChanged += (sender, args) => OnPropertyChanged();
                OnPropertyChanged();
            }
        }

        /// <summary>
        /// Gets or sets the maximum width of the deskband in the vertical orientation.
        /// </summary>
        /// <remarks>
        /// The maximum height will have to be addressed in your code as there is no limit to the height of the deskband when vertical.
        /// </remarks>
        /// <seealso cref="TaskbarOrientation"/>
        /// <value>
        /// The default value is <see cref="NoLimit"/>.
        /// </value>
        public int MaxVerticalWidth
        {
            get => _maxVerticalWidth;
            set
            {
                if (value.Equals(_maxVerticalWidth))
                {
                    return;
                }

                _maxVerticalWidth = value;
                OnPropertyChanged();
            }
        }

        /// <summary>
        /// Gets or sets the ideal <see cref="DeskBandSize"/> of the deskband in the vertical orientation.
        /// There is no guarantee that the deskband will be this size.
        /// </summary>
        /// <seealso cref="TaskbarOrientation"/>
        /// <value>
        /// The default value is <see cref="TaskbarVerticalWidth"/> for the width and 200 for the height.
        /// </value>
        public DeskBandSize VerticalSize
        {
            get => _verticalSize;
            set
            {
                if (value.Equals(_verticalSize))
                {
                    return;
                }

                _verticalSize = value;
                _verticalSize.PropertyChanged += (sender, args) => OnPropertyChanged();
                OnPropertyChanged();
            }
        }

        /// <summary>
        /// Gets or sets the minimum <see cref="DeskBandSize"/> of the deskband in the horizontal orientation.
        /// </summary>
        /// <seealso cref="TaskbarOrientation"/>
        /// <value>
        /// The default value is <see cref="NoLimit"/>.
        /// </value>
        public DeskBandSize MinHorizontalSize
        {
            get => _minHorizontalSize;
            set
            {
                if (value.Equals(_minHorizontalSize))
                {
                    return;
                }

                _minHorizontalSize = value;
                _minHorizontalSize.PropertyChanged += (sender, args) => OnPropertyChanged();
                OnPropertyChanged();
            }
        }

        /// <summary>
        /// Gets or sets the maximum height of the deskband in the horizontal orientation.
        /// </summary>
        /// <remarks>
        /// The maximum width will have to be addressed in your code as there is no limit to the width of the deskband when horizontal.
        /// </remarks>
        /// <seealso cref="TaskbarOrientation"/>
        /// <value>
        /// The default value is <see cref="NoLimit"/>.
        /// </value>
        public int MaxHorizontalHeight
        {
            get => _maxHorizontalHeight;
            set
            {
                if (value.Equals(_maxHorizontalHeight))
                {
                    return;
                }

                _maxHorizontalHeight = value;
                OnPropertyChanged();
            }
        }

        /// <summary>
        /// Gets or sets the ideal <see cref="DeskBandSize"/> of the deskband in the horizontal orientation.
        /// There is no guarantee that the deskband will be this size.
        /// </summary>
        /// <seealso cref="TaskbarOrientation"/>
        /// <value>
        /// The default value is 200 for the width and <see cref="TaskbarHorizontalHeightLarge"/> for the height.
        /// </value>
        public DeskBandSize HorizontalSize
        {
            get => _horizontalSize;
            set
            {
                if (value.Equals(_horizontalSize))
                {
                    return;
                }

                _horizontalSize = value;
                _horizontalSize.PropertyChanged += (sender, args) => OnPropertyChanged();
                OnPropertyChanged();
            }
        }

        /// <summary>
        /// Gets or sets the collection of <see cref="DeskBandMenuItem"/> the comprise the deskbands context menu.
        /// </summary>
        /// <value>
        /// A list of <see cref="DeskBandMenuItem"/> for the context menu. An empty collection indicates no context menu.
        /// </value>
        /// <remarks>
        /// These context menu items are in addition of the default ones that windows provides.
        /// The items will appear in their enumerated order.
        /// </remarks>
        public ICollection<DeskBandMenuItem> ContextMenuItems
        {
            get => _contextMenuItems;
            set
            {
                if (Equals(value, _contextMenuItems))
                {
                    return;
                }

                _contextMenuItems = value;
                OnPropertyChanged();
            }
        }

        private void OnPropertyChanged([CallerMemberName] string propertyName = null)
        {
            PropertyChanged?.Invoke(this, new PropertyChangedEventArgs(propertyName));
        }
    }
}
namespace CSDeskBand
{
    /// <summary>
    /// Specifies registration configuration for a deskband.
    /// </summary>
    [AttributeUsage(AttributeTargets.Class)]
    internal sealed class CSDeskBandRegistrationAttribute : Attribute
    {
        /// <summary>
        /// Gets or sets the name of the deskband in the toolbar menu.
        /// </summary>
        /// <value>
        /// The name is used to select the deskband from the toolbars menu.
        /// </value>
        public string Name { get; set; }

        /// <summary>
        /// Gets or sets a value indicating whether to automatically show the deskband after registration.
        /// </summary>
        /// <value>
        /// <see langword="true"/> if the deskband should be automatically shown after registration; <see langword="false"/> otherwise.
        /// </value>
        public bool ShowDeskBand { get; set; }
    }
}
#pragma warning disable 1591
#if DESKBAND_WINFORMS
namespace CSDeskBand
{
    using System;
    using System.Runtime.InteropServices;
    using System.Windows.Forms;
    using CSDeskBand.Interop;

    /// <summary>
    /// Winforms implementation of <see cref="ICSDeskBand"/>.
    /// The deskband should also have these attributes <see cref="ComVisibleAttribute"/>, <see cref="GuidAttribute"/>, <see cref="CSDeskBandRegistrationAttribute"/>.
    /// </summary>
    public abstract class CSDeskBandWin : ICSDeskBand, IDeskBandProvider
    {
        private readonly CSDeskBandImpl _impl;

        /// <summary>
        /// Initializes a new instance of the <see cref="CSDeskBandWin"/> class.
        /// </summary>
        public CSDeskBandWin()
        {
            Options.Title = RegistrationHelper.GetToolbarName(GetType());
            _impl = new CSDeskBandImpl(this);
            _impl.Closed += (o, e) => DeskbandOnClosed();
            TaskbarInfo = _impl.TaskbarInfo;
        }

        [ComRegisterFunction]
        private static void Register(Type t)
        {
            RegistrationHelper.Register(t);
        }

        [ComUnregisterFunction]
        private static void Unregister(Type t)
        {
            RegistrationHelper.Unregister(t);
        }

        /// <summary>
        /// Gets the taskbar information
        /// </summary>
        protected TaskbarInfo TaskbarInfo { get; }

        /// <summary>
        /// Gets the main control for the deskband.
        /// </summary>
        protected abstract Control Control { get; }

        /// <summary>
        /// Gets the options for this deskband.
        /// </summary>
        /// <seealso cref="CSDeskBandOptions"/>
        public CSDeskBandOptions Options { get; } = new CSDeskBandOptions();

        /// <summary>
        /// Gets the handle
        /// </summary>
        public IntPtr Handle => Control.Handle;

        /// <summary>
        /// Gets the deskband guid
        /// </summary>
        public Guid Guid => GetType().GUID;

        public bool HasFocus
        {
            get => Control?.ContainsFocus ?? false;
            set
            {
                if (value)
                {
                    Control?.Focus();
                }
            }
        }

        /// <summary>
        /// Updates the focus on this deskband.
        /// </summary>
        /// <param name="focused"><see langword="true"/> if focused.</param>
        public void UpdateFocus(bool focused)
        {
            _impl.UpdateFocus(focused);
        }

        /// <summary>
        /// Handle closing of the deskband.
        /// </summary>
        protected virtual void DeskbandOnClosed()
        {
        }

        public int GetWindow(out IntPtr phwnd)
        {
            return _impl.GetWindow(out phwnd);
        }

        public int ContextSensitiveHelp(bool fEnterMode)
        {
            return _impl.ContextSensitiveHelp(fEnterMode);
        }

        public int ShowDW([In] bool fShow)
        {
            return _impl.ShowDW(fShow);
        }

        public int CloseDW([In] uint dwReserved)
        {
            return _impl.CloseDW(dwReserved);
        }

        public int ResizeBorderDW(RECT prcBorder, [In, MarshalAs(UnmanagedType.IUnknown)] IntPtr punkToolbarSite, bool fReserved)
        {
            return _impl.ResizeBorderDW(prcBorder, punkToolbarSite, fReserved);
        }

        public int GetBandInfo(uint dwBandID, DESKBANDINFO.DBIF dwViewMode, ref DESKBANDINFO pdbi)
        {
            return _impl.GetBandInfo(dwBandID, dwViewMode, ref pdbi);
        }

        public int CanRenderComposited(out bool pfCanRenderComposited)
        {
            return _impl.CanRenderComposited(out pfCanRenderComposited);
        }

        public int SetCompositionState(bool fCompositionEnabled)
        {
            return _impl.SetCompositionState(fCompositionEnabled);
        }

        public int GetCompositionState(out bool pfCompositionEnabled)
        {
            return _impl.GetCompositionState(out pfCompositionEnabled);
        }

        public int SetSite([In, MarshalAs(UnmanagedType.IUnknown)] object pUnkSite)
        {
            return _impl.SetSite(pUnkSite);
        }

        public int GetSite(ref Guid riid, [MarshalAs(UnmanagedType.IUnknown)] out IntPtr ppvSite)
        {
            return _impl.GetSite(ref riid, out ppvSite);
        }

        public int QueryContextMenu(IntPtr hMenu, uint indexMenu, uint idCmdFirst, uint idCmdLast, QueryContextMenuFlags uFlags)
        {
            return _impl.QueryContextMenu(hMenu, indexMenu, idCmdFirst, idCmdLast, uFlags);
        }

        public int InvokeCommand(IntPtr pici)
        {
            return _impl.InvokeCommand(pici);
        }

        public int GetCommandString(ref uint idcmd, uint uflags, ref uint pwReserved, [MarshalAs(UnmanagedType.LPTStr)] out string pcszName, uint cchMax)
        {
            return _impl.GetCommandString(ref idcmd, uflags, ref pwReserved, out pcszName, cchMax);
        }

        public int HandleMenuMsg(uint uMsg, IntPtr wParam, IntPtr lParam)
        {
            return _impl.HandleMenuMsg(uMsg, wParam, lParam);
        }

        public int HandleMenuMsg2(uint uMsg, IntPtr wParam, IntPtr lParam, out IntPtr plResult)
        {
            return _impl.HandleMenuMsg2(uMsg, wParam, lParam, out plResult);
        }

        public int GetClassID(out Guid pClassID)
        {
            return _impl.GetClassID(out pClassID);
        }

        public int GetSizeMax(out ulong pcbSize)
        {
            return _impl.GetSizeMax(out pcbSize);
        }

        public int IsDirty()
        {
            return _impl.IsDirty();
        }

        public int Load(object pStm)
        {
            return _impl.Load(pStm);
        }

        public int Save(IntPtr pStm, bool fClearDirty)
        {
            return _impl.Save(pStm, fClearDirty);
        }

        public int UIActivateIO(int fActivate, ref MSG msg)
        {
            return _impl.UIActivateIO(fActivate, ref msg);
        }

        public int HasFocusIO()
        {
            return _impl.HasFocusIO();
        }

        public int TranslateAcceleratorIO(ref MSG msg)
        {
            return _impl.TranslateAcceleratorIO(ref msg);
        }
    }
}
#endif
#pragma warning disable 1591
#if DESKBAND_WPF
namespace CSDeskBand
{
    /// <summary>
    /// Wpf implementation of <see cref="ICSDeskBand"/>
    /// The deskband should also have these attributes <see cref="ComVisibleAttribute"/>, <see cref="GuidAttribute"/>, <see cref="CSDeskBandRegistrationAttribute"/>.
    /// </summary>
    public abstract class CSDeskBandWpf : ICSDeskBand, IDeskBandProvider
    {
        private readonly CSDeskBandImpl _impl;
        private readonly AdornerDecorator _rootVisual;

        private static readonly ILogger Logger = ToolbarLogger.GetLogger<CSDeskBandWpf>();

        /// <summary>
        /// Initializes a new instance of the <see cref="CSDeskBandWpf"/> class.
        /// </summary>
        public CSDeskBandWpf()
        {
            ToolbarLogger.Initialize("Deskband");

            Options.Title = RegistrationHelper.GetToolbarName(GetType());

            var hwndSourceParameters = new HwndSourceParameters("Deskband host for wpf")
            {
                TreatAsInputRoot = true,
                WindowStyle = unchecked((int)(WindowStyles.WS_VISIBLE | WindowStyles.WS_POPUP)),
                HwndSourceHook = HwndSourceHook,
            };

            HwndSource = new HwndSource(hwndSourceParameters);
            HwndSource.SizeToContent = SizeToContent.Manual;
            _rootVisual = new AdornerDecorator();
            HwndSource.RootVisual = _rootVisual;
            HwndSource.CompositionTarget.BackgroundColor = Colors.Transparent;

            _impl = new CSDeskBandImpl(this);

            _impl.Closed += (o, e) => DeskbandOnClosed();
            TaskbarInfo = _impl.TaskbarInfo;
        }

        /// <summary>
        /// The <see cref="System.Windows.Interop.HwndSourceHook"/>. for <see cref="HwndSource"/>.
        /// </summary>
        /// <param name="hwnd"></param>
        /// <param name="msg"></param>
        /// <param name="wparam"></param>
        /// <param name="lparam"></param>
        /// <param name="handled"></param>
        /// <returns></returns>
        protected virtual IntPtr HwndSourceHook(IntPtr hwnd, int msg, IntPtr wparam, IntPtr lparam, ref bool handled)
        {
            switch (msg)
            {
                // Handle hit testing against transparent areas
                case (int)WindowMessages.WM_NCHITTEST:
                    var mouseX = LowWord(lparam);
                    var mouseY = HighWord(lparam);
                    var relativepoint = HwndSource.RootVisual.PointFromScreen(new Point(mouseX, mouseY));
                    var result = VisualTreeHelper.HitTest(HwndSource.RootVisual, relativepoint);
                    if (result?.VisualHit != null)
                    {
                        handled = true;
                        return new IntPtr((int)HitTestMessageResults.HTCLIENT);
                    }

                    handled = true;
                    return new IntPtr((int)HitTestMessageResults.HTTRANSPARENT);
            }

            handled = false;
            return IntPtr.Zero;
        }

        protected static int LowWord(IntPtr value)
        {
            return unchecked((short)(long)value);
        }

        protected static int HighWord(IntPtr value)
        {
            return unchecked((short)((long)value >> 16));
        }

        /// <summary>
        /// Gets the <see cref="System.Windows.Interop.HwndSource"/> that hosts the wpf content.
        /// </summary>
        protected HwndSource HwndSource { get; }

        /// <summary>
        /// Gets the taskbar information
        /// </summary>
        protected TaskbarInfo TaskbarInfo { get; }

        /// <summary>
        /// Gets main UI element for the deskband.
        /// </summary>
        protected abstract UIElement UIElement { get; }

        /// <summary>
        /// Gets the options for this deskband.
        /// </summary>
        /// <seealso cref="CSDeskBandOptions"/>
        public CSDeskBandOptions Options { get; } = new CSDeskBandOptions();

        /// <summary>
        /// Gets the handle
        /// </summary>
        public IntPtr Handle
        {
            get
            {
                if (_rootVisual.Child == null)
                {
                    _rootVisual.Child = UIElement;
                }

                return HwndSource.Handle;
            }
        }

        /// <summary>
        /// Gets the deskband guid
        /// </summary>
        public Guid Guid => GetType().GUID;

        public bool HasFocus
        {
            get => UIElement?.IsKeyboardFocusWithin ?? false;
            set
            {
                if (value)
                {
                    UIElement?.Focus();
                }
            }
        }

        /// <summary>
        /// Updates the focus on this deskband.
        /// </summary>
        /// <param name="focused"><see langword="true"/> if focused.</param>
        public void UpdateFocus(bool focused)
        {
            _impl.UpdateFocus(focused);
        }

        /// <summary>
        /// Handle closing of the deskband.
        /// </summary>
        protected virtual void DeskbandOnClosed()
        {
        }

        public int GetWindow(out IntPtr phwnd)
        {
            return _impl.GetWindow(out phwnd);
        }

        public int ContextSensitiveHelp(bool fEnterMode)
        {
            return _impl.ContextSensitiveHelp(fEnterMode);
        }

        public int ShowDW([In] bool fShow)
        {
            return _impl.ShowDW(fShow);
        }

        public int CloseDW([In] uint dwReserved)
        {
            return _impl.CloseDW(dwReserved);
        }

        public int ResizeBorderDW(RECT prcBorder, [In, MarshalAs(UnmanagedType.IUnknown)] IntPtr punkToolbarSite, bool fReserved)
        {
            return _impl.ResizeBorderDW(prcBorder, punkToolbarSite, fReserved);
        }

        public int GetBandInfo(uint dwBandID, DESKBANDINFO.DBIF dwViewMode, ref DESKBANDINFO pdbi)
        {
            return _impl.GetBandInfo(dwBandID, dwViewMode, ref pdbi);
        }

        public int CanRenderComposited(out bool pfCanRenderComposited)
        {
            return _impl.CanRenderComposited(out pfCanRenderComposited);
        }

        public int SetCompositionState(bool fCompositionEnabled)
        {
            return _impl.SetCompositionState(fCompositionEnabled);
        }

        public int GetCompositionState(out bool pfCompositionEnabled)
        {
            return _impl.GetCompositionState(out pfCompositionEnabled);
        }

        public int SetSite([In, MarshalAs(UnmanagedType.IUnknown)] object pUnkSite)
        {
            return _impl.SetSite(pUnkSite);
        }

        public int GetSite(ref Guid riid, [MarshalAs(UnmanagedType.IUnknown)] out IntPtr ppvSite)
        {
            return _impl.GetSite(ref riid, out ppvSite);
        }

        public int QueryContextMenu(IntPtr hMenu, uint indexMenu, uint idCmdFirst, uint idCmdLast, QueryContextMenuFlags uFlags)
        {
            return _impl.QueryContextMenu(hMenu, indexMenu, idCmdFirst, idCmdLast, uFlags);
        }

        public int InvokeCommand(IntPtr pici)
        {
            return _impl.InvokeCommand(pici);
        }

        public int GetCommandString(ref uint idcmd, uint uflags, ref uint pwReserved, [MarshalAs(UnmanagedType.LPTStr)] out string pcszName, uint cchMax)
        {
            return _impl.GetCommandString(ref idcmd, uflags, ref pwReserved, out pcszName, cchMax);
        }

        public int HandleMenuMsg(uint uMsg, IntPtr wParam, IntPtr lParam)
        {
            return _impl.HandleMenuMsg(uMsg, wParam, lParam);
        }

        public int HandleMenuMsg2(uint uMsg, IntPtr wParam, IntPtr lParam, out IntPtr plResult)
        {
            return _impl.HandleMenuMsg2(uMsg, wParam, lParam, out plResult);
        }

        public int GetClassID(out Guid pClassID)
        {
            return _impl.GetClassID(out pClassID);
        }

        public int GetSizeMax(out ulong pcbSize)
        {
            return _impl.GetSizeMax(out pcbSize);
        }

        public int IsDirty()
        {
            return _impl.IsDirty();
        }

        public int Load(object pStm)
        {
            return _impl.Load(pStm);
        }

        public int Save(IntPtr pStm, bool fClearDirty)
        {
            return _impl.Save(pStm, fClearDirty);
        }

        public int UIActivateIO(int fActivate, ref MSG msg)
        {
            return _impl.UIActivateIO(fActivate, ref msg);
        }

        public int HasFocusIO()
        {
            return _impl.HasFocusIO();
        }

        public int TranslateAcceleratorIO(ref MSG msg)
        {
            return _impl.TranslateAcceleratorIO(ref msg);
        }

        [ComRegisterFunction]
        private static void Register(Type t)
        {
            RegistrationHelper.Register(t);
        }

        [ComUnregisterFunction]
        private static void Unregister(Type t)
        {
            RegistrationHelper.Unregister(t);
        }
    }
}
#endif
namespace CSDeskBand
{
    /// <summary>
    /// Size class that is independent of winforms or wpf.
    /// </summary>
    public sealed class DeskBandSize : INotifyPropertyChanged
    {
        private int _width;
        private int _height;

        /// <summary>
        /// Initializes a new instance of the <see cref="DeskBandSize"/> class.
        /// </summary>
        /// <param name="width">The <see cref="Width"/> component.</param>
        /// <param name="height">The <see cref="Height"/> component.</param>
        public DeskBandSize(int width, int height)
        {
            Width = width;
            Height = height;
        }

        /// <inheritdoc/>
        public event PropertyChangedEventHandler PropertyChanged;

        /// <summary>
        /// Gets or sets the width component of the size.
        /// </summary>
        public int Width
        {
            get => _width;
            set
            {
                if (value == _width)
                {
                    return;
                }

                _width = value;
                OnPropertyChanged();
            }
        }

        /// <summary>
        /// Gets or sets the height component of the size.
        /// </summary>
        public int Height
        {
            get => _height;
            set
            {
                if (value == _height)
                {
                    return;
                }

                _height = value;
                OnPropertyChanged();
            }
        }

        private void OnPropertyChanged([CallerMemberName] string propertyName = null)
        {
            PropertyChanged?.Invoke(this, new PropertyChangedEventArgs(propertyName));
        }

#if DESKBAND_WPF
        /// <summary>
        /// Converts from <see cref="System.Windows.Size"/> to <see cref="DeskBandSize"/>.
        /// </summary>
        /// <param name="size">The <see cref="System.Windows.Size"/> to convert.</param>
        public static implicit operator DeskBandSize(Size size)
        {
            return new DeskBandSize(Convert.ToInt32(size.Width), Convert.ToInt32(size.Height));
        }

        /// <summary>
        /// Converts from <see cref="DeskBandSize"/> to <see cref="System.Windows.Size"/>.
        /// </summary>
        /// <param name="size">The <see cref="DeskBandSize"/> to convert.</param>
        public static implicit operator Size(DeskBandSize size)
        {
            return new Size(size.Width, size.Height);
        }
#endif

#if DESKBAND_WINFORMS
        /// <summary>
        /// Converts from <see cref="System.Drawing.Size"/> to <see cref="DeskBandSize"/>.
        /// </summary>
        /// <param name="size">The <see cref="System.Drawing.Size"/> to convert.</param>
        public static implicit operator DeskBandSize(System.Drawing.Size size)
        {
            return new DeskBandSize(size.Width, size.Height);
        }

        /// <summary>
        /// Converts from <see cref="DeskBandSize"/> to <see cref="System.Drawing.Size"/>.
        /// </summary>
        /// <param name="size">The <see cref="DeskBandSize"/> to convert.</param>
        public static implicit operator System.Drawing.Size(DeskBandSize size)
        {
            return new System.Drawing.Size(size.Width, size.Height);
        }
#endif
    }
}
namespace CSDeskBand
{
    /// <summary>
    /// Deskband Interface
    /// </summary>
    public interface ICSDeskBand : IDeskBand2, IObjectWithSite, IContextMenu3, IPersistStream, IInputObject
    {
    }
}
namespace CSDeskBand
{
    internal interface IDeskBandProvider
    {
        IntPtr Handle { get; }
        CSDeskBandOptions Options { get; }
        Guid Guid { get; }
        bool HasFocus { get; set; }
    }
}
namespace CSDeskBand
{
    /// <summary>
    /// Helper class to register deskband.
    /// </summary>
    internal static class RegistrationHelper
    {
        /// <summary>
        /// Register the deskband.
        /// </summary>
        /// <param name="t">Type of the deskband.</param>
        [ComRegisterFunction]
        public static void Register(Type t)
        {
            var guid = t.GUID.ToString("B");
            try
            {
                var registryKey = Registry.ClassesRoot.CreateSubKey($@"CLSID\{guid}");
                registryKey.SetValue(null, GetToolbarName(t));

                var subKey = registryKey.CreateSubKey("Implemented Categories");
                subKey.CreateSubKey(ComponentCategoryManager.CATID_DESKBAND.ToString("B"));

                Console.WriteLine($"Succesfully registered deskband `{GetToolbarName(t)}` - GUID: {guid}");

                if (GetToolbarRequestToShow(t))
                {
                    Console.WriteLine("Request to show deskband.");

                    // https://www.pinvoke.net/default.aspx/Interfaces.ITrayDeskband
                    ITrayDeskband csdeskband = null;
                    try
                    {
                        var trayDeskbandType = Type.GetTypeFromCLSID(new Guid("E6442437-6C68-4f52-94DD-2CFED267EFB9"));
                        var deskbandGuid = t.GUID;
                        csdeskband = (ITrayDeskband)Activator.CreateInstance(trayDeskbandType);
                        if (csdeskband != null)
                        {
                            csdeskband.DeskBandRegistrationChanged();

                            if (csdeskband.IsDeskBandShown(ref deskbandGuid) == HRESULT.S_FALSE)
                            {
                                if (csdeskband.ShowDeskBand(ref deskbandGuid) != HRESULT.S_OK)
                                {
                                    Console.WriteLine("Error while trying to show deskband.");
                                }

                                if (csdeskband.DeskBandRegistrationChanged() == HRESULT.S_OK)
                                {
                                    Console.WriteLine($"The deskband was Succesfully shown with taskbar.{Environment.NewLine}You may see the alert notice box from explorer call.");
                                }
                            }
                        }
                    }
                    catch (Exception e)
                    {
                        Console.WriteLine($"Error while trying to show deskband: {e}");
                    }
                    finally
                    {
                        if (csdeskband != null && Marshal.IsComObject(csdeskband))
                        {
                            Marshal.ReleaseComObject(csdeskband);
                        }
                    }
                }
            }
            catch (Exception)
            {
                Console.Error.WriteLine($"Failed to register deskband `{GetToolbarName(t)}` - GUID: {guid}");
                throw;
            }
        }

        /// <summary>
        /// Unregister the deskband.
        /// </summary>
        /// <param name="t">Type of the deskband.</param>
        [ComUnregisterFunction]
        public static void Unregister(Type t)
        {
            var guid = t.GUID.ToString("B");
            try
            {
                Registry.ClassesRoot.OpenSubKey(@"CLSID", true)?.DeleteSubKeyTree(guid);

                Console.WriteLine($"Successfully unregistered deskband `{GetToolbarName(t)}` - GUID: {guid}");
            }
            catch (ArgumentException)
            {
                Console.Error.WriteLine($"Deskband `{GetToolbarName(t)}` is not registered");
            }
            catch (Exception)
            {
                Console.Error.WriteLine($"Failed to unregister deskband `{GetToolbarName(t)}` - GUID: {guid}");
                throw;
            }
        }

        /// <summary>
        /// Gets the name of the toolbar for the deskband.
        /// </summary>
        /// <param name="t">Type of the deskband.</param>
        /// <returns>The name of the toolbar.</returns>
        internal static string GetToolbarName(Type t)
        {
            return t.GetCustomAttribute<CSDeskBandRegistrationAttribute>(true)?.Name ?? t.Name;
        }

        /// <summary>
        /// Gets if the deskband should be shown after registration.
        /// </summary>
        /// <param name="t">Type of the deskband.</param>
        /// <returns>The value if it should be shown.</returns>
        internal static bool GetToolbarRequestToShow(Type t)
        {
            return t.GetCustomAttribute<CSDeskBandRegistrationAttribute>(true)?.ShowDeskBand ?? false;
        }
    }
}
namespace CSDeskBand
{
    /// <summary>
    /// The orientation of the taskbar.
    /// </summary>
    public enum TaskbarOrientation
    {
        /// <summary>
        /// Vertical if the taskbar is either on top or bottom.
        /// </summary>
        Vertical,

        /// <summary>
        /// Horizontal if the taskbar is either on the left or right.
        /// </summary>
        Horizontal,
    }

    /// <summary>
    /// The edge where the taskbar is located.
    /// </summary>
    public enum Edge : uint
    {
        /// <summary>
        /// Taskbar is on the left edge.
        /// </summary>
        Left,

        /// <summary>
        /// Taskbar is on the top edge.
        /// </summary>
        Top,

        /// <summary>
        /// Taskbar is on the right edge.
        /// </summary>
        Right,

        /// <summary>
        /// Taskbar is on the bottom edge.
        /// </summary>
        Bottom,
    }

    /// <summary>
    /// Provides information about the main taskbar.
    /// </summary>
    public sealed class TaskbarInfo
    {
        private TaskbarOrientation _orientation = TaskbarOrientation.Horizontal;
        private Edge _edge = Edge.Bottom;
        private DeskBandSize _size;

        /// <summary>
        /// Initializes a new instance of the <see cref="TaskbarInfo"/> class.
        /// </summary>
        internal TaskbarInfo()
        {
            UpdateInfo();
        }

        /// <summary>
        /// Occurs when the orientation of the main taskbar is changed.
        /// </summary>
        public event EventHandler<TaskbarOrientationChangedEventArgs> TaskbarOrientationChanged;

        /// <summary>
        /// Occurs when the edge of the main taskbar is changed.
        /// </summary>
        public event EventHandler<TaskbarEdgeChangedEventArgs> TaskbarEdgeChanged;

        /// <summary>
        /// Occurs when the size of the taskbar is changed.
        /// </summary>
        public event EventHandler<TaskbarSizeChangedEventArgs> TaskbarSizeChanged;

        /// <summary>
        /// Gets the current <see cref="TaskbarOrientation"/> of the main taskbar.
        /// </summary>
        /// <value>
        /// The current orientation.
        /// </value>
        public TaskbarOrientation Orientation
        {
            get => _orientation;
            private set
            {
                if (value == _orientation)
                {
                    return;
                }

                _orientation = value;
                TaskbarOrientationChanged?.Invoke(this, new TaskbarOrientationChangedEventArgs(value));
            }
        }

        /// <summary>
        /// Gets the current <see cref="CSDeskBand.Edge"/> of the main taskbar.
        /// </summary>
        /// <value>
        /// The current edge.
        /// </value>
        public Edge Edge
        {
            get => _edge;
            private set
            {
                if (value == _edge)
                {
                    return;
                }

                _edge = value;
                TaskbarEdgeChanged?.Invoke(this, new TaskbarEdgeChangedEventArgs(value));
            }
        }

        /// <summary>
        /// Gets the current <see cref="CSDeskBand.DeskBandSize"/> of the main taskbar.
        /// </summary>
        /// <value>
        /// The current size.
        /// </value>
        public DeskBandSize Size
        {
            get => _size;
            private set
            {
                if (value.Equals(_size))
                {
                    return;
                }

                _size = value;
                TaskbarSizeChanged?.Invoke(this, new TaskbarSizeChangedEventArgs(value));
            }
        }

        /// <summary>
        /// Get the latest taskbar information.
        /// </summary>
        internal void UpdateInfo()
        {
            var data = new APPBARDATA
            {
                hWnd = IntPtr.Zero,
                cbSize = Marshal.SizeOf<APPBARDATA>()
            };

            var res = Shell32.SHAppBarMessage(APPBARMESSAGE.ABM_GETTASKBARPOS, ref data);
            var rect = data.rc;
            Size = new DeskBandSize(rect.right - rect.left, rect.bottom - rect.top);
            Edge = (Edge)data.uEdge;
            Orientation = (Edge == Edge.Bottom || Edge == Edge.Top) ? TaskbarOrientation.Horizontal : TaskbarOrientation.Vertical;
        }
    }

    /// <summary>
    /// Provides data for a taskbar orientation change event.
    /// </summary>
    public sealed class TaskbarOrientationChangedEventArgs : EventArgs
    {
        /// <summary>
        /// Initializes a new instance of the <see cref="TaskbarOrientationChangedEventArgs"/> class
        /// with the new orientation.
        /// </summary>
        /// <param name="orientation">The new taskbar orientation.</param>
        public TaskbarOrientationChangedEventArgs(TaskbarOrientation orientation)
        {
            Orientation = orientation;
        }

        /// <summary>
        /// Gets the new orientation of the taskbar.
        /// </summary>
        public TaskbarOrientation Orientation { get; }
    }

        /// <summary>
    /// Provides data for a taskbar size change event.
    /// </summary>
    public sealed class TaskbarSizeChangedEventArgs : EventArgs
    {
        /// <summary>
        /// Initializes a new instance of the <see cref="TaskbarSizeChangedEventArgs"/> class
        /// with the new size of the taskbar.
        /// </summary>
        /// <param name="size">The new size of the taskbar.</param>
        public TaskbarSizeChangedEventArgs(DeskBandSize size)
        {
            Size = size;
        }

        /// <summary>
        /// Gets the new size of the taskbar.
        /// </summary>
        public DeskBandSize Size { get; }
    }

    /// <summary>
    /// Provides data for a taskbar edge change event.
    /// </summary>
    public sealed class TaskbarEdgeChangedEventArgs : EventArgs
    {
        /// <summary>
        /// Initializes a new instance of the <see cref="TaskbarEdgeChangedEventArgs"/> class
        /// with the new edge.
        /// </summary>
        /// <param name="edge">The new edge.</param>
        public TaskbarEdgeChangedEventArgs(Edge edge)
        {
            Edge = edge;
        }

        /// <summary>
        /// Gets the new edge location of the taskbar.
        /// </summary>
        public Edge Edge { get; }
    }
}
// <autogenerated/>
#region nativemethods
#pragma warning disable 1591
namespace CSDeskBand.Interop
{
    [ComImport]
    [InterfaceType(ComInterfaceType.InterfaceIsIUnknown)]
    [Guid("4CF504B0-DE96-11D0-8B3F-00A0C911E8E5")]
    internal interface IBandSite
    {
        [PreserveSig]
        int AddBand(ref object punk);

        [PreserveSig]
        int EnumBands(int uBand, out uint pdwBandID);

        [PreserveSig]
        int QueryBand(uint dwBandID, out IDeskBand ppstb, out BANDSITEINFO.BSSF pdwState, [MarshalAs(UnmanagedType.LPWStr)] out string pszName, int cchName);

        [PreserveSig]
        int SetBandState(uint dwBandID, BANDSITEINFO.BSIM dwMask, BANDSITEINFO.BSSF dwState);

        [PreserveSig]
        int RemoveBand(uint dwBandID);

        [PreserveSig]
        int GetBandObject(uint dwBandID, ref Guid riid, out IntPtr ppv);

        [PreserveSig]
        int SetBandSiteInfo([In] ref BANDSITEINFO pbsinfo);

        [PreserveSig]
        int GetBandSiteInfo([In, Out] ref BANDSITEINFO pbsinfo);
    }

    [ComImport]
    [InterfaceType(ComInterfaceType.InterfaceIsIUnknown)]
    [Guid("012DD920-7B26-11D0-8CA9-00A0C92DBFE8")]
    public interface IDockingWindow : IOleWindow
    {
        [PreserveSig]
        new int GetWindow(out IntPtr phwnd);

        [PreserveSig]
        new int ContextSensitiveHelp(bool fEnterMode);

        [PreserveSig]
        int ShowDW(bool fShow);

        [PreserveSig]
        int CloseDW(uint dwReserved);

        [PreserveSig]
        int ResizeBorderDW(RECT prcBorder, [MarshalAs(UnmanagedType.IUnknown)] IntPtr punkToolbarSite, bool fReserved);
    }

    [ComImport]
    [InterfaceType(ComInterfaceType.InterfaceIsIUnknown)]
    [Guid("EB0FE172-1A3A-11D0-89B3-00A0C90A90AC")]
    public interface IDeskBand : IDockingWindow
    {
        [PreserveSig]
        new int GetWindow(out IntPtr phwnd);

        [PreserveSig]
        new int ContextSensitiveHelp(bool fEnterMode);

        [PreserveSig]
        new int ShowDW(bool fShow);

        [PreserveSig]
        new int CloseDW(uint dwReserved);

        [PreserveSig]
        new int ResizeBorderDW(RECT prcBorder, [MarshalAs(UnmanagedType.IUnknown)] IntPtr punkToolbarSite, bool fReserved);

        [PreserveSig]
        int GetBandInfo(uint dwBandID, DESKBANDINFO.DBIF dwViewMode, ref DESKBANDINFO pdbi);
    }

    [ComImport]
    [InterfaceType(ComInterfaceType.InterfaceIsIUnknown)]
    [Guid("79D16DE4-ABEE-4021-8D9D-9169B261D657")]
    public interface IDeskBand2 : IDeskBand
    {
        [PreserveSig]
        new int GetWindow(out IntPtr phwnd);

        [PreserveSig]
        new int ContextSensitiveHelp(bool fEnterMode);

        [PreserveSig]
        new int ShowDW(bool fShow);

        [PreserveSig]
        new int CloseDW(uint dwReserved);

        [PreserveSig]
        new int ResizeBorderDW(RECT prcBorder, [MarshalAs(UnmanagedType.IUnknown)] IntPtr punkToolbarSite, bool fReserved);

        [PreserveSig]
        new int GetBandInfo(uint dwBandID, DESKBANDINFO.DBIF dwViewMode, ref DESKBANDINFO pdbi);

        [PreserveSig]
        int CanRenderComposited(out bool pfCanRenderComposited);

        [PreserveSig]
        int SetCompositionState(bool fCompositionEnabled);

        [PreserveSig]
        int GetCompositionState(out bool pfCompositionEnabled);
    }

    [ComImport]
    [InterfaceType(ComInterfaceType.InterfaceIsIUnknown)]
    [Guid("000214e4-0000-0000-c000-000000000046")]
    public interface IContextMenu
    {
        [PreserveSig]
        int QueryContextMenu(IntPtr hMenu, uint indexMenu, uint idCmdFirst, uint idCmdLast, QueryContextMenuFlags uFlags);

        [PreserveSig]
        int InvokeCommand(IntPtr pici);

        [PreserveSig]
        int GetCommandString(ref uint idcmd, uint uflags, ref uint pwReserved, [MarshalAs(UnmanagedType.LPTStr)] out string pcszName, uint cchMax);
    }

    [ComImport]
    [InterfaceType(ComInterfaceType.InterfaceIsIUnknown)]
    [Guid("000214f4-0000-0000-c000-000000000046")]
    public interface IContextMenu2 : IContextMenu
    {
        [PreserveSig]
        new int QueryContextMenu(IntPtr hMenu, uint indexMenu, uint idCmdFirst, uint idCmdLast, QueryContextMenuFlags uFlags);

        [PreserveSig]
        new int InvokeCommand(IntPtr pici);

        [PreserveSig]
        new int GetCommandString(ref uint idcmd, uint uflags, ref uint pwReserved, [MarshalAs(UnmanagedType.LPTStr)] out string pcszName, uint cchMax);

        [PreserveSig]
        int HandleMenuMsg(uint uMsg, IntPtr wParam, IntPtr lParam);
    }

    [ComImport]
    [InterfaceType(ComInterfaceType.InterfaceIsIUnknown)]
    [Guid("bcfce0a0-ec17-11d0-8d10-00a0c90f2719")]
    public interface IContextMenu3 : IContextMenu2
    {
        [PreserveSig]
        new int QueryContextMenu(IntPtr hMenu, uint indexMenu, uint idCmdFirst, uint idCmdLast, QueryContextMenuFlags uFlags);

        [PreserveSig]
        new int InvokeCommand(IntPtr pici);

        [PreserveSig]
        new int GetCommandString(ref uint idcmd, uint uflags, ref uint pwReserved, [MarshalAs(UnmanagedType.LPTStr)] out string pcszName, uint cchMax);

        [PreserveSig]
        new int HandleMenuMsg(uint uMsg, IntPtr wParam, IntPtr lParam);

        [PreserveSig]
        int HandleMenuMsg2(uint uMsg, IntPtr wParam, IntPtr lParam, out IntPtr plResult);
    }

    [ComImport]
    [InterfaceType(ComInterfaceType.InterfaceIsIUnknown)]
    [Guid("68284faa-6a48-11d0-8c78-00c04fd918b4")]
    public interface IInputObject
    {
        [PreserveSig]
        int UIActivateIO(int fActivate, ref MSG msg);

        [PreserveSig]
        int HasFocusIO();

        [PreserveSig]
        int TranslateAcceleratorIO(ref MSG msg);
    }

    //https://msdn.microsoft.com/en-us/library/windows/desktop/bb761789(v=vs.85).aspx
    [ComImport]
    [InterfaceType(ComInterfaceType.InterfaceIsIUnknown)]
    [Guid("F1DB8392-7331-11D0-8C99-00A0C92DBFE8")]
    public interface IInputObjectSite
    {
        [PreserveSig]
        int OnFocusChangeIS([MarshalAs(UnmanagedType.IUnknown)] object punkObj, Int32 fSetFocus);
    }

    //https://msdn.microsoft.com/en-us/library/windows/desktop/ms693765(v=vs.85).aspx
    [ComImport]
    [InterfaceType(ComInterfaceType.InterfaceIsIUnknown)]
    [Guid("FC4801A3-2BA9-11CF-A229-00AA003D7352")]
    public interface IObjectWithSite
    {
        [PreserveSig]
        int SetSite([MarshalAs(UnmanagedType.IUnknown)] object pUnkSite);

        [PreserveSig]
        int GetSite(ref Guid riid, [MarshalAs(UnmanagedType.IUnknown)] out IntPtr ppvSite);
    }

    //https://msdn.microsoft.com/en-us/library/windows/desktop/ms683797(v=vs.85).aspx
    [ComImport]
    [InterfaceType(ComInterfaceType.InterfaceIsIUnknown)]
    [Guid("b722bccb-4e68-101b-a2bc-00aa00404770")]
    internal interface IOleCommandTarget
    {
        [PreserveSig]
        void QueryStatus(ref Guid pguidCmdGroup, uint cCmds, [MarshalAs(UnmanagedType.LPArray), In, Out] OLECMD[] prgCmds, [In, Out] ref OLECMDTEXT pCmdText);

        [PreserveSig]
        int Exec(ref Guid pguidCmdGroup, uint nCmdID, uint nCmdExecOpt, IntPtr pvaIn, [In, Out] IntPtr pvaOut);
    }

    //https://msdn.microsoft.com/en-us/library/windows/desktop/ms680102(v=vs.85).aspx
    [ComImport]
    [InterfaceType(ComInterfaceType.InterfaceIsIUnknown)]
    [Guid("00000114-0000-0000-C000-000000000046")]
    public interface IOleWindow
    {
        [PreserveSig]
        int GetWindow(out IntPtr phwnd);

        [PreserveSig]
        int ContextSensitiveHelp(bool fEnterMode);
    }

    [ComImport]
    [InterfaceType(ComInterfaceType.InterfaceIsIUnknown)]
    [Guid("0000010c-0000-0000-C000-000000000046")]
    public interface IPersist
    {
        [PreserveSig]
        int GetClassID(out Guid pClassID);
    }

    [ComImport]
    [InterfaceType(ComInterfaceType.InterfaceIsIUnknown)]
    [Guid("00000109-0000-0000-C000-000000000046")]
    public interface IPersistStream : IPersist
    {
        [PreserveSig]
        new int GetClassID(out Guid pClassID);

        [PreserveSig]
        int GetSizeMax(out ulong pcbSize);

        [PreserveSig]
        int IsDirty();

        [PreserveSig]
        int Load([In, MarshalAs(UnmanagedType.Interface)] object pStm);

        [PreserveSig]
        int Save([In, MarshalAs(UnmanagedType.Interface)] IntPtr pStm, bool fClearDirty);
    }

    [ComImport, Guid("6D67E846-5B9C-4db8-9CBC-DDE12F4254F1"), InterfaceType(ComInterfaceType.InterfaceIsIUnknown)]
    public interface ITrayDeskband
    {
        [PreserveSig]
        int ShowDeskBand([In, MarshalAs(UnmanagedType.Struct)] ref Guid clsid);
        [PreserveSig]
        int HideDeskBand([In, MarshalAs(UnmanagedType.Struct)] ref Guid clsid);
        [PreserveSig]
        int IsDeskBandShown([In, MarshalAs(UnmanagedType.Struct)] ref Guid clsid);
        [PreserveSig]
        int DeskBandRegistrationChanged();
    }

    internal class User32
    {
        [DllImport("user32.dll", SetLastError = true)]
        public static extern int SetParent(IntPtr hWndChild, IntPtr hWndNewParent);

        [DllImport("user32.dll")]
        public static extern bool InsertMenuItem(IntPtr hMenu, uint uItem, bool fByPosition, ref MENUITEMINFO lpmii);

        [DllImport("user32.dll")]
        public static extern IntPtr CreateMenu();

        [DllImport("user32.dll")]
        public static extern bool DestroyMenu(IntPtr hMenu);

        [DllImport("user32.dll")]
        public static extern IntPtr CreatePopupMenu();

        [DllImport("user32.dll")]
        public static extern bool TranslateMessage([In] ref MSG lpMsg);

        [DllImport("user32.dll")]
        public static extern IntPtr DispatchMessage([In] ref MSG lpmsg);

        public static int HiWord(int val)
        {
            return Convert.ToInt32(BitConverter.ToInt16(BitConverter.GetBytes(val), 2));
        }

        public static int LoWord(int val)
        {
            return Convert.ToInt32(BitConverter.ToInt16(BitConverter.GetBytes(val), 0));
        }
    }

    internal class Shell32
    {
        [DllImport("shell32.dll")]
        public static extern IntPtr SHAppBarMessage(APPBARMESSAGE dwMessage, [In] ref APPBARDATA pData);
    }

    internal enum tagDESKBANDCID
    {
        DBID_BANDINFOCHANGED = 0,
        DBID_SHOWONLY = 1,
        DBID_MAXIMIZEBAND = 2,
        DBID_PUSHCHEVRON = 3
    }

    [StructLayout(LayoutKind.Sequential)]
    public struct RECT
    {
        public int left;
        public int top;
        public int right;
        public int bottom;

        public RECT(int left, int top, int right, int bottom)
        {
            this.left = left;
            this.top = top;
            this.right = right;
            this.bottom = bottom;
        }
    }

    [Flags]
    public enum QueryContextMenuFlags : uint
    {
        CMF_NORMAL = 0x00000000,
        CMF_DEFAULTONLY = 0x00000001,
        CMF_VERBSONLY = 0x00000002,
        CMF_EXPLORE = 0x00000004,
        CMF_NOVERBS = 0x00000008,
        CMF_CANRENAME = 0x00000010,
        CMF_NODEFAULT = 0x00000020,
        CMF_ITEMMENU = 0x00000080,
        CMF_EXTENDEDVERBS = 0x00000100,
        CMF_DISABLEDVERBS = 0x00000200,
        CMF_ASYNCVERBSTATE = 0x00000400,
        CMF_OPTIMIZEFORINVOKE = 0x00000800,
        CMF_SYNCCASCADEMENU = 0x00001000,
        CMF_DONOTPICKDEFAULT = 0x00002000,
        CMF_RESERVED = 0xffff0000,
    }

    [StructLayout(LayoutKind.Sequential, CharSet = CharSet.Unicode)]
    public struct POINT
    {
        public int X;
        public int Y;
    }

    [StructLayout(LayoutKind.Sequential)]
    internal struct OLECMDTEXT
    {
        public uint cmdtextf;
        public uint cwActual;
        public uint cwBuf;
        [MarshalAs(UnmanagedType.ByValTStr, SizeConst = 1)]
        public string rgwz;
    }

    [StructLayout(LayoutKind.Sequential)]
    internal struct OLECMD
    {
        public uint cmdID;
        public uint cmdf;
    }

    [StructLayout(LayoutKind.Sequential)]
    public struct MSG
    {
        public IntPtr hwnd;
        public uint message;
        public uint wParam;
        public int lParam;
        public uint time;
        public POINT pt;
    }

    [StructLayout(LayoutKind.Sequential)]
    internal struct MENUITEMINFO
    {
        public int cbSize;
        public MIIM fMask;
        public MFT fType;
        public MFS fState;
        public uint wID;
        public IntPtr hSubMenu;
        public IntPtr hbmpChecked;
        public IntPtr hbmpUnchecked;
        public IntPtr dwItemData;
        [MarshalAs(UnmanagedType.LPStr)] public string dwTypeData;
        public uint cch;
        public IntPtr hbmpItem;

        [Flags]
        public enum MIIM : uint
        {
            MIIM_BITMAP = 0x00000080,
            MIIM_CHECKMARKS = 0x00000008,
            MIIM_DATA = 0x00000020,
            MIIM_FTYPE = 0x00000100,
            MIIM_ID = 0x00000002,
            MIIM_STATE = 0x00000001,
            MIIM_STRING = 0x00000040,
            MIIM_SUBMENU = 0x00000004,
            MIIM_TYPE = 0x00000010
        }

        [Flags]
        public enum MFT : uint
        {
            MFT_BITMAP = 0x00000004,
            MFT_MENUBARBREAK = 0x00000020,
            MFT_MENUBREAK = 0x00000040,
            MFT_OWNERDRAW = 0x00000100,
            MFT_RADIOCHECK = 0x00000200,
            MFT_RIGHTJUSTIFY = 0x00004000,
            MFT_RIGHTORDER = 0x00002000,
            MFT_SEPARATOR = 0x00000800,
            MFT_STRING = 0x00000000,
        }

        [Flags]
        public enum MFS : uint
        {
            MFS_CHECKED = 0x00000008,
            MFS_DEFAULT = 0x00001000,
            MFS_DISABLED = 0x00000003,
            MFS_ENABLED = 0x00000000,
            MFS_GRAYED = 0x00000003,
            MFS_HILITE = 0x00000080,
            MFS_UNCHECKED = 0x00000000,
            MFS_UNHILITE = 0x00000000,
        }
    }

    internal class HRESULT
    {
        public static readonly int S_OK = 0;
        public static readonly int S_FALSE = 1;
        public static readonly int E_NOTIMPL = unchecked((int)0x80004001);
        public static readonly int E_FAIL = unchecked((int)0x80004005);

        public static int MakeHResult(uint sev, uint facility, uint errorNo)
        {
            var result = sev << 31 | facility << 16 | errorNo;
            return unchecked((int)result);
        }
    }

    [StructLayout(LayoutKind.Sequential, CharSet = CharSet.Unicode)]
    public struct DESKBANDINFO
    {
        public DBIM dwMask;
        public POINT ptMinSize;
        public POINT ptMaxSize;
        public POINT ptIntegral;
        public POINT ptActual;
        [MarshalAs(UnmanagedType.ByValTStr, SizeConst = 255)] public String wszTitle;
        public DBIMF dwModeFlags;
        public COLORREF crBkgnd;

        [Flags]
        public enum DBIF : uint
        {
            DBIF_VIEWMODE_NORMAL = 0x0000,
            DBIF_VIEWMODE_VERTICAL = 0x0001,
            DBIF_VIEWMODE_FLOATING = 0x0002,
            DBIF_VIEWMODE_TRANSPARENT = 0x0004
        }

        [Flags]
        public enum DBIM : uint
        {
            DBIM_MINSIZE = 0x0001,
            DBIM_MAXSIZE = 0x0002,
            DBIM_INTEGRAL = 0x0004,
            DBIM_ACTUAL = 0x0008,
            DBIM_TITLE = 0x0010,
            DBIM_MODEFLAGS = 0x0020,
            DBIM_BKCOLOR = 0x0040
        }

        [Flags]
        public enum DBIMF : uint
        {
            DBIMF_NORMAL = 0x0000,
            DBIMF_FIXED = 0x0001,
            DBIMF_FIXEDBMP = 0x0004,
            DBIMF_VARIABLEHEIGHT = 0x0008,
            DBIMF_UNDELETEABLE = 0x0010,
            DBIMF_DEBOSSED = 0x0020,
            DBIMF_BKCOLOR = 0x0040,
            DBIMF_USECHEVRON = 0x0080,
            DBIMF_BREAK = 0x0100,
            DBIMF_ADDTOFRONT = 0x0200,
            DBIMF_TOPALIGN = 0x0400,
            DBIMF_NOGRIPPER = 0x0800,
            DBIMF_ALWAYSGRIPPER = 0x1000,
            DBIMF_NOMARGINS = 0x2000
        }
    }

    [StructLayout(LayoutKind.Sequential)]
    public struct COLORREF
    {
        public byte R;
        public byte G;
        public byte B;
    }

    [StructLayout(LayoutKind.Sequential)]
    internal struct CMINVOKECOMMANDINFOEX
    {
        public uint cbSize;
        public CMIC fMask;
        public IntPtr hwnd;
        public IntPtr lpVerb;
        [MarshalAs(UnmanagedType.LPStr)] public string lpParameters;
        [MarshalAs(UnmanagedType.LPStr)] public string lpDirectory;
        public int nShow;
        public uint dwHotKey;
        public IntPtr hIcon;
        [MarshalAs(UnmanagedType.LPStr)] public string lpTitle;
        public IntPtr lpVerbW;
        [MarshalAs(UnmanagedType.LPWStr)] public string lpParametersW;
        [MarshalAs(UnmanagedType.LPWStr)] public string lpDirectoryW;
        [MarshalAs(UnmanagedType.LPWStr)] public string lpTitleW;
        public POINT ptInvoke;

        [Flags]
        public enum CMIC
        {
            CMIC_MASK_HOTKEY = 0x00000020,
            CMIC_MASK_ICON = 0x00000010,
            CMIC_MASK_FLAG_NO_UI = 0x00000400,
            CMIC_MASK_UNICODE = 0x00004000,
            CMIC_MASK_NO_CONSOLE = 0x00008000,
            CMIC_MASK_ASYNCOK = 0x00100000,
            CMIC_MASK_NOASYNC = 0x00000100,
            CMIC_MASK_SHIFT_DOWN = 0x10000000,
            CMIC_MASK_PTINVOKE = 0x20000000,
            CMIC_MASK_CONTROL_DOWN = 0x40000000,
            CMIC_MASK_FLAG_LOG_USAGE = 0x04000000,
            CMIC_MASK_NOZONECHECKS = 0x00800000,
        }
    }

    [StructLayout(LayoutKind.Sequential)]
    public class CMINVOKECOMMANDINFO
    {
        public int cbSize;
        public CMIC fMask;
        public IntPtr hwnd;
        public IntPtr lpVerb;
        [MarshalAs(UnmanagedType.LPStr)] public string lpParameters;
        [MarshalAs(UnmanagedType.LPStr)] public string lpDirectory;
        public int nShow;
        public int dwHotKey;
        public IntPtr hIcon;

        [Flags]
        public enum CMIC
        {
            CMIC_MASK_HOTKEY = 0x00000020,
            CMIC_MASK_ICON = 0x00000010,
            CMIC_MASK_FLAG_NO_UI = 0x00000400,
            CMIC_MASK_NO_CONSOLE = 0x00008000,
            CMIC_MASK_ASYNCOK = 0x00100000,
            CMIC_MASK_NOASYNC = 0x00000100,
            CMIC_MASK_SHIFT_DOWN = 0x10000000,
            CMIC_MASK_CONTROL_DOWN = 0x40000000,
            CMIC_MASK_FLAG_LOG_USAGE = 0x04000000,
            CMIC_MASK_NOZONECHECKS = 0x00800000,
        }
    }

    [StructLayout(LayoutKind.Sequential)]
    internal class CATEGORYINFO
    {
        public Guid catid;
        public uint lcidl;
        [MarshalAs(UnmanagedType.ByValTStr, SizeConst = 128)] public string szDescription;
    }

    [StructLayout(LayoutKind.Sequential)]
    struct BANDSITEINFO
    {
        public BSIM dwMask;
        public BSSF dwState;
        public BSIS dwStyle;

        [Flags]
        public enum BSIM : uint
        {
            BSIM_STATE = 0x00000001,
            BSIM_STYLE = 0x00000002,
        }

        [Flags]
        public enum BSSF : uint
        {
            BSSF_VISIBLE = 0x00000001,
            BSSF_NOTITLE = 0x00000002,
            BSSF_UNDELETEABLE = 0x00001000,
        }

        [Flags]
        public enum BSIS : uint
        {
            BSIS_AUTOGRIPPER = 0x00000000,
            BSIS_NOGRIPPER = 0x00000001,
            BSIS_ALWAYSGRIPPER = 0x00000002,
            BSIS_LEFTALIGN = 0x00000004,
            BSIS_SINGLECLICK = 0x00000008,
            BSIS_NOCONTEXTMENU = 0x00000010,
            BSIS_NODROPTARGET = 0x00000020,
            BSIS_NOCAPTION = 0x00000040,
            BSIS_PREFERNOLINEBREAK = 0x00000080,
            BSIS_LOCKED = 0x00000100,
            BSIS_PRESERVEORDERDURINGLAYOUT = 0x00000200,
            BSIS_FIXEDORDER = 0x00000400,
        }
    }

    internal enum APPBARMESSAGE : uint
    {
        ABM_NEW = 0x00000000,
        ABM_REMOVE = 0x00000001,
        ABM_QUERYPOS = 0x00000002,
        ABM_SETPOS = 0x00000003,
        ABM_GETSTATE = 0x00000004,
        ABM_GETTASKBARPOS = 0x00000005,
        ABM_ACTIVATE = 0x00000006,
        ABM_GETAUTOHIDEBAR = 0x00000007,
        ABM_SETAUTOHIDEBAR = 0x00000008,
        ABM_WINDOWPOSCHANGED = 0x00000009,
        ABM_SETSTATE = 0x0000000A,
        ABM_GETAUTOHIDEBAREX = 0x0000000B,
        ABM_SETAUTOHIDEBAREX = 0x0000000C,
    }

    [StructLayout(LayoutKind.Sequential)]
    internal struct APPBARDATA
    {
        public int cbSize;
        public IntPtr hWnd;
        public uint uCallbackMessage;
        public uint uEdge;
        public RECT rc;
        public int lParam;
    }

    internal class ComponentCategoryManager
    {
        public static readonly Guid CATID_DESKBAND = new Guid("00021492-0000-0000-C000-000000000046");

        private static readonly Guid _componentCategoryManager = new Guid("0002e005-0000-0000-c000-000000000046");
        private static readonly ICatRegister _catRegister;
        private Guid _classId;

        static ComponentCategoryManager()
        {
            _catRegister = Activator.CreateInstance(Type.GetTypeFromCLSID(_componentCategoryManager, true)) as ICatRegister;
        }

        private ComponentCategoryManager(Guid classId)
        {
            _classId = classId;
        }

        public static ComponentCategoryManager For(Guid classId)
        {
            return new ComponentCategoryManager(classId);
        }

        public void RegisterCategories(Guid[] categoryIds)
        {
            _catRegister.RegisterClassImplCategories(ref _classId, (uint)categoryIds.Length, categoryIds);
        }

        public void UnRegisterCategories(Guid[] categoryIds)
        {
            _catRegister.UnRegisterClassImplCategories(ref _classId, (uint)categoryIds.Length, categoryIds);
        }
    }

    [ComImport]
    [InterfaceType(ComInterfaceType.InterfaceIsIUnknown)]
    [Guid("0002E012-0000-0000-C000-000000000046")]
    internal interface ICatRegister
    {
        [PreserveSig]
        void RegisterCategories(uint cCategories, [MarshalAs(UnmanagedType.LPArray)] CATEGORYINFO[] rgCategoryInfo);

        [PreserveSig]
        void RegisterClassImplCategories([In] ref Guid rclsid, uint cCategories, [MarshalAs(UnmanagedType.LPArray)] Guid[] rgcatid);

        [PreserveSig]
        void RegisterClassReqCategories([In] ref Guid rclsid, uint cCategories, [MarshalAs(UnmanagedType.LPArray)] Guid[] rgcatid);

        [PreserveSig]
        void UnRegisterCategories(uint cCategories, [MarshalAs(UnmanagedType.LPArray)] Guid[] rgcatid);

        [PreserveSig]
        void UnRegisterClassImplCategories([In] ref Guid rclsid, uint cCategories, [MarshalAs(UnmanagedType.LPArray)] Guid[] rgcatid);

        [PreserveSig]
        void UnRegisterClassReqCategories([In] ref Guid rclsid, uint cCategories, [MarshalAs(UnmanagedType.LPArray)] Guid[] rgcatid);
    }

    [Flags]
    internal enum WindowStyles : uint
    {
        WS_BORDER = 0x800000,
        WS_CAPTION = 0xc00000,
        WS_CHILD = 0x40000000,
        WS_CLIPCHILDREN = 0x2000000,
        WS_CLIPSIBLINGS = 0x4000000,
        WS_DISABLED = 0x8000000,
        WS_DLGFRAME = 0x400000,
        WS_GROUP = 0x20000,
        WS_HSCROLL = 0x100000,
        WS_MAXIMIZE = 0x1000000,
        WS_MAXIMIZEBOX = 0x10000,
        WS_MINIMIZE = 0x20000000,
        WS_MINIMIZEBOX = 0x20000,
        WS_OVERLAPPED = 0x0,
        WS_OVERLAPPEDWINDOW = WS_OVERLAPPED | WS_CAPTION | WS_SYSMENU | WS_SIZEFRAME | WS_MINIMIZEBOX | WS_MAXIMIZEBOX,
        WS_POPUP = 0x80000000u,
        WS_POPUPWINDOW = WS_POPUP | WS_BORDER | WS_SYSMENU,
        WS_SIZEFRAME = 0x40000,
        WS_SYSMENU = 0x80000,
        WS_TABSTOP = 0x10000,
        WS_VISIBLE = 0x10000000,
        WS_VSCROLL = 0x200000
    }

    internal enum WindowMessages
    {
        WM_NCHITTEST = 0x0084,
    }

    internal enum HitTestMessageResults
    {
        HTCLIENT = 1,
        HTTRANSPARENT = -1,
    }
}
#endregion
namespace CSDeskBand.ContextMenu
{
    /// <summary>
    /// Base class for deskband menu items.
    /// </summary>
    public abstract class DeskBandMenuItem
    {
        /// <summary>
        /// Add this item to a menu.
        /// </summary>
        /// <param name="menu">The menu to add items to.</param>
        /// <param name="itemPosition">The position of the item to insert into the menu.</param>
        /// <param name="itemId">Unique id of the menu item. Should be incremented if used.</param>
        /// <param name="callbacks">Dictionary of callbacks assigned to a <paramref name="itemId"/>.</param>
        internal abstract void AddToMenu(IntPtr menu, uint itemPosition, ref uint itemId, Dictionary<uint, DeskBandMenuAction> callbacks);
    }

    /// <summary>
    /// A context menu seperator.
    /// </summary>
    internal sealed class DeskBandMenuSeparator : DeskBandMenuItem
    {
        private MENUITEMINFO _menuiteminfo;

        /// <inheritdoc/>
        internal override void AddToMenu(IntPtr menu, uint itemPosition, ref uint itemId, Dictionary<uint, DeskBandMenuAction> callbacks)
        {
            _menuiteminfo = new MENUITEMINFO
            {
                cbSize = Marshal.SizeOf<MENUITEMINFO>(),
                fMask = MENUITEMINFO.MIIM.MIIM_TYPE,
                fType = MENUITEMINFO.MFT.MFT_SEPARATOR,
            };

            User32.InsertMenuItem(menu, itemPosition, true, ref _menuiteminfo);
        }
    }

    /// <summary>
    /// A context menu item that can be clicked.
    /// </summary>
    internal sealed class DeskBandMenuAction : DeskBandMenuItem
    {
        private MENUITEMINFO _menuiteminfo;

        /// <summary>
        /// Initializes a new instance of the <see cref="DeskBandMenuAction"/> class
        /// with its display text.
        /// </summary>
        /// <param name="text">The text that is shown for this item in the context menu.</param>
        public DeskBandMenuAction(string text)
        {
            Text = text;
        }

        /// <summary>
        /// Occurs when the menu item has been clicked.
        /// </summary>
        public event EventHandler Clicked;

        /// <summary>
        /// Gets or sets a value indicating whether there is a checkmark next to the menu item.
        /// </summary>
        /// <value>
        /// <see langword="true"/> if the menu should have a checkmark. <see langword="false"/> if there should be no checkmark.
        /// The default value is <see langword="false"/>.
        /// </value>
        public bool Checked { get; set; } = false;

        /// <summary>
        /// Gets or sets a value indicating whether the menu item is enabled.
        /// </summary>
        /// <value>
        /// <see langword="true"/> if the menu item can be interacted with. <see langword="false"/> to disable interactions.
        /// The default value is <see langword="true"/>.
        /// </value>
        public bool Enabled { get; set; } = true;

        /// <summary>
        /// Gets or sets the text shown for this item in the context menu.
        /// </summary>
        /// <value>
        /// The text that will be displayed for this item in the context menu.
        /// </value>
        public string Text { get; set; }

        /// <summary>
        /// Performs the click action for this item.
        /// </summary>
        internal void DoAction()
        {
            Clicked?.Invoke(this, EventArgs.Empty);
        }

        /// <inheritdoc/>
        internal override void AddToMenu(IntPtr menu, uint itemPosition, ref uint itemId, Dictionary<uint, DeskBandMenuAction> callbacks)
        {
            _menuiteminfo = new MENUITEMINFO
            {
                cbSize = Marshal.SizeOf<MENUITEMINFO>(),
                fMask = MENUITEMINFO.MIIM.MIIM_TYPE | MENUITEMINFO.MIIM.MIIM_STATE | MENUITEMINFO.MIIM.MIIM_ID,
                fType = MENUITEMINFO.MFT.MFT_STRING,
                dwTypeData = Text,
                cch = (uint)Text.Length,
                wID = itemId++,
            };

            _menuiteminfo.fState |= Enabled ? MENUITEMINFO.MFS.MFS_ENABLED : MENUITEMINFO.MFS.MFS_DISABLED;
            _menuiteminfo.fState |= Checked ? MENUITEMINFO.MFS.MFS_CHECKED : MENUITEMINFO.MFS.MFS_UNCHECKED;

            callbacks[_menuiteminfo.wID] = this;

            User32.InsertMenuItem(menu, itemPosition, true, ref _menuiteminfo);
        }
    }

    /// <summary>
    /// A sub menu item that can contain other <see cref="DeskBandMenuItem"/>.
    /// </summary>
    internal sealed class DeskBandMenu : DeskBandMenuItem
    {
        private IntPtr _menu;
        private MENUITEMINFO _menuiteminfo;

        /// <summary>
        /// Initializes a new instance of the <see cref="DeskBandMenu"/> class
        /// with the display text.
        /// </summary>
        /// <param name="text">The text displayed for this item in the context menu.</param>
        public DeskBandMenu(string text)
            : this(text, null) { }

        /// <summary>
        /// Initializes a new instance of the <see cref="DeskBandMenu"/> class
        /// with a display text and a list of submenu items.
        /// </summary>
        /// <param name="text">The text displayed for this item in the context menu.</param>
        /// <param name="items">A <see cref="IEnumerable{T}"/> of <see cref="DeskBandMenuItem"/> that will appear in this submenu.</param>
        public DeskBandMenu(string text, IEnumerable<DeskBandMenuItem> items)
        {
            Text = text;
            if (items != null)
            {
                foreach (var item in items)
                {
                    Items.Add(item);
                }
            }
        }

        /// <summary>
        /// Finalizes an instance of the <see cref="DeskBandMenu"/> class.
        /// Frees up resoruces associated with the menu.
        /// </summary>
        ~DeskBandMenu()
        {
            ClearMenu();
        }

        /// <summary>
        /// Gets the collection of <see cref="DeskBandMenuItem"/> in the menu.
        /// </summary>
        public ICollection<DeskBandMenuItem> Items { get; } = new List<DeskBandMenuItem>();

        /// <summary>
        /// Gets or sets a value indicating whether the menu item is enabled.
        /// </summary>
        /// <value>
        /// <see langword="true"/> if the menu item can be interacted with. <see langword="false"/> to disable interactions.
        /// The default value is <see langword="true"/>;
        /// </value>
        public bool Enabled { get; set; } = true;

        /// <summary>
        /// Gets or sets the text shown in the menu item.
        /// </summary>
        /// <value>
        /// The text that will be displayed for this menu item.
        /// </value>
        public string Text { get; set; }

        /// <inheritdoc/>
        internal override void AddToMenu(IntPtr menu, uint itemPosition, ref uint itemId, Dictionary<uint, DeskBandMenuAction> callbacks)
        {
            ClearMenu();

            _menu = User32.CreatePopupMenu();
            uint index = 0;
            foreach (var item in Items)
            {
                item.AddToMenu(_menu, index++, ref itemId, callbacks);
            }

            _menuiteminfo = new MENUITEMINFO
            {
                cbSize = Marshal.SizeOf<MENUITEMINFO>(),
                fMask = MENUITEMINFO.MIIM.MIIM_SUBMENU | MENUITEMINFO.MIIM.MIIM_STRING | MENUITEMINFO.MIIM.MIIM_STATE,
                fType = MENUITEMINFO.MFT.MFT_MENUBREAK | MENUITEMINFO.MFT.MFT_STRING,
                fState = Enabled ? MENUITEMINFO.MFS.MFS_ENABLED : MENUITEMINFO.MFS.MFS_DISABLED,
                dwTypeData = Text,
                cch = (uint)Text.Length,
                hSubMenu = _menu,
            };

            User32.InsertMenuItem(menu, itemPosition, true, ref _menuiteminfo);
        }

        private void ClearMenu()
        {
            if (_menu != IntPtr.Zero)
            {
                User32.DestroyMenu(_menu);
            }
        }
    }
}
#pragma warning restore 1591
