[![crates.io](https://img.shields.io/crates/v/spring-sea-orm.svg)](https://crates.io/crates/spring-sea-orm)
[![Documentation](https://docs.rs/spring-sea-orm/badge.svg)](https://docs.rs/spring-sea-orm)

## Dependencies

```toml
spring-sea-orm = { version = "<version>", features = ["postgres"] }
sea-orm = { version = "1.0" }          # Mainly to adapt to the entity code generated by sea-orm-cli
```

You can replace `postgres`, `mysql`, `sqlite`feature to select the appropriate database driver.

optional features: `with-web`.

## Configuration items

```toml
[sea-orm]
uri = "postgres://root:123456@localhost:5432/pg_db" # Database address
min_connections = 1        # Minimum number of connections in the connection pool, the default value is 1
max_connections = 10       # Maximum number of connections in the connection pool, the default value is 10
acquire_timeout = 30000    # Connection timeout, in milliseconds, default 30s
idle_timeout = 600000      # Connection idle time, in milliseconds, default 10min
connect_timeout = 1800000  # Maximum connection survival time, in milliseconds, default 30min
enable_logging = true      # Print sql log
```

## Components

After configuring the above configuration items, the plugin will automatically register a [`DbConn`](https://docs.rs/spring-sea-orm/latest/spring_sea_orm/type.DbConn.html) connection pool object. This object is an alias of [`sea_orm::DbConn`](https://docs.rs/sea-orm/1.0.0/sea_orm/type.DbConn.html).

```rust
pub type DbConn = sea_orm::DbConn;
```

## Model generation

[sea-orm-cli](https://www.sea-ql.org/SeaORM/docs/generate-entity/sea-orm-cli/) provides a great model code generation function. You only need to define the table structure in the database, and after a simple configuration, the model code corresponding to the database structure can be generated, which can save a lot of code writing work.

## Extract the Component registered by the plugin

The `SeaOrmPlugin` plugin automatically registers a connection pool component for us. We can use `Component` to extract this connection pool from AppState. [`Component`](https://docs.rs/spring-web/latest/spring_web/extractor/struct.Component.html) is an axum [extractor](https://docs.rs/axum/latest/axum/extract/index.html).

```rust
use spring_sqlx::{sqlx::{self, Row}, ConnectPool};
use spring_web::get;
use spring_web::extractor::Component;
use spring_web::error::Result;
use anyhow::Context;

#[get("/:id")]
async fn get_todo_list(
    Component(db): Component<DbConn>,
    Path(id): Path<i32>
) -> Result<String> {
    let rows = TodoItem::find()
        .filter(todo_item::Column::ListId.eq(id))
        .all(&db)
        .await
        .context("query todo list failed")?;
    Ok(Json(rows))
}
```

## Pagination support

`spring-sea-orm` extends [SeaOrm's Select](https://docs.rs/sea-orm/latest/sea_orm/query/struct.Select.html) with the [PaginationExt feature](https://docs.rs/spring-sea-orm/latest/spring_sea_orm/pagination/trait.PaginationExt.html).

In addition, web pagination parameter parsing is also provided. Just add the `with-web` function to the dependency.

```toml
spring-sea-orm = { version = "<version>", features = ["postgres", "with-web"] }
```

The configuration is as follows:

```toml
# sea-orm-web configuration
[sea-orm-web]
one_indexed = false     # 1-based index, closed by default
max_page_size = 2000    # Maximum supported page size, to avoid OOM caused by server attacks, default value 2000
default_page_size = 20  # Default page size, 20
```

Use as follows:

```rust
#[get("/")]
async fn get_todo_list(
    Component(db): Component<DbConn>,
    Query(query): Query<TodoListQuery>,
    pagination: Pagination,
) -> Result<impl IntoResponse> {
    let rows = TodoList::find()
        .filter(query)
        .page(&db, pagination)
        .await
        .context("query todo list failed")?;
    Ok(Json(rows))
}
```

For the complete code, please refer to [`sea-orm-example`](https://github.com/spring-rs/spring-rs/tree/master/examples/sea-orm-example)