# frozen_string_literal: true

module Spree::Image::PaperclipAttachment
  extend ActiveSupport::Concern

  included do
    validate :no_attachment_errors

    has_attached_file :attachment,
                      styles: Spree::Config.product_image_styles,
                      default_style: Spree::Config.product_image_style_default,
                      default_url: 'noimage/:style.png',
                      url: '/spree/products/:id/:style/:basename.:extension',
                      path: ':rails_root/public/spree/products/:id/:style/:basename.:extension',
                      convert_options: { all: '-strip -auto-orient -colorspace sRGB' }
    validates_attachment :attachment,
      presence: true,
      content_type: { content_type: Spree::Config.allowed_image_mime_types }

    # save the w,h of the original image (from which others can be calculated)
    # we need to look at the write-queue for images which have not been saved yet
    after_post_process :find_dimensions, if: :valid?
  end

  def url(size)
    attachment.url(size)
  end

  def filename
    attachment_file_name
  end

  def attachment_present?
    attachment.present?
  end

  def find_dimensions
    temporary = attachment.queued_for_write[:original]
    filename = temporary.path unless temporary.nil?
    filename = attachment.path if filename.blank?
    geometry = Paperclip::Geometry.from_file(filename)
    self.attachment_width  = geometry.width
    self.attachment_height = geometry.height
  end

  # if there are errors from the plugin, then add a more meaningful message
  def no_attachment_errors
    unless attachment.errors.empty?
      # uncomment this to get rid of the less-than-useful interim messages
      # errors.clear
      errors.add :attachment, "Paperclip returned errors for file '#{attachment_file_name}' - check ImageMagick installation or image source file."
      false
    end
  end
end
