# frozen_string_literal: true

require "spec_helper"
require "solidus_admin/configuration"

RSpec.describe SolidusAdmin::Configuration do
  describe "#menu_items" do
    it "returns an array of hashes" do
      expect(described_class.new.menu_items).to be_an(Array)
      expect(described_class.new.menu_items).not_to be_empty
      expect(described_class.new.menu_items.first).to be_a(Hash)
    end

    it "returns the same instance every time" do
      config = described_class.new

      expect(config.menu_items).to be(config.menu_items)
    end
  end

  describe "#components" do
    it "returns a component class given a key" do
      config = described_class.new
      expect(config.components["ui/button"]).to eq(SolidusAdmin::UI::Button::Component)
    end

    it "suggests the correct name if a typo is found" do
      config = described_class.new
      expect{ config.components["ui/buton"] }.to raise_error("Unknown component ui/buton\nDid you mean?  ui/button")
    end
  end

  describe "#storefront_product_path" do
    it "returns the path generated by the `storefront_product_path_proc` preference" do
      product = build_stubbed(:product, slug: 'foo-bar-123')
      config = described_class.new
      config.storefront_product_path_proc = -> { "/foo/bar/#{_1.slug}" }

      expect(config.storefront_product_path(product)).to eq("/foo/bar/foo-bar-123")
    end
  end

  describe "#import_menu_items_from_backend!" do
    it "imports the menu items from the backend" do
      allow(Spree::Backend::Config).to receive(:menu_items).and_return([
        Spree::BackendConfiguration::MenuItem.new(
          label: :item1,
          icon: 'ri-inbox-line',
          condition: -> {},
          match_path: %r{/foo},
          url: :foo,
        ),
        Spree::BackendConfiguration::MenuItem.new(
          label: :item2,
          icon: 'shopping-cart',
          condition: -> {},
          match_path: -> {},
          url: "/bar"
        ),
        Spree::BackendConfiguration::MenuItem.new(
          label: :item3,
          icon: 'shopping-cart',
          condition: -> {},
          match_path: "foo",
          url: -> {},
        ),
        Spree::BackendConfiguration::MenuItem.new(
          label: :item4,
          icon: 'shopping-cart',
          condition: -> {},
          match_path: nil,
          url: nil,
          children: [
            Spree::BackendConfiguration::MenuItem.new(
              label: :item4_1,
              icon: 'shopping-cart',
              condition: -> {},
              match_path: nil,
              url: nil,
            ),
          ],
        ),
      ])

      config = described_class.new
      config.import_menu_items_from_backend!

      expect(config.menu_items).to match([
        {
          key: :item1,
          icon: 'inbox-line',
          route: a_kind_of(Proc),
          children: [],
          match_path: a_kind_of(Proc),
          position: 0,
        },
        {
          key: :item2,
          icon: 'record-circle-line',
          route: a_kind_of(Proc),
          children: [],
          match_path: a_kind_of(Proc),
          position: 1,
        },
        {
          key: :item3,
          icon: 'record-circle-line',
          route: a_kind_of(Proc),
          children: [],
          match_path: a_kind_of(Proc),
          position: 2,
        },
        {
          key: :item4,
          icon: 'record-circle-line',
          route: a_kind_of(Proc),
          children: [
            {
              key: :item4_1,
              icon: 'record-circle-line',
              route: a_kind_of(Proc),
              children: [],
              match_path: a_kind_of(Proc),
              position: 0,
            },
          ],
          match_path: a_kind_of(Proc),
          position: 3,
        },
      ])
    end
  end
end
