/*
 * Copyright (C) 2020 Nikolaos Garanis <nikolaos.garanis@heig-vd.ch>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

/*
 * Helper to deal with the PS/2 communication protocol.
 *
 * Documentation:
 *   https://wiki.osdev.org/Mouse_Input
 *   https://wiki.osdev.org/PS/2_Mouse
 *   https://wiki.osdev.org/PS/2_Keyboard
 */

#if 0
#define DEBUG
#endif

#include <common.h>
#include <device/input/ps2.h>

#define GET_DX(state, x) ((x) - (((state) << 4) & 0x100))
#define GET_DY(state, y) ((y) - (((state) << 3) & 0x100))

/* Keyboard scan codes */
uint8_t sc[] = {
/*         0     1     2     3     4     5     6     7     8     9     A     B     C     D     E     F */
/* 0 */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, '\t', 0x00, 0x00,
/* 1 */ 0x00, 0x00, 0x00, 0x00, 0x00,  'q',  '1', 0x00, 0x00, 0x00,  'y',  's',  'a',  'w',  '2', 0x00,
/* 2 */ 0x00,  'c',  'x',  'd',  'e',  '4',  '3', 0x00, 0x00,  ' ',  'v',  'f',  't',  'r',  '5', 0x00,
/* 3 */ 0x00,  'n',  'b',  'h',  'g',  'z',  '6', 0x00, 0x00, 0x00,  'm',  'j',  'u',  '7',  '8', 0x00,
/* 4 */ 0x00,  ',',  'k',  'i',  'o',  '0',  '9', 0x00, 0x00,  '.',  '-',  'l', 0x00,  'p', '\'', 0x00,
/* 5 */ 0x00, 0x00, 0x00, 0x00, 0x00,  '^', 0x00, 0x00, 0x00, 0x00, '\n',  '"', 0x00,  '$', 0x00, 0x00,
/* 6 */ 0x00,  '<', 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
/* 7 */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1b, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
/* 8 */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
/* 9 */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
/* A */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
/* B */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
/* C */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
/* D */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
/* E */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
/* F */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

/* Keyboard scan codes + shift modifier */
uint8_t scs[] = {
/*         0     1     2     3     4     5     6     7     8     9     A     B     C     D     E     F */
/* 0 */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
/* 1 */ 0x00, 0x00, 0x00, 0x00, 0x00,  'Q',  '+', 0x00, 0x00, 0x00,  'Y',  'S',  'A',  'W',  '"', 0x00,
/* 2 */ 0x00,  'C',  'X',  'D',  'E', 0x00,  '*', 0x00, 0x00,  ' ',  'V',  'F',  'T',  'R',  '%', 0x00,
/* 3 */ 0x00,  'N',  'B',  'H',  'G',  'Z',  '&', 0x00, 0x00, 0x00,  'M',  'J',  'U',  '/',  '(', 0x00,
/* 4 */ 0x00,  ';',  'K',  'I',  'O',  '=',  ')', 0x00, 0x00,  ':',  '_',  'L', 0x00,  'P',  '?', 0x00,
/* 5 */ 0x00, 0x00, 0x00, 0x00, 0x00,  '`', 0x00, 0x00, 0x00, 0x00, 0x00,  '!', 0x00, 0x00, 0x00, 0x00,
/* 6 */ 0x00,  '>', 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
/* 7 */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
/* 8 */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
/* 9 */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
/* A */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
/* B */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
/* C */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
/* D */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
/* E */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
/* F */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

/* Keyboard scan codes + extended modifier */
uint8_t sce[] = {
/*         0     1     2     3     4     5     6     7     8     9     A     B     C     D     E     F */
/* 0 */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
/* 1 */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
/* 2 */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
/* 3 */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
/* 4 */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
/* 5 */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
/* 6 */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0x00,
/* 7 */ 0x00, 0x7f, 0x12, 0x00, 0x13, 0x11, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
/* 8 */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
/* 9 */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
/* A */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
/* B */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
/* C */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
/* D */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
/* E */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
/* F */ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};


/* Extract mouse coordinates and button states from the given packet. */
void get_mouse_state(uint8_t *packet, struct ps2_mouse *state, uint16_t max_x, uint16_t max_y)
{
	/* Discard packet if there are overflows, as advised. */
	if ((packet[PS2_STATE] & AO) && ~(packet[PS2_STATE] & Y_OF) && ~(packet[PS2_STATE] & X_OF)) {

		/*
		 * Do not unset the button value, this is done in the ioctl.
		 * This way the driver has time to read the button state values.
		 */

		state->left = packet[PS2_STATE] & BL;
		state->right = packet[PS2_STATE] & BR;
		state->middle = packet[PS2_STATE] & BM;

		/* Retrieve dx and dy values to compute the mouse coordinates. */
		state->x += GET_DX(packet[PS2_STATE], packet[PS2_X]);
		state->y -= GET_DY(packet[PS2_STATE], packet[PS2_Y]);

		state->x = CLAMP(state->x, 0, max_x);
		state->y = CLAMP(state->y, 0, max_y);

		DBG("sign_dxy[%s, %s], dxy_val[%03u, %03u], computed_dxy[%03d, %03d]; xy[%03d, %03d]; %03s %03s %03s\n",
			packet[PS2_STATE] & X_BS ? "neg" : "pos",
			packet[PS2_STATE] & Y_BS ? "neg" : "pos",
			packet[PS2_X],
			packet[PS2_Y],
			GET_DX(packet[PS2_STATE], packet[PS2_X]),
			GET_DY(packet[PS2_STATE], packet[PS2_Y]),
			state->x, state->y,
			state->left ? "LFT" : "", state->middle ? "MID" : "", state->right ? "RGT" : "");
	}
	else {
		printk("%s: packet discarded.\n", __func__);
	}
}

/*
 * Parsing keyboard scan codes is a complex task but here we want only some
 * basic features such as:
 *   - lowercase letters
 *   - uppercase letters (using the shift modifier key)
 *   - some control keys (arrows, backspace, delete, etc.)
 */
void get_kb_key(uint8_t *packet, uint8_t len, struct ps2_key *key)
{
	uint8_t i = 0;

	/* If the first byte is KEY_EXT then we'll be using the extended scan codes array. */
	key->state &= ~KEY_ST_EXTENDED;
	if (packet[i] == KEY_EXT) {
		key->state |= KEY_ST_EXTENDED;
		i++;
	}

	/* Change the pressed/released state bit. */
	key->state |= KEY_ST_PRESSED;
	if (packet[i] == KEY_REL) {
		key->state &= ~KEY_ST_PRESSED;
		i++;
	}

	/* Change the shift modifier bit (left or right shift key). */
	if (packet[i] == KEY_LSH || packet[i] == KEY_RSH) {
		key->value = 0;
		key->state ^= KEY_ST_SHIFT;
	}
	else {
		/* Convert the scan code to the wanted value. */
		if (key->state & KEY_ST_SHIFT) {
			key->value = scs[packet[i]];
		}
		else if (key->state & KEY_ST_EXTENDED) {
			key->value = sce[packet[i]];
		}
		else {
			key->value = sc[packet[i]];
		}
	}

	DBG("%s sc: 0x%02x v: %c\n",
		key->state & KEY_ST_PRESSED ? "pressed" : "release",
		packet[i],
		key->value ? key->value : ' ');
}

