/*
 * Copyright (C) 2014-2018 Daniel Rossier <daniel.rossier@heig-vd.ch>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#include <avz/domain.h>

#include <asm/mmu.h>
#include <asm/processor.h>

#include <generated/asm-offsets.h>

.extern hypervisor_stack

.align  5
ENTRY(ret_from_hypercall)

	disable_irq

	/* Store the x0 value in the stack frame of G-stack */
	curdom	x8, x11
	ldr		x9, [x8, #OFFSET_G_SP]
	str		x0, [x9, #OFFSET_X0]

	bl		do_softirq

	curdom	x10, x11
	ldr		x11, [x10, #OFFSET_AVZ_SHARED]

	// If the softirq handling leads to trigger an interrupt in the guest,
	// it will be processed by do_evtchn_do_upcall. The way how to
	// process an interrupt with potentially IRQs off is under the
	// responsibility of the guest

	mov		x8, #0

	// are some IRQs pending?
	ldrb	w12, [x11, #OFFSET_EVTCHN_UPCALL_PENDING]
	tst		w12, #0xff

	b.eq	restore

	// Will go ahead with evtchn_do_upcall in the domain
	mov		x8, #1

restore:
	disable_irq

	current_cpu x11

#ifndef CONFIG_ARM64VT
	// Setting pseudo_usr_mode / x0, x1 re-assigned right after
	ldr 	x0, .LCpseudo_usr_mode
	mov		x1, #1
	str		x1, [x0, x11, lsl #3]
#endif

	// Restore saved registers

	ldr		x0, .LChypervisor_stack   	// running SVC hypervisor stack
	mov		x10, sp
	str		x10, [x0, x11, lsl #3]

	curdom	x10, x11

	// Get guest stack
	ldr		x11, [x10, #OFFSET_G_SP]
	mov		sp, x11

	// Store the evtchn pending state to give
	// the domain the possiblity to call evtchn_do_upcall()
	str 	x8, [sp, #OFFSET_X8]

	ldr		lr, [sp, #OFFSET_PC]

	// PSTATE contains the daif state at the entry in
	// the hypervisor.

	ldr		x9, [sp, #OFFSET_PSTATE]  // Get PSTATE

	cmp		x8, #0
	b.ne	save_pstate

restore_pstate:

	// Restore the IRQ status before the hypercall
	msr		daif, x9
	b		cont

save_pstate:
	str		x9, [sp, #OFFSET_X9]

cont:

	ldp		x0, x1, [sp, #OFFSET_X0]
	ldp		x2, x3, [sp, #OFFSET_X2]
	ldp		x4, x5, [sp, #OFFSET_X4]
	ldp		x6, x7, [sp, #OFFSET_X6]
	ldp		x8, x9, [sp, #OFFSET_X8]
	ldp		x10, x11, [sp, #OFFSET_X10]
	ldp		x12, x13, [sp, #OFFSET_X12]
	ldp		x14, x15, [sp, #OFFSET_X14]
	ldp		x16, x17, [sp, #OFFSET_X16]
	ldp		x18, x19, [sp, #OFFSET_X18]
	ldp		x20, x21, [sp, #OFFSET_X20]
	ldp		x22, x23, [sp, #OFFSET_X22]
	ldp		x24, x25, [sp, #OFFSET_X24]
	ldp		x26, x27, [sp, #OFFSET_X26]
	ldp		x28, x29, [sp, #OFFSET_X28]

	add		sp, sp, #S_FRAME_SIZE

	ret

.align  5

// Hypercall generated by the domain
// Always issued from a guest trampoline or from avz directly.
// Hence, we preserve the daif state at the entry in order
// to reconfigure correctly at the hypercall exit.

ENTRY(hypercall_entry)

	// According to AArch64 ABI, x8 is preserved by the caller if used.
	mrs		x8, daif

	disable_irq

	sub		sp, sp, #S_FRAME_SIZE

	str		x8, [sp, #OFFSET_PSTATE]

	stp		x0, x1, [sp, #OFFSET_X0]
	stp		x2, x3, [sp, #OFFSET_X2]
	stp		x4, x5, [sp, #OFFSET_X4]
	stp		x6, x7, [sp, #OFFSET_X6]
	stp		x8, x9, [sp, #OFFSET_X8]
	stp		x10, x11, [sp, #OFFSET_X10]
	stp		x12, x13, [sp, #OFFSET_X12]
	stp		x14, x15, [sp, #OFFSET_X14]
	stp		x16, x17, [sp, #OFFSET_X16]
	stp		x18, x19, [sp, #OFFSET_X18]
	stp		x20, x21, [sp, #OFFSET_X20]
	stp		x22, x23, [sp, #OFFSET_X22]
	stp		x24, x25, [sp, #OFFSET_X24]
	stp		x26, x27, [sp, #OFFSET_X26]
	stp		x28, x29, [sp, #OFFSET_X28]

	mov		x22, lr

	str		lr, [sp, #OFFSET_LR]
	str		x22, [sp, #OFFSET_PC]

	current_cpu x10

#ifndef CONFIG_ARM64VT

	// Check if we are coming from avz or the guest
	ldr		x8, .LCpseudo_usr_mode

	ldr		x9, [x8, x10, lsl #3]
	cmp		x9, #0
#endif

	b.eq	__do_hypercall_from_avz

	// readjust stack for future restore
	add		x8, sp, #S_FRAME_SIZE

	str		x8, [sp, #OFFSET_SP]

	str		lr, [sp, #OFFSET_LR]

#ifndef CONFIG_ARM64VT
	ldr		x8, .LCpseudo_usr_mode
	mov		x9, #0
	str		x9, [x8, x10, lsl #3]
#endif

	mov		x8, sp
	ldr 	x9, .LChypervisor_stack   // Get the running hypervisor EL1 stack
	ldr		x11, [x9, x10, lsl #3]
	mov		sp, x11

	curdom	x9, x11
	str		x8, [x9, #OFFSET_G_SP]


process_hypercalls:

	adr	lr, ret_from_hypercall
	adr	x11, hypercall_table
	add	x11, x11, x7, lsl #3
	ldr	x10, [x11]

	br	x10		// Perform the hypercall

	// x0 has the return value of the hypercall

// This is the entry point in case an hypercall is issued from the hypervisor
__do_hypercall_from_avz:

	adr lr, __end_hypercall_from_avz

	adr	x11, hypercall_table
	add	x11, x11, x7, lsl #3

	ldr	x10, [x11]

	br	x10		// Perform the hypercall

	// x0 has the return value of the hypercall

// Fast return path to the caller (in avz)
__end_hypercall_from_avz:

	ldr		lr, [sp, #OFFSET_PC]

	ldr		x1, [sp, #OFFSET_X1]
	ldp		x2, x3, [sp, #OFFSET_X2]
	ldp		x4, x5, [sp, #OFFSET_X4]
	ldp		x6, x7, [sp, #OFFSET_X6]
	ldp		x8, x9, [sp, #OFFSET_X8]
	ldp		x10, x11, [sp, #OFFSET_X10]
	ldp		x12, x13, [sp, #OFFSET_X12]
	ldp		x14, x15, [sp, #OFFSET_X14]
	ldp		x16, x17, [sp, #OFFSET_X16]
	ldp		x18, x19, [sp, #OFFSET_X18]
	ldp		x20, x21, [sp, #OFFSET_X20]
	ldp		x22, x23, [sp, #OFFSET_X22]
	ldp		x24, x25, [sp, #OFFSET_X24]
	ldp		x26, x27, [sp, #OFFSET_X26]
	ldp		x28, x29, [sp, #OFFSET_X28]

	add		sp, sp, #S_FRAME_SIZE

	ret

/*
 * hypercall_table
 */
.type hypercall_table, #object
ENTRY(hypercall_table)

__hypercall_entry:
	.quad do_event_channel_op
	.quad do_console_io
	.quad do_physdev_op
	.quad do_sched_op
	.quad do_domctl
	.quad do_soo_hypercall

__hypercall_end:

#ifndef CONFIG_ARM64VT
.LCpseudo_usr_mode:
	.quad	pseudo_usr_mode
#endif

.LChypervisor_stack:
	.quad	hypervisor_stack


